unit Sapphire;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TSapphire = class(TComponent)
  private
    { Private declarations }
    rotor,                            { Rotor and ratchet are used to help }
    ratchet,                          { Continually shuffle the "cards."   }
    avalanche,                        { Data dependent index. }
    last_plain,                       { Previous plain text byte. }
    last_cipher: byte;                { Previous cipher text byte. }
    cards: array[0..255] of byte;     { Array to hold a permutation of all }
    UserKey: string;
    procedure hash_init;              { Initialize state without key. }
  protected
    { Protected declarations }
  public
    { Public declarations }
    procedure SetKey(const user_key: string); { Initialize state with key. }
    procedure Reoriginate;            { Restart stream. }
    procedure HashInit;               { Prepare to compute hash. }
    function encrypt(const b: byte): byte;  { Encrypt b and return the result. }
    function decrypt(const b: byte): byte;  { Decrypt b and return the result. }
    procedure EncryptString(var s: string);
    procedure DecryptString(var s: string);
    function hash_final: string;      { Compute final hash value. }
    procedure burn;                   { Destroy internal key information. }
    function RandomByte: byte;
    function RandomFloat: extended;
    function RandomInt(const Range: Longint): Longint;
    function RandomInteger: Longint;
    function RandomString: string;
    procedure PoolByte(const b: byte);      { "Pool" procedures "stir" entropy into}
    procedure PoolInt(const li: longint);   { pool of random numbers. }
    procedure PoolFloat(r: extended);
    procedure PoolString(const s: string);
    constructor Create(Aowner: TComponent); override;
    destructor Destroy; override;

    property RandByte: byte read RandomByte write PoolByte;
    property RandFloat: extended read RandomFloat write PoolFloat;
    property RandInteger: longint read RandomInteger write PoolInt;
    property RandString: string read RandomString write PoolString;

  published
    { Published declarations }
    property Key: string read UserKey write SetKey;
    
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('MPJ', [TSapphire]);
end;

procedure TSapphire.hash_init;
  var i, j: integer;
  begin
    rotor := 1;           { Make sure we start key in a known place. }
    ratchet := 3;
    avalanche := 5;
    last_plain := 7;
    last_cipher := 11;

    j := 255;
    for i := 0 to 255 do  { Start with cards all in inverse order. }
      begin
        cards[i] := j;
        dec(j);
      end;
  end;

procedure TSapphire.HashInit;
begin
    UserKey := '';
    hash_init;
end;

procedure TSapphire.Reoriginate;
  { Sets up key for use by the stream cipher, based on user_key. }
  var keypos: integer;    { Index of current byte of user key. }
    toswap,               { Index for shuffling cards with. }
    swaptemp,             { For holding bytes while swapping. }
    rsum,                 { Function of key bytes processed to now. }
    i, j: byte;           { Indexes into key.cards. }

    function keyrand(const limit: integer): byte;
    var u, v, mask: word;
    begin
    if limit = 0 then
        keyrand := 0
    else
        begin
        v := 0;
        mask := 1;
        while mask < limit do
            mask := (mask shl 1) + 1;
        repeat
            rsum := (cards[rsum] + ord(UserKey[keypos])) and $FF;
            inc(keypos);
            if keypos > Length(UserKey) then
                begin
                keypos := 1;
                rsum := (rsum + Length(UserKey)) and $FF;
                end;
            u := mask and rsum;
            inc(v);
            if v > 11 then
                u := u mod limit;
        until u <= limit;
        keyrand := u;
        end;
    end;

  begin
    if length(UserKey) = 0 then
      hash_init
    else
      begin
        for i := 0 to 255 do  { Start with cards all in order. }
        cards[i] := i;
        keypos := 1;        { Start with first byte of user  }
        toswap := 0;
        rsum := 0;
        for i := 255 downto 1 do  { Swap elements.  Some may get swaped back }
          begin                   { and some may get swapped with self. }
            toswap := keyrand(i);
            swaptemp := cards[i];
            cards[i] := cards[toswap];
            cards[toswap] := swaptemp;
          end;
        rotor := cards[1];           { Make sure we start key in a known place. }
        ratchet := cards[3];
        avalanche := cards[5];
        last_plain := cards[7];
        last_cipher := cards[rsum];
        toswap := 0;
        swaptemp := 0;
        rsum := 0;
        keypos := 0;
      end
  end;

procedure TSapphire.SetKey(const user_key: string);
begin
    UserKey := user_key;
    Reoriginate;
end;

constructor TSapphire.Create(AOwner: TComponent);
begin
    Inherited Create(AOwner);
    Reoriginate;
end;

function TSapphire.encrypt(const b: byte): byte;
  var swaptemp: byte;
  begin
    ratchet := (ratchet + cards[rotor]) and $FF;
    inc(rotor);
    swaptemp := cards[last_cipher];      { Round-robin swap. }
    cards[last_cipher] := cards[ratchet];
    cards[ratchet] := cards[last_plain];
    cards[last_plain] := cards[rotor];
    cards[rotor] := swaptemp;
    avalanche := (avalanche + cards[swaptemp]) and $FF;
    last_cipher := b xor
                 cards[(cards[ratchet] + cards[rotor]) and $FF] xor
                 cards[cards[(cards[last_plain] +
                              cards[last_cipher] +
                              cards[avalanche]) and $FF]];
    last_plain := b;
    encrypt := last_cipher;
  end;

function TSapphire.decrypt(const b: byte): byte;
  var swaptemp: byte;
  begin
    ratchet := ratchet + cards[rotor];
    inc(rotor);
    swaptemp := cards[last_cipher];
    cards[last_cipher] := cards[ratchet];
    cards[ratchet] := cards[last_plain];
    cards[last_plain] := cards[rotor];
    cards[rotor] := swaptemp;
    avalanche := avalanche + cards[swaptemp];
    last_plain := b xor
                  cards[(cards[ratchet] + cards[rotor]) and $FF] xor
                  cards[cards[(cards[last_plain] +
                               cards[last_cipher] +
                               cards[avalanche]) and $FF]];
    last_cipher := b;
    decrypt := last_plain;
  end;

procedure TSapphire.EncryptString(var s: string);
var i: integer;
begin
    for i := 1 to length(s) do
        s[i] := chr(encrypt(ord(s[i])));
end;

procedure TSapphire.DecryptString(var s: string);
var i: integer;
begin
    for i := 1 to length(s) do
        s[i] := chr(decrypt(ord(s[i])));
end;

function TSapphire.hash_final: string;
  var i: integer;
  begin
    for i := 255 downto 0 do
      encrypt(i);
    result := '';
    for i := 1 to 20 do
      result := result + chr(encrypt(0));
  end;

procedure TSapphire.burn;
var i: integer;
  begin
    fillchar(cards, sizeof(cards), 0);
    for i := 1 to length(UserKey) do
        UserKey[i] := chr(0);
    UserKey := '';
    hash_init;
  end;

destructor TSapphire.Destroy;
begin
    burn;
    Inherited Destroy;
end;

function TSapphire.RandomByte: byte;
begin
    result := encrypt(0);
end;

function TSapphire.RandomFloat: extended;
var i: integer;
begin
    result := 0.0;
    for i := 1 to 8 do
        result := (result + encrypt(0)) / 256.0;
end;

function TSapphire.RandomInteger: longint;
begin
    result := encrypt(0);
    result := (result shl 8) + encrypt(0);
    result := (result shl 8) + encrypt(0);
    result := (result shl 8) + encrypt(0);
end;

function TSapphire.RandomInt(const Range: Longint): Longint;
var r: extended;
begin
    result := trunc(Range * RandomFloat);
    if result >= Range then result := Range;
end;

function TSapphire.RandomString: string;
var i, j: integer;
begin
    result := '';
    for i := 1 to 65 do
        begin
        j := RandomInt(62)+48;
        if j > 57 then j := j + 7;
        if j > 90 then j := j + 6;
        result := result + chr(j);
        end;
end;

procedure TSapphire.PoolByte(const b: byte);
begin
    encrypt(b);
end;

procedure  TSapphire.PoolInt(const li: longint);
begin
    encrypt(byte(li and $FF));
    encrypt(byte((li shr 8) and $FF));
    encrypt(byte((li shr 16) and $FF));
    encrypt(byte((li shr 24) and $FF));
end;

procedure TSapphire.PoolFloat(r: extended);
var i: integer;
    b: byte;
begin
    if r < 0.0 then
        begin
        r := -r;
        encrypt(7);
        end;
    while r >= 256.0 do
        begin
        r := r / 256.0;
        encrypt(13);
        end;
    while r < 1.0 do
        begin
        r := r * 256.0;
        encrypt(17);
        end;
    for i := 1 to 8 do
        begin
        b := trunc(r);
        encrypt(b);
        r := (r - b) * 256.0;
        end;
end;

procedure TSapphire.PoolString(const s: string);
var i: integer;
begin
    for i := 1 to length(s) do
        encrypt(ord(s[i]));
end;

end.
