/*********************************************************************
**
**     File name:               ssh_prepare.c
**
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 prepare the ssh connection (pty, exec shell)
**
**     Author/Date:             Tadayoshi Kohno, 21 February 1998
**
**     Notes:
**	These functions are to be called after authentication
**	but before switching to an interactive xxx process (via the request
**	for a shell).
**
**     Functions:
**	ssh_request_compression	request compression
**	ssh_request_pty		request a pty
**	ssh_request_exec_shell	request a shell
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_prepare.c,v 3.11 1998/05/09 16:30:34 kohno Exp $";
#endif

#include "ssh.h"
#include "ssh_types.h"

#include "ssh_prepare.h"
#include "ssh_compress.h"

/******************************************************************************
**
**     File/Function name:	ssh_request_compression
**
**     Purpose:			request compression
**
**     Preconditions:		connected, authenticated
**				shell not requested yet.
**
**				level between SSH_COMPRESS_{MIN,MAX}_LEVEL
**
**     Parameters:		sockfd		socket to server
**				ssh_info	info about connection
**
**				level		compression level
**
**     Exit (post) conditions:	SSH_COMPRESSION_STARTED
**				SSH_COMPRESSION_NOT_STARTED
**
**     Error conditions:	S_BAD		error in request
**
**     Side effects:		compression started (if server says OK)
**
**     Author/Date:		Tadayoshi Kohno, 4 April 1998
**
**     References:
**
**     Notes:
**	The reason we do not simply return S_GOOD or S_BAD is that the
**	server can legally deny compession.  This is different from
**	an internal error (like a recv() error).  For this reason,
**	we return SSH_COMPRESSION_{NOT_}STARTED for starting or not
**	starting compression and S_BAD for an error
**
**	The compression level can be anywhere between SSH_COMPRESS_MIN_LEVEL
**	and SSH_COMPRESS_MAX_LEVEL.  SSH_COMPRESS_DEFAULT_LEVEL may be used
**	for a nice balance between speed and efficiency (as per zlib 1.1.2
**	recommendations).
**
******************************************************************************/

int ssh_request_compression
(
	socket_type sockfd,		/* socket to server */
	struct ssh_struct * ssh_info,	/* information about ssh connection */

	uint32_t level			/* compression level */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data for packet */
	uint32_t data_len;		/* length of data */

	/*
	**	make sure the ssh_info struct is valid
	*/
	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	/*
	**	make sure compression level is OK
	*/
	if (level < SSH_COMPRESS_MIN_LEVEL || level > SSH_COMPRESS_MAX_LEVEL)
	{
		ssh_errno_set(SSH_ERRNO_COMPRESS_LEVEL);
		return(S_BAD);
	}

	/*
	**	pack a compression request
	*/
	if (ssh_cmsg_request_compression_encode(data, &data_len, level))
	{
		ssh_errno_set(SSH_ERRNO_ENCODE_MESSAGE);
		return(S_BAD);
	}

	/*
	**	send off the request
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len,
		SSH_CMSG_REQUEST_COMPRESSION))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending compression request",
			"ssh_request_compression");

		/* propagate ssh_errno */
		return(S_BAD);
	}

	/*
	**	try to recieve a reply
	*/
	switch (ssh_recv(sockfd, ssh_info, data, &data_len))
	{
		case SSH_SMSG_SUCCESS:
			break;

		case SSH_SMSG_FAILURE:
			ssh_debugger_new(&(ssh_info->debug_info),
				"compresion request denied",
				"ssh_request_compression");
			
			return(SSH_COMPRESSION_NOT_STARTED);
			break;

		default:
			ssh_debugger_new(&(ssh_info->debug_info),
				"error with compression request",
				"ssh_request_compression");

			ssh_errno_set(SSH_ERRNO_REQUEST_COMPRESS);
			return(S_BAD);
	}

	/*
	**	rather sloppy programming, but we get here if 
	**	our compression request is granted
	*/
	ssh_debugger_new(&(ssh_info->debug_info),
		"compression request accepted", "ssh_request_compression");

	/*
	**	now, tell our ssh_compress file to start compression
	**	this will probably be setting some flag/variables in
	**	the ssh_struct.
	**
	**	once we turn on compression, everything will be
	**	compressed (so we need to put this into ssh_send, ssh_recv)
	**	this is similar to the cipher in that the cipher starts out
	**	as NONE and changes to whatever we request (provided the
	**	server says OK).
	*/
	/* xxx xxx xxx xxx xxx xxx xxx */
	if (ssh_compression_set(ssh_info, level) == S_BAD)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"compression initialization failed",
			"ssh_request_compression");

		/* propogate ssh_errno */
		return(S_BAD);
	}

	ssh_debugger_new(&(ssh_info->debug_info),
		"compression initialized", "ssh_request_compression");

	return(SSH_COMPRESSION_STARTED);

}


/******************************************************************************
**
**     File/Function name:	ssh_request_pty
**
**     Purpose:			request a pty
**
**     Preconditions:		connected, authenticated
**				shell not requested yet
**
**     Parameters:		sockfd		socket to server
**				ssh_info	info about current connection
**
**				terminal_type	terminal type ("vt100", ...)
**				height		terminal height in rows
**				width		terminal width in columns
**				x_pixels	width in x_pixels
**				y_pixels	height in y_pixels
**
**				tty_modes	tty modes in binary format
**				tty_mode_len	bytes for tty_modes
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		error in request
**				ssh_errno set
**
**     Side effects:		pty requested and granted from server
**
**     Author/Date:		Tadayoshi Kohno, 21 February 1998
**
**     Notes:
**
******************************************************************************/

int ssh_request_pty
(
	socket_type sockfd,		/* socket to talk over */
	struct ssh_struct * ssh_info,	/* information about connection */

	const char * terminal_type,	/* terminal type */
	uint32_t height,		/* height of terminal */
	uint32_t width,			/* width of terminal */
	uint32_t x_pixels,		/* width in pixels */
	uint32_t y_pixels,		/* height in pixels */

	const uint8_t * tty_modes,	/* tty mode encoded in binary */
	int tty_mode_len		/* length of *tty_modes */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data for packet */
	uint32_t data_len;		/* length of data */

	int err;			/* error code */


	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	/*
	**	pack a pty request
	*/
	if (ssh_cmsg_request_pty_encode(data, &data_len, terminal_type,
		height, width, x_pixels, y_pixels, tty_modes, tty_mode_len))
	{
		ssh_errno_set(SSH_ERRNO_ENCODE_MESSAGE);
		return(S_BAD);
	}

	/*
	**	send the request off
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len, SSH_CMSG_REQUEST_PTY))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending request for pty", "ssh_request_pty");

		/* propagate ssh_errno */
		return(S_BAD);
	}

	/*
	**	try to recieve a reply
	*/
	if ((err = ssh_recv(sockfd, ssh_info, data, &data_len))
		!= SSH_SMSG_SUCCESS)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error receiving reply to pty request",
			"ssh_request_pty");

		if (err != SSH_MSG_ERROR)
		{
			ssh_errno_set(SSH_ERRNO_REQUEST_PTY);
		}

		/* propagate ssh_errno */
		return(S_BAD);
	}

	ssh_debugger_new(&(ssh_info->debug_info),
		"request for pty granted", "ssh_request_pty");

	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_request_exec_shell
**
**     Purpose:			request a shell from the server
**
**     Preconditions:		user authenticated
**
**     Parameters:		sockfd		socket to server
**				ssh_info	info about current connection
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		error forming request
**						or sending request
**				ssh_errno set
**
**     Side effects:		request for shell sent to server.
**				client should now enter interactive mode xxx.
**
**     Author/Date:		Tadayoshi Kohno, 21 February 1998
**
**     Notes:
**
******************************************************************************/

int ssh_request_exec_shell
(
	socket_type sockfd,		/* socket to server */
	struct ssh_struct * ssh_info	/* info about current connection */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data for packet */
	uint32_t data_len;		/* length of data */

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	/*
	**	compose request for a shell
	*/
	if (ssh_cmsg_exec_shell_encode(data, &data_len))
	{
		ssh_errno_set(SSH_ERRNO_ENCODE_MESSAGE);
		return(S_BAD);
	}
	
	/*
	**	send off request for shell
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len,
		SSH_CMSG_EXEC_SHELL))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending request for a shell",
			"ssh_request_exec_shell");

		return(S_BAD);
	}

	/*
	**	request is off and now we can enter an interactive session
	*/
	ssh_debugger_new(&(ssh_info->debug_info),
		"shell requested", "ssh_request_exec_shell");

	return(S_GOOD);
}

