
/*
 *
 * The statistics collection program -- finds out who is reading and
 * subscribing to what.
 *
 * Copyright 1990 by Looking Glass Software Limited.
 * See the file READ.ME for licence details.
 */

#include <stdio.h>
#include "db.h"
#include <sys/types.h>
#include <sys/stat.h>
#include "arbit.h"
#include <ctype.h>


long time_now;				/* current date */

int newsreaders = 0;
boolean statistics = FALSE;		/* collect statistics, too */
boolean arbitron = FALSE;		/* arbitron style report */
boolean full_report = FALSE;		/* full stat report */
char *prefix = NULL;			/* groups with this prefix */
int prefix_len = 0;

main( argc, argv )
int argc;	/* arg count */
char **argv;	/* arg vector */
{
	int argnum;
	char *strchr();
	FILE *inrc;
	FILE *rclist;

	init_bases();
	time( &time_now );
	read_active();

	for( argnum = 1; argnum < argc; argnum++ ) {
		char *argline;
		char *argstr;		/* argument string */
		int argval;
		int isplus;		/* boolean tells +arg vs -arg */
		argline = argv[argnum];

		if (argstr = strchr(argline, '=')) {
			argstr++;
			argval = atoi(argstr);
			switch( argline[0] ) {
				case 's':
					rclist = fopen( argstr, "r" );
					if( !rclist )
						error( "Could not open %s\n", argstr );
					do_rclist( rclist, FALSE );
					break;
				case 'p':
					/* only do groups with this prefix */
					prefix = argstr;
					prefix_len = strlen( prefix );
					break;
				default:
					error( "Bad Option %s\n", argline );
				}
			}
		else if( (isplus = argline[0] == '+') || argline[0] == '-' ) {
			switch( argline[1] ) {
				case 'a': /* arbitron style */
					arbitron = isplus;
					break;
				case 'f': /* full style report */
					statistics = full_report = isplus;
					break;
				default:
					error( "Bad Option %s\n", argline );
				}
			}
		else {
			/* code for untagged option */
			}
		}
	/* body of program */
	do_rclist( stdin, TRUE );

	print_report();
	
}

/* do a list of newsrc files */

do_rclist( desc, user )
FILE *desc;
boolean user;
{
	char buf[MAX_LLEN];
	FILE *rcf;
	struct stat sbuf;

	while( fgets( buf, sizeof(buf), desc ) ) {
		stripnl( buf );
		rcf = fopen( buf, "r" );
		if( !rcf ) {
			fprintf( stderr, "Could not open newsrc: %s\n", buf );
			continue;
			}
		/* don't do any .newsrc over 25 days old */
		if( fstat( fileno(rcf), &sbuf ) == 0 &&
				time_now - sbuf.st_mtime < 25 * DAY )
			do_a_newsrc( rcf, user );
		fclose( rcf );
		}
	fclose( desc );
}

dbptr newsgroups;

/* prepare the databases */

init_bases()
{
	newsgroups = init_db( 1200, sizeof( group ) );
}
	

/* feeder's list of patterns to exclude & include
   recipient's list of patterns to include & exclude
   recipient's list of specific groups desired
 */


error( form, a, b, c, d, e, f, g, h )
char *form;
{
	fprintf( stderr, form, a,b,c,d,e,f,g,h );
	exit(1);
}


/* Read the active file to note ranges of groups */

read_active()
{
	FILE *active;
	char actbuf[MAX_GNAME+40];
	char *groupp, *highp, *lowp, *flagp;
	group *grp;
	char *dir;
	struct stat sbuf;

	active = fopen( ACTFILE, "r" );
	if( !active )
		error( "Could not open %s\n", ACTFILE );
	
	while( fgets( actbuf, sizeof(actbuf), active ) ) {
		groupp = strtok( actbuf, " \t" );
		highp = strtok( NULL, " \t" );
		lowp = strtok( NULL, " \t" );
		flagp = strtok( NULL, " \t" );
		if( !( groupp && highp && lowp && flagp ) ) {
			fprintf( stderr, "Invalid active file line\n" );
			break;
			}
		grp = (group *)add_rec( newsgroups, groupp, AR_CREATE );
		if( statistics ) {
			dir = make_spoolname( groupp );
			/* record access time for group */
			if( stat( dir, &sbuf ) == 0 ) {
				grp->group_date = sbuf.st_mtime;
				}
		 	else	/* group not even created yet? */
				grp->group_date = time_now;
			}
		grp->gflags = 0;
		grp->active_highest = atol( highp );
		grp->active_lowest = atol( lowp );

		/* init this field if need be */
		grp->recent_article = grp->active_lowest;
		
		if( *flagp == 'x' )	/* C news junked group */
			grp->gflags |= GF_NOGROUP;
		/* check for group not updated in 2 months */
		if( time_now - grp->group_date > 60 * DAY ) {
			grp->gflags |= GF_DEAD;
			}
		/* normalize this for new groups */
		if( grp->recent_article < 1 )
			grp->recent_article = 1;

		grp->group_char = *flagp;
		}
	fclose( active );

	/* now read the old version of the active file to get low numbers */
	/* This should be the active file of approximately one month ago, so
	   that we can see where the groups were at that time */

	active = fopen( OLDACTFILE, "r" );
	if( !active ) {
		/* not a fatal error, just use current values */
		fprintf( stderr, "Could not open %s\n", OLDACTFILE );
		return;
		}
	
	while( fgets( actbuf, sizeof(actbuf), active ) ) {
		groupp = strtok( actbuf, " \t" );
		highp = strtok( NULL, " \t" );
		if( !( groupp && highp ) ) {
			fprintf( stderr, "Invalid old active file line\n" );
			break;
			}
		grp = (group *)get_rec( newsgroups, groupp );
		if( grp && atol(highp) < grp->recent_article )
			grp->recent_article = atol( highp );
		}
	fclose( active );

		
}

/* Scan over a .newsrc file and record information */

do_a_newsrc( desc, user )
FILE *desc;
boolean user;				/* is this a user or a system */
{
	char rclbuf[MAX_GNAME+40];	/* no long lines, please! */
	char *p;			/* pointer into rc line */
	char nbuf[15];			/* number reading buf */
	int ch, ddex;			/* storing number from .newsrc */
	group *grp;
	boolean readone;		/* read anything? */


	readone = FALSE;

	while( fgets( rclbuf, sizeof(rclbuf), desc ) ) {
		/* skip option lines and continuation lines */
		if( isspace( rclbuf[0] ) || strncmp( rclbuf, "options ", 8 )==0)
			continue;
		p = strchr( rclbuf, ':' );
		if( p != NULL ) {
			*p = 0;
			if(prefix && strncmp( rclbuf, prefix, prefix_len )==0)
				continue;
			grp = (group *)get_rec( newsgroups, rclbuf );
			if( !grp )
				continue;	/* unknown group? */
			p++;
			ch = *p;
			/* scan through lines, reading off numbers */
			while( ch != '\n' && ch != EOF ) {
				if( isdigit( ch ) && ddex < sizeof(nbuf)-1 ) {
					nbuf[ddex++] = ch;
					nbuf[ddex] = 0;
					}
				 else
					ddex = 0;
				/* read from line or file */
				ch = *p ? *++p : getc( desc );
				}
			if( atol(nbuf) >= grp->recent_article )	{
				grp->recent_readers++;
				readone = TRUE;
				}
			grp->subscribers++;
			if( user )
				grp->user_subscribers++;
			}
		/* else an NN line? */
		}
	/* increment count of people reading news */
	if( user && readone )
		newsreaders++;
}

/* zero out a region of memory */

zero( mem, bytes )
char *mem;
unsigned int bytes;
{
	memset( mem, 0, bytes );
}

static char statspool[MAX_GNAME+sizeof(SPOOLDIR)+1];

/* Turn a group name into a directory name */

char *
make_spoolname( grname )
char *grname;
{
	register char *p;
	sprintf( statspool, "%s/%s", SPOOLDIR, grname );
	for( p = statspool+sizeof(SPOOLDIR); *p; p++ )
		if( *p == '.' )
			*p = '/';
	return statspool;
}

/* output the readership report or subscription request */

print_report()
{
	group *grp;

	if( arbitron )
		printf( "Netreaders\t%d\n", newsreaders );

	for( grp = (group *)first_rec(newsgroups); grp;
					grp = next_rec(newsgroups,grp)) {

		if( arbitron )
			printf( "%d %s\n", grp->recent_readers, grp->name );
		else if( full_report )
			printf( "%-32s%d\t%d\t%d\t%d\n", grp->name,
				grp->subscribers, grp->user_subscribers,
				grp->recent_readers,
				(int)((time_now - grp->group_date)/DAY) );
		else	/* regular report, subscribed groups */
			if( grp->subscribers > 0 )
				printf( ": %s\n", grp->name );
		}
}

/* Remove the trailing nl from a line */

stripnl( buf )
char *buf;
{
	int len;

	len = strlen( buf );
	if( len > 0 && buf[len-1] == '\n' )
		buf[len-1] = 0;
}
