

#include "nl.h"

/*
 * These are the routines that handle the .newsrc file and all the
 * activities related to using it, including reading the active file and
 * the .nglas file.
 */

 /*
  * Newsclip(TM) Library Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip library for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed.
  * Use of this code for a short term evaluation of the product, as defined
  * in the associated file, 'Licence', is permitted.
  */



newsgroup main_newsgroup = NULL_NEWSGROUP;/* the group we are looping through */
int article_number;		/* number of the current article */
extern dbptr ng_base;		/* the database of known newsgroup names */

extern int group_count;		/* count of newsgroups in database */


/*
 * General initialization of newsgroup table
 */


/* First the routine to read the 'las' file */

read_las()
{

	FILE *howfile;
	char buf[MAX_NGLEN+20];		/* buffer for line */
	char ngname[MAX_NGLEN];
	char formats[10];
	long maxnum;
	ngrec *group;
	extern char *lasname;		/* name of las file */
	extern bool newsrc_allread;	/* mark all newsrc as read */

	/* First read the "how far" file that notes the last article
	 * seen by this program for each newsgroup in the .newsrc file.
	 */
	
	sprintf( formats, "%%%ds %%lu\n", MAX_NGLEN );
	
	howfile = fopen( lasname, "r" );

	if( !howfile ) {
		warning( 4, "LAS file %s missing\n", lasname );
		return;
		}

	while( fgets( buf, sizeof(buf), howfile ) ) {
		maxnum = 0;
		/* the maxnum entry might not be read if the last article
		   seen number is not present */
		sscanf( buf, formats, ngname, &maxnum );
		group = (ngrec *)add_rec( ng_base, ngname, AR_CREATE );
		if( group->ngnumber == 0 )
			group->ngnumber = group_count++;
		group->las = (int32)maxnum;
		group->gflag |= GF_LASGROUP;
		}

	fclose( howfile );

}


ngrec *rc_chain;		/* start of chain of groups from .newsrc */


read_newsrc()
{
	FILE *rcfile;			/* stream of the newsrc file */
	char buf[MAX_LLEN];		/* line buffer */
	char *subchar;			/* position of colon in line */
	ngrec *group;			/* record for this group */
	ngrec *lastgroup;		/* record for previous group */
	int len;			/* length of seen list */
	char *bslist;			/* seen list in buffer */
	char *slist;			/* final seen list */
	extern bool only_las;		/* only do groups in LAS file */
	extern bool allow_unsub;	/* allow unsubscribed groups? */
	bool unsub;			/* subscribed? */
	extern char *newsrcname;	/* newsrc file */


	/* Lock access to the .newsrc? */

	rcfile = mustopen( newsrcname, "r", ".newsrc file" );
	lastgroup = (ngrec *)0;

	while( fgets( buf, sizeof(buf), rcfile ) ) {
		/* skip option lines */
		if( strncmp( buf, "options ", 8 ) == 0 || isspace(buf[0]) ||
					buf[0] == '<' ) {
			(void) skiptonl( buf, rcfile );
			continue;
			}
		unsub = FALSE;
		subchar = strchr( buf, ':' );
		if( subchar == NULL ) {
			if( allow_unsub ) {
				subchar = strchr( buf, '!' );
				unsub = subchar != NULL;
				}
			if( subchar == NULL ) {
				(void) skiptonl( buf, rcfile );
				continue;	/* not a newsgroup line */
				}
			}
		*subchar = 0;		/* terminate group name */
		if( only_las ) {
			group = (ngrec *)get_rec(ng_base, buf);
			if( !group || !(group->gflag & GF_LASGROUP ) ) {
				(void) skiptonl( subchar+1, rcfile );
				continue;
				}
			}
		 else
			group = (ngrec *)add_rec( ng_base, buf, AR_CREATE );
		if( group->ngnumber == 0 )
			group->ngnumber = group_count++;
		group->gflag |= GF_RCGROUP;		/* group from .newsrc*/

		/* build the RC order chain */
		if( lastgroup ) 
			lastgroup->chain = group->ngnumber;
		 else
			rc_chain = group;
		lastgroup = group;
		if( unsub )
			group->gflag |= GF_UNSUB;
		/* now add the line from the .newsrc */

		bslist = whitestrip( subchar+1 );
			
		len = strlen( bslist );
		if( bslist[len-1] != '\n' ) {
			long seekback;
			int extras;
			char c;
			/* the line was too long for the buffer! */
			/* find out how long it really is */
			seekback = ftell( rcfile );

			extras = skiptonl( bslist, rcfile );

			/* go back and read it in for real */
			fseek( rcfile, seekback, 0 );
			slist = perm_alloc( len + extras + 1 + SLIST_EXTRAS );
			strcpy( slist, bslist );
			fgets( slist+len, extras+1, rcfile );
			len += extras;
			if( slist[len-1] == '\n' )
				slist[--len] = 0;
			 else {
				/* something strange is going on  -- EOF? */
				warning( 0, "Very strange line in .newsrc %s\n",
							slist );
				}
			}
		 else {
			slist = perm_alloc( len + 1 + SLIST_EXTRAS );
			strcpy( slist, bslist );
			slist[--len] = 0;
			}
		group->seenlist = slist;
		group->slistlen = len + SLIST_EXTRAS;
		}
	fclose( rcfile );		/* for now */

}


#define set_bit( map, i )	map[i>>3] |= 1 << (i&7)
#define is_bit_set( map, i )	((map[i>>3] >> (i&7)) & 1)
#define clear_bit( map, i )	map[i>>3] &= ~(1 << (i&7))

/*
 * Translate the SEEN list of a newsgroup to a bitmap
 * Bitmap maximum size 32767 bits or 4K bytes with 16 bit int -- no problem.
 */

int
seen_to_bitmap( group, bitmap, bmsize, num_unseen )
ngrec *group;		/* newsgroup, with seen article list */
char *bitmap;
int bmsize;		/* max size of bitmap */
int *num_unseen;	/* return number of unread articles */
{
	int bsize;
	char *p;		/* pointer into list */
	int num_new_seen;

	
	bsize = group->highest - group->lowest + 1;
	if( bsize < 0 || (bsize+7)/8 > bmsize )
		return -1;

	/* clear out the bitmap first */
	zero( bitmap, bmsize );
	num_new_seen = 0;

	p = group->seenlist;

	while( *p ) {
		int32 first,last;

		first = atoi32(p);
		while( isdigit(*p) )
			p++;
		if( *p == '-' ) {
			last = atoi32(++p);
			/* make sure last is within bounds of group */
			last = min( last, group->highest );
			while( isdigit(*p) )
				p++;
			}
		 else
			last = first;
		if( last >= group->lowest && last >= first ) {
			int topbit, i;
			/* move first up to within group */
			first = max( first, group->lowest );
			topbit = last - group->lowest;
			/* set all the bits for read articles */
			for( i = first - group->lowest; i <= topbit; i++ ) {
				set_bit( bitmap, i );
				if( i + group->lowest > group->las )
					num_new_seen++;
				}
			}
		if( *p == ',' )
			p++;
		}
	if( num_unseen )
		*num_unseen = group->highest - group->las - num_new_seen;
	return bsize;
}

/* scan through the newsrc for unread articles and process them */

do_newsrc()
{
	ngrec *group;
	char bitmap[BITMAP_SIZE];
	char newsfile[MAX_FNAME];
	int bsize;			/* number of bits in bitmap */
	int art;			/* article number (less lowest)*/
	int dirty;			/* was this article marked read? */
	extern char *news_spool_dir;
	extern bool list_always;	/* always list articles */
	int num_unread;			/* number of unread articles */
	extern bool newsrc_allread;	/* mark all articles read */
	bool foundone;			/* we liked an article here */
	extern int reading_mode;

	reading_mode = FILE_FULL;

	for( group = rc_chain; group; group = ngarray[group->chain] ) {

		/* test if group was found in active file */
		if( !( group->gflag & GF_ACTIVE ) )
			continue;

		/* adjust last article seen */

		group->las = max( group->lowest-1, group->las );

		if( group->las >= group->highest )
			continue;		/* all messages already done*/
		dirty = FALSE;
		foundone = FALSE;
		
		bsize = seen_to_bitmap( group, bitmap, sizeof(bitmap),
							&num_unread );
		if( bsize < 0 ) {
			warning( 1,"Invalid Active file line for %s\n",
				group->key);
			continue;		/* error on this group */
			}
		
		/* if no unread articles, don't bother, but update las */
		if( num_unread <= 0 ) {
			group->las = group->highest;
			continue;
			}

		main_newsgroup = group->ngnumber;

		Ustartgroup( num_unread );

		/* Here is the actual loop were we call the user code to
		   accept or reject the acticle */


		for( art = group->las - group->lowest+1 ; art < bsize; art++ ) {
			if( !is_bit_set( bitmap, art ) ) {
				/* an unread article! */
				reset_tempalloc();
				sprintf( newsfile, "%s/%s/%lu", news_spool_dir,
						ngdir(group->key),
						(long)(art + group->lowest) );
				article_number=makeint((long)art+group->lowest);
				main_newsgroup = group->ngnumber;
				if( !accept_article( newsfile ) ) {
					extern array *xref;
					set_bit( bitmap, art );
					dirty = TRUE;
					if( xref && xref->arsize > 2 )
						kill_xrefs( main_newsgroup );
					}
				 else {
					/* in list mode, assure we don't
					   list cross postings */
					if( list_always ) {
						extern array *xref;
						printf( "%s\n", newsfile );
						if( xref && xref->arsize > 2 )
							kill_xrefs(
								main_newsgroup);
						}
					/* mark it read anyway in batch mode */
					if( newsrc_allread ) {
						set_bit( bitmap, art );
						dirty = TRUE;
						}
					foundone = TRUE;
					}
				}
			}
		/* bitmap processed */
		group->las = group->highest;
		/* now turn bitmap into seen list again */
		if( dirty ) {
			bitmap_to_seen( bitmap, group, bsize );
			group->gflag |= GF_DIRTY;
			}
		if( foundone )
			/* turn off unsub it if it was on, resubscribe group */
			group->gflag &= ~GF_UNSUB;
		finish_group( );
		}
						

}

#define digits(n)   (n >= 10000 ? ( n >= 100000 ? 6 : 5 ) : n >= 100 ? (n >= 1000 ? 4 : 3) : ( n < 10 ? 1 : 2 ) )

/* convert a bitmap to a seen list, return pointer to that.
 * (A 'seen list' is the string of numbers found in a .newsrc entry.
 */


bitmap_to_seen( bitmap, group, numbits )
char *bitmap;		/* the bitmap itself */
ngrec *group;		/* group record */
int numbits;		/* the number of bits in the bitmap */
{
	int i;
	int size;		/* size of output string */
	bool oldstat;		/* status of old bit */
	int32 oldnum;		/* start of stream of read articles */
	bool bit;		/* bit in bitmap */
	char *p;		/* pointer storing slist */
	char *slist;		/*new seen list */
	int32 lowest;		/* lowest article in group */


	oldnum = 1;
	size = 0;
	/* clear the bit off the end */
	clear_bit( bitmap, numbits );

	lowest = group->lowest;

	/* set stat of 'previous' bit */
	oldstat = lowest > 1;
	for( i = 0; i <= numbits; i++ ) {
		bit = is_bit_set( bitmap, i );
		if( bit != oldstat ) {
			oldstat = bit;
			if( bit ) 
				oldnum = i + lowest;
			 else {
				size += 1 + digits(oldnum);
				if( i + lowest > oldnum+1 )
					size += 1 + digits( i + lowest-1 );
				}
			}
		}
	if( size >= group->slistlen ) {
		perm_free( group->seenlist );
		group->seenlist = perm_alloc( size+SLIST_EXTRAS+1 );
		group->slistlen = size+SLIST_EXTRAS;
		}

	/* now repeat the procedure, storing this time */
	/* set stat of 'previous' bit */

	p = group->seenlist;
	oldnum = 1;

	oldstat = lowest > 1;
	for( i = 0; i <= numbits; i++ ) {
		bit = is_bit_set( bitmap, i );
		if( bit != oldstat ) {
			oldstat = bit;
			if( bit ) 
				oldnum = i + lowest;
			 else {
				sprintf( p, "%lu", (long)oldnum );
				p += digits(oldnum);
				if( i + lowest > oldnum+1 ) {
					sprintf(p,"-%lu",(long)i+lowest-1);
					p += strlen(p);
					}
				*p++ = ',';
				}
			}
		/* sanity check */
		if( p - group->seenlist >= group->slistlen ) {
			fprintf( stderr,"Seen list %d bytes too big, only %d\n",
					p-group->seenlist, group->slistlen);
			fprintf(stderr,"%s: %s\n", group->key, group->seenlist);
			abort();
			}
		}
	if( p[-1] == ',' )
		p[-1] = 0;
	 else
		p[0] = 0;

}

/* structure for subscribe list */

struct sublist {
	char *gname;
	struct sublist *next;
	} *newgroups;

/* subscribe to a group for the next time around - for newsrc modes only */

subscribe( grp )
char *grp;
{
	ngrec *group;
	struct sublist *sptr;

	/* first see if we know about it already */
	group = (ngrec *)get_rec( ng_base, grp );
	if( group && group->gflag & GF_RCGROUP ) {
		/* turn off unsubscribed flag, if on */
		group->gflag &= ~GF_UNSUB;
		group->gflag |= GF_DIRTY;
		}
	 else {
		/* list this group for later concat to .newsrc */

		/* scan to see if already present */
		for( sptr = newgroups; sptr; sptr = sptr->next )
			if( strcmp( sptr->gname, grp ) == 0 )
				return;
		sptr = (struct sublist *)perm_alloc( sizeof(struct sublist) );	
		sptr->gname = allocstring(grp);
		/* stick on front of list */
		sptr->next = newgroups;
		if( newgroups == 0 )
			newgroups = sptr;
		}

}



/* Write out a new .newsrc to the specified descriptor.  Return true
   if a write error, false if all is well.  The descriptor must not
   point to the real, old .newsrc which is used to contruct the new one.
   */

int
write_newsrc( desc )
FILE *desc;			/* descriptor of output .newsrc */
{
	FILE *oldrc;		/* old .newsrc we replace */
	char rbuf[MAX_LLEN];
	char *subchar;
	ngrec *group;
	extern bool allow_unsub;	/* allow unsubscribed groups */
	bool we;			/* write error flag */
	extern char *strpbrk();
	extern char *newsrcname;
	struct sublist *sptr;


	oldrc = mustopen( newsrcname, "r", ".newsrc file for update" );
	we = FALSE;

	while( !we && fgets( rbuf, sizeof(rbuf), oldrc ) ) {
		subchar = strpbrk( rbuf, ":! \t," );
		if( !isspace(rbuf[0]) && subchar && (*subchar == ':' ||
					(allow_unsub && *subchar == '!') ) ) {
			char oldchar;		
			/* option lines have no colon or ! */
			oldchar = *subchar;
			*subchar = 0;
			group = (ngrec *)get_rec( ng_base, rbuf );
			*subchar = oldchar;
			if( group && group->gflag & GF_DIRTY ) {
				we |= fprintf( desc, "%s%c ", group->key,
					group->gflag&GF_UNSUB ? '!':':' ) < 0;
				/* beware of printf's with fixed sized buffers*/
				we |= fputs(group->seenlist, desc) == EOF;
				we |= fputs( "\n", desc ) == EOF;
				/* now skip over rest of any very long line */
				(void) skiptonl( rbuf, oldrc );
				continue;
				}
			}
		/* was not modified, so just copy what's there */
		we |= fputs( rbuf, desc ) == EOF;
		/* loop to copy extra parts of long lines */
		while( rbuf[strlen(rbuf)-1] != '\n' && fgets( rbuf,
					sizeof(rbuf), oldrc ) )
			we |= fputs( rbuf, desc ) == EOF;
		}
	/* write out new groups to subscribe to */
	for( sptr = newgroups; sptr; sptr = sptr->next )
		we |= fprintf( desc, "%s:\n", sptr->gname ) < 0;
	
	fclose( oldrc );
	fclose( desc );
	return we;
}

/* Output a new .nglas file */

write_las()
{
	FILE *lasfile;
	ngrec *group;
	bool we;			/* write error */
	extern char *lasname;		/* name of las file */
	struct sublist *sptr;

	lasfile = fopen( lasname, "w" );

	if( !lasfile ) {
		warning( 1, "Could not open file %s\n", lasname );
		return;
		}

	we = FALSE;

	for( group = rc_chain; group && !we; group = ngarray[group->chain] )
		we |= fprintf( lasfile, "%s %lu\n", group->key,
					(long)group->las ) < 0;
	/* write out new groups to subscribe to */
	for( sptr = newgroups; sptr; sptr = sptr->next )
		we |= fprintf( lasfile, "%s 0\n", sptr->gname ) < 0;
	fclose( lasfile );
	if( we )
		warning( 1, "Unable to write file %s\n", lasname );

}


/* the master loop for .newsrc mode */


process_newsrc()
{
	FILE *nrdesc;				/* newsrc descriptor */
	extern bool out_newsrc;		/* do we write a .newsrc? */
	extern char *newsrcname;
	extern char *temprc;		/* temporary write location */
	extern bool newsrc_allread;	/* newsrc to be marked all read? */

	/* init the newsgroups list and read in the data */
	initngs(TRUE);

	Uinit();

	/* go through the .newsrc and process the articles */
	do_newsrc();

	Uterminate();

	if( !out_newsrc )
		return;

	/* Now write out the .newsrc file */

	nrdesc = mustopen( temprc, "w", "temporary .newsrc for update" );

	if( !nrdesc || write_newsrc( nrdesc ) ) {
		char pidbuf[40];

		warning( 0, "Could not write out .newsrc\n" );
		sprintf( pidbuf, "/tmp/nr%d", getpid() );
		nrdesc = mustopen( pidbuf, "w", "emergency .newsrc" );
		if( !nrdesc || write_newsrc( nrdesc ) ) 
			warning( 0, "Attempt to write to %s failed\n",
							pidbuf );
		 else
			warning( 0, "Copy written to %s\n", pidbuf );
		}
	 else {
		/* release the old .newsrc */
		if( unlink( newsrcname ) )
			warning( 0, "Could not unlink old %s\n", newsrcname );
		 else {
			if( link( temprc, newsrcname ) )
				warning( 0, "Could not link in %s\n", temprc );
			 else
				unlink( temprc );
			}
		write_las();
		}
	
}

/* Mark articles listed as cross references as read in their groups */

kill_xrefs( mgroup )
newsgroup mgroup;				/* main newsgroup */
{
	extern array *xref;			/* array of xrefs */
	extern char *sitename;			/* short name of system */
	int i;
	char bitmap[BITMAP_SIZE];
	int num_unread;				/* a dummy */
	int bsize;				/* size of bitmap */
	int32 artbase;

	if( xref && cleq( xref->vals[0].ustring, sitename ) ) 
		for( i = 1; i < xref->arsize; i++ ) {
			ngrec *group;
			char *colon;
			char *xrpat;

			xrpat = xref->vals[i].ustring;
			colon = strchr( xrpat, ':' );
			if( colon ) {
				*colon++ = 0;
				group = (ngrec *)get_rec( ng_base, xrpat );
				/* if not a subscribed group or the current
				   group, then skip this one */
				if( group == 0 || group->ngnumber == mgroup ||
						!(group->gflag & GF_RCGROUP) )
					continue;
				artbase = atol(colon) - group->lowest;
				if( artbase < 0 )
					continue;
				/* ok, we have an article to mark unread */
				bsize = seen_to_bitmap( group, bitmap,
					sizeof(bitmap), &num_unread );
				/* if no change, keep going */
				if(artbase>bsize||is_bit_set(bitmap,artbase))
					continue;
				/* set the bit */
				set_bit( bitmap, artbase );
				/* put back new bitmap */
				bitmap_to_seen( bitmap, group, bsize );
				group->gflag |= GF_DIRTY;
				}
			}
}
