// -*- verilog -*-
//
//  USRP - Universal Software Radio Peripheral
//
//  Copyright (C) 2003,2004 Matt Ettus
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

// Top level module for a full setup with DUCs and DDCs

// Define DEBUG_OWNS_IO_PINS if we're using the daughterboard i/o pins
// for debugging info.  NB, This can kill the m'board and/or d'board if you
// have anything except basic d'boards installed.

//`define DEBUG_OWNS_IO_PINS


module usrp_std
(
 output MYSTERY_SIGNAL,
 input master_clk,
 input SCLK,
 input SDI,
 inout SDO,
 input SEN_FPGA,

 input FX2_1,
 output FX2_2,
 output FX2_3,
 
 input wire [11:0] rx_a_a,
 input wire [11:0] rx_b_a,
 input wire [11:0] rx_a_b,
 input wire [11:0] rx_b_b,

 output wire [13:0] tx_a,
 output wire [13:0] tx_b,

 output wire TXSYNC_A,
 output wire TXSYNC_B,
 
  // USB interface
 input usbclk,
 input wire [2:0] usbctl,
 output wire [1:0] usbrdy,
 inout [15:0] usbdata,  // NB Careful, inout

 // These are the general purpose i/o's that go to the daughterboard slots
 inout wire [15:0] io_tx_a,
 inout wire [15:0] io_tx_b,
 inout wire [15:0] io_rx_a,
 inout wire [15:0] io_rx_b
 );	

   assign MYSTERY_SIGNAL = 1'b0;
   
   wire clk64,clk128;
   
   wire WR = usbctl[0];
   wire RD = usbctl[1];
   wire OE = usbctl[2];

   wire have_space, have_pkt_rdy;
   assign usbrdy[0] = have_space;
   assign usbrdy[1] = have_pkt_rdy;

   wire   tx_underrun, rx_overrun;    
   wire   clear_status = FX2_1;
   assign FX2_2 = rx_overrun;
   assign FX2_3 = tx_underrun;
      
   wire [15:0] usbdata_out;
   
   wire [31:0] ch0rx_freq,ch1rx_freq,ch2rx_freq,ch3rx_freq;
   wire [3:0]  ddc0mux,ddc1mux,ddc2mux,ddc3mux;
   wire [3:0]  dac0mux,dac1mux,dac2mux,dac3mux;
   
   wire        rx_realsignals;
   wire        tx_realsignals;
   wire [2:0]  rx_numchan,tx_numchan;
      
   wire [7:0]  interp_rate, decim_rate;
   
   wire [15:0] tx_debugbus, rx_debugbus;
   wire [31:0] misc_settings;
   
   wire        enable_tx, enable_rx;
   wire        tx_dsp_reset, rx_dsp_reset, tx_bus_reset, rx_bus_reset;
   wire [7:0]  settings;
   
   // Tri-state bus macro
   bustri bustri( .data(usbdata_out),.enabledt(OE),.tridata(usbdata) );

   assign      clk64 = master_clk;

   wire [15:0] ch0tx,ch1tx,ch2tx,ch3tx; //,ch4tx,ch5tx,ch6tx,ch7tx;
   wire [15:0] ch0rx,ch1rx,ch2rx,ch3rx,ch4rx,ch5rx,ch6rx,ch7rx;
   
   // TX
   wire [15:0] i_out_0,i_out_1,q_out_0,q_out_1;
   wire [15:0] bb_tx_i0,bb_tx_q0,bb_tx_i1,bb_tx_q1;  // bb_tx_i2,bb_tx_q2,bb_tx_i3,bb_tx_q3;
   
   wire        strobe_interp, tx_sample_strobe;
   wire [7:0]  tx_sample_rate;

   // FIXME implement cool muxing...
   assign      bb_tx_i0 = ch0tx;
   assign      bb_tx_q0 = ch1tx;
   assign      bb_tx_i1 = ch2tx;
   assign      bb_tx_q1 = ch3tx;
   
   strobe_gen da_strobe_gen
     ( .clock(clk64),.reset(tx_dsp_reset),.enable(enable_tx),
       .rate(8'd1),.strobe_in(1'b1),.strobe(tx_sample_strobe) );

   strobe_gen tx_strobe_gen
     ( .clock(clk64),.reset(tx_dsp_reset),.enable(enable_tx),
       .rate(interp_rate),.strobe_in(tx_sample_strobe),.strobe(strobe_interp) );
   
   tx_buffer tx_buffer
     ( .usbclk(usbclk),.bus_reset(tx_bus_reset),.reset(tx_dsp_reset),
       .usbdata(usbdata),.WR(WR),.have_space(have_space),.tx_underrun(tx_underrun),
       .channels({tx_numchan,1'b0}),
       .tx_i_0(ch0tx),.tx_q_0(ch1tx),
       .tx_i_1(ch2tx),.tx_q_1(ch3tx),
       .tx_i_2(),.tx_q_2(),
       .tx_i_3(),.tx_q_3(),
       .txclk(clk64),.txstrobe(strobe_interp),
       .clear_status(clear_status),		    
       .debugbus(tx_debugbus) );

   tx_chain tx_chain_0
     ( .clock(clk64),.reset(tx_dsp_reset),.enable(enable_tx),
       .interp_rate(interp_rate),.sample_strobe(tx_sample_strobe),
       .interpolator_strobe(strobe_interp),.freq(),
       .i_in(bb_tx_i0),.q_in(bb_tx_q0),.i_out(i_out_0),.q_out(q_out_0) );

   tx_chain tx_chain_1
     ( .clock(clk64),.reset(tx_dsp_reset),.enable(enable_tx),
       .interp_rate(interp_rate),.sample_strobe(tx_sample_strobe),
       .interpolator_strobe(strobe_interp),.freq(),
       .i_in(bb_tx_i1),.q_in(bb_tx_q1),.i_out(i_out_1),.q_out(q_out_1) );

   wire [15:0] tx_a_a = dac0mux[3] ? (dac0mux[1] ? (dac0mux[0] ? q_out_1 : i_out_1) : (dac0mux[0] ? q_out_0 : i_out_0)) : 16'b0;
   wire [15:0] tx_b_a = dac1mux[3] ? (dac1mux[1] ? (dac1mux[0] ? q_out_1 : i_out_1) : (dac1mux[0] ? q_out_0 : i_out_0)) : 16'b0;
   wire [15:0] tx_a_b = dac2mux[3] ? (dac2mux[1] ? (dac2mux[0] ? q_out_1 : i_out_1) : (dac2mux[0] ? q_out_0 : i_out_0)) : 16'b0;
   wire [15:0] tx_b_b = dac3mux[3] ? (dac3mux[1] ? (dac3mux[0] ? q_out_1 : i_out_1) : (dac3mux[0] ? q_out_0 : i_out_0)) : 16'b0;

   wire txsync = tx_sample_strobe;
   assign TXSYNC_A = txsync;
   assign TXSYNC_B = txsync;

   assign tx_a = txsync ? tx_b_a[15:2] : tx_a_a[15:2];
   assign tx_b = txsync ? tx_b_b[15:2] : tx_a_b[15:2];
   
   // RX
   wire        rx_sample_strobe,strobe_decim;
   wire [7:0]  rx_sample_rate;

   // FIXME
   assign      rx_sample_strobe = 1'b1;
   //strobe_gen rx_strobe_gen
   //  ( .clock(clk64),.reset(rx_dsp_reset),.enable(enable_rx),
   //    .rate(rx_sample_rate),.strobe_in(1'b1),.strobe(rx_sample_strobe) );
   
   strobe_gen decim_strobe_gen
     ( .clock(clk64),.reset(rx_dsp_reset),.enable(enable_rx),
       .rate(decim_rate),.strobe_in(rx_sample_strobe),.strobe(strobe_decim) );
   
   wire [15:0] bb_rx_i0,bb_rx_q0,bb_rx_i1,bb_rx_q1,
	       bb_rx_i2,bb_rx_q2,bb_rx_i3,bb_rx_q3;
   
   wire [15:0] adc_offset_0, adc_offset_1, adc_offset_2, adc_offset_3;
   
   // Buffer at input to chip, then subtract dc offset
   reg [11:0] adc0,adc1,adc2,adc3;
   reg [15:0]  ddc0_in_i,ddc1_in_i,ddc2_in_i,ddc3_in_i;
   reg [15:0]  ddc0_in_q,ddc1_in_q,ddc2_in_q,ddc3_in_q;

   wire [15:0] adc0_corr = {{2{adc0[11]}},adc0,2'b0} - adc_offset_0;
   wire [15:0] adc1_corr = {{2{adc1[11]}},adc1,2'b0} - adc_offset_1;
   wire [15:0] adc2_corr = {{2{adc2[11]}},adc2,2'b0} - adc_offset_2;
   wire [15:0] adc3_corr = {{2{adc3[11]}},adc3,2'b0} - adc_offset_3;
   
   always @(negedge clk64)
     begin
	adc0 <= #1 rx_a_a;
	adc1 <= #1 rx_b_a;
	adc2 <= #1 rx_a_b;
	adc3 <= #1 rx_b_b;
end

always @(posedge clk64)
	begin
	ddc0_in_i <= #1 ddc0mux[1] ? (ddc0mux[0] ? adc3_corr : adc2_corr) : (ddc0mux[0] ? adc1_corr : adc0_corr);
	ddc0_in_q <= #1 rx_realsignals ? 16'd0 : ddc0mux[3] ? (ddc0mux[2] ? adc3_corr : adc2_corr) : (ddc0mux[2] ? adc1_corr : adc0_corr);
	ddc1_in_i <= #1 ddc1mux[1] ? (ddc1mux[0] ? adc3_corr : adc2_corr) : (ddc1mux[0] ? adc1_corr : adc0_corr);
	ddc1_in_q <= #1 rx_realsignals ? 16'd0 : ddc1mux[3] ? (ddc1mux[2] ? adc3_corr : adc2_corr) : (ddc1mux[2] ? adc1_corr : adc0_corr);
	ddc2_in_i <= #1 ddc2mux[1] ? (ddc2mux[0] ? adc3_corr : adc2_corr) : (ddc2mux[0] ? adc1_corr : adc0_corr);
	ddc2_in_q <= #1 rx_realsignals ? 16'd0 : ddc2mux[3] ? (ddc2mux[2] ? adc3_corr : adc2_corr) : (ddc2mux[2] ? adc1_corr : adc0_corr);
	ddc3_in_i <= #1 ddc3mux[1] ? (ddc3mux[0] ? adc3_corr : adc2_corr) : (ddc3mux[0] ? adc1_corr : adc0_corr);
	ddc3_in_q <= #1 rx_realsignals ? 16'd0 : ddc3mux[3] ? (ddc3mux[2] ? adc3_corr : adc2_corr) : (ddc3mux[2] ? adc1_corr : adc0_corr);
     end

   wire loopback = settings[0];
   wire counter = settings[1];

   reg [15:0] debug_counter;
   
   reg [15:0] loopback_i_0,loopback_q_0;

   always @(posedge clk64)
     if(rx_dsp_reset)
       debug_counter <= #1 16'd0;
     else if(~enable_rx)
       debug_counter <= #1 16'd0;
     else if(strobe_decim)
       debug_counter <=#1 debug_counter + 16'd2;
   
   always @(posedge clk64)
     if(strobe_interp)
       begin
	  loopback_i_0 <= #1 ch0tx;
	  loopback_q_0 <= #1 ch1tx;
       end
   
   assign ch0rx = counter ? debug_counter : loopback ? loopback_i_0 : bb_rx_i0;
   assign ch1rx = counter ? debug_counter + 16'd1 : loopback ? loopback_q_0 : bb_rx_q0;
   assign ch2rx = bb_rx_i1;
   assign ch3rx = bb_rx_q1;
   assign ch4rx = bb_rx_i2;
   assign ch5rx = bb_rx_q2;
   assign ch6rx = bb_rx_i3;
   assign ch7rx = bb_rx_q3;

   clk_doubler clk_doubler (.inclk0(clk64),.c0(clk128));

   rx_buffer rx_buffer
     ( .usbclk(usbclk),.bus_reset(rx_bus_reset),.reset(rx_dsp_reset),
       .usbdata(usbdata_out),.RD(RD),.have_pkt_rdy(have_pkt_rdy),.rx_overrun(rx_overrun),
       .channels({rx_numchan,1'b0}),
       .ch_0(ch0rx),.ch_1(ch1rx),
       .ch_2(ch2rx),.ch_3(ch3rx),
       .ch_4(ch4rx),.ch_5(ch5rx),
       .ch_6(ch6rx),.ch_7(ch7rx),
       .rxclk(clk64),.rxstrobe(strobe_decim),
       .clear_status(clear_status),
       .debugbus(rx_debugbus) );
   
/*   rx_chain_dual rx_chain_0
     ( .clock(clk64),.clock_2x(clk128),.reset(1'b0),.enable(enable_rx),
       .decim_rate(decim_rate),.sample_strobe(rx_sample_strobe),
       .decimator_strobe(strobe_decim),
       .freq0(ch0rx_freq),.i_in0(ddc0_in_i),.q_in0(ddc0_in_q),.i_out0(bb_rx_i0),.q_out0(bb_rx_q0),
       .freq1(ch1rx_freq),.i_in1(ddc1_in_i),.q_in1(ddc1_in_q),.i_out1(bb_rx_i1),.q_out1(bb_rx_q1)
       );
   
   rx_chain_dual rx_chain_1
     ( .clock(clk64),.clock_2x(clk128),.reset(1'b0),.enable(enable_rx),
       .decim_rate(decim_rate),.sample_strobe(rx_sample_strobe),
       .decimator_strobe(strobe_decim),
       .freq0(ch2rx_freq),.i_in0(ddc2_in_i),.q_in0(ddc2_in_q),.i_out0(bb_rx_i2),.q_out0(bb_rx_q2),
       .freq1(ch3rx_freq),.i_in1(ddc3_in_i),.q_in1(ddc3_in_q),.i_out1(bb_rx_i3),.q_out1(bb_rx_q3));
*/

   rx_chain rx_chain_0
     ( .clock(clk64),.reset(1'b0),.enable(enable_rx),
       .decim_rate(decim_rate),.sample_strobe(rx_sample_strobe),
       .decimator_strobe(strobe_decim),.freq(ch0rx_freq),
       .i_in(ddc0_in_i),.q_in(ddc0_in_q),.i_out(bb_rx_i0),.q_out(bb_rx_q0));
   
   rx_chain rx_chain_1
     ( .clock(clk64),.reset(1'b0),.enable(enable_rx),
       .decim_rate(decim_rate),.sample_strobe(rx_sample_strobe),
       .decimator_strobe(strobe_decim),.freq(ch1rx_freq),
       .i_in(ddc1_in_i),.q_in(ddc1_in_q),.i_out(bb_rx_i1),.q_out(bb_rx_q1));
   /*
   rx_chain rx_chain_2
     ( .clock(clk64),.reset(1'b0),.enable(enable_rx),
       .decim_rate(decim_rate),.sample_strobe(rx_sample_strobe),
       .decimator_strobe(strobe_decim),.freq(ch2rx_freq),
       .i_in(ddc2_in_i),.q_in(ddc2_in_q),.i_out(bb_rx_i2),.q_out(bb_rx_q2));

   rx_chain rx_chain_3
     ( .clock(clk64),.reset(1'b0),.enable(enable_rx),
       .decim_rate(decim_rate),.sample_strobe(rx_sample_strobe),
       .decimator_strobe(strobe_decim),.freq(ch3rx_freq),
       .i_in(ddc3_in_i),.q_in(ddc3_in_q),.i_out(bb_rx_i3),.q_out(bb_rx_q3));
*/
   // Control, clock and reset
   wire [7:0]  master_controls;
   wire [15:0] io_0, io_1, io_2, io_3;
   
   serial_io serial_io
     ( .serial_clock(SCLK),.serial_data_in(SDI),
       .enable(SEN_FPGA),.reset(1'b0),.serial_data_out(SDO),
       .ch0rx_freq(ch0rx_freq),.ch1rx_freq(ch1rx_freq),.ch2rx_freq(ch2rx_freq),.ch3rx_freq(ch3rx_freq),
       .interp_rate(interp_rate),.decim_rate(decim_rate),
       .tx_sample_rate(tx_sample_rate),.rx_sample_rate(rx_sample_rate),
       .adc_offset_0(adc_offset_0),.adc_offset_1(adc_offset_1),
       .adc_offset_2(adc_offset_2),.adc_offset_3(adc_offset_3),
       .master_controls(master_controls),
       .io_0(io_0),.io_1(io_1),.io_2(io_2),.io_3(io_3),
       .settings(settings),
       .rxmux({ddc3mux,ddc2mux,ddc1mux,ddc0mux,rx_realsignals,rx_numchan}),
       .txmux({dac3mux,dac2mux,dac1mux,dac0mux,tx_realsignals,tx_numchan}));

   master_control master_control
     ( .master_clk(clk64),.usbclk(usbclk),
       .master_controls(master_controls),
       .tx_bus_reset(tx_bus_reset),.rx_bus_reset(rx_bus_reset),
       .tx_dsp_reset(tx_dsp_reset),.rx_dsp_reset(rx_dsp_reset),
       .enable_tx(enable_tx),.enable_rx(enable_rx) );
   
   // DEBUG stuff

`ifdef DEBUG_OWNS_IO_PINS
   wire        dummy_0, dummy_1, dummy_2, dummy_3;

   bidir_reg db_bidir_0
     ( .tristate (io_tx_a), .oe(16'hffff),
       .reg_val(bb_rx_i0));
   
   bidir_reg db_bidir_1
     ( .tristate (io_rx_a), .oe(16'hffff),
       .reg_val(ddc0_in_i));
   
   bidir_reg db_bidir_2
     ( .tristate (io_tx_b), .oe(16'hffff),
       .reg_val({clk64,strobe_decim,adc0}));
   
   bidir_reg db_bidir_3
     ( .tristate (io_rx_b), .oe(16'hffff),
       .reg_val(bb_rx_q0));

`else // !`ifdef DEBUG_OWNS_IO_PINS
   assign      io_tx_a = io_0[15:0];
   assign      io_rx_a = io_1[15:0];
   assign      io_tx_b = io_2[15:0];
   assign      io_rx_b = io_3[15:0];
`endif // !`ifdef DEBUG_OWNS_IO_PINS
   
   
endmodule // usrp_std
