/* CSL - Common Sound Layer
 * Copyright (C) 2001 Jeff Tranter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * This example illustrates some of the "advanced" PCM API functions.
 * It currently doesn't do much more than just calling some of the
 * APIs. The program currently displays some errors due to
 * unimplemented APIs, depending on which back end is being used.
*/

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <csl/csl.h>

int main (int argc, char **argv)
{
	CslErrorType error;
	CslDriver *driver;
	CslPcmStream *stream;
	CslOptions options;
	CslPcmStatus status;
	char *title;
	int buffer_size, watermark;
	int packets, packet_size, packet_watermark;

	csl_options_parse (&options, &argc, &argv);
	csl_set_debug_mask (options.debug_flags);

	fprintf(stderr, "Data format: %d channels, %d samples/sec, %s\n",
			options.n_channels, options.rate, csl_describe_pcm_format(options.pcm_format));

	/* init driver */
	error = csl_driver_init (NULL, &driver);
	if (error)
		csl_error ("unable to initialize driver: %s", csl_strerror (error));
	
	/* open PCM input stream */
	error = csl_pcm_open_input (driver, "cslpcm2", options.rate, options.n_channels, options.pcm_format, &stream);
	if (error)
		csl_error ("failed to open input device: %s", csl_strerror (error));

	error = csl_pcm_get_status(stream, &status);
/* aRts driver does not implement this yet, so avoid the error check.
	if (error)
		csl_error ("failed to get pcm status: %s", csl_strerror (error));
*/

	printf("\nStream status:\n"
		   "  rate: %d\n"
		   "  channels: %d\n"
		   "  size: %d\n"
		   "  buffer size: %d\n"
		   "  bytes available: %d\n"
		   "  packet size: %d\n"
		   "  number of buffer packets: %d\n"
		   "  number of available packets: %d\n",
		   status.rate,
           status.n_channels,
		   status.n_bits,
		   status.buffer_size,
		   status.n_bytes_available,
		   status.packet_size,
		   status.n_buffer_packets,
		   status.n_packets_available);

	error = csl_pcm_flush(stream);
/* aRts driver does not implement this yet, so avoid the error check.
	if (error)
		csl_error ("failed to flush: %s", csl_strerror (error));
*/

	error = csl_pcm_sync(stream);
	if (error)
		csl_error ("failed to sync: %s", csl_strerror (error));

#if 0
	error = csl_pcm_suspend(stream);
	if (error)
		csl_error ("failed to suspend: %s", csl_strerror (error));
	error = csl_pcm_activate(stream);
	if (error)
		csl_error ("failed to activate: %s", csl_strerror (error));
#endif

	error = csl_pcm_set_title(stream, "Example Stream Title");
	if (error)
		csl_error ("failed to set title: %s", csl_strerror (error));

	title = csl_pcm_dup_title(stream);
	printf("\nStream title is `%s'\n", title);
	csl_free(title);

	error = csl_pcm_set_stream_mode (stream, 1024, 512);

	csl_pcm_get_stream_settings	(stream, &buffer_size, &watermark);
	printf("\nBuffer size: %d\nWatermark: %d\n", buffer_size, watermark);

	/* These will cause errors as packet mode is not implemented yet. */
	error = csl_pcm_set_packet_mode (stream, 10, 100, 5);

	csl_pcm_get_packet_settings	(stream, &packets, & packet_size, &packet_watermark);
	printf("\nNumber of packets: %d\nPacket size: %d\nPacket watermark: %d\n",
		   packets, packet_size, packet_watermark);

	/* clean up */
	csl_pcm_close (stream);
	csl_driver_shutdown (driver);
	
	return 0;
}
