/*
 * tif2ps/tifdump -- convert TIFF to PostScript
 *
 * written by:
 * Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *			     (andy@db0tui62.BITNET)
 *
 * Copyright (C) 1988 by the author.
 * Permission is granted to copy and distribute this program
 * without charge, provided this copyright notice is included
 * in the copy.
 * This Software is distributed on an as-is basis. There will be
 * ABSOLUTELY NO WARRANTY for any part of this software to work
 * correct. In no case will the author be liable to you for damages
 * caused by the usage of this software.
 */

/*
 * scantif.c -- read tif-file
 *
 * $Header: scantif.c[1.0] Thu Dec 29 20:10:59 1988 andy@coma published $
 */

#include <stdio.h>
#include "defs.h"
#include "tif.h"

STATIC IMAGE nullimage = {0, 0, 0, -1L, 0, 0L, 1, 1, 0, 1, 0L, 0L, 1, 1, 0, 0};
STATIC PHMETRIC nullphmetric = { 0, 1, 0, 0, 0 };
STATIC PHYS nullphys = { 0., 0., 2, 1 };
STATIC CONTEXT nullcontext = { '\0', '\0', 0., 0., 0., 0., 1, 1 };

#define FIELDSBASE 255
CHAR *fields[] = {
  "SubfileType        ",	/* 255  (ff) */
  "ImageWidth         ",	/* 256 (100) */
  "ImageLength        ",	/* 257 (101) */
  "BitsPerSample      ",	/* 258 (102) */
  "Compression        ",	/* 259 (103) */
  "                   ",	/* 260 (104) unused */
  "                   ",	/* 261 (105) unused */
  "PhotometricInterpr.",	/* 262 (106) */
  "Threshholding      ",	/* 263 (107) */
  "CellWidth          ",	/* 264 (108) */
  "CellLength         ",	/* 265 (109) */
  "FillOrder          ",	/* 266 (10a) */
  "                   ",	/* 267 (10b) unused */
  "                   ",	/* 268 (10c) unused */
  "DocumentName       ",	/* 269 (10d) */
  "ImageDescription   ",	/* 270 (10e) */
  "Make               ",	/* 271 (10f) */
  "Model              ",	/* 272 (110) */
  "StripOffsets       ",	/* 273 (111) */
  "Orientation        ",	/* 274 (112) */
  "                   ",	/* 275 (113) unused */
  "                   ",	/* 276 (114) unused */
  "SamplesPerPixel    ",        /* 277 (115) */
  "RowsPerStrip       ",	/* 278 (116) */
  "StripByteCounts    ",	/* 279 (117) */
  "MinSampleValue     ",	/* 280 (118) */
  "MaxSampleValue     ",	/* 281 (119) */
  "XResolution        ",	/* 282 (11a) */
  "YResolution        ",	/* 283 (11b) */
  "PlanarConfiguration",        /* 284 (11c) */
  "PageName           ",	/* 285 (11d) */
  "XPosition          ",	/* 286 (11e) */
  "YPosition          ",	/* 287 (11f) */
  "FreeOffsets        ",	/* 288 (120) */
  "FreeByteCounts     ",	/* 289 (121) */
  "GrayResponseUnit   ",	/* 290 (122) */
  "GrayResponseCurve  ",	/* 291 (123) */
  "Group3Options      ",	/* 292 (124) */
  "Group4Options      ",	/* 293 (125) */
  "                   ",	/* 294 (126) unused */
  "                   ",	/* 295 (127) unused */
  "ResolutionUnit     ",	/* 296 (128) */
  "PageNumber         ",	/* 297 (129) */
  "                   ",	/* 298 (12a) unused */
  "                   ",	/* 299 (12b) unused */
  "ColorResponseUnit  ",	/* 300 (12c) */
  "ColorResponseCurves",	/* 301 (12d) */
};

CHAR *malloc();
VOID logerr();

EXPORT INT scantif (filename, pics, tiff)
     CHAR    *filename;
     PICTURE *pics;  /* array of picture buffers -- out */
     TIFF    *tiff;
{
  DATAFILE infile;
  INT      ifdcount=0;
  VOID     EnterPicValue(), GetImageData();
  USHORT   entrycount, wordtemp, nrofentries;
  ULONG    location, dblwordtemp;
  ENTRY    *entry, stripOffsets, stripByteCounts;

#ifdef MSDOS
  if ((infile.fdes = fopen (filename, "rb")) == NULL)
#else
  if ((infile.fdes = fopen (filename, "r")) == NULL)
#endif
    { logerr ("scantif", "cannot open input file"); return ERROR; }

  /* read the first word, and determine the byte order
   */
  infile.order = INTELTIFF;
  if (GtData (&infile, 0L, 1, TIFFSHORT, (char *)&wordtemp))
    { logerr ("scantif", "can't read first word"); goto error; }
  infile.order = wordtemp;

  /* read the 8-byte header
   */
  if (GtTiffHdr (&infile, &(tiff->header)))
    { logerr ("scantif", "can't read header"); goto error; }

  location = tiff->header.offset;
 		
  /* loop through the IFD's
   */
  do {
    /* if ifd location is 0, quit
     */
    if (location == 0L)
      {
	(VOID) fclose (infile.fdes);
	return OK;
      }
	
    /* read the number of entries
     */
    if (GtData (&infile, location, 1, TIFFSHORT, (char *)&nrofentries))
      { logerr ("scantif", "can't read # of entries"); goto error; }
    tiff->ifdlist[ifdcount].entrycount = nrofentries;

    if ((tiff->ifdlist[ifdcount].entrylist = 
	 (ENTRY *)malloc (nrofentries * sizeof (ENTRY))) == (ENTRY *)0)
      { logerr ("scantif", "not enough memory"); goto error; }

    location += 2;

    pics[ifdcount].image = nullimage;
    pics[ifdcount].photoMetric = nullphmetric;
    pics[ifdcount].physWorld = nullphys;
    pics[ifdcount].context = nullcontext;
    
    /* loop through the entries
     */
    for (entrycount = 0; entrycount < nrofentries; entrycount++) {
      /* read the entry, and dump it
       */
      entry = &(tiff->ifdlist[ifdcount].entrylist[entrycount]);
      if (GtTiffEntry (&infile, location, entry))
	goto error;
      /* adjust the current location
       */
      location += (sizeof (ENTRY) - sizeof (CHAR *));

      if (entry->tag == STRIPOFFSETS)
	stripOffsets = *entry;
      else
	{
	  if (entry->tag == STRIPBYTECOUNTS)
	    stripByteCounts = *entry;
	  else
	    EnterPicValue (entry, &(pics[ifdcount]));
	}
    } /* end of entry loop */

    /* read the data */
    GetImageData (&infile, &(pics[ifdcount].image), &stripOffsets, &stripByteCounts);

    /* read the location of the next ifd */
    if (GtData(&infile, location, 1, TIFFLONG, (char *)&dblwordtemp))
      { logerr ("scantif", "can't read location of the next ifd"); goto error;}
    location = dblwordtemp;
    ifdcount++;

  } while (1); /* end of ifd loop */

 error: ;
  (VOID) fclose (infile.fdes);
  return ERROR;
}

EXPORT VOID dodump (filename, tiff)
     CHAR *filename;
     TIFF *tiff;
{
  INT    ifdcount=0, i;
  USHORT maxitems, item;
  IFD    *ifdptr;
  CHAR   *bufptr;

  printf ("File: %s\n", filename);

  /* print header */
  switch (tiff->header.byteorder)
    {
    case INTELTIFF:
      printf ("    Byte Order: INTELTIFF\n");
      break;
    case MOTOROLATIFF:
      printf ("    Byte Order: MOTOROLATIFF\n");
      break;
    default:
      printf ("    Byte Order: ***illegal***\n");
    }
  printf ("    Version: %d\n", tiff->header.version);
  printf ("    Ifd Offset: %d\n", tiff->header.offset);

  /* print IFDs */
  do {
    ifdptr = &(tiff->ifdlist[ifdcount]);
    printf ("    IFD Number %d\n", ifdcount);
    printf ("\tNumber of entries: %d\n", ifdptr->entrycount);
    for (i=0; i<ifdptr->entrycount; i++)
      {
	printf ("\t%s (%d) type=%d length=%3d val=",
		(ifdptr->entrylist[i].tag >= 32768) ? "-------------------" :
		fields[ifdptr->entrylist[i].tag-FIELDSBASE],
		ifdptr->entrylist[i].tag, ifdptr->entrylist[i].type,
		ifdptr->entrylist[i].length);

	bufptr = ifdptr->entrylist[i].value;
	maxitems = ifdptr->entrylist[i].length;
	switch (ifdptr->entrylist[i].type)
	  {
	  case TIFFBYTE:
	    for (item = 0; item < maxitems; item++)
	      printf ("%x", (unsigned)(*bufptr++));
	    printf ("\n");
	    break;
	  case TIFFASCII:
	    if (maxitems == 0)
	      break;
	    printf ("%.*s\n", maxitems, bufptr);
	    break;
	  case TIFFSHORT:
	    for (item = 0; item < maxitems; item++, bufptr += 2)
	      printf ("%u ", *((USHORT *)bufptr));
	    printf ("\n");
	    break;
	  case TIFFLONG:
	    for (item = 0; item < maxitems; item++, bufptr += 4)
	      printf ("%lu ", *((ULONG *)bufptr));
	    printf ("\n");
	    break;
	  case TIFFRATIONAL:
	    for (item = 0; item < maxitems; item++) {
	      printf ("% lu ", *((ULONG *)bufptr));
	      bufptr += 4;
	      printf ("%lu ", *((ULONG *)bufptr));
	      bufptr += 4;
	    }
	    printf ("\n");
	    break;
	  default:
	    logerr ("dodump", "internal error");
	    break;
	  }
      }
  }
  while (tiff->ifdlist[ifdcount++].nextifd);
  printf ("Ok -- that's all ! Bye Bye...\n");
}

/**********************************************************************/

/*
/* The following six routiones (swap, swaw, GtTiffSizeof, GtData,
/* GtTiffHdr, GtTiffEntry) are not originally written by me.
/* I copied them in a slightli modified form from a "tiffdump"
/* program, I received from Microsoft. There was no copyright
/* notice so I think the code is in the public domain.
/*       Andreas Lampen
/**/

/* swap bytes -- overlapping arrays are handled properly
 */

LOCAL VOID swab (lpSrc, lpDst, nbytes)
     register CHAR *lpSrc, *lpDst;	/* assumed to be word-aligned */
     USHORT        nbytes;              /* assumed to be even */
{
  register USHORT words;
  union {
    CHAR c[2];
    USHORT w;
  } wrd;
  
  words = nbytes/2;
  
  if (lpDst <= lpSrc || lpDst >= lpSrc + nbytes) {
    for (; words--; lpSrc += 2) {
      wrd.w = *(USHORT *)lpSrc;
      *lpDst++ = *(CHAR *)(wrd.c + 1);	/* W2 doesn't like wrd.c[1] */
      *lpDst++ = *(CHAR *)(wrd.c);
    }
  }
  else {		/* we'll have to go backward */
    lpSrc += nbytes - sizeof(USHORT);
    lpDst += nbytes - 1;
    for (; words--; lpSrc -= 2) {
      wrd.w = *(USHORT *)lpSrc;
      *lpDst-- = *(CHAR *)(wrd.c);
      *lpDst-- = *(CHAR *)(wrd.c + 1);
    }
  }
}

/* swap words -- overlapping ranges are handled properly
 */
LOCAL VOID swaw (lpSrc, lpDst, nbytes)
     register CHAR *lpSrc, *lpDst;	/* assumed to be word-aligned */
     USHORT  	   nbytes;	       	/* assumed to be multiple of 4 */
{
  register USHORT dwords;
  union {
    CHAR c[4];
    ULONG dw;
  } dwrd;
  
  dwords = nbytes/4;
  
  if (lpDst <= lpSrc || lpDst >= lpSrc + nbytes) {
    for (; dwords--; lpSrc += 4) {
      dwrd.dw = *(ULONG *)lpSrc;
      *lpDst++ = *(CHAR *)(dwrd.c + 3);
      *lpDst++ = *(CHAR *)(dwrd.c + 2);
      *lpDst++ = *(CHAR *)(dwrd.c + 1);
      *lpDst++ = *(CHAR *)(dwrd.c);
    }
  }
  else {		/* we'll have to go backward */
    lpSrc += nbytes - sizeof(ULONG);
    lpDst += nbytes - 1;
    for (; dwords--; lpSrc -= 4) {
      dwrd.dw = *(ULONG *)lpSrc;
      *lpDst-- = *(CHAR *)(dwrd.c);
      *lpDst-- = *(CHAR *)(dwrd.c + 1);
      *lpDst-- = *(CHAR *)(dwrd.c + 2);
      *lpDst-- = *(CHAR *)(dwrd.c + 3);
    }
  }
}

LOCAL INT GtTiffSizeof (n, p)
     USHORT n;	/* TIFFBYTE or ... */
     USHORT *p;	/* output */
{
  SHORT err = OK;
  
  switch (n) {
  case TIFFBYTE:
  case TIFFASCII:
    *p = 1;
    break;
  case TIFFSHORT:
    *p = 2;
    break;
  case TIFFLONG:
    *p = 4;
    break;
  case TIFFRATIONAL:
    *p = 8;
    break;
  default:
    *p = 1;
    err = ERROR;
    break;
  }
  return err;
}

/* get data -- handles file/table and byte-order problems
 * 64K max
 */
LOCAL INT GtData (pInfile, pos, n, dtype, lpData)
     DATAFILE *pInfile; /* data location - open file or locked-down table */
     ULONG  pos;       /* file/table position, with respect to its beginning */
     USHORT n;	       /* number of data elements to read */
     USHORT dtype;     /* data type: TIFFSHORT, etc */
     CHAR   *lpData;   /* where to put the data */
{
  INT    err;
  USHORT tsize, BytesToRead;

  /* read the data
   */
  if (err = GtTiffSizeof (dtype, &tsize))
    return err;

  BytesToRead = tsize * n;
  if (err = fseek (pInfile->fdes, (long) pos, 0))
    { logerr ("GtData", "fseek error"); return ERROR; }

  if ((fread (lpData, 1, (INT)BytesToRead, pInfile->fdes)) == 0)
    { logerr ("GtData", "fread error"); return ERROR; }

  /* change the byte order, if necessary
   */
#ifdef MOTOROLA
  if (pInfile->order == INTELTIFF) {
#else
  if (pInfile->order == MOTOROLATIFF) {
#endif
    if (dtype == TIFFSHORT)
      swab (lpData, lpData, BytesToRead);
    else if (dtype == TIFFLONG)
      swaw (lpData, lpData, BytesToRead);
    else if (dtype == TIFFRATIONAL)
      swaw (lpData, lpData, BytesToRead);
  }

  return OK;
}
  
/* get TIFF 8-byte header
 * currently only probably portable.  depends somewhat on compiler's 
 * structure organization.
 */
LOCAL INT GtTiffHdr (pInfile, pHdr)
     DATAFILE *pInfile;
     HEADER *pHdr;
{
  SHORT err;
  
  /* get the first 2 words
   */
  if (err = GtData (pInfile, (ULONG) 0, 2, TIFFSHORT, (CHAR *)&pHdr->byteorder))
    { logerr ("GtTiffHdr", "A"); return err; }
  
  /* get the double word (IFD offset)
   */
  if (err = GtData (pInfile, (ULONG)4, 1, TIFFLONG, (CHAR *)&pHdr->offset))
    { logerr ("GtTiffHdr", "B"); return err; }

  return OK;
}

/* get TIFF directory entry
 */
LOCAL INT GtTiffEntry (pInfile, EntryOffset, pDe)
     DATAFILE	*pInfile;
     ULONG	EntryOffset;
     ENTRY	*pDe;
{
  SHORT  err;
  USHORT tsize, BytesToRead, maxitems;
  ULONG  valpos;
  
  /* get the 2 words beginning with deTag
   */
  if (err = GtData (pInfile, EntryOffset, 2, TIFFSHORT, (CHAR *)&pDe->tag))
    return err;

  /* get the 2 dwords, beginning with deLength
   */
  if (err = GtData (pInfile, EntryOffset + 4L, 2, TIFFLONG,(CHAR *)&pDe->length))
    return err;

  /* get value
   */
  if (err = GtTiffSizeof (pDe->type, &tsize))
    return err;

  if ((BytesToRead = tsize * pDe->length) == 0)
    return OK; /* no need to read data */

  if ((pDe->value = malloc (BytesToRead)) == (CHAR *)0)
    { logerr ("GtTiffEntry", "not enough memory"); return err; }

  maxitems = (USHORT)(pDe->length);
  /* careful here: we can't just use valoffset to grab data out of, since
   * may already have been byte-reversed!
   */
  if (BytesToRead <= 4)
    valpos = EntryOffset + 8L;	/* valoffset starts on byte 8, wit de */
  else
    valpos = pDe->valoffset;
  if (err = GtData (pInfile, valpos, maxitems, pDe->type, pDe->value))
    return err;

  return OK;
}

LOCAL VOID EnterPicValue (entry, picture)
     ENTRY   *entry;
     PICTURE *picture;
{
  CHAR *bufptr;
  ULONG num, denom;
  SHORT i;

  switch (entry->tag)
    {
    case SUBFILETYPE:
      picture->image.subfileType = *((USHORT *)entry->value);
      break;
    case IMAGEWIDTH:
      picture->image.imWidth = *(USHORT *)entry->value;
      break;
    case IMAGELENGTH:
      picture->image.imLength = *(USHORT *)entry->value;
      break;
    case ROWSPERSTRIP:
      picture->image.rowsPerStrip = *(ULONG *)entry->value;
      break;
    case SAMPLESPERPIXEL:
      /* ignored */
      break;
    case BITSPERSAMPLE:
      picture->image.bitsPerSample = *(USHORT *)entry->value;
      break;
    case PLANARCONFIGURATION:
      picture->image.planConf = *(USHORT *)entry->value;
      break;
    case COMPRESSION:
      picture->image.compression = *(USHORT *)entry->value;
      break;
    case GROUP3OPTIONS:
      picture->image.gr3Options = *(ULONG *)entry->value;
      break;
    case GROUP4OPTIONS:
      picture->image.gr4Options = *(ULONG *)entry->value;
      break;
    case FILLORDER:
      picture->image.fillOrder = *(USHORT *)entry->value;
      break;
    case THRESHHOLDING:
      picture->image.threshholding = *(USHORT *)entry->value;
      break;
    case CELLWIDTH:
      picture->image.cellWidth = *(USHORT *)entry->value;
      break;
    case CELLLENGTH:
      picture->image.cellLength = *(USHORT *)entry->value;
      break;
    case MINSAMPLEVALUE:
      picture->photoMetric.minSampleValue = *(USHORT *)entry->value;
      break;
    case MAXSAMPLEVALUE:
      picture->photoMetric.maxSampleValue = *(USHORT *)entry->value;
      break;
    case PHOTOMETRICINTERPR:
      picture->photoMetric.photometInterpr = *(USHORT *)entry->value;
      break;
    case GRAYRESPONSEUNIT:
      picture->photoMetric.grayResponseUnit = *(USHORT *)entry->value;
      break;
    case GRAYRESPONSECURVE:
      bufptr = entry->value;
      picture->photoMetric.grayResponseCurve = 
	(USHORT *)malloc ((UINT)(entry->length+1) * sizeof (USHORT)); 
      for (i=0; i < entry->length; i++, bufptr += 2)
	picture->photoMetric.grayResponseCurve[i] = *(USHORT *)bufptr;
      picture->photoMetric.grayResponseCurve[entry->length] = -1;
      break;
    case XRESOLUTION:
      num = *(ULONG *)entry->value;
      bufptr = entry->value+4;
      denom = *(ULONG *)bufptr;
      picture->physWorld.xRes = (DOUBLE)num/(DOUBLE)denom;
      break;
    case YRESOLUTION:
      num = *(ULONG *)entry->value;
      bufptr = entry->value+4;
      denom = *(ULONG *)bufptr;
      picture->physWorld.yRes = (DOUBLE)num/(DOUBLE)denom;
      break;
    case RESOLUTIONUNIT:
      picture->physWorld.resUnit = *(USHORT *)entry->value;
      break;
    case ORIENTATION:
      picture->physWorld.orientation = *(USHORT *)entry->value;
      break;
    case DOCUMENTNAME:
      picture->context.docName = (CHAR *) entry->value;
      break;
    case PAGENAME:
      picture->context.pageName = (CHAR *) entry->value;
      break;
    case XPOSITION:
      num = *(ULONG *)entry->value;
      bufptr = entry->value+4;
      denom = *(ULONG *)bufptr;
      picture->context.xPos = (DOUBLE)num/(DOUBLE)denom;
      break;
    case YPOSITION:
      num = *(ULONG *)entry->value;
      bufptr = entry->value+4;
      denom = *(ULONG *)bufptr;
      picture->context.yPos = (DOUBLE)num/(DOUBLE)denom;
      break;
    case PAGENUMBER:
      picture->context.pageNo = *(USHORT *)entry->value;
      bufptr = entry->value+2;
      picture->context.noOfPages = *(USHORT *)bufptr;
      break;
    }
}

LOCAL VOID GetImageData (infile, image, stripOffsets, stripByteCounts)
     DATAFILE *infile;
     IMAGE    *image;
     ENTRY    *stripOffsets, *stripByteCounts;
{
  ULONG offset;

  if (stripOffsets->length != 1)
    { logerr ("GetImageData", "cannot handle multiple strips"); return; }

  if ((image->strips = (STRIP *)malloc (2 * sizeof (STRIP))) == (STRIP *)0)
    { logerr ("GetImageData", "not enough memory"); return; }

  image->strips[0].byteCount = *(ULONG *)stripByteCounts->value;
  offset = *(ULONG *)stripOffsets->value;

  if (fseek (infile->fdes, (LONG) offset, 0) == ERROR)
    { logerr ("GetImageData", "fseek error"); return; }

  if ((image->strips[0].data = 
       malloc ((UINT)image->strips[0].byteCount)) == (CHAR *)0)
    { logerr ("GetImageData", "not enough memory"); return; }

  if (fread (image->strips[0].data, sizeof (CHAR),
	     (INT)image->strips[0].byteCount, infile->fdes) == 0)
    { logerr ("GetImageData", "fread error"); return; }

  image->strips[1].byteCount = 0;
  image->strips[1].data = (char *)0;
}
