/*
 File: ReadRecord.c
 Author: K.R. Sloan
 Last Modified: 22 December 1992
 Purpose: read a tournament record into the standard in-memory
          data structure.  This includes both the information on the
          "pairing cards", and various header information, handled by
          "Header"

 Note: we are working here on reading the tournament record.  this
       format may (will!) change, so nothing here is cast in granite.
 */
#include <stdio.h>
#include <strings.h>
#include "Header.h"
#include "PairingCards.h"

extern int VERBOSE;

static void FatalError(s)
 {
  fprintf(stderr,"ReadRecord: FatalError(%s)\n",s);
  exit(-1);
 }

void ReadRecord(s)
 FILE *s;
 {
  int p,q,r;
  char name[NameLength];
  char value[ValueLength];

  if (VERBOSE) fprintf(stderr,"ReadRecord...\n");

  if (VERBOSE) fprintf(stderr,"Reading Header...\n");
  InitHeader(); 
  ReadHeader(s);

  strcpy(name,"PLAYERS");
  if  (SUCCESS != GetDescriptor(name,value))
   FatalError("No PLAYERS line");
  if (VERBOSE) fprintf(stderr,"[%s,%s]\n",name,value);
  PLAYERS = atoi(value);

  strcpy(name,"ROUNDS");
  if  (SUCCESS != GetDescriptor(name,value))
   FatalError("No ROUNDS line");
  if (VERBOSE) fprintf(stderr,"[%s,%s]\n",name,value);
  ROUNDS = atoi(value);

  if (VERBOSE) fprintf(stderr,"%d Players played %d Rounds\n",
                       PLAYERS,ROUNDS);

  /* allocate memory */

  if (!(PairingCards = (PlayerType *) malloc(PLAYERS*sizeof(PlayerType))))
   FatalError("allocating Pairing Cards");


  for(p=0;p<PLAYERS;p++)
   if (!(PairingCards[p].Games=(GameType *)malloc(ROUNDS*sizeof(GameType))))
    FatalError("allocating Games");

  /* read the individual "pairing cards" */
  for (p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;

    if (VERBOSE) fprintf(stderr,"Processing player %d\n",p+1);
    q = 0;
    for(;!feof(s);)
     {
      if (1 != fscanf(s,"%d",&q)) 
       { fscanf(s,"%*[^\n]\n"); continue;}
      break;
     }
    if (0 == q) FatalError("looking for pairing card");

    if ((p+1) != q) FatalError("pairing cards out of order");

    PC = &(PairingCards[p]);
    if (4 != fscanf(s," /%[^/]/%[^/]/ %ld %d",
                       (PC->Name),
                       (PC->Team),
                       &(PC->ID),
                       &(PC->Ro)))
     {
      fprintf(stderr,"Pairing Card %d is bad\n",q);
      FatalError("reading Pairing Card");  
     }

    if (VERBOSE) fprintf(stderr,"%s\n",PC->Name);

    if (1 != fscanf(s,"/%d",&(PC->Previous)))
     {
      if (PC->Ro > 0) PC->Previous = -1;
      else            PC->Previous =  0;
     }
 
    /* now for the rounds */

    for(r=0;r<ROUNDS;r++)
     {
      char WBbye[10];
      GameType *G;
      double score;

      if (VERBOSE) fprintf(stderr,"Processing player %d round %d\n",q,r+1);
      G = &(PC->Games[r]);
      if (1 != fscanf(s," %8s",WBbye)) FatalError("reading round");
      switch (WBbye[0])
       {
        case 'U':                     /* unplayed */
        case 'b': G->Opponent = 0;    /* also, "bye" */
                  G->Color = 'U';
                  break;
        case '?':
        case 'B':
        case 'W': if (1 != fscanf(s," %d",&(G->Opponent)))
                   FatalError("no  opponent");
                  G->Color = WBbye[0];
                  break;
        default: FatalError("Color not W B U ? bye");
       }   

      (G->Opponent)--;  /* zero based, internally */

      if (1 != fscanf(s," %lf",&score))
       FatalError("reading result");
      fscanf(s,"%*[^\n]\n"); /* not interested! */

      if (-1 == G->Opponent)
       {
        if      (1.0 == score) G->Result = 'B';
        else if (0.5 == score) G->Result = 'H';
        else if (0.0 == score) G->Result = 'F';
        else FatalError("invalid score");
       }
      else
       {
        if      (1.0 == score) G->Result = 'W';
        else if (0.5 == score) G->Result = 'D';
        else if (0.0 == score) G->Result = 'L';
        else FatalError("invalid score");
       }

     }
   }

  /* sanity checks */
  for (p=0;p<PLAYERS;p++)
   {
    PlayerType *PC1,*PC2;
    PC1 = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G1,*G2;
      int o1,o2;
      int OK;

      G1 = &(PC1->Games[r]);
      o1 = G1->Opponent;
      
      if (o1 != -1)
       {  /* played */
        if ((o1 < 0)  || (PLAYERS <= o1))
         {
          fprintf(stderr,"Bad Opponent (%d) in round %d for player %d\n",
                   o1+1,r,p+1);
          FatalError("bad opponent");
         }
        PC2 = &(PairingCards[G1->Opponent]);
        G2 = &(PC2->Games[r]);
        o2 = G2->Opponent;
        if (o2 != p)
         {
          fprintf(stderr,
          "Inconsistent Opponents (%d, %d) in round %d for players (%d,%d)\n",
                   o1+1,o2+1,r+1,p+1,o1+1);
          FatalError("inconsistent opponents");
         }

        OK = 0;
        if (  ('W' == G1->Color) 
            &&('B' == G2->Color) ) OK = -1;
        if (  ('B' == G1->Color) 
            &&('W' == G2->Color) ) OK = -1;
        if (  ('?' == G1->Color) 
            &&('?' == G2->Color) ) OK = -1;
        if (!OK)
         {
          fprintf(stderr,
          "Inconsistent Colors (%c,%c) in round %d for players (%d,%d)\n",
                    G1->Color,G2->Color,r+1,p+1,o1+1);
          FatalError("Inconsistent Colors");
         }

        OK = 0;
        if (  ('W' == G1->Result)
            &&('L' == G2->Result) ) OK = -1;
        if (  ('L' == G1->Result)
            &&('W' == G2->Result) ) OK = -1;
        if (  ('D' == G1->Result)
            &&('D' == G2->Result) ) OK = -1;

        if (!OK)
         {
          fprintf(stderr,
          "Inconsistent Results (%c,%c) in round %d for players (%d,%d)\n",
                    G1->Result,G2->Result,r+1,p+1,o1+1);
          FatalError("Inconsistent Results");
         }
       }
     }
   }
 }
