/****************************************************************************
 * painter.c
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 * Author Chris Nuuja
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* This module is the core of the Painter renderer */

#include <ctype.h>
#include <math.h>
#include "ge_error.h"
#include "p3dgen.h"
#include "pgen_objects.h"
#include "assist.h"
#include "paintr_strct.h"
#include "painter.h"

/* 
The following is designed to figure out what machine this is running
on, and include appropriate definition files.
*/
#ifdef unix
#define USE_UNIX
#endif
#ifdef CRAY
#undef USE_UNIX
#endif
#ifdef ardent
#undef USE_UNIX
#endif
#ifdef CRAY
#include "unicos_defs.h"
#endif
#ifdef ardent
#include "unicos_defs.h"
#endif
#ifdef USE_UNIX
#include "unix_defs.h"
#endif

/* This renderer caches some transformation information in local memory,
 * to accelerate rendering traversal.  The current_renderer variable
 * is used to test to see if these caches need to be updated.
 */
static P_Renderer *current_renderer= (P_Renderer *)0;

/* The following structures are used across all instances of the renderer.
 * They are buffers for storing coordinates in the process of being
 * transformed from world to screen space and clipped.
 */
static int TempCoordBuffSz= INITIAL_TEMP_COORDS
/*  Coordinate buffers used for transformations and clipping  */
float *pnt_Xcoord_buffer,*pnt_Ycoord_buffer,*pnt_Zcoord_buffer;
float *pnt_Xclip_buffer, *pnt_Yclip_buffer, *pnt_Zclip_buffer;

/*  
 * The following variables hold the values  of the transformation
 * matrix EyeToImage, the eye coordinates to 2d (virtual) screen coordinate
 * transform, during rendering traversal.  They are stored this way
 * to accelerate rendering;  the rest of the time they live in the
 * renderer's EyeMatrix slot.  The value at EI[3][3] is held in 
 * (3+3*4 =) EI15.  The current_renderer slot is used to identify when
 * these variables need to be read out of EyeMatrix.
 */
static float EI0,  EI1,  EI2,  EI3,   EI4,   EI5,   EI6,   EI7;
static float EI8,  EI9,  EI10, EI11,  EI12,  EI13,  EI14,  EI15;

/* 
   Utility function to enlarge the size of the temporary coordinate buffers if
   needed
*/
static void check_coordbuffsz(P_Renderer *self, int numcoords)
{
  if (numcoords >= TempCoordBuffSz)
    {
      while(numcoords >= TempCoordBuffSz)
	TempCoordBuffSz  *= 2;
      pnt_Xcoord_buffer = (float *) 
	realloc(pnt_Xcoord_buffer,TempCoordBuffSz*sizeof(float));
      pnt_Ycoord_buffer = (float *) 
	realloc(pnt_Ycoord_buffer,TempCoordBuffSz*sizeof(float));
      pnt_Zcoord_buffer = (float *) 
	realloc(pnt_Zcoord_buffer,TempCoordBuffSz*sizeof(float));
      pnt_Xclip_buffer = (float *) 
	realloc(pnt_Xclip_buffer,TempCoordBuffSz*sizeof(float));
      pnt_Yclip_buffer = (float *) 
	realloc(pnt_Yclip_buffer,TempCoordBuffSz*sizeof(float));
      pnt_Zclip_buffer = (float *) 
	realloc(pnt_Zclip_buffer,TempCoordBuffSz*sizeof(float));
      if ( !pnt_Xcoord_buffer || !pnt_Ycoord_buffer || !pnt_Zcoord_buffer
	  || !pnt_Xclip_buffer || !pnt_Yclip_buffer || !pnt_Zclip_buffer )
	ger_fatal("painter: pnt_init_renderer: memory allocation failed!\n");
    }
}

/*
   Global Data Used: pnt_Xcoord_buffer, pnt_Ycoord_buffer, pnt_Zcoord_buffer
   Expl:  This routine adds a record to the depth buffer, which is later
	  sorted to achieve painters algorithm for hidden surface handeling.
	  It creates a polygon record using the coordinates in the
	  global coordinate buffers and the input parameters.  This polygon
	  is then added to DepthBuffer at index DepthCount.
*/
static void insert_depthBuff(P_Renderer *self, int numcoords, float zdepth,
		      primtype type, int color)
{
	int i,newpoly,x_index,y_index,z_index;

	newpoly = pnt_new_DepthPoly(self);
	pnt_fillDpoly_rec(self,newpoly,numcoords,type);

	x_index = DPOLYBUFFER(self)[newpoly].x_index;
	y_index = DPOLYBUFFER(self)[newpoly].y_index;
	z_index = DPOLYBUFFER(self)[newpoly].z_index;
	for (i=0;i<numcoords;i++)
		{
		DCOORDBUFFER(self)[x_index++] = pnt_Xcoord_buffer[i];
		DCOORDBUFFER(self)[y_index++] = pnt_Ycoord_buffer[i];
		DCOORDBUFFER(self)[z_index++] = pnt_Zcoord_buffer[i];
		};
	DEPTHBUFFER(self)[DEPTHCOUNT(self)].poly = newpoly;
	DEPTHBUFFER(self)[DEPTHCOUNT(self)].key = zdepth;
	DEPTHCOUNT(self)++;
	DPOLYBUFFER(self)[newpoly].color = color;
	if (DEPTHCOUNT(self) > MAXDEPTHPOLY(self))
		printf("ERROR: Depth Buffer Overflow: %d > %d\n",
			DEPTHCOUNT(self), MAXDEPTHPOLY(self));

}

/*
   Global Data Used: none

   Expl:  This routine alters the intensity of <poly>'s color record
	  depending on its orientation to the lighting source(s).  
*/
static int calc_intensity(P_Renderer *self, Pnt_Colortype *oldcolor, 
			  Pnt_Polytype *polygon, float *trans, 
			  Pnt_Vectortype *normal)
{
  Pnt_Vectortype light;
  float red, green, blue, mx, my, mz, x, y, z, distance, percent;
  float source_red, source_green, source_blue;
  int newcolor, lightnum;
  register float *rtrans= trans;
  
  red   = oldcolor->r;
  green = oldcolor->g;
  blue  = oldcolor->b;
  
  source_red = 0.0;
  source_green = 0.0;
  source_blue = 0.0;
  
  /* Translate vertex coords from model to world coordinate system */
  mx= *(polygon->xcoords);
  my= *(polygon->ycoords);
  mz= *(polygon->zcoords);
  x = rtrans[0]*mx + rtrans[4]*my + rtrans[8]*mz + rtrans[12];
  y = rtrans[1]*mx + rtrans[5]*my + rtrans[9]*mz + rtrans[13];
  z = rtrans[2]*mx + rtrans[6]*my + rtrans[10]*mz + rtrans[14];
  
  for (lightnum = 0; lightnum < DLIGHTCOUNT(self); lightnum++)
    {
      light.x = DLIGHTBUFFER(self)[lightnum].loc.x - x;
      light.y = DLIGHTBUFFER(self)[lightnum].loc.y - y;
      light.z = DLIGHTBUFFER(self)[lightnum].loc.z - z;
      distance = sqrt( (light.x * light.x) + (light.y * light.y)
		      + (light.z * light.z) );
      if (distance == 0.0) percent= 1.0;
      else {
	light.x /= distance; light.y /= distance; light.z /= distance;
	percent = pnt_dotproduct( normal, &light );
      }
      if (percent < 0.0)
	percent = 0.0;
      source_red += (percent * DLIGHTBUFFER(self)[lightnum].color.r);
      source_green += (percent * DLIGHTBUFFER(self)[lightnum].color.g);
      source_blue += (percent * DLIGHTBUFFER(self)[lightnum].color.b);
    }
  
  if ((red= red*( source_red + AMBIENTCOLOR(self).r ) ) > 1.0)
    red = 1.0;
  if ((green= green*( source_green + AMBIENTCOLOR(self).g ) ) > 1.0)
    green = 1.0;
  if ((blue= blue*( source_blue + AMBIENTCOLOR(self).b ) ) > 1.0)
    blue = 1.0;
  /*	the <a> in a color rec is not used yet 		*/
  newcolor = pnt_makeDcolor_rec(self,red,green,blue,1.0);
  return(newcolor);
}


/*
   Global Data Used: NONE (The wire-frame model switch will go here so that
			   the interior style will be hollow)
   Expl:  Draws a polygon,polyline, or polymarker with it's recorded color 
	  wristl is a cgmgen routine to set the interior style;
	  wpgndc is a cgmgen routine to set the polygon fill color;
	  wrtpgn is a cgmgen routine to draw a polygon;
*/
void pnt_render_polyrec(P_Renderer *self, int depth_index)
{
  int i,error=0,poly_index,numcoords,x_index,y_index,color;
  float red,green,blue;

  poly_index = DEPTHBUFFER(self)[depth_index].poly;
  
  i=1;
  wristl(&i,&error);
  if (error) ger_error("ERROR: Bad interior style:%d\n",error);
  x_index = DPOLYBUFFER(self)[poly_index].x_index;
  y_index = DPOLYBUFFER(self)[poly_index].y_index;
  numcoords = DPOLYBUFFER(self)[poly_index].numcoords;
  
  color = DPOLYBUFFER(self)[poly_index].color;
  red   = DCOLORBUFFER(self)[color].r;
  green = DCOLORBUFFER(self)[color].g;
  blue  = DCOLORBUFFER(self)[color].b;
  
  switch(DPOLYBUFFER(self)[poly_index].type)
    {
    case POLYGON:
      wpgndc(&red,&green,&blue,&error);
      wrtpgn(DCOORDBUFFER(self)+x_index,
	     DCOORDBUFFER(self)+y_index, &numcoords,&error);
      break;
    case POLYLINE:
      wplndc(&red,&green,&blue,&error);
      wrplin(DCOORDBUFFER(self)+x_index,
	     DCOORDBUFFER(self)+y_index, &numcoords,&error);
      break;
    case POLYMARKER:
      wpmkdc(&red,&green,&blue,&error);
      wrtpmk(DCOORDBUFFER(self)+x_index,
	     DCOORDBUFFER(self)+y_index,&numcoords, &error);
      break;
    default:
      ger_error("ERROR: Unknown primitive:%d\n",error);
    }
}


/*
   Global Data Used:  pnt_Xcoord_buffer, pnt_Ycoord_buffer, pnt_Zcoord_buffer
   Expl:  This routine transforms the polygon cooridnates in the
	  global coordinate buffers, which are positions in 3 dimensional
	  space, to their corresponding positions in 2-dimensional space.
	  The perspective transformation happens here, as well as normalization
	  of the viewing fustrum.  All these transformations are contained
	  in the transformation matrix represented by the global variables
	  EIx where x is between 0 and 15.  Note that only the EIx values with
   	  non-zero values are used in the calculations.
*/
static void trans3Dto2D(numcoords)
int numcoords;
{ 

    float x,y;
    float z,w;
    int i;

    for (i=0;i<numcoords;i++)
	{
	x = pnt_Xcoord_buffer[i];
	y = pnt_Ycoord_buffer[i];
	z = pnt_Zcoord_buffer[i];

        pnt_Xcoord_buffer[i] = EI0*x + EI8*z;
        pnt_Ycoord_buffer[i] = EI5*y + EI9*z;
        pnt_Zcoord_buffer[i] = EI10*z + EI14;
        w =  EI11*z;
        if (w == 0.0) w= HUGE;

/* must normalize the coordinates */
        pnt_Xcoord_buffer[i] /=  w;
        pnt_Ycoord_buffer[i] /=  w;
        pnt_Zcoord_buffer[i] /=  w;
	}
}

/*
   Global Data Used: pnt_Xcoord_buffer, pnt_Ycoord_buffer, pnt_Zcoord_buffer

   Expl:  This routine transforms <polygon>'s coordinates from their position in
	  the world coordinate system to their positions with respect to the 
	  camera.  Also, any transformations of gob-children relative to gobs 
	  have been pre-concatinated onto ViewMatrix, so these transformations 
	  happen here as well.  The transformed coordinates are placed in the
	  global coordinate buffers.
*/
static void translateWtoI(P_Renderer *self, Pnt_Polytype *poly, int numcoords)
{
  register float *xp, *yp, *zp;
  register float *ViewM,x,y,z;
  int i;

  ViewM= VIEWMATRIX(self);
  if (poly->numcoords == -1)
    ger_error("ERROR, Bad Polygon In WtoI\n");
  xp= poly->xcoords;
  yp= poly->ycoords;
  zp= poly->zcoords;
  for (i=0; i<numcoords;i++)
    {
      x = *xp++;
      y = *yp++;
      z = *zp++;
      
      pnt_Xcoord_buffer[i] = ViewM[0]*x + ViewM[4]*y + ViewM[8]*z + ViewM[12];
      pnt_Ycoord_buffer[i] = ViewM[1]*x + ViewM[5]*y + ViewM[9]*z + ViewM[13];
      pnt_Zcoord_buffer[i] = ViewM[2]*x + ViewM[6]*y + ViewM[10]*z + ViewM[14];
    }
}

/*
   Global Data Used:NONE
   Expl:  This routine returns 1 if the polygon defined by normal vector
	  <normal> is facing the camera, and 0 if it is facing away.
*/
static int forward_facing(P_Renderer *self, Pnt_Vectortype *normal)
{
  /* All we have to do is calculate the z component of the normal in
   * the camera's frame.  If it is less than zero, the polygon is
   * facing backwards.
   */
  register float *ViewM;
  float zcomp;

  ViewM= VIEWMATRIX(self);
  zcomp= ViewM[2]*normal->x + ViewM[6]*normal->y 
    + ViewM[10]*normal->z;

  return( (zcomp>=0.0) );
}

static void render_polymarker(P_Renderer *self, Pnt_Polytype *polymarker, 
			      int numcoords, Pnt_Colortype *color)
{
  int newcolor;
  float z_depth,red,green,blue;

  if (numcoords != 1)
    {
      ger_error("ERROR: Marker with more than one coordinate\n");
      return;
    }
  check_coordbuffsz(self,numcoords);
  /*  
   * translateWtoI places the translated coordinates of the marker in
   * the buffers pnt_Xcoord_buffer, pnt_Ycoord_buffer, pnt_Zcoord_buffer
   */
  translateWtoI( self,polymarker,numcoords );
  z_depth = pnt_Zcoord_buffer[0];
  
  if (!pnt_insideZbound(self,z_depth,HITHER_PLANE))
    return;
  if (!pnt_insideZbound(self,z_depth,YON_PLANE))
    return;
  trans3Dto2D(numcoords);
  red   = color->r;
  green = color->g;
  blue  = color->b;
  newcolor = pnt_makeDcolor_rec(self,red,green,blue,1.0);
  /*  Add marker to global DepthBuffer, increment DepthCount  */
  insert_depthBuff(self,numcoords,z_depth,POLYMARKER,newcolor);
}

static void render_polyline(P_Renderer *self, Pnt_Polytype *polyline, 
			    int numcoords, Pnt_Colortype *color)
{
  int new_numcoords=0,newcolor;
  float z_depth,red,green,blue;
  
  /*  
    translateWtoI places the translated coordinates of the polygon in
    the buffers pnt_Xcoord_buffer, pnt_Ycoord_buffer, pnt_Zcoord_buffer
    */
  check_coordbuffsz(self,numcoords);
  translateWtoI( self,polyline,numcoords );
  
  /*  Clip on Far and near Z */
  /*  Clipped coordinates may contain more points than before 
      clipping */
  z_depth=0.0;
  pnt_clip_Zline(self,HITHER_PLANE,numcoords,&new_numcoords,&z_depth);
  pnt_clip_Zline(self,YON_PLANE,new_numcoords,&numcoords,&z_depth);
  if (numcoords < 2)
    return;
  trans3Dto2D(numcoords);
  red   = color->r;
  green = color->g;
  blue  = color->b;
  newcolor = pnt_makeDcolor_rec(self,red,green,blue,1.0);
  /*  Add polyline to global DepthBuffer, increment DepthCount  */
  insert_depthBuff(self,numcoords,z_depth,POLYLINE,newcolor);
}

/*
   Global Data Used: pnt_Xcoord_buffer,pnt_Ycoord_buffer,pnt_Zcoord_buffer;
   Expl:  This routine transforms the polygon at index <polygon> from
	  world coordinates to (virtual)screen coordinates, clips it against
	  the hither and yon planes, calculates its color with respect to
	  its orientation to the light, and inserts it into the depth buffer 
	  for later sorting and displaying.  If backface culling is on and a 
	  polygon is facing away from the camera, it is not added to the depth 
	  buffer.  The original polygon record is unaltered.
*/
static void render_polygon(P_Renderer *self, Pnt_Polytype *polygon, 
			   float *trans, int numcoords, int backcull, 
			   Pnt_Colortype *color)
{
	int new_numcoords=0,newcolor;
	float length,z_depth;
	Pnt_Vectortype normal;

	/*  
	  translateWtoI places the translated coordinates of the polygon in
	  the buffers pnt_Xcoord_buffer, pnt_Ycoord_buffer, pnt_Zcoord_buffer
	*/
	check_coordbuffsz(self,numcoords);
	translateWtoI( self,polygon,numcoords );

	/* calculate the polygon's normal */
	pnt_calc_normal(self,polygon,trans,&normal);
	length = pnt_vector_length(&normal);
	if (length == 0.0)
		length=1.0;

	/*  do backface culling */
	if ( backcull && (!forward_facing(self,&normal)) ) {
	  return;
	}
	else {
	  normal.x /= length;
	  normal.y /= length;
	  normal.z /= length;
	  newcolor = calc_intensity(self,color,polygon,trans,&normal);
	  /*  Clip on Far and near Z */
	  /*  Clipped coordinates may contain more points than before 
	      clipping */
	  /*  pnt_clip_Zpolygon assumes Hither plane is clipped before Yon */
	  z_depth=0.0;
	  pnt_clip_Zpolygon(self,HITHER_PLANE,numcoords,&new_numcoords,
			    &z_depth);
	  pnt_clip_Zpolygon(self,YON_PLANE,new_numcoords,&numcoords,
			    &z_depth);
	  /* Make sure we still have a polygon */
	  if (numcoords < 3)
	    return;
	  trans3Dto2D(numcoords);
	  /*  Add polygon to global DepthBuffer, increment DepthCount  */
	  insert_depthBuff(self,numcoords,z_depth,POLYGON,newcolor);
	};
}

void pnt_recache(P_Renderer *self)
/* This routine copies some renderer data into caches for faster access */
{
  ger_debug("painter: swap_in_vars");

  EI0= EYEMATRIX(self)[0];
  EI1= EYEMATRIX(self)[1];
  EI2= EYEMATRIX(self)[2];
  EI3= EYEMATRIX(self)[3];
  EI4= EYEMATRIX(self)[4];
  EI5= EYEMATRIX(self)[5];
  EI6= EYEMATRIX(self)[6];
  EI7= EYEMATRIX(self)[7];
  EI8= EYEMATRIX(self)[8];
  EI9= EYEMATRIX(self)[9];
  EI10= EYEMATRIX(self)[10];
  EI11= EYEMATRIX(self)[11];
  EI12= EYEMATRIX(self)[12];
  EI13= EYEMATRIX(self)[13];
  EI14= EYEMATRIX(self)[14];
  EI15= EYEMATRIX(self)[15];

  current_renderer= self;
}

/*
   Global Data Used: none
   Expl:  Transforms each polygon at index <object> from world coordinates to
	  (virtual)screen coordinates and stores them in DepthBuffer, which is 
	  later sorted and drawn in order.  <ViewMatrix> is altered to include
	  <object_trans> for drawing all polygons at index <object>, and
	  then returned to its original value.
*/
void pnt_render_primitive( P_Renderer *self, Pnt_Objecttype *object,
			  float *object_trans, int back_cull,
			  Pnt_Colortype *color)
{
  Pnt_Polytype *poly;
  Pnt_Colortype *newcolor;
  register int poly_index;
  int numcoords,numpolys;
  float *oldM;
  
  ger_debug("painter: pnt_render_primitive");

  if (current_renderer != self) pnt_recache(self);

  oldM = VIEWMATRIX(self);
  /*  
    Add gob-related transformations into ViewMatrix.  These occur BEFORE
    the translation from World to Eye coordinates (the ViewMatrix)
    */
  VIEWMATRIX(self) = pnt_mult3dMatrices(object_trans, VIEWMATRIX(self));
  poly = object->polygons;
  numpolys = object->num_polygons;
  for (poly_index=0; poly_index<numpolys; poly_index++)
    {
      numcoords = poly->numcoords;
      /* if no color of its own, inherit one */
      if (poly->color) newcolor = poly->color;
      else newcolor= color; 
      
      switch(poly->type)
	{
	case POLYGON:	
	  render_polygon(self,poly,object_trans,numcoords,
			 back_cull,newcolor);
	  break;
	case POLYLINE:
	  render_polyline(self,poly,numcoords,newcolor);
	  break;
	case POLYMARKER:
	  render_polymarker(self,poly,numcoords,newcolor);
	  break;
	default:
	  ger_error("ERROR: Undefined poly-type:%d\n", poly->type);
	  break;
	}
      poly++;
    }
  free(VIEWMATRIX(self));
  VIEWMATRIX(self) = oldM;
}

	
/*
   Global Data Used: none
   Expl: initialized the cgmgen routines.  It tells gplot that
	 the device is DeviceName, to output to file FileName
	 (which, if it is '-', will cause output to go to the screen)
	 and that all coordinates are normalized to between xmin,ymin
	 and xmax ymax.  If xmax and ymax are not equal, distortion
	 in size results.  If they are made smaller, magnification of
	 the image will occur.  They are set to 1.0 so that a 1x1 square
	 located at the camera position will just barely cover the entire
	 screen.  They have been set to values that produce results close to
	 Dore's renderer
*/
static void init_cgmgen(char *outfile, char *devname)
{

  int error=0;
  float xmin=0.0,ymin=0.0,xmax=2.0,ymax=2.0;
  
  csetdev(devname,&error);
  wrcopn(outfile,&error);
  setwcd(&xmin,&ymin,&xmax,&ymax,&error);
  if (error) ger_error("ERROR: Cgmgen initialization error:%d\n",error);
}

/*
   Global Data Used: none
   Expl: shuts down the cgmgen routines by calling wrtend.
*/
static void shutdown_cgmgen()
{
  int error=0;
  
  wrtend(&error);
  if (error)
    ger_error("ERROR: Cgmgen shutdown error:%d\n",error);
}

/*
   Global Data Used: LightLocation,LightIntensity,AmbientIntensity,
		     ViewMatrix, Zmin, Zmax,
		     pnt_Xcoord_buffer, pnt_Ycoord_buffer, pnt_Zcoord_buffer
   Expl:  calls init_cgmgen, and initializes all global variables.
	   Zmin, Zmax, and EyeMatrix will all be set to appropriate values
	   in ren_camera (located in painter_ren.c).  Light characteristics are
	   set, but are not (currently) used.
*/
void pnt_init_renderer(P_Renderer *self)
{
	int i;

	/*  
	    ViewMatrix is the matrix that translates  coordinates from
	    their position in the world to their position relative to the
	    Eye/Camera.  EyeMatrix hold projection information.
	*/
	VIEWMATRIX(self) = (float *) malloc(16*sizeof(float));
	if (!(VIEWMATRIX(self)))
	  ger_fatal(
		"painter: pnt_init_renderer: couldn't allocate 16 floats!");
	for (i=0;i<16;i++)
		VIEWMATRIX(self)[i]=0.0;
	VIEWMATRIX(self)[0] = 1.0;
	VIEWMATRIX(self)[5] = 1.0;
	VIEWMATRIX(self)[10] = 1.0;
	VIEWMATRIX(self)[15] = 1.0;

	EYEMATRIX(self) = (float *) malloc(16*sizeof(float));
	if (!(EYEMATRIX(self)))
	  ger_fatal(
		"painter: pnt_init_renderer: couldn't allocate 16 floats!");
	for (i=0;i<16;i++)
		EYEMATRIX(self)[i]=0.0;
	EYEMATRIX(self)[0] = 1.0;
	EYEMATRIX(self)[5] = 1.0;
	EYEMATRIX(self)[10] = 1.0;
	EYEMATRIX(self)[15] = 1.0;

	/* Set (default) Z clipping boundries,  Zmax is hither, Zmin is yon */
	ZMAX(self) = -0.01;
	ZMIN(self) = -900.0;

	/*
	   These buffers hold coordinates of polygons, polylines, markers, etc
	   as they are transformed from World coordinates to screen coordinates
	   The pnt_Xclip_buffer,pnt_Zclip_buffer,pnt_Yclip_buffer are used to
	   hold coordinates between clippings against the hither and
	   yon planes (Zmin and Zmax).
	*/
	pnt_Xcoord_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	pnt_Ycoord_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	pnt_Zcoord_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	pnt_Xclip_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	pnt_Yclip_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	pnt_Zclip_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	if ( !pnt_Xcoord_buffer || !pnt_Ycoord_buffer || !pnt_Zcoord_buffer
	    || !pnt_Xclip_buffer || !pnt_Yclip_buffer || !pnt_Zclip_buffer )
	  ger_fatal("painter: pnt_init_renderer: memory allocation failed!\n");
	init_cgmgen(OUTFILE(self),DEVICENAME(self));

}

/*
   Global Data Used: none
   Expl:  calls shutdown_cgmgen.
*/
void pnt_shutdown_renderer()
{
        shutdown_cgmgen();
}



