/****************************************************************************
 * emitp3d.c
 * Author Julie Roskies and Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include <stdio.h>
#include <math.h>
#include "err_defs.h"
#include "emit_defs.h"
#include "struct_defs.h"

#define FOV  30.0  	/* camera opening angle in degrees, change value here
			   if a different angle is wanted */
#define LESSER(a,b) (a>b)?b:a
#define BOUND(clr) (clr > 0.0) ? LESSER(1.0, clr) : 0.0	/* makes sure color values
						   are within bounds */


/* Some necessary snatches of lisp code */
/* The preamble follows */
static char preamble_text[]= "\
;P3D 1.0\n\
(load 'p3d.lsp)\n\
(defun vtx (xval yval zval) (make-vertex :x xval :y yval :z zval))\n\
(defun vtxn (xval yval zval xnrm ynrm znrm)\n\
	( make-vertex :x xval :y yval :z zval \n\
		:normal (make-vector :x xnrm :y ynrm :z znrm) ) )\n\
(defun vtxc (xval yval zval red green blue alpha)\n\
	( make-vertex :x xval :y yval :z zval \n\
		:clr (make-color :r red :g green :b blue :a alpha) ) )\n\
(defun vtxcn (xval yval zval red green blue alpha xnrm ynrm znrm )\n\
	( make-vertex :x xval :y yval :z zval \n\
		:clr (make-color :r red :g green :b blue :a alpha)\n\
		:normal (make-vector :x xnrm :y ynrm :z znrm) ) )\n\
(defun make-vertex-array (len vlist) ; number of vertices, and vertex list\n\
	(do	((varray (make-array len) varray)\n\
		 (vertlist vlist (cdr vertlist))\n\
		 (i 0 (+ i 1)))\n\
		((eq i len) varray)\n\
		(setf (aref varray i) (eval (car vertlist)))))\n\
(defun facet (&rest idlist) ; list of vertex id's in vertex-array\n\
	(do 	((vlist nil vlist)\n\
		 (ids idlist (cdr ids)))\n\
		 ((null ids) vlist)\n\
		 (setq vlist \n\
			(cons (aref vertex-array (car ids)) vlist))))\n\
(defun make-facet-array (len flist) ; number of facets, and facet list\n\
	(do	((farray (make-array len) farray)\n\
		 (facetlist flist (cdr facetlist))\n\
		 (i 0 (+ i 1)))\n\
		((eq i len) farray)\n\
		(setf (aref farray i) (eval (car facetlist)))))\n\
(defun part (min max) (mesh vertexlist\n\
	(do ((flist nil flist)\n\
		(i (- min 1) (+ i 1)))\n\
		((eq i max) flist)\n\
		(setq flist (cons (aref facet-array i) flist)))))\n\
(defun partc (min max rd grn blu alfa)\n\
	(def-gob :attr (list (cons 'color \n\
				(make-color :r rd :g grn :b blu :a alfa)))\n\
		:children (list (part min max))))\n\
(setq vertexlist (list\n\
"; /* end of preamble_text */


static char facet_intro[]= "\
)) ;to end vertex list\n\
(setq vertex-array (make-vertex-array %d vertexlist))\n\
(setq facet-array (make-facet-array\n\
";  /* End of facet_intro */


static char part_intro[]= "\
))) ;to end facet array\n\
(setq model (def-gob :children (list\n\
";  /* End of part_intro */

 
/* The postamble follows */
static char postamble_text[]= "\
\t))) ; to close earlier setq\n\
(setq best-lights (def-gob :children (list\n\
	(ambient white)\n\
	(light (make-point :x 20.0 :y 20.0 :z 0.0) white ))))\n\
(setq best-camera (make-camera :lookfrom (make-point :x %f :y %f :z %f)\n\
				:lookat (make-point :x %f :y %f :z %f)\n\
				:fovea %f :up y-vec :hither %f \n\
				:yon %f ))\n\
(snap model best-lights best-camera)\n\
"; /* end of postamble_text */

static void emit_vtxn( thisvertex )
Vertex *thisvertex;
/*  This routine emits a vertex with normal information  but no color */
{
	/* No debugging;  this will be called too often. */

	(void)fprintf(outfile,  "(vtxn %f %f %f %f %f %f )\n",thisvertex->x, 
		thisvertex->y, thisvertex->z, thisvertex->normal->x,
		thisvertex->normal->y, thisvertex->normal->z );
}

static void emit_dvtxn(  thisvertex, thisdisp, scale )
Vertex *thisvertex;
Vertex *thisdisp;
float scale;
/*  This routine emits a displaced vertex with normal information  but no color */
{
	/* No debugging;  this will be called too often. */

	(void)fprintf(outfile,  "(vtxn %f %f %f %f %f %f )\n",
		scale * thisdisp->x + thisvertex->x, 
		scale * thisdisp->y + thisvertex->y, 
		scale * thisdisp->z + thisvertex->z, 
		((thisvertex->normal) ? thisvertex->normal->x : 0.0) + 
		((thisdisp->normal) ? thisdisp->normal->x : 0.0),
		((thisvertex->normal) ? thisvertex->normal->y : 0.0) + 
		((thisdisp->normal) ? thisdisp->normal->y : 0.0),
		((thisvertex->normal) ? thisvertex->normal->z : 0.0) + 
		((thisdisp->normal) ? thisdisp->normal->z : 0.0) );
}

static void emit_vtxc(  thisvertex )
Vertex *thisvertex;
/*  This routine emits a vertex with color information  but no normal */
{
	/* No debugging;  this will be called too often. */

	(void)fprintf(outfile,  "(vtxc %f %f %f %f %f %f %f)\n",thisvertex->x, 
		thisvertex->y, thisvertex->z, thisvertex->color->r,
		thisvertex->color->g, thisvertex->color->b,
		thisvertex->color->a );
}

static void emit_dvtxc(  thisvertex, thisdisp, scale )
Vertex *thisvertex;
Vertex *thisdisp;
float scale;
/*  This routine emits a displaced vertex with color information  but no normal */
{
	/* No debugging;  this will be called too often. */
	(void)fprintf(outfile,  "(vtxc %f %f %f %f %f %f %f)\n",
		scale * thisdisp->x + thisvertex->x, 
		scale * thisdisp->y + thisvertex->y, 
		scale * thisdisp->z + thisvertex->z, 
		BOUND(((thisvertex->color) ? thisvertex->color->r : 0.0) +
		      ((thisdisp->color) ? thisdisp->color->r : 0.0)),
		BOUND(((thisvertex->color) ? thisvertex->color->g : 0.0) +
		      ((thisdisp->color) ? thisdisp->color->g : 0.0)),
		BOUND(((thisvertex->color) ? thisvertex->color->b : 0.0) +
		      ((thisdisp->color) ? thisdisp->color->b : 0.0)),
		BOUND(((thisvertex->color) ? thisvertex->color->a : 0.0) +
		      ((thisdisp->color) ? thisdisp->color->a : 0.0)) );
}

static void emit_vtxcn(  thisvertex )
Vertex *thisvertex;
/*  This routine emits a vertex with normal and color information */
{
	/* No debugging;  this will be called too often. */

	(void)fprintf(outfile,  "(vtxcn %f %f %f %f %f %f %f %f %f %f)\n",
		thisvertex->x, thisvertex->y, thisvertex->z, 
		thisvertex->color->r, thisvertex->color->g, 
		thisvertex->color->b, thisvertex->color->a,
		thisvertex->normal->x, thisvertex->normal->y, 
		thisvertex->normal->z );
}

static void emit_dvtxcn(  thisvertex, thisdisp, scale)
Vertex *thisvertex;
Vertex *thisdisp;
float scale;
/*  This routine emits a displaced vertex with normal and color information */
{
	/* No debugging;  this will be called too often. */

	(void)fprintf(outfile,  "(vtxcn %f %f %f %f %f %f %f %f %f %f)\n",
		scale * thisdisp->x + thisvertex->x, 
		scale * thisdisp->y + thisvertex->y, 
		scale * thisdisp->z + thisvertex->z, 
		BOUND(((thisvertex->color) ? thisvertex->color->r : 0.0) +
		      ((thisdisp->color) ? thisdisp->color->r : 0.0)),
		BOUND(((thisvertex->color) ? thisvertex->color->g : 0.0) +
		      ((thisdisp->color) ? thisdisp->color->g : 0.0)),
		BOUND(((thisvertex->color) ? thisvertex->color->b : 0.0) +
		      ((thisdisp->color) ? thisdisp->color->b : 0.0)),
		BOUND(((thisvertex->color) ? thisvertex->color->a : 0.0) +
		      ((thisdisp->color) ? thisdisp->color->a : 0.0)), 
		((thisvertex->normal) ? thisvertex->normal->x : 0.0) + 
		((thisdisp->normal) ? thisdisp->normal->x : 0.0),
		((thisvertex->normal) ? thisvertex->normal->y : 0.0) + 
		((thisdisp->normal) ? thisdisp->normal->y : 0.0),
		((thisvertex->normal) ? thisvertex->normal->z : 0.0) + 
		((thisdisp->normal) ? thisdisp->normal->z : 0.0) );
}

static void emit_vtx (  thisvertex )
Vertex *thisvertex;
/*  This routine emits a vertex with neither normal information nor color */
{
	/* No debugging;  this will be called too often. */

	(void)fprintf(outfile,  "(vtx %f %f %f)\n",thisvertex->x, 
		thisvertex->y, thisvertex->z );
}

static void emit_dvtx (  thisvertex, thisdisp, scale )
Vertex *thisvertex;
Vertex *thisdisp;
float scale;
/*  This routine emits a displaced vertex with neither normal information nor color */
{
	/* No debugging;  this will be called too often. */

	(void)fprintf(outfile,  "(vtx %f %f %f)\n",
		scale * thisdisp->x + thisvertex->x, 
		scale * thisdisp->y + thisvertex->y, 
		scale * thisdisp->z + thisvertex->z );
}

static void emit_facet (  thisfacet )
Facet *thisfacet;
/*  This routine emits a facet by printing its vertices node by node */
{
	Vnode *thisvnode;

	/* No debugging; this will be called too often */

	(void) fprintf (outfile, "(facet %ld ", thisfacet->myvertex);
	for(thisvnode= thisfacet; thisvnode->next; thisvnode= thisvnode->next)
		fprintf(outfile, "%ld ",thisvnode->next->myvertex);	
	fprintf(outfile, ")\n");
}

static void emit_part (  thispart )
Part *thispart;
/* This routine emits a part */
{
	/* No debugging; this will be called too often */

	(void) fprintf (outfile, "\t(part %d %d)\n", thispart->min, thispart->max );
}
	    
static void emit_partc (  thispart )
Part *thispart;
/* This routine emits a part with color information */
{
	/* No debugging; this will be called too often */

	(void) fprintf (outfile, "\t(partc %d %d %f %f %f %f)\n", 
		thispart->min, thispart->max, 
		(double)thispart->color->r, (double)thispart->color->g,
		(double)thispart->color->b, (double)thispart->color->a );
}
	    

void emit_p3d_model ( thismodel )
Model *thismodel;
/* This routine writes the p3d form of the model to standard output */
{
	Vertex *thisvertex;
	Vertex *thisdisp;
	Facet *thisfacet;
	Part *thispart;

	float fovrad= FOV * 3.1415926 / 180.0; 	/* FOV expressed in radians */
        float x_mid=
		(thismodel->bound_box->x_min + thismodel->bound_box->x_max)/2;
        float y_mid= 
		(thismodel->bound_box->y_min + thismodel->bound_box->y_max)/2;
        float z_mid= 
		(thismodel->bound_box->z_min + thismodel->bound_box->z_max)/2;
        float x_span = 
		(thismodel->bound_box->x_max - thismodel->bound_box->x_min);
        float y_span = 
		(thismodel->bound_box->y_max - thismodel->bound_box->y_min);

	float z_dist=
		thismodel->bound_box->z_max + 
		((x_span > y_span) ?  x_span / (2 * tan (fovrad * 9.0/20.0) ) :
				      y_span / (2 * tan (fovrad * 9.0/20.0) ));

		/* How far away the camera must be to get the whole picture
		   in, with a bit of room to spare */

	float hither= 0.25*( thismodel->bound_box->z_max - z_dist );
	float yon= 4.0*( thismodel->bound_box->z_min - z_dist );
	        /* hither and yon clipping distances, with wide margins */

	ger_debug("emit_p3d_model: %ld vertices, %ld facets, %d parts",
		thismodel->vertex_count, thismodel->facet_count, 
		thismodel->part_count );

	/* Emit the preamble */
     	(void)fprintf(outfile, preamble_text);

	/* Emit the vertex coordinates */
	if (thismodel->displace_list)
	for (thisvertex= thismodel->vertex_list, 
                thisdisp= thismodel->displace_list;
		thisvertex < thismodel->vertex_list + thismodel->vertex_count;
		thisvertex++, thisdisp++)
		if ((thisvertex->normal) || (thisdisp->normal))
			if ((thisvertex->color) || (thisdisp->color))
			    emit_dvtxcn ( thisvertex, thisdisp, 
					 thismodel->displace_scale);
			else emit_dvtxn ( thisvertex, thisdisp, 
					 thismodel->displace_scale);
		else
			if ((thisvertex->color) || (thisdisp->color))
			    emit_dvtxc ( thisvertex, thisdisp, 
					thismodel->displace_scale);
			else emit_dvtx ( thisvertex, thisdisp, 
					thismodel->displace_scale);
        else
	for (thisvertex= thismodel->vertex_list;
		thisvertex < thismodel->vertex_list + thismodel->vertex_count;
		thisvertex++ )
		if (thisvertex->normal)
			if (thisvertex->color) emit_vtxcn ( thisvertex);
			else emit_vtxn ( thisvertex);
		else
			if (thisvertex->color) emit_vtxc ( thisvertex);
			else emit_vtx ( thisvertex);

 	/* Emit the introductory facet code, including the vertex array def */
	(void)fprintf(outfile, facet_intro, thismodel->vertex_count);

	/* Emit the facet count, and an opening bracket for the facet list */
       	(void)fprintf(outfile, " %d '(\n", thismodel->facet_count);

	/* Emit the vertex ID's for each facet */
	for (thisfacet= thismodel->facet_list;
		thisfacet < thismodel->facet_list + thismodel->facet_count;
		thisfacet++ ) emit_facet(  thisfacet );

	/* Emit the introductory part code */
	(void)fprintf(outfile, part_intro);

	/* Emit the max and min for each part */
	for (thispart= thismodel->part_list;
		thispart < thismodel->part_list + thismodel->part_count;
		thispart++ ) 
		if (thispart->color) emit_partc(  thispart );
		    else emit_part(  thispart );

  
	/* Emit the post_amble */
	(void)fprintf(outfile, postamble_text, 
		      x_mid, y_mid, z_dist, x_mid, y_mid, z_mid, FOV,
		      hither, yon);

}

