/****************************************************************************
 * assist_spln.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
The assist modules provides support for renderers which do not have the
capability to do certain operations.  The operations are simulated using
a facility which the renderer (hopefully) does have.  This module simulates
spline patches in terms of meshes.

Bezier patch implementation notes:
     Colors and normals for the triangles making up the mesh for the patch
     are calculated by Bezier interpolation of the colors and normals of
     the control points (if given).  Note that this means that if one
     control point has a color and/or normal, they all must.  (In practice
     it's the first control point that gets checked).  Note that the
     actual (analytical) normals to the spline patch are *not* used;
     the computational cost seemed too great for the benefit.
*/


#include <math.h>
#include "alisp.h"
#include "ge_error.h"
#include "p3d.h"
#include "assist.h"

#define bezier_facets 18
#define bezier_vertices 16
#define bezier_knots 16

/* Initialization state */
static int bezier_initialized= 0;

/* A node structure suitable for handling in arrays */
typedef struct _cnode {
  float x, y, z;
  int has_color;
  float r, g, b;
  int has_normal;
  float nx, ny, nz;
} Cnode;

/* Unrefined Bezier connectivity data */
static int bezier_connect[bezier_facets][3]= {
  {0,1,4}, {1,5,4}, {1,2,5}, {2,6,5}, {2,7,6}, {2,3,7},
  {4,5,8}, {5,9,8}, {5,6,9}, {6,10,9}, {6,7,10}, {7,11,10},
  {8,13,12}, {8,9,13}, {9,10,13}, {10,14,13}, {10,11,14}, {11,15,14} };

/* Macro to calculate cubic Bezier interpolation */
#define mcr_bezier( s, v1, v2, v3, v4 ) \
  (((-(v1) + 3*(v2) - 3*(v3) + (v4))*(s) + (3*(v1) - 6*(v2) + 3*(v3)))*(s) \
    +(-3*(v1) + 3*(v2)))*(s) + (v1)

static void build_mesh( vcount, varray, fcount, vert_per_facet, connect,
		vlist, flist )
int vcount, fcount;
Cnode varray[];
int *connect, *vert_per_facet;
Vertex_list *vlist;
Facet_list *flist;
/* This routine produces mesh data from coordinate and connectivity lists */
{
	Vertex thisvertex, *vertex_array;
	Color thiscolor;
	Vector thisvector;
	Vertex_list vcopy, facet;
	Facet_list fcopy;
	int vloop, floop;

	ger_debug("build_mesh:");

	/* Allocate temporary vertex array */
	if (!(vertex_array= 
		(Vertex *)malloc(vcount*sizeof(Vertex))))
		ger_fatal(
		    "build_mesh: Unable to allocate space for mesh vertices");

	/* Assemble the vertex list */
	vcopy= *vlist= create_vertex_list( vcount );
	for (vloop=0; vloop<vcount; vloop++ ) {
		thisvertex= create_vertex();
		(void)set_vertex_x( thisvertex, varray[vloop].x );
		(void)set_vertex_y( thisvertex, varray[vloop].y );
		(void)set_vertex_z( thisvertex, varray[vloop].z );
		if (varray[vloop].has_color) {
		  thiscolor= create_color();
		  (void)set_color_red( thiscolor, varray[vloop].r );
		  (void)set_color_green( thiscolor, varray[vloop].g );
		  (void)set_color_blue( thiscolor, varray[vloop].b );
		  (void)set_vertex_color( thisvertex, thiscolor );
		  free_color( thiscolor ); /* won't get deallocated because
					      of other refs */
		}
		if (varray[vloop].has_normal) {
		  thisvector= create_vector();
		  (void)set_vector_x( thisvector, varray[vloop].nx );
		  (void)set_vector_y( thisvector, varray[vloop].ny );
		  (void)set_vector_z( thisvector, varray[vloop].nz );
		  (void)set_vertex_normal( thisvertex, thisvector );
		  free_vector( thisvector ); /* won't get deallocated because
					      of other refs */
		}
		vertex_array[vloop]= thisvertex;
		(void)set_first_vertex( vcopy, thisvertex );
		vcopy= rest_of_vertices(vcopy);
		free_vertex( thisvertex ); /* won't get deallocated
					   because of other refs */
	      };

	/* Assemble the facet list, additional pointers to same vertices */
	fcopy= *flist= create_facet_list( fcount );
	for (floop=0; floop<fcount; floop++ ) {
		vcopy= facet= create_vertex_list( *vert_per_facet );
		(void)set_first_facet( fcopy, facet );
		free_vertex_list( facet ); /* won't get deallocated because
						of other refs */
		fcopy= rest_of_facets(fcopy);
		for (vloop=0; vloop<*vert_per_facet; vloop++) 
			{
			thisvertex= vertex_array[ *connect++ ];
			(void)set_first_vertex( vcopy, thisvertex );
			vcopy= rest_of_vertices(vcopy);
			};
		vert_per_facet++;
		};

	free( (char *)vertex_array );
}

static float bezier2(s, t, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11,
		     v12, v13, v14, v15, v16)
float s, t, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, 
  v14, v15, v16;
/* This routine computes a bicubic Bezier interpolation.  The main 
 * purpose of this is to make sure the intermediates only get interpolated 
 * once.
 */
{
  float x1, x2, x3, x4;

  x1= mcr_bezier( s, v1, v2, v3, v4 );
  x2= mcr_bezier( s, v5, v6, v7, v8 );
  x3= mcr_bezier( s, v9, v10, v11, v12 );
  x4= mcr_bezier( s, v13, v14, v15, v16 );

  return( mcr_bezier( t, x1, x2, x3, x4 ) );
}

static void calc_bezier(s, t, knots, thisnode)
float s, t;
Cnode knots[bezier_knots], *thisnode;
/* This routine actually does the Bezier interpolation */
{
  thisnode->x= bezier2( s, t, 
		       knots[0].x, knots[1].x, knots[2].x, knots[3].x,
		       knots[4].x, knots[5].x, knots[6].x, knots[7].x,
		       knots[8].x, knots[9].x, knots[10].x, knots[11].x,
		       knots[12].x, knots[13].x, knots[14].x, knots[15].x );
  thisnode->y= bezier2( s, t, 
		       knots[0].y, knots[1].y, knots[2].y, knots[3].y,
		       knots[4].y, knots[5].y, knots[6].y, knots[7].y,
		       knots[8].y, knots[9].y, knots[10].y, knots[11].y,
		       knots[12].y, knots[13].y, knots[14].y, knots[15].y );
  thisnode->z= bezier2( s, t, 
		       knots[0].z, knots[1].z, knots[2].z, knots[3].z,
		       knots[4].z, knots[5].z, knots[6].z, knots[7].z,
		       knots[8].z, knots[9].z, knots[10].z, knots[11].z,
		       knots[12].z, knots[13].z, knots[14].z, knots[15].z );
  if (knots[0].has_color) { /* all get color if first knot has it */
    thisnode->has_color= 1;
    thisnode->r= bezier2( s, t, 
			 knots[0].r, knots[1].r, knots[2].r, knots[3].r,
			 knots[4].r, knots[5].r, knots[6].r, knots[7].r,
			 knots[8].r, knots[9].r, knots[10].r, knots[11].r,
			 knots[12].r, knots[13].r, knots[14].r, knots[15].r );
    thisnode->g= bezier2( s, t, 
			 knots[0].g, knots[1].g, knots[2].g, knots[3].g,
			 knots[4].g, knots[5].g, knots[6].g, knots[7].g,
			 knots[8].g, knots[9].g, knots[10].g, knots[11].g,
			 knots[12].g, knots[13].g, knots[14].g, knots[15].g );
    thisnode->b= bezier2( s, t, 
			 knots[0].b, knots[1].b, knots[2].b, knots[3].b,
			 knots[4].b, knots[5].b, knots[6].b, knots[7].b,
			 knots[8].b, knots[9].b, knots[10].b, knots[11].b,
			 knots[12].b, knots[13].b, knots[14].b, knots[15].b );
  } else thisnode->has_color= 0;
  if (knots[0].has_normal) { /* all get normal if first knot has it */
    thisnode->has_normal= 1;
    thisnode->nx= bezier2( s, t, 
			 knots[0].nx, knots[1].nx, knots[2].nx, knots[3].nx,
			 knots[4].nx, knots[5].nx, knots[6].nx, knots[7].nx,
			 knots[8].nx, knots[9].nx, knots[10].nx, knots[11].nx,
			 knots[12].nx, knots[13].nx, knots[14].nx, 
			 knots[15].nx );
    thisnode->ny= bezier2( s, t, 
			 knots[0].ny, knots[1].ny, knots[2].ny, knots[3].ny,
			 knots[4].ny, knots[5].ny, knots[6].ny, knots[7].ny,
			 knots[8].ny, knots[9].ny, knots[10].ny, knots[11].ny,
			 knots[12].ny, knots[13].ny, knots[14].ny, 
			 knots[15].ny );
    thisnode->nz= bezier2( s, t, 
			 knots[0].nz, knots[1].nz, knots[2].nz, knots[3].nz,
			 knots[4].nz, knots[5].nz, knots[6].nz, knots[7].nz,
			 knots[8].nz, knots[9].nz, knots[10].nz, knots[11].nz,
			 knots[12].nz, knots[13].nz, knots[14].nz, 
			 knots[15].nz );
  } else thisnode->has_normal= 0;
}

static void extract_knots( knotlist, knots )
Vertex_list knotlist;
Cnode knots[bezier_knots];
/* This routine extracts knots from a knot list */
{
  Vertex thisvertex;
  Color thiscolor;
  Vector thisvector;
  int i;

  for (i=0; i<bezier_knots; i++) {
    if ( null(knotlist) || null(thisvertex=first_vertex(knotlist)) ) {
      ger_error(
	 "ast_bezier: extract_knots: found %d control points, needed %d.",
	 i, bezier_knots);
      return;
    }
    knots[i].x= vertex_x(thisvertex);
    knots[i].y= vertex_y(thisvertex);
    knots[i].z= vertex_z(thisvertex);
    if ( !null(thiscolor= vertex_color( thisvertex )) ) {
      knots[i].has_color= 1;
      knots[i].r= color_red( thiscolor );
      knots[i].g= color_green( thiscolor );
      knots[i].b= color_blue( thiscolor );
    } else knots[i].has_color= 0;
    if ( !null(thisvector= vertex_normal( thisvertex )) ) {
      knots[i].has_normal= 1;
      knots[i].nx= vector_x( thisvector );
      knots[i].ny= vector_y( thisvector );
      knots[i].nz= vector_z( thisvector );
    } else knots[i].has_normal= 0;
    knotlist= rest_of_vertices(knotlist);
  }
}

static int match_knots( knots, oldknots )
Cnode knots[bezier_knots], oldknots[bezier_knots];
/* This routine checks to see if the two knot sets match */
{
  int i, matchflag= 1;

  i= 0;
  while (matchflag) {
    if (knots[i].x != oldknots[i].x) matchflag= 0;
    if (knots[i].y != oldknots[i].y) matchflag= 0;
    if (knots[i].z != oldknots[i].z) matchflag= 0;
    if (knots[i].has_color != oldknots[i].has_color) matchflag= 0;
    if (knots[i].has_color) {
      if (knots[i].r != oldknots[i].r) matchflag= 0;
      if (knots[i].g != oldknots[i].g) matchflag= 0;
      if (knots[i].b != oldknots[i].b) matchflag= 0;
    }
    if (knots[i].has_normal != oldknots[i].has_normal) matchflag= 0;
    if (knots[i].has_normal) {
      if (knots[i].nx != oldknots[i].nx) matchflag= 0;
      if (knots[i].ny != oldknots[i].ny) matchflag= 0;
      if (knots[i].nz != oldknots[i].nz) matchflag= 0;
    }
    if (++i == bezier_knots) break;
  }

  return(matchflag);
}

static void copy_knots( knots, oldknots )
Cnode knots[bezier_knots], oldknots[bezier_knots];
/* This routine copies its first argument to its second */
{
  int i;

  for (i=0; i<bezier_knots; i++) {
    oldknots[i].x= knots[i].x;
    oldknots[i].y= knots[i].y;
    oldknots[i].z= knots[i].z;
    oldknots[i].has_color= knots[i].has_color;
    if (knots[i].has_color) {
      oldknots[i].r= knots[i].r;
      oldknots[i].g= knots[i].g;
      oldknots[i].b= knots[i].b;
    }
    oldknots[i].has_normal= knots[i].has_normal;
    if (knots[i].has_normal) {
      oldknots[i].nx= knots[i].nx;
      oldknots[i].ny= knots[i].ny;
      oldknots[i].nz= knots[i].nz;
    }
  }
}

void ast_bezier( knotlist, mesh_fun )
Vertex_list knotlist;
void (*mesh_fun)();
/* This routine produces a bicubic Bezier patch, using the renderer's 
 * mesh function. 
 */
{
        static Vertex_list vlist;
        static Facet_list flist;
        static Cnode knots[bezier_knots], oldknots[bezier_knots];
	Cnode coords[bezier_vertices];
	int i, j;
	float s, t, sstep, tstep;
	int vertex_counts[bezier_facets];
	int connect[3*bezier_facets], *connect_copy;

	ger_debug("ast_bezier");

	/* Extract the knots from the knotlist */
	extract_knots( knotlist, knots );

	/* 
	If the knots match the last ones used, use the existing mesh.  
	Otherwise, generate a new mesh.
	*/
	if (!bezier_initialized || !(match_knots( knots, oldknots ))) {
	  bezier_initialized= 1;

	  copy_knots( knots, oldknots );

	  /* Fill the vertex_counts array */
	  for (i=0; i<bezier_facets; i++) vertex_counts[i]= 3;

	  /* Fill the coordinates array (calculating vertex positions) */
	  sstep= 1.0/3.0;
	  tstep= 1.0/3.0;
	  t= 0.0;
	  for (j=0; j<4; j++) {
	    s= 0.0;
	    for (i=0; i<4; i++) {
	      calc_bezier( s, t, knots, &coords[i+4*j] );
	      s += sstep;
	    }
	    t += tstep;
	  }

	  /* Copy the (precalculated) connectivity array into the
	   * appropriate slots.
	   */
	  connect_copy= connect;
	  for (i=0; i<bezier_facets; i++)
	    for (j=0; j<vertex_counts[i]; j++)
	      *connect_copy++= bezier_connect[i][j];

	  /* Generate the mesh from the torus data and draw it */
	  build_mesh( bezier_vertices, coords, bezier_facets, 
		     vertex_counts, connect, &vlist, &flist );
	}

	(*mesh_fun)( vlist, bezier_vertices, flist, bezier_facets );
}

/* 
This routine is needed to reset the spline patch emulation meshes
in the event of a hard reset.  It need never be called by a renderer;
the user interface should call it when carrying out the hard reset.
*/
void ast_spln_reset( hard )
int hard;
{
	ger_debug("ast_prim_reset: resetting primitive meshes; hard= %d",
		  hard);

	if (hard) {
	  bezier_initialized= 0;
        }
}
