// This contains a spline editor subroutine/object
#include <gl/gl.h>
#include <gl/sphere.h>
#include <gl/device.h>
#include <gl/image.h>
#include <stream.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <malloc.h>

#include "actor.h"
#include "../inc/debug.h"
#include "../inc/colors.h"

void plot_t_spline(float [4][3],float);
void plot_dbg_spline(float [4][2]);
void plot_point(long,float,float);

void matrix_mult(float a[4][4],float b[4][3],float c[4][3]);
typedef 	struct	{	float real,imag;} complex;
 


extern "C" {
#include <forms.h>
    extern  long bcopy(const void *, void *, unsigned long );

    extern void savemovieframe(); 
    void zporc_(long int*,float *,float *);
}


// used in drawing the axis for actor routines     
static	float p1[] = { 0.0,-700.0};
static	float p2[] = { 0.0, 700.0};
static	float p3[] = { 0.0, 0.0};
static	float p4[] = { 900.0,0.0};


/*** Actor - intializor ***/
Actor::Actor(int a,char *name,long id_num)
{
    
    status("Actor::Initializing Actor",name);
    
    if (strlen(name) < TITLE_LEN) {
	strcpy(title,name);
    }
    else {
	strncpy(title,name,TITLE_LEN-1);
	status("WARNING: Title too long",name);
	status("Truncated");
    }
    
    id = id_num;
    num_pts = a;
    pts_flag = TRUE;
    pnts = (float *) malloc(num_pts*3*sizeof(float)); 
    spl_coeff = (float *)new float [num_pts-3][4][2];
    reset();
};


/*** ~Actor - destructor ***/
Actor::~Actor()
{
    status("Delete cntrl_pnts");
    free( pnts);
}	



/*** add_pnt -- add a point to the spline ***/
void Actor::add_pnt(float x,float y)
{
    // add the point in correct place, readjust the rest of the points
    
    int i;
    float	*new_pts;
    
    
    if (x < xpos(0)) {
	//status("Adding a new first point--OK");
	new_pts = (float *)malloc(sizeof(float)*(num_pts+1)*3); 
    	st_x= x;
	new_pts[0] = x;
	new_pts[1] = y;
	new_pts[2] = 0;
	bcopy((void *)pnts,(void *)&new_pts[3],sizeof(float)*3*num_pts);
	free(pnts);
	pnts = new_pts;
    }
    else if (x>xpos(num_pts-1)) {
	//status("Adding as New end point");
	new_pts = (float *)malloc(sizeof(float)*3*(num_pts+1));
	end_x = x;
	new_pts[num_pts*3  ] = x;
	new_pts[num_pts*3+1] = y;
	new_pts[num_pts*3+2] = 0;
	bcopy((void *)pnts,(void *)new_pts,sizeof(float)*3*num_pts);
	free(pnts);
	pnts = new_pts;
    }
    else {
	i=0;
	while ((xpos(i) < x) && (i<num_pts))
	  i++;

	//status("Total Number of pnts",num_pts);
	//status("Inserting Point in position",i);

	new_pts =(float *)malloc(sizeof(float)*(num_pts+1)*3); 
	bcopy(pnts,new_pts,sizeof(float)*3*i);
	new_pts[i*3  ] = x;
	new_pts[i*3+1] = y;
	new_pts[i*3+2] = 0;
	
	bcopy(&(pnts[i*3]), &new_pts[(i+1)*3],sizeof(float)*3*(num_pts-i));
	free(pnts);
	
	pnts = new_pts;
    }
    
    pts_flag = TRUE;
    num_pts++;
}


/*** del_pnt -- delete a point from the spline ***/
void Actor::del_pnt(int i)
{
    float	*new_pts;
    
    if (i>=num_pts || i<0) {
	status("Illegal Point Number to Delete",i);
	return();
    }
    

    new_pts = (float *)malloc(sizeof(float)*(num_pts-1)*3);
   
    if (i==0) { // delete the first point
	status("Deleting the first point-OK");
	st_x = xpos(1);
	bcopy(&pnts[3],new_pts, sizeof(float)*3*(num_pts-1));
    }
    else if (i+1 == num_pts) { // last point
	status("Deleting the Last Point-OK");
	end_x = xpos(i-1);
	bcopy(pnts,new_pts, sizeof(float)*3*(num_pts-1));
    }
    else { // general case, deleting an internal control point
	status("Maybe-OKDeleting internal point #",i);
	bcopy(pnts,new_pts,sizeof(float)*3*i);
	bcopy(&pnts[(i+1)*3],&new_pts[i*3],(num_pts-i-1)*3*sizeof(float));
    }
    
    free(pnts);
    num_pts--;
    pts_flag = TRUE;
    pnts = new_pts;
    
}


/*** find_point -- Find the control point(if any) at curr cursor loc. ***/
     int Actor::find_pnt()
{
    
    long hits;
    short pick_buf[50];
    
    status("Actor::find_point",title);
    
    initnames();
    pick(pick_buf,45);
    draw_pts();			// This won't show on the screen
    hits = endpick(pick_buf);
    
    if (hits >1) {
	return(pick_buf[1]);
    }	
    
    else if (hits==0)
      return(-1);
    else {
	// pick_buf[0] = Number of points in set
	// pick_buf[1] = first point selected
	// pick_buf[2] = 2nd point in 1st set (never happen)
	
	return((int) pick_buf[1]);
    }
}



/*** plot -- Plot spine control points ***/
void Actor::draw_pts(void)
{
    int i;
    // Note: this plots in the current GL window,
    // maybe should keep track of its on gid??
    
   // status("Actor::Plotting points");
    ORTHO; 
    cpack(c_yellow);
    for (i=0;i<num_pts; i++) {
	loadname(i);
	circf(xpos(i), ypos(i), CNTRL_PNT_RADIUS);
    }
}


/*** Actor::draw_cursor(float xpos) - under construction ***/
void Actor::draw_cursor(float xpos)
{
    float pnt[2];
    

    // New OVERDRAW Version

    status("Actor",title);
   status("Plotting cursor at",xpos);
 
   // pushattributes();
    drawmode(OVERDRAW);
    ORTHO;
    linewidth(2);
    color(0);
    clear(); zclear();
    if (xpos >0) {
	color(1);
	bgnline();
	pnt[0]=xpos;
	pnt[1]=0;
	v2f(pnt);
	pnt[1]=500;
	v2f(pnt);
	endline();
    }
    linewidth(1);
    drawmode(NORMALDRAW);
    //popattributes();
}


/*** draw - this draws a nice window w/ spline and labels ***/
void Actor::draw(float a_st, float a_end,float crsr)
{
   
    // This is written so can be overridden and only display the
    // part specified but for now just do the default
    
    float	mark[2];
    float	temp1,temp2;
    
    temp1 = disp_st;
    temp2 = disp_end;
    disp_st  = a_st;
    disp_end = a_end;
    
   
    cpack(c_grey);    clear();    zclear();
    draw_pts();				// this set the ORTHO 
    
  
    cpack(c_white); 
    bgnline();    v2f(p1);	v2f(p2);    endline();
    bgnline();    v2f(p3);	v2f(p4);    endline();
    
    // Plot the x-axis 
    float x;
    for (x=0;x<= end()+25; x+=25) {
	bgnline();
	if	(((int)x % 100)==0) {
	    mark[0] = (float)x;
	    mark[1] = 40;
	    v2f(mark);
	}
	else {
	    mark[0] = (float)x;
	    mark[1] = 20;
	    v2f(mark);
	}
	mark[1] = 0.0;
	v2f(mark);
	endline();
    }
    
    
   
    cpack(c_red);
    crvn(num_pts,(Coord [][3])pnts);
    swapbuffers();
    disp_st = temp1;
    disp_end = temp2;

}


/*** reset -- set the cntrl points to initial values ***/
void Actor::reset()
{
    int		cnt,indx;
    float inc;		// space between each spline control point
    float xpos;
   

    // change this to delete all the old points
    // and start with a standard number of points.

    status("Actor::reseting",title);
    indx=0;
    cnt = 0;
    st_x = -10;		disp_st = 0;
    end_x   = 500;	disp_end= 500;
    xpos = st_x;
    inc = (end_x - st_x) / (num_pts);

    pts_flag = coeff_flag = TRUE;
    while (cnt < num_pts) {
	pnts[indx] = xpos;
	pnts[indx+1] = 0.0;
	pnts[indx+2] = 0.0;
	xpos+= inc;
	indx+= 3;
	cnt++;
    }
    
}



/*** setup_coeff - setup the coefficient table for all the spline segments ***/
void Actor::setup_coeff()
{
    int i;
    
    if (pts_flag==TRUE) { // # of spline control points has changed
	pts_flag = FALSE;
	coeff_flag=TRUE;
	delete spl_coeff;	// erase old coefficients structure
	spl_coeff = (float *)new float[num_pts-3][4][2];
    }

     
    if (coeff_flag==TRUE){  //	 this isn't set yet by the reset/move routine 
	coeff_flag=FALSE;
	for (i=0;i<num_pts-3; i++) { // for each spline segment 
	    calc_coeff((float [4][3])&pnts[i*3],
		       (float [4][2])&spl_coeff[i*8]);
				  
	    /*
	    plot_dbg_spline((float [4][2]) &spl_coeff[0]);
	    */
	}
    }

}
    

/*** calc_coeff - calculate the spline coefficients one spline segment ***/
void Actor::calc_coeff(float g[4][3], float r[4][2])
{
    
    // hard wired for the Cardinal Spline Bases Matrix
    // g[4][3] contains the x,y,z coordinated of 4 control points
    // it can safely be assumed that all z-coordinates are zero 
    
    // first do the X-coefficients

    //status("Calc_coeff",title);
    
    r[0][0]= ( - 0.5 	 * g[0][0] +
	         1.5	 * g[1][0] +
	       - 1.5	 * g[2][0] +
	         0.5	 * g[3][0]  );
    
    
    r[1][0]= (   1.0 	 * g[0][0] +
	      -2.5	 * g[1][0] +
	       2.0	 * g[2][0] +
	       -.5	 * g[3][0] );
    
    
    r[2][0] = ( -.5	* g[0][0] +
	       .0	* g[1][0] +
	       0.5	* g[2][0] +
	       0.0	* g[3][0]);
    
    r[3][0] = (  0.0	* g[0][0] +
	         1.0	* g[1][0] +
	         0.0	* g[2][0] +
	         0.0	* g[3][0]);
  
    
    r[0][1]= ( - 0.5 	 * g[0][1] +
	         1.5	 * g[1][1] +
	       - 1.5	 * g[2][1] +
	         0.5	 * g[3][1]  );
    
    
    r[1][1]= (   1.0 	 * g[0][1] +
	      -2.5	 * g[1][1] +
	       2.0	 * g[2][1] +
	       -.5	 * g[3][1] );
    
    
    r[2][1] = ( -.5	* g[0][1] +
	       .0	* g[1][1] +
	       0.5	* g[2][1] +
	       0.0	* g[3][1]);
    
    r[3][1] = (  0.0	* g[0][1] +
	         1.0	* g[1][1] +
	         0.0	* g[2][1] +
	         0.0	* g[3][1]);
  
  /*
    status(title,"Geometry Points");
    status("P0",g[0][0],g[0][1]);
    status("P1",g[1][0],g[1][1]);
    status("P2",g[2][0],g[2][1]);
    status("P3",g[3][0],g[3][1]);
    
    status(title,"X-Spline Coefficients");
    status("t*t*t	",r[0][0]);
    status("t*t		",r[1][0]);
    status("t		",r[2][0]);
    status("K		",r[3][0]);

    status(title,"Y-Spline Coefficients");
    status("t*t*t	",r[0][1]);
    status("t*t		",r[1][1]);
    status("t		",r[2][1]);
    status("K		",r[3][1]);
  */
}

/*** get_value - return a parameter value from spline at "xpos" ***/
float Actor::get_value(float xcoor)
{

    // Make sure its a valid location 
    if ((xcoor <xpos(1)) ||  (xcoor > xpos(num_pts-2))) { 
#ifdef SAFE_DEBUG
	status("Xcoor ",xcoor);
	status("WARNING: Does not exist on this spline",title);
#endif
	return(-1);
    }


    // Find which segment of the spline it is on

    int	i,segment;
    float val;  
    for (i=2; i <(num_pts-1); i++) {
	if (xpos(i) > xcoor) {
	    segment = i-2; 
	    break;
	}
    }
   
    if (i==num_pts-1) {
	status(title,xcoor);
	status("Spline Segment Doesn't exist == should never hapen");
	return(-1);
    }


    val = value(xcoor,(float [4][2])&spl_coeff[8*segment]);
    return(val);
}



/*** value -- return the spline y-value given an x-position ***/
float	Actor::value(float xcoor, float gen_coeff[4][2])
{
    /* !!! This routine is called by evaluate in callbacks.cc and is the
      most important routine to be optimized.
	Before being called the spline coefficients must be precalculated
	  by calling the routine: Actor::setup_coeff()
	    */

    int		segment,i;
    float	ycoor; 
    float	t,det,x1,x2;
    float	coeff[3+1];		/* for IMSL fortran routine */
   
    complex	roots[3];		/* for IMSL fortran routine */
    static long int	order=3;  	// for imsl routine 

    
    // I use a IMSL routine zporc to solve the t^3 t^2 T t^0 = 0
    // equation using the Jenkin-Traub 3-stage algorithm
    // Since I am lazy, you will have to roll your own 
    

    // Solve 3rd Order equation using IMSL Routine
    if (gen_coeff[0][0] != 0.0) { 
	t = -1;
	coeff[0] = gen_coeff[3][0]-xcoor;
	coeff[1] = gen_coeff[2][0];
	coeff[2] = gen_coeff[1][0];
	coeff[3] = gen_coeff[0][0];
	
	zporc_(&order,&coeff[0],(float *)&roots[0]);
	//status("Cooeficients 0,1 ",coeff[0],coeff[1]);
	//status("Coeeficients 2,3 ",coeff[2],coeff[3]);
	//status("Root 1 ",roots[0].real,roots[0].imag);
	//status("Root 2 ",roots[1].real,roots[1].imag);
	//status("Root 3 ",roots[2].real,roots[2].imag);
	for (i=0;i<3;i++)
	  if ((roots[i].imag==0)&&(roots[i].real>=0) &&(roots[i].real<=1.0)) {
	      t= roots[i].real;
	      break;
	  }
	
	if (t==-1) {
	    status(title,xcoor);
	    status("zporc_ Only Complex roots to spline equation : ");
	}
    }
    
    // 2nd order solver w/ quadratic formula
    else if (gen_coeff[1][0] !=0.0) {
	status("2nd Order Solver ");
	det = ((gen_coeff[2][0]*gen_coeff[2][0])-
	       4.0*gen_coeff[1][0]*(gen_coeff[3][0]-xcoor));
	
	if	 (det<0) {
	    status(title,xcoor);
	    status("Negative determinant in 2order solver");
	}
	else {
	    det = fsqrt(det);
	    x1 = (-gen_coeff[2][0] + det)/ 2*gen_coeff[1][0];
	    x2 = (-gen_coeff[2][0] - det)/ 2*gen_coeff[1][0];
	    status("2nd order results for t",x1,x2);
	    if (x1 > 0)	     	 	t = x1;
	    else if (x2 > 0)		t = x2;
	    else {
		status("WARNING:Negative t in 2nd order solver",x1,x2);
	    }
	}
    }
    
    // Linear Solver 
    else if (gen_coeff[2][0] !=0.0) {
	t = -((gen_coeff[3][0]-xcoor)/gen_coeff[2][0]); 
    }
    else  {
	status("WARNING:: \07 All polynomial gen_coefficients are zero",title);
    }
    
    if ((t<0) || (t>1)) {
	status("WARNING: \07 Bad T value in Actor",title);
	status("t = ",t);
    }
    
    ycoor= (gen_coeff[0][1]*t*t*t + gen_coeff[1][1]*t*t + 
	    gen_coeff[2][1]*t     + gen_coeff[3][1]);
    
  
    return((float) ycoor/YMAX);
}


/*** prepend -- prepend an actor to frontof  the spline ***/
void Actor::prepend(Actor *new_actor,float duration)
{
    // first shift the existing points right "duration" units
    int i;

    for (i=1;i<num_pnts(); i++) 
	eq(i,xpos(i)+duration,ypos(i));

    for (i=1;i<new_actor->num_pnts()-1;i++)
      add_pnt(new_actor->xpos(i),new_actor->ypos(i));

}
    
/*** append -- add an actor to end of me ***/
void Actor::append(Actor *new_actor,float x_pos)
{
    status("Appending Actor ",new_actor->title);
   
    float trans = 23;

    status("Transition Value ",trans);
    int i;
      
    eq(num_pnts()-1,new_actor->xpos(1)+x_pos,new_actor->ypos(0));
    for (i=2;i<new_actor->num_pnts();i++) {
	add_pnt(new_actor->xpos(i)+x_pos,new_actor->ypos(i));
    }
}



/*** gl_event -- handle any non-FORMS event on the spline window ***/
void Actor::gl_event(long dev,short val,int curr_win)
{
    /* Possibilities:
    
    LEFT MOUSE:		- Trying to move a point
      
      MIDDLE MOUSE:	- Delete point if on point, else add point

	RIGHT MOUSE:	- Not Used
	
	REDRAW:		

	  The general event manager will only send the request here
	    if it is actually in this window 
	
	      Maybe later add some keypad controls to scroll the window

	      */

    long	sx,sy,ox,oy,mx,my;
    float	wx,wy;
    int		pck_pnt;

    
//    status("Actor::calback",title);
    
    switch (dev) {
      case LEFTMOUSE:
	
	if (val) { 	// Mouse button pushed
	    pck_pnt = find_pnt();
	    if (pck_pnt == -1)	
	      return;
	    
	    getorigin(&ox,&oy);
	    getsize(&sx,&sy);
	    
	    while(getbutton(LEFTMOUSE) == TRUE) {
		mx = getvaluator(MOUSEX);
		my = getvaluator(MOUSEY);
		
		// convert to standard coordinates
		wx = (float) ((float)(mx-ox)/(float)sx) ;
		wx = wx * ((disp_end+R_BORDER)-(disp_st-L_BORDER))+
		  (disp_st-L_BORDER);
		wy = (float)(my-oy)/(float)sy * (float)(YMAX-YMIN);
		
		//status("Picket Point",pck_pnt);
		//status("Location",xpos(pck_pnt),ypos(pck_pnt));
		//status("World X",wx);
		//status("World Y",wy);
		
		// Insure correct ordering of points is maintained
		if ((pck_pnt > 0 ) && (wx <  xpos(pck_pnt-1))) 
		  wx = xpos(pck_pnt-1);
		else if ((pck_pnt < num_pts-1) && (wx> xpos(pck_pnt+1)))
		  wx = xpos(pck_pnt+1);
		
		// Keep Y coordinate within bounds	
		if (wy > YMAX )
		    wy = YMAX;
		else if (wy< YMIN)	
		    wy = YMIN;
		    
		    
		eq(pck_pnt,wx,wy);		// assign new point location
		// Maybe update the st and end of this frame is it
		// was the first or last control point????
		draw();
	    }
	    coeff_flag = TRUE;		// Spline coeff needs recalc
	}
	break;

      case MIDDLEMOUSE:
	if (val) {	// mouse pushed
	    pck_pnt = find_pnt();
	    if (pck_pnt == -1 ) 	{ // no point choosen so add a point
		// add point code
		mx = getvaluator(MOUSEX);
		my = getvaluator(MOUSEY);
		getorigin(&ox,&oy);
		getsize(&sx,&sy);

		// convert to world corridnates
		// convert to standard coordinates
		wx = (float) ((float)(mx-ox)/(float)sx) ;
		wx = wx * ((disp_end+R_BORDER)-(disp_st-L_BORDER))+
		  (disp_st-L_BORDER);
		wy = (float)(my-oy)/(float)sy * (float)(YMAX-YMIN);

		add_pnt(wx,wy);
		
	    }
	    else {			// delete the choosen point
		del_pnt(pck_pnt);
		status("Control Point Deleted",pck_pnt);
	    }
	    draw();
	}
	break;
	
      case REDRAW:
	status("READRAW Window #",val);
	winset(val);
	draw();
	break;

    }
}






#ifdef DEBUG


// Some debugging routines to see how our spline solver is working
void plot_point(long color, float xcoor,float ycoor)
{

    float       coor[4];
    coor[0] = xcoor;
    coor[1] = ycoor;
    coor[2] = 0;
    coor[3] = .25;
    // status("gid2",gid);

    pushmatrix();
    cpack(color);
    //   bgnpoint();
    //    v3f(coor);
    //   endpoint();
    circf(xcoor,ycoor,2.5);
    popmatrix();
}





void plot_dbg_spline(float coef[4][2])
{

    float       t,xcoor,ycoor;
    status("Plottting debugging spline");
    for (t=0.0; t<1.0;t+=.01) {
        xcoor = coef[0][0]*t*t*t + coef[1][0]*t*t +coef[2][0]*t + coef[3][0];
        ycoor = coef[0][1]*t*t*t + coef[1][1]*t*t +coef[2][1]*t + coef[3][1];
        plot_point(0xFF0000,xcoor,ycoor);
    }
}


void    plot_t_spline(float coef[4][3],float t)
{
    float       xcoor,ycoor;
    xcoor = coef[0][0]*t*t*t + coef[1][0]*t*t +coef[2][0]*t + coef[3][0];
    ycoor = coef[0][1]*t*t*t + coef[1][1]*t*t +coef[2][1]*t + coef[3][1];
    //    status("T-point",xcoor,ycoor);
    plot_point(0x00FF00,xcoor,ycoor);
}

#endif








