/*
    My own buffered I/O routines for reading LFF files, since stdio is
	horribly inefficient (particularly fread()).

	FD     - a file descriptor (int)
	BAD_FD - an invalid descriptor used as a marker

	these are defined in lffbufio.h

    Routines are:

    FD _open_lff(char *name, int mode, off_t offset)

	Opens the specified  file  for	read  (default)  or  write  as
	    specified by mode, and  returns  a	descriptor  for  other
	    calls. Offset is used as the virtual  base	of  the  image
	    within a larger file; normally it should be  specified  as
	    (off_t)0.

    FD _open_lff_fd(FD fd, int mode, off_t offset)

	Like _open_lff, but sets up for reading an already  open  file
	    descriptor; this descriptor MUST  have  been  opened  with
	    the right mode. The 'mode' argument is currently ignored.

    int _read_lff(FD desc; char *ptr, int size)

	Reads size bytes into ptr. Returns -1 on I/O error.

    int _write_lff(FD desc; char *ptr, int size)

	Writes size bytes from ptr. Returns -1 on I/O error.

    off_t _seek_lff(FD desc; off_t offset; int whence)

	Seeks to the specified absolute or relative position  (exactly
	    like fseek or lseek). Returns  the	new  position  in  the
	    file. This position is relative to the virtual image  base
	    specified when the image was opened.

    off_t _tell_lff(FD desc)

	Returns the current position in  the  file,  relative  to  the
	    virtual image base.

    void _close_lff(FD desc)

	Closes the file and frees any resources used  by  it.  If  the
	    file was opened by	fd  instead  of  name,	resources  are
	    released but the file is not closed.
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#ifdef sys5
#include <fcntl.h>
#endif
#include "pixlib.h"
#include "lffbufio.h"

/* Globally accessible symbols */
FD _open_lff(), _open_lff_fd();
int _read_lff(), _write_lff();
off_t _seek_lff();
off_t _tell_lff();
void _close_lff();

static int Opened_by_fd;    /* Was the file opened by descriptor or name? */
static off_t Offset;	    /* What is the virtual image offset? */

FD _open_lff(name, mode, offset)
char *name;
int mode;
off_t offset;
{
    FD fd;

    Opened_by_fd = FALSE;
    Offset = offset;
    fd = open(name, mode == LFF_READ ? O_RDONLY : O_WRONLY|O_CREAT, 0644);
    if (fd == BAD_FD)
	return BAD_FD;
    /* Seek to virtual image base */
    (void)lseek(fd, Offset, L_SET);
    return fd;
}

FD _open_lff_fd(fd, mode, offset)
FD fd;
int mode;
off_t offset;
{
    Opened_by_fd = TRUE;
    Offset = offset;
    /* Seek to virtual image base */
    (void)lseek(fd, Offset, L_SET);
    return fd;
}

int _read_lff(desc, ptr, size)
FD desc;
char *ptr;
int size;
{
    return read(desc, ptr, size) == size ? 0 : -1;
}

int _write_lff(desc, ptr, size)
FD desc;
char *ptr;
int size;
{
    return write(desc, ptr, size) == size ? 0 : -1;
}

off_t _seek_lff(desc, offset, whence)
FD desc;
off_t offset;
int whence;
{
    off_t pos = lseek(desc, offset + Offset, whence);

    /* Adjust for virtual image offset */
    if (pos != -1)
	return pos - Offset;
    else
	return pos;
}

off_t _tell_lff(desc)
FD desc;
{
    off_t pos = lseek(desc, 0L, L_INCR);

    /* Adjust for virtual image offset */
    if (pos != -1)
	return pos - Offset;
    else
	return pos;
}

void _close_lff(desc)
FD desc;
{
    if (!Opened_by_fd)
	close(desc);
}
