
/*  @(#)genps.c 1.2 91/12/30
 *
 *  PBM/PGM/PPM to PostScript.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

/*  Based on the pnmtops.c program, which is part of the PBMPLUS distribution.
 *
 *
 *  Copyright (C) 1989 by Jef Poskanzer.
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation.  This software is provided "as is" without express or
 *  implied warranty.
 */

#include <sys/types.h>
#include <ctype.h>
#include <time.h>

#ifdef      BSD
#include <sys/timeb.h>
#endif   /* BSD */

#include "popi.h"
#include "expr.h"
#include "libpbm.h"

#define  MARGIN  0.95

static FILE *ostr ;

static int bitspersample, item, bitsperitem, bitshift, itemsperline, items ;
static int rleitem, rlebitsperitem, rlebitshift ;
static int repeat, itembuf[128], count, repeatitem, repeatcount ;

static void putinit      P((char *, int, int, int, int, double,
                                    int, int, int, int, int, int, int)) ;
static void putitem      P(()) ;
static void putxelval    P((int)) ;
static void putrest      P(()) ;
static void rleputbuffer P(()) ;
static void rleputitem   P(()) ;
static void rleputxelval P((int)) ;
static void rleflush     P(()) ;
static void rleputrest   P(()) ;


void
genps(t)
Tree *t ;
{
  struct SRC *img ;
  FILE *ifp ;
  pixel *xelrow ;
  register pixel *xP ;
  int turnflag, turnokflag, rleflag ;
  int rows, cols, format, bps, padright, row, col ;
  pixval maxval, nmaxval ;
  double scale ;
  int dpi, pagewid, pagehgt ;
  char name[100], tmpname[MAXPATHLEN] ;

/* XXX: Shouldn't really have to write out to a temporary file first. */

  if (t->kids[1]) img = &Images[((Tree *) (t->kids[1]))->i] ;
  else            img = CurOld ;
  if ((ostr = EfopenW(lastfname)) == NULL) return ;
  SPRINTF(tmpname, "/tmp/.popipsXXXXXX") ;
  MKTEMP(tmpname) ;
  putpix(img, tmpname) ;

  scale      = 1.0 ;
  turnflag   = 0 ;
  turnokflag = 1 ;
  rleflag    = 0 ;

/* LaserWriter defaults. */

  dpi     = 300 ;
  pagewid = 612 ;
  pagehgt = 762 ;

  if ((ifp = fopen(tmpname, "r")) == NULL)
    {
      pm_message("couldn't open temporary file %s", tmpname,
                 (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
      return ;
    }
  STRCPY(name, img->str) ;

  ppm_readppminit(ifp, &cols, &rows, &maxval, &format) ;
  xelrow = (pixel *) pm_allocrow(cols, sizeof(pixel)) ;

/* Figure out bps. */

  bps = pm_maxvaltobits((int) maxval) ;
       if (bps > 2 && bps < 4) bps = 4 ;
  else if (bps > 4 && bps < 8) bps = 8 ;
  else if (bps > 8)
    pm_error("maxval of %d is too large for PostScript", maxval,
             (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
  nmaxval = pm_bitstomaxval(bps) ;
    
/* Compute padding to round cols * bps up to the nearest multiple of 8. */

  padright = (((cols * bps + 7) / 8) * 8 - cols * bps) / bps ;

  putinit(name, cols, rows, padright, bps, scale, dpi,
          pagewid, pagehgt, format, turnflag, turnokflag, rleflag) ;
  for (row = 0; row < rows; ++row)
    {
      ppm_readppmrow(ifp, xelrow, cols, maxval, format) ;
      switch (PPM_FORMAT_TYPE(format))
        {
          case PPM_FORMAT :                                    /* Color. */
                          for (col = 0, xP = xelrow; col < cols; ++col, ++xP)
                            if (maxval != nmaxval)
                              PPM_DEPTH(*xP, *xP, maxval, nmaxval) ;
/* First red. */
                          for (col = 0, xP = xelrow; col < cols; ++col, ++xP)
                            if (rleflag) rleputxelval(PPM_GETR(*xP)) ;
                            else         putxelval(PPM_GETR(*xP)) ;
                          for (col = 0; col < padright; ++col)
                            if (rleflag) rleputxelval(0) ;
                            else         putxelval(0) ;
                          if (rleflag) rleflush() ;
/* Then green. */
                          for (col = 0, xP = xelrow; col < cols; ++col, ++xP)
                            if (rleflag) rleputxelval(PPM_GETG(*xP)) ;
                            else         putxelval(PPM_GETG(*xP)) ;
                          for (col = 0; col < padright; ++col)
                            if (rleflag) rleputxelval(0) ;
                            else         putxelval(0) ;
                          if (rleflag) rleflush() ;
/* And blue. */
                          for (col = 0, xP = xelrow; col < cols; ++col, ++xP)
                            if (rleflag) rleputxelval(PPM_GETB(*xP)) ;
                            else         putxelval(PPM_GETB(*xP)) ;
                          for (col = 0; col < padright; ++col)
                            if (rleflag) rleputxelval(0) ;
                            else         putxelval(0) ;
                          if (rleflag) rleflush() ;
                          break ;
          default       :                                     /* Grayscale. */
                          for (col = 0, xP = xelrow; col < cols; ++col, ++xP)
                            {
                              if (maxval != nmaxval)
                                PPM_ASSIGN(*xP, 0, 0, (int) PPM_GETB(*xP) *
                                            nmaxval / maxval) ;
                              if (rleflag) rleputxelval(PPM_GETB(*xP)) ;
                              else         putxelval(PPM_GETB(*xP)) ;
                            }
                          for (col = 0; col < padright; ++col)
                            if (rleflag) rleputxelval(0) ;
                            else         putxelval(0) ;
                          if (rleflag) rleflush() ;
                          break ;
        }
    }

  if (rleflag) rleputrest() ;
  else         putrest() ;
  Efclose(ostr) ;
  FCLOSE(ifp) ;
  UNLINK(tmpname) ;
}


static void
putinit(name, cols, rows, padright, bps, scale, dpi, pagewid, pagehgt, format,
        turnflag, turnokflag, rleflag)
char *name ;
int cols, rows, padright, bps ;
double scale ;
int dpi, pagewid, pagehgt, format, turnflag, turnokflag, rleflag ;
{
  time_t time P((time_t *)) ;
  time_t clock ;
  int icols, irows, devpix ;
  double pixfac, scols, srows, llx, lly ;
#if       unix
  char *getlogin P((void)) ;
#endif /* unix */

/* Turn? */

  icols = cols ;
  irows = rows ;
  if (turnflag || (turnokflag && cols > rows))
    {
      turnflag = 1 ;
      cols     = irows ;
      rows     = icols ;
    }

/* Figure out size. */

  devpix = dpi / 72.0 + 0.5 ;       /* Device pixels per unit, approx. */
  pixfac = 72.0 / dpi * devpix ;    /* 1, approx. */
  scols  = scale * cols * pixfac ;
  srows  = scale * rows * pixfac ;
  if (scols > pagewid * MARGIN || srows > pagehgt * MARGIN)
    {
      if (scols > pagewid * MARGIN)
        {
          scale *= pagewid / scols * MARGIN ;
          scols = scale * cols * pixfac ;
          srows = scale * rows * pixfac ;
        }
      if (srows > pagehgt * MARGIN)
        {
          scale *= pagehgt / srows * MARGIN ;
          scols = scale * cols * pixfac ;
          srows = scale * rows * pixfac ;
        }
      pm_message("warning, image too large for page, rescaling to %g", scale,
                 (char *) 0, (char *) 0, (char *) 0, (char *) 0) ;
    }
  llx = (pagewid - scols) / 2 ;
  lly = (pagehgt - srows) / 2 ;

  clock = time((time_t *) 0) ;

  FPRINTF(ostr, "%%!PS-Adobe-2.0 EPSF-2.0\n") ;
  FPRINTF(ostr, "%%%%Creator: popi\n") ;
  FPRINTF(ostr, "%%%%CreationDate: %s", ctime(&clock)) ;    /* Includes \n */
  FPRINTF(ostr, "%%%%Title: %s.ps\n", name) ;
  FPRINTF(ostr, "%%%%Pages: 1\n") ;
  FPRINTF(ostr, "%%%%BoundingBox: %d %d %d %d\n", (int) llx, (int) lly,
         (int) (llx + scols + 0.5), (int) (lly + srows + 0.5)) ;

#if        unix
  FPRINTF(ostr, "%%%%For: %s\n", getlogin()) ;
#endif  /* unix */

  FPRINTF(ostr, "%%%%EndComments\n") ;
  if (rleflag)
    {
      FPRINTF(ostr, "/rlestr1 1 string def\n") ;
      FPRINTF(ostr, "/readrlestring {\n") ;                      /* s -- nr */
      FPRINTF(ostr, "  /rlestr exch def\n") ;                         /* - */
      FPRINTF(ostr, "  currentfile rlestr1 readhexstring pop\n") ;    /* s1 */
      FPRINTF(ostr, "  0 get\n") ;                                    /* c */
      FPRINTF(ostr, "  dup 127 le {\n") ;                      /* c */
      FPRINTF(ostr, "    currentfile rlestr 0\n") ;            /* c f s 0 */
      FPRINTF(ostr, "    4 3 roll\n") ;                        /* f s 0 c */
      FPRINTF(ostr, "    1 add  getinterval\n") ;              /* f s */
      FPRINTF(ostr, "    readhexstring pop\n") ;               /* s */
      FPRINTF(ostr, "    length\n") ;                          /* nr */
      FPRINTF(ostr, "  } {\n") ;                               /* c */
      FPRINTF(ostr, "    256 exch sub dup\n") ;                       /* n n */
      FPRINTF(ostr, "    currentfile rlestr1 readhexstring pop\n") ;  /* n n s1*/
      FPRINTF(ostr, "    0 get\n") ;                           /* n n c */
      FPRINTF(ostr, "    exch 0 exch 1 exch 1 sub {\n") ;      /* n c 0 1 n-1*/
      FPRINTF(ostr, "      rlestr exch 2 index put\n") ;
      FPRINTF(ostr, "    } for\n") ;                           /* n c */
      FPRINTF(ostr, "    pop\n") ;                             /* nr */
      FPRINTF(ostr, "  } ifelse\n") ;                          /* nr */
      FPRINTF(ostr, "} bind def\n") ;
      FPRINTF(ostr, "/readstring {\n") ;                       /* s -- s */
      FPRINTF(ostr, "  dup length 0 {\n") ;                    /* s l 0 */
      FPRINTF(ostr, "    3 copy exch\n") ;                     /* s l n s n l*/
      FPRINTF(ostr, "    1 index sub\n") ;                     /* s l n s n r*/
      FPRINTF(ostr, "    getinterval\n") ;                     /* s l n ss */
      FPRINTF(ostr, "    readrlestring\n") ;                   /* s l n nr */
      FPRINTF(ostr, "    add\n") ;                             /* s l n */
      FPRINTF(ostr, "    2 copy le { exit } if\n") ;           /* s l n */
      FPRINTF(ostr, "  } loop\n") ;                            /* s l l */
      FPRINTF(ostr, "  pop pop\n") ;                           /* s */
      FPRINTF(ostr, "} bind def\n") ;
    }
  else
    {
      FPRINTF(ostr, "/readstring {\n") ;                       /* s -- s */
      FPRINTF(ostr, "  currentfile exch readhexstring pop\n") ;
      FPRINTF(ostr, "} bind def\n") ;
    }
  if (PPM_FORMAT_TYPE(format) == PPM_FORMAT)
    {
      FPRINTF(ostr, "/rpicstr %d string def\n", (icols + padright) * bps / 8) ;
      FPRINTF(ostr, "/gpicstr %d string def\n", (icols + padright) * bps / 8) ;
      FPRINTF(ostr, "/bpicstr %d string def\n", (icols + padright) * bps / 8) ;
    }
  else FPRINTF(ostr, "/picstr %d string def\n", (icols + padright) * bps / 8) ;

  FPRINTF(ostr, "%%%%EndProlog\n") ;
  FPRINTF(ostr, "%%%%Page: 1 1\n") ;
  FPRINTF(ostr, "gsave\n") ;
  FPRINTF(ostr, "%g %g translate\n", llx, lly) ;
  FPRINTF(ostr, "%g %g scale\n", scols, srows) ;
  if (turnflag)
    FPRINTF(ostr, "0.5 0.5 translate  90 rotate  -0.5 -0.5 translate\n") ;

  FPRINTF(ostr, "%d %d %d\n", icols, irows, bps) ;
  FPRINTF(ostr, "[ %d 0 0 -%d 0 %d ]\n", icols, irows, irows) ;
  if (PPM_FORMAT_TYPE(format) == PPM_FORMAT)
    {
      FPRINTF(ostr, "{ rpicstr readstring }\n") ;
      FPRINTF(ostr, "{ gpicstr readstring }\n") ;
      FPRINTF(ostr, "{ bpicstr readstring }\n") ;
      FPRINTF(ostr, "true 3\n") ;
      FPRINTF(ostr, "colorimage\n") ;
      PRINTF("Writing color PostScript...\n") ;
    }
  else
    {
      FPRINTF(ostr, "{ picstr readstring }\n") ;
      FPRINTF(ostr, "image\n") ;
    }

  bitspersample = bps ;
  itemsperline = items = 0 ;
  if (rleflag)
    {
      rleitem        = 0 ;
      rlebitsperitem = 0 ;
      rlebitshift    = 8 - bitspersample ;
      repeat         = 1 ;
      count          = 0 ;
    }
  else
    {
      item        = 0 ;
      bitsperitem = 0 ;
      bitshift    = 8 - bitspersample ;
    }
}


static void
putitem()
{
  char *hexits = "0123456789abcdef" ;

  if (itemsperline == 30)
    {
      FPUTC('\n', ostr) ;
      itemsperline = 0 ;
    }
  FPUTC(hexits[item >> 4], ostr) ;
  FPUTC(hexits[item & 15], ostr) ;
  ++itemsperline ;
  ++items ;
  item        = 0 ;
  bitsperitem = 0 ;
  bitshift    = 8 - bitspersample ;
}


static void
putxelval(xv)
pixval xv ;
{
  if (bitsperitem == 8) putitem() ;
  item        += xv << bitshift ;
  bitsperitem += bitspersample ;
  bitshift    -= bitspersample ;
}


static void
putrest()
{
  if (bitsperitem > 0) putitem() ;
  FPRINTF(ostr, "\n") ;
  FPRINTF(ostr, "grestore\n") ;
  FPRINTF(ostr, "showpage\n") ;
  FPRINTF(ostr, "%%%%Trailer\n") ;
}


static void
rleputbuffer()
{
  int i ;

  if (repeat)
    {
      item = 256 - count ;
      putitem() ;
      item = repeatitem ;
      putitem() ;
    }
  else
    {
      item = count - 1 ;
      putitem() ;
      for (i = 0; i < count; ++i)
        {
          item = itembuf[i] ;
          putitem() ;
        }
    }
  repeat = 1 ;
  count  = 0 ;
}


static void
rleputitem()
{
  int i ;

  if (count == 128) rleputbuffer() ;

  if (repeat && count == 0)
    {                                  /* Still initializing a repeat buf. */
      itembuf[count] = repeatitem = rleitem ;
      ++count ;
    }
  else if (repeat)
    {                                 /* Repeating - watch for end of run. */
      if (rleitem == repeatitem)
        {                                                /* Run continues. */
          itembuf[count] = rleitem ;
          ++count ;
        }
      else
        {                        /* Run ended - is it long enough to dump? */
          if (count > 2)
            {       /* Yes, dump a repeat-mode buffer and start a new one. */
              rleputbuffer() ;
              itembuf[count] = repeatitem = rleitem ;
              ++count ;
            }
          else
            {             /* Not long enough - convert to non-repeat mode. */
              repeat         = 0 ;
              itembuf[count] = repeatitem = rleitem ;
              ++count ;
              repeatcount    = 1 ;
            }
        }
    }
  else
    {                  /* Not repeating - watch for a run worth repeating. */
      if (rleitem == repeatitem)
        {                                       /* Possible run continues. */
          ++repeatcount ;
          if (repeatcount > 3)
            {      /* Long enough - dump non-repeat part and start repeat. */
              count = count - (repeatcount - 1) ;
              rleputbuffer() ;
              count = repeatcount ;
              for (i = 0; i < count; ++i)
                itembuf[i] = rleitem ;
            }
          else
            {         /* Not long enough yet - continue as non-repeat buf. */
              itembuf[count] = rleitem ;
              ++count ;
            }
        }
      else
        {                                                   /* Broken run. */
          itembuf[count] = repeatitem = rleitem ;
          ++count ;
          repeatcount = 1 ;
        }
    }

  rleitem        = 0 ;
  rlebitsperitem = 0 ;
  rlebitshift    = 8 - bitspersample ;
}


static void
rleputxelval(xv)
pixval xv ;
{
  if (rlebitsperitem == 8) rleputitem() ;
  rleitem        += xv << rlebitshift ;
  rlebitsperitem += bitspersample ;
  rlebitshift    -= bitspersample ;
}


static void
rleflush()
{
  if (rlebitsperitem > 0) rleputitem() ;
  if (count > 0) rleputbuffer() ;
}


static void
rleputrest()
{
  rleflush() ;
  FPRINTF(ostr, "\n") ;
  FPRINTF(ostr, "grestore\n") ;
  FPRINTF(ostr, "showpage\n") ;
  FPRINTF(ostr, "%%%%Trailer\n") ;
}
