
/*
        encoder.c - GIF Encoder and associated routines

*/

#include "extra.h"
#include "showpix.h"

/*
                        Save-To-Disk Routines (GIF)

GIF and 'Graphics Interchange Format' are trademarks (tm) of Compuserve
Incorporated, an H&R Block Company.


The following routines perform the GIF encoding when the 's' key is pressed.
The routines refer to several variables that are declared elsewhere
[colors, xdots, ydots, and 'dacbox'], and rely on external routines to
actually read and write screen pixels [getcolor(x,y) and putcolor(x,y,color)].
(Writing pixels is just stuffed in here as a sort of visual status report,
and has nothing to do with any GIF function.)   They also rely on the
existence of an externally-defined 64K dataspace and they use the routines
'toextra()' and 'cmpextra()' to deal with that dataspace (in the same manner
as 'memcpy()' and 'memcmp()' would).   Otherwise, they perform a generic
GIF-encoder function.

Note that these routines use small string- and hash-tables, and "flush"
the GIF entries whenever the hash-table gets two-thirds full or the string
table gets full.   They also use the GIF encoding technique of limiting the
encoded string length to a specific size, "adding" a string to the hash table
at that point even if a matching string exists ("adding" is in quotes, because
if a matching string exists we can increment the code counter but safely throw
the duplicate string away, saving both string space and a hash table entry).

   This results in relatively good speed and small data space, but at the
expense of compression efficiency (filesize).   These trade-offs could be
adjusted by modifying the #DEFINEd variables below.

Note that the 'strlocn' and 'teststring' routines are declared
to be external just so that they can be defined (and the space re-used)
elsewhere.  The actual declarations are in the assembler code.

*/

#define MAXTEST   100           /* maximum single string length */
#define MAXSTRING 64000         /* total space reserved for strings */
                                /* maximum number of strings available */
#define MAXENTRY  5003          /* (a prime number is best for hashing) */

extern unsigned int strlocn[MAXENTRY];
extern unsigned char teststring[MAXTEST];
unsigned char block[266];   /* GIF-encoded blocks go here */

static FILE *out;

static int lentest, lastentry, numentries, numrealentries;
static unsigned int nextentry;
static int clearcode, endcode;
static unsigned int hashcode;

static unsigned char blockcount;
static int startbits, codebits, bytecount, bitcount;


int savegif(char *filename)                    /* save-to-disk routine */
{
char openfile[80], openfiletype[10];
int i, j, ydot, xdot, color, outcolor1, outcolor2;
unsigned int hashentry;
unsigned char bitsperpixel, x;
int entrynum;

if (extraseg == 0)                      /* not enough memory for this */
        return -1;

if ((out=fopen(filename,"wb")) == NULL) {
        return 0;
        }

bitsperpixel = 0;                       /* calculate bits / pixel */
for (i = colors; i >= 2; i /= 2 )
        bitsperpixel++;

startbits = bitsperpixel+1;             /* start coding with this many bits */
if (colors == 2)
        startbits++;                    /* B&W Klooge */

clearcode = 1 << (startbits - 1);       /* set clear and end codes */
endcode = clearcode+1;

outcolor1 = 2;                          /* use these colors to show progress */
outcolor2 = 3;                          /* (this has nothing to do with GIF) */

fwrite("GIF87a",1,6,out);               /* GIF Signature */

fwrite(&xdots,2,1,out);                 /* screen descriptor */
fwrite(&ydots,2,1,out);
x = 128 + ((6-1)<<4) + (bitsperpixel-1); /* color resolution == 6 bits worth */
fwrite(&x,1,1,out);
i = 0;
fwrite(&i,1,1,out);
fwrite(&i,1,1,out);

shftwrite(dacbox,colors);

fwrite(",",1,1,out);                    /* Image Descriptor */
i = 0;
fwrite(&i,2,1,out);
fwrite(&i,2,1,out);
fwrite(&xdots,2,1,out);
fwrite(&ydots,2,1,out);
i = 0;
fwrite(&i,1,1,out);

bitsperpixel = startbits - 1;           /* raster data starts here */
fwrite(&bitsperpixel,1,1,out);

codebits = startbits;                   /* start encoding */

raster(9999);                           /* initialize the raster routine */

inittable();                            /* initialize the LZW tables */

for (ydot = 0; ydot < ydots; ydot++) {  /* scan through the dots */
        for (xdot = 0; xdot < xdots; xdot++) {
                color = getcolor(xdot,ydot);    /* get the next dot */
                teststring[0] = ++lentest;
                teststring[lentest] = color;
                if (lentest == 1) {             /* root entry? */
                        lastentry = color;
                        continue;
                        }
                if (lentest == 2)               /* init   the hash code */
                        hashcode = 301 * (teststring[1]+1);
                hashcode *= (color + lentest);  /* update the hash code */
                hashentry = ++hashcode % MAXENTRY;
                for( i = 0; i < MAXENTRY; i++) {
                        if (++hashentry >= MAXENTRY) hashentry = 0;
                        if (cmpextra(strlocn[hashentry]+2,
                                teststring,lentest+1) == 0)
                                        break;
                        if (strlocn[hashentry] == 0) i = MAXENTRY;
                        }
                /* found an entry and string length isn't too bad */
                if (strlocn[hashentry] != 0 && lentest < MAXTEST-3) {
                        fromextra(strlocn[hashentry],&entrynum,2);
                        lastentry = entrynum;
                        continue;
                        }
                raster(lastentry);                      /* write entry */
                numentries++;           /* act like you added one, anyway */
                if (strlocn[hashentry] == 0) {  /* add new string, if any */
                        entrynum = numentries+endcode;
                        strlocn[hashentry] = nextentry;
                        toextra(nextentry, &entrynum,2);
                        toextra(nextentry+2,
                                teststring,lentest+1);
                        nextentry += lentest+3;
                        numrealentries++;
                        }
                teststring[0] = 1;              /* reset current entry */
                teststring[1] = color;
                lentest = 1;
                lastentry = color;

                if ((numentries+endcode) == (1<<codebits))
                        codebits++;              /* use longer encoding */

                if ( numentries + endcode > 4093 ||     /* out of room? */
                        numrealentries > (MAXENTRY*2)/3 ||
                        nextentry > MAXSTRING-MAXTEST-5) {
                        raster(lastentry);              /* flush & restart */
                        inittable();
                        }
                }

                if ((ydot & 4) == 0) {
                        if (++outcolor1 >= colors) outcolor1 = 0;
                        if (++outcolor2 >= colors) outcolor2 = 0;
                        }
                for (i = 0; 250*i < xdots; i++) {       /* display vert status bars */
                        putcolor(      i,ydot,outcolor1);       /*   (this is NOT   */
                        putcolor(xdots-1-i,ydot,outcolor2);     /*    GIF-related)  */
                        }
        if (kbhit())                            /* keyboard hit - bail out */
                ydot = 9999;
        }

raster(lastentry);                      /* tidy up - dump the last code */

raster(endcode);                        /* finish the map */

i = 0;                                  /* raster data ends here */
fwrite(&i,1,1,out);

fwrite(";",1,1,out);                    /* GIF Terminator */

fclose(out);
if (ydot < 9999)                        /* signal normal or interrupted end */
        return 1;
else
        return 0;
}

shftwrite(color,numcolors)              /* shift IBM colors to GIF format */
unsigned char color[];
int numcolors;
{
unsigned char thiscolor;
int i,j;
for (i = 0; i < numcolors; i++)
        for (j = 0; j < 3; j++) {
                thiscolor = color[3*i+j];
                thiscolor = thiscolor << 2;
                thiscolor += (thiscolor >> 6);
                fwrite(&thiscolor,1,1,out);
                }
}

inittable()                             /* routine to init tables */
{
int i;

raster(clearcode);                      /* signal that table is initialized */

numentries = 0;                         /* initialize the table */
numrealentries = 0;
nextentry = 1;
lentest = 0;
codebits = startbits;

toextra(0,"\0",1);                      /* clear the hash entries */
for (i = 0; i < MAXENTRY; i++)
        strlocn[i] = 0;

}

raster(code)                            /* routine to block and output codes */
unsigned int code;
{
unsigned int icode, i, j;

if (code == 9999) {                     /* special start-up signal */
        bytecount = 0;
        bitcount = 0;
        for (i = 0; i < 266; i++)
                block[i] = 0;
        return(0);
        }

icode = code << bitcount;               /* update the bit string */
block[bytecount  ] |= (icode & 255);
block[bytecount+1] |= ((icode>>8) & 255);
icode = (code>>8) << bitcount;
block[bytecount+2] |= ((icode>>8) & 255);
bitcount += codebits;
while (bitcount >= 8) {                 /* locate next starting point */
        bitcount -= 8;
        bytecount++;
        }

if (bytecount > 250 || code == endcode) {       /* time to write a block */
        if (code == endcode)
                while (bitcount > 0) {          /* if EOF, find the real end */
                        bitcount -= 8;
                        bytecount++;
                        }
        i = bytecount;
        blockcount = i;
        fwrite(&blockcount,1,1,out);            /* write the block */
        fwrite(block,i,1,out);
        bytecount = 0;                          /* now re-start the block */
        for (j = 0; j < 5; j++)                 /* (may have leftover bits) */
                block[j] = block[j+i];
        for (j = 5; j < 266; j++)
                block[j] = 0;
        }
}
