/* glx.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: glx.c,v 1.23 1995/06/12 15:29:06 brianp Exp $

$Log: glx.c,v $
 * Revision 1.23  1995/06/12  15:29:06  brianp
 * search for CI visuals from shallowest to deepest per GLX_BUFFER_SIZE
 *
 * Revision 1.22  1995/06/09  21:48:22  brianp
 * changed version string to 1.2.1
 *
 * Revision 1.21  1995/05/24  13:00:15  brianp
 * updated version query functions to return 1.2
 *
 * Revision 1.20  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.19  1995/05/22  16:44:48  brianp
 * added over/underlay error checking
 *
 * Revision 1.18  1995/05/19  14:22:18  brianp
 * added MESA_BACK_BUFFER environment variable
 *
 * Revision 1.17  1995/05/16  19:19:46  brianp
 * added casts to allow compiling with OpenGL's glx.h header
 * implemented GLX_color_SIZE attributes in glXGetConfig()
 *
 * Revision 1.16  1995/05/15  15:48:21  brianp
 * added share_list support to glXCreateContext
 *
 * Revision 1.15  1995/05/10  19:01:29  brianp
 * Better glXGetConfig() support from Armin Liebchen
 *
 * Revision 1.14  1995/04/17  14:40:07  brianp
 * Changed XMesaAttachTo... to XMesaBind...
 *
 * Revision 1.13  1995/04/17  14:31:26  brianp
 * GLXPixmaps implemented
 * uses new XMesaCreateContext() API
 *
 * Revision 1.12  1995/04/13  19:49:12  brianp
 * added SGI's multi-sample extension for GLX 1.1
 *
 * Revision 1.11  1995/04/11  13:40:35  brianp
 * better GLX visual handling
 * introduced GLX 1.1 functions
 *
 * Revision 1.10  1995/03/30  21:09:44  brianp
 * added 8-bit TrueColor test
 *
 * Revision 1.9  1995/03/27  20:33:12  brianp
 * added MESA_RGB_VISUAL, MESA_CI_VISUAL environment variable support
 *
 * Revision 1.8  1995/03/24  15:16:52  brianp
 * replaced ACCUM_BITS with ACC_TYPE
 *
 * Revision 1.7  1995/03/13  15:58:04  brianp
 * removed glXUseXFont stub
 *
 * Revision 1.6  1995/03/08  18:51:21  brianp
 * check if ctx is NULL in glXMakeCurrent per Thorsten Olh
 *
 * Revision 1.5  1995/03/07  19:10:19  brianp
 * look at color/index depth arguments when selecting the visual
 *
 * Revision 1.4  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.3  1995/03/04  19:18:17  brianp
 * new visual selection code
 *
 * Revision 1.2  1995/03/01  17:05:00  brianp
 * added error check to glXSwapBuffers
 *
 * Revision 1.1  1995/02/28  21:23:25  brianp
 * Initial revision
 *
 */


/*
 * A pseudo-GLX implementation to allow OpenGL/GLX programs to work with
 * Mesa.  Initial version contributed by Philip Brown
 * (philb@CSUA.Berkeley.EDU).  Thanks Philip!
 *
 * Better glXGetConfig() support contributed by Armin Liebchen
 * (liebchen@asylum.cs.utah.edu).  Thanks Armin!
 */



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "GL/gl.h"
#include "GL/glx.h"
#include "GL/xmesa.h"
#include "xmesaP.h"
#include "config.h"
#include "context.h"
#include "dd.h"



/*
 * Since we don't have a real GLX extension, the XVisualInfo doesn't
 * carry all the info we need such as double buffer mode, depth buffer, etc.
 * We use a table to associate this extra information with each XVisualInfo
 * structure we return from glXChooseVisual.
 */
struct glx_visual {
	XVisualInfo *visinfo;		/* Returned from glXChooseVisual */
	GLboolean rgba_buffer_flag;	/* RGBA mode? */
	GLboolean double_buffer_flag;	/* double buffered? */
	GLboolean depth_buffer_flag;	/* depth buffer? */
};

#define MAX_VISUALS 100

static struct glx_visual VisualTable[MAX_VISUALS];
static int NumVisuals = 0;



/*
 * We also need to keep a list of Pixmaps created with glXCreateGLXPixmap()
 * so when glXMakeCurrent() is called we know if the drawable is a pixmap
 * or a window.
 */
#define MAX_PIXMAPS 10
static Pixmap PixmapList[MAX_PIXMAPS];
static int NumPixmaps = 0;






/*
 * Save the GLX configuration associated with an XVisualInfo.
 */
static void save_glx_visual( XVisualInfo *vinfo, GLboolean rgba,
			     GLboolean dbl, GLboolean depth )
{
   int i;

   for (i=0;i<NumVisuals;i++) {
      if (VisualTable[i].visinfo==vinfo) {
	 VisualTable[i].rgba_buffer_flag = rgba;
	 VisualTable[i].double_buffer_flag = dbl;
	 VisualTable[i].depth_buffer_flag = depth;
	 return;
      }
   }
   if (NumVisuals+1<MAX_VISUALS) {
      VisualTable[NumVisuals].visinfo = vinfo;
      VisualTable[NumVisuals].rgba_buffer_flag = rgba;
      VisualTable[NumVisuals].double_buffer_flag = dbl;
      VisualTable[NumVisuals].depth_buffer_flag = depth;
      NumVisuals++;
   }
   else {
      fprintf( stderr, "GLX Error: maximum number of visuals exceeded\n");
   }
}



/*
 * Scan the list of visuals for this display and put all "Mesa-usable"
 * ones into the VisualTable[].
 */
static void load_all_glx_visuals( Display* dpy )
{
    int 	i;
    XVisualInfo		vis_templ;
    XVisualInfo*	vis_list;
    int			nvisuals;	/* number of visuals */

    memset( &vis_templ, 0, sizeof(XVisualInfo) );  /* init to zero */
    vis_templ.screen = DefaultScreen( dpy );       /* just match the screen */

    /* get list of XVisualInfos which match the template */
    vis_list = XGetVisualInfo( dpy, VisualScreenMask, &vis_templ,
			       &nvisuals );

    /* save those visuals which XMesa supports */
    for ( i=0; i < nvisuals; i++ ) {
	switch (vis_list[i].class) {
	   case StaticGray:
	      if (vis_list[i].depth==1) {
		 save_glx_visual( &vis_list[i], GL_TRUE, GL_TRUE, GL_TRUE );
	      }
	      break;
	   case PseudoColor:
	      if (vis_list[i].depth>=8) {
		 save_glx_visual( &vis_list[i], GL_TRUE, GL_TRUE, GL_TRUE );
	      }
	      else {
		 save_glx_visual( &vis_list[i], GL_FALSE, GL_TRUE, GL_TRUE );
	      }
	      break;
	   case TrueColor:
	   case DirectColor:
	      save_glx_visual( &vis_list[i], GL_TRUE, GL_TRUE, GL_TRUE );
	      break;
	   case GrayScale:
	   case StaticColor:
	      /* not supported */
	      break;
	}
    }

    /* don't XFree() the vis_list! */
}




/*
 * Find the GLX visual associated with an XVisualInfo.
 */
static GLboolean find_glx_visual( XVisualInfo *vinfo, GLboolean *rgba,
				  GLboolean *dbl, GLboolean *depth )
{
   int i;

   for (i=0;i<NumVisuals;i++) {
      if (   VisualTable[i].visinfo->visualid == vinfo->visualid
	  && VisualTable[i].visinfo->class    == vinfo->class
	  && VisualTable[i].visinfo->depth    == vinfo->depth ) {
	 *rgba  = VisualTable[i].rgba_buffer_flag;
	 *dbl   = VisualTable[i].double_buffer_flag;
	 *depth = VisualTable[i].depth_buffer_flag;
	 return GL_TRUE;
      }
   }
   return GL_FALSE;
}




/*
 * Try to get an X visual which matches the given arguments.
 */
static XVisualInfo *get_visual( Display *dpy, int scr,
			        unsigned int depth, int class )
{
   XVisualInfo template;
   long mask;
   int n;

   mask = VisualScreenMask | VisualDepthMask | VisualClassMask;
   template.screen = scr;
   template.depth = depth;
   template.class = class;

   return XGetVisualInfo( dpy, mask, &template, &n );
}



/*
 * Retrieve the value of the given environment variable and find
 * the X visual which matches it.
 * Input:  dpy - the display
 *         screen - the screen number
 *         varname - the name of the environment variable
 * Return:  an XVisualInfo pointer to NULL if error.
 */
static XVisualInfo *get_env_visual( Display *dpy, int scr, char *varname )
{
   char *value;
   char type[100];
   int depth, class = -1;
   XVisualInfo *vis;

   value = getenv( varname );
   if (!value) {
      return NULL;
   }

   sscanf( value, "%s %d", type, &depth );

   if (strcmp(type,"TrueColor")==0)          class = TrueColor;
   else if (strcmp(type,"DirectColor")==0)   class = DirectColor;
   else if (strcmp(type,"PseudoColor")==0)   class = PseudoColor;
   else if (strcmp(type,"StaticColor")==0)   class = StaticColor;
   else if (strcmp(type,"GrayScale")==0)     class = GrayScale;
   else if (strcmp(type,"StaticGray")==0)    class = StaticGray;

   if (class>-1 && depth>0) {
      vis = get_visual( dpy, scr, depth, class );
      if (vis) {
	 return vis;
      }
   }

   fprintf( stderr, "Mesa: GLX unable to find visual class=%s, depth=%d.\n",
	    type, depth );
   return NULL;
}



/*
 * Select an X visual which satisfies the RGBA/CI flag and minimum depth.
 * Input:  dpy, screen - X display and screen number
 *         rgba - GL_TRUE = RGBA mode, GL_FALSE = CI mode
 *         min_depth - minimum visual depth
 * Return:  pointer to an XVisualInfo or null.
 */
static XVisualInfo *choose_x_visual( Display *dpy, int screen,
				     GLboolean rgba, int min_depth )
{
   XVisualInfo *vis;

   if (rgba) {
      /* First see if the MESA_RGB_VISUAL env var is defined */
      if (vis = get_env_visual( dpy, screen, "MESA_RGB_VISUAL" ))  return vis;
      /* Look for 8, 12 or 24-bit depth */
      if (min_depth==8 || min_depth==12 || min_depth==24) {
	 if (vis = get_visual( dpy, screen, min_depth, TrueColor )) return vis;
      }
      /* Else, look for deepest TrueColor visual and work down */
      if (vis = get_visual( dpy, screen, 24, TrueColor   ))  return vis;
      if (vis = get_visual( dpy, screen, 16, TrueColor   ))  return vis;
      if (vis = get_visual( dpy, screen, 12, TrueColor   ))  return vis;
      if (vis = get_visual( dpy, screen,  8, PseudoColor ))  return vis;
      if (vis = get_visual( dpy, screen,  8, TrueColor   ))  return vis;
      if (vis = get_visual( dpy, screen,  1, StaticGray  ))  return vis;
   }
   else {
      /* First see if the MESA_CI_VISUAL env var is defined */
      if (vis = get_env_visual( dpy, screen, "MESA_CI_VISUAL" ))  return vis;
      /* Else, start with shallowest depth and go up */
      if (vis = get_visual( dpy, screen,  8, PseudoColor ))  return vis;
      if (vis = get_visual( dpy, screen, 12, PseudoColor ))  return vis;
      if (vis = get_visual( dpy, screen, 16, PseudoColor ))  return vis;
      if (vis = get_visual( dpy, screen,  1, StaticGray  ))  return vis;
   }

   /* didn't find a visual */
   return NULL;
}


/*
 * Return the number of bits set in n.
 */
static int bitcount( unsigned long n )
{
   int bits;
   for (bits=0; n>0; bits++) {
      n = n >> 1;
   }
   return bits;
}



XVisualInfo *glXChooseVisual(Display *dpy, int screen, int *list)
{
   int *parselist;
   XVisualInfo *vis;
   int min_depth = 0;
   int min_ci = 0;
   int min_red=0, min_green=0, min_blue=0, min_alpha=0;
   GLboolean rgba_flag = GL_FALSE;
   GLboolean double_flag = GL_FALSE;
   GLboolean depth_flag = GL_FALSE;

   parselist = list;

   while(*parselist) {

      switch(*parselist) {
	 case GLX_USE_GL:
	    /* ignore */
	    parselist++;
	    break;
	 case GLX_BUFFER_SIZE:
	    parselist++;
	    min_ci = *parselist++;
	    break;
	 case GLX_LEVEL:
	    /* ignore */
	    parselist++;
	    if (*parselist!=0) {
	       /* Overlays/underlays not supported */
	       return NULL;
	    }
	    parselist++;
	    break;
	 case GLX_RGBA:
	    rgba_flag = GL_TRUE;
	    parselist++;
	    break;
	 case GLX_DOUBLEBUFFER:
	    double_flag = GL_TRUE;
	    parselist++;
	    break;
	 case GLX_STEREO:
	    /* ignore */
	    parselist++;
	    break;
	 case GLX_AUX_BUFFERS:
	    /* ignore */
	    parselist++;
	    parselist++;
	    break;
	 case GLX_RED_SIZE:
	    parselist++;
	    min_red = *parselist++;
	    break;
	 case GLX_GREEN_SIZE:
	    parselist++;
	    min_green = *parselist++;
	    break;
	 case GLX_BLUE_SIZE:
	    parselist++;
	    min_blue = *parselist++;
	    break;
	 case GLX_ALPHA_SIZE:
	    /* ignore */
	    parselist++;
	    min_alpha = *parselist++;
	    break;
	 case GLX_DEPTH_SIZE:
	    depth_flag = GL_TRUE;
	    parselist++;
	    min_depth = *parselist++;
	    break;
	 case GLX_STENCIL_SIZE:
	    /* silently ignore */
	    parselist++;
	    parselist++;
	    break;
	 case GLX_ACCUM_RED_SIZE:
	 case GLX_ACCUM_GREEN_SIZE:
	 case GLX_ACCUM_BLUE_SIZE:
	 case GLX_ACCUM_ALPHA_SIZE:
	    /* silently ignore */
	    parselist++;
	    parselist++;
	    break;
	 case GLX_SAMPLES_SGIS:
	    /* silently ignored */
	    parselist++;
	    parselist++;
	    break;
	 case GLX_SAMPLE_BUFFER_SGIS:
	    /* silently ignored */
	    parselist++;
	    parselist++;
	    break;
	 case None:
	    break;
	 default:
	    puts("glx: unrecognised glXChooseVisual attribute");
	    printf("glx: %d\n", *parselist);
	    parselist++;
	    break;
      }
   }

   /*
    * Since we're only simulating the GLX extension this function will never
    * find any real GL visuals.  Instead, all we can do is try to find an RGB
    * or CI visual of appropriate depth.  Other requested attributes such as
    * double buffering, depth buffer, etc. will be associated with the X
    * visual and stored in the VisualTable[].
    */
   if (rgba_flag) {
      /* Get an RGB visual */
      int min_rgb = min_red + min_green + min_blue;
      vis = choose_x_visual( dpy, screen, rgba_flag, min_rgb );
   }
   else {
      /* Get a colormapped visual */
      vis = choose_x_visual( dpy, screen, rgba_flag, min_ci );
   }

   if (vis) {
      save_glx_visual( vis, rgba_flag, double_flag, depth_flag );
   }

   return vis;
}




GLXContext glXCreateContext( Display *dpy, XVisualInfo *visinfo,
			     GLXContext share_list, Bool direct )
{
   GLboolean rgba_flag, double_flag, depth_flag;

   if (find_glx_visual( visinfo, &rgba_flag, &double_flag, &depth_flag )) {
      XMesaContext ctx;
      GLboolean ximage_flag = GL_TRUE;
      if (double_flag) {
	 /* Check if the MESA_BACK_BUFFER env var is set */
	 char *backbuffer = getenv("MESA_BACK_BUFFER");
	 if (backbuffer) {
	    if (backbuffer[0]=='p' || backbuffer[0]=='P') {
	       ximage_flag = GL_FALSE;
	    }
	    else if (backbuffer[0]=='x' || backbuffer[0]=='X') {
	       ximage_flag = GL_TRUE;
	    }
	    else {
	       fprintf(stderr, "Mesa: invalid value for MESA_BACK_BUFFER ");
	       fprintf(stderr, "environment variable, using an XImage.\n");
	    }
	 }
      }
      ctx = XMesaCreateContext( dpy, visinfo, rgba_flag, double_flag,
			        ximage_flag, (XMesaContext) share_list );
      return (GLXContext) ctx;
   }
   else {
      fprintf(stderr,"Mesa: error in glXCreateContext: bad visual\n");
      return NULL;
   }
}



Bool glXMakeCurrent( Display *dpy, GLXDrawable drawable, GLXContext ctx )
{
   if (ctx && drawable) {
      /* determine if the drawable is a GLXPixmap */
      GLuint i;
      GLboolean pixmap_flag = GL_FALSE;
      for (i=0;i<NumPixmaps;i++) {
	 if (PixmapList[i]==drawable) {
	    pixmap_flag = GL_TRUE;
	    break;
	 }
      }
      if (pixmap_flag) {
	 XMesaBindPixmap( (XMesaContext) ctx, drawable );
      }
      else {
	 XMesaBindWindow( (XMesaContext) ctx, drawable );
      }
      XMesaMakeCurrent( (XMesaContext) ctx );
      return True;
   }
   else if (!ctx && !drawable) {
      /* release current context w/out assigning new one. */
      XMesaMakeCurrent( NULL );
      return True;
   }
   else {
      /* either ctx or drawable is NULL, this is an error */
      return False;
   }
}



GLXPixmap glXCreateGLXPixmap( Display *dpy, XVisualInfo *visual, Pixmap pixmap )
{
   if (NumPixmaps==MAX_PIXMAPS) {
      fprintf( stderr, "Mesa: glXCreateGLXPixmap: too many pixmaps\n");
      return 0;
   }
   PixmapList[NumPixmaps] = pixmap;
   NumPixmaps++;
   return pixmap;
}


void glXDestroyGLXPixmap( Display *dpy, GLXPixmap pixmap )
{
   int i, j;

   for (i=0;i<NumPixmaps;i++) {
      if (PixmapList[i]==pixmap) {
	 for (j=i+1;j<MAX_PIXMAPS;j++) {
	    PixmapList[j-1] = PixmapList[j];
	 }
	 NumPixmaps--;
	 return;
      }
   }
   fprintf( stderr, "Mesa: glXDestroyGLXPixmap: invalid pixmap\n");
}


void glXCopyContext( Display *dpy, GLXContext src, GLXContext dst,
		     GLuint mask )
{
   XMesaContext xm_src, xm_dst;
   xm_src = (XMesaContext) src;
   xm_dst = (XMesaContext) dst;
   gl_copy_context( xm_src->gl_ctx, xm_dst->gl_ctx, mask );
}



Bool glXQueryExtension( Display *dpy, int *errorb, int *event )
{
   /* Mesa's GLX isn't really an X extension but we try to act like one. */
   return True;
}


void glXDestroyContext( Display *dpy, GLXContext ctx )
{
   XMesaDestroyContext( (XMesaContext) ctx );
}



Bool glXIsDirect( Display *dpy, GLXContext ctx )
{
   /* This isn't true but... */
   return True;
}



void glXSwapBuffers( Display *dpy, GLXDrawable drawable )
{
   XMesaContext ctx = XMesaGetCurrentContext();
   if (ctx->frontbuffer!=drawable) {
      fprintf( stderr,
	  "Warning: glXSwapBuffers drawable doesn't match current context\n");
   }
   else {
      XMesaSwapBuffers();
   }
}



Bool glXQueryVersion( Display *dpy, int *maj, int *min )
{
   *maj = 1;
   *min = 1;
   return True;
}



/*
 * Query the GLX attributes of the given XVisualInfo.
 */
int glXGetConfig( Display *dpy, XVisualInfo *visual,
		  int attrib, int *value )
{
   static GLboolean load_visuals = GL_TRUE;
   GLboolean rgba_flag, double_flag, depth_flag;

   if (load_visuals) {
      /* load the visual list once. */
      load_all_glx_visuals( dpy );
      load_visuals = GL_FALSE;
   }

   if (!find_glx_visual( visual, &rgba_flag, &double_flag, &depth_flag )) {
      if (attrib==GLX_USE_GL) {
	 *value = (int) False;
	 return 0;
      }
      else {
	 fprintf( stderr, "Mesa: Error in glXGetConfig: bad visual\n");
	 return GLX_BAD_VISUAL;
      }
   }

   switch(attrib) {
      case GLX_USE_GL:
         *value = (int) True;
	 return 0;
      case GLX_BUFFER_SIZE:
	 *value = visual->depth;
	 return 0;
      case GLX_LEVEL:
	 *value = 0;
	 return 0;
      case GLX_RGBA:
	 if (rgba_flag) {
	    *value = True;
	 }
	 else {
	    *value = False;
	 }
	 return 0;
      case GLX_DOUBLEBUFFER:
	 *value = (int) double_flag;
	 return 0;
      case GLX_STEREO:
	 *value = (int) False;
	 return 0;
      case GLX_AUX_BUFFERS:
	 *value = (int) False;
	 return 0;
      case GLX_RED_SIZE:
	 {
	    int n = bitcount( visual->visual->red_mask );
	    *value = n>0 ? n : 8;
	 }
	 return 0;
      case GLX_GREEN_SIZE:
	 {
	    int n = bitcount( visual->visual->green_mask );
	    *value = n>0 ? n : 8;
	 }
	 return 0;
      case GLX_BLUE_SIZE:
	 {
	    int n = bitcount( visual->visual->blue_mask );
	    *value = n>0 ? n : 8;
	 }
	 return 0;
      case GLX_ALPHA_SIZE:
	 *value = 8;
	 return 0;
      case GLX_DEPTH_SIZE:
	 if (depth_flag) {
	    *value = (int) sizeof( GLint );
	 }
	 else {
	    *value = 0;
	 }
	 return 0;
      case GLX_STENCIL_SIZE:
	 *value = STENCIL_BITS;
	 return 0;
      case GLX_ACCUM_GREEN_SIZE:
      case GLX_ACCUM_BLUE_SIZE:
      case GLX_ACCUM_RED_SIZE:
      case GLX_ACCUM_ALPHA_SIZE:
	 *value = sizeof(ACC_TYPE)*8;
	 return 0;
      case GLX_SAMPLES_SGIS:
	 /* one sample per pixel */
	 *value = 1;
	 return 0;
      case GLX_SAMPLE_BUFFER_SGIS:
	 /* No multi-sample buffers available */
	 *value = 0;
	 return 0;
      default:
	 return GLX_BAD_ATTRIBUTE;
   }
}



GLXContext glXGetCurrentContext( void )
{
   return (GLXContext) XMesaGetCurrentContext();
}



GLXDrawable glXGetCurrentDrawable( void )
{
   XMesaContext ctx;

   ctx = XMesaGetCurrentContext();
   return ctx->frontbuffer;
}



void glXWaitGL( void )
{
   dd_flush();
}



void glXWaitX( void )
{
   dd_flush();
}




/*
 * New functions in GLX version 1.1
 */


const char *glXQueryExtensionsString( Display *dpy, int screen )
{
   static char *extensions = "";
   return extensions;
}



const char *glXQueryServerString( Display *dpy, int screen, int name )
{
   static char *extensions = "";
   static char *vendor = "Brian Paul";
   static char *version = "1.2.1 Mesa";

   switch (name) {
      case GLX_EXTENSIONS:
         return extensions;
      case GLX_VENDOR:
	 return vendor;
      case GLX_VERSION:
	 return version;
      default:
         return NULL;
   }
}



const char *glXGetClientString( Display *dpy, int name )
{
   static char *extensions = "";
   static char *vendor = "Brian Paul";
   static char *version = "1.2.1 Mesa";

   switch (name) {
      case GLX_EXTENSIONS:
         return extensions;
      case GLX_VENDOR:
	 return vendor;
      case GLX_VERSION:
	 return version;
      default:
         return NULL;
   }
}
