/* vertex.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: vertex.c,v 1.3 1995/05/22 21:02:41 brianp Exp $

$Log: vertex.c,v $
 * Revision 1.3  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.2  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.1  1995/02/24  14:28:31  brianp
 * Initial revision
 *
 */


/*
 * glVertex*, glNormal*, glIndex*, and glColor* functions.
 */



#include "context.h"
#include "draw.h"
#include "list.h"
#include "macros.h"



/*
 * Vertex
 */


/*** 2 arguments ***/

void glVertex2d( GLdouble x, GLdouble y )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = 0.0;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex2f( GLfloat x, GLfloat y )
{
   GLfloat v[4];

   v[0] = x;
   v[1] = y;
   v[2] = 0.0;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex2i( GLint x, GLint y )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = 0.0;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex2s( GLshort x, GLshort y )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = 0.0;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


/*** 3 arguments ***/

void glVertex3d( GLdouble x, GLdouble y, GLdouble z )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex3f( GLfloat x, GLfloat y, GLfloat z )
{
   GLfloat v[4];

   v[0] = x;
   v[1] = y;
   v[2] = z;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex3i( GLint x, GLint y, GLint z )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex3s( GLshort x, GLshort y, GLshort z )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


/*** 4 arguments ***/

void glVertex4d( GLdouble x, GLdouble y, GLdouble z, GLdouble w )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = (GLfloat) w;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex4f( GLfloat x, GLfloat y, GLfloat z, GLfloat w )
{
   GLfloat v[4];

   /* maybe:  GLfloat *v = &x ??? */
   v[0] = x;
   v[1] = y;
   v[2] = z;
   v[3] = w;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex4i( GLint x, GLint y, GLint z, GLint w )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = (GLfloat) w;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex4s( GLshort x, GLshort y, GLshort z, GLshort w )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = (GLfloat) w;

   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


/*** 2 element vector ***/

void glVertex2dv( const GLdouble *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = 0.0;
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


void glVertex2fv( const GLfloat *v )
{
   GLfloat vv[4];

   vv[0] = v[0];
   vv[1] = v[1];
   vv[2] = 0.0;
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


void glVertex2iv( const GLint *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = 0.0;
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


void glVertex2sv( const GLshort *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = 0.0;
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


/*** 3 element vector ***/

void glVertex3dv( const GLdouble *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


void glVertex3fv( const GLfloat *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


void glVertex3iv( const GLint *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


void glVertex3sv( const GLshort *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


/*** 4 element vector ***/

void glVertex4dv( const GLdouble *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = (GLfloat) v[3];

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


void glVertex4fv( const GLfloat *v )
{
   if (CC.CompileFlag) {
      gl_save_vertex( v );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( v );
   }
}


void glVertex4iv( const GLint *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = (GLfloat) v[3];

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}


void glVertex4sv( const GLshort *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = (GLfloat) v[3];

   if (CC.CompileFlag) {
      gl_save_vertex( vv );
   }
   if (CC.ExecuteFlag) {
      gl_vertex( vv );
   }
}



/*
 * Normal vectors
 */


/*** 3 arguments ***/

void glNormal3b( GLbyte nx, GLbyte ny, GLbyte nz )
{
   GLfloat n[3];

   n[0] = BYTE_TO_FLOAT(nx);
   n[1] = BYTE_TO_FLOAT(ny);
   n[2] = BYTE_TO_FLOAT(nz);

   if (CC.CompileFlag) {
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = n[0];
      CC.Current.Normal[1] = n[1];
      CC.Current.Normal[2] = n[2];
   }
}


void glNormal3d( GLdouble nx, GLdouble ny, GLdouble nz )
{
   GLfloat n[3];

   n[0] = (GLfloat) nx;
   n[1] = (GLfloat) ny;
   n[2] = (GLfloat) nz;

   if (CC.CompileFlag) {
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = n[0];
      CC.Current.Normal[1] = n[1];
      CC.Current.Normal[2] = n[2];
   }
}


void glNormal3f( GLfloat nx, GLfloat ny, GLfloat nz )
{
   if (CC.CompileFlag) {
      GLfloat n[3];
      n[0] = nx;
      n[1] = ny;
      n[2] = nz;
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = nx;
      CC.Current.Normal[1] = ny;
      CC.Current.Normal[2] = nz;
   }
}


void glNormal3i( GLint nx, GLint ny, GLint nz )
{
   GLfloat n[3];

   n[0] = INT_TO_FLOAT(nx);
   n[1] = INT_TO_FLOAT(ny);
   n[2] = INT_TO_FLOAT(nz);

   if (CC.CompileFlag) {
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = n[0];
      CC.Current.Normal[1] = n[1];
      CC.Current.Normal[2] = n[2];
   }
}


void glNormal3s( GLshort nx, GLshort ny, GLshort nz )
{
   GLfloat n[3];

   n[0] = SHORT_TO_FLOAT(nx);
   n[1] = SHORT_TO_FLOAT(ny);
   n[2] = SHORT_TO_FLOAT(nz);

   if (CC.CompileFlag) {
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = n[0];
      CC.Current.Normal[1] = n[1];
      CC.Current.Normal[2] = n[2];
   }
}


/*** vector argument ***/

void glNormal3bv( const GLbyte *v )
{
   GLfloat n[3];

   n[0] = BYTE_TO_FLOAT(v[0]);
   n[1] = BYTE_TO_FLOAT(v[0]);
   n[2] = BYTE_TO_FLOAT(v[0]);

   if (CC.CompileFlag) {
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = n[0];
      CC.Current.Normal[1] = n[1];
      CC.Current.Normal[2] = n[2];
   }
}


void glNormal3dv( const GLdouble *v )
{
   GLfloat n[3];

   n[0] = (GLfloat) v[0];
   n[1] = (GLfloat) v[1];
   n[2] = (GLfloat) v[2];

   if (CC.CompileFlag) {
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = n[0];
      CC.Current.Normal[1] = n[1];
      CC.Current.Normal[2] = n[2];
   }
}


void glNormal3fv( const GLfloat *v )
{
   if (CC.CompileFlag) {
      gl_save_normal( v );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = v[0];
      CC.Current.Normal[1] = v[1];
      CC.Current.Normal[2] = v[2];
   }
}


void glNormal3iv( const GLint *v )
{
   GLfloat n[3];

   n[0] = INT_TO_FLOAT(v[0]);
   n[1] = INT_TO_FLOAT(v[0]);
   n[2] = INT_TO_FLOAT(v[0]);

   if (CC.CompileFlag) {
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = n[0];
      CC.Current.Normal[1] = n[1];
      CC.Current.Normal[2] = n[2];
   }
}


void glNormal3sv( const GLshort *v )
{
   GLfloat n[3];

   n[0] = SHORT_TO_FLOAT(v[0]);
   n[1] = SHORT_TO_FLOAT(v[1]);
   n[2] = SHORT_TO_FLOAT(v[2]);

   if (CC.CompileFlag) {
      gl_save_normal( n );
   }
   if (CC.ExecuteFlag) {
      CC.Current.Normal[0] = n[0];
      CC.Current.Normal[1] = n[1];
      CC.Current.Normal[2] = n[2];
   }
}



/*
 * Color Index
 */


void glIndexd( GLdouble c )
{
   if (CC.CompileFlag) {
      gl_save_index( (GLuint) c );
   }
   if (CC.ExecuteFlag) {
      gl_index( (GLuint) c );
   }
}


void glIndexf( GLfloat c )
{
   if (CC.CompileFlag) {
      gl_save_index( (GLuint) c );
   }
   if (CC.ExecuteFlag) {
      gl_index( (GLuint) c );
   }
}


void glIndexi( GLint c )
{
   if (CC.CompileFlag) {
      gl_save_index( (GLuint) c );
   }
   if (CC.ExecuteFlag) {
      gl_index( (GLuint) c );
   }
}


void glIndexs( GLshort c )
{
   if (CC.CompileFlag) {
      gl_save_index( (GLuint) c );
   }
   if (CC.ExecuteFlag) {
      gl_index( (GLuint) c );
   }
}


void glIndexdv( const GLdouble *c )
{
   if (CC.CompileFlag) {
      gl_save_index( (GLuint) *c );
   }
   if (CC.ExecuteFlag) {
      gl_index( (GLuint) *c );
   }
}


void glIndexfv( const GLfloat *c )
{
   if (CC.CompileFlag) {
      gl_save_index( (GLuint) *c );
   }
   if (CC.ExecuteFlag) {
      gl_index( (GLuint) *c );
   }
}


void glIndexiv( const GLint *c )
{
   if (CC.CompileFlag) {
      gl_save_index( (GLuint) *c );
   }
   if (CC.ExecuteFlag) {
      gl_index( (GLuint) *c );
   }
}


void glIndexsv( const GLshort *c )
{
   if (CC.CompileFlag) {
      gl_save_index( (GLuint) *c );
   }
   if (CC.ExecuteFlag) {
      gl_index( (GLuint) *c );
   }
}



/*
 * Color
 */


void glColor3b( GLbyte red, GLbyte green, GLbyte blue )
{
   GLfloat c[4];

   c[0] = BYTE_TO_FLOAT(red);
   c[1] = BYTE_TO_FLOAT(green);
   c[2] = BYTE_TO_FLOAT(blue);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3d( GLdouble red, GLdouble green, GLdouble blue )
{
   GLfloat c[4];

   c[0] = (GLfloat) red;
   c[1] = (GLfloat) green;
   c[2] = (GLfloat) blue;
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3f( GLfloat red, GLfloat green, GLfloat blue )
{
   GLfloat c[4];

   c[0] = red;
   c[1] = green;
   c[2] = blue;
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3i( GLint red, GLint green, GLint blue )
{
   GLfloat c[4];

   c[0] = INT_TO_FLOAT(red);
   c[1] = INT_TO_FLOAT(green);
   c[2] = INT_TO_FLOAT(blue);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3s( GLshort red, GLshort green, GLshort blue )
{
   GLfloat c[4];

   c[0] = SHORT_TO_FLOAT(red);
   c[1] = SHORT_TO_FLOAT(green);
   c[2] = SHORT_TO_FLOAT(blue);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3ub( GLubyte red, GLubyte green, GLubyte blue )
{
   GLfloat c[4];

   c[0] = UBYTE_TO_FLOAT(red);
   c[1] = UBYTE_TO_FLOAT(green);
   c[2] = UBYTE_TO_FLOAT(blue);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3ui( GLuint red, GLuint green, GLuint blue )
{
   GLfloat c[4];

   c[0] = UINT_TO_FLOAT(red);
   c[1] = UINT_TO_FLOAT(green);
   c[2] = UINT_TO_FLOAT(blue);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3us( GLushort red, GLushort green, GLushort blue )
{
   GLfloat c[4];

   c[0] = USHORT_TO_FLOAT(red);
   c[1] = USHORT_TO_FLOAT(green);
   c[2] = USHORT_TO_FLOAT(blue);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4b( GLbyte red, GLbyte green, GLbyte blue, GLbyte alpha )
{
   GLfloat c[4];

   c[0] = BYTE_TO_FLOAT(red);
   c[1] = BYTE_TO_FLOAT(green);
   c[2] = BYTE_TO_FLOAT(blue);
   c[3] = BYTE_TO_FLOAT(alpha);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4d( GLdouble red, GLdouble green, GLdouble blue, GLdouble alpha )
{
   GLfloat c[4];

   c[0] = (GLfloat) red;
   c[1] = (GLfloat) green;
   c[2] = (GLfloat) blue;
   c[3] = (GLfloat) alpha;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4f( GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha )
{
   GLfloat v[4];

   v[0] = red;
   v[1] = green;
   v[2] = blue;
   v[3] = alpha;

   if (CC.CompileFlag) {
      gl_save_color( v );
   }
   if (CC.ExecuteFlag) {
      gl_color( v );
   }
}


void glColor4i( GLint red, GLint green, GLint blue, GLint alpha )
{
   GLfloat c[4];

   c[0] = INT_TO_FLOAT(red);
   c[1] = INT_TO_FLOAT(green);
   c[2] = INT_TO_FLOAT(blue);
   c[3] = INT_TO_FLOAT(alpha);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4s( GLshort red, GLshort green, GLshort blue, GLshort alpha )
{
   GLfloat c[4];

   c[0] = SHORT_TO_FLOAT(red);
   c[1] = SHORT_TO_FLOAT(green);
   c[2] = SHORT_TO_FLOAT(blue);
   c[3] = SHORT_TO_FLOAT(alpha);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4ub( GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha )
{
   GLfloat c[4];

   c[0] = UBYTE_TO_FLOAT(red);
   c[1] = UBYTE_TO_FLOAT(green);
   c[2] = UBYTE_TO_FLOAT(blue);
   c[3] = UBYTE_TO_FLOAT(alpha);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4ui( GLuint red, GLuint green, GLuint blue, GLuint alpha )
{
   GLfloat c[4];

   c[0] = UINT_TO_FLOAT(red);
   c[1] = UINT_TO_FLOAT(green);
   c[2] = UINT_TO_FLOAT(blue);
   c[3] = UINT_TO_FLOAT(alpha);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4us( GLushort red, GLushort green, GLushort blue, GLushort alpha )
{
   GLfloat c[4];

   c[0] = USHORT_TO_FLOAT(red);
   c[1] = USHORT_TO_FLOAT(green);
   c[2] = USHORT_TO_FLOAT(blue);
   c[3] = USHORT_TO_FLOAT(alpha);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3bv( const GLbyte *v )
{
   GLfloat c[4];

   c[0] = BYTE_TO_FLOAT(v[0]);
   c[1] = BYTE_TO_FLOAT(v[1]);
   c[2] = BYTE_TO_FLOAT(v[2]);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3dv( const GLdouble *v )
{
   GLfloat c[4];

   c[0] = (GLfloat) v[0];
   c[1] = (GLfloat) v[1];
   c[2] = (GLfloat) v[2];
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3fv( const GLfloat *v )
{
   GLfloat c[4];

   c[0] = v[0];
   c[1] = v[1];
   c[2] = v[2];
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3iv( const GLint *v )
{
   GLfloat c[4];

   c[0] = INT_TO_FLOAT(v[0]);
   c[1] = INT_TO_FLOAT(v[1]);
   c[2] = INT_TO_FLOAT(v[2]);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3sv( const GLshort *v )
{
   GLfloat c[4];

   c[0] = SHORT_TO_FLOAT(v[0]);
   c[1] = SHORT_TO_FLOAT(v[1]);
   c[2] = SHORT_TO_FLOAT(v[2]);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3ubv( const GLubyte *v )
{
   GLfloat c[4];

   c[0] = UBYTE_TO_FLOAT(v[0]);
   c[1] = UBYTE_TO_FLOAT(v[1]);
   c[2] = UBYTE_TO_FLOAT(v[2]);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3uiv( const GLuint *v )
{
   GLfloat c[4];

   c[0] = UINT_TO_FLOAT(v[0]);
   c[1] = UINT_TO_FLOAT(v[1]);
   c[2] = UINT_TO_FLOAT(v[2]);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor3usv( const GLushort *v )
{
   GLfloat c[4];

   c[0] = USHORT_TO_FLOAT(v[0]);
   c[1] = USHORT_TO_FLOAT(v[1]);
   c[2] = USHORT_TO_FLOAT(v[2]);
   c[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4bv( const GLbyte *v )
{
   GLfloat c[4];

   c[0] = BYTE_TO_FLOAT(v[0]);
   c[1] = BYTE_TO_FLOAT(v[1]);
   c[2] = BYTE_TO_FLOAT(v[2]);
   c[3] = BYTE_TO_FLOAT(v[3]);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4dv( const GLdouble *v )
{
   GLfloat c[4];

   c[0] = (GLfloat) v[0];
   c[1] = (GLfloat) v[1];
   c[2] = (GLfloat) v[2];
   c[3] = (GLfloat) v[3];

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4fv( const GLfloat *v )
{
   if (CC.CompileFlag) {
      gl_save_color( v );
   }
   if (CC.ExecuteFlag) {
      gl_color( v );
   }
}


void glColor4iv( const GLint *v )
{
   GLfloat c[4];

   c[0] = INT_TO_FLOAT(v[0]);
   c[1] = INT_TO_FLOAT(v[1]);
   c[2] = INT_TO_FLOAT(v[2]);
   c[3] = INT_TO_FLOAT(v[3]);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4sv( const GLshort *v )
{
   GLfloat c[4];

   c[0] = SHORT_TO_FLOAT(v[0]);
   c[1] = SHORT_TO_FLOAT(v[1]);
   c[2] = SHORT_TO_FLOAT(v[2]);
   c[3] = SHORT_TO_FLOAT(v[3]);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4ubv( const GLubyte *v )
{
   GLfloat c[4];

   c[0] = UBYTE_TO_FLOAT(v[0]);
   c[1] = UBYTE_TO_FLOAT(v[1]);
   c[2] = UBYTE_TO_FLOAT(v[2]);
   c[3] = UBYTE_TO_FLOAT(v[3]);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4uiv( const GLuint *v )
{
   GLfloat c[4];

   c[0] = UINT_TO_FLOAT(v[0]);
   c[1] = UINT_TO_FLOAT(v[1]);
   c[2] = UINT_TO_FLOAT(v[2]);
   c[3] = UINT_TO_FLOAT(v[3]);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}


void glColor4usv( const GLushort *v )
{
   GLfloat c[4];

   c[0] = USHORT_TO_FLOAT(v[0]);
   c[1] = USHORT_TO_FLOAT(v[1]);
   c[2] = USHORT_TO_FLOAT(v[2]);
   c[3] = USHORT_TO_FLOAT(v[3]);

   if (CC.CompileFlag) {
      gl_save_color( c );
   }
   if (CC.ExecuteFlag) {
      gl_color( c );
   }
}



/*
 * glRasterPos* functions
 */


void glRasterPos2d( GLdouble x, GLdouble y )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = 0.0;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos2f( GLfloat x, GLfloat y )
{
   GLfloat v[4];

   v[0] = x;
   v[1] = y;
   v[2] = 0.0;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos2i( GLint x, GLint y )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = 0.0;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos2s( GLshort x, GLshort y )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = 0.0;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


/*** 3 arguments ***/

void glRasterPos3d( GLdouble x, GLdouble y, GLdouble z )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos3f( GLfloat x, GLfloat y, GLfloat z )
{
   GLfloat v[4];

   v[0] = x;
   v[1] = y;
   v[2] = z;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos3i( GLint x, GLint y, GLint z )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos3s( GLshort x, GLshort y, GLshort z )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


/*** 4 arguments ***/

void glRasterPos4d( GLdouble x, GLdouble y, GLdouble z, GLdouble w )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = (GLfloat) w;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos4f( GLfloat x, GLfloat y, GLfloat z, GLfloat w )
{
   GLfloat v[4];

   /* maybe:  GLfloat *v = &x ??? */
   v[0] = x;
   v[1] = y;
   v[2] = z;
   v[3] = w;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos4i( GLint x, GLint y, GLint z, GLint w )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = (GLfloat) w;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos4s( GLshort x, GLshort y, GLshort z, GLshort w )
{
   GLfloat v[4];

   v[0] = (GLfloat) x;
   v[1] = (GLfloat) y;
   v[2] = (GLfloat) z;
   v[3] = (GLfloat) w;

   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


/*** 2 element vector ***/

void glRasterPos2dv( const GLdouble *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = 0.0;
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


void glRasterPos2fv( const GLfloat *v )
{
   GLfloat vv[4];

   vv[0] = v[0];
   vv[1] = v[1];
   vv[2] = 0.0;
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


void glRasterPos2iv( const GLint *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = 0.0;
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


void glRasterPos2sv( const GLshort *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = 0.0;
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


/*** 3 element vector ***/

void glRasterPos3dv( const GLdouble *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


void glRasterPos3fv( const GLfloat *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


void glRasterPos3iv( const GLint *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


void glRasterPos3sv( const GLshort *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = 1.0;

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


/*** 4 element vector ***/

void glRasterPos4dv( const GLdouble *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = (GLfloat) v[3];

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


void glRasterPos4fv( const GLfloat *v )
{
   if (CC.CompileFlag) {
      gl_save_rasterpos( v );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( v );
   }
}


void glRasterPos4iv( const GLint *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = (GLfloat) v[3];

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}


void glRasterPos4sv( const GLshort *v )
{
   GLfloat vv[4];

   vv[0] = (GLfloat) v[0];
   vv[1] = (GLfloat) v[1];
   vv[2] = (GLfloat) v[2];
   vv[3] = (GLfloat) v[3];

   if (CC.CompileFlag) {
      gl_save_rasterpos( vv );
   }
   if (CC.ExecuteFlag) {
      gl_rasterpos( vv );
   }
}



/*
 *
 * Texture coordinates
 *
 */


void glTexCoord1d( GLdouble s )
{
   glTexCoord4f( (GLfloat) s, 0.0, 0.0, 1.0 );
}

void glTexCoord1f( GLfloat s )
{
   glTexCoord4f( s, 0.0, 0.0, 1.0 );
}

void glTexCoord1i( GLint s )
{
   glTexCoord4f( (GLfloat) s, 0.0, 0.0, 1.0 );
}

void glTexCoord1s( GLshort s )
{
   glTexCoord4f( (GLfloat) s, 0.0, 0.0, 1.0 );
}

void glTexCoord2d( GLdouble s, GLdouble t )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, 0.0, 1.0 );
}

void glTexCoord2f( GLfloat s, GLfloat t )
{
   glTexCoord4f( s, t, 0.0, 1.0 );
}

void glTexCoord2i( GLint s, GLint t )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, 0.0, 1.0 );
}

void glTexCoord2s( GLshort s, GLshort t )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, 0.0, 1.0 );
}

void glTexCoord3d( GLdouble s, GLdouble t, GLdouble r )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, (GLfloat) r, 1.0 );
}

void glTexCoord3f( GLfloat s, GLfloat t, GLfloat r )
{
   glTexCoord4f( s, t, r, 1.0 );
}

void glTexCoord3i( GLint s, GLint t, GLint r )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, (GLfloat) r, 1.0 );
}

void glTexCoord3s( GLshort s, GLshort t, GLshort r )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, (GLfloat) r, 1.0 );
}

void glTexCoord4d( GLdouble s, GLdouble t, GLdouble r, GLdouble q )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, (GLfloat) r, (GLfloat) q );
}

void glTexCoord4f( GLfloat s, GLfloat t, GLfloat r, GLfloat q )
{
   if (CC.CompileFlag) {
      GLfloat tc[4];
      tc[0] = s;
      tc[1] = t;
      tc[2] = r;
      tc[3] = q;
      gl_save_texcoord( tc );
   }
   if (CC.ExecuteFlag) {
      CC.Current.TexCoord[0] = s;
      CC.Current.TexCoord[1] = t;
      CC.Current.TexCoord[2] = r;
      CC.Current.TexCoord[3] = q;
   }
}

void glTexCoord4i( GLint s, GLint t, GLint r, GLint q )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, (GLfloat) r, (GLfloat) q );
}

void glTexCoord4s( GLshort s, GLshort t, GLshort r, GLshort q )
{
   glTexCoord4f( (GLfloat) s, (GLfloat) t, (GLfloat) r, (GLfloat) q );
}

void glTexCoord1dv( const GLdouble *v )
{
   glTexCoord4f( (GLfloat) *v, 0.0, 0.0, 1.0 );
}

void glTexCoord1fv( const GLfloat *v )
{
   glTexCoord4f( *v, 0.0, 0.0, 1.0 );
}

void glTexCoord1iv( const GLint *v )
{
   glTexCoord4f( (GLfloat) *v, 0.0, 0.0, 1.0 );
}

void glTexCoord1sv( const GLshort *v )
{
   glTexCoord4f( (GLfloat) *v, 0.0, 0.0, 1.0 );
}

void glTexCoord2dv( const GLdouble *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1], 0.0, 1.0 );
}

void glTexCoord2fv( const GLfloat *v )
{
   glTexCoord4f( v[0], v[1], 0.0, 1.0 );
}

void glTexCoord2iv( const GLint *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1], 0.0, 1.0 );
}

void glTexCoord2sv( const GLshort *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1], 0.0, 1.0 );
}

void glTexCoord3dv( const GLdouble *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1], (GLfloat) v[2], 1.0 );
}

void glTexCoord3fv( const GLfloat *v )
{
   glTexCoord4f( v[0], v[1], v[2], 1.0 );
}

void glTexCoord3iv( const GLint *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1], (GLfloat) v[2], 1.0 );
}

void glTexCoord3sv( const GLshort *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1], (GLfloat) v[2], 1.0 );
}

void glTexCoord4dv( const GLdouble *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1],
		 (GLfloat) v[2], (GLfloat) v[3] );
}

void glTexCoord4fv( const GLfloat *v )
{
   glTexCoord4f( v[0], v[1], v[2], v[3] );
}

void glTexCoord4iv( const GLint *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1],
		 (GLfloat) v[2], (GLfloat) v[3] );
}

void glTexCoord4sv( const GLshort *v )
{
   glTexCoord4f( (GLfloat) v[0], (GLfloat) v[1],
		 (GLfloat) v[2], (GLfloat) v[3] );
}



/*
 *
 * Polygon Edge Flags
 *
 */


void glEdgeFlag( GLboolean flag )
{
   if (CC.ExecuteFlag) {
      CC.Current.EdgeFlag = flag;
   }
   if (CC.CompileFlag) {
      gl_save_set_boolean( &CC.Current.EdgeFlag, flag );
   }
}



void glEdgeFlagv( const GLboolean *flag )
{
   if (CC.ExecuteFlag) {
      CC.Current.EdgeFlag = *flag;
   }
   if (CC.CompileFlag) {
      gl_save_set_boolean( &CC.Current.EdgeFlag, *flag );
   }
}
