

/*C*

________________________________________________________________

        biff2pnm
        $Id: biff2pnm.c,v 1.9 1997/09/01 12:18:07 svein Exp $
        Copyright 1994, Blab, UiO
        Image processing lab, Department of Informatics
        University of Oslo
        E-mail: blab@ifi.uio.no
________________________________________________________________
  
  Permission to use, copy, modify and distribute this software and its
  documentation for any purpose and without fee is hereby granted, 
  provided that this copyright notice appear in all copies and that 
  both that copyright notice and this permission notice appear in supporting
  documentation and that the name of B-lab, Department of Informatics or
  University of Oslo not be used in advertising or publicity pertaining 
  to distribution of the software without specific, written prior permission.

  B-LAB DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL B-LAB
  BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
  OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 

*/

static char *Id = "$Id: biff2pnm.c,v 1.9 1997/09/01 12:18:07 svein Exp $, Blab, UiO";

#include <xite/biff.h>
#include <xite/readarg.h>
#include <xite/message.h>
#include <xite/convert.h>
#include <pnm.h>

#define  BIFF_BLACK 0
#define  BIFF_WHITE 255



/*P:biff2pnm*

________________________________________________________________

		biff2pnm
________________________________________________________________

Name:		biff2pnm -  BIFF to Portable Bit/Gray/Pixel Map conversion

Syntax:		| biff2pnm [<option>...] <BIFF-infile> <pnm-outfile>

Description:    Read a BIFF-image and convert the first band to one of the
                formats pbm (portable bitmap), pgm (portable graymap) or ppm
		(portable pixmap). Write the result to a 'pnm-file'
		(pnm: portable anymap).

		If the BIFF-image is two-valued, write a pbm-image.
		If the BIFF-image consists of more than two different
		values, write a pgm-image.
		Use option "-rgb" (see below) for rgb images and ppm-format.

		The data type of pixelvalue components in pbm/pgm/ppm is
		either unsigned char or unsigned short, depending on how the
		pbm package was compiled. 'biff2pnm' converts the BIFF band to
		the correct pixeltype before conversion to pbm/pgm/ppm. The
		pixeltype conversion does not imply any scaling. Values too
		large (or too small) to be represented by the new pixeltype are
		set to the maximum (or minimum) possible value. A warning is
		issued to this effect.

		User "-" as filename arguments for stdin and stdout.

Options:        &-rgb
                If the BIFF image consists of three bands, assume that they
		represent red, green and blue color components.
		Write a ppm-image.

		&-a
		Write a plain-format file (ascii), as opposed to a raw-format
		one (binary).

See also:       pnm2biff(1), XITE(1), biff2raw(1)

References:     Documentation for the pbmplus or netpbm free software
                packages.

Author:		Svein Be, Blab, Ifi, UiO

Examples:       | biff2pnm monabinary.pnm monabinary.img

Id: 		$Id: biff2pnm.c,v 1.9 1997/09/01 12:18:07 svein Exp $
________________________________________________________________

*/

#ifndef FUNCPROTO
int main( argc, argv )
int argc;
char* argv[];
#else /* FUNCPROTO */
int main(int argc, char **argv)
#endif /* FUNCPROTO */
{
  IMAGE i1;
  IUS_BAND ius_bnd = NULL, ius_bnd2 = NULL, ius_bnd3 = NULL;
  IBAND iub_bnd = NULL, iub_bnd2 = NULL, iub_bnd3 = NULL;
  FILE* ofp;
  xel** pnm;
  xelval maxval;
  int xsize, ysize, x, y;
  int rgb, ascii, binary, low, high;

  InitMessage(&argc, argv, xite_app_std_usage_text(
  "Usage: %s [<option>...] <BIFF-infile> <pnm-outfile>\n\
     where <option> is chosen from\n\
     -rgb         : Consider three-band image as rgb image. Write ppm-image.\n\
     -a           : Write a plain-format file (ascii), as opposed to a\n\
                    raw-format one (binary).\n"));

  Iset_message(1);		/* Warnings from BIFF-software */
  Iset_abort(1);

  if (argc == 1) Usage(1, NULL);

  pnm_init(&argc, argv);

  rgb   = read_bswitch(&argc, argv, "-rgb");
  ascii = read_bswitch(&argc, argv, "-a");

  if (argc != 3) Usage(2, "Illegal number of arguments.\n");

  i1 = Iread_image(argv[1]);

  if (rgb && Inbands(i1) != 3) {
    rgb = 0;
    Warning(1, "BIFF image is not three-band. Ignoring option -rgb.\n");
  }

  xsize=Ixsize(i1[1]);
  ysize=Iysize(i1[1]);
  pnm = pnm_allocarray(xsize, ysize);

  if (sizeof(xelval) == 2 && Ipixtyp(i1[1]) != Iu_short_typ) {
    if (Ipixtyp(i1[1]) > Iu_short_typ)
      Warning(0, "%s%s\n",
	      "Converting BIFF image to smaller pixeltype. ",
	      "May loose range.");
    else if (Ipixtyp(i1[1]) == Is_byte_typ)
      Warning(0, "%s%s\n",
	      "Converting BIFF image to unsigned pixeltype. ",
	      "May loose range.");

    ius_bnd = (IUS_BAND) mkConvertBand(i1[1], Iu_short_typ);
    if (Inbands(i1) == 3) {
      ius_bnd2 = (IUS_BAND) mkConvertBand(i1[2], Iu_short_typ);
      ius_bnd3 = (IUS_BAND) mkConvertBand(i1[3], Iu_short_typ);
    }
  } else if (sizeof(xelval) == 1 && Ipixtyp(i1[1]) != Iu_byte_typ) {
    if (Ipixtyp(i1[1]) > Iu_byte_typ)
      Warning(0, "%s%s\n",
	      "Converting BIFF image to smaller pixeltype. ",
	      "May loose range.");

    iub_bnd = mkConvertBand(i1[1], Iu_byte_typ);
    if (Inbands(i1) == 3) {
      iub_bnd2 = mkConvertBand(i1[2], Iu_byte_typ);
      iub_bnd3 = mkConvertBand(i1[3], Iu_byte_typ);
    }
  }

  maxval = 0;
  binary = 1;
  if (rgb) {
    if (Ixsize(i1[2]) != xsize || Ixsize(i1[3]) != xsize ||
	Iysize(i1[2]) != ysize || Iysize(i1[3]) != ysize)
      Error(2, "BIFF image bands do not have equal sizes.\n");

    if (iub_bnd != NULL) {
      for (y=1; y<=ysize; y++) {
	for (x=1; x<=xsize; x++) {
	  PPM_ASSIGN(pnm[y-1][x-1],
		     (pixval) iub_bnd[y][x],
		     (pixval) iub_bnd2[y][x],
		     (pixval) iub_bnd3[y][x]);
	  if ((pixval) iub_bnd[y][x]  > maxval) maxval = iub_bnd[y][x];
	  if ((pixval) iub_bnd2[y][x] > maxval) maxval = iub_bnd2[y][x];
	  if ((pixval) iub_bnd3[y][x] > maxval) maxval = iub_bnd3[y][x];
	}
      }
    } else {
      for (y=1; y<=ysize; y++) {
	for (x=1; x<=xsize; x++) {
	  PPM_ASSIGN(pnm[y-1][x-1],
		     (pixval) ius_bnd[y][x],
		     (pixval) ius_bnd2[y][x],
		     (pixval) ius_bnd3[y][x]);
	  if ((pixval) ius_bnd[y][x]  > maxval) maxval = ius_bnd[y][x];
	  if ((pixval) ius_bnd2[y][x] > maxval) maxval = ius_bnd2[y][x];
	  if ((pixval) ius_bnd3[y][x] > maxval) maxval = ius_bnd3[y][x];
	}
      }
    }
  } else {
    if (iub_bnd != NULL) {
      high = low = (pixval) iub_bnd[1][1];

      for (y=1; y<=ysize; y++) {
	for (x=1; x<=xsize; x++) {
	  PNM_ASSIGN1(pnm[y-1][x-1], (pixval) iub_bnd[y][x]);
	  if ((pixval) iub_bnd[y][x] > maxval) maxval = iub_bnd[y][x];
	  
	  if (binary) {
	    if (low != high && iub_bnd[y][x] != low && iub_bnd[y][x] != high)
	      /* At least three different values have been found. */
	      binary = 0;
	    else if (iub_bnd[y][x] != low) high = iub_bnd[y][x];
	  }
	}
      }
    } else {
      high = low = (pixval) ius_bnd[1][1];

      for (y=1; y<=ysize; y++) {
	for (x=1; x<=xsize; x++) {
	  PNM_ASSIGN1(pnm[y-1][x-1], (pixval) ius_bnd[y][x]);
	  if ((pixval) ius_bnd[y][x] > maxval) maxval = ius_bnd[y][x];
	  
	  if (binary) {
	    if (low != high && ius_bnd[y][x] != low && ius_bnd[y][x] != high)
	      /* At least three different values have been found. */
	      binary = 0;
	    else if (ius_bnd[y][x] != low) high = ius_bnd[y][x];
	  }
	}
      }
    }
  }

  if (strcmp(argv[2], "-") && strcmp(argv[2], "-1"))
    ofp = pm_openw(argv[2]);
  else ofp = stdout;

  pnm_writepnm(ofp, pnm, xsize, ysize, maxval,
	       rgb ? PPM_FORMAT : (binary ? PBM_FORMAT : PGM_FORMAT), ascii);

  return(0);
}
