/*
 * Label.c,v 2.2 1992/08/11 00:14:59 pete Exp
 * Label.c,v
 * Revision 2.2  1992/08/11  00:14:59  pete
 * Many changes to support adding the "Flag" object to indicate current
 * state.  Also attempted to improve geometry management requests.  Use
 * XoNameString() instead of XoName().  The Flag really complicated
 * things.
 *
 * Revision 2.1  1992/07/22  15:57:16  pete
 * Use XoNameString() instead of XtName().
 *
 * Revision 2.0  1992/04/23  02:51:02  ware
 * First public release.
 *
 * Revision 1.11  1992/04/23  02:18:38  ware
 * Added several classes.  Worked on geometry management
 *
 * Revision 1.10  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.9  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.8  1992/02/04  21:22:46  pete
 * Release 44
 *
 * Revision 1.7  1991/09/12  09:45:47  pete
 * Does a better job responding to QueryGeometry() requests.
 *
 * Revision 1.6  91/08/26  17:16:31  pete
 * Added some debuging info to Redisplay() to help problems with
 * geometry_handler in the Menu widget.
 *
 * Revision 1.5  91/08/26  11:57:46  pete
 * Use XoProto() for conditional prototypes.  Working on getting traversals
 * and menus to work more efficiently.  Changed to following naming
 * conventions.
 *
 * Revision 1.4  91/07/19  00:59:55  pete
 * Use shorter file names.  Various speedups.
 *
 * Revision 1.3  1991/05/22  17:49:39  pete
 * Get it to compile cleanely.  Menus almost work.
 *
 * Revision 1.2  91/05/16  18:40:55  pete
 * Added query_geometry
 *
 * Revision 1.1  91/05/15  08:56:43  pete
 * Initial revision
 *
 */
#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xo/LabelP.h>
#include <X11/Xo/dbug.h>

#include <X11/Xo/LabelRec.h>

/*
 * Private procedures
 */
XoProto (static void, DoLayout, (Widget gw));
XoProto (static void, CalcSize, (XoLabelWidget w, Dimension *width, Dimension *height));
XoProto (static void, CalcLocation, (Widget gw, Position *x_draw_ret,
				     Position *y_draw_ret,
				     Position *x_flag_ret,
				     Position *y_flag_ret));
XoProto (static void, AskNewSize, (XoLabelWidget w, unsigned int width, unsigned int height, int query));
XoProto (static void, LabelPostscript, (Widget tree, XoHardCopy * hardcopy));

/*
 *----------------------------------------------------------------------
 * Core Class Methods
 *----------------------------------------------------------------------
 */

static void
ClassInit ()
{
	_XoRegisterConverters ();
}

/*
 * ClassPartInitialize -
 * Installs a hardcopy extension for Postscript.
 */

static void
ClassPartInitialize (class)
	WidgetClassRec *class;
{
	XoHardcopyAddExtension (class, XoPostscript, 1L, LabelPostscript,
				(XtPointer) NULL);
}

static void
Redisplay (gw, event, region)
	Widget          gw;
	XEvent         *event;
	Region          region;
{
	XoLabelWidget   w = (XoLabelWidget) gw;

	_XoObjRedraw (w->simple.border_widget, event, region);
	_XoObjRedraw (w->label.draw_widget, event, region);
	_XoObjRedraw (w->label.flag_widget, event, region);
}

static void
Resize (gw)
	Widget          gw;
{
	DBUG_ENTER ("Label.Resize");
	DoLayout (gw);
	DBUG_VOID_RETURN;
}

static void
Initialize (request, new, arglist, num_args)
	Widget          request;	/* as first created */
	Widget          new;		/* after other parent classes */
	ArgList         arglist;	/* list of arguments */
	Cardinal       *num_args;	/* how many */
{
	XoLabelWidget   w = (XoLabelWidget) new;
	Arg             args[10];
	Cardinal        cnt;
	Dimension      *width_ptr;
	Dimension      *height_ptr;

	w->label.draw_widget = NULL;
	if (w->label.draw_class)
	{
		cnt = 0;
		w->label.draw_widget =
			XtCreateWidget (XtName ((Widget) w), w->label.draw_class,
					(Widget) w, args, cnt);
		_XoInheritedSet (new, w->label.draw_widget,
				 arglist, num_args);
	}
	w->label.flag_widget = NULL;
	if (w->label.flag_class)
	{
		cnt = 0;
		w->label.flag_widget =
			XtCreateWidget ("flag",
					w->label.flag_class,
					(Widget) w, args, cnt);
		_XoInheritedSet (new, w->label.flag_widget,
				 arglist, num_args);
	}

	/*
	 * If the width and height are already set, do not change them
	 */
	if (!w->core.width)
		width_ptr = &w->core.width;
	else
		width_ptr = NULL;
	if (!w->core.height)
		height_ptr = &w->core.height;
	else
		height_ptr = NULL;
	CalcSize (w, width_ptr, height_ptr);
	DoLayout (new);
}

static Boolean
SetValues (current, request, new, args, num_args)
	Widget          current;	/* widget before the XtSetValues() */
	Widget          request;	/* after args applied but no
					 * set_values */
	Widget          new;		/* the allowed changes */
	ArgList         args;		/* list of arguments */
	Cardinal       *num_args;	/* how many arguments */
{
	XoLabelWidget   w = (XoLabelWidget) new;
	XoLabelWidget   cur = (XoLabelWidget) current;
	Boolean         redisplay = False;
	Boolean		need_layout = False;
	Dimension	width,height;

	DBUG_ENTER ("Label.SetValues");
	if (w->label.gravity != cur->label.gravity)
	{
		need_layout = True;
	}
	if (w->label.draw_widget != cur->label.draw_widget)
	{
		if (cur->label.draw_widget)
			XtDestroyWidget (cur->label.draw_widget);
		redisplay = True;
		need_layout = True;
	}
	if (w->label.flag_widget != cur->label.flag_widget)
	{
		if (cur->label.flag_widget)
			XtDestroyWidget (cur->label.flag_widget);
		CalcSize (w, &width, &height);
		AskNewSize (w, width, height, False);
		redisplay = True;
		need_layout = True;
	}
		
	/*
	 * It doesn't matter if label.draw_class changes.
	 */

	/*
	 * Update the values in the drawing widget.  If they change, then try
	 * calculating a new size.
	 */
	if (_XoInheritedSet (new, w->label.draw_widget, args, num_args))
	{
		CalcSize (w, &width, &height);
		AskNewSize (w, width, height, False);
		need_layout = True;
	}
	if (need_layout)
		DoLayout (new);
	DBUG_RETURN (redisplay);
}

static XtGeometryResult
GeometryQuery (gw, proposed, desired)
	Widget          gw;
	XtWidgetGeometry *proposed;
	XtWidgetGeometry *desired;
{
	XoLabelWidget   w = (XoLabelWidget) gw;
	XtGeometryResult result;
	Dimension	width;
	Dimension	height;

	DBUG_ENTER ("Label.GeometryQuery");

	CalcSize (w, &width, &height);
	if (!proposed->request_mode)
	{
		/*
		 * I'm just being asked my ideal size
		 */
		desired->width = width;
		desired->height = height;
		desired->request_mode = CWWidth | CWHeight;
		if (width == w->core.width && height == w->core.height)
			result = XtGeometryNo;
		else
			result = XtGeometryAlmost;
	}
	else if (XoGeoDimFlags (proposed) == CWWidth)
	{
		desired->height = height;
		desired->request_mode = CWHeight;
		result = XtGeometryAlmost;
	}
	else if (XoGeoDimFlags (proposed) == CWHeight)
	{
		desired->width = width;
		desired->request_mode = CWWidth;
		result = XtGeometryAlmost;
	}
	else
	{
		/*
		 * Parent is specifying both width and height, so just return
		 * if it's what we'd like to be
		 */
		if (proposed->width == width && proposed->height == height)
			result = XtGeometryYes;
		else
			result = XtGeometryNo;
	}
	DBUG_RETURN (result);
}

/*
 *----------------------------------------------------------------------
 * Simple Class Methods
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * Label Class Methods
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * Private Utilities
 *----------------------------------------------------------------------
 */

/*
 * DoLayout -
 *	Place the draw, flag and border widgets.  Queries the draw and flag
 *	widgets to find their ideal size.  Places the draw widget according
 *	to the value of gravity.  Then places the flag widget in relation
 *	to the draw widget.
 */

static void
DoLayout (gw)
	Widget		gw;
{
	XoLabelWidget	w = (XoLabelWidget) gw;
	Position	x, y;
	Position	flag_x, flag_y;

	CalcLocation (gw, &x, &y, &flag_x, &flag_y);
	if (w->label.draw_widget)
	{
		if (w->label.resize_draw)
			XtResizeWidget (w->label.draw_widget,
					w->core.width, w->core.height,
					w->label.draw_widget->core.border_width);
		XtMoveWidget (w->label.draw_widget, x, y);
	}
	if (w->label.flag_widget)
	{
		XtMoveWidget (w->label.flag_widget, flag_x, flag_y);
	}
	if (w->simple.border_widget)
	{
		XtResizeWidget (w->simple.border_widget,
				w->core.width,
				w->core.height,
				w->simple.border_widget->core.border_width);
	}
}

/*
 * CalcLocation -
 *	Figure out where the draw & flag widget should be placed
 *	based on current size of the Label and the current size of
 *	this widget using the specified gravity.
 */

static void
CalcLocation (gw, x_draw_ret, y_draw_ret, x_flag_ret, y_flag_ret)
	Widget          gw;		/* the Label widget */
	Position	*x_draw_ret;	/* returned x & y for draw widget */
	Position	*y_draw_ret;
	Position	*x_flag_ret;	/* returned x & y for flag widget */
	Position	*y_flag_ret;
{
	XoLabelWidget   w = (XoLabelWidget) gw;
	Position        x, y;		/* new x and y coordinate */
	Dimension       width;		/* width and height of draw */
	Dimension       height;
	Dimension       border_width;	/* width of border objects borders */
	Dimension	flag_width;
	Dimension	flag_height;
	int		my_width;
	int		my_height;

	if (!w || !w->label.draw_widget)
		return;
	if (w->simple.border_widget)
		border_width = w->simple.border_widget->core.border_width;
	else
		border_width = 0;
	if (w->label.flag_widget)
	{
		flag_width = w->label.flag_widget->core.width
			+ 2 * w->label.flag_widget->core.border_width;
		flag_height = w->label.flag_widget->core.height
			+ 2 * w->label.flag_widget->core.border_width;
	}
	else
	{
		flag_width = flag_height = 0;
	}


	my_width = w->core.width - flag_width - 2*border_width;
	my_height = w->core.height - 2*border_width;
	width = w->label.draw_widget->core.width
		+ 2*w->label.draw_widget->core.border_width;
	height = w->label.draw_widget->core.height
		+ 2*w->label.draw_widget->core.border_width;
	if (w->label.flag_justify == XoJUST_LEFT ||
	    w->label.flag_justify == XoJUST_CENTER)
	{
		x = flag_width + border_width;
		*x_flag_ret = border_width;
	}
	else
	{
		x = border_width;
		*x_flag_ret = w->core.width - flag_width - border_width;
	}
	*y_flag_ret = (w->core.height - flag_height)/2;
	switch (w->label.gravity)
	{
	case XoNORTH:
		x += (my_width - width) / 2;
		y = border_width;
		break;
	case XoNORTHEAST:
		x += (my_width - width) - border_width;
		y = border_width;
		break;
	case XoEAST:
		x += (my_width - width) - border_width;
		y = border_width + (my_height - height) / 2;
		break;
	case XoSOUTHEAST:
		x += (my_width - width) - border_width;
		y = (my_height - height) - border_width;
		break;
	case XoSOUTH:
		x += (my_width - width) / 2;
		y = (my_height - height) - border_width;
		break;
	case XoSOUTHWEST:
		x += border_width;
		y = (my_height - height) - border_width;
		break;
	case XoWEST:
		x += border_width;
		y = border_width + (my_height - height) / 2;
		break;
	case XoNORTHWEST:
		x += border_width;
		y = border_width;
		break;
	case XoCENTER:
		x += (my_width - width) / 2;
		y = border_width + (my_height - height) / 2;
		break;
	default:
		x += 0;
		y = border_width;
		break;
	}
	*x_draw_ret = x;
	*y_draw_ret = y;
}

static void
CalcSize (w, ret_width, ret_height)
	XoLabelWidget   w;
	Dimension      *ret_width;
	Dimension      *ret_height;
{
	XtGeometryResult result;
	XtWidgetGeometry geo;
	Dimension       border;		/* border width of border Object */
	Dimension	width, height;
	Dimension       draw_width, draw_height;
	Dimension       flag_width, flag_height;

	DBUG_ENTER ("Label.GeometryQuery");
	if (w->simple.border_widget)
		border = w->simple.border_widget->core.border_width;
	else
		border = 0;
	if (!w->label.draw_widget)
	{
		draw_width = 2 * (border ? border : 1);
		draw_height = 2 * (border ? border : 1);
	}
	else
	{
		result = XtQueryGeometry (w->label.draw_widget,
					  (XtWidgetGeometry *) NULL, &geo);
		draw_width = geo.width + 2 * geo.border_width;
		draw_height = geo.height + 2 * geo.border_width;
	}

	if (w->label.flag_widget)
	{
		result = XtQueryGeometry (w->label.flag_widget,
					  (XtWidgetGeometry *) NULL, &geo);
		flag_width = geo.width + 2*geo.border_width;
		flag_height = geo.height + 2*geo.border_width;
	}
	else
	{
		flag_width = 0;
		flag_height = 0;
	}
	width = draw_width + flag_width;
	height = (draw_height > flag_height)?draw_height:flag_height;

	if (ret_width)
		*ret_width = width;
	if (ret_height)
		*ret_height = height;
	
	DBUG_VOID_RETURN;
}

static void
AskNewSize (w, width, height, query)
	XoLabelWidget		w;
	unsigned int		width;
	unsigned int		height;
	int			query;
{
	XtWidgetGeometry request;	/* what we want */
	XtWidgetGeometry reply;		/* what the parent says */
	XtGeometryResult result;
	int             i;
	for (i = 0; i < 5; i++)
	{
		request.request_mode = 0;
		if (w->core.width != width)
		{
			request.width = width;
			request.request_mode |= CWWidth;
		}
		if (w->core.height != height)
		{
			request.height = height;
			request.request_mode |= CWHeight;
		}
		if (query)
		{
			request.request_mode |= XtCWQueryOnly;
		}
		result = XtMakeGeometryRequest ((Widget) w, &request, &reply);
		if (result != XtGeometryAlmost)
			break;
		width = reply.width;
		height = reply.height;
	}
}


/*
 *----------------------------------------------------------------------
 * Printing extensions
 *----------------------------------------------------------------------
 */

static void
LabelPostscript (gw, hardcopy)
	Widget          gw;
	XoHardCopy     *hardcopy;
{
	XoLabelWidget   w = (XoLabelWidget) gw;

	if (!gw || !hardcopy || !XtIsSubclass (gw, xoLabelWidgetClass))
		return;
	XoHardcopy (w->simple.border_widget, hardcopy, 0);
	XoHardcopy (w->label.draw_widget, hardcopy, 0);
}
