/*
 * traversal - provide the interface to keyboard traversals
 *
 * traversal.c,v 2.1 1992/07/22 16:07:34 pete Exp
 * traversal.c,v
 * Revision 2.1  1992/07/22  16:07:34  pete
 * Use XoNameString() instead of XtName().
 *
 * Revision 2.0  1992/04/23  02:53:09  ware
 * First public release.
 *
 * Revision 1.8  1992/04/23  02:19:16  ware
 * Added several classes.  Worked on geometry management
 *
 * Revision 1.7  1992/03/05  01:12:09  ware
 * Sun cc had problems with casting to (signed), changed to (int)
 *
 * Revision 1.6  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.5  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.4  92/02/04  21:25:33  pete
 * Release 44
 *
 * Revision 1.3  91/11/30  15:51:19  pete
 * Cleaned up some nitpicky compile time warnings.
 *
 * Revision 1.2  1991/08/26  11:59:02  pete
 * Use XoProto() for conditional prototypes.  Working on getting traversals
 * and menus to work more efficiently.  Changed to following naming
 * conventions.
 *
 * Revision 1.1  91/07/19  00:58:43  pete
 * Initial revision
 *
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Xmu/CharSet.h>
#include <X11/CompositeP.h>
#include <X11/Xo/CompBase.h>
#include <X11/Xo/XoP.h>
#include <X11/Xo/dbug.h>

XoProto (static Boolean, CheckChild, (Widget gw, Time When, XoDirection direction));

typedef struct _str_direction_convert
{
	String          word;
	XoDirection     direction;
	XrmQuark        quark;
}               XoStrDirConvert;

static XoStrDirConvert dir_convert[] =
{
	"traverse_previous", XoTRAVERSE_PREVIOUS, NULLQUARK,
	"xotraverse_previous", XoTRAVERSE_PREVIOUS, NULLQUARK,
	"previous", XoTRAVERSE_PREVIOUS, NULLQUARK,
	"prev", XoTRAVERSE_PREVIOUS, NULLQUARK,
	"traverse_next", XoTRAVERSE_NEXT, NULLQUARK,
	"xotraverse_next", XoTRAVERSE_NEXT, NULLQUARK,
	"next", XoTRAVERSE_NEXT, NULLQUARK,
	"traverse_home", XoTRAVERSE_HOME, NULLQUARK,
	"xotraverse_home", XoTRAVERSE_HOME, NULLQUARK,
	"home", XoTRAVERSE_HOME, NULLQUARK,
	"traverse_end", XoTRAVERSE_END, NULLQUARK,
	"xotraverse_end", XoTRAVERSE_END, NULLQUARK,
	"end", XoTRAVERSE_END, NULLQUARK,
	"traverse_up", XoTRAVERSE_UP, NULLQUARK,
	"xotraverse_up", XoTRAVERSE_UP, NULLQUARK,
	"up", XoTRAVERSE_UP, NULLQUARK,
	"traverse_down", XoTRAVERSE_DOWN, NULLQUARK,
	"xotraverse_down", XoTRAVERSE_DOWN, NULLQUARK,
	"down", XoTRAVERSE_DOWN, NULLQUARK,
	"traverse_left", XoTRAVERSE_LEFT, NULLQUARK,
	"xotraverse_left", XoTRAVERSE_LEFT, NULLQUARK,
	"left", XoTRAVERSE_LEFT, NULLQUARK,
	"traverse_right", XoTRAVERSE_RIGHT, NULLQUARK,
	"xotraverse_right", XoTRAVERSE_RIGHT, NULLQUARK,
	"right", XoTRAVERSE_RIGHT, NULLQUARK,
};

/*
 * _XoDirectionFromString - Convert a String into an XoDirection enumerated
 *		type.
 */

XoDirection
_XoDirectionFromString (where)
	String          where;
{
	int             i;
	XrmQuark        q;
	char            lower[512];
	XoDirection     direction;
	static Boolean  dir_inited;	/* if we've initialized the quarks */

	if (!where || !*where)
		return XoTRAVERSE_PREVIOUS;
	if (!dir_inited)
	{
		for (i = 0; i < XtNumber (dir_convert); i++)
		{
			dir_convert[i].quark =
				XrmStringToQuark (dir_convert[i].word);
		}
		dir_inited = True;
	}
	XmuCopyISOLatin1Lowered (lower, where);
	q = XrmStringToQuark (lower);
	direction = XoTRAVERSE_PREVIOUS;
	for (i = 0; i < XtNumber (dir_convert); i++)
	{
		if (q == dir_convert[i].quark)
		{
			direction = dir_convert[i].direction;
			break;
		}
	}
	return (direction);
}

String
_XoStringFromDirection (direction)
	XoDirection     direction;
{
	String          ret = NULL;

	/*
	 * Do not add a "default" case because this way we can catch, at
	 * least with some compilers (gcc), when a new traversal type is
	 * added to the enum and we do not have a case statement for it.
	 */
	switch (direction)
	{
	case XoTRAVERSE_NEXT:
		ret = "next";
		break;
	case XoTRAVERSE_PREVIOUS:
		ret = "previous";
		break;
	case XoTRAVERSE_LEFT:
		ret = "left";
		break;
	case XoTRAVERSE_RIGHT:
		ret = "right";
		break;
	case XoTRAVERSE_HOME:
		ret = "home";
		break;
	case XoTRAVERSE_END:
		ret = "end";
		break;
	case XoTRAVERSE_UP:
		ret = "up";
		break;
	case XoTRAVERSE_DOWN:
		ret = "down";
		break;
	}
	return (ret);
}

/*
 * XoTraversalString - Takes a string argument indicating direction
 *		and executes a traversal to the appropriate next widget.
 */

Boolean
XoTraversalString (gw, where, when)
	Widget          gw;
	String          where;
	Time            when;
{
	if (!gw)
	{
		_XoWarn (gw, "XoTraversalString", "badArgument", "Widget is NULL");
		return False;
	}
	if (!where)
	{
		_XoWarn (gw, "XoTraversalString", "badArgument", "Direction is NULL");
		return False;
	}
	return XoTraverse (gw, _XoDirectionFromString (where), when);
}

/*
 * XoTraverse - Finds a parent that knows about traversal and executes
 *		the traverse callback to accomplish it.  Passes along
 *		the direction and child widget with current focus (this
 *		may have changed as the ancestors  were visited to find a
 *		parent that knew how to do a traversal) in an XoTraverseInfo
 *		pointer.
 */

Boolean
XoTraverse (gw, direction, when)
	Widget          gw;
	XoDirection     direction;
	Time            when;
{
	Widget          parent;		/* widget with traverse callback */
	Widget          startfrom;	/* widget to traverse from */
#ifndef DBUG_OFF
	char		buf1[80];
	char		buf2[80];
	char		buf3[80];
#endif

	DBUG_ENTER ("XoTraverse");
	if (!gw)
	{
		_XoWarn (gw, "XoTraverse", "badArgument", "Widget is NULL.");
		DBUG_RETURN (False);
	}
	startfrom = gw;
	for (parent = XtParent (gw); parent; parent = XtParent (parent))
	{
		if (XtHasCallbacks (parent, XtNtraverse) == XtCallbackHasSome)
			break;
		else
			startfrom = parent;
	}
	if (!parent)
	{
		DBUG_PRINT ("traverse", ("No parent of %s has traverse callbacks",
					 XoName (gw)));
		DBUG_RETURN (False);
	}
	DBUG_PRINT ("traverse", ("Started at %s.  Telling %s to find %s relative to %s",
				 XoNameString (gw, buf1, sizeof (buf1)),
				 XoNameString (parent, buf2, sizeof (buf2)),
				 _XoStringFromDirection (direction),
				 XoNameString (startfrom,buf3, sizeof (buf3))));
	DBUG_RETURN (XoTraverseWidget (parent, startfrom, direction, when));
}

Boolean
XoTraverseWidget (gw, startfrom, direction, when)
	Widget          gw;		/* widget to traverse */
	Widget          startfrom;	/* widget we started with */
	XoDirection     direction;	/* which way to move */
	Time            when;		/* when event occurred */
{
	XoTraverseInfo  info;
	Widget          old_focus;

	DBUG_ENTER ("XoTraverseWidget");
	if (!gw)
		DBUG_RETURN (False);
	if (!startfrom)
		startfrom = gw;
	info.t_start = startfrom;
	info.t_direction = direction;
	info.t_time = when;
	info.t_found = False;
	old_focus = XoFocusWidgetSet (XoShellGet (gw), (Widget) NULL);
	DBUG_PRINT ("traverse", ("Executing callbacks in %s", XoName (gw)));
	XtCallCallbacks (gw, XtNtraverse, (XtPointer) &info);
	if (XoFocusWidgetGet (XoShellGet (gw)) == NULL && old_focus != NULL)
		XoFocusWidgetSet (XoShellGet (old_focus), old_focus);
	DBUG_RETURN (info.t_found);
}


/*
 * _XoTraverseCallback - A conveniant default to install
 *		as the traverse callback.  It finds the child and
 *		in the list of children and changes to the appropriate
 *		next one.
 */

void
_XoTraverseCallback (gw, client_data, call_data)
	Widget          gw;
	XtPointer       client_data;
	XtPointer       call_data;
{
	Widget          change_to;	/* next widget to get focus */
	XoTraverseInfo *info;
	CompositeWidget w = (CompositeWidget) gw;
	int             i;

	DBUG_ENTER ("_XoTraverseCallback");
	if (!(info = (XoTraverseInfo *) call_data))
	{
		_XoWarn (gw, "_XoTraverseCallback", "badArgument", "call_data is NULL");
		DBUG_VOID_RETURN;
	}
	if (!XtIsComposite (gw))
	{
		_XoWarn (gw, "_XoTraverseCallback", "badParent", "This is a default traverse callback that only works with composites.");
		DBUG_VOID_RETURN;
	}
	change_to = NULL;	/* if this is not set, ask parent */
	/*
	 * Do not add a "default" case because this way we can catch, at
	 * least with some compilers (gcc), when a new traversal type is
	 * added to the enum and we do not have a case statement for it.
	 */
	switch (info->t_direction)
	{
	case XoTRAVERSE_DOWN:
	case XoTRAVERSE_RIGHT:
	case XoTRAVERSE_NEXT:
		i = _XoChildFind (gw, info->t_start);
		change_to = _XoTraverseAfter (gw, i + 1, info->t_time);
		break;
		break;
	case XoTRAVERSE_UP:
	case XoTRAVERSE_LEFT:
	case XoTRAVERSE_PREVIOUS:
		i = _XoChildFind (gw, info->t_start);
		change_to = _XoTraverseBefore (gw, i - 1, info->t_time);
		break;
	case XoTRAVERSE_HOME:
		change_to = _XoTraverseAfter (gw, 0, info->t_time);
		break;
	case XoTRAVERSE_END:
		change_to = _XoTraverseBefore (gw, (int) w->composite.num_children - 1, info->t_time);
		break;
	default:
		_XoWarn (gw, "_XoTraverseCallback", "badArgument", "Unknown direction");
		break;
	}
	if (change_to)
		info->t_found = True;
	else
	{
		if (XoTraverse (gw, info->t_direction, info->t_time))
			info->t_found = True;
		else
		{
			/*
			 * Do not add a "default" case because this way we
			 * can catch, at least with some compilers (gcc),
			 * when a new traversal type is added to the enum and
			 * we do not have a case statement for it.
			 */
			switch (info->t_direction)
			{
			case XoTRAVERSE_HOME:
			case XoTRAVERSE_NEXT:
			case XoTRAVERSE_RIGHT:
				change_to = _XoTraverseAfter (gw, 0, info->t_time);
				break;
			case XoTRAVERSE_END:
			case XoTRAVERSE_PREVIOUS:
			case XoTRAVERSE_DOWN:
			case XoTRAVERSE_UP:
			case XoTRAVERSE_LEFT:
				change_to = _XoTraverseBefore (gw, (int) w->composite.num_children - 1, info->t_time);
				break;
			}
		}
		if (change_to)
			info->t_found = True;
	}
	DBUG_VOID_RETURN;
}

Widget
_XoTraverseBefore (gw, indx, when)
	Widget          gw;
	int             indx;
	Time            when;
{
	int             i;
	CompositeWidget w = (CompositeWidget) gw;

	if (indx >= w->composite.num_children)
		indx = w->composite.num_children - 1;
	for (i = indx; i >= 0; i--)
	{
		if (CheckChild (w->composite.children[i], when, XoTRAVERSE_END))
			return w->composite.children[i];
	}
	return ((Widget) NULL);
}

Widget
_XoTraverseAfter (gw, indx, when)
	Widget          gw;
	int             indx;
	Time            when;
{
	int             i;
	CompositeWidget w = (CompositeWidget) gw;

	if (indx < 0)
		indx = 0;
	for (i = indx; i < w->composite.num_children; i++)
	{
		if (CheckChild (w->composite.children[i], when, XoTRAVERSE_HOME))
			return w->composite.children[i];
	}
	return ((Widget) NULL);
}

static Boolean
CheckChild (gw, when, direction)
	Widget          gw;
	Time            when;
	XoDirection     direction;
{
	XoTraverseInfo  info;
	Boolean         found = False;

	DBUG_ENTER ("CheckChild");

	if (!XtIsManaged (gw))
		DBUG_RETURN (False);
	if (XtIsComposite (gw)
	    && (XtHasCallbacks (gw, XtNtraverse) == XtCallbackHasSome))
	{
		info.t_start = gw;
		info.t_direction = direction;
		info.t_time = when;
		info.t_found = False;
		XtCallCallbacks (gw, XtNtraverse, (XtPointer) &info);
		found = info.t_found;
	}
	else if (XtCallAcceptFocus (gw, &when))
	{
		found = True;
	}
	if (found)
	{
		DBUG_PRINT ("traverse", ("Set focus to %s", XoName (gw)));
	}
	DBUG_RETURN (found);
}
