# GPL License and Copyright Notice ============================================
#  This file is part of Wrye Bash.
#
#  Wrye Bash is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  Wrye Bash is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with Wrye Bash; if not, write to the Free Software Foundation,
#  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#  Wrye Bash copyright (C) 2005, 2006, 2007 Wrye 
#
# =============================================================================

"""This module provides the GUI interface for Wrye Bash. (However, the Wrye
Bash application is actually launched by the bash module.)

The module is generally organized starting with lower level elements, working 
up to higher level elements (up the BashApp). This is followed by definition
of menus and buttons classes, and finally by a several initialization functions.

Non-GUI objects and functions are provided by the bosh module. Of those, the 
primary objects used are the plugins, modInfos and saveInfos singletons -- each 
representing external data structures (the plugins.txt file and the Data and 
Saves directories respectively). Persistent storage for the app is primarily 
provided through the settings singleton (however the modInfos singleton also
has its own data store)."""

# Imports ---------------------------------------------------------------------
#--Localization
#..Handled by bosh, so import that.
import bush,bosh
from bosh import _, Path, SubProgress, deprint, delist

#--Python
import cStringIO
import copy
import os
import re
import string
import struct
import sys
import textwrap
import time
from types import *

#--wxPython
import wx
from wx.lib.mixins.listctrl import ListCtrlAutoWidthMixin
from wx.lib.evtmgr import eventManager

#--Internet Explorer 
#  - Make sure that python root directory is in PATH, so can access dll's.
if sys.prefix not in set(os.environ['PATH'].split(';')):
    os.environ['PATH'] += ';'+sys.prefix

# Singletons ------------------------------------------------------------------
statusBar = None
modList = None
modDetails = None
saveList = None
saveDetails = None
screensList = None
gMessageList = None
bashFrame = None
docBrowser = None

# Settings --------------------------------------------------------------------
settings = None

#--Load config/defaults
settingDefaults = {
    #--Wrye Bash
    'bash.version': 0,
    'bash.readme': (0,'0'),
    'bash.framePos': (-1,-1),
    'bash.frameSize': (600,500),
    'bash.frameSize.min': (400,500),
    'bash.page':1,
    #--Wrye Bash: Windows
    'bash.window.sizes': {},
    #--Wrye Bash: Load Lists
    'bash.loadLists.data': {
        'Bethesda ESMs': [
            Path('Oblivion.esm'),
            ],
        },
    #--Wrye Bash: Statistics
    'bash.fileStats.cols': ['Type','Count','Size'],
    'bash.fileStats.sort': 'Type',
    'bash.fileStats.colReverse': {
        'Count':1,
        'Size':1,
        },
    'bash.fileStats.colWidths': {
        'Type':50,
        'Count':50,
        'Size':75,
        },
    'bash.fileStats.colAligns': {
        'Count':1,
        'Size':1,
        },
    #--Wrye Bash: Group and Rating
    'bash.mods.groups': ['Body','Bethesda','Clothes','Creature','Fix','Last','Test','Game','GFX','Location','Misc.','NPC','Quest','Race','Resource','Sound'],
    'bash.mods.ratings': ['+','1','2','3','4','5','=','~'],
    #--Wrye Bash: Col (Sort) Names
    'bash.colNames': {
        'Author': _('Author'),
        'Cell': _('Cell'),
        'Date': _('Date'),
        'Day': _('Day'),
        'File': _('File'),
        'Group': _('Group'),
        'Load Order': _('Load Order'),
        'Modified': _('Modified'),
        'Num': _('MI'),
        'PlayTime':_('Hours'),
        'Player': _('Player'),
        'Rating': _('Rating'),
        'Rating':_('Rating'),
        'Save Order': _('Save Order'),
        'Size': _('Size'),
        'Status': _('Status'),
        'Subject': _('Subject'),
        },
    #--Wrye Bash: Masters
    'bash.masters.cols': ['File','Num'],
    'bash.masters.esmsFirst': 1,
    'bash.masters.selectedFirst': 0,
    'bash.masters.sort': 'Save Order',
    'bash.masters.colReverse': {},
    'bash.masters.colWidths': {
        'File':80,
        'Num':20,
        },
    'bash.masters.colAligns': {
        'Num':1,
        },
    #--Wrye Bash: Mod Docs
    'bash.modDocs.show': False,
    'bash.modDocs.size': (300,400),
    'bash.modDocs.pos': wx.DefaultPosition,
    'bash.modDocs.dir': None,
    #--Wrye Bash: Mods
    'bash.mods.cols': ['File','Load Order','Rating','Group','Modified','Size','Author'],
    'bash.mods.esmsFirst': 1,
    'bash.mods.selectedFirst': 0,
    'bash.mods.sort': 'File',
    'bash.mods.colReverse': {},
    'bash.mods.colWidths': {
        'File':200,
        'Rating':20,
        'Group':20,
        'Rating':20,
        'Modified':150,
        'Size':75,
        'Load Order':20,
        'Author':100,
        },
    'bash.mods.colAligns': {
        'Size':1,
        'Load Order':1,
        },
    'bash.mods.renames': {},
    #--Wrye Bash: Saves
    'bash.saves.cols': ['File','Modified','Size','PlayTime','Player','Cell'],
    'bash.saves.sort': 'Modified',
    'bash.saves.colReverse': {
        'Modified':1,
        },
    'bash.saves.colWidths': {
        'File':150,
        'Modified':150,
        'Size':75,
        'Player':100,
        'PlayTime':75,
        'Cell':150,
        'Day':30,
        },
    'bash.saves.colAligns': {
        'Size':1,
        'PlayTime':1,
        },
    #--Wrye Bash: Replacers
    'bash.replacers.cols': ['File'],
    'bash.replacers.sort': 'File',
    'bash.replacers.colReverse': {
        },
    'bash.replacers.colWidths': {
        'File':150,
        },
    'bash.replacers.colAligns': {},
    'bash.replacers.autoChecked':False,
    #--Wrye Bash: Screens
    'bash.screens.cols': ['File'],
    'bash.screens.sort': 'File',
    'bash.screens.colReverse': {
        'Modified':1,
        },
    'bash.screens.colWidths': {
        'File':150,
        'Modified':150,
        'Size':75,
        },
    'bash.screens.colAligns': {},
    #--Wrye Bash: Messages
    'bash.messages.cols': ['Subject','Author','Date'],
    'bash.messages.sort': 'Date',
    'bash.messages.colReverse': {
        },
    'bash.messages.colWidths': {
        'Subject':250,
        'Author':100,
        'Date':150,
        },
    'bash.messages.colAligns': {},
    }

# Exceptions ------------------------------------------------------------------
class BashError(bosh.BoshError):
    pass

class InterfaceError(BashError):
    pass

# Gui Ids ---------------------------------------------------------------------
#------------------------------------------------------------------------------
class IdList:
    """List of ids.

    Id (baseId) through (baseId+size-1) are available for dynamic use through iter(IdList).
    names, if provided as assigned numbers (baseId+size+names.index(name))"""
    def __init__(self,baseId,size,*names):
        self.BASE = baseId
        self.MAX = baseId + size - 1
        #--Extra
        nextNameId = baseId + size
        for name in names:
            setattr(self,name,nextNameId)
            nextNameId += 1

    def __iter__(self):
        """Return iterator."""
        for id in range(self.BASE,self.MAX+1):
            yield id

#------------------------------------------------------------------------------
# Constants
#--Indexed
wxListAligns = [wx.LIST_FORMAT_LEFT, wx.LIST_FORMAT_RIGHT, wx.LIST_FORMAT_CENTRE]

#--Generic
ID_RENAME = 6000
ID_SET    = 6001
ID_SELECT = 6002
ID_BROWSER = 6003
#ID_NOTES  = 6004
ID_EDIT   = 6005
ID_BACK   = 6006
ID_NEXT   = 6007

#--File Menu
ID_REVERT_BACKUP = 6100
ID_REVERT_FIRST  = 6101
ID_BACKUP_NOW    = 6102

#--Label Menus
ID_LOADERS   = IdList(10000,90,'SAVE','EDIT','NONE') 
#ID_REMOVERS  = IdList(10100,90,'EDIT','EDIT_CELLS')
#ID_REPLACERS = IdList(10200,90,'EDIT')
ID_GROUPS    = IdList(10300,90,'EDIT','NONE')
ID_RATINGS   = IdList(10400,90,'EDIT','NONE')
ID_PROFILES  = IdList(10500,90,'EDIT','DEFAULT')

# GUI Utils -------------------------------------------------------------------
def fill(text,width=60):
    """Wraps paragraph to width characters."""
    pars = [textwrap.fill(text,width) for text in text.split('\n')]
    return '\n'.join(pars)

def tooltip(text,wrap=50):
    """Returns tolltip with wrapped copy of text."""
    text = textwrap.fill(text,wrap)
    return wx.ToolTip(text)

def bitmapButton(buttonArgs,onClick=None,tip=None):
    """Creates a button, binds click function, then returns bound button."""
    gButton = wx.BitmapButton(*buttonArgs)
    if onClick: gButton.Bind(wx.EVT_BUTTON,onClick)
    if tip: gButton.SetToolTip(tooltip(tip))
    return gButton

def button(buttonArgs,onClick=None,tip=None):
    """Creates a button, binds click function, then returns bound button."""
    gButton = wx.Button(*buttonArgs)
    if onClick: gButton.Bind(wx.EVT_BUTTON,onClick)
    if tip: gButton.SetToolTip(tooltip(tip))
    return gButton

def checkBox(checkBoxArgs,onCheck=None,tip=None):
    """Creates a checkBox, binds check function, then returns bound button."""
    gCheckBox = wx.CheckBox(*checkBoxArgs)
    if onCheck: gCheckBox.Bind(wx.EVT_CHECKBOX,onCheck)
    if tip: gCheckBox.SetToolTip(tooltip(tip))
    return gCheckBox

def leftSash(parent,defaultSize=(100,100),onSashDrag=None):
    """Creates a left sash window."""
    sash = wx.SashLayoutWindow(parent,style=wx.SW_3D)
    sash.SetDefaultSize(defaultSize)
    sash.SetOrientation(wx.LAYOUT_VERTICAL)
    sash.SetAlignment(wx.LAYOUT_LEFT)
    sash.SetSashVisible(wx.SASH_RIGHT, True)
    if onSashDrag:
        id = sash.GetId()
        sash.Bind(wx.EVT_SASH_DRAGGED_RANGE, onSashDrag,id=id,id2=id)
    return sash

def topSash(parent,defaultSize=(100,100),onSashDrag=None):
    """Creates a top sash window."""
    sash = wx.SashLayoutWindow(parent,style=wx.SW_3D)
    sash.SetDefaultSize(defaultSize)
    sash.SetOrientation(wx.LAYOUT_HORIZONTAL)
    sash.SetAlignment(wx.LAYOUT_TOP)
    sash.SetSashVisible(wx.SASH_BOTTOM, True)
    if onSashDrag:
        id = sash.GetId()
        sash.Bind(wx.EVT_SASH_DRAGGED_RANGE, onSashDrag,id=id,id2=id)
    return sash

def aSizer(sizer,*elements):
    """Adds elements to a sizer."""
    for element in elements:
        if isinstance(element,tuple):
            if element[0] != None:
                sizer.Add(*element)
        elif element != None:
            sizer.Add(element)
    return sizer

def hSizer(*elements):
    """Creates and returns a horizontal sizer, adding elements."""
    return aSizer(wx.BoxSizer(wx.HORIZONTAL),*elements)

def vSizer(*elements):
    """Creates and returns a vertical sizer, adding elements."""
    return aSizer(wx.BoxSizer(wx.VERTICAL),*elements)

def hsbSizer(boxArgs,*elements):
    """Creates and returns a horizontal static box sizer, adding elements."""
    return aSizer(wx.StaticBoxSizer(wx.StaticBox(*boxArgs),wx.HORIZONTAL),*elements)

def vsbSizer(boxArgs,*elements):
    """Creates and returns a vertical static box sizer, adding elements."""
    return aSizer(wx.StaticBoxSizer(wx.StaticBox(*boxArgs),wx.VERTICAL),*elements)

# Message Dialogs -------------------------------------------------------------
def Message(parent,message,title='',style=wx.OK):
    """Shows a modal MessageDialog. 
    Use ErrorMessage, WarningMessage or InfoMessage."""
    dialog = wx.MessageDialog(parent,message,title,style)
    result = dialog.ShowModal()
    dialog.Destroy()
    return result

def ListMessage(parent,header,items,maxItems=0,title='',style=wx.OK):
    """Formats a list of items into a message for use in a Message."""
    numItems = len(items)
    if maxItems <= 0: maxItems = numItems
    message = string.Template(header).substitute(count=numItems)
    message += '\n* '+'\n* '.join(items[:min(numItems,maxItems)])
    if numItems > maxItems:
        message += _('\n(And %d others.)') % (numItems - maxItems,)
    return Message(parent,message,title,style)

def ErrorMessage(parent,message,title=_('Error'),style=(wx.OK|wx.ICON_HAND)):
    """Shows a modal error message."""
    return Message(parent,message,title,style)

def WarningMessage(parent,message,title=_('Warning'),style=(wx.OK|wx.ICON_EXCLAMATION)):
    """Shows a modal warning message."""
    return Message(parent,message,title,style)
    
def InfoMessage(parent,message,title=_('Information'),style=(wx.OK|wx.ICON_INFORMATION)):
    """Shows a modal information message."""
    return Message(parent,message,title,style)

def YesQuery(parent,message,title='',style=wx.YES_NO|wx.ICON_EXCLAMATION,default=True):
    """Shows a modal warning message."""
    default = (wx.NO_DEFAULT,wx.YES_DEFAULT)[default]
    result = Message(parent,message,title,style=style|default)
    return result in (wx.ID_YES,wx.ID_OK)
    
def LogMessageOnClose(evt=None):
    """Handle log message closing."""
    window = evt.GetEventObject()
    if not window.IsIconized() and not window.IsMaximized():
        settings['bash.message.log.pos'] = window.GetPositionTuple()
        settings['bash.message.log.size'] = window.GetSizeTuple()
    window.Destroy()

def LogMessage(parent,logText,title='',style=0,asDialog=True,fixedFont=False):
    """Display text in a log window"""
    #--Sizing
    pos = settings.get('bash.message.log.pos',wx.DefaultPosition)
    size = settings.get('bash.message.log.size',(400,400))
    #--Dialog or Frame
    if asDialog:
        window = wx.Dialog(parent,-1,title,pos=pos,size=size,
            style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)
    else:
        window = wx.Frame(parent,-1,title,pos=pos,size=size,
            style= (wx.RESIZE_BORDER | wx.CAPTION | wx.SYSTEM_MENU | wx.CLOSE_BOX | wx.CLIP_CHILDREN))
        window.SetIcons(images['bash.icons2'].GetIconBundle())
    window.SetSizeHints(200,200)
    window.Bind(wx.EVT_CLOSE,LogMessageOnClose)
    window.SetBackgroundColour(wx.NullColour) #--Bug workaround to ensure that default colour is being used.
    #--Text
    textCtrl = wx.TextCtrl(window,-1,logText,style=wx.TE_READONLY|wx.TE_MULTILINE|wx.TE_RICH2|wx.SUNKEN_BORDER  )
    if fixedFont:
        fixedFont = wx.SystemSettings_GetFont(wx.SYS_ANSI_FIXED_FONT )
        fixedFont.SetPointSize(8)
        fixedStyle = wx.TextAttr()
        #fixedStyle.SetFlags(0x4|0x80)
        fixedStyle.SetFont(fixedFont)
        textCtrl.SetStyle(0,textCtrl.GetLastPosition(),fixedStyle)
    #--Buttons
    gOkButton = button((window,wx.ID_OK),lambda event: window.Close())
    gOkButton.SetDefault()
    #--Layout
    window.SetSizer(
        vSizer(
            (textCtrl,1,wx.EXPAND|wx.ALL^wx.BOTTOM,2),
            (gOkButton,0,wx.ALIGN_RIGHT|wx.ALL,4),
            )
        )
    #--Show
    if asDialog:
        window.ShowModal()
        window.Destroy()
    else:
        window.Show()

def WtxtLogMessage(parent,logText,title='',style=0,asDialog=True):
    """Convert logText from wtxt to html and display. Optionally, logText can be path to an html file."""
    import wx.lib.iewin
    import wtxt
    #--Sizing
    pos = settings.get('bash.message.log.pos',wx.DefaultPosition)
    size = settings.get('bash.message.log.size',(400,400))
    #--Dialog or Frame
    if asDialog:
        window = wx.Dialog(parent,-1,title,pos=pos,size=size,
            style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)
    else:
        window = wx.Frame(parent,-1,title,pos=pos,size=size,
            style= (wx.RESIZE_BORDER | wx.CAPTION | wx.SYSTEM_MENU | wx.CLOSE_BOX | wx.CLIP_CHILDREN))
        window.SetIcons(images['bash.icons2'].GetIconBundle())
    window.SetSizeHints(200,200)
    window.Bind(wx.EVT_CLOSE,LogMessageOnClose)
    #--Text
    textCtrl = wx.lib.iewin.IEHtmlWindow(window, -1, style = wx.NO_FULL_REPAINT_ON_RESIZE)
    if not isinstance(logText,Path):
        logPath = bosh.dirs['saveBase'].join('LogTemp.html')
        ins = cStringIO.StringIO(logText+'\n{{CSS:wtxt_sand_small.css}}')
        out = logPath.open('w')
        cssDir = bosh.modInfos.dir.join('Docs')
        wtxt.getHtml(ins,out,('',cssDir))
        out.close()
        logText = logPath
    textCtrl.Navigate(logText,0x2) #--0x2: Clear History
    #--Buttons
    bitmap = wx.ArtProvider_GetBitmap(wx.ART_GO_BACK,wx.ART_HELP_BROWSER, (16,16))
    gBackButton = bitmapButton((window,-1,bitmap),lambda evt: textCtrl.GoBack())
    bitmap = wx.ArtProvider_GetBitmap(wx.ART_GO_FORWARD,wx.ART_HELP_BROWSER, (16,16))
    gForwardButton = bitmapButton((window,-1,bitmap),lambda evt: textCtrl.GoForward())
    gOkButton = button((window,wx.ID_OK),lambda event: window.Close())
    gOkButton.SetDefault()
    #--Layout
    window.SetSizer(
        vSizer(
            (textCtrl,1,wx.EXPAND|wx.ALL^wx.BOTTOM,2),
            (hSizer(
                gBackButton,
                gForwardButton,
                ((0,0),1),
                gOkButton,
                ),0,wx.ALL|wx.EXPAND,4),
            )
        )
    #--Show
    if asDialog:
        window.ShowModal()
        window.Destroy()
    else:
        window.Show()

def ContinueQuery(parent,message,continueKey,title=_('Warning')):
    """Shows a modal continue query if value of continueKey is false. Returns True to continue.
    Also provides checkbox "Don't show this in future." to set continueKey to true."""
    #--ContinueKey set?
    if settings.get(continueKey): 
        return wx.ID_OK
    #--Generate/show dialog
    dialog = wx.Dialog(parent,-1,title,size=(350,200),style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)
    icon = wx.StaticBitmap(dialog,-1,
        wx.ArtProvider_GetBitmap(wx.ART_WARNING,wx.ART_MESSAGE_BOX, (32,32)))
    staticText = wx.StaticText(dialog,-1,message,style=wx.ST_NO_AUTORESIZE)
    checkBox = wx.CheckBox(dialog,-1,_("Don't show this in the future."))
    #--Layout
    sizer = vSizer(
        (hSizer(
            (icon,0,wx.ALL,6),
            (staticText,1,wx.EXPAND|wx.LEFT,6),
            ),1,wx.EXPAND|wx.ALL,6),
        (checkBox,0,wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM,6),
        (hSizer( #--Save/Cancel
            ((0,0),1),
            button((dialog,wx.ID_OK)),
            (button((dialog,wx.ID_CANCEL)),0,wx.LEFT,4),
            ),0,wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM,6),
        )
    dialog.SetSizer(sizer)
    #--Get continue key setting and return
    result = dialog.ShowModal()
    if checkBox.GetValue():
        settings[continueKey] = 1
    return result

def TextEntry(parent,message,title='',default=''):
    """Shows a text entry dialog and returns result or None if canceled."""
    dialog = wx.TextEntryDialog(parent,message,title,default)
    if dialog.ShowModal() != wx.ID_OK:
        dialog.Destroy()
        return None
    else:
        value = dialog.GetValue()
        dialog.Destroy()
        return value

#------------------------------------------------------------------------------
def OpenDialog(parent,title='',defaultDir='',defaultFile='',wildcard='',style=wx.OPEN):
    """Show as file dialog and return selected path(s)."""
    dialog = wx.FileDialog(parent,title,defaultDir,defaultFile,wildcard, style )
    if dialog.ShowModal() != wx.ID_OK: 
        result = False
    elif style & wx.MULTIPLE:
        result = map(Path,dialog.GetPaths())
    else:
        result = Path.get(dialog.GetPath())
    dialog.Destroy()
    return result

#------------------------------------------------------------------------------
def MultiOpenDialog(parent,title='',defaultDir='',defaultFile='',wildcard='',style=wx.OPEN|wx.MULTIPLE):
    """Show as save dialog and return selected path(s)."""
    return OpenDialog(parent,title,defaultDir,defaultFile,wildcard,style )

#------------------------------------------------------------------------------
def SaveDialog(parent,title='',defaultDir='',defaultFile='',wildcard='',style=wx.OVERWRITE_PROMPT):
    """Show as save dialog and return selected path(s)."""
    return OpenDialog(parent,title,defaultDir,defaultFile,wildcard,wx.SAVE|style )

#------------------------------------------------------------------------------
def DirDialog(parent,message=_('Choose a directory.'),defaultPath=''):
    """Shows a modal directory dialog and return the resulting path, or None if canceled."""
    dialog = wx.DirDialog(parent,message,defaultPath,style=wx.DD_NEW_DIR_BUTTON)
    if dialog.ShowModal() != wx.ID_OK:
        dialog.Destroy()
        return None
    else:
        path = dialog.GetPath()
        dialog.Destroy()
        return path

#------------------------------------------------------------------------------
class ProgressDialog(bosh.Progress):
    """Prints progress to file (stdout by default)."""
    def __init__(self,title=_('Progress'),message=' '*40,parent=None,
        style=wx.PD_APP_MODAL|wx.PD_ELAPSED_TIME|wx.PD_AUTO_HIDE):
        if sys.version[:3] != '2.4': style |= wx.PD_SMOOTH 
        self.dialog = wx.ProgressDialog(title,message,100,parent,style)
        bosh.Progress.__init__(self)
        self.message = message
        self.isDestroyed = False
        self.prevMessage = ''
        self.prevState = -1
        self.prevTime = 0

    def doProgress(self,state,message):
        if not self.dialog:
            raise InterfaceError(_('Dialog already destroyed.'))
        elif (state == 0 or state == 1 or (message != self.prevMessage) or 
            (state - self.prevState) > 0.05 or (time.time() - self.prevTime) > 0.5):
            if message != self.prevMessage:
                self.dialog.Update(int(state*100),message)
            else:
                self.dialog.Update(int(state*100))
            self.prevMessage = message
            self.prevState = state
            self.prevTime = time.time()

    def Destroy(self):
        if self.dialog:
            self.dialog.Destroy()
            self.dialog = None

# Colors ----------------------------------------------------------------------
class Colours:
    """Colour collection and wrapper for wx.ColourDatabase.

    Provides dictionary syntax access (colours[key]) and predefined colours."""
    def __init__(self): 
        self.data = {}
        self.database = None

    def Init(self):
        self.database = wx.ColourDatabase()
        #--Standard colours
        data = self.data
        data['bash.esm'] = wx.Colour(220,220,255)
        data['bash.doubleTime.not'] = self.database.Find('WHITE')
        data['bash.doubleTime.exists'] = wx.Colour(255,220,220)
        data['bash.doubleTime.load'] = wx.Colour(255,100,100)
        data['bash.exOverLoaded'] = wx.Colour(0xFF,0x99,0)
        data['bash.masters.remapped'] = wx.Colour(100,255,100)
        data['bash.masters.changed'] = wx.Colour(220,255,220)
        data['bash.mods.isMergeable'] = wx.Colour(0x00,0x99,0x00)
        data['bash.mods.groupHeader'] = wx.Colour(0xD8,0xD8,0xD8)
    
    def __getitem__(self,key):
        """Dictionary syntax: color = colours[key]."""
        data = self.data
        if not self.database: self.Init()
        if key not in data: 
            data[key] = self.database.Find(key)
        return data[key]
    
#--Singleton
colours = Colours()

# Images ----------------------------------------------------------------------
images = {}

#------------------------------------------------------------------------------
class Image: 
    """Wrapper for images, allowing access in various formats/classes.

    Allows image to be specified before wx.App is initialized."""
    def __init__(self,file,type=wx.BITMAP_TYPE_ANY ):
        self.file = file
        self.type = type
        self.bitmap = None
        self.icon = None
        if not Path.get(self.file).exists():
            raise bosh.BoshError(_("Missing resource file: %s.") % (self.file,))

    def GetBitmap(self):
        if not self.bitmap:
            self.bitmap = wx.Bitmap(self.file,self.type)
        return self.bitmap
    
    def GetIcon(self):
        if not self.icon:
            self.icon = wx.EmptyIcon()
            self.icon.CopyFromBitmap(self.GetBitmap())
        return self.icon

#------------------------------------------------------------------------------
class ImageBundle: 
    """Wrapper for bundle of images.

    Allows image bundle o be specified before wx.App is initialized."""
    def __init__(self):
        self.images = []
        self.iconBundle = None

    def Add(self,image):
        self.images.append(image)

    def GetIconBundle(self):
        if not self.iconBundle:
            self.iconBundle = wx.IconBundle()
            for image in self.images:
                self.iconBundle.AddIcon(image.GetIcon())
        return self.iconBundle

#------------------------------------------------------------------------------
class ImageList:
    """Wrapper for wx.ImageList.

    Allows ImageList to be specified before wx.App is initialized.
    Provides access to ImageList integers through imageList[key]."""
    def __init__(self,width,height):
        self.width = width
        self.height = height
        self.data = []
        self.indices = {}
        self.imageList = None

    def Add(self,image,key):
        self.data.append((key,image))

    def GetImageList(self):
        if not self.imageList:
            indices = self.indices
            imageList = self.imageList = wx.ImageList(self.width,self.height)
            for key,image in self.data:
                indices[key] = imageList.Add(image.GetBitmap())
        return self.imageList
    
    def __getitem__(self,key):
        self.GetImageList()
        return self.indices[key]

#------------------------------------------------------------------------------
class Checkboxes(ImageList):
    """Checkboxes ImageList. Used by several List classes."""
    def __init__(self):
        ImageList.__init__(self,16,16)
        for status in ('on','off','inc','imp'):
            for color in ('purple','blue','green','orange','yellow','red'):
                shortKey = color+'.'+status
                imageKey = 'checkbox.'+shortKey
                file = r'images\checkbox_'+color+'_'+status+'.png'
                image = images[imageKey] = Image(file,wx.BITMAP_TYPE_PNG)
                self.Add(image,shortKey)

    def Get(self,status,on):
        self.GetImageList()
        if on == 3:
            if status <= -20: shortKey = 'purple.imp'
            elif status <= -10: shortKey = 'blue.imp'
            elif status <= 0: shortKey = 'green.imp'
            elif status <=10: shortKey = 'yellow.imp'
            elif status <=20: shortKey = 'orange.imp'
            else: shortKey = 'red.on'
        elif on == 2:
            if status <= -20: shortKey = 'purple.inc'
            elif status <= -10: shortKey = 'blue.inc'
            elif status <= 0: shortKey = 'green.inc'
            elif status <=10: shortKey = 'yellow.inc'
            elif status <=20: shortKey = 'orange.inc'
            else: shortKey = 'red.on'
        elif on:
            if status <= -20: shortKey = 'purple.on'
            elif status <= -10: shortKey = 'blue.on'
            elif status <= 0: shortKey = 'green.on'
            elif status <=10: shortKey = 'yellow.on'
            elif status <=20: shortKey = 'orange.on'
            else: shortKey = 'red.on'
        else:
            if status <= -20: shortKey = 'purple.off'
            elif status <= -10: shortKey = 'blue.off'
            elif status == 0: shortKey = 'green.off'
            elif status <=10: shortKey = 'yellow.off'
            elif status <=20: shortKey = 'orange.off'
            else: shortKey = 'red.off'
        return self.indices[shortKey]

#------------------------------------------------------------------------------
class Picture(wx.Panel):
    """Picture panel."""
    def __init__(self, parent,width,height,scaling=1):
        """Initialize."""
        wx.Panel.__init__(self, parent, -1,size=(width,height))
        self.scaling=scaling
        self.bitmap = None
        self.scaled = None
        self.oldSize = (0,0)
        self.SetSizeHints(width,height,width,height)
        #--Events
        self.Bind(wx.EVT_PAINT,self.OnPaint)

    def SetBitmap(self,bitmap):
        """Set bitmap."""
        self.bitmap = bitmap
        self.Rescale()
        self.Refresh()

    def Rescale(self):
        """Updates scaled version of bitmap."""
        picWidth,picHeight = self.oldSize = self.GetSizeTuple()
        bitmap = self.scaled = self.bitmap
        if not bitmap: return
        imgWidth,imgHeight = bitmap.GetWidth(),bitmap.GetHeight()
        if self.scaling == 2 or (self.scaling == 1 and (imgWidth > picWidth or imgHeight > picHeight)):
            image = bitmap.ConvertToImage()
            factor = min(1.0*picWidth/imgWidth,1.0*picHeight/imgHeight)
            newWidth,newHeight = int(factor*imgWidth),int(factor*imgHeight)
            self.scaled = image.Scale(newWidth,newHeight).ConvertToBitmap()
        
    def OnPaint(self, event=None):
        """Draw bitmap or clear drawing area."""
        dc = wx.PaintDC(self)
        dc.SetBackground(wx.MEDIUM_GREY_BRUSH)
        if self.scaled:
            if self.GetSizeTuple() != self.oldSize:
                self.Rescale()
            panelWidth,panelHeight = self.GetSizeTuple()
            xPos = max(0,(panelWidth - self.scaled.GetWidth())/2)
            yPos = max(0,(panelHeight - self.scaled.GetHeight())/2)
            dc.Clear()
            dc.DrawBitmap(self.scaled,xPos,yPos,False)
        else:
            dc.Clear()
        dc.SetPen(wx.Pen("BLACK", 1))
        dc.SetBrush(wx.TRANSPARENT_BRUSH)
        (width,height) = self.GetSize()
        dc.DrawRectangle(0,0,width,height)

# Links -----------------------------------------------------------------------
#------------------------------------------------------------------------------
class Links(list):
    """List of menu or button links."""
    class LinksPoint:
        """Point in a link list. For inserting, removing, appending items."""
        def __init__(self,list,index):
            self._list = list
            self._index = index
        def remove(self):
            del self._list[self._index]
        def replace(self,item):
            self._list[self._index] = item
        def insert(self,item):
            self._list.insert(self._index,item)
            self._index += 1
        def append(self,item):
            self._list.insert(self._index+1,item)
            self._index += 1

    #--Access functions:
    def getClassPoint(self,classObj):
        """Returns index"""
        for index,item in enumerate(self):
            if isinstance(item,classObj):
                return Links.LinksPoint(self,index)
        else:
            return None

# Link Singletons
mastersMainMenu = Links()
mastersItemMenu = Links()

modsMainMenu = Links()
modsItemMenu = Links()

replacersMainMenu = Links()
replacersItemMenu = Links()

savesMainMenu = Links()
savesItemMenu = Links()

screensMainMenu = Links()
screensItemMenu = Links()

messagesMainMenu = Links()
messagesItemMenu = Links()

statusBarButtons = Links()

# Windows ---------------------------------------------------------------------
#------------------------------------------------------------------------------
class NotebookPanel(wx.Panel):
    """Parent class for notebook panels."""

    def SetStatusCount(self):
        """Sets status bar count field."""
        statusBar.SetStatusText('',2)

    def OnShow(self):
        """To be called when particular panel is changed to and/or shown for first time.
        Default version does nothing, but derived versions might update data."""
        self.SetStatusCount()

#------------------------------------------------------------------------------
class ListEditorData:
    """Data capsule for ListEditorDialog. [Abstract]"""
    def __init__(self,parent):
        """Initialize."""
        self.parent = parent #--Parent window.
        self.showAdd = False
        self.showEdit = False
        self.showRename = False
        self.showRemove = False
        self.showSave = False
        self.showCancel = False
        self.showInfo = False
        self.caption = None
    #--List
    def getInfo(self,item):
        """Returns string info on specified item."""
        return ''
    def getItemList(self):
        """Returns item list in correct order."""
        raise bosh.AbstractError
        return []
    def add(self):
        """Peforms add operation. Return new item on success."""
        raise bosh.AbstractError
        return None
    def edit(self,item=None):
        """Edits specified item. Return true on success."""
        raise bosh.AbstractError
        return False
    def rename(self,oldItem,newItem):
        """Renames oldItem to newItem. Return true on success."""
        raise bosh.AbstractError
        return False
    def remove(self,item):
        """Removes item. Return true on success."""
        raise bosh.AbstractError
        return False
    def close(self):
        """Called when dialog window closes."""
        pass

    #--Checklist
    def getChecks(self):
        """Returns checked state of items as array of True/False values matching Item list."""
        raise bosh.AbstractError
        return []
    def check(self,item):
        """Checks items. Return true on success."""
        raise bosh.AbstractError
        return False
    def uncheck(self,item):
        """Unchecks item. Return true on success."""
        raise bosh.AbstractError
        return False

    #--Save/Cancel
    def save(self):
        """Handles save button."""
        pass

    def cancel(self):
        """Handles cancel button."""
        pass

#------------------------------------------------------------------------------
class ListEditorDialog(wx.Dialog):
    """Dialog for editing lists."""
    def __init__(self,parent,id,title,data,type='list'):
        #--Data
        self.data = data #--Should be subclass of ListEditorData
        self.items = data.getItemList()
        #--GUI
        wx.Dialog.__init__(self,parent,id,title,
            style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)
        wx.EVT_CLOSE(self, self.OnCloseWindow)
        #--Caption
        if data.caption:
            captionText = wx.StaticText(self,-1,data.caption)
        else:
            captionText = None
        #--List Box
        if type == 'checklist':
            self.list = wx.CheckListBox(self,-1,choices=self.items,style=wx.LB_SINGLE)
            for index,checked in enumerate(self.data.getChecks()):
                self.list.Check(index,checked)
            self.Bind(wx.EVT_CHECKLISTBOX, self.DoCheck, self.list)
        else:
            self.list = wx.ListBox(self,-1,choices=self.items,style=wx.LB_SINGLE)
        self.list.SetSizeHints(125,150)
        #--Buttons and Events
        if data.showInfo:
            self.gInfoBox = wx.TextCtrl(self,-1," ",size=(130,-1),style=wx.TE_READONLY|wx.TE_MULTILINE|wx.SUNKEN_BORDER)
            self.list.Bind(wx.EVT_LISTBOX,self.OnShowInfo)
        else:
            self.gInfoBox = None
        if data.showAdd or data.showEdit or data.showRename or data.showRemove or data.showSave or data.showCancel or data.caption:
            buttons = wx.BoxSizer(wx.VERTICAL)
            if data.showAdd:
                buttons.Add(wx.Button(self,wx.ID_NEW,_('Add')),0,wx.LEFT|wx.TOP,4)
                wx.EVT_BUTTON(self,wx.ID_NEW,self.DoAdd)
            if data.showEdit:
                buttons.Add(wx.Button(self,wx.ID_REPLACE,_('Edit')),0,wx.LEFT|wx.TOP,4)
                wx.EVT_BUTTON(self,wx.ID_REPLACE,self.DoEdit)
            if data.showRename:
                buttons.Add(wx.Button(self,ID_RENAME,_('Rename')),0,wx.LEFT|wx.TOP,4)
                wx.EVT_BUTTON(self,ID_RENAME,self.DoRename)
            if data.showRemove:
                buttons.Add(wx.Button(self,wx.ID_DELETE,_('Remove')),0,wx.LEFT|wx.TOP,4)
                wx.EVT_BUTTON(self,wx.ID_DELETE,self.DoRemove)
            if data.showSave:
                saveName = (data.showSave,_('Save'))[data.showSave == True]
                buttons.Add(wx.Button(self,wx.ID_SAVE,saveName),0,wx.LEFT|wx.TOP,4)
                wx.EVT_BUTTON(self,wx.ID_SAVE,self.OnSave)
            if data.showCancel:
                buttons.Add(button((self,wx.ID_CANCEL),self.OnCancel),0,wx.LEFT|wx.TOP,4)
        else:
            buttons = None
        #--Layout
        sizer = vSizer(
            (captionText,0,wx.LEFT|wx.TOP,4),
            (hSizer(
                (self.list,1,wx.EXPAND|wx.TOP,4),
                (self.gInfoBox,1,wx.EXPAND|wx.TOP,4),
                (buttons,0,wx.EXPAND),
                ),1,wx.EXPAND)
            )
        #--Done
        className = data.__class__.__name__
        if className in settings['bash.window.sizes']:
            self.SetSizer(sizer)
            self.SetSize(settings['bash.window.sizes'][className])
        else:
            self.SetSizerAndFit(sizer)

    def GetSelected(self):
        return self.list.GetNextItem(-1,wx.LIST_NEXT_ALL,wx.LIST_STATE_SELECTED)

    #--Checklist commands
    def DoCheck(self,event):
        """Handles check/uncheck of listbox item."""
        index = event.GetSelection()
        item = self.items[index]
        if self.list.IsChecked(index):
            self.data.check(item)
        else:
            self.data.uncheck(item)
        #self.list.SetSelection(index)

    #--List Commands
    def DoAdd(self,event):
        """Adds a new item."""
        newItem = self.data.add()
        if newItem and newItem not in self.items:
            self.items = self.data.getItemList()
            index = self.items.index(newItem)
            self.list.InsertItems([newItem],index)

    def DoEdit(self,event):
        """Edits the selected item."""
        raise bosh.UncodedError

    def DoRename(self,event):
        """Renames selected item."""
        selections = self.list.GetSelections()
        if not selections:
            wx.Bell()
            return
        #--Rename it
        itemDex = selections[0]
        curName = self.list.GetString(itemDex)
        #--Dialog
        dialog = wx.TextEntryDialog(self,_('Rename to:'),_('Rename'),curName)
        result = dialog.ShowModal()
        #--Okay?
        if result != wx.ID_OK:
            dialog.Destroy()
            return
        newName = dialog.GetValue()
        dialog.Destroy()
        if newName == curName:
            pass
        elif newName in self.items:
            ErrorMessage(self,_('Name must be unique.'))
        elif self.data.rename(curName,newName):
            self.items[itemDex] = newName
            self.list.SetString(itemDex,newName)

    def DoRemove(self,event):
        """Removes selected item."""
        selections = self.list.GetSelections()
        if not selections:
            wx.Bell()
            return
        #--Data
        itemDex = selections[0]
        item = self.items[itemDex]
        if not self.data.remove(item): return
        #--GUI
        del self.items[itemDex]
        self.list.Delete(itemDex)
        
    #--Show Info
    def OnShowInfo(self,event):
        """Handle show info (item select) event."""
        index = event.GetSelection()
        item = self.items[index]
        self.gInfoBox.SetValue(self.data.getInfo(item))

    #--Save/Cancel
    def OnSave(self,event):
        """Handle save button."""
        self.data.save()
        self.EndModal(wx.ID_OK)

    def OnCancel(self,event):
        """Handle save button."""
        self.data.cancel()
        self.EndModal(wx.ID_CANCEL)

    #--Window Closing
    def OnCloseWindow(self, event):
        """Handle window close event.
        Remember window size, position, etc."""
        self.data.close()
        sizes = settings.getChanged('bash.window.sizes')
        sizes[self.data.__class__.__name__] = self.GetSizeTuple()
        self.Destroy()

#------------------------------------------------------------------------------
class ListCtrl(wx.ListCtrl, ListCtrlAutoWidthMixin):
    def __init__(self, parent, ID, pos=wx.DefaultPosition,
                 size=wx.DefaultSize, style=0):
        wx.ListCtrl.__init__(self, parent, ID, pos, size, style=style)
        ListCtrlAutoWidthMixin.__init__(self)

#------------------------------------------------------------------------------
class List(wx.Panel):
    def __init__(self,parent,id=-1,ctrlStyle=(wx.LC_REPORT | wx.LC_SINGLE_SEL)):
        wx.Panel.__init__(self,parent,id, style=wx.WANTS_CHARS)
        sizer = wx.BoxSizer(wx.VERTICAL)
        self.SetSizer(sizer)
        self.SetSizeHints(-1,50)
        #--ListCtrl
        listId = self.listId = wx.NewId()
        self.list = ListCtrl(self, listId, style=ctrlStyle)
        self.checkboxes = images['bash.checkboxes']
        #--Columns
        self.PopulateColumns()
        #--Items
        self.sortDirty = 0
        self.PopulateItems()
        #--Events
        wx.EVT_SIZE(self, self.OnSize)
        #--Events: Items
        self.hitIcon = 0
        wx.EVT_LEFT_DOWN(self.list,self.OnLeftDown)
        wx.EVT_COMMAND_RIGHT_CLICK(self.list, listId, self.DoItemMenu)
        #--Events: Columns
        wx.EVT_LIST_COL_CLICK(self, listId, self.DoItemSort)
        wx.EVT_LIST_COL_RIGHT_CLICK(self, listId, self.DoColumnMenu)
        wx.EVT_LIST_COL_END_DRAG(self,listId, self.OnColumnResize)
    
    #--Items ----------------------------------------------
    #--Populate Columns
    def PopulateColumns(self):
        """Create/name columns in ListCtrl."""
        cols = self.cols
        self.numCols = len(cols)
        for colDex in range(self.numCols):
            colKey = cols[colDex]
            colName = self.colNames.get(colKey,colKey)
            wxListAlign = wxListAligns[self.colAligns.get(colKey,0)]
            self.list.InsertColumn(colDex,colName,wxListAlign)
            self.list.SetColumnWidth(colDex, self.colWidths.get(colKey,30))

    def PopulateItem(self,itemDex,mode=0,selected=set()):
        """Populate ListCtrl for specified item. [ABSTRACT]"""
        raise bosh.AbstractError

    def GetItems(self):
        """Set and return self.items."""
        self.items = self.data.keys()
        return self.items

    def PopulateItems(self,col=None,reverse=-2,selected='SAME'):
        """Sort items and populate entire list."""
        #--Sort Dirty?
        if self.sortDirty:
            self.sortDirty = 0
            (col, reverse) = (None,-1)
        #--Items to select afterwards. (Defaults to current selection.)
        if selected == 'SAME': selected = set(self.GetSelected())
        #--Reget items
        self.GetItems()
        self.SortItems(col,reverse)
        #--Delete Current items
        listItemCount = self.list.GetItemCount()
        #--Populate items
        for itemDex in range(len(self.items)):
            mode = int(itemDex >= listItemCount)
            self.PopulateItem(itemDex,mode,selected)
        #--Delete items?
        while self.list.GetItemCount() > len(self.items):
            self.list.DeleteItem(self.list.GetItemCount()-1)

    def ClearSelected(self):
        for itemDex in range(self.list.GetItemCount()):
            self.list.SetItemState(itemDex, 0, wx.LIST_STATE_SELECTED)

    def GetSelected(self):
        """Return list of items selected (hilighted) in the interface."""
        #--No items?
        if not 'items' in self.__dict__: return []
        selected = []
        itemDex = -1
        while True:
            itemDex = self.list.GetNextItem(itemDex,
                wx.LIST_NEXT_ALL,wx.LIST_STATE_SELECTED)
            if itemDex == -1 or itemDex >= len(self.items): 
                break
            else:
                selected.append(self.items[itemDex])
        return selected

    def GetSortSettings(self,col,reverse):
        """Return parsed col, reverse arguments. Used by SortSettings.
        col: sort variable. 
          Defaults to last sort. (self.sort)
        reverse: sort order
          1: Descending order
          0: Ascending order
         -1: Use current reverse settings for sort variable, unless
             last sort was on same sort variable -- in which case, 
             reverse the sort order. 
         -2: Use current reverse setting for sort variable.
        """
        #--Sort Column
        if not col:
            col = self.sort
        #--Reverse
        oldReverse = self.colReverse.get(col,0)
        if col == 'Load Order': #--Disallow reverse for load
            reverse = 0
        elif reverse == -1 and col == self.sort:
            reverse = not oldReverse
        elif reverse < 0:
            reverse = oldReverse
        #--Done
        self.sort = col
        self.colReverse[col] = reverse
        return (col,reverse)

    #--Event Handlers -------------------------------------
    #--Column Menu
    def DoColumnMenu(self,event):
        if not self.colLinks: return
        #--Build Menu
        column = event.GetColumn()
        menu = wx.Menu()
        for link in self.colLinks:
            link.AppendToMenu(menu,self,column)
        #--Show/Destroy Menu
        self.PopupMenu(menu)
        menu.Destroy()

    #--Column Resize
    def OnColumnResize(self,event):
        pass

    #--Item Sort
    def DoItemSort(self, event):
        self.PopulateItems(self.cols[event.GetColumn()],-1)

    #--Item Menu
    def DoItemMenu(self,event):
        selected = self.GetSelected()
        if not selected: return
        #--Build Menu
        menu = wx.Menu()
        for link in self.itemLinks:
            link.AppendToMenu(menu,self,selected)
        #--Show/Destroy Menu
        self.PopupMenu(menu)
        menu.Destroy()

    #--Size Change
    def OnSize(self, event):
        size = self.GetClientSizeTuple()
        #print self,size
        self.list.SetSize(size)

    #--Event: Left Down
    def OnLeftDown(self,event):
        #self.hitTest = self.list.HitTest((event.GetX(),event.GetY()))
        event.Skip()
    
#------------------------------------------------------------------------------
class MasterList(List):
    colLinks = mastersMainMenu
    itemLinks = mastersItemMenu

    def __init__(self,parent,fileInfo):
        self.parent = parent
        #--Columns
        self.cols = settings['bash.masters.cols']
        self.colNames = settings['bash.colNames']
        self.colWidths = settings['bash.masters.colWidths']
        self.colAligns = settings['bash.masters.colAligns']
        self.colReverse = settings['bash.masters.colReverse'].copy()
        #--Data/Items
        self.edited = False
        self.fileInfo = fileInfo
        self.prevId = -1
        self.data = {}  #--masterInfo = self.data[item], where item is id number
        self.items = [] #--Item numbers in display order.
        self.fileOrderItems = []
        self.loadOrderNames = []
        self.sort = settings['bash.masters.sort']
        self.esmsFirst = settings['bash.masters.esmsFirst']
        self.selectedFirst = settings['bash.masters.selectedFirst']
        #--Links
        self.colLinks = MasterList.colLinks
        self.itemLinks = MasterList.itemLinks
        #--Parent init
        List.__init__(self,parent,-1,ctrlStyle=(wx.LC_REPORT|wx.LC_SINGLE_SEL|wx.LC_EDIT_LABELS))
        wx.EVT_LIST_END_LABEL_EDIT(self,self.listId,self.OnLabelEdited)
        #--Image List
        checkboxesIL = self.checkboxes.GetImageList()
        self.list.SetImageList(checkboxesIL,wx.IMAGE_LIST_SMALL)

    #--NewItemNum
    def newId(self):
        self.prevId += 1
        return self.prevId

    #--Set ModInfo
    def SetFileInfo(self,fileInfo):
        self.ClearSelected()
        self.edited = False
        self.fileInfo = fileInfo
        self.prevId = -1
        self.data.clear()
        del self.items[:]
        del self.fileOrderItems[:]
        #--Null fileInfo?
        if not fileInfo:
            self.PopulateItems()
            return
        #--Fill data and populate
        for masterName in fileInfo.header.masters:
            item = self.newId()
            masterInfo = bosh.MasterInfo(masterName,0)
            self.data[item] = masterInfo
            self.items.append(item)
            self.fileOrderItems.append(item)
        self.ReList()
        self.SortItems()
        self.PopulateItems()

    #--Get Master Status
    def GetMasterStatus(self,item):
        masterInfo = self.data[item]
        masterName = masterInfo.name
        status = masterInfo.getStatus()
        if status == 30:
            return status
        fileOrderIndex = self.fileOrderItems.index(item)
        loadOrderIndex = self.loadOrderNames.index(masterName)
        ordered = bosh.modInfos.ordered
        if fileOrderIndex != loadOrderIndex:
            return 20
        elif status > 0:
            return status
        elif ((fileOrderIndex < len(ordered)) and 
            (ordered[fileOrderIndex] == masterName)):
            return -10
        else:
            return status

    #--Get Items
    def GetItems(self):
        return self.items

    #--Populate Item
    def PopulateItem(self,itemDex,mode=0,selected=set()):
        itemId = self.items[itemDex]
        masterInfo = self.data[itemId]
        masterName = masterInfo.name
        cols = self.cols
        for colDex in range(self.numCols):
            #--Value
            col = cols[colDex]
            if col == 'File':
                value = masterName
                if value == 'Oblivion.esm':
                    voCurrent = bosh.modInfos.voCurrent
                    if voCurrent: value += ' ['+voCurrent+']'
            elif col == 'Num':
                value = '%02X' % (self.fileOrderItems.index(itemId),)
            #--Insert/Set Value
            if mode and (colDex == 0):
                self.list.InsertStringItem(itemDex, value)
            else:
                self.list.SetStringItem(itemDex, colDex, value)
        #--Font color
        item = self.list.GetItem(itemDex)
        if masterInfo.isEsm():
            item.SetTextColour(wx.BLUE)
        else:
            item.SetTextColour(wx.BLACK)
        #--Text BG
        if masterInfo.hasActiveTimeConflict():
            item.SetBackgroundColour(colours['bash.doubleTime.load'])
        elif masterInfo.isExOverLoaded():
            item.SetBackgroundColour(colours['bash.exOverLoaded'])
        elif masterInfo.hasTimeConflict():
            item.SetBackgroundColour(colours['bash.doubleTime.exists'])
        else:
            item.SetBackgroundColour(colours['bash.doubleTime.not'])
        self.list.SetItem(item)
        #--Image
        status = self.GetMasterStatus(itemId)
        oninc = (masterName in bosh.modInfos.ordered) or (masterName in bosh.modInfos.merged and 2)
        self.list.SetItemImage(itemDex,self.checkboxes.Get(status,oninc))
        #--Selection State
        if masterName in selected:
            self.list.SetItemState(itemDex,wx.LIST_STATE_SELECTED,wx.LIST_STATE_SELECTED)
        else:
            self.list.SetItemState(itemDex,0,wx.LIST_STATE_SELECTED)

    #--Sort Items
    def SortItems(self,col=None,reverse=-2):
        (col, reverse) = self.GetSortSettings(col,reverse)
        #--Sort
        data = self.data
        #--Start with sort by type
        self.items.sort(key=lambda a: data[a].name[:-4].lower())
        if col == 'File':
            pass #--Done by default
        elif col == 'Rating':
            self.items.sort(key=lambda a: bosh.modInfos.table.getItem(a,'rating',''))
        elif col == 'Group':
            self.items.sort(key=lambda a: bosh.modInfos.table.getItem(a,'group',''))
        elif col == 'Modified':
            self.items.sort(key=lambda a: data[a].mtime)
        elif col == 'Save Order':
            self.items.sort()
        elif col == 'Load Order':
            loadOrderNames = self.loadOrderNames
            data = self.data
            self.items.sort(key=lambda a: loadOrderNames.index(data[a].name))
        elif col == 'Status':
            self.items.sort(lambda a,b: cmp(self.GetMasterStatus(a),self.GetMasterStatus(b)))
        elif col == 'Author':
            self.items.sort(lambda a,b: cmp(data[a].author.lower(),data[b].author.lower()))
        else:
            raise BashError(_('Unrecognized sort key: ')+col)
        #--Ascending
        if reverse: self.items.reverse()
        #--ESMs First?
        settings['bash.masters.esmsFirst'] = self.esmsFirst
        if self.esmsFirst or col == 'Load Order':
            self.items.sort(key=lambda a:data[a].name[-1].lower())

    #--Relist
    def ReList(self):
        fileOrderNames = [self.data[item].name for item in self.fileOrderItems]
        self.loadOrderNames = bosh.modInfos.getOrdered(fileOrderNames,False)

    #--InitEdit
    def InitEdit(self):
        #--Pre-clean
        for itemId in self.items:
            masterInfo = self.data[itemId]
            #--Missing Master?
            if not masterInfo.modInfo:
                masterName = masterInfo.name
                newName = settings['bash.mods.renames'].get(masterName,None)
                #--Rename?
                if newName and newName in bosh.modInfos:
                    masterInfo.setName(newName)
        #--Done
        self.edited = True
        self.ReList()
        self.PopulateItems()
        self.parent.SetEdited()

    #--Item Sort
    def DoItemSort(self, event):
        pass #--Don't do column head sort.

    #--Column Menu
    def DoColumnMenu(self,event):
        if not self.fileInfo: return
        List.DoColumnMenu(self,event)

    #--Item Menu
    def DoItemMenu(self,event):
        if not self.edited:
            self.OnLeftDown(event)
        else:
            List.DoItemMenu(self,event)

    #--Column Resize
    def OnColumnResize(self,event):
        colDex = event.GetColumn()
        colName = self.cols[colDex]
        self.colWidths[colName] = self.list.GetColumnWidth(colDex)
        settings.setChanged('bash.masters.colWidths')

    #--Event: Left Down
    def OnLeftDown(self,event):
        #--Not edited yet?
        if not self.edited:
            message = (_("Edit/update the masters list? Note that the update process may automatically rename some files. Be sure to review the changes before saving."))
            if ContinueQuery(self,message,'bash.masters.update',_('Update Masters')) != wx.ID_OK:
                return
            self.InitEdit()
        #--Pass event on (for label editing)
        else:
            event.Skip()

    #--Label Edited
    def OnLabelEdited(self,event):
        itemDex = event.m_itemIndex
        newName = Path.get(event.GetText())
        #--No change?
        if newName in bosh.modInfos:
            masterInfo = self.data[self.items[itemDex]]
            oldName = masterInfo.name
            masterInfo.setName(newName)
            self.ReList()
            self.PopulateItem(itemDex)
            settings.getChanged('bash.mods.renames')[masterInfo.oldName] = newName
        elif newName == '':
            event.Veto()
        else:
            ErrorMessage(self,_('File "%s" does not exist.') % (newName,))
            event.Veto()

    #--GetMasters
    def GetNewMasters(self):
        """Returns new master list."""
        return [self.data[item].name for item in self.fileOrderItems]
    
#------------------------------------------------------------------------------
class ModList(List):
    #--Class Data
    colLinks = modsMainMenu #--Column menu
    itemLinks = modsItemMenu #--Single item menu

    def __init__(self,parent):
        #--Columns
        self.cols = settings['bash.mods.cols']
        self.colAligns = settings['bash.mods.colAligns']
        self.colNames = settings['bash.colNames']
        self.colReverse = settings.getChanged('bash.mods.colReverse')
        self.colWidths = settings['bash.mods.colWidths']
        #--Data/Items
        self.data = data = bosh.modInfos
        self.details = None #--Set by panel
        self.sort = settings['bash.mods.sort']
        self.esmsFirst = settings['bash.mods.esmsFirst']
        self.selectedFirst = settings['bash.mods.selectedFirst']
        #--Links
        self.colLinks = ModList.colLinks
        self.itemLinks = ModList.itemLinks
        #--Parent init
        List.__init__(self,parent,-1,ctrlStyle=(wx.LC_REPORT))#|wx.SUNKEN_BORDER))
        #--Image List
        checkboxesIL = images['bash.checkboxes'].GetImageList()
        self.list.SetImageList(checkboxesIL,wx.IMAGE_LIST_SMALL)
        #--Events
        wx.EVT_LIST_ITEM_SELECTED(self,self.listId,self.OnItemSelected)
        self.list.Bind(wx.EVT_CHAR, self.OnChar)
        self.list.Bind(wx.EVT_LEFT_DCLICK, self.OnDoubleClick)

    def RefreshUI(self,files='ALL',detail='SAME',refreshSaves=True):
        """Refreshes UI for specified file. Also calls saveList.RefreshUI()!"""
        #--Details
        if detail == 'SAME':
            selected = set(self.GetSelected())
        else:
            selected = set([detail])
        #--Populate
        if files == 'ALL':
            self.PopulateItems(selected=selected)
        elif isinstance(files,StringTypes):
            self.PopulateItem(files,selected=selected)
        else: #--Iterable
            for file in files:
                self.PopulateItem(file,selected=selected)
        modDetails.SetFile(detail)
        bashFrame.SetStatusCount()
        #--Saves
        if refreshSaves:
            saveList.RefreshUI()

    #--Populate Item
    def PopulateItem(self,itemDex,mode=0,selected=set()):
        #--String name of item?
        if not isinstance(itemDex,int):
            itemDex = self.items.index(itemDex)
        fileName = Path.get(self.items[itemDex])
        fileInfo = self.data[fileName]
        cols = self.cols
        for colDex in range(self.numCols):
            col = cols[colDex]
            #--Get Value
            if col == 'File':
                value = fileName
                if value == 'Oblivion.esm' and bosh.modInfos.voCurrent:
                    value += ' ['+bosh.modInfos.voCurrent+']'
            elif col == 'Rating':
                value = bosh.modInfos.table.getItem(fileName,'rating','')
            elif col == 'Group':
                value = bosh.modInfos.table.getItem(fileName,'group','')
            elif col == 'Modified':
                value = bosh.formatDate(fileInfo.mtime)
            elif col == 'Size':
                value = bosh.formatInteger(fileInfo.size/1024)+' KB'
            elif col == 'Author' and fileInfo.header:
                value = fileInfo.header.author
            elif col == 'Load Order':
                ordered = bosh.modInfos.ordered
                if fileName in ordered:
                    value = '%02X' % (list(ordered).index(fileName),)
                else:
                    value = ''
            else:
                value = '-'
            #--Insert/SetString
            if mode and (colDex == 0):
                self.list.InsertStringItem(itemDex, value)
            else:
                self.list.SetStringItem(itemDex, colDex, value)
        #--Font color
        item = self.list.GetItem(itemDex)
        if fileInfo.isEsm():
            item.SetTextColour(wx.BLUE)
        elif fileInfo.isMergeable:
            item.SetTextColour(colours['bash.mods.isMergeable'])
        else:
            item.SetTextColour(wx.BLACK)
        #--Text BG
        if fileInfo.hasActiveTimeConflict():
            item.SetBackgroundColour(colours['bash.doubleTime.load'])
        elif fileInfo.isExOverLoaded():
            item.SetBackgroundColour(colours['bash.exOverLoaded'])
        elif fileInfo.hasTimeConflict():
            item.SetBackgroundColour(colours['bash.doubleTime.exists'])
        elif fileName[0] in '.+=':
            item.SetBackgroundColour(colours['bash.mods.groupHeader'])
        else:
            item.SetBackgroundColour(colours['bash.doubleTime.not'])
        self.list.SetItem(item)
        #--Image
        status = fileInfo.getStatus()
        checkMark = (
            (fileName in bosh.modInfos.ordered and 1) or 
            (fileName in bosh.modInfos.merged and 2) or
            (fileName in bosh.modInfos.imported and 3))
        self.list.SetItemImage(itemDex,self.checkboxes.Get(status,checkMark))
        #--Selection State
        if fileName in selected:
            self.list.SetItemState(itemDex,wx.LIST_STATE_SELECTED,wx.LIST_STATE_SELECTED)
        else:
            self.list.SetItemState(itemDex,0,wx.LIST_STATE_SELECTED)

    #--Sort Items
    def SortItems(self,col=None,reverse=-2):
        (col, reverse) = self.GetSortSettings(col,reverse)
        settings['bash.mods.sort'] = col
        selected = bosh.modInfos.ordered
        data = self.data
        #--Start with sort by name
        self.items.sort(key = lambda a: a[:-4].lower())
        if col == 'File':
            pass #--Done by default
        elif col == 'Author':
            self.items.sort(key=lambda a: data[a].header.author.lower())
        elif col == 'Rating':
            self.items.sort(key=lambda a: bosh.modInfos.table.getItem(a,'rating',''))
        elif col == 'Group':
            self.items.sort(key=lambda a: bosh.modInfos.table.getItem(a,'group',''))
        elif col == 'Load Order':
            self.items = bosh.modInfos.getOrdered(self.items,False)
        elif col == 'Modified':
            self.items.sort(key=lambda a: data[a].mtime)
        elif col == 'Size':
            self.items.sort(key=lambda a: data[a].size)
        elif col == 'Status':
            self.items.sort(key=lambda a: data[a].getStatus())
        else:
            raise BashError(_('Unrecognized sort key: ')+col)
        #--Ascending
        if reverse: self.items.reverse()
        #--ESMs First?
        settings['bash.mods.esmsFirst'] = self.esmsFirst
        if self.esmsFirst or col == 'Load Order':
            self.items.sort(lambda a,b: cmp(a[-4:].lower(),b[-4:].lower()))
        #--Selected First?
        settings['bash.mods.selectedFirst'] = self.selectedFirst
        if self.selectedFirst:
            self.items.sort(lambda a,b: cmp(b in selected, a in selected))

    #--Events ---------------------------------------------
    def OnDoubleClick(self,event):
        """Handle doubclick event."""
        (hitItem,hitFlag) = self.list.HitTest(event.GetPosition())
        if hitItem < 0: return
        fileInfo = self.data[self.items[hitItem]]
        if not docBrowser: 
            DocBrowser().Show()
            settings['bash.modDocs.show'] = True
        docBrowser.SetMod(fileInfo.name)
        docBrowser.Raise()

    def OnChar(self,event):
        """Char event: Reordering."""
        if ((event.ControlDown() and event.GetKeyCode() in (wx.WXK_UP,wx.WXK_DOWN)) and
            (settings['bash.mods.sort'] == 'Load Order') and
            (self.list.GetSelectedItemCount() == 1)
            ):
                thisFile = self.GetSelected()[0]
                if Path(thisFile) in bosh.modInfos.autoSorted:
                    ErrorMessage(self,_("Auto-ordered files cannot be manually moved."))
                    event.Skip()
                    return
                thisItem = self.items.index(thisFile)
                swapItem = thisItem + (-1,1)[event.GetKeyCode() == wx.WXK_DOWN]
                if swapItem < 0 or swapItem >= len(self.items): return
                swapFile = self.items[swapItem]
                if thisFile.ext() != swapFile.ext(): return
                thisInfo, swapInfo = bosh.modInfos[thisFile], bosh.modInfos[swapFile]
                thisTime, swapTime = thisInfo.mtime, swapInfo.mtime
                thisInfo.setMTime(swapTime)
                swapInfo.setMTime(thisTime)
                bosh.modInfos.refreshInfoLists()
                self.RefreshUI()
        event.Skip()
    
    def OnColumnResize(self,event):
        """Column resize: Stored modified column widths."""
        colDex = event.GetColumn()
        colName = self.cols[colDex]
        self.colWidths[colName] = self.list.GetColumnWidth(colDex)
        settings.setChanged('bash.mods.colWidths')

    def OnLeftDown(self,event):
        """Left Down: Check/uncheck mods."""
        (hitItem,hitFlag) = self.list.HitTest((event.GetX(),event.GetY()))
        if hitFlag == 32:
            oldFiles = bosh.modInfos.ordered[:]
            fileName = Path.get(self.items[hitItem])
            #--Unselect?
            if self.data.isSelected(fileName):
                self.data.unselect(fileName)
                changed = bosh.listSubtract(oldFiles,bosh.modInfos.ordered)
                if len(changed) > 1:
                    changed.remove(fileName)
                    ListMessage(self,_('${count} Children deactivated:'),changed,10,_("Children Deactivated"))
            #--Select?
            else:
                try:
                    self.data.select(fileName)
                    changed = bosh.listSubtract(bosh.modInfos.ordered,oldFiles)
                    if len(changed) > 1:
                        changed.remove(fileName)
                        ListMessage(self,_('${count} Masters activated:'),changed,10,_("Masters activated"))
                except bosh.PluginsFullError:
                    ErrorMessage(self,_("Unable to add mod %s because load list is full." )
                        % (fileName,))
                    return
            #--Refresh 
            self.RefreshUI() 
            #--Mark sort as dirty
            if self.selectedFirst:
                self.sortDirty = 1
        #--Pass Event onward
        event.Skip()

    def OnItemSelected(self,event):
        """Item Selected: Set mod details."""
        modName = self.items[event.m_itemIndex]
        self.details.SetFile(modName)
        if docBrowser: 
            docBrowser.SetMod(modName)

#------------------------------------------------------------------------------
class ModDetails(wx.Window):
    """Details panel for mod tab."""
    def __init__(self,parent):
        wx.Window.__init__(self, parent, -1, style=wx.TAB_TRAVERSAL)
        #--Singleton
        global modDetails
        modDetails = self
        #--Data
        self.modInfo = None
        self.edited = False
        textWidth = 200
        #--Version
        self.version = wx.StaticText(self,-1,'v0.0')
        id = self.fileId = wx.NewId()
        #--File Name
        self.file = wx.TextCtrl(self,id,"",size=(textWidth,-1))
        self.file.SetMaxLength(200)
        self.file.Bind(wx.EVT_KILL_FOCUS, self.OnEditFile)
        self.file.Bind(wx.EVT_TEXT, self.OnTextEdit)
        #--Author
        id = self.authorId = wx.NewId()
        self.author = wx.TextCtrl(self,id,"",size=(textWidth,-1))
        self.author.SetMaxLength(512)
        wx.EVT_KILL_FOCUS(self.author,self.OnEditAuthor)
        wx.EVT_TEXT(self.author,id,self.OnTextEdit)
        #--Modified
        id = self.modifiedId = wx.NewId()
        self.modified = wx.TextCtrl(self,id,"",size=(textWidth,-1))
        self.modified.SetMaxLength(32)
        wx.EVT_KILL_FOCUS(self.modified,self.OnEditModified)
        wx.EVT_TEXT(self.modified,id,self.OnTextEdit)
        #--Description
        id = self.descriptionId = wx.NewId()
        self.description = (
            wx.TextCtrl(self,id,"",size=(textWidth,150),style=wx.TE_MULTILINE))
        self.description.SetMaxLength(512)
        wx.EVT_KILL_FOCUS(self.description,self.OnEditDescription)
        wx.EVT_TEXT(self.description,id,self.OnTextEdit)
        #--Masters
        id = self.mastersId = wx.NewId()
        self.masters = MasterList(self,None)
        #--Save/Cancel
        self.save = button((self,wx.ID_SAVE),self.OnSave)
        self.cancel = button((self,wx.ID_CANCEL),self.OnCancel)
        self.save.Disable()
        self.cancel.Disable()
        #--Layout
        sizer = vSizer(
            (hSizer(
                (wx.StaticText(self,-1,_("File:")),0,wx.TOP,4),
                ((0,0),1),
                (self.version,0,wx.TOP|wx.RIGHT,4)
                ),0,wx.EXPAND),
            self.file,
            (wx.StaticText(self,-1,_("Author:")),0,wx.TOP,4),
            self.author,
            (wx.StaticText(self,-1,_("Modified:")),0,wx.TOP,4),
            self.modified,
            (wx.StaticText(self,-1,_("Description:")),0,wx.TOP,4),
            self.description,
            (wx.StaticText(self,-1,_("Masters:")),0,wx.TOP,4),
            (self.masters,1,wx.EXPAND),
            (hSizer(
                ((0,0),1),
                self.save,
                (self.cancel,0,wx.LEFT,4)
                ),0,wx.EXPAND|wx.TOP,4),
            )
        self.SetSizer(sizer)

    def SetFile(self,fileName='SAME'):
        #--Reset?
        if fileName == 'SAME': 
            if not self.modInfo or self.modInfo.name not in bosh.modInfos:
                fileName = None
            else:
                fileName = self.modInfo.name
        #--Empty?
        if not fileName:
            modInfo = self.modInfo = None
            self.fileStr = ''
            self.authorStr = ''
            self.modifiedStr = ''
            self.descriptionStr = ''
            self.versionStr = ''
        #--Valid fileName?
        else:
            modInfo = self.modInfo = bosh.modInfos[fileName]
            #--Remember values for edit checks
            self.fileStr = modInfo.name
            self.authorStr = modInfo.header.author
            self.modifiedStr = bosh.formatDate(modInfo.mtime)
            self.descriptionStr = modInfo.header.description
            self.versionStr = 'v%0.1f' % (modInfo.header.version,)
        #--Editable mtime?
        if fileName in bosh.modInfos.autoSorted:
            self.modified.SetEditable(False)
            self.modified.SetBackgroundColour(self.GetBackgroundColour())
        else:
            self.modified.SetEditable(True)
            self.modified.SetBackgroundColour(self.author.GetBackgroundColour())
        #--Set fields
        self.file.SetValue(self.fileStr)
        self.author.SetValue(self.authorStr)
        self.modified.SetValue(self.modifiedStr)
        self.description.SetValue(self.descriptionStr)
        self.version.SetLabel(self.versionStr)
        self.masters.SetFileInfo(modInfo)
        #--Edit State
        self.edited = 0
        self.save.Disable()
        self.cancel.Disable()

    def SetEdited(self):
        self.edited = True
        self.save.Enable()
        self.cancel.Enable()
    
    def OnTextEdit(self,event):
        if self.modInfo and not self.edited:
            if ((self.fileStr != self.file.GetValue()) or
                (self.authorStr != self.author.GetValue()) or
                (self.modifiedStr != self.modified.GetValue()) or
                (self.descriptionStr != self.description.GetValue()) ):
                self.SetEdited()
        event.Skip()

    def OnEditFile(self,event):
        if not self.modInfo: return
        #--Changed?
        fileStr = self.file.GetValue()
        if fileStr == self.fileStr: return
        #--Extension Changed?
        if fileStr[-4:].lower() != self.fileStr[-4:].lower():
            ErrorMessage(self,_("Incorrect file extension: ")+fileStr[-3:])
            self.file.SetValue(self.fileStr)
        #--Else file exists?
        elif self.modInfo.dir.join(fileStr).exists():
            ErrorMessage(self,_("File %s already exists.") % (fileStr,))
            self.file.SetValue(self.fileStr)
        #--Okay?
        else:
            self.fileStr = fileStr
            self.SetEdited()

    def OnEditAuthor(self,event):
        if not self.modInfo: return
        authorStr = self.author.GetValue()
        if authorStr != self.authorStr:
            self.authorStr = authorStr
            self.SetEdited()

    def OnEditModified(self,event):
        if not self.modInfo: return
        modifiedStr = self.modified.GetValue()
        if modifiedStr == self.modifiedStr: return
        try:
            newTimeTup = time.strptime(modifiedStr,'%c')
            time.mktime(newTimeTup)
        except ValueError:
            ErrorMessage(self,_('Unrecognized date: ')+modifiedStr)
            self.modified.SetValue(self.modifiedStr)
            return
        except OverflowError:
            ErrorMessage(self,_('Bash cannot handle files dates greater than January 19, 2038.)'))
            self.modified.SetValue(self.modifiedStr)
            return
        #--Normalize format
        modifiedStr = time.strftime('%c',newTimeTup)
        self.modifiedStr = modifiedStr
        self.modified.SetValue(modifiedStr) #--Normalize format
        self.SetEdited()

    def OnEditDescription(self,event):
        if not self.modInfo: return
        descriptionStr = self.description.GetValue()
        if descriptionStr != self.descriptionStr:
            self.descriptionStr = descriptionStr
            self.SetEdited()
    
    def OnSave(self,event):
        modInfo = self.modInfo
        #--Change Tests
        changeName = (self.fileStr != modInfo.name)
        changeDate = (self.modifiedStr != bosh.formatDate(modInfo.mtime))
        changeHedr = ((self.authorStr != modInfo.header.author) or 
            (self.descriptionStr != modInfo.header.description ))
        changeMasters = self.masters.edited
        #--Warn on rename if file has bsa and/or dialog
        hasBsa, hasVoices = modInfo.hasResources()
        if changeName and (hasBsa or hasVoices):
            modName = modInfo.name
            if hasBsa and hasVoices:
                message = _("This mod has an associated archive (%s.bsa) and an associated voice directory (Sound\\Voices\\%s), which will become detached when the mod is renamed.\n\nNote that the BSA archive may also contain a voice directory (Sound\\Voices\\%s), which would remain detached even if the archive name is adjusted.") % (modName[:-4],modName,modName)
            elif hasBsa:
                message = _("This mod has an associated archive (%s.bsa), which will become detached when the mod is renamed.\n\nNote that this BSA archive may contain a voice directory (Sound\\Voices\\%s), which would remain detached even if the archive file name is adjusted.") % (modName[:-4],modName)
            else: #hasVoices
                message = _("This mod has an associated voice directory (Sound\\Voice\\%s), which will become detached when the mod is renamed.") % (modName,)
            if Message(self,message,style=wx.OK|wx.CANCEL|wx.ICON_EXCLAMATION) != wx.ID_OK:
                return
        #--Only change date?
        if changeDate and not (changeName or changeHedr or changeMasters):
            newTimeTup = time.strptime(self.modifiedStr,'%c')
            newTimeInt = int(time.mktime(newTimeTup))
            modInfo.setMTime(newTimeInt)
            self.SetFile(self.modInfo.name)
            bosh.modInfos.autoSort()
            bosh.modInfos.refreshInfoLists()
            modList.RefreshUI()
            return
        #--Backup
        modInfo.makeBackup()
        #--Change Name?
        fileName = modInfo.name
        if changeName:
            (oldName,newName) = (modInfo.name,Path.get(self.fileStr.strip()))
            modList.items[modList.items.index(oldName)] = newName
            settings.getChanged('bash.mods.renames')[oldName] = newName
            bosh.modInfos.rename(oldName,newName)
            fileName = Path.get(newName)
        #--Change hedr/masters?
        if changeHedr or changeMasters:
            modInfo.header.author = self.authorStr.strip()
            modInfo.header.description = bosh.winNewLines(self.descriptionStr.strip())
            modInfo.header.masters = self.masters.GetNewMasters()
            modInfo.header.changed = True
            modInfo.writeHeader()
        #--Change date?
        if (changeDate or changeHedr or changeMasters):
            newTimeTup = time.strptime(self.modifiedStr,'%c')
            newTimeInt = int(time.mktime(newTimeTup))
            modInfo.setMTime(newTimeInt)
        #--Done
        try:
            #bosh.modInfos.refresh()
            bosh.modInfos.refreshFile(fileName)
            self.SetFile(fileName)
        except bosh.FileError:
            ErrorMessage(self,_('File corrupted on save!'))
            self.SetFile(None)
        if bosh.modInfos.autoSort():
            bosh.modInfos.refreshInfoLists()
        modList.RefreshUI()

    def OnCancel(self,event):
        self.SetFile(self.modInfo.name)

#------------------------------------------------------------------------------
class ModPanel(NotebookPanel):
    def __init__(self,parent):
        wx.Panel.__init__(self, parent, -1)
        global modList
        modList = ModList(self)
        self.modDetails = ModDetails(self)
        modList.details = self.modDetails
        #--Events
        wx.EVT_SIZE(self,self.OnSize)
        #--Layout
        sizer = hSizer(
            (modList,1,wx.GROW),
            ((4,-1),0),
            (self.modDetails,0,wx.EXPAND))
        self.SetSizer(sizer)
        self.modDetails.Fit()

    def SetStatusCount(self):
        """Sets mod count in last field."""
        text = _("Mods: %d/%d") % (len(bosh.modInfos.ordered),len(bosh.modInfos.data))
        statusBar.SetStatusText(text,2)

    def OnSize(self,event):
        wx.Window.Layout(self)
        modList.Layout()
        self.modDetails.Layout()

#------------------------------------------------------------------------------
class SaveList(List):
    #--Class Data
    colLinks = savesMainMenu #--Column menu
    itemLinks = savesItemMenu #--Single item menu

    def __init__(self,parent):
        #--Columns
        self.cols = settings['bash.saves.cols']
        self.colAligns = settings['bash.saves.colAligns']
        self.colNames = settings['bash.colNames']
        self.colReverse = settings.getChanged('bash.saves.colReverse')
        self.colWidths = settings['bash.saves.colWidths']
        #--Data/Items
        self.data = data = bosh.saveInfos
        self.details = None #--Set by panel
        self.sort = settings['bash.saves.sort']
        #--Links
        self.colLinks = SaveList.colLinks
        self.itemLinks = SaveList.itemLinks
        #--Parent init
        List.__init__(self,parent,-1,ctrlStyle=(wx.LC_REPORT|wx.SUNKEN_BORDER))
        #--Image List
        checkboxesIL = self.checkboxes.GetImageList()
        self.list.SetImageList(checkboxesIL,wx.IMAGE_LIST_SMALL)
        #--Events
        wx.EVT_LIST_ITEM_SELECTED(self,self.listId,self.OnItemSelected)

    def RefreshUI(self,files='ALL',detail='SAME'):
        """Refreshes UI for specified files."""
        #--Details
        if detail == 'SAME':
            selected = set(self.GetSelected())
        else:
            selected = set([detail])
        #--Populate
        if files == 'ALL':
            self.PopulateItems(selected=selected)
        elif isinstance(files,StringTypes):
            self.PopulateItem(files,selected=selected)
        else: #--Iterable
            for file in files:
                self.PopulateItem(file,selected=selected)
        saveDetails.SetFile(detail)
        bashFrame.SetStatusCount()

    #--Populate Item
    def PopulateItem(self,itemDex,mode=0,selected=set()):
        #--String name of item?
        if not isinstance(itemDex,int):
            itemDex = self.items.index(itemDex)
        fileName = Path.get(self.items[itemDex])
        fileInfo = self.data[fileName]
        cols = self.cols
        for colDex in range(self.numCols):
            col = cols[colDex]
            if col == 'File':
                value = fileName
            elif col == 'Modified':
                value = bosh.formatDate(fileInfo.mtime)
            elif col == 'Size':
                value = bosh.formatInteger(fileInfo.size/1024)+' KB'
            elif col == 'Player' and fileInfo.header:
                value = fileInfo.header.pcName
            elif col == 'PlayTime' and fileInfo.header:
                playMinutes = fileInfo.header.gameTicks/60000
                value = "%d:%02d" % (playMinutes/60,(playMinutes % 60))
            elif col == 'Cell' and fileInfo.header:
                value = fileInfo.header.pcLocation
            else:
                value = '-'
            if mode and (colDex == 0):
                self.list.InsertStringItem(itemDex, value)
            else:
                self.list.SetStringItem(itemDex, colDex, value)
        #--Image
        status = fileInfo.getStatus()
        on = fileName.lower()[-4:] == '.ess'
        self.list.SetItemImage(itemDex,self.checkboxes.Get(status,on))
        #--Selection State
        if fileName in selected:
            self.list.SetItemState(itemDex,wx.LIST_STATE_SELECTED,wx.LIST_STATE_SELECTED)
        else:
            self.list.SetItemState(itemDex,0,wx.LIST_STATE_SELECTED)

    #--Sort Items
    def SortItems(self,col=None,reverse=-2):
        (col, reverse) = self.GetSortSettings(col,reverse)
        settings['bash.saves.sort'] = col
        data = self.data
        #--Start with sort by name
        self.items.sort(lambda a,b: cmp(a.lower(),b.lower()))
        if col == 'File':
            pass #--Done by default
        elif col == 'Modified':
            self.items.sort(key=lambda a: data[a].mtime)
        elif col == 'Size':
            self.items.sort(key=lambda a: data[a].size)
        elif col == 'Status':
            self.items.sort(key=lambda a: data[a].getStatus())
        elif col == 'Player':
            self.items.sort(key=lambda a: data[a].header.pcName)
        elif col == 'PlayTime':
            self.items.sort(key=lambda a: data[a].header.gameTicks)
        elif col == 'Cell':
            self.items.sort(key=lambda a: data[a].header.pcLocation)
        else:
            raise BashError(_('Unrecognized sort key: ')+col)
        #--Ascending
        if reverse: self.items.reverse()

    #--Events ---------------------------------------------
    #--Column Resize
    def OnColumnResize(self,event):
        colDex = event.GetColumn()
        colName = self.cols[colDex]
        self.colWidths[colName] = self.list.GetColumnWidth(colDex)
        settings.setChanged('bash.saves.colWidths')

    #--Event: Left Down
    def OnLeftDown(self,event):
        (hitItem,hitFlag) = self.list.HitTest((event.GetX(),event.GetY()))
        if hitFlag == 32:
            fileName = Path.get(self.items[hitItem])
            newEnabled = not self.data.isEnabled(fileName)
            newName = self.data.enable(fileName,newEnabled)
            if newName != fileName: self.RefreshUI()
        #--Pass Event onward
        event.Skip()

    def OnItemSelected(self,event=None):
        saveName = self.items[event.m_itemIndex]
        self.details.SetFile(saveName)

#------------------------------------------------------------------------------
class SaveDetails(wx.Window):
    """Savefile details panel."""
    def __init__(self,parent):
        """Initialize."""
        wx.Window.__init__(self, parent, -1, style=wx.TAB_TRAVERSAL)
        readOnlyColour = self.GetBackgroundColour()
        #--Singleton
        global saveDetails
        saveDetails = self
        #--Data
        self.saveInfo = None
        self.edited = False
        textWidth = 200
        #--File Name
        id = self.fileId = wx.NewId()
        self.file = wx.TextCtrl(self,id,"",size=(textWidth,-1))
        self.file.SetMaxLength(256)
        wx.EVT_KILL_FOCUS(self.file,self.OnEditFile)
        wx.EVT_TEXT(self.file,id,self.OnTextEdit)
        #--Player Info
        self.playerInfo = wx.StaticText(self,-1," \n \n ")
        #--Picture
        self.picture = Picture(self,textWidth,192*textWidth/256) #--Native: 256x192
        #--Masters
        id = self.mastersId = wx.NewId()
        self.masters = MasterList(self,None)
        #--Save/Cancel
        self.save = wx.Button(self,wx.ID_SAVE)
        self.cancel = wx.Button(self,wx.ID_CANCEL)
        self.save.Disable()
        self.cancel.Disable()
        wx.EVT_BUTTON(self,wx.ID_SAVE,self.OnSave)
        wx.EVT_BUTTON(self,wx.ID_CANCEL,self.OnCancel)
        #--Layout
        sizer = vSizer(
            #(wx.StaticText(self,-1,_("File:")),0,wx.TOP,4),
            (self.file,0,wx.EXPAND|wx.TOP,4),
            (self.playerInfo,0,wx.EXPAND|wx.TOP,4),
            (self.picture,0,wx.TOP,4),
            #(wx.StaticText(self,-1,_("Masters:")),0,wx.TOP,4),
            (self.masters,1,wx.EXPAND|wx.TOP,4),
            (hSizer(
                ((0,0),1),
                self.save,
                (self.cancel,0,wx.LEFT,4),
                ),0,wx.EXPAND|wx.TOP,4),
            )
        self.SetSizer(sizer)
    
    def SetFile(self,fileName='SAME'):
        """Set file to be viewed."""
        #--Reset?
        if fileName == 'SAME': 
            if not self.saveInfo or self.saveInfo.name not in bosh.saveInfos:
                fileName = None
            else:
                fileName = self.saveInfo.name
        #--Null fileName?
        if not fileName:
            saveInfo = self.saveInfo = None
            self.fileStr = ''
            self.playerNameStr = ''
            self.curCellStr = ''
            self.playerLevel = 0
            self.gameDays = 0
            self.playMinutes = 0
            self.picData = None
        #--Valid fileName?
        else:
            saveInfo = self.saveInfo = bosh.saveInfos[fileName]
            #--Remember values for edit checks
            self.fileStr = saveInfo.name
            self.playerNameStr = saveInfo.header.pcName
            self.curCellStr = saveInfo.header.pcLocation
            self.gameDays = saveInfo.header.gameDays
            self.playMinutes = saveInfo.header.gameTicks/60000
            self.playerLevel = saveInfo.header.pcLevel
            self.picData = saveInfo.header.image
        #--Set Fields
        self.file.SetValue(self.fileStr)
        self.playerInfo.SetLabel(_("%s\nLevel %d, Day %d, Play %d:%02d\n%s") % 
            (self.playerNameStr,self.playerLevel,int(self.gameDays),self.playMinutes/60,(self.playMinutes % 60),self.curCellStr))
        #self.playerInfo.SetLabel(_("%s\n%s [%d]") % (self.curCellStr,self.playerNameStr,self.playerLevel))
        self.masters.SetFileInfo(saveInfo)
        #--Picture
        if not self.picData:
            self.picture.SetBitmap(None)
        else:
            width,height,data = self.picData
            image = wx.EmptyImage(width,height)
            image.SetData(data)
            self.picture.SetBitmap(image.ConvertToBitmap())
        #--Edit State
        self.edited = 0
        self.save.Disable()
        self.cancel.Disable()

    def SetEdited(self):
        """Mark as edited."""
        self.edited = True
        self.save.Enable()
        self.cancel.Enable()
    
    def OnTextEdit(self,event):
        """Event: Editing file or save name text."""
        if self.saveInfo and not self.edited:
            if self.fileStr != self.file.GetValue():
                self.SetEdited()
        event.Skip()

    def OnEditFile(self,event):
        """Event: Finished editing file name."""
        if not self.saveInfo: return
        #--Changed?
        fileStr = self.file.GetValue()
        if fileStr == self.fileStr: return
        #--Extension Changed?
        if self.fileStr[-4:].lower() not in ('.ess','.bak'):
            ErrorMessage(self,"Incorrect file extension: "+fileStr[-3:])
            self.file.SetValue(self.fileStr)
        #--Else file exists?
        elif self.saveInfo.dir.join(fileStr).exists():
            ErrorMessage(self,"File %s already exists." % (fileStr,))
            self.file.SetValue(self.fileStr)
        #--Okay?
        else:
            self.fileStr = fileStr
            self.SetEdited()

    def OnSave(self,event):
        """Event: Clicked Save button."""
        saveInfo = self.saveInfo
        #--Change Tests
        changeName = (self.fileStr != saveInfo.name)
        changeMasters = self.masters.edited
        #--Backup
        saveInfo.makeBackup()
        prevMTime = saveInfo.mtime
        #--Change Name?
        if changeName:
            (oldName,newName) = (saveInfo.name,Path.get(self.fileStr.strip()))
            saveList.items[saveList.items.index(oldName)] = newName
            bosh.saveInfos.rename(oldName,newName)
        #--Change masters?
        if changeMasters:
            saveInfo.header.masters = self.masters.GetNewMasters()
            saveInfo.header.writeMasters(saveInfo.getPath())
            saveInfo.setMTime(prevMTime)
        #--Done
        try:
            bosh.saveInfos.refreshFile(saveInfo.name)
            self.SetFile(self.saveInfo.name)
        except bosh.FileError:
            ErrorMessage(self,_('File corrupted on save!'))
            self.SetFile(None)
        self.SetFile(self.saveInfo.name)
        saveList.RefreshUI(saveInfo.name)

    def OnCancel(self,event):
        """Event: Clicked cancel button."""
        self.SetFile(self.saveInfo.name)

#------------------------------------------------------------------------------
class SavePanel(NotebookPanel):
    """Savegames tab."""
    def __init__(self,parent):
        wx.Panel.__init__(self, parent, -1)
        global saveList
        saveList = SaveList(self)
        self.saveDetails = SaveDetails(self)
        saveList.details = self.saveDetails
        #--Events
        wx.EVT_SIZE(self,self.OnSize)
        #--Layout
        sizer = hSizer(
            (saveList,1,wx.GROW),
            ((4,-1),0),
            (self.saveDetails,0,wx.EXPAND))
        self.SetSizer(sizer)
        self.saveDetails.Fit()

    def SetStatusCount(self):
        """Sets mod count in last field."""
        text = _("Saves: %d") % (len(bosh.saveInfos.data))
        statusBar.SetStatusText(text,2)

    def OnSize(self,event=None):
        wx.Window.Layout(self)
        saveList.Layout()
        self.saveDetails.Layout()

#------------------------------------------------------------------------------
class ReplacersList(List):
    #--Class Data
    colLinks = replacersMainMenu #--Column menu
    itemLinks = replacersItemMenu #--Single item menu

    def __init__(self,parent):
        #--Columns
        self.cols = settings['bash.replacers.cols']
        self.colAligns = settings['bash.replacers.colAligns']
        self.colNames = settings['bash.colNames']
        self.colReverse = settings.getChanged('bash.replacers.colReverse')
        self.colWidths = settings['bash.replacers.colWidths']
        #--Data/Items
        self.data = bosh.replacersData = bosh.ReplacersData()
        self.sort = settings['bash.replacers.sort']
        #--Links
        self.colLinks = ReplacersList.colLinks
        self.itemLinks = ReplacersList.itemLinks
        #--Parent init
        List.__init__(self,parent,-1,ctrlStyle=(wx.LC_REPORT|wx.SUNKEN_BORDER))
        #--Image List
        checkboxesIL = images['bash.checkboxes'].GetImageList()
        self.list.SetImageList(checkboxesIL,wx.IMAGE_LIST_SMALL)
        #--Events
        #wx.EVT_LIST_ITEM_SELECTED(self,self.listId,self.OnItemSelected)

    def RefreshUI(self,files='ALL',detail='SAME'):
        """Refreshes UI for specified files."""
        #--Details
        if detail == 'SAME':
            selected = set(self.GetSelected())
        else:
            selected = set([detail])
        #--Populate
        if files == 'ALL':
            self.PopulateItems(selected=selected)
        elif isinstance(files,StringTypes):
            self.PopulateItem(files,selected=selected)
        else: #--Iterable
            for file in files:
                self.PopulateItem(file,selected=selected)

    #--Populate Item
    def PopulateItem(self,itemDex,mode=0,selected=set()):
        #--String name of item?
        if not isinstance(itemDex,int):
            itemDex = self.items.index(itemDex)
        fileName = Path.get(self.items[itemDex])
        fileInfo = self.data[fileName]
        cols = self.cols
        for colDex in range(self.numCols):
            col = cols[colDex]
            if col == 'File':
                value = fileName
            else:
                value = '-'
            if mode and (colDex == 0):
                self.list.InsertStringItem(itemDex, value)
            else:
                self.list.SetStringItem(itemDex, colDex, value)
        #--Image
        self.list.SetItemImage(itemDex,self.checkboxes.Get(0,fileInfo.isApplied()))

    #--Sort Items
    def SortItems(self,col=None,reverse=-2):
        (col, reverse) = self.GetSortSettings(col,reverse)
        settings['bash.screens.sort'] = col
        data = self.data
        #--Start with sort by name
        self.items.sort(lambda a,b: cmp(a.lower(),b.lower()))
        if col == 'File':
            pass #--Done by default
        else:
            raise BashError(_('Unrecognized sort key: ')+col)
        #--Ascending
        if reverse: self.items.reverse()

    #--Events ---------------------------------------------
    #--Column Resize
    def OnColumnResize(self,event):
        colDex = event.GetColumn()
        colName = self.cols[colDex]
        self.colWidths[colName] = self.list.GetColumnWidth(colDex)
        settings.setChanged('bash.screens.colWidths')

    #--Event: Left Down
    def OnLeftDown(self,event):
        (hitItem,hitFlag) = self.list.HitTest((event.GetX(),event.GetY()))
        if hitFlag == 32:
            item = Path.get(self.items[hitItem])
            replacer = self.data[item]
            #--Unselect?
            if replacer.isApplied():
                try:
                    wx.BeginBusyCursor()
                    replacer.remove()
                finally:
                    wx.EndBusyCursor()
            #--Select?
            else:
                progress = None
                try:
                    progress = ProgressDialog(item)
                    replacer.apply(progress)
                finally:
                    if progress != None: progress.Destroy()
            self.RefreshUI(item)
            bosh.modInfos.refresh()
            modList.RefreshUI()
            return True
        #--Pass Event onward
        event.Skip()

#------------------------------------------------------------------------------
class ReplacersPanel(NotebookPanel):
    """Replacers tab."""
    def __init__(self,parent):
        """Initialize."""
        wx.Panel.__init__(self, parent, -1)
        self.gList = ReplacersList(self)
        #--Buttons
        self.gAuto = checkBox((self,-1,_("Automatic")),self.OnAutomatic,
            _("Automatically update Textures BSA after adding/removing a replacer."))
        self.gAuto.SetValue(settings['bash.replacers.autoChecked'])
        self.gInvalidate = button((self,-1,_("Update")),self.OnInvalidateTextures,
            _("Enable replacement textures by updating Textures archive."))
        self.gReset = button((self,-1,_("Restore")),self.OnResetTextures,
            _("Restore Textures archive to its original state."))
        #--Layout
        self.gTexturesBsa = vsbSizer((self,-1,_("Textures BSA")),
            ((0,4),),
            (self.gAuto,0,wx.ALL^wx.BOTTOM,4),
            ((0,8),),
            (self.gInvalidate,0,wx.ALL^wx.BOTTOM,4),
            (self.gReset,0,wx.ALL,4),
            )
        sizer = hSizer(
            (self.gTexturesBsa,0,wx.ALL|10),
            (self.gList,1,wx.GROW|wx.LEFT,4))
        self.SetSizer(sizer)

    def SetStatusCount(self):
        """Sets status bar count field."""
        numUsed = len([info for info in self.gList.data.values() if info.isApplied()])
        text = _('Reps: %d/%d') % (numUsed,len(self.gList.data.data))
        statusBar.SetStatusText(text,2)

    def OnShow(self):
        """Panel is shown. Update self.data."""
        if bosh.replacersData.refresh():
            self.gList.RefreshUI()
        #--vs. OBMM?
        enableBsaEdits = not (
            bosh.dirs['mods'].join('ConsoleBSAEditData2').exists() or 
            bosh.dirs['app'].join('OBMM','BSAedits').exists())
        self.gAuto.Enable(enableBsaEdits)
        self.gInvalidate.Enable(enableBsaEdits)
        self.gReset.Enable(enableBsaEdits)
        if enableBsaEdits:
            self.gTexturesBsa.GetStaticBox().SetToolTip(None)
            settings['bash.replacers.autoEditBSAs'] = settings['bash.replacers.autoChecked']
        else:
            self.gTexturesBsa.GetStaticBox().SetToolTip(tooltip(
                _("BSA editing disabled becase OBMM or BSAPatch is in use.")))
            settings['bash.replacers.autoEditBSAs'] = False
        self.SetStatusCount()

    def ContinueEdit(self):
        """Continuation warning for Invalidate and Reset."""
        message = _("Edit Textures BSA?\n\nThis command directly edits the Oblivion - Textures - Compressed.bsa file. If the file becomes corrupted (very unlikely), you will need to reinstall Oblivion or restore it from another source.")
        result = ContinueQuery(self,message,'bash.replacers.editBSAs.continue',_('Textures BSA')) 
        return (result == wx.ID_OK)

    def OnAutomatic(self,event=None):
        """Automatic checkbox changed."""
        isChecked = self.gAuto.IsChecked()
        if isChecked and not self.ContinueEdit():
            self.gAuto.SetValue(False)
            return
        settings['bash.replacers.autoChecked'] = isChecked
        settings['bash.replacers.autoEditBSAs'] = isChecked

    def OnInvalidateTextures(self,event):
        """Invalid."""
        if not self.ContinueEdit(): return
        bsaPath = bosh.modInfos.dir.join('Oblivion - Textures - Compressed.bsa')
        bsaFile = bosh.BsaFile(bsaPath)
        bsaFile.scan()
        result = bsaFile.invalidate()
        Message(self,
            _("BSA Hashes reset: %d\nBSA Hashes Invalidated: %d.\nAIText entries: %d.") % 
            tuple(map(len,result)))

    def OnResetTextures(self,event):
        """Invalid."""
        if not self.ContinueEdit(): return
        bsaPath = bosh.modInfos.dir.join('Oblivion - Textures - Compressed.bsa')
        bsaFile = bosh.BsaFile(bsaPath)
        bsaFile.scan()
        resetCount = bsaFile.reset()
        Message(self,_("BSA Hashes reset: %d") % (resetCount,))

#------------------------------------------------------------------------------
class ScreensList(List):
    #--Class Data
    colLinks = screensMainMenu #--Column menu
    itemLinks = screensItemMenu #--Single item menu

    def __init__(self,parent):
        #--Columns
        self.cols = settings['bash.screens.cols']
        self.colAligns = settings['bash.screens.colAligns']
        self.colNames = settings['bash.colNames']
        self.colReverse = settings.getChanged('bash.screens.colReverse')
        self.colWidths = settings['bash.screens.colWidths']
        #--Data/Items
        self.data = bosh.screensData = bosh.ScreensData()
        self.sort = settings['bash.screens.sort']
        #--Links
        self.colLinks = ScreensList.colLinks
        self.itemLinks = ScreensList.itemLinks
        #--Parent init
        List.__init__(self,parent,-1,ctrlStyle=(wx.LC_REPORT|wx.SUNKEN_BORDER))
        #--Events
        wx.EVT_LIST_ITEM_SELECTED(self,self.listId,self.OnItemSelected)

    def RefreshUI(self,files='ALL',detail='SAME'):
        """Refreshes UI for specified files."""
        #--Details
        if detail == 'SAME':
            selected = set(self.GetSelected())
        else:
            selected = set([detail])
        #--Populate
        if files == 'ALL':
            self.PopulateItems(selected=selected)
        elif isinstance(files,StringTypes):
            self.PopulateItem(files,selected=selected)
        else: #--Iterable
            for file in files:
                self.PopulateItem(file,selected=selected)
        bashFrame.SetStatusCount()

    #--Populate Item
    def PopulateItem(self,itemDex,mode=0,selected=set()):
        #--String name of item?
        if not isinstance(itemDex,int):
            itemDex = self.items.index(itemDex)
        fileName = Path.get(self.items[itemDex])
        fileInfo = self.data[fileName]
        cols = self.cols
        for colDex in range(self.numCols):
            col = cols[colDex]
            if col == 'File':
                value = fileName
            elif col == 'Modified':
                value = bosh.formatDate(fileInfo[1])
            else:
                value = '-'
            if mode and (colDex == 0):
                self.list.InsertStringItem(itemDex, value)
            else:
                self.list.SetStringItem(itemDex, colDex, value)
        #--Image
        #--Selection State
        if fileName in selected:
            self.list.SetItemState(itemDex,wx.LIST_STATE_SELECTED,wx.LIST_STATE_SELECTED)
        else:
            self.list.SetItemState(itemDex,0,wx.LIST_STATE_SELECTED)

    #--Sort Items
    def SortItems(self,col=None,reverse=-2):
        (col, reverse) = self.GetSortSettings(col,reverse)
        settings['bash.screens.sort'] = col
        data = self.data
        #--Start with sort by name
        self.items.sort(lambda a,b: cmp(a.lower(),b.lower()))
        if col == 'File':
            pass #--Done by default
        elif col == 'Modified':
            self.items.sort(key=lambda a: data[a][1])
        else:
            raise BashError(_('Unrecognized sort key: ')+col)
        #--Ascending
        if reverse: self.items.reverse()

    #--Events ---------------------------------------------
    #--Column Resize
    def OnColumnResize(self,event):
        colDex = event.GetColumn()
        colName = self.cols[colDex]
        self.colWidths[colName] = self.list.GetColumnWidth(colDex)
        settings.setChanged('bash.screens.colWidths')

    def OnItemSelected(self,event=None):
        fileName = self.items[event.m_itemIndex]
        filePath = bosh.screensData.dir.join(fileName)
        bitmap = (filePath.exists() and 
            wx.Bitmap(filePath)) or None
        self.picture.SetBitmap(bitmap)

#------------------------------------------------------------------------------
class ScreensPanel(NotebookPanel):
    """Screenshots tab."""
    def __init__(self,parent):
        """Initialize."""
        wx.Panel.__init__(self, parent, -1)
        #--Left
        sashPos = settings.get('bash.screens.sashPos',120)
        left = self.left = leftSash(self,defaultSize=(sashPos,100),onSashDrag=self.OnSashDrag)
        right = self.right =  wx.Panel(self,style=wx.NO_BORDER)
        #--Contents
        global screensList
        screensList = ScreensList(left)
        screensList.SetSizeHints(100,100)
        screensList.picture = Picture(right,256,192)
        #--Events
        self.Bind(wx.EVT_SIZE,self.OnSize)
        #--Layout
        #left.SetSizer(hSizer((screensList,1,wx.GROW),((10,0),0)))
        right.SetSizer(hSizer((screensList.picture,1,wx.GROW)))
        wx.LayoutAlgorithm().LayoutWindow(self, right)

    def SetStatusCount(self):
        """Sets status bar count field."""
        text = _('Screens: %d') % (len(screensList.data.data),)
        statusBar.SetStatusText(text,2)

    def OnSashDrag(self,event):
        """Handle sash moved."""
        wMin,wMax = 80,self.GetSizeTuple()[0]-80
        sashPos = max(wMin,min(wMax,event.GetDragRect().width))
        self.left.SetDefaultSize((sashPos,10))
        wx.LayoutAlgorithm().LayoutWindow(self, self.right)
        screensList.picture.Refresh()
        settings['bash.screens.sashPos'] = sashPos

    def OnSize(self,event=None):
        wx.LayoutAlgorithm().LayoutWindow(self, self.right)

    def OnShow(self):
        """Panel is shown. Update self.data."""
        if bosh.screensData.refresh():
            screensList.RefreshUI()
            #self.Refresh()
        self.SetStatusCount()

#------------------------------------------------------------------------------
class MessageList(List):
    #--Class Data
    colLinks = messagesMainMenu #--Column menu
    itemLinks = messagesItemMenu #--Single item menu

    def __init__(self,parent):
        #--Columns
        self.cols = settings['bash.messages.cols']
        self.colAligns = settings['bash.messages.colAligns']
        self.colNames = settings['bash.colNames']
        self.colReverse = settings.getChanged('bash.messages.colReverse')
        self.colWidths = settings['bash.messages.colWidths']
        #--Data/Items
        self.data = bosh.messages = bosh.Messages()
        self.data.refresh()
        self.sort = settings['bash.messages.sort']
        #--Links
        self.colLinks = MessageList.colLinks
        self.itemLinks = MessageList.itemLinks
        #--Other
        self.gText = None
        self.searchResults = None
        #--Parent init
        List.__init__(self,parent,-1,ctrlStyle=(wx.LC_REPORT|wx.SUNKEN_BORDER))
        #--Events
        wx.EVT_LIST_ITEM_SELECTED(self,self.listId,self.OnItemSelected)

    def GetItems(self):
        """Set and return self.items."""
        if self.searchResults != None:
            self.items = list(self.searchResults)
        else:
            self.items = self.data.keys()
        return self.items

    def RefreshUI(self,files='ALL',detail='SAME'):
        """Refreshes UI for specified files."""
        #--Details
        if detail == 'SAME':
            selected = set(self.GetSelected())
        else:
            selected = set([detail])
        #--Populate
        if files == 'ALL':
            self.PopulateItems(selected=selected)
        elif isinstance(files,StringTypes):
            self.PopulateItem(files,selected=selected)
        else: #--Iterable
            for file in files:
                self.PopulateItem(file,selected=selected)
        bashFrame.SetStatusCount()

    #--Populate Item
    def PopulateItem(self,itemDex,mode=0,selected=set()):
        #--String name of item?
        if not isinstance(itemDex,int):
            itemDex = self.items.index(itemDex)
        item = self.items[itemDex]
        subject,author,date = self.data[item][:3]
        cols = self.cols
        for colDex in range(self.numCols):
            col = cols[colDex]
            if col == 'Subject':
                value = subject
            elif col == 'Author':
                value = author
            elif col == 'Date':
                value = bosh.formatDate(date)
            else:
                value = '-'
            if mode and (colDex == 0):
                self.list.InsertStringItem(itemDex, value)
            else:
                self.list.SetStringItem(itemDex, colDex, value)
        #--Image
        #--Selection State
        if item in selected:
            self.list.SetItemState(itemDex,wx.LIST_STATE_SELECTED,wx.LIST_STATE_SELECTED)
        else:
            self.list.SetItemState(itemDex,0,wx.LIST_STATE_SELECTED)

    #--Sort Items
    def SortItems(self,col=None,reverse=-2):
        (col, reverse) = self.GetSortSettings(col,reverse)
        settings['bash.messages.sort'] = col
        data = self.data
        #--Start with sort by date
        self.items.sort(key=lambda a: data[a][2])
        if col == 'Subject':
            reNoRe = re.compile('^Re: *')
            self.items.sort(key=lambda a: reNoRe.sub('',data[a][0]))
        elif col == 'Author':
            self.items.sort(key=lambda a: data[a][1])
        elif col == 'Date':
            pass #--Default sort
        else:
            raise BashError(_('Unrecognized sort key: ')+col)
        #--Ascending
        if reverse: self.items.reverse()

    #--Events ---------------------------------------------
    #--Column Resize
    def OnColumnResize(self,event):
        colDex = event.GetColumn()
        colName = self.cols[colDex]
        self.colWidths[colName] = self.list.GetColumnWidth(colDex)
        settings.setChanged('bash.messages.colWidths')

    def OnItemSelected(self,event=None):
        keys = self.GetSelected()
        path = bosh.dirs['saveBase'].join('Messages.html')
        bosh.messages.writeText(path,*keys)
        self.gText.Navigate(path,0x2) #--0x2: Clear History
        #self.list.SetFocus()

#------------------------------------------------------------------------------
class MessagePanel(NotebookPanel):
    """Messages tab."""
    def __init__(self,parent):
        """Initialize."""
        import wx.lib.iewin
        wx.Panel.__init__(self, parent, -1)
        #--Left
        sashPos = settings.get('bash.messages.sashPos',120)
        gTop = self.gTop =  topSash(self,defaultSize=(100,sashPos),onSashDrag=self.OnSashDrag)
        gBottom = self.gBottom =  wx.Panel(self,style=wx.NO_BORDER)
        #--Contents
        global gMessageList
        gMessageList = MessageList(gTop)
        gMessageList.SetSizeHints(100,100)
        gMessageList.gText = wx.lib.iewin.IEHtmlWindow(gBottom, -1, style = wx.NO_FULL_REPAINT_ON_RESIZE)
        #--Search
        gSearchBox = self.gSearchBox = wx.TextCtrl(gBottom,-1,"",style=wx.TE_PROCESS_ENTER)
        gSearchButton = button((gBottom,-1,_("Search")),self.DoSearch)
        gClearButton = button((gBottom,-1,_("Clear")),self.DoClear)
        #--Events
        #--Following line should use EVT_COMMAND_TEXT_ENTER, but that seems broken.
        gSearchBox.Bind(wx.EVT_CHAR,self.OnSearchChar) 
        self.Bind(wx.EVT_SIZE,self.OnSize)
        #--Layout
        gTop.SetSizer(hSizer(
            (gMessageList,1,wx.GROW)))
        gBottom.SetSizer(vSizer(
            (gMessageList.gText,1,wx.GROW),
            (hSizer(
                (gSearchBox,1,wx.GROW),
                (gSearchButton,0,wx.LEFT,4),
                (gClearButton,0,wx.LEFT,4),
                ),0,wx.GROW|wx.TOP,4),
            ))
        wx.LayoutAlgorithm().LayoutWindow(self, gTop)
        wx.LayoutAlgorithm().LayoutWindow(self, gBottom)

    def SetStatusCount(self):
        """Sets status bar count field."""
        if gMessageList.searchResults != None:
            numUsed = len(gMessageList.searchResults)
        else:
            numUsed = len(gMessageList.items)
        text = _('PMs: %d/%d') % (numUsed,len(gMessageList.data.keys()))
        statusBar.SetStatusText(text,2)

    def OnSashDrag(self,event):
        """Handle sash moved."""
        hMin,hMax = 80,self.GetSizeTuple()[1]-80
        sashPos = max(hMin,min(hMax,event.GetDragRect().height))
        self.gTop.SetDefaultSize((10,sashPos))
        wx.LayoutAlgorithm().LayoutWindow(self, self.gBottom)
        settings['bash.messages.sashPos'] = sashPos

    def OnSize(self,event=None):
        wx.LayoutAlgorithm().LayoutWindow(self, self.gTop)
        wx.LayoutAlgorithm().LayoutWindow(self, self.gBottom)

    def OnShow(self):
        """Panel is shown. Update self.data."""
        if bosh.messages.refresh():
            gMessageList.RefreshUI()
            #self.Refresh()
        self.SetStatusCount()

    def OnSearchChar(self,event):
        if event.GetKeyCode() == 13:
            self.DoSearch(None)
        else:
            event.Skip()

    def DoSearch(self,event):
        """Handle search button."""
        term = self.gSearchBox.GetValue()
        gMessageList.searchResults = gMessageList.data.search(term)
        gMessageList.RefreshUI()

    def DoClear(self,event):
        """Handle clear button."""
        self.gSearchBox.SetValue("")
        gMessageList.searchResults = None
        gMessageList.RefreshUI()

#------------------------------------------------------------------------------
class BashNotebook(wx.Notebook):
    def __init__(self, parent, id):
        wx.Notebook.__init__(self, parent, id)
        #--Pages
        self.AddPage(ReplacersPanel(self),_("Replacers"))
        self.AddPage(ModPanel(self),_("Mods"))
        self.AddPage(SavePanel(self),_("Saves"))
        self.AddPage(ScreensPanel(self),_("Screenshots"))
        if re.match('win',sys.platform):
            self.AddPage(MessagePanel(self),_("PM Archive"))
        self.Bind(wx.EVT_NOTEBOOK_PAGE_CHANGED,self.OnShowPage)
        #--Selection
        pageIndex = min(settings['bash.page'],self.GetPageCount()-1)
        self.SetSelection(pageIndex)
        #self.GetPage(pageIndex).OnShow()

    def OnShowPage(self,event):
        """Call page's OnShow command."""
        self.GetPage(event.GetSelection()).OnShow()
        event.Skip()

#------------------------------------------------------------------------------
class BashStatusBar(wx.StatusBar):
    #--Class Data
    links = statusBarButtons

    def __init__(self, parent):
        wx.StatusBar.__init__(self, parent, -1)
        global statusBar
        statusBar = self
        self.SetFieldsCount(3)
        links = BashStatusBar.links
        self.buttons = []
        for link in links:
            button = link.GetBitmapButton(self,style=wx.NO_BORDER)
            if button: self.buttons.append(button)
        self.SetStatusWidths([18*len(self.buttons),-1, 120])
        self.OnSize() #--Position buttons
        wx.EVT_SIZE(self,self.OnSize)
        #--Clear text notice
        self.Bind(wx.EVT_TIMER, self.OnTimer)
    
    def OnSize(self,event=None):
        rect = self.GetFieldRect(0)
        (xPos,yPos) = (rect.x+1,rect.y+1)
        for button in self.buttons:
            button.SetPosition((xPos,yPos))
            xPos += 18  
        if event: event.Skip()

    def SetText(self,text="",timeout=5):
        """Set's display text as specified. Empty string clears the field."""
        self.SetStatusText(text,1)
        if timeout > 0:
            wx.Timer(self).Start(timeout*1000,wx.TIMER_ONE_SHOT)

    def OnTimer(self,evt):
        """Clears display text as specified. Empty string clears the field."""
        self.SetStatusText("",1)

#------------------------------------------------------------------------------
class BashFrame(wx.Frame):
    """Main application frame."""
    def __init__(self, parent=None,pos=wx.DefaultPosition,size=(400,500),
             style = wx.DEFAULT_FRAME_STYLE):
        """Initialization."""
        #--Singleton
        global bashFrame
        bashFrame = self
        #--Window
        wx.Frame.__init__(self, parent, -1, 'Wrye Bash', pos, size,style)
        minSize = settings['bash.frameSize.min']
        self.SetSizeHints(minSize[0],minSize[1])
        self.SetTitle()
        #--Application Icons
        self.SetIcons(images['bash.icons'].GetIconBundle())
        #--Status Bar
        self.SetStatusBar(BashStatusBar(self))
        #--Notebook panel
        self.notebook = notebook = BashNotebook(self,-1)
        #--Events
        wx.EVT_CLOSE(self, self.OnCloseWindow)
        wx.EVT_ACTIVATE(self, self.RefreshData)
        #--Data
        self.knownCorrupted = set() 
        self.oblivionIniCorrupted = False
        self.incompleteInstallError = False
        #--Layout
        sizer = vSizer((notebook,1,wx.GROW))
        self.SetSizer(sizer)

    def SetTitle(self,title=None):
        """Set title. Set to default if no title supplied."""
        if not title:
            title = "Wrye Bash %s: " % (settings['bash.readme'][1],)
            maProfile = re.match(r'Saves\\(.+)\\$',bosh.saveInfos.localSave)
            if maProfile:
                title += maProfile.group(1)
            else:
                title += _("Default")
            if bosh.modInfos.voCurrent:
                title += ' ['+bosh.modInfos.voCurrent+']'
        wx.Frame.SetTitle(self,title)

    def SetStatusCount(self):
        """Sets the status bar count field. Actual work is done by current panel."""
        if hasattr(self,'notebook'): #--Hack to get around problem with screens tab.
            self.notebook.GetPage(self.notebook.GetSelection()).SetStatusCount()

    #--Events ---------------------------------------------
    def RefreshData(self, event=None):
        """Refreshes all data. Can be called manually, but is also triggered by window activation event."""
        #--Ignore deactivation events.
        if event and not event.GetActive(): return
        #--UPDATES-----------------------------------------
        popMods = popSaves = None
        #--Check plugins.txt and mods directory...
        if bosh.modInfos.refresh():
            popMods = 'ALL'
        #--Have any mtimes been reset?
        if bosh.modInfos.mtimesReset:
            resetList = '\n* '.join(bosh.modInfos.mtimesReset)
            del bosh.modInfos.mtimesReset[:]
            InfoMessage(self,_('Modified dates have been reset for some mod files:\n* %s')
                % (resetList,))
            popMods = 'ALL'
        #--Check savegames directory...
        if bosh.saveInfos.refresh():
            popSaves = 'ALL'
        #--Repopulate
        if popMods:
            modList.RefreshUI(popMods) #--Will repop saves too.
        elif popSaves:
            saveList.RefreshUI(popSaves)
        #--Current notebook panel
        self.notebook.GetPage(self.notebook.GetSelection()).OnShow()
        #--WARNINGS----------------------------------------
        #--Does plugins.txt have any bad or missing files?
        if bosh.modInfos.plugins.selectedBad:
            message = (_("Missing files have been removed from load list. (%s)") 
                % (', '.join(bosh.modInfos.plugins.selectedBad),))
            del bosh.modInfos.plugins.selectedBad[:]
            bosh.modInfos.plugins.save()
            WarningMessage(self,message)
        #--Was load list too long?
        if bosh.modInfos.plugins.selectedExtra:
            message = (_("Load list has been truncated because it was too long. (%s)") 
                % (', '.join(bosh.modInfos.plugins.selectedExtra),))
            del bosh.modInfos.plugins.selectedExtra[:]
            bosh.modInfos.plugins.save()
            WarningMessage(self,message)
        #--Any new corrupted files?
        message = ''
        corruptMods = set(bosh.modInfos.corrupted.keys())
        if not corruptMods <= self.knownCorrupted:
            message += _("The following mod files have corrupted headers: ")
            message += ','.join(sorted(corruptMods))+'.'
            self.knownCorrupted |= corruptMods
        corruptSaves = set(bosh.saveInfos.corrupted.keys())
        if not corruptSaves <= self.knownCorrupted:
            if message: message += '\n'
            message += _("The following save files have corrupted headers: ")
            message += ','.join(sorted(corruptSaves))+'.'
            self.knownCorrupted |= corruptSaves
        if message: WarningMessage(self,message)
        #--Corrupt Oblivion.ini
        if self.oblivionIniCorrupted != bosh.oblivionIni.isCorrupted:
            self.oblivionIniCorrupted = bosh.oblivionIni.isCorrupted
            if self.oblivionIniCorrupted:
                message = _('Your Oblivion.ini should begin with a section header (e.g. "[General]"), but does not. You should edit the file to correct this.')
                WarningMessage(self,fill(message))
        #--Any Y2038 Resets?
        if bosh.Path.mtimeResets:
            message = (_("Bash cannot handle dates greater than January 19, 2038. Accordingly, the dates for the following files have been reset to an earlier date: ") +
                ', '.join(sorted(bosh.Path.mtimeResets))+'.')
            del bosh.Path.mtimeResets[:]
            WarningMessage(self,message)
        #--OBMM Warning?
        if settings['bosh.modInfos.obmmWarn'] == 1:
            settings['bosh.modInfos.obmmWarn'] = 2
            message = _("Turn Lock Times Off?\n\nLock Times a feature which resets load order to a previously memorized state. While this feature is good for maintaining your load order, it will also undo any load order changes that you have made in OBMM.")
            result = YesQuery(self,message,_("Lock Times"))
            resetMTimes = settings['bosh.modInfos.resetMTimes'] = not result
            bosh.modInfos.resetMTimes = resetMTimes
            if resetMTimes:
                bosh.modInfos.refreshMTimes()
            else:
                bosh.modInfos.mtimes.clear()
            message = _("Lock Times is now %s. To change it in the future, right click on the main list header on the Mods tab and select 'Lock Times'.") 
            Message(self,message % ((_('off'),_('on'))[resetMTimes],),_("Lock Times"))
        #--Missing docs directory?
        testFile = Path.get(bosh.dirs['app']).join('Data','Docs','wtxt_teal.css')
        if not self.incompleteInstallError and not testFile.exists():
            self.incompleteInstallError = True
            message = _("Installation appears incomplete. Please re-unzip bash to Oblivion directory so that ALL files are installed.\n\nCorrect installation will create Oblivion\\Mopy, Oblivion\\Data\\Docs and Oblvion\\Data\\INI Tweaks directories.")
            WarningMessage(self,message,_("Incomplete Installation"))

    def OnCloseWindow(self, event):
        """Handle Close event. Save application data."""
        self.CleanSettings()
        if docBrowser: docBrowser.DoSave()
        if not self.IsIconized() and not self.IsMaximized():
            settings['bash.framePos'] = self.GetPositionTuple()
            settings['bash.frameSize'] = self.GetSizeTuple()
        settings['bash.page'] = self.notebook.GetSelection()
        bosh.modInfos.table.save()
        bosh.saveInfos.profiles.save()
        if bosh.messages: bosh.messages.save()
        settings.save()
        self.Destroy()

    def CleanSettings(self):
        """Cleans junk from settings before closing."""
        #--Clean rename dictionary.
        modNames = set(bosh.modInfos.data.keys())
        modNames.update(bosh.modInfos.table.data.keys())
        renames = bosh.settings.getChanged('bash.mods.renames')
        for key,value in renames.items():
            if value not in modNames:
                del renames[key]
        #--Clean backup
        for fileInfos in (bosh.modInfos,bosh.saveInfos):
            goodRoots = set(path.root() for path in fileInfos.data.keys())
            backupDir = fileInfos.dir.join('Bash','Backups')
            if not backupDir.isdir(): continue
            for name in backupDir.list():
                path = backupDir.join(name)
                if name.root() not in goodRoots and path.isfile():
                    path.remove()

#------------------------------------------------------------------------------
class DocBrowser(wx.Frame):
    """Doc Browser frame."""
    def __init__(self,modName=None):
        """Intialize.
        modName -- current modname (or None)."""
        import wx.lib.iewin
        #--Data
        self.modName = Path.get(modName or '')
        self.data = bosh.modInfos.table.getColumn('doc')
        self.docEdit = bosh.modInfos.table.getColumn('docEdit')
        self.docType = None
        self.docIsWtxt = False
        #--Singleton
        global docBrowser
        docBrowser = self
        #--Window
        pos = settings['bash.modDocs.pos']
        size = settings['bash.modDocs.size']
        wx.Frame.__init__(self, bashFrame, -1, _('Doc Browser'), pos, size,
            style=wx.DEFAULT_FRAME_STYLE)
        self.SetBackgroundColour(wx.NullColour)
        self.SetSizeHints(250,250)
        #--Mod Name
        self.modNameBox = wx.TextCtrl(self,-1,style=wx.TE_READONLY)
        self.modNameList = wx.ListBox(self,-1,choices=sorted(self.data.keys()),style=wx.LB_SINGLE|wx.LB_SORT)
        self.modNameList.Bind(wx.EVT_LISTBOX,self.DoSelectMod)
        #wx.EVT_COMBOBOX(self.modNameBox,ID_SELECT,self.DoSelectMod)
        #--Application Icons
        self.SetIcons(images['bash.icons2'].GetIconBundle())
        #--Set Doc
        self.setButton = button((self,ID_SET,_("Set Doc...")),self.DoSet)
        #--Forget Doc
        self.forgetButton = button((self,wx.ID_DELETE,_("Forget Doc...")),self.DoForget)
        #--Rename Doc
        self.renameButton = button((self,ID_RENAME,_("Rename Doc...")),self.DoRename)
        #--Edit Doc
        self.editButton = wx.ToggleButton(self,ID_EDIT,_("Edit Doc..."))
        wx.EVT_TOGGLEBUTTON(self.editButton,ID_EDIT,self.DoEdit)
        #--Html Back
        bitmap = wx.ArtProvider_GetBitmap(wx.ART_GO_BACK,wx.ART_HELP_BROWSER, (16,16))
        self.prevButton = bitmapButton((self,-1,bitmap),self.DoPrevPage)
        #--Html Forward
        bitmap = wx.ArtProvider_GetBitmap(wx.ART_GO_FORWARD,wx.ART_HELP_BROWSER, (16,16))
        self.nextButton = bitmapButton((self,-1,bitmap),self.DoNextPage)
        #--Doc Name
        self.docNameBox = wx.TextCtrl(self,-1,style=wx.TE_READONLY)
        #--Doc display
        self.plainText = wx.TextCtrl(self,-1,style=wx.TE_READONLY|wx.TE_MULTILINE|wx.TE_RICH2|wx.SUNKEN_BORDER)
        self.htmlText = wx.lib.iewin.IEHtmlWindow(self, -1, style = wx.NO_FULL_REPAINT_ON_RESIZE)
        #--Events
        wx.EVT_CLOSE(self, self.OnCloseWindow)
        #--Layout
        self.mainSizer = vSizer(
            (hSizer( #--Buttons
                (self.setButton,0,wx.GROW),
                (self.forgetButton,0,wx.GROW),
                (self.renameButton,0,wx.GROW),
                (self.editButton,0,wx.GROW),
                (self.prevButton,0,wx.GROW),
                (self.nextButton,0,wx.GROW),
                ),0,wx.GROW|wx.ALL^wx.BOTTOM,4),
            (hSizer( #--Mod name, doc name
                #(self.modNameBox,2,wx.GROW|wx.RIGHT,4),
                (self.docNameBox,2,wx.GROW),
                ),0,wx.GROW|wx.TOP|wx.BOTTOM,4),
            (self.plainText,3,wx.GROW),
            (self.htmlText,3,wx.GROW),
            )
        sizer = hSizer(
            (vSizer(
                (self.modNameBox,0,wx.GROW),
                (self.modNameList,1,wx.GROW|wx.TOP,4),
                ),0,wx.GROW|wx.TOP|wx.RIGHT,4),
            (self.mainSizer,1,wx.GROW),
            )
        #--Set
        self.SetSizer(sizer)
        self.SetMod(modName)
        self.SetDocType('txt')

    def GetIsWtxt(self,docPath=None):
        """Determines whether specified path is a wtxt file."""
        docPath = docPath or Path.get(self.data.get(self.modName,''))
        if not docPath.exists(): return False
        textFile = docPath.open()
        maText = re.match(r'^=.+=#\s*$',textFile.readline())
        textFile.close()
        return (maText != None)

    def DoHome(self, event):
        """Handle "Home" button click."""
        self.htmlText.GoHome()

    def DoPrevPage(self, event):
        """Handle "Back" button click."""
        self.htmlText.GoBack()

    def DoNextPage(self, event):
        """Handle "Next" button click."""
        self.htmlText.GoForward()

    def DoEdit(self,event):
        """Handle "Edit Doc" button click."""
        self.DoSave()
        editing = self.editButton.GetValue()
        self.docEdit[self.modName] = editing
        self.docIsWtxt = self.GetIsWtxt()
        if self.docIsWtxt:
            self.SetMod(self.modName)
        else:
            self.plainText.SetEditable(editing)

    def DoForget(self,event):
        """Handle "Forget Doc" button click.
        Sets help document for current mod name to None."""
        #--Already have mod data?
        modName = self.modName
        if modName not in self.data:
            return
        index = self.modNameList.FindString(modName)
        if index != wx.NOT_FOUND:
            self.modNameList.Delete(index)
        del self.data[modName]
        self.SetMod(modName)

    def DoSelectMod(self,event):
        """Handle mod name combobox selection."""
        self.SetMod(event.GetString())

    def DoSet(self,event):
        """Handle "Set Doc" button click."""
        #--Already have mod data?
        modName = self.modName
        if modName in self.data:
            (docsDir,fileName) = Path.get(self.data[modName]).split()
        else:
            docsDir = settings['bash.modDocs.dir'] or bosh.dirs['mods']
            fileName = ''
        #--Dialog
        path = OpenDialog(self,_("Select doc for %s:") % (modName,),
            docsDir,fileName, '*.*')
        if not path: return
        settings['bash.modDocs.dir'] = path.head()
        if modName not in self.data:
            self.modNameList.Append(modName)
        self.data[modName] = path
        self.SetMod(modName)

    def DoRename(self,event):
        """Handle "Rename Doc" button click."""
        modName = self.modName
        oldPath = Path.get(self.data[modName])
        (workDir,fileName) = oldPath.split()
        #--Dialog
        path = SaveDialog(self,_("Rename file to:"),workDir,fileName, '*.*')
        if not path or path == oldPath: return
        #--OS renaming
        path.remove()
        oldPath.rename(path)
        if self.docIsWtxt:
            oldHtml, newHtml = (xxx.root()+'.html' for xxx in (oldPath,path))
            newHtml.remove()
            if oldHtml.exists(): oldHtml.rename(newHtml)
        #--Remember change
        self.data[modName] = path
        self.SetMod(modName)

    def DoSave(self):
        """Saves doc, if necessary."""
        if not self.plainText.IsModified(): return
        try:
            docPath = self.data.get(self.modName,'')
            if not docPath: 
                raise mosh.Error(_('Filename not defined.'))
            self.plainText.SaveFile(docPath)
            self.plainText.DiscardEdits()
            if self.docIsWtxt:
                import wtxt
                docsDir = str(bosh.modInfos.dir.join('Docs'))
                wtxt.genHtml(docPath, cssDir=docsDir)
        except:
            ErrorMessage(self,_("Failed to save changes to %s doc file!" % (self.modName,)))

    def SetMod(self,modName=None):
        """Sets the mod to show docs for."""
        #--Save Current Edits
        self.DoSave()
        #--New modName
        self.modName = modName = Path.get(modName or '')
        #--ModName
        if modName:
            self.modNameBox.SetValue(modName)
            index = self.modNameList.FindString(modName)
            self.modNameList.SetSelection(index)
            self.setButton.Enable(True)
        else:
            self.modNameBox.SetValue('')
            self.modNameList.SetSelection(wx.NOT_FOUND)
            self.setButton.Enable(False)
        #--Doc Data
        docPath = Path.get(self.data.get(modName,''))
        docExt = docPath.ext().lower()
        self.docNameBox.SetValue(docPath.tail())
        self.forgetButton.Enable(docPath != '')
        self.renameButton.Enable(docPath != '')
        #--Edit defaults to false.
        self.editButton.SetValue(False)
        self.editButton.Enable(False)
        self.plainText.SetEditable(False)
        self.docIsWtxt = False
        #--View/edit doc.
        if not docPath:
            self.plainText.SetValue('')
            self.SetDocType('txt')
        elif not docPath.exists():
            myTemplate = bosh.modInfos.dir.join('Docs',_('My Readme Template.txt'))
            bashTemplate = bosh.modInfos.dir.join('Docs',_('Bash Readme Template.txt'))
            if myTemplate.exists():
                template = ''.join(myTemplate.open().readlines())
            elif bashTemplate.exists():
                template = ''.join(bashTemplate.open().readlines())
            else:
                template = '= $modName '+('='*(74-len(modName)))+'#\n'+docPath
            defaultText = string.Template(template).substitute(modName=str(modName))
            self.plainText.SetValue(defaultText)
            self.SetDocType('txt')
            if docExt in set(('.txt','.etxt')):
                self.editButton.Enable(True)
                editing = self.docEdit.get(modName,True)
                self.editButton.SetValue(editing)
                self.plainText.SetEditable(editing)
            self.docIsWtxt = (docExt == '.txt')
        elif docExt in set(('.htm','.html','.mht')):
            self.htmlText.Navigate(docPath,0x2) #--0x2: Clear History
            self.SetDocType('html')
        else:
            self.editButton.Enable(True)
            editing = self.docEdit.get(modName,False)
            self.editButton.SetValue(editing)
            self.plainText.SetEditable(editing)
            self.docIsWtxt = self.GetIsWtxt(docPath)
            htmlPath = self.docIsWtxt and (docPath.root()+'.html')
            if htmlPath and (not os.path.exists(htmlPath) or 
                (os.path.getmtime(docPath) > os.path.getmtime(htmlPath))
                ):
                import wtxt
                docsDir = str(bosh.modInfos.dir.join('Docs'))
                wtxt.genHtml(docPath,cssDir=docsDir)
            if not editing and htmlPath and os.path.exists(htmlPath):
                self.htmlText.Navigate(htmlPath,0x2) #--0x2: Clear History
                self.SetDocType('html')
            else:
                self.plainText.LoadFile(docPath)
                self.SetDocType('txt')
        
    #--Set Doc Type
    def SetDocType(self,docType):
        """Shows the plainText or htmlText view depending on document type (i.e. file name extension)."""
        if docType == self.docType: 
            return
        sizer = self.mainSizer
        if docType == 'html':
            sizer.Show(self.plainText,False)
            sizer.Show(self.htmlText,True)
            self.prevButton.Enable(True)
            self.nextButton.Enable(True)
        else:
            sizer.Show(self.plainText,True)
            sizer.Show(self.htmlText,False)
            self.prevButton.Enable(False)
            self.nextButton.Enable(False)
        self.Layout()

    #--Window Closing
    def OnCloseWindow(self, event):
        """Handle window close event.
        Remember window size, position, etc."""
        self.DoSave()
        settings['bash.modDocs.show'] = False
        if not self.IsIconized() and not self.IsMaximized():
            settings['bash.modDocs.pos'] = self.GetPositionTuple()
            settings['bash.modDocs.size'] = self.GetSizeTuple()
        self.Destroy()

#------------------------------------------------------------------------------
class BashApp(wx.App):
    """Bash Application class."""
    def OnInit(self):
        """wxWindows: Initialization handler."""
        #--Init Data
        oldErr,errLog = sys.stderr,None
        if isinstance(oldErr,wx.PyOnDemandOutputWindow):
            errLog = bosh.dirs['app'].join('Mopy','bash.log')
            sys.stderr = sys.stdout = errLog.open('w')
        progress = wx.ProgressDialog("Wrye Bash",_("Initializing Data"))
        self.InitData(progress)
        progress.Update(70,_("Initializing Version"))
        self.InitVersion()
        #--Locale (Only in later versions of wxPython??)
        #wx.Locale(wx.LOCALE_LOAD_DEFAULT) #~~Not clear that it's needed and/or does anything.
        progress.Destroy()
        #--MWFrame
        progress.Update(80,_("Initializing Windows"))
        frame = BashFrame(
             pos=settings['bash.framePos'], 
             size=settings['bash.frameSize'])
        sys.stderr = sys.stdout = oldErr
        self.SetTopWindow(frame)
        frame.Show()
        if errLog: errLog.remove()
        #--DocBrowser
        if settings['bash.modDocs.show']:
            DocBrowser().Show()
        return True
    
    def InitData(self,progress):
        """Initialize all data. Called by OnInit()."""
        progress.Update(5,_("Initializing ModInfos"))
        bosh.oblivionIni = bosh.OblivionIni()
        bosh.modInfos = bosh.ModInfos()
        bosh.modInfos.refresh()
        progress.Update(30,_("Initializing SaveInfos"))
        bosh.saveInfos = bosh.SaveInfos()
        bosh.saveInfos.refresh()
        #--Patch check
        firstBashed = settings.get('bash.patch.firstBashed',False)
        if not firstBashed:
            for modInfo in bosh.modInfos.values():
                if modInfo.header.author == 'BASHED PATCH': break
            else:
                progress.Update(68,_("Generating Blank Bashed Patch"))
                patchInfo = bosh.ModInfo(bosh.modInfos.dir,Path.get('Bashed Patch, 0.esp'))
                patchInfo.mtime = max([time.time()]+[info.mtime for info in bosh.modInfos.values()])
                patchFile = bosh.ModFile(patchInfo)
                patchFile.tes4.author = 'BASHED PATCH'
                patchFile.safeSave()
                bosh.modInfos.refresh()
            settings['bash.patch.firstBashed'] = True

    def InitVersion(self):
        """Perform any version to version conversion. Called by OnInit()."""
        #--Renames dictionary: Strings to Paths.
        if settings['bash.version'] < 40:
            #--Renames array
            newRenames = {}
            for key,value in settings['bash.mods.renames'].items():
                newRenames[Path.get(key)] = Path.get(value)
            settings['bash.mods.renames'] = newRenames
            #--Mod table data
            modTableData = bosh.modInfos.table.data
            for key in modTableData.keys():
                if not isinstance(key,Path):
                    modTableData[Path.get(key)] = modTableData[key]
                    del modTableData[key]
        #--Window sizes by class name rather than by class
        if settings['bash.version'] < 43:
            windowSizes = settings.getChanged('bash.window.sizes')
            keys = windowSizes.keys()
            for key in keys:
                if isinstance(key,ClassType):
                    windowSizes[key.__name__] = windowSizes[key]
                    del windowSizes[key]
        #--Current Version
        settings['bash.version'] = 43
        #--Version from readme
        readme = bosh.dirs['app'].join('Mopy','Wrye Bash.txt')
        if readme.exists() and readme.getmtime() != settings['bash.readme'][0]:
            reVersion = re.compile("^=== ([\.\d]+) \[")
            for line in readme.open():
                maVersion = reVersion.match(line)
                if maVersion:
                    settings['bash.readme'] = (readme.getmtime(),maVersion.group(1))
                    break

# Misc Dialogs ----------------------------------------------------------------
#------------------------------------------------------------------------------
class ImportFaceDialog(wx.Dialog):
    """Dialog for importing faces."""
    def __init__(self,parent,id,title,fileInfo,faces):
        #--Data
        self.fileInfo = fileInfo
        if faces and isinstance(faces.keys()[0],(IntType,LongType)):
            self.data = dict(('%08X %s' % (key,face.pcName),face) for key,face in faces.items())
        else:
            self.data = faces
        self.items = sorted(self.data.keys(),key=string.lower)
        #--GUI
        wx.Dialog.__init__(self,parent,id,title,
            style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)
        wx.EVT_CLOSE(self, self.OnCloseWindow)
        self.SetSizeHints(550,300)
        #--List Box
        self.list = wx.ListBox(self,wx.ID_OK,choices=self.items,style=wx.LB_SINGLE)
        self.list.SetSizeHints(175,150)
        wx.EVT_LISTBOX(self,wx.ID_OK,self.EvtListBox)
        #--Name,Race,Gender
        self.nameCheck = wx.CheckBox(self,-1,_('Name'))
        self.nameText  = wx.StaticText(self,-1,'-----------------------------')
        self.raceCheck = wx.CheckBox(self,-1,_('Race'))
        self.raceText  = wx.StaticText(self,-1,'')
        self.genderCheck = wx.CheckBox(self,-1,_('Gender'))
        self.genderText  = wx.StaticText(self,-1,'')
        self.statsCheck = wx.CheckBox(self,-1,_('Stats'))
        self.statsText  = wx.StaticText(self,-1,'')
        #self.nameText.SetSizeHints(40,-1)
        flags = settings.get('bash.faceImport.flags',0x4)
        self.nameCheck.SetValue(flags & 0x1)
        self.raceCheck.SetValue(flags & 0x2)
        self.genderCheck.SetValue(flags & 0x4)
        self.statsCheck.SetValue(flags & 0x40)
        #--Other
        importButton = button((self,-1,_('Import')),self.DoImport)
        importButton.SetDefault()
        self.picture = Picture(self,350,210,scaling=2)
        #--Layout
        fgSizer = wx.FlexGridSizer(3,2,2,4)
        fgSizer.AddGrowableCol(1,1)
        fgSizer.AddMany([
            self.nameCheck,
            self.nameText,
            self.raceCheck,
            self.raceText,
            self.genderCheck,
            self.genderText,
            self.statsCheck,
            self.statsText,
            ])
        sizer = hSizer(
            (self.list,1,wx.EXPAND|wx.TOP,4),
            (vSizer(
                self.picture,
                (hSizer(
                    (fgSizer,1),
                    (vSizer(
                        (importButton,0,wx.ALIGN_RIGHT),
                        (button((self,wx.ID_CANCEL)),0,wx.TOP,4),
                        )),
                    ),0,wx.EXPAND|wx.TOP,4),
                ),0,wx.EXPAND|wx.ALL,4),
            )
        #--Done
        if 'ImportFaceDialog' in settings['bash.window.sizes']:
            self.SetSizer(sizer)
            self.SetSize(settings['bash.window.sizes']['ImportFaceDialog'])
        else:
            self.SetSizerAndFit(sizer)

    def EvtListBox(self,event):
        """Responds to listbox selection."""
        itemDex = event.GetSelection()
        item = self.items[itemDex]
        face = self.data[item]
        self.nameText.SetLabel(face.pcName)
        self.raceText.SetLabel(face.getRaceName())
        self.genderText.SetLabel(face.getGenderName())
        self.statsText.SetLabel(_('Health ')+`face.health`)
        itemImagePath = bosh.dirs['mods'].join(r'Docs\Images\%s.jpg' % (item,))
        bitmap = (itemImagePath.exists() and 
            wx.Bitmap(itemImagePath,wx.BITMAP_TYPE_JPEG)) or None
        self.picture.SetBitmap(bitmap)

    def DoImport(self,event):
        """Imports selected face into save file."""
        selections = self.list.GetSelections()
        if not selections:
            wx.Bell()
            return
        itemDex = selections[0]
        item = self.items[itemDex]
        #--Do import
        flags = (
            (self.nameCheck.GetValue()   and 0x1) |
            (self.raceCheck.GetValue()   and 0x2) |
            (self.genderCheck.GetValue() and 0x4) |
            (self.statsCheck.GetValue() and 0x40) |
            0x18) #--Hair and eyes on by default.
        settings['bash.faceImport.flags'] = flags 
        bosh.PCFaces.save_setFace(self.fileInfo,self.data[item],flags)
        Message(self,_('Face imported.'))
        self.EndModal(wx.ID_OK)
        
    #--Window Closing
    def OnCloseWindow(self, event):
        """Handle window close event.
        Remember window size, position, etc."""
        sizes = settings.getChanged('bash.window.sizes')
        sizes['ImportFaceDialog'] = self.GetSizeTuple()
        self.Destroy()

# Patchers 00 ------------------------------------------------------------------
#------------------------------------------------------------------------------
class PatchDialog(wx.Dialog):
    """Bash Patch update dialog."""
    patchers = [] #--All patchers. These are copied as needed.

    def __init__(self,parent,patchInfo):
        """Initialized."""
        self.parent = parent
        size = settings['bash.window.sizes'].get(self.__class__.__name__,(400,400))
        wx.Dialog.__init__(self,parent,-1,_("Update ")+patchInfo.name, size=size,
            style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)
        self.SetSizeHints(400,300)
        #--Data
        groupOrder = dict([(group,index) for index,group in 
            enumerate((_('General'),_('Importers'),_('Tweakers'),_('Special')))])
        patchConfigs = bosh.modInfos.table.getItem(patchInfo.name,'bash.patch.configs',{})
        self.patchInfo = patchInfo
        self.patchers = [copy.deepcopy(patcher) for patcher in PatchDialog.patchers]
        self.patchers.sort(key=lambda a: a.__class__.name)
        self.patchers.sort(key=lambda a: groupOrder[a.__class__.group])
        for patcher in self.patchers:
            patcher.getConfig(patchConfigs) #--Will set patcher.isEnabled
            if 'UNDEFINED' in (patcher.__class__.group, patcher.__class__.group):
                raise bosh.UncodedError('Name or group not defined for: '+patcher.__class__.__name__)
        self.currentPatcher = None
        patcherNames = [patcher.getName() for patcher in self.patchers]
        #--GUI elements
        self.gExecute = button((self,wx.ID_OK),self.Execute)
        self.gPatchers = wx.CheckListBox(self,-1,choices=patcherNames,style=wx.LB_SINGLE)
        for index,patcher in enumerate(self.patchers):
            self.gPatchers.Check(index,patcher.isEnabled)
        self.gTipText = wx.StaticText(self,-1,'')
        #--Events
        self.Bind(wx.EVT_SIZE,self.OnSize)
        self.gPatchers.Bind(wx.EVT_LISTBOX, self.OnSelect)
        self.gPatchers.Bind(wx.EVT_CHECKLISTBOX, self.OnCheck)
        self.gPatchers.Bind(wx.EVT_MOTION,self.OnMouse)
        self.gPatchers.Bind(wx.EVT_LEAVE_WINDOW,self.OnMouse)
        self.mouseItem = -1
        #--Layout
        self.gConfigSizer = gConfigSizer = vSizer()
        sizer = vSizer(
            (hSizer(
                (self.gPatchers,0,wx.EXPAND),
                (self.gConfigSizer,1,wx.EXPAND|wx.LEFT,4),
                ),1,wx.EXPAND|wx.ALL,4),
            (self.gTipText,0,wx.EXPAND|wx.ALL^wx.TOP,4),
            (wx.StaticLine(self),0,wx.EXPAND|wx.BOTTOM,4),
            (hSizer(
                ((0,0),1),
                self.gExecute,
                (button((self,wx.ID_CANCEL)),0,wx.LEFT,4),
                ),0,wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM,4)
            )
        self.SetSizer(sizer)
        #--Patcher panels
        for patcher in self.patchers:
            gConfigPanel = patcher.GetConfigPanel(self,gConfigSizer,self.gTipText)
            gConfigSizer.Show(gConfigPanel,False)
        self.ShowPatcher(self.patchers[0])
        self.SetOkEnable()

    #--Core -------------------------------
    def SetOkEnable(self):
        """Sets enable state for Ok button."""
        for patcher in self.patchers:
            if patcher.isEnabled: 
                return self.gExecute.Enable(True)
        self.gExecute.Enable(False)

    def ShowPatcher(self,patcher):
        """Show patcher panel."""
        gConfigSizer = self.gConfigSizer
        if patcher == self.currentPatcher: return
        if self.currentPatcher != None:
            gConfigSizer.Show(self.currentPatcher.gConfigPanel,False)
        gConfigPanel = patcher.GetConfigPanel(self,gConfigSizer,self.gTipText)
        gConfigSizer.Show(gConfigPanel,True)
        self.Layout()
        patcher.Layout()
        self.currentPatcher = patcher

    def Execute(self,event=None):
        """Do the patch."""
        self.EndModal(wx.ID_OK)
        patchName = self.patchInfo.name
        progress = ProgressDialog(patchName)
        try:
            #--Save configs
            patchConfigs = {'ImportedMods':set()}
            for patcher in self.patchers:
                patcher.saveConfig(patchConfigs)
            bosh.modInfos.table.setItem(patchName,'bash.patch.configs',patchConfigs)
            #--Do it
            log = bosh.LogFile(cStringIO.StringIO())
            nullProgress = bosh.Progress()
            patchers = [patcher for patcher in self.patchers if patcher.isEnabled]
            patchFile = bosh.PatchFile(self.patchInfo,patchers)
            patchFile.loadSourceData(SubProgress(progress,0,0.1))
            patchFile.postSourceData(SubProgress(progress,0.1,0.2))
            patchFile.scanLoadMods(SubProgress(progress,0.2,0.8))
            patchFile.buildPatch(log,SubProgress(progress,0.8,0.9))
            #--Save
            progress(0.9,_('Saving: ')+patchName)
            patchFile.safeSave()
            patchFile.endPatch()
            #--Cleanup
            self.patchInfo.refresh()
            modList.RefreshUI(patchName)
            #--Done
            progress.Destroy()
            #--Readme and log
            import wtxt
            log.setHeader(None)
            log('{{CSS:wtxt_sand_small.css}}')
            logValue = log.out.getvalue()
            readme = bosh.modInfos.dir.join('Docs',patchName.root()+'.txt')
            readme.open('w').write(logValue)
            bosh.modInfos.table.setItem(patchName,'doc',readme)
            #--Convert log/readme to wtxt and show log
            docsDir = str(bosh.modInfos.dir.join('Docs'))
            wtxt.genHtml(readme,cssDir=docsDir)
            WtxtLogMessage(self.parent,readme.root()+'.html',patchName)
            #--Select?
            message = _("Activate %s?") % (patchName,)
            if not bosh.modInfos.isSelected(patchName) and YesQuery(self.parent,message):
                try:
                    oldFiles = bosh.modInfos.ordered[:]
                    bosh.modInfos.select(patchName)
                    changedFiles = bosh.listSubtract(bosh.modInfos.ordered,oldFiles)
                    if len(changedFiles) > 1:
                        statusBar.SetText(_("Masters Activated: ") + `len(changedFiles)-1`)
                except bosh.PluginsFullError:
                    ErrorMessage(self,_("Unable to add mod %s because load list is full." )
                        % (fileName,))
                modList.RefreshUI() 
        except:
            progress.Destroy()
            raise

    #--GUI --------------------------------
    def OnSize(self,event):
        sizes = settings.getChanged('bash.window.sizes')
        sizes[self.__class__.__name__] = self.GetSizeTuple()
        self.Layout()
        self.currentPatcher.Layout()

    def OnSelect(self,event):
        """Responds to patchers list selection."""
        itemDex = event.GetSelection()
        self.ShowPatcher(self.patchers[itemDex])

    def OnCheck(self,event):
        """Toggle patcher activity state."""
        index = event.GetSelection()
        patcher = self.patchers[index]
        patcher.isEnabled = self.gPatchers.IsChecked(index)
        self.gPatchers.SetSelection(index)
        self.ShowPatcher(patcher)
        self.SetOkEnable()

    def OnMouse(self,event):
        """Check mouse motion to detect right click event."""
        if event.Moving():
            mouseItem = (event.m_y/self.gPatchers.GetItemHeight() + 
                self.gPatchers.GetScrollPos(wx.VERTICAL))
            if mouseItem != self.mouseItem:
                self.mouseItem = mouseItem
                self.MouseEnteredItem(mouseItem)
        elif event.Leaving():
            self.gTipText.SetLabel('')
            self.mouseItem = -1
        event.Skip()

    def MouseEnteredItem(self,item):
        """Show tip text when changing item."""
        #--Following isn't displaying correctly.
        if item < len(self.patchers):
            patcherClass = self.patchers[item].__class__
            tip = patcherClass.tip or re.sub(r'\..*','.',patcherClass.text)
            self.gTipText.SetLabel(tip)
        else:
            self.gTipText.SetLabel('')

#------------------------------------------------------------------------------
class Patcher:
    """Basic patcher panel with no options."""
    def GetConfigPanel(self,parent,gConfigSizer,gTipText):
        """Show config."""
        if not self.gConfigPanel: 
            self.gTipText = gTipText
            gConfigPanel = self.gConfigPanel = wx.Window(parent,-1)
            text = textwrap.fill(self.__class__.text,70)
            gText = wx.StaticText(self.gConfigPanel,-1,text)
            #gText = wx.TextCtrl(self.gConfigPanel,-1,self.__class__.text*5,style=wx.TE_READONLY|wx.TE_MULTILINE)
            gSizer = vSizer(gText)
            gConfigPanel.SetSizer(gSizer)
            gConfigSizer.Add(gConfigPanel,1,wx.EXPAND)
        return self.gConfigPanel

    def Layout(self):
        """Layout control components."""
        if self.gConfigPanel:
            self.gConfigPanel.Layout()

#------------------------------------------------------------------------------
class AliasesPatcher(Patcher,bosh.AliasesPatcher):
    """Basic patcher panel with no options."""
    def GetConfigPanel(self,parent,gConfigSizer,gTipText):
        """Show config."""
        if self.gConfigPanel: return self.gConfigPanel
        #--Else...
        #--Tip
        self.gTipText = gTipText
        gConfigPanel = self.gConfigPanel = wx.Window(parent,-1)
        text = textwrap.fill(self.__class__.text,70)
        gText = wx.StaticText(gConfigPanel,-1,text)
        #gExample = wx.StaticText(gConfigPanel,-1,
        #    _("Example Mod 1.esp >> Example Mod 1.2.esp"))
        #--Aliases Text
        self.gAliases = wx.TextCtrl(gConfigPanel,-1,'',style=wx.TE_MULTILINE)
        self.gAliases.Bind(wx.EVT_KILL_FOCUS, self.OnEditAliases)
        self.SetAliasText()
        #--Sizing
        gSizer = vSizer(
            gText,
            #(gExample,0,wx.EXPAND|wx.TOP,8),
            (self.gAliases,1,wx.EXPAND|wx.TOP,4))
        gConfigPanel.SetSizer(gSizer)
        gConfigSizer.Add(gConfigPanel,1,wx.EXPAND)
        return self.gConfigPanel

    def SetAliasText(self):
        """Sets alias text according to current aliases."""
        self.gAliases.SetValue('\n'.join([
            '%s >> %s' % (key,value) for key,value in sorted(self.aliases.items())]))

    def OnEditAliases(self,event):
        text = self.gAliases.GetValue()
        self.aliases.clear()
        for line in text.split('\n'):
            fields = map(string.strip,line.split('>>'))
            if len(fields) != 2 or not fields[0] or not fields[1]: continue
            self.aliases[fields[0]] = fields[1]
        self.SetAliasText()

#------------------------------------------------------------------------------
class ListPatcher(Patcher):
    """Patcher panel with option to select source elements."""
    listLabel = _("Source Mods/Files")

    def GetConfigPanel(self,parent,gConfigSizer,gTipText):
        """Show config."""
        if self.gConfigPanel: return self.gConfigPanel
        #--Else...
        self.forceItemCheck = self.__class__.forceItemCheck
        self.gTipText = gTipText
        gConfigPanel = self.gConfigPanel = wx.Window(parent,-1)
        text = textwrap.fill(self.__class__.text,70)
        gText = wx.StaticText(self.gConfigPanel,-1,text)
        if self.forceItemCheck:
            self.gList =wx.ListBox(gConfigPanel,-1)
        else:
            self.gList =wx.CheckListBox(gConfigPanel,-1)
            self.gList.Bind(wx.EVT_CHECKLISTBOX,self.OnListCheck)
        #--Events
        self.gList.Bind(wx.EVT_MOTION,self.OnMouse)
        self.gList.Bind(wx.EVT_RIGHT_DOWN,self.OnMouse)
        self.gList.Bind(wx.EVT_RIGHT_UP,self.OnMouse)
        self.mouseItem = -1
        self.mouseState = None
        #--Manual controls
        if self.forceAuto:
            gManualSizer = None
            self.SetItems(self.getAutoItems())
        else:
            self.gAuto = checkBox((gConfigPanel,-1,_("Automatic")),self.OnAutomatic)
            self.gAuto.SetValue(self.autoIsChecked)
            self.gAdd = button((gConfigPanel,-1,_("Add")),self.OnAdd)
            self.gRemove = button((gConfigPanel,-1,_("Remove")),self.OnRemove)
            self.OnAutomatic()
            gManualSizer = (vSizer(
                (self.gAuto,0,wx.TOP,2),
                (self.gAdd,0,wx.TOP,12),
                (self.gRemove,0,wx.TOP,4),
                ),0,wx.EXPAND|wx.LEFT,4)
        #--Init GUI
        self.SetItems(self.configItems)
        #--Layout
        gSizer = vSizer(
            (gText,),
            (hsbSizer((gConfigPanel,-1,self.__class__.listLabel),
                ((4,0),0,wx.EXPAND),
                (self.gList,1,wx.EXPAND|wx.TOP,2),
                gManualSizer,
                ),1,wx.EXPAND|wx.TOP,4),
            )
        gConfigPanel.SetSizer(gSizer)
        gConfigSizer.Add(gConfigPanel,1,wx.EXPAND)
        return gConfigPanel

    def SetItems(self,items):
        """Set item to specified set of items."""
        items = self.items = self.sortConfig(items)
        forceItemCheck = self.forceItemCheck
        defaultItemCheck = self.__class__.defaultItemCheck
        self.gList.Clear()
        for index,item in enumerate(items):
            self.gList.Insert(self.getItemLabel(item),index)
            if forceItemCheck: 
                self.configChecks[item] = True
            else:
                self.gList.Check(index,self.configChecks.setdefault(item,defaultItemCheck))
        self.configItems = items
        
    def OnListCheck(self,event=None):
        """One of list items was checked. Update all configChecks states."""
        for index,item in enumerate(self.items):
            self.configChecks[item] = self.gList.IsChecked(index)

    def OnAutomatic(self,event=None):
        """Automatic checkbox changed."""
        self.autoIsChecked = self.gAuto.IsChecked()
        self.gAdd.Enable(not self.autoIsChecked)
        self.gRemove.Enable(not self.autoIsChecked)
        if self.autoIsChecked:
            self.SetItems(self.getAutoItems())

    def OnAdd(self,event):
        """Add button clicked."""
        srcDir = bosh.modInfos.dir
        wildcard = _('Oblivion Mod Files')+' (*.esp;*.esm)|*.esp;*.esm'
        #--File dialog
        title = _("Get ")+self.__class__.listLabel
        srcPaths = MultiOpenDialog(self.gConfigPanel,title,srcDir, '', wildcard)
        if not srcPaths: return
        #--Get new items
        for srcPath in srcPaths:
            root,tail = srcPath.split()
            if srcDir == root and tail not in self.configItems:
                self.configItems.append(tail)
        self.SetItems(self.configItems)

    def OnRemove(self,event):
        """Remove button clicked."""
        selected = self.gList.GetSelections()
        newItems = [item for index, item in enumerate(self.configItems) if index not in selected]
        self.SetItems(newItems)

    #--Choice stuff ---------------------------------------
    def OnMouse(self,event):
        """Check mouse motion to detect right click event."""
        if event.RightDown():
            self.mouseState = (event.m_x,event.m_y)
            event.Skip()
        elif event.RightUp() and self.mouseState:
            self.ShowChoiceMenu(event)
        elif event.Dragging():
            if self.mouseState:
                oldx,oldy = self.mouseState
                if max(abs(event.m_x-oldx),abs(event.m_y-oldy)) > 4:
                    self.mouseState = None
        else:
            self.mouseState = False
            event.Skip()

    def ShowChoiceMenu(self,event):
        """Displays a popup choice menu if applicable. 
        NOTE: Assume that configChoice returns a set of chosen items."""
        if not self.choiceMenu: return
        #--Item Index
        if self.forceItemCheck:
            itemHeight = self.gList.GetCharHeight()
        else:
            itemHeight = self.gList.GetItemHeight()
        itemIndex = event.m_y/itemHeight + self.gList.GetScrollPos(wx.VERTICAL)
        if itemIndex >= len(self.items): return
        self.rightClickItemIndex = itemIndex
        choiceSet = self.getChoice(self.items[itemIndex])
        #--Build Menu
        menu = wx.Menu()
        for index,label in enumerate(self.choiceMenu):
            if label == '----':
                menu.AppendSeparator()
            else:
                menuItem = wx.MenuItem(menu,index,label,kind=wx.ITEM_CHECK)
                menu.AppendItem(menuItem)
                if label in choiceSet: menuItem.Check()
                wx.EVT_MENU(self.gList,index,self.OnItemChoice)
        #--Show/Destroy Menu
        self.gList.PopupMenu(menu)
        menu.Destroy()

    def OnItemChoice(self,event):
        """Handle choice menu selection."""
        itemIndex = self.rightClickItemIndex
        item =self.items[itemIndex]
        choice = self.choiceMenu[event.GetId()]
        choiceSet = self.configChoices[item]
        choiceSet ^= set((choice,))
        if choice != 'Auto':
            choiceSet.discard('Auto')
        elif 'Auto' in self.configChoices[item]:
            self.getChoice(item)
        self.gList.SetString(itemIndex,self.getItemLabel(item))

#------------------------------------------------------------------------------
class TweakPatcher(Patcher):
    """Patcher panel with list of checkable, configurable tweaks."""
    listLabel = _("Tweaks")

    def GetConfigPanel(self,parent,gConfigSizer,gTipText):
        """Show config."""
        if self.gConfigPanel: return self.gConfigPanel
        #--Else...
        self.gTipText = gTipText
        gConfigPanel = self.gConfigPanel = wx.Window(parent,-1)
        text = textwrap.fill(self.__class__.text,70)
        gText = wx.StaticText(self.gConfigPanel,-1,text)
        self.gList =wx.CheckListBox(gConfigPanel,-1)
        #--Events
        self.gList.Bind(wx.EVT_CHECKLISTBOX,self.OnListCheck)
        self.gList.Bind(wx.EVT_MOTION,self.OnMouse)
        self.gList.Bind(wx.EVT_LEAVE_WINDOW,self.OnMouse)
        self.gList.Bind(wx.EVT_RIGHT_DOWN,self.OnMouse)
        self.gList.Bind(wx.EVT_RIGHT_UP,self.OnMouse)
        self.mouseItem = -1
        self.mouseState = None
        #--Init GUI
        self.SetItems()
        #--Layout
        gSizer = vSizer(
            (gText,),
            #(hsbSizer((gConfigPanel,-1,self.__class__.listLabel),
                #((4,0),0,wx.EXPAND),
                (self.gList,1,wx.EXPAND|wx.TOP,2),
                #),1,wx.EXPAND|wx.TOP,4),
            )
        gConfigPanel.SetSizer(gSizer)
        gConfigSizer.Add(gConfigPanel,1,wx.EXPAND)
        return gConfigPanel

    def SetItems(self):
        """Set item to specified set of items."""
        self.gList.Clear()
        for index,tweak in enumerate(self.tweaks):
            self.gList.Insert(tweak.getListLabel(),index)
            self.gList.Check(index,tweak.isEnabled)
        
    def OnListCheck(self,event=None):
        """One of list items was checked. Update all check states."""
        for index,tweak in enumerate(self.tweaks):
            tweak.isEnabled = self.gList.IsChecked(index)

    def OnMouse(self,event):
        """Check mouse motion to detect right click event."""
        if event.RightDown():
            self.mouseState = (event.m_x,event.m_y)
            event.Skip()
        elif event.RightUp() and self.mouseState:
            self.ShowChoiceMenu(event)
        elif event.Leaving():
            self.gTipText.SetLabel('')
            self.mouseState = False
            event.Skip()
        elif event.Dragging():
            if self.mouseState:
                oldx,oldy = self.mouseState
                if max(abs(event.m_x-oldx),abs(event.m_y-oldy)) > 4:
                    self.mouseState = None
        elif event.Moving():
            mouseItem = event.m_y/self.gList.GetItemHeight() + self.gList.GetScrollPos(wx.VERTICAL)
            self.mouseState = False
            if mouseItem != self.mouseItem:
                self.mouseItem = mouseItem
                self.MouseEnteredItem(mouseItem)
            event.Skip()
        else:
            self.mouseState = False
            event.Skip()

    def MouseEnteredItem(self,item):
        """Show tip text when changing item."""
        #--Following isn't displaying correctly.
        tip = item < len(self.tweaks) and self.tweaks[item].tip
        if tip:
            self.gTipText.SetLabel(tip)
        else:
            self.gTipText.SetLabel('')

    def ShowChoiceMenu(self,event):
        """Displays a popup choice menu if applicable."""
        #--Tweak Index
        tweakIndex = event.m_y/self.gList.GetItemHeight() + self.gList.GetScrollPos(wx.VERTICAL)
        self.rightClickTweakIndex = tweakIndex
        #--Tweaks
        tweaks = self.tweaks
        if tweakIndex >= len(tweaks): return
        choiceLabels = tweaks[tweakIndex].choiceLabels
        chosen = tweaks[tweakIndex].chosen
        if len(choiceLabels) <= 1: return
        #--Build Menu
        menu = wx.Menu()
        for index,label in enumerate(choiceLabels):
            if label == '----':
                menu.AppendSeparator()
            else:
                menuItem = wx.MenuItem(menu,index,label,kind=wx.ITEM_CHECK)
                menu.AppendItem(menuItem)
                if index == chosen: menuItem.Check()
                wx.EVT_MENU(self.gList,index,self.OnTweakChoice)
        #--Show/Destroy Menu
        self.gList.PopupMenu(menu)
        menu.Destroy()

    def OnTweakChoice(self,event):
        """Handle choice menu selection."""
        tweakIndex = self.rightClickTweakIndex
        self.tweaks[tweakIndex].chosen = event.GetId()
        self.gList.SetString(tweakIndex,self.tweaks[tweakIndex].getListLabel())

# Patchers 10 ------------------------------------------------------------------
class PatchMerger(bosh.PatchMerger,ListPatcher):
    listLabel = _("Mergeable Mods")

# Patchers 20 ------------------------------------------------------------------
class GraphicsPatcher(bosh.GraphicsPatcher,ListPatcher): pass

class NamesPatcher(bosh.NamesPatcher,ListPatcher): pass

class NpcFacePatcher(bosh.NpcFacePatcher,ListPatcher): pass

class RacePatcher(bosh.RacePatcher,ListPatcher):
    listLabel = _("Race Mods")

class StatsPatcher(bosh.StatsPatcher,ListPatcher): pass

# Patchers 30 ------------------------------------------------------------------
class AssortedTweaker(bosh.AssortedTweaker,TweakPatcher): pass

class ClothesTweaker(bosh.ClothesTweaker,TweakPatcher): pass

class GmstTweaker(bosh.GmstTweaker,TweakPatcher): pass

class NamesTweaker(bosh.NamesTweaker,TweakPatcher): pass

# Patchers 40 ------------------------------------------------------------------
class AlchemicalCatalogs(bosh.AlchemicalCatalogs,Patcher): pass

class ListsMerger(bosh.ListsMerger,ListPatcher):
    listLabel = _("Override Delev/Relev Tags")

class MFactMarker(bosh.MFactMarker,ListPatcher): pass

class PowerExhaustion(bosh.PowerExhaustion,Patcher): pass

class SEWorldEnforcer(bosh.SEWorldEnforcer,Patcher): pass

#------------------------------------------------------------------------------
#------------------------------------------------------------------------------
# Init Patchers 
PatchDialog.patchers.extend((
    AliasesPatcher(),
    AssortedTweaker(),
    PatchMerger(),
    AlchemicalCatalogs(),
    ClothesTweaker(),
    GmstTweaker(),
    GraphicsPatcher(),
    ListsMerger(),
    MFactMarker(),
    NamesPatcher(),
    NamesTweaker(),
    NpcFacePatcher(),
    PowerExhaustion(),
    RacePatcher(),
    StatsPatcher(),
    SEWorldEnforcer(),
    ))

# Links -----------------------------------------------------------------------
#------------------------------------------------------------------------------
class Link:
    """Abstract link."""
    def __init__(self):
        self.id = None

    def AppendToMenu(self,menu,window,data):
        self.window = window
        self.data = data
        if not self.id: self.id = wx.NewId()
        wx.EVT_MENU(window,self.id,self.Do)

    def Do(self, event):
        """Event: link execution."""
        raise bosh.AbstractError

#------------------------------------------------------------------------------
class SeparatorLink(Link):
    def AppendToMenu(self,menu,window,data):
        menu.AppendSeparator()

#------------------------------------------------------------------------------
class MenuLink(Link):
    def __init__(self,name):
        Link.__init__(self)
        self.name = name
        self.links = Links()

    def AppendToMenu(self,menu,window,data):
        subMenu = wx.Menu()
        for link in self.links:
            link.AppendToMenu(subMenu,window,data)
        menu.AppendMenu(-1,self.name,subMenu)

# Files Links -----------------------------------------------------------------
#------------------------------------------------------------------------------
class Files_Open(Link):
    """Opens data directory in explorer."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Open...'))
        menu.AppendItem(menuItem)

    def Do(self,event):
        """Handle selection."""
        dir = self.window.data.dir
        dir.makedirs()
        os.startfile(dir)

#------------------------------------------------------------------------------
class Files_SortBy(Link):
    """Sort files by specified key (sortCol)."""
    def __init__(self,sortCol,prefix=''):
        Link.__init__(self)
        self.sortCol = sortCol
        self.sortName = settings['bash.colNames'][sortCol]
        self.prefix = prefix

    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,self.prefix+self.sortName,kind=wx.ITEM_CHECK)
        menu.AppendItem(menuItem)
        if window.sort == self.sortCol: menuItem.Check()

    def Do(self,event):
        self.window.PopulateItems(self.sortCol,-1)

#------------------------------------------------------------------------------
class Files_Unhide(Link):
    """Unhide file(s). (Move files back to Data Files or Save directory.)"""
    def __init__(self,type='mod'):
        Link.__init__(self)
        self.type = type

    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_("Unhide..."))
        menu.AppendItem(menuItem)

    def Do(self,event):
        destDir = self.window.data.dir
        srcDir = destDir.join('Bash','Hidden')
        if self.type == 'mod':
            wildcard = 'Oblivion Mod Files (*.esp;*.esm)|*.esp;*.esm'
        elif self.type == 'save':
            wildcard = 'Oblivion Save files (*.ess)|*.ess'
        else:
            wildcard = '*.*'
        #--File dialog
        srcDir.makedirs()
        srcPaths = MultiOpenDialog(self.window,_('Unhide files:'),srcDir, '', wildcard)
        if not srcPaths: return
        #--Iterate over Paths
        for srcPath in srcPaths:
            #--Copy from dest directory?
            (newSrcDir,srcFileName) = srcPath.split()
            if newSrcDir == destDir:
                ErrorMessage(self.window,_("You can't unhide files from this directory."))
                return
            #--File already unhidden?
            destPath = destDir.join(srcFileName)
            if destPath.exists():
                WarningMessage(self.window,_("File skipped: %s. File is already present.") 
                    % (srcFileName,))
            #--Move it?
            else:
                srcPath.move(destPath)
        #--Repopulate
        bashFrame.RefreshData()

# File Links ------------------------------------------------------------------
#------------------------------------------------------------------------------
class File_Delete(Link):
    """Delete the file and all backups."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menu.AppendItem(wx.MenuItem(menu,self.id,_('Delete')))

    def Do(self,event):
        message = _(r'Delete these files? This operation cannot be undone.')
        message += '\n* ' + '\n* '.join(sorted(self.data))
        if not YesQuery(self.window,message,_('Delete Files')):
            return
        #--Do it
        for fileName in self.data:
            self.window.data.delete(fileName)
        #--Refresh stuff
        self.window.RefreshUI()

#------------------------------------------------------------------------------
class File_Duplicate(Link):
    """Create a duplicate of the file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Duplicate...'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        data = self.data
        fileName = Path.get(data[0])
        fileInfos = self.window.data
        fileInfo = fileInfos[fileName]
        #--Mod with resources?
        #--Warn on rename if file has bsa and/or dialog
        if fileInfo.isMod() and tuple(fileInfo.hasResources()) != (False,False):
            hasBsa, hasVoices = fileInfo.hasResources()
            modName = fileInfo.name
            if hasBsa and hasVoices:
                message = _("This mod has an associated archive (%s.bsa) and an associated voice directory (Sound\\Voices\\%s), which will not be attached to the duplicate mod.\n\nNote that the BSA archive may also contain a voice directory (Sound\\Voices\\%s), which would remain detached even if a duplicate archive were also created.") % (modName[:-4],modName,modName)
            elif hasBsa:
                message = _("This mod has an associated archive (%s.bsa), which will not be attached to the duplicate mod.\n\nNote that this BSA archive may contain a voice directory (Sound\\Voices\\%s), which would remain detached even if a duplicate archive were also created.") % (modName[:-4],modName)
            else: #hasVoices
                message = _("This mod has an associated voice directory (Sound\\Voice\\%s), which will not be attached to the duplicate mod.") % (modName,)
            if Message(self.window,message,style=wx.OK|wx.CANCEL|wx.ICON_EXCLAMATION) != wx.ID_OK:
                return
        #--Continue copy
        (root,ext) = fileName.splitext()
        if ext.lower() == '.bak': ext = '.ess'
        (destDir,destName,wildcard) = (fileInfo.dir, root+' Copy'+ext,'*'+ext)
        destDir.makedirs()
        destPath = SaveDialog(self.window,_('Duplicate as:'),
            destDir,destName,wildcard)
        if not destPath: return
        destDir,destName = destPath.split()
        if (destDir == fileInfo.dir) and (destName == fileName):
            ErrorMessage(self.window,_("Files cannot be duplicated to themselves!"))
            return
        fileInfos.copy(fileName,destDir,destName,mtime='+1')
        if destDir == fileInfo.dir:
            fileInfos.table.copyRow(fileName,destName)
            if fileInfos.table.getItem(fileName,'mtime'):
                destInfo = fileInfos[destName]
                fileInfos.table.setItem(destName,'mtime',destInfo.mtime)
            if fileInfo.isMod():
                fileInfos.autoSort()
        self.window.RefreshUI()

#------------------------------------------------------------------------------
class File_Hide(Link):
    """Hide the file. (Move it to Bash/Hidden directory.)"""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menu.AppendItem(wx.MenuItem(menu,self.id,_('Hide')))

    def Do(self,event):
        message = _(r'Hide these files? Note that hidden files are simply moved to the Bash\Hidden subdirectory.')
        if not YesQuery(self.window,message,_('Hide Files')): return
        #--Do it
        destRoot = self.window.data.dir.join('Bash','Hidden')
        fileInfos = self.window.data
        fileGroups = fileInfos.table.getColumn('group')
        for fileName in self.data:
            destDir = destRoot
            #--Use author subdirectory instead?
            author = getattr(fileInfos[fileName].header,'author','NOAUTHOR') #--Hack for save files.
            authorDir = destRoot.join(author)
            if author and authorDir.isdir():
                destDir = authorDir
            #--Use group subdirectory instead?
            elif fileName in fileGroups:
                groupDir = destRoot.join(fileGroups[fileName])
                if groupDir.isdir():
                    destDir = groupDir
            if not self.window.data.moveIsSafe(fileName,destDir):
                message = (_('A file named %s already exists in the hidden files directory. Overwrite it?') 
                    % (fileName,))
                if not YesQuery(self.window,message,_('Hide Files')): continue
            #--Do it
            self.window.data.move(fileName,destDir)
        #--Refresh stuff
        self.window.RefreshUI()

#------------------------------------------------------------------------------
class File_ListMasters(Link):
    """Copies list of masters to clipboard."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_("List Masters..."))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        text = bosh.modInfos.getModList(fileInfo)
        if (wx.TheClipboard.Open()):
            wx.TheClipboard.SetData(wx.TextDataObject(text))
            wx.TheClipboard.Close()
        LogMessage(self.window,text,fileName,asDialog=False,fixedFont=False)

#------------------------------------------------------------------------------
class File_Redate(Link):
    """Move the selected files to start at a specified date."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Redate...'))
        menu.AppendItem(menuItem)

    def Do(self,event):
        #--Get current start time.
        modInfos = self.window.data
        fileNames = [mod for mod in self.data if mod not in modInfos.autoSorted]
        if not fileNames: return
        #--Ask user for revised time.
        dialog = wx.TextEntryDialog(self.window,_('Redate selected mods starting at...'),
            _('Redate Mods'),bosh.formatDate(int(time.time())))
        result = dialog.ShowModal()
        newTimeStr = dialog.GetValue()
        dialog.Destroy()
        if result != wx.ID_OK: return
        try:
            newTimeTup = time.strptime(newTimeStr,'%c')
            newTime = int(time.mktime(newTimeTup))
        except ValueError:
            ErrorMessage(self.window,_('Unrecognized date: ')+newTimeStr)
            return
        except OverflowError:
            ErrorMessage(self,_('Bash cannot handle dates greater than January 19, 2038.)'))
            return
        #--Do it
        selInfos = [modInfos[fileName] for fileName in fileNames]
        selInfos.sort(key=lambda a: a.mtime)
        for fileInfo in selInfos:
            fileInfo.setMTime(newTime)
            newTime += 60
        #--Refresh
        modInfos.autoSort()
        modInfos.refreshInfoLists()
        self.window.RefreshUI()

#------------------------------------------------------------------------------
class File_Sort(Link):
    """Sort the selected files."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Sort'))
        menu.AppendItem(menuItem)
        if len(data) < 2: menuItem.Enable(False)

    def Do(self,event):
        message = _("Reorder selected mods in alphabetical order? The first file will be given the date/time of the current earliest file in the group, with consecutive files following at 1 minute increments.\n\nNote that this operation cannot be undone. Note also that some mods need to be in a specific order to work correctly, and this sort operation may break that order.")
        if ContinueQuery(self.window,message,'bash.sortMods.continue',_('Sort Mods')) != wx.ID_OK:
            return
        #--Get first time from first selected file.
        modInfos = self.window.data
        fileNames = [mod for mod in self.data if mod not in modInfos.autoSorted]
        if not fileNames: return
        dotTimes = [modInfos[fileName].mtime for fileName in fileNames if fileName[0] in '.=+']
        if dotTimes:
            newTime = min(dotTimes)
        else:
            newTime = min(modInfos[fileName].mtime for fileName in self.data)
        #--Do it
        fileNames.sort(key=lambda a: a[:-4].lower())
        fileNames.sort(key=lambda a: a[0] not in '.=')
        for fileName in fileNames:
            modInfos[fileName].setMTime(newTime)
            newTime += 60
        #--Refresh
        modInfos.autoSort()
        modInfos.refreshInfoLists()
        self.window.RefreshUI()

#------------------------------------------------------------------------------
class File_Snapshot(Link):
    """Take a snapshot of the file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Snapshot...'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        data = self.data
        fileName = Path.get(data[0])
        fileInfo = self.window.data[fileName]
        (destDir,destName,wildcard) = fileInfo.getNextSnapshot()
        destDir.makedirs()
        destPath = SaveDialog(self.window,_('Save snapshot as:'),
            destDir,destName,wildcard)
        if not destPath: return
        (destDir,destName) = destPath.split()
        #--Extract version number
        fileRoot = fileName.root()
        destRoot = destName.root()
        fileVersion = bosh.getMatch(re.search(r'[ _]+v?([\.0-9]+)$',fileRoot),1)
        snapVersion = bosh.getMatch(re.search(r'-[0-9\.]+$',destRoot))
        fileHedr = fileInfo.header
        if fileInfo.isMod() and (fileVersion or snapVersion) and bosh.reVersion.search(fileHedr.description):
            if fileVersion and snapVersion:
                newVersion = fileVersion+snapVersion
            elif snapVersion:
                newVersion = snapVersion[1:]
            else:
                newVersion = fileVersion
            newDescription = bosh.reVersion.sub(r'\1 '+newVersion, fileHedr.description,1)
            fileInfo.writeDescription(newDescription)
            self.window.details.SetFile(fileName)
        #--Copy file
        self.window.data.copy(fileName,destDir,destName)

#------------------------------------------------------------------------------
class File_RevertToSnapshot(Link):
    """Revert to Snapshot."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Revert to Snapshot...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data) == 1)

    def Do(self,event):
        """Handle menu item selection."""
        fileInfo = self.window.data[self.data[0]]
        fileName = fileInfo.name
        #--Snapshot finder
        destDir = self.window.data.dir
        srcDir = destDir.join('Bash','Snapshots')
        wildcard = fileInfo.getNextSnapshot()[2]
        #--File dialog
        srcDir.makedirs()
        snapPath = OpenDialog(self.window,_('Revert %s to snapshot:') % (fileName,),
            srcDir, '', wildcard)
        if not snapPath: return
        snapName = snapPath.tail()
        #--Warning box
        message = (_("Revert %s to snapshot %s dated %s?") 
            % (fileInfo.name,snapName,bosh.formatDate(snapPath.getmtime())))
        if not YesQuery(self.window,message,_('Revert to Snapshot')): return
        wx.BeginBusyCursor()
        destPath = fileInfo.dir.join(fileInfo.name)
        snapPath.copyfile(destPath)
        fileInfo.setMTime()
        try:
            self.window.data.refreshFile(fileName)
        except bosh.FileError:
            ErrorMessage(self,_('Snapshot file is corrupt!'))
            self.window.details.SetFile(None)
        wx.EndBusyCursor()
        self.window.RefreshUI(fileName)

#------------------------------------------------------------------------------
class File_Backup(Link):
    """Backup file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Backup'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data)==1)

    def Do(self,event):
        fileInfo = self.window.data[self.data[0]]
        fileInfo.makeBackup(True)

#------------------------------------------------------------------------------
class File_RevertToBackup:
    """Revert to last or first backup."""
    def AppendToMenu(self,menu,window,data):
        self.window = window
        self.data = data
        #--Backup Files
        singleSelect = len(data) == 1
        self.fileInfo = window.data[data[0]]
        #--Backup Item
        wx.EVT_MENU(window,ID_REVERT_BACKUP,self.Do)
        menuItem = wx.MenuItem(menu,ID_REVERT_BACKUP,_('Revert to Backup'))
        menu.AppendItem(menuItem)
        self.backup = self.fileInfo.dir.join('Bash','Backups',self.fileInfo.name)
        menuItem.Enable(singleSelect and self.backup.exists())
        #--First Backup item
        wx.EVT_MENU(window,ID_REVERT_FIRST,self.Do)
        menuItem = wx.MenuItem(menu,ID_REVERT_FIRST,_('Revert to First Backup'))
        menu.AppendItem(menuItem)
        self.firstBackup = self.backup +'f'
        menuItem.Enable(singleSelect and self.firstBackup.exists())

    def Do(self,event):
        fileInfo = self.fileInfo
        fileName = fileInfo.name
        #--Backup/FirstBackup?
        if event.GetId() ==  ID_REVERT_BACKUP:
            backup = self.backup
        else:
            backup = self.firstBackup
        #--Warning box
        message = _("Revert %s to backup dated %s?") % (fileName,
            bosh.formatDate(backup.getmtime()))
        if YesQuery(self.window,message,_('Revert to Backup')):
            wx.BeginBusyCursor()
            dest = fileInfo.dir.join(fileName)
            backup.copyfile(dest)
            fileInfo.setMTime()
            try:
                self.window.data.refreshFile(fileName)
            except bosh.FileError:
                ErrorMessage(self,_('Old file is corrupt!'))
            wx.EndBusyCursor()
        self.window.RefreshUI(fileName)
    
#------------------------------------------------------------------------------
class File_Open(Link):
    """Open specified file(s)."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Open...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data)>0)

    def Do(self,event):
        """Handle selection."""
        dir = self.window.data.dir
        for file in self.data:
            filePath = dir.join(file)
            os.startfile(filePath)

# Mods Links ------------------------------------------------------------------
class Mods_ReplacersData:
    """Empty version of a now removed class. Here for compatibility with 
    older settings files."""
    pass

class Mod_MergedLists_Data:
    """Empty version of a now removed class. Here for compatibility with 
    older settings files."""
    pass

#------------------------------------------------------------------------------
class Mods_LoadListData(ListEditorData):
    """Data capsule for load list editing dialog."""
    def __init__(self,parent):
        """Initialize."""
        self.data = settings['bash.loadLists.data']
        #--GUI
        ListEditorData.__init__(self,parent)
        self.showRename = True
        self.showRemove = True

    def getItemList(self):
        """Returns load list keys in alpha order."""
        return sorted(self.data.keys(),key=lambda a: a.lower())

    def rename(self,oldName,newName):
        """Renames oldName to newName."""
        #--Right length?
        if len(newName) == 0 or len(newName) > 64:
            ErrorMessage(self.parent,
                _('Name must be between 1 and 64 characters long.'))
            return False
        #--Rename
        settings.setChanged('bash.loadLists.data')
        self.data[newName] = self.data[oldName]
        del self.data[oldName]
        return newName

    def remove(self,item):
        """Removes load list."""
        settings.setChanged('bash.loadLists.data')
        del self.data[item]
        return True

#------------------------------------------------------------------------------
class Mods_LoadList:
    """Add load list links."""
    def __init__(self):
        self.data = settings['bash.loadLists.data']

    def GetItems(self):
        items = self.data.keys()
        items.sort(lambda a,b: cmp(a.lower(),b.lower()))
        return items

    def SortWindow(self):
        self.window.PopulateItems()

    def AppendToMenu(self,menu,window,data):
        self.window = window
        menu.Append(ID_LOADERS.NONE,_('None'))
        menu.Append(ID_LOADERS.SAVE,_('Save List...')) 
        menu.Append(ID_LOADERS.EDIT,_('Edit Lists...')) 
        menu.AppendSeparator()
        for id,item in zip(ID_LOADERS,self.GetItems()):
            menu.Append(id,item)
        #--Disable Save?
        if not bosh.modInfos.ordered:
            menu.FindItemById(ID_LOADERS.SAVE).Enable(False)
        #--Events
        wx.EVT_MENU(window,ID_LOADERS.NONE,self.DoNone)
        wx.EVT_MENU(window,ID_LOADERS.SAVE,self.DoSave)
        wx.EVT_MENU(window,ID_LOADERS.EDIT,self.DoEdit)
        wx.EVT_MENU_RANGE(window,ID_LOADERS.BASE,ID_LOADERS.MAX,self.DoList)

    def DoNone(self,event):
        """Unselect all mods."""
        bosh.modInfos.selectExact([])
        self.window.PopulateItems()

    def DoList(self,event):
        """Select mods in list."""
        item = self.GetItems()[event.GetId()-ID_LOADERS.BASE]
        selectList = [Path(modName) for modName in self.data[item]]
        errorMessage = bosh.modInfos.selectExact(selectList)
        self.window.PopulateItems()
        if errorMessage:
            ErrorMessage(self.window,errorMessage,item)

    def DoSave(self,event):
        #--No slots left?
        if len(self.data) >= (ID_LOADERS.MAX - ID_LOADERS.BASE + 1):
            ErrorMessage(self,_('All load list slots are full. Please delete an existing load list before adding another.'))
            return
        #--Dialog
        dialog = wx.TextEntryDialog(self.window,_('Save current load list as:'),
                'Wrye Bash')
        result = dialog.ShowModal()
        if result == wx.ID_OK:
            newItem = dialog.GetValue()
            dialog.Destroy()
            if len(newItem) == 0 or len(newItem) > 64:
                ErrorMessage(self.window,
                    _('Load list name must be between 1 and 64 characters long.'))
            else:
                self.data[newItem] = bosh.modInfos.ordered[:]
                settings.setChanged('bash.loadLists.data')
        #--Not Okay
        else:
            dialog.Destroy()

    def DoEdit(self,event):
        data = Mods_LoadListData(self.window)
        dialog = ListEditorDialog(self.window,-1,_('Load Lists'),data)
        dialog.ShowModal()
        dialog.Destroy()

#------------------------------------------------------------------------------
class Mods_EsmsFirst(Link):
    """Sort esms to the top."""
    def __init__(self,prefix=''):
        Link.__init__(self)
        self.prefix = prefix

    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,self.prefix+_('Type'),kind=wx.ITEM_CHECK)
        menu.AppendItem(menuItem)
        if window.esmsFirst: menuItem.Check()

    def Do(self,event):
        self.window.esmsFirst = not self.window.esmsFirst
        self.window.PopulateItems()

#------------------------------------------------------------------------------
class Mods_SelectedFirst(Link):
    """Sort loaded mods to the top."""
    def __init__(self,prefix=''):
        Link.__init__(self)
        self.prefix = prefix

    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,self.prefix+_('Selection'),kind=wx.ITEM_CHECK)
        menu.AppendItem(menuItem)
        if window.selectedFirst: menuItem.Check()

    def Do(self,event):
        self.window.selectedFirst = not self.window.selectedFirst
        self.window.PopulateItems()

#------------------------------------------------------------------------------
class Mods_Deprint(Link):
    """Turn on deprint/delist."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Debug Mode'),kind=wx.ITEM_CHECK)
        menu.AppendItem(menuItem)
        menuItem.Check(bosh.deprintOn)

    def Do(self,event):
        deprint(_('Debug Printing: Off'))
        bosh.deprintOn = not bosh.deprintOn
        deprint(_('Debug Printing: On'))

#------------------------------------------------------------------------------
class Mods_DumpTranslator(Link):
    """Dumps new translation key file using existing key, value pairs."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Dump Translator'))
        menu.AppendItem(menuItem)

    def Do(self,event):
        message = _("Generate Bash program translator file?\n\nThis function is for translating Bash itself (NOT mods) into non-English languages. For more info, see Internationalization section of Bash readme.")
        if ContinueQuery(self.window,message,'bash.dumpTranslator.continue',_('Dump Translator')) != wx.ID_OK:
            return
        import locale
        language = locale.getlocale()[0].split('_',1)[0]
        outPath = bosh.dirs['app'].join('Mopy','locale','NEW%s.txt' % (language,))
        outFile = open(outPath,'w')
        #--Scan for keys and dump to 
        keyCount = 0
        dumpedKeys = set()
        reKey = re.compile(r'_\([\'\"](.+?)[\'\"]\)')
        reTrans = bush.reTrans
        for pyFile in ('bush.py','bosh.py','bash.py','basher.py'):
            pyText = open(pyFile)
            for lineNum,line in enumerate(pyText):
                line = re.sub('#.*','',line)
                for key in reKey.findall(line):
                    key = reTrans.match(key).group(2)
                    if key in dumpedKeys: continue
                    outFile.write('=== %s, %d\n' % (pyFile,lineNum+1))
                    outFile.write(key+'\n>>>>\n')
                    value = _(key,False)
                    if value != key:
                        outFile.write(value)
                    outFile.write('\n')
                    dumpedKeys.add(key)
                    keyCount += 1
            pyText.close()
        outFile.close()
        Message(self.window,'%d translation keys written to Mopy\\locale\\%s.' % (keyCount,outPath.tail()))

#------------------------------------------------------------------------------
class Mods_IniTweaks(Link):
    """Applies ini tweaks to Oblivion.ini."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('INI Tweaks...'))
        menu.AppendItem(menuItem)

    def Do(self,event):
        """Handle menu selection."""
        #--Continue Query
        message = _("Apply an ini tweak to Oblivion.ini?\n\nWARNING: Incorrect tweaks can result in CTDs and even damage to you computer!")
        if ContinueQuery(self.window,message,'bash.iniTweaks.continue',_("INI Tweaks")) != wx.ID_OK:
            return
        #--File dialog
        tweakDir = bosh.modInfos.dir.join("INI Tweaks")
        tweakDir.makedirs()
        tweakPath = OpenDialog(self.window,_('INI Tweaks'),tweakDir,'', '*.ini')
        if not tweakPath: return
        bosh.oblivionIni.applyTweakFile(tweakPath)
        InfoMessage(self.window,tweakPath.tail()+_(' applied.'),_('INI Tweaks'))

#------------------------------------------------------------------------------
class Mods_ListMods(Link):
    """Copies list of mod files to clipboard."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_("List Mods..."))
        menu.AppendItem(menuItem)

    def Do(self,event):
        #--Get masters list
        text = bosh.modInfos.getModList()
        if (wx.TheClipboard.Open()):
            wx.TheClipboard.SetData(wx.TextDataObject(text))
            wx.TheClipboard.Close()
        LogMessage(self.window,text,_("Active Mod Files"),asDialog=False,fixedFont=False)

#------------------------------------------------------------------------------
class Mods_LockTimes(Link):
    """Turn on resetMTimes feature."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Lock Times'),kind=wx.ITEM_CHECK)
        menu.AppendItem(menuItem)
        if bosh.modInfos.resetMTimes: menuItem.Check()

    def Do(self,event):
        bosh.modInfos.resetMTimes = not bosh.modInfos.resetMTimes
        settings['bosh.modInfos.resetMTimes'] = bosh.modInfos.resetMTimes
        if bosh.modInfos.resetMTimes:
            bosh.modInfos.refreshMTimes()
        else:
            bosh.modInfos.mtimes.clear()

#------------------------------------------------------------------------------
class Mods_OblivionIni(Link):
    """Open Oblivion.ini."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Oblivion.ini...'))
        menu.AppendItem(menuItem)
        self.path = bosh.dirs['saveBase'].join('Oblivion.ini')
        menuItem.Enable(self.path.exists())

    def Do(self,event):
        """Handle selection."""
        os.startfile(self.path)

#------------------------------------------------------------------------------
class Mods_OblivionVersion(Link):
    """Specify/set Oblivion version."""
    def __init__(self,key,setProfile=False):
        Link.__init__(self)
        self.key = key
        self.setProfile = setProfile

    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,self.key,kind=wx.ITEM_CHECK)
        menu.AppendItem(menuItem)
        menuItem.Enable(bosh.modInfos.voCurrent != None and self.key in bosh.modInfos.voAvailable)
        if bosh.modInfos.voCurrent == self.key: menuItem.Check()

    def Do(self,event):
        """Handle selection."""
        bosh.modInfos.setOblivionVersion(self.key)
        bosh.modInfos.refresh()
        modList.RefreshUI()
        if self.setProfile:
            bosh.saveInfos.profiles.setItem(bosh.saveInfos.localSave,'vOblivion',self.key)
        bashFrame.SetTitle()

#------------------------------------------------------------------------------
class Mods_UpdateInvalidator(Link):
    """Mod Replacers dialog."""
    def AppendToMenu(self,menu,window,data):
        """Append ref replacer items to menu."""
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Update Archive Invalidator'))
        menu.AppendItem(menuItem)

    def Do(self,event):
        message = (_("Update ArchiveInvalidation.txt? This updates the file that forces the game engine to recognize replaced textures. Note that this feature is experimental and most probably somewhat incomplete. You may prefer to use another program to do AI.txt file updating."))
        if ContinueQuery(self.window,message,'bash.updateAI.continue',_('ArchiveInvalidation.txt')) != wx.ID_OK:
            return
        bosh.ResourceReplacer.updateInvalidator()
        Message(self.window,"ArchiveInvalidation.txt updated.")

# Mod Links -------------------------------------------------------------------
#------------------------------------------------------------------------------
class Mod_ActorLevels_Export(Link):
    """Export actor levels from mod to text file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('NPC Levels'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data)==1)

    def Do(self,event):
        message = (_("This command will export the level info for NPCs whose level is offset with respect to the PC. The exported file can be edited with most spreadsheet programs and then reimported.\n\nSee the Bash help file for more info."))
        if ContinueQuery(self.window,message,'bash.actorLevels.export.continue',_('Export NPC Levels')) != wx.ID_OK:
            return
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textName = fileName.splitext()[0]+_('_NPC_Levels.csv')
        textDir = settings.get('bash.workDir',bosh.dirs['app'])
        #--File dialog
        textPath = SaveDialog(self.window,_('Export NPC levels to:'),
            textDir,textName, '*.csv')
        if not destPath: return
        (textDir,textName) = textPath.split()
        settings['bash.workDir'] = textDir
        #--Export
        progress = ProgressDialog(_("Export NPC Levels"))
        try:
            bosh.ActorLevels.dumpText(fileInfo,textPath,progress)
        finally:
            progress = progress.Destroy()

#------------------------------------------------------------------------------
class Mod_ActorLevels_Import(Link):
    """Export actor levels from mod to text file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('NPC Levels...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data)==1)

    def Do(self,event):
        message = (_("This command will import NPC level info from a previously exported file.\n\nSee the Bash help file for more info.")) 
        if ContinueQuery(self.window,message,'bash.actorLevels.import.continue',_('Import NPC Levels')) != wx.ID_OK:
            return
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textName = fileName.splitext()[0]+_('_NPC_Levels.csv')
        textDir = settings.get('bash.workDir',bosh.dirs['app'])
        #--File dialog
        textPath = OpenDialog(self.window,_('Export NPC levels to:'),
            textDir, textName, '*.csv')
        if not textPath: return
        (textDir,textName) = textPath.split()
        settings['bash.workDir'] = textDir
        #--Export
        progress = ProgressDialog(_("Import NPC Levels"))
        try:
            bosh.ActorLevels.loadText(fileInfo,textPath, progress)
        finally:
            progress = progress.Destroy()

#------------------------------------------------------------------------------
class Mod_AddMaster(Link):
    """Adds master."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Add Master...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data)==1)

    def Do(self,event):
        message = _("WARNING! For advanced modders only! Adds specified master to list of masters, thus ceding ownership of new content of this mod to the new master. Useful for splitting mods into esm/esp pairs.")
        if ContinueQuery(self.window,message,'bash.addMaster.continue',_('Add Master...')) != wx.ID_OK:
            return
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        wildcard = _('Oblivion Masters')+' (*.esm;*.esp)|*.esm;*.esp'
        masterPath = OpenDialog(self.window,_('Add master:'),fileInfo.dir, '', wildcard)
        if not masterPath: return
        (dir,name) = masterPath.split()
        if dir != fileInfo.dir:
            ErrorMessage(self.window,
                _("File must be selected from Oblivion Data Files directory."))
            return
        if name in fileInfo.header.masters:
            ErrorMessage(self.window,_("%s is already a master!") % (name,))
            return
        fileInfo.header.masters.append(name)
        fileInfo.header.changed = True
        fileInfo.writeHeader()
        bosh.modInfos.refreshFile(fileInfo.name)
        self.window.RefreshUI()

#------------------------------------------------------------------------------
class Mod_CreateBlank(Link):
    """Create a duplicate of the file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('New Mod...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) == 1)

    def Do(self,event):
        data = self.data
        fileName = Path.get(data[0])
        fileInfos = self.window.data
        fileInfo = fileInfos[fileName]
        count = 0
        newName = Path('New Mod.esp')
        while newName in fileInfos:
            count += 1
            newName = Path('New Mod %d.esp' % (count,))
        newInfo = bosh.ModInfo(fileInfo.dir,newName)
        newInfo.mtime = fileInfo.mtime+20
        newFile = bosh.ModFile(newInfo,bosh.LoadFactory(True))
        newFile.masters = [Path('Oblivion.esm')]
        newFile.safeSave()
        bosh.modInfos.refresh()
        self.window.RefreshUI(detail=newName)

#------------------------------------------------------------------------------
class Mod_MarkLevelers(Link):
    """Marks (tags) selected mods as Delevs and/or Relevs according to Leveled Lists.csv."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Mark Levelers...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(bool(data))

    def Do(self,event):
        tagsByModName = bosh.ListsMerger.getDefaultTags()
        delevs = relevs = 0
        for fileName in map(Path,self.data):
            if fileName == 'Oblivion.esm': continue
            fileInfo = bosh.modInfos[fileName]
            fileInfo.shiftBashKeys() #--Move bash keys to top
            if fileName not in tagsByModName: continue
            tags = tagsByModName[fileName]
            bashKeys = fileInfo.getBashKeys()
            bashKeys -= set(('Relev','Delev'))
            if 'D' in tags: 
                bashKeys.add('Delev')
                delevs += 1
            if 'R' in tags: 
                bashKeys.add('Relev')
                relevs += 1
            fileInfo.setBashKeys(bashKeys)
        message = _('Mods tagged: %d Delevs, %d Relevs.') % (delevs,relevs)
        InfoMessage(self.window,message)

#------------------------------------------------------------------------------
class Mod_MarkMergeable(Link):
    """Returns true if can act as patch mod."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Mark Mergeable...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(bool(data))

    def Do(self,event):
        yes,no = [],[]
        for fileName in map(Path,self.data):
            if fileName == 'Oblivion.esm': continue
            fileInfo = bosh.modInfos[fileName]
            fileInfo.shiftBashKeys() #--Move bash keys to top
            canMerge = bosh.PatchFile.modIsMergeable(fileInfo)
            bashKeys = fileInfo.getBashKeys()
            if canMerge == True:
                yes.append(fileName)
                if 'Merge' not in bashKeys:
                    bashKeys.add('Merge')
                    fileInfo.setBashKeys(bashKeys)
            else:
                no.append("%s (%s)" % (fileName,canMerge))
                if 'Merge' in bashKeys:
                    bashKeys.remove('Merge')
                    fileInfo.setBashKeys(bashKeys)
        message = ''
        if yes:
            message += _('=== Mergeable\n* ') + '\n* '.join(yes)
        #else:
        #    message += _('No mergeable mods.')
        if yes and no:
            message += '\n\n'
        if no:
            message += _('=== Not Mergeable\n* ') + '\n* '.join(no)
        if yes:
            self.window.RefreshUI(yes)
        WtxtLogMessage(self.window,message,_('Mark Mergeable'))
        #InfoMessage(self.window,message)

#------------------------------------------------------------------------------
class Mod_CopyToEsmp(Link):
    """Create an esp(esm) copy of selected esm(esp)."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        fileInfo = self.fileInfo = bosh.modInfos[data[0]]
        self.label = (_('Copy to Esm'),_('Copy to Esp'))[fileInfo.isEsm()]
        menuItem = wx.MenuItem(menu,self.id,self.label)
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) == 1 and not fileInfo.isInvertedMod())

    def Do(self,event):
        fileInfo = self.fileInfo
        newType = (fileInfo.isEsm() and 'esp') or 'esm'
        modsDir = fileInfo.dir
        curName = fileInfo.name
        newName = Path.get(curName[:-3]+newType)
        #--Replace existing file?
        if modsDir.join(newName).exists():
            if not YesQuery(self.window,_('Replace existing %s?') % (newName,),self.label):
                return
            bosh.modInfos[newName].makeBackup()
        #--New Time
        modInfos = bosh.modInfos
        timeSource = (curName,newName)[newName in modInfos]
        newTime = modInfos[timeSource].mtime
        #--Copy, set type, update mtime.
        modInfos.copy(curName,modsDir,newName,newTime)
        modInfos.table.copyRow(curName,newName)
        newInfo = modInfos[newName]
        newInfo.setType(newType)
        newInfo.setMTime(newTime)
        #--Repopulate
        self.window.RefreshUI(detail=newName)

#------------------------------------------------------------------------------
class Mod_Face_Import(Link):
    """Imports a face from a save to an esp."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Face...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) == 1)

    def Do(self,event):
        #--Select source face file
        srcDir = bosh.saveInfos.dir
        wildcard = _('Oblivion Files')+' (*.ess;*.esr)|*.ess;*.esr'
        #--File dialog
        srcPath = OpenDialog(self.window,'Face Source:',srcDir, '', wildcard)
        if not srcPath: return
        #--Get face
        srcDir,srcName = srcPath.split()
        srcInfo = bosh.SaveInfo(srcDir,srcName)
        srcFace = bosh.PCFaces.save_getFace(srcInfo)
        #--Save Face
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        npc = bosh.PCFaces.mod_addFace(fileInfo,srcFace)
        #--Save Face picture?
        imagePath = bosh.modInfos.dir.join('Docs','Images',npc.eid+'.jpg')
        if not imagePath.exists():
            srcInfo.getHeader()
            width,height,data = srcInfo.header.image
            image = wx.EmptyImage(width,height)
            image.SetData(data)
            imagePath.head().makedirs()
            image.SaveFile(imagePath,wx.BITMAP_TYPE_JPEG)
        self.window.RefreshUI()
        Message(self.window,_('Imported face to: %s') % (npc.eid,))

#------------------------------------------------------------------------------
class Mod_FlipMasters(Link):
    """Swaps masters between esp and esm versions."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Esmify Masters'))
        menu.AppendItem(menuItem)
        #--FileInfo
        fileInfo = self.fileInfo = window.data[data[0]]
        menuItem.Enable(False)
        self.toEsp = False
        if len(data) == 1 and len(fileInfo.header.masters) > 1:
            espMasters = [master for master in fileInfo.header.masters if bosh.reEspExt.search(master)]
            if not espMasters: return
            for masterName in espMasters:
                masterInfo = bosh.modInfos.get(Path.get(masterName),None)
                if masterInfo and masterInfo.isInvertedMod():
                    menuItem.SetText(_('Espify Masters'))
                    self.toEsm = False
                    break
            else:
                self.toEsm = True
            menuItem.Enable(True)

    def Do(self,event):
        message = _("WARNING! For advanced modders only! Flips esp/esm bit of esp masters to convert them to/from esm state. Useful for building/analyzing esp mastered mods.")
        if ContinueQuery(self.window,message,'bash.flipMasters.continue') != wx.ID_OK:
            return
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        updated = [fileName]
        espMasters = [Path.get(master) for master in fileInfo.header.masters if bosh.reEspExt.search(master)]
        for masterPath in espMasters:
            masterInfo = bosh.modInfos.get(masterPath,None)
            if masterInfo:
                masterInfo.header.flags1.esm = self.toEsm
                masterInfo.writeHeader()
                updated.append(masterPath)
        self.window.RefreshUI(updated,fileName)

#------------------------------------------------------------------------------
class Mod_FlipSelf(Link):
    """Flip an esp(esm) to an esm(esp)."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Esmify Self'))
        menu.AppendItem(menuItem)
        menuItem.Enable(bool(len(data) == 1 and re.search('[Pp]$',data[0])))
        #--Filetype
        fileInfo = self.fileInfo = window.data[data[0]]
        if fileInfo.isEsm():
            menuItem.SetText(_('Espify Self'))

    def Do(self,event):
        message = _('WARNING! For advanced modders only!\n\nThis command flips an internal bit in the mod, converting an esp to an esm and vice versa. Note that it is this bit and NOT the file extension that determines the esp/esm state of the mod.')
        if ContinueQuery(self.window,message,'bash.flipToEsmp.continue',_('Flip to Esm')) != wx.ID_OK:
            return
        fileInfo = self.fileInfo
        header = fileInfo.header
        header.flags1.esm = not header.flags1.esm
        fileInfo.writeHeader()
        #--Repopulate
        self.window.RefreshUI(detail=fileInfo.name)


#------------------------------------------------------------------------------
class Mod_LabelsData(ListEditorData):
    """Data capsule for label editing dialog."""
    def __init__(self,parent,strings):
        """Initialize."""
        #--Strings
        self.column = strings.column
        self.setKey = strings.setKey
        self.addPrompt = strings.addPrompt
        #--Key/type
        self.data = settings[self.setKey]
        #--GUI
        ListEditorData.__init__(self,parent)
        self.showAdd = True
        self.showRename = True
        self.showRemove = True

    def getItemList(self):
        """Returns load list keys in alpha order."""
        return sorted(self.data,key=lambda a: a.lower())

    def add(self):
        """Adds a new group."""
        #--Name Dialog
        #--Dialog
        dialog = wx.TextEntryDialog(self.parent,self.addPrompt)
        result = dialog.ShowModal()
        #--Okay?
        if result != wx.ID_OK:
            dialog.Destroy()
            return
        newName = dialog.GetValue()
        dialog.Destroy()
        if newName in self.data:
            ErrorMessage(self.parent,_('Name must be unique.'))
            return False
        elif len(newName) == 0 or len(newName) > 64:
            ErrorMessage(self.parent,
                _('Name must be between 1 and 64 characters long.'))
            return False
        settings.setChanged(self.setKey)
        self.data.append(newName)
        self.data.sort()
        return newName

    def rename(self,oldName,newName):
        """Renames oldName to newName."""
        #--Right length?
        if len(newName) == 0 or len(newName) > 64:
            ErrorMessage(self.parent,
                _('Name must be between 1 and 64 characters long.'))
            return False
        #--Rename
        settings.setChanged(self.setKey)
        self.data.remove(oldName)
        self.data.append(newName)
        self.data.sort()
        #--Edit table entries.
        colGroup = bosh.modInfos.table.getColumn(self.column)
        for fileName in colGroup.keys():
            if colGroup[fileName] == oldName:
                colGroup[fileName] = newName
        self.parent.PopulateItems()
        #--Done
        return newName

    def remove(self,item):
        """Removes group."""
        settings.setChanged(self.setKey)
        self.data.remove(item)
        #--Edit table entries.
        colGroup = bosh.modInfos.table.getColumn(self.column)
        for fileName in colGroup.keys():
            if colGroup[fileName] == item:
                del colGroup[fileName]
        self.parent.PopulateItems()
        #--Done
        return True

#------------------------------------------------------------------------------
class Mod_Labels:
    """Add mod label links."""
    def __init__(self):
        """Initialize."""
        self.labels = settings[self.setKey]

    def GetItems(self):
        items = self.labels[:]
        items.sort(key=lambda a: a.lower())
        return items

    def AppendToMenu(self,menu,window,data):
        """Append label list to menu."""
        self.window = window
        self.data = data
        menu.Append(self.idList.EDIT,self.editMenu) 
        menu.AppendSeparator()
        menu.Append(self.idList.NONE,_('None'))
        for id,item in zip(self.idList,self.GetItems()):
            menu.Append(id,item)
        #--Events
        wx.EVT_MENU(window,self.idList.EDIT,self.DoEdit)
        wx.EVT_MENU(window,self.idList.NONE,self.DoNone)
        wx.EVT_MENU_RANGE(window,self.idList.BASE,self.idList.MAX,self.DoList)

    def DoNone(self,event):
        """Handle selection of None."""
        fileLabels = bosh.modInfos.table.getColumn(self.column)
        for fileName in self.data:
            del fileLabels[fileName]
        self.window.PopulateItems()

    def DoList(self,event):
        """Handle selection of label."""
        label = self.GetItems()[event.GetId()-self.idList.BASE]
        fileLabels = bosh.modInfos.table.getColumn(self.column)
        for fileName in self.data:
            fileLabels[fileName] = label
        if isinstance(self,Mod_Groups) and bosh.modInfos.autoSort():
            modList.SortItems()
        self.window.RefreshUI()

    def DoEdit(self,event):
        """Show label editing dialog."""
        data = Mod_LabelsData(self.window,self)
        dialog = ListEditorDialog(self.window,-1,self.editWindow,data)
        dialog.ShowModal()
        dialog.Destroy()

#------------------------------------------------------------------------------
class Mod_Groups(Mod_Labels):
    """Add mod group links."""
    def __init__(self):
        """Initialize."""
        self.column     = 'group'
        self.setKey     = 'bash.mods.groups'
        self.editMenu   = _('Edit Groups...')
        self.editWindow = _('Groups')
        self.addPrompt  = _('Add group:')
        self.idList     = ID_GROUPS
        Mod_Labels.__init__(self)

#------------------------------------------------------------------------------
class Mod_EditorIds_Export(Link):
    """Export editor ids from mod to text file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Editor Ids...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(bool(self.data))

    def Do(self,event):
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textName = fileName.root()+_('_Eids.csv')
        textDir = bosh.dirs['patches']
        textDir.makedirs()
        #--File dialog
        textPath = SaveDialog(self.window,_('Export eids to:'),textDir,textName, '*Eids.csv')
        if not textPath: return
        (textDir,textName) = textPath.split()
        #--Export
        progress = ProgressDialog(_("Export Editor Ids"))
        try:
            editorIds = bosh.EditorIds()
            readProgress = SubProgress(progress,0.1,0.8)
            readProgress.setFull(len(self.data))
            for index,fileName in enumerate(map(Path,self.data)):
                fileInfo = bosh.modInfos[fileName]
                readProgress(index,_("Reading %s.") % (fileName,))
                editorIds.readFromMod(fileInfo)
            progress(0.8,_("Exporting to %s.") % (textName,))
            editorIds.writeToText(textPath)
            progress(1.0,_("Done."))
        finally:
            progress = progress.Destroy()

#------------------------------------------------------------------------------
class Mod_EditorIds_Import(Link):
    """Import editor ids from text file or other mod."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Editor Ids...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data)==1)

    def Do(self,event):
        message = (_("Import editor ids from a text file. This will replace existing ids and is not reversible!"))
        if ContinueQuery(self.window,message,'bash.editorIds.import.continue',
            _('Import Editor Ids')) != wx.ID_OK:
            return
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textName = fileName.root()+_('_Eids.csv')
        textDir = bosh.dirs['patches']
        #--File dialog
        textPath = OpenDialog(self.window,_('Import names from:'),textDir,
            textName, '*Eids.csv')
        if not textPath: return
        (textDir,textName) = textPath.split()
        #--Extension error check
        if textName.ext().lower() != '.csv':
            ErrorMessage(self.window,_('Source file must be a csv file.'))
            return
        #--Export
        progress = ProgressDialog(_("Import Editor Ids"))
        changed = None
        try:
            editorIds = bosh.EditorIds()
            progress(0.1,_("Reading %s.") % (textName,))
            editorIds.readFromText(textPath)
            progress(0.2,_("Applying to %s.") % (fileName,))
            changed = editorIds.writeToMod(fileInfo)
            progress(1.0,_("Done."))
        finally:
            progress = progress.Destroy()
        #--Log
        if not changed:
            Message(self.window,_("No changes required."))
        else:
            buff = cStringIO.StringIO()
            format = '%s >> %s\n'
            for old_new in sorted(changed):
                buff.write(format % old_new)
            LogMessage(self.window,buff.getvalue(),_('Objects Changed'))

#------------------------------------------------------------------------------
class Mod_Formids_Replace(Link):
    """Replace formids according to text file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Formids...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data)==1)

    def Do(self,event):
        message = _("For advanced modders only! Systematically replaces one set of formids with another in npcs, creatures, containers and leveled lists according to a Replacers.csv file.")
        if ContinueQuery(self.window,message,'bash.formids.replace.continue',
            _('Import Formids')) != wx.ID_OK:
            return
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textDir = bosh.dirs['patches']
        #--File dialog
        textPath = OpenDialog(self.window,_('Formid mapper file:'),textDir,
            '', '*Formids.csv')
        if not textPath: return
        (textDir,textName) = textPath.split()
        #--Extension error check
        if textName.ext().lower() != '.csv':
            ErrorMessage(self.window,_('Source file must be a csv file.'))
            return
        #--Export
        progress = ProgressDialog(_("Import Formids"))
        changed = None
        try:
            replacer = bosh.FormidReplacer()
            progress(0.1,_("Reading %s.") % (textName,))
            replacer.readFromText(textPath)
            progress(0.2,_("Applying to %s.") % (fileName,))
            changed = replacer.updateMod(fileInfo)
            progress(1.0,_("Done."))
        finally:
            progress = progress.Destroy()
        #--Log
        if not changed:
            Message(self.window,_("No changes required."))
        else:
            LogMessage(self.window,changed,_('Objects Changed'))

#------------------------------------------------------------------------------
class Mod_FullNames_Export(Link):
    """Export full names from mod to text file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Names...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(bool(self.data))

    def Do(self,event):
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textName = fileName.root()+_('_Names.csv')
        textDir = bosh.dirs['patches']
        textDir.makedirs()
        #--File dialog
        textPath = SaveDialog(self.window,_('Export names to:'),
            textDir,textName, '*Names.csv')
        if not textPath: return
        (textDir,textName) = textPath.split()
        #--Export
        progress = ProgressDialog(_("Export Names"))
        try:
            fullNames = bosh.FullNames()
            readProgress = SubProgress(progress,0.1,0.8)
            readProgress.setFull(len(self.data))
            for index,fileName in enumerate(map(Path,self.data)):
                fileInfo = bosh.modInfos[fileName]
                readProgress(index,_("Reading %s.") % (fileName,))
                fullNames.readFromMod(fileInfo)
            progress(0.8,_("Exporting to %s.") % (textName,))
            fullNames.writeToText(textPath)
            progress(1.0,_("Done."))
        finally:
            progress = progress.Destroy()

#------------------------------------------------------------------------------
class Mod_FullNames_Import(Link):
    """Import full names from text file or other mod."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Names...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data)==1)

    def Do(self,event):
        message = (_("Import record names from a text file. This will replace existing names and is not reversible!"))
        if ContinueQuery(self.window,message,'bash.fullNames.import.continue',
            _('Import Names')) != wx.ID_OK:
            return
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textName = fileName.root()+_('_Names.csv')
        textDir = bosh.dirs['patches']
        #--File dialog
        textPath = OpenDialog(self.window,_('Import names from:'),
            textDir,textName, 'Mod/Text File|*Names.csv;*.esp;*.esm')
        if not textPath: return
        (textDir,textName) = textPath.split()
        #--Extension error check
        ext = textName.ext().lower()
        if ext not in ('.esp','.esm','.csv'):
            ErrorMessage(self.window,_('Source file must be mod (.esp or .esm) or csv file.'))
            return
        #--Export
        progress = ProgressDialog(_("Import Names"))
        renamed = None
        try:
            fullNames = bosh.FullNames()
            progress(0.1,_("Reading %s.") % (textName,))
            if ext == '.csv':
                fullNames.readFromText(textPath)
            else:
                srcInfo = bosh.ModInfo(textDir,textName)
                fullNames.readFromMod(srcInfo)
            progress(0.2,_("Applying to %s.") % (fileName,))
            renamed = fullNames.writeToMod(fileInfo)
            progress(1.0,_("Done."))
        finally:
            progress = progress.Destroy()
        #--Log
        if not renamed:
            Message(self.window,_("No changes required."))
        else:
            buff = cStringIO.StringIO()
            format = '%s:   %s >> %s\n'
            #buff.write(format % (_('Editor Id'),_('Name')))
            for eid in sorted(renamed.keys()):
                full,newFull = renamed[eid]
                buff.write(format % (eid,full,newFull))
            LogMessage(self.window,buff.getvalue(),_('Objects Renamed'))

#------------------------------------------------------------------------------
class Mod_Patch_Update(Link):
    """Updates a Bashed Patch."""
    def AppendToMenu(self,menu,window,data):
        """Append link to a menu."""
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Rebuild Patch...'))
        menu.AppendItem(menuItem)
        enable = (len(self.data) == 1 and 
            bosh.modInfos[self.data[0]].header.author in ('BASHED PATCH','BASHED LISTS'))
        menuItem.Enable(enable)

    def Do(self,event):
        """Handle activation event."""
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        patchDialog = PatchDialog(self.window,fileInfo)
        patchDialog.ShowModal()
        self.window.RefreshUI(detail=fileName)

#------------------------------------------------------------------------------
class Mod_Ratings(Mod_Labels):
    """Add mod rating links."""
    def __init__(self):
        """Initialize."""
        self.column     = 'rating'
        self.setKey     = 'bash.mods.ratings'
        self.editMenu   = _('Edit Ratings...')
        self.editWindow = _('Ratings')
        self.addPrompt  = _('Add rating:')
        self.idList     = ID_RATINGS
        Mod_Labels.__init__(self)

#------------------------------------------------------------------------------
class Mod_SetVersion(Link):
    """Sets version of file back to 0.8."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        self.fileInfo = window.data[data[0]]
        menuItem = wx.MenuItem(menu,self.id,_('Version 0.8'))
        menu.AppendItem(menuItem)
        #print self.fileInfo.header.version
        menuItem.Enable((len(data) == 1) and (int(10*self.fileInfo.header.version) != 8))

    def Do(self,event):
        message = _("WARNING! For advanced modders only! This feature allows you to edit newer official mods in the TES Construction Set by resetting the internal file version number back to 0.8. While this will make the mod editable, it may also break the mod in some way.")
        if ContinueQuery(self.window,message,'bash.setModVersion.continue',_('Set File Version')) != wx.ID_OK:
            return
        self.fileInfo.header.version = 0.8
        self.fileInfo.header.setChanged()
        self.fileInfo.writeHeader()
        #--Repopulate
        self.window.RefreshUI(detail=self.fileInfo.name)

#------------------------------------------------------------------------------
class Mod_Details(Link):
    """Show Mod Details"""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        self.fileInfo = window.data[data[0]]
        menuItem = wx.MenuItem(menu,self.id,_('Details...'))
        menu.AppendItem(menuItem)
        menuItem.Enable((len(data) == 1))

    def Do(self,event):
        modName = Path.get(self.data[0])
        modInfo = bosh.modInfos[modName]
        progress = ProgressDialog(_(modName))
        try:
            modDetails = bosh.ModDetails()
            modDetails.readFromMod(modInfo,SubProgress(progress,0.1,0.7))
            buff = cStringIO.StringIO()
            progress(0.7,_("Sorting records."))
            for group in sorted(modDetails.group_records):
                buff.write(group+'\n')
                if group in ('CELL','WRLD','DIAL'):
                    buff.write(_('  (Details not provided for this record type.)\n\n'))
                    continue
                records = modDetails.group_records[group]
                records.sort(key = lambda a: a[1].lower())
                #if group != 'GMST': records.sort(key = lambda a: a[0] >> 24)
                for formid,eid in records:
                    buff.write('  %08X %s\n' % (formid,eid))
                buff.write('\n')
            LogMessage(self.window,buff.getvalue(),modInfo.name,asDialog=False,fixedFont=True)
            progress.Destroy()
            buff.close()
        finally:
            if progress: progress.Destroy()

#------------------------------------------------------------------------------
class Mod_ShowReadme(Link):
    """Open the readme."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Readme...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) == 1)

    def Do(self,event):
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        if not docBrowser: 
            DocBrowser().Show()
            settings['bash.modDocs.show'] = True
        docBrowser.SetMod(fileInfo.name)
        docBrowser.Raise()

#------------------------------------------------------------------------------
class Mod_Stats_Export(Link):
    """Export armor and weapon stats from mod to text file."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Stats...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(bool(self.data))

    def Do(self,event):
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textName = fileName.root()+_('_Stats.csv')
        textDir = bosh.dirs['patches']
        textDir.makedirs()
        #--File dialog
        textPath = SaveDialog(self.window,_('Export stats to:'),
            textDir, textName, '*Stats.csv')
        if not textPath: return
        (textDir,textName) = textPath.split()
        #--Export
        progress = ProgressDialog(_("Export Stats"))
        try:
            itemStats = bosh.ItemStats()
            readProgress = SubProgress(progress,0.1,0.8)
            readProgress.setFull(len(self.data))
            for index,fileName in enumerate(map(Path,self.data)):
                fileInfo = bosh.modInfos[fileName]
                readProgress(index,_("Reading %s.") % (fileName,))
                itemStats.readFromMod(fileInfo)
            progress(0.8,_("Exporting to %s.") % (textName,))
            itemStats.writeToText(textPath)
            progress(1.0,_("Done."))
        finally:
            progress = progress.Destroy()

#------------------------------------------------------------------------------
class Mod_Stats_Import(Link):
    """Import stats from text file or other mod."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Stats...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(self.data)==1)

    def Do(self,event):
        message = (_("Import item stats from a text file. This will replace existing stats and is not reversible!"))
        if ContinueQuery(self.window,message,'bash.stats.import.continue',
            _('Import Stats')) != wx.ID_OK:
            return
        fileName = Path.get(self.data[0])
        fileInfo = bosh.modInfos[fileName]
        textName = fileName.root()+_('_Stats.csv')
        textDir = bosh.dirs['patches']
        #--File dialog
        textPath = OpenDialog(self.window,_('Import stats from:'),
            textDir, textName, '*Stats.csv')
        if not textPath: return
        (textDir,textName) = textPath.split()
        #--Extension error check
        ext = textName.ext().lower()
        if ext != '.csv':
            ErrorMessage(self.window,_('Source file must be a Stats.csv file.'))
            return
        #--Export
        progress = ProgressDialog(_("Import Stats"))
        changed = None
        try:
            itemStats = bosh.ItemStats()
            progress(0.1,_("Reading %s.") % (textName,))
            if ext == '.csv':
                itemStats.readFromText(textPath)
            else:
                srcInfo = bosh.ModInfo(textDir,textName)
                itemStats.readFromMod(srcInfo)
            progress(0.2,_("Applying to %s.") % (fileName,))
            changed = itemStats.writeToMod(fileInfo)
            progress(1.0,_("Done."))
        finally:
            progress = progress.Destroy()
        #--Log
        if not changed:
            Message(self.window,_("No relevant stats to import."))
        else:
            buff = cStringIO.StringIO()
            for modName in sorted(changed):
                buff.write('* %03d  %s:\n' % (changed[modName], modName))
            LogMessage(self.window,buff.getvalue(),_('Import Stats'))

#------------------------------------------------------------------------------
class Mod_TNRFaces_Import(Link):
    """Imports TNR faces."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('TNR Faces'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) == 1 and data[0] != 'Oblivion.esm')

    def Do(self,event):
        #--Continue?
        message = _("Import TNR Faces?\n\nThis command will resolve conflicts between this mod and TNR (Tamriel NPCs Revamped) by importing faces from TNR into this mod. Afterwards, you will still need the TNR mod, but it should load before (earlier) than this mod.")
        if ContinueQuery(self.window,message,'bash.tnrFaces.continue',_('Import TNR Faces')) != wx.ID_OK:
            return
        #--Get TNR mods
        reTnr = re.compile('^TNR ')
        faceInfos = bosh.byName([x for x in bosh.modInfos.data.values() if reTnr.match(x.name)])
        if not faceInfos:
            Message(self.window,_("No TNR mods found. TNR mods should be in the Data directory and have a mod name that starts with 'TNR '."))
            return
        progress = ProgressDialog(_("Import TNR Faces"))
        try:
            loadFactory = bosh.LoadFactory(True,bosh.MreNpc)
            #--Face source
            faces = {}
            for index,faceInfo in enumerate(faceInfos):
                progress(0.1+0.4*index/len(faceInfos),_('Reading: ')+faceInfo.name)
                faceFile = bosh.ModFile(faceInfo,loadFactory)
                faceFile.load(True)
                for record in faceFile.NPC_.records:
                    faces[record.eid] = (record.fggs,record.fgga,record.fgts)
            #--Mod info
            modName = Path.get(self.data[0])
            modInfo = self.window.data[modName]
            progress(0.5,_('Reading: ')+modName)
            modFile = bosh.ModFile(modInfo,loadFactory)
            modFile.load(True)
            refaced = 0
            progress(0.7,_('Updating Faces'))
            for record in modFile.NPC_.records:
                if record.eid in faces:
                    record.fggs,record.fgga,record.fgts = faces[record.eid]
                    record.setChanged()
                    refaced += 1
            if refaced:
                progress(0.8,_('Saving: ')+modName)
                modFile.safeSave()
                message = _('Faces updated: %d/%d') % (refaced,len(modFile.NPC_.records))
            else:
                message = _('No faces updated.')
            progress.Destroy()
            Message(self.window,message)
        finally:
            progress.Destroy()

# Saves Links ------------------------------------------------------------------
#------------------------------------------------------------------------------
class Saves_ProfilesData(ListEditorData):
    """Data capsule for save profiles editing dialog."""
    def __init__(self,parent):
        """Initialize."""
        self.baseSaves = bosh.dirs['saveBase'].join('Saves')
        #--GUI
        ListEditorData.__init__(self,parent)
        self.showAdd    = True
        self.showRename = True
        self.showRemove = True

    def getItemList(self):
        """Returns load list keys in alpha order."""
        #--Get list of directories in Hidden, but do not include default.
        items = bosh.saveInfos.getLocalSaveDirs()
        items.sort(key=lambda a: a.lower())
        return items

    def add(self):
        """Adds a new profile."""
        newName = TextEntry(self.parent,_("Enter profile name:"))
        if not newName: 
            return False
        if newName in self.getItemList():
            ErrorMessage(self.parent,_('Name must be unique.'))
            return False
        if len(newName) == 0 or len(newName) > 64:
            ErrorMessage(self.parent,
                _('Name must be between 1 and 64 characters long.'))
            return False
        self.baseSaves.join(newName).mkdir()
        newSaves = 'Saves\\'+newName+'\\'
        bosh.saveInfos.profiles.setItem(newSaves,'vOblivion',bosh.modInfos.voCurrent)
        return newName

    def rename(self,oldName,newName):
        """Renames profile oldName to newName."""
        newName = newName.strip()
        lowerNames = [name.lower() for name in self.getItemList()]
        #--Error checks
        if newName.lower() in lowerNames:
            ErrorMessage(self,_('Name must be unique.'))
            return False
        if len(newName) == 0 or len(newName) > 64:
            ErrorMessage(self.parent,
                _('Name must be between 1 and 64 characters long.'))
            return False
        #--Rename
        oldDir,newDir = (self.baseSaves.join(dir) for dir in (oldName,newName))
        oldDir.rename(newDir)
        oldSaves,newSaves = (('Saves\\'+name+'\\') for name in (oldName,newName))
        if bosh.saveInfos.localSave == oldSaves:
            bosh.saveInfos.setLocalSave(newSaves)
            bashFrame.SetTitle()
        bosh.saveInfos.profiles.moveRow(oldSaves,newSaves)
        return newName

    def remove(self,profile):
        """Removes load list."""
        profileSaves = 'Saves\\'+profile+'\\'
        #--Can't remove active or Default directory.
        if bosh.saveInfos.localSave == profileSaves:
            ErrorMessage(self.parent,_('Active profile cannot be removed.'))
            return False
        #--Get file count. If > zero, verify with user.
        profileDir = bosh.dirs['saveBase'].join(profileSaves)
        files = [file for file in profileDir.list() if bosh.reSaveExt.search(file)]
        if files:
            message = _('Delete profile %s and the %d save files it contains?') % (profile,len(files))
            if not YesQuery(self.parent,message,_('Delete Profile')):
                return False
        #--Remove directory
        if 'Oblivion' not in profileDir: 
            raise bosh.BoshError(_('Sanity check failed: No "Oblivion" in %s.') % (profileDir,))
        profileDir.rmtree() #--DO NOT SCREW THIS UP!!!
        bosh.saveInfos.profiles.delRow(profileSaves)
        return True

#------------------------------------------------------------------------------
class Saves_Profiles:
    """Select a save set profile -- i.e., the saves directory."""
    def __init__(self):
        """Initialize."""
        self.idList = ID_PROFILES

    def GetItems(self):
        items = bosh.saveInfos.getLocalSaveDirs()
        items.sort(key=lambda a: a.lower())
        return items

    def AppendToMenu(self,menu,window,data):
        """Append label list to menu."""
        self.window = window
        #--Edit
        menu.Append(self.idList.EDIT,_("Edit Profiles...")) 
        menu.AppendSeparator()
        #--List
        localSave = bosh.saveInfos.localSave
        menuItem = wx.MenuItem(menu,self.idList.DEFAULT,_('Default'),kind=wx.ITEM_CHECK)
        menu.AppendItem(menuItem)
        menuItem.Check(localSave == 'Saves\\')
        for id,item in zip(self.idList,self.GetItems()):
            menuItem = wx.MenuItem(menu,id,item,kind=wx.ITEM_CHECK)
            menu.AppendItem(menuItem)
            menuItem.Check(localSave == ('Saves\\'+item+'\\'))
        #--Events
        wx.EVT_MENU(window,self.idList.EDIT,self.DoEdit)
        wx.EVT_MENU(window,self.idList.DEFAULT,self.DoDefault)
        wx.EVT_MENU_RANGE(window,self.idList.BASE,self.idList.MAX,self.DoList)

    def DoEdit(self,event):
        """Show profiles editing dialog."""
        data = Saves_ProfilesData(self.window)
        dialog = ListEditorDialog(self.window,-1,_('Save Profiles'),data)
        dialog.ShowModal()
        dialog.Destroy()

    def DoDefault(self,event):
        """Handle selection of Default."""
        arcSaves,newSaves = bosh.saveInfos.localSave,'Saves\\'
        bosh.saveInfos.setLocalSave(newSaves)
        self.swapPlugins(arcSaves,newSaves)
        self.swapOblivionVersion(newSaves)
        bashFrame.SetTitle()
        self.window.details.SetFile(None)
        modList.RefreshUI()
        bashFrame.RefreshData()

    def DoList(self,event):
        """Handle selection of label."""
        profile = self.GetItems()[event.GetId()-self.idList.BASE]
        arcSaves = bosh.saveInfos.localSave
        newSaves = 'Saves\\%s\\' % (profile,)
        bosh.saveInfos.setLocalSave(newSaves)
        self.swapPlugins(arcSaves,newSaves)
        self.swapOblivionVersion(newSaves)
        bashFrame.SetTitle()
        self.window.details.SetFile(None)
        modList.RefreshUI()
        bashFrame.RefreshData()

    def swapPlugins(self,arcSaves,newSaves):
        """Saves current plugins into arcSaves directory and loads plugins 
        from newSaves directory (if present)."""
        arcPath,newPath = (bosh.dirs['saveBase'].join(saves,'plugins.txt') 
            for saves in (arcSaves,newSaves))
        #--Archive old Saves
        bosh.modInfos.plugins.path.copyfile(arcPath)
        if newPath.exists():
            newPath.copyfile(bosh.modInfos.plugins.path)

    def swapOblivionVersion(self,newSaves):
        """Swaps Oblivion version to memorized version."""
        voNew = bosh.saveInfos.profiles.setItemDefault(newSaves,'vOblivion',bosh.modInfos.voCurrent)
        if voNew in bosh.modInfos.voAvailable: 
            bosh.modInfos.setOblivionVersion(voNew)

#------------------------------------------------------------------------------
class Save_LoadMasters(Link):
    """Sets the load list to the save game's masters."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Load Masters'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        errorMessage = bosh.modInfos.selectExact(fileInfo.masterNames)
        modList.PopulateItems()
        saveList.PopulateItems()
        self.window.details.SetFile(fileName)
        if errorMessage:
            ErrorMessage(self.window,errorMessage,fileName)

#------------------------------------------------------------------------------
class Save_ImportFace(Link):
    """Imports a face from another save."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Import Face...'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        #--File Info
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        #--Select source face file
        srcDir = fileInfo.dir
        wildcard = _('Oblivion Files')+' (*.esp;*.esm;*.ess;*.esr)|*.esp;*.esm;*.ess;*.esr'
        #--File dialog
        srcPath = OpenDialog(self.window,'Face Source:',srcDir, '', wildcard)
        if not srcPath: return
        if bosh.reSaveExt.search(srcPath):
            self.FromSave(fileInfo,srcPath)
        elif bosh.reModExt.search(srcPath):
            self.FromMod(fileInfo,srcPath)

    def FromSave(self,fileInfo,srcPath):
        """Import from a save."""
        #--Get face
        srcDir,srcName = Path.get(srcPath).split()
        srcInfo = bosh.SaveInfo(srcDir,srcName)
        progress = ProgressDialog(srcName)
        try:
            saveFile = bosh.SaveFile(srcInfo)
            saveFile.load(progress)
            progress.Destroy()
            srcFaces = bosh.PCFaces.save_getFaces(saveFile)
            #--Dialog
            dialog = ImportFaceDialog(self.window,-1,srcName,fileInfo,srcFaces)
            dialog.ShowModal()
            dialog.Destroy()
        finally:
            if progress: progress.Destroy()

    def FromMod(self,fileInfo,srcPath):
        """Import from a mod."""
        #--Get faces
        srcDir,srcName = Path.get(srcPath).split()
        srcInfo = bosh.ModInfo(srcDir,srcName)
        srcFaces = bosh.PCFaces.mod_getFaces(srcInfo)
        #--No faces to import?
        if not srcFaces:
            Message(self.window,_('No player (PC) faces found in %s.') % (srcName,))
            return
        #--Dialog
        dialog = ImportFaceDialog(self.window,-1,srcName,fileInfo,srcFaces)
        dialog.ShowModal()
        dialog.Destroy()

#------------------------------------------------------------------------------
class Save_ImportNPCLevels(Link):
    """Imports NPC levels from a mod."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Import NPC Levels...'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        WarningMessage(self.window,_('Import NPC Levels has been replaced by "Update NPC Levels". Please use it instead.'))
        return

#--------------------------------------------------------------------------
class Save_EditCreatedData(ListEditorData):
    """Data capsule for custom item editing dialog."""
    def __init__(self,parent,saveFile,records):
        """Initialize."""
        self.changed = False
        self.saveFile = saveFile
        data = self.data = {}
        #--Get records
        for record in sorted(records):
            name = record.getSubString('FULL')
            if name == None: 
                raise bosh.SaveFileError(_('Created record with no name.'))
            if name not in data:
                data[name] = (name,[])
            data[name][1].append(record)
        #--Get enchantments
        self.enchantments = dict([(record.formid,record) 
            for record in saveFile.created if record.type == 'ENCH'])
        #--GUI
        ListEditorData.__init__(self,parent)
        self.showRename = True
        self.showInfo = True
        self.showSave = True
        self.showCancel = True

    def getItemList(self):
        """Returns load list keys in alpha order."""
        items = sorted(self.data.keys())
        items.sort(key=lambda a: self.data[a][1][0].type)
        return items

    def getInfo(self,item):
        """Returns string info on specified item."""
        buff = cStringIO.StringIO()
        name,records = self.data[item]
        record = records[0].getTypeCopy()
        #--Armor, clothing, weapons
        if record.type == 'ARMO':
            buff.write(_('Armor\nFlags: '))
            buff.write(', '.join(record.flags.getTrueAttrs())+'\n')
            for attr in ('strength','value','weight'):
                buff.write('%s: %s\n' % (attr,getattr(record,attr)))
        elif record.type == 'CLOT':
            buff.write(_('Clothing\nFlags: '))
            buff.write(', '.join(record.flags.getTrueAttrs())+'\n')
        elif record.type == 'WEAP':
            buff.write(bush.weaponTypes[record.weaponType]+'\n')
            for attr in ('damage','value','speed','reach','weight'):
                buff.write('%s: %s\n' % (attr,getattr(record,attr)))
        #--Enchanted? Switch record to enchantment.
        if hasattr(record,'enchantment') and record.enchantment in self.enchantments:
            buff.write('\nEnchantment:\n')
            record = self.enchantments[record.enchantment].getTypeCopy()
        #--Magic effects
        if record.type in ('ALCH','SPEL','ENCH'):
            buff.write(record.getEffectsSummary())
        #--Done
        return buff.getvalue()

    def rename(self,oldName,newName):
        """Renames oldName to newName."""
        #--Right length?
        if len(newName) == 0:
            return False
        elif len(newName) > 128:
            ErrorMessage(self.parent,_('Name is too long.'))
            return False
        elif newName in self.data:
            ErrorMessage(self.parent,_("Name is already used."))
            return False
        #--Rename
        self.data[newName] = self.data.pop(oldName)
        self.changed = True
        return newName

    def save(self):
        """Handles save button."""
        if not self.changed: 
            Message(self.parent,_("No changes made."))
        else:
            self.changed = False #--Allows graceful effort if close fails.
            count = 0
            for newName,(oldName,records) in self.data.items():
                if newName == oldName: continue
                for record in records:
                    record.setSubString('FULL',newName)
                    record.getSize()
                count += 1
            self.saveFile.safeSave()
            Message(self.parent, self.saveFile.fileInfo.name+_(": %d names modified.") % (count,))

#------------------------------------------------------------------------------
class Save_EditCreated(Link):
    """Allows user to rename custom items (spells, enchantments, etc."""
    menuNames = {'ENCH':_('Rename Enchanted...'),'SPEL':_('Rename Spells...'),'ALCH':_('Rename Potions...')}
    recordTypes = {'ENCH':('ARMO','CLOT','WEAP')}

    def __init__(self,type):
        if type not in Save_EditCreated.menuNames: 
            raise bosh.ArgumentError
        Link.__init__(self)
        self.type = type
        self.menuName = Save_EditCreated.menuNames[self.type]

    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id, self.menuName)
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        """Handle menu selection."""
        #--Get save info for file
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        #--Get SaveFile
        progress = ProgressDialog(_("Loading..."))
        try:
            saveFile = bosh.SaveFile(fileInfo)
            saveFile.load(progress)
        finally:
            if progress: progress.Destroy()
        #--No custom items?
        recordTypes = Save_EditCreated.recordTypes.get(self.type,(self.type,))
        records = [record for record in saveFile.created if record.type in recordTypes]
        if not records:
            Message(self.window,_('No items to edit.'))
            return
        #--Open editor dialog
        data = Save_EditCreatedData(self.window,saveFile,records)
        dialog = ListEditorDialog(self.window,-1,self.menuName,data)
        dialog.ShowModal()
        dialog.Destroy()

#--------------------------------------------------------------------------
class Save_EditPCSpellsData(ListEditorData):
    """Data capsule for pc spell editing dialog."""
    def __init__(self,parent,saveInfo):
        """Initialize."""
        self.saveSpells = bosh.SaveSpells(saveInfo)
        progress = ProgressDialog(_('Loading Masters'))
        try:
            self.saveSpells.load(progress)
        finally:
            progress = progress.Destroy()
        self.data = self.saveSpells.getPlayerSpells()
        self.removed = set()
        #--GUI
        ListEditorData.__init__(self,parent)
        self.showRemove = True
        self.showInfo = True
        self.showSave = True
        self.showCancel = True

    def getItemList(self):
        """Returns load list keys in alpha order."""
        return sorted(self.data.keys(),key=lambda a: a.lower())

    def getInfo(self,item):
        """Returns string info on specified item."""
        iref,record = self.data[item]
        return record.getEffectsSummary()

    def remove(self,item):
        """Removes item. Return true on success."""
        if not item in self.data: return False
        iref,record = self.data[item]
        self.removed.add(iref)
        del self.data[item]
        return True

    def save(self):
        """Handles save button click."""
        self.saveSpells.removePlayerSpells(self.removed)

#------------------------------------------------------------------------------
class Save_EditPCSpells(Link):
    """Save spell list editing dialog."""
    def AppendToMenu(self,menu,window,data):
        """Append ref replacer items to menu."""
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Delete Spells...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) == 1)

    def Do(self,event):
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        data = Save_EditPCSpellsData(self.window,fileInfo)
        dialog = ListEditorDialog(self.window,-1,_('Player Spells'),data)
        dialog.ShowModal()
        dialog.Destroy()

#------------------------------------------------------------------------------
class Save_RepairAbomb(Link):
    """Repairs animation slowing by resetting counter(?) at end of TesClass data."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Repair Abomb'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) == 1)

    def Do(self,event):
        #--File Info
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        #--Check current value
        saveFile = bosh.SaveFile(fileInfo)
        saveFile.load()
        (tcSize,abombCounter,abombFloat) = saveFile.getAbomb()
        #--Continue?
        newCounter = 0x41000000
        if abombCounter < newCounter:
            Message(self.window,_('Abomb counter too low (0x%08X) to reset.') % (abombCounter,))
            return
        message = (_("Reset Abomb counter from 0x%08X to 0x%08X?\n\nAbomb related animation slowing typically becomes noticeable after Abomb counter reaches 0x49000000.\n\nNOTE: While this action should repair animation slowing, it's possible that it may break something else, perhaps rendering your game unplayable in some other way.") % (abombCounter,newCounter))
        if ContinueQuery(self.window,message,'bash.abombRepair.continue',_('Repair Abomb')) != wx.ID_OK:
            return
        #--Do it
        saveFile.setAbomb(newCounter)
        saveFile.safeSave()
        Message(self.window,_('Abomb counter reset from 0x%08X to 0x%08X.') % (abombCounter,newCounter))

#------------------------------------------------------------------------------
class Save_RepairFactions(Link):
    """Repair factions from v 105 Bash error, plus mod faction changes."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Repair Factions'))
        menu.AppendItem(menuItem)
        menuItem.Enable(bool(bosh.modInfos.ordered) and len(data) == 1)

    def Do(self,event):
        debug = False
        message = _('This will (mostly) repair faction membership errors due to Wrye Bash v 105 bug and/or faction changes in underlying mods.\n\nWARNING! This repair is NOT perfect! Do not use it unless you have to!')
        if ContinueQuery(self.window,message,'bash.repairFactions.continue',_('Update NPC Levels')) != wx.ID_OK:
            return
        question = _("Restore dropped factions too? WARNING: This may involve clicking through a LOT of yes/no dialogs.")
        restoreDropped = YesQuery(self.window, question, _('Repair Factions'),default=False)
        progress = ProgressDialog(_('Repair Factions'))
        legitNullSpells = bush.repairFactions_legitNullSpells
        legitNullFactions = bush.repairFactions_legitNullFactions
        legitDroppedFactions = bush.repairFactions_legitDroppedFactions
        try:
            #--Loop over active mods
            log = bosh.LogFile(cStringIO.StringIO())
            offsetFlag = 0x80
            npc_info = {}
            fact_eid = {}
            loadFactory = bosh.LoadFactory(False,bosh.MreNpc,bosh.MreFact)
            ordered = list(bosh.modInfos.ordered)
            subProgress = SubProgress(progress,0,0.4,len(ordered))
            for index,modName in enumerate(ordered):
                subProgress(index,_("Scanning ") + modName)
                modInfo = bosh.modInfos[modName]
                modFile = bosh.ModFile(modInfo,loadFactory)
                modFile.load(True)
                #--Loop over mod NPCs
                mapToOrdered = bosh.MasterMap(modFile.tes4.masters+[modName], ordered)
                for npc in modFile.NPC_.getActiveRecords():
                    formid = mapToOrdered(npc.formid,None)
                    if not formid: continue
                    factions = []
                    for entry in npc.factions:
                        factionId = mapToOrdered(entry.faction,None)
                        if not factionId: continue
                        factions.append((factionId,entry.rank))
                    npc_info[formid] = (npc.eid,factions)
                #--Loop over mod factions
                for fact in modFile.FACT.getActiveRecords():
                    formid = mapToOrdered(fact.formid,None)
                    if not formid: continue
                    fact_eid[formid] = fact.eid
            #--Loop over savefiles
            subProgress = SubProgress(progress,0.4,1.0,len(self.data))
            message = _("NPC Factions Restored/UnNulled:")
            for index,saveName in enumerate(self.data):
                log.setHeader('== '+saveName,True)
                subProgress(index,_("Updating ") + saveName)
                saveInfo = self.window.data[saveName]
                saveFile = bosh.SaveFile(saveInfo)
                saveFile.load()
                records = saveFile.records
                mapToOrdered = bosh.MasterMap(saveFile.masters, ordered)
                mapToSave = bosh.MasterMap(ordered,saveFile.masters)
                refactionedCount = unNulledCount = 0
                for recNum in xrange(len(records)):
                    unFactioned = unSpelled = unModified = refactioned = False
                    (recId,recType,recFlags,version,data) = records[recNum]
                    if recType != 35: continue
                    orderedRecId = mapToOrdered(recId,None)
                    eid = npc_info.get(orderedRecId,('',))[0]
                    npc = bosh.SreNPC(recFlags,data)
                    recFlags = bosh.SreNPC.flags(recFlags)
                    #--Fix Bash v 105 null array bugs
                    if recFlags.factions and not npc.factions and recId not in legitNullFactions:
                        log('. %08X %s -- Factions' % (recId,eid))
                        npc.factions = None
                        unFactioned = True
                    if recFlags.modifiers and not npc.modifiers:
                        log('. %08X %s -- Modifiers' % (recId,eid))
                        npc.modifiers = None
                        unModified = True
                    if recFlags.spells and not npc.spells and recId not in legitNullSpells:
                        log('. %08X %s -- Spells' % (recId,eid))
                        npc.spells = None
                        unSpelled = True
                    unNulled = (unFactioned or unSpelled or unModified)
                    unNulledCount += (0,1)[unNulled]
                    #--Player, player faction
                    if recId == 7:
                        playerStartSpell = saveFile.getIref(0x00000136)
                        if npc.spells != None and playerStartSpell not in npc.spells:
                            log('. %08X %s -- **DefaultPlayerSpell**' % (recId,eid))
                            npc.spells.append(playerStartSpell)
                            refactioned = True #--I'm lying, but... close enough.
                        playerFactionIref = saveFile.getIref(0x0001dbcd)
                        if (npc.factions != None and 
                            playerFactionIref not in [iref for iref,level in npc.factions]
                            ):
                                log('. %08X %s -- **PlayerFaction, 0**' % (recId,eid))
                                npc.factions.append((playerFactionIref,0))
                                refactioned = True
                    #--Compare to mod data
                    elif orderedRecId in npc_info and restoreDropped:
                        (npcEid,factions) = npc_info[orderedRecId]
                        #--Refaction?
                        if npc.factions and factions:
                            curFactions = set([iref for iref,level in npc.factions])
                            for orderedId,level in factions:
                                formid = mapToSave(orderedId,None)
                                if not formid: continue
                                iref = saveFile.getIref(formid)
                                if iref not in curFactions and (recId,formid) not in legitDroppedFactions:
                                    factEid = fact_eid.get(orderedId,'------')
                                    question = _('Restore %s to %s faction?') % (npcEid,factEid)
                                    if debug: 
                                        print 'refactioned %08X %08X %s %s' % (recId,formid,npcEid,factEid)
                                    elif not YesQuery(self.window, question, saveName,default=False):
                                        continue
                                    log('. %08X %s -- **%s, %d**' % (recId,eid,factEid,level))
                                    npc.factions.append((iref,level))
                                    refactioned = True
                    refactionedCount += (0,1)[refactioned]
                    #--Save record changes?
                    if unNulled or refactioned:
                        saveFile.records[recNum] = (recId,recType,npc.getFlags(),version,npc.getData())
                #--Save changes?
                subProgress(index+0.5,_("Updating ") + saveName)
                if unNulledCount or refactionedCount:
                    saveFile.safeSave()
                message += '\n%d %d %s' % (refactionedCount,unNulledCount,saveName,)
            progress.Destroy()
            #Message(self.window,message,_('Repair Factions'))
            message = log.out.getvalue()
            WtxtLogMessage(self.window,message,_('Repair Factions'))
        finally:
            if progress: progress.Destroy()

#------------------------------------------------------------------------------
class Save_RepairFbomb(Link):
    """Repairs savegame formid rollover problem by resetting nextformid field."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Repair Fbomb'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        #--File Info
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        #--Check current value
        saveFile = bosh.SaveFile(fileInfo)
        saveFile.load()
        curNextFI, = struct.unpack('I',saveFile.preGlobals[:4])
        #--Continue?
        newNextFI = 0xFF200000
        message = _("Reset NextFormId (current value: 0x%08X)?\n\nWARNING: This repair is extremely experimental! Use at your own risk.\n\n") % (curNextFI,)
        if curNextFI >> 24 != 0xFF:
            message += _("Note: This savegame is already suffering from the FBomb error.")
        elif curNextFI >= 0xFFF00000:
            message += _("Note: Fbomb error is imminent for this savegame.")
        elif curNextFI >= 0xFFE00000:
            message += _("Note: Fbomb error is dangerously high for this savegame.")
        elif curNextFI >= 0xFFC00000:
            message += _("Note: Fbomb number is high, but not imminent for this savegame.")
        else:
            message += _("Your current FI is not dangerously high. It is recommended that you NOT proceed with the repair.")
            if curNextFI < newNextFI: newNextFI = 0xFF100000
        message = fill(message,60)
        result = TextEntry(self.window,message,_("Repair Fbomb"),'%08X' % (newNextFI,))
        if not result: return
        newNextFI = int(result,16)
        if newNextFI < 0 or newNextFI > 0xFFFFFFFF: return
        #--Do it
        saveFile.preGlobals = struct.pack('I',newNextFI)+saveFile.preGlobals[4:]
        saveFile.safeSave()
        Message(self.window,_('NextFormId reset from 0x%08X to 0x%08X.') % (curNextFI,newNextFI))

#------------------------------------------------------------------------------
class Save_RepairHair(Link):
    """Repairs hair that has been zeroed due to removal of a hair mod."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Repair Hair'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        #--File Info
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        if bosh.PCFaces.save_repairHair(fileInfo):
            Message(self.window,_('Hair repaired.'))
        else:
            Message(self.window,_('No repair necessary.'))

#------------------------------------------------------------------------------
class Save_ReweighPotions(Link):
    """Changes weight of all player potions to specified value."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Reweigh Potions...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) == 1)

    def Do(self,event):
        #--Query value
        result = TextEntry(self.window,
            _("Set weight of all player potions to..."),
            _("Reweigh Potions"),
            '%0.2f' % (settings.get('bash.reweighPotions.newWeight',0.2),))
        if not result: return
        newWeight = float(result.strip())
        if newWeight < 0 or newWeight > 100: 
            Message(self.window,_('Invalid weight: %f') % (newWeight,))
            return
        settings['bash.reweighPotions.newWeight'] = newWeight
        #--Do it
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        progress = ProgressDialog(_("Reweigh Potions"))
        try:
            saveFile = bosh.SaveFile(fileInfo)
            saveFile.load(SubProgress(progress,0,0.5))
            count = 0
            progress(0.5,_("Processing."))
            for index,record in enumerate(saveFile.created):
                if record.type == 'ALCH':
                    record = record.getTypeCopy()
                    record.weight = newWeight
                    record.getSize()
                    saveFile.created[index] = record
                    count += 1
            if count:
                saveFile.safeSave(SubProgress(progress,0.6,1.0))
                progress.Destroy()
                Message(self.window,_('Potions reweighed: %d.') % (count,))
            else:
                progress.Destroy()
                Message(self.window,_('No potions to reweigh!'))
        finally:
            if progress: progress.Destroy()

##------------------------------------------------------------------------------
class Save_Stats(Link):
    """Show savefile statistics."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Statistics'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        fileName = Path.get(self.data[0])
        fileInfo = self.window.data[fileName]
        saveFile = bosh.SaveFile(fileInfo)
        progress = ProgressDialog(_("Statistics"))
        try:
            saveFile.load(SubProgress(progress,0,0.9))
            log = bosh.LogFile(cStringIO.StringIO())
            progress(0.9,_("Calculating statistics."))
            saveFile.logStats(log)
            progress.Destroy()
            text = log.out.getvalue()
            LogMessage(self.window,text,fileName,asDialog=False,fixedFont=False)
        finally:
            progress.Destroy()

#------------------------------------------------------------------------------
class Save_Unbloat(Link):
    """Unbloats savegame."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Remove Bloat...'))
        menu.AppendItem(menuItem)
        if len(data) != 1: menuItem.Enable(False)

    def Do(self,event):
        #--File Info
        saveName = Path.get(self.data[0])
        saveInfo = self.window.data[saveName]
        progress = ProgressDialog(_("Scanning for Bloat"))
        delObjRefs = 0
        try:
            #--Scan and report
            saveFile = bosh.SaveFile(saveInfo)
            saveFile.load(SubProgress(progress,0,0.8))
            createdCounts,nullRefCount = saveFile.findBloating(SubProgress(progress,0.8,1.0))
            progress.Destroy()
            #--Dialog
            if not createdCounts and not nullRefCount:
                Message(self.window,_("No bloating found."))
                return
            message = ''
            if createdCounts:
                #message += _('Excess Created Objects\n')
                for type,name in sorted(createdCounts):
                    message += '  %s %s: %s\n' % (type,name,bosh.formatInteger(createdCounts[(type,name)]))
            if nullRefCount:
                message += _('  Null Ref Objects: %s\n') % (bosh.formatInteger(nullRefCount),)
            message = _("Remove savegame bloating?\n%s\nWARNING: This is a risky procedure that may corrupt your savegame! Use only if necessary!") % (message,)
            if not YesQuery(self.window,message,_("Remove bloating?")): 
                return
            #--Remove bloating
            progress = ProgressDialog(_("Removing Bloat"))
            nums = saveFile.removeBloating(createdCounts.keys(),True,SubProgress(progress,0,0.9))
            progress(0.9,_("Saving..."))
            saveFile.safeSave()
            progress.Destroy()
            Message(self.window,_("Uncreated Objects: %d\nUncreated Refs: %d\nUnNulled Refs: %d") % nums)
            self.window.RefreshUI(saveName)
        finally:
            progress.Destroy()


#------------------------------------------------------------------------------
class Save_UpdateNPCLevels(Link):
    """Update NPC levels from active mods."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Update NPC Levels...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(bool(data and bosh.modInfos.ordered))

    def Do(self,event):
        debug = True
        message = _('This will relevel the NPCs in the selected save game(s) according to the npc levels in the currently active mods. This supercedes the older "Import NPC Levels" command.')
        if ContinueQuery(self.window,message,'bash.updateNpcLevels.continue',_('Update NPC Levels')) != wx.ID_OK:
            return
        progress = ProgressDialog(_('Update NPC Levels'))
        try:
            #--Loop over active mods
            offsetFlag = 0x80
            npc_info = {}
            loadFactory = bosh.LoadFactory(False,bosh.MreNpc)
            ordered = list(bosh.modInfos.ordered)
            subProgress = SubProgress(progress,0,0.4,len(ordered))
            for index,modName in enumerate(ordered):
                subProgress(index,_("Scanning ") + modName)
                modInfo = bosh.modInfos[modName]
                modFile = bosh.ModFile(modInfo,loadFactory)
                modFile.load(True)
                if 'NPC_' not in modFile.tops: continue
                #--Loop over mod NPCs
                mapToOrdered = bosh.MasterMap(modFile.tes4.masters+[modName], ordered)
                for npc in modFile.NPC_.getActiveRecords():
                    formid = mapToOrdered(npc.formid,None)
                    if not formid: continue
                    npc_info[formid] = (npc.eid, npc.level, npc.calcMin, npc.calcMax, npc.flags.pcLevelOffset)
            #--Loop over savefiles
            subProgress = SubProgress(progress,0.4,1.0,len(self.data))
            message = _("NPCs Releveled:")
            for index,saveName in enumerate(self.data):
                deprint(saveName, '==============================')
                subProgress(index,_("Updating ") + saveName)
                saveInfo = self.window.data[saveName]
                saveFile = bosh.SaveFile(saveInfo)
                saveFile.load()
                records = saveFile.records
                mapToOrdered = bosh.MasterMap(saveFile.masters, ordered)
                releveledCount = 0
                #--Loop over change records
                for recNum in xrange(len(records)):
                    releveled = False
                    (recId,recType,recFlags,version,data) = records[recNum]
                    orderedRecId = mapToOrdered(recId,None)
                    if recType != 35 or recId == 7 or orderedRecId not in npc_info: continue
                    (eid,level,calcMin,calcMax,pcLevelOffset) = npc_info[orderedRecId]
                    npc = bosh.SreNPC(recFlags,data)
                    acbs = npc.acbs
                    if acbs and (
                        (acbs.level != level) or 
                        (acbs.calcMin != calcMin) or
                        (acbs.calcMax != calcMax) or
                        (acbs.flags.pcLevelOffset != pcLevelOffset)
                        ):
                        acbs.flags.pcLevelOffset = pcLevelOffset
                        acbs.level = level
                        acbs.calcMin = calcMin
                        acbs.calcMax = calcMax
                        (recId,recType,recFlags,version,data) = saveFile.records[recNum]
                        records[recNum] = (recId,recType,npc.getFlags(),version,npc.getData())
                        releveledCount += 1
                        saveFile.records[recNum] = npc.getTuple(recId,version) 
                        deprint(hex(recId), eid, acbs.level, acbs.calcMin, acbs.calcMax, acbs.flags.getTrueAttrs())
                #--Save changes?
                subProgress(index+0.5,_("Updating ") + saveName)
                if releveledCount:
                    saveFile.safeSave()
                message += '\n%d %s' % (releveledCount,saveName,)
            progress.Destroy()
            Message(self.window,message,_('Update NPC Levels'))
        finally:
            if progress: progress.Destroy()

# Screen Links ------------------------------------------------------------------
#------------------------------------------------------------------------------
class Screens_NextScreenShot(Link):
    """Sets screenshot base name and number."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Next Shot...'))
        menu.AppendItem(menuItem)

    def Do(self,event):
        oblivionIni = bosh.oblivionIni
        display = oblivionIni.getSettings()['Display']
        base,next = display.get('SScreenShotBaseName','ScreenShot'), display.get('iScreenShotIndex','0')
        rePattern = re.compile(r'^(.+?)(\d*)$',re.I)
        pattern = TextEntry(self.window,_("Screenshot base name, optionally with next screenshot number.\nE.g. ScreenShot or ScreenShot_101 or Subdir\\ScreenShot_201."),_("Next Shot..."),base+next)
        if not pattern: return
        maPattern = rePattern.match(pattern)
        newBase,newNext = maPattern.groups()
        settings = {'Display':{
            'SScreenShotBaseName': newBase,
            'iScreenShotIndex': (newNext or next),
            'bAllowScreenShot': '1',
            }}
        oblivionIni.saveSettings(settings)
        bosh.screensData.refresh()
        self.window.RefreshUI()

#------------------------------------------------------------------------------
class Screen_ConvertToJpg(Link):
    """Converts selected images to jpg files."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Convert to jpg'))
        menu.AppendItem(menuItem)
        convertable = [name for name in self.data if Path.get(name).ext() != '.jpg']
        menuItem.Enable(len(convertable) > 0)

    def Do(self,event):
        #--File Info
        srcDir = self.window.data.dir
        progress = ProgressDialog(_("Converting to Jpg"))
        try:
            progress.setFull(len(self.data))
            srcDir = bosh.screensData.dir
            for index,fileName in enumerate(self.data):
                progress(index,fileName)
                srcPath = srcDir.join(fileName)
                destPath = srcPath.root()+'.jpg'
                if srcPath == destPath or destPath.exists(): continue
                bitmap = wx.Bitmap(srcPath)
                result = bitmap.SaveFile(destPath,wx.BITMAP_TYPE_JPEG)
                if not result: continue
                srcPath.remove()
        finally:
            if progress: progress.Destroy()
            bosh.screensData.refresh()
            self.window.RefreshUI()

#------------------------------------------------------------------------------
class Screen_Rename(Link):
    """Renames files by pattern."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Rename...'))
        menu.AppendItem(menuItem)
        menuItem.Enable(len(data) > 0)

    def Do(self,event):
        #--File Info
        rePattern = re.compile(r'^([^\\/]+?)(\d*)(\.(jpg|bmp))$',re.I)
        fileName0 = self.data[0]
        pattern = TextEntry(self.window,_("Enter new name. E.g. Screenshot 123.bmp"),_("Rename..."))
        if not pattern: return
        maPattern = rePattern.match(pattern)
        if not maPattern:
            ErrorMessage(self.window,_("Bad extension or file root: ")+pattern)
            return
        root,numStr,ext = maPattern.groups()[:3]
        numLen = len(numStr)
        num = int(numStr or '0')
        screensDir = bosh.screensData.dir
        for oldName in map(Path,self.data):
            newName = root+numStr+oldName.ext()
            if newName != oldName: 
                oldPath = screensDir.join(oldName)
                newPath = screensDir.join(newName)
                if not newPath.exists():
                    oldPath.rename(newPath)
            num += 1
            numStr = `num`
            numStr = '0'*(numLen-len(numStr))+numStr
        bosh.screensData.refresh()
        self.window.RefreshUI()

# Messages Links ------------------------------------------------------------------
#------------------------------------------------------------------------------
class Messages_Archive_Import(Link):
    """Import messages from html message archive."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_('Import Archives...'))
        menu.AppendItem(menuItem)

    def Do(self,event):
        textDir = settings.get('bash.workDir',bosh.dirs['app'])
        #--File dialog
        paths = MultiOpenDialog(self.window,_('Import message archive(s):'),textDir,
            '', '*.html')
        if not paths: return
        settings['bash.workDir'] = paths[0].head()
        for path in paths:
            bosh.messages.importArchive(path)
        self.window.RefreshUI()

#------------------------------------------------------------------------------
class Message_Delete(Link):
    """Delete the file and all backups."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menu.AppendItem(wx.MenuItem(menu,self.id,_('Delete')))

    def Do(self,event):
        message = _(r'Delete these %d message(s)? This operation cannot be undone.') % (len(self.data),)
        if not YesQuery(self.window,message,_('Delete Messages')):
            return
        #--Do it
        for message in self.data:
            self.window.data.delete(message)
        #--Refresh stuff
        self.window.RefreshUI()

# Masters Links ---------------------------------------------------------------

# Master Links ----------------------------------------------------------------
#------------------------------------------------------------------------------
class Master_ChangeTo(Link):
    """Rename/replace master through file dialog."""
    def AppendToMenu(self,menu,window,data):
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_("Change to..."))
        menu.AppendItem(menuItem)
        menuItem.Enable(self.window.edited)

    def Do(self,event):
        itemId = self.data[0]
        masterInfo = self.window.data[itemId]
        masterName = masterInfo.name
        #--File Dialog
        wildcard = _('Oblivion Mod Files')+' (*.esp;*.esm)|*.esp;*.esm'
        newPath = OpenDialog(self.window,_('Change master name to:'),
            bosh.modInfos.dir, masterName, wildcard)
        if not newPath: return
        (newDir,newName) = newPath.split()
        #--Valid directory?
        if newDir != bosh.modInfos.dir:
            ErrorMessage(self.window,
                _("File must be selected from Oblivion Data Files directory."))
            return
        elif newName == masterName:
            return
        #--Save Name
        masterInfo.setName(newName)
        self.window.ReList()
        self.window.PopulateItems()
        settings.getChanged('bash.mods.renames')[masterName] = newName

#------------------------------------------------------------------------------
class Master_Disable(Link):
    """Rename/replace master through file dialog."""
    def AppendToMenu(self,menu,window,data):
        if window.fileInfo.isMod(): return #--Saves only
        Link.AppendToMenu(self,menu,window,data)
        menuItem = wx.MenuItem(menu,self.id,_("Disable"))
        menu.AppendItem(menuItem)
        menuItem.Enable(self.window.edited)

    def Do(self,event):
        itemId = self.data[0]
        masterInfo = self.window.data[itemId]
        masterName = masterInfo.name
        newName = Path.get(re.sub('[mM]$','p','XX'+masterName))
        #--Save Name
        masterInfo.setName(newName)
        self.window.ReList()
        self.window.PopulateItems()

# App Links -------------------------------------------------------------------
#------------------------------------------------------------------------------
class App_Button(Link):
    """Launch an application."""
    obseButtons = []

    def __init__(self,exePathArgs,image,tip,obseTip=None,obseArg=None):
        """Initialize
        exePathArgs (string): exePath
        exePathArgs (tuple): (exePath,*exeArgs)"""
        Link.__init__(self)
        self.gButton = None
        if isinstance(exePathArgs,tuple):
            self.exePath = exePathArgs[0]
            self.exeArgs = exePathArgs[1:]
        else:
            self.exePath = exePathArgs
            self.exeArgs = tuple()
        self.image = image
        self.tip = tip
        #--OBSE stuff
        self.obseTip = obseTip
        self.obseArg = obseArg
    
    def GetBitmapButton(self,window,style=0):
        if self.exePath.exists(): 
            gButton = self.gButton = wx.BitmapButton(window,-1,self.image.GetBitmap(),style=style)
            gButton.Bind(wx.EVT_BUTTON,self.Do)
            if self.tip: gButton.SetToolTip(tooltip(self.tip))
            if self.obseArg != None: 
                App_Button.obseButtons.append(self)
            return gButton
        else:
            return None

    def Do(self,event):
        exeObse = bosh.dirs['app'].join('obse_loader.exe')
        exeArgs = self.exeArgs
        if self.obseArg != None and settings.get('bash.obse.on',False) and exeObse.exists():
            exePath = exeObse
            if self.obseArg != '': exeArgs += (self.obseArg,)
        else:
            exePath = self.exePath
        cwd = os.getcwd()
        os.chdir(exePath.head())
        os.spawnv(os.P_NOWAIT,exePath,exeArgs)
        os.chdir(cwd)

#------------------------------------------------------------------------------
class Oblivion_Button(App_Button):
    """Will close app on execute if autoquit is on."""
    def Do(self,event):
        App_Button.Do(self,event)
        if settings.get('bash.autoQuit.on',False):
            bashFrame.Close()

#------------------------------------------------------------------------------
class Obse_Button(Link):
    """Obse on/off state button."""
    def __init__(self):
        Link.__init__(self)
        self.gButton = None
    
    def SetState(self,state=None):
        """Sets state related info. If newState != none, sets to new state first. 
        For convenience, returns state when done."""
        if state == None: #--Default
            state = settings.get('bash.obse.on',False)
        elif state == -1: #--Invert
            state = not settings.get('bash.obse.on',False)
        settings['bash.obse.on'] = state
        image = images[('checkbox.green.off','checkbox.green.on')[state]]
        tip = (_("OBSE Disabled"),_("OBSE Enabled"))[state]
        self.gButton.SetBitmapLabel(image.GetBitmap())
        self.gButton.SetToolTip(tooltip(tip))
        tipAttr = ('tip','obseTip')[state]
        for button in App_Button.obseButtons:
            button.gButton.SetToolTip(tooltip(getattr(button,tipAttr,'')))

    def GetBitmapButton(self,window,style=0):
        exeObse = bosh.dirs['app'].join('obse_loader.exe')
        if exeObse.exists(): 
            bitmap = images['checkbox.green.off'].GetBitmap()
            gButton = self.gButton = wx.BitmapButton(window,-1,bitmap,style=style)
            gButton.Bind(wx.EVT_BUTTON,self.Do)
            gButton.SetSize((16,16))
            self.SetState()
            return gButton
        else:
            return None

    def Do(self,event):
        """Invert state."""
        self.SetState(-1)

#------------------------------------------------------------------------------
class AutoQuit_Button(Link):
    """Button toggling application closure when launching Oblivion."""
    def __init__(self):
        Link.__init__(self)
        self.gButton = None
    
    def SetState(self,state=None):
        """Sets state related info. If newState != none, sets to new state first. 
        For convenience, returns state when done."""
        if state == None: #--Default
            state = settings.get('bash.autoQuit.on',False)
        elif state == -1: #--Invert
            state = not settings.get('bash.autoQuit.on',False)
        settings['bash.autoQuit.on'] = state
        image = images[('checkbox.red.off','checkbox.red.x')[state]]
        tip = (_("Auto-Quit Disabled"),_("Auto-Quit Enabled"))[state]
        self.gButton.SetBitmapLabel(image.GetBitmap())
        self.gButton.SetToolTip(tooltip(tip))

    def GetBitmapButton(self,window,style=0):
        bitmap = images['checkbox.red.off'].GetBitmap()
        gButton = self.gButton = wx.BitmapButton(window,-1,bitmap,style=style)
        gButton.Bind(wx.EVT_BUTTON,self.Do)
        gButton.SetSize((16,16))
        self.SetState()
        return gButton

    def Do(self,event):
        """Invert state."""
        self.SetState(-1)

#------------------------------------------------------------------------------
class App_Help(Link):
    """Show help browser."""
    def GetBitmapButton(self,window,style=0):
        if not self.id: self.id = wx.NewId()
        button = wx.BitmapButton(window,self.id,images['help'].GetBitmap(),style=style)
        button.SetToolTip(wx.ToolTip(_("Help File")))
        wx.EVT_BUTTON(button,self.id,self.Do)
        return button

    def Do(self,event):
        """Handle menu selection."""
        os.startfile(bosh.Path.getcwd().join(_('Wrye Bash.html')))

#------------------------------------------------------------------------------
class App_DocBrowser(Link):
    """Show doc browser."""
    def GetBitmapButton(self,window,style=0):
        if not self.id: self.id = wx.NewId()
        button = wx.BitmapButton(window,self.id,images['doc.on'].GetBitmap(),style=style)
        button.SetToolTip(wx.ToolTip(_("Doc Browser")))
        wx.EVT_BUTTON(button,self.id,self.Do)
        return button

    def Do(self,event):
        """Handle menu selection."""
        if not docBrowser: 
            DocBrowser().Show()
            settings['bash.modDocs.show'] = True
        docBrowser.Raise()

#------------------------------------------------------------------------------
class App_BashMon(Link):
    """Start bashmon."""
    def GetBitmapButton(self,window,style=0):
        if not self.id: self.id = wx.NewId()
        button = wx.BitmapButton(window,self.id,images['bashmon'].GetBitmap(),style=style)
        button.SetToolTip(wx.ToolTip(_("BashMon")))
        wx.EVT_BUTTON(button,self.id,self.Do)
        return button

    def Do(self,event):
        """Handle menu selection."""
        os.startfile(bosh.Path.getcwd().join(_('bashmon.py')))

# Initialization --------------------------------------------------------------
def InitSettings():
    """Initializes settings dictionary for bosh and basher."""
    bosh.initSettings()
    global settings
    settings = bosh.settings
    settings.loadDefaults(settingDefaults)

def InitImages():
    """Initialize image collection."""
    #--Standard
    images['save.on'] = Image(r'images\save_on.png',wx.BITMAP_TYPE_PNG)
    images['save.off'] = Image(r'images\save_off.png',wx.BITMAP_TYPE_PNG)
    #--Misc
    #images['oblivion'] = Image(r'images\oblivion.png',wx.BITMAP_TYPE_PNG)
    images['help'] = Image(r'images\help.png',wx.BITMAP_TYPE_PNG)
    #--Tools
    images['doc.on'] = Image(r'images\doc_on.png',wx.BITMAP_TYPE_PNG)
    images['bashmon'] = Image(r'images\dos.png',wx.BITMAP_TYPE_PNG)
    #--Checkboxes
    images['bash.checkboxes'] = Checkboxes()
    images['checkbox.red.x'] = Image(r'images\checkbox_red_x.png',wx.BITMAP_TYPE_PNG)
    images['checkbox.green.on.32'] = (
        Image(r'images\checkbox_green_on_32.png',wx.BITMAP_TYPE_PNG))
    images['checkbox.blue.on.32'] = (
        Image(r'images\checkbox_blue_on_32.png',wx.BITMAP_TYPE_PNG))
    #--Bash
    images['bash.16'] = Image(r'images\bash_16.png',wx.BITMAP_TYPE_PNG)
    images['bash.32'] = Image(r'images\bash_32.png',wx.BITMAP_TYPE_PNG)
    images['bash.16.blue'] = Image(r'images\bash_16_blue.png',wx.BITMAP_TYPE_PNG)
    images['bash.32.blue'] = Image(r'images\bash_32_blue.png',wx.BITMAP_TYPE_PNG)
    #--Applications Icons
    wryeBashIcons = ImageBundle()
    wryeBashIcons.Add(images['bash.16'])
    wryeBashIcons.Add(images['bash.32'])
    images['bash.icons'] = wryeBashIcons
    #--Application Subwindow Icons
    wryeBashIcons2 = ImageBundle()
    wryeBashIcons2.Add(images['bash.16.blue'])
    wryeBashIcons2.Add(images['bash.32.blue'])
    images['bash.icons2'] = wryeBashIcons2

def InitStatusBar():
    """Initialize status bar links."""
    #--Bash Status/LinkBar
    #statusBarButtons.append(App_Oblivion())
    statusBarButtons.append( 
        Oblivion_Button(
            bosh.dirs['app'].join('Oblivion.exe'),
            Image(r'images\oblivion.png'),
            _("Launch Oblivion"),
            _("Launch Oblivion + OBSE"),
            ''))
    statusBarButtons.append( 
        App_Button(
            bosh.dirs['app'].join('OblivionModManager.exe'),
            Image(r'images\obmm.png'),
            _("Launch OBMM")))
    statusBarButtons.append( 
        App_Button(
            bosh.dirs['app'].join('TESConstructionSet.exe'),
            Image(r'images\tescs.png'),
            _("Launch TESCS"),
            _("Launch TESCS + OBSE"),
            ' -editor'))
    statusBarButtons.append( 
        App_Button(
            bosh.dirs['app'].join('Tes4View.exe'),
            Image(r'images\tes4view.png'),
            _("Launch Tes4View")))
    statusBarButtons.append( 
        App_Button(
            bosh.dirs['app'].join('Tes4Edit.exe'),
            Image(r'images\tes4view.png'),
            _("Launch Tes4Edit")))
    statusBarButtons.append( 
        App_Button(
            bosh.dirs['app'].join('Tes4Trans.exe'),
            Image(r'images\tes4view.png'),
            _("Launch Tes4Trans")))
    statusBarButtons.append(Obse_Button())
    statusBarButtons.append(AutoQuit_Button())
    statusBarButtons.append(App_BashMon())
    statusBarButtons.append(App_Help())
    statusBarButtons.append(App_DocBrowser())

def InitMasterLinks():
    """Initialize master list menus."""
    #--MasterList: Column Links
    if True: #--Sort by
        sortMenu = MenuLink(_("Sort by"))
        sortMenu.links.append(Mods_EsmsFirst())
        sortMenu.links.append(SeparatorLink())
        sortMenu.links.append(Files_SortBy('File'))
        sortMenu.links.append(Files_SortBy('Author'))
        sortMenu.links.append(Files_SortBy('Group'))
        sortMenu.links.append(Files_SortBy('Load Order'))
        sortMenu.links.append(Files_SortBy('Modified'))
        sortMenu.links.append(Files_SortBy('Save Order'))
        sortMenu.links.append(Files_SortBy('Status'))
        MasterList.colLinks.append(sortMenu)
    #--------------------------------------------
    #mastersMainMenu.append(SeparatorLink())
    #mastersMainMenu.append(Masters_CopyList())
    
    #--MasterList: Item Links
    mastersItemMenu.append(Master_ChangeTo())
    mastersItemMenu.append(Master_Disable())
    
def InitModLinks():
    """Initialize Mods tab menus."""
    #--ModList: Column Links
    if True: #--Load
        loadMenu = MenuLink(_("Load"))
        loadMenu.links.append(Mods_LoadList())
        modsMainMenu.append(loadMenu)
    if True: #--Sort by
        sortMenu = MenuLink(_("Sort by"))
        sortMenu.links.append(Mods_EsmsFirst())
        sortMenu.links.append(Mods_SelectedFirst())
        sortMenu.links.append(SeparatorLink())
        sortMenu.links.append(Files_SortBy('File'))
        sortMenu.links.append(Files_SortBy('Author'))
        sortMenu.links.append(Files_SortBy('Group'))
        sortMenu.links.append(Files_SortBy('Load Order'))
        sortMenu.links.append(Files_SortBy('Modified'))
        sortMenu.links.append(Files_SortBy('Rating'))
        sortMenu.links.append(Files_SortBy('Size'))
        sortMenu.links.append(Files_SortBy('Status'))
        modsMainMenu.append(sortMenu)
    if True: #--Versions
        versionsMenu = MenuLink("Oblivion.esm")
        versionsMenu.links.append(Mods_OblivionVersion('1.1'))
        versionsMenu.links.append(Mods_OblivionVersion('SI'))
        modsMainMenu.append(versionsMenu)        
    #--------------------------------------------
    modsMainMenu.append(SeparatorLink())
    modsMainMenu.append(Files_Open())
    modsMainMenu.append(Files_Unhide('mod'))
    modsMainMenu.append(SeparatorLink())
    modsMainMenu.append(Mods_OblivionIni())
    modsMainMenu.append(Mods_IniTweaks())
    modsMainMenu.append(Mods_ListMods())
    modsMainMenu.append(Mods_LockTimes())
    modsMainMenu.append(SeparatorLink())
    modsMainMenu.append(Mods_Deprint())
    modsMainMenu.append(Mods_DumpTranslator())

    #--ModList: Item Links
    if True: #--File
        fileMenu = MenuLink(_("File"))
        fileMenu.links.append(Mod_CreateBlank())
        fileMenu.links.append(SeparatorLink())
        fileMenu.links.append(File_Backup())
        fileMenu.links.append(File_Duplicate())
        fileMenu.links.append(File_Snapshot())
        fileMenu.links.append(SeparatorLink())
        fileMenu.links.append(File_Delete())
        fileMenu.links.append(File_Hide())
        fileMenu.links.append(File_Redate())
        fileMenu.links.append(File_Sort())
        fileMenu.links.append(SeparatorLink())
        fileMenu.links.append(File_RevertToBackup())
        fileMenu.links.append(File_RevertToSnapshot())
        modsItemMenu.append(fileMenu)
    if True: #--Groups
        groupMenu = MenuLink(_("Group"))
        groupMenu.links.append(Mod_Groups())
        modsItemMenu.append(groupMenu)
    if True: #--Ratings
        ratingMenu = MenuLink(_("Rating"))
        ratingMenu.links.append(Mod_Ratings())
        modsItemMenu.append(ratingMenu)
    #--------------------------------------------
    modsItemMenu.append(SeparatorLink())
    if True: #--Export
        exportMenu = MenuLink(_("Export"))
        exportMenu.links.append(Mod_EditorIds_Export())
        exportMenu.links.append(Mod_FullNames_Export())
        exportMenu.links.append(Mod_ActorLevels_Export())
        exportMenu.links.append(Mod_Stats_Export())
        modsItemMenu.append(exportMenu)
    if True: #--Import
        importMenu = MenuLink(_("Import"))
        importMenu.links.append(Mod_Face_Import())
        importMenu.links.append(Mod_EditorIds_Import())
        importMenu.links.append(Mod_Formids_Replace())
        importMenu.links.append(Mod_FullNames_Import())
        importMenu.links.append(Mod_ActorLevels_Import())
        importMenu.links.append(Mod_Stats_Import())
        #importMenu.links.append(Mod_TNRFaces_Import())
        modsItemMenu.append(importMenu)
    modsItemMenu.append(Mod_Details())
    modsItemMenu.append(File_ListMasters())
    modsItemMenu.append(Mod_Patch_Update())
    modsItemMenu.append(SeparatorLink())
    modsItemMenu.append(Mod_ShowReadme())
    modsItemMenu.append(Mod_AddMaster())
    modsItemMenu.append(Mod_CopyToEsmp())
    modsItemMenu.append(Mod_FlipSelf())
    modsItemMenu.append(Mod_FlipMasters())
    modsItemMenu.append(Mod_MarkLevelers())
    modsItemMenu.append(Mod_MarkMergeable())
    modsItemMenu.append(Mod_SetVersion())
    
def InitReplacerLinks():
    """Initialize replacer tab menus."""
    replacersMainMenu.append(Files_Open())

    replacersItemMenu.append(File_Open())

def InitSaveLinks():
    """Initialize save tab menus."""
    #--SaveList: Column Links
    if True: #--Sort
        sortMenu = MenuLink(_("Sort by"))
        sortMenu.links.append(Files_SortBy('File'))
        sortMenu.links.append(Files_SortBy('Cell'))
        sortMenu.links.append(Files_SortBy('PlayTime'))
        sortMenu.links.append(Files_SortBy('Modified'))
        sortMenu.links.append(Files_SortBy('Player'))
        sortMenu.links.append(Files_SortBy('Status'))
        savesMainMenu.append(sortMenu)
    if True: #--Versions
        versionsMenu = MenuLink("Oblivion.esm")
        versionsMenu.links.append(Mods_OblivionVersion('1.1',True))
        versionsMenu.links.append(Mods_OblivionVersion('SI',True))
        savesMainMenu.append(versionsMenu)        
    if True: #--Save Profiles
        subDirMenu = MenuLink(_("Profile"))
        subDirMenu.links.append(Saves_Profiles())
        savesMainMenu.append(subDirMenu)
    savesMainMenu.append(SeparatorLink())
    savesMainMenu.append(Files_Open())
    savesMainMenu.append(Files_Unhide('save'))
    
    #--SaveList: Item Links
    if True: #--File
        fileMenu = MenuLink(_("File")) #>>
        fileMenu.links.append(File_Backup())
        fileMenu.links.append(File_Duplicate())
        fileMenu.links.append(File_Snapshot())
        fileMenu.links.append(SeparatorLink())
        fileMenu.links.append(File_Delete())
        fileMenu.links.append(File_Hide())
        fileMenu.links.append(SeparatorLink())
        fileMenu.links.append(File_RevertToBackup())
        fileMenu.links.append(File_RevertToSnapshot())
        savesItemMenu.append(fileMenu)
    #--------------------------------------------
    savesItemMenu.append(SeparatorLink())
    savesItemMenu.append(Save_LoadMasters())
    savesItemMenu.append(File_ListMasters())
    savesItemMenu.append(Save_Stats())
    #--------------------------------------------
    savesItemMenu.append(SeparatorLink())
    savesItemMenu.append(Save_EditPCSpells())
    savesItemMenu.append(Save_ImportFace())
    savesItemMenu.append(Save_ImportNPCLevels())
    savesItemMenu.append(Save_EditCreated('ENCH'))
    savesItemMenu.append(Save_EditCreated('ALCH'))
    savesItemMenu.append(Save_EditCreated('SPEL'))
    savesItemMenu.append(Save_ReweighPotions())
    savesItemMenu.append(Save_UpdateNPCLevels())
    #--------------------------------------------
    savesItemMenu.append(SeparatorLink())
    savesItemMenu.append(Save_Unbloat())
    savesItemMenu.append(Save_RepairAbomb())
    savesItemMenu.append(Save_RepairFactions())
    #savesItemMenu.append(Save_RepairFbomb())
    savesItemMenu.append(Save_RepairHair())

def InitScreenLinks():
    """Initialize screens tab menus."""
    #--SaveList: Column Links
    screensMainMenu.append(Files_Open())
    screensMainMenu.append(SeparatorLink())
    screensMainMenu.append(Screens_NextScreenShot())

    #--ScreensList: Item Links
    screensItemMenu.append(File_Open())
    screensItemMenu.append(Screen_Rename())
    screensItemMenu.append(File_Delete())
    screensItemMenu.append(SeparatorLink())
    screensItemMenu.append(Screen_ConvertToJpg())

def InitMessageLinks():
    """Initialize messages tab menus."""
    #--SaveList: Column Links
    messagesMainMenu.append(Messages_Archive_Import())

    #--ScreensList: Item Links
    messagesItemMenu.append(Message_Delete())

def InitLinks():
    """Call other link initializers."""
    InitStatusBar()
    InitMasterLinks()
    InitModLinks()
    InitReplacerLinks()
    InitSaveLinks()
    InitScreenLinks()
    InitMessageLinks()

# Main ------------------------------------------------------------------------
if __name__ == '__main__':
    print _('Compiled')
