# GPL License and Copyright Notice ============================================
#  This file is part of Wrye Bash.
#
#  Wrye Bash is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  Wrye Bash is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with Wrye Bash; if not, write to the Free Software Foundation,
#  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#  Wrye Bash copyright (C) 2005, 2006, 2007 Wrye 
#
# =============================================================================

"""bashmon is a monitor program which handles requests from Breeze582000's OBSE
extension. Current monitor commands are focused exclusively on shapeshifting the 
player, however other commands might be useful to include.

Note: bashmon is based on Breeze582000's brzMonitor (which it replaces).
"""
# Imports ----------------------------------------------------------------------
#--Standard
import cStringIO
import string
import struct
import sys
import time
import traceback

#--Local
import bosh
from bosh import _, Path, PCFaces, intArg

#--Debugging/Info
bosh.deprintOn = True

# Utils -----------------------------------------------------------------------
class Data: 
    """Some setup and data. Mostly various paths."""
    def __init__(self):
        #--Init bosh stuff
        bosh.initDirs()
        bosh.initSettings(safeMode=True)
        bosh.oblivionIni = bosh.OblivionIni()
        bosh.oblivionIni.mtime = 0
        bosh.modInfos = bosh.ModInfos()
        bosh.saveInfos = bosh.SaveInfos() #--Create, but don't fill
        self.savesDir = None
        self.morphFacts = {}
        self.update()

    def update(self):
        """Check to see if saves directory has changed. If so, update paths."""
        ini = bosh.oblivionIni
        if not self.savesDir or ini.mtime != ini.path.getmtime():
            ini.mtime = ini.path.getmtime()
            savesDir = bosh.dirs['saveBase'].join(ini.getSetting('General','SLocalSavePath','Saves\\'))
            if savesDir != self.savesDir:
                print '\nMonitoring:',savesDir
                self.savesDir = savesDir
                self.setSignals(savesDir)

    def setSignals(self,signalDir):
        """Set paths for signal files."""
        self.ping        = signalDir.join('.ping')
        self.pong        = signalDir.join('.pong')
        self.request     = signalDir.join('.request')
        self.completed   = signalDir.join('.completed')
        self.failed      = signalDir.join('.failed')
        self.shapeIsMale = signalDir.join('.shapeIsMale')
        #--Cleanup signal garbage
        self.request.remove()
        self.completed.remove()
        self.failed.remove()
        self.shapeIsMale.remove()

#--Data singleton
data = None

def touch(path):
    """Like unix 'touch' command. Creates a file with current date/time."""
    if path.exists():
        path.setmtime(time.time())
    else:
        tmpPath = path + '.touchTemp'
        tmpPath.open('w').close()
        tmpPath.rename(path)

def printFace(face):
    """Print data on face for debugging."""
    print face.pcName
    for attr in ('iclass','baseSpell','fatigue'):
        print ' ',attr,getattr(face,attr)
    for entry in face.factions:
        print ' %08X %2i' % entry
    print

# Monitor Commands ------------------------------------------------------------
monitorCommands = {}
def monitorCommand(func):
    """Add a function to monitor commands."""
    monitorCommands[func.__name__] = func
    return func

@monitorCommand
def ripAppearance(srcName,dstName,srcForm='player',dstForm='player',flags=`0x2|0x4|0x8|0x10`):
    """Rips a face from one save game and pastes it into another."""
    flags = intArg(flags)
    #--Get source face
    srcDir, srcName = data.savesDir.join(srcName).split()
    srcInfo = bosh.SaveInfo(srcDir,srcName)
    srcFile = bosh.SaveFile(srcInfo)
    srcFile.load()
    if srcForm == 'player':
        face = PCFaces.save_getPlayerFace(srcFile)
    else:
        srcForm = intArg(srcForm)
        face = PCFaces.save_getCreatedFace(srcFile,srcForm)
    #printFace(face)
    #--Set destination face
    if srcName == dstName:
        dstFile = srcFile
    else:
        dstDir, dstName = data.savesDir.join(dstName).split()
        dstInfo = bosh.SaveInfo(dstDir,dstName)
        dstFile = bosh.SaveFile(dstInfo)
        dstFile.load()
    if dstForm == 'player':
        PCFaces.save_setPlayerFace(dstFile,face,flags) #--Race, gender, hair, eyes
    else:
        dstForm = intArg(dstForm)
        PCFaces.save_setCreatedFace(dstFile,dstForm,face)
        if not face.gender: touch(data.shapeIsMale)
    dstFile.safeSave()
    #--Done
    print face.pcName,'...',
    touch(data.completed)

@monitorCommand
def swapPlayer(saveName,oldForm,newForm,flags=0x1|0x2|0x4|0x8|0x10|0x20|0x40):
    """Swaps the player between old and new forms. 
    Archives player's current form/stats to oldForm, then changes player into new form."""
    oldForm = intArg(oldForm)
    newForm = intArg(newForm)
    flags = intArg(flags)
    #--Open Save file
    saveDir, saveName = data.savesDir.join(saveName).split()
    saveInfo = bosh.SaveInfo(saveDir,saveName)
    saveFile = bosh.SaveFile(saveInfo)
    saveFile.load()
    #--player >> oldForm
    oldFace = PCFaces.save_getPlayerFace(saveFile)
    PCFaces.save_setCreatedFace(saveFile,oldForm,oldFace)
    if not oldFace.gender: touch(data.shapeIsMale)
    #--newForm >> player
    newFace = PCFaces.save_getCreatedFace(saveFile,newForm)
    PCFaces.save_setPlayerFace(saveFile,newFace,flags) #,morphFacts) 
    #--Checking
    #printFace(oldFace)
    #printFace(PCFaces.save_getCreatedFace(saveFile,oldForm))
    #printFace(newFace)
    #printFace(PCFaces.save_getPlayerFace(saveFile))
    #--Save and done
    saveFile.safeSave()
    print '%s >> %s...' % (oldFace.pcName,newFace.pcName),
    touch(data.completed)

# Monitor ---------------------------------------------------------------------
header = """== STARTING BASHMON ==
  bashmon is a monitor program which handles requests from Breeze582000's OBSE
  extension. Currently (Dec. 2007), the monitor is focused exclusively on 
  shapeshifting the  player, hence it is only useful in combination with 
  Breeze's Seducers/Succubi mod and Wrye's Morph mod.

  To stop the monitor, press Ctrl-c or close the command shell window."""

def monitor(sleepSeconds=0.25):
    """Monitor for requests to rip appearance, etc."""
    global data
    print header
    data = Data()
    running = True
    while running:
        try:
            time.sleep(sleepSeconds)
            data.update()
            if data.ping.exists(): 
                #print 'pingPong'
                data.ping.replace(data.pong)
            if not data.request.exists(): 
                continue
        except KeyboardInterrupt:
            print time.strftime('\n%H:%M:%S',time.localtime()),_("Bashmon stopped.")
            running = False
            continue
        #--Get request
        ins = data.request.open('r')
        line = ins.readline().strip()
        args = string.split(line)
        ins.close()
        data.request.remove()
        #--Handle request
        command,args = args[0],args[1:]
        print time.strftime('%H:%M:%S',time.localtime()), '='*60
        print line
        command = monitorCommands[command]
        try:
            command(*args)
            print '  Completed\n'
        except:
            print '  Failed\n'
            touch(data.failed)
            traceback.print_exc()
            running = False
    raw_input('\nSTOPPED: Press any key to exit.')

# Main -------------------------------------------------------------------------
if __name__ == '__main__':
    monitor(0.25) #--Call monitor with specified sleep interval
