(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Sat Jun 27 15:09:07 PDT 1992 by muller                   *)
(*      modified on Wed Jan 15 19:46:53 PST 1992 by meehan                   *)

INTERFACE RdUtils;

IMPORT IOFailure, Range, Rd, Thread;

PROCEDURE Find (rd        : Rd.T;
                pattern   : TEXT;
                ignoreCase: BOOLEAN  := FALSE;
                start     : CARDINAL := 0;
                length    : CARDINAL := LAST (CARDINAL)):
  [-1 .. LAST (CARDINAL)]
  RAISES {Range.Error, Rd.Failure, Thread.Alerted};
(* Finds the first occurrence of pattern, reading forward from the current
   position of rd. If no match is found, Find returns -1 and leaves rd
   positioned at the end. If Range.Error, Rd.Failure, or Rd.Error is raised
   while reading characters from rd, the exception propagates through to
   the caller of Find and the position of rd is undefined. If a match is
   found, Find returns the index of the first character of the match and
   leaves rd positioned to read the character following the match. A null
   pattern causes an instant match, with no advancement of rd. If
   ignoreCase is TRUE, Find treats lower case characters as upper case in
   both rd and pattern. *)

PROCEDURE FindChars (         rd     : Rd.T;
                     READONLY pattern: ARRAY OF CHAR;
                              start  : CARDINAL        := 0;
                              length : CARDINAL        := LAST (CARDINAL);
                     ignoreCase: BOOLEAN := FALSE): [-1 .. LAST (CARDINAL)]
  RAISES {Range.Error, Rd.Failure, Thread.Alerted};
(* = Find(rd, Text.FromSub(pattern, start, length), ignoreCase). *)

PROCEDURE FindChar (rd: Rd.T; pattern: CHAR; ignoreCase: BOOLEAN := FALSE):
  [-1 .. LAST (CARDINAL)] RAISES {Rd.Failure, Thread.Alerted};
(* = Find(rd, Text.FromChar(pattern), ignoreCase). *)

CONST
  IOFailureKind_names = ARRAY IOFailure.Kind OF
                          TEXT {"IOFailure: open failed",
                                "IOFailure: close failed",
                                "IOFailure: fcntl failed",
                                "IOFailure: read failed",
                                "IOFailure: write failed",
                                "IOFailure: lseek failed",
                                "IOFailure: fstat failed"};

PROCEDURE FailureText (f: REFANY): TEXT;
(* Rd.i3 says: "EXCEPTION Failure (REFANY)".

   FailureText returns a text describing the failure.  The only type
   for which an enumeration is defined (that I know about) is
   "IOFailure.T", which is a REF to the enumeration type
   "IOFailure.Kind".  Therefore, If "f" is an "IOFailure.T", then a text
   corresponding to the "IOFailure.Kind" will be returned (see
   IOFailureKind_names).  Concatenated to this string will be the
   current errno and its corresponding message (see strerrror(3)).  Of
   course, the CURRENT value of errno may be completely irrelevant when
   FailureText is called, but the intention is that FailureText would
   be used in exception-handlers, while errno is still 'fresh'. E.g.,
     "IOFailure: open failed (errno = 24: Too many open files)"

   If f is of type TEXT, it will be returned.  Otherwise, the text
   "unknown error" will be returned.
   
   *)

END RdUtils.

