(* Copyright (C) 1992, Digital Equipment Corporation          *)
(* All rights reserved.                                       *)
(* See the file COPYRIGHT for a full description.             *)
(* Created by stolfi on Mon Mar 28 20:52:56 1988              *)
(* Last modified on Wed Jun 17 11:45:38 PDT 1992 by stolfi    *)
(*      modified on Thu Jun 11 19:08:30 1992 by mhb           *)
(*      modified on Fri Mar 13 14:53:31 PST 1992 by muller    *)


(* An "RGB.T" is a color represented as an RGB triple.  
   This interface defines the standard representation of colors 
   and color mixing operations, based on a linear RGB model with
   coordinates in "[0.0..1.0]". *)

INTERFACE RGB;

IMPORT Color;

TYPE T = Color.T; (* ARRAY [0..2] OF REAL *)

(* An "RGB.T" value describes a color as a mixture of the three
   color TV primaries (Red, Green and Blue), in a linear scale
   (proportional to luminous power), where 0.0 = black and 1.0 =
   maximum screen intensity.

   The set of all colors with RGB coordinates in the range 0.0--1.0
   is the {\em unit RGB cube}.  RGB triples outside this set
   cannot be displayed on typical color monitors, but are still
   legal as far as this interface is concerned, make perfect
   physical sense, and are useful in some color computations. *)

CONST
  Black   = T{0.0, 0.0, 0.0};
  Red     = T{1.0, 0.0, 0.0};
  Green   = T{0.0, 1.0, 0.0};
  Blue    = T{0.0, 0.0, 1.0};
  Cyan    = T{0.0, 1.0, 1.0};
  Magenta = T{1.0, 0.0, 1.0};
  Yellow  = T{1.0, 1.0, 0.0};
  White   = T{1.0, 1.0, 1.0};
(* The vertices of the unit RGB cube *)

  Primary = ARRAY [0 .. 2] OF T{Red, Green, Blue};

PROCEDURE R (READONLY c: T): REAL;
PROCEDURE G (READONLY c: T): REAL;
PROCEDURE B (READONLY c: T): REAL;
(* Red, green, and blue components of "c". *)

PROCEDURE New (value: T := Undefined): REF T;
(* Allocate a new "RGB.T" from the heap and initialize it with
   the given value. *)

PROCEDURE NewArray (size: CARDINAL; value: T := Undefined):
  REF ARRAY OF T;
(* Allocate a new "ARRAY OF RGB.T" from the heap and initialize
   all its elements with the given value. *)

PROCEDURE Brightness (READONLY c: T): REAL;
(* Return the brightness (a.k.a. value, intensity, or luminance) of
   "c" in a linear scale.  Equal to the CIE 1931 Y coordinate,
   scaled so that "Brightness(White) = 1.0". *)

PROCEDURE Gray (y: REAL): T;
PROCEDURE Grey (y: REAL): T;
(* Return a grey color with brightness "y". *)

PROCEDURE Add (READONLY a, b: T): T;
PROCEDURE Sub (READONLY a, b: T): T;
(* Return the component-wise sum and difference of "a" and "b". *)

PROCEDURE Scale (alpha: REAL; READONLY a: T): T;
(* Returns color "a" with all components multiplied by "alpha". *)

PROCEDURE Weigh (READONLY a, b: T): T;
(* Returns the component-wise product:
| (a.r * b.r, a.g * b.g, a.b * b.b)
*)

PROCEDURE Mix (a: T; alpha: REAL; b: T; beta: REAL): T;
(* Returns a linear mix of colors "a" and "b" in the proportions "alpha"
   and "beta".  If "alpha + beta" is less than 1, the balance is
   completed with black. *)

PROCEDURE Interpolate (a: T; alpha: REAL; b: T; beta: REAL): T;
(* Returns a color that divides the segment "ab" in the ratio
   "alpha:beta".  Equivalent to
| Mix (a, beta / (alpha+beta), b, alpha / (alpha+beta)).
   *)

PROCEDURE Clip (a: T): T;
(* ``Clips'' an RGB triple against the unit color cube.  More
   precisely, returns the color nearest to the given one that has
   all three coordinates in 0.0--1.0. *)

PROCEDURE MaxStep (READONLY p, d: T): REAL;
(* Returns the maximum positive multiple of "d" that can be added
   to "p" without leaving the unit cube.  If "|d|" is too small,
   returns some large value.  If "p" is not inside the unit cube,
   the result is undefined. *)


(* The remainder of the interface defines the "Default" and
   "Undefined" triples, and a procedure for safely
   LOOPHOLE'ing a triple into an array of bytes.  As
   explained in the "Color" interface, "Default" is useful as a
   default value for generic modules, and "Undefined" is useful
   for applications that need a special color value meaning
   ``undefined'', ``vacant'', ``don't care'', and so on.  The
   loophole function, "ToBytes", is useful for marshalling and
   pickling color triples. *)

CONST 
  Default = Color.Default;
  Undefined = Color.Undefined;

TYPE 
  Bytes = Color.Bytes;

CONST (*PROCEDURE*) 
  ToBytes = Color.ToBytes;

END RGB.


