MODULE Modules EXPORTS Modules;

IMPORT Text, Rd, System, OList, ProgFileIO;

IMPORT Wr, Stdio, Thread;
PROCEDURE Log(z: TEXT) =
    <*FATAL Wr.Failure, Thread.Alerted*>
    BEGIN IF verbose THEN
	Wr.PutText(Stdio.stderr, z&"\n"); Wr.Flush(Stdio.stderr); END;
    END Log;

REVEAL
    T = Public BRANDED OBJECT
	    mods: OList.T;              (* Importliste *)
	    checked: OList.T;   (* Duftmarke: was haben wir schon *)
	    implList: OList.T;  (* schon untersuchte Implementations,
			   werden nur gefhrt, um oftmaliges
			   lesen von Implementation-Files zu
			   vermeiden (siehe SearchImpl.()!) *)
	    currentMod: ModList := NIL;
	    publicDirs: OList.T;        (* Dirs fr public interfaces*)
	    sourceDirs: OList.T;        (* Dirs fr .m3 und .i3 files*)
	OVERRIDES
	    getModules := GetModules;
	    beginScann := BeginScann;
	    next := Next;
	    publicInterface := PublicInterface;
	    uptodate := Uptodate;
	END;

TYPE
    ModList = OList.ElmT OBJECT
		  uptodate := TRUE;
		  public := FALSE;
	      END;
    ImplList = OList.ElmT OBJECT
		  exports: TEXT;
	      END;


(* Fr Dateinamen der Form: Module.m3 ****************)

PROCEDURE Basename (in: TEXT): TEXT = (* erster Teil (vor dem Punkt): *)
    VAR slashpos := Text.FindCharR(in, '/');
    BEGIN
	IF slashpos >= 0 THEN
	    in := Text.Sub(in, slashpos + 1, Text.Length(in) - slashpos);
	END;                     (*IF*)
	RETURN Text.Sub(in, 0, Text.FindCharR(in, '.'));
    END Basename;

PROCEDURE Extname (in: TEXT): TEXT = (* zweiter Teil (nach dem Punkt): *)
    VAR dotpos := Text.FindCharR(in, '.');
    BEGIN
	RETURN Text.Sub(in, dotpos + 1, Text.Length(in) - dotpos);
    END Extname;

(****************************************************************************)
(**                           Dateiensuchen                                      **)
(****************************************************************************)

PROCEDURE InitDirList (dirs: OList.T; path: TEXT) =
    VAR i: INTEGER;
    BEGIN
	i := Text.FindChar(path, ':');
	WHILE i > 0 DO
	    dirs.put(NEW(OList.ElmT, name := Text.Sub(path, 0, i)));
	    path := Text.Sub(path, i + 1, Text.Length(path) - i - 1);
	    i := Text.FindChar(path, ':');
	END;                     (*WHILE*)
	dirs.put(NEW(OList.ElmT, name := path));
    END InitDirList;

PROCEDURE IsImplementationFor (implFilename, interface: TEXT): BOOLEAN
    RAISES {Rd.Failure, ProgFileIO.SyntaxError} =
    VAR
	datei := NEW(ProgFileIO.T);
	word, moduleName: TEXT;
    BEGIN
	datei.open(implFilename);
	TRY
	    REPEAT               (* -> MODULE *)
		word := datei.getWord();
	    UNTIL Text.Equal(word, "MODULE");

	    REPEAT               (* -> Identifier *)
		word := datei.getWord();
	    UNTIL datei.tokenType() = ProgFileIO.TokenTypes.Identifier;
	    moduleName := word;

	    REPEAT               (* -> EXPORTS | ; *)
		word := datei.getWord();
	    UNTIL Text.Equal(word, "EXPORTS") OR Text.Equal(word, ";")
		OR Text.Equal(word, "END");

	    (* Gibt es Exportliste? *)
	    IF NOT Text.Equal(word, "EXPORTS") THEN
		datei.close();
		RETURN Text.Equal(moduleName, interface);
	    ELSE
		REPEAT           (* ber alle Export-Identifier *)
		    word := datei.getWord();
		    WHILE datei.tokenType()
			      # ProgFileIO.TokenTypes.Identifier
			      AND NOT Text.Equal(word, ";") DO
			word := datei.getWord();
		    END;
		    IF Text.Equal(word, interface) THEN
			datei.close();
			RETURN TRUE;
		    END;
		UNTIL Text.Equal(word, ";");

		(* Ende der Exports-Liste (;), nichts gefunden *)
		datei.close();
		RETURN FALSE;
	    END;

	EXCEPT
	  Rd.EndOfFile =>
	      RAISE ProgFileIO.SyntaxError(
			implFilename & " is no Modula-3 implementation!");
	END;
    END IsImplementationFor;


PROCEDURE SearchImplementations (modules: T;
							 result: OList.T;
							 interfaceName: TEXT): BOOLEAN
    RAISES {NotFound, ProgFileIO.SyntaxError, System.Error, Rd.Failure} =

    (* nach Implementierungen eines Interfaces suchen:
     |Abhngig vom 'modules.searchMode' werden entweder Implementierungen,
       die wie "Interfacename_xx.m3" heien gesucht und NotFound
       raisen, wenn es sie nicht gibt, oder (searchMode=TRUE) alle .m3
       Dateien im Pfad 'modules.publicDirs' untersucht, ob sie interfaceName
       exportieren.
     |D.h., ist searchMode FALSE, dann mu es eine Implementierung
       geben, oder es gibt eine Exception. Ist searchMode TRUE, dann
       wird mit FALSE retourniert, wenn keine Implementierung gefunden
       werden konnte.
     |In modules.implList werden schon untersuchte Implementations
       gespeichert, um zu viele offene Dateien zu vermeiden. *)

    VAR
	dir: System.Dir;
	path: OList.ElmT;
	filename: TEXT;
	treffer := FALSE;
    BEGIN
	(* Erster Versuch: Impl.  heit wie Interf. *)
	WITH searchdirs = modules.sourceDirs DO
	    filename := interfaceName & ".m3";
	    IF ExpandFilename(searchdirs, filename) THEN
		IF IsImplementationFor(filename, interfaceName) THEN
		    treffer := TRUE;
		    result.put(NEW(ModList, name := filename));
		ELSIF NOT modules.searchMode THEN
		    RAISE NotFound(
			      filename
				  & " does not contain an implenentation for "
				  & interfaceName);
		END;
	    END;

	    (* zweiter Versuch: Impl.  heien Interface_xx.m3 oder
	       searchMode *)
	    IF NOT treffer THEN
		searchdirs.first();
		path := searchdirs.getnext();
		WHILE path # NIL DO
		    IF modules.searchMode THEN
			dir := System.OpenDir(path.name, ".*\\.m3$");
		    ELSE
			dir := System.OpenDir(
				   path.name, interfaceName & "_.*\\.m3$");
		    END;
		    filename := System.GetnextDirentry(dir);
		    WHILE filename # NIL DO
			IF IsImplementationFor(
			       path.name & "/" & filename, interfaceName) THEN
			    treffer := TRUE;
			    result.put(NEW(ModList, name := filename));
			ELSIF NOT modules.searchMode THEN
			    RAISE
				NotFound("module " & path.name & "/" & filename
					& " does not contain an implenentation for "
					& interfaceName);
			END;
			filename := System.GetnextDirentry(dir);
		    END;
		    System.CloseDir(dir);
		    path := searchdirs.getnext();
		END;
	    END;
	END;

	IF NOT treffer AND NOT modules.searchMode THEN
	    RAISE NotFound("no implementation for interface "
			       & interfaceName & " found");
	END;
	RETURN treffer;
    END SearchImplementations;


PROCEDURE ExpandFilename (dirs: OList.T; VAR name: TEXT): BOOLEAN =
    VAR path: OList.ElmT;
    BEGIN
	IF System.FileIsReadable(name) THEN
	    RETURN TRUE;
	ELSE
	    dirs.first();
	    path := dirs.getnext();
	    WHILE path # NIL DO
		IF System.FileIsReadable(path.name & "/" & name) THEN
		    IF Text.Equal(path.name, ".")
			   OR Text.Equal(path.name, "./") THEN
			RETURN TRUE;
		    ELSIF Text.GetChar(
			      path.name, Text.Length(path.name) - 1) = '/' THEN
			name := path.name & name;
			RETURN TRUE;
		    ELSE
			name := path.name & "/" & name;
			RETURN TRUE
		    END;
		END;
		path := dirs.getnext();
	    END;

	    RETURN FALSE;        (* nichts gefunden *)
	END;
    END ExpandFilename;


PROCEDURE IsInDirlist (dirs: OList.T; name: TEXT): BOOLEAN =
    VAR path: OList.ElmT;
    BEGIN
	dirs.first();
	path := dirs.getnext();
	WHILE path # NIL DO
	    Log("...searching "&path.name & "/" & name);
	    IF System.FileIsReadable(path.name & "/" & name) THEN
		RETURN TRUE
	    END;
	    path := dirs.getnext();
	END;

	RETURN FALSE;
    END IsInDirlist;

(****************************************************************************)
(**                       G e t  M o d u l e s                               **)
(****************************************************************************)

PROCEDURE GetModules (self: T; RootDateiname: TEXT)
    RAISES {NotFound, Rd.Failure, Rd.EndOfFile, ProgFileIO.SyntaxError,
	    System.Error} =

    PROCEDURE NoMoreImports (word: TEXT): BOOLEAN =
	BEGIN
	    RETURN Text.Equal(word, "VAR") OR Text.Equal(word, "CONST")
		       OR Text.Equal(word, "EXCEPTION")
		       OR Text.Equal(word, "REVEAL")
		       OR Text.Equal(word, "PROCEDURE")
		       OR Text.Equal(word, "TYPE")
		       OR Text.Equal(word, "BEGIN")
		       OR Text.Equal(word, "END");
	END NoMoreImports;

    PROCEDURE ReadMod (name: TEXT)
	RAISES {NotFound, System.Error, Rd.Failure, Rd.EndOfFile,
		ProgFileIO.SyntaxError} =

	VAR
	    datei := NEW(ProgFileIO.T);
	    word: TEXT;
	    implList := NEW(OList.T);
	    implElm: OList.ElmT;

	BEGIN
	    IF NOT ExpandFilename(self.sourceDirs, name) THEN
		IF IsInDirlist(self.publicDirs, name) THEN
		    self.mods.put(NEW(ModList, name := name, public := TRUE));
		    self.checked.put(NEW(OList.ElmT, name := name));
		    Log(">"&name&" is public");
		    RETURN;
		ELSE
		    RAISE NotFound("file not found: " & name);
		END;
	    END;

	    self.mods.put(NEW(ModList, name := name));
	    (*Duftmarke:*)
	    self.checked.put(
		NEW(OList.ElmT, name := Basename(name) & "." & Extname(name)));
	    datei.open(name);

	    (* Wenn Interfacefile Implementations suchen *)
	    IF Text.Equal(Extname(name), "i3") THEN
		IF SearchImplementations(self, implList, Basename(name)) THEN
		    implList.first();
		    implElm := implList.getnext();
		    WHILE implElm # NIL DO
			IF self.checked.get(Basename(implElm.name) & "."
					   & Extname(implElm.name)) = NIL THEN
			    Log("ReadMod("&implElm.name&")");
			    ReadMod(implElm.name);
			END;
			implElm := implList.getnext();
		    END;
		ELSE Log(">no implementation for "&name);
		END;
	    END;

	    REPEAT
		word := datei.getWord();
		IF Text.Equal(word, "IMPORT") OR Text.Equal(word, "FROM") THEN
		    REPEAT
			WHILE datei.tokenType()
				  # ProgFileIO.TokenTypes.Identifier DO
			    word := datei.getWord();
			END;
			IF self.checked.get(word & ".i3") = NIL THEN
			    Log("ReadMod("&word & ".i3)");
			    ReadMod(word & ".i3");
			END;
			word := datei.getWord();
			WHILE NOT Text.Equal(word, ",")
				  AND NOT Text.Equal(word, ";")
				  AND NOT Text.Equal(word, "IMPORT") DO
			    word := datei.getWord();
			END;
		    UNTIL
			Text.Equal(word, ";") OR Text.Equal(word, "IMPORT");
		END;
	    UNTIL NoMoreImports(word);
	    datei.close();
	END ReadMod;

    BEGIN
	(* Interne Directory und Modullisten initialisieren: *)
	self.mods :=
	    NEW(OList.T, insertMode := OList.Mode.FirstIn_FirstOut);
	self.checked := NEW(OList.T);
	self.publicDirs := NEW(OList.T);
	self.sourceDirs := NEW(OList.T);
	InitDirList(self.publicDirs, self.publicPath);
	InitDirList(self.sourceDirs, self.sourcePath);

	ReadMod(RootDateiname & ".m3");
    END GetModules;


PROCEDURE BeginScann (self: T) =
    BEGIN
	self.mods.first();
	self.currentMod := NIL;
    END BeginScann;

PROCEDURE Next (self: T): TEXT =
    BEGIN
	self.currentMod := self.mods.getnext();
	IF self.currentMod = NIL THEN
	    RETURN NIL;
	ELSE
	    RETURN self.currentMod.name;
	END;                     (*IF*)
    END Next;

PROCEDURE PublicInterface (self: T): BOOLEAN =
    BEGIN
	RETURN self.currentMod.public;
    END PublicInterface;

PROCEDURE Uptodate (self: T): BOOLEAN =
    BEGIN
	RETURN self.currentMod.uptodate;
    END Uptodate;

BEGIN
END Modules.
