(* Copyright (C) 1990, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Thu Jul 30 12:08:37 PDT 1992 by muller     *)
(*      modified on Thu Mar  5 13:33:59 PST 1992 by kalsow     *)

UNSAFE MODULE RTStack;

IMPORT Unix, Umman, RTMisc;

VAR page_bytes : CARDINAL := 0;
VAR stack_slop : CARDINAL;

PROCEDURE Init () =
  BEGIN
    page_bytes := Unix.getpagesize ();
    stack_slop := 2 * (page_bytes DIV BYTESIZE (INTEGER));
  END Init;

PROCEDURE New (size: INTEGER): T =
  VAR t: T;  i: INTEGER;  start: ADDRESS;
  BEGIN
    IF (page_bytes = 0) THEN Init () END;

    (* allocate enough so that we're guaranteed to get a full, aligned page *)
    t := NEW (T, size + stack_slop);

    (* find the aligned page and unmap it *)
    (* the guard page is marked read-only so that when Thread.InitContext
       writes the "seal" on the non-protected pages it doesn't generate
       a NIL fault. *)
    start := RTMisc.Align (ADR (t[0]), page_bytes);
    i := Umman.mprotect (start, page_bytes, Umman.PROT_READ);
    <* ASSERT i = 0 *>

    (* finally, return the new stack *)
    RETURN t;
  END New;

PROCEDURE Dispose (t: T) =
  VAR i: INTEGER;  start := RTMisc.Align (ADR (t[0]), page_bytes);
  BEGIN
    (* find the aligned page and re-map it *)
    i := Umman.mprotect (start, page_bytes,
                         Umman.PROT_READ + Umman.PROT_WRITE);
    <* ASSERT i = 0 *>

    (* and finally, free the storage *)
    DISPOSE (t);
  END Dispose;

PROCEDURE GetBounds (t: T;  VAR(*OUT*) first, last: ADDRESS) =
  VAR start := RTMisc.Align (ADR (t[0]), page_bytes);
  BEGIN
    first := start + page_bytes;
    last  := ADR (t [0]) + NUMBER (t^) * ADRSIZE (t[0]);
  END GetBounds;

BEGIN
END RTStack.
