(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Last modified on Tue Feb 11 16:18:11 PST 1992 by muller                   *)

UNSAFE MODULE Dir;

IMPORT Text, List, Pathname, Ctypes, M3toC, Unix, Udir, Uerror;

PROCEDURE Delete (name: Text.T) RAISES {CannotAccess, Error} =
  VAR status: Ctypes.int; nm := M3toC.CopyTtoS (name);
  BEGIN
    status := Unix.unlink (nm);
    M3toC.FreeS (nm);
    IF status = -1 THEN
      RAISE Error; END;
  END Delete;

PROCEDURE Rename (old, new: Text.T) RAISES {CannotAccess, Error} =
  VAR status: Ctypes.int; o := M3toC.CopyTtoS (old); n := M3toC.CopyTtoS (new);
  BEGIN
    status := Unix.rename (o, n);
    M3toC.FreeS (o);
    M3toC.FreeS (n);
    IF status = -1 THEN
      RAISE Error; END;
  END Rename;

PROCEDURE Contents (name: Text.T): List.T RAISES {CannotAccess, Error} =
  VAR nm := M3toC.CopyTtoS (name);
      dir := Udir.opendir (nm);
      res : List.T := NIL;
  BEGIN
    M3toC.FreeS (nm);
    IF dir = NIL THEN
      RAISE Error;
    ELSE
      LOOP
        WITH file = Udir.readdir (dir) DO
          IF file = NIL THEN
            IF Uerror.errno = 0 THEN
              EXIT;
            ELSE
              RAISE Error; END;
          ELSE
            res := NEW (List.T,
                        first := M3toC.CopyStoT (LOOPHOLE (ADR (file.d_name),
                                                           Ctypes.char_star)),
                        tail := res); END; END; END;
      IF Udir.closedir (dir) = 0 THEN
        RETURN res;
      ELSE
        RAISE Error; END; END;
  END Contents;

PROCEDURE CreateAlias (name, alias: Text.T) RAISES {CannotAccess, Error} =
  VAR nm := M3toC.CopyTtoS (name); al := M3toC.CopyTtoS (alias);
      status: Ctypes.int;
  BEGIN
    status := Unix.symlink (nm, al);
    M3toC.FreeS (nm);
    M3toC.FreeS (al);
    IF status # 0 THEN
      RAISE Error; END;
  END CreateAlias;

PROCEDURE FindFile (READONLY searchPath: ARRAY OF Text.T; name: Text.T): Text.T
     RAISES {CannotAccess, Error} =
  BEGIN
    WITH nm = Pathname.Name (name), ext = Pathname.Ext (name) DO
      FOR i := 0 TO LAST (searchPath) DO
        WITH t = Pathname.Make (searchPath [i], nm, ext),
             n = M3toC.CopyTtoS (t),
             status = Unix.access (n, Unix.F_OK) DO
          M3toC.FreeS (n);
          IF status = 0 THEN
            RETURN t; END; END; END; END;
    RAISE CannotAccess;
  END FindFile;

BEGIN
END Dir.
