(* Copyright (C) 1989, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Modula-2+ version created by Benli Pierce and John Ellis.   *)
(* Ported to Modula-3 by J.Stolfi on May 1990.                 *)
(* Last modified on Mon Dec 16 22:08:41 PST 1991 by meehan     *)
(*      modified on Wed Nov 20 17:51:44 PST 1991 by stolfi     *)

INTERFACE FWr;

(*
  Formatted Writers (writers with automatic pretty printing)

  A formatted writer (an FWr.T) is a writer (Wr.T) in which the client
  can mark the start and end of logical objects and specify desirable
  places to break the objects into lines.  Formatted writers are a
  basic tool for building pretty printers.
  
  A FWr.T is simply a writer to a Formatter.T that in turn writes 
  to the underlying Wr.T.  See Formatter.i3 for more information.

  Index: formatted writers; writers, formatted

*)

IMPORT Wr, Formatter, Thread;

TYPE
  T <: Wr.T;

PROCEDURE New(
    underlyingWr: Wr.T; 
    lineWidth: CARDINAL := 75
  ): Wr.T;
  (*
    Returns a new FWr.T that feeds its formatted output into
    the given Wr.T.
    
    The writer will attempt to keep all lines shorter than
    /lineWidth/ characters.
    
    An FWr.T has internal buffering, because the decision to insert
    a newline at a break depends on the characters that follow.
    To force all buffered data out to the undelying writer,
    use FWr.Flush below (Wr.Flush is NOT enough).

    You MUST call FWr.Close when you are finished using the writer,
    in order to deallocate untraced storage (a bug in Thread).
    
    *)

PROCEDURE Close (fwr: T) RAISES {Wr.Failure, Thread.Alerted};
(* This flushes the writer and releases internal resources. *)
  
PROCEDURE UnderlyingWr (fwr: T): Wr.T;
  (*
    Returns the underlying writer of an FWr. *)

(**********************************************************)
(* FORMATTING OPERATIONS                                  *)
(**********************************************************)

(*
  The following procedures basically invoke the corresponding procedures 
  on the undelying Formatter.T: *)

PROCEDURE Group(fwr: T) 
  RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE Begin(fwr: T; offset := 0; width: CARDINAL := LAST (CARDINAL))
  RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE End(fwr: T) 
  RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE Flush(fwr: T) 
  RAISES {Wr.Failure, Thread.Alerted};
  (*
    Closes all unmatched /Begin/, /Group/, and /Align/ constructs, sends
    all pending output through the underlying Formatter.T into the
    underlying writer, and flushes the latter.  
    
    Note the difference between FWr.Flush(fwr) and Wr.Flush(fwr) *)

TYPE BreakType = Formatter.BreakType (* = {NonOptimal, OptimalBreak, OptimalNoBreak} *);

PROCEDURE Break(fwr: T; offset := 0; type := BreakType.OptimalBreak; freshLine := TRUE)
  RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE UnitedBreak(fwr: T;  offset := 0;  freshLine := TRUE)
  RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE PartialBreak (fwr: T;  offset := 0;  freshLine := TRUE)
  RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE NewLine (fwr: T;  offset := 0;  freshLine := TRUE)
  RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE Align (
    fwr: T;
    columns: CARDINAL;
    tryOneLine: BOOLEAN := TRUE;
    offset: INTEGER := 0;
    alignPred: Formatter.AlignPred := NIL;
  ) RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE NoAlign (fwr: T) 
  RAISES {Wr.Failure, Thread.Alerted};

PROCEDURE Col (fwr: T; column: CARDINAL; relative := FALSE; space := 0)
  RAISES {Wr.Failure, Thread.Alerted};
    
END FWr.
