(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created 1990-01-17 by Jorge Stolfi                          *)
(*      inspired on an original novel by John DeTreville       *)
(* Last modified on Tue Feb 11 17:11:59 PST 1992 by muller     *)
(*      modified on Wed Oct 17  1:10:59 PDT 1990 by stolfi     *)

INTERFACE LongRealTime;

(* Time values as LONGREALs.

  This interface implements time values as double-precision floating
  point numbers, which makes it possible to write time values directly as
  plain numeric literals, and operate on time values with the standard
  arithmetic and comparison operations.
  
  Absolute times are measured in microseconds from the same epoch used
  by Time.def, namely 1970 Jan 01 00:00:00.000000 GMT.  Time intervals
  are measured in microseconds.
  
  LongRealTime is primarily intended for applications such as 
  performance measurement and animation, where time is best thought
  of as a continuous physical quantity, and time values as 
  approximate numbers.

  Still, converting a Time.T value into a LongRealTime.T will normally
  involve no rounding or loss of precision.  Therefore, LongRealTime.T
  values should be usable also as timestamps, and other similar
  applications that depend on time being discrete and time values being
  exact.  (This feature hasn't been tested, however; use at your own
  risk.)
  
  Index: times; dates; threads, sleeping; statistics; 
         performance

*)

IMPORT Time, Thread;

TYPE
  T = LONGREAL; (* In microseconds *)

(**********************************************************)
(* Time units:                                            *)
(**********************************************************)

CONST
    Picosecond =       0.000001d+00;
    Nanosecond =       0.001d+00;
    Microsecond =      1.0d+00;
    Millisecond =   1000.0d+00;
    Second =     1000000.0d+00;
    
    Minute =   60.0d0 * Second;
    Hour =     60.0d0 * Minute;
    Day =        24.0d0 * Hour;
    Week =         7.0d0 * Day;
    
    (* Month, year, etc. are ambiguous --- leave out for now. *)

PROCEDURE Now (): T;
(*
  Current clock reading, i.e. elapsed time in microseconds from 1970
  Jan 01 00:00:00.000000 GMT.  
  
  CAUTION:  The accuracy of the result is only as good as that of
  Time.Now(), typically few milliseconds.*)

PROCEDURE Pause (rt: T) RAISES {Thread.Alerted};

PROCEDURE PauseUntil (rt: T) RAISES {Thread.Alerted};
(* 
  Pause and PauseUntil are no-ops if the requested interval is zero or
  negative. *)

(**********************************************************)
(* Conversion between LONGREAL times and Time.T values:   *)
(**********************************************************)

PROCEDURE FromTime (it: Time.T): T;

PROCEDURE ToTime (rt: T): Time.T;
(*
  ToTime raises a range fault if the argument is negative or too big. *)

END LongRealTime.

