(* Gene Rollins (rollins@cs.cmu.edu)
   School of Computer Science, Carnegie-Mellon Univ., Pittsburgh, PA 15213 *)

functor SourceActionFun
 (structure Hash :HASH
  structure Hasher :HASHER
  structure Pathname :PATHNAME)  = struct

fun printSep ([] :string list) (sep :string) :unit = ()
  | printSep (a::[]) sep = print a
  | printSep (a::(rest as (b::c))) sep =
      (print a; print sep; printSep rest sep)

fun standardBinary pathname = pathname ^ ".bin"

fun importBinary pathname =
  let val (dirname, filename) = Pathname.splitDirFile pathname in
    Pathname.mergeDirFile dirname ((Pathname.stripExtension filename) ^ ".bin")
  end

fun systemBinary system pathname =
  let val (dirname, filename) = Pathname.splitDirFile pathname in
     Pathname.mergeDirFile dirname (system^"/"^filename^".bin")
  end

val sysBinary = systemBinary ".@sys"

fun withOutputFile (filename:string) (operation :outstream -> unit) =
  let val strm = open_out filename in
    (operation strm; close_out strm) handle exn => close_out strm
  end;

fun doClean (SourceGroup.Source {targetName,...} :SourceGroup.sourceInfo) =
  (System.Unsafe.SysIO.unlink targetName;
   printSep ["rm ", targetName, "\n"]) ""
     handle _ => (printSep ["% ", targetName, " could not be removed"] "")

fun doLoadSource(SourceGroup.Source{envCurrent,loadSource,...}:SourceGroup.sourceInfo)=
  if envCurrent then () else loadSource()

fun doCompile (SourceGroup.Source {envCurrent, targetCurrent, loadTarget,
                                   compileSource,...}:SourceGroup.sourceInfo) =
 if targetCurrent
   then if envCurrent then () else loadTarget ()
   else compileSource ()

fun doLoadLibrary (SourceGroup.Source {envCurrent, targetCurrent, loadSource,
                                       loadTarget,...}:SourceGroup.sourceInfo) =
  if envCurrent then ()
    else if targetCurrent then loadTarget() else loadSource()

fun doNothing (_:SourceGroup.sourceInfo) = ()

fun showUses (group:SourceGroup.group) out =
  let val pr = (outputc out)
      fun doLoader (SourceGroup.Source {sourceName,toolName,...} :SourceGroup.sourceInfo) =
        case toolName of
           "sml" => (pr "use \""; pr sourceName; pr "\";\n")
         | "lex" => (pr "use \""; pr sourceName; pr ".sml\";\n")
         | "yacc" => (pr "use \""; pr sourceName; pr ".sig\";\n";
                      pr "use \""; pr sourceName; pr ".sml\";\n")
         | _ => (pr "(* load "; pr toolName; pr " "; pr sourceName; pr " *)\n")
  in
    SourceGroup.make doLoader group
  end

fun prJustified (pr:string->unit) (lineSize:int)
                (indent:string) ([]:string list) (sep:string) = ()
  | prJustified pr (lineSize:int)(indent:string)(a::[]) sep =
      (if (lineSize+(String.size a)) >= 80 then pr "\n  " else ();
       pr a)
  | prJustified pr (lineSize:int)(indent:string)(a::(rest as (b::c))) sep =
      let val textSize = (String.size a) + (String.size sep)
          val lineLength =
                if (lineSize+textSize) >= 80
                  then (pr "\n"; pr indent; size indent) 
                  else lineSize
      in
        pr a; pr sep;
        prJustified pr (lineLength+textSize) indent rest sep
      end

fun showDependencies (group:SourceGroup.group) out =
  let val pr = (outputc out)
      fun doDepends
            (SourceGroup.Source
               {sourceName, toolName ,dependsOn,...}:SourceGroup.sourceInfo) =
        (pr sourceName; pr ":";
         prJustified pr 80 "  " dependsOn " "; pr "\n")
  in
    SourceGroup.make doDepends group
  end

fun currentTime () =
  let val getTimeOfDay :unit -> System.Timer.time =
            System.Unsafe.CInterface.c_function "timeofday"
      val System.Timer.TIME {sec,...} = getTimeOfDay ()
  in
    sec
  end

fun getwd () =
  let val pwdFile = "/tmp/SG." ^ (Integer.makestring (currentTime ()))
      val status = System.Unsafe.CInterface.system("pwd > " ^ pwdFile ^ chr(0))
      val _ = if (status = 0) then () else
                (print "? Could not determine current working directory\n";
                 raise SourceGroup.CompilingError)
      val instream = open_in pwdFile
  in
    let val line = input_line instream
        val dirname = substring (line, 0, (size line)-1)
    in
      close_in instream;
      System.Unsafe.SysIO.unlink pwdFile;
      dirname
    end
      handle e =>
        (print "? Could not determine current working directory\n";
         close_in instream;
         System.Unsafe.SysIO.unlink pwdFile;
         raise e)
  end

fun absoluteName (filename:string) = 
  if substring(filename,0,1) = "/"
    then filename
    else Pathname.clearPath (Pathname.mergeDirFile (getwd()) filename)

type actionTable = (string,SourceGroup.sourceInfo -> unit) Hash.table

fun doAction (actionTable:actionTable) (defaultAction :SourceGroup.sourceInfo->unit)
             (info as SourceGroup.Source {sourceName,...} :SourceGroup.sourceInfo) =
  case Hash.lookup actionTable (Hasher.hasher sourceName) of
     NONE => defaultAction info
   | (SOME action) => action info

fun updateAction (actionTable:actionTable) filename (action:SourceGroup.sourceInfo->unit) =
  Hash.enter actionTable (Hasher.hasher (absoluteName filename)) action

fun clearAction (actionTable:actionTable) filename = 
  Hash.remove actionTable (Hasher.hasher (absoluteName filename))

fun actionTable () = Hash.createDefault ([]:(SourceGroup.sourceInfo->unit) list)

end
