(* Copyright (C) 1989, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Created September 1989 by Bill Kalsow                       *)
(* Based on Random.def by Mark R. Brown                        *)
(* Last modified on Tue Jan 30 11:02:59 1990 by muller     *)
(*      modified on Thu Jan 25 21:30:53 PST 1990 by stolfi     *)

(*      modified on Fri Sep 29 09:33:36 1989 by kalsow         *)

INTERFACE Random;

(* The interface to a pseudo-random number generator

   A pseudo-random number generator, implemented as in Algorithm A of Knuth
   Section 3.2.2 (second edition).

   Index: random number generator *)

TYPE T <: REFANY;

CONST Default = NIL;
  (* On any call to a procedure that takes a Random.T, passing Random.Default
     causes the procedure to use a pre-initialized Random.T that is held by
     the Random implementation module. *)

PROCEDURE New (seed: INTEGER := 0): T;
(* Creates a new generator and initializes it by calling Random.Start with the
   given seed. Individual generators are not protected against concurrent
   access from different threads. See note below about rules for the seed. *)

PROCEDURE Start (self: T := Default;  seed: INTEGER := 0);
(* Initializes an existing generator using the given seed. Either use the
   default seed to get a fixed default seed, pass -1 to have this module
   choose a random seed for you, or pick a seed satisfying
   LAST(INTEGER)/10 <= seed <= 9*LAST(INTEGER)/10
   If -1 is passed, a random seed is chosen in such a way that different seeds
   will result even if Start is called many times in close proximity. *)

PROCEDURE Integer (self: T := Default): INTEGER;
(* Returns a uniformly distributed INTEGER. To get a uniformly distributed
   BOOLEAN, write ODD(Random.Integer(r)). *)

PROCEDURE Cardinal (self: T := Default): CARDINAL;
(* Returns a uniformly distributed CARDINAL. *)

PROCEDURE Subrange (self: T; first, last: CARDINAL): CARDINAL;
(* Returns a uniformly distributed CARDINAL in [first..last]. The effect is
   not defined if first > last. *)

PROCEDURE Real (self: T := Default): REAL;
(* Returns a random REAL uniformly distributed between 0.0 (inclusive) and 1.0
   (exclusive). Each bit of the fraction part (except the most significant
   one) will be 0 or 1 with equal probability. *)

PROCEDURE LongReal (self: T := Default): LONGREAL;
(* Returns a random LONGREAL uniformly distributed between 0.0 (inclusive) and
   1.0 (exclusive). Each bit of the fraction part (except the most significant
   one) will be 0 or 1 with equal probability. *)

(* Private procedure *)
PROCEDURE Next55 (self: T);

END Random.
