INTERFACE SimpleIO;	(*29.07.93. LB*)

(*SimpleIO provides simple procedures
  to open, close, read and write files.
  Files are considered as sequences of characters or texts.

  It has a flexible, automatic mechanism for file naming.
  If the "name" paramtere is omitted at opening, then
  the opening procedures ask the user for a file name.
  If the user enters an empty line, the file defaults to standard i/o,
  i.e. normally to keyboard (standardin) and screen (standardout).
  The procedure GetFileName can be used if you want
  to have more explicit control over file naming. 

  The operative procedures default to standard input/output (via NIL),
  i.e. standard i/o can be used without any opening and closing.*)

  IMPORT Rd, Wr;

  TYPE
    Reader = Rd.T;
    Writer = Wr.T;

  VAR
    PromptText: TEXT := " file name, please = ";
    
  PROCEDURE OpenRead(name: TEXT := NIL): Reader;
  (*Opens the file, called "name" for reading.
    If "name" is NIL, or the file does not exists,
    it asks the user for a file name until the file can be opened.
    It returns Stdio.stdin if the user enters "return".*)

  PROCEDURE OpenWrite(name: TEXT := NIL): Writer;
  (*Opens the file, called "name" for writing.
    If "name" is NIL, or the file already exists,
    it asks the user for a file name until the file can be opened.
    It returns Stdio.stdout if the user enters "return".*)

  PROCEDURE FileExists(name: TEXT): BOOLEAN;
  (*Returns TRUE iff the file exists.*)

  PROCEDURE CloseRead(VAR rd: Reader);
  (*Closes the file. Assigns NIL to rd - except rd is stdin.*)

  PROCEDURE CloseWrite(VAR wr: Writer);
  (*Flushes and closes the file. Assigns NIL to wr - except wr is stdout.*)

  PROCEDURE Flush(wr: Writer := NIL);
  (*Flushes the file.*)

  PROCEDURE GetFileName(userInfo: TEXT := ""): TEXT;
  (*Asks the user for a file name - actually a simple text.
    Returns the empty text if the user enters "return".
    It displays PromptText preceded by userInfo.*)

  PROCEDURE GetChar(rd: Reader := NIL): CHAR;
  (*Returns the next char, or the nul char, if end of file reached.*)

  PROCEDURE GetLine(rd: Reader := NIL): TEXT;
  (*Reads a line; the terminating newline is discarded.
    Returns the empty text, if end of file reached.*)

  PROCEDURE EOF(rd: Reader := NIL): BOOLEAN;
  (*Returns TRUE iff end of file reached.
    Returns always TRUE for stdin.*)

  PROCEDURE Availabe(rd: Reader := NIL): BOOLEAN;
  (*Returns TRUE iff some characters are available.*)

  PROCEDURE Reset(rd: Reader := NIL);
  (*Sets the Reader on the beginning.*)

  PROCEDURE PutChar(ch: CHAR; wr: Writer := NIL);
  (*Outputs a single character. 
    Makes a flush if ch is newline and wr is stdout (or NIL)*)

  PROCEDURE PutText(t: TEXT; wr: Writer := NIL);
  (*Outputs all characters in t. 
    Makes a flush if text ends with newline and wr is stdout (or NIL)*)

END SimpleIO.
