(* Copyright (C) 1989, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Tue Jan 28 13:31:45 PST 1992 by kalsow     *)
(*      modified on Wed Nov 21 01:56:30 1990 by muller         *)
(*      modified on Fri Jan 20 15:27:49 PST 1989 by glassman   *)
(*      modified on Wed Apr 22 17:34:04 1987 by roberts        *)
(*      modified on Tue Nov 25 11:48:02 1986 by brooks         *)

INTERFACE Filename;
(***************************************************************)
(* Procedures for file names and search paths                  *)
(*                                                             *)
(* This interface provides several standard procedures for     *)
(* working with file names and search paths.  A search path    *)
(* is a list of directories separated by colons in the         *)
(* traditional Unix style.                                     *)
(*                                                             *)
(* The procedures exported by this package are listed          *)
(* below.  A more complete description of each operation is    *)
(* provided in the comments that following the declaration.    *)
(*                                                             *)
(*         Root(filename)                                      *)
(*         Extension(filename)                                 *)
(*         Head(filename)                                      *)
(*         Tail(filename)                                      *)
(*         DefaultExtension(filename, ".xxx")                  *)
(*         ExpandTilde(filename)                               *)
(*         SearchPath(dir, path, filename, [file predicate])   *)
(*         RdFromPath(dir, path, filename)                     *)
(*                                                             *)
(* This interface is similar to the C language interface in    *)
(* /proj/packages/clib/file.c and provides the same basic      *)
(* functionality.                                              *)
(*                                                             *)
(* Index: search path, manipulating from Modula-3; files,      *)
(* naming operations                                           *)
(***************************************************************)


IMPORT Rd;

TYPE
  FilePredicate = PROCEDURE ((* dir: OS.Dir; *)filename: TEXT): BOOLEAN;

EXCEPTION Error;

PROCEDURE FileIsReadable (filename: TEXT): BOOLEAN;
(* Return TRUE iff filename names a readable file.         *)

PROCEDURE Root (filename: TEXT): TEXT;
(* Returns the root part of the filename and is equivalent *)
(* to the :r substitution in the C shell.                  *)

PROCEDURE Extension (filename: TEXT): TEXT;
(* Returns the extension part of the filename and is       *)
(* equivalent to the :e substitution in the C shell.       *)

PROCEDURE Head (filename: TEXT): TEXT;
(* Returns the head part of the filename and is equivalent *)
(* to the :h substitution in the C shell.                  *)

PROCEDURE Tail (filename: TEXT): TEXT;
(* Returns the tail part of the filename and is equivalent *)
(* to the :t substitution in the C shell.                  *)

PROCEDURE DefaultExtension (filename, ext: TEXT): TEXT;
(* Adds an extension to a filename if none already         *)
(* exists.  Alternatively, if the extension field begins   *)
(* with a *, any old extension in the first filename is    *)
(* replaced with the given extension.                      *)

PROCEDURE ExpandTilde (filename: TEXT): TEXT  RAISES {Error};
(* Expands the "~user/" at the beginning of a file name    *)
(* to be the path of user's home directory.  If "user" is  *)
(* empty, the home directory of the user running the       *)
(* program is used.  The password file is used to map user *)
(* names to directories.  If there is any problem finding  *)
(* "user", Error is raised.                                *)

PROCEDURE SearchPath (path, filename: TEXT;
                      pred: FilePredicate := FileIsReadable): TEXT;
(* Most clients will not specify a filepred argument       *)
(* and will use this simply to find the first file         *)
(* that exists along a given search path.  More            *)
(* generally, SearchPath returns a pathname to the         *)
(* first instance of filename in the list of               *)
(* directories specified by the path for which the         *)
(* filepred returns TRUE.  If none exist, SearchPath       *)
(* returns NIL.                                            *)

PROCEDURE RdFromPath (path, filename: TEXT): Rd.T  RAISES {Rd.Failure};
(* This is a convenience routine and is a shorthand for    *)
(* FileStream.OpenRead(dir,                                *)
(*                     SearchPath(dir, path, filename))    *)

END Filename.

(***************************************************************)
(* Additional notes by Ken Brooks for those not entirely       *)
(* familiar with csh:                                          *)
(*                                                             *)
(* A pathname consists of components, separated by slashes.    *)
(* An extension is the part of the pathname following the      *)
(* last period in the last component.  If there is no period   *)
(* in the last component, the extension is the null string.    *)
(* The root of a pathname is all of the pathname except the    *)
(* extension and its period.                                   *)
(*                                                             *)
(* The tail of a pathname is the last component.  The head is  *)
(* everything except the last component and the slash that     *)
(* divides it from the rest.  Exception: If the pathname has   *)
(* only one component, that is both the head and the tail;     *)
(* the head is not the null string.                            *)
(* ************************************************************* *)
