(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by stolfi on Wed Apr 26 04:02:07 1989               *)
(* Last modified on Tue Feb 11 21:39:40 PST 1992 by muller     *)
(*      modified on Thu Oct 25  9:18:54 PDT 1990 by stolfi     *)
(*      modified on Thu Jul 20 11:32:57 PDT 1989 by glassman   *)

INTERFACE RGBScale;

(* Procedures for building color ramps, grey scales, RGB grids,etc.
   
  This interface contains extensions to RGB.def:  procedures for
  building color ramps, grey scales, RGB grids,etc
  
  Index: colors; graphics, colors; display, color.

*)

IMPORT Color, RGB, RGBDist, Intensity, IntensityScale;

TYPE T = ARRAY OF RGB.T;

PROCEDURE Straight(
    READONLY a, b: RGB.T; 
    READONLY s: IntensityScale.T;
  ): REF T;
  (* 
    Returns an array c of colors where
|      c^[i] = RGB.Interpolate(a, s[i], b, 1.0-s[i])
    i.e. the color s[i] of the way from a to b.
    If s[i] is Intensity.Undefined, c^[i] will be the Undefined color. *)

PROCEDURE Grey(
    n: CARDINAL; 
    a, b: Intensity.T; 
    ratio: REAL := 1.0
  ): REF T;
  (* 
    A shorthand for
|       Straight(Black, White, IntensityScale.New(n, a, b, ratio)^)
  *)

PROCEDURE Path(
    READONLY s: ARRAY OF REAL; 
    READONLY p: ARRAY OF RGB.T;
    dist: Color.DistFn := RGBDist.Perceptual;
    lumWeight: REAL := 1.0;
  ): REF T;
  (* 
    Given a polygonal path in RGB space with vertices p[0..m-1],
    and a list of real "positions" s[0..n-1], this procedure returns
    a list r[0..n-1] of colors at those prescribed positions
    along p.  
  
    More precisely, each returned color r[i] lies somewhere on 
    on the path p, and, approximately,
|
|        dist(p[0], r[0], lumWeight) is proportional to s[0],
|        dist(r[i], r[i+1], lumWeight) is proportional to s[i+1]-s[i],
|        dist(r[n-1], p[m-1], lumWeight) is proportional to 1.0-s[n-1].
|
    The positions s[i] must be between 0.0 and 1.0. *)

PROCEDURE Grid (READONLY rs, gs, bs: IntensityScale.T): REF T;
  (* 
    Returns an array of colors c consisting of a rectangular
    grid in RGB space with the given intensity levels along each axis.
    More precisely,
|
|     c^[i + NR*j + NR*NG*k] = rs[i]*Red + gs[j]*Green + bs[k]*Blue,
|
    for i in [0..NR-1], j in [0..NG-1], k in [0..NB-1],
    where NR = NUMBER(rs), NG = NUMBER(gs), and NB = NUMBER(bs). *)

END RGBScale.

