(* Copyright (C) 1989, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last Modified On Thu Oct 11 22:04:09 1990 By muller         *)

(*   based on Time.def *)


INTERFACE Time;

FROM Thread IMPORT Alerted;
IMPORT Word;

(* Routines for talking to other programs about times
   The Time interface provides procedures and types for dealing with dates and
   times. The type Time.T is the system representation for the current date and
   time, which happens to be an integer counting the number of microseconds
   since the start of January 1, 1970 GMT. The Time interface also includes
   procedures that allow a thread to sleep for a specified interval or until a
   specified moment, procedures to get the local time zone.
   Index: times; dates; time zone *)

(**************************)
(***  Timing Interface  ***)
(**************************)
(*****************************************************************************)
(*  This interface provides two units for expressing time.  The primary one  *)
(*  is Time.T, which expresses time in implementation-independent units      *)
(*  (microseconds).  The secondary unit is Time.Ticks, which depends on the  *)
(*  granularity of time measurement used by the implementation.  Both        *)
(*  units may be used to express an interval of time or to express an        *)
(*  instant relative to a particular epoch.  Some of the procedures in this  *)
(*  interface treat their arguments as intervals, while others regard them   *)
(*  as epoch-relative instants.                                              *)
(*                                                                           *)
(*  Time.T's are expressed in microseconds, with the idea that the           *)
(*  microsecond is (and will be for some years) a sufficiently fine unit     *)
(*  of measurement for timing purposes.  However, the procedures in this     *)
(*  interface that manipulate Time.T's are not required to be accurate to    *)
(*  within a microsecond; on the contrary, they may resolve time much more   *)
(*  coarsely.  The accuracy of clock values is typically a few milliseconds, *)
(*  and program(mer)s should not assume that, just because clock values have *)
(*  the precision of a microsecond, they are necessarily accurate to the     *)
(*  microsecond.  The procedure 'ClockGrain' may be used to determine the    *)
(*  approximate accuracy of the implementation.                              *)
(*****************************************************************************)




TYPE
  Seconds      = Word.T;
  Microseconds = Word.T;
  T = RECORD
        seconds     : Seconds;
        microseconds: Microseconds; END;

CONST 
  Epoch = T {0, 0};
       (*  The 'epoch' for Time.T is 1 January 1970, 00:00:00.000000 GMT. *)


(*  Facilities for manipulating T's.  *)



PROCEDURE Pause (interval: CARDINAL);
PROCEDURE AlertPause (interval: CARDINAL) RAISES {Alerted};
(* These procedures provides a mechanism for a thread of control to delay its
   execution by a specified amount. The parameter is a 31-bit positive value
   expressed in microseconds, which limits the maximum delay to approximately
   2147 seconds, or 35 minutes. For longer-duration waits, see 'LongPause' and
   'PauseUntil'. AlertPause is sensible to Alerts, Pause isn't. *)

PROCEDURE LongPause (interval: CARDINAL);
PROCEDURE AlertLongPause (interval: CARDINAL) RAISES {Alerted};
(* This procedure is functionally similar to 'Pause', but permits longer waits.
   The parameter is a 31-bit positive value expressed in seconds.
   'LongPause(interval)' is semantically equivalent to the sequence: 
       time := Now();
       INC (time.seconds, interval);
        PauseUntil(time); 
   AlertLongPause is sensible to Alerts, LongPause isn't. *)

PROCEDURE PauseUntil (t: T);
PROCEDURE AlertPauseUntil (t: T) RAISES {Alerted};
(* 'PauseUntil' is functionally similar to 'Pause', but specifies an absolute
   time (rather than an interval) after which the thread of control is to
   resume execution.
   AlertPauseUntil is sensible to Alerts, PauseUntil isn't. *)

PROCEDURE Now (): T;
(* The value returned is the number of seconds and microseconds that have
   elapsed since 'epoch' (see above). *)

PROCEDURE ClockGrain (): Microseconds;
(* This procedure returns the interval between clock updates. Calls of 'Now'
   that occur less than 'ClockGrain()' microseconds apart are likely to return
   identical values. It must be stressed that the value returned by ClockGrain
   is approximate; in reality, clock updates may occur more or less frequently.
   *)

PROCEDURE Add (t1, t2: T): T;
(*  Result is t1 + t2  *)

PROCEDURE Subtract (t1, t2: T): T;
(*  Result is t1 - t2.  If t1 < t2, the result is undefined.  *)

PROCEDURE Compare (t1, t2: T): [-1 .. 1];
(*  Result is the sign of t1-t2 *)


(***************************)
(***  Time Zone Support  ***)
(***************************)

TYPE
  TimeZone = RECORD
               minutesWest : INTEGER;
               dstAlgorithm: INTEGER; END;
             (* minutesWest is the number of minutes west of GMT to the local
                time zone, i.e. California is +480. dstAlgorithm indicates the
                algorithm to use for daylight savings time: 0 = no dst; 1 =
                USA; 2 = Australia; 3 = western Europe; 4 = middle Europe; 5 =
                eastern Europe *)


PROCEDURE GetTimeZone (): TimeZone;
(* This procedure sets 't' to the value set by the most recent call of
   'SetTimeZone' (see below). If 'SetTimeZone' has not been called since system
   initialization, 'GetTimeZone' returns (+480, 1). *)

(**********************)
(* Resetting the time *)
(**********************)


EXCEPTION
  NoPrivileges;


PROCEDURE SetTime (t: T; tz: TimeZone) RAISES {NoPrivileges};
(* Sets the time and the timezone of the system; can be called by the superuser
   only, otherwise raises NoPrivileges *)

END Time.
