(* Copyright (C) 1989, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Thu Nov  2 21:55:29 1989 by muller         *)
(*      modified on Fri Sep 29 15:46:46 1989 by kalsow         *)
(*      modified on Fri Jan 20 10:02:29 PST 1989 by glassman   *)
(*      modified on Wed May 27 23:11:56 1987 by mbrown         *)
(*      modified Mon May 13 20:11:50 1985 by Ellis             *)

INTERFACE Char;

(* Characters and Character Sets

   Char deals with individual characters.  It includes constant
   definitions for the character codes of exotic characters, such
   as Char.NL for new-line.  It classifies characters into groups,
   like digits or punctuation; each group is represented as a set
   of characters.  Finally, it provides mapping tables that translate
   lower-case letters into upper-case and the like.

   Index: characters; punctuation; case, converting characters;
          characters, case conversion; upper-case, converting to lower;
          lower-case, converting to upper
*)

CONST
  NUL = '\000';    SOH = '\001';    STX = '\002';    ETX = '\003';
  EOT = '\004';    ENQ = '\005';    ACK = '\006';    BEL = '\007';
  BS  = '\010';    HT  = '\011';    NL  = '\012';    VT  = '\013';
  NP  = '\014';    CR  = '\015';    SO  = '\016';    SI  = '\017';
  DLE = '\020';    DC1 = '\021';    DC2 = '\022';    DC3 = '\023';
  DC4 = '\024';    NAK = '\025';    SYN = '\026';    ETB = '\027';
  CAN = '\030';    EM  = '\031';    SUB = '\032';    ESC = '\033';
  FS  = '\034';    GS  = '\035';    RS  = '\036';    US  = '\037';
  SP  = '\040';    DEL = '\177';

TYPE T = CHAR;

CONST
  All           = SET OF T {'\000'..'\377'};
  Asciis        = SET OF T {'\000'..'\177'};
  Controls      = SET OF T {'\000'..'\037', '\177'};
  Spaces        = SET OF T {' ', '\t', '\n', '\r', '\f'};
  Digits        = SET OF T {'0'..'9'};
  Uppers        = SET OF T {'A'..'Z'};
  Lowers        = SET OF T {'a'..'z'};
  Letters       = Uppers + Lowers;
  AlphaNumerics = Letters + Digits;
  Graphics      = Asciis - Controls;
  Punctuation   = Graphics - AlphaNumerics;

VAR
  Upper   : ARRAY T OF T;
  Lower   : ARRAY T OF T;
  Control : ARRAY T OF T;
  (* These constant arrays implement character conversions (mappings):

         Upper[c]   = the upper-case equivalent of c if c is a letter, o.w. c
         Lower[c]   = the lower-case equivalent of c if c is a letter, o.w. c
         Control[c] = the control-shifted equivalent of c if c is in Graphics
                       (i.e. BitAnd( c, 037B )), o.w. c
    *)

PROCEDURE New (value: T): REF T;
(* Allocates and initializes a new heap value *)

PROCEDURE NewArray (size: CARDINAL;  value: T := NUL): REF ARRAY OF T;
(* Allocates a new array from the heap
  and initializes all its elements with the given value *)

PROCEDURE UntracedNew (value: T): UNTRACED REF T;
(* Allocates and initializes a new untraced value *)

PROCEDURE UntracedNewArray (size: CARDINAL;  value: T := NUL):
                                                       UNTRACED REF ARRAY OF T;
(* Allocates a new untraced array from the heap
  and initializes all its elements with the given value *)

PROCEDURE Compare (a, b: T): INTEGER;
(* == RETURN (a - b) *)

PROCEDURE Lt (a, b: T): BOOLEAN;
(* == RETURN (a < b) *)

PROCEDURE Eq (a, b: T): BOOLEAN;
(* == RETURN (a = b) *)

PROCEDURE Hash (a: T): INTEGER;
(* == RETURN ORD (a) *)

END Char.
