(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: RepeatStmt.m3                                         *)
(* Last modified on Mon Mar  2 11:11:19 PST 1992 by kalsow     *)
(*      modified on Tue Oct 10 18:42:03 1989 by muller         *)

MODULE RepeatStmt;

IMPORT M3, Expr, Type, Bool, Error, Emit, Stmt, StmtRep;
IMPORT Temp, Token, Scanner, Marker;

TYPE
  P = Stmt.T OBJECT
        body    : Stmt.T;
        expr    : Expr.T;
      OVERRIDES
        check    := Check;
	compile  := Compile;
        outcomes := GetOutcome;
      END;

PROCEDURE Parse (READONLY fail: Token.Set): Stmt.T =
  TYPE TK = Token.T;
  VAR p: P;
  BEGIN
    p := NEW (P);
    StmtRep.Init (p);
    Scanner.Match (TK.tREPEAT, fail, Token.Set {TK.tUNTIL} + Token.StmtStart);
    p.body := Stmt.Parse (fail + Token.Set {TK.tUNTIL});
    Scanner.Match1 (TK.tUNTIL, fail);
    p.expr := Expr.Parse (fail);
    RETURN p;
  END Parse;

PROCEDURE Check (p: P;  VAR cs: Stmt.CheckState) =
  VAR t: Type.T;
  BEGIN
    Marker.PushExit (0);
    Stmt.TypeCheck (p.body, cs);
    Marker.Pop ();
    Expr.TypeCheck (p.expr, cs);
    t := Expr.TypeOf (p.expr);
    IF (Type.Base (t) # Bool.T) THEN
      Error.Msg ("REPEAT condition must be a BOOLEAN");
    END;
  END Check;

PROCEDURE Compile (p: P): Stmt.Outcomes =
  VAR x: Temp.T;  label: INTEGER;  oc: Stmt.Outcomes;
  BEGIN
    label := M3.NextLabel;  INC (M3.NextLabel, 2);
    Emit.OpL ("@:;\n\001", label);

    Marker.PushExit (label+1);
    oc := Stmt.Compile (p.body);
    Marker.Pop ();
  
    IF (Stmt.Outcome.FallThrough IN oc) THEN
      x := Expr.Compile (p.expr);
      Emit.OpT ("\002if (! @) ", x);
      Emit.OpL ("goto @;\n", label);
      Temp.Free (x);
    ELSE
      Emit.OpL ("\002/* end @ */\n", label);
    END;

    IF (Stmt.Outcome.Exits IN oc) THEN
      Emit.OpL ("@:;\n", label+1);
      oc := oc  + Stmt.Outcomes {Stmt.Outcome.FallThrough}
                - Stmt.Outcomes {Stmt.Outcome.Exits};
    END;

    RETURN oc;
  END Compile;

PROCEDURE GetOutcome (p: P): Stmt.Outcomes =
  VAR oc := Stmt.GetOutcome (p.body);
  BEGIN
    IF (Stmt.Outcome.Exits IN oc) THEN
      oc := oc  + Stmt.Outcomes {Stmt.Outcome.FallThrough}
                - Stmt.Outcomes {Stmt.Outcome.Exits};
    END;
    RETURN oc;
  END GetOutcome;

BEGIN
END RepeatStmt.
