(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Last modified on Tue Feb 11 16:18:58 PST 1992 by muller                   *)

INTERFACE Pathname;

IMPORT Text;

(* 
   This interface exports procedures to manipulate paths.  The model of
paths is that they are made of four components:  the directory, the name,
the separator and the extension. 

   The Dir, Name, Sep and Ext procedures extract these components from a path.
They satisfy the property:

	Dir(x) & Name(x) & Sep(x) & Ext(x) = x

The precise rules depend on the underlying operating system.


   The Make procedure is an attempt to provide a simple, operating-system
independent, way of building file names from the components.  On some systems,
it may truncate components. 


   The companion interface SRCM3Path provides the default extensions used by the 
SRC Modula-3 system.
*)



(* Unix rules:
  
A path is made of simple paths, separated by '/'.  A simple path can be empty.
 
The 'directory' part of a path is the prefix that ends with the last '/';
if the path contains no '/'s, the directory part is empty.
  
The extension, separtor and name part of a path are those of its last
simple path.

The 'extension' part of a simple path is the suffix that starts after the
last ".", provided that it is neither the first nor the last character and
the preceding character is not "."; if there is no such suffix, the
extension is empty.

If the extension part of a simple path is non empty, its 'separator' is '.'.
Otherwise, the separator is empty.

The 'name' part of a simple path is obtained by removing the separator 
and the extension from the simple path.

  Examples:                                                               
                                                                          
    file name             Dir          Name       Sep      Ext
  -----------------------------------------------------------------------
    foo                                foo
    old/Foo.mod           old/         Foo         .       mod
    .tmp.csh                           .tmp        .       csh
  -----------------------------------------------------------------------
    123.0                              123         .       0
    /                     /                                                      
    /v.2/old              /v.2/        old
  -----------------------------------------------------------------------
    a.                                 a.
    a..                                a..
    a..z                               a..z
  -----------------------------------------------------------------------
    ivy.v22.4                          ivy.v22     .       4
    ivy.v22.4//.          ivy.v22.4//  . 
    /tmp/.login           /tmp/        .login
  -----------------------------------------------------------------------
    /..                   /            ..
    ...a                               ...a
    .,.login                           .,          .       login
  -----------------------------------------------------------------------
    ~/bin                 ~/           bin 
    ~joe/.cshrc           ~joe/        .cshrc
    $HOME/${F}.o          $HOME/       ${F}        .       o
  -----------------------------------------------------------------------
                                                                          
Note that these definitions are somewhat different from those used
by sh(1) and csh(1).  This is a feature, not a bug.

The Make procedure returns the Text.T : dir & name & "." & ext if ext is not
the empty string, dir & name otherwise.
*)


PROCEDURE Dir (path: Text.T): Text.T;
  (* Return the "directory" part of the given path. *)

PROCEDURE Name (path: Text.T): Text.T;
  (* Return the "name" part of the given path. *)

PROCEDURE Sep (path: Text.T): Text.T;
  (* Return the "separator" part of the given path. *)

PROCEDURE Ext (path: Text.T): Text.T;
  (* Return the "extension" part of the given path. *)

PROCEDURE Make (dir, name, ext: Text.T): Text.T;
  (* Return a path with these components and the standard separator. *)

END Pathname.
