SYNOPSIS
========

        my $stripe = Net::API::Stripe->new({
            debug => 3,
            conf_file => './stripe-settings.json',
            livemode => 0,
            ignore_unknown_parameters => 1,
            expand => 'all',
        }) || die( Net::API::Stripe->error );

A Stripe json settings file looks like this:

        {
            "livemode": false,
            "test_secret_key": "sk_test_1234567890abcdefg",
            "test_public_key": "pk_test_1234567890abcdefg",
            "live_secret_key": "sk_live_0987654321zyxwvut",
            "live_public_key": "pk_live_0987654321zyxwvut",
            "version": "2020-03-02",
        }

Create a customer:

        # Create an address object
        my $addr;
        if( $v->{street} || $v->{city} || $v->{postal_code} || $v->{country} )
        {
            $addr = $stripe->address({
                line1 => $v->{street},
                line2 => $v->{building},
                city => $v->{city},
                postal_code => $v->{postal_code},
                state => $v->{state},
                country => $v->{country},
            }) || bailout( "Unable to create a postal address object: ", $stripe->error );
        }
        my $cust_object = $stripe->customer({
            balance => 20000,
            address => $addr,
            # Must be set up previously before using it
            coupon => '2020DISCOUNT50',
            # Japanese Yen
            currency => 'jpy',
            description => 'VIP customer',
            email => 'john@example.com',
            invoice_prefix => 'VIP',
            # Default payment must be set up beforehand for it to be declared here
            invoice_settings => { default_payment_method => 'pm_fake1234567' },
            metadata => { db_id => 123, process_id => 456 },
            name => 'John Doe',
            phone => '+81-90-1234-5678',
            preferred_locales => [qw( en ja )],
            shipping => $addr,
        });

        # Submit this customer to Stripe for creation
        my $cust = $stripe->customers( create => $cust_object ) || die( sprintf( "Failed with error message %s and code %d\n", $stripe->error->message, $stripe->error->code ) );

Retrieve customer:

        my $cust = $stripe->customers( retrieve => 'cust_fake12345' );
        # or we can also pass a customer object
        my $cust = $stripe->customers( retrieve => $cust_object ) || do
        {
            if( $stripe->http_response->code == 404 )
            {
                die( "Customer ", $cust_object->id, " does not exist!\n" );
            }
            else
            {
                die( "Some unexpected error occurred: ", $stripe->error, "\n" );
            }
        };

Other methods are describe below and the parameters they take are
documented in their respective module.

VERSION
=======

        v1.3.0

DESCRIPTION
===========

This is a comprehensive Stripe API. It provides an object oriented
friendly interface for which I put a lot of hard work so you could spend
your time on other aspects of your development.

This distribution is almost completely auto-generated based on Stripe
API documentation version 2022-08-01 and contains 203 modules.

It inherits from
[Module::Generic](https://metacpan.org/pod/Module::Generic){.perl-module}
and
[Net::API::Stripe](https://metacpan.org/pod/Net::API::Stripe){.perl-module}
sub modules inherits from
[Net::API::Stripe::Generic](https://metacpan.org/pod/Net::API::Stripe::Generic){.perl-module}

This interface aims at making it easy to make api calls to Stripe,
however it is important and a time-saver to read Stripe documentation
comprehensively.

This interface will do minimal data integrity check. Thus, even though
this interface will check for proper data types like array, right
property names used, mandatory parameters in api calls, etc it does not
do any check on the data itself, so you should always check for return
value from Stripe api calls and look at the Stripe error returned. If an
error occured, the Stripe api method will return undef and set an error
message accordingly. See [\"ERROR
HANDLING\"](#error-handling){.perl-module}

CONSTRUCTOR
===========

new
---

Provided with an hash of parameters and this creates a new
[Net::API::Stripe](https://metacpan.org/pod/Net::API::Stripe){.perl-module}
objects.

Its arguments also have methods of the same name.

`api_uri`

:   The base uri of the Stripe API. This should not be changed.

`browser`

:   The user agent id to use when making http api calls

`conf_file`

:   The file path to the configuration file. Each property in this
    configuration file is same as the parameters to instantiate a new
    [Net::API::Stripe](https://metacpan.org/pod/Net::API::Stripe){.perl-module}
    object.

`debug`

:   Toggles debug mode on/off

`expand`

:   Integer. Sets the depth level of expansion of Stripe objects. If
    objects are not expanded, Stripe API will return the object id, but
    when they are expanded, Stripe returns the entire object and its
    properties. You can then chain your code and do something like:

            print $cust->invoice_settings->default_payment_method->type

`ignore_unknown_parameters`

:   Boolean. When this is on, this will ignore any properties sent back
    from Stripe API that are unknown to us. This happens frequently as
    Stripe updates its API. When this value is set to false, then
    unknown properties will cause this to stop processing and return an
    error.

`livemode`

:   Boolean value to toggle test or live mode

`verbose`

:   Toggles verbose mode on/off

`version`

:   The version of the Stripe API to use. Example `2020-03-02`

METHODS
=======

account
-------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::Account](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module}
object.

account\_bank\_account
----------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
object.

account\_card
-------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::ExternalAccount::Card](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}
object.

account\_link
-------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::Account::Link](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Link){.perl-module}
object.

address
-------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Address](https://metacpan.org/pod/Net::API::Stripe::Address){.perl-module}
object.

address\_kana
-------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::AddressKana](https://metacpan.org/pod/Net::API::Stripe::AddressKana){.perl-module}
object.

address\_kanji
--------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::AddressKanji](https://metacpan.org/pod/Net::API::Stripe::AddressKanji){.perl-module}
object.

amount
------

Provided with a number, this returns a
[Module::Generic::Number](https://metacpan.org/pod/Module::Generic::Number){.perl-module}
object, which extends
[Number::Format](https://metacpan.org/pod/Number::Format){.perl-module}

api\_uri
--------

Returns the [URI](https://metacpan.org/pod/URI){.perl-module} object of
the Stripe api.

application\_fee
----------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::ApplicationFee](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee){.perl-module}
object.

application\_fee\_refund
------------------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::ApplicationFee::Refund](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}
object.

apps\_secret
------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::AppsSecret](https://metacpan.org/pod/Net::API::Stripe::Connect::AppsSecret){.perl-module}
object.

authorization
-------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Issuing::Authorization](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module}
object.

balance
-------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Balance](https://metacpan.org/pod/Net::API::Stripe::Balance){.perl-module}
object.

balance\_transaction
--------------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Balance::Transaction](https://metacpan.org/pod/Net::API::Stripe::Balance::Transaction){.perl-module}
object.

bank\_account
-------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
object.

billing\_details
----------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::Details](https://metacpan.org/pod/Net::API::Stripe::Billing::Details){.perl-module}
object.

billing\_portal\_configuration
------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::PortalConfiguration](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalConfiguration){.perl-module}
object.

billing\_portal\_session
------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::PortalSession](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalSession){.perl-module}
object.

billing\_thresholds
-------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::Thresholds](https://metacpan.org/pod/Net::API::Stripe::Billing::Thresholds){.perl-module}
object.

browser
-------

Set or get the user agent string used when making calls to Stripe API.

business\_profile
-----------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::Business::Profile](https://metacpan.org/pod/Net::API::Stripe::Connect::Business::Profile){.perl-module}
object.

capability
----------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::Account::Capability](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Capability){.perl-module}
object.

card
----

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::ExternalAccount::Card](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}
object.

card\_holder
------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Issuing::Card::Holder](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}
object.

charge
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}
object.

checkout\_session
-----------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Checkout::Session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object.

code2error
----------

Given a code returned by Stripe upon error, this returns the
corresponding string.

        my $cust = $stripe->customers( retrieve => $id ) || 
          die( $stripe->code2error( $stripe->error->code ), "\n" );

code\_verification
------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Payment::Source::CodeVerification](https://metacpan.org/pod/Net::API::Stripe::Payment::Source::CodeVerification){.perl-module}
object.

company
-------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::Account::Company](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Company){.perl-module}
object.

conf\_file( \[ file path \] )
-----------------------------

Given a json configuration file, it will read the data, set the property
*conf\_data* to the decoded hash and return it. When called without
argument, it returns the current value of *conf\_data*.

connection\_token
-----------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Terminal::ConnectionToken](https://metacpan.org/pod/Net::API::Stripe::Terminal::ConnectionToken){.perl-module}
object.

country\_spec
-------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::CountrySpec](https://metacpan.org/pod/Net::API::Stripe::Connect::CountrySpec){.perl-module}
object.

coupon
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object.

credit\_note
------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object.

currency
--------

Set or get the 3-letter iso 4217 currency, such as `jpy` for Japanese
yen or `eur` for Euro.

customer
--------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object.

customer\_balance\_transaction
------------------------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Customer::BalanceTransaction](https://metacpan.org/pod/Net::API::Stripe::Customer::BalanceTransaction){.perl-module}
object.

customer\_cash\_balance\_transaction
------------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Cash::Transaction](https://metacpan.org/pod/Net::API::Stripe::Cash::Transaction){.perl-module}
object.

customer\_tax\_id
-----------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Customer::TaxId](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module}
object.

data
----

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Event::Data](https://metacpan.org/pod/Net::API::Stripe::Event::Data){.perl-module}
object.

delete( END POINT, HASH PAYLOAD )
---------------------------------

Given a Stripe end point as a URI absolute path, and a payload as a hash
reference, this will issue a `DELETE` http query and return a hash
reference corresponding to the json data returned by Stripe, or, in case
of error, it will return undef and set the error which can be accessed
with `$stripe-`error\> (a
[Module::Generic::Exception](https://metacpan.org/pod/Module::Generic::Exception){.perl-module}
object).

discount
--------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Discount](https://metacpan.org/pod/Net::API::Stripe::Billing::Discount){.perl-module}
object.

dispute
-------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
object.

document
--------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::Account::Document](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Document){.perl-module}
object.

encode\_with\_json
------------------

Takes a bollean value. This is used to set whether the payload should be
encoded with json. This should not be changed.

event
-----

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Event](https://metacpan.org/pod/Net::API::Stripe::Event){.perl-module}
object.

evidence
--------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Issuing::Dispute::Evidence](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute::Evidence){.perl-module}
object.

evidence\_details
-----------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Dispute::EvidenceDetails](https://metacpan.org/pod/Net::API::Stripe::Dispute::EvidenceDetails){.perl-module}
object.

expand
------

Integer. Sets or get the depth of Stripe object expansion. See Stripe
api documentation for more information:
<https://stripe.com/docs/api/expanding_objects>

fee\_refund
-----------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::ApplicationFee::Refund](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}
object.

fields
------

Given an object type, this returns an array reference of all the methods
(aka fields) for that module.

file
----

Provided with optional hash parameters, this returns a
[Net::API::Stripe::File](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
object.

file\_link
----------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::File::Link](https://metacpan.org/pod/Net::API::Stripe::File::Link){.perl-module}
object.

financial\_connections\_account
-------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Financial::Connections::Account](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Account){.perl-module}
object.

financial\_connections\_account\_owner
--------------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Financial::Connections::AccountOwner](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::AccountOwner){.perl-module}
object.

financial\_connections\_account\_ownership
------------------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Financial::Connections::AccountOwnership](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::AccountOwnership){.perl-module}
object.

financial\_connections\_session
-------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Financial::Connections::Session](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Session){.perl-module}
object.

fraud
-----

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Fraud](https://metacpan.org/pod/Net::API::Stripe::Fraud){.perl-module}
object.

funding\_instructions
---------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Issuing::FundingInstructions](https://metacpan.org/pod/Net::API::Stripe::Issuing::FundingInstructions){.perl-module}
object.

generate\_uuid
--------------

Returns a uuid version 4. This uses
[Data::UUID](https://metacpan.org/pod/Data::UUID){.perl-module} to
achieve that.

get( END POINT, HASH PAYLOAD )
------------------------------

Given a Stripe absolute uri and a hash reference, this will issue a http
`GET` request and return a hash reference representing the json data
returned by Stripe or undef if an error occurred. The error can then be
retrieved like `$stripe-`error\> which is a
[Module::Generic::Exception](https://metacpan.org/pod/Module::Generic::Exception){.perl-module}
object.

http\_client
------------

This returns the
[LWP::UserAgent](https://metacpan.org/pod/LWP::UserAgent){.perl-module}
object and create it if it is not yet instantiated.

http\_request
-------------

Get or set the
[HTTP::Request](https://metacpan.org/pod/HTTP::Request){.perl-module}
based on the data provided.

http\_response
--------------

Get or set the
[HTTP::Response](https://metacpan.org/pod/HTTP::Response){.perl-module}
based on the data provided.

identity\_verification\_report
------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Identity::VerificationReport](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationReport){.perl-module}
object.

identity\_verification\_session
-------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Identity::VerificationSession](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module}
object.

ignore\_unknown\_parameters
---------------------------

Boolean. When true, this module will ignore unknown properties returned
from calls made to Stripe api. if set to false, and an unknown property
is received, this will generate an error and return undef, stopping the
flow of the request instead of ignoring it.

invoice
-------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

invoice\_item
-------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Invoice::Item](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}
object.

invoice\_line\_item
-------------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Invoice::LineItem](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::LineItem){.perl-module}
object.

invoice\_settings
-----------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Invoice::Settings](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Settings){.perl-module}
object.

invoiceitem
-----------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::Invoice::Item](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}
object.

ip\_address\_location
---------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::GeoLocation](https://metacpan.org/pod/Net::API::Stripe::GeoLocation){.perl-module}
object.

issuing\_authorization
----------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Issuing::Authorization](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module}
object.

issuing\_card
-------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Issuing::Card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module}
object.

issuing\_cardholder
-------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Issuing::Card::Holder](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}
object.

issuing\_dispute
----------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Issuing::Dispute](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module}
object.

issuing\_transaction
--------------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Issuing::Transaction](https://metacpan.org/pod/Net::API::Stripe::Issuing::Transaction){.perl-module}
object.

item
----

Provided with optional hash parameters, this returns an
[Net::API::Stripe::List::Item](https://metacpan.org/pod/Net::API::Stripe::List::Item){.perl-module}
object.

json
----

This returns a `JSON` object with option *allow\_nonref* enabled.

key( STRIPE API SECRET KEY )
----------------------------

Provided with your Stripe api secret key, this will set this property
accordingly, but will also set the **auth** property as well. **auth**
is used to authenticate you when making calls to Stripe api. **auth**
would be something like this:

        Basic c2tfMTIzNDU2Nzg5MGFiY2RlZmdoaWo6Cg

line\_item
----------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::Invoice::LineItem](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::LineItem){.perl-module}
object.

livemode
--------

Boolean. Set or get the livemode status. If it is true, then all api
query will be mad in live mode.

location
--------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Terminal::Location](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}
object.

login\_link
-----------

Provided with optional hash parameters this returns a
[Net::API::Stripe::Connect::Account::LoginLink](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::LoginLink){.perl-module}
object.

merchant\_data
--------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Issuing::MerchantData](https://metacpan.org/pod/Net::API::Stripe::Issuing::MerchantData){.perl-module}
object.

next\_action
------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Payment::Intent::NextAction](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::NextAction){.perl-module}
object.

order
-----

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Order](https://metacpan.org/pod/Net::API::Stripe::Order){.perl-module}
object.

order\_item
-----------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Order::Item](https://metacpan.org/pod/Net::API::Stripe::Order::Item){.perl-module}
object.

outcome
-------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Charge::Outcome](https://metacpan.org/pod/Net::API::Stripe::Charge::Outcome){.perl-module}
object.

owner
-----

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Payment::Source::Owner](https://metacpan.org/pod/Net::API::Stripe::Payment::Source::Owner){.perl-module}
object.

package\_dimensions
-------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Order::SKU::PackageDimensions](https://metacpan.org/pod/Net::API::Stripe::Order::SKU::PackageDimensions){.perl-module}
object.

payment\_intent
---------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

payment\_method
---------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object.

payment\_method\_details
------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Payment::Method::Details](https://metacpan.org/pod/Net::API::Stripe::Payment::Method::Details){.perl-module}
object.

payout
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Payout](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module}
object.

period
------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::Invoice::Period](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Period){.perl-module}
object.

person
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::Person](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module}
object.

plan
----

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object.

post( END POINT, HASH PAYLOAD )
-------------------------------

Given a Stripe end point absolute uri and a hash reference, this will
issue a `POST` http request to the Stripe api and return a hash
reference representing the object provided by Stripe or undef with an
error set, which can be retrieved using the
[\"error\"](#error){.perl-module} method.

If no *idempotency* parameter was provided, **post** will automatically
create one.

post\_multipart( END POINT, HASH PAYLOAD )
------------------------------------------

Given a Stripe end point absolute uri and a hash reference, this will
issue a `POST` multipart http request to the Stripe api and return a
hash reference representing the object returned by Stripe. If an error
had occurred, it will return undef and set an error that can be
retrieved using the [\"error\"](#error){.perl-module} method.

This method is used primarily when upload file. See the section below on
[\"FILES\"](#files){.perl-module}

price
-----

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object.

product
-------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object.

radar\_early\_fraud\_warning
----------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Fraud](https://metacpan.org/pod/Net::API::Stripe::Fraud){.perl-module}
object.

radar\_value\_list
------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Fraud::ValueList](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}
object.

radar\_value\_list\_item
------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Fraud::ValueList::Item](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList::Item){.perl-module}
object.

reader
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Terminal::Reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}
object.

receiver
--------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Payment::Source::Receiver](https://metacpan.org/pod/Net::API::Stripe::Payment::Source::Receiver){.perl-module}
object.

redirect
--------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Payment::Source::Redirect](https://metacpan.org/pod/Net::API::Stripe::Payment::Source::Redirect){.perl-module}
object.

refund
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Refund](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module}
object.

relationship
------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::Account::Relationship](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Relationship){.perl-module}
object.

reporting\_report\_run
----------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Reporting::ReportRun](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportRun){.perl-module}
object.

reporting\_report\_type
-----------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Reporting::ReportType](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportType){.perl-module}
object.

request
-------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Event::Request](https://metacpan.org/pod/Net::API::Stripe::Event::Request){.perl-module}
object.

requirements
------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::Account::Requirements](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Requirements){.perl-module}
object.

return
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Order::Return](https://metacpan.org/pod/Net::API::Stripe::Order::Return){.perl-module}
object.

review
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Fraud::Review](https://metacpan.org/pod/Net::API::Stripe::Fraud::Review){.perl-module}
object.

schedule
--------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object.

schedule\_query
---------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Sigma::ScheduledQueryRun](https://metacpan.org/pod/Net::API::Stripe::Sigma::ScheduledQueryRun){.perl-module}
object.

scheduled\_query\_run
---------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Sigma::ScheduledQueryRun](https://metacpan.org/pod/Net::API::Stripe::Sigma::ScheduledQueryRun){.perl-module}
object.

session
-------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Checkout::Session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object.

settings
--------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::Account::Settings](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Settings){.perl-module}
object.

setup\_intent
-------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Payment::Intent::Setup](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}
object.

shipping
--------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Shipping](https://metacpan.org/pod/Net::API::Stripe::Shipping){.perl-module}
object.

sku
---

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Order::SKU](https://metacpan.org/pod/Net::API::Stripe::Order::SKU){.perl-module}
object.

source
------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object.

source\_order
-------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Order](https://metacpan.org/pod/Net::API::Stripe::Order){.perl-module}
object.

status\_transitions
-------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::Invoice::StatusTransition](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::StatusTransition){.perl-module}
object.

subscription
------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object.

subscription\_item
------------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::Subscription::Item](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module}
object.

subscription\_schedule
----------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object.

tax\_ids
--------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::TaxID](https://metacpan.org/pod/Net::API::Stripe::Billing::TaxID){.perl-module}
object.

tax\_rate
---------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Tax::Rate](https://metacpan.org/pod/Net::API::Stripe::Tax::Rate){.perl-module}
object.

terminal\_configuration
-----------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Terminal::Configuration](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module}
object.

terminal\_connection\_token
---------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Terminal::ConnectionToken](https://metacpan.org/pod/Net::API::Stripe::Terminal::ConnectionToken){.perl-module}
object.

terminal\_location
------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Terminal::Location](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}
object.

terminal\_reader
----------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Terminal::Reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}
object.

test\_helpers\_test\_clock
--------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::TestHelpersTestClock](https://metacpan.org/pod/Net::API::Stripe::Billing::TestHelpersTestClock){.perl-module}
object.

token
-----

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Token](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module}
object.

topup
-----

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::TopUp](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module}
object.

tos\_acceptance
---------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::Account::TosAcceptance](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::TosAcceptance){.perl-module}
object.

transfer
--------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::Transfer](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer){.perl-module}
object.

transfer\_data
--------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Payment::Intent::TransferData](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::TransferData){.perl-module}
object.

transfer\_reversal
------------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Connect::Transfer::Reversal](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer::Reversal){.perl-module}
object.

transform\_usage
----------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::Plan::TransformUsage](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan::TransformUsage){.perl-module}
object.

treasury\_credit\_reversal
--------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::CreditReversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::CreditReversal){.perl-module}
object.

treasury\_debit\_reversal
-------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::DebitReversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::DebitReversal){.perl-module}
object.

treasury\_financial\_account
----------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::FinancialAccount](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccount){.perl-module}
object.

treasury\_financial\_account\_features
--------------------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::FinancialAccountFeatures](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccountFeatures){.perl-module}
object.

treasury\_inbound\_transfer
---------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::InboundTransfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module}
object.

treasury\_outbound\_payment
---------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::OutboundPayment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module}
object.

treasury\_outbound\_transfer
----------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::OutboundTransfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module}
object.

treasury\_received\_credit
--------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::ReceivedCredit](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedCredit){.perl-module}
object.

treasury\_received\_debit
-------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::ReceivedDebit](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedDebit){.perl-module}
object.

treasury\_transaction
---------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::Transaction](https://metacpan.org/pod/Net::API::Stripe::Treasury::Transaction){.perl-module}
object.

treasury\_transaction\_entry
----------------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Treasury::TransactionEntry](https://metacpan.org/pod/Net::API::Stripe::Treasury::TransactionEntry){.perl-module}
object.

usage\_record
-------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Billing::UsageRecord](https://metacpan.org/pod/Net::API::Stripe::Billing::UsageRecord){.perl-module}
object.

usage\_record\_summary
----------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Billing::UserRecord::Summary](https://metacpan.org/pod/Net::API::Stripe::Billing::UserRecord::Summary){.perl-module}
object.

value\_list
-----------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Fraud::ValueList](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}
object.

value\_list\_item
-----------------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::Fraud::ValueList::Item](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList::Item){.perl-module}
object.

verification
------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::Account::Verification](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Verification){.perl-module}
object.

verification\_data
------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Issuing::Authorization::VerificationData](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization::VerificationData){.perl-module}
object.

verification\_fields
--------------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::Connect::CountrySpec::VerificationFields](https://metacpan.org/pod/Net::API::Stripe::Connect::CountrySpec::VerificationFields){.perl-module}
object.

version
-------

Set or get the api version. This must be set on the [Stripe
dashboard](https://dashboard.stripe.com/){.perl-module}

webhook
-------

Provided with optional hash parameters, this returns a
[Net::API::Stripe::WebHook::Object](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}
object.

webhook\_endpoint
-----------------

Provided with optional hash parameters, this returns an
[Net::API::Stripe::WebHook::Object](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}
object.

ACCOUNT
=======

You can create, delete, list, reject, retrieve or update account

create
------

        my $obj = $stripe->accounts( create => {
            capabilities =>
            {
                card_payments =>
                {
                    requested => "1",
                }
                transfers     =>
                {
                    requested => "1",
                }
            }
            country      => "US",
            email        => q{jenny.rosen@example.com},
            type         => "custom", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::Account](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module}
object or a hash reference, this will create a Stripe account and return
an
[Net::API::Stripe::Connect::Account](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module}
object.

Possible parameters are:

`account_token`

:   An [account
    token](https://stripe.com/docs/api/tokens/create_account){.perl-module},
    used to securely provide details to the account.

`business_profile`

:   Business information about the account.

`business_type`

:   The business type.

`capabilities`

:   **required for Custom accounts** Each key of the dictionary
    represents a capability, and each capability maps to its settings
    (e.g. whether it has been requested or not). Each capability will be
    inactive until you have provided its specific requirements and
    Stripe has verified them. An account may have some of its requested
    capabilities be active and some be inactive.

`company`

:   Information about the company or business. This field is available
    for any `business_type`.

`country`

:   The country in which the account holder resides, or in which the
    business is legally established. This should be an ISO 3166-1
    alpha-2 country code. For example, if you are in the United States
    and the business for which you\'re creating an account is legally
    represented in Canada, you would use `CA` as the country for the
    account being created. Available countries include [Stripe\'s global
    markets](https://stripe.com/global){.perl-module} as well as
    countries where [cross-border
    payouts](https://stripe.com/docs/connect/cross-border-payouts){.perl-module}
    are supported.

`default_currency`

:   Three-letter ISO currency code representing the default currency for
    the account. This must be a currency that [Stripe supports in the
    account\'s country](https://stripe.com/docs/payouts){.perl-module}.

`documents`

:   Documents that may be submitted to satisfy various informational
    requests.

`email`

:   The email address of the account holder. This is only to make the
    account easier to identify to you. Stripe only emails Custom
    accounts with your consent.

`external_account`

:   A card or bank account to attach to the account for receiving
    [payouts](https://stripe.com/docs/connect/bank-debit-card-payouts){.perl-module}
    (you won't be able to use it for top-ups). You can provide either a
    token, like the ones returned by
    [Stripe.js](https://stripe.com/docs/js){.perl-module}, or a
    dictionary, as documented in the `external_account` parameter for
    [bank
    account](https://stripe.com/docs/api#account_create_bank_account){.perl-module}
    creation. \<br\>\<br\>By default, providing an external account sets
    it as the new default external account for its currency, and deletes
    the old default if one exists. To add additional external accounts
    without replacing the existing default for the currency, use the
    [bank
    account](https://stripe.com/docs/api#account_create_bank_account){.perl-module}
    or [card
    creation](https://stripe.com/docs/api#account_create_card){.perl-module}
    APIs.

`individual`

:   Information about the person represented by the account. This field
    is null unless `business_type` is set to `individual`.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`settings`

:   Options for customizing how the account functions within Stripe.

`tos_acceptance`

:   Details on the account\'s acceptance of the [Stripe Services
    Agreement](https://stripe.com/docs/connect/updating-accounts#tos-acceptance){.perl-module}.

`type`

:   **Required**. The type of Stripe account to create. May be one of
    `custom`, `express` or `standard`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/accounts/create>

delete
------

        my $obj = $stripe->accounts( delete => $args ) || die( $stripe->error );

Provided with a
[account](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
account. It returns the account object that was deleted with its
property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/accounts/delete>

list
----

        my $obj = $stripe->accounts( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[account](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module}
object, this issue an api call to get the list of all account.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/accounts/list>

reject
------

        my $obj = $stripe->accounts( reject => {
            reason => "fraud", } ) || die( $stripe->error );

Provided with a
[account](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module},
or a hash reference, this will issue a reject api call.

Returns an account with `payouts_enabled` and `charges_enabled` set to
false on success. If the account ID does not exist, this call returns
[an error](https://stripe.com/docs/api/errors){.perl-module}.

Possible parameters are:

`reason`

:   **Required**. The reason for rejecting the account. Can be `fraud`,
    `terms_of_service`, or `other`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/account/reject>

retrieve
--------

        my $obj = $stripe->accounts( retrieve => $args ) || die( $stripe->error );

Provided with a
[account](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module}
object or a hash reference, this will retrieve a Stripe account and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/accounts/retrieve>

update
------

        my $obj = $stripe->accounts( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[account](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module}
object or a hash reference, this will update a Stripe account and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Account){.perl-module}

Possible parameters are:

`account_token`

:   An [account
    token](https://stripe.com/docs/api/tokens/create_account){.perl-module},
    used to securely provide details to the account.

`business_profile`

:   Business information about the account.

`business_type`

:   The business type.

`capabilities`

:   Each key of the dictionary represents a capability, and each
    capability maps to its settings (e.g. whether it has been requested
    or not). Each capability will be inactive until you have provided
    its specific requirements and Stripe has verified them. An account
    may have some of its requested capabilities be active and some be
    inactive.

`company`

:   Information about the company or business. This field is available
    for any `business_type`.

`default_currency`

:   Three-letter ISO currency code representing the default currency for
    the account. This must be a currency that [Stripe supports in the
    account\'s country](https://stripe.com/docs/payouts){.perl-module}.

`documents`

:   Documents that may be submitted to satisfy various informational
    requests.

`email`

:   The email address of the account holder. This is only to make the
    account easier to identify to you. Stripe only emails Custom
    accounts with your consent.

`external_account`

:   A card or bank account to attach to the account for receiving
    [payouts](https://stripe.com/docs/connect/bank-debit-card-payouts){.perl-module}
    (you won't be able to use it for top-ups). You can provide either a
    token, like the ones returned by
    [Stripe.js](https://stripe.com/docs/js){.perl-module}, or a
    dictionary, as documented in the `external_account` parameter for
    [bank
    account](https://stripe.com/docs/api#account_create_bank_account){.perl-module}
    creation. \<br\>\<br\>By default, providing an external account sets
    it as the new default external account for its currency, and deletes
    the old default if one exists. To add additional external accounts
    without replacing the existing default for the currency, use the
    [bank
    account](https://stripe.com/docs/api#account_create_bank_account){.perl-module}
    or [card
    creation](https://stripe.com/docs/api#account_create_card){.perl-module}
    APIs.

`individual`

:   Information about the person represented by the account. This field
    is null unless `business_type` is set to `individual`.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`settings`

:   Options for customizing how the account functions within Stripe.

`tos_acceptance`

:   Details on the account\'s acceptance of the [Stripe Services
    Agreement](https://stripe.com/docs/connect/updating-accounts#tos-acceptance){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/accounts/update>

ACCOUNT BANK ACCOUNT
====================

You can create, delete, list, retrieve or update account bank account

create
------

        my $obj = $stripe->account_bank_accounts( create => {
            external_account => "btok_1Le9F12eZvKYlo2CGBPhcUyo", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
object or a hash reference, this will create a Stripe account bank
account and return an
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
object.

Possible parameters are:

`default_for_currency`

:   When set to true, or if this is the first external account added in
    this currency, this account becomes the default external account for
    its currency.

`external_account`

:   **Required**. Either a token, like the ones returned by
    [Stripe.js](https://stripe.com/docs/stripe.js){.perl-module}, or a
    dictionary containing a user's bank account details (with the
    options shown below).

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_bank_accounts/create>

delete
------

        my $obj = $stripe->account_bank_accounts( delete => $args ) || die( $stripe->error );

Provided with a [account bank
account](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
account bank account. It returns the account bank account object that
was deleted with its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_bank_accounts/delete>

list
----

        my $obj = $stripe->account_bank_accounts( list => {
            limit  => "3",
            object => "bank_account", } ) || die( $stripe->error );

Provided with a [account bank
account](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
object, this issue an api call to get the list of all account bank
account.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_bank_accounts/list>

retrieve
--------

        my $obj = $stripe->account_bank_accounts( retrieve => $args ) || die( $stripe->error );

Provided with a [account bank
account](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
object or a hash reference, this will retrieve a Stripe account bank
account and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_bank_accounts/retrieve>

update
------

        my $obj = $stripe->account_bank_accounts( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [account bank
account](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
object or a hash reference, this will update a Stripe account bank
account and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}

Possible parameters are:

`account_holder_name`

:   The name of the person or business that owns the bank account.

`account_holder_type`

:   The type of entity that holds the account. This can be either
    `individual` or `company`.

`account_type`

:   The bank account type. This can only be `checking` or `savings` in
    most countries. In Japan, this can only be `futsu` or `toza`.

`default_for_currency`

:   When set to true, this becomes the default external account for its
    currency.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_bank_accounts/update>

ACCOUNT CARD
============

You can create, delete, list, retrieve or update account card

create
------

        my $obj = $stripe->account_cards( create => {
            external_account => "tok_mastercard_debit", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::ExternalAccount::Card](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}
object or a hash reference, this will create a Stripe account card and
return an
[Net::API::Stripe::Connect::ExternalAccount::Card](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}
object.

Possible parameters are:

`default_for_currency`

:   When set to true, or if this is the first external account added in
    this currency, this account becomes the default external account for
    its currency.

`external_account`

:   **Required**. A token, like the ones returned by
    [Stripe.js](https://stripe.com/docs/stripe.js){.perl-module}. Stripe
    will automatically validate the card.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_cards/create>

delete
------

        my $obj = $stripe->account_cards( delete => $args ) || die( $stripe->error );

Provided with a [account
card](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
account card. It returns the account card object that was deleted with
its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_cards/delete>

list
----

        my $obj = $stripe->account_cards( list => {
            limit  => "3",
            object => "card", } ) || die( $stripe->error );

Provided with a [account
card](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}
object, this issue an api call to get the list of all account card.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_cards/list>

retrieve
--------

        my $obj = $stripe->account_cards( retrieve => $args ) || die( $stripe->error );

Provided with a [account
card](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}
object or a hash reference, this will retrieve a Stripe account card and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_cards/retrieve>

update
------

        my $obj = $stripe->account_cards( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [account
card](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}
object or a hash reference, this will update a Stripe account card and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Card){.perl-module}

Possible parameters are:

`address_city`

:   City/District/Suburb/Town/Village.

`address_country`

:   Billing address country, if provided when creating card.

`address_line1`

:   Address line 1 (Street address/PO Box/Company name).

`address_line2`

:   Address line 2 (Apartment/Suite/Unit/Building).

`address_state`

:   State/County/Province/Region.

`address_zip`

:   ZIP or postal code.

`default_for_currency`

:   When set to true, this becomes the default external account for its
    currency.

`exp_month`

:   Two digit number representing the card's expiration month.

`exp_year`

:   Four digit number representing the card's expiration year.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   Cardholder name.

More information from Stripe api documentation at
<https://stripe.com/docs/api/external_account_cards/update>

ACCOUNT LINK
============

You can create account link

create
------

        my $obj = $stripe->account_links( create => {
            account     => "acct_1032D82eZvKYlo2C",
            refresh_url => "https://example.com/reauth",
            return_url  => "https://example.com/return",
            type        => "account_onboarding", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::Account::Link](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Link){.perl-module}
object or a hash reference, this will create a Stripe account link and
return an
[Net::API::Stripe::Connect::Account::Link](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Link){.perl-module}
object.

Possible parameters are:

`account`

:   **Required**. The identifier of the account to create an account
    link for.

`collect`

:   Which information the platform needs to collect from the user. One
    of `currently_due` or `eventually_due`. Default is `currently_due`.

`refresh_url`

:   **required** The URL the user will be redirected to if the account
    link is expired, has been previously-visited, or is otherwise
    invalid. The URL you specify should attempt to generate a new
    account link with the same parameters used to create the original
    account link, then redirect the user to the new account link\'s URL
    so they can continue with Connect Onboarding. If a new account link
    cannot be generated or the redirect fails you should display a
    useful error to the user.

`return_url`

:   **required** The URL that the user will be redirected to upon
    leaving or completing the linked flow.

`type`

:   **Required**. The type of account link the user is requesting.
    Possible values are `account_onboarding` or `account_update`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/account_links/create>

APPLICATION FEE
===============

You can list or retrieve application fee

list
----

        my $obj = $stripe->application_fees( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [application
fee](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee){.perl-module}
object, this issue an api call to get the list of all application fee.

Possible parameters are:

`charge`

:   Only return application fees for the charge specified by this charge
    ID.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/application_fees/list>

retrieve
--------

        my $obj = $stripe->application_fees( retrieve => $args ) || die( $stripe->error );

Provided with a [application
fee](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee){.perl-module}
object or a hash reference, this will retrieve a Stripe application fee
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/application_fees/retrieve>

APPS SECRET
===========

You can delete, find, list or set apps secret

delete
------

        my $obj = $stripe->apps_secrets( delete => {
            name  => "my-api-key",
            scope =>
            {
                type => "account",
            } } ) || die( $stripe->error );

Provided with a [apps
secret](https://metacpan.org/pod/Net::API::Stripe::Connect::AppsSecret){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
apps secret. It returns the apps secret object that was deleted with its
property `deleted` set to true.

Possible parameters are:

`name`

:   **Required**. A name for the secret that\'s unique within the scope.

`scope`

:   **Required**. Specifies the scoping of the secret. Requests
    originating from UI extensions can only access account-scoped
    secrets or secrets scoped to their own user.

More information from Stripe api documentation at
<https://stripe.com/docs/api/apps/secret_store/delete>

find
----

        my $obj = $stripe->apps_secrets( find => {
            name  => "my-api-key",
            scope =>
            {
                type => "account",
            } } ) || die( $stripe->error );

Provided with a [apps
secret](https://metacpan.org/pod/Net::API::Stripe::Connect::AppsSecret){.perl-module},
or a hash reference, this will issue a find api call.

Returns a secret object.

Possible parameters are:

`name`

:   **Required**. A name for the secret that\'s unique within the scope.

`scope`

:   **Required**. Specifies the scoping of the secret. Requests
    originating from UI extensions can only access account-scoped
    secrets or secrets scoped to their own user.

More information from Stripe api documentation at
<https://stripe.com/docs/api/apps/secret_store/find>

list
----

        my $obj = $stripe->apps_secrets( list => {
            limit => "2",
            scope =>
            {
                type => "account",
            } } ) || die( $stripe->error );

Provided with a [apps
secret](https://metacpan.org/pod/Net::API::Stripe::Connect::AppsSecret){.perl-module}
object, this issue an api call to get the list of all apps secret.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`scope`

:   **Required**. Specifies the scoping of the secret. Requests
    originating from UI extensions can only access account-scoped
    secrets or secrets scoped to their own user.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/apps/secret_store/list>

set
---

        my $obj = $stripe->apps_secrets( set => {
            name    => "my-api-key",
            payload => "secret_key_xxxxxx",
            scope   =>
            {
                type => "account",
            } } ) || die( $stripe->error );

Provided with a [apps
secret](https://metacpan.org/pod/Net::API::Stripe::Connect::AppsSecret){.perl-module},
or a hash reference, this will issue a set api call.

Returns a secret object.

Possible parameters are:

`expires_at`

:   The Unix timestamp for the expiry time of the secret, after which
    the secret deletes.

`name`

:   **Required**. A name for the secret that\'s unique within the scope.

`payload`

:   **Required**. The plaintext secret value to be stored.

`scope`

:   **Required**. Specifies the scoping of the secret. Requests
    originating from UI extensions can only access account-scoped
    secrets or secrets scoped to their own user.

More information from Stripe api documentation at
<https://stripe.com/docs/api/apps/secret_store/set>

BALANCE
=======

You can retrieve balance

retrieve
--------

        my $obj = $stripe->balances( retrieve => $args ) || die( $stripe->error );

Provided with a
[balance](https://metacpan.org/pod/Net::API::Stripe::Balance){.perl-module}
object or a hash reference, this will retrieve a Stripe balance and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Balance){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/balance/balance_retrieve>

BALANCE TRANSACTION
===================

You can **retrieve** or **list** the balance transactions.

list
----

This can take various parameter to influence the list of data returned
by Stripe. It returns a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object of
[Net::API::Stripe::Balance::Transaction](https://metacpan.org/pod/Net::API::Stripe::Balance::Transaction){.perl-module}
objects. Valid parameters are as follows. See Stripe API for more
information: <https://stripe.com/docs/api/balance_transactions/list>

        my $list = $stripe->balance_transactions( 'list' ) || die( $stripe->error );
        while( my $bt = $list->next )
        {
            printf( <<EOT, $bt->id, $bt->amount, $bt->created->iso8601, $bt->currency, $bt->customer->name, $bt->description );
    Id: %s
    Amount: %s
    Created on: $s
    Currency: %s
    Cusomer name: %s
    Description: %s
    EOT
        }

Possible parameters are:

*available\_on*

:   

*created*

:   

*currency*

:   3-letter iso 4217 currency

*ending\_before*

:   Stripe balance transaction id

*limit*

:   Integer

*payout*

:   

*source*

:   

*starting\_after*

:   

*type*

:   Only returns transactions of the given type

retrieve
--------

        my $trans = $stripe->balances( retrieve => 'txn_fake1234567890' ) || die( $stripe->error );

Provided a `balance_transaction` object or an id, this returns a
[Net::API::Stripe::Balance::Transaction](https://metacpan.org/pod/Net::API::Stripe::Balance::Transaction){.perl-module}
object or undef upon error.

BALANCES
========

You an **retrieve** balances.

retrieve
--------

        my $bal = $stripe->balances( 'retrieve' ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Balance](https://metacpan.org/pod/Net::API::Stripe::Balance){.perl-module}
object, or a hash reference, this will retrieve a Stripe balance and
return its
[Net::API::Stripe::Balance](https://metacpan.org/pod/Net::API::Stripe::Balance){.perl-module}
object.

There is no argument.

BANK ACCOUNT
============

You can create, delete, list, retrieve, update or verify bank account

create
------

        my $acct = $stripe->bank_accounts( create => $stripe->bank_account({
            account_holder_name => 'Big Corp, Inc',
            account_holder_type => 'company',
            bank_name => 'Big Bank, Corp'
            country => 'us',
            currency => 'usd',
            # Net::API::Stripe::Customer object
            customer => $customer_object,
            default_for_currency => $stripe->true,
            fingerprint => 'kshfkjhfkjsjdla',
            last4 => 1234,
            metadata => { transaction_id => 2222 },
            routing_number => 123,
            status => 'new',
        })) || die( "Oops: ", $stripe->error );

Provided wuth a bank account object
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
that has its *account* property set, or simply a hash reference this
will create a Stripe bank account and return its object as a
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}

Possible parameters are:

*account*

:   A Stripe account id. This is required.

*external\_account* This is required. Either a token, like the ones returned by Stripe.js, or a hash reference containing a userâs bank account details with the following properties:

:   

    *object* (required)

    :   

    *country* (required)

    :   

    *currency* (required)

    :   

    *account\_holder\_name*

    :   

    *account\_holder\_type*

    :   

    *routing\_number*

    :   

    *account\_number* (required)

    :   

*default\_for\_currency* Boolean

:   

*metadata*

:   An arbitrary hash reference

For more information see Stripe documentation here:
<https://stripe.com/docs/api/external_account_bank_accounts/create>

delete
------

        my $removed_acct = $stripe->bank_accounts( delete => 'ba_fake123456789' ) || die( "Oops: ", $stripe->error );

Provided wuth a bank account object
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
that has its *account* property set, or simply a hash reference this
will remove a Stripe bank account and return its object as a
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}

Possible parameters are:

*id*

:   A Stripe bank account id. This is required.

*account*

:   A Stripe account id. This is required.

For more information see Stripe documentation here:
<https://stripe.com/docs/api/external_account_bank_accounts/delete>

list
----

        my $list = $stripe->bank_accounts( 'list' ) || die( $stripe->error );
        printf( "%d total transaction(s) found\n", $list->count );
        while( my $acct = $list->next )
        {
            ## Do something with this object
        }

Provided wuth a bank account object
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
that has its *account* property set, or simply a hash reference this
will list all Stripe bank accounts and return a list object as a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}

Possible parameters are:

*account*

:   A Stripe account id. This is required.

For more information see Stripe documentation here:
<https://stripe.com/docs/api/external_account_bank_accounts/list>

retrieve
--------

        my $acct = $stripe->bank_accounts( retrieve => 'ba_fake123456789' ) || die( "Oops: ", $stripe->error );

Provided wuth a bank account object
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
that has its *account* property set, or simply a hash reference this
will retrieve a Stripe bank account and return its object as a
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}

Possible parameters are:

*id*

:   A Stripe bank account id. This is required.

*account*

:   A Stripe account id. This is required.

For more information see Stripe documentation here:
<https://stripe.com/docs/api/external_account_bank_accounts/retrieve>

update
------

        my $acct = $stripe->bank_accounts( update => $stripe->bank_account({
            account_holder_name => 'Big Corp, Co., Ltd.',
            default_for_currency => $stripe->false,
        })) || die( "Oops: ", $stripe->error );

        # or passing a hash rather than an object

        my $acct = $stripe->bank_accounts( update => {
            account_holder_name => 'Big Corp, Co., Ltd.',
            default_for_currency => $stripe->false,
        }) || die( "Oops: ", $stripe->error );

Provided wuth a bank account object
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}
that has its *account* property set, or simply a hash reference this
will update a Stripe bank account and return its object as a
[Net::API::Stripe::Connect::ExternalAccount::Bank](https://metacpan.org/pod/Net::API::Stripe::Connect::ExternalAccount::Bank){.perl-module}

Possible parameters are:

*id*

:   A Stripe bank account id. This is required.

*account*

:   A Stripe account id. This is required.

*account\_holder\_name* String

:   

*account\_holder\_type* String

:   

*default\_for\_currency* Boolean

:   

*metadata*

:   An arbitrary hash reference

For more information see Stripe documentation here:
<https://stripe.com/docs/api/external_account_bank_accounts/update>

verify
------

        my $obj = $stripe->bank_accounts( verify => {
            amounts => [qw( 32 45 )], } ) || die( $stripe->error );

Provided with a [bank
account](https://metacpan.org/pod/Net::API::Stripe::Customer::BankAccount){.perl-module},
or a hash reference, this will issue a verify api call.

Returns the bank account object with a `status` of
\<strong\>verified\</strong\>.

Possible parameters are:

`amounts`

:   Two positive integers, in *cents*, equal to the values of the
    microdeposits sent to the bank account.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_bank_accounts/verify>

BILLING PORTAL CONFIGURATION
============================

You can create, list, retrieve or update billing portal configuration

create
------

        my $obj = $stripe->billing_portal_configurations( create => {
            business_profile =>
            {
                privacy_policy_url   => "https://example.com/privacy",
                terms_of_service_url => "https://example.com/terms",
            }
            features         =>
            {
                customer_update =>
                {
                    allowed_updates => [qw( email tax_id )],
                    enabled         => "1",
                }
                invoice_history =>
                {
                    enabled => "1",
                }
            } } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::PortalConfiguration](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalConfiguration){.perl-module}
object or a hash reference, this will create a Stripe billing portal
configuration and return an
[Net::API::Stripe::Billing::PortalConfiguration](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalConfiguration){.perl-module}
object.

Possible parameters are:

`business_profile`

:   **Required**. The business information shown to customers in the
    portal.

`default_return_url`

:   The default URL to redirect customers to when they click on the
    portal\'s link to return to your website. This can be
    [overriden](https://stripe.com/docs/api/customer_portal/sessions/create#create_portal_session-return_url){.perl-module}
    when creating the session.

`features`

:   **Required**. Information about the features available in the
    portal.

`login_page`

:   The hosted login page for this configuration. Learn more about the
    portal login page in our [integration
    docs](https://stripe.com/docs/billing/subscriptions/integrating-customer-portal#share){.perl-module}.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_portal/configurations/create>

list
----

        my $obj = $stripe->billing_portal_configurations( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [billing portal
configuration](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalConfiguration){.perl-module}
object, this issue an api call to get the list of all billing portal
configuration.

Possible parameters are:

`active`

:   Only return configurations that are active or inactive (e.g., pass
    `true` to only list active configurations).

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`is_default`

:   Only return the default or non-default configurations (e.g., pass
    `true` to only list the default configuration).

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_portal/configurations/list>

retrieve
--------

        my $obj = $stripe->billing_portal_configurations( retrieve => $args ) || die( $stripe->error );

Provided with a [billing portal
configuration](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalConfiguration){.perl-module}
object or a hash reference, this will retrieve a Stripe billing portal
configuration and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalConfiguration){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_portal/configurations/retrieve>

update
------

        my $obj = $stripe->billing_portal_configurations( update => {
            business_profile =>
            {
                privacy_policy_url   => "https://example.com/privacy",
                terms_of_service_url => "https://example.com/terms",
            } } ) || die( $stripe->error );

Provided with a [billing portal
configuration](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalConfiguration){.perl-module}
object or a hash reference, this will update a Stripe billing portal
configuration and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalConfiguration){.perl-module}

Possible parameters are:

`active`

:   Whether the configuration is active and can be used to create portal
    sessions.

`business_profile`

:   The business information shown to customers in the portal.

`default_return_url`

:   The default URL to redirect customers to when they click on the
    portal\'s link to return to your website. This can be
    [overriden](https://stripe.com/docs/api/customer_portal/sessions/create#create_portal_session-return_url){.perl-module}
    when creating the session.

`features`

:   Information about the features available in the portal.

`login_page`

:   The hosted login page for this configuration. Learn more about the
    portal login page in our [integration
    docs](https://stripe.com/docs/billing/subscriptions/integrating-customer-portal#share){.perl-module}.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_portal/configurations/update>

BILLING PORTAL SESSION
======================

You can create billing portal session

create
------

        my $obj = $stripe->billing_portal_sessions( create => {
            customer   => "cus_AJ78ZaALpqgiuZ",
            return_url => "https://example.com/account", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::PortalSession](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalSession){.perl-module}
object or a hash reference, this will create a Stripe billing portal
session and return an
[Net::API::Stripe::Billing::PortalSession](https://metacpan.org/pod/Net::API::Stripe::Billing::PortalSession){.perl-module}
object.

Possible parameters are:

`configuration`

:   The ID of an existing
    [configuration](https://stripe.com/docs/api/customer_portal/configuration){.perl-module}
    to use for this session, describing its functionality and features.
    If not specified, the session uses the default configuration.

`customer`

:   **Required**. The ID of an existing customer.

`locale`

:   The IETF language tag of the locale Customer Portal is displayed in.
    If blank or auto, the customer's `preferred_locales` or browser's
    locale is used.

`on_behalf_of`

:   The `on_behalf_of` account to use for this session. When specified,
    only subscriptions and invoices with this `on_behalf_of` account
    appear in the portal. For more information, see the
    [docs](https://stripe.com/docs/connect/charges-transfers#on-behalf-of){.perl-module}.
    Use the [Accounts
    API](https://stripe.com/docs/api/accounts/object#account_object-settings-branding){.perl-module}
    to modify the `on_behalf_of` account\'s branding settings, which the
    portal displays.

`return_url`

:   The default URL to redirect customers to when they click on the
    portal\'s link to return to your website.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_portal/sessions/create>

CAPABILITY
==========

You can list, retrieve or update capability

list
----

        my $obj = $stripe->capabilitys( list => $args ) || die( $stripe->error );

Provided with a
[capability](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Capability){.perl-module}
object, this issue an api call to get the list of all capability.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/capabilities/list>

retrieve
--------

        my $obj = $stripe->capabilitys( retrieve => $args ) || die( $stripe->error );

Provided with a
[capability](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Capability){.perl-module}
object or a hash reference, this will retrieve a Stripe capability and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Capability){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/capabilities/retrieve>

update
------

        my $obj = $stripe->capabilitys( update => {
            requested => "1", } ) || die( $stripe->error );

Provided with a
[capability](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Capability){.perl-module}
object or a hash reference, this will update a Stripe capability and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::Capability){.perl-module}

Possible parameters are:

`requested`

:   Passing true requests the capability for the account, if it is not
    already requested. A requested capability may not immediately become
    active. Any requirements to activate the capability are returned in
    the `requirements` arrays.

More information from Stripe api documentation at
<https://stripe.com/docs/api/capabilities/update>

CARD
====

You can **create**, **retrieve**, **update**, **delete** or **list**
cards.

create
------

Provided a customer object
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
or a card object
[Net::API::Stripe::Payment::Card](https://metacpan.org/pod/Net::API::Stripe::Payment::Card){.perl-module}
that has its *customer* property set, or simply a hash reference this
will create a Stripe card and return its object as a
[Net::API::Stripe::Payment::Card](https://metacpan.org/pod/Net::API::Stripe::Payment::Card){.perl-module}

Possible parameters are:

*id*

:   A customer id

*source*

:   A hash reference with the following properties: object number
    exp\_month exp\_year cvc currency name metadata
    default\_for\_currency address\_line1 address\_line2 address\_city
    address\_state address\_zip address\_country

*metadata* An arbitrary hash reference

:   

delete
------

Provided with a
[customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
or a
[card](https://metacpan.org/pod/Net::API::Stripe::Payment::Card){.perl-module}
object, or a hash reference, this will issue an api call to Stripe to
remove the customer\'s card. It returns the card object that was deleted
with its property *deleted* set to true.

Possible parameters are:

*id*

:   Stripe customer id

*card\_id*

:   Stripe card id

list
----

Provided with a
[customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object, this issue an api call to get the list of all cards for a given
customer.

Possible parameters are:

*id*

:   Stripe customer id

*ending\_before*

:   A card id

*limit* Integer

:   

*starting\_after* A card id

:   

For more information, see Stripe api documentation here:
<https://stripe.com/docs/api/cards/list>

retrieve
--------

Provided a customer object
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
or a card object
[Net::API::Stripe::Payment::Card](https://metacpan.org/pod/Net::API::Stripe::Payment::Card){.perl-module}
that has its *customer* property set, or simply a hash reference this
will retrieve the customer card information as a
[Net::API::Stripe::Payment::Card](https://metacpan.org/pod/Net::API::Stripe::Payment::Card){.perl-module}
object

Possible parameters are:

*id*

:   Stripe card id

*customer*

:   Stripe customer id

update
------

Provided a customer object
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
or a card object
[Net::API::Stripe::Payment::Card](https://metacpan.org/pod/Net::API::Stripe::Payment::Card){.perl-module}
that has its *customer* property set, or simply a hash reference this
will update the customer card information, but what can be updated is
limited by Stripe and it is typically the expiration date or postal
address

Possible parameters are:

*id*

:   Stripe card id

*customer*

:   Stripe customer id

*address\_city* City

:   

*address\_country*

:   Country as 2-letters ISO 3166 country code

*address\_line1*

:   Address first line

*address\_line2*

:   Address second line

*address\_state*

:   State / region

*address\_zip*

:   Postal code

*exp\_month*

:   Expiration month

*exp\_year*

:   Expiration year

*metadata*

:   Arbitrary hash reference

*name*

:   Name for this credit card

CASH BALANCE
============

You can retrieve or update cash balance

retrieve
--------

        my $obj = $stripe->cash_balances( retrieve => $args ) || die( $stripe->error );

Provided with a [cash
balance](https://metacpan.org/pod/Net::API::Stripe::Cash::Balance){.perl-module}
object or a hash reference, this will retrieve a Stripe cash balance and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Cash::Balance){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/cash_balance/retrieve>

update
------

        my $obj = $stripe->cash_balances( update => {
            settings =>
            {
                reconciliation_mode => "manual",
            } } ) || die( $stripe->error );

Provided with a [cash
balance](https://metacpan.org/pod/Net::API::Stripe::Cash::Balance){.perl-module}
object or a hash reference, this will update a Stripe cash balance and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Cash::Balance){.perl-module}

Possible parameters are:

`settings`

:   A hash of settings for this cash balance.

More information from Stripe api documentation at
<https://stripe.com/docs/api/cash_balance/update>

CHARGE
======

You can **create**, **retrieve**, **update**, **capture** or **list**
charges.

capture
-------

Provided with a
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}
object or a hash reference, this will capture a Stripe charge and return
its corresponding charge object
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}

Possible parameters are:

*id* A Stripe charge id. This is required

:   

*amount* Integer

:   

*application\_fee\_amount* Integer

:   

*destination* A hash reference with one property: *amount*

:   

*receipt\_email* E-mail address

:   

*statement\_descriptor* Text

:   

*statement\_descriptor\_suffix* String

:   

*transfer\_data* A hash reference with one property: *amount*

:   

*transfer\_group* Text

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/charges/capture>

create
------

Provided with a
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}
object or a hash reference, this will create a Stripe charge and return
a charge object
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}

Possible parameters are:

*amount* Amount as integer. This is required

:   

*currency* A 3-letter iso 4217 code such sa `jpy` for Japanese Yen

:   

*application\_fee\_amount* Integer

:   

*capture* Boolean

:   

*customer* A customer id

:   

*description* An arbitrary text

:   

*destination* A hash with properties account and amount.

:   

*metadata* Arbitrary hash reference

:   

*on\_behalf\_of* Stripe account id

:   

*receipt\_email* E-mail address

:   

*shipping* Shipping address as a hash reference with the following properties: address name carrier phone tracking\_number. See also [Net::API::Stripe::Shipping](https://metacpan.org/pod/Net::API::Stripe::Shipping){.perl-module}

:   

*source* A source id

:   

*statement\_descriptor* Text

:   

*statement\_descriptor\_suffix* Text

:   

*transfer\_data* A date

:   

*transfer\_group* Text

:   

*idempotency* identifier

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/charges/create>

list
----

This will list all the charges for a given customer and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*created* A date that can also be expressed as a unix timestamp

:   

*customer* A Stripe customer id

:   

*ending\_before* A Stripe charge id

:   

*limit* Integer

:   

*payment\_intent* A payment intent Stripe id

:   

*source* A source Stripe id

:   

*starting\_after* A Stripe charge id

:   

*transfer\_group* Text

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/charges/list>

retrieve
--------

Provided with a
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}
object or a hash reference, this will retrieve a Stripe charge and
return its corresponding charge object
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}

Possible parameters are:

*id* Stripe charge id. This is required

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/charges/retrieve>

search
------

        my $obj = $stripe->charges( search => {
            query => "amount>999 AND metadata['order_id']:'6735'", } ) || die( $stripe->error );

Provided with a
[charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module},
or a hash reference, this will issue a search api call.

A dictionary with a `data` property that contains an array of up to
`limit` charges. If no objects match the query, the resulting array will
be empty. See the related guide on [expanding properties in
lists](https://stripe.com/docs/expand#lists){.perl-module}.

Possible parameters are:

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`page`

:   A cursor for pagination across multiple pages of results. Don\'t
    include this parameter on the first call. Use the next\_page value
    returned in a previous response to request subsequent results.

`query`

:   **Required**. The search query string. See [search query
    language](https://stripe.com/docs/search#search-query-language){.perl-module}
    and the list of supported [query fields for
    charges](https://stripe.com/docs/search#query-fields-for-charges){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/charges/search>

update
------

Provided with a
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}
object or a hash reference, this will update a Stripe charge and return
its corresponding charge object
[Net::API::Stripe::Charge](https://metacpan.org/pod/Net::API::Stripe::Charge){.perl-module}

Possible parameters are:

*id* A Stripe charge id. This is required

:   

*customer* A customer id

:   

*description* An arbitrary text

:   

*fraud\_details* A hash with one property: *user\_report*

:   

*metadata* Arbitrary hash reference

:   

*receipt\_email* E-mail address

:   

*shipping* Shipping address as a hash reference with the following properties: address name carrier phone tracking\_number. See also [Net::API::Stripe::Shipping](https://metacpan.org/pod/Net::API::Stripe::Shipping){.perl-module}

:   

*transfer\_group* Text

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/charges/update>

CHECKOUT SESSION
================

You can create, expire, items, list or retrieve checkout session

create
------

        my $obj = $stripe->checkout_sessions( create => {
            cancel_url  => "https://example.com/cancel",
            line_items  => [,
                price    => "price_H5ggYwtDq4fbrJ",
                quantity => "2",
            ],
            mode        => "payment",
            success_url => "https://example.com/success", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Checkout::Session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object or a hash reference, this will create a Stripe checkout session
and return an
[Net::API::Stripe::Checkout::Session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object.

Possible parameters are:

`after_expiration`

:   Configure actions after a Checkout Session has expired.

`allow_promotion_codes`

:   Enables user redeemable promotion codes.

`automatic_tax`

:   Settings for automatic tax lookup for this session and resulting
    payments, invoices, and subscriptions.

`billing_address_collection`

:   Specify whether Checkout should collect the customer\'s billing
    address.

`cancel_url`

:   **Required**. The URL the customer will be directed to if they
    decide to cancel payment and return to your website.

`client_reference_id`

:   A unique string to reference the Checkout Session. This can be a
    customer ID, a cart ID, or similar, and can be used to reconcile the
    session with your internal systems.

`consent_collection`

:   Configure fields for the Checkout Session to gather active consent
    from customers.

`currency`

:   Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`customer`

:   ID of an existing Customer, if one exists. In `payment` mode, the
    customer's most recent card payment method will be used to prefill
    the email, name, card details, and billing address on the Checkout
    page. In `subscription` mode, the customer's [default payment
    method](https://stripe.com/docs/api/customers/update#update_customer-invoice_settings-default_payment_method){.perl-module}
    will be used if it's a card, and otherwise the most recent card will
    be used. A valid billing address, billing name and billing email are
    required on the payment method for Checkout to prefill the
    customer\'s card details.

    If the Customer already has a valid
    [email](https://stripe.com/docs/api/customers/object#customer_object-email){.perl-module}
    set, the email will be prefilled and not editable in Checkout. If
    the Customer does not have a valid `email`, Checkout will set the
    email entered during the session on the Customer.

    If blank for Checkout Sessions in `payment` or `subscription` mode,
    Checkout will create a new Customer object based on information
    provided during the payment flow.

    You can set
    [`payment_intent_data.setup_future_usage`](https://stripe.com/docs/api/checkout/sessions/create#create_checkout_session-payment_intent_data-setup_future_usage){.perl-module}
    to have Checkout automatically attach the payment method to the
    Customer you pass in for future reuse.

`customer_creation`

:   Configure whether a Checkout Session creates a
    [Customer](https://stripe.com/docs/api/customers){.perl-module}
    during Session confirmation.

    When a Customer is not created, you can still retrieve email,
    address, and other customer data entered in Checkout with
    [customer\_details](https://stripe.com/docs/api/checkout/sessions/object#checkout_session_object-customer_details){.perl-module}.

    Sessions that don\'t create Customers instead create [Guest
    Customers](https://support.stripe.com/questions/guest-customer-faq){.perl-module}
    in the Dashboard. Promotion codes limited to first time customers
    will return invalid for these Sessions.

    Can only be set in `payment` and `setup` mode.

`customer_email`

:   If provided, this value will be used when the Customer object is
    created. If not provided, customers will be asked to enter their
    email address. Use this parameter to prefill customer data if you
    already have an email on file. To access information about the
    customer once a session is complete, use the `customer` field.

`customer_update`

:   Controls what fields on Customer can be updated by the Checkout
    Session. Can only be provided when `customer` is provided.

`discounts`

:   The coupon or promotion code to apply to this Session. Currently,
    only up to one may be specified.

`expires_at`

:   The Epoch time in seconds at which the Checkout Session will expire.
    It can be anywhere from 30 minutes to 24 hours after Checkout
    Session creation. By default, this value is 24 hours from creation.

`line_items`

:   A list of items the customer is purchasing. Use this parameter to
    pass one-time or recurring
    [Prices](https://stripe.com/docs/api/prices){.perl-module}.

    For `payment` mode, there is a maximum of 100 line items, however it
    is recommended to consolidate line items if there are more than a
    few dozen.

    For `subscription` mode, there is a maximum of 20 line items with
    recurring Prices and 20 line items with one-time Prices. Line items
    with one-time Prices will be on the initial invoice only.

`locale`

:   The IETF language tag of the locale Checkout is displayed in. If
    blank or `auto`, the browser\'s locale is used.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`mode`

:   **required conditionally** The mode of the Checkout Session.
    Required when using prices or `setup` mode. Pass `subscription` if
    the Checkout Session includes at least one recurring item.

`payment_intent_data`

:   A subset of parameters to be passed to PaymentIntent creation for
    Checkout Sessions in `payment` mode.

`payment_method_collection`

:   Specify whether Checkout should collect a payment method. When set
    to `if_required`, Checkout will not collect a payment method when
    the total due for the session is 0. This may occur if the Checkout
    Session includes a free trial or a discount.

    Can only be set in `subscription` mode.

    If you\'d like information on how to collect a payment method
    outside of Checkout, read the guide on configuring [subscriptions
    with a free
    trial](https://stripe.com/docs/payments/checkout/free-trials){.perl-module}.

`payment_method_options`

:   Payment-method-specific configuration.

`payment_method_types`

:   A list of the types of payment methods (e.g., `card`) this Checkout
    Session can accept.

    Do not include this attribute if you prefer to manage your payment
    methods from the [Stripe
    Dashboard](https://dashboard.stripe.com/settings/payment_methods){.perl-module}.

    Read more about the supported payment methods and their requirements
    in our [payment method details
    guide](https://stripe.com/docs/payments/checkout/payment-methods){.perl-module}.

    If multiple payment methods are passed, Checkout will dynamically
    reorder them to prioritize the most relevant payment methods based
    on the customer\'s location and other characteristics.

`phone_number_collection`

:   Controls phone number collection settings for the session.

    We recommend that you review your privacy policy and check with your
    legal contacts before using this feature. Learn more about
    [collecting phone numbers with
    Checkout](https://stripe.com/docs/payments/checkout/phone-numbers){.perl-module}.

`setup_intent_data`

:   A subset of parameters to be passed to SetupIntent creation for
    Checkout Sessions in `setup` mode.

`shipping_address_collection`

:   When set, provides configuration for Checkout to collect a shipping
    address from a customer.

`shipping_options`

:   The shipping rate options to apply to this Session.

`submit_type`

:   Describes the type of transaction being performed by Checkout in
    order to customize relevant text on the page, such as the submit
    button. `submit_type` can only be specified on Checkout Sessions in
    `payment` mode, but not Checkout Sessions in `subscription` or
    `setup` mode.

`subscription_data`

:   A subset of parameters to be passed to subscription creation for
    Checkout Sessions in `subscription` mode.

`success_url`

:   **Required**. The URL to which Stripe should send customers when
    payment or setup is complete. If you'd like to use information from
    the successful Checkout Session on your page, read the guide on
    [customizing your success
    page](https://stripe.com/docs/payments/checkout/custom-success-page){.perl-module}.

`tax_id_collection`

:   Controls tax ID collection settings for the session.

More information from Stripe api documentation at
<https://stripe.com/docs/api/checkout/sessions/create>

expire
------

        my $obj = $stripe->checkout_sessions( expire => $args ) || die( $stripe->error );

Provided with a [checkout
session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module},
or a hash reference, this will issue a expire api call.

Returns a Session object if the expiration succeeded. Returns an error
if the Session has already expired or isn't in an expireable state.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/checkout/sessions/expire>

items
-----

        my $obj = $stripe->checkout_sessions( items => $args ) || die( $stripe->error );

Provided with a [checkout
session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module},
or a hash reference, this will issue a items api call.

A dictionary with a `data` property that contains an array of up to
`limit` Checkout Session line items, starting after Line Item
`starting_after`. Each entry in the array is a separate Line Item
object. If no more line items are available, the resulting array will be
empty.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/checkout/sessions/line_items>

list
----

        my $obj = $stripe->checkout_sessions( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [checkout
session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object, this issue an api call to get the list of all checkout session.

Possible parameters are:

`customer`

:   Only return the Checkout Sessions for the Customer specified.

`customer_details`

:   Only return the Checkout Sessions for the Customer details
    specified.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`payment_intent`

:   Only return the Checkout Session for the PaymentIntent specified.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`subscription`

:   Only return the Checkout Session for the subscription specified.

More information from Stripe api documentation at
<https://stripe.com/docs/api/checkout/sessions/list>

retrieve
--------

        my $obj = $stripe->checkout_sessions( retrieve => $args ) || die( $stripe->error );

Provided with a [checkout
session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object or a hash reference, this will retrieve a Stripe checkout session
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/checkout/sessions/retrieve>

CHECKOUT SESSIONS
=================

You can **create** or **retrieve** checkout sessions

**create** {#createcreate}
----------

Provided with a
[Net::API::Stripe::Checkout::Session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object or an hash reference of parameters and this will create a Stripe
checkout session and return a
[Net::API::Stripe::Checkout::Session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object.

Possible parameters are:

*cancel\_url* URL

:   

*payment\_method\_types* String. One of card or ideal

:   

*success\_url* URL

:   

*billing\_address\_collection* String. One of auto or required

:   

*client\_reference\_id* String

:   

*customer* A Stripe customer id

:   

*customer\_email* String

:   

*line\_items* An array of hash reference with the following properties: amount currency name quantity description images

:   

*locale* a 2-letter iso 639, such as `fr` or `ja` or `local`

:   

*mode* String. One of setup or subscription

:   

*payment\_intent\_data* An hash reference with the following properties: application\_fee\_amount capture\_method description metadata on\_behalf\_of receipt\_email setup\_future\_usage

:   

*setup\_intent\_data* An hash reference with the following properties: description metadata on\_behalf\_of

:   

*submit\_type* String. One of auto, book, donate or pay

:   

*subscription\_data* An hash reference with the following properties: items application\_fee\_percent metadata trial\_end trial\_from\_plan trial\_period\_days

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/checkout/sessions/create>

**list** {#listlist}
--------

Provided with an hash reference of parameters, this will get the list of
checkout sessions and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*payment\_intent* A Stripe payment intent id

:   

*subscription* A Stripe subscription id

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/checkout/sessions/list>

**retrieve** {#retrieveretrieve}
------------

Provided with a
[Net::API::Stripe::Checkout::Session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object or an hash reference of parameters and this will retrieve a
Stripe checkout session and return a
[Net::API::Stripe::Checkout::Session](https://metacpan.org/pod/Net::API::Stripe::Checkout::Session){.perl-module}
object.

Possible parameters are:

*id* A Stripe checkout session

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/checkout/sessions/retrieve>

COUNTRY SPEC
============

You can list or retrieve country spec

list {#list-12}
----

        my $obj = $stripe->country_specs( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [country
spec](https://metacpan.org/pod/Net::API::Stripe::Connect::CountrySpec){.perl-module}
object, this issue an api call to get the list of all country spec.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/country_specs/list>

retrieve {#retrieve-14}
--------

        my $obj = $stripe->country_specs( retrieve => $args ) || die( $stripe->error );

Provided with a [country
spec](https://metacpan.org/pod/Net::API::Stripe::Connect::CountrySpec){.perl-module}
object or a hash reference, this will retrieve a Stripe country spec and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::CountrySpec){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/country_specs/retrieve>

COUPON
======

You can create, delete, list, retrieve or update coupon

create {#create-10}
------

        my $obj = $stripe->coupons( create => {
            duration           => "repeating",
            duration_in_months => "3",
            percent_off        => "25.5", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object or a hash reference, this will create a Stripe coupon and return
an
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object.

Possible parameters are:

`amount_off`

:   A positive integer representing the amount to subtract from an
    invoice total (required if `percent_off` is not passed).

`applies_to`

:   A hash containing directions for what this Coupon will apply
    discounts to.

`currency`

:   Three-letter [ISO code for the
    currency](https://stripe.com/docs/currencies){.perl-module} of the
    `amount_off` parameter (required if `amount_off` is passed).

`currency_options`

:   Coupons defined in each available currency option (only supported if
    `amount_off` is passed). Each key must be a three-letter [ISO
    currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module}
    and a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}. For
    example, to define your coupon in `eur`, pass the fields below in
    the `eur` key of `currency_options`.

`duration`

:   Specifies how long the discount will be in effect if used on a
    subscription. Can be `forever`, `once`, or `repeating`. Defaults to
    `once`.

`duration_in_months`

:   Required only if `duration` is `repeating`, in which case it must be
    a positive integer that specifies the number of months the discount
    will be in effect.

`id`

:   Unique string of your choice that will be used to identify this
    coupon when applying it to a customer. If you don\'t want to specify
    a particular code, you can leave the ID blank and we\'ll generate a
    random code for you.

`max_redemptions`

:   A positive integer specifying the number of times the coupon can be
    redeemed before it\'s no longer valid. For example, you might have a
    50% off coupon that the first 20 readers of your blog can use.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   Name of the coupon displayed to customers on, for instance invoices,
    or receipts. By default the `id` is shown if `name` is not set.

`percent_off`

:   A positive float larger than 0, and smaller or equal to 100, that
    represents the discount the coupon will apply (required if
    `amount_off` is not passed).

`redeem_by`

:   Unix timestamp specifying the last time at which the coupon can be
    redeemed. After the redeem\_by date, the coupon can no longer be
    applied to new customers.

More information from Stripe api documentation at
<https://stripe.com/docs/api/coupons/create>

delete
------

        my $obj = $stripe->coupons( delete => $args ) || die( $stripe->error );

Provided with a
[coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
coupon. It returns the coupon object that was deleted with its property
`deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/coupons/delete>

list {#list-13}
----

        my $obj = $stripe->coupons( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object, this issue an api call to get the list of all coupon.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/coupons/list>

retrieve {#retrieve-15}
--------

        my $obj = $stripe->coupons( retrieve => $args ) || die( $stripe->error );

Provided with a
[coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object or a hash reference, this will retrieve a Stripe coupon and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/coupons/retrieve>

update
------

        my $obj = $stripe->coupons( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object or a hash reference, this will update a Stripe coupon and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}

Possible parameters are:

`currency_options`

:   Coupons defined in each available currency option (only supported if
    the coupon is amount-based). Each key must be a three-letter [ISO
    currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module}
    and a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}. For
    example, to define your coupon in `eur`, pass the fields below in
    the `eur` key of `currency_options`.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   Name of the coupon displayed to customers on, for instance invoices,
    or receipts. By default the `id` is shown if `name` is not set.

More information from Stripe api documentation at
<https://stripe.com/docs/api/coupons/update>

COUPONS
=======

You can **create**, **retrieve**, **update**, **delete** or **list**
coupons.

create {#create-11}
------

Provided with a
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object or a hash reference, this will create a Stripe coupon and return
a coupon object
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}

Possible parameters are:

*duration* String that can be forever, once or repeating

:   

*amount\_off* Integer

:   

*currency* Three-letters iso 4217 currency code such as `jpy` for Japanese Yen

:   

*duration\_in\_months* Integer

:   

*id* A Coupon id, which is also the coupon code, so you are encouraged to create it

:   

*max\_redemptions* Integer

:   

*metadata* Arbitrary hash reference

:   

*name* String

:   

*percent\_off* Percentage such as \> 0 and \<= 100

:   

*redeem\_by* Date

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/coupons/create>

delete
------

Provided with a
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object or a hash reference, this will remove the Stripe coupon and
return a coupon object
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
with the property *deleted* set to true.

For more information see Stripe documentation here:
<https://stripe.com/docs/api/coupons/delete>

list {#list-14}
----

This will list all the coupons and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*created* A date that can also be expressed as a unix timestamp

:   

*customer* A Stripe customer id

:   

*ending\_before* A Stripe charge id

:   

*limit* Integer

:   

*starting\_after* A Stripe charge id

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/coupons/list>

retrieve {#retrieve-16}
--------

Provided with a
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object or a hash reference, this will retrieve a Stripe coupon and
return a coupon object
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}.

Possible parameters are:

*id* A Stripe coupon id

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/coupons/retrieve>

update
------

Provided with a
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}
object or a hash reference, this will update a Stripe coupon and return
a coupon object
[Net::API::Stripe::Billing::Coupon](https://metacpan.org/pod/Net::API::Stripe::Billing::Coupon){.perl-module}.

Possible parameters are:

*id* A Stripe coupon id

:   

*metadata* Arbitrary hash reference

:   

*name* String

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/coupons/update>

CREDIT NOTE
===========

You can create, list, preview, retrieve, update or void credit note

create {#create-12}
------

        my $obj = $stripe->credit_notes( create => {
            invoice => "in_1Le9F22eZvKYlo2C5gZ5i57e",
            lines   => [,
                invoice_line_item => "il_1Le9F22eZvKYlo2C8u1PfwX0",
                quantity          => "1",
                type              => "invoice_line_item",
            ], } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this will create a Stripe credit note and
return an
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object.

Possible parameters are:

`amount`

:   The integer amount in JPY representing the total amount of the
    credit note.

`credit_amount`

:   The integer amount in JPY representing the amount to credit the
    customer\'s balance, which will be automatically applied to their
    next invoice.

`invoice`

:   **Required**. ID of the invoice.

`lines`

:   Line items that make up the credit note.

`memo`

:   The credit note\'s memo appears on the credit note PDF.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`out_of_band_amount`

:   The integer amount in JPY representing the amount that is credited
    outside of Stripe.

`reason`

:   Reason for issuing this credit note, one of `duplicate`,
    `fraudulent`, `order_change`, or `product_unsatisfactory`

`refund`

:   ID of an existing refund to link this credit note to.

`refund_amount`

:   The integer amount in JPY representing the amount to refund. If set,
    a refund will be created for the charge associated with the invoice.

More information from Stripe api documentation at
<https://stripe.com/docs/api/credit_notes/create>

list {#list-15}
----

        my $obj = $stripe->credit_notes( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [credit
note](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object, this issue an api call to get the list of all credit note.

Possible parameters are:

`customer`

:   Only return credit notes for the customer specified by this customer
    ID.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`invoice`

:   Only return credit notes for the invoice specified by this invoice
    ID.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/credit_notes/list>

preview
-------

        my $obj = $stripe->credit_notes( preview => {
            invoice => "in_1Le9F22eZvKYlo2C5gZ5i57e",
            lines   => [,
                invoice_line_item => "il_1Le9F22eZvKYlo2C8u1PfwX0",
                quantity          => "1",
                type              => "invoice_line_item",
            ], } ) || die( $stripe->error );

Provided with a [credit
note](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module},
or a hash reference, this will issue a preview api call.

Returns a credit note object.

Possible parameters are:

`amount`

:   The integer amount in JPY representing the total amount of the
    credit note.

`credit_amount`

:   The integer amount in JPY representing the amount to credit the
    customer\'s balance, which will be automatically applied to their
    next invoice.

`invoice`

:   **Required**. ID of the invoice.

`lines`

:   Line items that make up the credit note.

`memo`

:   The credit note\'s memo appears on the credit note PDF.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`out_of_band_amount`

:   The integer amount in JPY representing the amount that is credited
    outside of Stripe.

`reason`

:   Reason for issuing this credit note, one of `duplicate`,
    `fraudulent`, `order_change`, or `product_unsatisfactory`

`refund`

:   ID of an existing refund to link this credit note to.

`refund_amount`

:   The integer amount in JPY representing the amount to refund. If set,
    a refund will be created for the charge associated with the invoice.

More information from Stripe api documentation at
<https://stripe.com/docs/api/credit_notes/preview>

retrieve {#retrieve-17}
--------

        my $obj = $stripe->credit_notes( retrieve => $args ) || die( $stripe->error );

Provided with a [credit
note](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this will retrieve a Stripe credit note and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/credit_notes/retrieve>

update
------

        my $obj = $stripe->credit_notes( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [credit
note](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this will update a Stripe credit note and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}

Possible parameters are:

`memo`

:   Credit note memo.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/credit_notes/update>

void
----

        my $obj = $stripe->credit_notes( void => $args ) || die( $stripe->error );

Provided with a [credit
note](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module},
or a hash reference, this will issue a void api call.

Returns the voided credit note object if the call succeeded.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/credit_notes/void>

CREDIT NOTE LINE ITEM
=====================

You can list credit note line item

list {#list-16}
----

        my $obj = $stripe->credit_note_line_items( list => {
            invoice => "in_1Le9F22eZvKYlo2C5gZ5i57e",
            lines   => [,
                invoice_line_item => "il_1Le9F22eZvKYlo2C8u1PfwX0",
                quantity          => "1",
                type              => "invoice_line_item",
            ], } ) || die( $stripe->error );

Provided with a [credit note line
item](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote::LineItem){.perl-module}
object, this issue an api call to get the list of all credit note line
item.

Possible parameters are:

`amount`

:   The integer amount in JPY representing the total amount of the
    credit note.

`credit_amount`

:   The integer amount in JPY representing the amount to credit the
    customer\'s balance, which will be automatically applied to their
    next invoice.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`invoice`

:   **Required**. ID of the invoice.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`lines`

:   Line items that make up the credit note.

`memo`

:   The credit note\'s memo appears on the credit note PDF.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`out_of_band_amount`

:   The integer amount in JPY representing the amount that is credited
    outside of Stripe.

`reason`

:   Reason for issuing this credit note, one of `duplicate`,
    `fraudulent`, `order_change`, or `product_unsatisfactory`

`refund`

:   ID of an existing refund to link this credit note to.

`refund_amount`

:   The integer amount in JPY representing the amount to refund. If set,
    a refund will be created for the charge associated with the invoice.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/credit_notes/preview_lines>

CREDIT NOTES
============

You can **preview**, **create**, **lines**, **lines\_preview**,
**retrieve**, **update**, **void** or **list** credit notes.

create {#create-13}
------

Provided with a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this will create a Stripe credit note and
return a credit note
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object.

Possible parameters are:

*invoice* A Stripe invoice id. This is required.

:   

*amount* Integer

:   

*credit\_amount* Integer

:   

*lines* An array of hash with properties: amount description invoice\_line\_item quantity tax\_rates type unit\_amount unit\_amount\_decimal

:   

*memo* Text

:   

*metadata* Arbitrary hash reference

:   

*out\_of\_band\_amount* Integer

:   

*reason* Text

:   

*refund* A Stripe refund id

:   

*refund\_amount* Integer

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/credit_notes/create>

lines
-----

Provided with a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this gets the list of all the credit note
line items and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*id* A Stripe credit note id. This is required.

:   

*ending\_before* A Stripe credit note id.

:   

*limit* Integer

:   

*starting\_after* A Stripe credit note id.

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/credit_notes/lines>

lines\_preview
--------------

Provided with a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this gets the list of all the credit note
preview line items and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*amount* Integer

:   

*credit\_amount* Integer

:   

*ending\_before* A Stripe credit note id.

:   

*invoice* A Stripe invoice id. This is required.

:   

*limit* Integer

:   

*lines* An array of hash with properties: amount description invoice\_line\_item quantity tax\_rates type unit\_amount unit\_amount\_decimal

:   

*memo* Text

:   

*metadata* Arbitrary hash reference

:   

*out\_of\_band\_amount* Integer

:   

*reason* Text

:   

*refund* A Stripe refund id

:   

*refund\_amount* Integer

:   

*starting\_after* A Stripe credit note id.

:   

For more information see Stripe documentation here:
<https://stripe.com/docs/api/credit_notes/lines>

list {#list-17}
----

Given a set of optional parameters, this get the list of credit notes
and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

*created* Date or unix timestamp

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/credit_notes/list>

preview
-------

Provided with a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
or a hash reference, this will create a Stripe credit note preview and
return a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object.

*invoice* A Stripe invoice id. This is required.

:   

*amount* Integer

:   

*credit\_amount* Integer

:   

*lines* An array of hash with properties: amount description invoice\_line\_item quantity tax\_rates type unit\_amount unit\_amount\_decimal

:   

*memo* Text

:   

*metadata* Arbitrary hash reference

:   

*out\_of\_band\_amount* Integer

:   

*reason* Text

:   

*refund* A Stripe refund id

:   

*refund\_amount* Integer

:   

retrieve {#retrieve-18}
--------

Provided with a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this will retrieve the Stripe credit note
and return a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object

Possible parameters are:

*id* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/credit_notes/retrieve>

update
------

Provided with a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this will update a Stripe credit note and
return a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object

Possible parameters are:

*id* A Stripe credit note id. This is required

:   

*memo* Text

:   

*metadata* Arbitrary hash reference

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/credit_notes/update>

void
----

Provided with a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object or a hash reference, this will void a Stripe credit note and
return a
[Net::API::Stripe::Billing::CreditNote](https://metacpan.org/pod/Net::API::Stripe::Billing::CreditNote){.perl-module}
object

Possible parameters are:

*id* A Stripe credit note id. This is required

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/credit_notes/void>

CUSTOMER
========

You can create, delete, list, retrieve, search or update customer

create {#create-14}
------

        my $obj = $stripe->customers( create => {
            description => "My First Test Customer (created for API docs at https://www.stripe.com/docs/api)", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a hash reference, this will create a Stripe customer and
return an
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object.

Possible parameters are:

`address`

:   The customer\'s address.

`balance`

:   An integer amount in JPY that represents the customer\'s current
    balance, which affect the customer\'s future invoices. A negative
    amount represents a credit that decreases the amount due on an
    invoice; a positive amount increases the amount due on an invoice.

`cash_balance`

:   Balance information and default balance settings for this customer.

`coupon`

:   If you provide a coupon code, the customer will have a discount
    applied on all recurring charges. Charges you create through the API
    will not have the discount.

`description`

:   An arbitrary string that you can attach to a customer object. It is
    displayed alongside the customer in the dashboard.

`email`

:   Customer\'s email address. It\'s displayed alongside the customer in
    your dashboard and can be useful for searching and tracking. This
    may be up to *512 characters*.

`invoice_prefix`

:   The prefix for the customer used to generate unique invoice numbers.
    Must be 3--12 uppercase letters or numbers.

`invoice_settings`

:   Default invoice settings for this customer.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   The customer\'s full name or business name.

`next_invoice_sequence`

:   The sequence to be used on the customer\'s next invoice. Defaults
    to 1.

`payment_method`

:   The ID of the PaymentMethod to attach to the customer.

`phone`

:   The customer\'s phone number.

`preferred_locales`

:   Customer\'s preferred languages, ordered by preference.

`promotion_code`

:   The API ID of a promotion code to apply to the customer. The
    customer will have a discount applied on all recurring payments.
    Charges you create through the API will not have the discount.

`shipping`

:   The customer\'s shipping information. Appears on invoices emailed to
    this customer.

`source`

:   When using payment sources created via the Token or Sources APIs,
    passing `source` will create a new source object, make it the new
    customer default source, and delete the old customer default if one
    exists. If you want to add additional sources instead of replacing
    the existing default, use the [card creation
    API](https://stripe.com/docs/api#create_card){.perl-module}.
    Whenever you attach a card to a customer, Stripe will automatically
    validate the card.

`tax`

:   Tax details about the customer.

`tax_exempt`

:   The customer\'s tax exemption. One of `none`, `exempt`, or
    `reverse`.

`tax_id_data`

:   The customer\'s tax IDs.

`test_clock`

:   ID of the test clock to attach to the customer.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customers/create>

delete
------

        my $obj = $stripe->customers( delete => $args ) || die( $stripe->error );

Provided with a
[customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
customer. It returns the customer object that was deleted with its
property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customers/delete>

list {#list-18}
----

        my $obj = $stripe->customers( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object, this issue an api call to get the list of all customer.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`email`

:   A case-sensitive filter on the list based on the customer\'s `email`
    field. The value must be a string.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`test_clock`

:   Provides a list of customers that are associated with the specified
    test clock. The response will not include customers with test clocks
    if this parameter is not set.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customers/list>

retrieve {#retrieve-19}
--------

        my $obj = $stripe->customers( retrieve => $args ) || die( $stripe->error );

Provided with a
[customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a hash reference, this will retrieve a Stripe customer and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customers/retrieve>

search
------

        my $obj = $stripe->customers( search => {
            query => "name:'fakename' AND metadata['foo']:'bar'", } ) || die( $stripe->error );

Provided with a
[customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module},
or a hash reference, this will issue a search api call.

A dictionary with a `data` property that contains an array of up to
`limit` customers. If no objects match the query, the resulting array
will be empty. See the related guide on [expanding properties in
lists](https://stripe.com/docs/expand#lists){.perl-module}.

Possible parameters are:

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`page`

:   A cursor for pagination across multiple pages of results. Don\'t
    include this parameter on the first call. Use the next\_page value
    returned in a previous response to request subsequent results.

`query`

:   **Required**. The search query string. See [search query
    language](https://stripe.com/docs/search#search-query-language){.perl-module}
    and the list of supported [query fields for
    customers](https://stripe.com/docs/search#query-fields-for-customers){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customers/search>

update
------

        my $obj = $stripe->customers( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a hash reference, this will update a Stripe customer and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}

Possible parameters are:

`address`

:   The customer\'s address.

`balance`

:   An integer amount in JPY that represents the customer\'s current
    balance, which affect the customer\'s future invoices. A negative
    amount represents a credit that decreases the amount due on an
    invoice; a positive amount increases the amount due on an invoice.

`cash_balance`

:   Balance information and default balance settings for this customer.

`coupon`

:   If you provide a coupon code, the customer will have a discount
    applied on all recurring charges. Charges you create through the API
    will not have the discount.

`default_source`

:   If you are using payment methods created via the PaymentMethods API,
    see the
    [invoice*settings.default*payment\_method](https://stripe.com/docs/api/customers/update#update_customer-invoice_settings-default_payment_method){.perl-module}
    parameter.

    Provide the ID of a payment source already attached to this customer
    to make it this customer\'s default payment source.

    If you want to add a new payment source and make it the default, see
    the
    [source](https://stripe.com/docs/api/customers/update#update_customer-source){.perl-module}
    property.

`description`

:   An arbitrary string that you can attach to a customer object. It is
    displayed alongside the customer in the dashboard.

`email`

:   Customer\'s email address. It\'s displayed alongside the customer in
    your dashboard and can be useful for searching and tracking. This
    may be up to *512 characters*.

`invoice_prefix`

:   The prefix for the customer used to generate unique invoice numbers.
    Must be 3--12 uppercase letters or numbers.

`invoice_settings`

:   Default invoice settings for this customer.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   The customer\'s full name or business name.

`next_invoice_sequence`

:   The sequence to be used on the customer\'s next invoice. Defaults
    to 1.

`phone`

:   The customer\'s phone number.

`preferred_locales`

:   Customer\'s preferred languages, ordered by preference.

`promotion_code`

:   The API ID of a promotion code to apply to the customer. The
    customer will have a discount applied on all recurring payments.
    Charges you create through the API will not have the discount.

`shipping`

:   The customer\'s shipping information. Appears on invoices emailed to
    this customer.

`source`

:   When using payment sources created via the Token or Sources APIs,
    passing `source` will create a new source object, make it the new
    customer default source, and delete the old customer default if one
    exists. If you want to add additional sources instead of replacing
    the existing default, use the [card creation
    API](https://stripe.com/docs/api#create_card){.perl-module}.
    Whenever you attach a card to a customer, Stripe will automatically
    validate the card.

`tax`

:   Tax details about the customer.

`tax_exempt`

:   The customer\'s tax exemption. One of `none`, `exempt`, or
    `reverse`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customers/update>

CUSTOMER BALANCE TRANSACTION
============================

You can create, list, retrieve or update customer balance transaction

create {#create-15}
------

        my $obj = $stripe->customer_balance_transactions( create => {
            amount   => "-500",
            currency => "usd", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Customer::BalanceTransaction](https://metacpan.org/pod/Net::API::Stripe::Customer::BalanceTransaction){.perl-module}
object or a hash reference, this will create a Stripe customer balance
transaction and return an
[Net::API::Stripe::Customer::BalanceTransaction](https://metacpan.org/pod/Net::API::Stripe::Customer::BalanceTransaction){.perl-module}
object.

Possible parameters are:

`amount`

:   **Required**. The integer amount in **JPY** to apply to the
    customer\'s credit balance.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}. If the
    customer\'s
    [`currency`](https://stripe.com/docs/api/customers/object#customer_object-currency){.perl-module}
    is set, this value must match it. If the customer\'s `currency` is
    not set, it will be updated to this value.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_balance_transactions/create>

list {#list-19}
----

        my $obj = $stripe->customer_balance_transactions( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [customer balance
transaction](https://metacpan.org/pod/Net::API::Stripe::Customer::BalanceTransaction){.perl-module}
object, this issue an api call to get the list of all customer balance
transaction.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_balance_transactions/list>

retrieve {#retrieve-20}
--------

        my $obj = $stripe->customer_balance_transactions( retrieve => $args ) || die( $stripe->error );

Provided with a [customer balance
transaction](https://metacpan.org/pod/Net::API::Stripe::Customer::BalanceTransaction){.perl-module}
object or a hash reference, this will retrieve a Stripe customer balance
transaction and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Customer::BalanceTransaction){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_balance_transactions/retrieve>

update
------

        my $obj = $stripe->customer_balance_transactions( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [customer balance
transaction](https://metacpan.org/pod/Net::API::Stripe::Customer::BalanceTransaction){.perl-module}
object or a hash reference, this will update a Stripe customer balance
transaction and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Customer::BalanceTransaction){.perl-module}

Possible parameters are:

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_balance_transactions/update>

CUSTOMER CASH BALANCE TRANSACTION
=================================

You can fund\_cash\_balance, list or retrieve customer cash balance
transaction

fund\_cash\_balance
-------------------

        my $obj = $stripe->customer_cash_balance_transactions( fund_cash_balance => {
            amount   => "5000",
            currency => "eur", } ) || die( $stripe->error );

Provided with a [customer cash balance
transaction](https://metacpan.org/pod/Net::API::Stripe::Cash::Transaction){.perl-module},
or a hash reference, this will issue a fund\_cash\_balance api call.

Returns a specific cash balance transaction, which funded the customer's
[cash
balance](https://stripe.com/docs/payments/customer-balance){.perl-module}.

Possible parameters are:

`amount`

:   **Required**. Amount to be used for this test cash balance
    transaction. A positive integer representing how much to fund in the
    [smallest currency
    unit](https://stripe.com/docs/currencies#zero-decimal){.perl-module}
    (e.g., 100 cents to fund \$1.00 or 100 to fund ¥100, a zero-decimal
    currency).

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`reference`

:   A description of the test funding. This simulates free-text
    references supplied by customers when making bank transfers to their
    cash balance. You can use this to test how Stripe\'s [reconciliation
    algorithm](https://stripe.com/docs/payments/customer-balance/reconciliation){.perl-module}
    applies to different user inputs.

More information from Stripe api documentation at
<https://stripe.com/docs/api/cash_balance_transactions/fund_cash_balance>

list {#list-20}
----

        my $obj = $stripe->customer_cash_balance_transactions( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [customer cash balance
transaction](https://metacpan.org/pod/Net::API::Stripe::Cash::Transaction){.perl-module}
object, this issue an api call to get the list of all customer cash
balance transaction.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/cash_balance_transactions/list>

retrieve {#retrieve-21}
--------

        my $obj = $stripe->customer_cash_balance_transactions( retrieve => $args ) || die( $stripe->error );

Provided with a [customer cash balance
transaction](https://metacpan.org/pod/Net::API::Stripe::Cash::Transaction){.perl-module}
object or a hash reference, this will retrieve a Stripe customer cash
balance transaction and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Cash::Transaction){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/cash_balance_transactions/retrieve>

CUSTOMERS
=========

You can **create**, **retrieve**, **update**, **delete**,
**delete\_discount** or **list** customers.

create {#create-16}
------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a hash reference, this will create a Stripe customer and
return its
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object.

Possible parameters are:

*account\_balance* Integer

:   

*address* A [Net::API::Stripe::Address](https://metacpan.org/pod/Net::API::Stripe::Address){.perl-module} object or a hash reference with the following properties: line1 city country line2 postal\_code state

:   

*balance* Integer

:   

*coupon* A string that matches an existing Stripe coupon.

:   

*default\_source*

:   

*description* Test

:   

*email* String

:   

*id* A customer id, or Stripe will create one

:   

*invoice\_prefix* String

:   

*invoice\_settings* A hash reference with the following properties: custom\_fields default\_payment\_method footer

:   

*metadata* An arbitrary hash reference

:   

*name* String. Customer name

:   

*payment\_method* A Stripe payment method id

:   

*phone* String.

:   

*preferred\_locales* An array of strings representing 2-letters ISO 639 language codes such as `[qw( en fr ja )]`

:   

*shipping* A [Net::API::Stripe::Address](https://metacpan.org/pod/Net::API::Stripe::Address){.perl-module} object or a hash reference with the following properties: line1 city country line2 postal\_code state

:   

*source*

:   

*tax\_exempt* String that can be either *none*, *exempt* or *reverse*

:   

*tax\_id\_data* An array reference of string representing tax id or [Net::API::Stripe::Customer::TaxId](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module} objects

:   

*tax\_info* A [Net::API::Stripe::Customer::TaxInfo](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxInfo){.perl-module} object or a hash reference with the following properties: tax\_id type

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customers/create>

delete
------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a hash reference, this will remove a Stripe customer and
return its
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object with the property *deleted* set to true.

Possible parameters are:

*id* A Stripe customer id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customers/delete>

delete\_discount
----------------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a hash reference, this will remove a Stripe customer discount
and return the discount removed as a
[Net::API::Stripe::Billing::Discount](https://metacpan.org/pod/Net::API::Stripe::Billing::Discount){.perl-module}
object.

Possible parameters are:

*id* A Stripe customer id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/discounts/delete>

list {#list-21}
----

Provided with some optional parameters, this will get a list of Stripe
customers and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*created* Date or unix timestamp

:   

*email* String. E-mail address

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customers/list>

retrieve {#retrieve-22}
--------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a hash reference, this will retrieve a Stripe customer and
return its
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object.

Possible parameters are:

*id* A Stripe customer id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customers/retrieve>

update
------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a hash reference, this will update a Stripe customer and
return its
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object.

Possible parameters are:

*account\_balance* Integer

:   

*address* A [Net::API::Stripe::Address](https://metacpan.org/pod/Net::API::Stripe::Address){.perl-module} object or a hash reference with the following properties: line1 city country line2 postal\_code state

:   

*balance* Integer

:   

*coupon* A string that matches an existing Stripe coupon.

:   

*default\_source*

:   

*description* Test

:   

*email* String

:   

*id* A customer id, or Stripe will create one

:   

*invoice\_prefix* String

:   

*invoice\_settings* A hash reference with the following properties: custom\_fields default\_payment\_method footer

:   

*metadata* An arbitrary hash reference

:   

*name* String. Customer name

:   

*next\_invoice\_sequence* String

:   

*payment\_method* A Stripe payment method id

:   

*phone* String.

:   

*preferred\_locales* An array of strings representing 2-letters ISO 639 language codes such as `[qw( en fr ja )]`

:   

*shipping* A [Net::API::Stripe::Address](https://metacpan.org/pod/Net::API::Stripe::Address){.perl-module} object or a hash reference with the following properties: line1 city country line2 postal\_code state

:   

*source*

:   

*tax\_exempt* String that can be either *none*, *exempt* or *reverse*

:   

*tax\_id\_data* An array reference of string representing tax id or [Net::API::Stripe::Customer::TaxId](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module} objects

:   

*tax\_info* A [Net::API::Stripe::Customer::TaxInfo](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxInfo){.perl-module} object or a hash reference with the following properties: tax\_id type

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customers/create>

DISCOUNT
========

You can delete, delete\_customer or delete\_subscription discount

delete
------

        my $obj = $stripe->discounts( delete => $args ) || die( $stripe->error );

Provided with a
[discount](https://metacpan.org/pod/Net::API::Stripe::Billing::Discount){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
discount. It returns the discount object that was deleted with its
property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/discounts/subscription_delete>

DISPUTE
=======

You can close, list, retrieve or update dispute

close
-----

        my $obj = $stripe->disputes( close => $args ) || die( $stripe->error );

Provided with a
[dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module},
or a hash reference, this will issue a close api call.

Returns the dispute object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/disputes/close>

list {#list-22}
----

        my $obj = $stripe->disputes( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
object, this issue an api call to get the list of all dispute.

Possible parameters are:

`charge`

:   Only return disputes associated to the charge specified by this
    charge ID.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`payment_intent`

:   Only return disputes associated to the PaymentIntent specified by
    this PaymentIntent ID.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/disputes/list>

retrieve {#retrieve-23}
--------

        my $obj = $stripe->disputes( retrieve => $args ) || die( $stripe->error );

Provided with a
[dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
object or a hash reference, this will retrieve a Stripe dispute and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/disputes/retrieve>

update
------

        my $obj = $stripe->disputes( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
object or a hash reference, this will update a Stripe dispute and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}

Possible parameters are:

`evidence`

:   Evidence to upload, to respond to a dispute. Updating any field in
    the hash will submit all fields in the hash for review. The combined
    character count of all fields is limited to 150,000.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`submit`

:   Whether to immediately submit evidence to the bank. If `false`,
    evidence is staged on the dispute. Staged evidence is visible in the
    API and Dashboard, and can be submitted to the bank by making
    another request with this attribute set to `true` (the default).

More information from Stripe api documentation at
<https://stripe.com/docs/api/disputes/update>

DISPUTES
========

You can **close**, **retrieve**, **update** or **list** disputes

close
-----

Provided with a
[Net::API::Stripe::Dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
object or an hash reference and this will close a Stripe dispute and
return
[Net::API::Stripe::Dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
object.

Possible parameters are:

*id* A Stripe dispute id.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/disputes/close>

list {#list-23}
----

Provided with some optional parameters and this will issue a Stripe api
call to get the list of disputes and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*created* Date or unix timestamp

:   

*charge* A Stripe charge id

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*payment\_intent* A Stripe payment intent id

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/disputes/list>

retrieve {#retrieve-24}
--------

Provided with a
[Net::API::Stripe::Dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
or a hash reference of parameters, this will retrieve the Stripe dispute
and return a
[Net::API::Stripe::Dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
object.

Possible parameters are:

*id* A Stripe dispute id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/disputes/retrieve>

update
------

Provided with a
[Net::API::Stripe::Dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
or a hash reference of parameters, this will update a Stripe dispute and
return a
[Net::API::Stripe::Dispute](https://metacpan.org/pod/Net::API::Stripe::Dispute){.perl-module}
object.

Possible parameters are:

*id* A Stripe dispute id. This is required.

:   

*evidence* This is a hash reference with following possible properties:

:   

    *access\_activity\_log*

    :   

    *billing\_address*

    :   

    *cancellation\_policy*

    :   

    *cancellation\_policy\_disclosure*

    :   

    *cancellation\_rebuttal*

    :   

    *customer\_communication*

    :   

    *customer\_email\_address*

    :   

    *customer\_name*

    :   

    *customer\_purchase\_ip*

    :   

    *customer\_signature*

    :   

    *duplicate\_charge\_documentation*

    :   

    *duplicate\_charge\_explanation*

    :   

    *duplicate\_charge\_id*

    :   

    *product\_description*

    :   

    *receipt*

    :   

    *refund\_policy*

    :   

    *refund\_policy\_disclosure*

    :   

    *refund\_refusal\_explanation*

    :   

    *service\_date*

    :   

    *service\_documentation*

    :   

    *shipping\_address*

    :   

    *shipping\_carrier*

    :   

    *shipping\_date*

    :   

    *shipping\_documentation*

    :   

    *shipping\_tracking\_number*

    :   

    *uncategorized\_file*

    :   

    *uncategorized\_text*

    :   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/disputes/update>

ERROR HANDLING
==============

[Net::API::Stripe](https://metacpan.org/pod/Net::API::Stripe){.perl-module}
never dies, or at least not voluntarily. Instead, when an error occurs
and is reported, it returns undef and the error can be retrieved with
the [\"error\"](#error){.perl-module} method, such as:

        my $prod = $stripe->products( retrieve => $prod_id ) || die( $stripe->error, "\n" );

The **error** method returns the
[Module::Generic::Exception](https://metacpan.org/pod/Module::Generic::Exception){.perl-module}
set. Please refer to the manual page of [\"error\" in
Module::Generic](https://metacpan.org/pod/Module::Generic#error){.perl-module}
for more information, but essentially, the following methods are
available with the error objects:

as\_string
----------

This is triggered when the error object is stringified

code
----

The error code returned by Stripe

file
----

The file containing the error

line
----

The line number of the error

message
-------

The actual error message

package
-------

The package name where the error occurred.

rethrow
-------

Used to re-trigger the error

subroutine
----------

The subroutine name where the error originates

trace
-----

The full stack trace object. This is a
[Devel::StackTrace](https://metacpan.org/pod/Devel::StackTrace){.perl-module}

type
----

The error type, if any

EVENT
=====

You can list or retrieve event

list {#list-24}
----

        my $obj = $stripe->events( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[event](https://metacpan.org/pod/Net::API::Stripe::Event){.perl-module}
object, this issue an api call to get the list of all event.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`delivery_success`

:   Filter events by whether all webhooks were successfully delivered.
    If false, events which are still pending or have failed all delivery
    attempts to a webhook endpoint will be returned.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`type`

:   A string containing a specific event name, or group of events
    using \* as a wildcard. The list will be filtered to include only
    events with a matching event property.

`types`

:   An array of up to 20 strings containing specific event names. The
    list will be filtered to include only events with a matching event
    property. You may pass either `type` or `types`, but not both.

More information from Stripe api documentation at
<https://stripe.com/docs/api/events/list>

retrieve {#retrieve-25}
--------

        my $obj = $stripe->events( retrieve => $args ) || die( $stripe->error );

Provided with a
[event](https://metacpan.org/pod/Net::API::Stripe::Event){.perl-module}
object or a hash reference, this will retrieve a Stripe event and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Event){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/events/retrieve>

FEE REFUND
==========

You can create, list, retrieve or update fee refund

create {#create-17}
------

        my $obj = $stripe->fee_refunds( create => $args ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::ApplicationFee::Refund](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}
object or a hash reference, this will create a Stripe fee refund and
return an
[Net::API::Stripe::Connect::ApplicationFee::Refund](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}
object.

Possible parameters are:

`amount`

:   A positive integer, in *JPY*, representing how much of this fee to
    refund. Can refund only up to the remaining unrefunded amount of the
    fee.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/fee_refunds/create>

list {#list-25}
----

        my $obj = $stripe->fee_refunds( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [fee
refund](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}
object, this issue an api call to get the list of all fee refund.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/fee_refunds/list>

retrieve {#retrieve-26}
--------

        my $obj = $stripe->fee_refunds( retrieve => $args ) || die( $stripe->error );

Provided with a [fee
refund](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}
object or a hash reference, this will retrieve a Stripe fee refund and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/fee_refunds/retrieve>

update
------

        my $obj = $stripe->fee_refunds( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [fee
refund](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}
object or a hash reference, this will update a Stripe fee refund and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::ApplicationFee::Refund){.perl-module}

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/fee_refunds/update>

FILE
====

You can create, list or retrieve file

create {#create-18}
------

        my $obj = $stripe->files( create => {
            file    => "{a file descriptor}",
            purpose => "dispute_evidence", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::File](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
object or a hash reference, this will create a Stripe file and return an
[Net::API::Stripe::File](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
object.

Possible parameters are:

`file`

:   **Required**. A file to upload. The file should follow the
    specifications of RFC 2388 (which defines file transfers for the
    `multipart/form-data` protocol).

`file_link_data`

:   Optional parameters to automatically create a [file
    link](https://stripe.com/docs/api/file_links){.perl-module} for the
    newly created file.

`purpose`

:   **Required**. The
    [purpose](https://stripe.com/docs/file-upload#uploading-a-file){.perl-module}
    of the uploaded file.

More information from Stripe api documentation at
<https://stripe.com/docs/api/files/create>

list {#list-26}
----

        my $obj = $stripe->files( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[file](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
object, this issue an api call to get the list of all file.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`purpose`

:   The file purpose to filter queries by. If none is provided, files
    will not be filtered by purpose.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/files/list>

retrieve {#retrieve-27}
--------

        my $obj = $stripe->files( retrieve => $args ) || die( $stripe->error );

Provided with a
[file](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
object or a hash reference, this will retrieve a Stripe file and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/files/retrieve>

FILE LINK
=========

You can create, list, retrieve or update file link

create {#create-19}
------

        my $obj = $stripe->file_links( create => {
            file => "file_1JXy922eZvKYlo2CV17dW6tI", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::File::Link](https://metacpan.org/pod/Net::API::Stripe::File::Link){.perl-module}
object or a hash reference, this will create a Stripe file link and
return an
[Net::API::Stripe::File::Link](https://metacpan.org/pod/Net::API::Stripe::File::Link){.perl-module}
object.

Possible parameters are:

`expires_at`

:   A future timestamp after which the link will no longer be usable.

`file`

:   **Required**. The ID of the file. The file\'s `purpose` must be one
    of the following: `business_icon`, `business_logo`,
    `customer_signature`, `dispute_evidence`, `finance_report_run`,
    `identity_document_downloadable`, `pci_document`, `selfie`,
    `sigma_scheduled_query`, `tax_document_user_upload`, or
    `terminal_reader_splashscreen`.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/file_links/create>

list {#list-27}
----

        my $obj = $stripe->file_links( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [file
link](https://metacpan.org/pod/Net::API::Stripe::File::Link){.perl-module}
object, this issue an api call to get the list of all file link.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`expired`

:   Filter links by their expiration status. By default, all links are
    returned.

`file`

:   Only return links for the given file.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/file_links/list>

retrieve {#retrieve-28}
--------

        my $obj = $stripe->file_links( retrieve => $args ) || die( $stripe->error );

Provided with a [file
link](https://metacpan.org/pod/Net::API::Stripe::File::Link){.perl-module}
object or a hash reference, this will retrieve a Stripe file link and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::File::Link){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/file_links/retrieve>

update
------

        my $obj = $stripe->file_links( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [file
link](https://metacpan.org/pod/Net::API::Stripe::File::Link){.perl-module}
object or a hash reference, this will update a Stripe file link and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::File::Link){.perl-module}

Possible parameters are:

`expires_at`

:   A future timestamp after which the link will no longer be usable, or
    `now` to expire the link immediately.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/file_links/update>

FILES
=====

You can **create**, **retrieve** or **list** files

create {#create-20}
------

Provided with a
[Net::API::Stripe::File](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
or a hash reference of parameters, this will create a Stripe file and
return a
[Net::API::Stripe::File](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
object.

Possible parameters are:

*file* File path.

:   It will check if the file exists, is not zero length, is readable
    and make the file path absolute if it is relative (using
    `Cwd::abs_path`)

*file\_link\_data* A hash reference with the following properties: create expires\_at metadata

:   

*purpose* String that can be either *business\_icon* *business\_logo* *customer\_signature* *dispute\_evidence* *identity\_document* *pci\_document* or *tax\_document\_user\_upload*

:   

For more information, see Stripe documentation here:
[httpshttps://stripe.com/docs/api/files/create](httpshttps://stripe.com/docs/api/files/create){.perl-module}

list {#list-28}
----

Provided with some optional parameters and this will issue a Stripe api
call to get the list of files and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*created* Date or unix timestamp

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*purpose* String.

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/files/list>

retrieve {#retrieve-29}
--------

Provided with a
[Net::API::Stripe::File](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
or a hash reference of parameters, this will retrieve the Stripe file
and return a
[Net::API::Stripe::File](https://metacpan.org/pod/Net::API::Stripe::File){.perl-module}
object.

Possible parameters are:

*id* A Stripe file id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/files/retrieve>

FINANCIAL CONNECTIONS ACCOUNT
=============================

You can disconnect, list, refresh or retrieve financial connections
account

disconnect
----------

        my $obj = $stripe->financial_connections_accounts( disconnect => $args ) || die( $stripe->error );

Provided with a [financial connections
account](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Account){.perl-module},
or a hash reference, this will issue a disconnect api call.

Returns an `Account` object if a valid identifier was provided, and
returns [an error](https://stripe.com/docs/api/errors){.perl-module}
otherwise.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/financial_connections/accounts/disconnect>

list {#list-29}
----

        my $obj = $stripe->financial_connections_accounts( list => {
            account_holder =>
            {
                customer => "cus_AJ78ZaALpqgiuZ",
            } } ) || die( $stripe->error );

Provided with a [financial connections
account](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Account){.perl-module}
object, this issue an api call to get the list of all financial
connections account.

Possible parameters are:

`account_holder`

:   If present, only return accounts that belong to the specified
    account holder. `account_holder[customer]` and
    `account_holder[account]` are mutually exclusive.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`session`

:   If present, only return accounts that were collected as part of the
    given session.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/financial_connections/accounts/list>

refresh
-------

        my $obj = $stripe->financial_connections_accounts( refresh => $args ) || die( $stripe->error );

Provided with a [financial connections
account](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Account){.perl-module},
or a hash reference, this will issue a refresh api call.

Returns an `Account` object if a valid identifier was provided and if
you have sufficient permissions to that account. Returns [an
error](https://stripe.com/docs/api/errors){.perl-module} otherwise.

Possible parameters are:

`features`

:   **Required**. The list of account features that you would like to
    refresh. Either: `balance` or `ownership`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/financial_connections/accounts/refresh>

retrieve {#retrieve-30}
--------

        my $obj = $stripe->financial_connections_accounts( retrieve => $args ) || die( $stripe->error );

Provided with a [financial connections
account](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Account){.perl-module}
object or a hash reference, this will retrieve a Stripe financial
connections account and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Account){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/financial_connections/accounts/retrieve>

FINANCIAL CONNECTIONS ACCOUNT OWNER
===================================

You can list financial connections account owner

list {#list-30}
----

        my $obj = $stripe->financial_connections_account_owners( list => {
            limit     => "3",
            ownership => "fcaowns_1Le9F42eZvKYlo2CfwkUGxZt", } ) || die( $stripe->error );

Provided with a [financial connections account
owner](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::AccountOwner){.perl-module}
object, this issue an api call to get the list of all financial
connections account owner.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`ownership`

:   **Required**. The ID of the ownership object to fetch owners from.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/financial_connections/ownership/list>

FINANCIAL CONNECTIONS SESSION
=============================

You can create or retrieve financial connections session

create {#create-21}
------

        my $obj = $stripe->financial_connections_sessions( create => {
            account_holder =>
            {
                customer => "cus_AJ78ZaALpqgiuZ",
                type     => "customer",
            }
            filters        =>
            {
                countries => [qw( US )],
            }
            permissions    => [qw( payment_method balances )], } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Financial::Connections::Session](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Session){.perl-module}
object or a hash reference, this will create a Stripe financial
connections session and return an
[Net::API::Stripe::Financial::Connections::Session](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Session){.perl-module}
object.

Possible parameters are:

`account_holder`

:   **Required**. The account holder to link accounts for.

`filters`

:   Filters to restrict the kinds of accounts to collect.

`permissions`

:   **Required**. List of data features that you would like to request
    access to.

    Possible values are `balances`, `transactions`, `ownership`, and
    `payment_method`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/financial_connections/sessions/create>

retrieve {#retrieve-31}
--------

        my $obj = $stripe->financial_connections_sessions( retrieve => $args ) || die( $stripe->error );

Provided with a [financial connections
session](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Session){.perl-module}
object or a hash reference, this will retrieve a Stripe financial
connections session and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Financial::Connections::Session){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/financial_connections/sessions/retrieve>

FUNDING INSTRUCTIONS
====================

You can create, fund or list funding instructions

create {#create-22}
------

        my $obj = $stripe->funding_instructionss( create => {
            bank_transfer =>
            {
                type => "eu_bank_transfer",
            }
            currency      => "eur",
            funding_type  => "bank_transfer", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Issuing::FundingInstructions](https://metacpan.org/pod/Net::API::Stripe::Issuing::FundingInstructions){.perl-module}
object or a hash reference, this will create a Stripe funding
instructions and return an
[Net::API::Stripe::Issuing::FundingInstructions](https://metacpan.org/pod/Net::API::Stripe::Issuing::FundingInstructions){.perl-module}
object.

Possible parameters are:

`bank_transfer`

:   **Required**. Additional parameters for `bank_transfer` funding
    types

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`funding_type`

:   **Required**. The `funding_type` to get the instructions for.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/funding_instructions/create>

fund
----

        my $obj = $stripe->funding_instructionss( fund => {
            amount   => "4200",
            currency => "eur", } ) || die( $stripe->error );

Provided with a [funding
instructions](https://metacpan.org/pod/Net::API::Stripe::Issuing::FundingInstructions){.perl-module},
or a hash reference, this will issue a fund api call.

Returns testmode funding instructions for an Issuing balance

Possible parameters are:

`amount`

:   **Required**. The amount to top up

`currency`

:   **Required**. The currency to top up

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/funding_instructions/fund>

list {#list-31}
----

        my $obj = $stripe->funding_instructionss( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [funding
instructions](https://metacpan.org/pod/Net::API::Stripe::Issuing::FundingInstructions){.perl-module}
object, this issue an api call to get the list of all funding
instructions.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/funding_instructions/list>

IDENTITY VERIFICATION REPORT
============================

You can list or retrieve identity verification report

list {#list-32}
----

        my $obj = $stripe->identity_verification_reports( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [identity verification
report](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationReport){.perl-module}
object, this issue an api call to get the list of all identity
verification report.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`type`

:   Only return VerificationReports of this type

`verification_session`

:   Only return VerificationReports created by this VerificationSession
    ID. It is allowed to provide a VerificationIntent ID.

More information from Stripe api documentation at
<https://stripe.com/docs/api/identity/verification_reports/list>

retrieve {#retrieve-32}
--------

        my $obj = $stripe->identity_verification_reports( retrieve => $args ) || die( $stripe->error );

Provided with a [identity verification
report](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationReport){.perl-module}
object or a hash reference, this will retrieve a Stripe identity
verification report and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationReport){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/identity/verification_reports/retrieve>

IDENTITY VERIFICATION SESSION
=============================

You can cancel, create, list, redact, retrieve or update identity
verification session

cancel
------

        my $obj = $stripe->identity_verification_sessions( cancel => $args ) || die( $stripe->error );

Provided with a [identity verification
session](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns the canceled VerificationSession object

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/identity/verification_sessions/cancel>

create {#create-23}
------

        my $obj = $stripe->identity_verification_sessions( create => {
            type => "document", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Identity::VerificationSession](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module}
object or a hash reference, this will create a Stripe identity
verification session and return an
[Net::API::Stripe::Identity::VerificationSession](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module}
object.

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`options`

:   A set of options for the session's verification checks.

`return_url`

:   The URL that the user will be redirected to upon completing the
    verification flow.

`type`

:   **Required**. The type of [verification
    check](https://stripe.com/docs/identity/verification-checks){.perl-module}
    to be performed.

More information from Stripe api documentation at
<https://stripe.com/docs/api/identity/verification_sessions/create>

list {#list-33}
----

        my $obj = $stripe->identity_verification_sessions( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [identity verification
session](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module}
object, this issue an api call to get the list of all identity
verification session.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return VerificationSessions with this status. [Learn more about
    the lifecycle of
    sessions](https://stripe.com/docs/identity/how-sessions-work){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/identity/verification_sessions/list>

redact
------

        my $obj = $stripe->identity_verification_sessions( redact => $args ) || die( $stripe->error );

Provided with a [identity verification
session](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module},
or a hash reference, this will issue a redact api call.

Returns the redacted VerificationSession object

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/identity/verification_sessions/redact>

retrieve {#retrieve-33}
--------

        my $obj = $stripe->identity_verification_sessions( retrieve => $args ) || die( $stripe->error );

Provided with a [identity verification
session](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module}
object or a hash reference, this will retrieve a Stripe identity
verification session and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/identity/verification_sessions/retrieve>

update
------

        my $obj = $stripe->identity_verification_sessions( update => {
            type => "id_number", } ) || die( $stripe->error );

Provided with a [identity verification
session](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module}
object or a hash reference, this will update a Stripe identity
verification session and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Identity::VerificationSession){.perl-module}

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`options`

:   A set of options for the session's verification checks.

`type`

:   The type of [verification
    check](https://stripe.com/docs/identity/verification-checks){.perl-module}
    to be performed.

More information from Stripe api documentation at
<https://stripe.com/docs/api/identity/verification_sessions/update>

INVOICE
=======

You can create, delete, finalise, finalize, invoice\_write\_off, lines,
lines\_upcoming, list, pay, retrieve, search, send, uncollectible,
upcoming, update or void invoice

create {#create-24}
------

        my $obj = $stripe->invoices( create => {
            customer => "cus_AJ78ZaALpqgiuZ", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or a hash reference, this will create a Stripe invoice and return
an
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

`account_tax_ids`

:   The account tax IDs associated with the invoice. Only editable when
    the invoice is a draft.

`application_fee_amount`

:   A fee in JPY that will be applied to the invoice and transferred to
    the application owner\'s Stripe account. The request must be made
    with an OAuth key or the Stripe-Account header in order to take an
    application fee. For more information, see the application fees
    [documentation](https://stripe.com/docs/billing/invoices/connect#collecting-fees){.perl-module}.

`auto_advance`

:   Controls whether Stripe will perform [automatic
    collection](https://stripe.com/docs/billing/invoices/workflow/#auto_advance){.perl-module}
    of the invoice. When `false`, the invoice\'s state will not
    automatically advance without an explicit action.

`automatic_tax`

:   Settings for automatic tax lookup for this invoice.

`collection_method`

:   Either `charge_automatically`, or `send_invoice`. When charging
    automatically, Stripe will attempt to pay this invoice using the
    default source attached to the customer. When sending an invoice,
    Stripe will email this invoice to the customer with payment
    instructions. Defaults to `charge_automatically`.

`currency`

:   The currency to create this invoice in. Defaults to that of
    `customer` if not specified.

`custom_fields`

:   A list of up to 4 custom fields to be displayed on the invoice.

`customer`

:   The ID of the customer who will be billed.

`days_until_due`

:   The number of days from when the invoice is created until it is due.
    Valid only for invoices where `collection_method=send_invoice`.

`default_payment_method`

:   ID of the default payment method for the invoice. It must belong to
    the customer associated with the invoice. If not set, defaults to
    the subscription\'s default payment method, if any, or to the
    default payment method in the customer\'s invoice settings.

`default_source`

:   ID of the default payment source for the invoice. It must belong to
    the customer associated with the invoice and be in a chargeable
    state. If not set, defaults to the subscription\'s default source,
    if any, or to the customer\'s default source.

`default_tax_rates`

:   The tax rates that will apply to any line item that does not have
    `tax_rates` set.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users. Referenced as \'memo\' in the Dashboard.

`discounts`

:   The coupons to redeem into discounts for the invoice. If not
    specified, inherits the discount from the invoice\'s customer. Pass
    an empty string to avoid inheriting any discounts.

`due_date`

:   The date on which payment for this invoice is due. Valid only for
    invoices where `collection_method=send_invoice`.

`footer`

:   Footer to be displayed on the invoice.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`on_behalf_of`

:   The account (if any) for which the funds of the invoice payment are
    intended. If set, the invoice will be presented with the branding
    and support information of the specified account. See the [Invoices
    with
    Connect](https://stripe.com/docs/billing/invoices/connect){.perl-module}
    documentation for details.

`payment_settings`

:   Configuration settings for the PaymentIntent that is generated when
    the invoice is finalized.

`pending_invoice_items_behavior`

:   How to handle pending invoice items on invoice creation. One of
    `include` or `exclude`. `include` will include any pending invoice
    items, and will create an empty draft invoice if no pending invoice
    items exist. `exclude` will always create an empty invoice draft
    regardless if there are pending invoice items or not. Defaults to
    `exclude` if the parameter is omitted.

`rendering_options`

:   Options for invoice PDF rendering.

`statement_descriptor`

:   Extra information about a charge for the customer\'s credit card
    statement. It must contain at least one letter. If not specified and
    this invoice is part of a subscription, the default
    `statement_descriptor` will be set to the first subscription item\'s
    product\'s `statement_descriptor`.

`subscription`

:   The ID of the subscription to invoice, if any. If set, the created
    invoice will only include pending invoice items for that
    subscription and pending invoice items not associated with any
    subscription if `pending_invoice_items_behavior` is `include`. The
    subscription\'s billing cycle and regular subscription events won\'t
    be affected.

`transfer_data`

:   If specified, the funds from the invoice will be transferred to the
    destination and the ID of the resulting transfer will be found on
    the invoice\'s charge.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/create>

delete
------

        my $obj = $stripe->invoices( delete => $args ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
invoice. It returns the invoice object that was deleted with its
property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/delete>

finalize
--------

        my $obj = $stripe->invoices( finalize => $args ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module},
or a hash reference, this will issue a finalize api call.

Returns an invoice object with `status=open`.

Possible parameters are:

`auto_advance`

:   Controls whether Stripe will perform [automatic
    collection](https://stripe.com/docs/invoicing/automatic-charging){.perl-module}
    of the invoice. When `false`, the invoice\'s state will not
    automatically advance without an explicit action.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/finalize>

list {#list-34}
----

        my $obj = $stripe->invoices( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object, this issue an api call to get the list of all invoice.

Possible parameters are:

`collection_method`

:   The collection method of the invoice to retrieve. Either
    `charge_automatically` or `send_invoice`.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`customer`

:   Only return invoices for the customer specified by this customer ID.

`due_date`

:   A filter on the list based on the object `due_date` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   The status of the invoice, one of `draft`, `open`, `paid`,
    `uncollectible`, or `void`. [Learn
    more](https://stripe.com/docs/billing/invoices/workflow#workflow-overview){.perl-module}

`subscription`

:   Only return invoices for the subscription specified by this
    subscription ID.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/list>

pay
---

        my $obj = $stripe->invoices( pay => $args ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module},
or a hash reference, this will issue a pay api call.

Returns the invoice object.

Possible parameters are:

`forgive`

:   In cases where the source used to pay the invoice has insufficient
    funds, passing `forgive=true` controls whether a charge should be
    attempted for the full amount available on the source, up to the
    amount to fully pay the invoice. This effectively forgives the
    difference between the amount available on the source and the amount
    due.

    Passing `forgive=false` will fail the charge if the source hasn\'t
    been pre-funded with the right amount. An example for this case is
    with ACH Credit Transfers and wires: if the amount wired is less
    than the amount due by a small amount, you might want to forgive the
    difference. Defaults to `false`.

`mandate`

:   ID of the mandate to be used for this invoice. It must correspond to
    the payment method used to pay the invoice, including the
    payment*method param or the invoice\'s default*payment*method or
    default*source, if set.

`off_session`

:   Indicates if a customer is on or off-session while an invoice
    payment is attempted. Defaults to `true` (off-session).

`paid_out_of_band`

:   Boolean representing whether an invoice is paid outside of Stripe.
    This will result in no charge being made. Defaults to `false`.

`payment_method`

:   A PaymentMethod to be charged. The PaymentMethod must be the ID of a
    PaymentMethod belonging to the customer associated with the invoice
    being paid.

`source`

:   A payment source to be charged. The source must be the ID of a
    source belonging to the customer associated with the invoice being
    paid.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/pay>

retrieve {#retrieve-34}
--------

        my $obj = $stripe->invoices( retrieve => $args ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or a hash reference, this will retrieve a Stripe invoice and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/retrieve>

search
------

        my $obj = $stripe->invoices( search => {
            query => "total>999 AND metadata['order_id']:'6735'", } ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module},
or a hash reference, this will issue a search api call.

A dictionary with a `data` property that contains an array of up to
`limit` invoices. If no objects match the query, the resulting array
will be empty. See the related guide on [expanding properties in
lists](https://stripe.com/docs/expand#lists){.perl-module}.

Possible parameters are:

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`page`

:   A cursor for pagination across multiple pages of results. Don\'t
    include this parameter on the first call. Use the next\_page value
    returned in a previous response to request subsequent results.

`query`

:   **Required**. The search query string. See [search query
    language](https://stripe.com/docs/search#search-query-language){.perl-module}
    and the list of supported [query fields for
    invoices](https://stripe.com/docs/search#query-fields-for-invoices){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/search>

send
----

        my $obj = $stripe->invoices( send => $args ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module},
or a hash reference, this will issue a send api call.

Returns the invoice object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/send>

uncollectible
-------------

        my $obj = $stripe->invoices( uncollectible => $args ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module},
or a hash reference, this will issue a uncollectible api call.

Returns the invoice object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/mark_uncollectible>

upcoming
--------

        my $obj = $stripe->invoices( upcoming => $args ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module},
or a hash reference, this will issue a upcoming api call.

Returns an invoice if valid customer information is provided. Returns
[an error](https://stripe.com/docs/api/errors){.perl-module} otherwise.

Possible parameters are:

`automatic_tax`

:   Settings for automatic tax lookup for this invoice preview.

`coupon`

:   The code of the coupon to apply. If `subscription` or
    `subscription_items` is provided, the invoice returned will preview
    updating or creating a subscription with that coupon. Otherwise, it
    will preview applying that coupon to the customer for the next
    upcoming invoice from among the customer\'s subscriptions. The
    invoice can be previewed without a coupon by passing this value as
    an empty string.

`currency`

:   The currency to preview this invoice in. Defaults to that of
    `customer` if not specified.

`customer`

:   **Required if subscription unset** The identifier of the customer
    whose upcoming invoice you\'d like to retrieve.

`customer_details`

:   **Required if subscription and customer unset** Details about the
    customer you want to invoice or overrides for an existing customer.

`discounts`

:   The coupons to redeem into discounts for the invoice preview. If not
    specified, inherits the discount from the customer or subscription.
    This only works for coupons directly applied to the invoice. To
    apply a coupon to a subscription, you must use the `coupon`
    parameter instead. Pass an empty string to avoid inheriting any
    discounts. To preview the upcoming invoice for a subscription that
    hasn\'t been created, use `coupon` instead.

`invoice_items`

:   List of invoice items to add or update in the upcoming invoice
    preview.

`schedule`

:   The identifier of the unstarted schedule whose upcoming invoice
    you\'d like to retrieve. Cannot be used with subscription or
    subscription fields.

`subscription`

:   The identifier of the subscription for which you\'d like to retrieve
    the upcoming invoice. If not provided, but a `subscription_items` is
    provided, you will preview creating a subscription with those items.
    If neither `subscription` nor `subscription_items` is provided, you
    will retrieve the next upcoming invoice from among the customer\'s
    subscriptions.

`subscription_billing_cycle_anchor`

:   For new subscriptions, a future timestamp to anchor the
    subscription\'s [billing
    cycle](https://stripe.com/docs/subscriptions/billing-cycle){.perl-module}.
    This is used to determine the date of the first full invoice, and,
    for plans with `month` or `year` intervals, the day of the month for
    subsequent invoices. For existing subscriptions, the value can only
    be set to `now` or `unchanged`.

`subscription_cancel_at`

:   Timestamp indicating when the subscription should be scheduled to
    cancel. Will prorate if within the current period and prorations
    have been enabled using `proration_behavior`.

`subscription_cancel_at_period_end`

:   Boolean indicating whether this subscription should cancel at the
    end of the current period.

`subscription_cancel_now`

:   This simulates the subscription being canceled or expired
    immediately.

`subscription_default_tax_rates`

:   If provided, the invoice returned will preview updating or creating
    a subscription with these default tax rates. The default tax rates
    will apply to any line item that does not have `tax_rates` set.

`subscription_items`

:   A list of up to 20 subscription items, each with an attached price.

`subscription_proration_behavior`

:   Determines how to handle
    [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations){.perl-module}
    when the billing cycle changes (e.g., when switching plans,
    resetting `billing_cycle_anchor=now`, or starting a trial), or if an
    item\'s `quantity` changes.

`subscription_proration_date`

:   If previewing an update to a subscription, and doing proration,
    `subscription_proration_date` forces the proration to be calculated
    as though the update was done at the specified time. The time given
    must be within the current subscription period, and cannot be before
    the subscription was on its current plan. If set, `subscription`,
    and one of `subscription_items`, or `subscription_trial_end` are
    required. Also, `subscription_proration_behavior` cannot be set to
    \'none\'.

`subscription_start_date`

:   Date a subscription is intended to start (can be future or past)

`subscription_trial_end`

:   If provided, the invoice returned will preview updating or creating
    a subscription with that trial end. If set, one of
    `subscription_items` or `subscription` is required.

`subscription_trial_from_plan`

:   Indicates if a plan\'s `trial_period_days` should be applied to the
    subscription. Setting `subscription_trial_end` per subscription is
    preferred, and this defaults to `false`. Setting this flag to `true`
    together with `subscription_trial_end` is not allowed. See [Using
    trial periods on
    subscriptions](https://stripe.com/docs/billing/subscriptions/trials){.perl-module}
    to learn more.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/upcoming>

update
------

        my $obj = $stripe->invoices( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or a hash reference, this will update a Stripe invoice and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}

Possible parameters are:

`account_tax_ids`

:   The account tax IDs associated with the invoice. Only editable when
    the invoice is a draft.

`application_fee_amount`

:   A fee in JPY that will be applied to the invoice and transferred to
    the application owner\'s Stripe account. The request must be made
    with an OAuth key or the Stripe-Account header in order to take an
    application fee. For more information, see the application fees
    [documentation](https://stripe.com/docs/billing/invoices/connect#collecting-fees){.perl-module}.

`auto_advance`

:   Controls whether Stripe will perform [automatic
    collection](https://stripe.com/docs/billing/invoices/workflow/#auto_advance){.perl-module}
    of the invoice.

`automatic_tax`

:   Settings for automatic tax lookup for this invoice.

`collection_method`

:   Either `charge_automatically` or `send_invoice`. This field can be
    updated only on `draft` invoices.

`custom_fields`

:   A list of up to 4 custom fields to be displayed on the invoice. If a
    value for `custom_fields` is specified, the list specified will
    replace the existing custom field list on this invoice. Pass an
    empty string to remove previously-defined fields.

`days_until_due`

:   The number of days from which the invoice is created until it is
    due. Only valid for invoices where `collection_method=send_invoice`.
    This field can only be updated on `draft` invoices.

`default_payment_method`

:   ID of the default payment method for the invoice. It must belong to
    the customer associated with the invoice. If not set, defaults to
    the subscription\'s default payment method, if any, or to the
    default payment method in the customer\'s invoice settings.

`default_source`

:   ID of the default payment source for the invoice. It must belong to
    the customer associated with the invoice and be in a chargeable
    state. If not set, defaults to the subscription\'s default source,
    if any, or to the customer\'s default source.

`default_tax_rates`

:   The tax rates that will apply to any line item that does not have
    `tax_rates` set. Pass an empty string to remove previously-defined
    tax rates.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users. Referenced as \'memo\' in the Dashboard.

`discounts`

:   The discounts that will apply to the invoice. Pass an empty string
    to remove previously-defined discounts.

`due_date`

:   The date on which payment for this invoice is due. Only valid for
    invoices where `collection_method=send_invoice`. This field can only
    be updated on `draft` invoices.

`footer`

:   Footer to be displayed on the invoice.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`on_behalf_of`

:   The account (if any) for which the funds of the invoice payment are
    intended. If set, the invoice will be presented with the branding
    and support information of the specified account. See the [Invoices
    with
    Connect](https://stripe.com/docs/billing/invoices/connect){.perl-module}
    documentation for details.

`payment_settings`

:   Configuration settings for the PaymentIntent that is generated when
    the invoice is finalized.

`rendering_options`

:   Options for invoice PDF rendering.

`statement_descriptor`

:   Extra information about a charge for the customer\'s credit card
    statement. It must contain at least one letter. If not specified and
    this invoice is part of a subscription, the default
    `statement_descriptor` will be set to the first subscription item\'s
    product\'s `statement_descriptor`.

`transfer_data`

:   If specified, the funds from the invoice will be transferred to the
    destination and the ID of the resulting transfer will be found on
    the invoice\'s charge. This will be unset if you POST an empty
    value.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/update>

void
----

        my $obj = $stripe->invoices( void => $args ) || die( $stripe->error );

Provided with a
[invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module},
or a hash reference, this will issue a void api call.

Returns the voided invoice object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/void>

INVOICEITEM
===========

You can create, delete, list, retrieve or update invoiceitem

create {#create-25}
------

        my $obj = $stripe->invoiceitems( create => {
            customer => "cus_AJ78ZaALpqgiuZ",
            price    => "price_1Le4Uy2eZvKYlo2Cby2Mg3bR", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::Invoice::Item](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}
object or a hash reference, this will create a Stripe invoiceitem and
return an
[Net::API::Stripe::Billing::Invoice::Item](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}
object.

Possible parameters are:

`amount`

:   The integer amount in JPY of the charge to be applied to the
    upcoming invoice. Passing in a negative `amount` will reduce the
    `amount_due` on the invoice.

`currency`

:   Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`customer`

:   **Required**. The ID of the customer who will be billed when this
    invoice item is billed.

`description`

:   An arbitrary string which you can attach to the invoice item. The
    description is displayed in the invoice for easy tracking.

`discountable`

:   Controls whether discounts apply to this invoice item. Defaults to
    false for prorations or negative invoice items, and true for all
    other invoice items.

`discounts`

:   The coupons to redeem into discounts for the invoice item or invoice
    line item.

`invoice`

:   The ID of an existing invoice to add this invoice item to. When left
    blank, the invoice item will be added to the next upcoming scheduled
    invoice. This is useful when adding invoice items in response to an
    invoice.created webhook. You can only add invoice items to draft
    invoices and there is a maximum of 250 items per invoice.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`period`

:   The period associated with this invoice item. When set to different
    values, the period will be rendered on the invoice.

`price`

:   The ID of the price object.

`price_data`

:   Data used to generate a new
    [Price](https://stripe.com/docs/api/prices){.perl-module} object
    inline.

`quantity`

:   Non-negative integer. The quantity of units for the invoice item.

`subscription`

:   The ID of a subscription to add this invoice item to. When left
    blank, the invoice item will be be added to the next upcoming
    scheduled invoice. When set, scheduled invoices for subscriptions
    other than the specified subscription will ignore the invoice item.
    Use this when you want to express that an invoice item has been
    accrued within the context of a particular subscription.

`tax_rates`

:   The tax rates which apply to the invoice item. When set, the
    `default_tax_rates` on the invoice do not apply to this invoice
    item.

`unit_amount`

:   The integer unit amount in JPY of the charge to be applied to the
    upcoming invoice. This `unit_amount` will be multiplied by the
    quantity to get the full amount. Passing in a negative `unit_amount`
    will reduce the `amount_due` on the invoice.

`unit_amount_decimal`

:   Same as `unit_amount`, but accepts a decimal value in JPY with at
    most 12 decimal places. Only one of `unit_amount` and
    `unit_amount_decimal` can be set.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoiceitems/create>

delete
------

        my $obj = $stripe->invoiceitems( delete => $args ) || die( $stripe->error );

Provided with a
[invoiceitem](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
invoiceitem. It returns the invoiceitem object that was deleted with its
property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoiceitems/delete>

list {#list-35}
----

        my $obj = $stripe->invoiceitems( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[invoiceitem](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}
object, this issue an api call to get the list of all invoiceitem.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`customer`

:   The identifier of the customer whose invoice items to return. If
    none is provided, all invoice items will be returned.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`invoice`

:   Only return invoice items belonging to this invoice. If none is
    provided, all invoice items will be returned. If specifying an
    invoice, no customer identifier is needed.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`pending`

:   Set to `true` to only show pending invoice items, which are not yet
    attached to any invoices. Set to `false` to only show invoice items
    already attached to invoices. If unspecified, no filter is applied.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoiceitems/list>

retrieve {#retrieve-35}
--------

        my $obj = $stripe->invoiceitems( retrieve => $args ) || die( $stripe->error );

Provided with a
[invoiceitem](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}
object or a hash reference, this will retrieve a Stripe invoiceitem and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoiceitems/retrieve>

update
------

        my $obj = $stripe->invoiceitems( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[invoiceitem](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}
object or a hash reference, this will update a Stripe invoiceitem and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::Item){.perl-module}

Possible parameters are:

`amount`

:   The integer amount in JPY of the charge to be applied to the
    upcoming invoice. If you want to apply a credit to the customer\'s
    account, pass a negative amount.

`description`

:   An arbitrary string which you can attach to the invoice item. The
    description is displayed in the invoice for easy tracking.

`discountable`

:   Controls whether discounts apply to this invoice item. Defaults to
    false for prorations or negative invoice items, and true for all
    other invoice items. Cannot be set to true for prorations.

`discounts`

:   The coupons & existing discounts which apply to the invoice item or
    invoice line item. Item discounts are applied before invoice
    discounts. Pass an empty string to remove previously-defined
    discounts.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`period`

:   The period associated with this invoice item. When set to different
    values, the period will be rendered on the invoice.

`price`

:   The ID of the price object.

`price_data`

:   Data used to generate a new
    [Price](https://stripe.com/docs/api/prices){.perl-module} object
    inline.

`quantity`

:   Non-negative integer. The quantity of units for the invoice item.

`tax_rates`

:   The tax rates which apply to the invoice item. When set, the
    `default_tax_rates` on the invoice do not apply to this invoice
    item. Pass an empty string to remove previously-defined tax rates.

`unit_amount`

:   The integer unit amount in JPY of the charge to be applied to the
    upcoming invoice. This unit*amount will be multiplied by the
    quantity to get the full amount. If you want to apply a credit to
    the customer\'s account, pass a negative unit*amount.

`unit_amount_decimal`

:   Same as `unit_amount`, but accepts a decimal value in JPY with at
    most 12 decimal places. Only one of `unit_amount` and
    `unit_amount_decimal` can be set.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoiceitems/update>

INVOICES
========

You can **create** **delete** **finalise** **lines** **lines\_upcoming**
**invoice\_write\_off** **upcoming** **pay** **retrieve** **send**
**update** **void** **list** invoices

create {#create-26}
------

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference, this will create a Stripe invoice and
return a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*customer* A Stripe customer id. This is required.

:   

*application\_fee\_amount* Integer

:   

*auto\_advance* Boolean

:   

*collection\_method* String. Either `charge_automatically`, or `send_invoice`

:   

*custom\_fields* An array of hash reference with key and value properties.

:   

*days\_until\_due* Integer

:   

*default\_payment\_method* A Stripe payment method id

:   

*default\_source* String

:   

*default\_tax\_rates* Array reference of decimal amount

:   

*description* Text

:   

*due\_date* Date or unix timestamp

:   

*footer* Text

:   

*metadata* An arbitrary hash reference

:   

*statement\_descriptor* Text

:   

*subscription* A Stripe subscription id

:   

*tax\_percent* Decimal value

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/create>

delete
------

This is to remove draft invoice. When provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will remove the draft
invoice and return a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*id* A Stripe draft invoice id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/delete>

finalise
--------

When provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will set the draft
invoice as finalised and return a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*id* A Stripe draft invoice id

:   

*auto\_advance* Boolean

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/finalize>

invoice\_write\_off
-------------------

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will write off an
invoice and return its
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*id* A Stripe invoice id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/mark_uncollectible>

lines
-----

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will retrieve the list
of invoice lines and return a
[Net:API::Stripe::List](Net:API::Stripe::List){.perl-module}

Possible parameters are:

*id* A Stripe invoice id. This is required.

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/invoice_lines>

lines\_upcoming
---------------

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will retrieve the list
of upcoming invoice lines and return a
[Net:API::Stripe::List](Net:API::Stripe::List){.perl-module}

Possible parameters are:

*customer* A Stripe customer id. This is required

:   

*coupon* String

:   

*ending\_before* A Stripe invoice id

:   

*invoice\_items* An array of hash with the following properties:

:   

    *amount*

    :   

    *currency*

    :   

    *description*

    :   

    *discountable*

    :   

    *invoiceitem*

    :   

    *metadata*

    :   

    *period.end*

    :   

    *period.start*

    :   

    *quantity*

    :   

    *tax\_rates*

    :   

    *unit\_amount*

    :   

    *unit\_amount\_decimal*

    :   

*limit* Integer

:   

*schedule* A Stripe schedule id

:   

*starting\_after* A Stripe invoice id

:   

*subscription* A Stripe subscription id

:   

*subscription\_billing\_cycle\_anchor* A timestamp

:   

*subscription\_cancel\_at* A timestamp

:   

*subscription\_cancel\_at\_period\_end* Boolean

:   

*subscription\_cancel\_now* Boolean

:   

*subscription\_default\_tax\_rates* Array of tax rates

:   

*subscription\_items* List of subscription items, each with an attached plan.

:   

*subscription\_prorate* String. If previewing an update to a subscription, this decides whether the preview will show the result of applying prorations or not. If set, one of subscription\_items or subscription, and one of subscription\_items or subscription\_trial\_end are required.

:   

*subscription\_proration\_behavior* String. Determines how to handle prorations when the billing cycle changes.

:   

*subscription\_proration\_date* Date/timestamp. If previewing an update to a subscription, and doing proration, subscription\_proration\_date forces the proration to be calculated as though the update was done at the specified time.

:   

*subscription\_start\_date* Date/timestamp.

:   

*subscription\_tax\_percent* Decimal

:   

*subscription\_trial\_end* Boolean. If set, one of subscription\_items or subscription is required.

:   

*subscription\_trial\_from\_plan* Boolean. Indicates if a planâs trial\_period\_days should be applied to the subscription. Setting this flag to true together with subscription\_trial\_end is not allowed.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/upcoming_invoice_lines>

list {#list-36}
----

Provided with an hash reference of parameters, this returns a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*collection\_method* String that can be charge\_automatically or send\_invoice.

:   

*created* Date or unix timestamp

:   

*customer* A Stripe customer id.

:   

*due\_date* Date / timestamp

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*starting\_after* A Stripe credit note id

:   

*status* String. Status of the invoice, which can be one of draft, open, paid, uncollectible and void

:   

*subscription* A Stripe subscription id.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/list>

pay
---

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will mark an invoice as
paid and return its
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*id* A Stripe invoice id. This is required.

:   

*forgive* Boolean

:   

*off\_session* Boolean

:   

*paid\_out\_of\_band* Boolean to signify this was paid outside of Stripe

:   

*payment\_method* A Stripe payment method id

:   

*source* A Stripe source id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/pay>

retrieve {#retrieve-36}
--------

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will retrieve an invoice
and return its
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*id* A Stripe invoice id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/retrieve>

send
----

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will send an invoice to
a recipient to get payment and return its
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*id* A Stripe invoice id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/send>

upcoming
--------

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will retrieve an
upcoming invoice and return its
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*customer* A Stripe customer id. This is required

:   

*coupon* String

:   

*invoice\_items* An array of hash with the following properties:

:   

    *amount*

    :   

    *currency*

    :   

    *description*

    :   

    *discountable*

    :   

    *invoiceitem*

    :   

    *metadata*

    :   

    *period.end*

    :   

    *period.start*

    :   

    *quantity*

    :   

    *tax\_rates*

    :   

    *unit\_amount*

    :   

    *unit\_amount\_decimal*

    :   

*schedule* A Stripe schedule id

:   

*subscription* A Stripe subscription id

:   

*subscription\_billing\_cycle\_anchor* A timestamp

:   

*subscription\_cancel\_at* A timestamp

:   

*subscription\_cancel\_at\_period\_end* Boolean

:   

*subscription\_cancel\_now* Boolean

:   

*subscription\_default\_tax\_rates* Array of tax rates

:   

*subscription\_items* List of subscription items, each with an attached plan.

:   

*subscription\_prorate* String. If previewing an update to a subscription, this decides whether the preview will show the result of applying prorations or not. If set, one of subscription\_items or subscription, and one of subscription\_items or subscription\_trial\_end are required.

:   

*subscription\_proration\_behavior* String. Determines how to handle prorations when the billing cycle changes.

:   

*subscription\_proration\_date* Date/timestamp. If previewing an update to a subscription, and doing proration, subscription\_proration\_date forces the proration to be calculated as though the update was done at the specified time.

:   

*subscription\_start\_date* Date/timestamp.

:   

*subscription\_tax\_percent* Decimal

:   

*subscription\_trial\_end* Boolean. If set, one of subscription\_items or subscription is required.

:   

*subscription\_trial\_from\_plan* Boolean. Indicates if a planâs trial\_period\_days should be applied to the subscription. Setting this flag to true together with subscription\_trial\_end is not allowed.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/upcoming>

update
------

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will update an invoice
and return its
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*id*

:   

*application\_fee\_amount* Integer

:   

*auto\_advance* Boolean

:   

*collection\_method* String. Either `charge_automatically`, or `send_invoice`

:   

*custom\_fields* An array of hash reference with key and value properties.

:   

*days\_until\_due* Integer

:   

*default\_payment\_method* A Stripe payment method id

:   

*default\_source* String

:   

*default\_tax\_rates* Array reference of decimal amount

:   

*description* Text

:   

*due\_date* Date or unix timestamp

:   

*footer* Text

:   

*metadata* An arbitrary hash reference

:   

*statement\_descriptor* Text

:   

*tax\_percent* Decimal value

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/update>

void
----

Provided with a
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object or an hash reference of parameters, this will void (i.e. cancel)
an invoice and return its
[Net::API::Stripe::Billing::Invoice](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice){.perl-module}
object.

Possible parameters are:

*id* A Stripe invoice id. This is required.

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/invoices/void>

ISSUING AUTHORIZATION
=====================

You can approve, decline, list, retrieve or update issuing authorization

approve
-------

        my $obj = $stripe->issuing_authorizations( approve => $args ) || die( $stripe->error );

Provided with a [issuing
authorization](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module},
or a hash reference, this will issue a approve api call.

Returns an approved Issuing `Authorization` object.

Possible parameters are:

`amount`

:   If the authorization\'s `pending_request.is_amount_controllable`
    property is `true`, you may provide this value to control how much
    to hold for the authorization. Must be positive (use
    [`decline`](https://stripe.com/docs/api/issuing/authorizations/decline){.perl-module}
    to decline an authorization request).

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/authorizations/approve>

decline
-------

        my $obj = $stripe->issuing_authorizations( decline => $args ) || die( $stripe->error );

Provided with a [issuing
authorization](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module},
or a hash reference, this will issue a decline api call.

Returns a declined Issuing `Authorization` object.

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/authorizations/decline>

list {#list-37}
----

        my $obj = $stripe->issuing_authorizations( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [issuing
authorization](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module}
object, this issue an api call to get the list of all issuing
authorization.

Possible parameters are:

`card`

:   Only return authorizations that belong to the given card.

`cardholder`

:   Only return authorizations that belong to the given cardholder.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return authorizations with the given status. One of `pending`,
    `closed`, or `reversed`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/authorizations/list>

retrieve {#retrieve-37}
--------

        my $obj = $stripe->issuing_authorizations( retrieve => $args ) || die( $stripe->error );

Provided with a [issuing
authorization](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module}
object or a hash reference, this will retrieve a Stripe issuing
authorization and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/authorizations/retrieve>

update
------

        my $obj = $stripe->issuing_authorizations( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [issuing
authorization](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module}
object or a hash reference, this will update a Stripe issuing
authorization and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Authorization){.perl-module}

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/authorizations/update>

ISSUING CARD
============

You can create, deliver, fail, list, retrieve, return, ship or update
issuing card

create {#create-27}
------

        my $obj = $stripe->issuing_cards( create => {
            cardholder => "ich_1KwsZz2eZvKYlo2Cz5eys5Kb",
            currency   => "usd",
            type       => "virtual", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Issuing::Card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module}
object or a hash reference, this will create a Stripe issuing card and
return an
[Net::API::Stripe::Issuing::Card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module}
object.

Possible parameters are:

`cardholder`

:   **required** The
    [Cardholder](https://stripe.com/docs/api#issuing_cardholder_object){.perl-module}
    object with which the card will be associated.

`currency`

:   **Required**. The currency for the card.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`replacement_for`

:   The card this is meant to be a replacement for (if any).

`replacement_reason`

:   If `replacement_for` is specified, this should indicate why that
    card is being replaced.

`shipping`

:   The address where the card will be shipped.

`spending_controls`

:   Rules that control spending for this card. Refer to our
    [documentation](https://stripe.com/docs/issuing/controls/spending-controls){.perl-module}
    for more details.

`status`

:   Whether authorizations can be approved on this card. Defaults to
    `inactive`.

`type`

:   **Required**. The type of card to issue. Possible values are
    `physical` or `virtual`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cards/create>

deliver
-------

        my $obj = $stripe->issuing_cards( deliver => $args ) || die( $stripe->error );

Provided with a [issuing
card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module},
or a hash reference, this will issue a deliver api call.

Returns an updated Issuing `Card` object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cards/test_mode_deliver>

fail
----

        my $obj = $stripe->issuing_cards( fail => $args ) || die( $stripe->error );

Provided with a [issuing
card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module},
or a hash reference, this will issue a fail api call.

Returns an updated Issuing `Card` object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cards/test_mode_fail>

list {#list-38}
----

        my $obj = $stripe->issuing_cards( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [issuing
card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module}
object, this issue an api call to get the list of all issuing card.

Possible parameters are:

`cardholder`

:   Only return cards belonging to the Cardholder with the provided ID.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`exp_month`

:   Only return cards that have the given expiration month.

`exp_year`

:   Only return cards that have the given expiration year.

`last4`

:   Only return cards that have the given last four digits.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return cards that have the given status. One of `active`,
    `inactive`, or `canceled`.

`type`

:   Only return cards that have the given type. One of `virtual` or
    `physical`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cards/list>

retrieve {#retrieve-38}
--------

        my $obj = $stripe->issuing_cards( retrieve => $args ) || die( $stripe->error );

Provided with a [issuing
card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module}
object or a hash reference, this will retrieve a Stripe issuing card and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cards/retrieve>

return
------

        my $obj = $stripe->issuing_cards( return => $args ) || die( $stripe->error );

Provided with a [issuing
card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module},
or a hash reference, this will issue a return api call.

Returns an updated Issuing `Card` object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cards/test_mode_return>

ship
----

        my $obj = $stripe->issuing_cards( ship => $args ) || die( $stripe->error );

Provided with a [issuing
card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module},
or a hash reference, this will issue a ship api call.

Returns an updated Issuing `Card` object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cards/test_mode_ship>

update
------

        my $obj = $stripe->issuing_cards( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [issuing
card](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module}
object or a hash reference, this will update a Stripe issuing card and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card){.perl-module}

Possible parameters are:

`cancellation_reason`

:   Reason why the `status` of this card is `canceled`.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`pin`

:   The desired new PIN for this card.

`spending_controls`

:   Rules that control spending for this card. Refer to our
    [documentation](https://stripe.com/docs/issuing/controls/spending-controls){.perl-module}
    for more details.

`status`

:   Dictates whether authorizations can be approved on this card. If
    this card is being canceled because it was lost or stolen, this
    information should be provided as `cancellation_reason`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cards/update>

ISSUING CARDHOLDER
==================

You can create, list, retrieve or update issuing cardholder

create {#create-28}
------

        my $obj = $stripe->issuing_cardholders( create => {
            billing      =>
            {
                address =>
                {
                    city        => "San Francisco",
                    country     => "US",
                    line1       => "1234 Main Street",
                    postal_code => "94111",
                    state       => "CA",
                }
            }
            email        => q{jenny.rosen@example.com},
            name         => "Jenny Rosen",
            phone_number => "+18888675309",
            type         => "individual", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Issuing::Card::Holder](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}
object or a hash reference, this will create a Stripe issuing cardholder
and return an
[Net::API::Stripe::Issuing::Card::Holder](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}
object.

Possible parameters are:

`billing`

:   **Required**. The cardholder\'s billing address.

`company`

:   Additional information about a `company` cardholder.

`email`

:   The cardholder\'s email address.

`individual`

:   Additional information about an `individual` cardholder.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   **Required**. The cardholder\'s name. This will be printed on cards
    issued to them. The maximum length of this field is 24 characters.
    This field cannot contain any special characters or numbers.

`phone_number`

:   The cardholder\'s phone number. This will be transformed to
    [E.164](https://en.wikipedia.org/wiki/E.164){.perl-module} if it is
    not provided in that format already. This is required for all
    cardholders who will be creating EU cards. See the [3D Secure
    documentation](https://stripe.com/docs/issuing/3d-secure#when-is-3d-secure-applied){.perl-module}
    for more details.

`spending_controls`

:   Rules that control spending across this cardholder\'s cards. Refer
    to our
    [documentation](https://stripe.com/docs/issuing/controls/spending-controls){.perl-module}
    for more details.

`status`

:   Specifies whether to permit authorizations on this cardholder\'s
    cards. Defaults to `active`.

`type`

:   **Required**. One of `individual` or `company`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cardholders/create>

list {#list-39}
----

        my $obj = $stripe->issuing_cardholders( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [issuing
cardholder](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}
object, this issue an api call to get the list of all issuing
cardholder.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`email`

:   Only return cardholders that have the given email address.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`phone_number`

:   Only return cardholders that have the given phone number.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return cardholders that have the given status. One of `active`,
    `inactive`, or `blocked`.

`type`

:   Only return cardholders that have the given type. One of
    `individual` or `company`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cardholders/list>

retrieve {#retrieve-39}
--------

        my $obj = $stripe->issuing_cardholders( retrieve => $args ) || die( $stripe->error );

Provided with a [issuing
cardholder](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}
object or a hash reference, this will retrieve a Stripe issuing
cardholder and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cardholders/retrieve>

update
------

        my $obj = $stripe->issuing_cardholders( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [issuing
cardholder](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}
object or a hash reference, this will update a Stripe issuing cardholder
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Card::Holder){.perl-module}

Possible parameters are:

`billing`

:   The cardholder\'s billing address.

`company`

:   Additional information about a `company` cardholder.

`email`

:   The cardholder\'s email address.

`individual`

:   Additional information about an `individual` cardholder.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`phone_number`

:   The cardholder\'s phone number. This is required for all cardholders
    who will be creating EU cards. See the [3D Secure
    documentation](https://stripe.com/docs/issuing/3d-secure){.perl-module}
    for more details.

`spending_controls`

:   Rules that control spending across this cardholder\'s cards. Refer
    to our
    [documentation](https://stripe.com/docs/issuing/controls/spending-controls){.perl-module}
    for more details.

`status`

:   Specifies whether to permit authorizations on this cardholder\'s
    cards.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/cardholders/update>

ISSUING DISPUTE
===============

You can create, list, retrieve, submit or update issuing dispute

create {#create-29}
------

        my $obj = $stripe->issuing_disputes( create => {
            evidence    =>
            {
                fraudulent =>
                {
                    explanation => "Purchase was unrecognized.",
                }
                reason     => "fraudulent",
            }
            transaction => "ipi_1JIdAl2eZvKYlo2Cfr8US8uB", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Issuing::Dispute](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module}
object or a hash reference, this will create a Stripe issuing dispute
and return an
[Net::API::Stripe::Issuing::Dispute](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module}
object.

Possible parameters are:

`evidence`

:   Evidence provided for the dispute.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`transaction`

:   The ID of the issuing transaction to create a dispute for. For
    transaction on Treasury FinancialAccounts, use
    `treasury.received_debit`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/disputes/create>

list {#list-40}
----

        my $obj = $stripe->issuing_disputes( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [issuing
dispute](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module}
object, this issue an api call to get the list of all issuing dispute.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Select Issuing disputes with the given status.

`transaction`

:   Select the Issuing dispute for the given transaction.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/disputes/list>

retrieve {#retrieve-40}
--------

        my $obj = $stripe->issuing_disputes( retrieve => $args ) || die( $stripe->error );

Provided with a [issuing
dispute](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module}
object or a hash reference, this will retrieve a Stripe issuing dispute
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/disputes/retrieve>

submit
------

        my $obj = $stripe->issuing_disputes( submit => $args ) || die( $stripe->error );

Provided with a [issuing
dispute](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module},
or a hash reference, this will issue a submit api call.

Returns an Issuing `Dispute` object in `submitted` status if submission
succeeds.

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/dispute/submit>

update
------

        my $obj = $stripe->issuing_disputes( update => {
            evidence =>
            {
                not_received =>
                {
                    expected_at         => "1590000000",
                    explanation         => "",
                    product_description => "Baseball cap",
                    product_type        => "merchandise",
                }
                reason       => "not_received",
            } } ) || die( $stripe->error );

Provided with a [issuing
dispute](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module}
object or a hash reference, this will update a Stripe issuing dispute
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Dispute){.perl-module}

Possible parameters are:

`evidence`

:   Evidence provided for the dispute.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/disputes/update>

ISSUING TRANSACTION
===================

You can list, retrieve or update issuing transaction

list {#list-41}
----

        my $obj = $stripe->issuing_transactions( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [issuing
transaction](https://metacpan.org/pod/Net::API::Stripe::Issuing::Transaction){.perl-module}
object, this issue an api call to get the list of all issuing
transaction.

Possible parameters are:

`card`

:   Only return transactions that belong to the given card.

`cardholder`

:   Only return transactions that belong to the given cardholder.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`type`

:   Only return transactions that have the given type. One of `capture`
    or `refund`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/transactions/list>

retrieve {#retrieve-41}
--------

        my $obj = $stripe->issuing_transactions( retrieve => $args ) || die( $stripe->error );

Provided with a [issuing
transaction](https://metacpan.org/pod/Net::API::Stripe::Issuing::Transaction){.perl-module}
object or a hash reference, this will retrieve a Stripe issuing
transaction and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Transaction){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/transactions/retrieve>

update
------

        my $obj = $stripe->issuing_transactions( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [issuing
transaction](https://metacpan.org/pod/Net::API::Stripe::Issuing::Transaction){.perl-module}
object or a hash reference, this will update a Stripe issuing
transaction and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Issuing::Transaction){.perl-module}

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/issuing/transactions/update>

LINE ITEM
=========

You can lines line item

lines
-----

        my $obj = $stripe->line_items( lines => $args ) || die( $stripe->error );

Provided with a [line
item](https://metacpan.org/pod/Net::API::Stripe::Billing::Invoice::LineItem){.perl-module},
or a hash reference, this will issue a lines api call.

Returns a list of [line\_item
objects](https://stripe.com/docs/api/invoices/line_item){.perl-module}.

Possible parameters are:

`automatic_tax`

:   Settings for automatic tax lookup for this invoice preview.

`coupon`

:   The code of the coupon to apply. If `subscription` or
    `subscription_items` is provided, the invoice returned will preview
    updating or creating a subscription with that coupon. Otherwise, it
    will preview applying that coupon to the customer for the next
    upcoming invoice from among the customer\'s subscriptions. The
    invoice can be previewed without a coupon by passing this value as
    an empty string.

`currency`

:   The currency to preview this invoice in. Defaults to that of
    `customer` if not specified.

`customer`

:   **Required if subscription unset** The identifier of the customer
    whose upcoming invoice you\'d like to retrieve.

`customer_details`

:   **Required if subscription and customer unset** Details about the
    customer you want to invoice or overrides for an existing customer.

`discounts`

:   The coupons to redeem into discounts for the invoice preview. If not
    specified, inherits the discount from the customer or subscription.
    This only works for coupons directly applied to the invoice. To
    apply a coupon to a subscription, you must use the `coupon`
    parameter instead. Pass an empty string to avoid inheriting any
    discounts. To preview the upcoming invoice for a subscription that
    hasn\'t been created, use `coupon` instead.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`invoice_items`

:   List of invoice items to add or update in the upcoming invoice
    preview.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`schedule`

:   The identifier of the unstarted schedule whose upcoming invoice
    you\'d like to retrieve. Cannot be used with subscription or
    subscription fields.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`subscription`

:   The identifier of the subscription for which you\'d like to retrieve
    the upcoming invoice. If not provided, but a `subscription_items` is
    provided, you will preview creating a subscription with those items.
    If neither `subscription` nor `subscription_items` is provided, you
    will retrieve the next upcoming invoice from among the customer\'s
    subscriptions.

`subscription_billing_cycle_anchor`

:   For new subscriptions, a future timestamp to anchor the
    subscription\'s [billing
    cycle](https://stripe.com/docs/subscriptions/billing-cycle){.perl-module}.
    This is used to determine the date of the first full invoice, and,
    for plans with `month` or `year` intervals, the day of the month for
    subsequent invoices. For existing subscriptions, the value can only
    be set to `now` or `unchanged`.

`subscription_cancel_at`

:   Timestamp indicating when the subscription should be scheduled to
    cancel. Will prorate if within the current period and prorations
    have been enabled using `proration_behavior`.

`subscription_cancel_at_period_end`

:   Boolean indicating whether this subscription should cancel at the
    end of the current period.

`subscription_cancel_now`

:   This simulates the subscription being canceled or expired
    immediately.

`subscription_default_tax_rates`

:   If provided, the invoice returned will preview updating or creating
    a subscription with these default tax rates. The default tax rates
    will apply to any line item that does not have `tax_rates` set.

`subscription_items`

:   A list of up to 20 subscription items, each with an attached price.

`subscription_proration_behavior`

:   Determines how to handle
    [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations){.perl-module}
    when the billing cycle changes (e.g., when switching plans,
    resetting `billing_cycle_anchor=now`, or starting a trial), or if an
    item\'s `quantity` changes.

`subscription_proration_date`

:   If previewing an update to a subscription, and doing proration,
    `subscription_proration_date` forces the proration to be calculated
    as though the update was done at the specified time. The time given
    must be within the current subscription period, and cannot be before
    the subscription was on its current plan. If set, `subscription`,
    and one of `subscription_items`, or `subscription_trial_end` are
    required. Also, `subscription_proration_behavior` cannot be set to
    \'none\'.

`subscription_start_date`

:   Date a subscription is intended to start (can be future or past)

`subscription_trial_end`

:   If provided, the invoice returned will preview updating or creating
    a subscription with that trial end. If set, one of
    `subscription_items` or `subscription` is required.

`subscription_trial_from_plan`

:   Indicates if a plan\'s `trial_period_days` should be applied to the
    subscription. Setting `subscription_trial_end` per subscription is
    preferred, and this defaults to `false`. Setting this flag to `true`
    together with `subscription_trial_end` is not allowed. See [Using
    trial periods on
    subscriptions](https://stripe.com/docs/billing/subscriptions/trials){.perl-module}
    to learn more.

More information from Stripe api documentation at
<https://stripe.com/docs/api/invoices/upcoming_invoice_lines>

LOGIN LINK
==========

You can create login link

create {#create-30}
------

        my $obj = $stripe->login_links( create => $args ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::Account::LoginLink](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::LoginLink){.perl-module}
object or a hash reference, this will create a Stripe login link and
return an
[Net::API::Stripe::Connect::Account::LoginLink](https://metacpan.org/pod/Net::API::Stripe::Connect::Account::LoginLink){.perl-module}
object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/account/create_login_link>

MANDATE
=======

You can retrieve mandate

retrieve {#retrieve-42}
--------

        my $obj = $stripe->mandates( retrieve => $args ) || die( $stripe->error );

Provided with a
[mandate](https://metacpan.org/pod/Net::API::Stripe::Mandate){.perl-module}
object or a hash reference, this will retrieve a Stripe mandate and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Mandate){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/mandates/retrieve>

PAYMENT INTENT
==============

You can apply\_customer\_balance, cancel, capture, confirm, create,
increment, increment\_authorization, list, reconcile, retrieve, search,
update, verify or verify\_microdeposits payment intent

apply\_customer\_balance
------------------------

        my $obj = $stripe->payment_intents( apply_customer_balance => $args ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module},
or a hash reference, this will issue a apply\_customer\_balance api
call.

Returns a PaymentIntent object.

Possible parameters are:

`amount`

:   Amount intended to be applied to this PaymentIntent from the
    customer's cash balance.

    A positive integer representing how much to charge in the [smallest
    currency
    unit](https://stripe.com/docs/currencies#zero-decimal){.perl-module}
    (e.g., 100 cents to charge \$1.00 or 100 to charge ¥100, a
    zero-decimal currency).

    The maximum amount is the amount of the PaymentIntent.

    When omitted, the amount defaults to the remaining amount requested
    on the PaymentIntent.

`currency`

:   Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/apply_customer_balance>

cancel
------

        my $obj = $stripe->payment_intents( cancel => $args ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns a PaymentIntent object if the cancellation succeeded. Returns an
error if the PaymentIntent has already been canceled or is not in a
cancelable state.

Possible parameters are:

`cancellation_reason`

:   Reason for canceling this PaymentIntent. Possible values are
    `duplicate`, `fraudulent`, `requested_by_customer`, or `abandoned`

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/cancel>

capture
-------

        my $obj = $stripe->payment_intents( capture => $args ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module},
or a hash reference, this will issue a capture api call.

Returns a PaymentIntent object with `status="succeeded"` if the
PaymentIntent was capturable. Returns an error if the PaymentIntent was
not capturable or an invalid amount to capture was provided.

Possible parameters are:

`amount_to_capture`

:   The amount to capture from the PaymentIntent, which must be less
    than or equal to the original amount. Any additional amount will be
    automatically refunded. Defaults to the full `amount_capturable` if
    not provided.

`application_fee_amount`

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    owner\'s Stripe account. The amount of the application fee collected
    will be capped at the total payment amount. For more information,
    see the PaymentIntents [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

`statement_descriptor`

:   For non-card charges, you can use this value as the complete
    description that appears on your customers' statements. Must contain
    at least one letter, maximum 22 characters.

`statement_descriptor_suffix`

:   Provides information about a card payment that customers see on
    their statements. Concatenated with the prefix (shortened
    descriptor) or statement descriptor that's set on the account to
    form the complete statement descriptor. Maximum 22 characters for
    the concatenated descriptor.

`transfer_data`

:   The parameters used to automatically create a Transfer when the
    payment is captured. For more information, see the PaymentIntents
    [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/capture>

confirm
-------

        my $obj = $stripe->payment_intents( confirm => {
            payment_method => "pm_card_visa", } ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module},
or a hash reference, this will issue a confirm api call.

Returns the resulting PaymentIntent after all possible transitions are
applied.

Possible parameters are:

`capture_method`

:   Controls when the funds will be captured from the customer\'s
    account.

`error_on_requires_action`

:   Set to `true` to fail the payment attempt if the PaymentIntent
    transitions into `requires_action`. This parameter is intended for
    simpler integrations that do not handle customer actions, like
    [saving cards without
    authentication](https://stripe.com/docs/payments/save-card-without-authentication){.perl-module}.

`mandate`

:   ID of the mandate to be used for this payment.

`mandate_data`

:   This hash contains details about the Mandate to create

`off_session`

:   Set to `true` to indicate that the customer is not in your checkout
    flow during this payment attempt, and therefore is unable to
    authenticate. This parameter is intended for scenarios where you
    collect card details and [charge them
    later](https://stripe.com/docs/payments/cards/charging-saved-cards){.perl-module}.

`payment_method`

:   ID of the payment method (a PaymentMethod, Card, or [compatible
    Source](https://stripe.com/docs/payments/payment-methods/transitioning#compatibility){.perl-module}
    object) to attach to this PaymentIntent.

`payment_method_data`

:   If provided, this hash will be used to create a PaymentMethod. The
    new PaymentMethod will appear in the
    [payment\_method](https://stripe.com/docs/api/payment_intents/object#payment_intent_object-payment_method){.perl-module}
    property on the PaymentIntent.

`payment_method_options`

:   Payment-method-specific configuration for this PaymentIntent.

`payment_method_types`

:   The list of payment method types (e.g. card) that this PaymentIntent
    is allowed to use. Use automatic*payment*methods to manage payment
    methods from the [Stripe
    Dashboard](https://dashboard.stripe.com/settings/payment_methods){.perl-module}.

`radar_options`

:   Options to configure Radar. See [Radar
    Session](https://stripe.com/docs/radar/radar-session){.perl-module}
    for more information.

`receipt_email`

:   Email address that the receipt for the resulting payment will be
    sent to. If `receipt_email` is specified for a payment in live mode,
    a receipt will be sent regardless of your [email
    settings](https://dashboard.stripe.com/account/emails){.perl-module}.

`return_url`

:   The URL to redirect your customer back to after they authenticate or
    cancel their payment on the payment method\'s app or site. If you\'d
    prefer to redirect to a mobile application, you can alternatively
    supply an application URI scheme. This parameter is only used for
    cards and other redirect-based payment methods.

`setup_future_usage`

:   Indicates that you intend to make future payments with this
    PaymentIntent\'s payment method.

    Providing this parameter will [attach the payment
    method](https://stripe.com/docs/payments/save-during-payment){.perl-module}
    to the PaymentIntent\'s Customer, if present, after the
    PaymentIntent is confirmed and any required actions from the user
    are complete. If no Customer was provided, the payment method can
    still be
    [attached](https://stripe.com/docs/api/payment_methods/attach){.perl-module}
    to a Customer after the transaction completes.

    When processing card payments, Stripe also uses `setup_future_usage`
    to dynamically optimize your payment flow and comply with regional
    legislation and network rules, such as
    [SCA](https://stripe.com/docs/strong-customer-authentication){.perl-module}.

    If `setup_future_usage` is already set and you are performing a
    request using a publishable key, you may only update the value from
    `on_session` to `off_session`.

`shipping`

:   Shipping information for this PaymentIntent.

`use_stripe_sdk`

:   Set to `true` only when using manual confirmation and the iOS or
    Android SDKs to handle additional authentication steps.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/confirm>

create {#create-31}
------

        my $obj = $stripe->payment_intents( create => {
            amount               => "2000",
            currency             => "usd",
            payment_method_types => [qw( card )], } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or a hash reference, this will create a Stripe payment intent and
return an
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

Possible parameters are:

`amount`

:   **Required**. Amount intended to be collected by this PaymentIntent.
    A positive integer representing how much to charge in the [smallest
    currency
    unit](https://stripe.com/docs/currencies#zero-decimal){.perl-module}
    (e.g., 100 cents to charge \$1.00 or 100 to charge ¥100, a
    zero-decimal currency). The minimum amount is \$0.50 US or
    [equivalent in charge
    currency](https://stripe.com/docs/currencies#minimum-and-maximum-charge-amounts){.perl-module}.
    The amount value supports up to eight digits (e.g., a value of
    99999999 for a USD charge of \$999,999.99).

`application_fee_amount`

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    owner\'s Stripe account. The amount of the application fee collected
    will be capped at the total payment amount. For more information,
    see the PaymentIntents [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

`automatic_payment_methods`

:   When enabled, this PaymentIntent will accept payment methods that
    you have enabled in the Dashboard and are compatible with this
    PaymentIntent\'s other parameters.

`capture_method`

:   Controls when the funds will be captured from the customer\'s
    account.

`confirm`

:   Set to `true` to attempt to
    [confirm](https://stripe.com/docs/api/payment_intents/confirm){.perl-module}
    this PaymentIntent immediately. This parameter defaults to `false`.
    When creating and confirming a PaymentIntent at the same time,
    parameters available in the
    [confirm](https://stripe.com/docs/api/payment_intents/confirm){.perl-module}
    API may also be provided.

`confirmation_method`

:   

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`customer`

:   ID of the Customer this PaymentIntent belongs to, if one exists.

    Payment methods attached to other Customers cannot be used with this
    PaymentIntent.

    If present in combination with
    [setup*future*usage](https://stripe.com/docs/api/payment_intents/object#payment_intent_object-setup_future_usage){.perl-module},
    this PaymentIntent\'s payment method will be attached to the
    Customer after the PaymentIntent has been confirmed and any required
    actions from the user are complete.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`error_on_requires_action`

:   Set to `true` to fail the payment attempt if the PaymentIntent
    transitions into `requires_action`. This parameter is intended for
    simpler integrations that do not handle customer actions, like
    [saving cards without
    authentication](https://stripe.com/docs/payments/save-card-without-authentication){.perl-module}.
    This parameter can only be used with
    [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm){.perl-module}.

`mandate`

:   ID of the mandate to be used for this payment. This parameter can
    only be used with
    [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm){.perl-module}.

`mandate_data`

:   This hash contains details about the Mandate to create. This
    parameter can only be used with
    [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm){.perl-module}.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`off_session`

:   Set to `true` to indicate that the customer is not in your checkout
    flow during this payment attempt, and therefore is unable to
    authenticate. This parameter is intended for scenarios where you
    collect card details and [charge them
    later](https://stripe.com/docs/payments/cards/charging-saved-cards){.perl-module}.
    This parameter can only be used with
    [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm){.perl-module}.

`on_behalf_of`

:   The Stripe account ID for which these funds are intended. For
    details, see the PaymentIntents [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

`payment_method`

:   ID of the payment method (a PaymentMethod, Card, or [compatible
    Source](https://stripe.com/docs/payments/payment-methods/transitioning#compatibility){.perl-module}
    object) to attach to this PaymentIntent.

    If this parameter is omitted with `confirm=true`,
    `customer.default_source` will be attached as this PaymentIntent\'s
    payment instrument to improve the migration experience for users of
    the Charges API. We recommend that you explicitly provide the
    `payment_method` going forward.

`payment_method_data`

:   If provided, this hash will be used to create a PaymentMethod. The
    new PaymentMethod will appear in the
    [payment\_method](https://stripe.com/docs/api/payment_intents/object#payment_intent_object-payment_method){.perl-module}
    property on the PaymentIntent.

`payment_method_options`

:   Payment-method-specific configuration for this PaymentIntent.

`payment_method_types`

:   The list of [payment method
    types](https://stripe.com/docs/payments/payment-methods/overview){.perl-module}
    that this PaymentIntent is allowed to use. If this is not provided,
    defaults to `[“card”]`. Valid payment method types include:
    `acss_debit`, `affirm`, `afterpay_clearpay`, `alipay`,
    `au_becs_debit`, `bacs_debit`, `bancontact`, `blik`, `boleto`,
    `card`, `card_present`, `eps`, `fpx`, `giropay`, `grabpay`, `ideal`,
    `klarna`, `konbini`, `link`, `oxxo`, `p24`, `paynow`, `promptpay`,
    `sepa_debit`, `sofort`, `us_bank_account`, and `wechat_pay`.

`radar_options`

:   Options to configure Radar. See [Radar
    Session](https://stripe.com/docs/radar/radar-session){.perl-module}
    for more information.

`receipt_email`

:   Email address that the receipt for the resulting payment will be
    sent to. If `receipt_email` is specified for a payment in live mode,
    a receipt will be sent regardless of your [email
    settings](https://dashboard.stripe.com/account/emails){.perl-module}.

`return_url`

:   The URL to redirect your customer back to after they authenticate or
    cancel their payment on the payment method\'s app or site. If you\'d
    prefer to redirect to a mobile application, you can alternatively
    supply an application URI scheme. This parameter can only be used
    with
    [`confirm=true`](https://stripe.com/docs/api/payment_intents/create#create_payment_intent-confirm){.perl-module}.

`setup_future_usage`

:   Indicates that you intend to make future payments with this
    PaymentIntent\'s payment method.

    Providing this parameter will [attach the payment
    method](https://stripe.com/docs/payments/save-during-payment){.perl-module}
    to the PaymentIntent\'s Customer, if present, after the
    PaymentIntent is confirmed and any required actions from the user
    are complete. If no Customer was provided, the payment method can
    still be
    [attached](https://stripe.com/docs/api/payment_methods/attach){.perl-module}
    to a Customer after the transaction completes.

    When processing card payments, Stripe also uses `setup_future_usage`
    to dynamically optimize your payment flow and comply with regional
    legislation and network rules, such as
    [SCA](https://stripe.com/docs/strong-customer-authentication){.perl-module}.

`shipping`

:   Shipping information for this PaymentIntent.

`statement_descriptor`

:   For non-card charges, you can use this value as the complete
    description that appears on your customers' statements. Must contain
    at least one letter, maximum 22 characters.

`statement_descriptor_suffix`

:   Provides information about a card payment that customers see on
    their statements. Concatenated with the prefix (shortened
    descriptor) or statement descriptor that's set on the account to
    form the complete statement descriptor. Maximum 22 characters for
    the concatenated descriptor.

`transfer_data`

:   The parameters used to automatically create a Transfer when the
    payment succeeds. For more information, see the PaymentIntents [use
    case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

`transfer_group`

:   A string that identifies the resulting payment as part of a group.
    See the PaymentIntents [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}
    for details.

`use_stripe_sdk`

:   Set to `true` only when using manual confirmation and the iOS or
    Android SDKs to handle additional authentication steps.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/create>

increment\_authorization
------------------------

        my $obj = $stripe->payment_intents( increment_authorization => {
            amount => "2099", } ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module},
or a hash reference, this will issue a increment\_authorization api
call.

Returns a PaymentIntent object with the updated amount if the
incremental authorization succeeded. Returns an error if the incremental
authorization failed or the PaymentIntent isn't eligible for incremental
authorizations.

Possible parameters are:

`amount`

:   **Required**. The updated total amount you intend to collect from
    the cardholder. This amount must be greater than the currently
    authorized amount.

`application_fee_amount`

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    owner\'s Stripe account. The amount of the application fee collected
    will be capped at the total payment amount. For more information,
    see the PaymentIntents [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`transfer_data`

:   The parameters used to automatically create a Transfer when the
    payment is captured. For more information, see the PaymentIntents
    [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/increment_authorization>

list {#list-42}
----

        my $obj = $stripe->payment_intents( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object, this issue an api call to get the list of all payment intent.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`customer`

:   Only return PaymentIntents for the customer specified by this
    customer ID.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/list>

retrieve {#retrieve-43}
--------

        my $obj = $stripe->payment_intents( retrieve => $args ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or a hash reference, this will retrieve a Stripe payment intent
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}

Possible parameters are:

`client_secret`

:   **Required**. The client secret of the PaymentIntent. Required if a
    publishable key is used to retrieve the source.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/retrieve>

search
------

        my $obj = $stripe->payment_intents( search => {
            query => "status:'succeeded' AND metadata['order_id']:'6735'", } ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module},
or a hash reference, this will issue a search api call.

A dictionary with a `data` property that contains an array of up to
`limit` PaymentIntents. If no objects match the query, the resulting
array will be empty. See the related guide on [expanding properties in
lists](https://stripe.com/docs/expand#lists){.perl-module}.

Possible parameters are:

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`page`

:   A cursor for pagination across multiple pages of results. Don\'t
    include this parameter on the first call. Use the next\_page value
    returned in a previous response to request subsequent results.

`query`

:   **Required**. The search query string. See [search query
    language](https://stripe.com/docs/search#search-query-language){.perl-module}
    and the list of supported [query fields for payment
    intents](https://stripe.com/docs/search#query-fields-for-payment-intents){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/search>

update
------

        my $obj = $stripe->payment_intents( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or a hash reference, this will update a Stripe payment intent and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}

Possible parameters are:

`amount`

:   Amount intended to be collected by this PaymentIntent. A positive
    integer representing how much to charge in the [smallest currency
    unit](https://stripe.com/docs/currencies#zero-decimal){.perl-module}
    (e.g., 100 cents to charge \$1.00 or 100 to charge ¥100, a
    zero-decimal currency). The minimum amount is \$0.50 US or
    [equivalent in charge
    currency](https://stripe.com/docs/currencies#minimum-and-maximum-charge-amounts){.perl-module}.
    The amount value supports up to eight digits (e.g., a value of
    99999999 for a USD charge of \$999,999.99).

`application_fee_amount`

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    owner\'s Stripe account. The amount of the application fee collected
    will be capped at the total payment amount. For more information,
    see the PaymentIntents [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

`capture_method`

:   Controls when the funds will be captured from the customer\'s
    account.

`currency`

:   Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`customer`

:   ID of the Customer this PaymentIntent belongs to, if one exists.

    Payment methods attached to other Customers cannot be used with this
    PaymentIntent.

    If present in combination with
    [setup*future*usage](https://stripe.com/docs/api/payment_intents/object#payment_intent_object-setup_future_usage){.perl-module},
    this PaymentIntent\'s payment method will be attached to the
    Customer after the PaymentIntent has been confirmed and any required
    actions from the user are complete.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`payment_method`

:   ID of the payment method (a PaymentMethod, Card, or [compatible
    Source](https://stripe.com/docs/payments/payment-methods/transitioning#compatibility){.perl-module}
    object) to attach to this PaymentIntent.

`payment_method_data`

:   If provided, this hash will be used to create a PaymentMethod. The
    new PaymentMethod will appear in the
    [payment\_method](https://stripe.com/docs/api/payment_intents/object#payment_intent_object-payment_method){.perl-module}
    property on the PaymentIntent.

`payment_method_options`

:   Payment-method-specific configuration for this PaymentIntent.

`payment_method_types`

:   The list of payment method types (e.g. card) that this PaymentIntent
    is allowed to use. Use automatic*payment*methods to manage payment
    methods from the [Stripe
    Dashboard](https://dashboard.stripe.com/settings/payment_methods){.perl-module}.

`receipt_email`

:   Email address that the receipt for the resulting payment will be
    sent to. If `receipt_email` is specified for a payment in live mode,
    a receipt will be sent regardless of your [email
    settings](https://dashboard.stripe.com/account/emails){.perl-module}.

`setup_future_usage`

:   Indicates that you intend to make future payments with this
    PaymentIntent\'s payment method.

    Providing this parameter will [attach the payment
    method](https://stripe.com/docs/payments/save-during-payment){.perl-module}
    to the PaymentIntent\'s Customer, if present, after the
    PaymentIntent is confirmed and any required actions from the user
    are complete. If no Customer was provided, the payment method can
    still be
    [attached](https://stripe.com/docs/api/payment_methods/attach){.perl-module}
    to a Customer after the transaction completes.

    When processing card payments, Stripe also uses `setup_future_usage`
    to dynamically optimize your payment flow and comply with regional
    legislation and network rules, such as
    [SCA](https://stripe.com/docs/strong-customer-authentication){.perl-module}.

    If `setup_future_usage` is already set and you are performing a
    request using a publishable key, you may only update the value from
    `on_session` to `off_session`.

`shipping`

:   Shipping information for this PaymentIntent.

`statement_descriptor`

:   For non-card charges, you can use this value as the complete
    description that appears on your customers' statements. Must contain
    at least one letter, maximum 22 characters.

`statement_descriptor_suffix`

:   Provides information about a card payment that customers see on
    their statements. Concatenated with the prefix (shortened
    descriptor) or statement descriptor that's set on the account to
    form the complete statement descriptor. Maximum 22 characters for
    the concatenated descriptor.

`transfer_data`

:   The parameters used to automatically create a Transfer when the
    payment succeeds. For more information, see the PaymentIntents [use
    case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}.

`transfer_group`

:   A string that identifies the resulting payment as part of a group.
    `transfer_group` may only be provided if it has not been set. See
    the PaymentIntents [use case for connected
    accounts](https://stripe.com/docs/payments/connected-accounts){.perl-module}
    for details.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/update>

verify\_microdeposits
---------------------

        my $obj = $stripe->payment_intents( verify_microdeposits => $args ) || die( $stripe->error );

Provided with a [payment
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module},
or a hash reference, this will issue a verify\_microdeposits api call.

Returns a PaymentIntent object.

Possible parameters are:

`amounts`

:   Two positive integers, in *cents*, equal to the values of the
    microdeposits sent to the bank account.

`client_secret`

:   **Required**. The client secret of the PaymentIntent.

`descriptor_code`

:   A six-character code starting with SM present in the microdeposit
    sent to the bank account.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_intents/verify_microdeposits>

PAYMENT INTENTS
===============

You can **create** **retrieve** **update** **confirm** **capture**
**cancel** **list** payment intents

cancel
------

Provided with a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or an hash reference, this will cancel the Stripe payment intent
and return a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

Possible parameters are:

*cancellation\_reason* String

:   Reason for canceling this PaymentIntent. Possible values are
    `duplicate`, `fraudulent`, `requested_by_customer`, or `abandoned`

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_intents/cancel>

capture
-------

Provided with a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or an hash reference, this will capture the Stripe payment intent
and return a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

Possible parameters are:

*amount\_to\_capture* Integer

:   The amount to capture from the PaymentIntent, which must be less
    than or equal to the original amount. Any additional amount will be
    automatically refunded. Defaults to the full amount\_capturable if
    not provided.

*application\_fee\_amount* Integer

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    ownerâs Stripe account. The amount of the application fee
    collected will be capped at the total payment amount. For more
    information, see the PaymentIntents use case for connected accounts.

*statement\_descriptor* String

:   For non-card charges, you can use this value as the complete
    description that appears on your customersâ statements. Must
    contain at least one letter, maximum 22 characters.

*statement\_descriptor\_suffix* String

:   Provides information about a card payment that customers see on
    their statements. Concatenated with the prefix (shortened
    descriptor) or statement descriptor thatâs set on the account to
    form the complete statement descriptor. Maximum 22 characters for
    the concatenated descriptor.

*transfer\_data* Hash

:   The parameters used to automatically create a Transfer when the
    payment is captured. For more information, see the PaymentIntents
    use case for connected accounts.

    It has the following properties:

    *amount* Integer

    :   The amount that will be transferred automatically when a charge
        succeeds.

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_intents/capture>

confirm
-------

Provided with a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or an hash reference, this will confirm the Stripe payment intent
and return a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

Possible parameters are:

*error\_on\_requires\_action* Boolean

:   Set to true to fail the payment attempt if the PaymentIntent
    transitions into requires\_action. This parameter is intended for
    simpler integrations that do not handle customer actions, like
    saving cards without authentication.

*mandate* String

:   ID of the mandate to be used for this payment.

*mandate\_data* Hash

:   This hash contains details about the Mandate to create. This
    parameter can only be used with confirm=true.

    *customer\_acceptance* Hash required

    :   This hash contains details about the customer acceptance of the
        Mandate.

        *type* String required

        :   The type of customer acceptance information included with
            the Mandate. One of online or offline.

        *accepted\_at* Datetime

        :   The time at which the customer accepted the Mandate.

        *offline* Hash

        :   If this is a Mandate accepted offline, this hash contains
            details about the offline acceptance.

        *online* Hash

        :   If this is a Mandate accepted online, this hash contains
            details about the online acceptance.

            *ip\_address* String

            :   The IP address from which the Mandate was accepted by
                the customer.

            *user\_agent* String

            :   The user agent of the browser from which the Mandate was
                accepted by the customer.

*off\_session* Boolean

:   Set to true to indicate that the customer is not in your checkout
    flow during this payment attempt, and therefore is unable to
    authenticate. This parameter is intended for scenarios where you
    collect card details and charge them later.

*payment\_method* String

:   ID of the payment method (a PaymentMethod, Card, or compatible
    Source object) to attach to this PaymentIntent.

*payment\_method\_data* Hash

:   If provided, this hash will be used to create a PaymentMethod. The
    new PaymentMethod will appear in the payment\_method property on the
    PaymentIntent.

    *alipay* Hash

    :   If this is an Alipay PaymentMethod, this hash contains details
        about the Alipay payment method.

        No documented property.

    *au\_becs\_debit* Hash

    :   If this is an au\_becs\_debit PaymentMethod, this hash contains
        details about the bank account.

        *account\_number* String

        :   The account number for the bank account.

        *bsb\_number* String

        :   Bank-State-Branch number of the bank account.

    *bacs\_debit* Hash

    :   If this is a bacs\_debit PaymentMethod, this hash contains
        details about the Bacs Direct Debit bank account.

        *account\_number* String

        :   Account number of the bank account that the funds will be
            debited from.

        *sort\_code* String

        :   Sort code of the bank account. (e.g., 10-20-30)

    *bancontact* Hash

    :   If this is a bancontact PaymentMethod, this hash contains
        details about the Bancontact payment method.

        No documented property.

    *billing\_details* Hash

    :   Billing information associated with the PaymentMethod that may
        be used or required by particular types of payment methods.

        *address* Hash

        :   Billing address.

            *city* String

            :   City, district, suburb, town, or village.

            *country* String

            :   Two-letter country code (ISO 3166-1 alpha-2).

            *line1* String

            :   Address line 1 (e.g., street, PO Box, or company name).

            *line2* String

            :   Address line 2 (e.g., apartment, suite, unit, or
                building).

            *postal\_code* String

            :   ZIP or postal code.

            *state* String

            :   State, county, province, or region.

        *email* String

        :   Email address.

        *name* String

        :   Full name.

        *phone* String

        :   Billing phone number (including extension).

    *eps* hash

    :   If this is an eps PaymentMethod, this hash contains details
        about the EPS payment method.

        No documented property.

    *fpx* Hash

    :   If this is an fpx PaymentMethod, this hash contains details
        about the FPX payment method.

        *bank* String required

        :   The customerâs bank.

    *giropay* Hash

    :   If this is a giropay PaymentMethod, this hash contains details
        about the Giropay payment method.

        No documented property.

    *grabpay* Hash

    :   If this is a grabpay PaymentMethod, this hash contains details
        about the GrabPay payment method.

        No documented property.

    *ideal* Hash

    :   If this is an ideal PaymentMethod, this hash contains details
        about the iDEAL payment method.

        *bank* String

        :   The customerâs bank.

    *interac\_present* Hash

    :   If this is an interac\_present PaymentMethod, this hash contains
        details about the Interac Present payment method.

        No documented property.

    *metadata* Hash

    :   Set of key-value pairs that you can attach to an object. This
        can be useful for storing additional information about the
        object in a structured format. Individual keys can be unset by
        posting an empty value to them. All keys can be unset by posting
        an empty value to metadata.

    *oxxo* Hash

    :   If this is an oxxo PaymentMethod, this hash contains details
        about the OXXO payment method.

        No documented property.

    *p24* Hash

    :   If this is a p24 PaymentMethod, this hash contains details about
        the P24 payment method.

        *bank* String

        :   The customerâs bank.

    *sepa\_debit* Hash

    :   If this is a sepa\_debit PaymentMethod, this hash contains
        details about the SEPA debit bank account.

        *iban* String required

        :   IBAN of the bank account.

    *sofort* Hash

    :   If this is a sofort PaymentMethod, this hash contains details
        about the SOFORT payment method.

        *country* String required

        :   Two-letter ISO code representing the country the bank
            account is located in.

    *type* String

    :   The type of the PaymentMethod. An additional hash is included on
        the PaymentMethod with a name matching this value. It contains
        additional information specific to the PaymentMethod type.

*payment\_method\_options* Hash

:   Payment-method-specific configuration for this PaymentIntent.

    *alipay* Hash

    :   If this is a alipay PaymentMethod, this sub-hash contains
        details about the Alipay payment method options.

        No documented property.

    *bancontact* Hash

    :   If this is a bancontact PaymentMethod, this sub-hash contains
        details about the Bancontact payment method options.

        *preferred\_language* String

        :   Preferred language of the Bancontact authorization page that
            the customer is redirected to.

    *card* Hash

    :   Configuration for any card payments attempted on this
        PaymentIntent.

        *cvc\_token*

        :   A single-use cvc\_update Token that represents a card CVC
            value. When provided, the CVC value will be verified during
            the card payment attempt. This parameter can only be
            provided during confirmation.

        *installments*

        :   Installment configuration for payments attempted on this
            PaymentIntent (Mexico Only).

            For more information, see the [installments integration
            guide](https://stripe.com/docs/payments/installments){.perl-module}.

            *enabled* Boolean

            :   Setting to true enables installments for this
                PaymentIntent. This will cause the response to contain a
                list of available installment plans. Setting to false
                will prevent any selected plan from applying to a
                charge.

            *plan* Hash

            :   The selected installment plan to use for this payment
                attempt. This parameter can only be provided during
                confirmation.

                *count* Integer required

                :   For fixed\_count installment plans, this is the
                    number of installment payments your customer will
                    make to their credit card.

                *interval* String required

                :   For fixed\_count installment plans, this is the
                    interval between installment payments your customer
                    will make to their credit card. One of month.

                *type* String required

                :   Type of installment plan, one of fixed\_count.

        *network* String

        :   Selected network to process this PaymentIntent on. Depends
            on the available networks of the card attached to the
            PaymentIntent. Can be only set confirm-time.

        *request\_three\_d\_secure* String

        :   We strongly recommend that you rely on our SCA Engine to
            automatically prompt your customers for authentication based
            on risk level and other requirements. However, if you wish
            to request 3D Secure based on logic from your own fraud
            engine, provide this option. Permitted values include:
            automatic or any. If not provided, defaults to automatic.
            Read our guide on manually requesting 3D Secure for more
            information on how this configuration interacts with Radar
            and our SCA Engine.

    *oxxo* Hash

    :   If this is a oxxo PaymentMethod, this sub-hash contains details
        about the OXXO payment method options.

        *expires\_after\_days* Integer

        :   The number of calendar days before an OXXO voucher expires.
            For example, if you create an OXXO voucher on Monday and you
            set expires\_after\_days to 2, the OXXO invoice will expire
            on Wednesday at 23:59 America/Mexico\_City time.

    *p24* Hash

    :   If this is a p24 PaymentMethod, this sub-hash contains details
        about the Przelewy24 payment method options.

        No property documented.

    *sepa\_debit* Hash

    :   If this is a sepa\_debit PaymentIntent, this sub-hash contains
        details about the SEPA Debit payment method options.

        *mandate\_options* Hash

        :   Additional fields for Mandate creation

            No property provided.

    *sofort* Hash

    :   If this is a sofort PaymentMethod, this sub-hash contains
        details about the SOFORT payment method options.

        *preferred\_language* String

        :   Language shown to the payer on redirect.

*payment\_method\_types* Array of strings

:   The list of payment method types (e.g. card) that this PaymentIntent
    is allowed to use.

*receipt\_email* String

:   Email address that the receipt for the resulting payment will be
    sent to. If receipt\_email is specified for a payment in live mode,
    a receipt will be sent regardless of your email settings.

*return\_url* String

:   The URL to redirect your customer back to after they authenticate or
    cancel their payment on the payment methodâs app or site. If
    youâd prefer to redirect to a mobile application, you can
    alternatively supply an application URI scheme. This parameter is
    only used for cards and other redirect-based payment methods.

*setup\_future\_usage* String

:   Indicates that you intend to make future payments with this
    PaymentIntentâs payment method.

    Providing this parameter will attach the payment method to the
    PaymentIntentâs Customer, if present, after the PaymentIntent is
    confirmed and any required actions from the user are complete. If no
    Customer was provided, the payment method can still be attached to a
    Customer after the transaction completes.

    When processing card payments, Stripe also uses setup\_future\_usage
    to dynamically optimize your payment flow and comply with regional
    legislation and network rules, such as SCA.

    If setup\_future\_usage is already set and you are performing a
    request using a publishable key, you may only update the value from
    on\_session to off\_session.

*shipping* Hash

:   Shipping information for this PaymentIntent.

    *address* String required

    :   Shipping address.

        *city* String

        :   City, district, suburb, town, or village.

        *country* String

        :   Two-letter country code (ISO 3166-1 alpha-2).

        *line1* String required

        :   Address line 1 (e.g., street, PO Box, or company name).

        *line2* String

        :   Address line 2 (e.g., apartment, suite, unit, or building).

        *postal\_code* String

        :   ZIP or postal code.

        *state* String

        :   State, county, province, or region.

    *use\_stripe\_sdk* Boolean

    :   Set to true only when using manual confirmation and the iOS or
        Android SDKs to handle additional authentication steps.

    For more information, see Stripe documentation here:
    <https://stripe.com/docs/api/payment_intents/confirm>

create {#create-32}
------

Provided with a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or an hash reference, this will create a Stripe payment intent
and return a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

Possible parameters are:

*amount* Integer

:   Amount intended to be collected by this PaymentIntent. A positive
    integer representing how much to charge in the smallest currency
    unit (e.g., 100 cents to charge \$1.00 or 100 to charge Â¥100, a
    zero-decimal currency). The minimum amount is \$0.50 US or
    equivalent in charge currency. The amount value supports up to eight
    digits (e.g., a value of 99999999 for a USD charge of \$999,999.99).

*application\_fee\_amount* Integer

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    ownerâs Stripe account. The amount of the application fee
    collected will be capped at the total payment amount. For more
    information, see the PaymentIntents use case for connected accounts.

*capture\_method* String

:   Controls when the funds will be captured from the customerâs
    account.

    Possible enum values

    *automatic* String

    :   (Default) Stripe automatically captures funds when the customer
        authorizes the payment.

    *manual* String

    :   Place a hold on the funds when the customer authorizes the
        payment, but donât capture the funds until later. (Not all
        payment methods support this.)

*confirmation\_method* String

:   Possible enum values

    *automatic* String

    :   (Default) PaymentIntent can be confirmed using a publishable
        key. After next\_actions are handled, no additional confirmation
        is required to complete the payment.

    *manual* String

    :   All payment attempts must be made using a secret key. The
        PaymentIntent returns to the requires\_confirmation state after
        handling next\_actions, and requires your server to initiate
        each payment attempt with an explicit confirmation.

*currency* String

:   Three-letter ISO currency code, in lowercase. Must be a supported
    currency.

*confirm* Boolean

:   Set to true to attempt to confirm this PaymentIntent immediately.
    This parameter defaults to false. When creating and confirming a
    PaymentIntent at the same time, parameters available in the confirm
    API may also be provided.

*customer* String.

:   ID of the Customer this PaymentIntent belongs to, if one exists.

    Payment methods attached to other Customers cannot be used with this
    PaymentIntent.

    If present in combination with setup\_future\_usage, this
    PaymentIntentâs payment method will be attached to the Customer
    after the PaymentIntent has been confirmed and any required actions
    from the user are complete.

*description* String

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

*error\_on\_requires\_action* Boolean

:   Set to true to fail the payment attempt if the PaymentIntent
    transitions into requires\_action. This parameter is intended for
    simpler integrations that do not handle customer actions, like
    saving cards without authentication. This parameter can only be used
    with confirm=true.

*mandate* String

:   ID of the mandate to be used for this payment. This parameter can
    only be used with confirm=true.

*mandate\_data* Hash

:   This hash contains details about the Mandate to create. This
    parameter can only be used with confirm=true.

    *customer\_acceptance* Hash required

    :   This hash contains details about the customer acceptance of the
        Mandate.

        *type* String required

        :   The type of customer acceptance information included with
            the Mandate. One of online or offline.

        *accepted\_at* Datetime

        :   The time at which the customer accepted the Mandate.

        *offline* Hash

        :   If this is a Mandate accepted offline, this hash contains
            details about the offline acceptance.

        *online* Hash

        :   If this is a Mandate accepted online, this hash contains
            details about the online acceptance.

            *ip\_address* String

            :   The IP address from which the Mandate was accepted by
                the customer.

            *user\_agent* String

            :   The user agent of the browser from which the Mandate was
                accepted by the customer.

*metadata* Hash

:   Set of key-value pairs that you can attach to an object. This can be
    useful for storing additional information about the object in a
    structured format. Individual keys can be unset by posting an empty
    value to them. All keys can be unset by posting an empty value to
    metadata.

*on\_behalf\_of* String

:   The Stripe account ID for which these funds are intended. For
    details, see the PaymentIntents use case for connected accounts.

*off\_session* String

:   Set to true to indicate that the customer is not in your checkout
    flow during this payment attempt, and therefore is unable to
    authenticate. This parameter is intended for scenarios where you
    collect card details and charge them later. This parameter can only
    be used with confirm=true.

*payment\_method* String

:   ID of the payment method (a PaymentMethod, Card, or compatible
    Source object) to attach to this PaymentIntent.

    If this parameter is omitted with confirm=true,
    customer.default\_source will be attached as this PaymentIntentâs
    payment instrument to improve the migration experience for users of
    the Charges API. We recommend that you explicitly provide the
    payment\_method going forward.

*payment\_method\_data* Hash

:   If provided, this hash will be used to create a PaymentMethod. The
    new PaymentMethod will appear in the payment\_method property on the
    PaymentIntent.

    *alipay* Hash

    :   If this is an Alipay PaymentMethod, this hash contains details
        about the Alipay payment method.

        No documented property.

    *au\_becs\_debit* Hash

    :   If this is an au\_becs\_debit PaymentMethod, this hash contains
        details about the bank account.

        *account\_number* String

        :   The account number for the bank account

        *bsb\_number* String

        :   Bank-State-Branch number of the bank account.

    *bacs\_debit* Hash

    :   If this is a bacs\_debit PaymentMethod, this hash contains
        details about the Bacs Direct Debit bank account.

        *account\_number* String

        :   Account number of the bank account that the funds will be
            debited from.

        *sort\_code* String

        :   Sort code of the bank account. (e.g., 10-20-30)

    *bancontact* Hash

    :   If this is a bancontact PaymentMethod, this hash contains
        details about the Bancontact payment method.

        No documented property.

    *billing\_details* Hash

    :   Billing information associated with the PaymentMethod that may
        be used or required by particular types of payment methods.

        *address* Hash

        :   Billing address.

            *city* String

            :   City, district, suburb, town, or village.

            *country* String

            :   Two-letter country code (ISO 3166-1 alpha-2).

            *line1* String

            :   Address line 1 (e.g., street, PO Box, or company name).

            *line2* String

            :   Address line 2 (e.g., apartment, suite, unit, or
                building).

            *postal\_code* String

            :   ZIP or postal code.

            *state* String

            :   State, county, province, or region.

        *email* String

        :   Email address.

        *name* String

        :   Full name.

        *phone* String

        :   Billing phone number (including extension).

    *eps* hash

    :   If this is an eps PaymentMethod, this hash contains details
        about the EPS payment method.

        No documented property.

    *fpx* Hash

    :   If this is an fpx PaymentMethod, this hash contains details
        about the FPX payment method.

        *bank* String

        :   The customerâs bank.

    *giropay* Hash

    :   If this is a giropay PaymentMethod, this hash contains details
        about the Giropay payment method.

        No documented property.

    *grabpay* Hash

    :   If this is a grabpay PaymentMethod, this hash contains details
        about the GrabPay payment method.

        No documented property.

    *ideal* Hash

    :   If this is an ideal PaymentMethod, this hash contains details
        about the iDEAL payment method.

        *bank* String

        :   The customerâs bank.

    *interac\_present* Hash

    :   If this is an interac\_present PaymentMethod, this hash contains
        details about the Interac Present payment method.

        No documented property.

    *metadata* Hash

    :   Set of key-value pairs that you can attach to an object. This
        can be useful for storing additional information about the
        object in a structured format. Individual keys can be unset by
        posting an empty value to them. All keys can be unset by posting
        an empty value to metadata.

    *oxxo* Hash

    :   If this is an oxxo PaymentMethod, this hash contains details
        about the OXXO payment method.

        No documented property.

    *p24* Hash

    :   If this is a p24 PaymentMethod, this hash contains details about
        the P24 payment method.

        *bank* String

        :   The customerâs bank.

    *sepa\_debit* Hash

    :   If this is a sepa\_debit PaymentMethod, this hash contains
        details about the SEPA debit bank account.

        *iban* String required

        :   IBAN of the bank account.

    *sofort* Hash

    :   If this is a sofort PaymentMethod, this hash contains details
        about the SOFORT payment method.

        *country*

        :   Two-letter ISO code representing the country the bank
            account is located in.

    *type* String

    :   The type of the PaymentMethod. An additional hash is included on
        the PaymentMethod with a name matching this value. It contains
        additional information specific to the PaymentMethod type.

*payment\_method\_options* Hash

:   Payment-method-specific configuration for this PaymentIntent.

    *alipay* Hash

    :   If this is a alipay PaymentMethod, this sub-hash contains
        details about the Alipay payment method options.

        No documented property.

    *bancontact* Hash

    :   If this is a bancontact PaymentMethod, this sub-hash contains
        details about the Bancontact payment method options.

        *preferred\_language* String

        :   Preferred language of the Bancontact authorization page that
            the customer is redirected to.

    *card* Hash

    :   Configuration for any card payments attempted on this
        PaymentIntent.

        *cvc\_token*

        :   A single-use cvc\_update Token that represents a card CVC
            value. When provided, the CVC value will be verified during
            the card payment attempt. This parameter can only be
            provided during confirmation.

        *installments*

        :   Installment configuration for payments attempted on this
            PaymentIntent (Mexico Only).

            For more information, see the [installments integration
            guide](https://stripe.com/docs/payments/installments){.perl-module}.

            *enabled* Boolean

            :   Setting to true enables installments for this
                PaymentIntent. This will cause the response to contain a
                list of available installment plans. Setting to false
                will prevent any selected plan from applying to a
                charge.

            *plan* Hash

            :   The selected installment plan to use for this payment
                attempt. This parameter can only be provided during
                confirmation.

                *count* Integer required

                :   For fixed\_count installment plans, this is the
                    number of installment payments your customer will
                    make to their credit card.

                *interval* String required

                :   For fixed\_count installment plans, this is the
                    interval between installment payments your customer
                    will make to their credit card. One of month.

                *type* String required

                :   Type of installment plan, one of fixed\_count.

        *network* String

        :   Selected network to process this PaymentIntent on. Depends
            on the available networks of the card attached to the
            PaymentIntent. Can be only set confirm-time.

        *request\_three\_d\_secure* String

        :   We strongly recommend that you rely on our SCA Engine to
            automatically prompt your customers for authentication based
            on risk level and other requirements. However, if you wish
            to request 3D Secure based on logic from your own fraud
            engine, provide this option. Permitted values include:
            automatic or any. If not provided, defaults to automatic.
            Read our guide on manually requesting 3D Secure for more
            information on how this configuration interacts with Radar
            and our SCA Engine.

    *oxxo* Hash

    :   If this is a oxxo PaymentMethod, this sub-hash contains details
        about the OXXO payment method options.

        *expires\_after\_days* Integer

        :   The number of calendar days before an OXXO voucher expires.
            For example, if you create an OXXO voucher on Monday and you
            set expires\_after\_days to 2, the OXXO invoice will expire
            on Wednesday at 23:59 America/Mexico\_City time.

    *p24* Hash

    :   If this is a p24 PaymentMethod, this sub-hash contains details
        about the Przelewy24 payment method options.

        No property documented.

    *sepa\_debit* Hash

    :   If this is a sepa\_debit PaymentIntent, this sub-hash contains
        details about the SEPA Debit payment method options.

        *mandate\_options* Hash

        :   Additional fields for Mandate creation

            No property provided.

    *sofort* Hash

    :   If this is a sofort PaymentMethod, this sub-hash contains
        details about the SOFORT payment method options.

        *preferred\_language* String

        :   Language shown to the payer on redirect.

*payment\_method\_types* array of strings

:   The list of payment method types that this PaymentIntent is allowed
    to use. If this is not provided, defaults to \[âcardâ\]. Valid
    payment method types include: alipay, au\_becs\_debit, bancontact,
    card, card\_present, eps, giropay, ideal, interac\_present, p24,
    sepa\_debit, and sofort.

*receipt\_email* String

:   Email address that the receipt for the resulting payment will be
    sent to. If receipt\_email is specified for a payment in live mode,
    a receipt will be sent regardless of your email settings.

*return\_url* String

:   The URL to redirect your customer back to after they authenticate or
    cancel their payment on the payment methodâs app or site. If
    youâd prefer to redirect to a mobile application, you can
    alternatively supply an application URI scheme. This parameter can
    only be used with confirm=true.

*setup\_future\_usage* String

:   Indicates that you intend to make future payments with this
    PaymentIntentâs payment method.

    Providing this parameter will attach the payment method to the
    PaymentIntentâs Customer, if present, after the PaymentIntent is
    confirmed and any required actions from the user are complete. If no
    Customer was provided, the payment method can still be attached to a
    Customer after the transaction completes.

    When processing card payments, Stripe also uses setup\_future\_usage
    to dynamically optimize your payment flow and comply with regional
    legislation and network rules, such as SCA.

    Possible enum values

    *on\_session* String

    :   Use on\_session if you intend to only reuse the payment method
        when your customer is present in your checkout flow.

    *off\_session* String

    :   Use off\_session if your customer may or may not be present in
        your checkout flow.

*shipping* Hash

:   Shipping information for this PaymentIntent.

    *address* Hash required

    :   Shipping address.

        *city* String

        :   City, district, suburb, town, or village.

        *country* String

        :   Two-letter country code (ISO 3166-1 alpha-2).

        *line1* String required

        :   Address line 1 (e.g., street, PO Box, or company name).

        *line2* String

        :   Address line 2 (e.g., apartment, suite, unit, or building).

        *postal\_code* String

        :   ZIP or postal code.

        *state* String

        :   State, county, province, or region.

    *name* String required

    :   Recipient name.

    *carrier* String

    :   The delivery service that shipped a physical product, such as
        Fedex, UPS, USPS, etc.

    *phone* String

    :   Recipient phone (including extension).

    *tracking\_number* String

    :   The tracking number for a physical product, obtained from the
        delivery service. If multiple tracking numbers were generated
        for this purchase, please separate them with commas.

*statement\_descriptor* String

:   For non-card charges, you can use this value as the complete
    description that appears on your customersâ statements. Must
    contain at least one letter, maximum 22 characters.

*statement\_descriptor\_suffix* Text

:   Provides information about a card payment that customers see on
    their statements. Concatenated with the prefix (shortened
    descriptor) or statement descriptor thatâs set on the account to
    form the complete statement descriptor. Maximum 22 characters for
    the concatenated descriptor.

*transfer\_data* Hash

:   The parameters used to automatically create a Transfer when the
    payment succeeds. For more information, see the PaymentIntents use
    case for connected accounts.

    *destination* String required

    :   If specified, successful charges will be attributed to the
        destination account for tax reporting, and the funds from
        charges will be transferred to the destination account. The ID
        of the resulting transfer will be returned on the successful
        chargeâs transfer field.

    *amount* Integer

    :   The amount that will be transferred automatically when a charge
        succeeds. The amount is capped at the total transaction amount
        and if no amount is set, the full amount is transferred.

        If you intend to collect a fee and you need a more robust
        reporting experience, using application\_fee\_amount might be a
        better fit for your integration.

*transfer\_group* String

:   A string that identifies the resulting payment as part of a group.
    See the PaymentIntents use case for connected accounts for details.

*use\_stripe\_sdk* Boolean

:   Set to true only when using manual confirmation and the iOS or
    Android SDKs to handle additional authentication steps.

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_intents/create>

list {#list-43}
----

Provided with an hash reference of parameters, and this will get a list
of payment methods and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Provided with a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or an hash reference, this will update the Stripe payment intent
and return a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

Possible parameters are:

*customer* A Stripe customer id

:   Only return PaymentIntents for the customer specified by this
    customer ID.

*created* Hash

:   A filter on the list based on the object created field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

    *gt* Datetime

    :   Return results where the created field is greater than this
        value.

    *gte* Datetime

    :   Return results where the created field is greater than or equal
        to this value.

    *lt* Datetime

    :   Return results where the created field is less than this value.

    *lte* Datetime

    :   Return results where the created field is less than or equal to
        this value.

*ending\_before* String

:   A cursor for use in pagination. ending\_before is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with obj\_bar, your
    subsequent call can include ending\_before=obj\_bar in order to
    fetch the previous page of the list.

*limit* Integer

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

*starting\_after* String

:   A cursor for use in pagination. starting\_after is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, ending with obj\_foo, your
    subsequent call can include starting\_after=obj\_foo in order to
    fetch the next page of the list.

retrieve {#retrieve-44}
--------

Provided with a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or an hash reference, this will retrieve a Stripe payment intent
and return a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

Possible parameters are:

*client\_secret* String required

:   The client secret of the PaymentIntent. Required if a publishable
    key is used to retrieve the source.

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_intents/retrieve>

update
------

Provided with a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object or an hash reference, this will update the Stripe payment intent
and return a
[Net::API::Stripe::Payment::Intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent){.perl-module}
object.

Possible parameters are:

*amount* Integer

:   Amount intended to be collected by this PaymentIntent. A positive
    integer representing how much to charge in the smallest currency
    unit (e.g., 100 cents to charge \$1.00 or 100 to charge Â¥100, a
    zero-decimal currency). The minimum amount is \$0.50 US or
    equivalent in charge currency. The amount value supports up to eight
    digits (e.g., a value of 99999999 for a USD charge of \$999,999.99).

*application\_fee\_amount* Integer

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    ownerâs Stripe account. The amount of the application fee
    collected will be capped at the total payment amount. For more
    information, see the PaymentIntents use case for connected accounts.

*capture\_method* String

:   Controls when the funds will be captured from the customerâs
    account.

    Possible enum values

    *automatic* String

    :   (Default) Stripe automatically captures funds when the customer
        authorizes the payment.

    *manual* String

    :   Place a hold on the funds when the customer authorizes the
        payment, but donât capture the funds until later. (Not all
        payment methods support this.)

*confirmation\_method* String

:   Possible enum values

    *automatic* String

    :   (Default) PaymentIntent can be confirmed using a publishable
        key. After next\_actions are handled, no additional confirmation
        is required to complete the payment.

    *manual* String

    :   All payment attempts must be made using a secret key. The
        PaymentIntent returns to the requires\_confirmation state after
        handling next\_actions, and requires your server to initiate
        each payment attempt with an explicit confirmation.

*currency* String

:   Three-letter ISO currency code, in lowercase. Must be a supported
    currency.

*confirm* Boolean

:   Set to true to attempt to confirm this PaymentIntent immediately.
    This parameter defaults to false. When creating and confirming a
    PaymentIntent at the same time, parameters available in the confirm
    API may also be provided.

*customer* String.

:   ID of the Customer this PaymentIntent belongs to, if one exists.

    Payment methods attached to other Customers cannot be used with this
    PaymentIntent.

    If present in combination with setup\_future\_usage, this
    PaymentIntentâs payment method will be attached to the Customer
    after the PaymentIntent has been confirmed and any required actions
    from the user are complete.

*description* String

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

*error\_on\_requires\_action* Boolean

:   Set to true to fail the payment attempt if the PaymentIntent
    transitions into requires\_action. This parameter is intended for
    simpler integrations that do not handle customer actions, like
    saving cards without authentication. This parameter can only be used
    with confirm=true.

*mandate* String

:   ID of the mandate to be used for this payment. This parameter can
    only be used with confirm=true.

*mandate\_data* Hash

:   This hash contains details about the Mandate to create. This
    parameter can only be used with confirm=true.

    *customer\_acceptance* Hash required

    :   This hash contains details about the customer acceptance of the
        Mandate.

        *type* String required

        :   The type of customer acceptance information included with
            the Mandate. One of online or offline.

        *accepted\_at* Datetime

        :   The time at which the customer accepted the Mandate.

        *offline* Hash

        :   If this is a Mandate accepted offline, this hash contains
            details about the offline acceptance.

        *online* Hash

        :   If this is a Mandate accepted online, this hash contains
            details about the online acceptance.

            *ip\_address* String

            :   The IP address from which the Mandate was accepted by
                the customer.

            *user\_agent* String

            :   The user agent of the browser from which the Mandate was
                accepted by the customer.

*metadata* Hash

:   Set of key-value pairs that you can attach to an object. This can be
    useful for storing additional information about the object in a
    structured format. Individual keys can be unset by posting an empty
    value to them. All keys can be unset by posting an empty value to
    metadata.

*on\_behalf\_of* String

:   The Stripe account ID for which these funds are intended. For
    details, see the PaymentIntents use case for connected accounts.

*payment\_method* String

:   ID of the payment method (a PaymentMethod, Card, or compatible
    Source object) to attach to this PaymentIntent.

*payment\_method\_data* Hash

:   If provided, this hash will be used to create a PaymentMethod. The
    new PaymentMethod will appear in the payment\_method property on the
    PaymentIntent.

    *alipay* Hash

    :   If this is an Alipay PaymentMethod, this hash contains details
        about the Alipay payment method.

        No documented property.

    *au\_becs\_debit* Hash

    :   If this is an au\_becs\_debit PaymentMethod, this hash contains
        details about the bank account.

        *account\_number* String

        :   The account number for the bank account.

        *bsb\_number* String

        :   Bank-State-Branch number of the bank account.

    *bacs\_debit* Hash

    :   If this is a bacs\_debit PaymentMethod, this hash contains
        details about the Bacs Direct Debit bank account.

        *account\_number* String

        :   Account number of the bank account that the funds will be
            debited from.

        *sort\_code* String

        :   Sort code of the bank account. (e.g., 10-20-30)

    *bancontact* Hash

    :   If this is a bancontact PaymentMethod, this hash contains
        details about the Bancontact payment method.

        No documented property.

    *billing\_details* Hash

    :   Billing information associated with the PaymentMethod that may
        be used or required by particular types of payment methods.

        *address* Hash

        :   Billing address.

            *city* String

            :   City, district, suburb, town, or village.

            *country* String

            :   Two-letter country code (ISO 3166-1 alpha-2).

            *line1* String

            :   Address line 1 (e.g., street, PO Box, or company name).

            *line2* String

            :   Address line 2 (e.g., apartment, suite, unit, or
                building).

            *postal\_code* String

            :   ZIP or postal code.

            *state* String

            :   State, county, province, or region.

        *email* String

        :   Email address.

        *name* String

        :   Full name.

        *phone* String

        :   Billing phone number (including extension).

    *eps* hash

    :   If this is an eps PaymentMethod, this hash contains details
        about the EPS payment method.

        No documented property.

    *fpx* Hash

    :   If this is an fpx PaymentMethod, this hash contains details
        about the FPX payment method.

        *bank* String required

        :   The customerâs bank.

    *giropay* Hash

    :   If this is a giropay PaymentMethod, this hash contains details
        about the Giropay payment method.

        No documented property.

    *grabpay* Hash

    :   If this is a grabpay PaymentMethod, this hash contains details
        about the GrabPay payment method.

        No documented property.

    *ideal* Hash

    :   If this is an ideal PaymentMethod, this hash contains details
        about the iDEAL payment method.

        *bank* String

        :   The customerâs bank.

    *interac\_present* Hash

    :   If this is an interac\_present PaymentMethod, this hash contains
        details about the Interac Present payment method.

        No documented property.

    *metadata* Hash

    :   Set of key-value pairs that you can attach to an object. This
        can be useful for storing additional information about the
        object in a structured format. Individual keys can be unset by
        posting an empty value to them. All keys can be unset by posting
        an empty value to metadata.

    *oxxo* Hash

    :   If this is an oxxo PaymentMethod, this hash contains details
        about the OXXO payment method.

        No documented property.

    *p24* Hash

    :   If this is a p24 PaymentMethod, this hash contains details about
        the P24 payment method.

        *bank* String

        :   The customerâs bank.

    *sepa\_debit* Hash

    :   If this is a sepa\_debit PaymentMethod, this hash contains
        details about the SEPA debit bank account.

        *iban* String required

        :   IBAN of the bank account.

    *sofort* Hash

    :   If this is a sofort PaymentMethod, this hash contains details
        about the SOFORT payment method.

        *country* String required

        :   Two-letter ISO code representing the country the bank
            account is located in.

    *type* String

    :   The type of the PaymentMethod. An additional hash is included on
        the PaymentMethod with a name matching this value. It contains
        additional information specific to the PaymentMethod type.

*payment\_method\_options* Hash

:   Payment-method-specific configuration for this PaymentIntent.

    *alipay* Hash

    :   If this is a alipay PaymentMethod, this sub-hash contains
        details about the Alipay payment method options.

        No documented property.

    *bancontact* Hash

    :   If this is a bancontact PaymentMethod, this sub-hash contains
        details about the Bancontact payment method options.

        *preferred\_language* String

        :   Preferred language of the Bancontact authorization page that
            the customer is redirected to.

    *card* Hash

    :   Configuration for any card payments attempted on this
        PaymentIntent.

        *cvc\_token*

        :   A single-use cvc\_update Token that represents a card CVC
            value. When provided, the CVC value will be verified during
            the card payment attempt. This parameter can only be
            provided during confirmation.

        *installments*

        :   Installment configuration for payments attempted on this
            PaymentIntent (Mexico Only).

            For more information, see the [installments integration
            guide](https://stripe.com/docs/payments/installments){.perl-module}.

            *enabled* Boolean

            :   Setting to true enables installments for this
                PaymentIntent. This will cause the response to contain a
                list of available installment plans. Setting to false
                will prevent any selected plan from applying to a
                charge.

            *plan* Hash

            :   The selected installment plan to use for this payment
                attempt. This parameter can only be provided during
                confirmation.

                *count* Integer required

                :   For fixed\_count installment plans, this is the
                    number of installment payments your customer will
                    make to their credit card.

                *interval* String required

                :   For fixed\_count installment plans, this is the
                    interval between installment payments your customer
                    will make to their credit card. One of month.

                *type* String required

                :   Type of installment plan, one of fixed\_count.

        *network* String

        :   Selected network to process this PaymentIntent on. Depends
            on the available networks of the card attached to the
            PaymentIntent. Can be only set confirm-time.

        *request\_three\_d\_secure* String

        :   We strongly recommend that you rely on our SCA Engine to
            automatically prompt your customers for authentication based
            on risk level and other requirements. However, if you wish
            to request 3D Secure based on logic from your own fraud
            engine, provide this option. Permitted values include:
            automatic or any. If not provided, defaults to automatic.
            Read our guide on manually requesting 3D Secure for more
            information on how this configuration interacts with Radar
            and our SCA Engine.

    *oxxo* Hash

    :   If this is a oxxo PaymentMethod, this sub-hash contains details
        about the OXXO payment method options.

        *expires\_after\_days* Integer

        :   The number of calendar days before an OXXO voucher expires.
            For example, if you create an OXXO voucher on Monday and you
            set expires\_after\_days to 2, the OXXO invoice will expire
            on Wednesday at 23:59 America/Mexico\_City time.

    *p24* Hash

    :   If this is a p24 PaymentMethod, this sub-hash contains details
        about the Przelewy24 payment method options.

        No property documented.

    *sepa\_debit* Hash

    :   If this is a sepa\_debit PaymentIntent, this sub-hash contains
        details about the SEPA Debit payment method options.

        *mandate\_options* Hash

        :   Additional fields for Mandate creation

            No property provided.

    *sofort* Hash

    :   If this is a sofort PaymentMethod, this sub-hash contains
        details about the SOFORT payment method options.

        *preferred\_language* String

        :   Language shown to the payer on redirect.

*off\_session* String

:   Set to true to indicate that the customer is not in your checkout
    flow during this payment attempt, and therefore is unable to
    authenticate. This parameter is intended for scenarios where you
    collect card details and charge them later. This parameter can only
    be used with confirm=true.

*payment\_method\_types* Array of strings

:   The list of payment method types (e.g. card) that this PaymentIntent
    is allowed to use.

*receipt\_email* String

:   Email address that the receipt for the resulting payment will be
    sent to. If receipt\_email is specified for a payment in live mode,
    a receipt will be sent regardless of your email settings.

*setup\_future\_usage* String

:   Indicates that you intend to make future payments with this
    PaymentIntentâs payment method.

    Providing this parameter will attach the payment method to the
    PaymentIntentâs Customer, if present, after the PaymentIntent is
    confirmed and any required actions from the user are complete. If no
    Customer was provided, the payment method can still be attached to a
    Customer after the transaction completes.

    When processing card payments, Stripe also uses setup\_future\_usage
    to dynamically optimize your payment flow and comply with regional
    legislation and network rules, such as SCA.

    If setup\_future\_usage is already set and you are performing a
    request using a publishable key, you may only update the value from
    on\_session to off\_session.

    Possible enum values

    *on\_session* String

    :   Use on\_session if you intend to only reuse the payment method
        when your customer is present in your checkout flow.

    *off\_session* String

    :   Use off\_session if your customer may or may not be present in
        your checkout flow.

*return\_url* String

:   The URL to redirect your customer back to after they authenticate or
    cancel their payment on the payment methodâs app or site. If
    youâd prefer to redirect to a mobile application, you can
    alternatively supply an application URI scheme. This parameter can
    only be used with confirm=true.

*shipping* Hash

:   Shipping information for this PaymentIntent.

    *address* String required

    :   Shipping address.

        *city* String

        :   City, district, suburb, town, or village.

        *country* String

        :   Two-letter country code (ISO 3166-1 alpha-2).

        *line1* String required

        :   Address line 1 (e.g., street, PO Box, or company name).

        *line2* String

        :   Address line 2 (e.g., apartment, suite, unit, or building).

        *postal\_code* String

        :   ZIP or postal code.

        *state* String

        :   State, county, province, or region.

    *name* String required

    :   Recipient name.

    *carrier* String

    :   The delivery service that shipped a physical product, such as
        Fedex, UPS, USPS, etc.

    *phone* String

    :   Recipient phone (including extension).

    *tracking\_number* String

    :   The tracking number for a physical product, obtained from the
        delivery service. If multiple tracking numbers were generated
        for this purchase, please separate them with commas.

*statement\_descriptor* String

:   For non-card charges, you can use this value as the complete
    description that appears on your customersâ statements. Must
    contain at least one letter, maximum 22 characters.

*statement\_descriptor\_suffix* Text

:   Provides information about a card payment that customers see on
    their statements. Concatenated with the prefix (shortened
    descriptor) or statement descriptor thatâs set on the account to
    form the complete statement descriptor. Maximum 22 characters for
    the concatenated descriptor.

*transfer\_data* Hash

:   The parameters used to automatically create a Transfer when the
    payment succeeds. For more information, see the PaymentIntents use
    case for connected accounts.

    *amount* Integer

    :   The amount that will be transferred automatically when a charge
        succeeds.

*transfer\_group* String

:   A string that identifies the resulting payment as part of a group.
    transfer\_group may only be provided if it has not been set. See the
    PaymentIntents use case for connected accounts for details.

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_intents/create>

PAYMENT LINK
============

You can create, items, line\_items, list, retrieve or update payment
link

create {#create-33}
------

        my $obj = $stripe->payment_links( create => {
            line_items => [,
                price    => "price_1Le1oa2eZvKYlo2CuD7mwpZu",
                quantity => "1",
            ], } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Payment::Link](https://metacpan.org/pod/Net::API::Stripe::Payment::Link){.perl-module}
object or a hash reference, this will create a Stripe payment link and
return an
[Net::API::Stripe::Payment::Link](https://metacpan.org/pod/Net::API::Stripe::Payment::Link){.perl-module}
object.

Possible parameters are:

`after_completion`

:   Behavior after the purchase is complete.

`allow_promotion_codes`

:   Enables user redeemable promotion codes.

`application_fee_amount`

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    owner\'s Stripe account. Can only be applied when there are no line
    items with recurring prices.

`application_fee_percent`

:   A non-negative decimal between 0 and 100, with at most two decimal
    places. This represents the percentage of the subscription invoice
    subtotal that will be transferred to the application owner\'s Stripe
    account. There must be at least 1 line item with a recurring price
    to use this field.

`automatic_tax`

:   Configuration for automatic tax collection.

`billing_address_collection`

:   Configuration for collecting the customer\'s billing address.

`consent_collection`

:   Configure fields to gather active consent from customers.

`currency`

:   Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module} and
    supported by each line item\'s price.

`customer_creation`

:   Configures whether [checkout
    sessions](https://stripe.com/docs/api/checkout/sessions){.perl-module}
    created by this payment link create a
    [Customer](https://stripe.com/docs/api/customers){.perl-module}.

`line_items`

:   **Required**. The line items representing what is being sold. Each
    line item represents an item being sold. Up to 20 line items are
    supported.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`. Metadata associated
    with this Payment Link will automatically be copied to [checkout
    sessions](https://stripe.com/docs/api/checkout/sessions){.perl-module}
    created by this payment link.

`on_behalf_of`

:   The account on behalf of which to charge.

`payment_intent_data`

:   A subset of parameters to be passed to PaymentIntent creation for
    Checkout Sessions in `payment` mode.

`payment_method_collection`

:   Specify whether Checkout should collect a payment method. When set
    to `if_required`, Checkout will not collect a payment method when
    the total due for the session is 0.This may occur if the Checkout
    Session includes a free trial or a discount.

    Can only be set in `subscription` mode.

    If you\'d like information on how to collect a payment method
    outside of Checkout, read the guide on [configuring subscriptions
    with a free
    trial](https://stripe.com/docs/payments/checkout/free-trials){.perl-module}.

`payment_method_types`

:   The list of payment method types that customers can use. If no value
    is passed, Stripe will dynamically show relevant payment methods
    from your [payment method
    settings](https://dashboard.stripe.com/settings/payment_methods){.perl-module}
    (20+ payment methods
    [supported](https://stripe.com/docs/payments/payment-methods/integration-options#payment-method-product-support){.perl-module}).

`phone_number_collection`

:   Controls phone number collection settings during checkout.

    We recommend that you review your privacy policy and check with your
    legal contacts.

`shipping_address_collection`

:   Configuration for collecting the customer\'s shipping address.

`shipping_options`

:   The shipping rate options to apply to [checkout
    sessions](https://stripe.com/docs/api/checkout/sessions){.perl-module}
    created by this payment link.

`submit_type`

:   Describes the type of transaction being performed in order to
    customize relevant text on the page, such as the submit button.
    Changing this value will also affect the hostname in the
    [url](https://stripe.com/docs/api/payment_links/payment_links/object#url){.perl-module}
    property (example: `donate.stripe.com`).

`subscription_data`

:   When creating a subscription, the specified configuration data will
    be used. There must be at least one line item with a recurring price
    to use `subscription_data`.

`tax_id_collection`

:   Controls tax ID collection during checkout.

`transfer_data`

:   The account (if any) the payments will be attributed to for tax
    reporting, and where funds from each payment will be transferred to.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_links/payment_links/create>

line\_items
-----------

        my $obj = $stripe->payment_links( line_items => $args ) || die( $stripe->error );

Provided with a [payment
link](https://metacpan.org/pod/Net::API::Stripe::Payment::Link){.perl-module},
or a hash reference, this will issue a line\_items api call.

A dictionary with a `data` property that contains an array of up to
`limit` payment link line items, starting after Line Item
`starting_after`. Each entry in the array is a separate Line Item
object. If no more line items are available, the resulting array will be
empty.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_links/line_items>

list {#list-44}
----

        my $obj = $stripe->payment_links( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [payment
link](https://metacpan.org/pod/Net::API::Stripe::Payment::Link){.perl-module}
object, this issue an api call to get the list of all payment link.

Possible parameters are:

`active`

:   Only return payment links that are active or inactive (e.g., pass
    `false` to list all inactive payment links).

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_links/payment_links/list>

retrieve {#retrieve-45}
--------

        my $obj = $stripe->payment_links( retrieve => $args ) || die( $stripe->error );

Provided with a [payment
link](https://metacpan.org/pod/Net::API::Stripe::Payment::Link){.perl-module}
object or a hash reference, this will retrieve a Stripe payment link and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Link){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_links/payment_links/retrieve>

update
------

        my $obj = $stripe->payment_links( update => {
            active => "0", } ) || die( $stripe->error );

Provided with a [payment
link](https://metacpan.org/pod/Net::API::Stripe::Payment::Link){.perl-module}
object or a hash reference, this will update a Stripe payment link and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Link){.perl-module}

Possible parameters are:

`active`

:   Whether the payment link\'s `url` is active. If `false`, customers
    visiting the URL will be shown a page saying that the link has been
    deactivated.

`after_completion`

:   Behavior after the purchase is complete.

`allow_promotion_codes`

:   Enables user redeemable promotion codes.

`automatic_tax`

:   Configuration for automatic tax collection.

`billing_address_collection`

:   Configuration for collecting the customer\'s billing address.

`customer_creation`

:   Configures whether [checkout
    sessions](https://stripe.com/docs/api/checkout/sessions){.perl-module}
    created by this payment link create a
    [Customer](https://stripe.com/docs/api/customers){.perl-module}.

`line_items`

:   The line items representing what is being sold. Each line item
    represents an item being sold. Up to 20 line items are supported.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`. Metadata associated
    with this Payment Link will automatically be copied to [checkout
    sessions](https://stripe.com/docs/api/checkout/sessions){.perl-module}
    created by this payment link.

`payment_method_collection`

:   Specify whether Checkout should collect a payment method. When set
    to `if_required`, Checkout will not collect a payment method when
    the total due for the session is 0.This may occur if the Checkout
    Session includes a free trial or a discount.

    Can only be set in `subscription` mode.

    If you\'d like information on how to collect a payment method
    outside of Checkout, read the guide on [configuring subscriptions
    with a free
    trial](https://stripe.com/docs/payments/checkout/free-trials){.perl-module}.

`payment_method_types`

:   The list of payment method types that customers can use. Pass an
    empty string to enable automatic payment methods that use your
    [payment method
    settings](https://dashboard.stripe.com/settings/payment_methods){.perl-module}.

`shipping_address_collection`

:   Configuration for collecting the customer\'s shipping address.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_links/payment_links/update>

PAYMENT METHOD
==============

You can attach, create, detach, list, list\_customer\_payment\_methods,
retrieve, retrieve\_customer\_payment\_method or update payment method

attach
------

        my $obj = $stripe->payment_methods( attach => {
            customer => "cus_AJ78ZaALpqgiuZ", } ) || die( $stripe->error );

Provided with a [payment
method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module},
or a hash reference, this will issue a attach api call.

Returns a PaymentMethod object.

Possible parameters are:

`customer`

:   **Required**. The ID of the customer to which to attach the
    PaymentMethod.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_methods/attach>

create {#create-34}
------

        my $obj = $stripe->payment_methods( create => {
            card =>
            {
                cvc       => "314",
                exp_month => "9",
                exp_year  => "2023",
                number    => "4242424242424242",
            }
            type => "card", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object or a hash reference, this will create a Stripe payment method and
return an
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object.

Possible parameters are:

`acss_debit`

:   If this is an `acss_debit` PaymentMethod, this hash contains details
    about the ACSS Debit payment method.

`affirm`

:   If this is an `affirm` PaymentMethod, this hash contains details
    about the Affirm payment method.

`afterpay_clearpay`

:   If this is an `AfterpayClearpay` PaymentMethod, this hash contains
    details about the AfterpayClearpay payment method.

`alipay`

:   If this is an `Alipay` PaymentMethod, this hash contains details
    about the Alipay payment method.

`au_becs_debit`

:   If this is an `au_becs_debit` PaymentMethod, this hash contains
    details about the bank account.

`bacs_debit`

:   If this is a `bacs_debit` PaymentMethod, this hash contains details
    about the Bacs Direct Debit bank account.

`bancontact`

:   If this is a `bancontact` PaymentMethod, this hash contains details
    about the Bancontact payment method.

`billing_details`

:   Billing information associated with the PaymentMethod that may be
    used or required by particular types of payment methods.

`blik`

:   If this is a `blik` PaymentMethod, this hash contains details about
    the BLIK payment method.

`boleto`

:   If this is a `boleto` PaymentMethod, this hash contains details
    about the Boleto payment method.

`card`

:   If this is a `card` PaymentMethod, this hash contains the user\'s
    card details. For backwards compatibility, you can alternatively
    provide a Stripe token (e.g., for Apple Pay, Amex Express Checkout,
    or legacy Checkout) into the card hash with format
    `card: {token: "tok_visa"}`. When providing a card number, you must
    meet the requirements for [PCI
    compliance](https://stripe.com/docs/security#validating-pci-compliance){.perl-module}.
    We strongly recommend using Stripe.js instead of interacting with
    this API directly.

`customer_balance`

:   If this is a `customer_balance` PaymentMethod, this hash contains
    details about the CustomerBalance payment method.

`eps`

:   If this is an `eps` PaymentMethod, this hash contains details about
    the EPS payment method.

`fpx`

:   If this is an `fpx` PaymentMethod, this hash contains details about
    the FPX payment method.

`giropay`

:   If this is a `giropay` PaymentMethod, this hash contains details
    about the Giropay payment method.

`grabpay`

:   If this is a `grabpay` PaymentMethod, this hash contains details
    about the GrabPay payment method.

`ideal`

:   If this is an `ideal` PaymentMethod, this hash contains details
    about the iDEAL payment method.

`interac_present`

:   If this is an `interac_present` PaymentMethod, this hash contains
    details about the Interac Present payment method.

`klarna`

:   If this is a `klarna` PaymentMethod, this hash contains details
    about the Klarna payment method.

`konbini`

:   If this is a `konbini` PaymentMethod, this hash contains details
    about the Konbini payment method.

`link`

:   If this is an `Link` PaymentMethod, this hash contains details about
    the Link payment method.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`oxxo`

:   If this is an `oxxo` PaymentMethod, this hash contains details about
    the OXXO payment method.

`p24`

:   If this is a `p24` PaymentMethod, this hash contains details about
    the P24 payment method.

`paynow`

:   If this is a `paynow` PaymentMethod, this hash contains details
    about the PayNow payment method.

`promptpay`

:   If this is a `promptpay` PaymentMethod, this hash contains details
    about the PromptPay payment method.

`radar_options`

:   Options to configure Radar. See [Radar
    Session](https://stripe.com/docs/radar/radar-session){.perl-module}
    for more information.

`sepa_debit`

:   If this is a `sepa_debit` PaymentMethod, this hash contains details
    about the SEPA debit bank account.

`sofort`

:   If this is a `sofort` PaymentMethod, this hash contains details
    about the SOFORT payment method.

`type`

:   **Required**. The type of the PaymentMethod. An additional hash is
    included on the PaymentMethod with a name matching this value. It
    contains additional information specific to the PaymentMethod type.
    Required unless `payment_method` is specified (see the [Cloning
    PaymentMethods](https://stripe.com/docs/payments/payment-methods/connect#cloning-payment-methods){.perl-module}
    guide).

`us_bank_account`

:   If this is an `us_bank_account` PaymentMethod, this hash contains
    details about the US bank account payment method.

`wechat_pay`

:   If this is an `wechat_pay` PaymentMethod, this hash contains details
    about the wechat\_pay payment method.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_methods/create>

detach
------

        my $obj = $stripe->payment_methods( detach => $args ) || die( $stripe->error );

Provided with a [payment
method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module},
or a hash reference, this will issue a detach api call.

Returns a PaymentMethod object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_methods/detach>

list {#list-45}
----

        my $obj = $stripe->payment_methods( list => {
            customer => "cus_AJ78ZaALpqgiuZ",
            type     => "card", } ) || die( $stripe->error );

Provided with a [payment
method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object, this issue an api call to get the list of all payment method.

Possible parameters are:

`customer`

:   The ID of the customer whose PaymentMethods will be retrieved.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`type`

:   **Required**. A required filter on the list, based on the object
    `type` field.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_methods/list>

list\_customer\_payment\_methods
--------------------------------

        my $obj = $stripe->payment_methods( list_customer_payment_methods => {
            type => "card", } ) || die( $stripe->error );

Provided with a [payment
method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module},
or a hash reference, this will issue a list\_customer\_payment\_methods
api call.

A dictionary with a `data` property that contains an array of up to
`limit` PaymentMethods of type `type`, starting after PaymentMethods
`starting_after`. Each entry in the array is a separate PaymentMethod
object. If no more PaymentMethods are available, the resulting array
will be empty. This request should never return an error.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`type`

:   **Required**. A required filter on the list, based on the object
    `type` field.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_methods/customer_list>

retrieve {#retrieve-46}
--------

        my $obj = $stripe->payment_methods( retrieve => $args ) || die( $stripe->error );

Provided with a [payment
method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object or a hash reference, this will retrieve a Stripe payment method
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_methods/retrieve>

retrieve\_customer\_payment\_method
-----------------------------------

        my $obj = $stripe->payment_methods( retrieve_customer_payment_method => {
            type => "card", } ) || die( $stripe->error );

Provided with a [payment
method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module},
or a hash reference, this will issue a
retrieve\_customer\_payment\_method api call.

Returns a PaymentMethod object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_methods/customer>

update
------

        my $obj = $stripe->payment_methods( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [payment
method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object or a hash reference, this will update a Stripe payment method and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}

Possible parameters are:

`billing_details`

:   Billing information associated with the PaymentMethod that may be
    used or required by particular types of payment methods.

`card`

:   If this is a `card` PaymentMethod, this hash contains the user\'s
    card details.

`link`

:   If this is an `Link` PaymentMethod, this hash contains details about
    the Link payment method.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`us_bank_account`

:   If this is an `us_bank_account` PaymentMethod, this hash contains
    details about the US bank account payment method.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payment_methods/update>

PAYMENT METHODS
===============

You can **create**, **retrieve**, **update**, **list**, **attach**,
**detach**, payment methods.

attach
------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
or a
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object or an hash reference and this will attach the Stripe payment
method to the given customer and return its
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object.

Possible parameters are:

*id* A Stripe payment method id

:   

*customer* A Stripe customer id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_methods/attach>

create {#create-35}
------

Provided with a
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object or an hash reference and this will create a Stripe payment method
and return its
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object.

Possible parameters are:

*type* String. Any of card, fpx, ideal or sepa\_debit

:   

*billing\_details* An hash reference with the following properties: address.city address.country address.line1 address.line2 address.postal\_code address.state email name phone

:   

*metadata* An arbitrary hash reference

:   

*card* An hash reference with the following properties: exp\_month exp\_year number cvc

:   

*fpx* An hash reference with the property *bank*

:   

*ideal* An hash reference with the property *bank*

:   

*sepa\_debit* An hash reference with the property *iban*

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_methods/create>

detach
------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
or a
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object or an hash reference and this will dettach the Stripe payment
method from the given customer and return its
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object.

Possible parameters are:

*id* A Stripe payment method id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_methods/detach>

list {#list-46}
----

Provided with an hash reference of parameters, and this will get a list
of payment methods and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*customer* A Stripe customer id

:   

*type* String. One of card fpx ideal or sepa\_debit

:   

*ending\_before* A Stripe payment method id

:   

*limit* Integer

:   

*starting\_after* A Stripe payment method id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_methods/list>

retrieve {#retrieve-47}
--------

Provided with a
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object or an hash reference and this will retrieve a Stripe payment
method and return its
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object.

Possible parameters are:

*id* A Stripe payment method id. This is required

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_methods/retrieve>

update
------

Provided with a
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object or an hash reference and this will update the Stripe payment
method and return its
[Net::API::Stripe::Payment::Method](https://metacpan.org/pod/Net::API::Stripe::Payment::Method){.perl-module}
object.

Possible parameters are:

*id* A Stripe payment method id. This is required.

:   

*billing\_details* An hash reference with the following properties: address.city address.country address.line1 address.line2 address.postal\_code address.state email name phone

:   

*metadata* An arbitrary hash reference.

:   

*card* An hash reference with the following properties: exp\_month exp\_year

:   

*sepa\_debit* An hash reference with the following property: iban

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/payment_methods/update>

PAYOUT
======

You can cancel, create, list, retrieve, reverse or update payout

cancel
------

        my $obj = $stripe->payouts( cancel => $args ) || die( $stripe->error );

Provided with a
[payout](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns the payout object if the cancellation succeeded. Returns an
error if the payout has already been canceled or cannot be canceled.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payouts/cancel>

create {#create-36}
------

        my $obj = $stripe->payouts( create => {
            amount   => "1100",
            currency => "usd", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Payout](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module}
object or a hash reference, this will create a Stripe payout and return
an
[Net::API::Stripe::Payout](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module}
object.

Possible parameters are:

`amount`

:   **Required**. A positive integer in cents representing how much to
    payout.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`destination`

:   The ID of a bank account or a card to send the payout to. If no
    destination is supplied, the default external account for the
    specified currency will be used.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`method`

:   The method used to send this payout, which can be `standard` or
    `instant`. `instant` is only supported for payouts to debit cards.
    (See [Instant payouts for marketplaces for more
    information](https://stripe.com/blog/instant-payouts-for-marketplaces).){.perl-module}

`source_type`

:   The balance type of your Stripe balance to draw this payout from.
    Balances for different payment sources are kept separately. You can
    find the amounts with the balances API. One of `bank_account`,
    `card`, or `fpx`.

`statement_descriptor`

:   A string to be displayed on the recipient\'s bank or card statement.
    This may be at most 22 characters. Attempting to use a
    `statement_descriptor` longer than 22 characters will return an
    error. Note: Most banks will truncate this information and/or
    display it inconsistently. Some may not display it at all.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payouts/create>

list {#list-47}
----

        my $obj = $stripe->payouts( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[payout](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module}
object, this issue an api call to get the list of all payout.

Possible parameters are:

`arrival_date`

:   A filter on the list based on the object `arrival_date` field. The
    value can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`destination`

:   The ID of an external account - only return payouts sent to this
    external account.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return payouts that have the given status: `pending`, `paid`,
    `failed`, or `canceled`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payouts/list>

retrieve {#retrieve-48}
--------

        my $obj = $stripe->payouts( retrieve => $args ) || die( $stripe->error );

Provided with a
[payout](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module}
object or a hash reference, this will retrieve a Stripe payout and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payouts/retrieve>

reverse
-------

        my $obj = $stripe->payouts( reverse => $args ) || die( $stripe->error );

Provided with a
[payout](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module},
or a hash reference, this will issue a reverse api call.

Returns the reversing payout object if the reversal was successful.
Returns an error if the payout has already been reversed or cannot be
reversed.

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payouts/reverse>

update
------

        my $obj = $stripe->payouts( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[payout](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module}
object or a hash reference, this will update a Stripe payout and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payout){.perl-module}

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/payouts/update>

PERSON
======

You can create, delete, list, retrieve or update person

create {#create-37}
------

        my $obj = $stripe->persons( create => {
            first_name => "Jane",
            last_name  => "Diaz", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::Person](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module}
object or a hash reference, this will create a Stripe person and return
an
[Net::API::Stripe::Connect::Person](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module}
object.

Possible parameters are:

`address`

:   The person\'s address.

`address_kana`

:   The Kana variation of the person\'s address (Japan only).

`address_kanji`

:   The Kanji variation of the person\'s address (Japan only).

`dob`

:   The person\'s date of birth.

`documents`

:   Documents that may be submitted to satisfy various informational
    requests.

`email`

:   The person\'s email address.

`first_name`

:   The person\'s first name.

`first_name_kana`

:   The Kana variation of the person\'s first name (Japan only).

`first_name_kanji`

:   The Kanji variation of the person\'s first name (Japan only).

`full_name_aliases`

:   A list of alternate names or aliases that the person is known by.

`gender`

:   The person\'s gender (International regulations require either
    \"male\" or \"female\").

`id_number`

:   The person\'s ID number, as appropriate for their country. For
    example, a social security number in the U.S., social insurance
    number in Canada, etc. Instead of the number itself, you can also
    provide a [PII token provided by
    Stripe.js](https://stripe.com/docs/js/tokens_sources/create_token?type=pii){.perl-module}.

`id_number_secondary`

:   The person\'s secondary ID number, as appropriate for their country,
    will be used for enhanced verification checks. In Thailand, this
    would be the laser code found on the back of an ID card. Instead of
    the number itself, you can also provide a [PII token provided by
    Stripe.js](https://stripe.com/docs/js/tokens_sources/create_token?type=pii){.perl-module}.

`last_name`

:   The person\'s last name.

`last_name_kana`

:   The Kana variation of the person\'s last name (Japan only).

`last_name_kanji`

:   The Kanji variation of the person\'s last name (Japan only).

`maiden_name`

:   The person\'s maiden name.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`nationality`

:   The country where the person is a national. Two-letter country code
    ([ISO 3166-1
    alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)){.perl-module},
    or \"XX\" if unavailable.

`person_token`

:   A [person
    token](https://stripe.com/docs/connect/account-tokens){.perl-module},
    used to securely provide details to the person.

`phone`

:   The person\'s phone number.

`political_exposure`

:   Indicates if the person or any of their representatives, family
    members, or other closely related persons, declares that they hold
    or have held an important public job or function, in any
    jurisdiction.

`registered_address`

:   The person\'s registered address.

`relationship`

:   The relationship that this person has with the account\'s legal
    entity.

`ssn_last_4`

:   The last four digits of the person\'s Social Security number (U.S.
    only).

`verification`

:   The person\'s verification status.

More information from Stripe api documentation at
<https://stripe.com/docs/api/persons/create>

delete
------

        my $obj = $stripe->persons( delete => $args ) || die( $stripe->error );

Provided with a
[person](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
person. It returns the person object that was deleted with its property
`deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/persons/delete>

list {#list-48}
----

        my $obj = $stripe->persons( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[person](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module}
object, this issue an api call to get the list of all person.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`relationship`

:   Filters on the list of people returned based on the person\'s
    relationship to the account\'s company.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/persons/list>

retrieve {#retrieve-49}
--------

        my $obj = $stripe->persons( retrieve => $args ) || die( $stripe->error );

Provided with a
[person](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module}
object or a hash reference, this will retrieve a Stripe person and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/persons/retrieve>

update
------

        my $obj = $stripe->persons( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[person](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module}
object or a hash reference, this will update a Stripe person and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Person){.perl-module}

Possible parameters are:

`address`

:   The person\'s address.

`address_kana`

:   The Kana variation of the person\'s address (Japan only).

`address_kanji`

:   The Kanji variation of the person\'s address (Japan only).

`dob`

:   The person\'s date of birth.

`documents`

:   Documents that may be submitted to satisfy various informational
    requests.

`email`

:   The person\'s email address.

`first_name`

:   The person\'s first name.

`first_name_kana`

:   The Kana variation of the person\'s first name (Japan only).

`first_name_kanji`

:   The Kanji variation of the person\'s first name (Japan only).

`full_name_aliases`

:   A list of alternate names or aliases that the person is known by.

`gender`

:   The person\'s gender (International regulations require either
    \"male\" or \"female\").

`id_number`

:   The person\'s ID number, as appropriate for their country. For
    example, a social security number in the U.S., social insurance
    number in Canada, etc. Instead of the number itself, you can also
    provide a [PII token provided by
    Stripe.js](https://stripe.com/docs/js/tokens_sources/create_token?type=pii){.perl-module}.

`id_number_secondary`

:   The person\'s secondary ID number, as appropriate for their country,
    will be used for enhanced verification checks. In Thailand, this
    would be the laser code found on the back of an ID card. Instead of
    the number itself, you can also provide a [PII token provided by
    Stripe.js](https://stripe.com/docs/js/tokens_sources/create_token?type=pii){.perl-module}.

`last_name`

:   The person\'s last name.

`last_name_kana`

:   The Kana variation of the person\'s last name (Japan only).

`last_name_kanji`

:   The Kanji variation of the person\'s last name (Japan only).

`maiden_name`

:   The person\'s maiden name.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`nationality`

:   The country where the person is a national. Two-letter country code
    ([ISO 3166-1
    alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)){.perl-module},
    or \"XX\" if unavailable.

`person_token`

:   A [person
    token](https://stripe.com/docs/connect/account-tokens){.perl-module},
    used to securely provide details to the person.

`phone`

:   The person\'s phone number.

`political_exposure`

:   Indicates if the person or any of their representatives, family
    members, or other closely related persons, declares that they hold
    or have held an important public job or function, in any
    jurisdiction.

`registered_address`

:   The person\'s registered address.

`relationship`

:   The relationship that this person has with the account\'s legal
    entity.

`ssn_last_4`

:   The last four digits of the person\'s Social Security number (U.S.
    only).

`verification`

:   The person\'s verification status.

More information from Stripe api documentation at
<https://stripe.com/docs/api/persons/update>

PLAN
====

You can create, delete, list, retrieve or update plan

create {#create-38}
------

        my $obj = $stripe->plans( create => {
            amount   => "1200",
            currency => "usd",
            interval => "month",
            product  => "prod_MMlLy1NOplKgdo", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object or a hash reference, this will create a Stripe plan and return an
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object.

Possible parameters are:

`active`

:   Whether the plan is currently available for new subscriptions.
    Defaults to `true`.

`aggregate_usage`

:   Specifies a usage aggregation strategy for plans of
    `usage_type=metered`. Allowed values are `sum` for summing up all
    usage during a period, `last_during_period` for using the last usage
    record reported within a period, `last_ever` for using the last
    usage record ever (across period bounds) or `max` which uses the
    usage record with the maximum reported usage during a period.
    Defaults to `sum`.

`amount`

:   **Required unless billing\_scheme=tiered** A positive integer in JPY
    (or 0 for a free plan) representing how much to charge on a
    recurring basis.

`amount_decimal`

:   Same as `amount`, but accepts a decimal value with at most 12
    decimal places. Only one of `amount` and `amount_decimal` can be
    set.

`billing_scheme`

:   Describes how to compute the price per period. Either `per_unit` or
    `tiered`. `per_unit` indicates that the fixed amount (specified in
    `amount`) will be charged per unit in `quantity` (for plans with
    `usage_type=licensed`), or per unit of total usage (for plans with
    `usage_type=metered`). `tiered` indicates that the unit pricing will
    be computed using a tiering strategy as defined using the `tiers`
    and `tiers_mode` attributes.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`id`

:   An identifier randomly generated by Stripe. Used to identify this
    plan when subscribing a customer. You can optionally override this
    ID, but the ID must be unique across all plans in your Stripe
    account. You can, however, use the same plan ID in both live and
    test modes.

`interval`

:   **Required**. Specifies billing frequency. Either `day`, `week`,
    `month` or `year`.

`interval_count`

:   The number of intervals between subscription billings. For example,
    `interval=month` and `interval_count=3` bills every 3 months.
    Maximum of one year interval allowed (1 year, 12 months, or 52
    weeks).

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`nickname`

:   A brief description of the plan, hidden from customers.

`product`

:   **Required**. The product whose pricing the created plan will
    represent. This can either be the ID of an existing product, or a
    dictionary containing fields used to create a [service
    product](https://stripe.com/docs/api#product_object-type){.perl-module}.

`tiers`

:   **Required if billing\_scheme=tiered** Each element represents a
    pricing tier. This parameter requires `billing_scheme` to be set to
    `tiered`. See also the documentation for `billing_scheme`.

`tiers_mode`

:   **Required if billing\_scheme=tiered** Defines if the tiering price
    should be `graduated` or `volume` based. In `volume`-based tiering,
    the maximum quantity within a period determines the per unit price,
    in `graduated` tiering pricing can successively change as the
    quantity grows.

`transform_usage`

:   Apply a transformation to the reported usage or set quantity before
    computing the billed price. Cannot be combined with `tiers`.

`trial_period_days`

:   Default number of trial days when subscribing a customer to this
    plan using
    [`trial_from_plan=true`](https://stripe.com/docs/api#create_subscription-trial_from_plan){.perl-module}.

`usage_type`

:   Configures how the quantity per period should be determined. Can be
    either `metered` or `licensed`. `licensed` automatically bills the
    `quantity` set when adding it to a subscription. `metered`
    aggregates the total usage based on usage records. Defaults to
    `licensed`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/plans/create>

delete
------

        my $obj = $stripe->plans( delete => $args ) || die( $stripe->error );

Provided with a
[plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
plan. It returns the plan object that was deleted with its property
`deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/plans/delete>

list {#list-49}
----

        my $obj = $stripe->plans( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object, this issue an api call to get the list of all plan.

Possible parameters are:

`active`

:   Only return plans that are active or inactive (e.g., pass `false` to
    list all inactive plans).

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`product`

:   Only return plans for the given product.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/plans/list>

retrieve {#retrieve-50}
--------

        my $obj = $stripe->plans( retrieve => $args ) || die( $stripe->error );

Provided with a
[plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object or a hash reference, this will retrieve a Stripe plan and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/plans/retrieve>

update
------

        my $obj = $stripe->plans( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object or a hash reference, this will update a Stripe plan and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}

Possible parameters are:

`active`

:   Whether the plan is currently available for new subscriptions.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`nickname`

:   A brief description of the plan, hidden from customers.

`product`

:   The product the plan belongs to. This cannot be changed once it has
    been used in a subscription or subscription schedule.

`trial_period_days`

:   Default number of trial days when subscribing a customer to this
    plan using
    [`trial_from_plan=true`](https://stripe.com/docs/api#create_subscription-trial_from_plan){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/plans/update>

PLANS
=====

You can **create**, **retrieve**, **update**, **list**, **delete**
**plans**.

create {#create-39}
------

Provided with a
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object or an hash reference of parameters and this will create a Stripe
plan and return its
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object.

Possible parameters are:

*id* A Stripe plan id (optional)

:   

*active* Boolean

:   

*aggregate\_usage* String

:   

*amount* Integer

:   

*amount\_decimal* Decimal

:   

*billing\_scheme* String. One of per\_unit or tiered

:   

*currency* A 3-letter ISO 4217 code such as `jpy` for Japanese Yen or `eur` for Euro

:   

*interval* String. One of day, week, month or year

:   

*interval\_count* Integer

:   

*metadata* An arbitrary hash reference

:   

*nickname* String

:   

*product* A Stripe product id

:   

*tiers* An hash reference with the following properties: up\_to flat\_amount flat\_amount\_decimal unit\_amount unit\_amount\_decimal

:   

*tiers\_mode* String. One of graduated or volume

:   

*transform\_usage* An hash reference with the following properties: divide\_by round

:   

*trial\_period\_days* Integer

:   

*usage\_type* String. One of metered\|licensed

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/plans/create>

delete
------

Provided with a
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object or an hash reference of parameters and this will remove a Stripe
plan and return its
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object.

Possible parameters are:

*id* A Stripe plan id. This is required

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/plans/delete>

list {#list-50}
----

Provided with an hash reference of parameters, this will get the list of
Stripe plans and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*created* Date or unix timestamp

:   

*email* String. E-mail address

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*product* A Stripe product id

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/plans/list>

retrieve {#retrieve-51}
--------

Provided with a
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object or an hash reference of parameters and this will retrieve a
Stripe plan and return its
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object.

Possible parameters are:

*id* A Stripe plan id

:   

For more information, see Stripe documentation here:
[hhttps://stripe.com/docs/api/plans/retrieve](hhttps://stripe.com/docs/api/plans/retrieve){.perl-module}

update
------

Provided with a
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object or an hash reference of parameters and this will update a Stripe
plan and return its
[Net::API::Stripe::Billing::Plan](https://metacpan.org/pod/Net::API::Stripe::Billing::Plan){.perl-module}
object.

Possible parameters are:

*id* A Stripe plan id (optional)

:   

*active* Boolean

:   

*metadata* An arbitrary hash reference

:   

*nickname* String

:   

*product* A Stripe product id

:   

*trial\_period\_days* Integer

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/plans/update>

PRICE
=====

You can create, list, retrieve, search or update price

create {#create-40}
------

        my $obj = $stripe->prices( create => {
            currency    => "usd",
            product     => "prod_MMlLy1NOplKgdo",
            recurring   =>
            {
                interval => "month",
            }
            unit_amount => "1200", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object or a hash reference, this will create a Stripe price and return
an
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object.

Possible parameters are:

`active`

:   Whether the price can be used for new purchases. Defaults to `true`.

`billing_scheme`

:   Describes how to compute the price per period. Either `per_unit` or
    `tiered`. `per_unit` indicates that the fixed amount (specified in
    `unit_amount` or `unit_amount_decimal`) will be charged per unit in
    `quantity` (for prices with `usage_type=licensed`), or per unit of
    total usage (for prices with `usage_type=metered`). `tiered`
    indicates that the unit pricing will be computed using a tiering
    strategy as defined using the `tiers` and `tiers_mode` attributes.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`currency_options`

:   Prices defined in each available currency option. Each key must be a
    three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module}
    and a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}. For
    example, to define your price in `eur`, pass the fields below in the
    `eur` key of `currency_options`.

`custom_unit_amount`

:   **Required unless unit\_amount is provided** When set, provides
    configuration for the amount to be adjusted by the customer during
    Checkout Sessions and Payment Links.

`lookup_key`

:   A lookup key used to retrieve prices dynamically from a static
    string. This may be up to 200 characters.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`nickname`

:   A brief description of the price, hidden from customers.

`product`

:   **required unless product\_data is provided** The ID of the product
    that this price will belong to.

`product_data`

:   **required unless product is provided** These fields can be used to
    create a new product that this price will belong to.

`recurring`

:   The recurring components of a price such as `interval` and
    `usage_type`.

`tax_behavior`

:   Specifies whether the price is considered inclusive of taxes or
    exclusive of taxes. One of `inclusive`, `exclusive`, or
    `unspecified`. Once specified as either `inclusive` or `exclusive`,
    it cannot be changed.

`tiers`

:   **Required if billing\_scheme=tiered** Each element represents a
    pricing tier. This parameter requires `billing_scheme` to be set to
    `tiered`. See also the documentation for `billing_scheme`.

`tiers_mode`

:   **Required if billing\_scheme=tiered** Defines if the tiering price
    should be `graduated` or `volume` based. In `volume`-based tiering,
    the maximum quantity within a period determines the per unit price,
    in `graduated` tiering pricing can successively change as the
    quantity grows.

`transfer_lookup_key`

:   If set to true, will atomically remove the lookup key from the
    existing price, and assign it to this price.

`transform_quantity`

:   Apply a transformation to the reported usage or set quantity before
    computing the billed price. Cannot be combined with `tiers`.

`unit_amount`

:   **Required conditionally** A positive integer in JPY (or 0 for a
    free price) representing how much to charge. One of `unit_amount` or
    `custom_unit_amount` is required, unless `billing_scheme=tiered`.

`unit_amount_decimal`

:   Same as `unit_amount`, but accepts a decimal value in JPY with at
    most 12 decimal places. Only one of `unit_amount` and
    `unit_amount_decimal` can be set.

More information from Stripe api documentation at
<https://stripe.com/docs/api/prices/create>

list {#list-51}
----

        my $obj = $stripe->prices( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object, this issue an api call to get the list of all price.

Possible parameters are:

`active`

:   Only return prices that are active or inactive (e.g., pass `false`
    to list all inactive prices).

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`currency`

:   Only return prices for the given currency.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`lookup_keys`

:   Only return the price with these lookup\_keys, if any exist.

`product`

:   Only return prices for the given product.

`recurring`

:   Only return prices with these recurring fields.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`type`

:   Only return prices of type `recurring` or `one_time`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/prices/list>

retrieve {#retrieve-52}
--------

        my $obj = $stripe->prices( retrieve => $args ) || die( $stripe->error );

Provided with a
[price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object or a hash reference, this will retrieve a Stripe price and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/prices/retrieve>

search
------

        my $obj = $stripe->prices( search => {
            query => "active:'true' AND metadata['order_id']:'6735'", } ) || die( $stripe->error );

Provided with a
[price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module},
or a hash reference, this will issue a search api call.

A dictionary with a `data` property that contains an array of up to
`limit` prices. If no objects match the query, the resulting array will
be empty. See the related guide on [expanding properties in
lists](https://stripe.com/docs/expand#lists){.perl-module}.

Possible parameters are:

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`page`

:   A cursor for pagination across multiple pages of results. Don\'t
    include this parameter on the first call. Use the next\_page value
    returned in a previous response to request subsequent results.

`query`

:   **Required**. The search query string. See [search query
    language](https://stripe.com/docs/search#search-query-language){.perl-module}
    and the list of supported [query fields for
    prices](https://stripe.com/docs/search#query-fields-for-prices){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/prices/search>

update
------

        my $obj = $stripe->prices( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object or a hash reference, this will update a Stripe price and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}

Possible parameters are:

`active`

:   Whether the price can be used for new purchases. Defaults to `true`.

`currency_options`

:   Prices defined in each available currency option. Each key must be a
    three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module}
    and a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}. For
    example, to define your price in `eur`, pass the fields below in the
    `eur` key of `currency_options`.

`lookup_key`

:   A lookup key used to retrieve prices dynamically from a static
    string. This may be up to 200 characters.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`nickname`

:   A brief description of the price, hidden from customers.

`tax_behavior`

:   Specifies whether the price is considered inclusive of taxes or
    exclusive of taxes. One of `inclusive`, `exclusive`, or
    `unspecified`. Once specified as either `inclusive` or `exclusive`,
    it cannot be changed.

`transfer_lookup_key`

:   If set to true, will atomically remove the lookup key from the
    existing price, and assign it to this price.

More information from Stripe api documentation at
<https://stripe.com/docs/api/prices/update>

PRICES
======

You can **create**, **retrieve**, **update**, **list** products

create {#create-41}
------

Provided with a
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object or an hash reference of parameters and this will create a Stripe
product and return its
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object

Possible parameters are:

*active* boolean

:   Whether the price is currently active. Defaults to true.

*billing\_scheme* string

:   

*currency* String

:   Three-letter ISO currency code, in lowercase. Must be a supported
    currency.

*lookup\_key* string

:   

*metadata* Hash

:   Set of key-value pairs that you can attach to an object. This can be
    useful for storing additional information about the object in a
    structured format. Individual keys can be unset by posting an empty
    value to them. All keys can be unset by posting an empty value to
    metadata.

*nickname* String

:   A brief description of the price, hidden from customers.

*product\_data*

:   A hash or
    [Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}

*unit\_amount* A number

:   A positive integer in JPY (or 0 for a free price) representing how
    much to charge.

    This is required only if *billing-scheme* is set to `perl-unit`

*recurring*

:   The recurring components of a price such as interval and
    usage\_type.

    Possible properties are :

    *interval* string

    :   

    *aggregate\_usage* string

    :   

    *interval\_count* number

    :   

    *trial\_period\_days* number

    :   

    *usage\_type* string

    :   

*tiers* hash

:   See
    [Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
    for details

*tiers\_mode* string

:   

*transfer\_lookup\_key* string

:   

*transform\_quantity* number

:   

*unit\_amount\_decimal* number

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/prices/create#create_price>

list {#list-52}
----

Provided with an hash reference of parameters, this will retrieve a list
of Stripe prices and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object

Possible parameters are:

*active* Boolean

:   

*created* Date or unix timestamp

:   

*currency* String

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*lookup\_key* String

:   

*product* [Net::API::String::Product](https://metacpan.org/pod/Net::API::String::Product){.perl-module} id or object

:   

*recurring* Hash with `inerval` and `usage_type` properties

:   

*starting\_after* A Stripe credit note id

:   

*type* String. One of `recurring` or `one_time`

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/prices/list>

retrieve {#retrieve-53}
--------

Provided with a
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object or an hash reference of parameters and this will retrieve a
Stripe price and return its
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object

Possible parameters are:

*id* A Stripe price id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/prices/retrieve#retrieve_price>

update
------

Provided with a
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object or an hash reference of parameters and this will update a Stripe
price and return its updated
[Net::API::Stripe::Price](https://metacpan.org/pod/Net::API::Stripe::Price){.perl-module}
object

As per the Stripe documentation, \"After prices are created, you can
only update their metadata, nickname, and active fields.\" (see
[Products and
Prices](https://stripe.com/docs/billing/prices-guide|){.perl-module})

Possible parameters are:

*id* A Stripe price id

:   

*active* boolean

:   Whether the price is currently active. Defaults to true.

*lookup\_key* string

:   

*metadata* Hash

:   Set of key-value pairs that you can attach to an object. This can be
    useful for storing additional information about the object in a
    structured format. Individual keys can be unset by posting an empty
    value to them. All keys can be unset by posting an empty value to
    metadata.

*nickname* String

:   A brief description of the price, hidden from customers.

*recurring*

:   The recurring components of a price such as interval and
    usage\_type.

    Possible properties are :

    *interval* string

    :   

    *aggregate\_usage* string

    :   

    *interval\_count* number

    :   

    *trial\_period\_days* number

    :   

    *usage\_type* string

    :   

*transfer\_lookup\_key* string

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/prices/update>

PRODUCT
=======

You can create, delete, list, retrieve, search or update product

create {#create-42}
------

        my $obj = $stripe->products( create => {
            name => "Gold Special", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object or a hash reference, this will create a Stripe product and return
an
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object.

Possible parameters are:

`active`

:   Whether the product is currently available for purchase. Defaults to
    `true`.

`default_price_data`

:   Data used to generate a new
    [Price](https://stripe.com/docs/api/prices){.perl-module} object.
    This Price will be set as the default price for this product.

`description`

:   The product\'s description, meant to be displayable to the customer.
    Use this field to optionally store a long form explanation of the
    product being sold for your own rendering purposes.

`id`

:   An identifier will be randomly generated by Stripe. You can
    optionally override this ID, but the ID must be unique across all
    products in your Stripe account.

`images`

:   A list of up to 8 URLs of images for this product, meant to be
    displayable to the customer.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   **Required**. The product\'s name, meant to be displayable to the
    customer.

`package_dimensions`

:   The dimensions of this product for shipping purposes.

`shippable`

:   Whether this product is shipped (i.e., physical goods).

`statement_descriptor`

:   An arbitrary string to be displayed on your customer\'s credit card
    or bank statement. While most banks display this information
    consistently, some may display it incorrectly or not at all.

    This may be up to 22 characters. The statement description may not
    include `&lt;`, `&gt;`, `\`, `"`, `'` characters, and will appear on
    your customer\'s statement in capital letters. Non-ASCII characters
    are automatically stripped. It must contain at least one letter.

`tax_code`

:   A [tax
    code](https://stripe.com/docs/tax/tax-categories){.perl-module} ID.

`unit_label`

:   A label that represents units of this product in Stripe and on
    customers' receipts and invoices. When set, this will be included in
    associated invoice line item descriptions.

`url`

:   A URL of a publicly-accessible webpage for this product.

More information from Stripe api documentation at
<https://stripe.com/docs/api/products/create>

delete
------

        my $obj = $stripe->products( delete => $args ) || die( $stripe->error );

Provided with a
[product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
product. It returns the product object that was deleted with its
property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/products/delete>

list {#list-53}
----

        my $obj = $stripe->products( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object, this issue an api call to get the list of all product.

Possible parameters are:

`active`

:   Only return products that are active or inactive (e.g., pass `false`
    to list all inactive products).

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`ids`

:   Only return products with the given IDs. Cannot be used with
    [starting\_after](https://stripe.com/docs/api/products/list#list_products-starting_after){.perl-module}
    or
    [ending\_before](https://stripe.com/docs/api/products/list#list_products-ending_before){.perl-module}.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`shippable`

:   Only return products that can be shipped (i.e., physical, not
    digital products).

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`url`

:   Only return products with the given url.

More information from Stripe api documentation at
<https://stripe.com/docs/api/products/list>

retrieve {#retrieve-54}
--------

        my $obj = $stripe->products( retrieve => $args ) || die( $stripe->error );

Provided with a
[product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object or a hash reference, this will retrieve a Stripe product and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/products/retrieve>

search
------

        my $obj = $stripe->products( search => {
            query => "active:'true' AND metadata['order_id']:'6735'", } ) || die( $stripe->error );

Provided with a
[product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module},
or a hash reference, this will issue a search api call.

A dictionary with a `data` property that contains an array of up to
`limit` products. If no objects match the query, the resulting array
will be empty. See the related guide on [expanding properties in
lists](https://stripe.com/docs/expand#lists){.perl-module}.

Possible parameters are:

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`page`

:   A cursor for pagination across multiple pages of results. Don\'t
    include this parameter on the first call. Use the next\_page value
    returned in a previous response to request subsequent results.

`query`

:   **Required**. The search query string. See [search query
    language](https://stripe.com/docs/search#search-query-language){.perl-module}
    and the list of supported [query fields for
    products](https://stripe.com/docs/search#query-fields-for-products){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/products/search>

update
------

        my $obj = $stripe->products( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object or a hash reference, this will update a Stripe product and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}

Possible parameters are:

`active`

:   Whether the product is available for purchase.

`default_price`

:   The ID of the
    [Price](https://stripe.com/docs/api/prices){.perl-module} object
    that is the default price for this product.

`description`

:   The product\'s description, meant to be displayable to the customer.
    Use this field to optionally store a long form explanation of the
    product being sold for your own rendering purposes.

`images`

:   A list of up to 8 URLs of images for this product, meant to be
    displayable to the customer.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   The product\'s name, meant to be displayable to the customer.

`package_dimensions`

:   The dimensions of this product for shipping purposes.

`shippable`

:   Whether this product is shipped (i.e., physical goods).

`statement_descriptor`

:   An arbitrary string to be displayed on your customer\'s credit card
    or bank statement. While most banks display this information
    consistently, some may display it incorrectly or not at all.

    This may be up to 22 characters. The statement description may not
    include `&lt;`, `&gt;`, `\`, `"`, `'` characters, and will appear on
    your customer\'s statement in capital letters. Non-ASCII characters
    are automatically stripped. It must contain at least one letter. May
    only be set if `type=service`.

`tax_code`

:   A [tax
    code](https://stripe.com/docs/tax/tax-categories){.perl-module} ID.

`unit_label`

:   A label that represents units of this product in Stripe and on
    customers' receipts and invoices. When set, this will be included in
    associated invoice line item descriptions. May only be set if
    `type=service`.

`url`

:   A URL of a publicly-accessible webpage for this product.

More information from Stripe api documentation at
<https://stripe.com/docs/api/products/update>

PRODUCTS
========

You can **create**, **retrieve**, **update**, **list**, **delete**
products

create {#create-43}
------

Provided with a
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object or an hash reference of parameters and this will create a Stripe
product and return its
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object

Possible parameters are:

*id* An id to be used as a Stripe product id

:   

*active* Boolean

:   

*attributes* An array of up to 5 elements

:   

*caption* String

:   

*deactivate\_on* Date or timestamp

:   

*description* Text

:   

*images* An array of up to 8 images

:   

*metadata* An arbitrary of hash reference

:   

*name* Stripe. Max length of 250 characters

:   

*package\_dimensions* An hash reference with the following properties: height, length, weight and width

:   

*shippable* Boolean

:   

*statement\_descriptor* Text

:   

*type* String. One of good or service

:   

*unit\_label* String

:   

*url* An url. For goods

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/service_products/create>

delete
------

Provided with a
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object or an hash reference of parameters and this will remove a Stripe
product and return its
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object with its property *deleted* set to true.

Possible parameters are:

*id* A Stripe product id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/service_products/delete>

list {#list-54}
----

Provided with an hash reference of parameters, this will retrieve a list
of Stripe products and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object

Possible parameters are:

*active* Boolean

:   

*created* Date or unix timestamp

:   

*email* String. E-mail address

:   

*ending\_before* A Stripe credit note id

:   

*ids* An array

:   

*limit* Integer

:   

*shippable* Boolean

:   

*starting\_after* A Stripe credit note id

:   

*type* String. One of service or good

:   

*url* The product url

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/service_products/list>

retrieve {#retrieve-55}
--------

Provided with a
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object or an hash reference of parameters and this will retrieve a
Stripe product and return its
[Net::API::Stripe::Product](https://metacpan.org/pod/Net::API::Stripe::Product){.perl-module}
object

Possible parameters are:

*id* A Stripe product id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/service_products/retrieve>

update
------

Possible parameters are:

*id* A Stripe product id

:   

*active* Boolean

:   

*attributes* An array of up to 5 elements

:   

*caption* String

:   

*deactivate\_on* Date or timestamp

:   

*description* Text

:   

*images* An array of up to 8 images

:   

*metadata* An arbitrary of hash reference

:   

*name* Stripe. Max length of 250 characters

:   

*package\_dimensions* An hash reference with the following properties: height, length, weight and width

:   

*shippable* Boolean

:   

*statement\_descriptor* Text

:   

*type* String. One of good or service

:   

*unit\_label* String

:   

*url* An url. For goods

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/service_products/update>

PROMOTION CODE
==============

You can create, list, retrieve or update promotion code

create {#create-44}
------

        my $obj = $stripe->promotion_codes( create => {
            coupon => "Z4OV52SU", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::PromotionCode](https://metacpan.org/pod/Net::API::Stripe::Billing::PromotionCode){.perl-module}
object or a hash reference, this will create a Stripe promotion code and
return an
[Net::API::Stripe::Billing::PromotionCode](https://metacpan.org/pod/Net::API::Stripe::Billing::PromotionCode){.perl-module}
object.

Possible parameters are:

`active`

:   Whether the promotion code is currently active.

`code`

:   The customer-facing code. Regardless of case, this code must be
    unique across all active promotion codes for a specific customer. If
    left blank, we will generate one automatically.

`coupon`

:   **Required**. The coupon for this promotion code.

`customer`

:   The customer that this promotion code can be used by. If not set,
    the promotion code can be used by all customers.

`expires_at`

:   The timestamp at which this promotion code will expire. If the
    coupon has specified a `redeems_by`, then this value cannot be after
    the coupon\'s `redeems_by`.

`max_redemptions`

:   A positive integer specifying the number of times the promotion code
    can be redeemed. If the coupon has specified a `max_redemptions`,
    then this value cannot be greater than the coupon\'s
    `max_redemptions`.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`restrictions`

:   Settings that restrict the redemption of the promotion code.

More information from Stripe api documentation at
<https://stripe.com/docs/api/promotion_codes/create>

list {#list-55}
----

        my $obj = $stripe->promotion_codes( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [promotion
code](https://metacpan.org/pod/Net::API::Stripe::Billing::PromotionCode){.perl-module}
object, this issue an api call to get the list of all promotion code.

Possible parameters are:

`active`

:   Filter promotion codes by whether they are active.

`code`

:   Only return promotion codes that have this case-insensitive code.

`coupon`

:   Only return promotion codes for this coupon.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`customer`

:   Only return promotion codes that are restricted to this customer.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/promotion_codes/list>

retrieve {#retrieve-56}
--------

        my $obj = $stripe->promotion_codes( retrieve => $args ) || die( $stripe->error );

Provided with a [promotion
code](https://metacpan.org/pod/Net::API::Stripe::Billing::PromotionCode){.perl-module}
object or a hash reference, this will retrieve a Stripe promotion code
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::PromotionCode){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/promotion_codes/retrieve>

update
------

        my $obj = $stripe->promotion_codes( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [promotion
code](https://metacpan.org/pod/Net::API::Stripe::Billing::PromotionCode){.perl-module}
object or a hash reference, this will update a Stripe promotion code and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::PromotionCode){.perl-module}

Possible parameters are:

`active`

:   Whether the promotion code is currently active. A promotion code can
    only be reactivated when the coupon is still valid and the promotion
    code is otherwise redeemable.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`restrictions`

:   Settings that restrict the redemption of the promotion code.

More information from Stripe api documentation at
<https://stripe.com/docs/api/promotion_codes/update>

QUOTE
=====

You can accept, cancel, create, download, finalize, line\_items, lines,
list, retrieve, update, upfront\_line\_items or upfront\_lines quote

accept
------

        my $obj = $stripe->quotes( accept => $args ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module},
or a hash reference, this will issue a accept api call.

Returns an accepted quote and creates an invoice, subscription or
subscription schedule. Returns [an
error](https://stripe.com/docs/api/errors){.perl-module} otherwise.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/accept>

cancel
------

        my $obj = $stripe->quotes( cancel => $args ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns a canceled quote. Returns [an
error](https://stripe.com/docs/api/errors){.perl-module} otherwise.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/cancel>

create {#create-45}
------

        my $obj = $stripe->quotes( create => {
            customer   => "cus_AJ78ZaALpqgiuZ",
            line_items => [,
                price    => "price_1Le1oa2eZvKYlo2CuD7mwpZu",
                quantity => "2",
            ], } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::Quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module}
object or a hash reference, this will create a Stripe quote and return
an
[Net::API::Stripe::Billing::Quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module}
object.

Possible parameters are:

`application_fee_amount`

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    owner\'s Stripe account. There cannot be any line items with
    recurring prices when using this field.

`application_fee_percent`

:   A non-negative decimal between 0 and 100, with at most two decimal
    places. This represents the percentage of the subscription invoice
    subtotal that will be transferred to the application owner\'s Stripe
    account. There must be at least 1 line item with a recurring price
    to use this field.

`automatic_tax`

:   Settings for automatic tax lookup for this quote and resulting
    invoices and subscriptions.

`collection_method`

:   Either `charge_automatically`, or `send_invoice`. When charging
    automatically, Stripe will attempt to pay invoices at the end of the
    subscription cycle or at invoice finalization using the default
    payment method attached to the subscription or customer. When
    sending an invoice, Stripe will email your customer an invoice with
    payment instructions. Defaults to `charge_automatically`.

`customer`

:   The customer for which this quote belongs to. A customer is required
    before finalizing the quote. Once specified, it cannot be changed.

`default_tax_rates`

:   The tax rates that will apply to any line item that does not have
    `tax_rates` set.

`description`

:   A description that will be displayed on the quote PDF. If no value
    is passed, the default description configured in your [quote
    template
    settings](https://dashboard.stripe.com/settings/billing/quote){.perl-module}
    will be used.

`discounts`

:   The discounts applied to the quote. You can only set up to one
    discount.

`expires_at`

:   A future timestamp on which the quote will be canceled if in `open`
    or `draft` status. Measured in seconds since the Unix epoch. If no
    value is passed, the default expiration date configured in your
    [quote template
    settings](https://dashboard.stripe.com/settings/billing/quote){.perl-module}
    will be used.

`footer`

:   A footer that will be displayed on the quote PDF. If no value is
    passed, the default footer configured in your [quote template
    settings](https://dashboard.stripe.com/settings/billing/quote){.perl-module}
    will be used.

`from_quote`

:   Clone an existing quote. The new quote will be created in
    `status=draft`. When using this parameter, you cannot specify any
    other parameters except for `expires_at`.

`header`

:   A header that will be displayed on the quote PDF. If no value is
    passed, the default header configured in your [quote template
    settings](https://dashboard.stripe.com/settings/billing/quote){.perl-module}
    will be used.

`invoice_settings`

:   All invoices will be billed using the specified settings.

`line_items`

:   A list of line items the customer is being quoted for. Each line
    item includes information about the product, the quantity, and the
    resulting cost.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`on_behalf_of`

:   The account on behalf of which to charge.

`subscription_data`

:   When creating a subscription or subscription schedule, the specified
    configuration data will be used. There must be at least one line
    item with a recurring price for a subscription or subscription
    schedule to be created. A subscription schedule is created if
    `subscription_data[effective_date]` is present and in the future,
    otherwise a subscription is created.

`test_clock`

:   ID of the test clock to attach to the quote.

`transfer_data`

:   The data with which to automatically create a Transfer for each of
    the invoices.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/create>

download
--------

        my $obj = $stripe->quotes( download => $args ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module},
or a hash reference, this will issue a download api call.

The PDF file for the quote.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/pdf>

finalize
--------

        my $obj = $stripe->quotes( finalize => $args ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module},
or a hash reference, this will issue a finalize api call.

Returns an open quote. Returns [an
error](https://stripe.com/docs/api/errors){.perl-module} otherwise.

Possible parameters are:

`expires_at`

:   A future timestamp on which the quote will be canceled if in `open`
    or `draft` status. Measured in seconds since the Unix epoch.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/finalize>

line\_items
-----------

        my $obj = $stripe->quotes( line_items => $args ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module},
or a hash reference, this will issue a line\_items api call.

A dictionary with a `data` property that contains an array of up to
`limit` quote line items, starting after Line Item `starting_after`.
Each entry in the array is a separate Line Item object. If no more line
items are available, the resulting array will be empty.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/line_items/list>

list {#list-56}
----

        my $obj = $stripe->quotes( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module}
object, this issue an api call to get the list of all quote.

Possible parameters are:

`customer`

:   The ID of the customer whose quotes will be retrieved.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   The status of the quote.

`test_clock`

:   Provides a list of quotes that are associated with the specified
    test clock. The response will not include quotes with test clocks if
    this and the customer parameter is not set.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/list>

retrieve {#retrieve-57}
--------

        my $obj = $stripe->quotes( retrieve => $args ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module}
object or a hash reference, this will retrieve a Stripe quote and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/retrieve>

update
------

        my $obj = $stripe->quotes( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module}
object or a hash reference, this will update a Stripe quote and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module}

Possible parameters are:

`application_fee_amount`

:   The amount of the application fee (if any) that will be requested to
    be applied to the payment and transferred to the application
    owner\'s Stripe account. There cannot be any line items with
    recurring prices when using this field.

`application_fee_percent`

:   A non-negative decimal between 0 and 100, with at most two decimal
    places. This represents the percentage of the subscription invoice
    subtotal that will be transferred to the application owner\'s Stripe
    account. There must be at least 1 line item with a recurring price
    to use this field.

`automatic_tax`

:   Settings for automatic tax lookup for this quote and resulting
    invoices and subscriptions.

`collection_method`

:   Either `charge_automatically`, or `send_invoice`. When charging
    automatically, Stripe will attempt to pay invoices at the end of the
    subscription cycle or at invoice finalization using the default
    payment method attached to the subscription or customer. When
    sending an invoice, Stripe will email your customer an invoice with
    payment instructions. Defaults to `charge_automatically`.

`customer`

:   The customer for which this quote belongs to. A customer is required
    before finalizing the quote. Once specified, it cannot be changed.

`default_tax_rates`

:   The tax rates that will apply to any line item that does not have
    `tax_rates` set.

`description`

:   A description that will be displayed on the quote PDF.

`discounts`

:   The discounts applied to the quote. You can only set up to one
    discount.

`expires_at`

:   A future timestamp on which the quote will be canceled if in `open`
    or `draft` status. Measured in seconds since the Unix epoch.

`footer`

:   A footer that will be displayed on the quote PDF.

`header`

:   A header that will be displayed on the quote PDF.

`invoice_settings`

:   All invoices will be billed using the specified settings.

`line_items`

:   A list of line items the customer is being quoted for. Each line
    item includes information about the product, the quantity, and the
    resulting cost.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`on_behalf_of`

:   The account on behalf of which to charge.

`subscription_data`

:   When creating a subscription or subscription schedule, the specified
    configuration data will be used. There must be at least one line
    item with a recurring price for a subscription or subscription
    schedule to be created. A subscription schedule is created if
    `subscription_data[effective_date]` is present and in the future,
    otherwise a subscription is created.

`transfer_data`

:   The data with which to automatically create a Transfer for each of
    the invoices.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/update>

upfront\_line\_items
--------------------

        my $obj = $stripe->quotes( upfront_line_items => $args ) || die( $stripe->error );

Provided with a
[quote](https://metacpan.org/pod/Net::API::Stripe::Billing::Quote){.perl-module},
or a hash reference, this will issue a upfront\_line\_items api call.

A dictionary with a `data` property that contains an array of up to
`limit` upfront line items, starting after Line Item `starting_after`.
Each entry in the array is a separate Line Item object. If no more
upfront line items are available, the resulting array will be empty.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/quotes/line_items/upfront/list>

RADAR EARLY FRAUD WARNING
=========================

You can list or retrieve radar early fraud warning

list {#list-57}
----

        my $obj = $stripe->radar_early_fraud_warnings( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [radar early fraud
warning](https://metacpan.org/pod/Net::API::Stripe::Fraud){.perl-module}
object, this issue an api call to get the list of all radar early fraud
warning.

Possible parameters are:

`charge`

:   Only return early fraud warnings for the charge specified by this
    charge ID.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`payment_intent`

:   Only return early fraud warnings for charges that were created by
    the PaymentIntent specified by this PaymentIntent ID.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/early_fraud_warnings/list>

retrieve {#retrieve-58}
--------

        my $obj = $stripe->radar_early_fraud_warnings( retrieve => $args ) || die( $stripe->error );

Provided with a [radar early fraud
warning](https://metacpan.org/pod/Net::API::Stripe::Fraud){.perl-module}
object or a hash reference, this will retrieve a Stripe radar early
fraud warning and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Fraud){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/early_fraud_warnings/retrieve>

RADAR VALUE LIST
================

You can create, delete, list, retrieve or update radar value list

create {#create-46}
------

        my $obj = $stripe->radar_value_lists( create => {
            alias     => "custom_ip_blocklist",
            item_type => "ip_address",
            name      => "Custom IP Blocklist", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Fraud::ValueList](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}
object or a hash reference, this will create a Stripe radar value list
and return an
[Net::API::Stripe::Fraud::ValueList](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}
object.

Possible parameters are:

`alias`

:   **Required**. The name of the value list for use in rules.

`item_type`

:   Type of the items in the value list. One of `card_fingerprint`,
    `card_bin`, `email`, `ip_address`, `country`, `string`,
    `case_sensitive_string`, or `customer_id`. Use `string` if the item
    type is unknown or mixed.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   **Required**. The human-readable name of the value list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_lists/create>

delete
------

        my $obj = $stripe->radar_value_lists( delete => $args ) || die( $stripe->error );

Provided with a [radar value
list](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
radar value list. It returns the radar value list object that was
deleted with its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_lists/delete>

list {#list-58}
----

        my $obj = $stripe->radar_value_lists( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [radar value
list](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}
object, this issue an api call to get the list of all radar value list.

Possible parameters are:

`alias`

:   The alias used to reference the value list when writing rules.

`contains`

:   A value contained within a value list - returns all value lists
    containing this value.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_lists/list>

retrieve {#retrieve-59}
--------

        my $obj = $stripe->radar_value_lists( retrieve => $args ) || die( $stripe->error );

Provided with a [radar value
list](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}
object or a hash reference, this will retrieve a Stripe radar value list
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_lists/retrieve>

update
------

        my $obj = $stripe->radar_value_lists( update => {
            name => "Updated IP Block List", } ) || die( $stripe->error );

Provided with a [radar value
list](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}
object or a hash reference, this will update a Stripe radar value list
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList){.perl-module}

Possible parameters are:

`alias`

:   The name of the value list for use in rules.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`name`

:   The human-readable name of the value list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_lists/update>

RADAR VALUE LIST ITEM
=====================

You can create, delete, list or retrieve radar value list item

create {#create-47}
------

        my $obj = $stripe->radar_value_list_items( create => {
            value      => "1.2.3.4",
            value_list => "rsl_1Le9F32eZvKYlo2CWsVfMmYL", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Fraud::ValueList::Item](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList::Item){.perl-module}
object or a hash reference, this will create a Stripe radar value list
item and return an
[Net::API::Stripe::Fraud::ValueList::Item](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList::Item){.perl-module}
object.

Possible parameters are:

`value`

:   **Required**. The value of the item (whose type must match the type
    of the parent value list).

`value_list`

:   **Required**. The identifier of the value list which the created
    item will be added to.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_list_items/create>

delete
------

        my $obj = $stripe->radar_value_list_items( delete => $args ) || die( $stripe->error );

Provided with a [radar value list
item](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList::Item){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
radar value list item. It returns the radar value list item object that
was deleted with its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_list_items/delete>

list {#list-59}
----

        my $obj = $stripe->radar_value_list_items( list => {
            limit      => "3",
            value_list => "rsl_1Le9F32eZvKYlo2CWsVfMmYL", } ) || die( $stripe->error );

Provided with a [radar value list
item](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList::Item){.perl-module}
object, this issue an api call to get the list of all radar value list
item.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`value`

:   Return items belonging to the parent list whose value matches the
    specified value (using an \"is like\" match).

`value_list`

:   **Required**. Identifier for the parent value list this item belongs
    to.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_list_items/list>

retrieve {#retrieve-60}
--------

        my $obj = $stripe->radar_value_list_items( retrieve => $args ) || die( $stripe->error );

Provided with a [radar value list
item](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList::Item){.perl-module}
object or a hash reference, this will retrieve a Stripe radar value list
item and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Fraud::ValueList::Item){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/value_list_items/retrieve>

REFUND
======

You can cancel, create, list, retrieve or update refund

cancel
------

        my $obj = $stripe->refunds( cancel => $args ) || die( $stripe->error );

Provided with a
[refund](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns the refund object if the cancelation succeeded. This call will
return [an error](https://stripe.com/docs/api/errors){.perl-module} if
the refund is unable to be canceled.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/refunds/cancel>

create {#create-48}
------

        my $obj = $stripe->refunds( create => {
            charge => "ch_3Le9Ez2eZvKYlo2C1rWzICyb", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Refund](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module}
object or a hash reference, this will create a Stripe refund and return
an
[Net::API::Stripe::Refund](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module}
object.

Possible parameters are:

`amount`

:   A positive integer in *JPY* representing how much of this charge to
    refund. Can refund only up to the remaining, unrefunded amount of
    the charge.

`charge`

:   The identifier of the charge to refund.

`metadata`

:   A set of key-value pairs that you can attach to a `Refund` object.
    This can be useful for storing additional information about the
    refund in a structured format. You can unset individual keys if you
    POST an empty value for that key. You can clear all keys if you POST
    an empty value for `metadata`

`payment_intent`

:   ID of the PaymentIntent to refund.

`reason`

:   String indicating the reason for the refund. If set, possible values
    are `duplicate`, `fraudulent`, and `requested_by_customer`. If you
    believe the charge to be fraudulent, specifying `fraudulent` as the
    reason will add the associated card and email to your [block
    lists](https://stripe.com/docs/radar/lists){.perl-module}, and will
    also help us improve our fraud detection algorithms.

`refund_application_fee`

:   Boolean indicating whether the application fee should be refunded
    when refunding this charge. If a full charge refund is given, the
    full application fee will be refunded. Otherwise, the application
    fee will be refunded in an amount proportional to the amount of the
    charge refunded.

    An application fee can be refunded only by the application that
    created the charge.

`reverse_transfer`

:   Boolean indicating whether the transfer should be reversed when
    refunding this charge. The transfer will be reversed proportionally
    to the amount being refunded (either the entire or partial amount).

    A transfer can be reversed only by the application that created the
    charge.

More information from Stripe api documentation at
<https://stripe.com/docs/api/refunds/create>

list {#list-60}
----

        my $obj = $stripe->refunds( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[refund](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module}
object, this issue an api call to get the list of all refund.

Possible parameters are:

`charge`

:   Only return refunds for the charge specified by this charge ID.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`payment_intent`

:   Only return refunds for the PaymentIntent specified by this ID.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/refunds/list>

retrieve {#retrieve-61}
--------

        my $obj = $stripe->refunds( retrieve => $args ) || die( $stripe->error );

Provided with a
[refund](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module}
object or a hash reference, this will retrieve a Stripe refund and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/refunds/retrieve>

update
------

        my $obj = $stripe->refunds( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[refund](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module}
object or a hash reference, this will update a Stripe refund and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Refund){.perl-module}

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/refunds/update>

REPORTING REPORT RUN
====================

You can create, list or retrieve reporting report run

create {#create-49}
------

        my $obj = $stripe->reporting_report_runs( create => {
            parameters  =>
            {
                interval_end   => "1525132800",
                interval_start => "1522540800",
            }
            report_type => "balance.summary.1", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Reporting::ReportRun](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportRun){.perl-module}
object or a hash reference, this will create a Stripe reporting report
run and return an
[Net::API::Stripe::Reporting::ReportRun](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportRun){.perl-module}
object.

Possible parameters are:

`parameters`

:   Parameters specifying how the report should be run. Different Report
    Types have different required and optional parameters, listed in the
    [API Access to
    Reports](https://stripe.com/docs/reporting/statements/api){.perl-module}
    documentation.

`report_type`

:   **Required**. The ID of the [report
    type](https://stripe.com/docs/reporting/statements/api#report-types){.perl-module}
    to run, such as `"balance.summary.1"`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/reporting/report_run/create>

list {#list-61}
----

        my $obj = $stripe->reporting_report_runs( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [reporting report
run](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportRun){.perl-module}
object, this issue an api call to get the list of all reporting report
run.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/reporting/report_run/list>

retrieve {#retrieve-62}
--------

        my $obj = $stripe->reporting_report_runs( retrieve => $args ) || die( $stripe->error );

Provided with a [reporting report
run](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportRun){.perl-module}
object or a hash reference, this will retrieve a Stripe reporting report
run and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportRun){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/reporting/report_run/retrieve>

REPORTING REPORT TYPE
=====================

You can list or retrieve reporting report type

list {#list-62}
----

        my $obj = $stripe->reporting_report_types( list => $args ) || die( $stripe->error );

Provided with a [reporting report
type](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportType){.perl-module}
object, this issue an api call to get the list of all reporting report
type.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/reporting/report_type/list>

retrieve {#retrieve-63}
--------

        my $obj = $stripe->reporting_report_types( retrieve => $args ) || die( $stripe->error );

Provided with a [reporting report
type](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportType){.perl-module}
object or a hash reference, this will retrieve a Stripe reporting report
type and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Reporting::ReportType){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/reporting/report_type/retrieve>

REVIEW
======

You can approve, list or retrieve review

approve
-------

        my $obj = $stripe->reviews( approve => $args ) || die( $stripe->error );

Provided with a
[review](https://metacpan.org/pod/Net::API::Stripe::Fraud::Review){.perl-module},
or a hash reference, this will issue a approve api call.

Returns the approved `Review` object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/reviews/approve>

list {#list-63}
----

        my $obj = $stripe->reviews( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[review](https://metacpan.org/pod/Net::API::Stripe::Fraud::Review){.perl-module}
object, this issue an api call to get the list of all review.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/reviews/list>

retrieve {#retrieve-64}
--------

        my $obj = $stripe->reviews( retrieve => $args ) || die( $stripe->error );

Provided with a
[review](https://metacpan.org/pod/Net::API::Stripe::Fraud::Review){.perl-module}
object or a hash reference, this will retrieve a Stripe review and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Fraud::Review){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/radar/reviews/retrieve>

SCHEDULED QUERY RUN
===================

You can list or retrieve scheduled query run

list {#list-64}
----

        my $obj = $stripe->scheduled_query_runs( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [scheduled query
run](https://metacpan.org/pod/Net::API::Stripe::Sigma::ScheduledQueryRun){.perl-module}
object, this issue an api call to get the list of all scheduled query
run.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/sigma/scheduled_queries/list>

retrieve {#retrieve-65}
--------

        my $obj = $stripe->scheduled_query_runs( retrieve => $args ) || die( $stripe->error );

Provided with a [scheduled query
run](https://metacpan.org/pod/Net::API::Stripe::Sigma::ScheduledQueryRun){.perl-module}
object or a hash reference, this will retrieve a Stripe scheduled query
run and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Sigma::ScheduledQueryRun){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/sigma/scheduled_queries/retrieve>

SETUP ATTEMPT
=============

You can list setup attempt

list {#list-65}
----

        my $obj = $stripe->setup_attempts( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [setup
attempt](https://metacpan.org/pod/Net::API::Stripe::SetupAttempt){.perl-module}
object, this issue an api call to get the list of all setup attempt.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`setup_intent`

:   **Required**. Only return SetupAttempts created by the SetupIntent
    specified by this ID.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/setup_attempts/list>

SETUP INTENT
============

You can cancel, confirm, create, list, retrieve, update, verify or
verify\_microdeposits setup intent

cancel
------

        my $obj = $stripe->setup_intents( cancel => $args ) || die( $stripe->error );

Provided with a [setup
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns a SetupIntent object if the cancellation succeeded. Returns an
error if the SetupIntent has already been canceled or is not in a
cancelable state.

Possible parameters are:

`cancellation_reason`

:   Reason for canceling this SetupIntent. Possible values are
    `abandoned`, `requested_by_customer`, or `duplicate`

More information from Stripe api documentation at
<https://stripe.com/docs/api/setup_intents/cancel>

confirm
-------

        my $obj = $stripe->setup_intents( confirm => {
            payment_method => "pm_card_visa", } ) || die( $stripe->error );

Provided with a [setup
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module},
or a hash reference, this will issue a confirm api call.

Returns the resulting SetupIntent after all possible transitions are
applied.

Possible parameters are:

`mandate_data`

:   This hash contains details about the Mandate to create

`payment_method`

:   ID of the payment method (a PaymentMethod, Card, or saved Source
    object) to attach to this SetupIntent.

`payment_method_data`

:   When included, this hash creates a PaymentMethod that is set as the
    [`payment_method`](https://stripe.com/docs/api/setup_intents/object#setup_intent_object-payment_method){.perl-module}
    value in the SetupIntent.

`payment_method_options`

:   Payment-method-specific configuration for this SetupIntent.

`return_url`

:   The URL to redirect your customer back to after they authenticate on
    the payment method\'s app or site. If you\'d prefer to redirect to a
    mobile application, you can alternatively supply an application URI
    scheme. This parameter is only used for cards and other
    redirect-based payment methods.

More information from Stripe api documentation at
<https://stripe.com/docs/api/setup_intents/confirm>

create {#create-50}
------

        my $obj = $stripe->setup_intents( create => {
            payment_method_types => [qw( card )], } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Payment::Intent::Setup](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}
object or a hash reference, this will create a Stripe setup intent and
return an
[Net::API::Stripe::Payment::Intent::Setup](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}
object.

Possible parameters are:

`attach_to_self`

:   If present, the SetupIntent\'s payment method will be attached to
    the in-context Stripe Account.

    It can only be used for this Stripe Account's own money movement
    flows like InboundTransfer and OutboundTransfers. It cannot be set
    to true when setting up a PaymentMethod for a Customer, and defaults
    to false when attaching a PaymentMethod to a Customer.

`confirm`

:   Set to `true` to attempt to confirm this SetupIntent immediately.
    This parameter defaults to `false`. If the payment method attached
    is a card, a return\_url may be provided in case additional
    authentication is required.

`customer`

:   ID of the Customer this SetupIntent belongs to, if one exists.

    If present, the SetupIntent\'s payment method will be attached to
    the Customer on successful setup. Payment methods attached to other
    Customers cannot be used with this SetupIntent.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`flow_directions`

:   Indicates the directions of money movement for which this payment
    method is intended to be used.

    Include `inbound` if you intend to use the payment method as the
    origin to pull funds from. Include `outbound` if you intend to use
    the payment method as the destination to send funds to. You can
    include both if you intend to use the payment method for both
    purposes.

`mandate_data`

:   This hash contains details about the Mandate to create. This
    parameter can only be used with
    [`confirm=true`](https://stripe.com/docs/api/setup_intents/create#create_setup_intent-confirm){.perl-module}.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`on_behalf_of`

:   The Stripe account ID for which this SetupIntent is created.

`payment_method`

:   ID of the payment method (a PaymentMethod, Card, or saved Source
    object) to attach to this SetupIntent.

`payment_method_data`

:   When included, this hash creates a PaymentMethod that is set as the
    [`payment_method`](https://stripe.com/docs/api/setup_intents/object#setup_intent_object-payment_method){.perl-module}
    value in the SetupIntent.

`payment_method_options`

:   Payment-method-specific configuration for this SetupIntent.

`payment_method_types`

:   The list of [payment method
    types](https://stripe.com/docs/payments/payment-methods/overview){.perl-module}
    that this SetupIntent is allowed to set up. If this is not provided,
    defaults to `[“card”]`. Valid payment method types include:
    `acss_debit`, `au_becs_debit`, `bacs_debit`, `bancontact`, `blik`,
    `boleto`, `card`, `card_present`, `ideal`, `link`, `sepa_debit`,
    `sofort`, and `us_bank_account`.

`return_url`

:   The URL to redirect your customer back to after they authenticate or
    cancel their payment on the payment method\'s app or site. If you\'d
    prefer to redirect to a mobile application, you can alternatively
    supply an application URI scheme. This parameter can only be used
    with
    [`confirm=true`](https://stripe.com/docs/api/setup_intents/create#create_setup_intent-confirm){.perl-module}.

`single_use`

:   If this hash is populated, this SetupIntent will generate a
    single\_use Mandate on success.

`usage`

:   Indicates how the payment method is intended to be used in the
    future. If not provided, this value defaults to `off_session`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/setup_intents/create>

list {#list-66}
----

        my $obj = $stripe->setup_intents( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [setup
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}
object, this issue an api call to get the list of all setup intent.

Possible parameters are:

`attach_to_self`

:   If present, the SetupIntent\'s payment method will be attached to
    the in-context Stripe Account.

    It can only be used for this Stripe Account's own money movement
    flows like InboundTransfer and OutboundTransfers. It cannot be set
    to true when setting up a PaymentMethod for a Customer, and defaults
    to false when attaching a PaymentMethod to a Customer.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`customer`

:   Only return SetupIntents for the customer specified by this customer
    ID.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`payment_method`

:   Only return SetupIntents associated with the specified payment
    method.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/setup_intents/list>

retrieve {#retrieve-66}
--------

        my $obj = $stripe->setup_intents( retrieve => $args ) || die( $stripe->error );

Provided with a [setup
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}
object or a hash reference, this will retrieve a Stripe setup intent and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}

Possible parameters are:

`client_secret`

:   **Required**. The client secret of the SetupIntent. Required if a
    publishable key is used to retrieve the SetupIntent.

More information from Stripe api documentation at
<https://stripe.com/docs/api/setup_intents/retrieve>

update
------

        my $obj = $stripe->setup_intents( update => {
            metadata =>
            {
                user_id => "3435453",
            } } ) || die( $stripe->error );

Provided with a [setup
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}
object or a hash reference, this will update a Stripe setup intent and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}

Possible parameters are:

`attach_to_self`

:   If present, the SetupIntent\'s payment method will be attached to
    the in-context Stripe Account.

    It can only be used for this Stripe Account's own money movement
    flows like InboundTransfer and OutboundTransfers. It cannot be set
    to true when setting up a PaymentMethod for a Customer, and defaults
    to false when attaching a PaymentMethod to a Customer.

`customer`

:   ID of the Customer this SetupIntent belongs to, if one exists.

    If present, the SetupIntent\'s payment method will be attached to
    the Customer on successful setup. Payment methods attached to other
    Customers cannot be used with this SetupIntent.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`flow_directions`

:   Indicates the directions of money movement for which this payment
    method is intended to be used.

    Include `inbound` if you intend to use the payment method as the
    origin to pull funds from. Include `outbound` if you intend to use
    the payment method as the destination to send funds to. You can
    include both if you intend to use the payment method for both
    purposes.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`payment_method`

:   ID of the payment method (a PaymentMethod, Card, or saved Source
    object) to attach to this SetupIntent.

`payment_method_data`

:   When included, this hash creates a PaymentMethod that is set as the
    [`payment_method`](https://stripe.com/docs/api/setup_intents/object#setup_intent_object-payment_method){.perl-module}
    value in the SetupIntent.

`payment_method_options`

:   Payment-method-specific configuration for this SetupIntent.

`payment_method_types`

:   The list of payment method types (e.g. card) that this SetupIntent
    is allowed to set up. If this is not provided, defaults to
    \[\"card\"\].

More information from Stripe api documentation at
<https://stripe.com/docs/api/setup_intents/update>

verify\_microdeposits
---------------------

        my $obj = $stripe->setup_intents( verify_microdeposits => $args ) || die( $stripe->error );

Provided with a [setup
intent](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module},
or a hash reference, this will issue a verify\_microdeposits api call.

Returns a SetupIntent object.

Possible parameters are:

`amounts`

:   Two positive integers, in *cents*, equal to the values of the
    microdeposits sent to the bank account.

`client_secret`

:   **Required**. The client secret of the SetupIntent.

`descriptor_code`

:   A six-character code starting with SM present in the microdeposit
    sent to the bank account.

More information from Stripe api documentation at
<https://stripe.com/docs/api/setup_intents/verify_microdeposits>

SETUP INTENTS
=============

You can **create**, **retrieve**, **update**, **confirm** or **cancel**
and **list** setup inetnts

cancel
------

confirm
-------

Confirm that your customer intends to set up the current or provided
payment method. For example, you would confirm a SetupIntent when a
customer hits the âSaveâ button on a payment method management page
on your website.

If the selected payment method does not require any additional steps
from the customer, the SetupIntent will transition to the `succeeded`
status.

Otherwise, it will transition to the `requires_action` status and
suggest additional actions via `next_action`. If setup fails, the
SetupIntent will transition to the `requires_payment_method` status.

>         =item I<payment_method> optional
>
>         ID of the payment method (a PaymentMethod, Card, or saved Source object) to attach to this SetupIntent.
>
>         =item I<mandate_data> optional
>
>         This hash contains details about the Mandate to create
>
>         =over 8
>
>             =item I<customer_acceptance> required
>
>             =over 12
>
>                 =item type
>
>                 The type of customer acceptance information included with the Mandate. One of C<online> or C<offline>.
>
>                 =item accepted_at optional
>
>                 The time at which the customer accepted the Mandate.
>
>                 =item offline optional hash
>
>                 If this is a Mandate accepted offline, this hash contains details about the offline acceptance.
>
>                 =item online optional hash
>
>                 If this is a Mandate accepted online, this hash contains details about the online acceptance.
>
>                 =over 16
>
>                     =item ip_address required
>
>                     The IP address from which the Mandate was accepted by the customer.
>
>                     =item user_agent required
>
>                     The user agent of the browser from which the Mandate was accepted by the customer.
>
>                 =back
>
>             =back
>
>         =back
>
>         =item I<payment_method_options> optional hash
>
>         Payment-method-specific configuration for this SetupIntent.
>
>         =over 8
>
>             =item acss_debit
>
>             If this is a C<acss_debit> SetupIntent, this sub-hash contains details about the ACSS Debit payment method options.
>
>             =over 12
>
>                 =item currency
>
>                 Three-letter ISO currency code, in lowercase. Must be a supported currency.
>
>                 Possible enum values
>
>                 =over 16
>
>                     =item cad
>
>                     Canadian dollars
>
>                     =item usd
>
>                     US dollars
>
>                 =back
>
>                 =item mandate_options
>
>                 Additional fields for Mandate creation
>
>                 =over 16
>
>                     =item custom_mandate_url
>
>                     A URL for custom mandate text to render during confirmation step. The URL will be rendered with additional GET parameters C<payment_intent> and C<payment_intent_client_secret> when confirming a Payment Intent, or C<setup_intent> and C<setup_intent_client_secret> when confirming a Setup Intent.
>
>                     =item default_for
>
>                     List of Stripe products where this mandate can be selected automatically.
>
>                     Possible enum values
>
>                     =over 20
>
>                         =item invoice
>
>                         Enables payments for Stripe Invoices. âsubscriptionâ must also be provided.
>
>                         =item subscription
>
>                         Enables payments for Stripe Subscriptions. âinvoiceâ must also be provided.
>
>                     =back
>
>                     =item interval_description
>
>                     Description of the mandate interval. Only required if âpayment_scheduleâ parameter is âintervalâ or âcombinedâ.
>
>                     =item payment_schedule
>
>                     Payment schedule for the mandate.
>
>                     Possible enum values
>
>                     =over 20
>
>                         =item interval
>
>                         Payments are initiated at a regular pre-defined interval
>
>                         =item sporadic
>
>                         Payments are initiated sporadically
>
>                         =item combined
>
>                         Payments can be initiated at a pre-defined interval or sporadically
>
>                     =back
>
>                     =item transaction_type
>
>                     Transaction type of the mandate.
>
>                     Possible enum values
>
>                     =over 20
>
>                         =item personal
>
>                         Transactions are made for personal reasons
>
>                         =item business
>
>                         Transactions are made for business reasons
>
>                     =back
>
>                 =back
>
>                 =item verification_method
>
>                 Verification method for the intent
>
>                 Possible enum values
>
>                 =over 16
>
>                     =item automatic
>
>                     Instant verification with fallback to microdeposits.
>
>                     =item instant
>
>                     Instant verification.
>
>                     =item microdeposits
>
>                     Verification using microdeposits.
>
>                 =back
>
>                 =item card
>
>                 Configuration for any card setup attempted on this SetupIntent.
>
>                 =over 16
>
>                     =item request_three_d_secure
>
>                     We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and other requirements. However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Permitted values include: automatic or any. If not provided, defaults to automatic. Read our guide on manually requesting 3D Secure for more information on how this configuration interacts with Radar and our SCA Engine.
>
>                 =back
>
>                 =item sepa_debit
>
>                 If this is a C<sepa_debit> SetupIntent, this sub-hash contains details about the SEPA Debit payment method options.
>
>                 =over 16
>
>                     =item mandate_options
>
>                     Additional fields for Mandate creation
>
>                 =back
>
>             =back
>
>             =item I<return_url> optional
>
>             The URL to redirect your customer back to after they authenticate on the payment methodâs app or site. If youâd prefer to redirect to a mobile application, you can alternatively supply an application URI scheme. This parameter is only used for cards and other redirect-based payment methods.
>
>         =back

create {#create-51}
------

Provided with
[Net::API::Stripe::Payment::Intent::Setup](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}
object or an hash reference of parameters, this will create a Stripe
setup intent and return a
[Net::API::Stripe::Payment::Intent::Setup](https://metacpan.org/pod/Net::API::Stripe::Payment::Intent::Setup){.perl-module}
object.

Possible parameters are:

*confirm* optional.

:   Set to true to attempt to confirm this SetupIntent immediately. This
    parameter defaults to false. If the payment method attached is a
    card, a return\_url may be provided in case additional
    authentication is required.

*customer* optional.

:   ID of the Customer this SetupIntent belongs to, if one exists.

    If present, the SetupIntentâs payment method will be attached to
    the Customer on successful setup. Payment methods attached to other
    Customers cannot be used with this SetupIntent.

*description* optional

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

*metadata* hash

:   Set of key-value pairs that you can attach to an object. This can be
    useful for storing additional information about the object in a
    structured format. Individual keys can be unset by posting an empty
    value to them. All keys can be unset by posting an empty value to
    metadata.

*payment\_method* optional

:   ID of the payment method (a PaymentMethod, Card, or saved Source
    object) to attach to this SetupIntent.

*payment\_method\_types* optional

:   The list of payment method types that this SetupIntent is allowed to
    set up. If this is not provided, defaults to `[âcardâ]`. Valid
    payment method types include: `au_becs_debit`, `bancontact`, `card`,
    `card_present`, `ideal`, `sepa_debit`, and `sofort`.

*usage* optional

:   Indicates how the payment method is intended to be used in the
    future. If not provided, this value defaults to `off_session`.

    Possible enum values

    on\_session

    :   Use `on_session` if you intend to only reuse the payment method
        when the customer is in your checkout flow.

    off\_session

    :   Use `off_session` if your customer may or may not be in your
        checkout flow.

*mandate\_data* optional hash

:   This hash contains details about the Mandate to create. This
    parameter can only be used with `confirm=true`.

    customer\_acceptance required

    :   This hash contains details about the customer acceptance of the
        Mandate.

        type

        :   The type of customer acceptance information included with
            the Mandate. One of online or `offline`.

        accepted\_at

        :   The time at which the customer accepted the Mandate.

        offline

        :   If this is a Mandate accepted offline, this hash contains
            details about the offline acceptance.

        online

        :   If this is a Mandate accepted online, this hash contains
            details about the online acceptance.

            ip\_address required

            :   The IP address from which the Mandate was accepted by
                the customer.

            user\_agent required

            :   The user agent of the browser from which the Mandate was
                accepted by the customer.

*on\_behalf\_of* optional

:   The Stripe account ID for which this SetupIntent is created.

*payment\_method\_options* optional hash

:   Payment-method-specific configuration for this SetupIntent.

    acss\_debit

    :   If this is a `acss_debit` SetupIntent, this sub-hash contains
        details about the ACSS Debit payment method options.

        currency

        :   Three-letter ISO currency code, in lowercase. Must be a
            supported currency.

            Possible enum values

            cad

            :   Canadian dollars

            usd

            :   US dollars

        mandate\_options

        :   Additional fields for Mandate creation

            custom\_mandate\_url

            :   A URL for custom mandate text to render during
                confirmation step. The URL will be rendered with
                additional GET parameters `payment_intent` and
                `payment_intent_client_secret` when confirming a Payment
                Intent, or `setup_intent` and
                `setup_intent_client_secret` when confirming a Setup
                Intent.

            default\_for

            :   List of Stripe products where this mandate can be
                selected automatically.

                Possible enum values

                invoice

                :   Enables payments for Stripe Invoices.
                    âsubscriptionâ must also be provided.

                subscription

                :   Enables payments for Stripe Subscriptions.
                    âinvoiceâ must also be provided.

            interval\_description

            :   Description of the mandate interval. Only required if
                âpayment\_scheduleâ parameter is âintervalâ or
                âcombinedâ.

            payment\_schedule

            :   Payment schedule for the mandate.

                Possible enum values

                interval

                :   Payments are initiated at a regular pre-defined
                    interval

                sporadic

                :   Payments are initiated sporadically

                combined

                :   Payments can be initiated at a pre-defined interval
                    or sporadically

            transaction\_type

            :   Transaction type of the mandate.

                Possible enum values

                personal

                :   Transactions are made for personal reasons

                business

                :   Transactions are made for business reasons

        verification\_method

        :   Verification method for the intent

            Possible enum values

            automatic

            :   Instant verification with fallback to microdeposits.

            instant

            :   Instant verification.

            microdeposits

            :   Verification using microdeposits.

        card

        :   Configuration for any card setup attempted on this
            SetupIntent.

            request\_three\_d\_secure

            :   We strongly recommend that you rely on our SCA Engine to
                automatically prompt your customers for authentication
                based on risk level and other requirements. However, if
                you wish to request 3D Secure based on logic from your
                own fraud engine, provide this option. Permitted values
                include: automatic or any. If not provided, defaults to
                automatic. Read our guide on manually requesting 3D
                Secure for more information on how this configuration
                interacts with Radar and our SCA Engine.

        sepa\_debit

        :   If this is a `sepa_debit` SetupIntent, this sub-hash
            contains details about the SEPA Debit payment method
            options.

            mandate\_options

            :   Additional fields for Mandate creation

    *return\_url* optional

    :   The URL to redirect your customer back to after they
        authenticate or cancel their payment on the payment methodâs
        app or site. If youâd prefer to redirect to a mobile
        application, you can alternatively supply an application URI
        scheme. This parameter can only be used with `confirm=true`.

    *single\_use* optional hash

    :   If this hash is populated, this SetupIntent will generate a
        single\_use Mandate on success.

        amount

        :   Amount the customer is granting permission to collect later.
            A positive integer representing how much to charge in the
            smallest currency unit (e.g., 100 cents to charge \$1.00 or
            100 to charge Â¥100, a zero-decimal currency). The minimum
            amount is \$0.50 US or equivalent in charge currency. The
            amount value supports up to eight digits (e.g., a value of
            99999999 for a USD charge of \$999,999.99).

        currency

        :   Three-letter ISO currency code, in lowercase. Must be a
            supported currency.

list {#list-67}
----

retrieve {#retrieve-67}
--------

Retrieves the details of a SetupIntent that has previously been created.

Client-side retrieval using a publishable key is allowed when the
client\_secret is provided in the query string.

When retrieved with a publishable key, only a subset of properties will
be returned. Please refer to the SetupIntent object reference for more
details.

*client\_secret* required if using publishable key

:   The client secret of the SetupIntent. Required if a publishable key
    is used to retrieve the SetupIntent.

update
------

*customer* optional

:   ID of the Customer this SetupIntent belongs to, if one exists.

    If present, the SetupIntentâs payment method will be attached to
    the Customer on successful setup. Payment methods attached to other
    Customers cannot be used with this SetupIntent.

*description* optional

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

*metadata* optional hash

:   Set of `key-value pairs` that you can attach to an object. This can
    be useful for storing additional information about the object in a
    structured format. Individual keys can be unset by posting an empty
    value to them. All keys can be unset by posting an empty value to
    `metadata`.

*payment\_method* optional

:   ID of the payment method (a PaymentMethod, Card, or saved Source
    object) to attach to this SetupIntent.

*payment\_method\_types* optional

:   The list of payment method types (e.g. card) that this SetupIntent
    is allowed to set up. If this is not provided, defaults to
    \[âcardâ\].

*payment\_method\_options* optional hash

:   Payment-method-specific configuration for this SetupIntent.

    acss\_debit

    :   If this is a `acss_debit` SetupIntent, this sub-hash contains
        details about the ACSS Debit payment method options.

        currency

        :   Three-letter ISO currency code, in lowercase. Must be a
            supported currency.

            Possible enum values

            cad

            :   Canadian dollars

            usd

            :   US dollars

        mandate\_options

        :   Additional fields for Mandate creation

            custom\_mandate\_url

            :   A URL for custom mandate text to render during
                confirmation step. The URL will be rendered with
                additional GET parameters `payment_intent` and
                `payment_intent_client_secret` when confirming a Payment
                Intent, or `setup_intent` and
                `setup_intent_client_secret` when confirming a Setup
                Intent.

            default\_for

            :   List of Stripe products where this mandate can be
                selected automatically.

                Possible enum values

                invoice

                :   Enables payments for Stripe Invoices.
                    âsubscriptionâ must also be provided.

                subscription

                :   Enables payments for Stripe Subscriptions.
                    âinvoiceâ must also be provided.

            interval\_description

            :   Description of the mandate interval. Only required if
                âpayment\_scheduleâ parameter is âintervalâ or
                âcombinedâ.

            payment\_schedule

            :   Payment schedule for the mandate.

                Possible enum values

                interval

                :   Payments are initiated at a regular pre-defined
                    interval

                sporadic

                :   Payments are initiated sporadically

                combined

                :   Payments can be initiated at a pre-defined interval
                    or sporadically

            transaction\_type

            :   Transaction type of the mandate.

                Possible enum values

                personal

                :   Transactions are made for personal reasons

                business

                :   Transactions are made for business reasons

        verification\_method

        :   Verification method for the intent

            Possible enum values

            automatic

            :   Instant verification with fallback to microdeposits.

            instant

            :   Instant verification.

            microdeposits

            :   Verification using microdeposits.

        card

        :   Configuration for any card setup attempted on this
            SetupIntent.

            request\_three\_d\_secure

            :   We strongly recommend that you rely on our SCA Engine to
                automatically prompt your customers for authentication
                based on risk level and other requirements. However, if
                you wish to request 3D Secure based on logic from your
                own fraud engine, provide this option. Permitted values
                include: automatic or any. If not provided, defaults to
                automatic. Read our guide on manually requesting 3D
                Secure for more information on how this configuration
                interacts with Radar and our SCA Engine.

        sepa\_debit

        :   If this is a `sepa_debit` SetupIntent, this sub-hash
            contains details about the SEPA Debit payment method
            options.

            mandate\_options

            :   Additional fields for Mandate creation

SHIPPING RATE
=============

You can create, list, retrieve or update shipping rate

create {#create-52}
------

        my $obj = $stripe->shipping_rates( create => {
            display_name => "Ground shipping",
            fixed_amount =>
            {
                amount   => "500",
                currency => "usd",
            }
            type         => "fixed_amount", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Shipping::Rate](https://metacpan.org/pod/Net::API::Stripe::Shipping::Rate){.perl-module}
object or a hash reference, this will create a Stripe shipping rate and
return an
[Net::API::Stripe::Shipping::Rate](https://metacpan.org/pod/Net::API::Stripe::Shipping::Rate){.perl-module}
object.

Possible parameters are:

`delivery_estimate`

:   The estimated range for how long shipping will take, meant to be
    displayable to the customer. This will appear on CheckoutSessions.

`display_name`

:   **Required**. The name of the shipping rate, meant to be displayable
    to the customer. This will appear on CheckoutSessions.

`fixed_amount`

:   Describes a fixed amount to charge for shipping. Must be present if
    type is `fixed_amount`.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`tax_behavior`

:   Specifies whether the rate is considered inclusive of taxes or
    exclusive of taxes. One of `inclusive`, `exclusive`, or
    `unspecified`.

`tax_code`

:   A [tax
    code](https://stripe.com/docs/tax/tax-categories){.perl-module} ID.
    The Shipping tax code is `txcd_92010001`.

`type`

:   **required** The type of calculation to use on the shipping rate.
    Can only be `fixed_amount` for now.

More information from Stripe api documentation at
<https://stripe.com/docs/api/shipping_rates/create>

list {#list-68}
----

        my $obj = $stripe->shipping_rates( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [shipping
rate](https://metacpan.org/pod/Net::API::Stripe::Shipping::Rate){.perl-module}
object, this issue an api call to get the list of all shipping rate.

Possible parameters are:

`active`

:   Only return shipping rates that are active or inactive.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`currency`

:   Only return shipping rates for the given currency.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/shipping_rates/list>

retrieve {#retrieve-68}
--------

        my $obj = $stripe->shipping_rates( retrieve => $args ) || die( $stripe->error );

Provided with a [shipping
rate](https://metacpan.org/pod/Net::API::Stripe::Shipping::Rate){.perl-module}
object or a hash reference, this will retrieve a Stripe shipping rate
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Shipping::Rate){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/shipping_rates/retrieve>

update
------

        my $obj = $stripe->shipping_rates( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [shipping
rate](https://metacpan.org/pod/Net::API::Stripe::Shipping::Rate){.perl-module}
object or a hash reference, this will update a Stripe shipping rate and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Shipping::Rate){.perl-module}

Possible parameters are:

`active`

:   Whether the shipping rate can be used for new purchases. Defaults to
    `true`.

`fixed_amount`

:   Describes a fixed amount to charge for shipping. Must be present if
    type is `fixed_amount`.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`tax_behavior`

:   Specifies whether the rate is considered inclusive of taxes or
    exclusive of taxes. One of `inclusive`, `exclusive`, or
    `unspecified`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/shipping_rates/update>

SOURCE
======

You can attach, create, detach, retrieve or update source

attach
------

        my $obj = $stripe->sources( attach => $args ) || die( $stripe->error );

Provided with a
[source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module},
or a hash reference, this will issue a attach api call.

Returns the attached Source object.

Possible parameters are:

`source`

:   **Required**. The identifier of the source to be attached.

More information from Stripe api documentation at
<https://stripe.com/docs/api/sources/attach>

create {#create-53}
------

        my $obj = $stripe->sources( create => $args ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object or a hash reference, this will create a Stripe source and return
an
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object.

Possible parameters are:

`amount`

:   Amount associated with the source. This is the amount for which the
    source will be chargeable once ready. Required for `single_use`
    sources. Not supported for `receiver` type sources, where charge
    amount may not be specified until funds land.

`currency`

:   Three-letter [ISO code for the
    currency](https://stripe.com/docs/currencies){.perl-module}
    associated with the source. This is the currency for which the
    source will be chargeable once ready.

`flow`

:   The authentication `flow` of the source to create. `flow` is one of
    `redirect`, `receiver`, `code_verification`, `none`. It is generally
    inferred unless a type supports multiple flows.

`mandate`

:   Information about a mandate possibility attached to a source object
    (generally for bank debits) as well as its acceptance status.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`owner`

:   Information about the owner of the payment instrument that may be
    used or required by particular source types.

`receiver`

:   Optional parameters for the receiver flow. Can be set only if the
    source is a receiver (`flow` is `receiver`).

`redirect`

:   Parameters required for the redirect flow. Required if the source is
    authenticated by a redirect (`flow` is `redirect`).

`source_order`

:   Information about the items and shipping associated with the source.
    Required for transactional credit (for example Klarna) sources
    before you can charge it.

`statement_descriptor`

:   An arbitrary string to be displayed on your customer\'s statement.
    As an example, if your website is `RunClub` and the item you\'re
    charging for is a race ticket, you may want to specify a
    `statement_descriptor` of `RunClub 5K race ticket.` While many
    payment types will display this information, some may not display it
    at all.

`token`

:   An optional token used to create the source. When passed, token
    properties will override source parameters.

`type`

:   **Required**. The `type` of the source to create. Required unless
    `customer` and `original_source` are specified (see the [Cloning
    card
    Sources](https://stripe.com/docs/sources/connect#cloning-card-sources){.perl-module}
    guide)

`usage`

:   Either `reusable` or `single_use`. Whether this source should be
    reusable or not. Some source types may or may not be reusable by
    construction, while others may leave the option at creation. If an
    incompatible value is passed, an error will be returned.

More information from Stripe api documentation at
<https://stripe.com/docs/api/sources/create>

detach
------

        my $obj = $stripe->sources( detach => $args ) || die( $stripe->error );

Provided with a
[source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module},
or a hash reference, this will issue a detach api call.

Returns the detached Source object.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/sources/detach>

retrieve {#retrieve-69}
--------

        my $obj = $stripe->sources( retrieve => $args ) || die( $stripe->error );

Provided with a
[source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object or a hash reference, this will retrieve a Stripe source and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}

Possible parameters are:

`client_secret`

:   The client secret of the source. Required if a publishable key is
    used to retrieve the source.

More information from Stripe api documentation at
<https://stripe.com/docs/api/sources/retrieve>

update
------

        my $obj = $stripe->sources( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object or a hash reference, this will update a Stripe source and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}

Possible parameters are:

`amount`

:   Amount associated with the source.

`mandate`

:   Information about a mandate possibility attached to a source object
    (generally for bank debits) as well as its acceptance status.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`owner`

:   Information about the owner of the payment instrument that may be
    used or required by particular source types.

`source_order`

:   Information about the items and shipping associated with the source.
    Required for transactional credit (for example Klarna) sources
    before you can charge it.

More information from Stripe api documentation at
<https://stripe.com/docs/api/sources/update>

SOURCES
=======

You can **create**, **retrieve**, **update**, **detach** or **attach**
sources

attach
------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object or an hash reference of parameters, this will attach a Stripe
source to a customer and return a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object.

Possible parameters are:

*id* A Stripe customer id

:   

*source* A Stripe source id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/sources/attach>

create {#create-54}
------

Provided with
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object or an hash reference of parameters, this will create a Stripe
source and return a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object.

Possible parameters are:

*type* String. This is required.

:   

*amount* Integer

:   

*currency* A 3-letter iso 4217 code such as `jpy` or `eur`

:   

*flow* String. One of redirect, receiver, code\_verification, none

:   

*mandate* An hash reference with the following properties: acceptance amount currency interval notification\_method

:   

*metadata* An arbitrary hash reference

:   

*owner* An hash reference with the following properties: address email name phone

:   

*receiver* An hash reference with the following property: refund\_attributes\_method

:   

*redirect* An hash reference with the following property: return\_url

:   

*source\_order* An hash reference with the following properties: items shipping

:   

*statement\_descriptor* Text

:   

*token* String

:   

*usage* String. One of reusable or single\_use

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/sources/create>

detach
------

Provided with a
[Net::API::Stripe::Customer](https://metacpan.org/pod/Net::API::Stripe::Customer){.perl-module}
object or a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object or an hash reference of parameters, this will detach a Stripe
source from the customer and return a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object.

Possible parameters are:

*id* A Stripe customer id

:   

*source* A Stripe source id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/sources/detach>

retrieve {#retrieve-70}
--------

Provided with
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object or an hash reference of parameters, this will retrieve a Stripe
source and return a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object.

Possible parameters are:

*id* A Stripe source id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/sources/retrieve>

update
------

Provided with
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object or an hash reference of parameters, this will update a Stripe
source and return a
[Net::API::Stripe::Payment::Source](https://metacpan.org/pod/Net::API::Stripe::Payment::Source){.perl-module}
object.

Possible parameters are:

*id* A Stripe source id

:   

*amount* Integer

:   

*mandate* An hash reference with the following properties: acceptance amount currency interval notification\_method

:   

*metadata* An arbitrary hash reference

:   

*owner* An hash reference with the following properties: address email name phone

:   

*source\_order* An hash reference with the following properties: items shipping

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/sources/update>

SUBSCRIPTION
============

You can cancel, create, delete, delete\_discount, list, retrieve, search
or update subscription

create {#create-55}
------

        my $obj = $stripe->subscriptions( create => {
            customer => "cus_AJ78ZaALpqgiuZ",
            items    => [,
                price => "price_1Le1oa2eZvKYlo2CuD7mwpZu",
            ], } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or a hash reference, this will create a Stripe subscription and
return an
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object.

Possible parameters are:

`add_invoice_items`

:   A list of prices and quantities that will generate invoice items
    appended to the next invoice for this subscription. You may pass up
    to 20 items.

`application_fee_percent`

:   A non-negative decimal between 0 and 100, with at most two decimal
    places. This represents the percentage of the subscription invoice
    subtotal that will be transferred to the application owner\'s Stripe
    account. The request must be made by a platform account on a
    connected account in order to set an application fee percentage. For
    more information, see the application fees
    [documentation](https://stripe.com/docs/connect/subscriptions#collecting-fees-on-subscriptions){.perl-module}.

`automatic_tax`

:   Automatic tax settings for this subscription. We recommend you only
    include this parameter when the existing value is being changed.

`backdate_start_date`

:   For new subscriptions, a past timestamp to backdate the
    subscription\'s start date to. If set, the first invoice will
    contain a proration for the timespan between the start date and the
    current time. Can be combined with trials and the billing cycle
    anchor.

`billing_cycle_anchor`

:   A future timestamp to anchor the subscription\'s [billing
    cycle](https://stripe.com/docs/subscriptions/billing-cycle){.perl-module}.
    This is used to determine the date of the first full invoice, and,
    for plans with `month` or `year` intervals, the day of the month for
    subsequent invoices. The timestamp is in UTC format.

`billing_thresholds`

:   Define thresholds at which an invoice will be sent, and the
    subscription advanced to a new billing period. Pass an empty string
    to remove previously-defined thresholds.

`cancel_at`

:   A timestamp at which the subscription should cancel. If set to a
    date before the current period ends, this will cause a proration if
    prorations have been enabled using `proration_behavior`. If set
    during a future period, this will always cause a proration for that
    period.

`cancel_at_period_end`

:   Boolean indicating whether this subscription should cancel at the
    end of the current period.

`collection_method`

:   Either `charge_automatically`, or `send_invoice`. When charging
    automatically, Stripe will attempt to pay this subscription at the
    end of the cycle using the default source attached to the customer.
    When sending an invoice, Stripe will email your customer an invoice
    with payment instructions. Defaults to `charge_automatically`.

`coupon`

:   The ID of the coupon to apply to this subscription. A coupon applied
    to a subscription will only affect invoices created for that
    particular subscription.

`currency`

:   Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`customer`

:   **Required**. The identifier of the customer to subscribe.

`days_until_due`

:   Number of days a customer has to pay invoices generated by this
    subscription. Valid only for subscriptions where `collection_method`
    is set to `send_invoice`.

`default_payment_method`

:   ID of the default payment method for the subscription. It must
    belong to the customer associated with the subscription. This takes
    precedence over `default_source`. If neither are set, invoices will
    use the customer\'s
    [invoice*settings.default*payment\_method](https://stripe.com/docs/api/customers/object#customer_object-invoice_settings-default_payment_method){.perl-module}
    or
    [default\_source](https://stripe.com/docs/api/customers/object#customer_object-default_source){.perl-module}.

`default_source`

:   ID of the default payment source for the subscription. It must
    belong to the customer associated with the subscription and be in a
    chargeable state. If `default_payment_method` is also set,
    `default_payment_method` will take precedence. If neither are set,
    invoices will use the customer\'s
    [invoice*settings.default*payment\_method](https://stripe.com/docs/api/customers/object#customer_object-invoice_settings-default_payment_method){.perl-module}
    or
    [default\_source](https://stripe.com/docs/api/customers/object#customer_object-default_source){.perl-module}.

`default_tax_rates`

:   The tax rates that will apply to any subscription item that does not
    have `tax_rates` set. Invoices created will have their
    `default_tax_rates` populated from the subscription.

`description`

:   The subscription\'s description, meant to be displayable to the
    customer. Use this field to optionally store an explanation of the
    subscription for rendering in Stripe surfaces.

`items`

:   **required** A list of up to 20 subscription items, each with an
    attached price.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`off_session`

:   Indicates if a customer is on or off-session while an invoice
    payment is attempted.

`payment_behavior`

:   Use `allow_incomplete` to create subscriptions with
    `status=incomplete` if the first invoice cannot be paid. Creating
    subscriptions with this status allows you to manage scenarios where
    additional user actions are needed to pay a subscription\'s invoice.
    For example, SCA regulation may require 3DS authentication to
    complete payment. See the [SCA Migration
    Guide](https://stripe.com/docs/billing/migration/strong-customer-authentication){.perl-module}
    for Billing to learn more. This is the default behavior.

    Use `default_incomplete` to create Subscriptions with
    `status=incomplete` when the first invoice requires payment,
    otherwise start as active. Subscriptions transition to
    `status=active` when successfully confirming the payment intent on
    the first invoice. This allows simpler management of scenarios where
    additional user actions are needed to pay a subscription's invoice.
    Such as failed payments, [SCA
    regulation](https://stripe.com/docs/billing/migration/strong-customer-authentication){.perl-module},
    or collecting a mandate for a bank debit payment method. If the
    payment intent is not confirmed within 23 hours subscriptions
    transition to `status=incomplete_expired`, which is a terminal
    state.

    Use `error_if_incomplete` if you want Stripe to return an HTTP 402
    status code if a subscription\'s first invoice cannot be paid. For
    example, if a payment method requires 3DS authentication due to SCA
    regulation and further user action is needed, this parameter does
    not create a subscription and returns an error instead. This was the
    default behavior for API versions prior to 2019-03-14. See the
    [changelog](https://stripe.com/docs/upgrades#2019-03-14){.perl-module}
    to learn more.

    `pending_if_incomplete` is only used with updates and cannot be
    passed when creating a subscription.

`payment_settings`

:   Payment settings to pass to invoices created by the subscription.

`pending_invoice_item_interval`

:   Specifies an interval for how often to bill for any pending invoice
    items. It is analogous to calling [Create an
    invoice](https://stripe.com/docs/api#create_invoice){.perl-module}
    for the given subscription at the specified interval.

`promotion_code`

:   The API ID of a promotion code to apply to this subscription. A
    promotion code applied to a subscription will only affect invoices
    created for that particular subscription.

`proration_behavior`

:   Determines how to handle
    [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations){.perl-module}
    resulting from the `billing_cycle_anchor`. If no value is passed,
    the default is `create_prorations`.

`transfer_data`

:   If specified, the funds from the subscription\'s invoices will be
    transferred to the destination and the ID of the resulting transfers
    will be found on the resulting charges.

`trial_end`

:   Unix timestamp representing the end of the trial period the customer
    will get before being charged for the first time. This will always
    overwrite any trials that might apply via a subscribed plan. If set,
    trial\_end will override the default trial period of the plan the
    customer is being subscribed to. The special value `now` can be
    provided to end the customer\'s trial immediately. Can be at most
    two years from `billing_cycle_anchor`. See [Using trial periods on
    subscriptions](https://stripe.com/docs/billing/subscriptions/trials){.perl-module}
    to learn more.

`trial_from_plan`

:   Indicates if a plan\'s `trial_period_days` should be applied to the
    subscription. Setting `trial_end` per subscription is preferred, and
    this defaults to `false`. Setting this flag to `true` together with
    `trial_end` is not allowed. See [Using trial periods on
    subscriptions](https://stripe.com/docs/billing/subscriptions/trials){.perl-module}
    to learn more.

`trial_period_days`

:   Integer representing the number of trial period days before the
    customer is charged for the first time. This will always overwrite
    any trials that might apply via a subscribed plan. See [Using trial
    periods on
    subscriptions](https://stripe.com/docs/billing/subscriptions/trials){.perl-module}
    to learn more.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscriptions/create>

delete
------

        my $obj = $stripe->subscriptions( delete => $args ) || die( $stripe->error );

Provided with a
[subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
subscription. It returns the subscription object that was deleted with
its property `deleted` set to true.

Possible parameters are:

`invoice_now`

:   Will generate a final invoice that invoices for any un-invoiced
    metered usage and new/pending proration invoice items.

`prorate`

:   Will generate a proration invoice item that credits remaining unused
    time until the subscription period end.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscriptions/cancel>

list {#list-69}
----

        my $obj = $stripe->subscriptions( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object, this issue an api call to get the list of all subscription.

Possible parameters are:

`collection_method`

:   The collection method of the subscriptions to retrieve. Either
    `charge_automatically` or `send_invoice`.

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`current_period_end`

:   A filter on the list based on the object `current_period_end` field.
    The value can be a string with an integer Unix timestamp, or it can
    be a dictionary with the following options:

`current_period_start`

:   A filter on the list based on the object `current_period_start`
    field. The value can be a string with an integer Unix timestamp, or
    it can be a dictionary with the following options:

`customer`

:   The ID of the customer whose subscriptions will be retrieved.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`price`

:   Filter for subscriptions that contain this recurring price ID.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   The status of the subscriptions to retrieve. Passing in a value of
    `canceled` will return all canceled subscriptions, including those
    belonging to deleted customers. Pass `ended` to find subscriptions
    that are canceled and subscriptions that are expired due to
    [incomplete
    payment](https://stripe.com/docs/billing/subscriptions/overview#subscription-statuses){.perl-module}.
    Passing in a value of `all` will return subscriptions of all
    statuses. If no value is supplied, all subscriptions that have not
    been canceled are returned.

`test_clock`

:   Filter for subscriptions that are associated with the specified test
    clock. The response will not include subscriptions with test clocks
    if this and the customer parameter is not set.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscriptions/list>

retrieve {#retrieve-71}
--------

        my $obj = $stripe->subscriptions( retrieve => $args ) || die( $stripe->error );

Provided with a
[subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or a hash reference, this will retrieve a Stripe subscription and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscriptions/retrieve>

search
------

        my $obj = $stripe->subscriptions( search => {
            query => "status:'active' AND metadata['order_id']:'6735'", } ) || die( $stripe->error );

Provided with a
[subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module},
or a hash reference, this will issue a search api call.

A dictionary with a `data` property that contains an array of up to
`limit` subscriptions. If no objects match the query, the resulting
array will be empty. See the related guide on [expanding properties in
lists](https://stripe.com/docs/expand#lists){.perl-module}.

Possible parameters are:

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`page`

:   A cursor for pagination across multiple pages of results. Don\'t
    include this parameter on the first call. Use the next\_page value
    returned in a previous response to request subsequent results.

`query`

:   **Required**. The search query string. See [search query
    language](https://stripe.com/docs/search#search-query-language){.perl-module}
    and the list of supported [query fields for
    subscriptions](https://stripe.com/docs/search#query-fields-for-subscriptions){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscriptions/search>

update
------

        my $obj = $stripe->subscriptions( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or a hash reference, this will update a Stripe subscription and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}

Possible parameters are:

`add_invoice_items`

:   A list of prices and quantities that will generate invoice items
    appended to the next invoice for this subscription. You may pass up
    to 20 items.

`application_fee_percent`

:   A non-negative decimal between 0 and 100, with at most two decimal
    places. This represents the percentage of the subscription invoice
    subtotal that will be transferred to the application owner\'s Stripe
    account. The request must be made by a platform account on a
    connected account in order to set an application fee percentage. For
    more information, see the application fees
    [documentation](https://stripe.com/docs/connect/subscriptions#collecting-fees-on-subscriptions){.perl-module}.

`automatic_tax`

:   Automatic tax settings for this subscription. We recommend you only
    include this parameter when the existing value is being changed.

`billing_cycle_anchor`

:   Either `now` or `unchanged`. Setting the value to `now` resets the
    subscription\'s billing cycle anchor to the current time (in UTC).
    For more information, see the billing cycle
    [documentation](https://stripe.com/docs/billing/subscriptions/billing-cycle){.perl-module}.

`billing_thresholds`

:   Define thresholds at which an invoice will be sent, and the
    subscription advanced to a new billing period. Pass an empty string
    to remove previously-defined thresholds.

`cancel_at`

:   A timestamp at which the subscription should cancel. If set to a
    date before the current period ends, this will cause a proration if
    prorations have been enabled using `proration_behavior`. If set
    during a future period, this will always cause a proration for that
    period.

`cancel_at_period_end`

:   Boolean indicating whether this subscription should cancel at the
    end of the current period.

`collection_method`

:   Either `charge_automatically`, or `send_invoice`. When charging
    automatically, Stripe will attempt to pay this subscription at the
    end of the cycle using the default source attached to the customer.
    When sending an invoice, Stripe will email your customer an invoice
    with payment instructions. Defaults to `charge_automatically`.

`coupon`

:   The ID of the coupon to apply to this subscription. A coupon applied
    to a subscription will only affect invoices created for that
    particular subscription.

`days_until_due`

:   Number of days a customer has to pay invoices generated by this
    subscription. Valid only for subscriptions where `collection_method`
    is set to `send_invoice`.

`default_payment_method`

:   ID of the default payment method for the subscription. It must
    belong to the customer associated with the subscription. This takes
    precedence over `default_source`. If neither are set, invoices will
    use the customer\'s
    [invoice*settings.default*payment\_method](https://stripe.com/docs/api/customers/object#customer_object-invoice_settings-default_payment_method){.perl-module}
    or
    [default\_source](https://stripe.com/docs/api/customers/object#customer_object-default_source){.perl-module}.

`default_source`

:   ID of the default payment source for the subscription. It must
    belong to the customer associated with the subscription and be in a
    chargeable state. If `default_payment_method` is also set,
    `default_payment_method` will take precedence. If neither are set,
    invoices will use the customer\'s
    [invoice*settings.default*payment\_method](https://stripe.com/docs/api/customers/object#customer_object-invoice_settings-default_payment_method){.perl-module}
    or
    [default\_source](https://stripe.com/docs/api/customers/object#customer_object-default_source){.perl-module}.

`default_tax_rates`

:   The tax rates that will apply to any subscription item that does not
    have `tax_rates` set. Invoices created will have their
    `default_tax_rates` populated from the subscription. Pass an empty
    string to remove previously-defined tax rates.

`description`

:   The subscription\'s description, meant to be displayable to the
    customer. Use this field to optionally store an explanation of the
    subscription for rendering in Stripe surfaces.

`items`

:   A list of up to 20 subscription items, each with an attached price.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`off_session`

:   Indicates if a customer is on or off-session while an invoice
    payment is attempted.

`pause_collection`

:   If specified, payment collection for this subscription will be
    paused.

`payment_behavior`

:   Use `allow_incomplete` to transition the subscription to
    `status=past_due` if a payment is required but cannot be paid. This
    allows you to manage scenarios where additional user actions are
    needed to pay a subscription\'s invoice. For example, SCA regulation
    may require 3DS authentication to complete payment. See the [SCA
    Migration
    Guide](https://stripe.com/docs/billing/migration/strong-customer-authentication){.perl-module}
    for Billing to learn more. This is the default behavior.

    Use `default_incomplete` to transition the subscription to
    `status=past_due` when payment is required and await explicit
    confirmation of the invoice\'s payment intent. This allows simpler
    management of scenarios where additional user actions are needed to
    pay a subscription's invoice. Such as failed payments, [SCA
    regulation](https://stripe.com/docs/billing/migration/strong-customer-authentication){.perl-module},
    or collecting a mandate for a bank debit payment method.

    Use `pending_if_incomplete` to update the subscription using
    [pending
    updates](https://stripe.com/docs/billing/subscriptions/pending-updates){.perl-module}.
    When you use `pending_if_incomplete` you can only pass the
    parameters [supported by pending
    updates](https://stripe.com/docs/billing/pending-updates-reference#supported-attributes){.perl-module}.

    Use `error_if_incomplete` if you want Stripe to return an HTTP 402
    status code if a subscription\'s invoice cannot be paid. For
    example, if a payment method requires 3DS authentication due to SCA
    regulation and further user action is needed, this parameter does
    not update the subscription and returns an error instead. This was
    the default behavior for API versions prior to 2019-03-14. See the
    [changelog](https://stripe.com/docs/upgrades#2019-03-14){.perl-module}
    to learn more.

`payment_settings`

:   Payment settings to pass to invoices created by the subscription.

`pending_invoice_item_interval`

:   Specifies an interval for how often to bill for any pending invoice
    items. It is analogous to calling [Create an
    invoice](https://stripe.com/docs/api#create_invoice){.perl-module}
    for the given subscription at the specified interval.

`promotion_code`

:   The promotion code to apply to this subscription. A promotion code
    applied to a subscription will only affect invoices created for that
    particular subscription.

`proration_behavior`

:   Determines how to handle
    [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations){.perl-module}
    when the billing cycle changes (e.g., when switching plans,
    resetting `billing_cycle_anchor=now`, or starting a trial), or if an
    item\'s `quantity` changes.

`proration_date`

:   If set, the proration will be calculated as though the subscription
    was updated at the given time. This can be used to apply exactly the
    same proration that was previewed with [upcoming
    invoice](https://stripe.com#retrieve_customer_invoice){.perl-module}
    endpoint. It can also be used to implement custom proration logic,
    such as prorating by day instead of by second, by providing the time
    that you wish to use for proration calculations.

`transfer_data`

:   If specified, the funds from the subscription\'s invoices will be
    transferred to the destination and the ID of the resulting transfers
    will be found on the resulting charges. This will be unset if you
    POST an empty value.

`trial_end`

:   Unix timestamp representing the end of the trial period the customer
    will get before being charged for the first time. This will always
    overwrite any trials that might apply via a subscribed plan. If set,
    trial\_end will override the default trial period of the plan the
    customer is being subscribed to. The special value `now` can be
    provided to end the customer\'s trial immediately. Can be at most
    two years from `billing_cycle_anchor`.

`trial_from_plan`

:   Indicates if a plan\'s `trial_period_days` should be applied to the
    subscription. Setting `trial_end` per subscription is preferred, and
    this defaults to `false`. Setting this flag to `true` together with
    `trial_end` is not allowed. See [Using trial periods on
    subscriptions](https://stripe.com/docs/billing/subscriptions/trials){.perl-module}
    to learn more.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscriptions/update>

SUBSCRIPTION ITEM
=================

You can create, delete, list, retrieve or update subscription item

create {#create-56}
------

        my $obj = $stripe->subscription_items( create => {
            price        => "price_1Le1oa2eZvKYlo2CuD7mwpZu",
            quantity     => "2",
            subscription => "sub_1LduLW2eZvKYlo2CkXEi23Ew", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::Subscription::Item](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module}
object or a hash reference, this will create a Stripe subscription item
and return an
[Net::API::Stripe::Billing::Subscription::Item](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module}
object.

Possible parameters are:

`billing_thresholds`

:   Define thresholds at which an invoice will be sent, and the
    subscription advanced to a new billing period. When updating, pass
    an empty string to remove previously-defined thresholds.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`payment_behavior`

:   Use `allow_incomplete` to transition the subscription to
    `status=past_due` if a payment is required but cannot be paid. This
    allows you to manage scenarios where additional user actions are
    needed to pay a subscription\'s invoice. For example, SCA regulation
    may require 3DS authentication to complete payment. See the [SCA
    Migration
    Guide](https://stripe.com/docs/billing/migration/strong-customer-authentication){.perl-module}
    for Billing to learn more. This is the default behavior.

    Use `default_incomplete` to transition the subscription to
    `status=past_due` when payment is required and await explicit
    confirmation of the invoice\'s payment intent. This allows simpler
    management of scenarios where additional user actions are needed to
    pay a subscription's invoice. Such as failed payments, [SCA
    regulation](https://stripe.com/docs/billing/migration/strong-customer-authentication){.perl-module},
    or collecting a mandate for a bank debit payment method.

    Use `pending_if_incomplete` to update the subscription using
    [pending
    updates](https://stripe.com/docs/billing/subscriptions/pending-updates){.perl-module}.
    When you use `pending_if_incomplete` you can only pass the
    parameters [supported by pending
    updates](https://stripe.com/docs/billing/pending-updates-reference#supported-attributes){.perl-module}.

    Use `error_if_incomplete` if you want Stripe to return an HTTP 402
    status code if a subscription\'s invoice cannot be paid. For
    example, if a payment method requires 3DS authentication due to SCA
    regulation and further user action is needed, this parameter does
    not update the subscription and returns an error instead. This was
    the default behavior for API versions prior to 2019-03-14. See the
    [changelog](https://stripe.com/docs/upgrades#2019-03-14){.perl-module}
    to learn more.

`price`

:   The ID of the price object.

`price_data`

:   Data used to generate a new
    [Price](https://stripe.com/docs/api/prices){.perl-module} object
    inline.

`proration_behavior`

:   Determines how to handle
    [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations){.perl-module}
    when the billing cycle changes (e.g., when switching plans,
    resetting `billing_cycle_anchor=now`, or starting a trial), or if an
    item\'s `quantity` changes.

`proration_date`

:   If set, the proration will be calculated as though the subscription
    was updated at the given time. This can be used to apply the same
    proration that was previewed with the [upcoming
    invoice](https://stripe.com#retrieve_customer_invoice){.perl-module}
    endpoint.

`quantity`

:   The quantity you\'d like to apply to the subscription item you\'re
    creating.

`subscription`

:   **Required**. The identifier of the subscription to modify.

`tax_rates`

:   A list of [Tax
    Rate](https://stripe.com/docs/api/tax_rates){.perl-module} ids.
    These Tax Rates will override the
    [`default_tax_rates`](https://stripe.com/docs/api/subscriptions/create#create_subscription-default_tax_rates){.perl-module}
    on the Subscription. When updating, pass an empty string to remove
    previously-defined tax rates.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_items/create>

delete
------

        my $obj = $stripe->subscription_items( delete => $args ) || die( $stripe->error );

Provided with a [subscription
item](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
subscription item. It returns the subscription item object that was
deleted with its property `deleted` set to true.

Possible parameters are:

`clear_usage`

:   Delete all usage for the given subscription item. Allowed only when
    the current plan\'s `usage_type` is `metered`.

`proration_behavior`

:   Determines how to handle
    [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations){.perl-module}
    when the billing cycle changes (e.g., when switching plans,
    resetting `billing_cycle_anchor=now`, or starting a trial), or if an
    item\'s `quantity` changes.

`proration_date`

:   If set, the proration will be calculated as though the subscription
    was updated at the given time. This can be used to apply the same
    proration that was previewed with the [upcoming
    invoice](https://stripe.com#retrieve_customer_invoice){.perl-module}
    endpoint.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_items/delete>

list {#list-70}
----

        my $obj = $stripe->subscription_items( list => {
            subscription => "sub_1LduLW2eZvKYlo2CkXEi23Ew", } ) || die( $stripe->error );

Provided with a [subscription
item](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module}
object, this issue an api call to get the list of all subscription item.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`subscription`

:   **Required**. The ID of the subscription whose items will be
    retrieved.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_items/list>

retrieve {#retrieve-72}
--------

        my $obj = $stripe->subscription_items( retrieve => $args ) || die( $stripe->error );

Provided with a [subscription
item](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module}
object or a hash reference, this will retrieve a Stripe subscription
item and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_items/retrieve>

update
------

        my $obj = $stripe->subscription_items( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [subscription
item](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module}
object or a hash reference, this will update a Stripe subscription item
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Item){.perl-module}

Possible parameters are:

`billing_thresholds`

:   Define thresholds at which an invoice will be sent, and the
    subscription advanced to a new billing period. When updating, pass
    an empty string to remove previously-defined thresholds.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`off_session`

:   Indicates if a customer is on or off-session while an invoice
    payment is attempted.

`payment_behavior`

:   Use `allow_incomplete` to transition the subscription to
    `status=past_due` if a payment is required but cannot be paid. This
    allows you to manage scenarios where additional user actions are
    needed to pay a subscription\'s invoice. For example, SCA regulation
    may require 3DS authentication to complete payment. See the [SCA
    Migration
    Guide](https://stripe.com/docs/billing/migration/strong-customer-authentication){.perl-module}
    for Billing to learn more. This is the default behavior.

    Use `default_incomplete` to transition the subscription to
    `status=past_due` when payment is required and await explicit
    confirmation of the invoice\'s payment intent. This allows simpler
    management of scenarios where additional user actions are needed to
    pay a subscription's invoice. Such as failed payments, [SCA
    regulation](https://stripe.com/docs/billing/migration/strong-customer-authentication){.perl-module},
    or collecting a mandate for a bank debit payment method.

    Use `pending_if_incomplete` to update the subscription using
    [pending
    updates](https://stripe.com/docs/billing/subscriptions/pending-updates){.perl-module}.
    When you use `pending_if_incomplete` you can only pass the
    parameters [supported by pending
    updates](https://stripe.com/docs/billing/pending-updates-reference#supported-attributes){.perl-module}.

    Use `error_if_incomplete` if you want Stripe to return an HTTP 402
    status code if a subscription\'s invoice cannot be paid. For
    example, if a payment method requires 3DS authentication due to SCA
    regulation and further user action is needed, this parameter does
    not update the subscription and returns an error instead. This was
    the default behavior for API versions prior to 2019-03-14. See the
    [changelog](https://stripe.com/docs/upgrades#2019-03-14){.perl-module}
    to learn more.

`price`

:   The ID of the price object. When changing a subscription item\'s
    price, `quantity` is set to 1 unless a `quantity` parameter is
    provided.

`price_data`

:   Data used to generate a new
    [Price](https://stripe.com/docs/api/prices){.perl-module} object
    inline.

`proration_behavior`

:   Determines how to handle
    [prorations](https://stripe.com/docs/subscriptions/billing-cycle#prorations){.perl-module}
    when the billing cycle changes (e.g., when switching plans,
    resetting `billing_cycle_anchor=now`, or starting a trial), or if an
    item\'s `quantity` changes.

`proration_date`

:   If set, the proration will be calculated as though the subscription
    was updated at the given time. This can be used to apply the same
    proration that was previewed with the [upcoming
    invoice](https://stripe.com#retrieve_customer_invoice){.perl-module}
    endpoint.

`quantity`

:   The quantity you\'d like to apply to the subscription item you\'re
    creating.

`tax_rates`

:   A list of [Tax
    Rate](https://stripe.com/docs/api/tax_rates){.perl-module} ids.
    These Tax Rates will override the
    [`default_tax_rates`](https://stripe.com/docs/api/subscriptions/create#create_subscription-default_tax_rates){.perl-module}
    on the Subscription. When updating, pass an empty string to remove
    previously-defined tax rates.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_items/update>

SUBSCRIPTION SCHEDULE
=====================

You can cancel, create, list, release, retrieve or update subscription
schedule

cancel
------

        my $obj = $stripe->subscription_schedules( cancel => $args ) || die( $stripe->error );

Provided with a [subscription
schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module},
or a hash reference, this will issue a cancel api call.

The canceled `subscription_schedule` object. Its status will be
`canceled` and `canceled_at` will be the current time.

Possible parameters are:

`invoice_now`

:   If the subscription schedule is `active`, indicates if a final
    invoice will be generated that contains any un-invoiced metered
    usage and new/pending proration invoice items. Defaults to `true`.

`prorate`

:   If the subscription schedule is `active`, indicates if the
    cancellation should be prorated. Defaults to `true`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_schedules/cancel>

create {#create-57}
------

        my $obj = $stripe->subscription_schedules( create => {
            customer     => "cus_AJ78ZaALpqgiuZ",
            end_behavior => "release",
            phases       => [,
                items      => [,
                    price    => "plan_JiX4v6L7JY0Vyt",
                    quantity => "1",
                ],
                iterations => "12",
            ],
            start_date   => "1662865884", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object or a hash reference, this will create a Stripe subscription
schedule and return an
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object.

Possible parameters are:

`customer`

:   The identifier of the customer to create the subscription schedule
    for.

`default_settings`

:   Object representing the subscription schedule\'s default settings.

`end_behavior`

:   Configures how the subscription schedule behaves when it ends.
    Possible values are `release` or `cancel` with the default being
    `release`. `release` will end the subscription schedule and keep the
    underlying subscription running.`cancel` will end the subscription
    schedule and cancel the underlying subscription.

`from_subscription`

:   Migrate an existing subscription to be managed by a subscription
    schedule. If this parameter is set, a subscription schedule will be
    created using the subscription\'s item(s), set to auto-renew using
    the subscription\'s interval. When using this parameter, other
    parameters (such as phase values) cannot be set. To create a
    subscription schedule with other modifications, we recommend making
    two separate API calls.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`phases`

:   List representing phases of the subscription schedule. Each phase
    can be customized to have different durations, plans, and coupons.
    If there are multiple phases, the `end_date` of one phase will
    always equal the `start_date` of the next phase.

`start_date`

:   When the subscription schedule starts. We recommend using `now` so
    that it starts the subscription immediately. You can also use a Unix
    timestamp to backdate the subscription so that it starts on a past
    date, or set a future date for the subscription to start on.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_schedules/create>

list {#list-71}
----

        my $obj = $stripe->subscription_schedules( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [subscription
schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object, this issue an api call to get the list of all subscription
schedule.

Possible parameters are:

`canceled_at`

:   A filter on the list based on the object `canceled_at` field. The
    value can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`completed_at`

:   A filter on the list based on the object `completed_at` field. The
    value can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`customer`

:   Only return subscription schedules for the given customer.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`released_at`

:   A filter on the list based on the object `released_at` field. The
    value can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`scheduled`

:   Only return subscription schedules that have not started yet.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_schedules/list>

release
-------

        my $obj = $stripe->subscription_schedules( release => $args ) || die( $stripe->error );

Provided with a [subscription
schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module},
or a hash reference, this will issue a release api call.

The released `subscription_schedule` object. Its status will be
`released`, `released_at` will be the current time, and
`released_subscription` will be the ID of the subscription the
subscription schedule managed prior to being released.

Possible parameters are:

`preserve_cancel_date`

:   Keep any cancellation on the subscription that the schedule has set

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_schedules/release>

retrieve {#retrieve-73}
--------

        my $obj = $stripe->subscription_schedules( retrieve => $args ) || die( $stripe->error );

Provided with a [subscription
schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object or a hash reference, this will retrieve a Stripe subscription
schedule and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_schedules/retrieve>

update
------

        my $obj = $stripe->subscription_schedules( update => {
            end_behavior => "release", } ) || die( $stripe->error );

Provided with a [subscription
schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object or a hash reference, this will update a Stripe subscription
schedule and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}

Possible parameters are:

`default_settings`

:   Object representing the subscription schedule\'s default settings.

`end_behavior`

:   Configures how the subscription schedule behaves when it ends.
    Possible values are `release` or `cancel` with the default being
    `release`. `release` will end the subscription schedule and keep the
    underlying subscription running.`cancel` will end the subscription
    schedule and cancel the underlying subscription.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`phases`

:   List representing phases of the subscription schedule. Each phase
    can be customized to have different durations, plans, and coupons.
    If there are multiple phases, the `end_date` of one phase will
    always equal the `start_date` of the next phase. Note that past
    phases can be omitted.

`proration_behavior`

:   If the update changes the current phase, indicates whether the
    changes should be prorated. The default value is
    `create_prorations`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/subscription_schedules/update>

SUBSCRIPTION SCHEDULES
======================

You can **create**, **retrieve**, **update**, **list**, **cancel** or
**release** schedules

cancel
------

Provided with a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object or an hash reference of parameters and this will cancel a Stripe
subscription schedule and return a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object.

Possible parameters are:

*id* A Stripe subscription schedule. This is required.

:   

*invoice\_now* Boolean

:   

*prorate* Boolean

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscription_schedules/cancel>

create {#create-58}
------

Provided with a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object or an hash reference of parameters and this will create a Stripe
subscription schedule and return a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object.

Possible parameters are:

*customer* A Stripe customer id

:   

*default\_settings* An hash reference with the following properties:

:   

    billing\_thresholds.amount\_gte

    :   

    billing\_thresholds.reset\_billing\_cycle\_anchor

    :   

    collection\_method

    :   

    default\_payment\_method

    :   

    invoice\_settings.days\_until\_due

    :   

*end\_behavior* String. One of release or cancel

:   

*from\_subscription* Stripe subscription id

:   

*metadata* An aribitrary hash reference

:   

*phases* An array of hash reference with following properties:

:   

    plan

    :   

    price

    :   

    application\_fee\_percent

    :   

    billing\_thresholds

    :   

    collection\_method

    :   

    coupon

    :   

    default\_payment\_method

    :   

    default\_tax\_rates

    :   

    end\_date

    :   

    invoice\_settings

    :   

    iterations

    :   

    tax\_percent

    :   

    trial

    :   

    trial\_end

    :   

*start\_date* Date or timestamp or the word \'now\'

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscription_schedules/create>

list {#list-72}
----

Provided with an hash reference of parameters this will get the list of
subscription schedules and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object

Possible parameters are:

*canceled\_at* Unix timestamp

:   

*completed\_at* Unix timestamp

:   

*created* Unix timestamp

:   

*customer* A Stripe customer id

:   

*email* String. E-mail address

:   

*ending\_before* A Stripe subscription schedule id

:   

*limit* Integer

:   

*released\_at* Unix timestamp

:   

*scheduled* Boolean

:   

*starting\_after* A Stripe subscription schedule id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscription_schedules/list>

release
-------

Provided with a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object or an hash reference of parameters and this will release a Stripe
subscription schedule and return a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object.

Possible parameters are:

*id* A Stripe subscription schedule. This is required.

:   

*preserve\_cancel\_date* Boolean

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscription_schedules/release>

retrieve {#retrieve-74}
--------

Provided with a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object or an hash reference of parameters and this will retrieve a
Stripe subscription schedule and return a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object.

Possible parameters are:

*id* A Stripe subscription schedule id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscription_schedules/retrieve>

update
------

Provided with a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object or an hash reference of parameters and this will update a Stripe
subscription schedule and return a
[Net::API::Stripe::Billing::Subscription::Schedule](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription::Schedule){.perl-module}
object.

Possible parameters are:

*id* A Stripe subscription id

:   

*default\_settings* An hash reference with the following properties: billing\_thresholds.amount\_gte billing\_thresholds.reset\_billing\_cycle\_anchor collection\_method default\_payment\_method invoice\_settings.days\_until\_due

:   

*end\_behavior* String. One of release or cancel

:   

*from\_subscription* Stripe subscription id

:   

*metadata* An aribitrary hash reference

:   

*phases* An array of hash reference with following properties: plan application\_fee\_percent billing\_thresholds collection\_method coupon default\_payment\_method default\_tax\_rates end\_date invoice\_settings iterations tax\_percent trial trial\_end

:   

*prorate* Boolean

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscription_schedules/update>

SUBSCRIPTIONS
=============

You can **create**, **delete\_discount**, **retrieve**, **update**,
**list** or **cancel** subscriptions

cancel
------

Provided with a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or an hash reference of parameters, this will cancel a Stripe
subscription and return a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object.

Possible parameters are:

*id* A Stripe subscription id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscriptions/create>

create {#create-59}
------

Provided with a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or an hash reference of parameters, this will create a Stripe
subscription and return a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object.

Possible parameters are:

*customer* A Strip customer id. This is required.

:   

*application\_fee\_percent* Decimal

:   

*backdate\_start\_date* Date or timestamp

:   

*billing\_cycle\_anchor* Date or timestamp

:   

*billing\_thresholds* An hash reference with the following properties: amount\_gte reset\_billing\_cycle\_anchor

:   

*cancel\_at* Date or timestamp

:   

*cancel\_at\_period\_end* Boolean

:   

*collection\_method* String. One of charge\_automatically, or send\_invoice

:   

*coupon* String

:   

*days\_until\_due* Integer

:   

*default\_payment\_method* A Stripe payment method id

:   

*default\_source* A Stripe source id

:   

*default\_tax\_rates* Array of string

:   

*items* An array of hash reference with the following properties:

:   

    billing\_thresholds.usage\_gte

    :   

    plan

    :   

    price

    :   

    price\_data.currency

    :   

    price\_data.product

    :   

    price\_data.recurring.interval

    :   

    price\_data.recurring.interval\_count

    :   

    price\_data.unit\_amount\_decimal

    :   

    price\_data.unit\_amount

    :   

    billing\_thresholds

    :   

    metadata

    :   

    quantity

    :   

    tax\_rates

    :   

*metadata* An arbitrary hash reference

:   

*off\_session* Boolean

:   

*payment\_behavior* String. One of allow\_incomplete error\_if\_incomplete or pending\_if\_incomplete

:   

*pending\_invoice\_item\_interval* An hash reference with the following properties: interval interval\_count

:   

*prorate* Boolean

:   

*proration\_behavior* String. One of billing\_cycle\_anchor, create\_prorations or none

:   

*tax\_percent* Decimal

:   

*trial\_end* Unix timestamp or \'now\'

:   

*trial\_from\_plan* Boolean

:   

*trial\_period\_days* Integer

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscriptions/create>

delete\_discount
----------------

Provided with a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or an hash reference of parameters, this will remove its discount
and return a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object.

Possible parameters are:

*id* A Stripe subscription id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscriptions/create>

list {#list-73}
----

Provided with a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or an hash reference of parameters, this will create a Stripe
subscription and return a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object.

Possible parameters are:

*active* Boolean

:   

*created* Date or unix timestamp

:   

*ids* Array reference

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*shippable* Boolean

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscriptions/create>

retrieve {#retrieve-75}
--------

Provided with a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or an hash reference of parameters, this will retrieve a Stripe
subscription and return a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object.

Possible parameters are:

*id* A Stripe subscription id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscriptions/create>

update
------

Provided with a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object or an hash reference of parameters, this will update a Stripe
subscription and return a
[Net::API::Stripe::Billing::Subscription](https://metacpan.org/pod/Net::API::Stripe::Billing::Subscription){.perl-module}
object.

Possible parameters are:

*id* A Stripe subscription id

:   

*application\_fee\_percent* Decimal

:   

*billing\_cycle\_anchor* Date or timestamp

:   

*billing\_thresholds* An hash reference with the following properties: amount\_gte reset\_billing\_cycle\_anchor

:   

*cancel\_at* Date or timestamp

:   

*cancel\_at\_period\_end* Boolean

:   

*collection\_method* String. One of charge\_automatically, or send\_invoice

:   

*coupon* String

:   

*days\_until\_due* Integer

:   

*default\_payment\_method* A Stripe payment method id

:   

*default\_source* A Stripe source id

:   

*default\_tax\_rates* Array of string

:   

*items* An array of hash reference with the following properties: plan billing\_thresholds metadata quantity tax\_rates

:   

*metadata* An arbitrary hash reference

:   

*off\_session* Boolean

:   

*pause\_collection* An hash reference with the following properties: behavior resumes\_at

:   

*payment\_behavior* String. One of allow\_incomplete error\_if\_incomplete or pending\_if\_incomplete

:   

*pending\_invoice\_item\_interval* An hash reference with the following properties: interval interval\_count

:   

*prorate* Boolean

:   

*prorate\_date* A Date or timestamp

:   

*tax\_percent* Decimal

:   

*trial\_end* Unix timestamp or \'now\'

:   

*trial\_from\_plan* Boolean

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/subscriptions/create>

TAX CODE
========

You can list or retrieve tax code

list {#list-74}
----

        my $obj = $stripe->tax_codes( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [tax
code](https://metacpan.org/pod/Net::API::Stripe::Product::TaxCode){.perl-module}
object, this issue an api call to get the list of all tax code.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tax_codes/list>

retrieve {#retrieve-76}
--------

        my $obj = $stripe->tax_codes( retrieve => $args ) || die( $stripe->error );

Provided with a [tax
code](https://metacpan.org/pod/Net::API::Stripe::Product::TaxCode){.perl-module}
object or a hash reference, this will retrieve a Stripe tax code and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Product::TaxCode){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tax_codes/retrieve>

TAX ID
======

You can create, delete, list or retrieve tax id

create {#create-60}
------

        my $obj = $stripe->tax_ids( create => {
            type  => "eu_vat",
            value => "DE123456789", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Customer::TaxId](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module}
object or a hash reference, this will create a Stripe tax id and return
an
[Net::API::Stripe::Customer::TaxId](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module}
object.

Possible parameters are:

`type`

:   **Required**. Type of the tax ID, one of `ae_trn`, `au_abn`,
    `au_arn`, `bg_uic`, `br_cnpj`, `br_cpf`, `ca_bn`, `ca_gst_hst`,
    `ca_pst_bc`, `ca_pst_mb`, `ca_pst_sk`, `ca_qst`, `ch_vat`, `cl_tin`,
    `es_cif`, `eu_oss_vat`, `eu_vat`, `gb_vat`, `ge_vat`, `hk_br`,
    `hu_tin`, `id_npwp`, `il_vat`, `in_gst`, `is_vat`, `jp_cn`, `jp_rn`,
    `kr_brn`, `li_uid`, `mx_rfc`, `my_frp`, `my_itn`, `my_sst`,
    `no_vat`, `nz_gst`, `ru_inn`, `ru_kpp`, `sa_vat`, `sg_gst`,
    `sg_uen`, `si_tin`, `th_vat`, `tw_vat`, `ua_vat`, `us_ein`, or
    `za_vat`

`value`

:   **Required**. Value of the tax ID.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_tax_ids/create>

delete
------

        my $obj = $stripe->tax_ids( delete => $args ) || die( $stripe->error );

Provided with a [tax
id](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
tax id. It returns the tax id object that was deleted with its property
`deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_tax_ids/delete>

list {#list-75}
----

        my $obj = $stripe->tax_ids( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [tax
id](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module}
object, this issue an api call to get the list of all tax id.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_tax_ids/list>

retrieve {#retrieve-77}
--------

        my $obj = $stripe->tax_ids( retrieve => $args ) || die( $stripe->error );

Provided with a [tax
id](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module}
object or a hash reference, this will retrieve a Stripe tax id and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Customer::TaxId){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/customer_tax_ids/retrieve>

TAX IDS
=======

You can **create**, **retrieve**, **delete** or **list** tax ids

create {#create-61}
------

Provided with a
[Net::API::Stripe::Billing::TaxID](https://metacpan.org/pod/Net::API::Stripe::Billing::TaxID){.perl-module}
object or an hash reference of parameters, this will cancel a Stripe tax
id and return a
[Net::API::Stripe::Billing::TaxID](https://metacpan.org/pod/Net::API::Stripe::Billing::TaxID){.perl-module}
object.

Possible parameters are:

*customer* A Stripe customer id

:   

*type* String, such as au\_abn, ch\_vat, eu\_vat, in\_gst, mx\_rfc, no\_vat, nz\_gst, or za\_vat

:   

*value* String

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customer_tax_ids/create>

delete
------

Provided with a
[Net::API::Stripe::Billing::TaxID](https://metacpan.org/pod/Net::API::Stripe::Billing::TaxID){.perl-module}
object or an hash reference of parameters, this will cancel a Stripe tax
id and return a
[Net::API::Stripe::Billing::TaxID](https://metacpan.org/pod/Net::API::Stripe::Billing::TaxID){.perl-module}
object.

Possible parameters are:

*id* A Stripe tax id. This is required

:   

*customer* A Stripe customer id. This is required

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customer_tax_ids/delete>

list {#list-76}
----

Provided with an hash reference of parameters, this will get the list of
Stripe tax id and return a
[Net::API::Stripe::List](https://metacpan.org/pod/Net::API::Stripe::List){.perl-module}
object.

Possible parameters are:

*id* A Stripe customer id. This is required

:   

*ending\_before* A Stripe credit note id

:   

*limit* Integer

:   

*starting\_after* A Stripe credit note id

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customer_tax_ids/list>

retrieve {#retrieve-78}
--------

Provided with a
[Net::API::Stripe::Billing::TaxID](https://metacpan.org/pod/Net::API::Stripe::Billing::TaxID){.perl-module}
object or an hash reference of parameters, this will cancel a Stripe tax
id and return a
[Net::API::Stripe::Billing::TaxID](https://metacpan.org/pod/Net::API::Stripe::Billing::TaxID){.perl-module}
object.

Possible parameters are:

*id* A Stripe tax id. This is required

:   

*customer* A Stripe customer id. This is required

:   

For more information, see Stripe documentation here:
<https://stripe.com/docs/api/customer_tax_ids/retrieve>

TAX RATE
========

You can create, list, retrieve or update tax rate

create {#create-62}
------

        my $obj = $stripe->tax_rates( create => {
            description  => "VAT Germany",
            display_name => "VAT",
            inclusive    => "0",
            jurisdiction => "DE",
            percentage   => "16", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Tax::Rate](https://metacpan.org/pod/Net::API::Stripe::Tax::Rate){.perl-module}
object or a hash reference, this will create a Stripe tax rate and
return an
[Net::API::Stripe::Tax::Rate](https://metacpan.org/pod/Net::API::Stripe::Tax::Rate){.perl-module}
object.

Possible parameters are:

`active`

:   Flag determining whether the tax rate is active or inactive
    (archived). Inactive tax rates cannot be used with new applications
    or Checkout Sessions, but will still work for subscriptions and
    invoices that already have it set.

`country`

:   Two-letter country code ([ISO 3166-1
    alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)){.perl-module}.

`description`

:   An arbitrary string attached to the tax rate for your internal use
    only. It will not be visible to your customers.

`display_name`

:   **Required**. The display name of the tax rate, which will be shown
    to users.

`inclusive`

:   **Required**. This specifies if the tax rate is inclusive or
    exclusive.

`jurisdiction`

:   The jurisdiction for the tax rate. You can use this label field for
    tax reporting purposes. It also appears on your customer's invoice.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`percentage`

:   **Required**. This represents the tax rate percent out of 100.

`state`

:   [ISO 3166-2 subdivision
    code](https://en.wikipedia.org/wiki/ISO_3166-2:US){.perl-module},
    without country prefix. For example, \"NY\" for New York, United
    States.

`tax_type`

:   The high-level tax type, such as `vat` or `sales_tax`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tax_rates/create>

list {#list-77}
----

        my $obj = $stripe->tax_rates( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [tax
rate](https://metacpan.org/pod/Net::API::Stripe::Tax::Rate){.perl-module}
object, this issue an api call to get the list of all tax rate.

Possible parameters are:

`active`

:   Optional flag to filter by tax rates that are either active or
    inactive (archived).

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`inclusive`

:   Optional flag to filter by tax rates that are inclusive (or those
    that are not inclusive).

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tax_rates/list>

retrieve {#retrieve-79}
--------

        my $obj = $stripe->tax_rates( retrieve => $args ) || die( $stripe->error );

Provided with a [tax
rate](https://metacpan.org/pod/Net::API::Stripe::Tax::Rate){.perl-module}
object or a hash reference, this will retrieve a Stripe tax rate and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Tax::Rate){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tax_rates/retrieve>

update
------

        my $obj = $stripe->tax_rates( update => {
            active => "0", } ) || die( $stripe->error );

Provided with a [tax
rate](https://metacpan.org/pod/Net::API::Stripe::Tax::Rate){.perl-module}
object or a hash reference, this will update a Stripe tax rate and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Tax::Rate){.perl-module}

Possible parameters are:

`active`

:   Flag determining whether the tax rate is active or inactive
    (archived). Inactive tax rates cannot be used with new applications
    or Checkout Sessions, but will still work for subscriptions and
    invoices that already have it set.

`country`

:   Two-letter country code ([ISO 3166-1
    alpha-2](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2)){.perl-module}.

`description`

:   An arbitrary string attached to the tax rate for your internal use
    only. It will not be visible to your customers.

`display_name`

:   The display name of the tax rate, which will be shown to users.

`jurisdiction`

:   The jurisdiction for the tax rate. You can use this label field for
    tax reporting purposes. It also appears on your customer's invoice.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`state`

:   [ISO 3166-2 subdivision
    code](https://en.wikipedia.org/wiki/ISO_3166-2:US){.perl-module},
    without country prefix. For example, \"NY\" for New York, United
    States.

`tax_type`

:   The high-level tax type, such as `vat` or `sales_tax`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tax_rates/update>

TERMINAL CONFIGURATION
======================

You can create, delete, list, retrieve or update terminal configuration

create {#create-63}
------

        my $obj = $stripe->terminal_configurations( create => {
            bbpos_wisepos_e =>
            {
                splashscreen => "file_1Le9F32eZvKYlo2CupWuSTyz",
            } } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Terminal::Configuration](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module}
object or a hash reference, this will create a Stripe terminal
configuration and return an
[Net::API::Stripe::Terminal::Configuration](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module}
object.

Possible parameters are:

`bbpos_wisepos_e`

:   An object containing device type specific settings for BBPOS WisePOS
    E readers

`tipping`

:   Tipping configurations for readers supporting on-reader tips

`verifone_p400`

:   An object containing device type specific settings for Verifone P400
    readers

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/configuration/create>

delete
------

        my $obj = $stripe->terminal_configurations( delete => $args ) || die( $stripe->error );

Provided with a [terminal
configuration](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
terminal configuration. It returns the terminal configuration object
that was deleted with its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/configuration/delete>

list {#list-78}
----

        my $obj = $stripe->terminal_configurations( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [terminal
configuration](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module}
object, this issue an api call to get the list of all terminal
configuration.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`is_account_default`

:   if present, only return the account default or non-default
    configurations.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/configuration/list>

retrieve {#retrieve-80}
--------

        my $obj = $stripe->terminal_configurations( retrieve => $args ) || die( $stripe->error );

Provided with a [terminal
configuration](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module}
object or a hash reference, this will retrieve a Stripe terminal
configuration and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/configuration/retrieve>

update
------

        my $obj = $stripe->terminal_configurations( update => {
            bbpos_wisepos_e =>
            {
                splashscreen => "file_1Le9F32eZvKYlo2CupWuSTyz",
            } } ) || die( $stripe->error );

Provided with a [terminal
configuration](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module}
object or a hash reference, this will update a Stripe terminal
configuration and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Terminal::Configuration){.perl-module}

Possible parameters are:

`bbpos_wisepos_e`

:   An object containing device type specific settings for BBPOS WisePOS
    E readers

`tipping`

:   Tipping configurations for readers supporting on-reader tips

`verifone_p400`

:   An object containing device type specific settings for Verifone P400
    readers

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/configuration/update>

TERMINAL CONNECTION TOKEN
=========================

You can create terminal connection token

create {#create-64}
------

        my $obj = $stripe->terminal_connection_tokens( create => $args ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Terminal::ConnectionToken](https://metacpan.org/pod/Net::API::Stripe::Terminal::ConnectionToken){.perl-module}
object or a hash reference, this will create a Stripe terminal
connection token and return an
[Net::API::Stripe::Terminal::ConnectionToken](https://metacpan.org/pod/Net::API::Stripe::Terminal::ConnectionToken){.perl-module}
object.

Possible parameters are:

`location`

:   The id of the location that this connection token is scoped to. If
    specified the connection token will only be usable with readers
    assigned to that location, otherwise the connection token will be
    usable with all readers. Note that location scoping only applies to
    internet-connected readers. For more details, see [the docs on
    scoping connection
    tokens](https://stripe.com/docs/terminal/fleet/locations#connection-tokens){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/connection_tokens/create>

TERMINAL LOCATION
=================

You can create, delete, list, retrieve or update terminal location

create {#create-65}
------

        my $obj = $stripe->terminal_locations( create => {
            address      =>
            {
                city        => "San Francisco",
                country     => "US",
                line1       => "1234 Main Street",
                postal_code => "94111",
            }
            display_name => "My First Store", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Terminal::Location](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}
object or a hash reference, this will create a Stripe terminal location
and return an
[Net::API::Stripe::Terminal::Location](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}
object.

Possible parameters are:

`address`

:   **Required**. The full address of the location.

`configuration_overrides`

:   The ID of a configuration that will be used to customize all readers
    in this location.

`display_name`

:   **Required**. A name for the location.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/locations/create>

delete
------

        my $obj = $stripe->terminal_locations( delete => $args ) || die( $stripe->error );

Provided with a [terminal
location](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
terminal location. It returns the terminal location object that was
deleted with its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/locations/delete>

list {#list-79}
----

        my $obj = $stripe->terminal_locations( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [terminal
location](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}
object, this issue an api call to get the list of all terminal location.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/locations/list>

retrieve {#retrieve-81}
--------

        my $obj = $stripe->terminal_locations( retrieve => $args ) || die( $stripe->error );

Provided with a [terminal
location](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}
object or a hash reference, this will retrieve a Stripe terminal
location and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/locations/retrieve>

update
------

        my $obj = $stripe->terminal_locations( update => {
            display_name => "My First Store", } ) || die( $stripe->error );

Provided with a [terminal
location](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}
object or a hash reference, this will update a Stripe terminal location
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Terminal::Location){.perl-module}

Possible parameters are:

`address`

:   The full address of the location.

`configuration_overrides`

:   The ID of a configuration that will be used to customize all readers
    in this location.

`display_name`

:   A name for the location.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/locations/update>

TERMINAL READER
===============

You can cancel\_action, create, delete, list, present\_payment\_method,
process\_payment\_intent, process\_setup\_intent, retrieve,
set\_reader\_display or update terminal reader

cancel\_action
--------------

        my $obj = $stripe->terminal_readers( cancel_action => $args ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module},
or a hash reference, this will issue a cancel\_action api call.

Returns an updated `Reader` resource.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/cancel_action>

create {#create-66}
------

        my $obj = $stripe->terminal_readers( create => {
            label             => "Blue Rabbit",
            location          => "tml_1234",
            registration_code => "puppies-plug-could", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Terminal::Reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}
object or a hash reference, this will create a Stripe terminal reader
and return an
[Net::API::Stripe::Terminal::Reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}
object.

Possible parameters are:

`label`

:   Custom label given to the reader for easier identification. If no
    label is specified, the registration code will be used.

`location`

:   **required** The location to assign the reader to.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`registration_code`

:   **Required**. A code generated by the reader used for registering to
    an account.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/create>

delete
------

        my $obj = $stripe->terminal_readers( delete => $args ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
terminal reader. It returns the terminal reader object that was deleted
with its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/delete>

list {#list-80}
----

        my $obj = $stripe->terminal_readers( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}
object, this issue an api call to get the list of all terminal reader.

Possible parameters are:

`device_type`

:   Filters readers by device type

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`location`

:   A location ID to filter the response list to only readers at the
    specific location

`serial_number`

:   Filters readers by serial number

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   A status filter to filter readers to only offline or online readers

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/list>

present\_payment\_method
------------------------

        my $obj = $stripe->terminal_readers( present_payment_method => $args ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module},
or a hash reference, this will issue a present\_payment\_method api
call.

Returns an updated `Reader` resource.

Possible parameters are:

`card_present`

:   Simulated data for the card\_present payment method

`type`

:   Simulated payment type

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/present_payment_method>

process\_payment\_intent
------------------------

        my $obj = $stripe->terminal_readers( process_payment_intent => {
            payment_intent => "pi_1DsTej2eZvKYlo2C5PX0hXuO", } ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module},
or a hash reference, this will issue a process\_payment\_intent api
call.

Returns an updated `Reader` resource.

Possible parameters are:

`payment_intent`

:   **Required**. PaymentIntent ID

`process_config`

:   Configuration overrides

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/process_payment_intent>

process\_setup\_intent
----------------------

        my $obj = $stripe->terminal_readers( process_setup_intent => {
            customer_consent_collected => "true",
            setup_intent               => "seti_1Le8Ey2eZvKYlo2CVYgmKCc3", } ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module},
or a hash reference, this will issue a process\_setup\_intent api call.

Returns an updated `Reader` resource.

Possible parameters are:

`customer_consent_collected`

:   **Required**. Customer Consent Collected

`setup_intent`

:   **Required**. SetupIntent ID

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/process_setup_intent>

retrieve {#retrieve-82}
--------

        my $obj = $stripe->terminal_readers( retrieve => $args ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}
object or a hash reference, this will retrieve a Stripe terminal reader
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/retrieve>

set\_reader\_display
--------------------

        my $obj = $stripe->terminal_readers( set_reader_display => {
            cart =>
            {
                currency   => "usd",
                line_items => [,
                    amount      => "5100",
                    description => "Red t-shirt",
                    quantity    => "1",
                ],
                tax        => "100",
                total      => "5200",
            }
            type => "cart", } ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module},
or a hash reference, this will issue a set\_reader\_display api call.

Returns an updated `Reader` resource.

Possible parameters are:

`cart`

:   Cart

`type`

:   **Required**. Type

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/set_reader_display>

update
------

        my $obj = $stripe->terminal_readers( update => {
            label => "Blue Rabbit", } ) || die( $stripe->error );

Provided with a [terminal
reader](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}
object or a hash reference, this will update a Stripe terminal reader
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Terminal::Reader){.perl-module}

Possible parameters are:

`label`

:   The new label of the reader.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/terminal/readers/update>

TEST HELPERS TEST CLOCK
=======================

You can advance, create, delete, list or retrieve test helpers test
clock

advance
-------

        my $obj = $stripe->test_helpers_test_clocks( advance => {
            frozen_time => "1662347484", } ) || die( $stripe->error );

Provided with a [test helpers test
clock](https://metacpan.org/pod/Net::API::Stripe::Billing::TestHelpersTestClock){.perl-module},
or a hash reference, this will issue a advance api call.

A `TestClock` object with status `Advancing` is returned upon success.
Otherwise, this call returns [an
error](https://stripe.com/docs/api/errors){.perl-module}.

Possible parameters are:

`frozen_time`

:   **Required**. The time to advance the test clock. Must be after the
    test clock\'s current frozen time. Cannot be more than two intervals
    in the future from the shortest subscription in this test clock. If
    there are no subscriptions in this test clock, it cannot be more
    than two years in the future.

More information from Stripe api documentation at
<https://stripe.com/docs/api/test_clocks/advance>

create {#create-67}
------

        my $obj = $stripe->test_helpers_test_clocks( create => {
            frozen_time => "1577836800", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::TestHelpersTestClock](https://metacpan.org/pod/Net::API::Stripe::Billing::TestHelpersTestClock){.perl-module}
object or a hash reference, this will create a Stripe test helpers test
clock and return an
[Net::API::Stripe::Billing::TestHelpersTestClock](https://metacpan.org/pod/Net::API::Stripe::Billing::TestHelpersTestClock){.perl-module}
object.

Possible parameters are:

`frozen_time`

:   **Required**. The initial frozen time for this test clock.

`name`

:   The name for this test clock.

More information from Stripe api documentation at
<https://stripe.com/docs/api/test_clocks/create>

delete
------

        my $obj = $stripe->test_helpers_test_clocks( delete => $args ) || die( $stripe->error );

Provided with a [test helpers test
clock](https://metacpan.org/pod/Net::API::Stripe::Billing::TestHelpersTestClock){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
test helpers test clock. It returns the test helpers test clock object
that was deleted with its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/test_clocks/delete>

list {#list-81}
----

        my $obj = $stripe->test_helpers_test_clocks( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [test helpers test
clock](https://metacpan.org/pod/Net::API::Stripe::Billing::TestHelpersTestClock){.perl-module}
object, this issue an api call to get the list of all test helpers test
clock.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/test_clocks/list>

retrieve {#retrieve-83}
--------

        my $obj = $stripe->test_helpers_test_clocks( retrieve => $args ) || die( $stripe->error );

Provided with a [test helpers test
clock](https://metacpan.org/pod/Net::API::Stripe::Billing::TestHelpersTestClock){.perl-module}
object or a hash reference, this will retrieve a Stripe test helpers
test clock and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Billing::TestHelpersTestClock){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/test_clocks/retrieve>

TOKEN
=====

You can create, create\_account, create\_bank\_account, create\_card,
create\_cvc\_update, create\_person, create\_pii or retrieve token

create\_account
---------------

        my $obj = $stripe->tokens( create_account => {
            account =>
            {
                individual             =>
                {
                    first_name => "Jane",
                    last_name  => "Doe",
                }
                tos_shown_and_accepted => "1",
            } } ) || die( $stripe->error );

Provided with a
[token](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module},
or a hash reference, this will issue a create\_account api call.

Returns the created account token if successful. Otherwise, this call
returns [an error](https://stripe.com/docs/api/errors){.perl-module}.

Possible parameters are:

`account`

:   **Required**. Information for the account this token will represent.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tokens/create_account>

create\_bank\_account
---------------------

        my $obj = $stripe->tokens( create_bank_account => {
            bank_account =>
            {
                account_holder_name => "Jenny Rosen",
                account_holder_type => "individual",
                account_number      => "000123456789",
                country             => "US",
                currency            => "usd",
                routing_number      => "110000000",
            } } ) || die( $stripe->error );

Provided with a
[token](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module},
or a hash reference, this will issue a create\_bank\_account api call.

Returns the created bank account token if successful. Otherwise, this
call returns [an
error](https://stripe.com/docs/api/errors){.perl-module}.

Possible parameters are:

`bank_account`

:   The bank account this token will represent.

`customer`

:   The customer (owned by the application\'s account) for which to
    create a token. This can be used only with an [OAuth access
    token](https://stripe.com/docs/connect/standard-accounts){.perl-module}
    or [Stripe-Account
    header](https://stripe.com/docs/connect/authentication){.perl-module}.
    For more details, see [Cloning Saved Payment
    Methods](https://stripe.com/docs/connect/cloning-saved-payment-methods){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tokens/create_bank_account>

create\_card
------------

        my $obj = $stripe->tokens( create_card => {
            card =>
            {
                cvc       => "314",
                exp_month => "9",
                exp_year  => "2023",
                number    => "4242424242424242",
            } } ) || die( $stripe->error );

Provided with a
[token](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module},
or a hash reference, this will issue a create\_card api call.

Returns the created card token if successful. Otherwise, this call
raises [an error](https://stripe.com/docs/api/errors){.perl-module}.

Possible parameters are:

`card`

:   The card this token will represent. If you also pass in a customer,
    the card must be the ID of a card belonging to the customer.
    Otherwise, if you do not pass in a customer, this is a dictionary
    containing a user\'s credit card details, with the options described
    below.

`customer`

:   The customer (owned by the application\'s account) for which to
    create a token. Also, this can be used only with an [OAuth access
    token](https://stripe.com/docs/connect/standard-accounts){.perl-module}
    or [Stripe-Account
    header](https://stripe.com/docs/connect/authentication){.perl-module}.
    For more details, see [Cloning Saved Payment
    Methods](https://stripe.com/docs/connect/cloning-saved-payment-methods){.perl-module}.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tokens/create_card>

create\_cvc\_update
-------------------

        my $obj = $stripe->tokens( create_cvc_update => {
            cvc_update =>
            {
                cvc => "123",
            } } ) || die( $stripe->error );

Provided with a
[token](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module},
or a hash reference, this will issue a create\_cvc\_update api call.

Returns the created CVC update token if successful. Otherwise, this call
raises [an error](https://stripe.com/docs/api/errors){.perl-module}.

Possible parameters are:

`cvc_update`

:   **Required**. The updated CVC value this token will represent.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tokens/create_cvc_update>

create\_person
--------------

        my $obj = $stripe->tokens( create_person => {
            person =>
            {
                first_name   => "Jane",
                last_name    => "Doe",
                relationship =>
                {
                    owner => "1",
                }
            } } ) || die( $stripe->error );

Provided with a
[token](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module},
or a hash reference, this will issue a create\_person api call.

Returns the created person token if successful. Otherwise, this call
returns [an error](https://stripe.com/docs/api/errors){.perl-module}.

Possible parameters are:

`person`

:   **Required**. Information for the person this token will represent.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tokens/create_person>

create\_pii
-----------

        my $obj = $stripe->tokens( create_pii => {
            pii =>
            {
                id_number => "000000000",
            } } ) || die( $stripe->error );

Provided with a
[token](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module},
or a hash reference, this will issue a create\_pii api call.

Returns the created PII token if successful. Otherwise, this call
returns [an error](https://stripe.com/docs/api/errors){.perl-module}.

Possible parameters are:

`pii`

:   **Required**. The PII this token will represent.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tokens/create_pii>

retrieve {#retrieve-84}
--------

        my $obj = $stripe->tokens( retrieve => $args ) || die( $stripe->error );

Provided with a
[token](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module}
object or a hash reference, this will retrieve a Stripe token and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Token){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/tokens/retrieve>

TOPUP
=====

You can cancel, create, list, retrieve or update topup

cancel
------

        my $obj = $stripe->topups( cancel => $args ) || die( $stripe->error );

Provided with a
[topup](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns the canceled top-up. If the top-up is already canceled or can't
be canceled, an error is returned.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/topups/cancel>

create {#create-68}
------

        my $obj = $stripe->topups( create => {
            amount               => "2000",
            currency             => "usd",
            description          => "Top-up for Jenny Rosen",
            statement_descriptor => "Top-up", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::TopUp](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module}
object or a hash reference, this will create a Stripe topup and return
an
[Net::API::Stripe::Connect::TopUp](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module}
object.

Possible parameters are:

`amount`

:   **Required**. A positive integer representing how much to transfer.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`source`

:   The ID of a source to transfer funds from. For most users, this
    should be left unspecified which will use the bank account that was
    set up in the dashboard for the specified currency. In test mode,
    this can be a test bank token (see [Testing
    Top-ups](https://stripe.com/docs/connect/testing#testing-top-ups){.perl-module}).

`statement_descriptor`

:   Extra information about a top-up for the source\'s bank statement.
    Limited to 15 ASCII characters.

`transfer_group`

:   A string that identifies this top-up as part of a group.

More information from Stripe api documentation at
<https://stripe.com/docs/api/topups/create>

list {#list-82}
----

        my $obj = $stripe->topups( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[topup](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module}
object, this issue an api call to get the list of all topup.

Possible parameters are:

`amount`

:   A filter on the list based on the object `amount` field. The value
    can be a string with an integer amount, or it can be a dictionary
    with the following options:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return top-ups that have the given status. One of `canceled`,
    `failed`, `pending` or `succeeded`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/topups/list>

retrieve {#retrieve-85}
--------

        my $obj = $stripe->topups( retrieve => $args ) || die( $stripe->error );

Provided with a
[topup](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module}
object or a hash reference, this will retrieve a Stripe topup and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/topups/retrieve>

update
------

        my $obj = $stripe->topups( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[topup](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module}
object or a hash reference, this will update a Stripe topup and return
its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::TopUp){.perl-module}

Possible parameters are:

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/topups/update>

TRANSFER
========

You can create, list, retrieve or update transfer

create {#create-69}
------

        my $obj = $stripe->transfers( create => {
            amount         => "400",
            currency       => "usd",
            destination    => "acct_1032D82eZvKYlo2C",
            transfer_group => "ORDER_95", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::Transfer](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer){.perl-module}
object or a hash reference, this will create a Stripe transfer and
return an
[Net::API::Stripe::Connect::Transfer](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer){.perl-module}
object.

Possible parameters are:

`amount`

:   **required** A positive integer in JPY representing how much to
    transfer.

`currency`

:   **Required**. 3-letter [ISO code for
    currency](https://stripe.com/docs/payouts){.perl-module}.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`destination`

:   **Required**. The ID of a connected Stripe account. \<a
    href=\"/docs/connect/charges-transfers\"\>See the Connect
    documentation\</a\> for details.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`source_transaction`

:   You can use this parameter to transfer funds from a charge before
    they are added to your available balance. A pending balance will
    transfer immediately but the funds will not become available until
    the original charge becomes available. [See the Connect
    documentation](https://stripe.com/docs/connect/charges-transfers#transfer-availability){.perl-module}
    for details.

`source_type`

:   The source balance to use for this transfer. One of `bank_account`,
    `card`, or `fpx`. For most users, this will default to `card`.

`transfer_group`

:   A string that identifies this transaction as part of a group. See
    the [Connect
    documentation](https://stripe.com/docs/connect/charges-transfers#transfer-options){.perl-module}
    for details.

More information from Stripe api documentation at
<https://stripe.com/docs/api/transfers/create>

list {#list-83}
----

        my $obj = $stripe->transfers( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a
[transfer](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer){.perl-module}
object, this issue an api call to get the list of all transfer.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`destination`

:   Only return transfers for the destination specified by this account
    ID.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`transfer_group`

:   Only return transfers with the specified transfer group.

More information from Stripe api documentation at
<https://stripe.com/docs/api/transfers/list>

retrieve {#retrieve-86}
--------

        my $obj = $stripe->transfers( retrieve => $args ) || die( $stripe->error );

Provided with a
[transfer](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer){.perl-module}
object or a hash reference, this will retrieve a Stripe transfer and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/transfers/retrieve>

update
------

        my $obj = $stripe->transfers( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a
[transfer](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer){.perl-module}
object or a hash reference, this will update a Stripe transfer and
return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer){.perl-module}

Possible parameters are:

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/transfers/update>

TRANSFER REVERSAL
=================

You can create, list, retrieve or update transfer reversal

create {#create-70}
------

        my $obj = $stripe->transfer_reversals( create => {
            amount => "100", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Connect::Transfer::Reversal](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer::Reversal){.perl-module}
object or a hash reference, this will create a Stripe transfer reversal
and return an
[Net::API::Stripe::Connect::Transfer::Reversal](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer::Reversal){.perl-module}
object.

Possible parameters are:

`amount`

:   A positive integer in JPY representing how much of this transfer to
    reverse. Can only reverse up to the unreversed amount remaining of
    the transfer. Partial transfer reversals are only allowed for
    transfers to Stripe Accounts. Defaults to the entire transfer
    amount.

`description`

:   An arbitrary string which you can attach to a reversal object. It is
    displayed alongside the reversal in the Dashboard. This will be
    unset if you POST an empty value.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`refund_application_fee`

:   Boolean indicating whether the application fee should be refunded
    when reversing this transfer. If a full transfer reversal is given,
    the full application fee will be refunded. Otherwise, the
    application fee will be refunded with an amount proportional to the
    amount of the transfer reversed.

More information from Stripe api documentation at
<https://stripe.com/docs/api/transfer_reversals/create>

list {#list-84}
----

        my $obj = $stripe->transfer_reversals( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [transfer
reversal](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer::Reversal){.perl-module}
object, this issue an api call to get the list of all transfer reversal.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/transfer_reversals/list>

retrieve {#retrieve-87}
--------

        my $obj = $stripe->transfer_reversals( retrieve => $args ) || die( $stripe->error );

Provided with a [transfer
reversal](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer::Reversal){.perl-module}
object or a hash reference, this will retrieve a Stripe transfer
reversal and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer::Reversal){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/transfer_reversals/retrieve>

update
------

        my $obj = $stripe->transfer_reversals( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [transfer
reversal](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer::Reversal){.perl-module}
object or a hash reference, this will update a Stripe transfer reversal
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Connect::Transfer::Reversal){.perl-module}

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/transfer_reversals/update>

TREASURY CREDIT REVERSAL
========================

You can create, list or retrieve treasury credit reversal

create {#create-71}
------

        my $obj = $stripe->treasury_credit_reversals( create => {
            received_credit => "rc_1Le9F42eZvKYlo2CM2wIU5bz", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Treasury::CreditReversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::CreditReversal){.perl-module}
object or a hash reference, this will create a Stripe treasury credit
reversal and return an
[Net::API::Stripe::Treasury::CreditReversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::CreditReversal){.perl-module}
object.

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`received_credit`

:   **Required**. The ReceivedCredit to reverse.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/credit_reversals/create>

list {#list-85}
----

        my $obj = $stripe->treasury_credit_reversals( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury credit
reversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::CreditReversal){.perl-module}
object, this issue an api call to get the list of all treasury credit
reversal.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. Returns objects associated with this FinancialAccount.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`received_credit`

:   Only return CreditReversals for the ReceivedCredit ID.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return CreditReversals for a given status.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/credit_reversals/list>

retrieve {#retrieve-88}
--------

        my $obj = $stripe->treasury_credit_reversals( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury credit
reversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::CreditReversal){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury credit
reversal and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::CreditReversal){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/credit_reversals/retrieve>

TREASURY DEBIT REVERSAL
=======================

You can create, list or retrieve treasury debit reversal

create {#create-72}
------

        my $obj = $stripe->treasury_debit_reversals( create => {
            received_debit => "rd_1Le9F42eZvKYlo2C0TIJJqNP", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Treasury::DebitReversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::DebitReversal){.perl-module}
object or a hash reference, this will create a Stripe treasury debit
reversal and return an
[Net::API::Stripe::Treasury::DebitReversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::DebitReversal){.perl-module}
object.

Possible parameters are:

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`received_debit`

:   **Required**. The ReceivedDebit to reverse.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/debit_reversals/create>

list {#list-86}
----

        my $obj = $stripe->treasury_debit_reversals( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury debit
reversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::DebitReversal){.perl-module}
object, this issue an api call to get the list of all treasury debit
reversal.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. Returns objects associated with this FinancialAccount.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`received_debit`

:   Only return DebitReversals for the ReceivedDebit ID.

`resolution`

:   Only return DebitReversals for a given resolution.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return DebitReversals for a given status.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/debit_reversals/list>

retrieve {#retrieve-89}
--------

        my $obj = $stripe->treasury_debit_reversals( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury debit
reversal](https://metacpan.org/pod/Net::API::Stripe::Treasury::DebitReversal){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury debit
reversal and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::DebitReversal){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/debit_reversals/retrieve>

TREASURY FINANCIAL ACCOUNT
==========================

You can create, list, retrieve or update treasury financial account

create {#create-73}
------

        my $obj = $stripe->treasury_financial_accounts( create => {
            features             =>
            {
            }
            supported_currencies => [qw( usd )], } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Treasury::FinancialAccount](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccount){.perl-module}
object or a hash reference, this will create a Stripe treasury financial
account and return an
[Net::API::Stripe::Treasury::FinancialAccount](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccount){.perl-module}
object.

Possible parameters are:

`features`

:   Encodes whether a FinancialAccount has access to a particular
    feature. Stripe or the platform can control features via the
    requested field.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`platform_restrictions`

:   The set of functionalities that the platform can restrict on the
    FinancialAccount.

`supported_currencies`

:   **Required**. The currencies the FinancialAccount can hold a balance
    in.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/financial_accounts/create>

list {#list-87}
----

        my $obj = $stripe->treasury_financial_accounts( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [treasury financial
account](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccount){.perl-module}
object, this issue an api call to get the list of all treasury financial
account.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   An object ID cursor for use in pagination.

`limit`

:   A limit ranging from 1 to 100 (defaults to 10).

`starting_after`

:   An object ID cursor for use in pagination.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/financial_accounts/list>

retrieve {#retrieve-90}
--------

        my $obj = $stripe->treasury_financial_accounts( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury financial
account](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccount){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury
financial account and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccount){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/financial_accounts/retrieve>

update
------

        my $obj = $stripe->treasury_financial_accounts( update => {
            metadata =>
            {
                order_id => "6735",
            } } ) || die( $stripe->error );

Provided with a [treasury financial
account](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccount){.perl-module}
object or a hash reference, this will update a Stripe treasury financial
account and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccount){.perl-module}

Possible parameters are:

`features`

:   Encodes whether a FinancialAccount has access to a particular
    feature, with a status enum and associated `status_details`. Stripe
    or the platform may control features via the requested field.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`platform_restrictions`

:   The set of functionalities that the platform can restrict on the
    FinancialAccount.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/financial_accounts/update>

TREASURY FINANCIAL ACCOUNT FEATURES
===================================

You can retrieve or update treasury financial account features

retrieve {#retrieve-91}
--------

        my $obj = $stripe->treasury_financial_account_featuress( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury financial account
features](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccountFeatures){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury
financial account features and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccountFeatures){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/financial_account_features/retrieve>

update
------

        my $obj = $stripe->treasury_financial_account_featuress( update => {
            card_issuing =>
            {
                requested => "0",
            } } ) || die( $stripe->error );

Provided with a [treasury financial account
features](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccountFeatures){.perl-module}
object or a hash reference, this will update a Stripe treasury financial
account features and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::FinancialAccountFeatures){.perl-module}

Possible parameters are:

`card_issuing`

:   Encodes the FinancialAccount\'s ability to be used with the Issuing
    product, including attaching cards to and drawing funds from the
    FinancialAccount.

`deposit_insurance`

:   Represents whether this FinancialAccount is eligible for deposit
    insurance. Various factors determine the insurance amount.

`financial_addresses`

:   Contains Features that add FinancialAddresses to the
    FinancialAccount.

`inbound_transfers`

:   Contains settings related to adding funds to a FinancialAccount from
    another Account with the same owner.

`intra_stripe_flows`

:   Represents the ability for the FinancialAccount to send money to, or
    receive money from other FinancialAccounts (for example, via
    OutboundPayment).

`outbound_payments`

:   Includes Features related to initiating money movement out of the
    FinancialAccount to someone else\'s bucket of money.

`outbound_transfers`

:   Contains a Feature and settings related to moving money out of the
    FinancialAccount into another Account with the same owner.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/financial_account_features/update>

TREASURY INBOUND TRANSFER
=========================

You can cancel, create, fail, list, retrieve, return or succeed treasury
inbound transfer

cancel
------

        my $obj = $stripe->treasury_inbound_transfers( cancel => $args ) || die( $stripe->error );

Provided with a [treasury inbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns the InboundTransfer object if the cancellation succeeded.
Returns an error if the InboundTransfer has already been canceled or
cannot be canceled.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/inbound_transfers/cancel>

create {#create-74}
------

        my $obj = $stripe->treasury_inbound_transfers( create => {
            amount                => "10000",
            currency              => "usd",
            description           => "InboundTransfer from my bank account",
            financial_account     => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            origin_payment_method => "pm_1KMDdkGPnV27VyGeAgGz8bsi", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Treasury::InboundTransfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module}
object or a hash reference, this will create a Stripe treasury inbound
transfer and return an
[Net::API::Stripe::Treasury::InboundTransfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module}
object.

Possible parameters are:

`amount`

:   **Required**. Amount (in cents) to be transferred.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`financial_account`

:   **Required**. The FinancialAccount to send funds to.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`origin_payment_method`

:   **Required**. The origin payment method to be debited for the
    InboundTransfer.

`statement_descriptor`

:   The complete description that appears on your customers\'
    statements. Maximum 10 characters.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/inbound_transfers/create>

fail
----

        my $obj = $stripe->treasury_inbound_transfers( fail => {
            failure_details =>
            {
                code => "insufficient_funds",
            } } ) || die( $stripe->error );

Provided with a [treasury inbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module},
or a hash reference, this will issue a fail api call.

Returns the InboundTransfer object in the returned state. Returns an
error if the InboundTransfer has already failed or cannot be failed.

Possible parameters are:

`failure_details`

:   Details about a failed InboundTransfer.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/inbound_transfers/test_mode_fail>

list {#list-88}
----

        my $obj = $stripe->treasury_inbound_transfers( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury inbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module}
object, this issue an api call to get the list of all treasury inbound
transfer.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. Returns objects associated with this FinancialAccount.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return InboundTransfers that have the given status:
    `processing`, `succeeded`, `failed` or `canceled`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/inbound_transfers/list>

retrieve {#retrieve-92}
--------

        my $obj = $stripe->treasury_inbound_transfers( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury inbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury inbound
transfer and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/inbound_transfers/retrieve>

return
------

        my $obj = $stripe->treasury_inbound_transfers( return => $args ) || die( $stripe->error );

Provided with a [treasury inbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module},
or a hash reference, this will issue a return api call.

Returns the InboundTransfer object with `returned` set to `true`.
Returns an error if the InboundTransfer has already been returned or
cannot be returned.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/inbound_transfers/test_mode_return>

succeed
-------

        my $obj = $stripe->treasury_inbound_transfers( succeed => $args ) || die( $stripe->error );

Provided with a [treasury inbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::InboundTransfer){.perl-module},
or a hash reference, this will issue a succeed api call.

Returns the InboundTransfer object in the succeeded state. Returns an
error if the InboundTransfer has already succeeded or cannot be
succeeded.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/inbound_transfers/test_mode_succeed>

TREASURY OUTBOUND PAYMENT
=========================

You can cancel, create, fail, list, post, retrieve or return treasury
outbound payment

cancel
------

        my $obj = $stripe->treasury_outbound_payments( cancel => $args ) || die( $stripe->error );

Provided with a [treasury outbound
payment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns the OutboundPayment object if the cancellation succeeded.
Returns an error if the OutboundPayment has already been canceled or
cannot be canceled.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_payments/cancel>

create {#create-75}
------

        my $obj = $stripe->treasury_outbound_payments( create => {
            amount                     => "10000",
            currency                   => "usd",
            customer                   => "cus_AJ78ZaALpqgiuZ",
            description                => "OutboundPayment to a 3rd party",
            destination_payment_method => "pm_1Le9F42eZvKYlo2Cy9Yph0k5",
            financial_account          => "fa_1Le9F32eZvKYlo2CjbQcDQUE", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Treasury::OutboundPayment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module}
object or a hash reference, this will create a Stripe treasury outbound
payment and return an
[Net::API::Stripe::Treasury::OutboundPayment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module}
object.

Possible parameters are:

`amount`

:   **Required**. Amount (in cents) to be transferred.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`customer`

:   ID of the customer to whom the OutboundPayment is sent. Must match
    the Customer attached to the `destination_payment_method` passed in.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`destination_payment_method`

:   The PaymentMethod to use as the payment instrument for the
    OutboundPayment. Exclusive with `destination_payment_method_data`.

`destination_payment_method_data`

:   Hash used to generate the PaymentMethod to be used for this
    OutboundPayment. Exclusive with `destination_payment_method`.

`destination_payment_method_options`

:   Payment method-specific configuration for this OutboundPayment.

`end_user_details`

:   End user details.

`financial_account`

:   **Required**. The FinancialAccount to pull funds from.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`statement_descriptor`

:   The description that appears on the receiving end for this
    OutboundPayment (for example, bank statement for external bank
    transfer). Maximum 10 characters for `ach` payments, 140 characters
    for `wire` payments, or 500 characters for `stripe` network
    transfers. The default value is `payment`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_payments/create>

fail
----

        my $obj = $stripe->treasury_outbound_payments( fail => $args ) || die( $stripe->error );

Provided with a [treasury outbound
payment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module},
or a hash reference, this will issue a fail api call.

Returns the OutboundPayment object in the failed state. Returns an error
if the OutboundPayment has already been failed or cannot be failed.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_payments/test_mode_fail>

list {#list-89}
----

        my $obj = $stripe->treasury_outbound_payments( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury outbound
payment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module}
object, this issue an api call to get the list of all treasury outbound
payment.

Possible parameters are:

`customer`

:   Only return OutboundPayments sent to this customer.

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. Returns objects associated with this FinancialAccount.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return OutboundPayments that have the given status:
    `processing`, `failed`, `posted`, `returned`, or `canceled`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_payments/list>

post
----

        my $obj = $stripe->treasury_outbound_payments( post => $args ) || die( $stripe->error );

Provided with a [treasury outbound
payment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module},
or a hash reference, this will issue a post api call.

Returns the OutboundPayment object in the posted state. Returns an error
if the OutboundPayment has already been posted or cannot be posted.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_payments/test_mode_post>

retrieve {#retrieve-93}
--------

        my $obj = $stripe->treasury_outbound_payments( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury outbound
payment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury
outbound payment and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_payments/retrieve>

return
------

        my $obj = $stripe->treasury_outbound_payments( return => {
            return_details =>
            {
                code => "account_closed",
            } } ) || die( $stripe->error );

Provided with a [treasury outbound
payment](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundPayment){.perl-module},
or a hash reference, this will issue a return api call.

Returns the OutboundPayment object in the returned state. Returns an
error if the OutboundPayment has already been returned or cannot be
returned.

Possible parameters are:

`returned_details`

:   Optional hash to set the the return code.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_payments/test_mode_return>

TREASURY OUTBOUND TRANSFER
==========================

You can cancel, create, fail, list, post, retrieve or return treasury
outbound transfer

cancel
------

        my $obj = $stripe->treasury_outbound_transfers( cancel => $args ) || die( $stripe->error );

Provided with a [treasury outbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module},
or a hash reference, this will issue a cancel api call.

Returns the OutboundTransfer object if the cancellation succeeded.
Returns an error if the object has already been canceled or cannot be
canceled.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_transfers/cancel>

create {#create-76}
------

        my $obj = $stripe->treasury_outbound_transfers( create => {
            amount                     => "500",
            currency                   => "usd",
            description                => "OutboundTransfer to my external bank account",
            destination_payment_method => "pm_1234567890",
            financial_account          => "fa_1Le9F32eZvKYlo2CjbQcDQUE", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Treasury::OutboundTransfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module}
object or a hash reference, this will create a Stripe treasury outbound
transfer and return an
[Net::API::Stripe::Treasury::OutboundTransfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module}
object.

Possible parameters are:

`amount`

:   **Required**. Amount (in cents) to be transferred.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`destination_payment_method`

:   The PaymentMethod to use as the payment instrument for the
    OutboundTransfer.

`destination_payment_method_options`

:   Hash describing payment method configuration details.

`financial_account`

:   **Required**. The FinancialAccount to pull funds from.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`statement_descriptor`

:   Statement descriptor to be shown on the receiving end of an
    OutboundTransfer. Maximum 10 characters for `ach` transfers or 140
    characters for `wire` transfers. The default value is `transfer`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_transfers/create>

fail
----

        my $obj = $stripe->treasury_outbound_transfers( fail => $args ) || die( $stripe->error );

Provided with a [treasury outbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module},
or a hash reference, this will issue a fail api call.

Returns the OutboundTransfer object in the failed state. Returns an
error if the OutboundTransfer has already been failed or cannot be
failed.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_transfers/test_mode_fail>

list {#list-90}
----

        my $obj = $stripe->treasury_outbound_transfers( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury outbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module}
object, this issue an api call to get the list of all treasury outbound
transfer.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. Returns objects associated with this FinancialAccount.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return OutboundTransfers that have the given status:
    `processing`, `canceled`, `failed`, `posted`, or `returned`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_transfers/list>

post
----

        my $obj = $stripe->treasury_outbound_transfers( post => $args ) || die( $stripe->error );

Provided with a [treasury outbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module},
or a hash reference, this will issue a post api call.

Returns the OutboundTransfer object in the posted state. Returns an
error if the OutboundTransfer has already been posted or cannot be
posted.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_transfers/test_mode_post>

retrieve {#retrieve-94}
--------

        my $obj = $stripe->treasury_outbound_transfers( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury outbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury
outbound transfer and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_transfers/retrieve>

return
------

        my $obj = $stripe->treasury_outbound_transfers( return => {
            code => "declined", } ) || die( $stripe->error );

Provided with a [treasury outbound
transfer](https://metacpan.org/pod/Net::API::Stripe::Treasury::OutboundTransfer){.perl-module},
or a hash reference, this will issue a return api call.

Returns the OutboundTransfer object in the returned state. Returns an
error if the OutboundTransfer has already been returned or cannot be
returned.

Possible parameters are:

`returned_details`

:   Details about a returned OutboundTransfer.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/outbound_transfers/test_mode_return>

TREASURY RECEIVED CREDIT
========================

You can list, received\_credit or retrieve treasury received credit

list {#list-91}
----

        my $obj = $stripe->treasury_received_credits( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury received
credit](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedCredit){.perl-module}
object, this issue an api call to get the list of all treasury received
credit.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. The FinancialAccount that received the funds.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`linked_flows`

:   Only return ReceivedCredits described by the flow.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return ReceivedCredits that have the given status: `succeeded`
    or `failed`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/received_credits/list>

received\_credit
----------------

        my $obj = $stripe->treasury_received_credits( received_credit => {
            amount            => "1000",
            currency          => "usd",
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            network           => "ach", } ) || die( $stripe->error );

Provided with a [treasury received
credit](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedCredit){.perl-module},
or a hash reference, this will issue a received\_credit api call.

A test mode ReceivedCredit object.

Possible parameters are:

`amount`

:   **Required**. Amount (in cents) to be transferred.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`financial_account`

:   **Required**. The FinancialAccount to send funds to.

`initiating_payment_method_details`

:   Initiating payment method details for the object.

`network`

:   **Required**. The rails used for the object.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/received_credits/test_mode_create>

retrieve {#retrieve-95}
--------

        my $obj = $stripe->treasury_received_credits( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury received
credit](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedCredit){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury
received credit and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedCredit){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/received_credits/retrieve>

TREASURY RECEIVED DEBIT
=======================

You can list, received\_debit or retrieve treasury received debit

list {#list-92}
----

        my $obj = $stripe->treasury_received_debits( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury received
debit](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedDebit){.perl-module}
object, this issue an api call to get the list of all treasury received
debit.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. The FinancialAccount that funds were pulled from.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return ReceivedDebits that have the given status: `succeeded`
    or `failed`.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/received_debits/list>

received\_debit
---------------

        my $obj = $stripe->treasury_received_debits( received_debit => {
            amount            => "1000",
            currency          => "usd",
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            network           => "ach", } ) || die( $stripe->error );

Provided with a [treasury received
debit](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedDebit){.perl-module},
or a hash reference, this will issue a received\_debit api call.

A test mode ReceivedDebit object.

Possible parameters are:

`amount`

:   **Required**. Amount (in cents) to be transferred.

`currency`

:   **Required**. Three-letter [ISO currency
    code](https://www.iso.org/iso-4217-currency-codes.html){.perl-module},
    in lowercase. Must be a [supported
    currency](https://stripe.com/docs/currencies){.perl-module}.

`description`

:   An arbitrary string attached to the object. Often useful for
    displaying to users.

`financial_account`

:   **Required**. The FinancialAccount to pull funds from.

`initiating_payment_method_details`

:   Initiating payment method details for the object.

`network`

:   **Required**. The rails used for the object.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/received_debits/test_mode_create>

retrieve {#retrieve-96}
--------

        my $obj = $stripe->treasury_received_debits( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury received
debit](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedDebit){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury
received debit and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::ReceivedDebit){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/received_debits/retrieve>

TREASURY TRANSACTION
====================

You can list or retrieve treasury transaction

list {#list-93}
----

        my $obj = $stripe->treasury_transactions( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury
transaction](https://metacpan.org/pod/Net::API::Stripe::Treasury::Transaction){.perl-module}
object, this issue an api call to get the list of all treasury
transaction.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. Returns objects associated with this FinancialAccount.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`order_by`

:   The results are in reverse chronological order by `created` or
    `posted_at`. The default is `created`.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`status`

:   Only return Transactions that have the given status: `open`,
    `posted`, or `void`.

`status_transitions`

:   A filter for the `status_transitions.posted_at` timestamp. When
    using this filter, `status=posted` and `order_by=posted_at` must
    also be specified.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/transactions/list>

retrieve {#retrieve-97}
--------

        my $obj = $stripe->treasury_transactions( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury
transaction](https://metacpan.org/pod/Net::API::Stripe::Treasury::Transaction){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury
transaction and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::Transaction){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/transactions/retrieve>

TREASURY TRANSACTION ENTRY
==========================

You can list or retrieve treasury transaction entry

list {#list-94}
----

        my $obj = $stripe->treasury_transaction_entrys( list => {
            financial_account => "fa_1Le9F32eZvKYlo2CjbQcDQUE",
            limit             => "3", } ) || die( $stripe->error );

Provided with a [treasury transaction
entry](https://metacpan.org/pod/Net::API::Stripe::Treasury::TransactionEntry){.perl-module}
object, this issue an api call to get the list of all treasury
transaction entry.

Possible parameters are:

`created`

:   A filter on the list based on the object `created` field. The value
    can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`effective_at`

:   A filter on the list based on the object `effective_at` field. The
    value can be a string with an integer Unix timestamp, or it can be a
    dictionary with the following options:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`financial_account`

:   **Required**. Returns objects associated with this FinancialAccount.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`order_by`

:   The results are in reverse chronological order by `created` or
    `effective_at`. The default is `created`.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

`transaction`

:   Only return TransactionEntries associated with this Transaction.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/transaction_entries/list>

retrieve {#retrieve-98}
--------

        my $obj = $stripe->treasury_transaction_entrys( retrieve => $args ) || die( $stripe->error );

Provided with a [treasury transaction
entry](https://metacpan.org/pod/Net::API::Stripe::Treasury::TransactionEntry){.perl-module}
object or a hash reference, this will retrieve a Stripe treasury
transaction entry and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::Treasury::TransactionEntry){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/treasury/transaction_entries/retrieve>

USAGE RECORD
============

You can create or list usage record

create {#create-77}
------

        my $obj = $stripe->usage_records( create => {
            quantity  => "100",
            timestamp => "1571252444", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::Billing::UsageRecord](https://metacpan.org/pod/Net::API::Stripe::Billing::UsageRecord){.perl-module}
object or a hash reference, this will create a Stripe usage record and
return an
[Net::API::Stripe::Billing::UsageRecord](https://metacpan.org/pod/Net::API::Stripe::Billing::UsageRecord){.perl-module}
object.

Possible parameters are:

`action`

:   Valid values are `increment` (default) or `set`. When using
    `increment` the specified `quantity` will be added to the usage at
    the specified timestamp. The `set` action will overwrite the usage
    quantity at that timestamp. If the subscription has [billing
    thresholds](https://stripe.com/docs/api/subscriptions/object#subscription_object-billing_thresholds){.perl-module},
    `increment` is the only allowed value.

`quantity`

:   **Required**. The usage quantity for the specified timestamp.

`timestamp`

:   The timestamp for the usage event. This timestamp must be within the
    current billing period of the subscription of the provided
    `subscription_item`, and must not be in the future. When passing
    `"now"`, Stripe records usage for the current time. Default is
    `"now"` if a value is not provided.

More information from Stripe api documentation at
<https://stripe.com/docs/api/usage_records/create>

list {#list-95}
----

        my $obj = $stripe->usage_records( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [usage
record](https://metacpan.org/pod/Net::API::Stripe::Billing::UsageRecord){.perl-module}
object, this issue an api call to get the list of all usage record.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/usage_records/subscription_item_summary_list>

WEBHOOK ENDPOINT
================

You can create, delete, list, retrieve or update webhook endpoint

create {#create-78}
------

        my $obj = $stripe->webhook_endpoints( create => {
            enabled_events => [qw( charge.failed charge.succeeded )],
            url            => "https://example.com/my/webhook/endpoint", } ) || die( $stripe->error );

Provided with a
[Net::API::Stripe::WebHook::Object](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}
object or a hash reference, this will create a Stripe webhook endpoint
and return an
[Net::API::Stripe::WebHook::Object](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}
object.

Possible parameters are:

`api_version`

:   Events sent to this endpoint will be generated with this Stripe
    Version instead of your account\'s default Stripe Version.

`connect`

:   Whether this endpoint should receive events from connected accounts
    (`true`), or from your account (`false`). Defaults to `false`.

`description`

:   An optional description of what the webhook is used for.

`enabled_events`

:   **Required**. The list of events to enable for this endpoint. You
    may specify `['*']` to enable all events, except those that require
    explicit selection.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`url`

:   **Required**. The URL of the webhook endpoint.

More information from Stripe api documentation at
<https://stripe.com/docs/api/webhook_endpoints/create>

delete
------

        my $obj = $stripe->webhook_endpoints( delete => $args ) || die( $stripe->error );

Provided with a [webhook
endpoint](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module},
or a hash reference, this will issue an api call to Stripe to remove the
webhook endpoint. It returns the webhook endpoint object that was
deleted with its property `deleted` set to true.

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/webhook_endpoints/delete>

list {#list-96}
----

        my $obj = $stripe->webhook_endpoints( list => {
            limit => "3", } ) || die( $stripe->error );

Provided with a [webhook
endpoint](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}
object, this issue an api call to get the list of all webhook endpoint.

Possible parameters are:

`ending_before`

:   A cursor for use in pagination. `ending_before` is an object ID that
    defines your place in the list. For instance, if you make a list
    request and receive 100 objects, starting with `obj_bar`, your
    subsequent call can include `ending_before=obj_bar` in order to
    fetch the previous page of the list.

`limit`

:   A limit on the number of objects to be returned. Limit can range
    between 1 and 100, and the default is 10.

`starting_after`

:   A cursor for use in pagination. `starting_after` is an object ID
    that defines your place in the list. For instance, if you make a
    list request and receive 100 objects, ending with `obj_foo`, your
    subsequent call can include `starting_after=obj_foo` in order to
    fetch the next page of the list.

More information from Stripe api documentation at
<https://stripe.com/docs/api/webhook_endpoints/list>

retrieve {#retrieve-99}
--------

        my $obj = $stripe->webhook_endpoints( retrieve => $args ) || die( $stripe->error );

Provided with a [webhook
endpoint](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}
object or a hash reference, this will retrieve a Stripe webhook endpoint
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}

There are no query parameter.

More information from Stripe api documentation at
<https://stripe.com/docs/api/webhook_endpoints/retrieve>

update
------

        my $obj = $stripe->webhook_endpoints( update => {
            url => "https://example.com/new_endpoint", } ) || die( $stripe->error );

Provided with a [webhook
endpoint](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}
object or a hash reference, this will update a Stripe webhook endpoint
and return its corresponding
[object](https://metacpan.org/pod/Net::API::Stripe::WebHook::Object){.perl-module}

Possible parameters are:

`description`

:   An optional description of what the webhook is used for.

`disabled`

:   Disable the webhook endpoint if set to true.

`enabled_events`

:   The list of events to enable for this endpoint. You may specify
    `['*']` to enable all events, except those that require explicit
    selection.

`metadata`

:   Set of [key-value
    pairs](https://stripe.com/docs/api/metadata){.perl-module} that you
    can attach to an object. This can be useful for storing additional
    information about the object in a structured format. Individual keys
    can be unset by posting an empty value to them. All keys can be
    unset by posting an empty value to `metadata`.

`url`

:   The URL of the webhook endpoint.

More information from Stripe api documentation at
<https://stripe.com/docs/api/webhook_endpoints/update>

API SAMPLE
==========

        {
          "object": "balance",
          "available": [
            {
              "amount": 0,
              "currency": "jpy",
              "source_types": {
                "card": 0
              }
            }
          ],
          "connect_reserved": [
            {
              "amount": 0,
              "currency": "jpy"
            }
          ],
          "livemode": false,
          "pending": [
            {
              "amount": 7712,
              "currency": "jpy",
              "source_types": {
                "card": 7712
              }
            }
          ]
        }

HISTORY
=======

<https://stripe.com/docs/upgrades> for Stripe API version history.

AUTHOR
======

Jacques Deguest \<`jack@deguest.jp`{classes="ARRAY(0x55bbe6b0ed60)"}\>

SEE ALSO
========

Stripe API documentation:

<https://stripe.com/docs/api>

List of server-side libraries:
<https://stripe.com/docs/libraries#server-side-libraries>

[Net::Stripe](https://metacpan.org/pod/Net::Stripe){.perl-module},
another Stripe API, but which uses Moose

COPYRIGHT & LICENSE
===================

Copyright (c) 2018-2019 DEGUEST Pte. Ltd.

You can use, copy, modify and redistribute this package and associated
files under the same terms as Perl itself.
