package SLUB::LZA::Rosetta::TA::Command::search;
use SLUB::LZA::Rosetta::TA -command;
use strict;
use warnings;
use feature qw(say);
use SLUB::LZA::Rosetta::TA::common_sru;
use SLUB::LZA::Rosetta::TA::SRU;



# VERSION

# ABSTRACT: search IEs module for ta-tool

sub abstract {"searches Rosetta based Archival Information System";}
my $description=<<"DESCR";
Searches a Rosetta-based AIS for descriptive oder source metadata on behalf of the Technical Analyst.
Returns SRU-response including dc-records. Only the first 10 matches will be presented by default.

Examples:

  * Is this dc identifier in Archiv?
    $0 count -d SLUB:LZA:Kitodo:kitodo:422766
  * Which IE PID has this dc identifier?
    $0 search -d SLUB:LZA:Kitodo:kitodo:422766
  * Which IE PID are public domain?
    $0 search -s publicdomain
  * Which IE PID are copyrighted?
    $0 search -s copyrighted
  * Which IEs were modified in 2021-05-31?
    $0 search -m 2021-05-31
    Which IEs have at least one invalid file?
    $0 search --with-invalid-files
  * Which IEs with mkv-files are in archive?
    $0 search --with-format=fmt/569
  * Which IEs have at least one file without virus check?
    $0 search --with-missed-viruscheck
DESCR

sub description {
    "$description"
}
sub opt_spec {
    return(
        ["verbose|v" => "enable verbose output"],
        ["debug" => "enable user agent debug output"],
        ["datemode" => hidden => {one_of => [
            ["creationdate|c=s" => "search based on creationdate string"],
            ["modificationdate|m=s" => "search based on modificationdate string"]
        ] } ],
        [ "descriptive|d=s", "search descriptive metadata (dc identifier)"],
        [ "source|s=s", "search source metadata"],
        ["ie|i=s" => "search a specific IE"],
        ["maxrecords=i", "set maxrecords, default is 10"],
        ["startrecord=i", "set startrecord, default is 1"],
        ["format" => hidden => {one_of => [
            ["with-format=s" => "with pronom format id"],
            # unsupported by Rosetta <= 7.0
            #["without-format=s" => "without pronom format id"],
        ] } ],
        ["validation" => hidden => {one_of => [
            ["with-valid-files" => "with valid files"],
            ["with-invalid-files" => "with at least one invalid file"],
        ] } ],
        ["viruscheck" => hidden => {one_of => [
            ["with-passed-viruschecks" => "with passed virus checks"],
            ["with-missed-viruschecks" => "with missed virus checks"],
            ["with-failed-viruschecks" => "with failed virus checks"],
        ]}],
    );
}
sub validate_args {
    my ($self, $opt, $args) = @_;
    # no args allowed but options!
    $self->usage_error("No args allowed") if @$args;
    1;
}

sub execute {
    my ($self, $opt, $args) = @_;
    my $maxrecords="10";
    if (exists $opt->{maxrecords}) {
        $maxrecords = $opt->{maxrecords};
    }
    my $startrecord=1;
    if (exists $opt->{startrecord}) {
        $startrecord = $opt->{startrecord};
    }
    my $query = SLUB::LZA::Rosetta::TA::common_sru::prepare_query($opt);
    my $response = SLUB::LZA::Rosetta::TA::SRU::sru_search('ie', $query, $startrecord, $maxrecords, $opt->{verbose}, $opt->{debug}, $opt->{user}, $opt->{password});
    say $response;
}

1;
