package RackMan::Format::Generic;

use strict;
use Carp;
use RackMan::File;
use RackMan::Template;


use constant {
    CONFIG_SECTION  => "format:generic",
};


#
# write()
# -----
sub write {
    my ($class, $args) = @_;

    my $rackdev = $args->{rackdev};
    my $rackman = $args->{rackman};
    my $output  = RackMan::File->new;

    # load and populate the template
    my $tmpl_path = $rackman->options->{template}
        || $rackman->config->val(CONFIG_SECTION, "template");
    print "  = reading $tmpl_path\n" if $args->{verbose};
    my $tmpl = RackMan::Template->new(filename => $tmpl_path);
    $tmpl->populate_from($rackdev, $rackman);

    # generate the output file
    $output->add_content($tmpl->output);

    # get the output path. the little juggle is to allow %name% expansion
    my $fullpath = $rackman->options->{output}
        || $rackman->config->val(CONFIG_SECTION, "path");
    $rackman->config->newval(CONFIG_SECTION, "path", $fullpath);
    $fullpath = $rackman->config->val(CONFIG_SECTION, "path");

    # write the output file
    if ($fullpath) {
        # ... on disk
        $output->name($fullpath);
        print "  + writing ", $output->fullpath, $/ if $args->{verbose};
        my $scm = $rackman->get_scm({ path => $output->path });
        $scm->update;
        $output->write;
        $scm->add($output->name);
        $scm->commit($output->name, "generated by $class / $::PROGRAM v$::VERSION");
    }
    else {
        # ... on screen
        print $output->content;
    }
}


__PACKAGE__

__END__

=pod

=head1 NAME

RackMan::Format::Generic - Generate a file from a generic for a given RackObject

=head1 SYNOPSIS

on the command line:

    rack write -G Generic -t input.tmpl -o %name%.conf device-name

in code:

    use RackMan::Format::Generic;

    $rackman->options->{template} = "path/to/input.tmpl";
    $rackman->options->{output}   = "path/to/%name%.conf";

    RackMan::Format::Generic->write({
        rackdev => $rackdev,  # a RackMan::Device instance
        rackman => $rackman,  # a RackMan instance
    });


=head1 DESCRIPTION

This module is a generic file generator: it takes a template file as input,
processes it and write the result on disk if a path was provided, otherwise
on screen. As shown in the synopsis, this allows to use rack(1) as a simple
generic templating processor.


=head1 METHODS

=head2 write

Generate the file.

B<Arguments>

Arguments are expected as a hashref with the following keys:

=over

=item *

C<rackdev> - I<(mandatory)> a RackMan::Device instance

=item *

C<rackman> - I<(mandatory)> a RackMan instance

=item *

C<verbose> - I<(optional)> boolean, set to true to be verbose

=back


=head1 TEMPLATE PARAMETERS

See L<RackMan::Template/"TEMPLATE PARAMETERS"> for more details about
the available parameters.


=head1 CONFIGURATION

This module gets its configuration from the C<[format:dhcp]> section
of the main F<rack.conf>, with the following parameters:

=over

=item *

C<path> - specify the path of the output file; can be overriden by
the C<--output> option

=item *

C<template> - specify the path of the template; can be overriden by
the C<--template> option

=back


=head1 SEE ALSO

L<RackMan::Template>


=head1 AUTHOR

Sebastien Aperghis-Tramoni

=cut

