#!/usr/bin/perl

use 5.010;
use strict;
use warnings;
use utf8;

use Test::More;

use Math::Polygon::Tree;


our @TESTS = (
    [
        triangle => [[0,0],[2,0],[0,2]],
        [ [0,0], -1, 'vertex point' ],
        [ [0,1], -1, 'edge point 1' ],
        [ [1,1], -1, 'edge point 2' ],
        [ [0.5,0.5], 1, 'inside' ],
        [ [1.5,1.5], 0, 'outside' ],
        [ [8,8], 0, 'far outside' ],
    ],
    [
        square => [[0,0],[2,0],[2,2],[0,2]],
        [ [2,2], -1, 'vertex point' ],
        [ [1,2], -1, 'edge point' ],
        [ [1,1], 1, 'inside' ],
        [ [3,0], 0, 'outside' ],
        [ [8,8], 0, 'far outside' ],
    ],
    [
        big_poly_file => \*DATA,
        [ [34.9821818,54.9026445], -1, 'vertex' ],
        [ [35.5846168,54.4395806], 1, 'deep inside' ],
        [ [33.7432200,55.2087300], 0, 'far outside but in bbox' ],
        [ [34.4161186,54.5149376], 0, 'outside near border' ],
        [ [34.4141363,54.5091271], 1, 'inside near border' ],
        [ [8,8], 0, 'very far outside' ],
#        [ [], 1, '' ],
    ],
);


for my $item ( @TESTS ) {
    my ($case, $contour, @tests) = @$item;
    my $t = Math::Polygon::Tree->new($contour);

    for my $test ( @tests ) { 
        my ($in, $expected, $name) = @$test;
        my $got = $t->contains($in);
        is( $got, $expected, "$case: $name" );
    }
}


done_testing();



__DATA__
RU-KLU

1
   34.9819063   54.9142909 
   34.9807074   54.9139512 
   34.9797337   54.9139033 
   34.9781759   54.9139832 
   34.9762564   54.9142391 
   34.9759782   54.9135675 
   34.9728068   54.9122242 
   34.9736136   54.9104172 
   34.974003    54.9090259 
   34.9744481   54.9082423 
   34.9784819   54.9060352 
   34.9787322   54.9055714 
   34.9793721   54.9052515 
   34.9792886   54.9048837 
   34.9796224   54.9037961 
   34.980763    54.90386   
   34.9821818   54.9026445 
   34.9826825   54.9019727 
   34.9832945   54.9017967 
   34.9848802   54.9012689 
   34.9856313   54.9016048 
   34.9895816   54.900837  
   34.9906108   54.8994933 
   34.9909157   54.8978438 
   34.9902167   54.8979784 
   34.9885802   54.8980729 
   34.9876017   54.897868  
   34.9867035   54.8973708 
   34.9858779   54.8965327 
   34.9855133   54.8960633 
   34.9852213   54.8953272 
   34.9848007   54.8948614 
   34.984003    54.8943489 
   34.9830846   54.8939139 
   34.9819539   54.8936063 
   34.9797531   54.8933125 
   34.9777198   54.8932976 
   34.9746653   54.8930599 
   34.9727775   54.8929832 
   34.9709319   54.8931285 
   34.9691845   54.8931043 
   34.9657521   54.8933225 
   34.9594847   54.8935961 
   34.9579167   54.893634  
   34.9554038   54.8939848 
   34.9537786   54.8946062 
   34.9530372   54.8952294 
   34.9526754   54.8959923 
   34.952712    54.8967993 
   34.9529056   54.8975561 
   34.9532812   54.8987667 
   34.9532752   54.9000143 
   34.9534534   54.9015683 
   34.9533086   54.9026048 
   34.952774    54.903322  
   34.9498029   54.9050824 
   34.9485821   54.9057077 
   34.947757    54.9060278 
   34.9466924   54.9062231 
   34.9454705   54.9067308 
   34.9439395   54.9078624 
   34.9428887   54.908933  
   34.9421998   54.9095541 
   34.9418817   54.9104742 
   34.9414542   54.9108513 
   34.9408677   54.9116228 
   34.9406092   54.9125772 
   34.9404185   54.9131411 
   34.940393    54.9136186 
   34.9405048   54.9140901 
   34.9407242   54.9149603 
   34.9415238   54.9172328 
   34.9422793   54.9184001 
   34.9435866   54.9193683 
   34.9440091   54.9194683 
   34.9441427   54.9197475 
   34.9438103   54.920561  
   34.9430552   54.9211715 
   34.9409273   54.9223464 
   34.9388834   54.9231361 
   34.9372242   54.923946  
   34.9347346   54.9246944 
   34.9319304   54.9250029 
   34.9301286   54.9253242 
   34.9279739   54.9255933 
   34.9265497   54.9257384 
   34.9254313   54.926024  
   34.9240661   54.9265453 
   34.9229087   54.9271951 
   34.9222749   54.9276664 
   34.9220761   54.9281591 
   34.9218585   54.9284169 
   34.921082    54.928866  
   34.9204731   54.9293586 
   34.9198145   54.9300084 
   34.919409    54.9302972 
   34.9189322   54.9306367 
   34.9171159   54.9331893 
   34.9163517   54.9349247 
   34.9098177   54.9330755 
   34.9061715   54.9300706 
   34.9060367   54.9256139 
   34.9080675   54.9241098 
   34.9096832   54.922913  
   34.9107637   54.920212  
   34.9172462   54.9187265 
   34.9218381   54.9133246 
   34.9206228   54.9104885 
   34.9067131   54.90171   
   34.9025264   54.9038708 
   34.8988801   54.9041408 
   34.8913175   54.899684  
   34.8868608   54.9011695 
   34.8757866   54.9034652 
   34.8719713   54.9033842 
   34.867279    54.891817  
   34.875382    54.8912769 
   34.8745719   54.8869215 
   34.8767328   54.8848958 
   34.8752474   54.8813845 
   34.8765981   54.8784134 
   34.8791641   54.8767928 
   34.8803796   54.8753073 
   34.8790292   54.8730114 
   34.8706564   54.866799  
   34.8763288   54.8601479 
   34.876059    54.8552861 
   34.8770045   54.8521799 
   34.8896193   54.8474061 
   34.8851078   54.8453938 
   34.879706    54.840937  
   34.8836227   54.8344547 
   34.8809219   54.8317536 
   34.8814622   54.8289176 
   34.8768707   54.8252712 
   34.8830833   54.8179448 
   34.8942922   54.8230769 
   34.8971285   54.8180801 
   34.8883507   54.8091666 
   34.8926726   54.8015701 
   34.9002354   54.8018403 
   34.9045574   54.7938724 
   34.9084738   54.7931972 
   34.9034771   54.7910364 
   34.905368    54.7875251 
   34.9242748   54.7896862 
   34.9284617   54.7837103 
   34.9204939   54.7804689 
   34.9118507   54.7797936 
   34.90348     54.7780069 
   34.9075295   54.7719268 
   34.9067194   54.7677402 
   34.8991569   54.7615278 
   34.8937356   54.7575312 
   34.8932713   54.7574486 
   34.8925558   54.7573419 
   34.8918428   54.757248  
   34.8909234   54.757109  
   34.8901813   54.7569784 
   34.8899405   54.7569673 
   34.8887705   54.756898  
   34.8878878   54.7568423 
   34.8870603   54.7567772 
   34.8860304   54.7566842 
   34.885376    54.7566295 
   34.8844094   54.7565342 
   34.8837007   54.7564379 
   34.8829707   54.7563427 
   34.8823791   54.7557603 
   34.8809798   54.7555015 
   34.8795973   54.7552258 
   34.8781525   54.7548885 
   34.8777321   54.7547422 
   34.8767178   54.7544534 
   34.8759308   54.7542174 
   34.8757336   54.7541259 
   34.8754858   54.7540171 
   34.8746462   54.7536413 
   34.8738889   54.7536814 
   34.8731736   54.7537513 
   34.8727166   54.7537806 
   34.8720942   54.7538607 
   34.8714304   54.7538794 
   34.8712623   54.7535289 
   34.871198    54.7532914 
   34.8710428   54.7529459 
   34.870956    54.7528136 
   34.8709494   54.7526913 
   34.8710907   54.7522837 
   34.8713134   54.7518228 
   34.8711798   54.7511571 
   34.8710367   54.750555  
   34.8710399   54.7503376 
   34.8706856   54.750013  
   34.8705408   54.7498118 
   34.8701291   54.7494008 
   34.8699254   54.749171  
   34.8694091   54.7486253 
   34.8689266   54.7481118 
   34.8684192   54.747569  
   34.8680016   54.7471433 
   34.8677631   54.7468463 
   34.8673901   54.7464652 
   34.8669826   54.7460232 
   34.8665243   54.7455868 
   34.8664588   54.745452  
   34.8664653   54.745303  
   34.8665185   54.7447153 
   34.866543    54.744081  
   34.8666101   54.7434275 
   34.8666765   54.743363  
   34.866517    54.7430236 
   34.8662236   54.7424378 
   34.8659862   54.7419367 
   34.8656543   54.7413995 
   34.8655425   54.7409772 
   34.8654204   54.7403752 
   34.8652806   54.7398539 
   34.8652385   54.7395664 
   34.8652891   54.739426  
   34.8654611   54.7389564 
   34.8655998   54.7385854 
   34.8657444   54.7382451 
   34.8658416   54.7379694 
   34.8658851   54.7377423 
   34.8664099   54.7372706 
   34.866877    54.7368355 
   34.867197    54.7365717 
   34.8673824   54.7364349 
   34.867234    54.7363728 
   34.8668949   54.7361511 
   34.8663646   54.7358299 
   34.8660216   54.7356107 
   34.865577    54.7353544 
   34.865342    54.7352757 
   34.8649883   54.7350951 
   34.8645432   54.7348835 
   34.8641078   54.7346638 
   34.8640164   54.7346132 
   34.8636566   54.734389  
   34.8635058   54.7342881 
   34.8633145   54.7341787 
   34.8631358   54.7340856 
   34.8628603   54.7339233 
   34.8623064   54.7335914 
   34.862144    54.7335291 
   34.8616764   54.7333076 
   34.861342    54.7331635 
   34.8606595   54.732835  
   34.8603158   54.732666  
   34.8597551   54.7324092 
   34.8593628   54.7322091 
   34.8591735   54.7321156 
   34.8590229   54.7320659 
   34.8581648   54.7317187 
   34.8576332   54.7314928 
   34.8572834   54.7313791 
   34.8568624   54.7311953 
   34.8567177   54.7311558 
   34.8563582   54.7310058 
   34.8559894   54.730915  
   34.8554897   54.7307976 
   34.8549764   54.7306527 
   34.8545105   54.7305137 
   34.854338    54.7304682 
   34.8539641   54.7303434 
   34.8532444   54.7300802 
   34.8527171   54.729888  
   34.8522667   54.7297153 
   34.8518263   54.7295569 
   34.8512946   54.7293391 
   34.8506657   54.7290852 
   34.8502228   54.7289973 
   34.8494313   54.7288604 
   34.8486278   54.7287066 
   34.8481708   54.7286292 
   34.8473265   54.7284823 
   34.8469213   54.7284018 
   34.8458416   54.7282903 
   34.8446617   54.728179  
   34.8438222   54.7280855 
   34.8425934   54.7279569 
   34.8420218   54.7278877 
   34.8418313   54.7277444 
   34.8417997   54.7276214 
   34.8417751   54.7274094 
   34.8416743   54.7273304 
   34.8416124   54.7272622 
   34.8415913   54.7271941 
   34.8415966   54.7271246 
   34.8415644   54.727038  
   34.8415085   54.7269406 
   34.8414922   54.7267392 
   34.8414399   54.7266945 
   34.8413594   54.7266633 
   34.8412051   54.7266148 
   34.84114     54.7265789 
   34.8410919   54.7265284 
   34.8410739   54.7264305 
   34.8410413   54.726373  
   34.8409625   54.7263208 
   34.8407643   54.7263125 
   34.8405699   54.726326  
   34.8404542   54.72631   
   34.840339    54.7262649 
   34.8401754   54.7261841 
   34.840112    54.7261198 
   34.8400249   54.7260733 
   34.8398692   54.7260273 
   34.8397474   54.7259587 
   34.8396922   54.7259114 
   34.8396583   54.7258482 
   34.839603    54.7256272 
   34.8395526   54.7255565 
   34.839472    54.7255173 
   34.8393581   54.7254909 
   34.8392183   54.7254868 
   34.8389355   54.7255356 
   34.8387618   54.7255621 
   34.8385325   54.7255763 
   34.838361    54.7255559 
   34.8382154   54.725492  
   34.8380881   54.725417  
   34.8380125   54.725339  
   34.8379143   54.7252786 
   34.8376974   54.7252111 
   34.8375053   54.7251721 
   34.8372918   54.7251452 
   34.8371159   54.7251475 
   34.8369611   54.7251305 
   34.8368485   54.7250967 
   34.836757    54.7250541 
   34.8366627   54.7249171 
   34.836507    54.7247901 
   34.8363553   54.724668  
   34.8361919   54.7245749 
   34.8361332   54.7244922 
   34.8361584   54.7243889 
   34.8362286   54.724293  
   34.8362643   54.7242438 
   34.836268    54.7241881 
   34.836245    54.7241355 
   34.8361787   54.7240897 
   34.8359395   54.7240214 
   34.8356972   54.7239764 
   34.8355781   54.723912  
   34.8355082   54.7238226 
   34.835465    54.7237375 
   34.8354626   54.7236227 
   34.8354642   54.7235217 
   34.8354189   54.7234736 
   34.8353384   54.7234409 
   34.8352216   54.7234071 
   34.8351046   54.7233879 
   34.8350366   54.72336   
   34.8349953   54.7233186 
   34.8349207   54.7231848 
   34.8348127   54.7230354 
   34.8347559   54.7229956 
   34.8346768   54.722974  
   34.8345638   54.7229653 
   34.8344378   54.7229801 
   34.8342276   54.7230234 
   34.8340932   54.7230429 
   34.8339718   54.7230383 
   34.8338798   54.7230127 
   34.8338263   54.7229639 
   34.8337574   54.7229077 
   34.8337112   54.7228267 
   34.8336722   54.7227335 
   34.8336585   54.7226258 
   34.8336495   54.7225781 
   34.8336098   54.7225213 
   34.833502    54.722443  
   34.8333619   54.7223825 
   34.8331816   54.7223177 
   34.832983    54.7222357 
   34.8327808   54.7221343 
   34.8326011   54.7220105 
   34.8324688   54.7218951 
   34.83238     54.7217822 
   34.8323373   54.7216518 
   34.8323234   54.7214779 
   34.8323363   54.7212014 
   34.8323229   54.7209887 
   34.8323208   54.7208545 
   34.8322993   54.7207153 
   34.8322428   54.720398  
   34.8321805   54.7202029 
   34.8320889   54.7199922 
   34.8320014   54.7198616 
   34.8318839   54.719709  
   34.8318519   54.7196101 
   34.8318491   54.7195251 
   34.831864    54.7194622 
   34.8319048   54.7193663 
   34.8319737   54.7192477 
   34.8320309   54.7191648 
   34.8321356   54.7190739 
   34.832261    54.7189994 
   34.8323546   54.7189216 
   34.8323755   54.7188416 
   34.8323659   54.7187422 
   34.8323395   54.7186433 
   34.8323457   54.7185228 
   34.8323662   54.7184631 
   34.8324146   54.7183948 
   34.8324816   54.7183215 
   34.8325705   54.7182622 
   34.8326021   54.7182074 
   34.8325989   54.7181482 
   34.8325639   54.7180672 
   34.8325739   54.7179614 
   34.832656    54.7178825 
   34.8327575   54.7178379 
   34.8328569   54.7178206 
   34.8329797   54.7178269 
   34.8331414   54.7178439 
   34.8332971   54.7178827 
   34.8334686   54.7179079 
   34.83366     54.7179058 
   34.8337468   54.717895  
   34.8338436   54.7178656 
   34.8339831   54.7177895 
   34.8340161   54.7177355 
   34.8340086   54.7176747 
   34.8339722   54.7175985 
   34.833901    54.7175069 
   34.8336183   54.717283  
   34.8335662   54.7172123 
   34.8335455   54.7171192 
   34.8335421   54.7169819 
   34.8335348   54.7169147 
   34.8335078   54.7168539 
   34.8334549   54.7167484 
   34.8333755   54.7166583 
   34.8333586   54.7165807 
   34.8333838   54.7164902 
   34.8334321   54.7164372 
   34.8335365   54.716373  
   34.8336827   54.7163107 
   34.8338069   54.7162322 
   34.8339172   54.7161445 
   34.8340598   54.7160499 
   34.8341078   54.716005  
   34.8341326   54.7159461 
   34.8341528   54.7158201 
   34.8341411   54.7157658 
   34.8341042   54.7157049 
   34.8340193   54.7156082 
   34.8339907   54.7155564 
   34.8340016   54.7154878 
   34.8340435   54.7154088 
   34.8340632   54.7153118 
   34.8340506   54.7152276 
   34.8339962   54.7151238 
   34.8339307   54.7150394 
   34.8338049   54.714957  
   34.8336533   54.7149167 
   34.8335168   54.7148957 
   34.833401    54.7148887 
   34.8332165   54.7148877 
   34.8329164   54.7148925 
   34.8327503   54.7148844 
   34.8326027   54.7148471 
   34.8324193   54.7148    
   34.8322235   54.7147311 
   34.8319253   54.7145993 
   34.8317581   54.7144949 
   34.8316785   54.7144048 
   34.8316242   54.7143082 
   34.8316028   54.7141627 
   34.8316193   54.7140058 
   34.8315993   54.7139427 
   34.8315458   54.7138834 
   34.8314349   54.7138334 
   34.8312818   54.7138027 
   34.8310739   54.7137815 
   34.8308534   54.7137835 
   34.8307198   54.7137497 
   34.8306118   54.7136964 
   34.8305752   54.7136259 
   34.8305119   54.713569  
   34.830402    54.7135433 
   34.8303028   54.7135348 
   34.8301743   54.7134491 
   34.830113    54.7133631 
   34.8300699   54.7132666 
   34.8299787   54.7132063 
   34.8298568   54.7131426 
   34.8298029   54.7130995 
   34.8297693   54.7130266 
   34.8297569   54.7129286 
   34.829791    54.7127955 
   34.8298031   54.7127405 
   34.829857    54.7126923 
   34.8299451   54.7126798 
   34.8300401   54.7126835 
   34.8300698   54.7126538 
   34.8300537   54.7126141 
   34.8299573   54.7125383 
   34.8298425   54.7124698 
   34.8297028   54.7123859 
   34.8294771   54.7122674 
   34.8293306   54.712172  
   34.8292948   54.7121297 
   34.8292736   54.7120673 
   34.8292561   54.7119387 
   34.8292847   54.7118176 
   34.8293065   54.7117538 
   34.8293452   54.7117007 
   34.829417    54.7116607 
   34.8295207   54.7116394 
   34.8297123   54.7116234 
   34.8298151   54.7116434 
   34.8299173   54.7116416 
   34.8299845   54.711629  
   34.8300144   54.7115928 
   34.8299723   54.7115166 
   34.8298969   54.7114288 
   34.8298589   54.7113655 
   34.8298671   54.7112863 
   34.8299314   54.7111944 
   34.8300262   54.7111174 
   34.830157    54.7110607 
   34.8303167   54.7110275 
   34.8304957   54.7110148 
   34.8306728   54.7110295 
   34.8308106   54.7110545 
   34.8308761   54.7110638 
   34.8309709   54.711061  
   34.8310165   54.7110185 
   34.8310061   54.7109675 
   34.8309513   54.7109017 
   34.830867    54.7108382 
   34.8307431   54.7107275 
   34.8306378   54.7106009 
   34.8306256   54.7104868 
   34.8306319   54.7103598 
   34.8306444   54.7102743 
   34.8306807   54.7101911 
   34.8307557   54.710126  
   34.8308473   54.7100772 
   34.830943    54.7100285 
   34.8310691   54.7099992 
   34.8312804   54.7099729 
   34.8315374   54.7099662 
   34.8316651   54.7099264 
   34.8317983   54.7098947 
   34.8319376   54.709826  
   34.8320435   54.7097563 
   34.832163    54.7096963 
   34.8323189   54.7096494 
   34.8324814   54.7096178 
   34.8326378   54.7096172 
   34.8327903   54.7096705 
   34.8328885   54.7097438 
   34.8328892   54.7097866 
   34.8328593   54.7098244 
   34.8328206   54.7098825 
   34.8328282   54.709931  
   34.8328781   54.7099547 
   34.8329381   54.7099591 
   34.833022    54.7099482 
   34.8331317   54.7099003 
   34.8332208   54.7098344 
   34.8333401   54.7097058 
   34.8334231   54.7095834 
   34.8334904   54.7094778 
   34.8335262   54.709419  
   34.8335342   54.7093584 
   34.8335204   54.7092588 
   34.833521    54.7091318 
   34.8335447   54.7090471 
   34.8335971   54.7090022 
   34.8336798   54.7089823 
   34.8338097   54.7089791 
   34.8339602   54.7089039 
   34.8339861   54.708861  
   34.8339686   54.7088157 
   34.833908    54.7087702 
   34.8338496   54.7087456 
   34.8337467   54.7086416 
   34.8337209   54.7085848 
   34.8337332   54.7085258 
   34.8337272   54.7084563 
   34.8337396   54.7083868 
   34.8337226   54.7083067 
   34.8336814   54.7082709 
   34.833602    54.7082502 
   34.8334785   54.70821   
   34.8332089   54.7081236 
   34.8329644   54.7079542 
   34.8329283   54.7078642 
   34.8329768   54.7077124 
   34.8330683   54.7075803 
   34.833168    54.7074621 
   34.8334817   54.7071282 
   34.8335385   54.7069935 
   34.8335144   54.7069279 
   34.8334315   54.7068765 
   34.8333245   54.7068377 
   34.8332096   54.7068267 
   34.8330684   54.7067934 
   34.8329681   54.7067459 
   34.8328862   54.7067175 
   34.832767    54.7066843 
   34.8326158   54.706659  
   34.8324987   54.7066774 
   34.8323858   54.7067102 
   34.8322485   54.7067431 
   34.8320786   54.7067525 
   34.8319786   54.706722  
   34.8319232   54.7066651 
   34.8318858   54.7065556 
   34.8318451   54.7064821 
   34.8317534   54.7064174 
   34.8316015   54.7063242 
   34.8315769   54.7062284 
   34.8316046   54.706128  
   34.8316115   54.706067  
   34.8315877   54.7060424 
   34.8315414   54.7060387 
   34.8314769   54.7060971 
   34.8313785   54.7061396 
   34.8312781   54.7061458 
   34.8311772   54.7061319 
   34.8310593   54.7061156 
   34.8309528   54.7060708 
   34.8309076   54.7060013 
   34.8309203   54.7059332 
   34.8309482   54.7058773 
   34.8309698   54.7057703 
   34.8309371   54.7057285 
   34.8308656   54.7057102 
   34.8307029   54.7057083 
   34.8305417   54.7057339 
   34.8304148   54.7057385 
   34.8303136   54.7057202 
   34.8301672   54.7056432 
   34.8300106   54.7055396 
   34.829876    54.70544   
   34.8297695   54.7053354 
   34.8297325   54.7052379 
   34.8297356   54.7051309 
   34.8297685   54.7050339 
   34.8298336   54.7049654 
   34.829861    54.7048651 
   34.8298616   54.70474   
   34.8298469   54.704662  
   34.8298103   54.7045921 
   34.8297473   54.704558  
   34.8296517   54.7045402 
   34.829518    54.7045194 
   34.829346    54.7045088 
   34.8292579   54.7044872 
   34.8292123   54.7044485 
   34.8292028   54.7043745 
   34.8292199   54.704265  
   34.8292142   54.7041846 
   34.8291963   54.7041134 
   34.8291328   54.7040678 
   34.8289573   54.7040243 
   34.8287836   54.703988  
   34.8286718   54.7039507 
   34.8285818   54.7039039 
   34.8285549   54.7038399 
   34.8285838   54.7037726 
   34.8286865   54.7036991 
   34.8287479   54.7036173 
   34.8287102   54.7035208 
   34.8286208   54.7034685 
   34.8284717   54.7034314 
   34.8283185   54.7034047 
   34.828134    54.7033927 
   34.8279356   54.703387  
   34.8277414   54.7033994 
   34.8274814   54.7034587 
   34.8273321   54.7034502 
   34.8272092   54.7034144 
   34.8269813   54.7033184 
   34.826773    54.7031739 
   34.8266945   54.703108  
   34.8266141   54.7030057 
   34.8266027   54.7029386 
   34.8266209   54.7028823 
   34.8266685   54.7028158 
   34.8266938   54.7027367 
   34.8267247   54.7026771 
   34.8268089   54.7026265 
   34.8269091   54.7026155 
   34.8270219   54.7026333 
   34.8271644   54.702708  
   34.8272809   54.7027902 
   34.8273938   54.7028597 
   34.8275099   54.7028984 
   34.8275813   54.702912  
   34.8276375   54.7028948 
   34.8276422   54.7028381 
   34.8276147   54.7027826 
   34.827545    54.702705  
   34.827439    54.7026268 
   34.8274463   54.7025395 
   34.8274143   54.7024538 
   34.8273693   54.7024022 
   34.8272932   54.7023638 
   34.8270121   54.702314  
   34.8268424   54.7022662 
   34.8266657   54.7021456 
   34.8266233   54.7020911 
   34.8265779   54.7020294 
   34.8265683   54.7019623 
   34.8265389   54.7018985 
   34.8265337   54.7018274 
   34.8265633   54.7017512 
   34.8266366   54.7016808 
   34.8267318   54.701623  
   34.8267495   54.7015663 
   34.826719    54.7015023 
   34.8266313   54.7014034 
   34.8265164   54.7013152 
   34.8264057   54.7012692 
   34.8262579   54.7012415 
   34.8261063   54.7012251 
   34.8259919   54.7011887 
   34.8259144   54.7011464 
   34.8258732   54.7010877 
   34.8258834   54.701031  
   34.8258491   54.7009102 
   34.8257687   54.700847  
   34.8256624   54.7007999 
   34.825604    54.7007578 
   34.8255564   54.7006815 
   34.8255419   54.7004841 
   34.825483    54.7003747 
   34.8254255   54.7003324 
   34.8253373   54.700303  
   34.8252252   54.7002951 
   34.8251119   54.7003045 
   34.8249639   54.7003594 
   34.8248607   54.7003826 
   34.824788    54.7003723 
   34.8247316   54.7003394 
   34.8246919   54.7002923 
   34.8246773   54.7002156 
   34.8246909   54.7001606 
   34.8247659   54.7000997 
   34.8248859   54.7000338 
   34.8249275   54.6999956 
   34.824927    54.6999509 
   34.824889    54.6998837 
   34.824828    54.6998262 
   34.824776    54.6997827 
   34.8246653   54.6997275 
   34.8245539   54.699702  
   34.8244095   54.6996827 
   34.8241558   54.6996618 
   34.8240476   54.6996449 
   34.8239811   54.6996233 
   34.8239182   54.69959   
   34.8238949   54.6995063 
   34.8239385   54.69943   
   34.8239823   54.6993718 
   34.824009    54.6993096 
   34.824012    54.6992415 
   34.8239196   54.6991618 
   34.8238207   54.6991003 
   34.8236632   54.6990291 
   34.8235592   54.6989843 
   34.8234956   54.6989474 
   34.8234305   54.6988896 
   34.8234201   54.6988388 
   34.8234539   54.6987911 
   34.8235476   54.6987183 
   34.823701    54.6986227 
   34.823729    54.6985772 
   34.8237198   54.6985235 
   34.8236428   54.6984094 
   34.8236339   54.6983665 
   34.8236608   54.6983174 
   34.8237068   54.698279  
   34.8237738   54.6982531 
   34.8238518   54.6982625 
   34.8239314   54.6982914 
   34.823983    54.6983236 
   34.8240631   54.6983451 
   34.8241379   54.6983541 
   34.824215    54.6983417 
   34.8242579   54.6982696 
   34.8242466   54.698202  
   34.8242063   54.6981374 
   34.8241819   54.6979725 
   34.8242991   54.6978513 
   34.8245179   54.6977986 
   34.8246276   54.6976814 
   34.8245244   54.6975418 
   34.8244845   54.6974394 
   34.8244452   54.6972888 
   34.8244171   54.6972133 
   34.8243424   54.6971139 
   34.8240643   54.6969645 
   34.8239156   54.696911  
   34.8236196   54.6968315 
   34.823593    54.6968244 
   34.8234125   54.6967834 
   34.8232678   54.6967637 
   34.823219    54.6967569 
   34.8230325   54.6967178 
   34.8227304   54.696671  
   34.8224782   54.6966417 
   34.8221025   54.6965977 
   34.8217439   54.6965803 
   34.8214287   54.6965908 
   34.8213344   54.6965826 
   34.8212201   54.6965722 
   34.8211352   54.6965613 
   34.8210229   54.6965465 
   34.820898    54.6965304 
   34.8208245   54.6965207 
   34.8207348   54.696515  
   34.8206127   54.696507  
   34.8203766   54.6964872 
   34.8202272   54.6964757 
   34.8201136   54.6964667 
   34.8198764   54.69646   
   34.8197572   54.6964565 
   34.8193775   54.6964763 
   34.8193033   54.6964803 
   34.8191569   54.6964882 
   34.8190466   54.6964979 
   34.8188997   54.6965107 
   34.8186642   54.6965435 
   34.8185486   54.6965646 
   34.8184084   54.69659   
   34.8181499   54.6966356 
   34.8181333   54.6966384 
   34.8177993   54.6966892 
   34.8176617   54.6967011 
   34.8175297   54.6967004 
   34.8174533   54.696681  
   34.8173981   54.6966669 
   34.8173663   54.6966416 
   34.8173305   54.6966128 
   34.8173228   54.696586  
   34.8173134   54.6965523 
   34.8173438   54.6964878 
   34.817368    54.696442  
   34.8173696   54.696439  
   34.8173793   54.6964319 
   34.8174149   54.6964066 
   34.8176149   54.6962921 
   34.8176174   54.6962908 
   34.8180264   54.6961385 
   34.8180606   54.6961259 
   34.8181908   54.6960785 
   34.8182998   54.6960194 
   34.8183472   54.695994  
   34.8184108   54.6959535 
   34.8185122   54.6958896 
   34.8186056   54.69581   
   34.8186711   54.6957057 
   34.8186905   54.6956753 
   34.8186974   54.6956529 
   34.8187093   54.6956165 
   34.8187009   54.6955348 
   34.8186932   54.6955107 
   34.8186906   54.695501  
   34.8186816   54.6954919 
   34.8186573   54.6954666 
   34.8185771   54.6954218 
   34.8185026   54.6953681 
   34.8184951   54.6953627 
   34.818461    54.695348  
   34.8183903   54.6953173 
   34.8183249   54.6952979 
   34.8182842   54.6952855 
   34.8181666   54.6952753 
   34.8179011   54.6952666 
   34.8176615   54.6952711 
   34.8174927   54.6952675 
   34.8171155   54.6952592 
   34.8168992   54.6952658 
   34.8167522   54.6952701 
   34.8166526   54.6952813 
   34.8165019   54.695298  
   34.8162655   54.6953395 
   34.8160895   54.6953523 
   34.8159256   54.6953437 
   34.8157911   54.6953255 
   34.8157671   54.6953168 
   34.8156926   54.6952894 
   34.8154009   54.6952061 
   34.8153001   54.6951468 
   34.815241    54.6951114 
   34.815105    54.6949657 
   34.8150895   54.694949  
   34.814989    54.6948724 
   34.8148658   54.6948062 
   34.8147509   54.6947539 
   34.8147166   54.694746  
   34.814613    54.6947221 
   34.8143436   54.6946196 
   34.8143404   54.6946184 
   34.8142452   54.6945415 
   34.8141592   54.6944296 
   34.8142061   54.6942956 
   34.8142513   54.6942434 
   34.8143264   54.6941573 
   34.8145038   54.693945  
   34.814506    54.6939425 
   34.8145637   54.693743  
   34.8145411   54.6936944 
   34.8145249   54.693659  
   34.8144093   54.6935513 
   34.8143429   54.6935787 
   34.8140321   54.6937223 
   34.8135875   54.6939441 
   34.8130777   54.6942649 
   34.8124485   54.6946997 
   34.8120264   54.6950237 
   34.8117643   54.6953287 
   34.8116601   54.6955566 
   34.8115765   54.695864  
   34.8114567   54.6960749 
   34.8112464   54.6963504 
   34.8110091   54.69663   
   34.8108511   54.696839  
   34.8108135   54.6968927 
   34.8108066   54.6970276 
   34.8108443   54.6971144 
   34.8110407   54.6972319 
   34.8111471   54.6973233 
   34.811258    54.6975708 
   34.8112969   54.6981882 
   34.811338    54.6985218 
   34.8113926   54.6987689 
   34.8114366   54.6992261 
   34.8113846   54.699396  
   34.8112431   54.6995101 
   34.8110628   54.6995957 
   34.8107205   54.6997115 
   34.810369    54.699775  
   34.8098957   54.6998093 
   34.8093413   54.6998486 
   34.8087678   54.6998752 
   34.8082723   54.6999094 
   34.8078772   54.6999625 
   34.8073168   54.7000588 
   34.8068498   54.7001498 
   34.8064615   54.700227  
   34.8061222   54.7003089 
   34.8057038   54.7004257 
   34.8054135   54.7004794 
   34.8050821   54.7005343 
   34.8046128   54.700614  
   34.8043584   54.7007345 
   34.8040499   54.7010181 
   34.8038346   54.7011389 
   34.8035762   54.701214  
   34.8032371   54.7012773 
   34.80273     54.7014164 
   34.8024915   54.701459  
   34.8019778   54.7015683 
   34.8018001   54.7016509 
   34.8014348   54.7018135 
   34.8008802   54.7020033 
   34.8005328   54.702129  
   34.8001876   54.7022591 
   34.7998225   54.7024046 
   34.7994449   54.7025642 
   34.7992514   54.702712  
   34.7991386   54.7028647 
   34.7990218   54.7031081 
   34.7990132   54.7033265 
   34.799089    54.7034844 
   34.7991656   54.7035926 
   34.7992256   54.7036654 
   34.7993306   54.703822  
   34.799378    54.7039272 
   34.7993835   54.704028  
   34.7993529   54.7041046 
   34.7993103   54.7041652 
   34.7992433   54.7042202 
   34.7991762   54.7042581 
   34.7991172   54.7042806 
   34.7990409   54.7042958 
   34.79896     54.7043051 
   34.7987738   54.7043084 
   34.7985924   54.7043003 
   34.7984274   54.7042789 
   34.7982686   54.7042581 
   34.7981524   54.7042484 
   34.7979024   54.704252  
   34.7975958   54.7042769 
   34.7974113   54.7042896 
   34.797291    54.7043018 
   34.7968708   54.7043366 
   34.796669    54.7043504 
   34.796408    54.7043947 
   34.7961252   54.7044445 
   34.7958545   54.7044916 
   34.7956928   54.7045219 
   34.7955554   54.7045392 
   34.7954007   54.70458   
   34.7952691   54.7046168 
   34.7949126   54.7047298 
   34.7947387   54.7048016 
   34.7946027   54.7048634 
   34.7944835   54.7049317 
   34.7943541   54.7050028 
   34.7942318   54.705079  
   34.7941188   54.7051381 
   34.7940293   54.705166  
   34.7938813   54.7052061 
   34.7936888   54.7053033 
   34.7934598   54.7054056 
   34.7933036   54.7055047 
   34.793125    54.7056702 
   34.7929072   54.7058271 
   34.7926745   54.7059695 
   34.7924482   54.7060876 
   34.7923753   54.706103  
   34.7923575   54.7060871 
   34.7923376   54.7060168 
   34.7922831   54.7059972 
   34.7921648   54.7060016 
   34.7920032   54.7060622 
   34.7918591   54.7061648 
   34.7917854   54.7062276 
   34.7917045   54.7063483 
   34.7915331   54.7064508 
   34.7911976   54.7065841 
   34.7909498   54.7067229 
   34.7906764   54.7069355 
   34.7904993   54.7071906 
   34.790321    54.7073528 
   34.7899361   54.7075208 
   34.7897114   54.707565  
   34.7894626   54.7075795 
   34.7890475   54.7075754 
   34.7886327   54.7075501 
   34.7883485   54.7075134 
   34.7879406   54.7074444 
   34.7873786   54.7073587 
   34.7868801   54.7072733 
   34.7864003   54.7071601 
   34.7860471   54.7070844 
   34.7857261   54.707065  
   34.7852322   54.7070709 
   34.7843558   54.7071097 
   34.7837763   54.7071573 
   34.7825634   54.7073836 
   34.7818791   54.7075252 
   34.781463    54.707579  
   34.7811626   54.7076088 
   34.7809953   54.70765   
   34.7808391   54.7077456 
   34.7806386   54.7079391 
   34.7804661   54.7081084 
   34.7801106   54.7083467 
   34.7796515   54.7086054 
   34.7793496   54.7087213 
   34.7790694   54.7088004 
   34.7787622   54.708867  
   34.7784817   54.7089619 
   34.7781515   54.7091266 
   34.7779225   54.7092341 
   34.7776119   54.70932   
   34.7772739   54.7094093 
   34.777058    54.7094555 
   34.7768822   54.7094561 
   34.776658    54.7094496 
   34.7764097   54.7094552 
   34.7762637   54.7094823 
   34.7759622   54.7095525 
   34.7756007   54.7096189 
   34.7752422   54.7096588 
   34.7748541   54.7096776 
   34.7745509   54.7096899 
   34.7743261   54.7097149 
   34.7741257   54.7097524 
   34.7738212   54.7098347 
   34.7734561   54.7099098 
   34.773198    54.7099382 
   34.7730314   54.7099372 
   34.7729842   54.7099311 
   34.7727625   54.7099022 
   34.7724938   54.7098428 
   34.7721415   54.7097021 
   34.7719075   54.709562  
   34.7717461   54.7094401 
   34.771618    54.7093269 
   34.7715708   54.7092444 
   34.7715542   54.7091565 
   34.7715858   54.7090848 
   34.7717496   54.7089787 
   34.7719009   54.7089075 
   34.772011    54.7088539 
   34.7720572   54.7088292 
   34.7722226   54.7087147 
   34.7722329   54.70865   
   34.7722067   54.708576  
   34.7721233   54.7085019 
   34.7717914   54.7084227 
   34.7716953   54.7083677 
   34.7716345   54.7082129 
   34.7716425   54.7080832 
   34.7715781   54.7079635 
   34.771501    54.7078735 
   34.7714333   54.7077503 
   34.7714108   54.7076537 
   34.77144     54.7075503 
   34.7715324   54.7074649 
   34.7716854   54.7073833 
   34.7719108   54.7073039 
   34.7722787   54.707236  
   34.7727345   54.7071667 
   34.7728989   54.7071167 
   34.7729878   54.7070647 
   34.7730165   54.7069824 
   34.7729907   54.7068962 
   34.7729078   54.7067817 
   34.7728547   54.7067076 
   34.7727614   54.7064879 
   34.7726602   54.7063838 
   34.7724449   54.7062246 
   34.7722867   54.7060885 
   34.7721702   54.7059931 
   34.772011    54.7059114 
   34.7717933   54.7058821 
   34.7716881   54.7058445 
   34.7716281   54.7057986 
   34.7715722   54.7057124 
   34.77156     54.7055421 
   34.77147     54.7054872 
   34.7712856   54.7054686 
   34.7710913   54.7054849 
   34.7708811   54.7055503 
   34.7706183   54.7056786 
   34.7703917   54.7058124 
   34.7701598   54.7059005 
   34.7698917   54.7059849 
   34.7696305   54.7060343 
   34.7693566   54.7060993 
   34.7691159   54.7061681 
   34.7687439   54.7063045 
   34.7684203   54.7064359 
   34.7680148   54.7065861 
   34.767658    54.7067069 
   34.7672373   54.7068553 
   34.7670331   54.7069295 
   34.7666982   54.7070047 
   34.76648     54.7070261 
   34.7661712   54.7070227 
   34.7659008   54.7070105 
   34.7654089   54.706897  
   34.7650835   54.7067955 
   34.7647975   54.7067097 
   34.7645777   54.7066556 
   34.7639823   54.7065448 
   34.76363     54.7064623 
   34.7631775   54.7063403 
   34.7626343   54.7061842 
   34.7620802   54.706004  
   34.7615326   54.7057969 
   34.7611501   54.7056332 
   34.7607776   54.7054593 
   34.7605826   54.7053915 
   34.7602535   54.7053333 
   34.7598687   54.7053043 
   34.7596289   54.7052837 
   34.7594814   54.7052551 
   34.7593683   54.7052274 
   34.7593396   54.7052203 
   34.7592554   54.7051656 
   34.7591665   54.7050742 
   34.7589799   54.7050346 
   34.7587886   54.7050383 
   34.7586792   54.7050643 
   34.7583528   54.7052039 
   34.758179    54.7052419 
   34.7579244   54.7052668 
   34.7571965   54.7052941 
   34.7566058   54.7053009 
   34.7560024   54.7053167 
   34.7555203   54.7053381 
   34.7550172   54.705365  
   34.7545851   54.7054028 
   34.7542001   54.7054579 
   34.7536992   54.7055081 
   34.7530615   54.7055755 
   34.7523547   54.7056435 
   34.7517989   54.7057124 
   34.750641    54.7058582 
   34.7502586   54.7059257 
   34.749887    54.7060132 
   34.7496806   54.7060798 
   34.7494416   54.7062086 
   34.7487631   54.7065497 
   34.7481554   54.7068007 
   34.7478867   54.7069114 
   34.7474852   54.7071087 
   34.7472205   54.7072316 
   34.7468566   54.7073547 
   34.7466014   54.7074178 
   34.7462426   54.7074892 
   34.7460244   54.7075106 
   34.7457142   54.7075232 
   34.7455532   54.7075449 
   34.7454382   54.7075741 
   34.7453465   54.7076277 
   34.7451459   54.7076774 
   34.7450061   54.7076855 
   34.7448247   54.7076827 
   34.7445185   54.7076275 
   34.7442716   54.707537  
   34.7440472   54.7074387 
   34.743885    54.707373  
   34.7437276   54.7073542 
   34.7436032   54.7073624 
   34.7433563   54.7074223 
   34.7430467   54.707477  
   34.7428299   54.7075032 
   34.7425596   54.7075444 
   34.7422603   54.7075741 
   34.7419162   54.7076068 
   34.7416921   54.7076442 
   34.74136     54.7077085 
   34.7411724   54.7077372 
   34.740983    54.7077886 
   34.740583    54.7079778 
   34.7403036   54.7081353 
   34.7401615   54.7082679 
   34.7400625   54.7084128 
   34.7399718   54.7085634 
   34.739822    54.7088132 
   34.7397178   54.7090163 
   34.7395617   54.7094559 
   34.739498    54.7096585 
   34.7393918   54.7098147 
   34.7392834   54.7099361 
   34.7391564   54.7100849 
   34.7390207   54.7101755 
   34.7389054   54.7102216 
   34.7387654   54.7102385 
   34.7386567   54.7102258 
   34.7384971   54.7101739 
   34.7383578   54.7101552 
   34.7381482   54.710166  
   34.7378778   54.7102121 
   34.7376619   54.7102617 
   34.737542    54.7103224 
   34.7373869   54.7104039 
   34.7372072   54.7104626 
   34.7369446   54.7105394 
   34.7366065   54.7106271 
   34.7363385   54.7106942 
   34.735886    54.7107755 
   34.7354599   54.7108642 
   34.7351222   54.7109284 
   34.7347477   54.7110158 
   34.7343774   54.7111065 
   34.7341209   54.7111591 
   34.7339517   54.7111718 
   34.7336054   54.7111697 
   34.7332578   54.7111627 
   34.7330947   54.7111495 
   34.7330338   54.7111192 
   34.7330345   54.7110837 
   34.7330638   54.7110055 
   34.7328069   54.7109303 
   34.7326595   54.7108954 
   34.7325443   54.7108567 
   34.7324797   54.7108013 
   34.7324501   54.7107397 
   34.7324727   54.7106525 
   34.7324222   54.7105875 
   34.7322722   54.7105478 
   34.7319493   54.7104859 
   34.7317492   54.7104354 
   34.7315933   54.7104085 
   34.7314681   54.7103835 
   34.7311677   54.7103922 
   34.7306133   54.7103919 
   34.7302615   54.7103914 
   34.7300705   54.7103764 
   34.7296355   54.710339  
   34.7292819   54.710281  
   34.7289646   54.71022   
   34.7285624   54.7101488 
   34.7283594   54.7101038 
   34.728067    54.7100535 
   34.7276615   54.7100106 
   34.7274471   54.7099785 
   34.7270977   54.7099197 
   34.7267957   54.7098636 
   34.7266829   54.7098476 
   34.7264111   54.7098192 
   34.7261561   54.7097942 
   34.7259737   54.7097647 
   34.7258208   54.7097266 
   34.725736    54.7097058 
   34.7255433   54.7096343 
   34.7253645   54.7095588 
   34.7252704   54.7095145 
   34.7251855   54.7095043 
   34.7250933   54.7095053 
   34.7249185   54.7095188 
   34.7247618   54.7095332 
   34.7246109   54.7095395 
   34.7243526   54.7095379 
   34.7241894   54.7095264 
   34.723966    54.7095266 
   34.723769    54.7095334 
   34.7235076   54.7095488 
   34.7230969   54.7095567 
   34.7228563   54.7095819 
   34.7223847   54.7096363 
   34.7221746   54.7096714 
   34.7217129   54.7097194 
   34.721206    54.7097914 
   34.7208045   54.7098342 
   34.7204965   54.7098783 
   34.7201183   54.7099366 
   34.719876    54.7099763 
   34.7196636   54.7099839 
   34.7194178   54.7099872 
   34.7191443   54.7099806 
   34.7190015   54.7099999 
   34.7188638   54.7100443 
   34.7187594   54.7100994 
   34.7187292   54.7101461 
   34.7187223   54.7102197 
   34.7187432   54.7102926 
   34.7187617   54.7104253 
   34.7187419   54.7105125 
   34.7187059   54.710568  
   34.7186321   54.7106314 
   34.7185502   54.7106794 
   34.7183931   54.7107189 
   34.7182993   54.7107296 
   34.7181863   54.7107257 
   34.7180386   54.7107061 
   34.717805    54.7106529 
   34.7175966   54.7105974 
   34.7174409   54.71056   
   34.7171666   54.7105179 
   34.7169813   54.7104981 
   34.7168264   54.7104939 
   34.7166964   54.7104995 
   34.7165621   54.7105141 
   34.7164372   54.7105448 
   34.7163318   54.7105813 
   34.7162119   54.7106444 
   34.7161662   54.7106991 
   34.7161482   54.7107637 
   34.7161638   54.7108277 
   34.716216    54.7108733 
   34.7163284   54.7109104 
   34.7165456   54.7109457 
   34.7167739   54.7109811 
   34.7169742   54.7110244 
   34.7171214   54.711069  
   34.7171865   54.7110961 
   34.7172622   54.7111556 
   34.7172862   54.711214  
   34.7172901   54.7112997 
   34.7172544   54.7113431 
   34.7171753   54.7113936 
   34.71705     54.7114461 
   34.7168339   54.7115006 
   34.7166126   54.711538  
   34.7163994   54.7115867 
   34.7160968   54.7116374 
   34.7159313   54.7116768 
   34.7158189   54.711714  
   34.7156477   54.7117582 
   34.7155597   54.7117601 
   34.7154733   54.7117474 
   34.7153779   54.7116967 
   34.71533     54.7116487 
   34.7153171   54.711592  
   34.7153318   54.7115501 
   34.7153831   54.7114962 
   34.7153925   54.7114429 
   34.7153695   54.711404  
   34.7153085   54.7113785 
   34.7152221   54.7113723 
   34.7150879   54.7113803 
   34.7148009   54.7114182 
   34.7145265   54.7114536 
   34.7144006   54.7114674 
   34.7142987   54.7114651 
   34.7141888   54.7114409 
   34.7140834   54.7114023 
   34.7139338   54.7113407 
   34.7138254   54.7113141 
   34.7137376   54.7113039 
   34.7136026   54.7112812 
   34.7133668   54.7112684 
   34.7130584   54.7112591 
   34.712803    54.7112526 
   34.7125106   54.7112088 
   34.7123278   54.7111963 
   34.7121351   54.7112007 
   34.7118233   54.7112181 
   34.7115106   54.7112177 
   34.7113211   54.7111955 
   34.7111807   54.7111615 
   34.7110605   54.7110936 
   34.7109949   54.7110196 
   34.7109193   54.7108809 
   34.7108688   54.7108183 
   34.7107681   54.710757  
   34.7106821   54.7107249 
   34.7105846   54.7107106 
   34.7104868   54.7107148 
   34.7104054   54.7107337 
   34.7102959   54.7107669 
   34.7102208   54.7108239 
   34.7101326   54.7109074 
   34.7100642   54.710983  
   34.7099595   54.7110494 
   34.7098537   54.7111101 
   34.7097019   54.7111617 
   34.7095297   54.7111849 
   34.7093226   54.7112062 
   34.7092313   54.7112331 
   34.7091497   54.7112649 
   34.7090818   54.7113098 
   34.7090119   54.7113878 
   34.70896     54.7114764 
   34.7088881   54.7115842 
   34.7088381   54.7116413 
   34.7087661   54.7116861 
   34.7086692   54.7117155 
   34.7085349   54.7117267 
   34.7084134   54.7117251 
   34.708214    54.7117101 
   34.7080451   54.7117082 
   34.7078884   54.7117218 
   34.707778    54.7117292 
   34.7076861   54.7117156 
   34.7075847   54.7116883 
   34.7074906   54.71164   
   34.7074634   54.7116002 
   34.7074445   54.711495  
   34.7074112   54.7114115 
   34.7073783   54.7113814 
   34.7073188   54.7113535 
   34.7072659   54.7113442 
   34.7071344   54.7113515 
   34.7070029   54.7113652 
   34.7068696   54.7113991 
   34.7067865   54.7114374 
   34.7066693   54.7115038 
   34.7065806   54.7115436 
   34.7065021   54.7115585 
   34.7064168   54.7115636 
   34.7063374   54.7115526 
   34.7062389   54.7115196 
   34.7060883   54.7114345 
   34.7059721   54.7113804 
   34.7058611   54.7113425 
   34.7057415   54.7113175 
   34.7056582   54.7112919 
   34.7056017   54.7112511 
   34.7055843   54.7112138 
   34.7055422   54.7109298 
   34.7055027   54.7107322 
   34.7054013   54.7105602 
   34.7052545   54.7103506 
   34.7051415   54.7102019 
   34.704983    54.7100198 
   34.7049024   54.7099247 
   34.7047627   54.7097103 
   34.7046884   54.7095773 
   34.7046278   54.7094597 
   34.7045451   54.709333  
   34.7044317   54.7092013 
   34.7042897   54.709037  
   34.7042089   54.7088821 
   34.7041808   54.7088132 
   34.7041841   54.708645  
   34.7041986   54.7084665 
   34.7041759   54.7083418 
   34.7041264   54.7081604 
   34.7040951   54.7079669 
   34.7040751   54.7077024 
   34.7040489   54.7075398 
   34.704001    54.7073406 
   34.7039318   54.7071622 
   34.7038774   54.7070915 
   34.7037687   54.7070067 
   34.7036198   54.706908  
   34.703514    54.7068167 
   34.7034435   54.7067088 
   34.7034242   54.7066213 
   34.7034229   54.7065437 
   34.7034536   54.7064735 
   34.7034897   54.7064115 
   34.7035618   54.7062907 
   34.7036069   54.7061956 
   34.7036188   54.7060121 
   34.7036146   54.7057938 
   34.7036002   54.7056708 
   34.7036081   54.7055528 
   34.7036612   54.7054764 
   34.7037267   54.7054097 
   34.70383     54.7053376 
   34.7040318   54.7052265 
   34.7042022   54.7051499 
   34.7043109   54.7050868 
   34.7044116   54.705009  
   34.7045082   54.7049215 
   34.7045719   54.7048031 
   34.7046153   54.7046473 
   34.7046289   54.7045213 
   34.7046199   54.7044128 
   34.7045851   54.7042598 
   34.7045931   54.7041353 
   34.7046187   54.7040352 
   34.7046759   54.7038925 
   34.7047759   54.7036999 
   34.7048168   54.703532  
   34.7048109   54.7034026 
   34.7047994   54.7032004 
   34.7047512   54.7030222 
   34.7046984   54.7028626 
   34.704618    54.7026899 
   34.7045202   54.7024726 
   34.7043931   54.7021831 
   34.7043511   54.702043  
   34.7043078   54.701898  
   34.7043106   54.7016093 
   34.7043395   54.7014818 
   34.7044069   54.7013892 
   34.7044667   54.7013282 
   34.7045807   54.7012812 
   34.7048107   54.7012293 
   34.7048562   54.7011859 
   34.7048631   54.7011189 
   34.7048489   54.7010582 
   34.7047897   54.7010125 
   34.7046889   54.7009528 
   34.7046004   54.7009102 
   34.7044844   54.7008424 
   34.7044283   54.7007846 
   34.7043659   54.700688  
   34.7043436   54.7006159 
   34.7043687   54.7005425 
   34.7044343   54.7004717 
   34.7045544   54.7003941 
   34.7046314   54.7003153 
   34.7046384   54.7002394 
   34.7046173   54.700177  
   34.7045283   54.700081  
   34.7044408   54.6999859 
   34.7043363   54.6998324 
   34.7043025   54.6997756 
   34.7042639   54.6996775 
   34.7042601   54.6995813 
   34.7042955   54.6994828 
   34.7043794   54.6994009 
   34.704481    54.6993458 
   34.7046317   54.699278  
   34.7047766   54.6992167 
   34.704825    54.6991685 
   34.7048343   54.69912   
   34.7048169   54.6990843 
   34.7047548   54.6990468 
   34.7046218   54.6989925 
   34.7045448   54.698933  
   34.7045454   54.6989007 
   34.7047613   54.6988536 
   34.704887    54.6987735 
   34.7049514   54.6986227 
   34.7050317   54.6982166 
   34.7050682   54.6981327 
   34.7051268   54.6980627 
   34.7052057   54.6980244 
   34.7053208   54.6979936 
   34.7055323   54.6979522 
   34.7056156   54.6979058 
   34.7056882   54.6977616 
   34.7056857   54.6976726 
   34.7056194   54.6975582 
   34.7054698   54.6974295 
   34.7051988   54.6972191 
   34.7049046   54.6969803 
   34.704621    54.6967732 
   34.7044325   54.6966345 
   34.704304    54.6964906 
   34.7042396   54.6963543 
   34.7042082   54.6961722 
   34.7041327   54.6960343 
   34.7040559   54.6959594 
   34.703896    54.6958557 
   34.7036683   54.6957216 
   34.7035225   54.6956083 
   34.703456    54.6955084 
   34.7034373   54.6953927 
   34.7034678   54.695181  
   34.7034759   54.6950533 
   34.7034534   54.6949893 
   34.7033882   54.6948967 
   34.7033139   54.6948372 
   34.7031262   54.6947261 
   34.7029522   54.6946328 
   34.7028155   54.6945518 
   34.7027501   54.6944706 
   34.7027459   54.6943962 
   34.7027818   54.6943447 
   34.7028399   54.6942981 
   34.7028966   54.6942548 
   34.7029979   54.6942134 
   34.7032242   54.6941324 
   34.7034015   54.694056  
   34.7034841   54.6939708 
   34.7035399   54.6938248 
   34.7035556   54.6937343 
   34.7035106   54.6936799 
   34.7034244   54.6933689 
   34.7033831   54.6931972 
   34.703379    54.693118  
   34.7034097   54.6930462 
   34.7034199   54.6929533 
   34.703354    54.6927499 
   34.7032263   54.6924936 
   34.7031679   54.692338  
   34.703161    54.6921115 
   34.7031693   54.691827  
   34.7031477   54.6917128 
   34.7030979   54.6916347 
   34.7029526   54.6914982 
   34.7026902   54.6913034 
   34.702419    54.6911427 
   34.7021822   54.6910118 
   34.7022397   54.69095   
   34.7022602   54.6907132 
   34.7022559   54.6905232 
   34.7022623   54.6904749 
   34.7022301   54.6903345 
   34.7022063   54.6896116 
   34.7022145   54.6889044 
   34.7022995   54.6881103 
   34.7023344   54.6876962 
   34.7023528   54.6870202 
   34.7023545   54.6863753 
   34.7023866   54.6858208 
   34.7024035   54.6852135 
   34.7024712   54.6844973 
   34.7025256   54.6839119 
   34.7025673   54.6834107 
   34.7024718   54.6830641 
   34.702429    54.682771  
   34.7023865   54.6824684 
   34.7014773   54.6824844 
   34.7010901   54.6824913 
   34.7003579   54.6825364 
   34.699422    54.682546  
   34.6986689   54.6825629 
   34.6981152   54.6825561 
   34.697428    54.6824988 
   34.6964294   54.6824174 
   34.6961124   54.682403  
   34.695858    54.6822143 
   34.6954987   54.681891  
   34.6951123   54.6815739 
   34.6948921   54.6812702 
   34.6945481   54.680997  
   34.6943378   54.6807525 
   34.6940631   54.6804859 
   34.6939267   54.6803168 
   34.6935474   54.6799186 
   34.693347    54.6797086 
   34.6925085   54.6794196 
   34.6913811   54.6790508 
   34.6910232   54.6789301 
   34.6906173   54.6785161 
   34.6902225   54.6780835 
   34.6899095   54.6778666 
   34.6895013   54.6775648 
   34.68905     54.677272  
   34.6888859   54.6771401 
   34.6884982   54.6768947 
   34.6882222   54.6767152 
   34.6876809   54.6763565 
   34.6872936   54.6760891 
   34.6868475   54.6758088 
   34.6863048   54.6755092 
   34.6857732   54.6752005 
   34.6854168   54.6750111 
   34.6853211   54.6749606 
   34.6850098   54.6747872 
   34.6846161   54.6745759 
   34.6838208   54.6745582 
   34.6826809   54.6745694 
   34.6817241   54.6745568 
   34.6808747   54.6745574 
   34.6798853   54.6745694 
   34.6787833   54.6745622 
   34.6783856   54.6745502 
   34.6776868   54.6745394 
   34.6776268   54.6745903 
   34.6775322   54.6747564 
   34.6773636   54.6749811 
   34.6772069   54.6752885 
   34.6770436   54.6755289 
   34.6767999   54.6758981 
   34.6766687   54.6761402 
   34.6765503   54.6762702 
   34.6764066   54.6764749 
   34.6761578   54.6766837 
   34.6758256   54.6769058 
   34.6754136   54.6772022 
   34.6751277   54.6773856 
   34.6748956   54.6775585 
   34.6747515   54.6776494 
   34.6745265   54.6777352 
   34.6741965   54.6778452 
   34.6736137   54.6780781 
   34.6729505   54.6783104 
   34.6723381   54.678554  
   34.6719648   54.6786716 
   34.6716333   54.6788484 
   34.6710086   54.6791714 
   34.670595    54.6794116 
   34.6701466   54.6796517 
   34.6696301   54.679804  
   34.6692396   54.6795599 
   34.6690138   54.6794321 
   34.668877    54.6794126 
   34.6683263   54.6791299 
   34.667594    54.6787762 
   34.6670595   54.6784796 
   34.6668361   54.6783628 
   34.6657436   54.6778757 
   34.6649255   54.6775166 
   34.6638993   54.6770812 
   34.6638222   54.677034  
   34.6635024   54.6769057 
   34.663323    54.6768608 
   34.6629306   54.676855  
   34.6624715   54.6768346 
   34.6617999   54.6768068 
   34.661148    54.6766124 
   34.6597901   54.6762246 
   34.6591992   54.6760679 
   34.6589266   54.6759896 
   34.6578625   54.675705  
   34.6571861   54.6755152 
   34.6562984   54.6752831 
   34.6557055   54.6750936 
   34.6554784   54.6750298 
   34.6549338   54.6748345 
   34.6542956   54.6746101 
   34.6534723   54.6743458 
   34.6530289   54.6741983 
   34.6526833   54.6741019 
   34.6521501   54.6739851 
   34.6514379   54.673851  
   34.6511938   54.6738028 
   34.6503737   54.6736628 
   34.649496    54.6734889 
   34.648497    54.6733139 
   34.6479838   54.6731928 
   34.6476455   54.6730126 
   34.6468837   54.6725706 
   34.6461275   54.6721387 
   34.6454452   54.6717645 
   34.6450679   54.6715166 
   34.6442192   54.671074  
   34.6434975   54.6706626 
   34.6428563   54.6702787 
   34.6422321   54.6699114 
   34.641951    54.6697785 
   34.6416449   54.669726  
   34.6411367   54.6696351 
   34.6405128   54.6695367 
   34.6400239   54.6693621 
   34.6396388   54.6692082 
   34.639081    54.6689995 
   34.6383962   54.6687528 
   34.637897    54.6685209 
   34.6371448   54.6681797 
   34.6366789   54.6679481 
   34.636433    54.6678523 
   34.636341    54.6678113 
   34.6361425   54.6681694 
   34.635972    54.6685644 
   34.6357119   54.6690833 
   34.6355922   54.6689819 
   34.6352911   54.6686976 
   34.6348593   54.668288  
   34.6344783   54.6679394 
   34.6341829   54.6676417 
   34.6340072   54.6674591 
   34.6337352   54.6671615 
   34.6333768   54.6665645 
   34.6331803   54.6662575 
   34.6329517   54.6658292 
   34.6328894   54.6652041 
   34.6328452   54.6642732 
   34.6328306   54.66386   
   34.6327683   54.6635135 
   34.6326559   54.6633515 
   34.6325162   54.6631087 
   34.6322838   54.6628484 
   34.6320034   54.6624064 
   34.6318882   54.6622901 
   34.6312518   54.6621    
   34.6302479   54.661776  
   34.6289375   54.661325  
   34.6280869   54.6609915 
   34.627874    54.6609227 
   34.6276894   54.6608812 
   34.627192    54.6608475 
   34.6265381   54.6608024 
   34.6261515   54.6607292 
   34.6258575   54.6606499 
   34.6257835   54.6607166 
   34.6257262   54.6607914 
   34.6257248   54.6608593 
   34.6257466   54.6609459 
   34.6257773   54.6610141 
   34.6257273   54.6610703 
   34.6256306   54.6610907 
   34.6254763   54.6610685 
   34.6252751   54.6610178 
   34.6251749   54.6610074 
   34.6250491   54.6610186 
   34.6249357   54.6610413 
   34.6247261   54.6611263 
   34.6245967   54.6611771 
   34.6244412   54.6612092 
   34.6243045   54.6612106 
   34.6241057   54.6611785 
   34.6239694   54.661163  
   34.6238396   54.6611669 
   34.6237092   54.6612016 
   34.6235868   54.66125   
   34.623464    54.6613219 
   34.623256    54.6613932 
   34.6231426   54.6614158 
   34.6229432   54.661412  
   34.622597    54.6613602 
   34.6223067   54.6613064 
   34.622084    54.6612814 
   34.6219141   54.6612729 
   34.6215947   54.6612731 
   34.6212904   54.661283  
   34.6211168   54.6613165 
   34.6209993   54.6613327 
   34.6208644   54.6613802 
   34.6205574   54.6615195 
   34.6203705   54.6615861 
   34.6201941   54.6616811 
   34.6199599   54.6618056 
   34.6197557   54.6618987 
   34.6194758   54.6619436 
   34.619188    54.661969  
   34.6190497   54.6619777 
   34.6189639   54.6620143 
   34.6189312   54.6620424 
   34.6189419   54.6621274 
   34.6189364   54.6621904 
   34.6188915   54.6622046 
   34.618802    54.6622128 
   34.6186854   54.6621886 
   34.6185521   54.6621618 
   34.6184311   54.6621447 
   34.618297    54.6621527 
   34.6179686   54.6621835 
   34.6177292   54.6622286 
   34.6175133   54.6622789 
   34.617416    54.6623275 
   34.6173577   54.6623812 
   34.6172916   54.6624721 
   34.6172122   54.6625346 
   34.6171191   54.6625808 
   34.6167554   54.6626979 
   34.6166249   54.6627325 
   34.6165243   54.6627432 
   34.6162903   54.6627277 
   34.6162057   54.6627061 
   34.6161317   54.662708  
   34.6159776   54.6627417 
   34.6158723   54.6627741 
   34.6156255   54.6627699 
   34.6155402   54.6627806 
   34.6154253   54.6628056 
   34.6153163   54.6628145 
   34.6151952   54.662804  
   34.6150426   54.6627633 
   34.6149542   54.6627254 
   34.6148931   54.6627129 
   34.6148123   54.6627099 
   34.614727    54.662719  
   34.6146343   54.662749  
   34.6145307   54.6628299 
   34.6143883   54.6629017 
   34.6142492   54.66295   
   34.6141063   54.6629821 
   34.6140325   54.6630422 
   34.613926    54.6631304 
   34.6138553   54.6631727 
   34.613774    54.6631908 
   34.6136166   54.6631824 
   34.6135402   54.6631656 
   34.613475    54.6631531 
   34.6133253   54.6631714 
   34.6132524   54.6631894 
   34.6131491   54.663259  
   34.6130554   54.663336  
   34.6129889   54.6633824 
   34.6128795   54.6634091 
   34.6127651   54.6634107 
   34.6125844   54.6633795 
   34.6124784   54.6633811 
   34.6123898   54.6634177 
   34.6123144   54.6634851 
   34.6122395   54.6635953 
   34.6121724   54.6637307 
   34.6121006   54.6638312 
   34.6120098   54.6639009 
   34.6118867   54.6639817 
   34.6118294   54.6640516 
   34.6117943   54.6641945 
   34.611759    54.6642791 
   34.6116652   54.6643601 
   34.6115834   54.6643999 
   34.6114645   54.6644177 
   34.6112825   54.6644495 
   34.6111285   54.6644759 
   34.6109866   54.6644611 
   34.6108214   54.6644227 
   34.6106837   54.6644048 
   34.610575    54.6644    
   34.6104686   54.6644178 
   34.6103267   54.6644661 
   34.6102075   54.6645614 
   34.6100514   54.6646185 
   34.6099256   54.6646346 
   34.6097236   54.6646186 
   34.6093925   54.6645831 
   34.6091942   54.6645913 
   34.6089832   54.6646108 
   34.6088861   54.6646481 
   34.6087953   54.6647178 
   34.6087131   54.6647803 
   34.608602    54.664824  
   34.6084205   54.6648323 
   34.6082423   54.6648165 
   34.6081529   54.6648231 
   34.6080925   54.6648413 
   34.6080426   54.6648927 
   34.6079998   54.6650031 
   34.608034    54.6650972 
   34.6080763   54.665205  
   34.6080767   54.6653813 
   34.6081194   54.665473  
   34.6082893   54.6656501 
   34.6084395   54.6658061 
   34.6085665   54.665929  
   34.6085769   54.6660112 
   34.6085549   54.6661002 
   34.6084585   54.6662426 
   34.60831     54.6663731 
   34.6081824   54.6664753 
   34.6080259   54.6665985 
   34.6078598   54.6667921 
   34.6077922   54.6669135 
   34.607789    54.6670755 
   34.6077487   54.6672488 
   34.6075909   54.6674308 
   34.6075276   54.6675547 
   34.6075123   54.667707  
   34.6074771   54.6678358 
   34.6074306   54.6679247 
   34.6073525   54.6679781 
   34.6072819   54.6680598 
   34.6072358   54.6681368 
   34.607221    54.668254  
   34.6072067   54.6683571 
   34.6071136   54.6685442 
   34.607039    54.6686328 
   34.6069605   54.6687003 
   34.6068164   54.6688049 
   34.6067337   54.6688886 
   34.6067078   54.6689682 
   34.6066922   54.6691255 
   34.6066424   54.669179  
   34.6065406   54.6692111 
   34.6064594   54.6692222 
   34.6062745   54.6692663 
   34.6061306   54.6693355 
   34.6059399   54.6693589 
   34.6057964   54.6693991 
   34.6057428   54.6694649 
   34.6057082   54.6696313 
   34.6056428   54.6696873 
   34.6054826   54.6697426 
   34.6054285   54.669829  
   34.6055018   54.6699535 
   34.6056583   54.6700813 
   34.6058067   54.6701457 
   34.6058532   54.6701901 
   34.6058401   54.6702396 
   34.6057132   54.6702827 
   34.6056745   54.6703252 
   34.6056699   54.6704147 
   34.6057043   54.6704797 
   34.6057161   54.6705927 
   34.6056663   54.6707025 
   34.6055302   54.6708309 
   34.6054098   54.67091   
   34.6052853   54.6709558 
   34.6051852   54.6709744 
   34.6046626   54.6709595 
   34.6045859   54.6709922 
   34.6044697   54.6710849 
   34.6044474   54.6712473 
   34.6044427   54.6714649 
   34.6043834   54.6715608 
   34.6041947   54.6717138 
   34.6040974   54.6718287 
   34.604085    54.6719663 
   34.6041119   54.6720326 
   34.6042154   54.672083  
   34.6043949   54.6721339 
   34.604565    54.6721763 
   34.6046546   54.6722101 
   34.6047858   54.6722937 
   34.6048555   54.6723768 
   34.604777    54.6724865 
   34.604694    54.6725988 
   34.6046825   54.6726868 
   34.6047477   54.6727425 
   34.6049355   54.6728484 
   34.6050388   54.6729098 
   34.6051127   54.6730039 
   34.6051206   54.6730839 
   34.605067    54.6731386 
   34.6049428   54.6731652 
   34.60481     54.6731532 
   34.604626    54.6730968 
   34.604527    54.6730575 
   34.604385    54.6730372 
   34.6042705   54.6730503 
   34.6041936   54.6730936 
   34.6041965   54.6731764 
   34.6042183   54.6732618 
   34.6042397   54.6733778 
   34.6041668   54.6734516 
   34.6040519   54.6734948 
   34.6040032   54.6735386 
   34.6039481   54.6736732 
   34.6038845   54.6737525 
   34.6037787   54.6738151 
   34.6036402   54.6738472 
   34.6035583   54.6738907 
   34.6034853   54.6739782 
   34.6033881   54.6740713 
   34.6033063   54.6741284 
   34.6031627   54.6741659 
   34.6030629   54.6741625 
   34.6029645   54.6741011 
   34.602849    54.6739461 
   34.6027319   54.6738709 
   34.6025998   54.6738176 
   34.6024866   54.6737809 
   34.6023394   54.6737716 
   34.6021395   54.6737894 
   34.6019434   54.6738459 
   34.6017127   54.673957  
   34.6015446   54.6740454 
   34.6014411   54.6742154 
   34.601418    54.6743998 
   34.6015296   54.6745218 
   34.6018302   54.6746865 
   34.602122    54.6748126 
   34.6022198   54.6749069 
   34.6022077   54.6750335 
   34.602083    54.6750822 
   34.6019266   54.6750536 
   34.6016344   54.6749441 
   34.6014407   54.6748902 
   34.6013352   54.674928  
   34.6012479   54.6750155 
   34.6012119   54.6751337 
   34.6011103   54.6752212 
   34.6008618   54.6752799 
   34.6006513   54.6753501 
   34.600478    54.6754507 
   34.6003165   54.6756671 
   34.6002524   54.6757741 
   34.6002116   54.6758949 
   34.6001517   54.6760267 
   34.6001569   54.6762306 
   34.6000563   54.6764778 
   34.5999866   54.676626  
   34.5999081   54.6767424 
   34.5998288   54.6771027 
   34.5998056   54.6773065 
   34.5997551   54.6774328 
   34.5995644   54.6774533 
   34.5994165   54.677491  
   34.599246    54.677462  
   34.5989144   54.6774019 
   34.5986257   54.6773475 
   34.5985161   54.6773604 
   34.5984283   54.6774671 
   34.5982989   54.677513  
   34.5982139   54.6774849 
   34.5981919   54.6774048 
   34.5981937   54.6773249 
   34.5981231   54.6772859 
   34.5979993   54.6773042 
   34.597941    54.6773617 
   34.5978776   54.6774301 
   34.5977822   54.6774486 
   34.5977501   54.6773905 
   34.597728    54.6773161 
   34.5976481   54.6772686 
   34.5975051   54.6772924 
   34.5974227   54.6773689 
   34.597277    54.6775028 
   34.5968097   54.6777997 
   34.5966405   54.6781482 
   34.5965195   54.6784559 
   34.5965166   54.6785908 
   34.5965325   54.6787342 
   34.5965263   54.6788002 
   34.5963781   54.6788404 
   34.5961507   54.6788139 
   34.5957947   54.6787865 
   34.5955809   54.6787795 
   34.595357    54.6787999 
   34.5952183   54.6788348 
   34.5951069   54.6789275 
   34.5950903   54.6790457 
   34.5951545   54.679151  
   34.5953235   54.6792487 
   34.5954979   54.6793133 
   34.5957676   54.6793649 
   34.5959997   54.6794078 
   34.59607     54.6794634 
   34.5960013   54.6795483 
   34.5957318   54.6797089 
   34.5953613   54.6799099 
   34.5951597   54.6799967 
   34.5948744   54.6800055 
   34.5946528   54.6799157 
   34.5945403   54.6798434 
   34.5944175   54.6798067 
   34.5941603   54.6798269 
   34.5940688   54.6798867 
   34.5940904   54.6799833 
   34.5941598   54.6800775 
   34.5941089   54.6801596 
   34.5940375   54.6801694 
   34.5939354   54.6801171 
   34.5938937   54.6800672 
   34.5938081   54.6800694 
   34.5937266   54.6800963 
   34.5936595   54.6801206 
   34.59355     54.6801308 
   34.5934938   54.680108  
   34.5934766   54.6800145 
   34.5934258   54.6799399 
   34.5932985   54.679892  
   34.5931797   54.6798939 
   34.5930794   54.6799125 
   34.5929541   54.679997  
   34.5928184   54.6801089 
   34.5926399   54.6802316 
   34.5924573   54.6803211 
   34.5922884   54.6803926 
   34.5922697   54.680435  
   34.5923483   54.68052   
   34.5923793   54.6806332 
   34.5923982   54.6808508 
   34.5922226   54.6810535 
   34.5919191   54.6812358 
   34.5915356   54.6813789 
   34.5913628   54.6814547 
   34.5911941   54.6815664 
   34.5910588   54.6816591 
   34.5909148   54.6817269 
   34.5907151   54.6817172 
   34.5905265   54.681658  
   34.5903993   54.6817956 
   34.590013    54.682081  
   34.5889645   54.6827242 
   34.5885202   54.682964  
   34.5884975   54.6829787 
   34.5883965   54.6828994 
   34.588289    54.6828521 
   34.5881325   54.6828106 
   34.5879616   54.6827914 
   34.5878601   54.6827968 
   34.5877583   54.6828145 
   34.5876511   54.6829295 
   34.5875975   54.6830359 
   34.5875564   54.6831177 
   34.587376    54.6833825 
   34.5872667   54.6834325 
   34.5871112   54.6835087 
   34.5870513   54.6835307 
   34.5869708   54.6835383 
   34.5868624   54.6835417 
   34.5867865   54.6835386 
   34.5867471   54.683537  
   34.5866639   54.6835348 
   34.5865514   54.6835318 
   34.5863974   54.6835329 
   34.5862543   54.6835239 
   34.5861502   54.6834848 
   34.586106    54.6834298 
   34.5860553   54.6833402 
   34.5859776   54.6832078 
   34.5859652   54.6831347 
   34.5859981   54.6830658 
   34.5860379   54.6829972 
   34.5860221   54.6829179 
   34.5859848   54.6828608 
   34.5859127   54.6827955 
   34.585712    54.6826928 
   34.5856506   54.6826234 
   34.5856098   54.6825603 
   34.585601    54.6824791 
   34.5856443   54.6824164 
   34.5856997   54.6822768 
   34.5857183   54.682216  
   34.5857128   54.6821449 
   34.5856687   54.6820838 
   34.5856066   54.6820408 
   34.5854782   54.6819913 
   34.5853564   54.6819643 
   34.5852551   54.6819596 
   34.5851469   54.6819508 
   34.5850674   54.6819098 
   34.5850368   54.681865  
   34.5850277   54.6817979 
   34.5850534   54.6817372 
   34.5850654   54.6816642 
   34.5850316   54.6816092 
   34.5849178   54.6815274 
   34.5847481   54.6814512 
   34.5846069   54.6813469 
   34.5845064   54.6813077 
   34.5843741   54.6812745 
   34.5842736   54.6812353 
   34.5842497   54.6812068 
   34.5842537   54.6811805 
   34.5842998   54.6811502 
   34.5844123   54.6811185 
   34.5845783   54.6810383 
   34.5847059   54.6809498 
   34.5847639   54.6808526 
   34.5847683   54.6808174 
   34.5847766   54.6807472 
   34.5847716   54.6806518 
   34.5846869   54.6805193 
   34.5846129   54.6804388 
   34.5846082   54.6804337 
   34.5844759   54.6802401 
   34.5842914   54.680028  
   34.5841364   54.6799154 
   34.5839429   54.6798026 
   34.5836606   54.6795939 
   34.5836093   54.6795347 
   34.583468    54.6794324 
   34.5832631   54.6793662 
   34.5830619   54.67929   
   34.582924    54.6791958 
   34.5828389   54.6790836 
   34.5827896   54.678927  
   34.5827579   54.6787685 
   34.5827102   54.6787114 
   34.5826266   54.6786926 
   34.5824693   54.6786937 
   34.5823293   54.678697  
   34.5820984   54.6787037 
   34.5818538   54.6786982 
   34.581501    54.6786778 
   34.5813544   54.6786688 
   34.5811406   54.6786939 
   34.5809612   54.6787476 
   34.580807    54.6787589 
   34.5806499   54.6787559 
   34.5805245   54.6787349 
   34.5802504   54.6786298 
   34.5799519   54.6785184 
   34.5796939   54.6784884 
   34.5794711   54.6784384 
   34.5791653   54.6783432 
   34.5790138   54.6782882 
   34.578721    54.6781822 
   34.5784849   54.6781036 
   34.5782282   54.6780046 
   34.5779433   54.6779217 
   34.5777493   54.6778231 
   34.5773203   54.6776014 
   34.5771615   54.677507  
   34.5770803   54.6773725 
   34.5769837   54.6773111 
   34.5768174   54.677233  
   34.576672    54.6771631 
   34.5765689   54.6770813 
   34.5765005   54.6769996 
   34.576441    54.6768368 
   34.5763102   54.6767386 
   34.5761781   54.6766972 
   34.5760528   54.6766721 
   34.575913    54.6766632 
   34.5758375   54.6765937 
   34.5757918   54.6764391 
   34.5757177   54.6763007 
   34.5756847   54.676203  
   34.5756021   54.6761375 
   34.575505    54.6761004 
   34.5753723   54.6760914 
   34.5751828   54.6761268 
   34.5750134   54.676203  
   34.5749012   54.6762145 
   34.5747829   54.6761894 
   34.5747208   54.6761444 
   34.5746886   54.6760102 
   34.5746138   54.6759124 
   34.5744886   54.6758791 
   34.5743417   54.6758782 
   34.5741251   54.6758647 
   34.573993    54.6758314 
   34.573784    54.6757896 
   34.5736511   54.6757928 
   34.5734685   54.6758242 
   34.5732379   54.6758228 
   34.5729448   54.6757845 
   34.5727352   54.6757791 
   34.5725881   54.6757945 
   34.5725306   54.6758591 
   34.5725427   54.6759566 
   34.5725825   54.6760624 
   34.5726289   54.6761846 
   34.5726617   54.6762903 
   34.5726323   54.6763592 
   34.5724915   54.6764111 
   34.5723448   54.676398  
   34.5721573   54.6763319 
   34.5721006   54.6762822 
   34.5720267   54.6762174 
   34.5717722   54.6760128 
   34.5716214   54.6758699 
   34.571504    54.675792  
   34.5713856   54.675771  
   34.5712665   54.6757783 
   34.5711255   54.6758343 
   34.5709273   54.6759468 
   34.5707866   54.6759907 
   34.5705341   54.6760256 
   34.5702396   54.6760644 
   34.5697494   54.6760899 
   34.569456    54.676076  
   34.5691909   54.6760418 
   34.5689959   54.6760001 
   34.5687673   54.6758931 
   34.5685324   54.6757495 
   34.5683753   54.6755659 
   34.5682264   54.675419  
   34.5681475   54.6755623 
   34.5682008   54.6759831 
   34.5682365   54.6762607 
   34.5681764   54.6763274 
   34.5682642   54.6773201 
   34.5613315   54.6782919 
   34.560662    54.6784009 
   34.5582392   54.6787964 
   34.5529159   54.6795358 
   34.5491149   54.6801477 
   34.5468392   54.6805069 
   34.5456517   54.6806869 
   34.5439966   54.6809377 
   34.542704    54.6811387 
   34.5401378   54.678373  
   34.5377883   54.6755458 
   34.5363299   54.6738252 
   34.5347217   54.6720095 
   34.5346795   54.6719656 
   34.5345977   54.6720045 
   34.534479    54.6720085 
   34.5343508   54.672005  
   34.5342286   54.6719766 
   34.534101    54.6719465 
   34.5339618   54.6719325 
   34.5338408   54.671976  
   34.5337247   54.6720494 
   34.5336462   54.6721248 
   34.5335989   54.672181  
   34.5335663   54.6722625 
   34.5335516   54.6723553 
   34.5334312   54.6724336 
   34.5333247   54.6724522 
   34.5332234   54.6724296 
   34.5331123   54.6724069 
   34.5330532   54.6724282 
   34.5330771   54.6724802 
   34.5331052   54.6725297 
   34.5331725   54.6725795 
   34.533205    54.6726809 
   34.5331268   54.6727441 
   34.5329851   54.6727754 
   34.5328229   54.6727911 
   34.5326877   54.6727876 
   34.5325466   54.6727962 
   34.5324149   54.6728178 
   34.5323075   54.672876  
   34.5321603   54.6729654 
   34.532061    54.673039  
   34.5319311   54.6731018 
   34.5317862   54.6731549 
   34.5316199   54.6731657 
   34.5314939   54.6731259 
   34.5313878   54.6730709 
   34.5313125   54.6730703 
   34.5312806   54.6731218 
   34.5312452   54.6731999 
   34.53118     54.6732463 
   34.5311027   54.67327   
   34.530967    54.6732843 
   34.5308177   54.6732855 
   34.5307613   54.6733142 
   34.5308073   54.6733744 
   34.5309631   54.6734516 
   34.5310672   54.6735341 
   34.5311159   54.6736008 
   34.5311486   54.6736924 
   34.5311209   54.6738029 
   34.5310631   54.6738882 
   34.5309586   54.6740038 
   34.5308969   54.6740769 
   34.5308313   54.6741354 
   34.5307326   54.6741815 
   34.5305753   54.6742247 
   34.5304593   54.6742335 
   34.5303535   54.674223  
   34.5302094   54.6742413 
   34.5301316   54.6742851 
   34.5299798   54.6743923 
   34.5299143   54.6744483 
   34.5297652   54.6745014 
   34.5295257   54.6745415 
   34.5292651   54.6745864 
   34.5290255   54.6746314 
   34.5288506   54.674655  
   34.5286132   54.6746629 
   34.5283025   54.6746451 
   34.5280568   54.6746488 
   34.5277925   54.6746758 
   34.5275653   54.6747258 
   34.5273599   54.6747977 
   34.5271571   54.6749392 
   34.5269491   54.6750613 
   34.5267645   54.6751407 
   34.5265262   54.6751881 
   34.5263431   54.6752012 
   34.5261381   54.6751972 
   34.5260117   54.6751768 
   34.5258295   54.6751478 
   34.5256976   54.675125  
   34.5255186   54.6751406 
   34.5254039   54.675155  
   34.5252382   54.6751408 
   34.5251373   54.6751011 
   34.5250748   54.6750287 
   34.5249834   54.6749398 
   34.5248541   54.674862  
   34.5247904   54.6748421 
   34.5246812   54.6747992 
   34.5245447   54.6747884 
   34.5244089   54.6748092 
   34.5243392   54.6748652 
   34.524287    54.6749489 
   34.5241913   54.6750419 
   34.5240909   54.6750977 
   34.5239422   54.6751353 
   34.5237805   54.6751268 
   34.5236712   54.6750895 
   34.5235657   54.6750661 
   34.5235301   54.6750981 
   34.5235678   54.675155  
   34.5235874   54.6752093 
   34.5235606   54.6752827 
   34.523456    54.6753393 
   34.5232945   54.6753817 
   34.5231231   54.6754369 
   34.5230244   54.6754757 
   34.5228113   54.675639  
   34.5226375   54.6757904 
   34.5224431   54.6759296 
   34.522335    54.6760176 
   34.5222727   54.6761158 
   34.5222569   54.6762507 
   34.5221729   54.676381  
   34.5221173   54.6764323 
   34.5220311   54.6764761 
   34.5218606   54.6764917 
   34.5216095   54.6764873 
   34.5214867   54.6764888 
   34.5213155   54.6765319 
   34.5210331   54.6766129 
   34.5207844   54.6766877 
   34.5205072   54.6767276 
   34.5203493   54.6767385 
   34.520162    54.6767515 
   34.5199504   54.6767305 
   34.5197683   54.6767039 
   34.5194777   54.6767194 
   34.5191886   54.6767317 
   34.518903    54.6767714 
   34.5185697   54.676823  
   34.5182028   54.676879  
   34.517572    54.6769452 
   34.5170606   54.6769783 
   34.5167172   54.6769828 
   34.5163685   54.6769776 
   34.5161052   54.6769609 
   34.5158173   54.6769215 
   34.5155195   54.6768875 
   34.5152518   54.6768822 
   34.5150183   54.6769005 
   34.514825    54.6769338 
   34.5146718   54.676981  
   34.514445    54.677014  
   34.5141855   54.6770143 
   34.5139645   54.6770352 
   34.5137468   54.6770383 
   34.5134618   54.677053  
   34.5131476   54.6770626 
   34.5126841   54.6770743 
   34.5119877   54.6774333 
   34.5116304   54.6776706 
   34.5112242   54.6788438 
   34.5109425   54.6797721 
   34.510526    54.6812039 
   34.5101672   54.6820832 
   34.510071    54.6823161 
   34.5099458   54.682708  
   34.5096745   54.6835443 
   34.5093296   54.6858272 
   34.507594    54.6867964 
   34.5073366   54.6869399 
   34.505161    54.6881497 
   34.5032717   54.689175  
   34.5026353   54.6895208 
   34.50135     54.6902098 
   34.4991664   54.6913353 
   34.4979666   54.6913118 
   34.496077    54.6913013 
   34.4940299   54.6912846 
   34.4920413   54.6912683 
   34.4886002   54.6912927 
   34.4848693   54.6912935 
   34.482223    54.6912935 
   34.4801488   54.6912998 
   34.475936    54.6913125 
   34.4757279   54.690419  
   34.4752522   54.6883832 
   34.4708367   54.6864067 
   34.4664387   54.6843057 
   34.4641299   54.6831074 
   34.4589793   54.6815314 
   34.454293    54.6807675 
   34.4474051   54.6798567 
   34.4412747   54.6797377 
   34.4415668   54.6817099 
   34.4415714   54.6829865 
   34.4415165   54.68513   
   34.4359189   54.688284  
   34.4347981   54.6888726 
   34.4310565   54.6894857 
   34.4273731   54.6902136 
   34.424154    54.6907376 
   34.4205756   54.6912372 
   34.4185656   54.6909451 
   34.4141791   54.6901752 
   34.411136    54.6896317 
   34.4093872   54.6892938 
   34.4061221   54.6887055 
   34.405634    54.6861466 
   34.4050913   54.68366   
   34.4057202   54.6836805 
   34.4056525   54.6831049 
   34.4045528   54.6829738 
   34.4027195   54.682732  
   34.4028966   54.6806781 
   34.4088165   54.6798453 
   34.4125882   54.679479  
   34.4154172   54.6791432 
   34.4206573   54.6788542 
   34.4241332   54.6783196 
   34.4265251   54.6783306 
   34.4284803   54.6783207 
   34.4320412   54.6784226 
   34.4365287   54.678302  
   34.4409978   54.6784439 
   34.4480933   54.6787976 
   34.451391    54.679252  
   34.4598857   54.680602  
   34.4647551   54.6822144 
   34.4672735   54.6792124 
   34.4683386   54.6780199 
   34.4704185   54.6756844 
   34.4697406   54.6735523 
   34.4690617   54.6715714 
   34.4711586   54.6698828 
   34.4732418   54.6680665 
   34.4745063   54.6682104 
   34.4756455   54.668963  
   34.4779233   54.6694242 
   34.478689    54.6702755 
   34.4797502   54.6707467 
   34.4805323   54.6690124 
   34.4831271   54.6644656 
   34.4801757   54.6634467 
   34.4767565   54.6614366 
   34.4770729   54.6596976 
   34.4812436   54.6585954 
   34.4847365   54.6582224 
   34.4867614   54.6580006 
   34.4906368   54.6577785 
   34.4959764   54.6578505 
   34.50093     54.6584883 
   34.5047289   54.6609252 
   34.5090388   54.661209  
   34.5109934   54.6612393 
   34.5167006   54.661006  
   34.5178703   54.660966  
   34.5181396   54.6566676 
   34.5181501   54.6565166 
   34.5184628   54.6519142 
   34.5185561   54.648688  
   34.5124677   54.6485165 
   34.5124709   54.646878  
   34.5127174   54.6457656 
   34.5129164   54.6422258 
   34.5128995   54.6409124 
   34.5130064   54.6391229 
   34.5131484   54.6372123 
   34.5189737   54.637374  
   34.5188703   54.6362722 
   34.5189415   54.634958  
   34.5190658   54.6329462 
   34.5191572   54.6292159 
   34.5193333   54.6276086 
   34.5194571   54.6261326 
   34.5194771   54.6238277 
   34.5197967   54.6186114 
   34.5200289   54.6135567 
   34.5200913   54.6120814 
   34.5200908   54.6120813 
   34.5201184   54.6114438 
   34.5202427   54.6093718 
   34.5203677   54.6064296 
   34.5205106   54.6037505 
   34.5206167   54.6011931 
   34.5206895   54.598645  
   34.5116107   54.5983195 
   34.5059655   54.5981932 
   34.5005992   54.5981667 
   34.491939    54.5979303 
   34.4881923   54.597902  
   34.4884389   54.5961867 
   34.4885275   54.5950053 
   34.4886697   54.5929463 
   34.4889697   54.5900598 
   34.489236    54.5872434 
   34.4891187   54.5831455 
   34.4892289   54.5797036 
   34.4893502   54.5784942 
   34.4811672   54.5783305 
   34.4780313   54.5782697 
   34.473574    54.5781784 
   34.4693062   54.5780772 
   34.463878    54.5779055 
   34.4607034   54.5779401 
   34.4583127   54.5778148 
   34.4584717   54.5769704 
   34.458531    54.576104  
   34.4594351   54.5750201 
   34.462917    54.5709034 
   34.4643703   54.5693809 
   34.4646283   54.5689741 
   34.4672552   54.5655202 
   34.4665326   54.5648632 
   34.466017    54.564618  
   34.4652457   54.5640252 
   34.4660089   54.5624316 
   34.4628335   54.5613302 
   34.4616758   54.5607803 
   34.4606074   54.5600517 
   34.4582925   54.5588799 
   34.4571941   54.558134  
   34.4561426   54.556795  
   34.4548452   54.5551701 
   34.4528328   54.5535488 
   34.4526583   54.5534083 
   34.4520504   54.5532996 
   34.4501838   54.5529643 
   34.4470465   54.5523553 
   34.4464988   54.5507176 
   34.4460769   54.5491494 
   34.4458127   54.5481423 
   34.4447284   54.5471336 
   34.4428747   54.5454847 
   34.4358401   54.545207  
   34.4350758   54.5451376 
   34.435473    54.543822  
   34.4359748   54.5415157 
   34.4391883   54.5316516 
   34.4398387   54.5296976 
   34.4399788   54.5291411 
   34.4402852   54.5282155 
   34.4412965   54.5253659 
   34.4436207   54.5191201 
   34.444682    54.5166374 
   34.4420808   54.5140401 
   34.4386833   54.5130848 
   34.4363528   54.5131295 
   34.4355426   54.5102934 
   34.4264943   54.5106984 
   34.4185262   54.5136694 
   34.3832782   54.5146142 
   34.3524868   54.5146137 
   34.3392363   54.5152013 
   34.3327113   54.5154164 
   34.3184579   54.5155753 
   34.3103063   54.5156992 
   34.3019792   54.5156373 
   34.2985393   54.5147097 
   34.2987474   54.5127296 
   34.297351    54.5116767 
   34.2964095   54.5119334 
   34.2960214   54.5126796 
   34.2944543   54.5125748 
   34.2937148   54.5113843 
   34.2923596   54.5116499 
   34.2901156   54.5129275 
   34.287182    54.514324  
   34.2868914   54.5167254 
   34.2853484   54.5176322 
   34.2824981   54.5167047 
   34.2794876   54.5171211 
   34.2775653   54.5178978 
   34.2739331   54.5197529 
   34.2724861   54.5208227 
   34.2702106   54.5210299 
   34.2634733   54.5194761 
   34.2599746   54.519502  
   34.2524788   54.5213148 
   34.2506941   54.5203307 
   34.2504264   54.5176892 
   34.2469462   54.5149438 
   34.2484632   54.5123019 
   34.2427522   54.5081574 
   34.2312408   54.5066031 
   34.2224957   54.5042197 
   34.2183016   54.5083646 
   34.2128582   54.5090382 
   34.2000083   54.514374  
   34.1968032   54.5148038 
   34.1815837   54.520074  
   34.171655    54.5182572 
   34.1666579   54.5241994 
   34.1665225   54.5301416 
   34.1576092   54.5323022 
   34.1584193   54.5350033 
   34.163686    54.5406755 
   34.1586891   54.5427349 
   34.1527464   54.553944  
   34.1403124   54.5557966 
   34.1378907   54.5587042 
   34.1381606   54.5615403 
   34.1350544   54.5643763 
   34.1261412   54.5623504 
   34.1195237   54.5624854 
   34.1177682   54.5595142 
   34.1123663   54.5574884 
   34.1053437   54.5573532 
   34.102778    54.5529978 
   34.0842765   54.54611   
   34.0779292   54.5458398 
   34.0791445   54.5482707 
   34.0788742   54.5534026 
   34.0799544   54.5577242 
   34.0692853   54.5612691 
   34.0629378   54.5647803 
   34.0625325   54.566536  
   34.0578059   54.56424   
   34.0553751   54.5627544 
   34.0521339   54.5631595 
   34.0494329   54.5624843 
   34.0456515   54.561944  
   34.0445711   54.5615388 
   34.0393042   54.562214  
   34.0355229   54.5592428 
   34.0341724   54.5589727 
   34.0290242   54.5558959 
   34.0251242   54.5566767 
   34.019587    54.5597828 
   34.0168858   54.5635641 
   34.0085127   54.5641042 
   34.0027054   54.5678855 
   33.9990589   54.571937  
   33.9970331   54.5730173 
   33.9902804   54.5770687 
   33.9870391   54.5797697 
   33.9740744   54.5778788 
   33.9739394   54.5762582 
   33.9635407   54.5747725 
   33.9620548   54.58166   
   33.9530064   54.5839557 
   33.9508456   54.5823351 
   33.9134371   54.5762572 
   33.9056046   54.5685592 
   33.9045244   54.5639675 
   33.8992576   54.5618066 
   33.8966918   54.5589705 
   33.8902096   54.5547839 
   33.8883517   54.5552138 
   33.872653    54.5573496 
   33.867251    54.5585649 
   33.8644149   54.5596453 
   33.862389    54.5626164 
   33.859958    54.5642369 
   33.8584722   54.5695039 
   33.8559062   54.5719347 
   33.8375396   54.5674778 
   33.8228193   54.5655868 
   33.821334    54.560151  
   33.8233598   54.5594758 
   33.8190383   54.5565046 
   33.8037777   54.5562343 
   33.7920284   54.5552888 
   33.7917581   54.5593403 
   33.7816291   54.5656875 
   33.7758219   54.5675781 
   33.7719051   54.5736553 
   33.7608312   54.570819  
   33.7606964   54.5648768 
   33.745166    54.5590694 
   33.7288253   54.5519115 
   33.7092434   54.545834  
   33.707218    54.5379672 
   33.7043819   54.5382373 
   33.7004654   54.5408032 
   33.6994574   54.5405793 
   33.6980345   54.540263  
   33.6973594   54.5379671 
   33.6995202   54.5372919 
   33.6991151   54.5359414 
   33.6943884   54.5339155 
   33.695199    54.5286486 
   33.6979001   54.5251373 
   33.6933088   54.5176757 
   33.6930389   54.5142995 
   33.6673796   54.5098424 
   33.649148    54.5078164 
   33.6510388   54.5056219 
   33.6506337   54.5041363 
   33.6542802   54.5017054 
   33.6541452   54.5003549 
   33.6560359   54.4992746 
   33.6596825   54.4950881 
   33.6595475   54.4929273 
   33.6565765   54.4911716 
   33.655226    54.4911715 
   33.6509046   54.4876602 
   33.6482036   54.48739   
   33.6475285   54.4852292 
   33.6442873   54.4841488 
   33.6442874   54.4818529 
   33.6336532   54.4768606 
   33.655632    54.4728047 
   33.6706227   54.4696988 
   33.6864233   54.4751011 
   33.7029436   54.4692905 
   33.7022245   54.4658842 
   33.6991184   54.4639934 
   33.6962824   54.4638583 
   33.6956072   54.4622377 
   33.6872343   54.4589964 
   33.6834529   54.4587262 
   33.6814273   54.4567004 
   33.6773758   54.4553498 
   33.6762955   54.4534254 
   33.660765    54.4486984 
   33.66036     54.4464025 
   33.6594147   54.4459973 
   33.6591447   54.4434314 
   33.6573891   54.442351  
   33.6561737   54.4405953 
   33.6519873   54.4389409 
   33.6460453   54.4350243 
   33.6402382   54.4332686 
   33.632405    54.4398859 
   33.6315946   54.4415065 
   33.619575    54.4450176 
   33.6170093   54.4405609 
   33.6278137   54.4307024 
   33.6345663   54.429217  
   33.6287594   54.423511  
   33.627139    54.4194595 
   33.6294351   54.415273  
   33.6374032   54.4106814 
   33.6314611   54.4085205 
   33.6318665   54.4036587 
   33.6365933   54.4019031 
   33.6318666   54.4002824 
   33.6284905   54.3982229 
   33.6257895   54.3972775 
   33.6218731   54.3980877 
   33.6203876   54.3968722 
   33.616201    54.3970072 
   33.6140402   54.3978175 
   33.6031011   54.3982225 
   33.5989146   54.3976822 
   33.5947281   54.3979522 
   33.594323    54.3960615 
   33.5831139   54.3957912 
   33.5804129   54.3966015 
   33.5762264   54.3957911 
   33.5723278   54.3936172 
   33.5812352   54.390176  
   33.5866256   54.3880934 
   33.5883813   54.3866079 
   33.5970245   54.3848524 
   33.6005362   54.3764118 
   33.6037775   54.3749263 
   33.6149866   54.3739811 
   33.6256561   54.362367  
   33.627547    54.3583155 
   33.6315986   54.3564249 
   33.6360556   54.3488621 
   33.6337598   54.3479168 
   33.6330846   54.3464312 
   33.641998    54.3431901 
   33.6428084   54.3404891 
   33.6514517   54.339105  
   33.6553681   54.338835  
   33.6573938   54.3397804 
   33.6622558   54.335864  
   33.6768412   54.3349189 
   33.6754908   54.3331632 
   33.6827837   54.3289767 
   33.6842692   54.3299221 
   33.6899413   54.329382  
   33.6862951   54.3254655 
   33.6881859   54.3242163 
   33.6888938   54.3216198 
   33.6821794   54.3149288 
   33.6815857   54.3117129 
   33.6716625   54.3051318 
   33.6877771   54.2953323 
   33.655371    54.2765093 
   33.652805    54.2782649 
   33.6518601   54.2686426 
   33.6433519   54.2703981 
   33.6212038   54.2685071 
   33.6234997   54.2663125 
   33.6225545   54.2650971 
   33.6241751   54.2642868 
   33.6226896   54.2633414 
   33.6230948   54.2617208 
   33.6302879   54.2576702 
   33.6180981   54.2586146 
   33.6082395   54.2579392 
   33.6033777   54.2588845 
   33.6027026   54.2551031 
   33.6076996   54.251997  
   33.6041887   54.2424084 
   33.6085104   54.2416994 
   33.6095909   54.2385933 
   33.6252567   54.2368379 
   33.6205302   54.2317059 
   33.6113708   54.2236279 
   33.615804    54.2206317 
   33.6286343   54.2077684 
   33.6209366   54.204257  
   33.6263388   54.2007457 
   33.6337667   54.1964242 
   33.6337669   54.1932843 
   33.6294456   54.187207  
   33.6214777   54.1854512 
   33.6170211   54.185046  
   33.6106739   54.1807243 
   33.5986544   54.1836952 
   33.5642171   54.175693  
   33.558651    54.1798615 
   33.5609759   54.1751527 
   33.5655678   54.1697508 
   33.5635421   54.1690755 
   33.5689443   54.1642138 
   33.5628672   54.1614789 
   33.5592212   54.1541861 
   33.5652986   54.1499997 
   33.5590865   54.1471635 
   33.5638137   54.1360557 
   33.5715119   54.1297085 
   33.569081    54.128493  
   33.5728626   54.1239014 
   33.5363994   54.1203557 
   33.5238401   54.1133329 
   33.5237054   54.1052299 
   33.5270392   54.1027397 
   33.56368     54.106581  
   33.557603    54.1022256 
   33.5480146   54.0981739 
   33.5412624   54.0939873 
   33.5260019   54.091286  
   33.5250566   54.0893953 
   33.520735    54.08872   
   33.5174938   54.0903405 
   33.5174937   54.0930415 
   33.5114164   54.0930414 
   33.510066    54.0914208 
   33.5103362   54.0880446 
   33.506555    54.0857486 
   33.4984519   54.0875042 
   33.4931849   54.0880443 
   33.4848114   54.0988482 
   33.4826507   54.0965523 
   33.4785992   54.0956069 
   33.469956    54.0966871 
   33.4687777   54.0877075 
   33.4683359   54.085613  
   33.4588826   54.0825067 
   33.4559114   54.0833169 
   33.4525352   54.0825741 
   33.4524003   54.0808185 
   33.45159     54.0804133 
   33.4532107   54.0773072 
   33.4519953   54.076767  
   33.4525355   54.0758216 
   33.4501047   54.0736608 
   33.4510501   54.0728505 
   33.4499698   54.0713649 
   33.4517255   54.0701495 
   33.4542914   54.0704196 
   33.4560472   54.068529  
   33.4560473   54.0663682 
   33.4499702   54.063397  
   33.4513208   54.0604259 
   33.446324    54.0598856 
   33.4449737   54.0554289 
   33.4414624   54.0546186 
   33.4424078   54.0534031 
   33.4510509   54.0556991 
   33.4544273   54.0532683 
   33.4586139   54.0519178 
   33.4634756   54.052053  
   33.4652894   54.0490885 
   33.4738745   54.0519181 
   33.4760352   54.0536738 
   33.478736    54.0581305 
   33.4852184   54.0576916 
   33.4898101   54.0580969 
   33.4919709   54.0563412 
   33.494402    54.0521547 
   33.4945371   54.0506692 
   33.4925115   54.0486434 
   33.4968334   54.042026  
   33.4989942   54.0414858 
   33.5019654   54.0389199 
   33.4981842   54.0348684 
   33.4865699   54.0350032 
   33.4817081   54.0355434 
   33.473335    54.0352731 
   33.4687436   54.0283855 
   33.458615    54.02744   
   33.4601006   54.0254143 
   33.4656376   54.0266298 
   33.4702293   54.027035  
   33.4741456   54.0290608 
   33.4911618   54.0302766 
   33.4941332   54.0243006 
   33.4991299   54.0267316 
   33.5011556   54.0271368 
   33.5037217   54.0251111 
   33.5103392   54.0235919 
   33.5057477   54.0194052 
   33.506288    54.0169744 
   33.5045916   54.0165863 
   33.508251    54.0165619 
   33.5104309   54.016126  
   33.511303    54.0146312 
   33.5124864   54.0141329 
   33.5145418   54.0139461 
   33.5166595   54.0115794 
   33.5167841   54.010832  
   33.5153516   54.0095863 
   33.5137323   54.0092748 
   33.5116146   54.0097731 
   33.5106804   54.0095862 
   33.5106181   54.008652  
   33.5121753   54.0067212 
   33.5130473   54.0066589 
   33.5144175   54.0066589 
   33.516286    54.0072195 
   33.5173448   54.0072195 
   33.5183414   54.0069081 
   33.5182791   54.006223  
   33.5164729   54.0059116 
   33.5154764   54.0056624 
   33.5163484   54.0051642 
   33.52021     54.0048528 
   33.5208328   54.0044168 
   33.5205214   54.0037317 
   33.5175318   54.0031088 
   33.5215805   53.998749  
   33.5239472   53.9986868 
   33.5291166   54.0015519 
   33.5296771   54.0027976 
   33.5290542   54.003981  
   33.5291788   54.0051021 
   33.5337877   54.0066592 
   33.5349087   54.0074067 
   33.5270609   54.0100225 
   33.5327909   54.0122648 
   33.542943    54.0146317 
   33.5476141   54.0161889 
   33.5576417   54.0181198 
   33.5610049   54.0205956 
   33.5708456   54.0207204 
   33.5746448   54.0243951 
   33.5795028   54.026326  
   33.5824924   54.0266375 
   33.5779454   54.0319315 
   33.5887826   54.0347967 
   33.590153    54.0328037 
   33.5981254   54.0296896 
   33.5970667   54.0256412 
   33.5983126   54.0222779 
   33.6085893   54.0220289 
   33.6123263   54.0226518 
   33.6141325   54.0232124 
   33.6132605   54.0249563 
   33.6196131   54.0298768 
   33.6262776   54.0257662 
   33.6274612   54.0214064 
   33.642347    54.0212197 
   33.641288    54.0240848 
   33.6502569   54.0230884 
   33.6550525   54.0270746 
   33.6579798   54.0277598 
   33.6609071   54.0282581 
   33.6638968   54.0270747 
   33.6751078   54.0270749 
   33.6827687   54.0262031 
   33.6951631   54.0247085 
   33.6995853   54.0235251 
   33.7033846   54.0240858 
   33.7068724   54.0248955 
   33.7108586   54.024662  
   33.7157167   54.023977  
   33.7180213   54.0226067 
   33.7218206   54.0204269 
   33.7248725   54.0203024 
   33.7282359   54.0192436 
   33.7312879   54.0172506 
   33.7331565   54.0150707 
   33.7356478   54.0140897 
   33.7392604   54.0125327 
   33.7412535   54.0102282 
   33.7421255   54.0099168 
   33.7420633   54.0071141 
   33.744181    54.0059307 
   33.7456759   54.0040622 
   33.744991    54.0007612 
   33.7454893   53.9990173 
   33.7488525   54.0009481 
   33.7497245   54.0010104 
   33.7506588   54.0006367 
   33.7510948   53.9994533 
   33.7499114   53.9988305 
   33.7489772   53.9987059 
   33.7497869   53.9979585 
   33.7510948   53.9981454 
   33.7536484   54.0000762 
   33.7538975   54.0010727 
   33.754894    54.0011973 
   33.7572608   54.0014465 
   33.7603748   54.0032528 
   33.7633644   54.0049968 
   33.7673505   54.0053082 
   33.7686585   54.0033152 
   33.7651708   54.0013843 
   33.7649217   53.9986439 
   33.7807419   53.9951095 
   33.7989288   53.9909368 
   33.8115101   53.9913107 
   33.8174269   53.992681  
   33.8181746   53.9868887 
   33.8191713   53.9841482 
   33.8150608   53.9789786 
   33.8136284   53.9754907 
   33.8107011   53.9748522 
   33.7974975   53.9655718 
   33.7984947   53.9496895 
   33.801173    53.9472916 
   33.8041626   53.9465442 
   33.8055952   53.9461706 
   33.80709     53.9464197 
   33.8092076   53.947354  
   33.8142524   53.9495963 
   33.8199825   53.9500324 
   33.8240309   53.9496587 
   33.8269583   53.9488491 
   33.8315674   53.9462332 
   33.8377336   53.9418112 
   33.8365504   53.937825  
   33.8318169   53.9379495 
   33.8225989   53.9393196 
   33.8198583   53.9403784 
   33.8016717   53.9375754 
   33.8006135   53.9237484 
   33.8011119   53.9214283 
   33.8069673   53.9044873 
   33.8067806   53.9018091 
   33.8062203   53.8963904 
   33.8382346   53.8810224 
   33.8480754   53.8806489 
   33.8502554   53.8797769 
   33.8519993   53.8795901 
   33.8536187   53.8800261 
   33.8557986   53.8801507 
   33.8590373   53.8795279 
   33.8614042   53.8785937 
   33.8844489   53.8804626 
   33.88775     53.8804626 
   33.8896808   53.8792793 
   33.8893694   53.8785319 
   33.8893694   53.8780336 
   33.8898055   53.8769748 
   33.8899924   53.8757914 
   33.8882486   53.8720543 
   33.8938541   53.8718053 
   33.8960963   53.8722102 
   33.8979648   53.8722102 
   33.8995842   53.8718365 
   33.9018264   53.8713383 
   33.9034458   53.8704663 
   33.9039305   53.8702586 
   33.9078057   53.8685979 
   33.9115428   53.8672277 
   33.915529    53.8671032 
   33.9251206   53.8680999 
   33.9287329   53.8697193 
   33.9314732   53.8743284 
   33.9643589   53.8738306 
   33.9634868   53.8759482 
   33.963362    53.880059  
   33.9666008   53.8803081 
   33.9712098   53.8796854 
   33.9775627   53.87981   
   33.9788083   53.8814294 
   33.981175    53.8823014 
   33.9866559   53.8827998 
   33.9898946   53.8840455 
   33.9941299   53.8840456 
   33.9986552   53.8823122 
   34.0019861   53.8804387 
   34.0036516   53.875859  
   34.0113543   53.8687813 
   34.0119789   53.8656587 
   34.0024033   53.8596216 
   34.0265512   53.8592056 
   34.0375841   53.8608712 
   34.0457027   53.8621203 
   34.0454944   53.8664919 
   34.0477841   53.8694064 
   34.054862    53.8681574 
   34.0625643   53.8671167 
   34.068393    53.8687822 
   34.0740137   53.8679496 
   34.0844221   53.8704478 
   34.0885854   53.8729459 
   34.087128    53.8769011 
   34.0964766   53.8808775 
   34.1023243   53.8814811 
   34.1065429   53.881701  
   34.1060303   53.8847071 
   34.107112    53.886061  
   34.1081099   53.888774  
   34.1102175   53.8893639 
   34.1108475   53.8921149 
   34.1143485   53.888774  
   34.1148143   53.8868938 
   34.1224322   53.8869908 
   34.1324246   53.8869697 
   34.137       53.8882664 
   34.1437428   53.8900326 
   34.1534825   53.8923383 
   34.1582274   53.8934666 
   34.1617237   53.8949873 
   34.1663022   53.8971456 
   34.1721293   53.8998923 
   34.1787889   53.902786  
   34.1806203   53.9008242 
   34.1914421   53.9067094 
   34.197186    53.910093  
   34.2011817   53.9121524 
   34.2045948   53.9102401 
   34.2123365   53.901707  
   34.2182748   53.8941814 
   34.2249897   53.8860095 
   34.2264882   53.881201  
   34.229152    53.873938  
   34.2329812   53.8673611 
   34.2337305   53.8643667 
   34.2314828   53.8638758 
   34.2309001   53.8604885 
   34.2303174   53.8564135 
   34.2293185   53.8510123 
   34.228486    53.8447264 
   34.227645    53.8394325 
   34.2319823   53.8356887 
   34.2425864   53.8353303 
   34.2428218   53.8342882 
   34.2533583   53.8302933 
   34.2594212   53.8304323 
   34.26466     53.8307797 
   34.2690158   53.8311271 
   34.2714881   53.8264371 
   34.2653075   53.8221635 
   34.2641891   53.8139973 
   34.2668379   53.8142058 
   34.2662493   53.8148313 
   34.2664847   53.8171597 
   34.2686038   53.8210168 
   34.2708406   53.8224067 
   34.2797878   53.8262286 
   34.2851443   53.8284868 
   34.2917958   53.8259854 
   34.2953864   53.8197659 
   34.2983296   53.8145881 
   34.2992125   53.8108346 
   34.3029209   53.8072198 
   34.3074534   53.8071851 
   34.3079831   53.8013104 
   34.3214039   53.7999199 
   34.3215216   53.7932097 
   34.3238173   53.7917841 
   34.3294092   53.7888632 
   34.3378855   53.7838902 
   34.3395337   53.7823947 
   34.3495404   53.773525  
   34.3559564   53.7678892 
   34.3523658   53.7590513 
   34.3494288   53.7517942 
   34.3490125   53.7509615 
   34.3567216   53.7417527 
   34.3583109   53.7377838 
   34.35784     53.7357993 
   34.3613718   53.7341976 
   34.3635497   53.7339191 
   34.3692006   53.7321432 
   34.3717906   53.7323521 
   34.3736153   53.7324566 
   34.3891551   53.7321084 
   34.3920394   53.7321084 
   34.3944528   53.732387  
   34.4065197   53.7335709 
   34.4075441   53.7344982 
   34.4092621   53.7336424 
   34.4096013   53.7332257 
   34.4102535   53.7329479 
   34.4115105   53.7318697 
   34.411354    53.7307122 
   34.4136627   53.7281426 
   34.4149931   53.7270854 
   34.4159844   53.7253259 
   34.4170671   53.7246159 
   34.4177752   53.7244972 
   34.4188973   53.7203691 
   34.4202627   53.7197073 
   34.4216845   53.7179861 
   34.4219715   53.7173686 
   34.4237584   53.7166044 
   34.4276845   53.7155237 
   34.426315    53.7146746 
   34.4274106   53.7133392 
   34.4275672   53.7124823 
   34.4258976   53.7092245 
   34.4255063   53.7089465 
   34.4254802   53.7085451 
   34.4250237   53.7083366 
   34.4238889   53.7083134 
   34.4232106   53.7085837 
   34.4229106   53.7091472 
   34.4225193   53.7096027 
   34.4219193   53.7096105 
   34.4209149   53.7091472 
   34.4203541   53.7086223 
   34.4203801   53.7081513 
   34.4209801   53.7078502 
   34.4213975   53.7075337 
   34.4213454   53.7073329 
   34.4209019   53.7071554 
   34.4205627   53.7071631 
   34.4199758   53.7073793 
   34.4191019   53.7074179 
   34.4186193   53.7072557 
   34.4172758   53.7064914 
   34.4170279   53.7062057 
   34.4161018   53.7055186 
   34.416141    53.7052252 
   34.4165062   53.7050399 
   34.4169236   53.7050399 
   34.4174714   53.7052869 
   34.4178758   53.7052174 
   34.4181497   53.7048237 
   34.4186062   53.7043913 
   34.4195584   53.7038045 
   34.4211758   53.703318  
   34.4216845   53.7029783 
   34.4215019   53.7026926 
   34.4205236   53.702654  
   34.4185671   53.7027544 
   34.4179149   53.7028779 
   34.4171453   53.7028239 
   34.4166366   53.7026771 
   34.4165062   53.702376  
   34.416741    53.7021135 
   34.417654    53.701627  
   34.4177714   53.70121   
   34.4174845   53.7008934 
   34.4171714   53.7006618 
   34.4174323   53.7003915 
   34.4187627   53.7001599 
   34.420341    53.6999668 
   34.4211106   53.6997892 
   34.4214236   53.6994649 
   34.4213062   53.6990865 
   34.420928    53.6987931 
   34.4207454   53.6984301 
   34.4215149   53.6978741 
   34.424241    53.6966694 
   34.425428    53.6956732 
   34.4254671   53.6949936 
   34.4249715   53.6946846 
   34.4254541   53.693727  
   34.4266541   53.6934721 
   34.4271889   53.694005  
   34.4274367   53.6953334 
   34.4297846   53.6962447 
   34.4313107   53.6935107 
   34.432602    53.692051  
   34.4336976   53.6916417 
   34.4347542   53.6913328 
   34.4372064   53.6886526 
   34.4396194   53.6867988 
   34.4402846   53.6845277 
   34.4418629   53.6843501 
   34.4432586   53.6843269 
   34.4431803   53.6837398 
   34.4407803   53.6830677 
   34.4416412   53.6826351 
   34.446363    53.6833922 
   34.4523369   53.6834385 
   34.4600326   53.6833304 
   34.4627979   53.6837939 
   34.4653936   53.684605  
   34.4684458   53.6847749 
   34.4697762   53.6845895 
   34.4705588   53.6839793 
   34.4705849   53.6833536 
   34.4701284   53.6827124 
   34.4680805   53.6815922 
   34.4683023   53.6805647 
   34.4684588   53.6785483 
   34.4686284   53.6782161 
   34.4692414   53.6780848 
   34.4702979   53.6780848 
   34.472711    53.677598  
   34.4734154   53.677598  
   34.4748762   53.6772194 
   34.4757762   53.6765782 
   34.4763371   53.6760219 
   34.475998    53.6749479 
   34.4754632   53.6740593 
   34.4745241   53.6736498 
   34.4735458   53.6737503 
   34.4729197   53.6730239 
   34.4733762   53.6726144 
   34.4748502   53.6719962 
   34.4773023   53.6718958 
   34.4783719   53.6722821 
   34.4785415   53.672568  
   34.4785415   53.6730317 
   34.4782806   53.6732171 
   34.4778893   53.6732635 
   34.4776415   53.6734953 
   34.4776284   53.6738275 
   34.4778502   53.6740207 
   34.4783067   53.6739589 
   34.4786067   53.673673  
   34.4789328   53.6734721 
   34.4794545   53.6733562 
   34.4807719   53.6732944 
   34.482872    53.673333  
   34.4851937   53.6735957 
   34.4870329   53.674098  
   34.4882459   53.6742911 
   34.4888329   53.6742139 
   34.4889894   53.6739666 
   34.4886894   53.673503  
   34.4887155   53.6727921 
   34.4892503   53.672174  
   34.4901372   53.6717567 
   34.4908155   53.6716562 
   34.4913112   53.6717567 
   34.4920416   53.6721044 
   34.4929546   53.6722744 
   34.4938416   53.672313  
   34.494559    53.6725835 
   34.4950677   53.6729698 
   34.4953155   53.6738893 
   34.4957982   53.6740902 
   34.4964895   53.6739821 
   34.4967242   53.6737193 
   34.4967634   53.6733987 
   34.4970308   53.6730123 
   34.4973373   53.6721546 
   34.4977025   53.671834  
   34.4984982   53.671579  
   34.4993264   53.6715287 
   34.4998743   53.6713897 
   34.5003634   53.6710651 
   34.5006634   53.6707406 
   34.5005982   53.6703233 
   34.5009504   53.6681827 
   34.5008525   53.6677229 
   34.5006047   53.6673133 
   34.4997504   53.6666681 
   34.4982569   53.6658952 
   34.4972945   53.6673171 
   34.4965382   53.6670329 
   34.4955236   53.6654591 
   34.4966304   53.6632841 
   34.4941955   53.6605187 
   34.4915208   53.6591742 
   34.4920742   53.6567802 
   34.4931625   53.6554356 
   34.492148    53.6548234 
   34.4913916   53.6548234 
   34.4903955   53.6551404 
   34.489713    53.655206  
   34.4883849   53.6546922 
   34.4884771   53.6537083 
   34.4887723   53.6531507 
   34.4896392   53.6527681 
   34.4903586   53.6529321 
   34.4910227   53.6535224 
   34.4909489   53.6538942 
   34.4911703   53.6540363 
   34.4921848   53.6538176 
   34.492978    53.6528337 
   34.4943062   53.6522433 
   34.4958926   53.6526697 
   34.4963537   53.6521231 
   34.4956712   53.651478  
   34.4925353   53.6506143 
   34.4930703   53.6502644 
   34.4965751   53.6503519 
   34.4990654   53.6493241 
   34.5006149   53.6480777 
   34.5002828   53.6477168 
   34.4984935   53.647531  
   34.4958188   53.6488102 
   34.4943062   53.6486572 
   34.4935868   53.6477387 
   34.4942508   53.6469405 
   34.4965382   53.6465578 
   34.4973867   53.6457705 
   34.4906722   53.6409372 
   34.489879    53.6397123 
   34.4901742   53.638739  
   34.4909858   53.6381375 
   34.4922402   53.6380938 
   34.4934392   53.6384437 
   34.4944538   53.6380063 
   34.4947673   53.6374594 
   34.4944538   53.6364642 
   34.4934761   53.6360048 
   34.492314    53.6360048 
   34.490414    53.6372079 
   34.4894917   53.6370438 
   34.4879422   53.6353486 
   34.486448    53.6350095 
   34.4850276   53.635458  
   34.4845849   53.6361579 
   34.4862635   53.6370876 
   34.4863004   53.6376344 
   34.4858208   53.6378641 
   34.4844742   53.6375469 
   34.4821869   53.6375579 
   34.4797151   53.6365735 
   34.4788665   53.6368688 
   34.4791432   53.6376454 
   34.4797704   53.6381594 
   34.4793461   53.6383781 
   34.4776491   53.6379844 
   34.4756753   53.6379188 
   34.4753433   53.6364751 
   34.4741442   53.6359064 
   34.4727239   53.6359611 
   34.4726501   53.635447  
   34.4734986   53.6345173 
   34.4747345   53.6318594 
   34.4758967   53.6303828 
   34.4753617   53.6279215 
   34.4774646   53.6258321 
   34.4790141   53.6256023 
   34.4800655   53.6243442 
   34.4799549   53.6238848 
   34.4812646   53.622922  
   34.483017    53.6226266 
   34.4841238   53.6218389 
   34.4848247   53.6206791 
   34.4878868   53.6204713 
   34.4885693   53.6202524 
   34.4876839   53.6192896 
   34.4899159   53.6180532 
   34.4927382   53.6178891 
   34.4972945   53.6157772 
   34.4987887   53.6156131 
   34.5000061   53.6160289 
   34.4983091   53.6174623 
   34.4989916   53.618272  
   34.5029391   53.6179219 
   34.5039168   53.6175827 
   34.5050051   53.6176046 
   34.5064808   53.6179766 
   34.5079381   53.6169699 
   34.509008    53.6165979 
   34.5103546   53.6166417 
   34.5111293   53.6169809 
   34.5118118   53.6169918 
   34.5120161   53.6166935 
   34.5118303   53.616193  
   34.5119041   53.6156022 
   34.5122914   53.6151426 
   34.5124206   53.6145188 
   34.5120516   53.613917  
   34.5119963   53.6134683 
   34.5122914   53.6128665 
   34.51314     53.6122646 
   34.5130846   53.6120129 
   34.5128264   53.6119254 
   34.5114798   53.6119035 
   34.5106682   53.6115861 
   34.5101885   53.611017  
   34.5101885   53.6105136 
   34.5107235   53.6094192 
   34.5099856   53.6090143 
   34.5087128   53.6098789 
   34.5074585   53.6092004 
   34.5073662   53.6084671 
   34.5082701   53.6076353 
   34.5075323   53.6072194 
   34.5062226   53.6069896 
   34.5056692   53.6060921 
   34.5105206   53.5995794 
   34.5105369   53.5980514 
   34.5101848   53.5968361 
   34.5100674   53.5960697 
   34.5104717   53.5952414 
   34.5110195   53.5948311 
   34.5119065   53.5943434 
   34.5119587   53.5940183 
   34.5117891   53.5937783 
   34.5112413   53.5937706 
   34.5105108   53.5939873 
   34.5098456   53.5943589 
   34.5093891   53.5943357 
   34.509063    53.5941886 
   34.5088152   53.5939254 
   34.508763    53.5936777 
   34.5090108   53.5934222 
   34.5093891   53.5933216 
   34.5100935   53.5933216 
   34.5107195   53.5931977 
   34.5113587   53.5928184 
   34.5120239   53.5926171 
   34.5128196   53.5925861 
   34.5133804   53.5924313 
   34.5142674   53.592261  
   34.5149065   53.5918352 
   34.51565     53.5915952 
   34.5164848   53.5916571 
   34.5172413   53.592021  
   34.5178674   53.592261  
   34.5185848   53.5920829 
   34.519224    53.5916107 
   34.5192501   53.5907203 
   34.5190414   53.5898145 
   34.5188196   53.5896055 
   34.5181283   53.5894042 
   34.5175805   53.5892029 
   34.5174109   53.5889551 
   34.5173979   53.5886532 
   34.5176066   53.5884983 
   34.5179587   53.5884054 
   34.5187935   53.5884132 
   34.5190414   53.5882583 
   34.5186109   53.5879099 
   34.5181022   53.5876854 
   34.5165631   53.5874144 
   34.5160674   53.5871743 
   34.51445     53.5855484 
   34.5119196   53.584867  
   34.5080195   53.5845353 
   34.5036891   53.5839159 
   34.5014716   53.5838075 
   34.4975783   53.5830401 
   34.4981554   53.5758955 
   34.4986951   53.5721446 
   34.4990424   53.5667248 
   34.499825    53.5607131 
   34.5015467   53.5430761 
   34.504912    53.5436251 
   34.5086164   53.542974  
   34.5108599   53.543005  
   34.5176947   53.5439816 
   34.5207469   53.5433306 
   34.5281818   53.5428035 
   34.5292513   53.5410828 
   34.5260687   53.5402457 
   34.5220774   53.5411603 
   34.5206426   53.5406953 
   34.5222339   53.5403852 
   34.5217121   53.5337341 
   34.5197556   53.5313773 
   34.5194947   53.5281365 
   34.5185556   53.5267564 
   34.5191769   53.5224708 
   34.5174299   53.5210416 
   34.5186951   53.5177923 
   34.5179647   53.5166289 
   34.5186951   53.5153493 
   34.5160864   53.513674  
   34.5163864   53.5115565 
   34.5152777   53.5111687 
   34.5154733   53.5083762 
   34.5176777   53.5074996 
   34.5196473   53.5031011 
   34.5197125   53.5008434 
   34.5226473   53.5000598 
   34.5234821   53.4982287 
   34.5276039   53.4981511 
   34.5303691   53.4988494 
   34.5310996   53.4986942 
   34.53543     53.4997184 
   34.5410388   53.496995  
   34.5475368   53.4869141 
   34.5447194   53.482754  
   34.5446537   53.4819172 
   34.5550607   53.4784185 
   34.5561824   53.4757325 
   34.5557394   53.4714696 
   34.5685366   53.4704165 
   34.5669237   53.4641946 
   34.5653589   53.4571937 
   34.5613595   53.4568186 
   34.5560457   53.4566267 
   34.554924    53.4552598 
   34.5548327   53.4546384 
   34.5556153   53.4527821 
   34.5605197   53.4497295 
   34.5650197   53.4492867 
   34.5669502   53.4482613 
   34.5686328   53.4470961 
   34.5662913   53.4404706 
   34.5659623   53.4397803 
   34.5652376   53.4371486 
   34.5668977   53.43583   
   34.5683919   53.4342695 
   34.5693511   53.4328629 
   34.5717123   53.4286536 
   34.5740918   53.4280711 
   34.5829461   53.428181  
   34.5876684   53.4264773 
   34.5903616   53.4265763 
   34.5994741   53.4277743 
   34.6060041   53.4266752 
   34.6081993   53.4227181 
   34.6072401   53.4224653 
   34.6051187   53.4137804 
   34.6056867   53.4134891 
   34.6057052   53.4132197 
   34.6075867   53.4125985 
   34.6084629   53.412615  
   34.6100032   53.4121037 
   34.6108425   53.411532  
   34.6113406   53.4107128 
   34.6115435   53.4100586 
   34.6123367   53.4097397 
   34.6142735   53.4094923 
   34.6147716   53.4093438 
   34.6149192   53.408849  
   34.6158046   53.4083927 
   34.6171327   53.4082772 
   34.617474    53.4080738 
   34.6179075   53.4080903 
   34.6183871   53.4079584 
   34.6183133   53.4077549 
   34.6171327   53.4072051 
   34.6171051   53.4070347 
   34.6178983   53.4070237 
   34.6186546   53.4066388 
   34.6192356   53.4065233 
   34.620158    53.4066443 
   34.6209696   53.4065233 
   34.6220948   53.4064684 
   34.6231924   53.4065783 
   34.6237642   53.4065838 
   34.6244283   53.4064739 
   34.6248987   53.4065123 
   34.6253875   53.4063694 
   34.6259778   53.4057866 
   34.6260239   53.4052477 
   34.6265496   53.4050058 
   34.6271123   53.4049783 
   34.6276472   53.4047749 
   34.6283482   53.4046649 
   34.6300478   53.4042198 
   34.6306347   53.404352  
   34.6323956   53.4040332 
   34.6327087   53.4037066 
   34.6330478   53.4036599 
   34.6333217   53.4034422 
   34.6333217   53.4031311 
   34.6337261   53.4028434 
   34.6336869   53.4022368 
   34.6339478   53.4016925 
   34.6346913   53.4009692 
   34.6349391   53.4006271 
   34.6360609   53.3999194 
   34.6365696   53.3998494 
   34.6370131   53.3995305 
   34.6375609   53.3994294 
   34.638187    53.3989784 
   34.6386435   53.3984106 
   34.6386174   53.3981229 
   34.6382131   53.3979207 
   34.6382913   53.397734  
   34.6388261   53.397454  
   34.6388653   53.3970496 
   34.6390218   53.3967385 
   34.6388392   53.3964274 
   34.6389435   53.3961318 
   34.6386957   53.3956807 
   34.6391522   53.3950274 
   34.639158    53.3932251 
   34.6387245   53.3925486 
   34.6390104   53.3916576 
   34.6374148   53.390717  
   34.6370828   53.3886929 
   34.6373041   53.3881758 
   34.6370274   53.3878183 
   34.6377284   53.3873782 
   34.6377468   53.3860965 
   34.637424    53.3858545 
   34.637175    53.3853374 
   34.6373134   53.3849083 
   34.6375255   53.3845728 
   34.6373134   53.3839566 
   34.6379498   53.3830269 
   34.6383925   53.3829939 
   34.6387706   53.3830654 
   34.639075    53.3830599 
   34.639573    53.383291  
   34.640025    53.383258  
   34.6405046   53.3827519 
   34.640975    53.3823723 
   34.6419803   53.3818716 
   34.6442584   53.381261  
   34.6465366   53.3811729 
   34.6473851   53.381349  
   34.657051    53.3804687 
   34.6589694   53.3805348 
   34.6614044   53.3802157 
   34.6678791   53.3802377 
   34.6735119   53.3806164 
   34.673935    53.3806274 
   34.6744432   53.380647  
   34.6751096   53.380666  
   34.6761763   53.3806894 
   34.6764953   53.3806997 
   34.6769183   53.3807187 
   34.6771613   53.3807446 
   34.6774176   53.3807787 
   34.6778449   53.380856  
   34.6785366   53.3810386 
   34.6789227   53.3811487 
   34.6790923   53.3811919 
   34.6792299   53.381209  
   34.679403    53.3812118 
   34.6795841   53.3812179 
   34.6798164   53.3812356 
   34.6801334   53.3812798 
   34.6805512   53.3813619 
   34.6809366   53.3814348 
   34.6814726   53.3814885 
   34.6814915   53.3814895 
   34.6818756   53.3814921 
   34.6821532   53.3814729 
   34.6823833   53.3814615 
   34.6826847   53.3814021 
   34.683061    53.381314  
   34.6833869   53.3812434 
   34.6837263   53.381177  
   34.6839801   53.3811318 
   34.6842461   53.3810785 
   34.6843685   53.3810438 
   34.6846516   53.3809486 
   34.6852086   53.3807453 
   34.6857502   53.3805686 
   34.6861439   53.3804192 
   34.6864002   53.3803077 
   34.6866188   53.3801927 
   34.6868589   53.3800811 
   34.6871466   53.3799593 
   34.687441    53.3798351 
   34.6878499   53.379668  
   34.6882575   53.379496  
   34.6886932   53.379338  
   34.6891439   53.379176  
   34.6895358   53.3790476 
   34.6899057   53.3789353 
   34.6903038   53.378836  
   34.690654    53.3787664 
   34.6909415   53.3787262 
   34.6912003   53.3786972 
   34.6915362   53.3786719 
   34.6918883   53.3786484 
   34.6923363   53.3786303 
   34.6926531   53.3786114 
   34.6931567   53.3785856 
   34.6936616   53.3785631 
   34.6940866   53.3785433 
   34.6946888   53.3785262 
   34.6953316   53.3785046 
   34.695954    53.3784893 
   34.6963694   53.3784759 
   34.6969485   53.3784603 
   34.6977643   53.3784478 
   34.6986369   53.3784277 
   34.6993433   53.3784097 
   34.7002781   53.37839   
   34.7011476   53.3783989 
   34.701603    53.3784164 
   34.7022649   53.3784596 
   34.7025063   53.3785    
   34.7029318   53.3785982 
   34.7034246   53.3787195 
   34.7041606   53.3789289 
   34.7046407   53.3790776 
   34.705286    53.3792904 
   34.7058893   53.3795087 
   34.7064134   53.3796916 
   34.7069172   53.3798777 
   34.7071673   53.3799569 
   34.7074714   53.380043  
   34.7077689   53.3801169 
   34.7082785   53.3802092 
   34.7087436   53.3802915 
   34.7092127   53.380377  
   34.7097231   53.3805016 
   34.7100436   53.3805765 
   34.7105769   53.3806996 
   34.7111876   53.3808143 
   34.7116162   53.3808948 
   34.7122214   53.3810103 
   34.7130742   53.3811937 
   34.7137681   53.3813437 
   34.7143069   53.3814612 
   34.7147177   53.3815593 
   34.7149414   53.3816028 
   34.7151115   53.3816249 
   34.7152574   53.3816307 
   34.7157199   53.3816369 
   34.716151    53.3816542 
   34.716705    53.3816837 
   34.7172926   53.3817263 
   34.7177924   53.3817634 
   34.7182397   53.3817873 
   34.7186124   53.3818172 
   34.7190538   53.3818653 
   34.7198039   53.3819736 
   34.7202826   53.3820535 
   34.7206416   53.3820995 
   34.7208252   53.3821185 
   34.7216183   53.382164  
   34.7220141   53.3821908 
   34.7220174   53.38216   
   34.7222818   53.3821989 
   34.7225787   53.3822332 
   34.7229417   53.382284  
   34.7233327   53.3823512 
   34.7237751   53.3824203 
   34.7240975   53.3824693 
   34.7243404   53.3824983 
   34.7246262   53.3825519 
   34.7247907   53.382582  
   34.7249392   53.3825975 
   34.7252461   53.382606  
   34.7255844   53.3825968 
   34.7260781   53.3825935 
   34.7266218   53.3825937 
   34.726887    53.3825921 
   34.7272589   53.3825929 
   34.7275755   53.38259   
   34.727673    53.3825825 
   34.7278413   53.3825513 
   34.7280599   53.3825041 
   34.7282038   53.3824743 
   34.7283148   53.3824653 
   34.7283717   53.3824592 
   34.7284025   53.3824012 
   34.728433    53.3823625 
   34.7284768   53.3823288 
   34.7285395   53.382305  
   34.7286289   53.3822975 
   34.7287641   53.3822983 
   34.7288219   53.3823181 
   34.7288972   53.3823469 
   34.7289615   53.3823829 
   34.7290591   53.3824466 
   34.7291802   53.3825581 
   34.7292784   53.3826614 
   34.7293506   53.382712  
   34.7293868   53.382726  
   34.7294247   53.3827303 
   34.7294668   53.3827168 
   34.7294996   53.3826968 
   34.729523    53.3826751 
   34.7295294   53.3826202 
   34.7295417   53.3825345 
   34.7295493   53.3824885 
   34.7295908   53.382437  
   34.7296634   53.3823816 
   34.7297839   53.3823015 
   34.7298795   53.3822496 
   34.7299692   53.3822243 
   34.7300711   53.3821966 
   34.730197    53.3821885 
   34.7302743   53.3821809 
   34.7303634   53.3821863 
   34.7304647   53.3821958 
   34.7305713   53.3822127 
   34.730644    53.3822309 
   34.730737    53.3822461 
   34.7308461   53.3822759 
   34.730936    53.3823169 
   34.7310125   53.3823481 
   34.7311145   53.3823956 
   34.7312376   53.3824724 
   34.731307    53.3825198 
   34.731367    53.3825719 
   34.7314367   53.3826104 
   34.7314904   53.3826325 
   34.7315483   53.3826475 
   34.7316075   53.3826608 
   34.7316466   53.3826723 
   34.7316908   53.3826944 
   34.731727    53.3827149 
   34.7317456   53.3827336 
   34.7317651   53.3827782 
   34.7317724   53.3828227 
   34.7317599   53.3829173 
   34.7317713   53.3829642 
   34.7318033   53.3829879 
   34.7318301   53.3830042 
   34.7318759   53.3830159 
   34.7319312   53.3830243 
   34.7319974   53.3830279 
   34.7320935   53.3830237 
   34.732168    53.3830193 
   34.7322345   53.3830052 
   34.7323457   53.3829865 
   34.7325249   53.382948  
   34.7325942   53.3829346 
   34.732674    53.3829335 
   34.7327374   53.3829396 
   34.7327873   53.3829512 
   34.732842    53.382992  
   34.7328847   53.3830271 
   34.7328977   53.3830562 
   34.7329285   53.3830742 
   34.7329689   53.383085  
   34.733012    53.3830917 
   34.7330958   53.3830955 
   34.7331444   53.383099  
   34.7331903   53.383109  
   34.7332519   53.383141  
   34.7333431   53.3831844 
   34.7334022   53.3832115 
   34.733475    53.3832216 
   34.7335224   53.3832203 
   34.7335577   53.3832116 
   34.7335906   53.3831876 
   34.7336074   53.3831545 
   34.7336205   53.3831012 
   34.7336336   53.3830504 
   34.7336573   53.3830109 
   34.7336891   53.3829715 
   34.733741    53.3829411 
   34.7338367   53.3828842 
   34.7339405   53.3828291 
   34.734013    53.382781  
   34.7340702   53.3827587 
   34.7341149   53.3827517 
   34.7341663   53.3827569 
   34.7342173   53.3827758 
   34.7342615   53.3828044 
   34.7343066   53.3828508 
   34.7343722   53.3828925 
   34.7344218   53.382917  
   34.7344594   53.3829302 
   34.734558    53.3829389 
   34.7346446   53.3829354 
   34.7347233   53.3829213 
   34.7347955   53.3828967 
   34.7348528   53.3828647 
   34.7349133   53.3828117 
   34.7349462   53.3827828 
   34.7349742   53.3827272 
   34.7349993   53.3826894 
   34.7350376   53.3826605 
   34.7350817   53.3826179 
   34.7351323   53.3825818 
   34.7351774   53.3825595 
   34.7352154   53.38255   
   34.7352722   53.3825504 
   34.735318    53.3825636 
   34.7353474   53.3825824 
   34.735374    53.3826068 
   34.7353936   53.3826474 
   34.7354145   53.3826912 
   34.7354341   53.3827293 
   34.7354728   53.3827562 
   34.7355213   53.3827711 
   34.7355846   53.3827828 
   34.7357077   53.3827836 
   34.7357621   53.3827653 
   34.7358194   53.3827365 
   34.7358744   53.3826868 
   34.7358909   53.3826707 
   34.7359252   53.382641  
   34.7359675   53.382621  
   34.7360287   53.3826052 
   34.7360734   53.382599  
   34.7361208   53.3825993 
   34.7361841   53.3826127 
   34.736238    53.382626  
   34.7362932   53.3826384 
   34.7363243   53.3826386 
   34.7363676   53.3826389 
   34.7364204   53.3826368 
   34.7364936   53.3826259 
   34.7365587   53.3826166 
   34.7366294   53.382592  
   34.7366708   53.3825478 
   34.7366862   53.382518  
   34.7367098   53.3824833 
   34.736747    53.3824456 
   34.7367882   53.3824086 
   34.7368538   53.3823686 
   34.7368772   53.3823453 
   34.7369094   53.3822865 
   34.736916    53.382217  
   34.7369333   53.3821556 
   34.7369781   53.3820661 
   34.7370654   53.381947  
   34.737133    53.3818746 
   34.7372195   53.3817967 
   34.73731     53.3817261 
   34.7373865   53.3816862 
   34.7374776   53.3816584 
   34.7376024   53.3816341 
   34.7377312   53.3816204 
   34.7378696   53.3815946 
   34.7379836   53.3815718 
   34.7380937   53.3815394 
   34.7382448   53.3814829 
   34.7383759   53.381415  
   34.7384647   53.3813638 
   34.7385602   53.3813134 
   34.738631    53.3812864 
   34.7386799   53.3812737 
   34.7387342   53.3812644 
   34.7388193   53.3812681 
   34.738933    53.381268  
   34.7389965   53.3812684 
   34.7390764   53.3812641 
   34.7391388   53.3812556 
   34.7391944   53.3812422 
   34.7392365   53.3812359 
   34.7392938   53.3812096 
   34.7393675   53.3811672 
   34.7394264   53.3811247 
   34.7394843   53.3810652 
   34.7395406   53.3810122 
   34.7396054   53.3809423 
   34.7396849   53.3808813 
   34.7397574   53.380834  
   34.7398422   53.380782  
   34.7399172   53.3807445 
   34.7400043   53.3807183 
   34.7401073   53.380702  
   34.740213    53.3806937 
   34.7402874   53.3806934 
   34.7403524   53.3806873 
   34.7404527   53.3806726 
   34.7405205   53.3806633 
   34.7405668   53.3806434 
   34.7407088   53.3805682 
   34.7408564   53.3804834 
   34.7409356   53.3804354 
   34.741012    53.3803979 
   34.7410799   53.3803862 
   34.741134    53.3803841 
   34.7411894   53.380386  
   34.7412527   53.3804034 
   34.7413634   53.3804146 
   34.7415134   53.3804228 
   34.741627    53.3804227 
   34.741662    53.3804326 
   34.7417063   53.3804507 
   34.7417237   53.3804613 
   34.7417397   53.3804752 
   34.7417418   53.3805084 
   34.7417421   53.3805674 
   34.7417375   53.3806005 
   34.7417505   53.3806305 
   34.74178     53.3806461 
   34.7418243   53.3806625 
   34.7418729   53.3806693 
   34.7419162   53.3806696 
   34.7419542   53.3806625 
   34.7420099   53.3806491 
   34.7420779   53.3806245 
   34.7421776   53.3805677 
   34.7422185   53.3805485 
   34.7422591   53.380548  
   34.7423415   53.3805566 
   34.7424141   53.380578  
   34.7425133   53.3806304 
   34.7426618   53.3807251 
   34.7427504   53.3807637 
   34.7428136   53.3807851 
   34.7428419   53.3807893 
   34.7428798   53.3807888 
   34.7429232   53.3807817 
   34.7429708   53.3807651 
   34.7430119   53.3807378 
   34.7430439   53.3806871 
   34.7430692   53.380629  
   34.7431091   53.3805921 
   34.7431447   53.3805648 
   34.7431733   53.3805512 
   34.743229    53.3805427 
   34.7433143   53.3805343 
   34.7434441   53.3805383 
   34.7435587   53.3805576 
   34.7437416   53.3806194 
   34.7438785   53.3806761 
   34.7441025   53.3807866 
   34.7442807   53.3808807 
   34.7444508   53.3809805 
   34.7444895   53.3810074 
   34.7445082   53.381022  
   34.7445117   53.381056  
   34.7445055   53.3811045 
   34.7445044   53.3811635 
   34.7445082   53.3811806 
   34.7445282   53.3811977 
   34.7445658   53.3812108 
   34.7446264   53.381229  
   34.7447034   53.3812311 
   34.7447427   53.3812281 
   34.7447712   53.3812234 
   34.7448216   53.3812003 
   34.7448966   53.3811684 
   34.7450113   53.3811012 
   34.7450768   53.3810668 
   34.7451449   53.3810422 
   34.7452047   53.3810247 
   34.7452684   53.3810203 
   34.7453713   53.3810104 
   34.7454579   53.3810077 
   34.7455824   53.3810077 
   34.7456607   53.3810154 
   34.7457362   53.381028  
   34.7458881   53.3810807 
   34.7459498   53.3811094 
   34.7460234   53.3811576 
   34.7460754   53.3811991 
   34.7461107   53.3812697 
   34.7461595   53.3813444 
   34.746207    53.381411  
   34.7462577   53.3814526 
   34.7463314   53.3814894 
   34.7464176   53.3815142 
   34.7465253   53.3815416 
   34.7466225   53.3815519 
   34.7467453   53.3815672 
   34.7468372   53.3815742 
   34.7469033   53.3815851 
   34.7470055   53.3816149 
   34.7470931   53.3816381 
   34.7471991   53.3816832 
   34.74729     53.3817477 
   34.7473651   53.3817829 
   34.7474217   53.381797  
   34.7474649   53.3817981 
   34.7475083   53.3817951 
   34.7475532   53.381776  
   34.7476015   53.3817205 
   34.7476308   53.3816729 
   34.7476611   53.3816391 
   34.7476829   53.3816288 
   34.7477452   53.3816251 
   34.7478195   53.3816256 
   34.7478872   53.3816284 
   34.7479356   53.3816416 
   34.7479963   53.3816542 
   34.7480782   53.3816854 
   34.7481467   53.3817141 
   34.7482485   53.3817722 
   34.7483209   53.3818074 
   34.7483679   53.3818287 
   34.7484124   53.3818371 
   34.7484732   53.3818399 
   34.7485245   53.3818426 
   34.7485531   53.3818363 
   34.7485926   53.3818188 
   34.7486257   53.3817818 
   34.7486386   53.3817357 
   34.7486463   53.3816857 
   34.7486575   53.3816647 
   34.7486848   53.3816503 
   34.7487106   53.3816416 
   34.7487567   53.381633  
   34.7488163   53.3816325 
   34.7488836   53.3816499 
   34.7489815   53.3816966 
   34.749054    53.3817278 
   34.749093    53.381741  
   34.7491631   53.3817576 
   34.7492008   53.3817627 
   34.7492463   53.3817961 
   34.7493102   53.3818499 
   34.7493542   53.3818898 
   34.7494197   53.3819354 
   34.7494922   53.3819682 
   34.7495701   53.3819946 
   34.7496674   53.3820009 
   34.7497514   53.3819957 
   34.7498096   53.3819912 
   34.7498774   53.3819819 
   34.7499483   53.3819508 
   34.7500069   53.3819196 
   34.7502151   53.3817673 
   34.7503561   53.3816719 
   34.7505079   53.3815766 
   34.7506278   53.3815232 
   34.750711    53.3814808 
   34.7507764   53.3814553 
   34.7508484   53.3814339 
   34.7509285   53.3814199 
   34.7510017   53.3814074 
   34.7510694   53.3814038 
   34.7511303   53.3814033 
   34.751183    53.3814069 
   34.7512383   53.3814153 
   34.751334    53.3814369 
   34.7514469   53.3814772 
   34.7515543   53.3815199 
   34.7516739   53.381566  
   34.7518152   53.3816065 
   34.7518826   53.3816214 
   34.7519353   53.3816274 
   34.7520151   53.3816279 
   34.7521625   53.3816256 
   34.7522303   53.3816179 
   34.7522953   53.381611  
   34.7523549   53.3816098 
   34.7524575   53.3816161 
   34.752502    53.3816252 
   34.7526085   53.3816461 
   34.7526759   53.3816603 
   34.7527725   53.3817021 
   34.7528464   53.3817317 
   34.7528813   53.3817472 
   34.7529421   53.3817525 
   34.7529976   53.381752  
   34.7530627   53.3817394 
   34.7531552   53.3817101 
   34.7532108   53.3816975 
   34.7532745   53.3816914 
   34.753334    53.381691  
   34.7534083   53.3816979 
   34.7535109   53.3817066 
   34.7536054   53.3817201 
   34.7537715   53.3817373 
   34.7539173   53.3817528 
   34.7540239   53.3817664 
   34.7541023   53.3817684 
   34.7542132   53.3817691 
   34.7543336   53.3817691 
   34.7544121   53.3817639 
   34.754492    53.3817547 
   34.7545722   53.3817374 
   34.7546795   53.3817105 
   34.7548236   53.3816645 
   34.7549325   53.3816263 
   34.7550782   53.381569  
   34.7551803   53.3815284 
   34.7552744   53.381482  
   34.7553301   53.3814678 
   34.7554188   53.3814239 
   34.7555527   53.3813422 
   34.7556413   53.3813031 
   34.7557066   53.3812825 
   34.7557702   53.3812756 
   34.7558635   53.3812827 
   34.7559159   53.3812983 
   34.7560007   53.3813247 
   34.7561067   53.3813763 
   34.7562288   53.3814337 
   34.7563319   53.3814942 
   34.7564539   53.3815547 
   34.7565801   53.3816105 
   34.7567024   53.3816598 
   34.7567696   53.3816828 
   34.7568652   53.3817093 
   34.7569918   53.3817367 
   34.7571011   53.3817528 
   34.757263    53.381778  
   34.7573899   53.3817909 
   34.7574723   53.3818011 
   34.7575844   53.3818058 
   34.757656    53.3818144 
   34.7577072   53.3818252 
   34.7577771   53.3818507 
   34.7578738   53.3818901 
   34.757984    53.381932  
   34.7580607   53.3819567 
   34.7581226   53.3819717 
   34.7582508   53.3819886 
   34.7584955   53.3819974 
   34.7586781   53.3819953 
   34.7588297   53.3819913 
   34.7589907   53.3819834 
   34.7590854   53.3819848 
   34.7591543   53.3819893 
   34.7592579   53.3820222 
   34.759356    53.382056  
   34.7594203   53.3820968 
   34.7595071   53.382162  
   34.7595672   53.3822061 
   34.7597053   53.3822789 
   34.7597924   53.3823287 
   34.7599212   53.3823869 
   34.7600953   53.3824866 
   34.7602226   53.3825618 
   34.7602831   53.382584  
   34.7603921   53.3826154 
   34.7604689   53.3826304 
   34.7605863   53.3826473 
   34.7607335   53.3826579 
   34.7609417   53.3826697 
   34.7610728   53.3826745 
   34.7612012   53.3826769 
   34.7613324   53.3826777 
   34.7614879   53.3826803 
   34.7616747   53.3826701 
   34.7618076   53.3826515 
   34.7619188   53.3826319 
   34.7620288   53.3826116 
   34.7621726   53.3825833 
   34.7622824   53.3825662 
   34.762353    53.3825504 
   34.7624169   53.3825322 
   34.7624537   53.3825147 
   34.7625087   53.3824632 
   34.7625596   53.3824126 
   34.7626089   53.3823781 
   34.7626607   53.3823542 
   34.7627107   53.3823561 
   34.7627607   53.3823596 
   34.7628145   53.3823729 
   34.7628428   53.3823828 
   34.7628816   53.3824097 
   34.7629203   53.3824366 
   34.7629588   53.3824789 
   34.7630095   53.3825229 
   34.7630415   53.3825481 
   34.7631087   53.3825744 
   34.7631746   53.3825966 
   34.7632447   53.3826116 
   34.7633415   53.3826437 
   34.7633967   53.3826594 
   34.7634466   53.3826686 
   34.7634923   53.3826843 
   34.7635367   53.3826999 
   34.7636238   53.3827482 
   34.7637215   53.3828078 
   34.763838    53.3828796 
   34.763945    53.3829523 
   34.7639918   53.3829792 
   34.7640091   53.383002  
   34.764026    53.3830369 
   34.7640364   53.3830628 
   34.7640411   53.3831049 
   34.7640305   53.3831736 
   34.764023    53.383218  
   34.7640307   53.3832415 
   34.7640516   53.3832845 
   34.7640769   53.3833089 
   34.7641303   53.3833496 
   34.7642284   53.383389  
   34.7643266   53.3834244 
   34.764383    53.383445  
   34.7644313   53.3834671 
   34.7644848   53.3835046 
   34.7645168   53.3835291 
   34.7645647   53.3835827 
   34.7646154   53.3836227 
   34.764654    53.3836577 
   34.7647078   53.3836758 
   34.7647657   53.3836923 
   34.7648357   53.3837105 
   34.7649046   53.383715  
   34.7649912   53.3837139 
   34.7650535   53.3837078 
   34.7650834   53.3836991 
   34.7651327   53.383667  
   34.765178    53.3836252 
   34.7652178   53.3835883 
   34.7652534   53.3835634 
   34.7653024   53.3835459 
   34.7653541   53.3835293 
   34.7654125   53.3835183 
   34.7654598   53.3835194 
   34.7655097   53.3835286 
   34.7655824   53.3835476 
   34.7656537   53.3835699 
   34.7657074   53.3835928 
   34.7657608   53.3836328 
   34.7658115   53.3836768 
   34.7658636   53.3837143 
   34.7659346   53.3837543 
   34.7659751   53.3837586 
   34.7660413   53.3837679 
   34.7661535   53.383767  
   34.7662348   53.3837594 
   34.7663068   53.3837404 
   34.7663859   53.3837021 
   34.7664349   53.3836829 
   34.7664812   53.3836679 
   34.7665315   53.3836528 
   34.7665803   53.3836434 
   34.7666304   53.3836437 
   34.7666654   53.3836504 
   34.7667018   53.3836595 
   34.7667799   53.3836802 
   34.7668508   53.3837259 
   34.7669218   53.3837675 
   34.7669782   53.3837954 
   34.7670078   53.383802  
   34.7670645   53.3838056 
   34.7671294   53.3838084 
   34.7671998   53.3838056 
   34.7672418   53.3837994 
   34.7672949   53.3837819 
   34.7673263   53.3837667 
   34.7673658   53.3837484 
   34.7673989   53.3837089 
   34.7674444   53.3836566 
   34.7674667   53.3836196 
   34.7674835   53.3835841 
   34.7675059   53.3835414 
   34.7675306   53.3835213 
   34.7675716   53.3834924 
   34.7676086   53.3834635 
   34.7676562   53.3834485 
   34.7677147   53.383427  
   34.767811    53.383413  
   34.7679411   53.383396  
   34.7680211   53.383386  
   34.7680957   53.3833743 
   34.7681745   53.3833497 
   34.7682347   53.3833137 
   34.7682869   53.3832622 
   34.7683326   53.383201  
   34.7683716   53.3831341 
   34.7683902   53.3830696 
   34.7684183   53.3830115 
   34.7684651   53.3829624 
   34.768509    53.3829271 
   34.7685731   53.3828927 
   34.7686492   53.3828697 
   34.768709    53.3828539 
   34.7687928   53.3828544 
   34.7688712   53.3828614 
   34.7689278   53.3828722 
   34.7689766   53.382866  
   34.7690161   53.3828509 
   34.7691007   53.3828045 
   34.7691582   53.3827628 
   34.7692105   53.382713  
   34.7692746   53.3826794 
   34.7693305   53.3826498 
   34.7694145   53.3826406 
   34.7695446   53.3826268 
   34.7696694   53.3826074 
   34.7697277   53.3825972 
   34.7697807   53.382583  
   34.7698434   53.3825542 
   34.7699021   53.3825214 
   34.7699335   53.3825054 
   34.7699837   53.382496  
   34.7700364   53.382498  
   34.7701011   53.3825097 
   34.7701427   53.3825293 
   34.7701801   53.3825562 
   34.7702309   53.3825921 
   34.7702768   53.3826005 
   34.7703295   53.3826032 
   34.7703702   53.3825938 
   34.7704094   53.382594  
   34.7704473   53.3825966 
   34.7704823   53.3826041 
   34.770513    53.3826294 
   34.7705638   53.3826693 
   34.7706082   53.3826777 
   34.7706557   53.3826715 
   34.7706869   53.3826611 
   34.7707722   53.382612  
   34.7708113   53.3825883 
   34.7708675   53.3825409 
   34.7709219   53.3825235 
   34.7709924   53.3825134 
   34.7710993   53.3825108 
   34.7712103   53.3825017 
   34.7713121   53.3824845 
   34.7713934   53.3824721 
   34.7714383   53.3824586 
   34.77155     53.3824116 
   34.7716249   53.3823813 
   34.7716656   53.3823726 
   34.7717603   53.38237   
   34.7718697   53.3823803 
   34.7719301   53.3824082 
   34.7719851   53.3824344 
   34.772079    53.3824835 
   34.7721379   53.3825218 
   34.7721929   53.382548  
   34.7722736   53.3825711 
   34.7723492   53.3825837 
   34.7724411   53.3825851 
   34.7725117   53.3825734 
   34.7725865   53.3825479 
   34.7726573   53.3825176 
   34.772716    53.3824889 
   34.7727694   53.3824471 
   34.7728084   53.3823819 
   34.7728854   53.3822238 
   34.7729321   53.3821004 
   34.7729655   53.3820456 
   34.7730232   53.3819861 
   34.7730617   53.3819523 
   34.7731436   53.3819067 
   34.7732143   53.3818829 
   34.7733447   53.3818521 
   34.773483    53.3818295 
   34.7735903   53.3818018 
   34.7737463   53.3817761 
   34.7738697   53.3817582 
   34.7739957   53.3817452 
   34.7740823   53.3817393 
   34.7741894   53.3817261 
   34.7743383   53.3817189 
   34.7744331   53.3817114 
   34.774517    53.3817054 
   34.7745955   53.3817027 
   34.7746792   53.3817145 
   34.7747519   53.3817319 
   34.7748205   53.3817582 
   34.7749358   53.3818211 
   34.7749985   53.3818717 
   34.7751174   53.3819605 
   34.7752231   53.3820275 
   34.775293    53.3820529 
   34.7753591   53.3820671 
   34.7754427   53.3820821 
   34.775521    53.3820883 
   34.7756306   53.3820889 
   34.7757239   53.382087  
   34.775789    53.3820809 
   34.7758325   53.3820658 
   34.7758639   53.3820482 
   34.7758899   53.3820322 
   34.7759133   53.3820024 
   34.7759465   53.381963  
   34.7760054   53.3819148 
   34.7760682   53.381882  
   34.7761199   53.3818597 
   34.7761934   53.3818359 
   34.7762477   53.3818217 
   34.7763451   53.3818222 
   34.776395    53.3818314 
   34.7764637   53.3818472 
   34.7765228   53.3818678 
   34.7765926   53.3819046 
   34.7766636   53.3819462 
   34.776736    53.3819806 
   34.7768113   53.3820102 
   34.7768733   53.3820235 
   34.7769664   53.3820354 
   34.7770569   53.3820424 
   34.7771136   53.3820492 
   34.777261    53.3820533 
   34.77738     53.3820556 
   34.7774774   53.3820529 
   34.7775694   53.382047  
   34.7776386   53.3820377 
   34.7776753   53.3820266 
   34.7777067   53.3820098 
   34.7777532   53.3819769 
   34.7777782   53.381935  
   34.7777926   53.3818826 
   34.7777905   53.3817645 
   34.7777958   53.3816893 
   34.7778229   53.3816045 
   34.7778608   53.3815255 
   34.7779055   53.381436  
   34.777958    53.3813684 
   34.7780089   53.3813161 
   34.7780693   53.3812655 
   34.7781798   53.3812047 
   34.7782591   53.3811526 
   34.7782827   53.3811196 
   34.7782859   53.3810881 
   34.7782758   53.381046  
   34.7782453   53.3810094 
   34.77818     53.3809508 
   34.7781602   53.3809175 
   34.7781661   53.3808925 
   34.7781989   53.38087   
   34.778233    53.3808557 
   34.77832     53.3808279 
   34.7784517   53.3807971 
   34.7785753   53.3807655 
   34.7787355   53.3807277 
   34.7788727   53.3806888 
   34.7789542   53.3806723 
   34.7790287   53.3806615 
   34.7791126   53.3806595 
   34.7791719   53.380672  
   34.7792122   53.3806884 
   34.7792403   53.3807072 
   34.779259    53.3807202 
   34.7792871   53.3807382 
   34.779318    53.3807505 
   34.7793625   53.3807621 
   34.7793989   53.3807671 
   34.7794718   53.3807757 
   34.7795461   53.3807834 
   34.7796149   53.3807943 
   34.7796607   53.3808035 
   34.7796997   53.3808183 
   34.7797412   53.3808444 
   34.7798053   53.3808884 
   34.7798614   53.3809357 
   34.779896    53.380969  
   34.7799175   53.3809772 
   34.7799473   53.3809774 
   34.7799878   53.3809776 
   34.7800109   53.3809713 
   34.7800465   53.3809465 
   34.780088    53.3808885 
   34.7801475   53.380812  
   34.7802143   53.3806968 
   34.7802528   53.3806574 
   34.7802951   53.380635  
   34.7803766   53.3806128 
   34.7804322   53.3806066 
   34.7805351   53.3805976 
   34.7806283   53.3806046 
   34.7807432   53.3806069 
   34.7808784   53.3806109 
   34.780999    53.380597  
   34.7810641   53.3805877 
   34.7811253   53.3805663 
   34.7811582   53.380539  
   34.7812212   53.3804932 
   34.7812717   53.3804644 
   34.7813342   53.3804486 
   34.7813979   53.3804377 
   34.7815034   53.3804367 
   34.7815722   53.3804484 
   34.7816476   53.3804715 
   34.7817079   53.3805034 
   34.7817559   53.3805457 
   34.7818064   53.3805969 
   34.7818272   53.3806521 
   34.781837    53.3807111 
   34.781839    53.3807581 
   34.7818329   53.3807977 
   34.7818184   53.3808558 
   34.7818314   53.3808923 
   34.7818781   53.3809265 
   34.7819372   53.3809511 
   34.7820165   53.3809815 
   34.7820946   53.3810046 
   34.7821564   53.3810292 
   34.7822248   53.3810636 
   34.7822999   53.381102  
   34.7823955   53.3811269 
   34.7824656   53.3811418 
   34.7825075   53.3811453 
   34.7827494   53.3811581 
   34.7828966   53.3811678 
   34.7830169   53.3811734 
   34.7831235   53.3811886 
   34.7832101   53.3811874 
   34.783282    53.3811749 
   34.7833256   53.3811534 
   34.7833699   53.3810938 
   34.7834309   53.3810027 
   34.7835036   53.380936  
   34.7835556   53.3809056 
   34.7836005   53.3808897 
   34.7836626   53.3808917 
   34.783741    53.3808938 
   34.7837747   53.3809069 
   34.7838416   53.3809445 
   34.7839059   53.3809845 
   34.783958    53.3810269 
   34.78406     53.3810727 
   34.7840937   53.3810778 
   34.7841394   53.3810983 
   34.7841607   53.381121  
   34.7841731   53.3811866 
   34.784188    53.381266  
   34.7842088   53.381317  
   34.7842394   53.3813512 
   34.7842794   53.3813813 
   34.7843333   53.3813994 
   34.7843833   53.3813981 
   34.7844429   53.3813952 
   34.7845244   53.3813747 
   34.7845883   53.3813548 
   34.7846592   53.3813197 
   34.7847861   53.3812484 
   34.7848601   53.3811906 
   34.7849408   53.3811369 
   34.7850063   53.3810985 
   34.7850784   53.3810771 
   34.7851597   53.3810679 
   34.7852749   53.3810524 
   34.7853492   53.3810576 
   34.7854461   53.3810825 
   34.7855203   53.3810967 
   34.7855686   53.3811228 
   34.7856518   53.3811589 
   34.7857554   53.3811959 
   34.7858102   53.3812334 
   34.7858719   53.3812677 
   34.7859365   53.3812851 
   34.7859783   53.3812934 
   34.7860375   53.3813099 
   34.7861052   53.3813103 
   34.7861795   53.3813108 
   34.786277    53.3813081 
   34.7863838   53.3813087 
   34.7864662   53.3813173 
   34.786528    53.3813419 
   34.7865844   53.3813649 
   34.7866367   53.3813935 
   34.7866768   53.3814228 
   34.7867433   53.3814936 
   34.786798    53.381536  
   34.7869226   53.381607  
   34.7870033   53.381635  
   34.7870773   53.3816613 
   34.7871582   53.3816755 
   34.7872404   53.3816954 
   34.7873212   53.3817169 
   34.7874142   53.3817369 
   34.7874869   53.3817551 
   34.7875381   53.3817708 
   34.7875742   53.3817968 
   34.7876828   53.3819374 
   34.78775     53.3820486 
   34.787829    53.3821784 
   34.7878588   53.382257  
   34.7878731   53.3823792 
   34.7878556   53.3824576 
   34.7878575   53.3825029 
   34.787868    53.3825239 
   34.7879083   53.3825412 
   34.7879485   53.3825608 
   34.7879643   53.3825868 
   34.7879796   53.3826459 
   34.787984    53.3827106 
   34.7880088   53.3827633 
   34.7880486   53.3828089 
   34.7880885   53.3828519 
   34.7881606   53.3829114 
   34.7882607   53.3829912 
   34.7883208   53.3830377 
   34.7883474   53.3830686 
   34.7883547   53.3831196 
   34.7883559   53.3832093 
   34.7883552   53.383253  
   34.7883901   53.383271  
   34.7884358   53.3832858 
   34.7884885   53.3832869 
   34.7885509   53.3832784 
   34.7886468   53.3832838 
   34.788763    53.3832966 
   34.7888047   53.3833114 
   34.7888665   53.3833376 
   34.7889148   53.383359  
   34.788959    53.3833875 
   34.7890557   53.3834301 
   34.7891267   53.3834677 
   34.7891994   53.3834884 
   34.7892602   53.3834968 
   34.7893117   53.3834874 
   34.7893823   53.3834757 
   34.7894678   53.3834535 
   34.7895332   53.3834281 
   34.7895767   53.3834105 
   34.7895945   53.3833977 
   34.7896305   53.3833486 
   34.7896746   53.3832954 
   34.789713    53.3832625 
   34.7897553   53.3832401 
   34.7898015   53.3832291 
   34.7898693   53.3832165 
   34.7899197   53.383199  
   34.7899363   53.3831724 
   34.7899371   53.3831255 
   34.7899362   53.3830115 
   34.7899527   53.3829962 
   34.7900072   53.3829723 
   34.7900955   53.3829469 
   34.7902094   53.3829314 
   34.7903406   53.3829322 
   34.7903797   53.3829421 
   34.7904389   53.3829602 
   34.7905556   53.3830183 
   34.7906711   53.3830691 
   34.7907663   53.3831174 
   34.7908229   53.3831339 
   34.7909188   53.3831409 
   34.7909812   53.3831284 
   34.7910507   53.3830972 
   34.791108    53.38307   
   34.7911799   53.3830527 
   34.791253    53.3830531 
   34.7912934   53.383063  
   34.791335    53.3830811 
   34.7913832   53.3831088 
   34.791415    53.3831519 
   34.7914592   53.3832613 
   34.7914746   53.383314  
   34.7915074   53.3833789 
   34.7915394   53.3834074 
   34.7916036   53.3834498 
   34.7917016   53.3834932 
   34.7917553   53.3835178 
   34.7917793   53.3835406 
   34.7917786   53.3835826 
   34.7917565   53.3836156 
   34.7917044   53.3836525 
   34.7916321   53.3836917 
   34.7915314   53.3837348 
   34.791482    53.3837717 
   34.7914748   53.3838016 
   34.7914864   53.3838405 
   34.7915156   53.3838738 
   34.7915559   53.3838886 
   34.7916285   53.3839165 
   34.791692    53.3839218 
   34.7917988   53.3839272 
   34.7918881   53.3839197 
   34.7919424   53.383907  
   34.7920333   53.3838914 
   34.792101    53.3838885 
   34.7922023   53.3838956 
   34.7922847   53.3839074 
   34.792355    53.3839078 
   34.7923983   53.3839081 
   34.7924784   53.3838891 
   34.792549    53.3838693 
   34.7926343   53.3838633 
   34.7927182   53.3838638 
   34.7927464   53.3838753 
   34.7928138   53.3838927 
   34.7929442   53.3839387 
   34.7930827   53.3839921 
   34.7931821   53.3840315 
   34.793275    53.3840611 
   34.7933477   53.3840785 
   34.7934165   53.3840935 
   34.7934676   53.3841116 
   34.7935158   53.3841393 
   34.7935316   53.3841661 
   34.7935514   53.3841986 
   34.7935863   53.3842166 
   34.7936453   53.3842476 
   34.7937784   53.3842945 
   34.7939142   53.3843487 
   34.7939923   53.3843693 
   34.7940379   53.3843898 
   34.7940917   53.3844095 
   34.7942288   53.3844669 
   34.7943428   53.3845218 
   34.7944409   53.384562  
   34.7945312   53.3846684 
   34.7945559   53.3847308 
   34.79457     53.3847794 
   34.7945221   53.3848107 
   34.7943891   53.3848398 
   34.7943452   53.384876  
   34.7943318   53.3849511 
   34.794361    53.3849869 
   34.7944202   53.3850082 
   34.7945225   53.3850363 
   34.7946086   53.3850667 
   34.7947658   53.3851348 
   34.7949875   53.3852274 
   34.7951033   53.3852604 
   34.7954424   53.3853659 
   34.7958302   53.3854781 
   34.7961455   53.3855503 
   34.7964385   53.3855868 
   34.7966302   53.3856105 
   34.7968424   53.3856214 
   34.7972027   53.3856736 
   34.7974615   53.3857293 
   34.797653    53.385762  
   34.7978977   53.3857739 
   34.7981114   53.3857719 
   34.7983524   53.3857587 
   34.7986287   53.3857352 
   34.7988115   53.3857274 
   34.7990846   53.3857346 
   34.7993208   53.3857651 
   34.7994624   53.385795  
   34.79962     53.3858315 
   34.7998108   53.3859102 
   34.7999374   53.3859489 
   34.8000469   53.3859495 
   34.8001974   53.3859261 
   34.8003061   53.3858977 
   34.8004544   53.3858451 
   34.8005947   53.3857845 
   34.8006632   53.3857331 
   34.8007634   53.3856358 
   34.8008494   53.3855918 
   34.8009433   53.385556  
   34.8010518   53.3855388 
   34.801329    53.385542  
   34.8013792   53.3855302 
   34.8014068   53.3854972 
   34.8014463   53.3853988 
   34.8014685   53.3853625 
   34.8015598   53.3853185 
   34.8016916   53.3852805 
   34.8019933   53.385279  
   34.8021917   53.3853076 
   34.8023885   53.3853508 
   34.8025243   53.3854009 
   34.8026303   53.38545   
   34.8027422   53.3855638 
   34.8027745   53.385653  
   34.8028435   53.3857367 
   34.8029225   53.3857889 
   34.8030474   53.3858486 
   34.8031689   53.3858631 
   34.8034177   53.3858677 
   34.8034938   53.3858479 
   34.8035685   53.3858233 
   34.8037068   53.3858022 
   34.8037989   53.3857987 
   34.8038839   53.3858113 
   34.8039995   53.3858549 
   34.8041956   53.3859482 
   34.8043236   53.3859764 
   34.8044547   53.3859836 
   34.8045476   53.3860108 
   34.8046456   53.3860558 
   34.8047639   53.3861042 
   34.804881    53.3861389 
   34.8050565   53.3861601 
   34.8051596   53.3861445 
   34.8054574   53.3860475 
   34.805608    53.3860185 
   34.8058559   53.3859956 
   34.8062657   53.3859995 
   34.8061615   53.385318  
   34.8060833   53.3848736 
   34.8061103   53.3847953 
   34.8073563   53.382902  
   34.8078494   53.3822554 
   34.809665    53.3797385 
   34.8114324   53.3796733 
   34.8138893   53.3796135 
   34.8157425   53.3795972 
   34.8160065   53.3795793 
   34.8229786   53.3792994 
   34.8260387   53.3789815 
   34.8325505   53.3774872 
   34.8375391   53.3777803 
   34.8375548   53.3779025 
   34.8376737   53.3779072 
   34.8378086   53.3781125 
   34.8380434   53.3782319 
   34.8383242   53.3782698 
   34.8386584   53.3781681 
   34.8388651   53.3781805 
   34.8395082   53.3784072 
   34.8399783   53.3787105 
   34.8399966   53.37884   
   34.8396385   53.3789974 
   34.839853    53.3791239 
   34.8398679   53.3792146 
   34.8395966   53.3792681 
   34.8396674   53.379685  
   34.8394683   53.3798828 
   34.8396721   53.380002  
   34.8397995   53.3801644 
   34.8401982   53.3801811 
   34.8403848   53.3804514 
   34.8407069   53.3806141 
   34.8409614   53.3809616 
   34.8411866   53.3810978 
   34.8411882   53.3814423 
   34.8414448   53.3815577 
   34.8414868   53.381731  
   34.8411882   53.3820699 
   34.8417093   53.3823137 
   34.8418174   53.3824986 
   34.8416786   53.3827356 
   34.8416243   53.383016  
   34.8412933   53.3831743 
   34.8413143   53.3833054 
   34.841451    53.3833895 
   34.8419613   53.3835426 
   34.8423693   53.3834825 
   34.842561    53.3835094 
   34.8428548   53.3836744 
   34.8428209   53.3838586 
   34.8429427   53.3839393 
   34.8434616   53.384158  
   34.8438633   53.3842491 
   34.8439329   53.3843878 
   34.8438436   53.384477  
   34.8440892   53.3846951 
   34.8443244   53.384795  
   34.8445734   53.3847001 
   34.8447583   53.3847278 
   34.8451375   53.3851463 
   34.8450536   53.3854184 
   34.8452233   53.3854726 
   34.8464294   53.3854977 
   34.8465483   53.385601  
   34.8465439   53.3858048 
   34.8466789   53.3860028 
   34.8466925   53.3860919 
   34.8464925   53.3862598 
   34.8465685   53.386333  
   34.8474909   53.3865174 
   34.8482418   53.3865101 
   34.8485482   53.3864665 
   34.8490751   53.3865962 
   34.8497528   53.3868675 
   34.84983     53.3869569 
   34.8497091   53.3871689 
   34.8499686   53.3872754 
   34.8509694   53.387195  
   34.8511355   53.3870374 
   34.8514713   53.3869243 
   34.8517248   53.3866992 
   34.852017    53.3866086 
   34.8523849   53.3866113 
   34.8526339   53.386696  
   34.8528397   53.3868669 
   34.8528225   53.3870188 
   34.8526906   53.3871629 
   34.8527587   53.3873128 
   34.8532952   53.3876213 
   34.8535501   53.387854  
   34.853764    53.3879335 
   34.8547502   53.3878377 
   34.8554985   53.3874526 
   34.8559471   53.3873046 
   34.8562481   53.3872568 
   34.8567899   53.3873009 
   34.8569989   53.3874403 
   34.8571847   53.387679  
   34.8570701   53.3880245 
   34.857315    53.3882951 
   34.8572685   53.388607  
   34.8575519   53.388839  
   34.8580965   53.3890626 
   34.8589872   53.3891117 
   34.8592309   53.3892812 
   34.8593459   53.3892794 
   34.8596306   53.3891475 
   34.8599965   53.3891938 
   34.860153    53.3893135 
   34.860051    53.3894448 
   34.8601131   53.3895438 
   34.860578    53.3894848 
   34.8603541   53.3891634 
   34.8604239   53.3889236 
   34.8607856   53.3886052 
   34.86099     53.3883208 
   34.8611354   53.388273  
   34.8613544   53.3882855 
   34.8615432   53.3883269 
   34.861948    53.3848695 
   34.85977     53.3827588 
   34.8604658   53.3807698 
   34.8606086   53.380532  
   34.8616453   53.3796486 
   34.8654709   53.3781854 
   34.8654255   53.3779644 
   34.8654056   53.3778389 
   34.8652187   53.377022  
   34.8615844   53.3763117 
   34.8603702   53.3760199 
   34.8577915   53.3755463 
   34.8589541   53.3736285 
   34.8591833   53.373303  
   34.859242    53.3731771 
   34.8598742   53.3721097 
   34.8602172   53.3715859 
   34.8616482   53.3712787 
   34.8620263   53.3712241 
   34.864563    53.3709454 
   34.8646685   53.3709435 
   34.863687    53.3695288 
   34.8633739   53.3691172 
   34.8631439   53.3683882 
   34.862731    53.3668382 
   34.8617947   53.3650292 
   34.8615673   53.3645856 
   34.8612645   53.3642072 
   34.8600005   53.3622589 
   34.8598997   53.3621242 
   34.8651038   53.3617113 
   34.8671303   53.3610474 
   34.8674144   53.3613125 
   34.8683303   53.3621113 
   34.8692158   53.3629585 
   34.8706173   53.3626123 
   34.8706498   53.3626085 
   34.8707893   53.3625922 
   34.8719788   53.3623209 
   34.8728092   53.3626778 
   34.8756567   53.3638859 
   34.8762603   53.3637459 
   34.8767186   53.3636487 
   34.8779176   53.3643293 
   34.8816575   53.3653623 
   34.8824597   53.3665745 
   34.8825328   53.366663  
   34.8838164   53.3665352 
   34.8840216   53.366457  
   34.8875668   53.3640204 
   34.8898915   53.3656324 
   34.8900461   53.3656873 
   34.8906297   53.3664108 
   34.8910132   53.3680972 
   34.8932077   53.3689709 
   34.8941104   53.3695997 
   34.8946861   53.3709894 
   34.895929    53.3717622 
   34.8963543   53.3720012 
   34.8975072   53.3727355 
   34.8991244   53.3733305 
   34.9015561   53.3723113 
   34.9021638   53.3722206 
   34.9047069   53.3718997 
   34.9051633   53.3718509 
   34.9064354   53.3733726 
   34.9074313   53.3737292 
   34.9082695   53.3736378 
   34.9092762   53.3732933 
   34.9104877   53.3730031 
   34.9105046   53.3729498 
   34.9103634   53.3728901 
   34.9099144   53.3728936 
   34.9098178   53.3728365 
   34.909881    53.3727641 
   34.9102441   53.3728111 
   34.911234    53.3727059 
   34.9114361   53.3727594 
   34.9119034   53.3730091 
   34.9119467   53.3731063 
   34.9118492   53.3733169 
   34.911932    53.3734977 
   34.912384    53.3737748 
   34.9126567   53.3738117 
   34.9131637   53.3736152 
   34.9134548   53.3735842 
   34.9137155   53.3736097 
   34.9138995   53.3737004 
   34.9138897   53.3738297 
   34.9135785   53.3739406 
   34.9134667   53.3741172 
   34.9137944   53.3742797 
   34.913706    53.37442   
   34.9138435   53.3745533 
   34.9140595   53.3746804 
   34.9143948   53.374792  
   34.914577    53.3748147 
   34.9151213   53.3747647 
   34.9158985   53.3747976 
   34.9161848   53.3747261 
   34.9165262   53.3745701 
   34.9166074   53.3744678 
   34.9165099   53.3742764 
   34.9168272   53.3740062 
   34.9169536   53.3737489 
   34.9167789   53.373465  
   34.9167857   53.3733583 
   34.9170475   53.3730943 
   34.9175347   53.3728596 
   34.9176185   53.3728448 
   34.9183819   53.3730811 
   34.9195386   53.373345  
   34.9230616   53.3725625 
   34.9258636   53.3736665 
   34.9297123   53.3730529 
   34.9371511   53.3744844 
   34.9428591   53.3761564 
   34.9449368   53.3763763 
   34.9464575   53.3760024 
   34.9468535   53.3759001 
   34.9469193   53.3758763 
   34.9469667   53.3758334 
   34.9469716   53.3757678 
   34.9468013   53.3756854 
   34.9468716   53.3755585 
   34.9476307   53.375273  
   34.9481113   53.3749391 
   34.9483605   53.3749083 
   34.9488711   53.3749585 
   34.9488734   53.3747729 
   34.9490598   53.3748473 
   34.9491148   53.37495   
   34.9492504   53.3750146 
   34.9496157   53.3751155 
   34.9497255   53.3753369 
   34.9499199   53.3754129 
   34.9500886   53.3754025 
   34.9504889   53.3752731 
   34.9507755   53.3752615 
   34.9510508   53.3752883 
   34.95136     53.3753858 
   34.9518333   53.3754247 
   34.9522998   53.3751386 
   34.9527208   53.3750669 
   34.9531717   53.3750506 
   34.9534727   53.3750751 
   34.9535711   53.375125  
   34.9537406   53.3753554 
   34.9539734   53.3754612 
   34.9540855   53.3753777 
   34.954182    53.3751412 
   34.954512    53.3749683 
   34.9546438   53.3747016 
   34.9548704   53.374553  
   34.9551222   53.3745397 
   34.955395    53.3744333 
   34.9557388   53.3744508 
   34.9560425   53.3742873 
   34.9569597   53.374407  
   34.9573192   53.3745582 
   34.9576147   53.3745579 
   34.9576209   53.3744971 
   34.957014    53.3742295 
   34.9570974   53.3741931 
   34.9576288   53.3742955 
   34.9578441   53.3743989 
   34.9579926   53.3748356 
   34.9584513   53.3749721 
   34.9586039   53.3750784 
   34.9587171   53.3755638 
   34.9591028   53.3755399 
   34.9589969   53.3754338 
   34.9591726   53.3753017 
   34.959485    53.3752478 
   34.959711    53.3752617 
   34.9602456   53.3754224 
   34.9608931   53.3755397 
   34.9611659   53.3756682 
   34.9613793   53.3756131 
   34.9611586   53.3758169 
   34.9611991   53.3760092 
   34.9617287   53.3761547 
   34.9619186   53.3760452 
   34.9621908   53.3760015 
   34.9621142   53.3761364 
   34.9621771   53.3762495 
   34.9623822   53.3763201 
   34.9627541   53.3763312 
   34.9630786   53.3762783 
   34.9633922   53.376132  
   34.9637991   53.376019  
   34.9638971   53.3756569 
   34.964082    53.3755225 
   34.9643887   53.3755022 
   34.9649533   53.3756199 
   34.9651342   53.3756102 
   34.9656881   53.374946  
   34.9668353   53.3744597 
   34.9681077   53.3741484 
   34.9687618   53.3741648 
   34.9690897   53.3743719 
   34.9692589   53.3743246 
   34.9693356   53.3741801 
   34.9695164   53.3741793 
   34.9699888   53.3742757 
   34.9703886   53.3745239 
   34.9708384   53.3746226 
   34.9709785   53.3745377 
   34.9707251   53.3744717 
   34.9707334   53.3741188 
   34.9713256   53.374051  
   34.9717362   53.3740656 
   34.9718392   53.3738443 
   34.9720481   53.37383   
   34.9720779   53.3736877 
   34.9724063   53.3737516 
   34.9727051   53.373716  
   34.9733938   53.3737518 
   34.9738613   53.3740498 
   34.9743184   53.3740942 
   34.9744554   53.3740435 
   34.9746881   53.3740461 
   34.9748068   53.3738634 
   34.9756685   53.3736422 
   34.9757266   53.3735824 
   34.9755375   53.3735152 
   34.9755349   53.373392  
   34.9756901   53.3732806 
   34.9759824   53.373229  
   34.9766447   53.3734703 
   34.9767065   53.3736658 
   34.9769212   53.3737252 
   34.9770127   53.3739177 
   34.9773463   53.3742039 
   34.9775089   53.3741501 
   34.9776376   53.3740179 
   34.9784696   53.3740366 
   34.9786921   53.3740023 
   34.9786968   53.3738439 
   34.9791933   53.3735956 
   34.9796911   53.3737097 
   34.9800888   53.373689  
   34.9801104   53.3735659 
   34.9803105   53.3735051 
   34.9804228   53.3732775 
   34.9805605   53.373174  
   34.9805862   53.3730469 
   34.9810717   53.3730489 
   34.9818399   53.3731611 
   34.9823504   53.3733561 
   34.982381    53.3734827 
   34.9823061   53.3735977 
   34.9827729   53.3736092 
   34.9832714   53.3734296 
   34.9835651   53.3733813 
   34.983611    53.373235  
   34.9832702   53.3730824 
   34.9832245   53.372867  
   34.9836201   53.3725629 
   34.9841672   53.3723555 
   34.9844878   53.3721617 
   34.9847385   53.371893  
   34.9848733   53.3715879 
   34.9849906   53.3715084 
   34.9857332   53.3711562 
   34.9868612   53.3710393 
   34.9871761   53.3711141 
   34.9874307   53.3710873 
   34.9880012   53.3708127 
   34.987895    53.3705018 
   34.9879948   53.3704463 
   34.9885391   53.3703428 
   34.9888654   53.3701202 
   34.9894674   53.3699002 
   34.9899823   53.3695774 
   34.9904481   53.3690864 
   34.9908599   53.3691154 
   34.9914047   53.3693265 
   34.9915654   53.3693096 
   34.9918057   53.3691209 
   34.9921497   53.369111  
   34.99253     53.3693143 
   34.9928742   53.3693893 
   34.9930503   53.3695565 
   34.9934381   53.3695693 
   34.9937116   53.3696368 
   34.9938914   53.3696031 
   34.9944012   53.3692452 
   34.9947751   53.3691795 
   34.9951442   53.3689762 
   34.9960105   53.3689093 
   34.9961609   53.3689835 
   34.9960933   53.3693873 
   34.9963117   53.3697131 
   34.9965427   53.3698725 
   34.9969573   53.3700126 
   34.9971633   53.3700086 
   34.9977408   53.3694693 
   34.9978969   53.3693859 
   34.9982227   53.369328  
   34.9989584   53.3693398 
   34.9994714   53.3694099 
   34.9997269   53.3696366 
   34.9999823   53.3697752 
   35.0005707   53.369912  
   35.0010269   53.369917  
   35.0016732   53.369802  
   35.0018956   53.3696549 
   35.0021627   53.3693423 
   35.0022161   53.3685231 
   35.0025376   53.3681235 
   35.0027284   53.3680595 
   35.0031342   53.3680204 
   35.0039798   53.3681318 
   35.0041797   53.3680837 
   35.0041742   53.3679797 
   35.0038541   53.3678554 
   35.0038126   53.3677401 
   35.0041955   53.3674759 
   35.0044228   53.3672431 
   35.0048365   53.3671056 
   35.0048998   53.366949  
   35.0045317   53.3668467 
   35.0045543   53.3667348 
   35.0047207   53.3666874 
   35.0051214   53.3667514 
   35.0054785   53.3668793 
   35.005765    53.3670901 
   35.0059386   53.3671244 
   35.0063094   53.3669834 
   35.0067377   53.3667339 
   35.0068902   53.3664912 
   35.0071925   53.3664957 
   35.007363    53.3664467 
   35.0077062   53.3664613 
   35.007924    53.3662525 
   35.0081674   53.3661375 
   35.0081586   53.3658438 
   35.0083145   53.365666  
   35.008256    53.3655066 
   35.0085718   53.3653558 
   35.0083264   53.3648629 
   35.0083043   53.3644444 
   35.0083622   53.3642966 
   35.0092752   53.3637489 
   35.0102392   53.3633142 
   35.0107597   53.3631962 
   35.0127118   53.3629127 
   35.0148341   53.3623593 
   35.0151114   53.3623268 
   35.0161534   53.3623381 
   35.0168818   53.3623962 
   35.0196871   53.3627407 
   35.0209829   53.3628458 
   35.0215803   53.3628333 
   35.0221472   53.3627479 
   35.0232857   53.362393  
   35.0236458   53.3622    
   35.0239968   53.361912  
   35.0243699   53.3614036 
   35.0243739   53.3608704 
   35.0241534   53.3602923 
   35.0237849   53.3598018 
   35.0229632   53.3589434 
   35.0218421   53.357928  
   35.0215725   53.3576197 
   35.0209547   53.3571349 
   35.0194479   53.3557407 
   35.0191781   53.3553702 
   35.0190002   53.3548781 
   35.0192416   53.3540374 
   35.0193674   53.3538325 
   35.02002     53.3529678 
   35.0206651   53.3521536 
   35.0210623   53.3519235 
   35.0228226   53.3513361 
   35.0231357   53.3511927 
   35.0235981   53.3508545 
   35.0239101   53.3504478 
   35.0240506   53.349812  
   35.0240767   53.3492472 
   35.0239721   53.34885   
   35.0238503   53.3486628 
   35.0233988   53.3482599 
   35.0228823   53.3479488 
   35.0213751   53.3472992 
   35.0206016   53.3468924 
   35.0198562   53.3460559 
   35.0196708   53.345183  
   35.019802    53.3446575 
   35.0202713   53.3441473 
   35.0207281   53.3438756 
   35.022448    53.3430536 
   35.0230162   53.3426436 
   35.023211    53.3424273 
   35.0235023   53.341887  
   35.023627    53.341357  
   35.0235369   53.3406614 
   35.023359    53.3402987 
   35.0235758   53.3395834 
   35.0239048   53.3377064 
   35.0251406   53.3368831 
   35.0272377   53.3370385 
   35.0282741   53.3366004 
   35.0287371   53.336234  
   35.0304791   53.3344864 
   35.0513315   53.3340403 
   35.0667263   53.3338143 
   35.0666831   53.3294354 
   35.0665122   53.3249203 
   35.0678253   53.3248664 
   35.0717834   53.324831  
   35.0841343   53.3248097 
   35.0842945   53.3247968 
   35.0843761   53.3245298 
   35.0843401   53.3243232 
   35.0836714   53.3235486 
   35.0837402   53.3230652 
   35.0831599   53.3226831 
   35.0829563   53.3224041 
   35.0828769   53.3220255 
   35.0825345   53.3219886 
   35.0823979   53.3220671 
   35.082336    53.3222231 
   35.0821313   53.3222867 
   35.0819339   53.3222971 
   35.0815188   53.3222004 
   35.0812575   53.322044  
   35.0812657   53.3217908 
   35.081068    53.3216188 
   35.0807067   53.3215047 
   35.0803272   53.3214684 
   35.0801533   53.3213707 
   35.0801185   53.321225  
   35.0797324   53.32095   
   35.0796513   53.3206421 
   35.0795701   53.3205617 
   35.0794072   53.3205448 
   35.0791513   53.320751  
   35.0790128   53.32074   
   35.0789078   53.3206782 
   35.078949    53.3202497 
   35.0787559   53.3201134 
   35.0793788   53.3197237 
   35.0793026   53.3196223 
   35.0789492   53.3194965 
   35.0789002   53.3194098 
   35.0789386   53.3192842 
   35.0793364   53.3190487 
   35.079393    53.3189478 
   35.0791191   53.3186877 
   35.0791207   53.3186112 
   35.0795563   53.3184033 
   35.079624    53.318162  
   35.0798905   53.3181752 
   35.0800352   53.3180429 
   35.0799931   53.3178521 
   35.0798507   53.3177394 
   35.0797161   53.3175126 
   35.0799307   53.3170256 
   35.0797045   53.3168722 
   35.0789645   53.3168423 
   35.0786492   53.3166321 
   35.0786111   53.3164413 
   35.0790872   53.3161704 
   35.0788738   53.3158101 
   35.0789742   53.3154122 
   35.0788678   53.3152083 
   35.0786817   53.3151644 
   35.0783937   53.3152681 
   35.0783095   53.3148672 
   35.0779287   53.3147578 
   35.0777077   53.3145786 
   35.0770979   53.3143333 
   35.076478    53.3136996 
   35.076208    53.313545  
   35.0761728   53.3134489 
   35.0762269   53.3134059 
   35.0765977   53.3133328 
   35.076743    53.3132515 
   35.0768236   53.3129901 
   35.0763734   53.3126704 
   35.0762587   53.3118156 
   35.0757824   53.3113854 
   35.0757367   53.3112507 
   35.0760634   53.3109928 
   35.0764343   53.3107964 
   35.07646     53.3106626 
   35.0762409   53.3105917 
   35.0760576   53.3106805 
   35.0758864   53.3107015 
   35.0756539   53.3106189 
   35.0756526   53.3104534 
   35.0759832   53.3100844 
   35.0760079   53.3098423 
   35.0759091   53.3097713 
   35.0753615   53.3097349 
   35.0752277   53.3096566 
   35.0751966   53.3095196 
   35.0754299   53.3092387 
   35.0754357   53.3091375 
   35.0752614   53.3090808 
   35.0747025   53.3090233 
   35.0745406   53.3089011 
   35.0745297   53.3088157 
   35.0746665   53.308708  
   35.0752931   53.3086189 
   35.0752842   53.3085463 
   35.0750056   53.3083793 
   35.0749848   53.308291  
   35.0752139   53.3079921 
   35.0751169   53.3076849 
   35.0753793   53.307588  
   35.0754792   53.3073885 
   35.0753005   53.3072066 
   35.0756884   53.306901  
   35.0756825   53.3068063 
   35.0754321   53.3065664 
   35.075436    53.3065006 
   35.0757877   53.3062319 
   35.0757683   53.3059176 
   35.0758536   53.3058098 
   35.076996    53.3048312 
   35.0774877   53.3032179 
   35.0828668   53.3003721 
   35.0827027   53.2989796 
   35.0870546   53.2970661 
   35.0889586   53.2960249 
   35.0897831   53.295728  
   35.0925087   53.2952529 
   35.0931961   53.2949039 
   35.0935734   53.2948001 
   35.0939756   53.2948066 
   35.094932    53.2949343 
   35.0953569   53.2949006 
   35.0955327   53.2948435 
   35.0956351   53.2947512 
   35.0956704   53.2946442 
   35.0955518   53.2942668 
   35.096031    53.2937308 
   35.0968502   53.2936346 
   35.0977461   53.2936087 
   35.0989437   53.2932767 
   35.0995403   53.2930706 
   35.0999306   53.2926864 
   35.100854    53.2924347 
   35.1020608   53.2918598 
   35.1027439   53.2915901 
   35.1041778   53.2913536 
   35.1051302   53.2913382 
   35.1055041   53.291435  
   35.1060716   53.2917948 
   35.1064434   53.2919401 
   35.1071055   53.2919073 
   35.1076964   53.2919441 
   35.1083868   53.2914158 
   35.1090635   53.2909975 
   35.1094562   53.2907979 
   35.1101989   53.2905778 
   35.110998    53.2902054 
   35.1114859   53.2900545 
   35.1124036   53.2898627 
   35.1132447   53.2895995 
   35.1145086   53.2893347 
   35.1156374   53.2889973 
   35.116174    53.2889567 
   35.1167308   53.2885681 
   35.1171628   53.2884931 
   35.1172836   53.2884286 
   35.1175107   53.2882089 
   35.1177516   53.2881736 
   35.1179519   53.2882556 
   35.1181805   53.2882656 
   35.1184107   53.2880768 
   35.1187979   53.2880131 
   35.1189694   53.2878673 
   35.1190185   53.2875595 
   35.1194362   53.2873466 
   35.1195719   53.2871513 
   35.1195338   53.2865724 
   35.1195881   53.2864541 
   35.1197544   53.2863073 
   35.1201064   53.2861364 
   35.1248974   53.2845823 
   35.1266248   53.2841528 
   35.1270769   53.2841274 
   35.1315751   53.2825959 
   35.1366767   53.2797174 
   35.1396219   53.2793233 
   35.1410996   53.2791226 
   35.1415901   53.2792342 
   35.1418289   53.2792328 
   35.1420477   53.2791716 
   35.1426693   53.2792403 
   35.1435404   53.2792552 
   35.1446499   53.279109  
   35.1448057   53.279404  
   35.1450266   53.2794923 
   35.1451556   53.2794874 
   35.1454743   53.2793699 
   35.1455593   53.2792795 
   35.1458049   53.2792905 
   35.1461013   53.2791997 
   35.146593    53.279166  
   35.146912    53.2787941 
   35.147403    53.278859  
   35.1478144   53.2786569 
   35.1482889   53.2787447 
   35.1484803   53.2786592 
   35.1496803   53.2784813 
   35.1501576   53.2783188 
   35.1507386   53.2782886 
   35.151064    53.2780969 
   35.1518919   53.2781342 
   35.1523391   53.2779782 
   35.1528834   53.2779238 
   35.1531764   53.2777958 
   35.1540798   53.2771835 
   35.1573565   53.2772172 
   35.1579208   53.2771301 
   35.1588387   53.2773047 
   35.1592687   53.2772399 
   35.159726    53.2769898 
   35.1598806   53.2769923 
   35.1599528   53.2769991 
   35.1578879   53.2811066 
   35.1579032   53.2813559 
   35.1580675   53.2814409 
   35.1613413   53.2811923 
   35.1646762   53.2824537 
   35.1667193   53.2827145 
   35.1690551   53.2824365 
   35.1717821   53.2822396 
   35.1749886   53.2818072 
   35.1762098   53.2821358 
   35.1791029   53.2822771 
   35.1797154   53.2831356 
   35.1798933   53.2832576 
   35.1798291   53.2834675 
   35.1798656   53.2836468 
   35.1798217   53.2837652 
   35.1795987   53.2839737 
   35.1785301   53.2848783 
   35.1785361   53.285006  
   35.1803305   53.2854556 
   35.1830894   53.2866284 
   35.1844789   53.2870047 
   35.1852882   53.287291  
   35.1871473   53.2883689 
   35.1886052   53.2888174 
   35.1893535   53.2889634 
   35.1917996   53.2891828 
   35.1930983   53.2894867 
   35.1951104   53.2898177 
   35.1955625   53.2899254 
   35.1967482   53.2903748 
   35.2020052   53.2909752 
   35.2021138   53.2909447 
   35.2024457   53.2904876 
   35.2022251   53.2903222 
   35.2008024   53.2896387 
   35.2001628   53.289122  
   35.19988     53.2887754 
   35.1991483   53.2863787 
   35.197668    53.2855769 
   35.1964543   53.2849888 
   35.194419    53.2841621 
   35.1917434   53.2833362 
   35.1901109   53.2827469 
   35.1886571   53.2823902 
   35.1881096   53.2823333 
   35.187199    53.281246  
   35.1885761   53.2803665 
   35.1920739   53.2802188 
   35.2002172   53.282365  
   35.2047215   53.2824893 
   35.2057395   53.282464  
   35.2110355   53.2813992 
   35.2120195   53.2813912 
   35.2294467   53.2807404 
   35.2327283   53.2804956 
   35.2383809   53.2803512 
   35.2409888   53.2816466 
   35.2453461   53.2836976 
   35.2456558   53.2839925 
   35.2469844   53.2845005 
   35.2493387   53.285652  
   35.2494167   53.286557  
   35.2512453   53.287966  
   35.2530971   53.2887285 
   35.2557236   53.2899142 
   35.2548833   53.2917289 
   35.2542256   53.2935724 
   35.2536032   53.2956181 
   35.2527646   53.298007  
   35.2524439   53.2991958 
   35.2513281   53.3021093 
   35.2508914   53.3035758 
   35.2507388   53.3038733 
   35.2493207   53.3081058 
   35.2492027   53.3088936 
   35.248686    53.3104869 
   35.2490891   53.3106    
   35.250877    53.3109    
   35.2490713   53.3157496 
   35.2474435   53.3194654 
   35.2440075   53.3231122 
   35.2421837   53.3249337 
   35.2398339   53.3274709 
   35.2414425   53.3270097 
   35.2502643   53.3242145 
   35.2523831   53.3236118 
   35.262348    53.320562  
   35.2663906   53.3192872 
   35.2665913   53.3191786 
   35.2668241   53.3191397 
   35.271695    53.3209731 
   35.2715492   53.3211223 
   35.2717215   53.321194  
   35.2718523   53.3210368 
   35.2792301   53.3238475 
   35.3000318   53.3321893 
   35.3045529   53.334022  
   35.3062115   53.3347658 
   35.3068888   53.333934  
   35.3237942   53.3340465 
   35.3370323   53.3340723 
   35.3658466   53.3342175 
   35.3689168   53.3342894 
   35.3788117   53.3342776 
   35.3816545   53.3361649 
   35.3835638   53.3373935 
   35.3834962   53.3392155 
   35.3834857   53.3399611 
   35.3835406   53.3407512 
   35.3835559   53.3416747 
   35.3835494   53.3435556 
   35.3835855   53.3458866 
   35.3835337   53.3474274 
   35.3835603   53.349141  
   35.3835367   53.3498617 
   35.3832949   53.3498583 
   35.3833301   53.3501698 
   35.3832901   53.3506332 
   35.3835197   53.3506596 
   35.3835836   53.3551187 
   35.3836799   53.3578683 
   35.3837674   53.3615597 
   35.383759    53.3676565 
   35.3837955   53.3687902 
   35.3837913   53.3705787 
   35.3837636   53.3728753 
   35.3837754   53.3734572 
   35.3839071   53.3796008 
   35.3839248   53.381312  
   35.3839835   53.3885836 
   35.3840373   53.3916356 
   35.3840582   53.3931195 
   35.3840737   53.3957623 
   35.3840992   53.3975624 
   35.3871855   53.3975471 
   35.3924581   53.3975521 
   35.395072    53.3975646 
   35.3993205   53.3975539 
   35.4055375   53.3975606 
   35.4097413   53.39756   
   35.4139937   53.3975723 
   35.4140647   53.4004803 
   35.4142265   53.406674  
   35.4146753   53.4069904 
   35.4180927   53.4094135 
   35.419043    53.4101216 
   35.4204586   53.4110879 
   35.4214416   53.4117426 
   35.4217918   53.4119709 
   35.4230158   53.4126152 
   35.4239368   53.4130936 
   35.4246536   53.4134732 
   35.4251974   53.4137524 
   35.4252839   53.4137945 
   35.4255775   53.4139346 
   35.4260468   53.4141742 
   35.4277492   53.4144232 
   35.4286451   53.4145563 
   35.4288833   53.414592  
   35.429886    53.4147323 
   35.4305816   53.4148378 
   35.4318998   53.4150389 
   35.4320554   53.4150786 
   35.4324465   53.4151038 
   35.4329825   53.4151243 
   35.4341573   53.4151603 
   35.4344145   53.415175  
   35.4353904   53.4152053 
   35.4357463   53.4152257 
   35.4358966   53.4152144 
   35.4366074   53.4150635 
   35.4384012   53.4146954 
   35.4397686   53.4144097 
   35.4413728   53.4140908 
   35.4417681   53.4140149 
   35.4430732   53.4137534 
   35.443865    53.4137795 
   35.4443225   53.4137821 
   35.4449017   53.4138057 
   35.4459845   53.4138376 
   35.4468832   53.4138654 
   35.4471634   53.4138752 
   35.4476682   53.4138972 
   35.4482638   53.4139176 
   35.4485602   53.4139435 
   35.4494643   53.4139778 
   35.4498812   53.4139852 
   35.4501045   53.4139982 
   35.4508218   53.4141318 
   35.4518246   53.4143132 
   35.452979    53.4145302 
   35.4537856   53.4146873 
   35.4546396   53.414846  
   35.4554205   53.4149926 
   35.4560349   53.4151067 
   35.4567806   53.415246  
   35.4574181   53.4153771 
   35.4576359   53.415475  
   35.4581813   53.4156992 
   35.4588579   53.4159864 
   35.4599189   53.4164443 
   35.4603627   53.4166344 
   35.4610691   53.4169467 
   35.4614291   53.4171004 
   35.4616659   53.4172129 
   35.4620584   53.417382  
   35.4626309   53.4176352 
   35.4631871   53.4178787 
   35.4637217   53.4181076 
   35.4636878   53.4182928 
   35.4636539   53.4183793 
   35.4636146   53.418482  
   35.4635183   53.4188144 
   35.4634343   53.419044  
   35.4633855   53.4191871 
   35.463212    53.4196869 
   35.4631225   53.4199998 
   35.4630913   53.420096  
   35.4630236   53.4201195 
   35.4629843   53.4201453 
   35.4629816   53.4201898 
   35.4629842   53.4202731 
   35.462972    53.4203095 
   35.4629057   53.420354  
   35.462621    53.4212151 
   35.4628916   53.4215475 
   35.4635951   53.4223943 
   35.4637778   53.4226208 
   35.4648129   53.4238978 
   35.4658981   53.4252323 
   35.4662851   53.4256893 
   35.4662878   53.4257903 
   35.4659801   53.4267106 
   35.4658011   53.427292  
   35.4655015   53.4281394 
   35.4654771   53.4282162 
   35.465347    53.428589  
   35.4654413   53.4297583 
   35.4654629   53.429891  
   35.465514    53.4305889 
   35.4656219   53.4317736 
   35.4656891   53.432896  
   35.4645922   53.4329217 
   35.4623037   53.4329844 
   35.4605921   53.4330221 
   35.4602739   53.4330479 
   35.4598988   53.4330648 
   35.4582467   53.4331267 
   35.4563834   53.4332153 
   35.4550563   53.4332675 
   35.453805    53.4333166 
   35.4523385   53.4334019 
   35.4512456   53.4334558 
   35.4503194   53.4335049 
   35.4499794   53.4335638 
   35.4485506   53.4337955 
   35.4460939   53.434174  
   35.445995    53.4341764 
   35.4450091   53.4342917 
   35.4434015   53.4344861 
   35.44309     53.4345256 
   35.4422747   53.4346523 
   35.4410408   53.434816  
   35.4402851   53.4349273 
   35.439765    53.4349926 
   35.4391745   53.4351194 
   35.4387802   53.4353424 
   35.4387056   53.4353836 
   35.4386094   53.4354548 
   35.4385116   53.4357596 
   35.4384967   53.4357887 
   35.4384533   53.4358186 
   35.4383463   53.435825  
   35.4383476   53.4358727 
   35.4383611   53.435952  
   35.4383718   53.436024  
   35.4383704   53.4361008 
   35.4383527   53.4361687 
   35.4382984   53.4362746 
   35.4382645   53.4363789 
   35.4382563   53.4364339 
   35.438286    53.4364663 
   35.4383428   53.4365383 
   35.4383739   53.4366006 
   35.4384077   53.436641  
   35.4384375   53.4366879 
   35.4384388   53.4367292 
   35.4384049   53.4367728 
   35.4383629   53.4368181 
   35.4383358   53.436827  
   35.4382992   53.4368261 
   35.4382762   53.4368358 
   35.4382139   53.4368463 
   35.4380906   53.4368447 
   35.4379986   53.4368252 
   35.4379485   53.4368074 
   35.4379106   53.4367823 
   35.437874    53.4367775 
   35.4378293   53.4367572 
   35.4377901   53.4367232 
   35.4377211   53.4366868 
   35.4376683   53.4366601 
   35.4376128   53.4366544 
   35.4375451   53.4366593 
   35.4374868   53.4366859 
   35.4373351   53.4367255 
   35.4372402   53.4368055 
   35.4366523   53.4369614 
   35.4364611   53.4371869 
   35.4363987   53.4372734 
   35.43743     53.4377418 
   35.4390487   53.4385394 
   35.4394361   53.4387279 
   35.4431432   53.4404869 
   35.4401589   53.4414495 
   35.4382842   53.4408356 
   35.4365317   53.4405636 
   35.4300401   53.4432915 
   35.4307168   53.4435953 
   35.4402809   53.4442322 
   35.441194    53.444322  
   35.44209     53.4443945 
   35.4426079   53.444397  
   35.4436627   53.4444764 
   35.4449292   53.4445467 
   35.4458857   53.44459   
   35.4474661   53.4446536 
   35.4492959   53.4447287 
   35.4498365   53.4447491 
   35.4506078   53.4448103 
   35.4519045   53.4448987 
   35.4525586   53.4449327 
   35.4550955   53.4450733 
   35.4564111   53.4451526 
   35.4572798   53.445203  
   35.4580652   53.4452489 
   35.4593695   53.4454997 
   35.4802719   53.445975  
   35.4839129   53.4461012 
   35.4896833   53.4462434 
   35.4930236   53.4463345 
   35.4935031   53.4463531 
   35.4940531   53.4463482 
   35.4953615   53.4462308 
   35.4965589   53.4461062 
   35.4979811   53.445971  
   35.4993302   53.4458552 
   35.500579    53.4457717 
   35.5024347   53.4456339 
   35.5034817   53.4455569 
   35.5048091   53.4454645 
   35.5060376   53.4453737 
   35.5088237   53.4452131 
   35.5105899   53.4450906 
   35.512371    53.4449753 
   35.5125159   53.4449615 
   35.5125647   53.4449551 
   35.5126121   53.4449623 
   35.5145232   53.4448478 
   35.5161255   53.4447237 
   35.5177332   53.4446416 
   35.5196389   53.4445318 
   35.5220945   53.4443871 
   35.5244836   53.4442311 
   35.5266046   53.444092  
   35.5284303   53.4439701 
   35.5285021   53.4439652 
   35.5285997   53.444046  
   35.5294552   53.4447832 
   35.5313058   53.4462947 
   35.5342995   53.448738  
   35.5362101   53.4503011 
   35.5372435   53.4511967 
   35.5384518   53.4521625 
   35.5407139   53.4539647 
   35.5407397   53.4540036 
   35.5424826   53.4554397 
   35.5434675   53.4563554 
   35.5440128   53.4568403 
   35.5440117   53.4531892 
   35.5440894   53.4487707 
   35.5439379   53.4479694 
   35.5437968   53.4478295 
   35.5433046   53.4474683 
   35.5426808   53.4469632 
   35.5419838   53.4464015 
   35.5415756   53.4460516 
   35.5408636   53.4454503 
   35.5404337   53.4450615 
   35.5399985   53.4447488 
   35.5388325   53.4438889 
   35.5384657   53.4430052 
   35.538114    53.4422937 
   35.5373929   53.4419018 
   35.5374273   53.4412347 
   35.5366698   53.4409277 
   35.535869    53.4406054 
   35.5355655   53.4404826 
   35.5354006   53.439687  
   35.5343951   53.4392111 
   35.5337013   53.4388394 
   35.5333332   53.4378797 
   35.5334658   53.4377729 
   35.5336106   53.4376815 
   35.5342317   53.4373019 
   35.5347689   53.4369435 
   35.5354661   53.4366917 
   35.5360766   53.4365127 
   35.5371203   53.4362203 
   35.5372821   53.4356008 
   35.5360453   53.4352973 
   35.5352465   53.4353963 
   35.5339832   53.4355464 
   35.5330205   53.4356657 
   35.5325655   53.4357096 
   35.5324605   53.4351209 
   35.5323053   53.434326  
   35.5321704   53.4336525 
   35.531944    53.4334358 
   35.5316878   53.4331424 
   35.5313258   53.4327803 
   35.5308909   53.432512  
   35.5305168   53.4322461 
   35.5299491   53.4318711 
   35.5295276   53.4315769 
   35.5289598   53.4311129 
   35.5285098   53.4307136 
   35.5288261   53.4301854 
   35.5279011   53.4299893 
   35.5262299   53.4296405 
   35.5256785   53.4293472 
   35.5253237   53.4293085 
   35.5243324   53.42916   
   35.5231421   53.4289801 
   35.5220723   53.428809  
   35.5216254   53.4287606 
   35.5189352   53.4289418 
   35.5178276   53.4289365 
   35.5167891   53.4288995 
   35.5161865   53.4288876 
   35.5159632   53.4289208 
   35.5150385   53.429048  
   35.5143874   53.4292552 
   35.513521    53.4295926 
   35.5121091   53.4301121 
   35.5114824   53.4304438 
   35.5107149   53.4307982 
   35.5094208   53.4313815 
   35.5076637   53.4321962 
   35.5062038   53.4318811 
   35.5050391   53.4316032 
   35.5054368   53.4308219 
   35.5057304   53.4303391 
   35.5053035   53.4295879 
   35.5051071   53.4293438 
   35.5049174   53.4291271 
   35.5049822   53.4288473 
   35.5045011   53.4278899 
   35.5039307   53.4269786 
   35.5036366   53.4264991 
   35.5025456   53.4269821 
   35.5007633   53.4258421 
   35.5002135   53.4254993 
   35.5014751   53.4248514 
   35.5017417   53.4247155 
   35.5019949   53.4246111 
   35.5022967   53.4244542 
   35.502172    53.4240928 
   35.5021218   53.4238833 
   35.502004    53.4238445 
   35.5019187   53.4237936 
   35.501855    53.4237378 
   35.5018198   53.4236723 
   35.5018117   53.4236125 
   35.5018157   53.4235405 
   35.5018387   53.423479  
   35.501886    53.4234143 
   35.5019348   53.423361  
   35.501566    53.4221965 
   35.5036278   53.4221606 
   35.5038972   53.4221606 
   35.5040394   53.4222584 
   35.5041153   53.4223352 
   35.5041478   53.4224298 
   35.5041384   53.4225018 
   35.5041005   53.422598  
   35.5040274   53.4226805 
   35.503907    53.4227582 
   35.5037419   53.4228342 
   35.5035794   53.4229127 
   35.5034305   53.4229645 
   35.5031287   53.4230486 
   35.5028742   53.4231182 
   35.502575    53.4232024 
   35.502437    53.4232436 
   35.5023138   53.4233011 
   35.5021974   53.423365  
   35.5021094   53.423424  
   35.5020377   53.423517  
   35.5020161   53.4235599 
   35.5020107   53.4236448 
   35.5020297   53.423703  
   35.5020622   53.4237329 
   35.5022165   53.4237499 
   35.5022964   53.4237523 
   35.5023817   53.4237458 
   35.5026646   53.4236641 
   35.5029285   53.4235662 
   35.5032331   53.4234287 
   35.5036662   53.4232264 
   35.5039897   53.4230808 
   35.5041589   53.4229838 
   35.504416    53.422839  
   35.504504    53.4227759 
   35.5046258   53.4227031 
   35.5047436   53.4226375 
   35.5048464   53.422589  
   35.5049791   53.4225558 
   35.5051361   53.4225518 
   35.5053365   53.4225857 
   35.5055518   53.4226552 
   35.5057386   53.4227061 
   35.505958    53.4227627 
   35.5064305   53.4228871 
   35.5066634   53.4229582 
   35.5067826   53.4230189 
   35.5068625   53.4230722 
   35.5069465   53.4231733 
   35.5070345   53.4232744 
   35.5070901   53.4233285 
   35.50717     53.4233657 
   35.5072688   53.4233988 
   35.5074381   53.4234271 
   35.5078131   53.4234869 
   35.5081949   53.423541  
   35.5083831   53.4235862 
   35.5087081   53.4236654 
   35.5089152   53.423726  
   35.509316    53.4238294 
   35.5095381   53.4238957 
   35.5097466   53.4239733 
   35.5099362   53.4240355 
   35.5100987   53.4240864 
   35.5102572   53.4241422 
   35.5104576   53.4242133 
   35.5107067   53.424299  
   35.5110398   53.4244089 
   35.5112985   53.4244921 
   35.5115165   53.4245462 
   35.5116573   53.4245785 
   35.5118442   53.4246108 
   35.5119877   53.4246326 
   35.5121001   53.4246423 
   35.5122314   53.4246431 
   35.5124061   53.4246697 
   35.5126159   53.4247279 
   35.5127541   53.424778  
   35.5128881   53.4248419 
   35.5130128   53.4249146 
   35.513197    53.4250367 
   35.5133162   53.4251191 
   35.5133636   53.425183  
   35.5133758   53.4252542 
   35.5133799   53.4253269 
   35.5134003   53.425373  
   35.5134911   53.4254814 
   35.5135886   53.4255889 
   35.5136564   53.4256617 
   35.513693    53.4257692 
   35.5137012   53.4258517 
   35.5136945   53.4259851 
   35.5137839   53.426083  
   35.5138151   53.4261331 
   35.5138192   53.4261654 
   35.5138111   53.4262261 
   35.513688    53.426387  
   35.5135825   53.4265051 
   35.5135514   53.4265666 
   35.5135529   53.4267089 
   35.5135624   53.4268715 
   35.5135747   53.4269798 
   35.5136005   53.4270485 
   35.5136506   53.4271076 
   35.5137333   53.4271852 
   35.5138213   53.4272337 
   35.5139391   53.4272765 
   35.5140204   53.4272894 
   35.51413     53.4272894 
   35.5142207   53.4272748 
   35.514593    53.4271777 
   35.5147216   53.4271558 
   35.5149667   53.4271574 
   35.5151278   53.4271743 
   35.5152131   53.4271969 
   35.5153174   53.4272276 
   35.5154989   53.4272931 
   35.5156316   53.4273432 
   35.5157684   53.4273884 
   35.5158632   53.427411  
   35.515958    53.4274215 
   35.5160094   53.4274377 
   35.5160717   53.4274854 
   35.516107    53.427559  
   35.5161288   53.4277627 
   35.5161776   53.4278719 
   35.5162048   53.4279576 
   35.5162726   53.42807   
   35.5163728   53.4281775 
   35.5164474   53.4282479 
   35.516553    53.4283149 
   35.5167115   53.4283796 
   35.5168347   53.4284103 
   35.5169227   53.4284143 
   35.517096    53.4283892 
   35.5173803   53.4283446 
   35.5177174   53.4283138 
   35.5181223   53.4282902 
   35.5185271   53.4282683 
   35.5186909   53.4282488 
   35.5188033   53.428227  
   35.5189251   53.4282124 
   35.5190063   53.4282148 
   35.5191377   53.4282414 
   35.5192623   53.4282891 
   35.5194695   53.4283181 
   35.5197809   53.4283229 
   35.5200422   53.4283455 
   35.5205175   53.4283534 
   35.520726    53.4283671 
   35.5208438   53.4283582 
   35.5210509   53.4283063 
   35.5212472   53.4282222 
   35.5213514   53.4281744 
   35.5214475   53.4281129 
   35.5216328   53.4279859 
   35.5216951   53.4279374 
   35.521783    53.4278622 
   35.5218642   53.4278088 
   35.5218791   53.427757  
   35.5218723   53.4277141 
   35.5218248   53.4276495 
   35.5217422   53.4275937 
   35.5216595   53.4275509 
   35.5215512   53.4275258 
   35.521432    53.4275073 
   35.5213197   53.4275008 
   35.521191    53.4274847 
   35.5211098   53.427475  
   35.5210556   53.4274589 
   35.520973    53.4274322 
   35.5209174   53.4273991 
   35.5208416   53.4273473 
   35.5207806   53.4273061 
   35.5207115   53.4272778 
   35.5206276   53.4272536 
   35.5205409   53.4272374 
   35.5204366   53.4272278 
   35.5203243   53.4272254 
   35.5202281   53.427202  
   35.5201577   53.4271777 
   35.5200886   53.4271405 
   35.5200222   53.4270953 
   35.5199288   53.4270419 
   35.5198583   53.4270137 
   35.5197865   53.4269854 
   35.5196809   53.4269547 
   35.519559    53.4269013 
   35.5194859   53.4268577 
   35.5194303   53.4268173 
   35.5193842   53.4267623 
   35.5193517   53.4267146 
   35.5193327   53.4266451 
   35.5193299   53.4265788 
   35.5193542   53.4265043 
   35.5193961   53.4263846 
   35.5194448   53.4263199 
   35.5195016   53.4262593 
   35.519576    53.4262083 
   35.5197236   53.426142  
   35.5198548   53.426095  
   35.5199916   53.4260675 
   35.5202312   53.4260472 
   35.5212114   53.4260461 
   35.5215648   53.4260678 
   35.5218369   53.4260758 
   35.5221727   53.4260967 
   35.5223555   53.4261274 
   35.5225031   53.4261637 
   35.5226277   53.4262098 
   35.5228851   53.4263205 
   35.5230964   53.4264126 
   35.5233971   53.4265961 
   35.5235787   53.4267125 
   35.5237101   53.4267925 
   35.5238171   53.4268474 
   35.5239755   53.4269113 
   35.5241313   53.4269597 
   35.5243656   53.4270082 
   35.524635    53.4270477 
   35.5248056   53.4270662 
   35.5250819   53.4270855 
   35.5253391   53.4271008 
   35.5255815   53.4270991 
   35.5257006   53.4271055 
   35.5258564   53.4271475 
   35.5259688   53.4271904 
   35.5260243   53.4272114 
   35.5260744   53.4272146 
   35.5261977   53.4272469 
   35.5263304   53.4273002 
   35.5264645   53.4273681 
   35.5265702   53.4274214 
   35.5267612   53.4275467 
   35.5268913   53.427634  
   35.5270051   53.427718  
   35.5270877   53.427773  
   35.5271826   53.4278571 
   35.5272545   53.4279363 
   35.5272978   53.4279888 
   35.527405    53.4281546 
   35.527462    53.428292  
   35.5275366   53.4284901 
   35.5275828   53.4285572 
   35.527641    53.4286259 
   35.527721    53.4287019 
   35.5277901   53.4287553 
   35.5279093   53.4288345 
   35.528053    53.428925  
   35.5281492   53.4289864 
   35.5282061   53.4290236 
   35.5283388   53.429064  
   35.5284295   53.4290874 
   35.5285595   53.4290946 
   35.528657    53.4291059 
   35.5287884   53.4291042 
   35.5289278   53.4290913 
   35.5291038   53.4290588 
   35.5292987   53.4290038 
   35.5294287   53.4289544 
   35.5296046   53.4288791 
   35.5297142   53.4288257 
   35.529836    53.4287747 
   35.5299849   53.4287221 
   35.5301446   53.42868   
   35.5303572   53.4286573 
   35.5306374   53.4286563 
   35.5307864   53.4286886 
   35.5309381   53.4287379 
   35.5310506   53.4287839 
   35.5311373   53.4288324 
   35.5312768   53.4289351 
   35.5313432   53.42899   
   35.5314083   53.4290499 
   35.5314747   53.4291355 
   35.5315276   53.4292059 
   35.5315548   53.4292867 
   35.5315603   53.4293765 
   35.5315496   53.4294452 
   35.5315347   53.4294824 
   35.5315388   53.4295245 
   35.5315592   53.4295746 
   35.5315714   53.429632  
   35.5315811   53.4297663 
   35.5315961   53.4298552 
   35.5316313   53.4299304 
   35.5316626   53.4299846 
   35.5317683   53.4300791 
   35.5318591   53.4301486 
   35.5319404   53.430202  
   35.532015    53.4303645 
   35.5320666   53.4304793 
   35.5321019   53.430527  
   35.5321507   53.4305682 
   35.5322171   53.4306167 
   35.5323133   53.4306692 
   35.5323973   53.4307048 
   35.532488    53.4307525 
   35.5325883   53.4308082 
   35.5327252   53.4308931 
   35.5328363   53.4309618 
   35.5329908   53.4310725 
   35.5331304   53.4311768 
   35.5332266   53.4312681 
   35.5333839   53.4314289 
   35.533514    53.4315817 
   35.5336754   53.4317725 
   35.5337879   53.4319035 
   35.5339412   53.4320886 
   35.5341161   53.4323157 
   35.5342124   53.4324394 
   35.5343521   53.4326068 
   35.5344334   53.4327119 
   35.5344904   53.4328218 
   35.5345271   53.4329035 
   35.5345367   53.4329989 
   35.534526    53.4330903 
   35.5344841   53.4331582 
   35.5344639   53.4332399 
   35.5345331   53.4334065 
   35.5345739   53.4335035 
   35.534639    53.4336215 
   35.5346906   53.4337177 
   35.5347286   53.4337735 
   35.5348004   53.433839  
   35.5349034   53.4339141 
   35.5350322   53.4339966 
   35.5351744   53.4340782 
   35.5353221   53.4341493 
   35.5354156   53.4341921 
   35.535547    53.4342349 
   35.5356825   53.4342704 
   35.535841    53.4343003 
   35.5360658   53.4343155 
   35.5361836   53.4343203 
   35.5363826   53.4343203 
   35.5365194   53.4343129 
   35.5369987   53.4342787 
   35.5371707   53.434256  
   35.5375661   53.434234  
   35.5379615   53.4342257 
   35.5384882   53.434219  
   35.5388444   53.4342407 
   35.5390584   53.4342721 
   35.5392995   53.4343214 
   35.5397072   53.4344045 
   35.5398955   53.4344497 
   35.5400324   53.4344965 
   35.5401123   53.4345312 
   35.5402112   53.4345959 
   35.5402858   53.4346791 
   35.5403306   53.4347422 
   35.5403904   53.4348958 
   35.5404353   53.4350535 
   35.5405102   53.4353535 
   35.5405728   53.4355847 
   35.5406232   53.4358079 
   35.5406953   53.4360949 
   35.5407905   53.436369  
   35.5408802   53.4366237 
   35.5409454   53.4367555 
   35.5410295   53.4368646 
   35.5411217   53.4369503 
   35.5412342   53.4370376 
   35.5413887   53.4371264 
   35.5415757   53.4372129 
   35.5419457   53.4373655 
   35.5422248   53.437464  
   35.5425053   53.4375787 
   35.5427085   53.437666  
   35.5428739   53.4377443 
   35.542985    53.4378033 
   35.5432005   53.4379439 
   35.5432832   53.4380086 
   35.5434243   53.4381492 
   35.5435044   53.4382858 
   35.5435628   53.4384532 
   35.5435793   53.4385907 
   35.5435686   53.4387063 
   35.5435093   53.4389109 
   35.5434567   53.4390832 
   35.5433595   53.4392789 
   35.5433055   53.4393792 
   35.5432041   53.4395071 
   35.5431203   53.4396211 
   35.542985    53.4397627 
   35.5428795   53.439859  
   35.5427754   53.4399496 
   35.5426374   53.4400556 
   35.5424818   53.4401681 
   35.5423356   53.4402676 
   35.5422017   53.4403421 
   35.542115    53.4403907 
   35.5418971   53.4404846 
   35.5417712   53.4405356 
   35.5416359   53.4405882 
   35.5414179   53.440653  
   35.5412297   53.440696  
   35.5410848   53.4407243 
   35.5409034   53.4407487 
   35.5407666   53.4407649 
   35.5405472   53.4407812 
   35.5400191   53.4408219 
   35.5398457   53.4408406 
   35.5396738   53.4408697 
   35.5396156   53.4409078 
   35.5395845   53.4409579 
   35.5395791   53.4410073 
   35.5395995   53.4410744 
   35.539666    53.4411665 
   35.5397284   53.4412401 
   35.5398233   53.4413169 
   35.5399657   53.4414058 
   35.5401175   53.4414922 
   35.5402422   53.4415617 
   35.5403384   53.4416094 
   35.5404793   53.4416748 
   35.5405972   53.4417216 
   35.5407205   53.4417685 
   35.5408574   53.4418121 
   35.540997    53.4418525 
   35.541146    53.441888  
   35.5412801   53.4419186 
   35.5415619   53.4419638 
   35.5419033   53.4420267 
   35.5423409   53.4420968 
   35.5426687   53.4421533 
   35.5429383   53.4422033 
   35.5430535   53.4422348 
   35.5431375   53.4422614 
   35.5432134   53.4422937 
   35.5432879   53.4423422 
   35.5433652   53.4423996 
   35.5434534   53.4424893 
   35.543543    53.4426275 
   35.543581    53.4427059 
   35.5435785   53.4428669 
   35.5435705   53.4429526 
   35.5435395   53.4430197 
   35.5434867   53.4430836 
   35.5434285   53.4431281 
   35.54335     53.4431654 
   35.5432241   53.443197  
   35.5431307   53.4432172 
   35.5429248   53.4432351 
   35.5424197   53.4432621 
   35.5421447   53.4432671 
   35.5419226   53.4432817 
   35.5416883   53.4433101 
   35.5415421   53.443345  
   35.5414487   53.4433701 
   35.5413783   53.4433976 
   35.5412917   53.4434486 
   35.5412348   53.4434939 
   35.5411794   53.4435781 
   35.5411497   53.4436614 
   35.5411567   53.4437988 
   35.5411975   53.4439169 
   35.5412545   53.4440381 
   35.5413332   53.4441343 
   35.5413929   53.4441877 
   35.5414824   53.4442507 
   35.5416125   53.4443202 
   35.5417548   53.4443767 
   35.5420394   53.4444518 
   35.5422548   53.444497  
   35.5424472   53.444543  
   35.5427385   53.4446237 
   35.5429553   53.4446851 
   35.5430976   53.4447278 
   35.5432548   53.444786  
   35.5434391   53.4448716 
   35.5435787   53.4449459 
   35.5436926   53.4450203 
   35.5437794   53.4450704 
   35.5439082   53.4451698 
   35.5439896   53.4452555 
   35.5441537   53.4454009 
   35.5442067   53.4455149 
   35.5442176   53.4455828 
   35.5442123   53.4456758 
   35.5441706   53.44584   
   35.544137    53.4460074 
   35.5441357   53.4461109 
   35.5440656   53.4463075 
   35.5440076   53.4464967 
   35.544001    53.4466334 
   35.543993    53.4467321 
   35.5440108   53.4468639 
   35.5440327   53.4469924 
   35.544064    53.4471299 
   35.5441401   53.4472406 
   35.544174    53.4472625 
   35.5442743   53.4473158 
   35.5443651   53.4473707 
   35.5444655   53.4474604 
   35.5445265   53.4475202 
   35.5445754   53.4475687 
   35.5446337   53.4476221 
   35.5447435   53.4476964 
   35.5448357   53.4477627 
   35.5449008   53.4478144 
   35.5450039   53.4479227 
   35.545111    53.4480108 
   35.5452507   53.4481126 
   35.5453646   53.4481975 
   35.5455219   53.4483058 
   35.5456643   53.4484051 
   35.5457497   53.448431  
   35.545873    53.4484657 
   35.5461467   53.4485068 
   35.5462876   53.4485383 
   35.5463798   53.4485778 
   35.5464733   53.4486263 
   35.5465601   53.4486934 
   35.5466171   53.4487564 
   35.5466877   53.4488542 
   35.5468451   53.449049  
   35.5469021   53.4491056 
   35.546974    53.449163  
   35.5470648   53.4492171 
   35.5471665   53.449264  
   35.5472411   53.4493019 
   35.5473386   53.449335  
   35.5474376   53.4493544 
   35.5476299   53.4493745 
   35.5477966   53.4494035 
   35.5479145   53.4494455 
   35.5480908   53.449536  
   35.5482223   53.4496184 
   35.5483579   53.4497307 
   35.5484651   53.4498051 
   35.5485532   53.449856  
   35.5486671   53.4499069 
   35.5487863   53.4499416 
   35.5488649   53.449952  
   35.5489327   53.449973  
   35.5490099   53.4500086 
   35.5491116   53.450074  
   35.549174    53.4501282 
   35.5493001   53.4502025 
   35.5493923   53.450259  
   35.5495821   53.4503519 
   35.5496404   53.4503802 
   35.5498193   53.4504319 
   35.5499603   53.4504601 
   35.550112    53.4504859 
   35.5503898   53.4505197 
   35.5505619   53.4505463 
   35.550734    53.4505826 
   35.5510023   53.4506237 
   35.551196    53.4506518 
   35.551341    53.4506873 
   35.5514766   53.4507415 
   35.551642    53.4508085 
   35.5517342   53.4508513 
   35.5517938   53.4508884 
   35.5519037   53.4509967 
   35.5518998   53.4511011 
   35.5519609   53.4511665 
   35.5519881   53.4512094 
   35.5520993   53.4513217 
   35.5521211   53.4513864 
   35.5521293   53.4514487 
   35.5521149   53.4517139 
   35.5520908   53.4518862 
   35.5520504   53.4520455 
   35.552033    53.4522161 
   35.5520386   53.452301  
   35.5520577   53.4523657 
   35.5521134   53.4524951 
   35.5521664   53.4525646 
   35.5522315   53.452622  
   35.5523359   53.4526777 
   35.5524484   53.4527237 
   35.5526463   53.4527867 
   35.5527886   53.4528368 
   35.5529757   53.4528787 
   35.5532155   53.4529101 
   35.553435    53.4529431 
   35.5537589   53.4529664 
   35.5538591   53.4529865 
   35.5540028   53.4530253 
   35.5541167   53.4530721 
   35.5542279   53.4531206 
   35.5543133   53.4531787 
   35.554369    53.4532563 
   35.5544138   53.4533267 
   35.554437    53.4534083 
   35.5544385   53.4535054 
   35.5544265   53.4536428 
   35.5544187   53.4538377 
   35.5543906   53.4540488 
   35.5543691   53.4541232 
   35.5543016   53.454272  
   35.5542719   53.4543683 
   35.5542655   53.4546052 
   35.5542684   53.4547152 
   35.5542916   53.4547864 
   35.5543459   53.4548405 
   35.5544422   53.4549084 
   35.5545358   53.4549827 
   35.5545833   53.455015  
   35.5547296   53.455019  
   35.5548244   53.4550125 
   35.554926    53.4549882 
   35.5550614   53.4549355 
   35.5552252   53.4548796 
   35.5553309   53.454861  
   35.5554243   53.4548544 
   35.555549    53.4548641 
   35.555671    53.4548899 
   35.5557333   53.4549238 
   35.5557646   53.4549779 
   35.5558054   53.4550936 
   35.5558138   53.4552019 
   35.5558126   53.4553256 
   35.5557277   53.4555667 
   35.5557251   53.4556516 
   35.555728    53.4557891 
   35.5557608   53.4559362 
   35.555833    53.4561958 
   35.5558821   53.4563769 
   35.5559435   53.4566218 
   35.5559899   53.4568159 
   35.5559713   53.4570237 
   35.555954    53.4572389 
   35.5559533   53.4576173 
   35.555963    53.4576949 
   35.5559671   53.4577701 
   35.5559729   53.4579408 
   35.5560152   53.4581219 
   35.5560017   53.4581769 
   35.5559557   53.4582351 
   35.5558691   53.4582983 
   35.5558028   53.4583266 
   35.5556375   53.4583598 
   35.5554926   53.4583955 
   35.5553545   53.458453  
   35.5552489   53.458504  
   35.5551826   53.4585558 
   35.5551204   53.4586311 
   35.5550583   53.4587718 
   35.5550343   53.4590015 
   35.5550373   53.4591575 
   35.5550783   53.4593589 
   35.5551055   53.4594082 
   35.555206    53.4595496 
   35.5552414   53.4596596 
   35.5552821   53.4597218 
   35.5553405   53.4597817 
   35.5553798   53.4598075 
   35.5554151   53.4598269 
   35.5554667   53.4599061 
   35.5555265   53.4600104 
   35.5555619   53.4601212 
   35.5555702   53.4602255 
   35.5555758   53.460312  
   35.5556235   53.4604551 
   35.5556724   53.4605262 
   35.5557402   53.4605876 
   35.5558081   53.4606474 
   35.5558461   53.4606701 
   35.5559113   53.4607614 
   35.5559588   53.4608067 
   35.5559928   53.4608988 
   35.5559969   53.4609409 
   35.5559807   53.4609692 
   35.5559103   53.4609822 
   35.5557518   53.4610033 
   35.5556285   53.4610042 
   35.5555404   53.4610123 
   35.5555011   53.4610301 
   35.555447    53.4610884 
   35.5554431   53.4611741 
   35.5554093   53.4612113 
   35.5552956   53.4612915 
   35.555186    53.4613441 
   35.5551223   53.461374  
   35.5550235   53.4614436 
   35.5549748   53.4614647 
   35.5549084   53.4614777 
   35.5548352   53.4614907 
   35.5546347   53.4614908 
   35.5544924   53.4614876 
   35.5543719   53.4615233 
   35.5542541   53.4615589 
   35.5541511   53.4615986 
   35.5541065   53.4616318 
   35.5540659   53.461707  
   35.5540661   53.4617717 
   35.554077    53.4618404 
   35.5540582   53.461931  
   35.5539935   53.4621235 
   35.553957    53.4621939 
   35.5539382   53.4622812 
   35.5539044   53.4623743 
   35.5538571   53.4624406 
   35.553822    53.4625207 
   35.5537829   53.4626444 
   35.5536789   53.4628572 
   35.5536506   53.4629275 
   35.553617    53.4630788 
   35.553586    53.4631855 
   35.5535563   53.4632494 
   35.5535103   53.4633077 
   35.5534264   53.4633846 
   35.5533926   53.4634404 
   35.5533359   53.4635488 
   35.5533211   53.4636167 
   35.5533145   53.4637526 
   35.5532279   53.4638076 
   35.5531332   53.4638942 
   35.5530778   53.4639694 
   35.5530331   53.4640293 
   35.5529276   53.4641563 
   35.5528384   53.4642583 
   35.5527531   53.4643384 
   35.5526611   53.4644104 
   35.5525555   53.4644751 
   35.5524079   53.4645577 
   35.5523619   53.4646063 
   35.5523417   53.4646669 
   35.5523324   53.4647656 
   35.5523433   53.4648173 
   35.5523813   53.4648602 
   35.5524844   53.4649305 
   35.5526703   53.4650387 
   35.5528195   53.4651268 
   35.5529578   53.4652003 
   35.5531151   53.4652819 
   35.5533009   53.4653691 
   35.5533552   53.4653933 
   35.5535355   53.465462  
   35.5536291   53.4655185 
   35.5537255   53.4655953 
   35.5538843   53.4657456 
   35.5539304   53.4657973 
   35.5539347   53.4658968 
   35.5539281   53.4660011 
   35.5539187   53.4660706 
   35.5538321   53.4661499 
   35.5537536   53.4662042 
   35.5536709   53.4662277 
   35.5535043   53.4662561 
   35.5534244   53.4662618 
   35.5533471   53.4662562 
   35.5532413   53.4662231 
   35.5531654   53.4661722 
   35.5530867   53.4661067 
   35.5530025   53.4660146 
   35.5529495   53.4659475 
   35.5529115   53.4659224 
   35.5528532   53.465899  
   35.5525727   53.4658911 
   35.5524724   53.4658766 
   35.5524141   53.4658451 
   35.5523435   53.465791  
   35.5523096   53.4657546 
   35.5523311   53.4656632 
   35.552335    53.4655759 
   35.5523336   53.4655136 
   35.5523064   53.4654918 
   35.5522536   53.4654878 
   35.5522035   53.4655015 
   35.5521371   53.4655347 
   35.552041    53.4656019 
   35.5519964   53.4656585 
   35.5519707   53.4657079 
   35.5519789   53.4657604 
   35.5520143   53.4658615 
   35.5520254   53.4659828 
   35.5520946   53.4660572 
   35.5521746   53.4661081 
   35.5523414   53.4661832 
   35.5524337   53.4662268 
   35.5527375   53.4663673 
   35.5528324   53.4664214 
   35.5530006   53.4665119 
   35.5532041   53.4666218 
   35.553341    53.4666937 
   35.5534767   53.466751  
   35.5535784   53.4668059 
   35.5536299   53.466839  
   35.5536829   53.4668948 
   35.5537114   53.4669352 
   35.5537332   53.4669934 
   35.5537794   53.467063  
   35.5538296   53.4670856 
   35.553999    53.4671097 
   35.5541915   53.4671282 
   35.5543419   53.4671556 
   35.5544409   53.4671911 
   35.5545142   53.4672299 
   35.5545766   53.4672679 
   35.5546092   53.4673107 
   35.5546242   53.4673471 
   35.5545998   53.4673948 
   35.5545322   53.4674668 
   35.5544469   53.4675275 
   35.5543982   53.4675591 
   35.5543047   53.4675964 
   35.5541936   53.4676134 
   35.5539402   53.467633  
   35.5537126   53.4676679 
   35.5535067   53.4676979 
   35.5533143   53.4677182 
   35.5531164   53.4677216 
   35.5527613   53.4677186 
   35.5526922   53.4677251 
   35.5526137   53.4677429 
   35.5524619   53.467764  
   35.5523128   53.4677641 
   35.5522355   53.4677447 
   35.5521149   53.4676947 
   35.5519942   53.4676487 
   35.5518559   53.4676293 
   35.551719    53.4675971 
   35.5516091   53.4675648 
   35.5515427   53.4675648 
   35.5514601   53.4675818 
   35.5513219   53.4676256 
   35.5512176   53.4676669 
   35.5510835   53.4676726 
   35.5509615   53.4676654 
   35.5508639   53.4676412 
   35.5507527   53.4676016 
   35.5506375   53.4675653 
   35.5505358   53.4675355 
   35.5504883   53.4674983 
   35.5504597   53.4674465 
   35.5504556   53.4673996 
   35.5504908   53.4673616 
   35.5505286   53.4672977 
   35.5505285   53.4672443 
   35.5505027   53.4672136 
   35.5504593   53.4671975 
   35.5503983   53.4671919 
   35.550294    53.4671927 
   35.5502195   53.4672122 
   35.5500624   53.4672875 
   35.5499121   53.4673522 
   35.5498159   53.4673749 
   35.5497319   53.4673968 
   35.5496533   53.4673863 
   35.5495326   53.4673719 
   35.5494147   53.4673719 
   35.549305    53.4673793 
   35.5492156   53.4673979 
   35.5491262   53.4674457 
   35.5490653   53.4674966 
   35.5490369   53.4675549 
   35.5489951   53.4677118 
   35.5489682   53.4678048 
   35.5489737   53.4678962 
   35.5489983   53.4679706 
   35.5490025   53.4680676 
   35.5489944   53.4681169 
   35.5489458   53.4682544 
   35.548912    53.4683102 
   35.548862    53.468362  
   35.5488105   53.4683871 
   35.5486276   53.4684026 
   35.548465    53.4684124 
   35.5483742   53.4684213 
   35.5483105   53.4684367 
   35.5482496   53.4684691 
   35.5482144   53.4685168 
   35.5481738   53.4685524 
   35.5481644   53.4686268 
   35.5481659   53.4686907 
   35.5482433   53.468816  
   35.5483411   53.4689664 
   35.5484539   53.469124  
   35.5484946   53.4692048 
   35.5485354   53.4692792 
   35.5485464   53.4693568 
   35.5485411   53.4694515 
   35.5485235   53.4694919 
   35.5484694   53.4695316 
   35.5483935   53.4695567 
   35.5482757   53.4695891 
   35.5481578   53.4696094 
   35.5480521   53.4696183 
   35.5479816   53.4696272 
   35.5479098   53.4696168 
   35.5477065   53.4696031 
   35.5476116   53.4695838 
   35.5475478   53.4695709 
   35.5473987   53.4695103 
   35.5472658   53.469461  
   35.5471438   53.4694239 
   35.5470014   53.4693957 
   35.5469513   53.4694305 
   35.5469256   53.4694572 
   35.546927    53.4695057 
   35.5469664   53.4695445 
   35.5470451   53.4695857 
   35.5471495   53.4696277 
   35.5473203   53.4696697 
   35.5474965   53.4696696 
   35.5475968   53.4696727 
   35.5477053   53.4697058 
   35.5478083   53.4697559 
   35.5478789   53.4698246 
   35.5479197   53.4698893 
   35.5479279   53.4699443 
   35.5479159   53.470047  
   35.5479132   53.4701141 
   35.5478794   53.470144  
   35.5477413   53.4702128 
   35.5476641   53.4702655 
   35.5476411   53.4702881 
   35.5476101   53.4703924 
   35.5476074   53.4704466 
   35.5476292   53.4704887 
   35.5477052   53.4705379 
   35.5477662   53.4705695 
   35.547914    53.4706276 
   35.5480686   53.4706793 
   35.5481798   53.4707002 
   35.5484062   53.4707535 
   35.5486299   53.470793  
   35.5487356   53.4708099 
   35.5489525   53.4708082 
   35.5491164   53.4707895 
   35.5496409   53.4707811 
   35.549729    53.4707859 
   35.5497521   53.470798  
   35.5498742   53.4709047 
   35.5499571   53.4710001 
   35.5500046   53.4710461 
   35.5500779   53.4711076 
   35.5501064   53.4711399 
   35.5501092   53.4711933 
   35.5500902   53.4712272 
   35.5500537   53.4712531 
   35.5499873   53.4712677 
   35.5498423   53.4712872 
   35.5497109   53.4713229 
   35.5495917   53.4713779 
   35.5494685   53.4714597 
   35.5494252   53.4714945 
   35.5494198   53.4715244 
   35.5494308   53.4716424 
   35.5494485   53.4716974 
   35.5494501   53.4718179 
   35.5494841   53.471898  
   35.5495507   53.4720127 
   35.5495753   53.4721745 
   35.5496042   53.4724106 
   35.5495934   53.4724672 
   35.5495555   53.4725254 
   35.5493933   53.4727697 
   35.5493581   53.4728175 
   35.5493392   53.4728676 
   35.5493393   53.4729291 
   35.5493584   53.4730018 
   35.5494032   53.47306   
   35.5494995   53.4731214 
   35.5495606   53.4731473 
   35.5496772   53.4731804 
   35.5497721   53.4731819 
   35.5498466   53.4731738 
   35.5499401   53.4731495 
   35.5499997   53.4731204 
   35.5500443   53.4730823 
   35.5500903   53.4730039 
   35.5501023   53.4728947 
   35.550159    53.4727459 
   35.5502334   53.4726431 
   35.5503038   53.4726018 
   35.5503783   53.4725792 
   35.5504732   53.4725694 
   35.5505722   53.4725758 
   35.5507918   53.4726    
   35.5510399   53.4726313 
   35.5511538   53.4726612 
   35.5512229   53.4726854 
   35.5512907   53.4727169 
   35.5513694   53.4727662 
   35.5514332   53.4728333 
   35.5514374   53.4729077 
   35.5514253   53.4729449 
   35.551382    53.472987  
   35.5513021   53.4730517 
   35.551275    53.4730808 
   35.5512561   53.4731205 
   35.5512481   53.4731989 
   35.5512685   53.473249  
   35.5513215   53.4733331 
   35.5513894   53.4734204 
   35.5514342   53.4734608 
   35.5515197   53.473498  
   35.5515929   53.4735189 
   35.551822    53.4735334 
   35.5520958   53.4735429 
   35.5522111   53.4735849 
   35.552264    53.4736172 
   35.5522925   53.4736487 
   35.5522912   53.4736989 
   35.5522588   53.4737345 
   35.5521924   53.473779  
   35.5520136   53.4738535 
   35.5519486   53.4738794 
   35.5517996   53.4739604 
   35.5517116   53.4740081 
   35.5516629   53.4740292 
   35.551526    53.4740616 
   35.551419    53.4741061 
   35.5513012   53.4741749 
   35.5511752   53.4742486 
   35.5511346   53.474285  
   35.5511239   53.4743319 
   35.5511266   53.4743723 
   35.5511891   53.474437  
   35.551219    53.4744839 
   35.5512164   53.4745551 
   35.5511921   53.4746351 
   35.5511475   53.4747176 
   35.5510677   53.4748139 
   35.5509622   53.4749474 
   35.5508606   53.4749944 
   35.5508445   53.475072  
   35.5508392   53.475199  
   35.5508448   53.4752636 
   35.5508652   53.4753235 
   35.5509019   53.4754076 
   35.5509359   53.4754601 
   35.5509793   53.4755029 
   35.5510336   53.4755336 
   35.5510879   53.4755554 
   35.5511977   53.4755845 
   35.5513238   53.4755949 
   35.5513726   53.4756014 
   35.5514376   53.4756029 
   35.551458    53.4755965 
   35.5514756   53.4755827 
   35.5514891   53.4755504 
   35.5515063   53.475311  
   35.5515238   53.4752018 
   35.5515372   53.4751573 
   35.5515778   53.475091  
   35.5516089   53.475053  
   35.5516576   53.4750101 
   35.5517064   53.4749793 
   35.551789    53.4749202 
   35.5518784   53.4748692 
   35.5519312   53.4748555 
   35.5520112   53.474849  
   35.5521101   53.4748529 
   35.5521888   53.474865  
   35.552262    53.4748803 
   35.5523501   53.4748843 
   35.5524165   53.4748883 
   35.5525142   53.4749764 
   35.5525564   53.475067  
   35.5525877   53.4751543 
   35.5526108   53.4751899 
   35.5526801   53.4752666 
   35.5527059   53.475286  
   35.5527805   53.4753321 
   35.5528212   53.4753539 
   35.5528578   53.4753628 
   35.5529405   53.475374  
   35.5530435   53.4753756 
   35.5531669   53.4753658 
   35.553278    53.4753585 
   35.5533417   53.4753584 
   35.5535559   53.4753583 
   35.553625    53.4753623 
   35.5537118   53.475372  
   35.5538121   53.4753929 
   35.5539287   53.4754212 
   35.5539898   53.4754454 
   35.5540779   53.4754963 
   35.5541377   53.4755407 
   35.554173    53.4755787 
   35.5541988   53.475628  
   35.5542125   53.4757016 
   35.55421     53.475814  
   35.5541817   53.4759151 
   35.5541479   53.4760106 
   35.5540845   53.4761578 
   35.5540386   53.4762686 
   35.5539899   53.4763422 
   35.5539412   53.4764377 
   35.5539129   53.4765185 
   35.55389     53.4766172 
   35.5538902   53.4767094 
   35.5538998   53.4767571 
   35.5539161   53.4767951 
   35.5539772   53.4768889 
   35.5540397   53.4769535 
   35.5541401   53.4770327 
   35.5542324   53.4770852 
   35.5543301   53.4771264 
   35.5544074   53.4771539 
   35.5545308   53.4771829 
   35.5546542   53.4771974 
   35.5548792   53.4772166 
   35.5550948   53.4772367 
   35.5552751   53.4772674 
   35.5554012   53.4772826 
   35.5554812   53.4772915 
   35.5555639   53.4772882 
   35.5556303   53.4772744 
   35.5556899   53.4772428 
   35.5557751   53.4771611 
   35.5558184   53.4771199 
   35.5559078   53.4770567 
   35.5559701   53.4770138 
   35.5559958   53.4769855 
   35.5560133   53.476937  
   35.5560105   53.4768691 
   35.5559901   53.4768214 
   35.5559276   53.4767527 
   35.5559018   53.476705  
   35.5558691   53.4766371 
   35.5558528   53.4765942 
   35.5558634   53.4764761 
   35.555881    53.4764333 
   35.5559094   53.476392  
   35.5559554   53.4763548 
   35.5560326   53.4763151 
   35.5561288   53.4762859 
   35.5561898   53.476281  
   35.5563416   53.476281  
   35.5564094   53.4762914 
   35.5565097   53.4763148 
   35.5565843   53.4763455 
   35.5566535   53.4763786 
   35.5567173   53.4764231 
   35.5567675   53.4764756 
   35.556792    53.4765322 
   35.5568029   53.476571  
   35.5567922   53.476609  
   35.5567409   53.476723  
   35.556726    53.4767845 
   35.5567262   53.4768549 
   35.5567371   53.4768791 
   35.5567724   53.4769284 
   35.5568172   53.4769656 
   35.5569475   53.4770634 
   35.5570506   53.4771434 
   35.5570737   53.4771652 
   35.557105    53.4772274 
   35.5571472   53.4773333 
   35.5571676   53.4773948 
   35.5571624   53.4774676 
   35.5571421   53.4775379 
   35.5571138   53.4775905 
   35.5570773   53.4776423 
   35.5570001   53.4777103 
   35.5568958   53.4777791 
   35.5567306   53.4778851 
   35.556579    53.4779758 
   35.5564707   53.4780478 
   35.5564355   53.4780769 
   35.5564043   53.4781077 
   35.5563651   53.4781433 
   35.5563353   53.4781773 
   35.5562934   53.4782493 
   35.5562502   53.4783156 
   35.5562326   53.4783585 
   35.5561961   53.4784183 
   35.5561596   53.4784491 
   35.5561135   53.4784887 
   35.5560445   53.4785227 
   35.5559415   53.4785754 
   35.5559104   53.4786029 
   35.555882    53.4786385 
   35.5558794   53.4786733 
   35.5558875   53.478704  
   35.5559039   53.4787428 
   35.5559297   53.4787719 
   35.5559555   53.4788083 
   35.5559555   53.4788349 
   35.5559421   53.4788916 
   35.5559056   53.4789684 
   35.5558963   53.4790646 
   35.555899    53.4790986 
   35.5559045   53.4791172 
   35.5559249   53.4791431 
   35.5559452   53.479156  
   35.5560049   53.4791697 
   35.5560795   53.4791696 
   35.556112    53.4791664 
   35.5561608   53.4791478 
   35.5562122   53.4791202 
   35.5562488   53.4790863 
   35.5563151   53.4790126 
   35.5563773   53.4789115 
   35.5564232   53.4788371 
   35.5564489   53.4788144 
   35.5565058   53.4787772 
   35.5565491   53.4787545 
   35.5565857   53.4787448 
   35.5566603   53.4787407 
   35.556747    53.4787447 
   35.5567918   53.4787528 
   35.5568325   53.4787649 
   35.5568555   53.4787786 
   35.556933    53.4788675 
   35.5570213   53.4789968 
   35.5571191   53.4791059 
   35.5571734   53.4791657 
   35.557202    53.4792102 
   35.5572197   53.4792684 
   35.5572118   53.4794018 
   35.5572064   53.4794277 
   35.5571861   53.4794585 
   35.5571374   53.4795022 
   35.5570765   53.4795467 
   35.5570345   53.4795653 
   35.5569627   53.479588  
   35.5568936   53.4796002 
   35.5568136   53.4796018 
   35.5566997   53.479593  
   35.5565668   53.4795793 
   35.5565099   53.4795729 
   35.5564611   53.4795746 
   35.5563811   53.4795851 
   35.5562863   53.4796086 
   35.5562063   53.4796281 
   35.5561494   53.479633  
   35.5560382   53.4796314 
   35.5559894   53.4796258 
   35.5558891   53.4795975 
   35.5557887   53.4795596 
   35.5556856   53.4795152 
   35.5555757   53.4794692 
   35.5555079   53.4794506 
   35.5554048   53.4794353 
   35.5553357   53.4794353 
   35.5552706   53.4794418 
   35.5551554   53.4794597 
   35.5549955   53.4794986 
   35.5548926   53.4795278 
   35.5547665   53.4795554 
   35.5547096   53.4795724 
   35.5546771   53.4795942 
   35.5546569   53.4796306 
   35.5546434   53.4797002 
   35.5546516   53.4797479 
   35.5546693   53.4797794 
   35.5547345   53.4798796 
   35.5547753   53.4799378 
   35.5548011   53.4799839 
   35.5548094   53.4800316 
   35.5548041   53.4801004 
   35.5547879   53.4801465 
   35.5547446   53.4802201 
   35.5546594   53.4803083 
   35.5546242   53.4803568 
   35.5545958   53.4804126 
   35.5545877   53.4804418 
   35.5545959   53.4804725 
   35.5546095   53.4804983 
   35.5546272   53.4805234 
   35.5546475   53.4805339 
   35.5547736   53.4805524 
   35.5550394   53.480566  
   35.5552088   53.480578  
   35.555301    53.4805796 
   35.5553458   53.4805747 
   35.5554528   53.4805504 
   35.555572    53.480522  
   35.5556384   53.480501  
   35.5557265   53.4804904 
   35.555816    53.4804871 
   35.5558784   53.4804927 
   35.5559191   53.4805008 
   35.5559448   53.4805178 
   35.5559964   53.4805638 
   35.5560196   53.4806115 
   35.5560372   53.4806406 
   35.556059    53.4806544 
   35.5561024   53.4806665 
   35.5562027   53.4806801 
   35.5562339   53.4806866 
   35.5562569   53.4806971 
   35.5562855   53.4807294 
   35.5562856   53.4807844 
   35.5562721   53.4808232 
   35.5562288   53.480909  
   35.5562154   53.4809656 
   35.5562181   53.4809947 
   35.5562372   53.4810238 
   35.5562575   53.481044  
   35.5562725   53.4810577 
   35.5563037   53.4810658 
   35.5564434   53.4811005 
   35.5565871   53.4811328 
   35.5567051   53.4811715 
   35.5567852   53.4812006 
   35.5568367   53.4812345 
   35.5568883   53.4812862 
   35.5569427   53.4813541 
   35.5569631   53.4813937 
   35.5569605   53.4814827 
   35.5569538   53.4815086 
   35.5569227   53.4815579 
   35.5568862   53.4816121 
   35.5568253   53.481655  
   35.5567372   53.4817084 
   35.55666     53.481736  
   35.5565936   53.481749  
   35.5564797   53.481749  
   35.5564201   53.481741  
   35.5563197   53.4817184 
   35.5562383   53.4817031 
   35.5561475   53.481695  
   35.5560323   53.4816951 
   35.5559835   53.4817016 
   35.5559415   53.4817122 
   35.5558954   53.4817445 
   35.5558521   53.4817874 
   35.5558237   53.4818279 
   35.5558089   53.4818707 
   35.5557981   53.4819193 
   35.5557983   53.4820147 
   35.5558161   53.4821497 
   35.5558394   53.4823066 
   35.5558477   53.4823955 
   35.555848    53.4825443 
   35.5558372   53.4825872 
   35.5557967   53.4826632 
   35.5557453   53.4827288 
   35.5556857   53.4828088 
   35.5556506   53.4828622 
   35.5556303   53.4828914 
   35.5556222   53.4829132 
   35.5556276   53.4829286 
   35.5556426   53.4829504 
   35.5556738   53.4829698 
   35.555705    53.4829884 
   35.5557281   53.4829932 
   35.5558609   53.4829947 
   35.5559057   53.4829915 
   35.5559951   53.4829696 
   35.5561292   53.4829113 
   35.5562471   53.4828603 
   35.5563866   53.4828173 
   35.5564978   53.4827898 
   35.5566075   53.4827695 
   35.5566984   53.4827541 
   35.5568529   53.4827556 
   35.5569451   53.4827685 
   35.5571594   53.4827974 
   35.5573085   53.4828168 
   35.5575038   53.4828263 
   35.5576637   53.4828295 
   35.557756    53.4828383 
   35.5578739   53.482852  
   35.5579906   53.4828778 
   35.5580448   53.4828915 
   35.5581425   53.4829149 
   35.5582198   53.4829375 
   35.5582578   53.4829455 
   35.5582998   53.4829455 
   35.558377    53.4829422 
   35.5584069   53.4829317 
   35.5584285   53.4829147 
   35.5584257   53.4828808 
   35.5583999   53.4828331 
   35.5583306   53.4827336 
   35.5582885   53.4826754 
   35.5582626   53.4826229 
   35.5582367   53.4825412 
   35.5582446   53.4824248 
   35.5582568   53.4823738 
   35.558273    53.4823447 
   35.558296    53.4823237 
   35.5583474   53.4822945 
   35.5583989   53.4822759 
   35.5584829   53.4822572 
   35.5585887   53.4822572 
   35.5586741   53.4822587 
   35.5587663   53.4822692 
   35.5588775   53.4822958 
   35.5589603   53.4823233 
   35.5590186   53.4823386 
   35.5591665   53.4823951 
   35.5592818   53.4824427 
   35.5594337   53.4825195 
   35.5596197   53.4826156 
   35.5597377   53.4826818 
   35.5598355   53.4827521 
   35.5599495   53.4828402 
   35.5599835   53.4828765 
   35.5600107   53.4829258 
   35.5600365   53.4829938 
   35.5600816   53.4831749 
   35.5601118   53.4833576 
   35.5601134   53.4835355 
   35.5601026   53.4835646 
   35.5600878   53.4836156 
   35.5600405   53.4836754 
   35.5599714   53.483741  
   35.5598794   53.4838025 
   35.5597629   53.4838608 
   35.5596111   53.4839086 
   35.5594932   53.4839338 
   35.5594024   53.4839508 
   35.5592506   53.4839549 
   35.5591529   53.4839477 
   35.5590838   53.4839381 
   35.5590146   53.4839203 
   35.5589522   53.4838937 
   35.5589006   53.4838492 
   35.5588409   53.4838032 
   35.5588164   53.4837619 
   35.5587878   53.4836803 
   35.5587592   53.4836035 
   35.5587361   53.4835703 
   35.5586967   53.4835396 
   35.5586479   53.4835178 
   35.5585909   53.483509  
   35.5584784   53.4835042 
   35.5584065   53.4835091 
   35.5583645   53.483518  
   35.5583035   53.4835415 
   35.5582385   53.4835747 
   35.5582074   53.4836071 
   35.558198    53.4836329 
   35.558198    53.4836612 
   35.5582157   53.4836887 
   35.5582877   53.4837687 
   35.5583298   53.4838237 
   35.5583529   53.4838714 
   35.5583505   53.4840202 
   35.5583424   53.4840574 
   35.5583452   53.4840849 
   35.5583601   53.4841261 
   35.5583914   53.4841795 
   35.5584376   53.4842215 
   35.5585666   53.484316  
   35.5586439   53.4843653 
   35.5586779   53.4843928 
   35.5587037   53.4844381 
   35.5587241   53.4844825 
   35.5587296   53.4845327 
   35.5587216   53.4845755 
   35.5587013   53.4846232 
   35.5586837   53.4846435 
   35.5585903   53.4847171 
   35.5584901   53.484794  
   35.5584441   53.484845  
   35.5584157   53.4848895 
   35.5584009   53.4849372 
   35.558401    53.4850092 
   35.5584187   53.4850601 
   35.558465    53.4851336 
   35.5585166   53.4852007 
   35.5585682   53.4852549 
   35.5586402   53.4853163 
   35.5587827   53.4854019 
   35.5589808   53.4855166 
   35.5591925   53.4856273 
   35.5594449   53.4857533 
   35.5596091   53.485834  
   35.5597313   53.485893  
   35.5599294   53.4859972 
   35.5601194   53.4860909 
   35.5602225   53.4861369 
   35.5604165   53.4862257 
   35.5605115   53.486262  
   35.5605739   53.4862944 
   35.5606893   53.4863371 
   35.5607951   53.4863678 
   35.560947    53.4864    
   35.5611559   53.4864444 
   35.5614176   53.4865008 
   35.5616347   53.4865468 
   35.5618327   53.486596  
   35.5620308   53.4866573 
   35.5622221   53.4867324 
   35.5625463   53.4868502 
   35.5627838   53.4869382 
   35.5629276   53.4869996 
   35.563047    53.4870561 
   35.5631828   53.4871312 
   35.5632656   53.4871878 
   35.5633552   53.4872613 
   35.563438    53.4873187 
   35.5634814   53.4873445 
   35.5635235   53.4873631 
   35.5635751   53.4873736 
   35.5636388   53.4873751 
   35.563693    53.4873735 
   35.5637242   53.4873678 
   35.5637675   53.4873443 
   35.5638041   53.4873095 
   35.5638622   53.4872213 
   35.5639041   53.4871558 
   35.5639447   53.4871113 
   35.5639866   53.4870539 
   35.5640245   53.487011  
   35.5640908   53.4869454 
   35.5641761   53.486871  
   35.5642533   53.486824  
   35.5643251   53.4867852 
   35.5644159   53.4867487 
   35.5645337   53.4867066 
   35.5647194   53.486662  
   35.564817    53.4866344 
   35.5649715   53.4866052 
   35.5650894   53.4865776 
   35.5652208   53.4865557 
   35.5652859   53.4865468 
   35.5653889   53.4865362 
   35.5655408   53.4865401 
   35.5656927   53.4865635 
   35.5658459   53.4865892 
   35.5659395   53.4866126 
   35.5660548   53.4866368 
   35.5661837   53.4866691 
   35.5663099   53.4866997 
   35.5664835   53.486753  
   35.5666504   53.4868111 
   35.566782    53.4868635 
   35.5669109   53.4869184 
   35.5670371   53.4869782 
   35.5670887   53.4870105 
   35.5671376   53.4870549 
   35.5671757   53.4871075 
   35.5671893   53.4871414 
   35.5671894   53.4871916 
   35.5671692   53.4872563 
   35.5671557   53.4872943 
   35.5671219   53.4873469 
   35.5670989   53.4873776 
   35.567061    53.4874003 
   35.5669689   53.4874343 
   35.5668035   53.4874773 
   35.5666206   53.48753   
   35.5665379   53.4875527 
   35.5664404   53.4875899 
   35.5663591   53.4876345 
   35.5663171   53.487662  
   35.566305    53.4876855 
   35.5663023   53.4877049 
   35.5663078   53.4877194 
   35.5663349   53.4877348 
   35.5664801   53.4877864 
   35.5666389   53.4878494 
   35.5667325   53.4878768 
   35.5668125   53.487914  
   35.5668478   53.4879261 
   35.5668709   53.4879487 
   35.5669307   53.4880133 
   35.5669647   53.4880578 
   35.567       53.4881208 
   35.5670286   53.488179  
   35.5670721   53.4882389 
   35.5671088   53.488276  
   35.5671428   53.488297  
   35.5672256   53.4883423 
   35.5672948   53.4883713 
   35.5673436   53.4883972 
   35.5674128   53.4884278 
   35.5675106   53.4884949 
   35.5676301   53.4885902 
   35.5676817   53.4886411 
   35.5677021   53.4886775 
   35.5677117   53.4887252 
   35.5677147   53.4888392 
   35.5677079   53.48887   
   35.5676768   53.4889007 
   35.5676023   53.4889485 
   35.5675021   53.4889882 
   35.5673909   53.4890085 
   35.5672079   53.4890102 
   35.5671509   53.4890006 
   35.5670695   53.4889764 
   35.5669243   53.4889223 
   35.5667886   53.4888641 
   35.5666312   53.4887996 
   35.5665254   53.48876   
   35.5664331   53.4887382 
   35.5663477   53.4887278 
   35.5662582   53.4887311 
   35.5661931   53.4887384 
   35.5661498   53.4887514 
   35.5660929   53.4887821 
   35.5660563   53.488808  
   35.5660266   53.4888558 
   35.5660105   53.4889213 
   35.5660134   53.4890426 
   35.5660243   53.4890935 
   35.5660448   53.4891412 
   35.5660733   53.4891703 
   35.5661398   53.4892196 
   35.5661765   53.4892422 
   35.5662688   53.4892826 
   35.5663339   53.4893084 
   35.5664519   53.4893439 
   35.566532    53.489373  
   35.5666989   53.4894189 
   35.5668386   53.4894738 
   35.5670055   53.4895425 
   35.5671318   53.4895982 
   35.5672349   53.4896442 
   35.5673177   53.4896684 
   35.567395    53.4896918 
   35.5675781   53.4897345 
   35.5676812   53.4897555 
   35.5679023   53.4897893 
   35.5681424   53.4898279 
   35.5684788   53.4898794 
   35.5687988   53.4899221 
   35.5689765   53.4899502 
   35.5691949   53.4899816 
   35.5694187   53.4900203 
   35.5695842   53.4900444 
   35.5696506   53.4900597 
   35.5697999   53.4900847 
   35.5700345   53.4901104 
   35.5703776   53.4901344 
   35.5708617   53.4901494 
   35.5710678   53.4901477 
   35.5712603   53.4901354 
   35.5715694   53.490115  
   35.5717416   53.4900889 
   35.57185     53.4900767 
   35.5719327   53.4900662 
   35.5720506   53.4900386 
   35.5721767   53.4900118 
   35.5722959   53.4899875 
   35.5723623   53.4899656 
   35.5724165   53.4899566 
   35.5725602   53.4899226 
   35.5726266   53.4898999 
   35.5727459   53.4898707 
   35.5729004   53.4898301 
   35.5730589   53.4897952 
   35.5731674   53.4897822 
   35.5733097   53.4897805 
   35.5733884   53.4897901 
   35.573543    53.4898207 
   35.5737113   53.4898708 
   35.5738605   53.4899184 
   35.5739243   53.4899474 
   35.5741727   53.4900394 
   35.574307    53.4900951 
   35.5743884   53.4901242 
   35.5744861   53.49015   
   35.5746123   53.490179  
   35.5746774   53.4901911 
   35.5748089   53.4902007 
   35.5751331   53.490219  
   35.5753216   53.4902326 
   35.5753677   53.4902399 
   35.5753962   53.490252  
   35.5754531   53.4902584 
   35.5754911   53.4902656 
   35.5755915   53.4902947 
   35.5757326   53.4903423 
   35.5758439   53.4903931 
   35.5759295   53.4904529 
   35.5759865   53.4904973 
   35.5760124   53.4905313 
   35.5760424   53.4905984 
   35.5760454   53.4907642 
   35.5760104   53.4908871 
   35.5759768   53.4909874 
   35.5758944   53.4911411 
   35.5757998   53.4913021 
   35.5757026   53.4914744 
   35.5756053   53.4916354 
   35.5755351   53.491794  
   35.5754865   53.4918967 
   35.5754717   53.4919323 
   35.5754637   53.4920132 
   35.5754692   53.4920536 
   35.5754843   53.4921053 
   35.5755265   53.4921821 
   35.5755496   53.4922314 
   35.5755647   53.4923115 
   35.5755594   53.4923883 
   35.5755555   53.4924352 
   35.5755447   53.4924506 
   35.5755244   53.4924692 
   35.5755014   53.4925169 
   35.5754676   53.4925477 
   35.5754419   53.4925679 
   35.5753958   53.4925874 
   35.5753308   53.4926109 
   35.5752792   53.4926182 
   35.5751382   53.4926159 
   35.5750378   53.4926022 
   35.5749171   53.4925788 
   35.5748289   53.4925619 
   35.5747285   53.4925256 
   35.5745969   53.4924812 
   35.5744503   53.4924353 
   35.574316    53.4923982 
   35.5741356   53.4923538 
   35.574023    53.4923305 
   35.5739063   53.4923152 
   35.5738453   53.4923031 
   35.5736649   53.4922992 
   35.5736053   53.4923049 
   35.5735483   53.4923171 
   35.5734697   53.4923462 
   35.5734386   53.4923713 
   35.5734075   53.4924094 
   35.5733899   53.4924385 
   35.57339     53.4924636 
   35.5733995   53.492491  
   35.5734281   53.492525  
   35.5734715   53.4925573 
   35.5735734   53.4926106 
   35.5736263   53.4926348 
   35.573781    53.4926703 
   35.5739031   53.4926993 
   35.5741323   53.4927452 
   35.5742626   53.4927742 
   35.5744511   53.4928032 
   35.5746058   53.4928265 
   35.5748458   53.4928433 
   35.5750763   53.4928448 
   35.575216    53.4928317 
   35.5754682   53.4928226 
   35.5756797   53.4928103 
   35.5760553   53.492802  
   35.5764648   53.4927944 
   35.5767048   53.4927845 
   35.5772526   53.4927768 
   35.5776282   53.49277   
   35.5778086   53.4927674 
   35.5779035   53.4927593 
   35.5780201   53.4927487 
   35.5781665   53.4927453 
   35.5782899   53.4927485 
   35.5783903   53.4927573 
   35.5785558   53.4927846 
   35.5786195   53.4928016 
   35.5787648   53.492854  
   35.5789113   53.492917  
   35.5790267   53.4929751 
   35.5791639   53.4930591 
   35.5792522   53.493131  
   35.5793215   53.493207  
   35.5793556   53.4932724 
   35.5793734   53.4933492 
   35.5793682   53.4934802 
   35.5793509   53.4935935 
   35.579328    53.4936962 
   35.5792943   53.4937714 
   35.5792376   53.4938774 
   35.5791524   53.4939972 
   35.579051    53.4941339 
   35.5789848   53.4942132 
   35.5788955   53.494333  
   35.5788265   53.494409  
   35.5787738   53.4944721 
   35.5787034   53.4945353 
   35.5786466   53.4945766 
   35.5786263   53.4945984 
   35.5785856   53.4946251 
   35.5785315   53.494664  
   35.5784882   53.4946947 
   35.5784421   53.4947142 
   35.5783825   53.4947466 
   35.5783215   53.4947677 
   35.5782402   53.494792  
   35.5781846   53.4948058 
   35.578125    53.4948155 
   35.5780341   53.4948213 
   35.5779609   53.4948278 
   35.5778416   53.4948287 
   35.5777317   53.4948207 
   35.5775948   53.4948087 
   35.5775079   53.494799  
   35.5773764   53.4947797 
   35.5772624   53.4947653 
   35.5771458   53.4947548 
   35.5770942   53.4947508 
   35.5768908   53.4947502 
   35.5767864   53.4947608 
   35.5766983   53.4947714 
   35.5766129   53.4947949 
   35.5765235   53.4948144 
   35.5764001   53.4948411 
   35.5763405   53.4948565 
   35.5762659   53.4948809 
   35.5761806   53.4949036 
   35.5761196   53.4949376 
   35.5760708   53.4949546 
   35.5760316   53.4949821 
   35.5759571   53.4950347 
   35.5759138   53.4950825 
   35.5758854   53.495127  
   35.5758773   53.4951601 
   35.575883    53.4952709 
   35.5759007   53.4953202 
   35.5759266   53.4953736 
   35.5759661   53.4954245 
   35.5760177   53.4954754 
   35.5760585   53.4955093 
   35.5761277   53.4955441 
   35.5761983   53.4955731 
   35.5762498   53.4955868 
   35.576334    53.4955981 
   35.5763963   53.4956037 
   35.5765997   53.4955971 
   35.5767367   53.4955865 
   35.5768574   53.4955702 
   35.5769671   53.4955483 
   35.5771759   53.4955125 
   35.5772478   53.4954971 
   35.5774823   53.4954508 
   35.5775826   53.4954281 
   35.5777059   53.4954062 
   35.5778537   53.4953875 
   35.5779201   53.4953834 
   35.5782293   53.4953815 
   35.5784273   53.4953919 
   35.5784789   53.4954015 
   35.578605    53.4954168 
   35.5787081   53.4954329 
   35.5788112   53.4954514 
   35.5789021   53.4954634 
   35.5790717   53.4954803 
   35.5791436   53.4954956 
   35.5792847   53.4955222 
   35.5794095   53.4955496 
   35.5795045   53.495577  
   35.5795615   53.4955972 
   35.5796565   53.4956416 
   35.5797312   53.4956763 
   35.5798032   53.4957199 
   35.5798602   53.4957627 
   35.5799499   53.4958241 
   35.5800205   53.4958855 
   35.5800898   53.4959315 
   35.5801469   53.4959865 
   35.580185    53.4960398 
   35.5801987   53.4961029 
   35.5802029   53.4961587 
   35.5801881   53.4962048 
   35.5801475   53.4962477 
   35.5801218   53.4962841 
   35.5800595   53.4963229 
   35.5799931   53.4963553 
   35.5799416   53.4963691 
   35.5798766   53.4963983 
   35.5797993   53.4964178 
   35.5797329   53.4964429 
   35.5797045   53.4964566 
   35.5796503   53.4964688 
   35.5795663   53.4964996 
   35.5794931   53.4965304 
   35.5794321   53.4965523 
   35.5793806   53.4965693 
   35.5793671   53.4965855 
   35.5793088   53.4966122 
   35.5792723   53.4966381 
   35.5792209   53.4966875 
   35.5791694   53.4967369 
   35.5791411   53.4967838 
   35.5791289   53.4968162 
   35.5791209   53.4968736 
   35.5791319   53.4969197 
   35.5791496   53.4969803 
   35.5791782   53.497011  
   35.5792217   53.4970562 
   35.5792815   53.4970958 
   35.5793425   53.4971265 
   35.5794145   53.4971523 
   35.5794823   53.4971725 
   35.5795651   53.4971797 
   35.5796519   53.4971812 
   35.5798173   53.497169  
   35.5798864   53.4971503 
   35.5799976   53.4971227 
   35.5801182   53.4970782 
   35.5801927   53.4970458 
   35.5803159   53.4969826 
   35.5803823   53.4969445 
   35.5804907   53.4968983 
   35.5806018   53.496857  
   35.5807075   53.4968149 
   35.5807915   53.4967784 
   35.5809026   53.4967185 
   35.5809825   53.4966691 
   35.5810285   53.4966294 
   35.5810826   53.496572  
   35.5811706   53.4964935 
   35.5812193   53.4964587 
   35.5813006   53.496423  
   35.5813683   53.4963922 
   35.5814171   53.4963712 
   35.5814713   53.4963614 
   35.5815256   53.4963581 
   35.5816897   53.496358  
   35.58179     53.4963709 
   35.5818321   53.4963829 
   35.5819393   53.4964241 
   35.5819991   53.4964499 
   35.5821131   53.496504  
   35.5821457   53.4965315 
   35.5822258   53.4965694 
   35.5823345   53.4966356 
   35.582363    53.4966575 
   35.5823889   53.4967051 
   35.5824174   53.4967342 
   35.5824949   53.4968304 
   35.5825955   53.4969136 
   35.5827      53.4969685 
   35.5827652   53.4969959 
   35.5828222   53.4970113 
   35.5828887   53.4970193 
   35.5829809   53.4970176 
   35.5830487   53.4970006 
   35.5830812   53.4969803 
   35.5831028   53.4969577 
   35.5831257   53.4969018 
   35.583135    53.4968161 
   35.5831539   53.4967805 
   35.5831823   53.4967376 
   35.5832378   53.4966883 
   35.5832866   53.4966656 
   35.5833747   53.4966453 
   35.5834546   53.4966347 
   35.5835387   53.4966314 
   35.5838452   53.4966295 
   35.5839212   53.4966359 
   35.5840107   53.4966545 
   35.5840745   53.4966665 
   35.584145    53.4966867 
   35.5842061   53.4967174 
   35.584274    53.4967521 
   35.584323    53.4968078 
   35.5843665   53.4968579 
   35.5843978   53.4969105 
   35.5844074   53.4969517 
   35.5844131   53.4970779 
   35.5843983   53.4971207 
   35.5843754   53.497179  
   35.5843592   53.4972049 
   35.5843295   53.4972623 
   35.5842849   53.4973408 
   35.5842417   53.4974201 
   35.5842025   53.4974799 
   35.5841795   53.4974937 
   35.5841389   53.4975107 
   35.5840535   53.4975464 
   35.5839735   53.4975634 
   35.5839044   53.4975756 
   35.5838732   53.4975845 
   35.5838068   53.4976129 
   35.5837689   53.4976339 
   35.5837405   53.4976647 
   35.5837121   53.4977108 
   35.5836893   53.4977909 
   35.583661    53.4978783 
   35.5836353   53.497926  
   35.5836123   53.4979414 
   35.5835351   53.4979754 
   35.5834578   53.4979997 
   35.5833928   53.4980119 
   35.5833033   53.4980241 
   35.5832436   53.4980242 
   35.5831595   53.4980153 
   35.5830347   53.4980017 
   35.5829628   53.4979832 
   35.5828624   53.4979606 
   35.5828013   53.4979388 
   35.582633    53.4978654 
   35.5825121   53.4978072 
   35.5824117   53.4977564 
   35.5823193   53.4976893 
   35.5821794   53.4975956 
   35.5820136   53.497468  
   35.581852    53.4973671 
   35.5817121   53.4972782 
   35.5816687   53.497258  
   35.5815452   53.4972104 
   35.581457    53.4971895 
   35.5814081   53.4971863 
   35.5812332   53.4971864 
   35.5811613   53.497197  
   35.5810448   53.4972319 
   35.5809296   53.4972684 
   35.5807968   53.497317  
   35.5806925   53.4973575 
   35.5806234   53.4973956 
   35.5805665   53.4974466 
   35.5805381   53.4974813 
   35.5805057   53.4975412 
   35.5804828   53.4975833 
   35.580472    53.4976229 
   35.5804641   53.4977151 
   35.5804752   53.4978178 
   35.5804956   53.4978671 
   35.5805351   53.4979302 
   35.5805868   53.4979972 
   35.5806248   53.4980344 
   35.5806792   53.4980804 
   35.5807566   53.4981313 
   35.580849    53.4981879 
   35.5809291   53.4982201 
   35.5810173   53.4982548 
   35.5810919   53.4982815 
   35.5812385   53.4983194 
   35.5814135   53.498358  
   35.5816401   53.4984039 
   35.5818776   53.4984466 
   35.5820933   53.4984812 
   35.5822873   53.4985166 
   35.5825288   53.4985504 
   35.582697    53.4985777 
   35.5828408   53.4985946 
   35.5829643   53.4986115 
   35.5830185   53.4986219 
   35.5831271   53.4986469 
   35.5832193   53.498659  
   35.5832994   53.4986646 
   35.5835882   53.4986619 
   35.5837496   53.4986432 
   35.5837889   53.498631  
   35.5839177   53.4985986 
   35.5840206   53.4985564 
   35.5841236   53.4985046 
   35.584213    53.4984584 
   35.5843105   53.4983904 
   35.5844391   53.4982981 
   35.5845813   53.4982034 
   35.5847194   53.4981167 
   35.5847844   53.4980843 
   35.5848969   53.4980373 
   35.5849741   53.4980154 
   35.5850446   53.4980032 
   35.5850934   53.4979967 
   35.5851599   53.4979967 
   35.5851911   53.4980031 
   35.5852494   53.4980152 
   35.5853119   53.4980443 
   35.5853974   53.4980838 
   35.5854612   53.4981193 
   35.5855034   53.4981533 
   35.5855387   53.498184  
   35.5855537   53.4982114 
   35.5855565   53.4982656 
   35.5855282   53.4983206 
   35.5854904   53.498391  
   35.5854498   53.498442  
   35.5853673   53.4985173 
   35.585328    53.4985432 
   35.5852535   53.4986063 
   35.5851412   53.4987067 
   35.5850505   53.4987804 
   35.584995    53.4988281 
   35.5849504   53.4988645 
   35.5848705   53.4989228 
   35.5848177   53.4989706 
   35.5847757   53.4989997 
   35.5847406   53.499037  
   35.5846987   53.4991025 
   35.5846636   53.4991624 
   35.5846474   53.4992117 
   35.584645    53.499333  
   35.5846614   53.4993912 
   35.5846737   53.4994333 
   35.5847173   53.4995189 
   35.5847649   53.4995941 
   35.5848398   53.4997089 
   35.5848725   53.499763  
   35.584878    53.499814  
   35.5848783   53.4999102 
   35.5848499   53.4999579 
   35.5847985   53.4999992 
   35.5847091   53.5000438 
   35.5846169   53.5000673 
   35.5845288   53.5000827 
   35.5844081   53.5000917 
   35.5842847   53.5000951 
   35.5842019   53.5000846 
   35.5841327   53.5000725 
   35.5840499   53.5000524 
   35.5839536   53.5000218 
   35.5838273   53.4999847 
   35.5837486   53.499954  
   35.5837038   53.4999451 
   35.5835084   53.4998903 
   35.5832763   53.499804  
   35.5831269   53.4997386 
   35.5829382   53.4996555 
   35.5827685   53.499582  
   35.5826287   53.499528  
   35.5825283   53.4994868 
   35.5824278   53.4994368 
   35.5823436   53.4993915 
   35.5821861   53.4993245 
   35.5821114   53.4992923 
   35.5820395   53.4992616 
   35.5819485   53.4992188 
   35.5818507   53.4991647 
   35.581776    53.4991163 
   35.5816877   53.4990468 
   35.5815804   53.4989628 
   35.5814867   53.4988998 
   35.5814066   53.4988643 
   35.5812966   53.4988296 
   35.5811799   53.4988095 
   35.5811216   53.498799  
   35.5810538   53.498799  
   35.5809738   53.4988031 
   35.5808952   53.4988162 
   35.5808301   53.4988388 
   35.5807705   53.498868  
   35.580715    53.4989141 
   35.5806785   53.4989587 
   35.5806637   53.4990064 
   35.5806612   53.4991188 
   35.5806709   53.4991891 
   35.5806846   53.4992659 
   35.5806998   53.4993767 
   35.5807029   53.499536  
   35.5806936   53.4996452 
   35.5806939   53.499777  
   35.5807144   53.4998465 
   35.5807472   53.4999581 
   35.5807868   53.5000446 
   35.5808535   53.5001658 
   35.5809025   53.5002653 
   35.5809651   53.5003525 
   35.5809992   53.5004277 
   35.5810197   53.5004989 
   35.5810431   53.5006396 
   35.5810639   53.5008546 
   35.5810643   53.5010099 
   35.5810563   53.5010819 
   35.5810239   53.5011401 
   35.5809752   53.5012032 
   35.5809022   53.5012914 
   35.5808441   53.5013756 
   35.580798    53.5014185 
   35.5807791   53.5014597 
   35.5807765   53.5015091 
   35.5807847   53.5015349 
   35.5808309   53.5015673 
   35.5808934   53.5015979 
   35.5809544   53.50161   
   35.5810453   53.5016116 
   35.5811823   53.5016025 
   35.5812827   53.5016057 
   35.5814129   53.5016177 
   35.5815065   53.5016379 
   35.5816192   53.5016725 
   35.5816789   53.5016992 
   35.5817251   53.5017355 
   35.5817415   53.5017646 
   35.5817429   53.501814  
   35.5817336   53.5018706 
   35.5817052   53.5019078 
   35.581674    53.5019321 
   35.5816049   53.5019588 
   35.5815453   53.501975  
   35.58143     53.5019921 
   35.5812551   53.5020125 
   35.5810029   53.5020466 
   35.5808768   53.5020694 
   35.5807969   53.5021002 
   35.5807197   53.5021431 
   35.5806682   53.5021852 
   35.5806344   53.502237  
   35.5806169   53.5022799 
   35.5806116   53.5023308 
   35.5806225   53.5023753 
   35.5806525   53.5024197 
   35.5806865   53.5024504 
   35.5807761   53.5025134 
   35.5808441   53.5025643 
   35.5809786   53.5026685 
   35.5810914   53.5027622 
   35.5811865   53.5028341 
   35.581249    53.5028834 
   35.5813128   53.5029303 
   35.5813604   53.5029909 
   35.5814094   53.5030458 
   35.5814271   53.5030814 
   35.58143     53.5031655 
   35.5814125   53.5032181 
   35.581376    53.5032666 
   35.5813219   53.5033265 
   35.5812704   53.5033637 
   35.5812244   53.5033824 
   35.5811579   53.5033913 
   35.5810698   53.5033962 
   35.5809355   53.5033883 
   35.5808201   53.5033673 
   35.5806885   53.5033262 
   35.5805988   53.503281  
   35.5805472   53.5032535 
   35.5804956   53.5032075 
   35.5804643   53.5031816 
   35.5803963   53.5031065 
   35.5802835   53.5029869 
   35.5801979   53.502919  
   35.5801408   53.5028827 
   35.5800743   53.5028552 
   35.5799793   53.502827  
   35.5798015   53.5027892 
   35.5797242   53.5027787 
   35.5795614   53.5027756 
   35.5794068   53.5027862 
   35.579274    53.5028122 
   35.5791547   53.502843  
   35.5790395   53.5028957 
   35.5789542   53.5029556 
   35.5788933   53.5030187 
   35.5788514   53.5030665 
   35.5788176   53.5031199 
   35.5787744   53.5032186 
   35.5787259   53.5033763 
   35.5787003   53.5034499 
   35.5786883   53.5035421 
   35.5786981   53.5036513 
   35.5787349   53.5037652 
   35.5787868   53.5038905 
   35.5788359   53.5040304 
   35.5788645   53.5040951 
   35.5789312   53.5041961 
   35.5790347   53.5043594 
   35.5791177   53.504483  
   35.5792851   53.5047473 
   35.5793709   53.504875  
   35.5794198   53.5049284 
   35.5795231   53.5050067 
   35.5795978   53.505073  
   35.5796671   53.5051368 
   35.5797093   53.5051926 
   35.5797638   53.5052847 
   35.5798196   53.5053736 
   35.5798849   53.5054488 
   35.5799978   53.5055926 
   35.5800971   53.5057405 
   35.5801666   53.5058488 
   35.5801898   53.5059183 
   35.5801954   53.506021  
   35.5801739   53.5060825 
   35.5801225   53.5061699 
   35.5800996   53.5062071 
   35.5800332   53.5062605 
   35.5799736   53.5062994 
   35.5797961   53.5063869 
   35.5795522   53.5064946 
   35.5794086   53.5065821 
   35.5792542   53.506676  
   35.579158    53.5067448 
   35.579112    53.5067885 
   35.5790484   53.5068541 
   35.578932    53.5069738 
   35.5787886   53.5071324 
   35.578637    53.5073056 
   35.5785681   53.5074043 
   35.578537    53.5074569 
   35.5785236   53.5075119 
   35.5785265   53.5075904 
   35.5785456   53.5076405 
   35.5785755   53.507676  
   35.5786353   53.5077188 
   35.5786977   53.5077519 
   35.5787764   53.5077656 
   35.5788904   53.507789  
   35.5790248   53.5077994 
   35.5791251   53.5077945 
   35.5792485   53.5077766 
   35.5793285   53.5077563 
   35.5793949   53.507736  
   35.579521    53.5076898 
   35.5795671   53.5076655 
   35.5796212   53.5076315 
   35.5797039   53.5075749 
   35.5798014   53.5075052 
   35.5798948   53.5074332 
   35.5799896   53.5073717 
   35.5800465   53.5073344 
   35.5801251   53.5073052 
   35.5802417   53.5072744 
   35.5803109   53.5072622 
   35.5803963   53.5072516 
   35.5804682   53.5072484 
   35.5807232   53.5072449 
   35.580787    53.5072529 
   35.5808602   53.5072666 
   35.5809499   53.5073005 
   35.5810042   53.5073264 
   35.5810966   53.5073877 
   35.5811958   53.5074693 
   35.5812746   53.5075461 
   35.5812978   53.5075776 
   35.5813141   53.5076196 
   35.581317    53.5076811 
   35.5813035   53.5077296 
   35.5812887   53.5077499 
   35.5812291   53.5077895 
   35.5811776   53.5078179 
   35.5810719   53.5078641 
   35.5810231   53.5078851 
   35.580977    53.507911  
   35.5809432   53.5079377 
   35.5809202   53.507962  
   35.5809107   53.5079944 
   35.5809136   53.5080437 
   35.5809313   53.5080817 
   35.5809653   53.5081205 
   35.5810142   53.5081544 
   35.5811038   53.508194  
   35.5811812   53.5082246 
   35.5812328   53.5082521 
   35.581351    53.5083272 
   35.5814679   53.5084193 
   35.581563    53.5084912 
   35.5816255   53.5085388 
   35.5816745   53.508597  
   35.581729    53.5086819 
   35.5817753   53.508774  
   35.5818216   53.5088508 
   35.5818556   53.508913  
   35.5818884   53.5090133 
   35.5819021   53.5090764 
   35.5818996   53.5091346 
   35.5818971   53.509247  
   35.5818823   53.5092834 
   35.5818458   53.5093263 
   35.5817686   53.5094032 
   35.5817091   53.5094647 
   35.5816305   53.5095327 
   35.581571    53.5095812 
   35.581525    53.5096354 
   35.5814736   53.5097212 
   35.5814628   53.5097447 
   35.581463    53.5098239 
   35.5814766   53.5098563 
   35.5814943   53.5098716 
   35.5815405   53.5099039 
   35.5815948   53.5099346 
   35.5816722   53.5099685 
   35.5818094   53.5100209 
   35.5819071   53.5100556 
   35.5820171   53.5100911 
   35.582142    53.5101387 
   35.5822466   53.5101831 
   35.5823688   53.5102412 
   35.5825046   53.5103042 
   35.5825793   53.5103365 
   35.5826282   53.5103712 
   35.5826826   53.5104084 
   35.5827505   53.5104633 
   35.5827831   53.5104884 
   35.582828    53.5105417 
   35.5829232   53.5106265 
   35.582987    53.5106677 
   35.5830726   53.5107226 
   35.5831419   53.5107517 
   35.5832274   53.5107839 
   35.5833021   53.510813  
   35.5833822   53.5108331 
   35.5834718   53.5108549 
   35.5835682   53.5108839 
   35.5836483   53.5109017 
   35.5837148   53.510917  
   35.5838098   53.5109525 
   35.5838872   53.5109742 
   35.5839782   53.5110186 
   35.5840814   53.5110598 
   35.5842077   53.5111244 
   35.5843028   53.5111858 
   35.5843803   53.5112544 
   35.5844524   53.5113247 
   35.5845217   53.5113877 
   35.5845734   53.5114556 
   35.5846129   53.511517  
   35.5846225   53.5115397 
   35.58462     53.5116472 
   35.5846107   53.5117224 
   35.5845769   53.5117807 
   35.5845418   53.5118398 
   35.584466    53.5119086 
   35.5843969   53.5119652 
   35.5843306   53.5120195 
   35.5842683   53.5120608 
   35.5841843   53.5121158 
   35.5841585   53.5121377 
   35.5841247   53.5121749 
   35.5840787   53.5122332 
   35.5840585   53.5122696 
   35.5840396   53.5123343 
   35.5840344   53.512437  
   35.5840401   53.5125389 
   35.5840593   53.5126027 
   35.5840866   53.5127103 
   35.5840978   53.5128178 
   35.5841035   53.5129625 
   35.5841009   53.5129989 
   35.5840833   53.5130256 
   35.5840441   53.5130669 
   35.583975    53.51313   
   35.5839155   53.5132021 
   35.5838979   53.5132296 
   35.5838845   53.5132724 
   35.5838846   53.5133185 
   35.5838901   53.5133606 
   35.5858562   53.5123237 
   35.5924521   53.5148362 
   35.5935525   53.515887  
   35.5948451   53.5175964 
   35.5961058   53.5188698 
   35.5980193   53.519562  
   35.6018481   53.5195491 
   35.6045805   53.5200069 
   35.6065341   53.5201917 
   35.608308    53.5205847 
   35.6084782   53.5206398 
   35.6087894   53.5213182 
   35.6100609   53.5226098 
   35.6109834   53.5226088 
   35.6122095   53.5224632 
   35.6128613   53.5232578 
   35.6232682   53.5238573 
   35.622995    53.5254008 
   35.6244337   53.5256325 
   35.6246293   53.5270971 
   35.6254536   53.5285641 
   35.6186079   53.5282312 
   35.6187241   53.5275063 
   35.6189587   53.5267383 
   35.6128601   53.526269  
   35.6116128   53.5262197 
   35.612849    53.5293606 
   35.6129572   53.5293943 
   35.613009    53.5294725 
   35.612905    53.5300162 
   35.6124716   53.5307395 
   35.6126708   53.5309589 
   35.6127777   53.5313503 
   35.6129691   53.5316065 
   35.6139633   53.5322304 
   35.6140411   53.5324207 
   35.6139208   53.5326972 
   35.6136559   53.532848  
   35.6132515   53.5329222 
   35.6121332   53.5330232 
   35.611794    53.5333568 
   35.6117767   53.5335656 
   35.6124011   53.534692  
   35.6126417   53.5350004 
   35.6155709   53.5345011 
   35.6234077   53.5349052 
   35.6237613   53.5358538 
   35.6236642   53.5360872 
   35.6234433   53.536327  
   35.6260326   53.5383094 
   35.6193235   53.5377077 
   35.6139816   53.5372941 
   35.6135186   53.537668  
   35.6126945   53.5379054 
   35.6123646   53.537958  
   35.6121382   53.5380534 
   35.612016    53.5385849 
   35.6139441   53.5385566 
   35.6153259   53.5386012 
   35.6171873   53.5387594 
   35.618894    53.5387974 
   35.6187757   53.5388896 
   35.6265009   53.5395757 
   35.6279542   53.5409099 
   35.6285998   53.5412592 
   35.6296666   53.5415863 
   35.6329801   53.541866  
   35.6347227   53.54196   
   35.643198    53.5425968 
   35.6446227   53.5427988 
   35.6522727   53.5433131 
   35.6569296   53.543726  
   35.6587077   53.5439776 
   35.6591779   53.543871  
   35.6609912   53.5438898 
   35.6624551   53.5440342 
   35.6640573   53.5440857 
   35.6695793   53.5446046 
   35.6731011   53.5448039 
   35.6759156   53.5450277 
   35.6851231   53.5454137 
   35.7001184   53.5301252 
   35.6994552   53.5294264 
   35.6927039   53.5240752 
   35.6905012   53.5248693 
   35.6840636   53.5193354 
   35.6861242   53.5192678 
   35.6877158   53.5190143 
   35.6984168   53.5208478 
   35.6989853   53.521059  
   35.69971     53.5221151 
   35.7039308   53.5262886 
   35.7112243   53.5187806 
   35.7126253   53.5191454 
   35.7138585   53.5194117 
   35.7144647   53.5202623 
   35.7150663   53.5202808 
   35.7155262   53.5205684 
   35.7174394   53.5213752 
   35.7183883   53.5214792 
   35.7198416   53.5219931 
   35.7228203   53.5221624 
   35.7238666   53.522402  
   35.7298213   53.5206934 
   35.7333373   53.5204418 
   35.7382541   53.52125   
   35.7395996   53.521098  
   35.7433498   53.5204411 
   35.7487018   53.5208912 
   35.7498273   53.5217262 
   35.7512823   53.5219637 
   35.7525029   53.52235   
   35.7534059   53.5223281 
   35.7543651   53.5218893 
   35.7550126   53.5220011 
   35.7555229   53.5216388 
   35.7576508   53.5205493 
   35.7582811   53.5203296 
   35.7582933   53.5198767 
   35.758831    53.5193247 
   35.7595817   53.5190876 
   35.7630386   53.5156274 
   35.7720571   53.5059516 
   35.7787322   53.5098285 
   35.780695    53.5111606 
   35.7830008   53.512385  
   35.7837963   53.5144567 
   35.7830238   53.5158594 
   35.7838077   53.517569  
   35.7882133   53.5190971 
   35.7900067   53.5204636 
   35.7908382   53.522024  
   35.7927859   53.522816  
   35.7943409   53.5241219 
   35.7949638   53.5247601 
   35.7949335   53.5253649 
   35.7942531   53.5281623 
   35.795854    53.532625  
   35.7949255   53.5376294 
   35.7946682   53.53834   
   35.7943946   53.5399023 
   35.7985174   53.5417892 
   35.8005475   53.5435646 
   35.802954    53.5454316 
   35.8034282   53.5455004 
   35.8043082   53.5461593 
   35.8046471   53.5466949 
   35.8065791   53.5485998 
   35.8091748   53.550662  
   35.8092553   53.5515191 
   35.8095768   53.5533052 
   35.8101192   53.553788  
   35.8105241   53.5546646 
   35.8113965   53.5558449 
   35.8128408   53.5574313 
   35.8121244   53.5582608 
   35.8110588   53.5609637 
   35.8091284   53.5642781 
   35.8107144   53.565033  
   35.813108    53.5657929 
   35.8151101   53.5678278 
   35.815233    53.5694475 
   35.8158511   53.5706965 
   35.8180345   53.5715512 
   35.8198754   53.5720374 
   35.8213192   53.5726992 
   35.8232093   53.5736579 
   35.8233169   53.575027  
   35.8236039   53.5763703 
   35.8242212   53.577471  
   35.8245077   53.5785529 
   35.8243889   53.579927  
   35.8247055   53.5821275 
   35.8245523   53.5829761 
   35.8239743   53.5831952 
   35.8234482   53.5834949 
   35.8229506   53.5856582 
   35.8227337   53.585905  
   35.8225181   53.5876468 
   35.8272941   53.5892811 
   35.8286576   53.5903706 
   35.8326808   53.5911155 
   35.8336129   53.590605  
   35.8348358   53.5898387 
   35.8354375   53.5895484 
   35.8368606   53.5894338 
   35.83907     53.589447  
   35.840788    53.589378  
   35.8435917   53.5891226 
   35.8458992   53.5893116 
   35.8498328   53.5901451 
   35.8528455   53.5906542 
   35.8535327   53.5908398 
   35.8543348   53.5912013 
   35.8566734   53.5919766 
   35.8597431   53.5929275 
   35.8616616   53.5933222 
   35.8626176   53.5938295 
   35.8654574   53.5955464 
   35.8704831   53.5961233 
   35.8740935   53.5961979 
   35.8753312   53.5962657 
   35.8744256   53.5974807 
   35.8741406   53.5981887 
   35.8748772   53.5986284 
   35.8758051   53.6003073 
   35.8761814   53.6008112 
   35.8761459   53.6023611 
   35.878094    53.6066801 
   35.878003    53.6072896 
   35.8799645   53.6091134 
   35.880319    53.6095408 
   35.8817883   53.6097812 
   35.8841605   53.6107508 
   35.8848977   53.6109028 
   35.8852743   53.6112629 
   35.8853413   53.6114747 
   35.8847078   53.6115927 
   35.8860261   53.611989  
   35.8878634   53.6132532 
   35.8879905   53.6137425 
   35.8879443   53.6139709 
   35.8882821   53.6143767 
   35.8888779   53.6147342 
   35.8890431   53.6150984 
   35.8906384   53.6160032 
   35.8910947   53.617414  
   35.8911833   53.6186078 
   35.8911134   53.6192002 
   35.8911426   53.6197158 
   35.8912472   53.6200614 
   35.891262    53.6206633 
   35.8943077   53.62383   
   35.8957595   53.6254711 
   35.8982263   53.6291886 
   35.898724    53.6293755 
   35.9020995   53.6309227 
   35.9031467   53.6314994 
   35.9050858   53.6317711 
   35.9060492   53.6322663 
   35.9057384   53.6336934 
   35.9069572   53.6352372 
   35.9077235   53.636576  
   35.9081132   53.6383482 
   35.9091121   53.6406502 
   35.9106841   53.640947  
   35.9143885   53.6405852 
   35.9156017   53.6405235 
   35.9190479   53.6401956 
   35.9208361   53.6400783 
   35.9242531   53.6396978 
   35.9264013   53.6395258 
   35.9267966   53.6395922 
   35.9270606   53.6397279 
   35.9270042   53.6400621 
   35.9270707   53.6401742 
   35.9276225   53.6402337 
   35.9278267   53.6403768 
   35.9280047   53.6411282 
   35.9284178   53.6413272 
   35.9284894   53.641407  
   35.9284844   53.6415785 
   35.9283016   53.6417991 
   35.9283062   53.6419786 
   35.9282026   53.6420914 
   35.9280725   53.642138  
   35.9284348   53.6423032 
   35.9285616   53.6424499 
   35.9288483   53.6425434 
   35.9289901   53.6426925 
   35.9298119   53.6430671 
   35.9301825   53.6433706 
   35.9306093   53.6435776 
   35.930619    53.6438533 
   35.9309007   53.6439792 
   35.9309043   53.6440641 
   35.9307319   53.6442297 
   35.9307031   53.644339  
   35.9309412   53.6445904 
   35.9309539   53.6447602 
   35.9306901   53.6449164 
   35.9306489   53.6451317 
   35.9308415   53.6453323 
   35.9308035   53.6454707 
   35.9309705   53.6458138 
   35.9305368   53.6462456 
   35.9305472   53.6464542 
   35.9307206   53.6466379 
   35.9306217   53.6469407 
   35.9302892   53.6472889 
   35.9294215   53.6477247 
   35.9292148   53.6478767 
   35.9286026   53.6484175 
   35.9281952   53.6490183 
   35.9276253   53.6494449 
   35.9274286   53.6497707 
   35.9272725   53.6504586 
   35.9271003   53.6506525 
   35.9266767   53.6508805 
   35.9259751   53.6510748 
   35.9258652   53.6517666 
   35.9255298   53.6522271 
   35.9253042   53.6523937 
   35.9262294   53.6524906 
   35.9276576   53.6522581 
   35.9293964   53.6520599 
   35.9305832   53.6516625 
   35.9317218   53.6514739 
   35.9323417   53.6517805 
   35.9349026   53.6515865 
   35.9361358   53.6515899 
   35.9385301   53.6513374 
   35.9413801   53.6511972 
   35.9423301   53.6510779 
   35.9440779   53.6501906 
   35.945775    53.6497885 
   35.9461283   53.6497293 
   35.9483227   53.649688  
   35.9514905   53.6489812 
   35.9544766   53.6471326 
   35.9541187   53.6467608 
   35.9540158   53.6465359 
   35.9523785   53.6447056 
   35.9503444   53.642552  
   35.9501071   53.6420681 
   35.9495007   53.6403781 
   35.9492756   53.6392373 
   35.9493936   53.6385442 
   35.9498326   53.6382278 
   35.950186    53.6376704 
   35.9502446   53.636841  
   35.9509712   53.6368416 
   35.9532166   53.6366364 
   35.9544009   53.6365938 
   35.9585352   53.6324889 
   35.9593222   53.6314991 
   35.9615488   53.628214  
   35.9606372   53.6251555 
   35.9645573   53.6251565 
   35.9718705   53.6253303 
   35.9811144   53.6253766 
   35.9835534   53.6254336 
   35.9895356   53.626093  
   35.9911887   53.6262728 
   35.9913996   53.6268547 
   35.9915218   53.6270483 
   35.991772    53.6276261 
   35.992062    53.6282727 
   35.99214     53.6284952 
   35.9922433   53.6288055 
   35.9926733   53.6299471 
   35.9930734   53.6307494 
   35.9945437   53.6312026 
   35.9958973   53.6314472 
   35.9959341   53.6314545 
   35.9961062   53.6320935 
   35.9961869   53.632544  
   35.9963255   53.6329752 
   35.9962946   53.6336048 
   35.9962259   53.6349604 
   35.996187    53.6360564 
   35.9961776   53.63671   
   35.9961671   53.6368053 
   35.9962377   53.6373656 
   35.9963116   53.6378798 
   35.996261    53.6388184 
   35.9961059   53.6392407 
   35.9959606   53.639703  
   35.9959338   53.639872  
   35.9954725   53.6402517 
   35.9943103   53.640927  
   35.9939364   53.6409337 
   35.9938173   53.6409923 
   35.9936464   53.6411635 
   35.9934397   53.6412551 
   35.9932836   53.6413012 
   35.9931709   53.6413212 
   35.9930649   53.6412918 
   35.992539    53.64142   
   35.993523    53.6419211 
   35.9941995   53.6425477 
   35.9954394   53.6437507 
   35.9970871   53.6454    
   35.9973449   53.6456263 
   35.9978584   53.6460845 
   35.9986882   53.6469424 
   35.999314    53.6476007 
   35.999508    53.6482959 
   35.9996951   53.6488183 
   36.0001885   53.6503146 
   36.0004497   53.6511129 
   36.0004708   53.6512575 
   36.0006398   53.6517989 
   36.0007288   53.6521225 
   36.0012091   53.6537675 
   36.0012927   53.6539555 
   36.0017968   53.6553385 
   36.0023118   53.6566583 
   36.0027292   53.6577588 
   36.0032683   53.6597629 
   36.0056435   53.6610621 
   36.0071631   53.662289  
   36.0081963   53.6623258 
   36.0114519   53.6633369 
   36.012474    53.6640387 
   36.0135758   53.665225  
   36.0149422   53.6662274 
   36.0183186   53.6680312 
   36.0216855   53.6698314 
   36.0228376   53.6704861 
   36.0251622   53.6716793 
   36.027956    53.6730864 
   36.0285844   53.6733546 
   36.0287887   53.6734195 
   36.0337068   53.6749813 
   36.0354513   53.6754012 
   36.0364536   53.6757159 
   36.0374134   53.6760244 
   36.0397915   53.6768105 
   36.0422312   53.6775665 
   36.0439402   53.6780707 
   36.0462708   53.6787559 
   36.0478958   53.6793214 
   36.0488946   53.6796555 
   36.0496466   53.6797599 
   36.050486    53.6798754 
   36.0511136   53.6799448 
   36.0525755   53.6800332 
   36.0538494   53.680104  
   36.0558598   53.6802621 
   36.058106    53.6804076 
   36.0584064   53.6804463 
   36.0587088   53.6804943 
   36.061022    53.6807302 
   36.0614425   53.680774  
   36.0617021   53.6807992 
   36.0621241   53.680862  
   36.0664201   53.6812088 
   36.0677878   53.6813469 
   36.0701472   53.6815443 
   36.0740036   53.6818608 
   36.0739922   53.6821106 
   36.0739537   53.6825416 
   36.0739242   53.6827955 
   36.0738182   53.6832303 
   36.0738282   53.6833933 
   36.0736971   53.6839774 
   36.0736484   53.6843705 
   36.0735065   53.6850282 
   36.073454    53.6854167 
   36.0733096   53.6861525 
   36.0731675   53.6870675 
   36.0730494   53.6877595 
   36.0729491   53.688332  
   36.0728103   53.6892067 
   36.0726567   53.6899909 
   36.0722739   53.6920854 
   36.0723851   53.6931244 
   36.0726793   53.6932642 
   36.0730819   53.6935551 
   36.0744632   53.6944867 
   36.0759834   53.6955152 
   36.0769682   53.6961788 
   36.0774179   53.6965051 
   36.0779511   53.6969917 
   36.0785597   53.6975974 
   36.0790387   53.6982154 
   36.0791446   53.698213  
   36.0793288   53.698227  
   36.079503    53.6982319 
   36.0797135   53.6982056 
   36.080065    53.6981476 
   36.0803474   53.6981381 
   36.0809282   53.6981386 
   36.0814468   53.6981475 
   36.0819269   53.6981451 
   36.082248    53.698132  
   36.0826003   53.6981302 
   36.0829598   53.6981044 
   36.0833003   53.69809   
   36.0837608   53.6980739 
   36.0845421   53.6980367 
   36.0849174   53.6980245 
   36.0851089   53.6980098 
   36.0856914   53.6980114 
   36.086075    53.698029  
   36.0864041   53.6980377 
   36.0864931   53.6980396 
   36.0866216   53.6983651 
   36.0866527   53.6985016 
   36.0867112   53.6988068 
   36.0867683   53.6990052 
   36.0870868   53.7004463 
   36.0873363   53.7015109 
   36.0876502   53.7028853 
   36.0861614   53.7028584 
   36.0853499   53.7028383 
   36.0845443   53.7028137 
   36.0838805   53.7028158 
   36.0836115   53.7028194 
   36.0820253   53.7027643 
   36.0812274   53.7027349 
   36.0798834   53.7026831 
   36.0795067   53.702198  
   36.0790893   53.7016946 
   36.0787895   53.7012975 
   36.0788065   53.7008621 
   36.0788909   53.7002542 
   36.0789348   53.6996635 
   36.079017    53.6989119 
   36.0790916   53.6982936 
   36.0769349   53.6985272 
   36.0753581   53.6987187 
   36.0746121   53.6987913 
   36.0742569   53.6988551 
   36.0720986   53.6992563 
   36.0707432   53.6995018 
   36.0703076   53.7001862 
   36.0690628   53.7011572 
   36.0683619   53.7016911 
   36.0675152   53.7023362 
   36.06707     53.7037328 
   36.066608    53.7053201 
   36.0658838   53.7055786 
   36.0646128   53.7060602 
   36.0635674   53.7064903 
   36.0627267   53.7068631 
   36.0615389   53.707497  
   36.0613614   53.7076069 
   36.0608565   53.7079171 
   36.0591699   53.7089851 
   36.0588206   53.7092096 
   36.0581255   53.7096333 
   36.0578797   53.7099256 
   36.0576492   53.7102047 
   36.0570591   53.7109334 
   36.0559502   53.712255  
   36.055552    53.7127324 
   36.0552796   53.7131248 
   36.0545167   53.7142678 
   36.053978    53.7150951 
   36.0539329   53.7152039 
   36.053901    53.7152968 
   36.0538363   53.7152853 
   36.0537704   53.7152839 
   36.0537076   53.7152974 
   36.0536588   53.7153298 
   36.0536246   53.7153874 
   36.0535912   53.7154984 
   36.0535628   53.7155341 
   36.0535118   53.7155631 
   36.0534038   53.7155929 
   36.0532832   53.7156228 
   36.0531419   53.7156274 
   36.0529793   53.7156294 
   36.0528216   53.7156387 
   36.0527222   53.7156553 
   36.0525748   53.7157157 
   36.0525007   53.7157454 
   36.0524225   53.7157624 
   36.0523035   53.7157681 
   36.0521234   53.7157701 
   36.0520298   53.7157947 
   36.0519344   53.7158313 
   36.0518456   53.7158461 
   36.051769    53.7158373 
   36.0517058   53.7158117 
   36.0516212   53.7157961 
   36.0515651   53.715794  
   36.0514918   53.7158116 
   36.0513865   53.7158362 
   36.0512717   53.7158603 
   36.0510733   53.7158607 
   36.0509732   53.7158382 
   36.0508692   53.7158082 
   36.0507961   53.715773  
   36.0507288   53.7157296 
   36.0506708   53.7156742 
   36.0506331   53.7156095 
   36.0506073   53.7155464 
   36.0505776   53.7154363 
   36.0505663   53.7153157 
   36.0505251   53.715208  
   36.0504818   53.7151502 
   36.0504123   53.7150925 
   36.0502975   53.7150483 
   36.0502228   53.7150383 
   36.0501007   53.7150326 
   36.0499799   53.7150406 
   36.0499007   53.715064  
   36.0498238   53.7150976 
   36.0497548   53.7151496 
   36.049678    53.7152103 
   36.0496127   53.7152439 
   36.0495356   53.7152696 
   36.0494205   53.7152747 
   36.0492743   53.715273  
   36.0492033   53.7152493 
   36.0491402   53.7152456 
   36.0490402   53.7152208 
   36.0489945   53.7151997 
   36.0489163   53.7151547 
   36.0488741   53.7151027 
   36.0488216   53.7150144 
   36.0488072   53.7149566 
   36.0488103   53.7148951 
   36.0488182   53.7148307 
   36.0488479   53.7146738 
   36.048843    53.7145871 
   36.0488128   53.7145074 
   36.0487712   53.7144415 
   36.0486911   53.7143759 
   36.0485721   53.7143104 
   36.048432    53.7142645 
   36.0482251   53.7142116 
   36.0481525   53.7142038 
   36.0480466   53.7141785 
   36.047848    53.7141794 
   36.0477864   53.7141912 
   36.04771     53.7142134 
   36.0476052   53.7142609 
   36.0475009   53.7142868 
   36.0474285   53.7142957 
   36.0473152   53.7142968 
   36.047187    53.7142687 
   36.0471036   53.7142513 
   36.0470413   53.7142246 
   36.0469516   53.714175  
   36.0468891   53.71413   
   36.046833    53.7140722 
   36.0467916   53.714011  
   36.0467809   53.7139358 
   36.046791    53.7138995 
   36.0468417   53.7138505 
   36.0468739   53.7138004 
   36.0469233   53.7137215 
   36.046972    53.7136707 
   36.0470563   53.7136089 
   36.0471186   53.7135729 
   36.0471977   53.713541  
   36.0472399   53.7135092 
   36.0472858   53.7134676 
   36.0472938   53.7134148 
   36.0472755   53.7133505 
   36.047217    53.7132469 
   36.0471661   53.7132052 
   36.047094    53.7131705 
   36.0469803   53.7131343 
   36.0469042   53.7130979 
   36.0468006   53.713018  
   36.0466473   53.7129256 
   36.04651     53.7128626 
   36.0463813   53.7127937 
   36.0462765   53.7127103 
   36.0462225   53.7126486 
   36.0461918   53.7125918 
   36.0461695   53.7125225 
   36.0461489   53.7124973 
   36.0460678   53.7124385 
   36.0459326   53.7123915 
   36.0458355   53.7123707 
   36.0457213   53.7123707 
   36.0455954   53.7123643 
   36.0454585   53.7123467 
   36.045342    53.7123213 
   36.0452574   53.7122936 
   36.0451435   53.7122447 
   36.0450499   53.7121912 
   36.0449717   53.7121255 
   36.0448805   53.7120415 
   36.044796    53.7119541 
   36.0447077   53.7118614 
   36.0446223   53.711771  
   36.0445097   53.7116809 
   36.0444318   53.7116393 
   36.0443501   53.7116145 
   36.0441827   53.7116175 
   36.0440963   53.7116047 
   36.0440422   53.7116107 
   36.0439783   53.711611  
   36.0439151   53.7115866 
   36.0438273   53.711545  
   36.0437105   53.7114882 
   36.043568    53.7114084 
   36.0434831   53.7113503 
   36.0433813   53.7112726 
   36.0433343   53.7112246 
   36.0432624   53.7111353 
   36.043188    53.7110657 
   36.0431086   53.7110034 
   36.0429817   53.7109145 
   36.0428908   53.7108528 
   36.0428449   53.7108203 
   36.042791    53.7107786 
   36.042731    53.7107037 
   36.0427012   53.7106401 
   36.0426984   53.7105752 
   36.0427094   53.7105241 
   36.0427348   53.7104797 
   36.0427902   53.7104174 
   36.0428467   53.7103718 
   36.0429141   53.7103508 
   36.0429853   53.7103178 
   36.0430238   53.7102939 
   36.0430407   53.7102566 
   36.0430397   53.7101819 
   36.0430147   53.7101252 
   36.0429491   53.7100692 
   36.0428681   53.7100134 
   36.0427785   53.709973  
   36.0426744   53.7099332 
   36.0425916   53.7099009 
   36.0425313   53.7098719 
   36.0424243   53.7098362 
   36.0423299   53.7097948 
   36.0422365   53.7097653 
   36.0421492   53.7097508 
   36.0420328   53.7097312 
   36.0419147   53.7097306 
   36.0417163   53.709735  
   36.0416137   53.7097372 
   36.0415062   53.7097314 
   36.0414052   53.7097071 
   36.0412857   53.7096704 
   36.0412155   53.7096276 
   36.0410981   53.7095386 
   36.0410096   53.7095097 
   36.0408545   53.7095013 
   36.040692    53.7095089 
   36.0405404   53.7095315 
   36.0403426   53.7095737 
   36.0401401   53.7096436 
   36.0399745   53.7097126 
   36.0398486   53.709777  
   36.0396554   53.7098778 
   36.0395574   53.7099397 
   36.0394343   53.7099937 
   36.0392819   53.7100404 
   36.0390685   53.7100861 
   36.0387884   53.710127  
   36.038482    53.7101543 
   36.0380357   53.7101564 
   36.0374956   53.7101623 
   36.0369424   53.7102028 
   36.0365      53.7102686 
   36.0361743   53.7103792 
   36.0358749   53.7105052 
   36.0357468   53.7105534 
   36.0354848   53.7105903 
   36.0353045   53.7106451 
   36.0351591   53.7107003 
   36.0349917   53.7107775 
   36.0348137   53.7108656 
   36.0346609   53.7109519 
   36.0342146   53.7111733 
   36.0339655   53.7112927 
   36.033747    53.7113862 
   36.0332711   53.7115722 
   36.0329328   53.7116806 
   36.0325285   53.7117749 
   36.0322483   53.7118186 
   36.0319244   53.7118476 
   36.0312596   53.7118691 
   36.031103    53.7118826 
   36.0307684   53.7119109 
   36.0304123   53.7119786 
   36.0304994   53.7124284 
   36.0305844   53.7127084 
   36.0307617   53.7133944 
   36.030864    53.7136662 
   36.0309061   53.7139255 
   36.0310407   53.7143051 
   36.0310158   53.7147562 
   36.0309563   53.7158379 
   36.0309312   53.7164365 
   36.0308305   53.7165931 
   36.0305007   53.7171334 
   36.0303955   53.7173865 
   36.0302584   53.717564  
   36.0298308   53.7181092 
   36.0295791   53.7184217 
   36.0296922   53.7184728 
   36.0302342   53.7187523 
   36.0311859   53.7191959 
   36.0322925   53.7197273 
   36.0333369   53.720221  
   36.0341477   53.7206031 
   36.0355479   53.7213571 
   36.0367252   53.721942  
   36.0373771   53.7223249 
   36.0381266   53.7227683 
   36.0357989   53.7234377 
   36.0345399   53.7238042 
   36.0341982   53.7238804 
   36.0333514   53.7241554 
   36.0327962   53.7243405 
   36.0322361   53.7245406 
   36.0312873   53.7248413 
   36.0309094   53.7249786 
   36.0300824   53.7254245 
   36.0287517   53.7261622 
   36.0273253   53.7260918 
   36.0262731   53.7260391 
   36.0256255   53.7259915 
   36.0253948   53.7259707 
   36.0249178   53.7259293 
   36.0246555   53.7258672 
   36.0233991   53.7260544 
   36.0218363   53.7262176 
   36.0211751   53.7263124 
   36.0201391   53.7264779 
   36.0193465   53.726439  
   36.0192689   53.7264266 
   36.0181023   53.7263734 
   36.0173228   53.7263044 
   36.0174677   53.7256664 
   36.0176223   53.7250236 
   36.0177075   53.7244168 
   36.0178204   53.7238489 
   36.0179283   53.7233453 
   36.0180272   53.7229038 
   36.0180403   53.722565  
   36.0180756   53.7224384 
   36.0180258   53.7220964 
   36.0178055   53.7221444 
   36.0173616   53.7222636 
   36.016812    53.7224394 
   36.0165557   53.7225359 
   36.0161603   53.7226732 
   36.0154374   53.7229349 
   36.0150264   53.7230573 
   36.014476    53.7231792 
   36.0141849   53.7232769 
   36.0137063   53.7234158 
   36.0134579   53.7235179 
   36.0131568   53.7236123 
   36.0124067   53.7238729 
   36.0111958   53.7242975 
   36.0109681   53.7243535 
   36.0107114   53.7244248 
   36.0103331   53.7245356 
   36.010181    53.7246052 
   36.0098267   53.7247675 
   36.0095769   53.7249306 
   36.0092527   53.7248653 
   36.007788    53.7245134 
   36.0065609   53.724258  
   36.006114    53.7241382 
   36.0051178   53.7239267 
   36.0043273   53.7237429 
   36.0040781   53.7236337 
   36.0036589   53.7232461 
   36.003344    53.7229925 
   36.003109    53.7227868 
   36.0027533   53.7225138 
   36.0020936   53.7220986 
   36.0015243   53.7217916 
   36.0008719   53.721491  
   36.0001203   53.7211601 
   35.9996864   53.7209988 
   35.9993041   53.7210878 
   35.9981208   53.7213961 
   35.9969976   53.7217088 
   35.9956036   53.722057  
   35.9943747   53.7224459 
   35.9929767   53.7229411 
   35.9926584   53.7230401 
   35.9910436   53.7232433 
   35.9902529   53.723187  
   35.9901117   53.7231968 
   35.9897351   53.7232891 
   35.9888509   53.723526  
   35.9881269   53.723714  
   35.9877567   53.7238442 
   35.987582    53.7238139 
   35.9863492   53.7235549 
   35.9852714   53.7233101 
   35.9842039   53.7231056 
   35.9835324   53.7228247 
   35.9839221   53.7220489 
   35.9839594   53.7209675 
   35.9839601   53.7208578 
   35.9833428   53.7208185 
   35.982731    53.7211565 
   35.9821519   53.7214839 
   35.9816104   53.7217662 
   35.9814052   53.7219404 
   35.9812724   53.7220133 
   35.9808487   53.7222138 
   35.9802576   53.7225034 
   35.9788893   53.7231574 
   35.9782382   53.723453  
   35.9780502   53.7235244 
   35.9745751   53.7237845 
   35.9737927   53.7231239 
   35.9732664   53.7221565 
   35.9729975   53.7215955 
   35.9729405   53.7215302 
   35.97249     53.7207728 
   35.9724605   53.7207504 
   35.971973    53.7207635 
   35.9714402   53.7207346 
   35.9700964   53.7204185 
   35.9695384   53.7200555 
   35.9687252   53.7195557 
   35.9682502   53.719004  
   35.9675993   53.7183038 
   35.9670288   53.7178673 
   35.9665525   53.717527  
   35.9663457   53.7173968 
   35.9650798   53.7167791 
   35.964784    53.7166448 
   35.9632492   53.7158676 
   35.9626711   53.7162312 
   35.9615252   53.7169605 
   35.9601101   53.7178391 
   35.9596402   53.7180684 
   35.9584907   53.7184726 
   35.9579316   53.7190422 
   35.9574068   53.7195611 
   35.9579293   53.7202081 
   35.9583168   53.7209209 
   35.9586281   53.7215675 
   35.9587673   53.7218885 
   35.9585927   53.7222063 
   35.9578319   53.7237782 
   35.9576707   53.7245943 
   35.9576663   53.724988  
   35.9577098   53.7251153 
   35.9577268   53.7251698 
   35.9577288   53.7253461 
   35.9577289   53.7255327 
   35.9577471   53.7257055 
   35.957766    53.7258363 
   35.9577662   53.7260308 
   35.957756    53.7261536 
   35.9577361   53.7262481 
   35.9576898   53.7263086 
   35.9576578   53.7263804 
   35.9576489   53.7264304 
   35.9576621   53.7265805 
   35.95771     53.7267499 
   35.9577549   53.7269943 
   35.9578349   53.727276  
   35.9578778   53.7273463 
   35.9578999   53.727435  
   35.9579247   53.7275931 
   35.9579475   53.7277339 
   35.9579419   53.7279274 
   35.9579335   53.7280309 
   35.9579113   53.7281015 
   35.9578455   53.7282155 
   35.9578042   53.7283044 
   35.9577098   53.7284424 
   35.9576345   53.7285599 
   35.9576045   53.7286316 
   35.9575675   53.7287455 
   35.9575607   53.7288252 
   35.9575611   53.7288673 
   35.9575703   53.7290003 
   35.9575501   53.7290709 
   35.9575106   53.7291462 
   35.9574672   53.7292067 
   35.957401    53.7292865 
   35.9573155   53.7293642 
   35.9572958   53.7294053 
   35.9572717   53.729485  
   35.9572502   53.7296239 
   35.9571773   53.7297754 
   35.9571092   53.7298678 
   35.9570395   53.7299807 
   35.9569639   53.7300845 
   35.9568842   53.7301644 
   35.9567566   53.7302451 
   35.9566178   53.7303531 
   35.9564678   53.730488  
   35.956413    53.730561  
   35.9563592   53.7306391 
   35.9562881   53.7307128 
   35.9561836   53.7307996 
   35.9560618   53.7309002 
   35.9559384   53.7310031 
   35.9557234   53.7311597 
   35.9554345   53.7314009 
   35.9553625   53.7314797 
   35.9553209   53.7315345 
   35.9552702   53.7316404 
   35.9552291   53.7317282 
   35.9552106   53.7317931 
   35.9551956   53.7318995 
   35.9551599   53.7319804 
   35.9551066   53.7321051 
   35.9550052   53.7323012 
   35.9549075   53.7324859 
   35.9548868   53.7326008 
   35.9548286   53.7327137 
   35.9547663   53.7327969 
   35.9545968   53.7329102 
   35.9545418   53.7329809 
   35.9544531   53.7331099 
   35.9544137   53.733176  
   35.9544009   53.7332397 
   35.9544076   53.7333216 
   35.954463    53.7334135 
   35.9544843   53.7334839 
   35.9544969   53.7335852 
   35.9544986   53.7337387 
   35.9545058   53.7338683 
   35.954518    53.7340969 
   35.9545371   53.7342675 
   35.9545227   53.7343506 
   35.9545364   53.734537  
   35.9545458   53.7347054 
   35.9545473   53.7348339 
   35.9545516   53.734884  
   35.9545812   53.7349464 
   35.9546181   53.734986  
   35.9546456   53.7350496 
   35.9546495   53.7352293 
   35.9546369   53.7353045 
   35.9546319   53.7353716 
   35.9546408   53.7354751 
   35.954684    53.7355682 
   35.954766    53.7356987 
   35.9547803   53.7357714 
   35.9548319   53.7359247 
   35.9548518   53.7359793 
   35.9549181   53.7360825 
   35.9549632   53.73617   
   35.9550568   53.7363061 
   35.9551214   53.7364127 
   35.9551974   53.7365342 
   35.9552281   53.736608  
   35.9553156   53.7367237 
   35.9554185   53.736829  
   35.9555406   53.7369354 
   35.9556145   53.7370285 
   35.9556576   53.7371011 
   35.9556774   53.7371613 
   35.9556882   53.7372614 
   35.9556849   53.7373137 
   35.9556677   53.7374855 
   35.9556362   53.7375936 
   35.955618    53.7378531 
   35.9556224   53.73791   
   35.9556117   53.7379783 
   35.9555951   53.738025  
   35.9555457   53.7380752 
   35.9555006   53.7381697 
   35.9554423   53.7382678 
   35.9553745   53.7383808 
   35.955343    53.7384764 
   35.9552618   53.7385882 
   35.9552147   53.7386726 
   35.9551541   53.7387422 
   35.9550932   53.7387845 
   35.9550528   53.7388609 
   35.955016    53.7389998 
   35.9550065   53.739175  
   35.9550196   53.7393171 
   35.955021    53.7394423 
   35.9550259   53.7395446 
   35.9550721   53.7397309 
   35.9550907   53.739848  
   35.9550685   53.739922  
   35.9550849   53.740021  
   35.955111    53.7401221 
   35.9551246   53.7401448 
   35.9551653   53.7401821 
   35.9551813   53.7402196 
   35.9551788   53.7403482 
   35.9551625   53.7404404 
   35.9551355   53.7405895 
   35.9551098   53.7407    
   35.9550327   53.7408424 
   35.9549872   53.7408916 
   35.9549494   53.7409394 
   35.9549121   53.7410352 
   35.9548824   53.7411263 
   35.9548657   53.7411809 
   35.954851    53.7412402 
   35.9548516   53.741289  
   35.9548582   53.741363  
   35.9548899   53.741447  
   35.9549081   53.7415322 
   35.9549034   53.7416232 
   35.954912    53.7417222 
   35.9549513   53.7418028 
   35.9550277   53.7419401 
   35.9550786   53.7420366 
   35.9551566   53.7421353 
   35.9552613   53.7422418 
   35.9554085   53.7423618 
   35.9554881   53.7424366 
   35.9556602   53.7425666 
   35.9558232   53.7427139 
   35.9560225   53.7428666 
   35.9561587   53.7429588 
   35.9562335   53.7430353 
   35.9564675   53.7432016 
   35.9567211   53.7434008 
   35.9568568   53.7435197 
   35.9569018   53.7435912 
   35.9569604   53.7436911 
   35.9570113   53.7437819 
   35.9570531   53.7439114 
   35.9570982   53.7439896 
   35.9571139   53.7440283 
   35.9571323   53.7441249 
   35.9570964   53.7441797 
   35.9570395   53.7442424 
   35.9570228   53.7442948 
   35.9570199   53.7443767 
   35.95699     53.7444416 
   35.9569485   53.7445113 
   35.956905    53.7445615 
   35.9568325   53.7445936 
   35.956682    53.7446898 
   35.9565907   53.7447708 
   35.9564995   53.7448531 
   35.9564595   53.7448828 
   35.956343    53.7449243 
   35.9561426   53.7450171 
   35.9560433   53.7450597 
   35.9559078   53.7451307 
   35.9557514   53.7452269 
   35.9556088   53.7453366 
   35.9555156   53.7454052 
   35.9554016   53.7455092 
   35.9552916   53.7456189 
   35.9552042   53.7457113 
   35.9551613   53.7458162 
   35.9551182   53.7459108 
   35.9550729   53.7459769 
   35.9550333   53.7460452 
   35.9549327   53.7461446 
   35.9548588   53.7462235 
   35.9547141   53.7463286 
   35.9546113   53.7463927 
   35.9544931   53.7464649 
   35.9543597   53.7465462 
   35.954274    53.7466079 
   35.9542266   53.7466615 
   35.9541984   53.7467048 
   35.9541644   53.7467516 
   35.9541325   53.7468166 
   35.9540894   53.7469191 
   35.9540671   53.7469887 
   35.9540373   53.7470592 
   35.9540229   53.74714   
   35.9540218   53.7472152 
   35.9540223   53.7472607 
   35.9540307   53.7473323 
   35.9540621   53.7473924 
   35.9541282   53.7474638 
   35.9541883   53.7475182 
   35.9542716   53.747585  
   35.9543296   53.7476337 
   35.9543589   53.7476779 
   35.9543594   53.7477279 
   35.9543544   53.7477951 
   35.9543063   53.7479454 
   35.9542727   53.7480321 
   35.9542545   53.7481151 
   35.9542533   53.7481799 
   35.9542633   53.748222  
   35.9542836   53.7483169 
   35.9543673   53.7484332 
   35.9543856   53.7485235 
   35.9544559   53.7486296 
   35.9544865   53.7487051 
   35.9545015   53.7487625 
   35.9545253   53.7488261 
   35.9545743   53.7489284 
   35.9547246   53.7489346 
   35.9549873   53.7489164 
   35.9552631   53.7488516 
   35.955736    53.7487691 
   35.9566975   53.7485935 
   35.9569702   53.7487039 
   35.9574597   53.7488771 
   35.9580881   53.7491294 
   35.9584466   53.7492622 
   35.9587317   53.7493521 
   35.9590436   53.7494556 
   35.9584488   53.7501472 
   35.958571    53.7511044 
   35.9585649   53.7512433 
   35.9585016   53.7515074 
   35.9584212   53.7518626 
   35.9561392   53.7516919 
   35.9539824   53.7513842 
   35.9509803   53.751339  
   35.9503133   53.7510664 
   35.9488623   53.7510743 
   35.9473507   53.7514783 
   35.9461388   53.7515694 
   35.9447388   53.7516681 
   35.9434884   53.7517456 
   35.942944    53.7518114 
   35.9412916   53.751977  
   35.9390826   53.7521697 
   35.9368431   53.7523738 
   35.9367472   53.7523742 
   35.9343361   53.7526813 
   35.9334505   53.7527847 
   35.9335304   53.753087  
   35.933984    53.7546107 
   35.9345163   53.757011  
   35.9347827   53.7582248 
   35.9347962   53.7583254 
   35.9349122   53.7587458 
   35.9348727   53.7588325 
   35.934756    53.7588727 
   35.9345853   53.7588949 
   35.9343649   53.758914  
   35.9341654   53.7589284 
   35.9340063   53.7589472 
   35.9338357   53.758974  
   35.9335753   53.759033  
   35.9334395   53.7590779 
   35.9333668   53.7590986 
   35.9332445   53.7591628 
   35.933121    53.7591792 
   35.9328706   53.7591852 
   35.9327535   53.759201  
   35.9325982   53.7592061 
   35.9323238   53.7592174 
   35.9320647   53.7592149 
   35.9319686   53.7592028 
   35.9319147   53.7591951 
   35.9317881   53.7591956 
   35.9316767   53.7591994 
   35.9316096   53.7592065 
   35.9315041   53.7592159 
   35.9314293   53.7592242 
   35.931347    53.7592438 
   35.9312285   53.7592841 
   35.9310735   53.7593438 
   35.9309857   53.7593817 
   35.930806    53.7594529 
   35.9305844   53.7595606 
   35.930367    53.7596899 
   35.9302298   53.7597803 
   35.930142    53.7598307 
   35.9299223   53.7599328 
   35.9298155   53.7600002 
   35.9296763   53.7600873 
   35.9295525   53.7601752 
   35.9294078   53.7602907 
   35.9292897   53.7603741 
   35.9290744   53.7605183 
   35.9289258   53.760636  
   35.92884     53.7606921 
   35.9286898   53.7608279 
   35.928566    53.7609194 
   35.9284114   53.7610008 
   35.9283291   53.7610352 
   35.9281111   53.7611088 
   35.9280272   53.7611603 
   35.9278439   53.7612519 
   35.9275707   53.7613815 
   35.9273107   53.7614894 
   35.9271599   53.7615491 
   35.9270251   53.7616122 
   35.9269439   53.7616443 
   35.9268234   53.7616903 
   35.9266307   53.7617979 
   35.9265447   53.7618323 
   35.9264453   53.7618714 
   35.9263133   53.7619185 
   35.9262674   53.7619426 
   35.9262064   53.7619763 
   35.9259823   53.7620255 
   35.9258656   53.7620566 
   35.9256721   53.7621029 
   35.9255497   53.7621386 
   35.9253792   53.7621802 
   35.9252453   53.7622205 
   35.9250328   53.7622918 
   35.9248091   53.7623666 
   35.9246982   53.7624113 
   35.9245091   53.7624927 
   35.9242375   53.7625927 
   35.9241458   53.7626397 
   35.9238567   53.7627215 
   35.9237727   53.7627457 
   35.9235773   53.762801  
   35.9235162   53.7628285 
   35.9234589   53.7628549 
   35.9232527   53.7629672 
   35.923167    53.7630363 
   35.9230659   53.7630912 
   35.9229456   53.763144  
   35.9229613   53.7631826 
   35.9229601   53.7632623 
   35.9229394   53.7632998 
   35.9227522   53.7635611 
   35.922616    53.7637697 
   35.9223323   53.7641688 
   35.9221349   53.7643811 
   35.9220568   53.7644497 
   35.9220343   53.764493  
   35.9220193   53.7645339 
   35.9220197   53.7645738 
   35.922022    53.7646125 
   35.9220397   53.7646488 
   35.9221136   53.7647407 
   35.9221804   53.7648758 
   35.9222429   53.7649939 
   35.9222438   53.7650849 
   35.9222392   53.7651826 
   35.9222185   53.7652329 
   35.9221863   53.7652659 
   35.9221386   53.7653082 
   35.9220605   53.7653619 
   35.9220071   53.7653917 
   35.921969    53.7654169 
   35.9218642   53.7654969 
   35.9217119   53.765618  
   35.9216282   53.7656957 
   35.9215597   53.7657505 
   35.9213707   53.7658615 
   35.9212659   53.7659347 
   35.9210657   53.7660639 
   35.9209839   53.7661371 
   35.9208942   53.7661909 
   35.9207663   53.7662595 
   35.9207014   53.7662905 
   35.9206691   53.7663224 
   35.9206367   53.766359  
   35.9206066   53.7664092 
   35.9205958   53.7664718 
   35.9206057   53.7665127 
   35.9206486   53.7665648 
   35.9206992   53.7666352 
   35.920707    53.766659  
   35.9207411   53.7667851 
   35.9207768   53.7668976 
   35.9208461   53.7670964 
   35.9208428   53.7671619 
   35.9208977   53.7672657 
   35.9209503   53.767344  
   35.9209685   53.767419  
   35.9209846   53.7674929 
   35.9209912   53.767576  
   35.9209956   53.7676327 
   35.9209963   53.7676988 
   35.9209628   53.7678058 
   35.9209349   53.7678935 
   35.9209238   53.7679345 
   35.9209288   53.7680289 
   35.9209313   53.7681005 
   35.9209456   53.7681733 
   35.9209521   53.7682472 
   35.9209935   53.7683505 
   35.9210366   53.7684414 
   35.9210684   53.7685402 
   35.9211566   53.7687083 
   35.9211496   53.76878   
   35.9211223   53.7689279 
   35.9211023   53.7690287 
   35.9210951   53.7690764 
   35.9211168   53.7691275 
   35.9211811   53.7692263 
   35.9212606   53.7693    
   35.9213036   53.7693658 
   35.9213099   53.7694192 
   35.9213161   53.7694663 
   35.9213246   53.7695334 
   35.9213214   53.7696097 
   35.9213128   53.7696962 
   35.9212981   53.7697678 
   35.9212738   53.7698248 
   35.9212511   53.7698636 
   35.9211981   53.7699275 
   35.9211656   53.769956  
   35.9211142   53.7699858 
   35.9210435   53.7700258 
   35.9210014   53.7700465 
   35.9209731   53.7700785 
   35.9209351   53.7701287 
   35.9209163   53.7701617 
   35.9209089   53.7701925 
   35.9209151   53.7702311 
   35.92095     53.7702617 
   35.9210257   53.7703376 
   35.9210703   53.7703852 
   35.9210823   53.770425  
   35.9210999   53.7704579 
   35.9210946   53.7705057 
   35.9210395   53.7705582 
   35.9209249   53.7706189 
   35.9208808   53.7706214 
   35.920827    53.7706273 
   35.9207028   53.7706777 
   35.9206226   53.770719  
   35.9205558   53.7707579 
   35.9205195   53.7707888 
   35.9203865   53.7709121 
   35.9203145   53.771009  
   35.9202049   53.7711732 
   35.9201651   53.7712211 
   35.9201232   53.7712623 
   35.9199784   53.7713674 
   35.9198906   53.7714144 
   35.9197226   53.7715083 
   35.9196729   53.7715414 
   35.9196503   53.7715722 
   35.9196222   53.7716383 
   35.9196226   53.7716793 
   35.9196404   53.7717281 
   35.9196659   53.771786  
   35.919678    53.7718428 
   35.9196845   53.7719214 
   35.9196793   53.7719702 
   35.9196625   53.7720192 
   35.9196455   53.7720466 
   35.9195923   53.7720888 
   35.919512    53.7721358 
   35.9193747   53.7722137 
   35.9193003   53.772269  
   35.9192588   53.7723238 
   35.9191728   53.7723764 
   35.9191002   53.7724176 
   35.9190833   53.7724473 
   35.9190761   53.7724985 
   35.9190631   53.7725418 
   35.9190217   53.7726159 
   35.9189499   53.7727344 
   35.9189013   53.7728518 
   35.9188695   53.7729531 
   35.9188434   53.7730237 
   35.9188443   53.7731068 
   35.9188278   53.7731875 
   35.9187884   53.7732719 
   35.9187892   53.7733481 
   35.918802    53.7734777 
   35.9188315   53.7735448 
   35.9188785   53.7736242 
   35.9189118   53.7736991 
   35.9189186   53.7738003 
   35.9189159   53.7739004 
   35.9188958   53.7739989 
   35.9188586   53.7741276 
   35.9188309   53.7742141 
   35.918791    53.7742631 
   35.9186823   53.7743386 
   35.9186176   53.7743901 
   35.9185699   53.7744289 
   35.9185531   53.7744756 
   35.9185536   53.7745246 
   35.9185598   53.7745632 
   35.9186007   53.7746131 
   35.918626    53.7746403 
   35.9186226   53.7746972 
   35.9186134   53.7747324 
   35.9185887   53.7747632 
   35.9185505   53.7747748 
   35.9184775   53.7747796 
   35.9183393   53.7747836 
   35.9182685   53.7748065 
   35.9182132   53.7748408 
   35.918154    53.7748797 
   35.9181389   53.7748991 
   35.918147    53.7749389 
   35.9181684   53.7749639 
   35.9182225   53.7749898 
   35.9182975   53.7750111 
   35.9183745   53.7750222 
   35.9184552   53.7750402 
   35.9185152   53.775082  
   35.9185752   53.7751148 
   35.9186022   53.7751283 
   35.918633    53.7751419 
   35.9186622   53.7751759 
   35.9187068   53.7752156 
   35.9187342   53.7752655 
   35.9187556   53.7752984 
   35.9187755   53.775354  
   35.9187798   53.7754042 
   35.9187669   53.7754621 
   35.9187389   53.7755419 
   35.9186974   53.7756035 
   35.9186383   53.7756571 
   35.9185775   53.7757142 
   35.9185279   53.7757486 
   35.9185013   53.7757725 
   35.918496    53.775817  
   35.9184981   53.7758431 
   35.9185119   53.7758737 
   35.918545    53.7759237 
   35.9185763   53.7759634 
   35.9185824   53.7760032 
   35.9185789   53.7760452 
   35.9185467   53.7760851 
   35.9185049   53.7761172 
   35.9184648   53.7761378 
   35.9184035   53.7761539 
   35.9182963   53.7761861 
   35.9182484   53.7762068 
   35.9181527   53.7762435 
   35.9180706   53.7762802 
   35.9180114   53.7763192 
   35.9179544   53.7763648 
   35.9179048   53.7764048 
   35.9178611   53.7764494 
   35.9178365   53.7764802 
   35.9177948   53.7765338 
   35.9177513   53.7765999 
   35.9177191   53.7766444 
   35.9176872   53.7767139 
   35.9176568   53.7767539 
   35.9175979   53.7768166 
   35.9175523   53.7768543 
   35.9174682   53.7768899 
   35.9173935   53.7769185 
   35.9173785   53.7769402 
   35.91735     53.7769756 
   35.9175325   53.777077  
   35.9250262   53.7854995 
   35.9284136   53.7882518 
   35.9314589   53.7895407 
   35.9333889   53.7875109 
   35.9345533   53.7883578 
   35.9361412   53.7878285 
   35.9412223   53.7904751 
   35.9453508   53.7882521 
   35.9473622   53.7857115 
   35.9492646   53.7837108 
   35.9533961   53.7870878 
   35.9657811   53.7949215 
   35.9698036   53.7970387 
   35.9731969   53.800197  
   35.9702269   53.801273  
   35.9696975   53.8028609 
   35.9675803   53.802755  
   35.964828    53.8037077 
   35.9614404   53.8073068 
   35.9656746   53.8099533 
   35.9684268   53.8131291 
   35.9728246   53.8163851 
   35.9695909   53.8188455 
   35.9651447   53.8242442 
   35.9569936   53.8243499 
   35.9488425   53.8255142 
   35.9381508   53.8274195 
   35.9347632   53.8296425 
   35.9298936   53.8326064 
   35.9255535   53.8327122 
   35.9254474   53.8375817 
   35.9186724   53.8400163 
   35.9139088   53.8400162 
   35.9006765   53.8408629 
   35.899512    53.8432976 
   35.900147    53.846791  
   35.901161    53.8489956 
   35.9025815   53.8520839 
   35.9049102   53.8549421 
   35.92153     53.8512373 
   35.9263993   53.8548895 
   35.9328568   53.8529842 
   35.9365619   53.8515022 
   35.943866    53.8529843 
   35.950535    53.8534079 
   35.9534991   53.8521376 
   35.9586862   53.850444  
   35.9653551   53.8525612 
   35.9676839   53.854255  
   35.9701186   53.8556312 
   35.9755173   53.8561606 
   35.9775286   53.8579072 
   35.9809158   53.8623533 
   35.9921365   53.8680699 
   35.9986996   53.871034  
   36.0015576   53.8731512 
   35.9991228   53.8761152 
   35.9993343   53.8799261 
   35.996582    53.8803495 
   35.9939355   53.881408  
   35.9909714   53.8822548 
   35.9874781   53.8823606 
   35.9879014   53.8864891 
   35.9875837   53.8891355 
   35.9846196   53.8906175 
   35.9818672   53.8922053 
   35.9811261   53.8949576 
   35.9844075   53.8981334 
   35.984725    53.9001447 
   35.9823961   53.9023677 
   35.980808    53.9064962 
   35.9815489   53.9087192 
   35.9857832   53.9103071 
   35.9886346   53.9103684 
   35.9910759   53.9129537 
   35.993934    53.914965  
   35.9979566   53.9152827 
   36.0026143   53.9161296 
   36.0093891   53.9177176 
   36.011612    53.9221637 
   36.0200805   53.9247044 
   36.0243149   53.9215287 
   36.0257969   53.9217404 
   36.0267495   53.9240693 
   36.0314073   53.924387  
   36.0343713   53.9247046 
   36.0298192   53.9297857 
   36.0318304   53.930315  
   36.0286547   53.9313736 
   36.0257964   53.9333848 
   36.035747    53.9349729 
   36.036276    53.9418007 
   36.0464383   53.9429653 
   36.0517311   53.9443415 
   36.0558595   53.9463529 
   36.0570238   53.9495287 
   36.0537421   53.9503755 
   36.0482376   53.9482582 
   36.046332    53.9512222 
   36.0479199   53.9513281 
   36.0471787   53.95588   
   36.0503544   53.9561976 
   36.0542711   53.9569387 
   36.0585055   53.9567271 
   36.0637806   53.9566954 
   36.0652803   53.9573623 
   36.0628456   53.9579974 
   36.0660212   53.9612791 
   36.0770303   53.9621261 
   36.0802649   53.9606927 
   36.0822175   53.9604325 
   36.0827468   53.9595856 
   36.0857109   53.9582095 
   36.0910038   53.9574686 
   36.0943912   53.9592682 
   36.0993665   53.9578921 
   36.1002135   53.9559867 
   36.1043419   53.9561985 
   36.1058239   53.9567278 
   36.1068825   53.9563044 
   36.1107994   53.9538697 
   36.1129169   53.9463538 
   36.1155634   53.9446601 
   36.1145049   53.9431781 
   36.116728    53.9414844 
   36.1201155   53.9385204 
   36.1224442   53.942596  
   36.1319714   53.9437606 
   36.141287    53.9403732 
   36.140334    53.947995  
   36.1432979   53.9497947 
   36.1514489   53.9518061 
   36.1558948   53.9537116 
   36.1567739   53.9562375 
   36.1550478   53.957099  
   36.1531423   53.959322  
   36.1537774   53.9606982 
   36.1504958   53.9609099 
   36.1493312   53.9624977 
   36.1471082   53.9631328 
   36.1454144   53.9644031 
   36.1439324   53.9645089 
   36.1417094   53.9647206 
   36.135993    53.9660967 
   36.1341935   53.9646146 
   36.1283713   53.9647733 
   36.1261481   53.9684783 
   36.1264655   53.9727127 
   36.1245598   53.9780056 
   36.1274124   53.9809205 
   36.1238499   53.9826701 
   36.1203656   53.9835679 
   36.1161375   53.9858469 
   36.1122226   53.9861001 
   36.1054106   53.9873431 
   36.1002429   53.9871359 
   36.0934692   53.9873392 
   36.0933918   53.9910623 
   36.0969153   53.9941694 
   36.0973459   53.9972762 
   36.0955059   53.9974603 
   36.0979723   54.0056751 
   36.1037272   54.0068715 
   36.1157068   54.0089421 
   36.1237324   54.0104605 
   36.1211486   54.0137501 
   36.1192694   54.0177066 
   36.1196218   54.0187876 
   36.1219707   54.0184656 
   36.125033    54.0190959 
   36.1301001   54.0222752 
   36.1315486   54.0221832 
   36.1311571   54.0254719 
   36.127986    54.0269897 
   36.1311963   54.0269667 
   36.1343282   54.0284384 
   36.1317835   54.0301401 
   36.1291201   54.0319922 
   36.126929    54.0335661 
   36.1343673   54.0378426 
   36.1381257   54.0376816 
   36.1397699   54.0373598 
   36.1411401   54.0380265 
   36.1424712   54.0399576 
   36.1446244   54.0414519 
   36.1452508   54.0423254 
   36.1407878   54.0449    
   36.1387129   54.0446931 
   36.1360899   54.0466239 
   36.1383997   54.0489224 
   36.1410618   54.0503474 
   36.1432151   54.0502555 
   36.1484219   54.052209  
   36.155547    54.05476   
   36.1724986   54.0531513 
   36.1804067   54.0532892 
   36.1842825   54.0556333 
   36.1882757   54.0555873 
   36.1902249   54.0557341 
   36.1919921   54.056679  
   36.1925821   54.0578394 
   36.1913293   54.06039   
   36.1896068   54.0605738 
   36.1882365   54.0634459 
   36.1947353   54.0621592 
   36.1966145   54.0624809 
   36.1963013   54.0615848 
   36.2009992   54.0608725 
   36.2042485   54.0591032 
   36.2101992   54.0611253 
   36.2142707   54.0628026 
   36.2150928   54.0639054 
   36.2113345   54.0658354 
   36.2190078   54.0699706 
   36.2167763   54.0705909 
   36.2132137   54.0722908 
   36.2107081   54.0724976 
   36.2080069   54.0730029 
   36.212235    54.0799396 
   36.2171286   54.0891483 
   36.209886    54.0896994 
   36.2093876   54.092103  
   36.2091813   54.0930976 
   36.2034608   54.0927614 
   36.1966251   54.0923115 
   36.1894893   54.0903653 
   36.1796237   54.0902964 
   36.1806025   54.0978041 
   36.1766571   54.1026548 
   36.1714024   54.1019591 
   36.1595402   54.1026019 
   36.1580917   54.1009721 
   36.1531981   54.0997324 
   36.1470908   54.0993422 
   36.1438414   54.100467  
   36.1416099   54.0998472 
   36.1402006   54.098378  
   36.1365597   54.0970694 
   36.135072    54.0950721 
   36.1358942   54.0933961 
   36.1353852   54.0919266 
   36.1318618   54.0896764 
   36.1267724   54.0896764 
   36.1260677   54.0908245 
   36.1247367   54.0911919 
   36.1247367   54.0921792 
   36.1230924   54.0925466 
   36.1232696   54.0942734 
   36.1205014   54.0962865 
   36.1202245   54.0979747 
   36.1191172   54.0986241 
   36.1190065   54.0997603 
   36.1165151   54.0995006 
   36.1070407   54.0979317 
   36.1057742   54.1019679 
   36.1002931   54.1020328 
   36.1005796   54.1077219 
   36.1005145   54.108882  
   36.0988536   54.1088495 
   36.0945351   54.1071617 
   36.089861    54.106566  
   36.087393    54.1049868 
   36.0806868   54.1055765 
   36.073941    54.106206  
   36.0718353   54.1068371 
   36.0728873   54.1086547 
   36.0727765   54.1106346 
   36.0706173   54.1112513 
   36.0698975   54.1136854 
   36.0677383   54.1162167 
   36.0710602   54.1184559 
   36.0744929   54.118391  
   36.0773719   54.1184883 
   36.1057742   54.1188128 
   36.1057188   54.1161843 
   36.1076013   54.1157624 
   36.1082656   54.1137503 
   36.1087363   54.111657  
   36.107795    54.1098557 
   36.1120305   54.1079731 
   36.116349    54.1105373 
   36.120446    54.1125333 
   36.121664    54.114659  
   36.1275604   54.1167197 
   36.1289722   54.1188128 
   36.1285362   54.1214973 
   36.1280034   54.123534  
   36.1322388   54.1258538 
   36.1357268   54.1262432 
   36.1423153   54.1258214 
   36.1423706   54.1278005 
   36.1398792   54.129163  
   36.1411526   54.1319367 
   36.1435056   54.1335262 
   36.1302733   54.1380511 
   36.1314637   54.1413918 
   36.1317405   54.1439214 
   36.109982    54.1437593 
   36.1066047   54.141781  
   36.1032159   54.1401526 
   36.0990636   54.1383039 
   36.096323    54.1369741 
   36.0861358   54.1326599 
   36.0730142   54.1279884 
   36.0711041   54.1289779 
   36.0667026   54.1290914 
   36.0665365   54.1308108 
   36.0690002   54.1311677 
   36.0694708   54.1333735 
   36.0736509   54.1332924 
   36.0778033   54.1405351 
   36.0831184   54.1435837 
   36.0886549   54.1449133 
   36.0854991   54.1468266 
   36.0819003   54.1489992 
   36.0816235   54.1510096 
   36.0802394   54.1529875 
   36.0747029   54.1554516 
   36.0654015   54.1579155 
   36.0611668   54.1529767 
   36.0544945   54.1552895 
   36.0606955   54.1583694 
   36.0569306   54.1593419 
   36.0529443   54.1621622 
   36.0464666   54.1634264 
   36.0433661   54.1658251 
   36.0352275   54.1703302 
   36.0296356   54.1704923 
   36.0288605   54.1715618 
   36.0253171   54.1724044 
   36.0217737   54.1755154 
   36.0196145   54.1776865 
   36.0186732   54.1789179 
   36.0162925   54.1781078 
   36.0118633   54.1767792 
   36.0105899   54.1747053 
   36.0082646   54.1733118 
   36.0058839   54.1727285 
   36.0047766   54.1733766 
   36.0021744   54.1732794 
   36.0018976   54.1724368 
   35.9873919   54.1649823 
   35.9775369   54.1697793 
   35.9778137   54.1677698 
   35.9769988   54.1670755 
   35.9739756   54.1675057 
   35.9711598   54.1673409 
   35.9708427   54.1648713 
   35.9696205   54.1642368 
   35.9683194   54.1646582 
   35.9676965   54.1643016 
   35.9683748   54.1632968 
   35.9658833   54.1610276 
   35.9631151   54.1623243 
   35.9638902   54.1630212 
   35.9636369   54.1637998 
   35.9643582   54.1640945 
   35.9645848   54.1647397 
   35.9643782   54.165224  
   35.9633303   54.1658775 
   35.9626717   54.1666054 
   35.9618749   54.1687206 
   35.9613112   54.1707483 
   35.9617302   54.1707531 
   35.9702571   54.167977  
   35.9706063   54.1679101 
   35.9711241   54.1678806 
   35.9716218   54.1679028 
   35.9673669   54.1723209 
   35.9647831   54.1749332 
   35.9670537   54.1756665 
   35.9693244   54.1767205 
   35.9688546   54.1796992 
   35.968698    54.1813488 
   35.9716733   54.1817612 
   35.9737874   54.1804324 
   35.977624    54.1803865 
   35.9803644   54.181807  
   35.9799729   54.1831358 
   35.9833398   54.184098  
   35.981304    54.1906954 
   35.9812257   54.1929401 
   35.9862368   54.1959176 
   35.991013    54.1957344 
   35.9947713   54.1933982 
   35.9980598   54.1926652 
   36.0040888   54.1919323 
   36.008082    54.1916574 
   36.0069858   54.1931691 
   36.0068292   54.194864  
   36.0078471   54.1958718 
   36.009648    54.1966047 
   36.0134846   54.196055  
   36.0173995   54.1947724 
   36.0192004   54.1958718 
   36.0209229   54.1975207 
   36.0241331   54.1989406 
   36.0275783   54.1995361 
   36.0294411   54.1996255 
   36.0286744   54.1982536 
   36.0264932   54.1969057 
   36.0223409   54.1965799 
   36.021471    54.1950931 
   36.0197484   54.194635  
   36.0196206   54.1933543 
   36.0167731   54.1929859 
   36.0154705   54.1911869 
   36.0124808   54.1906686 
   36.0123147   54.1877854 
   36.0132559   54.186522  
   36.0151383   54.1854853 
   36.0159688   54.1840273 
   36.0186817   54.1824722 
   36.0195676   54.1810141 
   36.0254363   54.1809493 
   36.0260453   54.1785516 
   36.0280384   54.1768666 
   36.0275955   54.1761213 
   36.030495    54.174383  
   36.0349591   54.1719732 
   36.0391115   54.1745982 
   36.0429972   54.1755524 
   36.0515687   54.1803013 
   36.0558318   54.1799125 
   36.0574374   54.1809817 
   36.0608147   54.1815002 
   36.0620327   54.1805605 
   36.0601503   54.1778063 
   36.0610361   54.1773526 
   36.0628632   54.1777415 
   36.0638044   54.176737  
   36.0630293   54.1751491 
   36.072386    54.1762833 
   36.0773689   54.1769314 
   36.078144    54.1791996 
   36.0815213   54.1796857 
   36.0853415   54.1804633 
   36.0891617   54.1812085 
   36.0922067   54.181889  
   36.0951965   54.1812409 
   36.098463    54.1812409 
   36.1014527   54.1810141 
   36.1041656   54.1801717 
   36.1085948   54.1799773 
   36.1087056   54.185712  
   36.1155155   54.1854529 
   36.1167335   54.183023  
   36.1196125   54.1814353 
   36.1205537   54.1788108 
   36.1234881   54.1779683 
   36.1259242   54.1796209 
   36.1409281   54.1771258 
   36.1497866   54.1689267 
   36.1625759   54.1715195 
   36.174258    54.1736584 
   36.1816011   54.1682691 
   36.189871    54.1619903 
   36.1970131   54.1567709 
   36.1989509   54.1573545 
   36.2061484   54.1557335 
   36.2091935   54.1588458 
   36.2138995   54.1633193 
   36.2163356   54.1663986 
   36.2258584   54.1694453 
   36.2344954   54.1721028 
   36.2386478   54.1750195 
   36.2402534   54.1797829 
   36.2428002   54.1807225 
   36.2431323   54.182051  
   36.2352705   54.1860684 
   36.2351598   54.1872023 
   36.2381748   54.189092  
   36.2421597   54.1927716 
   36.2420031   54.1940771 
   36.2431384   54.195291  
   36.2423946   54.1959781 
   36.2437256   54.1967568 
   36.2441171   54.1982912 
   36.2464269   54.1976041 
   36.2490891   54.1983141 
   36.2492848   54.1995966 
   36.2477084   54.2008749 
   36.2471707   54.2019782 
   36.2481495   54.2033979 
   36.2486193   54.2093283 
   36.2471316   54.211137  
   36.2441171   54.2115033 
   36.2419639   54.2118238 
   36.2413375   54.2126709 
   36.2419248   54.2155325 
   36.2607154   54.2186193 
   36.2611893   54.2092334 
   36.2630391   54.20917   
   36.2628503   54.2154497 
   36.2627432   54.2206959 
   36.2637142   54.2251804 
   36.2673371   54.2248523 
   36.2685166   54.2229549 
   36.2685117   54.220848  
   36.271446    54.220622  
   36.2752028   54.2202656 
   36.2830439   54.2196819 
   36.2833849   54.2205673 
   36.2899742   54.2196443 
   36.2901659   54.2144274 
   36.2916547   54.2141113 
   36.2918126   54.2133418 
   36.30192     54.2126608 
   36.3035554   54.2152907 
   36.3078484   54.2161872 
   36.3089216   54.2178307 
   36.3113236   54.2186973 
   36.309218    54.2200334 
   36.3090188   54.2213118 
   36.3121925   54.2219543 
   36.3101293   54.2235495 
   36.3067565   54.2261126 
   36.3041208   54.2287152 
   36.3038842   54.2339057 
   36.3129399   54.2333858 
   36.3159435   54.2321066 
   36.3230415   54.2208389 
   36.3280386   54.2210446 
   36.330509    54.220423  
   36.3322989   54.2164155 
   36.3362763   54.2133522 
   36.3371205   54.2135175 
   36.3369274   54.2144032 
   36.339735    54.2143173 
   36.3372222   54.2119028 
   36.3409349   54.2118204 
   36.341101    54.2125797 
   36.3423576   54.2125295 
   36.3432117   54.2141675 
   36.3486438   54.2136258 
   36.3478654   54.2121616 
   36.3514651   54.2115327 
   36.3625311   54.2111634 
   36.3626968   54.2099274 
   36.3664954   54.206408  
   36.3699536   54.2047743 
   36.3728227   54.2046094 
   36.3771388   54.202666  
   36.3815902   54.2062352 
   36.3790863   54.2078611 
   36.3778362   54.209458  
   36.3786182   54.2101458 
   36.3802666   54.209406  
   36.3803432   54.2105493 
   36.3830266   54.2123427 
   36.3823366   54.2139118 
   36.3862083   54.2145171 
   36.3939014   54.2170818 
   36.3924261   54.2184549 
   36.3922455   54.2195991 
   36.3927874   54.2206377 
   36.3934498   54.2221339 
   36.3922878   54.2229462 
   36.3918173   54.2237768 
   36.4062222   54.2218441 
   36.4129689   54.2207683 
   36.4142723   54.2211717 
   36.4245139   54.2184188 
   36.4247056   54.2172085 
   36.4263923   54.2169619 
   36.4263923   54.216424  
   36.4251656   54.2159757 
   36.4252806   54.2151015 
   36.4342123   54.2157067 
   36.436474    54.2167826 
   36.4332156   54.2193602 
   36.4328323   54.2219152 
   36.4302639   54.2233271 
   36.4300339   54.2261283 
   36.4304939   54.2291311 
   36.433139    54.2306099 
   36.4382373   54.2285261 
   36.4421473   54.2323576 
   36.4458274   54.2320216 
   36.4506957   54.2315734 
   36.4534986   54.2330331 
   36.447399    54.2383396 
   36.4660572   54.2424674 
   36.4831069   54.2461118 
   36.4847875   54.2459851 
   36.4864138   54.2455733 
   36.4873881   54.2468948 
   36.4882826   54.2468156 
   36.4892329   54.2454149 
   36.4916724   54.2451615 
   36.4951962   54.2474738 
   36.4973646   54.2467453 
   36.5017016   54.248234  
   36.5049543   54.2465553 
   36.5056048   54.2538717 
   36.5050085   54.2548218 
   36.5001836   54.2552652 
   36.4927024   54.2564686 
   36.4870516   54.2640759 
   36.4828773   54.2670204 
   36.4790282   54.2700913 
   36.4786488   54.2711993 
   36.4759382   54.2712943 
   36.4746371   54.2752512 
   36.476914    54.2774353 
   36.4743118   54.2809169 
   36.4720891   54.2833854 
   36.4808714   54.2856957 
   36.4863468   54.2873728 
   36.4922559   54.2872779 
   36.495346    54.2872146 
   36.5019598   54.2865817 
   36.5051041   54.2827208 
   36.5271141   54.2802839 
   36.545004    54.2787963 
   36.5463593   54.2809169 
   36.55113     54.2822145 
   36.5530816   54.2817081 
   36.5551958   54.2824993 
   36.550202    54.2872992 
   36.5536199   54.2871193 
   36.5534399   54.2881986 
   36.5573075   54.2889182 
   36.5572176   54.2878389 
   36.5622952   54.2842147 
   36.5792049   54.2805273 
   36.5879294   54.2801677 
   36.5878396   54.2775593 
   36.5895486   54.2756706 
   36.5970471   54.2793565 
   36.6019039   54.2815152 
   36.6120676   54.2826847 
   36.6145859   54.2835842 
   36.6135965   54.2845735 
   36.6405797   54.2856083 
   36.647088    54.279076  
   36.6486984   54.2787563 
   36.6606372   54.2837198 
   36.6636953   54.2837198 
   36.675118    54.2886669 
   36.6802448   54.2892966 
   36.6800648   54.290286  
   36.6788956   54.2903759 
   36.6778162   54.291905  
   36.673139    54.2925345 
   36.6719696   54.2969417 
   36.6704404   54.3002697 
   36.6727788   54.3021585 
   36.6722391   54.3042272 
   36.6747573   54.3080949 
   36.6820882   54.307743  
   36.6852585   54.3082316 
   36.6890238   54.3064584 
   36.6902467   54.3052283 
   36.6907678   54.3048442 
   36.6907404   54.3031318 
   36.6952111   54.3019635 
   36.7002692   54.2993526 
   36.6978007   54.2940384 
   36.6993092   54.2933821 
   36.712886    54.2891238 
   36.715711    54.2880832 
   36.7143945   54.2874428 
   36.7126117   54.2858417 
   36.7128585   54.284801  
   36.7135991   54.2841285 
   36.7120631   54.2821751 
   36.7125294   54.2804297 
   36.7142848   54.2792288 
   36.7143671   54.278268  
   36.7148333   54.2774833 
   36.7176858   54.2775473 
   36.7188927   54.2776594 
   36.7202092   54.278268  
   36.722952    54.2803337 
   36.7219097   54.2815026 
   36.7245976   54.2846889 
   36.730522    54.2859218 
   36.7382567   54.2824473 
   36.748021    54.2823833 
   36.7489261   54.2838083 
   36.7511752   54.2835041 
   36.7530677   54.2834881 
   36.7568527   54.2830878 
   36.7578401   54.282047  
   36.7636548   54.2805098 
   36.7654376   54.2796931 
   36.767961    54.2794529 
   36.7671381   54.2757217 
   36.7705392   54.2745846 
   36.7784932   54.2743604 
   36.7807423   54.2747928 
   36.7836497   54.2751932 
   36.7879833   54.276058  
   36.7906438   54.2763463 
   36.7917957   54.2744885 
   36.7948951   54.2683062 
   36.7988172   54.2670729 
   36.8002709   54.2687227 
   36.8016149   54.2736077 
   36.8061405   54.2733194 
   36.8085267   54.2738319 
   36.8111049   54.2719421 
   36.8124489   54.2718299 
   36.8144785   54.2742804 
   36.818867    54.2789405 
   36.8297558   54.2790526 
   36.8338974   54.279549  
   36.8355431   54.2816947 
   36.8363659   54.2829117 
   36.8371613   54.2831679 
   36.8379567   54.2836162 
   36.8385327   54.2843367 
   36.838615    54.284705  
   36.8391361   54.2849932 
   36.8391361   54.2853134 
   36.8395201   54.2855215 
   36.8403155   54.2854255 
   36.8407543   54.2856816 
   36.8406995   54.2864021 
   36.8405075   54.2869945 
   36.8454445   54.2880832 
   36.8493118   54.2845128 
   36.8523289   54.2858577 
   36.8600909   54.2858898 
   36.8639582   54.2861299 
   36.8651102   54.2865302 
   36.8716655   54.2823512 
   36.8842548   54.2792128 
   36.8863942   54.2814385 
   36.8884513   54.281967  
   36.8891918   54.28336   
   36.8918249   54.2860819 
   36.8965973   54.2883073 
   36.899395    54.2886435 
   36.9020006   54.2896041 
   36.906197    54.2889797 
   36.9125877   54.2897482 
   36.9189784   54.2908688 
   36.9262468   54.2917333 
   36.9346671   54.2913491 
   36.9347219   54.2887876 
   36.9454462   54.289428  
   36.9514803   54.2916532 
   36.9556768   54.2915412 
   36.9584744   54.2906127 
   36.9630242   54.2929019 
   36.9636391   54.2927437 
   36.9668461   54.2916582 
   36.9688449   54.2937239 
   36.9723472   54.2974099 
   36.9740278   54.2989156 
   36.9744802   54.2993251 
   36.974671    54.2994783 
   36.974841    54.2996281 
   36.9744695   54.2997508 
   36.9760156   54.3013936 
   36.9759774   54.3014473 
   36.9757495   54.3025958 
   36.971817    54.3048342 
   36.9702267   54.30541   
   36.9695248   54.3051071 
   36.9671208   54.3053926 
   36.965091    54.3062938 
   36.9637637   54.3073027 
   36.9651889   54.3079403 
   36.9667216   54.3083511 
   36.9671393   54.3085459 
   36.9686724   54.3093571 
   36.9687005   54.3116314 
   36.9679497   54.3133664 
   36.9727812   54.3153494 
   36.9734615   54.3172557 
   36.9702245   54.3193813 
   36.9681089   54.320786  
   36.9673992   54.3214505 
   36.9674791   54.3224709 
   36.9676765   54.3228281 
   36.967795    54.3249661 
   36.96821     54.3259897 
   36.9672835   54.3269871 
   36.9663347   54.3276645 
   36.968598    54.3289726 
   36.9677615   54.3307517 
   36.9697789   54.3317325 
   36.9674342   54.3328164 
   36.9625971   54.3314585 
   36.9606234   54.3322793 
   36.9603886   54.3335699 
   36.9584732   54.3332663 
   36.9564932   54.334567  
   36.9552532   54.3350966 
   36.9526255   54.3339572 
   36.9495085   54.3337212 
   36.9436388   54.3370784 
   36.9417389   54.3386992 
   36.9405926   54.3398471 
   36.9416868   54.3401748 
   36.9420365   54.3403184 
   36.9425841   54.3405567 
   36.9431216   54.3408427 
   36.9432524   54.3409416 
   36.9440585   54.341225  
   36.9449238   54.3415746 
   36.9458789   54.3421173 
   36.9457323   54.3421902 
   36.9450132   54.3422521 
   36.9437992   54.3422927 
   36.9424578   54.3423028 
   36.9416846   54.342285  
   36.9410825   54.3422402 
   36.9400912   54.3421809 
   36.9399177   54.3421217 
   36.9385951   54.3419461 
   36.9382673   54.3419284 
   36.937624    54.3419732 
   36.9370144   54.3419303 
   36.9360357   54.34182   
   36.9352723   54.3417764 
   36.9343811   54.3418026 
   36.932827    54.3420382 
   36.9324372   54.3421434 
   36.9319217   54.3424302 
   36.9311418   54.3429833 
   36.9335691   54.3444346 
   36.9363761   54.3456155 
   36.9366933   54.3457489 
   36.9370055   54.3458885 
   36.94009     54.3472584 
   36.9385392   54.3486835 
   36.9384157   54.3491202 
   36.9393947   54.3493169 
   36.9384421   54.3498427 
   36.938001    54.3500343 
   36.9372453   54.3502665 
   36.9371884   54.3502925 
   36.9373051   54.3504759 
   36.9373849   54.3506399 
   36.9374841   54.3506935 
   36.9378169   54.3506682 
   36.9385371   54.3505611 
   36.9386462   54.3505597 
   36.9387626   54.3505963 
   36.9393802   54.3508791 
   36.9398785   54.3511414 
   36.9399596   54.3511548 
   36.9401082   54.3510989 
   36.9404545   54.3509169 
   36.9407683   54.350801  
   36.9409999   54.3507232 
   36.9417292   54.3509222 
   36.941814    54.3507012 
   36.9419841   54.3503868 
   36.9421305   54.3500234 
   36.9430377   54.3500344 
   36.9436217   54.3505551 
   36.9441321   54.3510435 
   36.9448518   54.3513484 
   36.9455663   54.3516661 
   36.9462915   54.3520098 
   36.9470702   54.3523316 
   36.9478328   54.3526584 
   36.9475667   54.3528848 
   36.9473591   54.3530189 
   36.9471003   54.3532309 
   36.9474323   54.3535104 
   36.9479994   54.3540346 
   36.9487026   54.3546289 
   36.950238    54.356062  
   36.9518024   54.3574848 
   36.9519198   54.357631  
   36.9523567   54.3580042 
   36.9526616   54.3582442 
   36.9529868   54.3583901 
   36.9532517   54.3585242 
   36.9534406   54.358643  
   36.9543642   54.359108  
   36.9532175   54.360648  
   36.9523901   54.3620407 
   36.9521845   54.3624358 
   36.9593877   54.363503  
   36.9590489   54.3648747 
   36.9592766   54.3649849 
   36.9618627   54.3656392 
   36.961918    54.3659006 
   36.9619111   54.366179  
   36.9623144   54.3665528 
   36.9635469   54.367568  
   36.9692024   54.3732684 
   36.9742241   54.3783399 
   36.9807269   54.38411   
   36.9806973   54.3843455 
   36.9806119   54.3845337 
   36.9804452   54.3848147 
   36.980188    54.3851153 
   36.9801305   54.3852111 
   36.9802286   54.3853922 
   36.9804349   54.3857071 
   36.9806953   54.3859846 
   36.982085    54.3870403 
   36.9827306   54.3874718 
   36.9836009   54.3881097 
   36.9837341   54.388226  
   36.984129    54.3884777 
   36.9847261   54.3889059 
   36.9835245   54.3906576 
   36.9833033   54.3910094 
   36.9832236   54.3911614 
   36.9831136   54.3919261 
   36.9830286   54.3924004 
   36.9828751   54.3930613 
   36.9826173   54.3946315 
   36.9800019   54.3959668 
   36.9787126   54.3960748 
   36.9774336   54.397028  
   36.977303    54.3971484 
   36.9765032   54.3977683 
   36.9756939   54.3984414 
   36.974098    54.3997034 
   36.9727502   54.4007398 
   36.9708294   54.4024254 
   36.9700328   54.4031034 
   36.9690975   54.4031097 
   36.9684954   54.4042379 
   36.9683042   54.4045196 
   36.96833     54.4045814 
   36.9683213   54.4046643 
   36.9682058   54.4047572 
   36.96808     54.4048343 
   36.9681054   54.4049559 
   36.9682278   54.4050259 
   36.9683698   54.4050763 
   36.9685211   54.4050696 
   36.9688997   54.4048306 
   36.9690005   54.4048314 
   36.9691745   54.4048802 
   36.9693706   54.4049825 
   36.9695402   54.4052367 
   36.9696258   54.4054416 
   36.9697032   54.4060982 
   36.9701356   54.4063852 
   36.9706046   54.4064554 
   36.9716874   54.406344  
   36.9722885   54.4067874 
   36.9718658   54.4071645 
   36.9705075   54.408054  
   36.9699171   54.4087421 
   36.9675206   54.4103592 
   36.9665944   54.4110621 
   36.965682    54.4117688 
   36.9644044   54.413739  
   36.9639744   54.4144414 
   36.9633888   54.4154539 
   36.9629387   54.4161506 
   36.9617333   54.4164782 
   36.9587643   54.4201684 
   36.9735619   54.4298573 
   36.9695345   54.4329011 
   36.9651325   54.4333694 
   36.958951    54.4349615 
   36.9500534   54.4358979 
   36.9430448   54.4355019 
   36.9419048   54.4417982 
   36.9435906   54.4439524 
   36.9422793   54.4449826 
   36.9411059   54.4467659 
   36.9339879   54.4458292 
   36.9217187   54.4436749 
   36.9140388   54.4417079 
   36.9072019   54.4383127 
   36.9015826   54.434098  
   36.894315    54.4296791 
   36.8932876   54.4279033 
   36.8918841   54.4276549 
   36.8813504   54.434093  
   36.879756    54.437961  
   36.8784139   54.4433799 
   36.8747612   54.4431793 
   36.8744112   54.4478805 
   36.8764318   54.4479672 
   36.8763265   54.4494173 
   36.8787181   54.4495852 
   36.8788494   54.4518358 
   36.8786881   54.4533434 
   36.8794584   54.4539591 
   36.882249    54.4554436 
   36.8843236   54.4567665 
   36.8777222   54.460009  
   36.8721518   54.4624364 
   36.8711058   54.4632362 
   36.8701067   54.4663264 
   36.8697875   54.4681981 
   36.8694532   54.4693201 
   36.8692688   54.4701326 
   36.8671455   54.4702381 
   36.8649605   54.4693833 
   36.8555889   54.4790339 
   36.8630868   54.4820506 
   36.8647727   54.4818633 
   36.8668331   54.4826126 
   36.8704874   54.4832041 
   36.8705919   54.4832335 
   36.8706637   54.4833089 
   36.8710047   54.4833716 
   36.8712062   54.4835858 
   36.8715703   54.4836608 
   36.8721481   54.4837999 
   36.872567    54.4838524 
   36.8728091   54.4838663 
   36.8730586   54.4837732 
   36.8731996   54.4837452 
   36.8733196   54.4837738 
   36.8735435   54.4839493 
   36.8737508   54.4839714 
   36.8741201   54.4838705 
   36.8742851   54.4838502 
   36.8746218   54.4838602 
   36.8747701   54.4839071 
   36.8749216   54.4839063 
   36.8750388   54.4838966 
   36.8753796   54.4841085 
   36.8757307   54.4841391 
   36.8760359   54.4842838 
   36.8763136   54.4843982 
   36.8765967   54.4844697 
   36.876712    54.4845608 
   36.8768343   54.4845427 
   36.8769526   54.484447  
   36.8770443   54.4844073 
   36.8770955   54.4844413 
   36.8771494   54.4846404 
   36.8772772   54.4846011 
   36.8774018   54.4845867 
   36.8775687   54.484618  
   36.877773    54.4846352 
   36.8779441   54.4847272 
   36.8780753   54.4849077 
   36.8782676   54.4849925 
   36.878441    54.4850298 
   36.878601    54.4849751 
   36.8787612   54.484875  
   36.8789209   54.4848346 
   36.879244    54.4845942 
   36.8794797   54.4844889 
   36.8797716   54.4844805 
   36.8800719   54.4845159 
   36.8804213   54.4845346 
   36.8806124   54.484654  
   36.8808512   54.4848743 
   36.8810541   54.4849323 
   36.8812373   54.484946  
   36.8818187   54.4851877 
   36.8782996   54.4870644 
   36.8770491   54.489534  
   36.8742481   54.4918001 
   36.8716411   54.4931514 
   36.8776123   54.5004254 
   36.8774574   54.5032419 
   36.8782391   54.5039362 
   36.8800386   54.5037621 
   36.8821191   54.5039529 
   36.8849718   54.5040748 
   36.8885799   54.5042172 
   36.8916333   54.5057363 
   36.8933061   54.5066695 
   36.8975509   54.5063204 
   36.8974787   54.509052  
   36.8983959   54.5098788 
   36.902692    54.5105588 
   36.9039618   54.5107424 
   36.9049263   54.5110014 
   36.9051399   54.5109617 
   36.907195    54.5090559 
   36.9113385   54.5072032 
   36.9121045   54.5070473 
   36.91576     54.5063031 
   36.9171531   54.5064412 
   36.9189988   54.5096254 
   36.9234246   54.5097108 
   36.9239687   54.5093057 
   36.9248233   54.5087727 
   36.9249772   54.5080126 
   36.9266868   54.5078574 
   36.9296884   54.5071903 
   36.9317857   54.5064569 
   36.9335269   54.5064682 
   36.9345306   54.5074442 
   36.9366965   54.5084948 
   36.9419259   54.5097501 
   36.9408495   54.5112897 
   36.9397467   54.5128249 
   36.9375228   54.5165681 
   36.9378385   54.5169271 
   36.9378831   54.5171587 
   36.9385277   54.5175995 
   36.9385321   54.5177152 
   36.9387695   54.5176989 
   36.9390233   54.5177047 
   36.9394161   54.5177234 
   36.9397408   54.5177787 
   36.9401327   54.5178665 
   36.9404855   54.5179644 
   36.9411139   54.5182004 
   36.9412734   54.5182344 
   36.9414176   54.5182723 
   36.9420123   54.5185092 
   36.9425901   54.5188432 
   36.9440734   54.5201769 
   36.9445396   54.5207447 
   36.9439801   54.5208309 
   36.9423777   54.5208061 
   36.9410221   54.5211679 
   36.9403399   54.521068  
   36.9388113   54.521935  
   36.9374119   54.5238717 
   36.938435    54.5246097 
   36.9405858   54.5252327 
   36.94164     54.524633  
   36.9432804   54.525592  
   36.9416861   54.5268246 
   36.9429007   54.5272708 
   36.9433155   54.5269332 
   36.9453515   54.5276778 
   36.9479533   54.5273661 
   36.9483561   54.5270069 
   36.9505879   54.5261506 
   36.9498091   54.5256748 
   36.9504745   54.5252942 
   36.9506929   54.524822  
   36.9495249   54.5245952 
   36.9490185   54.5241845 
   36.9484821   54.5238015 
   36.9468935   54.5228393 
   36.9461861   54.5224736 
   36.9471618   54.522075  
   36.948118    54.5219399 
   36.9492119   54.521229  
   36.9506063   54.5213241 
   36.9513931   54.5221745 
   36.9524767   54.5225492 
   36.9531593   54.5231233 
   36.9544053   54.5239644 
   36.9554193   54.5240126 
   36.9564783   54.5240356 
   36.9569073   54.5241179 
   36.9573245   54.5242978 
   36.9574451   54.5243822 
   36.9575844   54.524525  
   36.9576282   54.524643  
   36.9576052   54.5247665 
   36.9588357   54.5240412 
   36.9594079   54.5235161 
   36.9616522   54.5220718 
   36.9623727   54.5235282 
   36.9626716   54.5243952 
   36.9625687   54.525613  
   36.9624395   54.5257373 
   36.9623082   54.52577   
   36.9616635   54.526115  
   36.9614845   54.5262288 
   36.9612473   54.5264952 
   36.9610057   54.526998  
   36.9609018   54.5274584 
   36.9608072   54.5276705 
   36.960711    54.5277858 
   36.960607    54.5278911 
   36.9604747   54.5279693 
   36.9603177   54.5280353 
   36.9600158   54.5280394 
   36.9598375   54.5280125 
   36.9597308   54.5279807 
   36.9595707   54.5279098 
   36.9594766   54.5278389 
   36.9593869   54.5277477 
   36.9567332   54.5273068 
   36.9557909   54.5274131 
   36.9552119   54.5280787 
   36.9527009   54.5295818 
   36.952274    54.5293206 
   36.9513221   54.5290968 
   36.9498407   54.5298069 
   36.9489241   54.5301715 
   36.9479422   54.5293477 
   36.947057    54.5292106 
   36.9464114   54.5290499 
   36.9450139   54.5291631 
   36.9433645   54.5288763 
   36.942882    54.5280158 
   36.9401397   54.5286644 
   36.9388703   54.5289362 
   36.9367371   54.5294099 
   36.934304    54.5299289 
   36.9340587   54.5300278 
   36.9339928   54.5300828 
   36.9309218   54.5318515 
   36.9294833   54.532679  
   36.9284586   54.5332199 
   36.9283678   54.5332846 
   36.9282692   54.5333476 
   36.9278938   54.5335265 
   36.9277427   54.533611  
   36.928721    54.5349445 
   36.9289418   54.5352046 
   36.9342613   54.5371924 
   36.9326779   54.5381199 
   36.9301428   54.5394662 
   36.9290437   54.5400676 
   36.9282624   54.5405363 
   36.927462    54.5409978 
   36.9261507   54.5417778 
   36.922713    54.5437903 
   36.9192323   54.5449099 
   36.9174233   54.5442574 
   36.916211    54.5439776 
   36.9157108   54.544358  
   36.9150975   54.5446544 
   36.9149399   54.544758  
   36.9145124   54.545005  
   36.9140735   54.545262  
   36.9136528   54.5455689 
   36.9142389   54.5473878 
   36.9161631   54.5484793 
   36.9162589   54.5485344 
   36.9164631   54.5486687 
   36.9179155   54.5492661 
   36.9189633   54.5494919 
   36.9211773   54.5499197 
   36.9213041   54.549943  
   36.9220827   54.5500731 
   36.9234829   54.5492978 
   36.9245627   54.5494827 
   36.9258984   54.5484031 
   36.9267096   54.548339  
   36.9273495   54.5475003 
   36.9283153   54.5480223 
   36.9284496   54.5483398 
   36.9285589   54.5485096 
   36.9290958   54.5488686 
   36.9291998   54.5490219 
   36.9295531   54.5492241 
   36.9294281   54.5493264 
   36.9294017   54.5493758 
   36.9293938   54.549415  
   36.9293505   54.5494958 
   36.9293369   54.5495363 
   36.9292854   54.5495856 
   36.9291746   54.5496217 
   36.9290578   54.5496272 
   36.9289073   54.5496119 
   36.9286319   54.5495487 
   36.9285218   54.549479  
   36.9284128   54.5494189 
   36.928368    54.5494022 
   36.9282542   54.5494167 
   36.9282407   54.5494549 
   36.928237    54.5495366 
   36.9282087   54.5495846 
   36.9281193   54.5496166 
   36.9280078   54.5496217 
   36.9278516   54.5496022 
   36.9276311   54.5495222 
   36.9275463   54.5495041 
   36.9274452   54.5495012 
   36.9274007   54.5495373 
   36.9273787   54.5495732 
   36.9273591   54.549612  
   36.9273903   54.5496527 
   36.9275149   54.5497094 
   36.9276149   54.5498033 
   36.9276322   54.5498501 
   36.927626    54.5499286 
   36.9276053   54.5499529 
   36.9275085   54.5499815 
   36.9273043   54.5500012 
   36.9272459   54.5500148 
   36.9271396   54.5500915 
   36.9270576   54.5501722 
   36.9269914   54.5501993 
   36.9269069   54.5502177 
   36.9268235   54.5502229 
   36.9267088   54.5501849 
   36.9266366   54.5501169 
   36.9266012   54.550077  
   36.9265291   54.5500376 
   36.9263989   54.5500308 
   36.9262813   54.5500632 
   36.9259664   54.5503654 
   36.9258062   54.5504624 
   36.9256372   54.5505555 
   36.9255121   54.5506095 
   36.9252786   54.5506794 
   36.9251137   54.550773  
   36.9250864   54.5508416 
   36.9250879   54.5508863 
   36.9251134   54.5509356 
   36.9251497   54.5509847 
   36.9251361   54.5510379 
   36.9251144   54.551061  
   36.9250404   54.5510837 
   36.9249217   54.551103  
   36.9248285   54.5511125 
   36.9245864   54.5511695 
   36.9244542   54.5511971 
   36.9243484   54.5512636 
   36.9243412   54.5513101 
   36.9243854   54.5514485 
   36.9243716   54.5514974 
   36.9243425   54.5515335 
   36.9241921   54.551551  
   36.9239167   54.5515649 
   36.9236627   54.5515638 
   36.9234172   54.5515976 
   36.9230947   54.5516714 
   36.922823    54.5517597 
   36.9225396   54.5518818 
   36.9222049   54.5520445 
   36.9219504   54.5522202 
   36.9215124   54.5524209 
   36.9210887   54.5526006 
   36.9206716   54.5527728 
   36.9201091   54.5530345 
   36.919619    54.5533307 
   36.9193911   54.5534549 
   36.9189739   54.5536476 
   36.9188225   54.5537071 
   36.9185993   54.5538068 
   36.9182817   54.5538811 
   36.9180309   54.5539346 
   36.9177441   54.5539811 
   36.9174471   54.5539976 
   36.9169877   54.5539519 
   36.9167157   54.5539492 
   36.9162094   54.5540234 
   36.916014    54.5540234 
   36.9157619   54.5539867 
   36.913831    54.5543607 
   36.9132697   54.5558706 
   36.9122515   54.5564836 
   36.9117666   54.5571471 
   36.9114369   54.5574114 
   36.910868    54.55804   
   36.909944    54.558521  
   36.9096145   54.5590574 
   36.9057935   54.5612439 
   36.9056782   54.5612724 
   36.9055559   54.5612552 
   36.9037668   54.5601615 
   36.9024674   54.5592302 
   36.8989861   54.5611917 
   36.8973253   54.5620597 
   36.8972617   54.5621486 
   36.8960704   54.5628171 
   36.8949683   54.563494  
   36.8958362   54.564108  
   36.8965102   54.5646353 
   36.8972471   54.5653718 
   36.8988666   54.568121  
   36.8992593   54.5683234 
   36.9001563   54.5687028 
   36.9018872   54.5689614 
   36.9063286   54.5702347 
   36.9072837   54.57078   
   36.907531    54.5708643 
   36.9075746   54.570929  
   36.9075262   54.5709768 
   36.907371    54.5709177 
   36.9071965   54.5709037 
   36.9070461   54.5709515 
   36.9069007   54.5710751 
   36.9067067   54.5711426 
   36.9064837   54.571241  
   36.9062607   54.5713815 
   36.9060522   54.5716794 
   36.9059552   54.5717975 
   36.9057225   54.5718846 
   36.9056352   54.5719858 
   36.9056158   54.572132  
   36.9056013   54.5722978 
   36.9054413   54.5724214 
   36.9054267   54.5725367 
   36.9055091   54.5726603 
   36.9055285   54.5727756 
   36.9054073   54.572843  
   36.9050825   54.572843  
   36.9048691   54.5728739 
   36.9045125   54.5729876 
   36.9050425   54.5733887 
   36.9052139   54.5735954 
   36.9055465   54.5737822 
   36.9060519   54.5742092 
   36.9064976   54.5745649 
   36.9070462   54.5750637 
   36.9074267   54.5753419 
   36.907507    54.5754739 
   36.9075002   54.5759071 
   36.9073904   54.5762767 
   36.9075824   54.5768808 
   36.9083436   54.5788282 
   36.9083232   54.5800669 
   36.9095267   54.5801967 
   36.9103316   54.5802599 
   36.9107026   54.5803173 
   36.9116214   54.5801146 
   36.912458    54.5798961 
   36.9146453   54.5794271 
   36.9147825   54.5794688 
   36.9149093   54.5795861 
   36.9150533   54.5798166 
   36.9150328   54.5799636 
   36.9148716   54.5801524 
   36.9148065   54.5802954 
   36.9148819   54.5804147 
   36.9150396   54.5804961 
   36.915139    54.5805935 
   36.915115    54.5807227 
   36.9151082   54.5808677 
   36.9152248   54.5809472 
   36.9155402   54.5810187 
   36.9156499   54.5811499 
   36.9156362   54.5813565 
   36.9158419   54.5815274 
   36.9160407   54.5815671 
   36.9161573   54.5816545 
   36.916315    54.5819168 
   36.916315    54.5821791 
   36.916123    54.5828427 
   36.916219    54.5830851 
   36.9161504   54.5832123 
   36.9159516   54.5833752 
   36.9162602   54.5834745 
   36.9167196   54.5837368 
   36.9167813   54.5838997 
   36.9171357   54.5841308 
   36.917366    54.5844975 
   36.9175478   54.5846661 
   36.9177563   54.5847026 
   36.9180545   54.584777  
   36.9182751   54.5848852 
   36.9183551   54.5849723 
   36.9183575   54.5851044 
   36.9184351   54.5851957 
   36.9186485   54.5853165 
   36.9189141   54.5856527 
   36.9189004   54.5857878 
   36.9186261   54.5862209 
   36.9190649   54.5869798 
   36.9190992   54.5871904 
   36.9192089   54.58725   
   36.9192775   54.587393  
   36.9191746   54.5875678 
   36.9191952   54.5877387 
   36.9194421   54.5880089 
   36.9196683   54.5880804 
   36.9197781   54.5881837 
   36.9197643   54.5884579 
   36.9198741   54.5886049 
   36.9201278   54.5886168 
   36.9202855   54.5886724 
   36.920354    54.5888035 
   36.9202375   54.5889903 
   36.9202443   54.5891889 
   36.9205803   54.5893717 
   36.9207344   54.5893737 
   36.9325764   54.5907659 
   36.9328573   54.5916089 
   36.9421295   54.5922646 
   36.9431596   54.5952617 
   36.9462503   54.5968539 
   36.949622    54.5970413 
   36.951495    54.6001321 
   36.9605798   54.6005069 
   36.9667612   54.6020055 
   36.9705074   54.6048387 
   36.9839006   54.604277  
   36.9839007   54.6025911 
   36.9906441   54.6024039 
   36.9913      54.5976273 
   36.9929858   54.5972527 
   36.9962638   54.5989386 
   37.0019771   54.5950987 
   37.0022582   54.5936938 
   37.0045061   54.591329  
   37.0114113   54.5905926 
   37.0191425   54.5929513 
   37.025848    54.5929635 
   37.0270944   54.5936957 
   37.0312927   54.5848435 
   37.0367248   54.5859675 
   37.039441    54.5840007 
   37.048994    54.5871853 
   37.0536771   54.5843756 
   37.0599524   54.581004  
   37.0564871   54.5781005 
   37.0535838   54.5770702 
   37.0496845   54.5742945 
   37.0503191   54.5728168 
   37.0531967   54.5704652 
   37.0607023   54.5666508 
   37.0602341   54.5640283 
   37.0554577   54.5611951 
   37.0627632   54.5577298 
   37.0620978   54.5568315 
   37.0621027   54.556461  
   37.0617842   54.5562202 
   37.0617916   54.55602   
   37.0620063   54.5559181 
   37.0626315   54.5559248 
   37.0628577   54.55596   
   37.0657916   54.552703  
   37.0680192   54.5535995 
   37.0701406   54.5537059 
   37.0713143   54.5534543 
   37.0719008   54.553318  
   37.0723654   54.5531016 
   37.0730489   54.5526653 
   37.0725547   54.5511739 
   37.0740372   54.550871  
   37.076344    54.5498627 
   37.0780204   54.5469228 
   37.0781468   54.5448723 
   37.0935368   54.5467186 
   37.1048616   54.5487518 
   37.1220326   54.5509004 
   37.1271824   54.5538871 
   37.1340489   54.5592626 
   37.141602    54.567225  
   37.1450777   54.5770532 
   37.1491436   54.5887799 
   37.1490792   54.5911234 
   37.1488118   54.6008489 
   37.1478232   54.6077324 
   37.1512123   54.6128803 
   37.1584248   54.6187438 
   37.1765504   54.628128  
   37.1926511   54.6350785 
   37.1984876   54.6413995 
   37.2009998   54.6493458 
   37.2040867   54.6644396 
   37.2114438   54.6778593 
   37.2123265   54.6791308 
   37.2177147   54.6833277 
   37.2281341   54.6944838 
   37.2313067   54.6969302 
   37.2420545   54.7046654 
   37.246346    54.7090293 
   37.2453161   54.7121035 
   37.2331281   54.7179539 
   37.2317269   54.7183032 
   37.2257161   54.7198016 
   37.2147603   54.7212258 
   37.209464    54.7218437 
   37.2069207   54.7221404 
   37.2078095   54.7237746 
   37.2063351   54.724514  
   37.2049275   54.7249284 
   37.2010428   54.7265416 
   37.1976276   54.7286716 
   37.196203    54.7307789 
   37.1959883   54.7325706 
   37.1959102   54.7341481 
   37.1968275   54.7355453 
   37.2034042   54.7391507 
   37.2103518   54.7412124 
   37.2061106   54.7449993 
   37.2264027   54.7507238 
   37.2316242   54.7521819 
   37.2370457   54.7536958 
   37.2415089   54.7570639 
   37.2445988   54.7620164 
   37.2439122   54.7669683 
   37.2433925   54.7684672 
   37.2421955   54.7719196 
   37.238419    54.7832063 
   37.2336125   54.790927  
   37.228806    54.8004273 
   37.2325825   54.8079468 
   37.2415089   54.8172453 
   37.2565636   54.83028   
   37.269308    54.8416291 
   37.2720347   54.8397773 
   37.2782822   54.8458072 
   37.2771994   54.8471718 
   37.2750186   54.8475744 
   37.2709856   54.8470811 
   37.269469    54.8470663 
   37.2680173   54.8473892 
   37.2662131   54.8479082 
   37.2651704   54.8483731 
   37.2653729   54.8485385 
   37.2650889   54.8487213 
   37.2635148   54.8493163 
   37.2620951   54.8502141 
   37.2615385   54.8506063 
   37.2613686   54.8507958 
   37.2612153   54.8505535 
   37.2600157   54.8508641 
   37.258986    54.850795  
   37.2567929   54.8508321 
   37.2548446   54.850695  
   37.2538878   54.8504655 
   37.2526608   54.8498801 
   37.2511864   54.8485565 
   37.2492116   54.8479223 
   37.247445    54.8476518 
   37.2453766   54.8470478 
   37.2399665   54.845976  
   37.2357282   54.845297  
   37.2326564   54.8453915 
   37.2301132   54.8465603 
   37.2293355   54.8472863 
   37.2289832   54.8486785 
   37.2294899   54.8492815 
   37.2303989   54.8498009 
   37.2312475   54.8500507 
   37.2319648   54.8501225 
   37.2331078   54.8500138 
   37.2335594   54.8502816 
   37.2333342   54.8507071 
   37.231898    54.8510998 
   37.2302213   54.8514699 
   37.229791    54.8514396 
   37.2288918   54.850374  
   37.2280726   54.8499092 
   37.2274033   54.8497926 
   37.2266372   54.8498845 
   37.2266471   54.8503903 
   37.2271883   54.8515455 
   37.2272667   54.8519492 
   37.2267264   54.852564  
   37.2255261   54.8529916 
   37.223933    54.8532033 
   37.223342    54.8531716 
   37.222198    54.8528489 
   37.2210262   54.8527372 
   37.2198993   54.8524384 
   37.2192929   54.8515059 
   37.2202197   54.8505674 
   37.219661    54.8497735 
   37.2184558   54.8496159 
   37.217565    54.8502396 
   37.2174794   54.8515029 
   37.2168507   54.8521762 
   37.215607    54.85279   
   37.2139882   54.853182  
   37.2116835   54.8529933 
   37.2097399   54.8534616 
   37.2089264   54.8531343 
   37.2086873   54.8514493 
   37.2051768   54.848997  
   37.2037021   54.8487663 
   37.200069    54.8495407 
   37.1967092   54.8512712 
   37.1966838   54.8517783 
   37.1982963   54.8520237 
   37.200809    54.8517501 
   37.2014346   54.8523463 
   37.2011002   54.8534529 
   37.1975325   54.8558422 
   37.1957058   54.8576239 
   37.1946579   54.8575703 
   37.194073    54.8564247 
   37.1932558   54.8554918 
   37.191663    54.8557776 
   37.1903405   54.8558193 
   37.1900366   54.856257  
   37.1907017   54.857191  
   37.1907233   54.8578153 
   37.1896703   54.8582024 
   37.1888579   54.857984  
   37.1884183   54.8573708 
   37.1881007   54.857373  
   37.1869782   54.8582053 
   37.1850208   54.8596556 
   37.1853896   54.8610009 
   37.1852927   54.861123  
   37.185275    54.8611884 
   37.1848458   54.8613722 
   37.1840164   54.8615536 
   37.18032     54.862507  
   37.17852     54.8629562 
   37.1767177   54.8634288 
   37.1725913   54.8641139 
   37.1721957   54.8641716 
   37.1717537   54.8642871 
   37.1689708   54.8652873 
   37.1654877   54.8665328 
   37.1652849   54.8665797 
   37.1664456   54.867909  
   37.1677062   54.8694109 
   37.1709876   54.8728448 
   37.1738928   54.87601   
   37.1759095   54.8785471 
   37.169302    54.879562  
   37.1694529   54.8813887 
   37.1694088   54.8834225 
   37.1702808   54.8849581 
   37.1718811   54.8863304 
   37.1722042   54.8866508 
   37.1728254   54.8878992 
   37.1741524   54.8895129 
   37.1736225   54.8899834 
   37.1739015   54.890489  
   37.173138    54.8918642 
   37.1693629   54.8935754 
   37.1681953   54.8938508 
   37.1675986   54.8945666 
   37.1671501   54.8951087 
   37.1678886   54.895816  
   37.168088    54.8967841 
   37.1687864   54.8980225 
   37.1713198   54.9001465 
   37.1729908   54.9015666 
   37.1743414   54.9027225 
   37.1757387   54.9028047 
   37.1763194   54.9045731 
   37.1760899   54.9059582 
   37.1763472   54.9072392 
   37.1767851   54.9083122 
   37.1770045   54.9088499 
   37.178487    54.9100699 
   37.1802466   54.9115179 
   37.1834707   54.9147714 
   37.1856331   54.916836  
   37.1929945   54.9232556 
   37.1881449   54.9288628 
   37.1851242   54.9311151 
   37.1808045   54.9332783 
   37.1732577   54.9367229 
   37.1725073   54.9373548 
   37.173124    54.9377014 
   37.1714746   54.9383842 
   37.1725512   54.9393196 
   37.1735991   54.9396547 
   37.1739039   54.9403702 
   37.1733947   54.940628  
   37.1715221   54.9438206 
   37.1706391   54.9439006 
   37.1697681   54.9449661 
   37.1684899   54.9459918 
   37.1668941   54.9462976 
   37.1668806   54.9474908 
   37.1676141   54.9492957 
   37.1604962   54.9504442 
   37.1610946   54.9518192 
   37.1637239   54.9527908 
   37.1644668   54.9536396 
   37.16635     54.9532813 
   37.1687408   54.956087  
   37.1672134   54.9574962 
   37.1623854   54.9586912 
   37.1581582   54.9576563 
   37.1577291   54.9579767 
   37.161291    54.962498  
   37.1612696   54.9650726 
   37.1643595   54.9654299 
   37.1665696   54.9653313 
   37.1677283   54.9650603 
   37.166355    54.9640502 
   37.1660117   54.9634096 
   37.1640805   54.9631017 
   37.1623425   54.9609088 
   37.1621279   54.9597631 
   37.1629647   54.9597631 
   37.1630935   54.9592703 
   37.169917    54.9573483 
   37.1736077   54.9601943 
   37.1717409   54.9681521 
   37.1680287   54.9686818 
   37.1674708   54.9700243 
   37.1647028   54.9710343 
   37.1637158   54.9706894 
   37.1624068   54.97149   
   37.1641449   54.9725122 
   37.1613125   54.9753695 
   37.1521501   54.9801721 
   37.1473006   54.9813172 
   37.1524939   54.9827726 
   37.154961    54.9851096 
   37.1389536   54.987892  
   37.1342978   54.9913977 
   37.1566435   55.0004043 
   37.1513668   54.9996928 
   37.1498994   55.0001591 
   37.127839    55.0185246 
   37.1244159   55.0198404 
   37.1223176   55.0233021 
   37.1196288   55.021913  
   37.113193    55.0235108 
   37.1078361   55.0261301 
   37.0958678   55.0324086 
   37.094554    55.0366414 
   37.0962944   55.0412665 
   37.0968052   55.0442306 
   37.0935577   55.0483614 
   37.0838062   55.05655   
   37.0844975   55.057676  
   37.0860388   55.0580101 
   37.0874784   55.0579162 
   37.0884033   55.0582002 
   37.0928318   55.0589095 
   37.0985077   55.0604026 
   37.1033858   55.0616009 
   37.1110239   55.0615102 
   37.1152765   55.0626958 
   37.1164727   55.0632215 
   37.1168642   55.0633007 
   37.1181588   55.0634876 
   37.1168692   55.0648455 
   37.115261    55.0656157 
   37.1144477   55.0658957 
   37.1134224   55.0660505 
   37.1128446   55.0666653 
   37.1126303   55.0668898 
   37.1119883   55.0682596 
   37.1121139   55.0686187 
   37.1128569   55.0690586 
   37.1141444   55.0694236 
   37.1153759   55.0703345 
   37.1167164   55.0708586 
   37.1187618   55.0715427 
   37.1201821   55.073376  
   37.1203819   55.073933  
   37.1195996   55.0743447 
   37.1187661   55.0744578 
   37.1177454   55.074334  
   37.1162954   55.0744025 
   37.114672    55.0748009 
   37.1136216   55.0753715 
   37.1117519   55.0765405 
   37.1112917   55.0771716 
   37.1109983   55.0793327 
   37.1105718   55.0807769 
   37.1106994   55.0831224 
   37.1106713   55.0852217 
   37.1113145   55.0874868 
   37.1124443   55.0888147 
   37.1130401   55.0894024 
   37.1146271   55.0906231 
   37.1168534   55.0927838 
   37.1171475   55.0933516 
   37.1169619   55.0957619 
   37.1170343   55.096302  
   37.1164767   55.097209  
   37.1163222   55.098063  
   37.1157285   55.0987393 
   37.1145589   55.0992044 
   37.1111816   55.0995079 
   37.1088121   55.0996753 
   37.1077194   55.0997278 
   37.1058326   55.0991698 
   37.1049432   55.098827  
   37.1039939   55.0984848 
   37.1024446   55.097777  
   37.1011573   55.0972913 
   37.099973    55.0971138 
   37.098692    55.097106  
   37.098062    55.0974638 
   37.097323    55.0991579 
   37.0974964   55.0996606 
   37.0976371   55.1000884 
   37.0972311   55.1011538 
   37.0961728   55.1023036 
   37.095221    55.1033963 
   37.0949718   55.1038557 
   37.095019    55.1045224 
   37.0953042   55.1049621 
   37.0952328   55.1054795 
   37.0946027   55.1058839 
   37.091905    55.106354  
   37.0898286   55.1067329 
   37.0860744   55.1072112 
   37.0845487   55.107262  
   37.0832332   55.1074471 
   37.083139    55.1077254 
   37.0831761   55.109318  
   37.0841159   55.1093288 
   37.0841003   55.1097757 
   37.0831078   55.1099879 
   37.0827097   55.1104908 
   37.0814368   55.1105781 
   37.0814344   55.1112014 
   37.0811933   55.111245  
   37.0787618   55.1116843 
   37.0763811   55.1122467 
   37.0746765   55.1126839 
   37.0730181   55.1131182 
   37.0728669   55.1131577 
   37.0727592   55.1132261 
   37.0721684   55.1136004 
   37.0714379   55.1140634 
   37.070936    55.1143814 
   37.0667599   55.113375  
   37.0607971   55.1120472 
   37.0572372   55.109546  
   37.0553465   55.1105532 
   37.0550941   55.1108852 
   37.0524268   55.1124107 
   37.0518839   55.1131729 
   37.0518245   55.1133711 
   37.0536622   55.1161526 
   37.0524831   55.1180237 
   37.0550492   55.1212946 
   37.0552515   55.1213955 
   37.0556166   55.121646  
   37.0573977   55.1229282 
   37.0573582   55.1230755 
   37.0561781   55.1239303 
   37.0549867   55.1248806 
   37.055394    55.1250713 
   37.0551134   55.1252505 
   37.0560879   55.1258475 
   37.0560202   55.1258787 
   37.0566371   55.1261393 
   37.0567632   55.1260851 
   37.057184    55.1259033 
   37.0598745   55.1271684 
   37.0556091   55.1281321 
   37.0416623   55.1338717 
   37.0410969   55.1354736 
   37.0432582   55.137156  
   37.0430002   55.1378448 
   37.0450029   55.1422006 
   37.04189     55.1426406 
   37.0393655   55.1423777 
   37.0378396   55.1460157 
   37.0325737   55.147703  
   37.0322169   55.1481244 
   37.0309138   55.148647  
   37.0279371   55.1501468 
   37.0274341   55.1526132 
   37.0232266   55.1555999 
   37.0208478   55.1560649 
   37.0181058   55.1575885 
   37.0152269   55.1599986 
   37.0185474   55.1601772 
   37.0198904   55.1611199 
   37.0198202   55.167195  
   37.0192303   55.1689537 
   37.0194652   55.1714944 
   37.0192618   55.1729636 
   37.0184235   55.1741781 
   37.0183986   55.1771845 
   37.0153315   55.1761794 
   37.0152477   55.183382  
   37.0106727   55.1831993 
   37.0084864   55.1827456 
   37.0069467   55.1808962 
   37.005765    55.1805064 
   37.0027341   55.1812719 
   36.9988575   55.1805468 
   36.9971491   55.1793938 
   36.994939    55.1820743 
   36.9915128   55.1824866 
   36.9904592   55.1813609 
   36.9854969   55.1803889 
   36.9836763   55.1811592 
   36.9829384   55.1803549 
   36.9818157   55.182364  
   36.9818028   55.183191  
   36.9813978   55.1847233 
   36.981452    55.1855115 
   36.9811941   55.1859444 
   36.9807657   55.1859338 
   36.9799131   55.1860511 
   36.9767334   55.1886165 
   36.9757102   55.1895425 
   36.9752051   55.1903448 
   36.9750289   55.1910523 
   36.9738279   55.1920271 
   36.971173    55.1954763 
   36.9690566   55.19847   
   36.9670967   55.1997366 
   36.9666855   55.200997  
   36.9663012   55.2025022 
   36.9662318   55.203586  
   36.9658661   55.2042117 
   36.9663044   55.2053499 
   36.9657007   55.205819  
   36.9634592   55.2075608 
   36.9616364   55.2089772 
   36.9627733   55.2096038 
   36.9629503   55.2099642 
   36.9640409   55.2109361 
   36.9646003   55.2119494 
   36.9630311   55.2122551 
   36.9621688   55.2123669 
   36.9623803   55.2134141 
   36.9622748   55.2141413 
   36.9600924   55.2149835 
   36.9572593   55.2166115 
   36.9539406   55.2175068 
   36.9514108   55.2181218 
   36.9488992   55.2182259 
   36.9477081   55.2181681 
   36.9475129   55.2183871 
   36.9484587   55.2203112 
   36.9469775   55.2221053 
   36.946344    55.2224439 
   36.9458962   55.2228045 
   36.9475583   55.2235781 
   36.9481228   55.2239704 
   36.9496407   55.2246956 
   36.9495966   55.2248671 
   36.9486662   55.2259452 
   36.9480559   55.2267812 
   36.9471149   55.2278003 
   36.9467311   55.2287351 
   36.9463168   55.2294515 
   36.9458295   55.2307229 
   36.9459449   55.231378  
   36.9470101   55.2321353 
   36.9477926   55.2328744 
   36.9480327   55.2333192 
   36.9488563   55.2343118 
   36.949031    55.2351167 
   36.9489336   55.2356424 
   36.9494871   55.236975  
   36.9501322   55.2382617 
   36.949803    55.2387804 
   36.9494394   55.2402372 
   36.9467799   55.2413959 
   36.9449046   55.2414142 
   36.9414029   55.2416936 
   36.9398902   55.2415049 
   36.9372397   55.2413907 
   36.9374636   55.2411893 
   36.9380466   55.2409336 
   36.9389696   55.2406665 
   36.9397267   55.2404183 
   36.940496    55.240187  
   36.9410386   55.2399754 
   36.9415304   55.2398048 
   36.942001    55.2395865 
   36.9421866   55.239473  
   36.9422315   55.2393071 
   36.9422011   55.2390861 
   36.9421272   55.2388425 
   36.941878    55.2386355 
   36.9417415   55.2384065 
   36.9414439   55.2380026 
   36.9410496   55.237664  
   36.9407403   55.2375365 
   36.9403094   55.2375858 
   36.9397513   55.2377279 
   36.9389468   55.238069  
   36.9386788   55.2384078 
   36.9386794   55.2386896 
   36.9385088   55.2387841 
   36.9382677   55.2387995 
   36.9378246   55.2387164 
   36.937463    55.2385647 
   36.937166    55.2382877 
   36.9368704   55.2379161 
   36.9366495   55.2375016 
   36.935685    55.2376228 
   36.9337853   55.2380445 
   36.9304717   55.2391904 
   36.9295486   55.2388994 
   36.92869     55.2383676 
   36.9280587   55.2380622 
   36.9261016   55.2349284 
   36.9232868   55.2351743 
   36.9212284   55.2352294 
   36.9191043   55.2358585 
   36.9156052   55.2351537 
   36.9133477   55.2347733 
   36.9112274   55.234157  
   36.9082458   55.2328325 
   36.907475    55.2324151 
   36.9065902   55.2324874 
   36.9047716   55.2325568 
   36.9038633   55.2325954 
   36.9029717   55.2327356 
   36.9018453   55.2330056 
   36.901311    55.2332757 
   36.9011326   55.2335595 
   36.9007908   55.2340027 
   36.9004424   55.2346725 
   36.9000831   55.2358793 
   36.8996633   55.2365303 
   36.8990649   55.236426  
   36.8971101   55.2356259 
   36.8933791   55.2336902 
   36.8856912   55.2298811 
   36.8840551   55.2303654 
   36.8823293   55.2295961 
   36.8780092   55.2303637 
   36.8771503   55.2298587 
   36.875221    55.2298993 
   36.8722344   55.2326734 
   36.8633369   55.2295853 
   36.8610981   55.2289407 
   36.8588925   55.2282643 
   36.856699    55.2276244 
   36.8558391   55.2273425 
   36.8574321   55.225922  
   36.8592099   55.2247784 
   36.8603024   55.2251852 
   36.8613473   55.2246787 
   36.8637742   55.2254651 
   36.8651448   55.2247945 
   36.8673432   55.223848  
   36.8691247   55.2213056 
   36.8699303   55.2207539 
   36.8707187   55.219623  
   36.8722187   55.2182589 
   36.8724971   55.2173526 
   36.8721011   55.2150762 
   36.8689084   55.2148874 
   36.8660149   55.2147163 
   36.8648609   55.2146113 
   36.8635117   55.2148839 
   36.8609246   55.2149039 
   36.8589644   55.2149629 
   36.8580954   55.2136073 
   36.8591294   55.2134153 
   36.8601788   55.2122691 
   36.8608216   55.2115669 
   36.8601451   55.2102767 
   36.8591613   55.2098086 
   36.8563519   55.2100961 
   36.8537451   55.2104706 
   36.8490413   55.211482  
   36.8493618   55.210606  
   36.8469714   55.2099053 
   36.8433928   55.2088171 
   36.8403393   55.2078885 
   36.8402231   55.2067518 
   36.840162    55.2052698 
   36.8395611   55.2048758 
   36.8371217   55.2033539 
   36.8355369   55.2023883 
   36.8344286   55.2013341 
   36.8329117   55.1992641 
   36.8311291   55.1969859 
   36.8289583   55.1974113 
   36.828844    55.1959785 
   36.8258264   55.1949367 
   36.8239283   55.1948536 
   36.8206862   55.1946025 
   36.817665    55.1938665 
   36.8136057   55.1940542 
   36.8086576   55.1954613 
   36.8069477   55.1962196 
   36.8028709   55.1977587 
   36.80502     55.1998453 
   36.8072525   55.2014818 
   36.8070883   55.201881  
   36.8064737   55.202446  
   36.8058145   55.2028703 
   36.8052289   55.2032496 
   36.802579    55.2035193 
   36.8023922   55.2053669 
   36.8017903   55.2060893 
   36.8036096   55.2087704 
   36.805164    55.2107422 
   36.8053209   55.2108399 
   36.8090348   55.212048  
   36.8117168   55.2125197 
   36.8123023   55.2149441 
   36.8115349   55.2150496 
   36.8107794   55.2151904 
   36.8089813   55.2176204 
   36.8073237   55.2190053 
   36.8063018   55.2200174 
   36.8057515   55.2213704 
   36.804864    55.2239168 
   36.8060077   55.2255588 
   36.8038694   55.2257176 
   36.8008423   55.2259892 
   36.796573    55.226852  
   36.7940964   55.2269773 
   36.7941301   55.2289065 
   36.7941131   55.2309287 
   36.7940742   55.2319617 
   36.7943071   55.2326874 
   36.7944991   55.2344253 
   36.794543    55.2354343 
   36.7955951   55.236713  
   36.799951    55.2424725 
   36.7954099   55.2438653 
   36.7920259   55.2450174 
   36.7893066   55.2459402 
   36.7879896   55.2468686 
   36.788843    55.2478026 
   36.7891847   55.2484308 
   36.7905557   55.2517355 
   36.7905161   55.252001  
   36.7894135   55.2536326 
   36.7886126   55.2546768 
   36.7883164   55.2556378 
   36.7887213   55.256409  
   36.7813609   55.2573401 
   36.777847    55.2580278 
   36.7764846   55.2598261 
   36.7741133   55.2619303 
   36.7701657   55.2647861 
   36.7689013   55.2660848 
   36.7632458   55.2698276 
   36.7610787   55.2699211 
   36.7586385   55.2697524 
   36.7568458   55.2696285 
   36.7546651   55.2689031 
   36.7531303   55.2681731 
   36.7520064   55.2674635 
   36.7498373   55.2668141 
   36.7495386   55.2665891 
   36.7468795   55.2652081 
   36.7443152   55.2642621 
   36.7420433   55.2636829 
   36.7385635   55.2637797 
   36.7384396   55.2635705 
   36.7365889   55.2635079 
   36.7351287   55.263032  
   36.7341808   55.2626256 
   36.7312073   55.2622515 
   36.7292928   55.2612662 
   36.7266835   55.2603249 
   36.719309    55.2620062 
   36.7191133   55.2621678 
   36.7184108   55.2625879 
   36.7174174   55.2631504 
   36.7147267   55.2637617 
   36.711551    55.2643326 
   36.709384    55.2638242 
   36.7065056   55.2634535 
   36.7058817   55.2633655 
   36.7036646   55.2633964 
   36.7022021   55.2634111 
   36.7002369   55.2628108 
   36.6967066   55.2617681 
   36.694923    55.2638525 
   36.6934084   55.2664652 
   36.6926604   55.2668396 
   36.6917932   55.2673074 
   36.6885415   55.268667  
   36.6876668   55.2690697 
   36.6854617   55.2689674 
   36.6852145   55.2689423 
   36.6805512   55.2682113 
   36.6768061   55.2684501 
   36.6749648   55.2690832 
   36.6727283   55.2683551 
   36.6715844   55.2690309 
   36.6714988   55.2693272 
   36.6700193   55.2701397 
   36.6619311   55.2750038 
   36.6599582   55.2769004 
   36.6593265   55.2774903 
   36.6545143   55.2775949 
   36.6531138   55.2775904 
   36.6487157   55.2777312 
   36.647377    55.2777635 
   36.6460559   55.277779  
   36.6446493   55.2777344 
   36.6439178   55.2776568 
   36.6429877   55.2773842 
   36.6410157   55.2713202 
   36.6407302   55.2707875 
   36.6398319   55.2691007 
   36.6390989   55.2672848 
   36.6377624   55.2671737 
   36.63745     55.2671129 
   36.6364609   55.2669189 
   36.6357109   55.2667649 
   36.6315236   55.2660707 
   36.6300335   55.2658763 
   36.6288383   55.2659386 
   36.6213295   55.2653907 
   36.619281    55.2685264 
   36.618306    55.2686021 
   36.6178178   55.268807  
   36.6170561   55.2689744 
   36.6162378   55.2694319 
   36.6141387   55.2694668 
   36.6133445   55.2703927 
   36.6124084   55.2720288 
   36.611736    55.2740063 
   36.6111661   55.2743819 
   36.6065684   55.2784205 
   36.6035706   55.2806557 
   36.6005531   55.2829004 
   36.6030978   55.2847726 
   36.6045195   55.2858768 
   36.605619    55.2864646 
   36.6060648   55.286703  
   36.6065867   55.2875545 
   36.6084231   55.2902316 
   36.6065512   55.2916543 
   36.6129895   55.2943104 
   36.6108927   55.2999255 
   36.6094323   55.3025941 
   36.6077348   55.308118  
   36.6078947   55.3084124 
   36.6077513   55.3111754 
   36.6136545   55.313149  
   36.6197178   55.3133271 
   36.6182443   55.3100813 
   36.6214227   55.3089791 
   36.6235818   55.3080493 
   36.6258848   55.3101209 
   36.6246835   55.312322  
   36.6224187   55.3138134 
   36.625819    55.3146893 
   36.627067    55.3160181 
   36.6229338   55.316037  
   36.6203113   55.3181801 
   36.6192475   55.3188882 
   36.6178063   55.3206277 
   36.6171865   55.3210714 
   36.6170983   55.3211496 
   36.6165013   55.3214491 
   36.6163248   55.3223043 
   36.6158074   55.3229991 
   36.6127428   55.3248543 
   36.609386    55.3268861 
   36.6096683   55.3272456 
   36.6065669   55.3298604 
   36.6051063   55.3312525 
   36.6003288   55.3346929 
   36.6029342   55.3362918 
   36.6032915   55.3359248 
   36.6018606   55.3344601 
   36.6049412   55.3330679 
   36.6062522   55.3340105 
   36.6086285   55.335376  
   36.6121067   55.3364301 
   36.6129551   55.337846  
   36.6131547   55.3381791 
   36.6112642   55.3387138 
   36.6107579   55.3400634 
   36.6093332   55.340548  
   36.6129964   55.3418563 
   36.6151812   55.3426909 
   36.6171274   55.343401  
   36.6185355   55.3439639 
   36.6200798   55.3446215 
   36.6173433   55.3487205 
   36.6100707   55.3496693 
   36.6093045   55.3483171 
   36.6008253   55.3512798 
   36.59906     55.35006   
   36.599089    55.349938  
   36.5926539   55.3462302 
   36.5900255   55.3377039 
   36.5815609   55.3365674 
   36.578054    55.3341054 
   36.5756421   55.332557  
   36.5742675   55.3320245 
   36.5740688   55.3313044 
   36.5738327   55.3305856 
   36.5729184   55.3303602 
   36.571281    55.3299562 
   36.5694716   55.3295326 
   36.5681283   55.3292093 
   36.5671324   55.3290668 
   36.5666922   55.329029  
   36.5660653   55.3289644 
   36.5637789   55.3286383 
   36.5636534   55.328645  
   36.5635229   55.3286979 
   36.5629169   55.3290951 
   36.5593289   55.3281057 
   36.5585605   55.3262847 
   36.5570569   55.3232558 
   36.5565265   55.3220337 
   36.5546579   55.3218323 
   36.5497923   55.3196515 
   36.5487718   55.3204496 
   36.5475129   55.3214945 
   36.5467775   55.3222461 
   36.5451766   55.3231262 
   36.5411529   55.3238438 
   36.5369936   55.3240668 
   36.536896    55.3235837 
   36.5355961   55.3226606 
   36.5342964   55.3217375 
   36.5326287   55.3208227 
   36.5328966   55.3195197 
   36.5335947   55.3193803 
   36.5343381   55.3187401 
   36.534069    55.3174084 
   36.5345528   55.3167734 
   36.5342721   55.3149038 
   36.5364698   55.313578  
   36.5383563   55.3121686 
   36.5351953   55.3070732 
   36.5322139   55.3056872 
   36.530872    55.3047925 
   36.527598    55.3019233 
   36.5261287   55.3009064 
   36.5212147   55.2985102 
   36.5164304   55.2963007 
   36.5085542   55.3027873 
   36.5091936   55.3032812 
   36.5023577   55.3047069 
   36.4998738   55.3035009 
   36.4967661   55.3028178 
   36.4931231   55.3023364 
   36.4855382   55.3045008 
   36.4812563   55.3030996 
   36.4785036   55.3026973 
   36.4780212   55.3026491 
   36.4763415   55.3025063 
   36.4743575   55.3022805 
   36.4735355   55.3021568 
   36.4729755   55.3020634 
   36.4707517   55.3016906 
   36.4689571   55.3018365 
   36.4679717   55.3018688 
   36.467728    55.3019013 
   36.4672509   55.301909  
   36.4665928   55.3020067 
   36.4656797   55.3020439 
   36.4634633   55.3016797 
   36.4611429   55.3011737 
   36.4608788   55.3009081 
   36.4608745   55.3007329 
   36.461311    55.2988673 
   36.4613151   55.2982891 
   36.4614614   55.297324  
   36.4613973   55.2968963 
   36.4615715   55.2960101 
   36.4616964   55.2955424 
   36.4619741   55.2951973 
   36.4619973   55.2951346 
   36.4620017   55.2946236 
   36.4622465   55.2941053 
   36.4625726   55.2931344 
   36.4598685   55.2929814 
   36.4568233   55.2926486 
   36.4508462   55.2921614 
   36.4491155   55.2924144 
   36.446395    55.2910625 
   36.4434443   55.2911867 
   36.4413108   55.2911888 
   36.4413038   55.2917078 
   36.4374356   55.2905597 
   36.435488    55.2909082 
   36.4361782   55.293058  
   36.4358169   55.2936825 
   36.4354793   55.2946705 
   36.431434    55.2955124 
   36.4295872   55.2948311 
   36.4277413   55.2940162 
   36.4254898   55.2942593 
   36.4234732   55.2943878 
   36.4218913   55.2976202 
   36.419928    55.3004657 
   36.4187788   55.3019551 
   36.4181277   55.3028899 
   36.4173662   55.3039839 
   36.4195837   55.3049023 
   36.4229031   55.3060086 
   36.4205241   55.306761  
   36.4185904   55.3084264 
   36.4178158   55.3088775 
   36.4154938   55.3086231 
   36.4167889   55.3052625 
   36.4089683   55.3010053 
   36.4045799   55.2998526 
   36.4004649   55.298913  
   36.398502    55.2976724 
   36.3929824   55.2967913 
   36.3892634   55.2955406 
   36.3849938   55.2944235 
   36.3853644   55.2941863 
   36.384862    55.2940182 
   36.3843398   55.2939774 
   36.3847643   55.2936414 
   36.3852934   55.2933909 
   36.3854249   55.2932327 
   36.3854444   55.2929059 
   36.3856261   55.2925809 
   36.3858501   55.292456  
   36.3863132   55.2923925 
   36.3870762   55.2923375 
   36.3873504   55.2921985 
   36.3875686   55.2921366 
   36.3878928   55.2920849 
   36.3883278   55.2918991 
   36.3888763   55.2918616 
   36.389144    55.2915482 
   36.389821    55.2902124 
   36.3859896   55.288618  
   36.3838924   55.2875074 
   36.3835941   55.2873494 
   36.3859548   55.2846978 
   36.3895068   55.282997  
   36.3898542   55.2811038 
   36.3848081   55.2798485 
   36.3822676   55.2781685 
   36.3798815   55.2770874 
   36.3781974   55.2783041 
   36.3752481   55.2799308 
   36.3720566   55.280774  
   36.3692903   55.280448  
   36.3696222   55.2790071 
   36.371013    55.2778426 
   36.368929    55.2772182 
   36.3741886   55.2741349 
   36.3794161   55.2734802 
   36.379921    55.2728847 
   36.3754799   55.2712837 
   36.3730668   55.2666965 
   36.3731841   55.2663905 
   36.3721372   55.2644687 
   36.3748344   55.2630682 
   36.3798616   55.2653276 
   36.3823993   55.261299  
   36.3804426   55.2598263 
   36.3835472   55.2583722 
   36.37965     55.2566583 
   36.3799532   55.2556253 
   36.3799392   55.2555095 
   36.3794231   55.2551928 
   36.3790773   55.2549701 
   36.3781978   55.2545638 
   36.3771318   55.2540713 
   36.3756233   55.2534934 
   36.372535    55.2529759 
   36.3707238   55.2526996 
   36.3689128   55.2524231 
   36.368965    55.2519963 
   36.3675363   55.251133  
   36.366667    55.2499121 
   36.3655925   55.2495076 
   36.3651329   55.2493757 
   36.3646928   55.2492491 
   36.3638133   55.249118  
   36.363763    55.2491209 
   36.3632066   55.2491537 
   36.3625283   55.2491348 
   36.362625    55.2493557 
   36.3627954   55.2494664 
   36.3628345   55.2496054 
   36.3629877   55.2498411 
   36.3632506   55.249977  
   36.3636523   55.25001   
   36.363776    55.2501265 
   36.3638228   55.2502258 
   36.3638421   55.2503714 
   36.3635637   55.2508044 
   36.3633664   55.2512809 
   36.3632527   55.2513879 
   36.3630483   55.2513711 
   36.3628544   55.2512778 
   36.3626954   55.2513166 
   36.3626249   55.2514724 
   36.3627179   55.2516088 
   36.3628302   55.2516838 
   36.3630098   55.251706  
   36.363109    55.2519372 
   36.3633894   55.2521476 
   36.3634694   55.2524131 
   36.3633447   55.252744  
   36.3632258   55.2529745 
   36.3632287   55.2531058 
   36.3633016   55.2532377 
   36.3633052   55.2534033 
   36.3634186   55.2535376 
   36.3633337   55.2538302 
   36.3630748   55.2540562 
   36.3627867   55.2542293 
   36.3624742   55.254292  
   36.3621736   55.2542634 
   36.3616858   55.2541767 
   36.3613358   55.2540387 
   36.3609361   55.2537258 
   36.3607828   55.2534854 
   36.3603856   55.2530599 
   36.3599461   55.2526776 
   36.359244    55.252358  
   36.359106    55.2522823 
   36.3588988   55.2522758 
   36.357704    55.2520342 
   36.3570043   55.251949  
   36.3567937   55.2519602 
   36.3564627   55.2520026 
   36.3563244   55.2520732 
   36.3560167   55.2522684 
   36.3556235   55.2527138 
   36.3552641   55.2530989 
   36.3550306   55.2532062 
   36.3545524   55.2533735 
   36.3541846   55.2534692 
   36.3536112   55.2536749 
   36.3533835   55.2537787 
   36.3530032   55.2539436 
   36.3521668   55.2542791 
   36.351986    55.2543115 
   36.3518762   55.2543145 
   36.3518001   55.2543322 
   36.3517498   55.2543622 
   36.351568    55.2544627 
   36.3508661   55.2547065 
   36.3505029   55.2549484 
   36.3501309   55.2550621 
   36.3493957   55.2551995 
   36.3489518   55.2552619 
   36.3486356   55.2552506 
   36.3483402   55.2552138 
   36.3481637   55.2551682 
   36.3478896   55.2548984 
   36.3477801   55.254564  
   36.3477706   55.2542997 
   36.3477679   55.2540804 
   36.3475815   55.2539709 
   36.3472434   55.2538694 
   36.3468341   55.2538003 
   36.3463908   55.2538309 
   36.3461689   55.2539068 
   36.3459269   55.2540169 
   36.3454952   55.2542963 
   36.3454009   55.2544456 
   36.3453806   55.2545889 
   36.3449407   55.2544285 
   36.3438869   55.2540321 
   36.3428332   55.2536357 
   36.3412056   55.2510473 
   36.341347    55.2491442 
   36.3398957   55.2486289 
   36.3387505   55.2474716 
   36.3376919   55.2466107 
   36.3372657   55.2455095 
   36.3374641   55.2443511 
   36.3366456   55.2418201 
   36.33461     55.2391769 
   36.3334702   55.2376917 
   36.3329956   55.2356731 
   36.3326924   55.2343835 
   36.3326085   55.232945  
   36.3318793   55.2306168 
   36.331551    55.2298856 
   36.3315411   55.2294233 
   36.3309666   55.2290266 
   36.3316773   55.2274529 
   36.3332203   55.2259689 
   36.3344356   55.2243022 
   36.3340375   55.2228255 
   36.3332969   55.220969  
   36.3280464   55.2159136 
   36.3273066   55.2143911 
   36.3246201   55.2144979 
   36.322327    55.2138435 
   36.3204573   55.2133508 
   36.3185876   55.212858  
   36.3144708   55.2116825 
   36.3106797   55.2105447 
   36.3109759   55.2104588 
   36.3112026   55.210393  
   36.3119373   55.2103297 
   36.3122459   55.2103031 
   36.3099457   55.2082235 
   36.3094629   55.2076808 
   36.3091255   55.2073015 
   36.3083599   55.206441  
   36.3050836   55.2027728 
   36.3026804   55.1998358 
   36.3002776   55.1968988 
   36.2997959   55.1963305 
   36.2986458   55.1947384 
   36.2969903   55.1928154 
   36.2958098   55.1912419 
   36.295486    55.1908393 
   36.2954166   55.190131  
   36.2960732   55.1895711 
   36.2979479   55.1881404 
   36.3012284   55.1895702 
   36.3061298   55.1880138 
   36.3134709   55.1857342 
   36.312317    55.1854831 
   36.3089532   55.1843734 
   36.3059428   55.1833803 
   36.2936159   55.1804653 
   36.2869989   55.1796599 
   36.2851258   55.1793178 
   36.2794405   55.1786169 
   36.2780676   55.1779663 
   36.2757234   55.1766708 
   36.2770902   55.1749867 
   36.2777934   55.1738933 
   36.2767751   55.1736619 
   36.2746937   55.172982  
   36.2679989   55.1721783 
   36.264784    55.1714839 
   36.2628082   55.1728212 
   36.2594779   55.1752626 
   36.2557374   55.1779633 
   36.2539373   55.1793593 
   36.2536898   55.1802816 
   36.253386    55.1810118 
   36.2542787   55.1833114 
   36.2547202   55.1840717 
   36.2531369   55.1861538 
   36.2525375   55.1869422 
   36.2519156   55.1879568 
   36.2513327   55.1887637 
   36.2510701   55.1891267 
   36.2492266   55.1907814 
   36.2484102   55.1906202 
   36.2480063   55.1906587 
   36.2471537   55.1906635 
   36.2449828   55.1903903 
   36.2433951   55.1900542 
   36.2420205   55.1896943 
   36.2411739   55.1893301 
   36.2404516   55.1890998 
   36.2398298   55.1889083 
   36.2392082   55.1887168 
   36.2382287   55.1885949 
   36.238109    55.1885    
   36.2381354   55.188371  
   36.2386428   55.1880012 
   36.2400053   55.1872505 
   36.2401134   55.1870959 
   36.239954    55.187038  
   36.2382232   55.1876654 
   36.2375587   55.188104  
   36.2369979   55.1881483 
   36.2358956   55.1884403 
   36.2353303   55.1884944 
   36.2339146   55.1884679 
   36.2326297   55.188506  
   36.2312792   55.1888446 
   36.2304218   55.1889881 
   36.230403    55.1890635 
   36.2301297   55.1894289 
   36.2299247   55.189718  
   36.2294951   55.1898756 
   36.2293605   55.1899227 
   36.2291879   55.1899455 
   36.2288974   55.1899043 
   36.2286808   55.1897306 
   36.2280862   55.1889733 
   36.2279826   55.1886542 
   36.2280145   55.1884844 
   36.2280896   55.1883798 
   36.2281663   55.18834   
   36.2282638   55.1883266 
   36.228425    55.1883323 
   36.2285367   55.188374  
   36.2286365   55.1885353 
   36.2287638   55.1886014 
   36.2290125   55.1886412 
   36.2295557   55.1886152 
   36.2298769   55.1885598 
   36.2302251   55.1884522 
   36.230566    55.1880748 
   36.2309281   55.1878347 
   36.2313591   55.187283  
   36.231435    55.1868497 
   36.2310839   55.1865566 
   36.2306506   55.1864804 
   36.2303769   55.1865756 
   36.2300393   55.1867501 
   36.2297561   55.1868466 
   36.2292618   55.1868843 
   36.228715    55.186861  
   36.2280947   55.1867726 
   36.2275955   55.1867971 
   36.2268145   55.1869253 
   36.2263381   55.1869788 
   36.2259305   55.1870984 
   36.2258422   55.187119  
   36.2257143   55.1872991 
   36.2258573   55.1878055 
   36.2257353   55.187954  
   36.2255618   55.1880421 
   36.2253695   55.188125  
   36.2253802   55.1881952 
   36.2253486   55.188233  
   36.2252781   55.1882424 
   36.2251152   55.1882045 
   36.2250151   55.1882356 
   36.224739    55.188561  
   36.224731    55.1886381 
   36.2248606   55.1888169 
   36.2247817   55.1889621 
   36.2242322   55.1890651 
   36.2239783   55.1890812 
   36.2238441   55.1889935 
   36.223571    55.1890485 
   36.2234845   55.1891856 
   36.2234296   55.1894765 
   36.2233261   55.1896711 
   36.2229619   55.1897639 
   36.2226937   55.1897381 
   36.2224425   55.1896776 
   36.2222129   55.1895349 
   36.2219145   55.1893099 
   36.2146697   55.1906884 
   36.2113781   55.1902565 
   36.2086629   55.1907817 
   36.2064338   55.1912079 
   36.2001281   55.1926619 
   36.1927762   55.1933911 
   36.1893338   55.1932956 
   36.1829504   55.1924564 
   36.1822271   55.1923184 
   36.1789793   55.193082  
   36.1772127   55.1931867 
   36.174793    55.1930271 
   36.174593    55.1930257 
   36.1738738   55.1926101 
   36.1734203   55.192486  
   36.1727171   55.1924055 
   36.1720289   55.1923664 
   36.1705558   55.1921516 
   36.1696208   55.1920082 
   36.1691974   55.1917676 
   36.167267    55.1912258 
   36.1670159   55.191094  
   36.1661934   55.1909725 
   36.1657075   55.1907441 
   36.1650255   55.1902799 
   36.1646286   55.1900043 
   36.1635967   55.1893056 
   36.1630173   55.189043  
   36.1621966   55.1887648 
   36.1605783   55.1885371 
   36.159967    55.1881895 
   36.1594055   55.1877782 
   36.159279    55.1873546 
   36.1589734   55.1869682 
   36.1585644   55.1864879 
   36.1580363   55.1860227 
   36.157489    55.1856723 
   36.156863    55.1854397 
   36.1557935   55.185149  
   36.155171    55.1850058 
   36.1545486   55.1848625 
   36.1538518   55.1846869 
   36.1534489   55.1846105 
   36.1530459   55.1845341 
   36.151999    55.1842448 
   36.1510577   55.1838415 
   36.1493559   55.1825656 
   36.1468699   55.1811028 
   36.1461601   55.1805274 
   36.1445061   55.1788187 
   36.1438658   55.1781273 
   36.1420725   55.1761341 
   36.1408852   55.1750048 
   36.1398804   55.1740767 
   36.139083    55.1740502 
   36.1343219   55.1756676 
   36.1293887   55.177654  
   36.1227183   55.1793243 
   36.1230291   55.18007   
   36.1240863   55.1808335 
   36.1263145   55.1819892 
   36.1247926   55.1822945 
   36.121031    55.1830986 
   36.1197491   55.1833375 
   36.1178718   55.1835786 
   36.1166219   55.1842389 
   36.1153474   55.1847064 
   36.11554     55.1854178 
   36.1157094   55.1863306 
   36.1155753   55.1879713 
   36.1151673   55.1888416 
   36.1138168   55.1917203 
   36.113482    55.1937973 
   36.1134358   55.1940829 
   36.1134071   55.1942671 
   36.113364    55.1949485 
   36.1133208   55.19563   
   36.1124474   55.1975697 
   36.1112123   55.199067  
   36.1102432   55.1996379 
   36.1086284   55.2012499 
   36.1076831   55.2027375 
   36.10714     55.2038433 
   36.106791    55.2048774 
   36.1065487   55.2051242 
   36.1059624   55.2058237 
   36.0998618   55.2064    
   36.099303    55.2064561 
   36.0981175   55.2067015 
   36.0976591   55.2072497 
   36.097152    55.2076649 
   36.0976308   55.2085278 
   36.0982032   55.2094037 
   36.0995071   55.2092773 
   36.1000068   55.2096538 
   36.1011902   55.2095751 
   36.1014931   55.2095603 
   36.1040561   55.2094143 
   36.1037506   55.2108536 
   36.1043539   55.2107643 
   36.1050537   55.2106607 
   36.1052547   55.2100606 
   36.1064323   55.2099047 
   36.1072561   55.2099485 
   36.1076847   55.2095574 
   36.1093377   55.2088681 
   36.1098651   55.2088486 
   36.1119316   55.2087718 
   36.1125703   55.2094803 
   36.1135756   55.2100824 
   36.1099995   55.210184  
   36.1098282   55.209909  
   36.1064906   55.2111702 
   36.1033115   55.212347  
   36.1022452   55.2127849 
   36.1017002   55.2129753 
   36.1014411   55.2130658 
   36.100982    55.2131434 
   36.0992512   55.2133892 
   36.0985244   55.2133841 
   36.0977041   55.2134102 
   36.0972405   55.2135126 
   36.0967982   55.2137011 
   36.0963736   55.213966  
   36.0958485   55.2142064 
   36.0956729   55.2142822 
   36.0954389   55.2142337 
   36.0951077   55.2139067 
   36.0947427   55.2125463 
   36.0945562   55.2119817 
   36.0941634   55.2114128 
   36.0938597   55.2106004 
   36.0936372   55.2101319 
   36.0935645   55.2099391 
   36.0933653   55.2098803 
   36.0904044   55.2099015 
   36.0870142   55.210091  
   36.0865504   55.2102865 
   36.0858394   55.2108597 
   36.0843759   55.2105611 
   36.081493    55.2115808 
   36.0806213   55.2120878 
   36.0803792   55.212283  
   36.0799009   55.2125419 
   36.0785624   55.2134058 
   36.0782076   55.2136346 
   36.077225    55.2142688 
   36.0768261   55.214526  
   36.0766788   55.2146573 
   36.0765523   55.2147374 
   36.0764641   55.2149649 
   36.076238    55.215317  
   36.0758164   55.2157398 
   36.0750978   55.2160279 
   36.0746012   55.2163091 
   36.0742944   55.2166003 
   36.0741445   55.2170131 
   36.0742695   55.2173267 
   36.0747223   55.2177362 
   36.0751829   55.2180899 
   36.0748829   55.2182363 
   36.0731893   55.2187668 
   36.0707934   55.2181566 
   36.0659947   55.2154388 
   36.0653501   55.2153936 
   36.0625162   55.216263  
   36.0614855   55.2174448 
   36.0604442   55.2172106 
   36.0582089   55.2165754 
   36.0567121   55.2162309 
   36.0545397   55.2156223 
   36.05288     55.215461  
   36.0514104   55.2152748 
   36.0486529   55.215264  
   36.0470236   55.2166753 
   36.0454946   55.2176108 
   36.0455132   55.2183435 
   36.0466887   55.2182124 
   36.0464785   55.2194203 
   36.0453791   55.2207604 
   36.0477986   55.2226664 
   36.0455111   55.2234785 
   36.0438323   55.2239424 
   36.0441326   55.2248294 
   36.0453665   55.2264422 
   36.0428116   55.2267868 
   36.0415508   55.226856  
   36.0386649   55.227703  
   36.0360707   55.226757  
   36.0353471   55.2265256 
   36.0333467   55.2258989 
   36.0311588   55.225357  
   36.0302766   55.2244989 
   36.0292279   55.2235866 
   36.0279936   55.2226373 
   36.0262504   55.2212309 
   36.0246138   55.2209658 
   36.0232581   55.2207933 
   36.022267    55.2206489 
   36.020911    55.2204367 
   36.0213534   55.2178706 
   36.021796    55.2153046 
   36.0220261   55.2143613 
   36.0202238   55.2144422 
   36.0191033   55.214589  
   36.0178793   55.2148616 
   36.0177541   55.2153323 
   36.0166199   55.2149862 
   36.014431    55.2156321 
   36.0132965   55.2164161 
   36.0111841   55.2170312 
   36.009075    55.2174448 
   36.0083638   55.2182191 
   36.0073201   55.2195245 
   36.0049448   55.2195019 
   36.0041908   55.2196072 
   35.9989415   55.2197853 
   35.9986166   55.2192798 
   35.9986721   55.2190862 
   35.9984829   55.2188295 
   35.998006    55.2186577 
   35.9973372   55.2185325 
   35.996996    55.2182321 
   35.9969611   55.2178651 
   35.9964488   55.2176679 
   35.9959832   55.2175168 
   35.9955764   55.2177514 
   35.9949621   55.2185247 
   35.9937779   55.2199362 
   35.9897609   55.2230548 
   35.9881818   55.2236376 
   35.984835    55.2242884 
   35.9823963   55.2264271 
   35.9821395   55.2265763 
   35.982155    55.2267065 
   35.9803997   55.2284564 
   35.9782799   55.230595  
   35.9751038   55.2336906 
   35.9733722   55.2353034 
   35.9720866   55.2365827 
   35.9704659   55.2366756 
   35.9697679   55.2367158 
   35.9692318   55.2367466 
   35.9662599   55.2372992 
   35.9647386   55.2373717 
   35.9634921   55.2381739 
   35.9628538   55.2385967 
   35.9592278   55.2409553 
   35.9591694   55.2409991 
   35.9586283   55.2414113 
   35.9496384   55.2389831 
   35.9465524   55.2370074 
   35.9450301   55.2361055 
   35.9450019   55.2360318 
   35.9439786   55.2334521 
   35.9397088   55.2336409 
   35.9371622   55.2336322 
   35.9357357   55.2334135 
   35.9343092   55.2331947 
   35.9330688   55.2330351 
   35.9328763   55.2329713 
   35.9302517   55.2320773 
   35.92957     55.2338726 
   35.9285526   55.2351736 
   35.9265465   55.236667  
   35.9246979   55.2374758 
   35.9236725   55.2380692 
   35.9214146   55.2392095 
   35.9186477   55.2391251 
   35.9172544   55.2392043 
   35.9151412   55.2391722 
   35.9146919   55.2392529 
   35.9141869   55.239545  
   35.9137381   55.2396719 
   35.9132653   55.2397264 
   35.9118103   55.2397051 
   35.9109787   55.2396717 
   35.910483    55.2397427 
   35.9091939   55.24011   
   35.9078466   55.2404413 
   35.9074889   55.2404491 
   35.9068298   55.2403525 
   35.9064133   55.2402814 
   35.9058947   55.2403689 
   35.9054922   55.2405221 
   35.9050314   55.240616  
   35.9037613   55.2406137 
   35.9018904   55.2405542 
   35.901822    55.2406468 
   35.9019592   55.2416095 
   35.9020005   55.2416984 
   35.9019552   55.2417843 
   35.9018338   55.2445191 
   35.9005887   55.2455229 
   35.8972219   55.2465568 
   35.8952505   55.2468603 
   35.8914644   55.247005  
   35.8885982   55.2478985 
   35.8879464   55.2462514 
   35.8872581   55.2444264 
   35.8865895   55.242819  
   35.8863488   55.2401414 
   35.8864033   55.2386701 
   35.8818384   55.238502  
   35.8794875   55.2384899 
   35.8762104   55.2385236 
   35.8721131   55.238629  
   35.8689505   55.2387083 
   35.8663984   55.2386471 
   35.8656868   55.2384646 
   35.8642856   55.2383288 
   35.8637475   55.2391023 
   35.8622691   55.2393394 
   35.8613281   55.239321  
   35.8603866   55.239271  
   35.8605345   55.2384229 
   35.861134    55.2371215 
   35.8602894   55.2369262 
   35.8579025   55.2359934 
   35.857001    55.2358906 
   35.8549268   55.2356562 
   35.852495    55.2354788 
   35.8509873   55.2359584 
   35.8495577   55.2342821 
   35.8486092   55.2324972 
   35.8459599   55.2324788 
   35.8428331   55.2323148 
   35.8357999   55.2351983 
   35.8315934   55.2373214 
   35.8283818   55.2390705 
   35.822632    55.2420766 
   35.82185     55.2424416 
   35.8207115   55.2429923 
   35.8183142   55.2441797 
   35.8144267   55.2460111 
   35.81488     55.2477744 
   35.8149699   55.2481601 
   35.8150938   55.2491625 
   35.8155827   55.250342  
   35.8173797   55.2511618 
   35.8182197   55.2521754 
   35.8185619   55.2530584 
   35.8180607   55.2566319 
   35.8176762   55.2576027 
   35.8212498   55.258147  
   35.8220693   55.2567199 
   35.8229579   55.2552595 
   35.8267411   55.254675  
   35.8278469   55.2556746 
   35.8277561   55.2571987 
   35.8290142   55.2597547 
   35.8293316   55.2610418 
   35.8289113   55.262209  
   35.8284283   55.2624677 
   35.8244521   55.2648768 
   35.8222636   55.2642826 
   35.8198154   55.2636989 
   35.8156811   55.2624832 
   35.814615    55.2620904 
   35.8117271   55.2620915 
   35.8083211   55.2619473 
   35.8068115   55.2624064 
   35.8054774   55.2627711 
   35.8044087   55.2629331 
   35.8034223   55.2630924 
   35.8017953   55.262963  
   35.8004825   55.262757  
   35.7983381   55.2640143 
   35.7974279   55.2643333 
   35.7954766   55.2644735 
   35.7949317   55.2634854 
   35.7940919   55.2620692 
   35.7934115   55.2611045 
   35.7923679   55.2620867 
   35.7904303   55.2639948 
   35.7886209   55.2654886 
   35.787069    55.2668762 
   35.7815576   55.2687269 
   35.7779354   55.2706752 
   35.7766045   55.27195   
   35.7757946   55.273357  
   35.773899    55.2732922 
   35.7705146   55.273452  
   35.7659925   55.2777669 
   35.7645904   55.2788685 
   35.7656834   55.2802266 
   35.7669654   55.2817228 
   35.7652403   55.2835496 
   35.7636568   55.2856906 
   35.7620013   55.2869063 
   35.756879    55.2884578 
   35.754822    55.2890705 
   35.7529777   55.2900596 
   35.752732    55.2904658 
   35.7517743   55.2918    
   35.7503357   55.2929803 
   35.7490856   55.2940418 
   35.7485294   55.2947159 
   35.7473829   55.2948586 
   35.7411561   55.2955777 
   35.7380965   55.2937605 
   35.7349081   55.2920146 
   35.7292768   55.2886882 
   35.726497    55.2868472 
   35.7242352   55.285228  
   35.721289    55.2826087 
   35.7224166   55.2794001 
   35.718431    55.2794779 
   35.7183553   55.2817076 
   35.7177659   55.2838656 
   35.7163525   55.2855733 
   35.7128759   55.2882405 
   35.7114392   55.2891847 
   35.7067615   55.2909036 
   35.7017891   55.289547  
   35.6986136   55.2891312 
   35.6973778   55.2887879 
   35.6979079   55.2884704 
   35.6963921   55.288186  
   35.6923347   55.2870551 
   35.6902032   55.286155  
   35.6891836   55.2856916 
   35.6886274   55.2854187 
   35.6863287   55.2845946 
   35.6848808   55.2839638 
   35.682873    55.2831853 
   35.6786395   55.2816081 
   35.6766096   55.2811893 
   35.672025    55.2802226 
   35.6634794   55.2783376 
   35.6586233   55.2798901 
   35.6533758   55.2812342 
   35.6475906   55.2831445 
   35.6460756   55.2839719 
   35.644942    55.2846351 
   35.6432217   55.2845656 
   35.642689    55.2846868 
   35.6428433   55.2856958 
   35.6429763   55.2864263 
   35.6421597   55.2876477 
   35.6415703   55.2880397 
   35.6398764   55.2882835 
   35.6392348   55.2882613 
   35.6372452   55.2889993 
   35.6368902   55.2895168 
   35.6352299   55.2883479 
   35.6327174   55.2867589 
   35.6295128   55.2877689 
   35.6270055   55.2878187 
   35.6251615   55.2878661 
   35.6237081   55.2879469 
   35.6220411   55.2880709 
   35.6193391   55.2882455 
   35.6163315   55.2882532 
   35.6160825   55.2896314 
   35.6159027   55.291169  
   35.6160756   55.292077  
   35.6152909   55.2923349 
   35.6135136   55.2936809 
   35.6100763   55.2956693 
   35.6068317   55.2927865 
   35.599991    55.2897039 
   35.5899187   55.2898663 
   35.5730798   55.2902669 
   35.575861    55.286684  
   35.5748567   55.280911  
   35.5730622   55.2801928 
   35.5708002   55.2793229 
   35.5689322   55.2785681 
   35.5661885   55.2776274 
   35.5633461   55.2770888 
   35.5606366   55.2758549 
   35.5669831   55.2728302 
   35.5699302   55.2718683 
   35.5768354   55.2690493 
   35.5766487   55.26702   
   35.5761556   55.2657885 
   35.576384    55.2633879 
   35.5764925   55.2622457 
   35.5735542   55.2617252 
   35.5701956   55.2596711 
   35.5677968   55.256519  
   35.5661299   55.2547726 
   35.5647239   55.2531333 
   35.5623649   55.2526797 
   35.5606456   55.2504208 
   35.5591733   55.2508986 
   35.5568746   55.2505776 
   35.5565469   55.2505135 
   35.5564691   55.2505877 
   35.5563712   55.2507411 
   35.5563079   55.2508798 
   35.5561811   55.251015  
   35.5559878   55.2510812 
   35.5559619   55.2511966 
   35.5562284   55.2515634 
   35.556116    55.2516855 
   35.5554784   55.2519546 
   35.5556293   55.2523702 
   35.5554678   55.2525253 
   35.5551618   55.2525584 
   35.5548788   55.252529  
   35.5547691   55.2525554 
   35.5547058   55.2526544 
   35.5546713   55.2527435 
   35.5545501   55.2527667 
   35.5544173   55.2527799 
   35.5543078   55.2528294 
   35.5541113   55.2527999 
   35.55399     55.2527471 
   35.553811    55.2527275 
   35.5535858   55.2527606 
   35.5534011   55.2528597 
   35.5531647   55.2530411 
   35.5529918   55.2532293 
   35.5527608   55.2532228 
   35.5526106   55.2531438 
   35.5524503   55.2531216 
   35.5521733   55.2532512 
   35.5519294   55.2532736 
   35.5516879   55.2534514 
   35.5515218   55.2534441 
   35.5512705   55.2533309 
   35.5510212   55.2532502 
   35.5508337   55.2532866 
   35.5507098   55.2535126 
   35.5506863   55.2536437 
   35.5503582   55.2537717 
   35.550286    55.2537651 
   35.5502207   55.2536699 
   35.5501955   55.253611  
   35.5499364   55.2536828 
   35.5490592   55.2540494 
   35.5489325   55.2541781 
   35.5489053   55.2543456 
   35.5490239   55.2544519 
   35.5491871   55.2545516 
   35.54918     55.2546415 
   35.5490561   55.2547273 
   35.5489001   55.2547291 
   35.5487095   55.2546549 
   35.5485969   55.25465   
   35.5484844   55.2547045 
   35.5484065   55.2547804 
   35.5483779   55.2549552 
   35.5482251   55.2550906 
   35.5480648   55.2550898 
   35.5474681   55.2547908 
   35.5473591   55.2547706 
   35.5472552   55.2547756 
   35.5470965   55.2548714 
   35.547004    55.2548706 
   35.5467974   55.2547867 
   35.5466633   55.2547892 
   35.5463907   55.2549344 
   35.5461814   55.2549675 
   35.54592     55.2549833 
   35.5456459   55.2550964 
   35.545222    55.2553968 
   35.5450576   55.2555816 
   35.5445948   55.2550145 
   35.5423204   55.2556225 
   35.5422954   55.2563663 
   35.5403849   55.2570022 
   35.5395133   55.2573324 
   35.5374824   55.2576624 
   35.5350043   55.2583874 
   35.5312954   55.2582696 
   35.5291335   55.258196  
   35.5261616   55.2576742 
   35.5177077   55.2554942 
   35.5158969   55.255028  
   35.5133086   55.2543553 
   35.5094571   55.2547078 
   35.5070498   55.2549627 
   35.5038151   55.2552962 
   35.5032188   55.253088  
   35.5009587   55.2534937 
   35.4998795   55.2522838 
   35.4991731   55.2523324 
   35.4969598   55.2500285 
   35.4982888   55.2486525 
   35.4992281   55.2477938 
   35.500638    55.2469081 
   35.5026889   55.2465882 
   35.5050304   55.2446837 
   35.5056823   55.2432455 
   35.5074995   55.2421798 
   35.5094217   55.2414137 
   35.5082323   55.240189  
   35.4975817   55.2388058 
   35.4914297   55.2397051 
   35.4817556   55.2399121 
   35.481539    55.239919  
   35.4815489   55.237455  
   35.4764668   55.2366775 
   35.4720541   55.2366863 
   35.4694212   55.2366112 
   35.4622567   55.2362728 
   35.4608247   55.2359401 
   35.4592409   55.234891  
   35.4571189   55.2339529 
   35.455039    55.2328227 
   35.4528234   55.2318442 
   35.4512976   55.2310786 
   35.4485211   55.2299143 
   35.4462712   55.2290131 
   35.4414992   55.2300783 
   35.4384164   55.2300934 
   35.437242    55.2300452 
   35.4319948   55.2309551 
   35.4294263   55.2310904 
   35.4243323   55.231205  
   35.422191    55.2317244 
   35.4203748   55.2311252 
   35.419505    55.2300305 
   35.4184993   55.2287057 
   35.4160369   55.2310225 
   35.4143749   55.2324683 
   35.412961    55.233603  
   35.4108909   55.2355011 
   35.4085326   55.2375461 
   35.4076425   55.2377338 
   35.4048744   55.2378573 
   35.4018149   55.237918  
   35.4004364   55.2370594 
   35.3977074   55.235525  
   35.3961473   55.234589  
   35.3936305   55.2350597 
   35.3920731   55.2355638 
   35.3907495   55.2358219 
   35.3874883   55.2365174 
   35.3858086   55.2375339 
   35.3843798   55.2376404 
   35.3828867   55.2380374 
   35.3823235   55.2381269 
   35.3814189   55.2386852 
   35.3813837   55.2388534 
   35.3810831   55.2390312 
   35.3810534   55.2393347 
   35.381023    55.2394286 
   35.3807743   55.2396132 
   35.3808155   55.2398655 
   35.3806983   55.2399627 
   35.3805093   55.2399782 
   35.3803893   55.2400597 
   35.3804033   55.2401538 
   35.3801319   55.2402566 
   35.3801484   55.2405535 
   35.380073    55.2406589 
   35.3799048   55.2409293 
   35.3796104   55.2409917 
   35.3793905   55.2411515 
   35.3790962   55.2411496 
   35.3789892   55.2412105 
   35.3788734   55.2413556 
   35.378873    55.2415056 
   35.378769    55.2415319 
   35.3785843   55.2415104 
   35.3784773   55.2415547 
   35.3784091   55.2417436 
   35.3781419   55.2418117 
   35.37812     55.2419024 
   35.378055    55.241942  
   35.3778847   55.2419434 
   35.3777371   55.242067  
   35.3776245   55.2420727 
   35.3774854   55.2420042 
   35.3774371   55.2420067 
   35.3773521   55.2422246 
   35.3772422   55.2422856 
   35.3770979   55.242287  
   35.3769938   55.242343  
   35.3769643   55.2425524 
   35.3767911   55.2426166 
   35.3766454   55.2425942 
   35.3764579   55.2425075 
   35.3763367   55.2425032 
   35.3762296   55.2426483 
   35.3760837   55.242658  
   35.3759363   55.2427222 
   35.375763    55.242765  
   35.3757297   55.2428037 
   35.3757381   55.2428738 
   35.3758475   55.2430157 
   35.3758009   55.2431476 
   35.3757389   55.2431789 
   35.3754871   55.2431387 
   35.375381    55.2431357 
   35.3752604   55.2431608 
   35.3751219   55.2431606 
   35.3749191   55.2431559 
   35.374844    55.2431946 
   35.3747946   55.2432984 
   35.3746734   55.2433074 
   35.3746085   55.2432884 
   35.3744771   55.2432891 
   35.3743687   55.2433517 
   35.374342    55.2435995 
   35.3740642   55.2436033 
   35.3739153   55.2436914 
   35.3738926   55.2437887 
   35.3736978   55.2437819 
   35.3736299   55.2438289 
   35.3736224   55.2439196 
   35.3735544   55.2439501 
   35.3734086   55.2439912 
   35.3733982   55.2440908 
   35.3732362   55.2442606 
   35.3731524   55.2442589 
   35.3730083   55.2441548 
   35.3729204   55.2441218 
   35.3727847   55.24412   
   35.3725858   55.2440126 
   35.3721906   55.243924  
   35.3720898   55.2438275 
   35.3717366   55.2437034 
   35.3715259   55.243695  
   35.371399    55.2436702 
   35.3713267   55.2436981 
   35.3715108   55.2439407 
   35.3716462   55.2440283 
   35.3716632   55.244119  
   35.371542    55.2441453 
   35.3711237   55.2440394 
   35.3710088   55.2440924 
   35.3710101   55.2441522 
   35.3711749   55.2442838 
   35.3709838   55.2445245 
   35.3708077   55.2445622 
   35.3707236   55.244643  
   35.3706413   55.2446668 
   35.3704666   55.2446725 
   35.3703091   55.2447391 
   35.3701157   55.2447488 
   35.3699553   55.2447924 
   35.3694819   55.2448002 
   35.369318    55.2446202 
   35.3691421   55.2445425 
   35.368978    55.2443659 
   35.3686809   55.2443112 
   35.3685281   55.244232  
   35.3683987   55.244062  
   35.3682458   55.2440305 
   35.3681243   55.2440897 
   35.3680378   55.2440897 
   35.3678792   55.2440512 
   35.3677443   55.2439863 
   35.367676    55.243905  
   35.3676805   55.243872  
   35.3675939   55.2438484 
   35.3673629   55.2438577 
   35.367193    55.2437552 
   35.3671853   55.2435941 
   35.367131    55.2435494 
   35.3669455   55.2435434 
   35.3668217   55.2434748 
   35.3666426   55.2434706 
   35.3665211   55.2435521 
   35.3661113   55.2435484 
   35.3660192   55.243484  
   35.3660077   55.2434213 
   35.3659523   55.2434027 
   35.365828    55.2434219 
   35.3655907   55.2433883 
   35.3653963   55.2432773 
   35.3653392   55.243278  
   35.365234    55.2432321 
   35.3651396   55.2432209 
   35.3649644   55.2431082 
   35.3647855   55.2431056 
   35.3645921   55.2431177 
   35.3644019   55.243021  
   35.363913    55.2428986 
   35.3638353   55.2428041 
   35.363566    55.2426545 
   35.36341     55.2426461 
   35.3632396   55.242688  
   35.3632608   55.2428217 
   35.3631988   55.242824  
   35.3630041   55.2427422 
   35.3629392   55.2427429 
   35.3627556   55.2428282 
   35.3626502   55.2428639 
   35.3626125   55.242962  
   35.3625113   55.242999  
   35.3624174   55.2429948 
   35.3622359   55.2429302 
   35.3621294   55.242929  
   35.3619981   55.2430003 
   35.3619726   55.243113  
   35.3617898   55.2431966 
   35.3616684   55.2432006 
   35.3615716   55.2432557 
   35.3615077   55.2433695 
   35.3613307   55.2434641 
   35.3610744   55.2434792 
   35.3609089   55.2435535 
   35.3608086   55.2435617 
   35.3606005   55.2436527 
   35.3604365   55.2436917 
   35.360396    55.2437411 
   35.3603639   55.2438565 
   35.3601802   55.2439808 
   35.3600936   55.2439989 
   35.3598741   55.2440044 
   35.3597759   55.2440604 
   35.3597767   55.2442426 
   35.3594645   55.2443693 
   35.359464    55.2445516 
   35.3593462   55.2446062 
   35.3590322   55.2446076 
   35.3589441   55.2446496 
   35.3588887   55.2448614 
   35.359047    55.2449704 
   35.3591247   55.2450595 
   35.3591072   55.2451189 
   35.3589626   55.2452029 
   35.3589537   55.2452779 
   35.3591581   55.2454191 
   35.3592157   55.2455247 
   35.3593611   55.2456172 
   35.3594564   55.2456173 
   35.3596586   55.245568  
   35.3597812   55.2455723 
   35.359902    55.2457241 
   35.3597774   55.2458939 
   35.3598056   55.2460688 
   35.3599286   55.246216  
   35.3601785   55.2464089 
   35.3605609   55.2464513 
   35.3607395   55.2465406 
   35.3606958   55.2466922 
   35.3604497   55.2469031 
   35.3600448   55.247183  
   35.3598937   55.2474797 
   35.3600232   55.2475912 
   35.360092    55.2477859 
   35.3598201   55.2479308 
   35.3593807   55.24811   
   35.3588467   55.2481227 
   35.3579395   55.2483114 
   35.3575636   55.248527  
   35.3575602   55.2486574 
   35.3576407   55.2487646 
   35.3576058   55.2488718 
   35.3571237   55.2488729 
   35.3569762   55.2489305 
   35.356782    55.2491546 
   35.3566056   55.2492616 
   35.3566427   55.2494035 
   35.3568475   55.2494878 
   35.357       55.2496578 
   35.3573258   55.2497555 
   35.357686    55.2499422 
   35.3577345   55.2501962 
   35.3580196   55.2503763 
   35.3581073   55.2504959 
   35.3580591   55.2506789 
   35.3581627   55.2507796 
   35.3584998   55.2509754 
   35.3583866   55.2516713 
   35.3583719   55.252218  
   35.3585482   55.2525925 
   35.3585      55.2532192 
   35.3591509   55.2537707 
   35.3589338   55.2539536 
   35.3585023   55.2543291 
   35.3581125   55.2543402 
   35.3577936   55.2547555 
   35.3578534   55.2550211 
   35.3580432   55.255272  
   35.358447    55.2554208 
   35.3583916   55.2555989 
   35.3582756   55.2557241 
   35.3580762   55.2557618 
   35.3580348   55.2560652 
   35.358124    55.2562105 
   35.3581726   55.2563573 
   35.3583801   55.256501  
   35.358562    55.2567292 
   35.3588458   55.2568927 
   35.3590227   55.2571609 
   35.3591492   55.2572814 
   35.3592006   55.2574893 
   35.3594795   55.2578722 
   35.3595312   55.2579444 
   35.3595895   55.2582793 
   35.3596452   55.2584549 
   35.3596241   55.2587254 
   35.3595486   55.2588638 
   35.3595389   55.25922   
   35.3592462   55.2595249 
   35.3583147   55.2600351 
   35.3579568   55.2600249 
   35.3577718   55.2600708 
   35.3570719   55.2603769 
   35.3566145   55.2607129 
   35.3566701   55.2609777 
   35.3566525   55.2610329 
   35.3563995   55.2611629 
   35.3526154   55.2667158 
   35.3526206   55.2669138 
   35.3524062   55.2670917 
   35.3524116   55.267217  
   35.3522508   55.2673669 
   35.351973    55.2675085 
   35.3518857   55.2677063 
   35.3517626   55.2678347 
   35.3514444   55.2679416 
   35.3513892   55.2680999 
   35.3511198   55.2682925 
   35.3510284   55.2688795 
   35.3508862   55.2690624 
   35.3497174   55.2696516 
   35.3493392   55.2697703 
   35.3491469   55.2698526 
   35.3487434   55.2700311 
   35.3486725   55.2700356 
   35.3485298   55.2701789 
   35.348111    55.270152  
   35.3474169   55.2704118 
   35.3471195   55.2703818 
   35.3465381   55.2714531 
   35.3462355   55.2720795 
   35.3430604   55.2730587 
   35.3407847   55.2728879 
   35.3368241   55.2725509 
   35.3342215   55.271752  
   35.3298102   55.2708613 
   35.3286737   55.2666608 
   35.3279452   55.2660962 
   35.3252932   55.2575609 
   35.3232533   55.2571623 
   35.3208928   55.2540897 
   35.3145399   55.253525  
   35.3124417   55.2538074 
   35.2973462   55.2534918 
   35.2962097   55.253143  
   35.2955977   55.2500037 
   35.2972297   55.2466149 
   35.2998524   55.2452195 
   35.3003478   55.2437243 
   35.3047191   55.2419134 
   35.308391    55.2391388 
   35.3111595   55.2404846 
   35.3195232   55.2425448 
   35.3234865   55.2427441 
   35.3239527   55.2389892 
   35.3271583   55.2400194 
   35.335114    55.2357159 
   35.3425161   55.2337218 
   35.3389899   55.230581  
   35.3344438   55.230581  
   35.332637    55.2284703 
   35.3311799   55.2266089 
   35.3334238   55.2253789 
   35.3363671   55.2234675 
   35.3290982   55.2198819 
   35.3329104   55.2168961 
   35.3315091   55.2155795 
   35.3301079   55.2138278 
   35.3315297   55.2129108 
   35.3339407   55.2126522 
   35.3345177   55.2118763 
   35.3356304   55.2116999 
   35.3362486   55.2120056 
   35.3367638   55.2114883 
   35.3373614   55.2114765 
   35.3384329   55.2103949 
   35.3409263   55.2083843 
   35.3408644   55.2079023 
   35.3412147   55.207526  
   35.3413178   55.2069851 
   35.3417917   55.2068558 
   35.3422245   55.2063149 
   35.3432136   55.2062796 
   35.3436463   55.2058681 
   35.3452124   55.2058798 
   35.3409057   55.202211  
   35.3399578   55.2021051 
   35.3383505   55.1994356 
   35.3266254   55.1978244 
   35.3215769   55.2010115 
   35.3123853   55.1970849 
   35.3090059   55.1917214 
   35.3039367   55.1859102 
   35.2997484   55.1857321 
   35.295215    55.1838968 
   35.2889506   55.1824144 
   35.2804608   55.1796376 
   35.2786062   55.177096  
   35.2790184   55.1752603 
   35.2831809   55.1732362 
   35.2840875   55.1712356 
   35.2838815   55.1696114 
   35.2856124   55.1669279 
   35.2861894   55.1668808 
   35.2889919   55.163844  
   35.2896925   55.1623844 
   35.2897337   55.1608305 
   35.2873021   55.1595592 
   35.2854888   55.1515767 
   35.2844585   55.1482797 
   35.2856536   55.1482562 
   35.2872197   55.1476909 
   35.2851179   55.1469373 
   35.2837166   55.1467253 
   35.2834025   55.1454405 
   35.2816716   55.143886  
   35.2788897   55.140553  
   35.2752012   55.1365483 
   35.2795285   55.1350523 
   35.2765532   55.1270562 
   35.2754897   55.1284435 
   35.2748303   55.128632  
   35.2736351   55.1282903 
   35.2729139   55.1265937 
   35.2733054   55.1259811 
   35.2747066   55.125062  
   35.2754073   55.1246732 
   35.2756046   55.1238308 
   35.2747211   55.1223921 
   35.2752492   55.1220571 
   35.2741209   55.1214695 
   35.2742025   55.1211208 
   35.2748123   55.120871  
   35.273302    55.1190384 
   35.2733165   55.1185884 
   35.2738265   55.1182551 
   35.2738994   55.1177802 
   35.2732583   55.1171885 
   35.2732291   55.1162052 
   35.2736808   55.1159885 
   35.2747299   55.1162385 
   35.275677    55.1162135 
   35.276391    55.1159302 
   35.27674     55.1155509 
   35.2768281   55.1154552 
   35.276391    55.1148718 
   35.2761724   55.1143135 
   35.2756187   55.1138718 
   35.2754439   55.1134551 
   35.2757645   55.1129967 
   35.2758519   55.1126217 
   35.2743219   55.1114549 
   35.2727774   55.1112466 
   35.2698778   55.1115299 
   35.268479    55.1108048 
   35.2679399   55.1107882 
   35.267663    55.1110215 
   35.2675465   55.1117216 
   35.2671968   55.111855  
   35.2641369   55.11188   
   35.2617878   55.1103203 
   35.2603866   55.1102967 
   35.2564302   55.1114283 
   35.2551526   55.1115225 
   35.2547542   55.1111673 
   35.2562429   55.1102526 
   35.2563821   55.1097967 
   35.2545383   55.1088409 
   35.2546055   55.1085663 
   35.2549369   55.1084839 
   35.2560316   55.1084976 
   35.256747    55.108363  
   35.2568815   55.1081213 
   35.2562258   55.1076584 
   35.2551481   55.1072589 
   35.2548408   55.1070089 
   35.2549753   55.106693  
   35.2559548   55.1063799 
   35.2560748   55.1059871 
   35.2552105   55.1053362 
   35.2545767   55.1050148 
   35.2542358   55.1042402 
   35.2532971   55.1036823 
   35.2525394   55.1033405 
   35.2520319   55.102158  
   35.2515806   55.1018009 
   35.2510524   55.1017679 
   35.2508892   55.1014328 
   35.2506683   55.1010756 
   35.2507259   55.1003339 
   35.2496981   55.099906  
   35.2480661   55.0996559 
   35.2475905   55.099246  
   35.2474465   55.0989164 
   35.2476144   55.0983636 
   35.249007    55.0976279 
   35.2488629   55.0974109 
   35.2479787   55.0971214 
   35.2471344   55.0968147 
   35.2458572   55.0963201 
   35.2457131   55.0960042 
   35.2459676   55.0958119 
   35.2467167   55.095702  
   35.2473313   55.0958036 
   35.2476818   55.0955948 
   35.247773    55.0952459 
   35.247845    55.0948173 
   35.2478767   55.0944701 
   35.2496106   55.0941782 
   35.2516286   55.0940425 
   35.2526177   55.0938501 
   35.254231    55.093416  
   35.2549705   55.0929352 
   35.2547833   55.0924689 
   35.2539525   55.0917207 
   35.2527871   55.0909263 
   35.2507713   55.0910519 
   35.2493338   55.0908513 
   35.2472947   55.0916037 
   35.2449859   55.0928549 
   35.2435929   55.0931026 
   35.2426976   55.0930103 
   35.2419642   55.0928044 
   35.2415025   55.0920777 
   35.2408167   55.0911102 
   35.2404228   55.0906672 
   35.2398728   55.0906749 
   35.2388282   55.090968  
   35.2375437   55.0910635 
   35.236498    55.0918291 
   35.2358869   55.0925207 
   35.2355541   55.0926684 
   35.23478     55.0926878 
   35.2340942   55.0925829 
   35.2335578   55.0924003 
   35.2329415   55.0922104 
   35.2320978   55.0923226 
   35.2316633   55.0922021 
   35.2306107   55.090399  
   35.2301422   55.0894314 
   35.2293274   55.0887163 
   35.2273044   55.088687  
   35.2261223   55.0868152 
   35.2249858   55.0859145 
   35.2225379   55.0837128 
   35.2211974   55.0832374 
   35.2210225   55.0815527 
   35.2216382   55.0816463 
   35.222338    55.0813632 
   35.2235724   55.0812774 
   35.2240824   55.081194  
   35.2242135   55.080777  
   35.2239658   55.0801514 
   35.2237843   55.0797128 
   35.2230938   55.07826   
   35.2229165   55.0769193 
   35.2225526   55.0749429 
   35.2225379   55.0746795 
   35.222338    55.0742111 
   35.2234764   55.0737677 
   35.2239149   55.0732655 
   35.2247173   55.0726886 
   35.2242508   55.0719407 
   35.2237327   55.071451  
   35.2232664   55.0709421 
   35.2218528   55.0702472 
   35.2207748   55.0702413 
   35.2201191   55.0703831 
   35.2195015   55.0705143 
   35.2188671   55.0706372 
   35.218254    55.0711507 
   35.2163105   55.0724322 
   35.2164131   55.0730091 
   35.2164971   55.0736341 
   35.2164131   55.0742164 
   35.2166464   55.0745049 
   35.2165437   55.0750337 
   35.2160865   55.0755412 
   35.2147616   55.0762196 
   35.2139395   55.076534  
   35.2126151   55.0762728 
   35.2116971   55.075839  
   35.2106889   55.075195  
   35.2098758   55.0746795 
   35.2096871   55.0745186 
   35.2091539   55.0743087 
   35.2085545   55.0740394 
   35.2085126   55.0733404 
   35.2087894   55.0730985 
   35.2092411   55.0730568 
   35.2097802   55.0731903 
   35.2101532   55.073336  
   35.2107286   55.0734896 
   35.210954    55.0731445 
   35.211158    55.0727608 
   35.2118574   55.0723687 
   35.2122276   55.0718998 
   35.2121262   55.0715334 
   35.21145     55.0710125 
   35.2114362   55.0704546 
   35.2115806   55.0700828 
   35.2124694   55.0699409 
   35.2129211   55.0695905 
   35.2127608   55.0691066 
   35.2089141   55.0665869 
   35.207253    55.0656942 
   35.2067284   55.0652352 
   35.2066556   55.0646762 
   35.2073841   55.0640254 
   35.2087975   55.0631576 
   35.2090452   55.0628905 
   35.2089286   55.0625568 
   35.2081564   55.0622647 
   35.2070563   55.0619518 
   35.2054025   55.0616889 
   35.2039454   55.0616555 
   35.202838    55.0617724 
   35.2012935   55.0615888 
   35.1986853   55.0609045 
   35.1980733   55.060796  
   35.1973448   55.0611799 
   35.1964997   55.0609629 
   35.195436    55.0611048 
   35.194824    55.0609629 
   35.1945326   55.0606542 
   35.1945617   55.0600617 
   35.195538    55.0596611 
   35.195946    55.0591354 
   35.1966891   55.0589851 
   35.1975051   55.0583676 
   35.1980879   55.0577028 
   35.1981438   55.057163  
   35.197668    55.0569493 
   35.1970157   55.0569279 
   35.1967825   55.0566358 
   35.1972948   55.056356  
   35.1980599   55.0557735 
   35.1993102   55.0552979 
   35.1995061   55.0548543 
   35.1989827   55.0544992 
   35.1982251   55.0543824 
   35.1973881   55.0542022 
   35.196399    55.0538762 
   35.1957272   55.0537319 
   35.1949434   55.0536411 
   35.1940757   55.0538442 
   35.1933852   55.0542664 
   35.1930213   55.0547901 
   35.193096    55.0566553 
   35.1928534   55.0570936 
   35.1923495   55.0573501 
   35.1901569   55.0582158 
   35.1876611   55.0588473 
   35.1859768   55.0588678 
   35.1849218   55.0594732 
   35.1843681   55.0595734 
   35.1835813   55.0593147 
   35.1827017   55.0586433 
   35.1822408   55.0580796 
   35.1820579   55.0572165 
   35.1822912   55.0565431 
   35.1824591   55.054865  
   35.1837934   55.0535662 
   35.1841946   55.0527538 
   35.1844745   55.0522675 
   35.185249    55.0516956 
   35.1864993   55.0513749 
   35.1869565   55.050509  
   35.1875816   55.0499905 
   35.1875737   55.0497166 
   35.1872677   55.0494828 
   35.1861895   55.0491907 
   35.185155    55.0489486 
   35.1835583   55.0487165 
   35.1817745   55.0487984 
   35.1808573   55.0491216 
   35.1799677   55.0493827 
   35.1789788   55.0500974 
   35.1783649   55.0501506 
   35.1774761   55.0499753 
   35.1775699   55.0496056 
   35.1772284   55.0494161 
   35.1751159   55.0494667 
   35.1743042   55.049627  
   35.1729699   55.0504448 
   35.1720742   55.0506319 
   35.1708705   55.0510435 
   35.1686592   55.0519468 
   35.1673493   55.0520287 
   35.1667373   55.0518367 
   35.1658776   55.0512524 
   35.1654706   55.0506019 
   35.1644351   55.0501506 
   35.1637357   55.0497917 
   35.1629489   55.049391  
   35.1623369   55.0484895 
   35.1617006   55.0481736 
   35.1609527   55.0480054 
   35.1598453   55.0480304 
   35.1576369   55.0473838 
   35.1570039   55.0469619 
   35.1566834   55.0465028 
   35.1572954   55.0459435 
   35.1602047   55.0444931 
   35.161858    55.0440122 
   35.162063    55.0434067 
   35.1614302   55.0429301 
   35.1592488   55.0427212 
   35.1567549   55.042803  
   35.1559419   55.042655  
   35.1546065   55.0422531 
   35.1538916   55.0414878 
   35.1536818   55.0401412 
   35.1551243   55.0385465 
   35.1554303   55.0376364 
   35.1553734   55.0370593 
   35.1542338   55.0368217 
   35.1507156   55.0366589 
   35.1498351   55.0360499 
   35.1484945   55.0363088 
   35.1470666   55.036935  
   35.1453618   55.037269  
   35.1427827   55.0379787 
   35.1408011   55.0388638 
   35.139548    55.0415605 
   35.1369252   55.0431551 
   35.136051    55.0426375 
   35.1339673   55.0432887 
   35.1305432   55.0422451 
   35.1313883   55.0408175 
   35.1326122   55.0392144 
   35.1284212   55.0388039 
   35.1280503   55.04015   
   35.1274321   55.0400791 
   35.1228575   55.0420155 
   35.1214562   55.043149  
   35.1207968   55.0421808 
   35.1191483   55.0413779 
   35.1183653   55.0401972 
   35.1172937   55.0396068 
   35.1146973   55.0376467 
   35.1222393   55.0278447 
   35.122569    55.0238996 
   35.1187362   55.0232381 
   35.1198629   55.0215068 
   35.1185379   55.0189629 
   35.1174655   55.0169778 
   35.1144275   55.0158392 
   35.1076336   55.0154381 
   35.1042449   55.0150244 
   35.1016464   55.0141561 
   35.1002318   55.0128224 
   35.0997365   55.0112027 
   35.0999082   55.0094793 
   35.1011989   55.0075052 
   35.10312     55.0066139 
   35.1065764   55.0067933 
   35.1090067   55.0076348 
   35.1102441   55.0084068 
   35.1102005   55.0097264 
   35.1107237   55.0110199 
   35.1156291   55.0108035 
   35.1212188   55.0086804 
   35.1221618   55.0074766 
   35.1245965   55.006719  
   35.1267777   55.0048573 
   35.1263018   55.0042041 
   35.1225938   55.002874  
   35.1211406   55.0026503 
   35.1194674   55.0019673 
   35.117591    55.0019204 
   35.1133489   55.001054  
   35.1118106   55.0006133 
   35.10965     54.9999754 
   35.1093458   54.9991327 
   35.1079193   54.9988017 
   35.1069819   54.9987856 
   35.1059721   54.9985019 
   35.1052921   54.9978518 
   35.1051067   54.9974026 
   35.1050861   54.9968234 
   35.1055188   54.9960551 
   35.1061782   54.9951331 
   35.1071427   54.9944995 
   35.1081358   54.9936555 
   35.1098735   54.9927861 
   35.1118862   54.9925679 
   35.1139056   54.9925206 
   35.1156984   54.9928516 
   35.1169347   54.9933718 
   35.1181093   54.9936909 
   35.1190778   54.9937737 
   35.120267    54.9935273 
   35.1214085   54.9930306 
   35.1220863   54.991965  
   35.1215871   54.9906408 
   35.1210766   54.9896715 
   35.1198402   54.9876735 
   35.1186451   54.9864557 
   35.1178208   54.9856517 
   35.1165226   54.9848122 
   35.1156365   54.9842211 
   35.1149771   54.9834998 
   35.1147711   54.9825302 
   35.1149565   54.9815842 
   35.1164536   54.9803769 
   35.1181917   54.9794321 
   35.1202524   54.9787817 
   35.1217566   54.9783796 
   35.1231785   54.9780721 
   35.1242912   54.9774926 
   35.1246827   54.976925  
   35.1247033   54.9762391 
   35.124353    54.9755177 
   35.1237967   54.9750446 
   35.1223748   54.9743705 
   35.1217673   54.9743992 
   35.1213239   54.974607  
   35.1194899   54.9738501 
   35.1156777   54.9721588 
   35.112731    54.9722061 
   35.1116801   54.9725254 
   35.1109589   54.9723953 
   35.1100316   54.9725727 
   35.1073322   54.9728684 
   35.106137    54.9729039 
   35.1044061   54.9725254 
   35.1011296   54.9715319 
   35.0994811   54.971662  
   35.0981211   54.9721706 
   35.0971076   54.9727424 
   35.0950714   54.9736727 
   35.0935842   54.9749315 
   35.0926033   54.9757173 
   35.0924369   54.9756197 
   35.0922168   54.975339  
   35.0918525   54.9750881 
   35.0911483   54.9749354 
   35.0904537   54.9748707 
   35.0902934   54.9745863 
   35.0905557   54.9743271 
   35.0910074   54.9742936 
   35.0914008   54.9741347 
   35.0915174   54.9738922 
   35.0912988   54.9736748 
   35.090716    54.9736664 
   35.0903517   54.9739006 
   35.0897543   54.9737751 
   35.0896815   54.9732901 
   35.0894338   54.9729472 
   35.0889238   54.9729054 
   35.0884866   54.973081  
   35.0879621   54.9730726 
   35.0869938   54.9726104 
   35.0864162   54.9726742 
   35.0861496   54.9727464 
   35.0858237   54.9726912 
   35.0857201   54.9728102 
   35.0857327   54.9733403 
   35.0853393   54.9735995 
   35.0847711   54.9735493 
   35.0842465   54.9730392 
   35.0836782   54.9730308 
   35.083416    54.9732483 
   35.0830954   54.9734573 
   35.0826      54.9730977 
   35.0821483   54.9732315 
   35.081478    54.9731981 
   35.0804726   54.9724538 
   35.0804581   54.9719938 
   35.0810555   54.9716927 
   35.0810846   54.9715254 
   35.0806329   54.9712076 
   35.0807058   54.9706975 
   35.0818007   54.9697619 
   35.0788615   54.9684013 
   35.0758581   54.9671986 
   35.0714133   54.970802  
   35.0682601   54.9703881 
   35.0626335   54.9706981 
   35.0605764   54.9702018 
   35.057877    54.9699889 
   35.0565994   54.9704147 
   35.055928    54.9708234 
   35.0550333   54.9703438 
   35.0536939   54.9704502 
   35.0526018   54.9702728 
   35.0517569   54.9699298 
   35.0510151   54.9698351 
   35.0498405   54.9692556 
   35.0492635   54.9686286 
   35.0492429   54.966251  
   35.0499023   54.9658488 
   35.0510357   54.9656241 
   35.0535291   54.9640862 
   35.0535497   54.9632108 
   35.0533436   54.9621816 
   35.0528078   54.9611641 
   35.0521896   54.9603833 
   35.0510769   54.9598627 
   35.0495108   54.9595078 
   35.0481508   54.9594368 
   35.0428343   54.9593658 
   35.0383009   54.9598982 
   35.0361579   54.9600875 
   35.0346164   54.9600303 
   35.0328654   54.9595504 
   35.0309239   54.9585613 
   35.0302644   54.9577804 
   35.0302644   54.9556506 
   35.0310887   54.9548342 
   35.031439    54.954207  
   35.0329433   54.9532959 
   35.0341591   54.9524202 
   35.0358076   54.9515919 
   35.0370233   54.9513552 
   35.0387258   54.9512099 
   35.0404262   54.9509077 
   35.0431022   54.9505505 
   35.0439059   54.950231  
   35.0443798   54.9485505 
   35.0444416   54.9470594 
   35.0441325   54.9463138 
   35.0435144   54.9455208 
   35.041598    54.9448698 
   35.0406501   54.9446923 
   35.0395766   54.9446813 
   35.0384864   54.9452841 
   35.0372294   54.945509  
   35.0364258   54.9454143 
   35.0350451   54.9444438 
   35.0341385   54.9435916 
   35.0331906   54.9423251 
   35.032799    54.9421831 
   35.0326136   54.9414137 
   35.031913    54.9413072 
   35.0302232   54.9412717 
   35.028719    54.940739  
   35.0273796   54.9409876 
   35.0244741   54.9410231 
   35.0230484   54.9413913 
   35.0213007   54.9419345 
   35.0180449   54.943343  
   35.0173236   54.9442899 
   35.0165818   54.9446923 
   35.0148463   54.9451082 
   35.0140472   54.9452367 
   35.0131611   54.9454498 
   35.0115745   54.9455326 
   35.0094108   54.9452722 
   35.008092    54.9447041 
   35.0071235   54.9439348 
   35.0065465   54.9426802 
   35.0066083   54.9422068 
   35.0070617   54.9416978 
   35.0072471   54.940526  
   35.0077417   54.9396855 
   35.0084011   54.9391292 
   35.0086071   54.9384663 
   35.0085041   54.9378389 
   35.0095962   54.9367143 
   35.0104435   54.9360769 
   35.0110814   54.9356833 
   35.012393    54.9351452 
   35.0148646   54.9344151 
   35.0165485   54.934049  
   35.0180414   54.9340286 
   35.020074    54.9338921 
   35.0207533   54.9337469 
   35.0212768   54.933515  
   35.021664    54.9331604 
   35.0226047   54.9319588 
   35.0230934   54.9313987 
   35.0235262   54.9302384 
   35.0238353   54.9293504 
   35.0236977   54.928611  
   35.0235394   54.9279694 
   35.0237075   54.926686  
   35.0236755   54.9264145 
   35.0237555   54.9261615 
   35.0236158   54.9258043 
   35.0234031   54.9254245 
   35.0220103   54.9244409 
   35.0202334   54.9239553 
   35.0189334   54.9235953 
   35.018293    54.9234619 
   35.0171368   54.9235383 
   35.0151147   54.924129  
   35.0118488   54.9247505 
   35.0084553   54.926149  
   35.0077573   54.92619   
   35.0070688   54.9263142 
   35.0061756   54.9262674 
   35.0052674   54.926167  
   35.004699    54.9259968 
   35.0039832   54.92562   
   35.0026415   54.9246534 
   35.0022124   54.9241652 
   35.0018169   54.9237425 
   35.0008002   54.9229282 
   35.0004319   54.9228316 
   34.9996841   54.9220389 
   34.9993525   54.9211642 
   34.9996473   54.9200388 
   35.0002638   54.9188149 
   35.0006733   54.9182726 
   35.0014651   54.9175436 
   35.0017769   54.9170894 
   35.0017769   54.916689  
   35.0016822   54.9161685 
   35.0010964   54.9156905 
   35.000037    54.9152355 
   34.9994072   54.9151521 
   34.9989908   54.9151889 
   34.9981582   54.9151475 
   34.9969397   54.9150306 
   34.9955323   54.91483   
   34.9934242   54.9148454 
   34.9926902   54.9147471 
   34.9919617   54.9145263 
   34.9916257   54.9143701 
   34.9907048   54.9142133 
   34.9886153   54.9141995 
   34.9878633   54.9143169 
   34.9873343   54.9145079 
   34.9863452   54.9148929 
   34.985063    54.9151501 
   34.9843001   54.9152395 
   34.9832673   54.9152027 
   34.982779    54.9151107 
   34.9823489   54.914912  
   34.9820681   54.9147419 
   34.981981    54.9145681 
   34.9819063   54.9142909 
END

END
