package Linux::Sys::CPU::Affinity;

use 5.026001;
use strict;
use warnings;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
  	cpu_setaffinity
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw();

our $VERSION = '0.01';

require XSLoader;
XSLoader::load('Linux::Sys::CPU::Affinity', $VERSION);

1;
__END__
=head1 NAME

Linux::Sys::CPU::Affinity - Perl XS extension for setupping CPU affinity

=head1 SYNOPSIS

  use Linux::Sys::CPU::Affinity ();

  my @cpus = (0, 10, 70 .. 80);
  my $ca = Linux::Sys::CPU::Affinity->new(\@cpus);

  # the same as @cpus used above is empty
  my $ca = Linux::Sys::CPU::Affinity->new();

  $ca->cpu_zero();

  $ca->reset(\@new_cpus);

  # the same as @new_cpus used above is empty
  $ca->reset();

  $ca->cpu_clr($cpu);

  $ca->cpu_set($cpu);

  my $result = $ca->set_affinity($pid);

  my $isset = $ca->cpu_isset($cpu);

  $ca->DESTROY();

=head1 DESCRIPTION

This module allows you to pin any process by its PID to some CPU's group.

=head2 new

Constructor. It receives an array reference with CPU's number to be used in set creation.
In case if the argument isn't specified, the empty array will be used.

According to the cpu_set(3) Linux man page, the first available CPU on the system corresponds to a cpu value of 0,
the next CPU corresponds to a cpu value of 1, and so on. The constant CPU_SETSIZE (currently 1024) specifies a value
one greater than the maximum CPU number that can be used in set.

The size of created set will be equal to the amoumt of available CPU cores.
If code is failed to get that amount, then the CPU_SETSIZE constant will be used instead.

Returns an instance of class.

=head2 reset

It receives an array reference with CPU's number to be used in set creation.
In case if the argument isn't specified, the empty array will be used.

If the set had been set set before this method was invoked, then the old set will be destroyed,
but it won't be applied automatically.

=head2 cpu_zero

Clears set, so that it contains no CPUs.
New set won't be applied automatically.

See CPU_ZERO, CPU_ZERO_S in cpu_set(3) Linux man page.

=head2 set_affinity

It receives the PID number and applies previously created set to the specified process.

Returns -1 on error, otherwise returns 0.

See the method analog in sched_setaffinity(2) Linux man page.

=head2 cpu_count

Returns the number of CPUs in set.

See CPU_COUNT, CPU_COUNT_S in cpu_set(3) Linux man page.

=head2 cpu_isset

It receives the CPU number.

Returns nonzero if cpu is in set; otherwise, it returns 0.

=head2 cpu_set

Adds the received CPU in to the set.
New set won't be applied automatically.

=head2 cpu_clr

Removes the received CPU from the set.
New set won't be applied automatically.

=head2 DESTROY

Destructor.

=cut

=head1 AUTHOR

Chernenko Dmitriy <d.chernenko@corp.mail.ru>

=cut
