/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1996-2001
 *      Sleepycat Software.  All rights reserved.
 */
/*
 * This code originally written by Adam Stubblefield - astubble@rice.edu
 */
#ifndef lint
static const char revid[] = "$Id: aes_method.c,v 1.2 2002/01/08 18:54:17 sue Exp $";
#endif /* not lint */


#include "db_config.h"

#include "db_int.h"
#include "crypto.h"
#include "crypto_ext.h"

static void __aes_err __P((DB_ENV *, int));
static int __aes_derivekeys __P((DB_ENV *, u_int8_t *, size_t, AES_CIPHER *));

/*
 * __aes_setup --
 *	Setup AES functions.
 *
 * PUBLIC: int __aes_setup __P((DB_ENV *, DB_CIPHER *));
 */
int
__aes_setup(dbenv, db_cipher)
	DB_ENV *dbenv;
	DB_CIPHER *db_cipher;
{
	AES_CIPHER *aes_cipher;
	int ret;

	db_cipher->close = __aes_close;
	db_cipher->decrypt = __aes_decrypt;
	db_cipher->encrypt = __aes_encrypt;
	db_cipher->init = __aes_init;
	if ((ret = __os_calloc(dbenv, 1, sizeof(AES_CIPHER), &aes_cipher)) != 0)
		return (ret);
	db_cipher->data = aes_cipher;
	return (0);
}

/*
 * __aes_close --
 *	Destroy the AES encryption instantiation.
 *
 * PUBLIC: int __aes_close __P((DB_ENV *, void *));
 */
int
__aes_close(dbenv, data)
	DB_ENV *dbenv;
	void *data;
{
	__os_free(dbenv, data, sizeof(AES_CIPHER));
	return (0);
}

/*
 * __aes_decrypt --
 *	Decrypt data with AES.
 *
 * PUBLIC: int __aes_decrypt __P((DB_ENV *, void *, void *,
 * PUBLIC:     u_int8_t *, size_t));
 */
int
__aes_decrypt(dbenv, aes_data, iv, cipher, cipher_len)
	DB_ENV *dbenv;
	void *aes_data;
	void *iv;
	u_int8_t *cipher;
	size_t cipher_len;
{
	AES_CIPHER *aes;
	cipherInstance c;
	int ret;

	aes = (AES_CIPHER *)aes_data;
	if (iv == NULL || cipher == NULL)
		return (EINVAL);
	if ((cipher_len % 16) != 0)
		return (EINVAL);
	/*
	 * Initialize the cipher
	 */
	if ((ret = __db_cipherInit(&c, MODE_CBC, iv)) < 0) {
		__aes_err(dbenv, ret);
		return (EAGAIN);
	}

	/* Do the decryption */
	if ((ret = __db_blockDecrypt(&c, &aes->decrypt_ki, cipher,
	    cipher_len * 8, cipher)) < 0) {
		__aes_err(dbenv, ret);
		return (EAGAIN);
	}
	return (0);
}

/*
 * __aes_encrypt --
 *	Encrypt data with AES.
 *
 * PUBLIC: int __aes_encrypt __P((DB_ENV *, void *, void *,
 * PUBLIC:     u_int8_t *, size_t));
 */
int
__aes_encrypt(dbenv, aes_data, iv, data, data_len)
	DB_ENV *dbenv;
	void *aes_data;
	void *iv;
	u_int8_t *data;
	size_t data_len;
{
	AES_CIPHER *aes;
	cipherInstance c;
	int ret;

	/*
	 * XXX:
	 * Need to know where to put IV and checksum after encrypting.
	 * Need to 
	 */
	aes = (AES_CIPHER *)aes_data;
	if (aes == NULL || data == NULL)
		return (EINVAL);
	if ((data_len % 16) != 0)
		return (EINVAL);
	if ((ret = __db_generate_iv(dbenv, iv)) != 0)
		return (ret);

	/*
	 * Initialize the cipher
	 */
	if ((ret = __db_cipherInit(&c, MODE_CBC, iv)) < 0) {
		__aes_err(dbenv, ret);
		return (EAGAIN);
	}

	/* Do the encryption */
	if ((ret = __db_blockEncrypt(&c, &aes->encrypt_ki, data, data_len * 8,
	    data)) < 0) {
		__aes_err(dbenv, ret);
		return (EAGAIN);
	}
	return (0);
}

/*
 * __aes_init --
 *	Initialize the AES encryption instantiation.
 *
 * PUBLIC: int __aes_init __P((DB_ENV *, void *));
 */
int
__aes_init(dbenv, data)
	DB_ENV *dbenv;
	void *data;
{
	AES_CIPHER *aes;

	aes = (AES_CIPHER *)data;
	return (__aes_derivekeys(dbenv, dbenv->passwd,
	    dbenv->passwd_len, aes));
}

static int
__aes_derivekeys(dbenv, passwd, plen, aes)
	DB_ENV *dbenv;
	u_int8_t *passwd;
	size_t plen;
	AES_CIPHER *aes;
{
	SHA1_CTX ctx;
	int ret;
	u_int8_t temp[DB_MAC_KEY];

	if (passwd == NULL)
		return (EINVAL);

	/* Compute the MAC key. mac_key must be 20 bytes. */
	__db_SHA1Init(&ctx);
	__db_SHA1Update(&ctx, passwd, plen);
	__db_SHA1Update(&ctx, DB_MAC_MAGIC, strlen(DB_MAC_MAGIC));
	__db_SHA1Update(&ctx, passwd, plen);
	__db_SHA1Final(aes->mac_key, &ctx);

	/* Derive the crypto keys */
	__db_SHA1Init(&ctx);
	__db_SHA1Update(&ctx, passwd, plen);
	__db_SHA1Update(&ctx, DB_ENC_MAGIC, strlen(DB_ENC_MAGIC));
	__db_SHA1Update(&ctx, passwd, plen);
	__db_SHA1Final(temp, &ctx);

	if ((ret = __db_makeKey(&aes->encrypt_ki, DIR_ENCRYPT,
	    DB_AES_KEYLEN, temp)) != TRUE) {
		__aes_err(dbenv, ret);
		return (EAGAIN);
	}
	if ((ret = __db_makeKey(&aes->decrypt_ki, DIR_DECRYPT,
	    DB_AES_KEYLEN, temp)) != TRUE) {
		__aes_err(dbenv, ret);
		return (EAGAIN);
	}
	return (0);
}

/*
 * __aes_err --
 *	Handle AES-specific errors.  Codes and messages derived from
 *	rijndael/rijndael-api-fst.h.
 */
static void
__aes_err(dbenv, err)
	DB_ENV *dbenv;
	int err;
{
	char *errstr;

	switch (err) {
	case BAD_KEY_DIR:
		errstr = "AES key direction is invalid";
		break;
	case BAD_KEY_MAT:
		errstr = "AES key material not of correct length";
		break;
	case BAD_KEY_INSTANCE:
		errstr = "AES key passwd not valid";
		break;
	case BAD_CIPHER_MODE:
		errstr = "AES cipher in wrong state (not initialized)";
		break;
	case BAD_BLOCK_LENGTH:
		errstr = "AES bad block length";
		break;
	case BAD_CIPHER_INSTANCE:
		errstr = "AES cipher instance is invalid";
		break;
	case BAD_DATA:
		errstr = "AES data contents are invalid";
		break;
	case BAD_OTHER:
		errstr = "AES unknown error";
		break;
	default:
		errstr = "AES error unrecognized";
		break;
	}
	__db_err(dbenv, errstr);
	return;
}

