;==============================================================================

; file: "host.scm"

;------------------------------------------------------------------------------
;
; Host system interface:
; ---------------------

; This package contains definitions to interface to the host system in which
; the compiler is loaded.  This is the only package that contains non-portable
; scheme code.  So one should be able to port the compiler to another system by
; adjusting this file.  The global variable 'host-system' is assumed to contain
; the name of the host system.

;------------------------------------------------------------------------------

; The host dependent variables:
; ----------------------------

; 'open-input-file*' is like open-input-file but returns #f when the named
; file does not exist.

(define open-input-file* '())

; 'pp-expression' is used to pretty print an expression on a given port.

(define pp-expression '())

; 'write-returning-len' is like 'write' but it returns the number of
; characters that were written out.

(define write-returning-len '())

; 'display-returning-len' is like 'display' but it returns the number of
; characters that were written out.

(define display-returning-len '())

; 'write-word' is used to write out files containing binary data.

(define write-word '())

; Various characters

(define char-newline '())
(define char-tab     '())
(define char-esc     '())

; 'character-encoding' is used to convert Scheme characters into their
; corresponding machine representation.

(define character-encoding '())

; Highest value returned by 'character-encoding'.

(define max-character-encoding '())

; 'fatal-err' is used to signal non recoverable errors.

(define (fatal-err msg arg)
  (display msg)
  (display " ")
  (write arg)
  (newline)
  (display "*** Entering infinite loop... (send interrupt to quit)")
  (newline)
  (let loop () (begin (read-char) (loop))))


; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Set the variables according to host:
; -----------------------------------

(case host-system


  ((GAMBIT)

   (set! open-input-file* open-input-file)

   (set! pp-expression
     (lambda (expr port)
       (newline port)
       (write expr port)
       (newline port)))

   (set! write-returning-len write)
   (set! display-returning-len display)

   (set! write-word
     (lambda (w p)
       (write-char (integer->char (quotient w 256)) p)
       (write-char (integer->char (modulo w 256)) p)))

   (set! char-newline (integer->char 10))
   (set! char-tab     (integer->char 9))
   (set! char-esc     (integer->char 27))

   (set! character-encoding
     (lambda (c) (char->integer c)))

   (set! max-character-encoding 255)

   (set! fatal-err
     (lambda (msg arg) (write (list msg arg)) (newline) (car msg)))

  )


  ((MIT)

   (set! open-input-file* open-input-file)

   (set! pp-expression
     (lambda (expr port)
       (pp expr port)
       (newline port)))

   (set! write-returning-len
     (lambda (x p)
       (write x p)
       1))

   (set! display-returning-len
     (lambda (x p)
       (display x p)
       1))

   (set! write-word
     (lambda (w p)
       (write-char (integer->char (quotient w 256)) p)
       (write-char (integer->char (modulo w 256)) p)))

   (set! char-newline (integer->char 10))
   (set! char-tab     (integer->char 9))
   (set! char-esc     (integer->char 27))

   (set! character-encoding
     (lambda (c) (char->integer c)))

   (set! max-character-encoding 255)

   (set! fatal-err
     (lambda (msg arg) (error msg arg)))

  )


  ((T)

   (set! open-input-file* open-input-file)

   (set! pp-expression
     (lambda (expr port)
       (newline port)
       (write expr port)
       (newline port)))

   (set! write-returning-len
     (lambda (x p)
       (write x p)
       1))

   (set! display-returning-len
     (lambda (x p)
       (display x p)
       1))

   (set! write-word
     (lambda (w p)
       (write-char (integer->char (quotient w 256)) p)
       (write-char (integer->char (modulo w 256)) p)))

   (set! char-newline (integer->char 10))
   (set! char-tab     (integer->char 9))
   (set! char-esc     (integer->char 27))

   (set! character-encoding
     (lambda (c) (char->integer c)))

   (set! max-character-encoding 255)

   (set! fatal-err
     (lambda (msg arg) (error msg arg)))

   (set! case-fell-off-end '())

   (let ((original-number->string number->string))
     (set! number->string
       (lambda (n) (original-number->string n '(HEUR)))))

   (let ((original-string->number string->number))
     (set! string->number
       (lambda (str . radix)
         (let ((r (cond ((null? radix)      "")
                        ((= (car radix) 2)  "#b")
                        ((= (car radix) 8)  "#o")
                        ((= (car radix) 16) "#x")
                        (else               ""))))
           (let ((n (original-string->number (string-append r str))))
             (if (number? n) n #f))))))

   (let ((original-make-vector make-vector))
     (set! make-vector
       (lambda (len . init)
         (let ((v (original-make-vector len)))
           (if (not (null? init))
             (let loop ((i (- len 1)))
               (if (>= i 0)
                 (begin (vector-set! v i (car init)) (loop (- i 1))))))
           v))))

   (set! exact?
     (lambda (x) (or (integer? x) (rational? x))))

   (set! inexact?
     (lambda (x) (not (exact? x))))

   (set! make-string
     (lambda l (list->string (vector->list (apply make-vector l)))))

   (set! string
     (lambda l (list->string l)))

   (set! list?
     (lambda (x)
       (let loop ((l1 x) (l2 x))
         (if (not (pair? l1))
           (null? l1)
           (let ((l1 (cdr l1)))
             (cond ((eq? l1 l2) #f)
                   ((pair? l1)  (loop (cdr l1) (cdr l2)))
                   (else        (null? l1))))))))

   (set! lcm          (lambda l (fatal-err "not implemented" 'lcm)))
   (set! numerator    (lambda l (fatal-err "not implemented" 'numerator)))
   (set! denominator  (lambda l (fatal-err "not implemented" 'denominator)))
   (set! floor        (lambda l (fatal-err "not implemented" 'floor)))
   (set! ceiling      (lambda l (fatal-err "not implemented" 'ceiling)))
   (set! truncate     (lambda l (fatal-err "not implemented" 'truncate)))
   (set! round        (lambda l (fatal-err "not implemented" 'round)))
   (set! rationalize  (lambda l (fatal-err "not implemented" 'rationalize)))
   (set! make-rectangular
     (lambda l (fatal-err "not implemented" 'make-rectangular)))
   (set! make-polar   (lambda l (fatal-err "not implemented" 'make-polar)))
   (set! real-part    (lambda l (fatal-err "not implemented" 'real-part)))
   (set! imag-part    (lambda l (fatal-err "not implemented" 'imag-part)))
   (set! magnitude    (lambda l (fatal-err "not implemented" 'magnitude)))
   (set! angle        (lambda l (fatal-err "not implemented" 'angle)))
   (set! exact->inexact
     (lambda l (fatal-err "not implemented" 'exact->inexact)))
   (set! inexact->exact
     (lambda l (fatal-err "not implemented" 'inexact->exact)))
   (set! string<?     (lambda l (fatal-err "not implemented" 'string<?)))
   (set! string>?     (lambda l (fatal-err "not implemented" 'string>?)))
   (set! string<=?    (lambda l (fatal-err "not implemented" 'string<=?)))
   (set! string>=?    (lambda l (fatal-err "not implemented" 'string>=?)))
   (set! string-ci=?  (lambda l (fatal-err "not implemented" 'string-ci=?)))
   (set! string-ci<?  (lambda l (fatal-err "not implemented" 'string-ci<?)))
   (set! string-ci>?  (lambda l (fatal-err "not implemented" 'string-ci>?)))
   (set! string-ci<=? (lambda l (fatal-err "not implemented" 'string-ci<=?)))
   (set! string-ci>=? (lambda l (fatal-err "not implemented" 'string-ci>=?)))

  )


  (else

   (display "The host system '")
   (display host-system)
   (display "' is not known.")
   (newline)
   (display "You must edit file 'host.scm' to account for that system.")
   (newline)
   (fatal-err "Unknown host system" host-system)))

;==============================================================================
