/* GAMBIT Scheme program loader for M68000 machines */

#include "params.h"
#include "gambit.h"
#include "struct.h"
#include "os.h"
#include "mem.h"
#include "strings.h"
#include "load.h"
#include "run.h"
#include "link.h"


/*---------------------------------------------------------------------------*/

/* Global data structures */

SSTATE_PTR sstate;  /* pointer to system state           */
PSTATE_PTR pstate;  /* pointer to this processor's state */


/*---------------------------------------------------------------------------*/


void usage_err( name )
char *name;
{ os_warn( "Usage: %s [arg]...\n", (long)name );
  os_warn( "           [--\n", 0L );
  os_warn( "            [-sSTACK_SIZE_IN_KILOBYTES]\n", 0L );
  os_warn( "            [-hHEAP_SIZE_IN_KILOBYTES]\n", 0L );
  os_warn( "            [-cCONST_SIZE_IN_KILOBYTES]\n", 0L );
  os_warn( "            [-d[DEBUG_LEVEL]]\n", 0L );
  os_warn( "            [-vGLOBAL_VARIABLE]...\n", 0L );
  os_warn( "            [-p]\n", 0L );
  os_warn( "           ]\n", 0L );
  os_quit();
}


void main_gambit( argc, argv, envp )
int argc;
char *argv[], *envp[];
{ long nb_processors;
  long stack_len, heap_len, const_len;
  long i, nb_args;


  /* compute number of arguments to the program */

  nb_args = argc;
  for(i=1; i<argc; i++)
  { char *arg = argv[i];
    if ((arg[0] == '-') && (arg[1] == '-') && (arg[2] == '\0'))
    { nb_args = i; break; }
  }


  /* get size of stack, heap and constant area */

  if (link_stack_length_in_k < 0)
    stack_len = ((long)DEFAULT_STACK_LENGTH_IN_K)*K;
  else
    stack_len = link_stack_length_in_k*K;

  if (link_heap_length_in_k < 0)
    heap_len = ((long)DEFAULT_HEAP_LENGTH_IN_K)*K;
  else
    heap_len = link_heap_length_in_k*K;

  if (link_const_length_in_k < 0)
  { const_len = 0;
    for (i=0; i<link_nb_ofiles; i++) const_len += *(link_sizeof_ofiles[i]);
    const_len += ((long)ADDITIONAL_CONST_LENGTH_IN_K)*K;
  }
  else
    const_len = link_const_length_in_k*K;

  for(i=nb_args+1; i<argc; i++)
  { char *arg = argv[i];
    if (*arg == '-')
    { if (arg[1] == 's')
        stack_len = string_to_int( &arg[2] )*K;
      else if (arg[1] == 'h')
        heap_len = string_to_int( &arg[2] )*K;
      else if (arg[1] == 'c')
        const_len = string_to_int( &arg[2] )*K;
    }
  }

  if (stack_len < ((long)MIN_STACK_LENGTH_IN_K)*K)
  { stack_len = ((long)MIN_STACK_LENGTH_IN_K)*K;
    os_warn( "Minimum size stack (%dK) is being allocated\n",
             (long)MIN_STACK_LENGTH_IN_K );
  }

  if (heap_len < ((long)MIN_HEAP_LENGTH_IN_K)*K)
  { heap_len = ((long)MIN_HEAP_LENGTH_IN_K)*K;
    os_warn( "Minimum size heap (%dK) is being allocated\n",
             (long)MIN_HEAP_LENGTH_IN_K );
  }


  /* setup global system memory */

  nb_processors = init_system_mem( const_len );


  { void (*kernel_startup)();

    /* handle arguments */

    sstate->program_filename = argv[0];
    sstate->profiling = 0;
    sstate->debug = 0;

    for(i=nb_args+1; i<argc; i++)
    { char *arg = argv[i];
      if (*arg == '-')
      { if ((arg[1] == 's') || (arg[1] == 'h') || (arg[1] == 'c'))
          ;
        else if (arg[1] == 'd')
          if (arg[2] != '\0')
            sstate->debug = string_to_int( &arg[2] );
          else
            sstate->debug = 1;
        else if (arg[1] == 'v')
          ; /* will be handled later */
        else if (arg[1] == 'p')
          sstate->profiling = 1;
        else
          usage_err( argv[0] );
      }
      else
        usage_err( argv[0] );
    }


    /* setup each processor's memory */

    init_processor_mem( nb_processors, stack_len, heap_len, const_len );


    /* setup table of object files to load */

    init_runtime();

    for (i=0; i<link_nb_ofiles; i++)
    { long size = *(link_sizeof_ofiles[i]);
      if (size < 0)
        ((void (*)())link_ofiles[i])();
      else
        init_ofile( (char *)(link_ofiles[i]), size );
    }


    /* load the program */

    kernel_startup = (void (*)())init_program( nb_args, argv, envp );


    /* print value of global variables */

    for(i=nb_args+1; i<argc; i++)
    { char *arg = argv[i];
      if ((*arg == '-') && (arg[1] == 'v')) print_global_var( &arg[2] );
    }


    /* start executing the program */

    start_program( kernel_startup );

  }

  os_quit();
}


/*---------------------------------------------------------------------------*/
