/* Xlib interface for Gambit Scheme. */

/*
 * One common vector argument contains a Display pointer, a Screen, a
 * root Window, a Black Pixel, a White Pixel, and a Connection number.
 * See below for which index each is stored at.  Individual calls must
 * be made to initialize each item. They do not have to all be
 * initialized, only if they are used, e.g. the connection number may
 * be infrequently used so it does not have to be initialized. The
 * vector does not even have to be long enough for unused items at the
 * end of the vector. The indices are fixed for each call, e.g. the
 * Display pointer will always be expected in the first element, the
 * root Window in the second, and so on.
 */

/*
 * Some functions take unscanned vectors of Windows along with an
 * index indicating which Window to use in the function. There are
 * also similar uses of vectors of XFontStructs, GCs, GCValues, etc.
 * These vectors do not have to be homogeneous, e.g. there can be a
 * vector of a window, a GC, and a XFontStruct. As long as the index
 * for a window results in a window, etc. Vectors are used to save the
 * overhead of storing a 32-bit C pointer in Scheme. The pointer
 * cannot be stored as an immediate, and subtypes have a 32-bit header
 * in addition to the data. Better to have the 32-bit overhead per N
 * objects than for each one. This is meant as a low level interface.
 * Higher level systems should be built on top.
 */

#include "../../LOADER/params.h"
#include "../../LOADER/gambit.h"
#include "../../LOADER/struct.h"
#include "../../LOADER/os.h"
#include "../../LOADER/mem.h"
#include "../../LOADER/strings.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#ifndef ceiling4
#define ceiling4(x) ((((long)(x)) + 3L) & -4L)
#endif

static char *malloc4( length ) /* alloc of blocks starting at an octuple adr */
long length;
{ 
  char *ptr;
  ptr = (char *)malloc( length+3 );
  if (ptr != NULL) ptr = (char *) ceiling4(ptr);
  return ptr;
}

#define DISP_INDEX 0
#define SCRN_INDEX 1
#define ROOT_INDEX 2
#define BPIX_INDEX 3
#define WPIX_INDEX 4
#define CONN_INDEX 5

SCM_obj xDconnectionDnumber( vect )
SCM_obj vect;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * vect[CONN_INDEX] --- MOD --- Connection number (net socket) (int).
    * Returns SCM_undef.
    */

   int fd;
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   fd = ConnectionNumber(dpy);
   SCM_obj_to_vect(vect)[CONN_INDEX] = (long) fd;
   
   return SCM_undef;
}

SCM_obj xDdefaultDscreen( vect )
SCM_obj vect;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * vect[SCRN_INDEX] --- MOD --- default Screen (int).
    * Returns SCM_undef.
    */

   int scr;
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   scr = DefaultScreen(dpy);
   SCM_obj_to_vect(vect)[SCRN_INDEX] = (long) scr;

   return SCM_undef;
}

SCM_obj xDdefaultDrootDwindow( vect )
SCM_obj vect;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * vect[ROOT_INDEX] --- MOD --- root Window.
    * Returns SCM_undef.
    */

   Window root;
   Display *dpy;

   dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   root = DefaultRootWindow(dpy);
   SCM_obj_to_vect(vect)[ROOT_INDEX] = (long) root;

   return SCM_undef;
}

SCM_obj xDblackDpixel( vect )
SCM_obj vect;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * vect[SCRN_INDEX] --- IN  --- Screen.
    * vect[BPIX_INDEX] --- MOD --- black pixel (unsigned long).
    * Returns SCM_undef.
    */

   int scr;
   unsigned long pixel;

   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   scr = (int) SCM_obj_to_vect(vect)[SCRN_INDEX];
   pixel = BlackPixel(dpy, scr);
   SCM_obj_to_vect(vect)[BPIX_INDEX] = (long) pixel;

   return SCM_undef;
}

SCM_obj xDwhiteDpixel( vect )
SCM_obj vect;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * vect[SCRN_INDEX] --- IN  --- Screen.
    * vect[WPIX_INDEX] --- MOD --- white pixel (unsigned long).
    * Returns SCM_undef.
    */

   int scr;
   unsigned long pixel;

   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   scr = (int) SCM_obj_to_vect(vect)[SCRN_INDEX];
   pixel = WhitePixel(dpy, scr);
   SCM_obj_to_vect(vect)[WPIX_INDEX] = (long) pixel;

   return SCM_undef;
}

SCM_obj xDopenDdisplay( str, vect )
SCM_obj str;
SCM_obj vect;
{
   /*
    * vect[DISP_INDEX] --- OUT --- Display pointer.
    * str ---------------- IN  --- Scheme string, display name.
    * Returns SCM_undef.
    */

   Display *dpy;
   char *name;
   char *mark = local_mark();       /* put mark on local C heap          */
   name = string_to_c_str( str );   /* convert Scheme string to C string */
   dpy = XOpenDisplay(name);
   SCM_obj_to_vect(vect)[DISP_INDEX] = (long) dpy;
   local_release( mark );           /* get rid of converted string       */

   return SCM_undef;
}

SCM_obj xDcloseDdisplay( vect )
SCM_obj vect;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * Returns SCM_undef.
    */

   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   XCloseDisplay(dpy);
   
   return SCM_undef;
}

SCM_obj xDsynchronize( vect, onoffp )
SCM_obj vect;
SCM_obj onoffp;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * onoffp ------------- IN  --- Scheme boolean.
    * Returns SCM_undef.
    */

   Display *dpy;
   int onoff;

   dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   onoff = (SCM_false != onoffp);
   XSynchronize(dpy, onoff);
   
   return SCM_undef;            /* don't return the after function? */
}

SCM_obj xDmapDsubwindows( vect, windows, win_index )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * windows ------------ IN  --- unscanned vector.
    * win_index ---------- IN  --- index of Window in windows vector, Scheme integer.
    * Returns SCM_undef.
    */

   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window win = (Window) SCM_obj_to_vect(windows)[i];
   XMapSubwindows(dpy, win);

   return SCM_undef;
}

SCM_obj xDmapDraised( vect, windows, win_index )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * windows ------------ IN  --- unscanned vector.
    * win_index ---------- IN  --- index of Window in windows vector, Scheme integer.
    * Returns SCM_undef.
    */

   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window win = (Window) SCM_obj_to_vect(windows)[i];
   XMapRaised(dpy, win);

   return SCM_undef;
}

SCM_obj makeDxDevent( events, ev_index )
SCM_obj events;
SCM_obj ev_index;
{
   /*
    * events ----- MOD --- unscanned vector for new XEvent pointer.
    * ev_index --- IN  --- index of XEvent in events vector, Scheme integer.
    * Returns SCM_undef.
    */

   XEvent *event = (XEvent *)malloc4(sizeof(XEvent)); /* use c-free to free this space. */
   int i = SCM_obj_to_int(ev_index);
   SCM_obj_to_vect(events)[i] = (long) event;
   
   return SCM_undef;
}

SCM_obj xDeventDtype( events, ev_index )
SCM_obj events;
SCM_obj ev_index;
{
   /*
    * events ----- IN  --- unscanned vector.
    * ev_index --- IN  --- index of XEvent in events vector, Scheme integer.
    * Returns the indexed event's type, a Scheme integer.
    */

   int i = SCM_obj_to_int(ev_index);
   XEvent *event = (XEvent *) SCM_obj_to_vect(events)[i];

   return SCM_int_to_obj(event->type);
}

SCM_obj xDeventDwindow( events, ev_index, windows, win_index )
SCM_obj events;
SCM_obj ev_index;
SCM_obj windows;
SCM_obj win_index;
{
   /*
    * Store the window in the windows vector at win_index.
    * events ----- IN  --- unscanned vector.
    * ev_index --- IN  --- index of XEvent in events vector, Scheme integer.
    * windows ---- MOD --- unscanned vector.
    * win_index -- IN  --- index for Window result in windows vector, Scheme integer.
    * Returns SCM_undef.
    */

   int i = SCM_obj_to_int(ev_index);
   int j = SCM_obj_to_int(win_index);
   XEvent *event = (XEvent *) SCM_obj_to_vect(events)[i];
   SCM_obj_to_vect(windows)[j] = event->xany.window;

   return SCM_undef;
}

SCM_obj xDeventDwindowDindex( events, ev_index, windows )
SCM_obj events;
SCM_obj ev_index;
SCM_obj windows;
{
   /*
    * Lookup the window in the windows vector and return the index or #f.
    * events ----- IN  --- unscanned vector.
    * ev_index --- IN  --- index of XEvent in events vector, Scheme integer.
    * windows ---- IN  --- unscanned vector.
    * Returns SCM_false or a Scheme integer.
    */

   int j;
   int len = SCM_length(windows);
   int i = SCM_obj_to_int(ev_index);
   XEvent *event = (XEvent *) SCM_obj_to_vect(events)[i];
   Window win = event->xany.window;

   for (j = 0; j < len; j++) {
      if (win == SCM_obj_to_vect(windows)[j])
         return SCM_int_to_obj(j);
   }

   return SCM_false;
}

SCM_obj xDbuttonDeventDx( events, ev_index )
SCM_obj events;
SCM_obj ev_index;
{
   /*
    * events ----- IN  --- unscanned vector.
    * ev_index --- IN  --- index of XEvent in events vector, Scheme integer.
    * Returns x coordinate of the event, a Scheme integer.
    */

   int i = SCM_obj_to_int(ev_index);
   XEvent *event = (XEvent *) SCM_obj_to_vect(events)[i];
   int x = event->xbutton.x;

   return SCM_int_to_obj(x);
}

SCM_obj xDbuttonDeventDy( events, ev_index )
SCM_obj events;
SCM_obj ev_index;
{
   /*
    * events ----- IN  --- unscanned vector.
    * ev_index --- IN  --- index of XEvent in events vector, Scheme integer.
    * Returns y coordinate of the event, a Scheme integer.
    */

   int i = SCM_obj_to_int(ev_index);
   XEvent *event = (XEvent *) SCM_obj_to_vect(events)[i];
   int y = event->xbutton.y;

   return SCM_int_to_obj(y);
}

SCM_obj xDeventDexposeDcount( events, ev_index )
SCM_obj events;
SCM_obj ev_index;
{
   /*
    * events ----- IN  --- unscanned vector.
    * ev_index --- IN  --- index of XEvent in events vector, Scheme integer.
    * Returns expose count of the expose event, a Scheme integer.
    */

   int i = SCM_obj_to_int(ev_index);
   XExposeEvent *event = (XExposeEvent *) SCM_obj_to_vect(events)[i];

   return SCM_int_to_obj(event->count);
}

/* More XEvent setters and getters to be implemented as needed... */

SCM_obj cDfree( vect, index )
SCM_obj vect;
SCM_obj index;
{
   /*
    * vect ---- IN --- unscanned vector.
    * index --- IN --- index of pointer to be free'd.
    *
    * N.B. --- Space allocated by Xlib should be freed with x-free
    * (XFree) or another X function.
    */

   int i = SCM_obj_to_int(index);
   char *ptr = (char *) SCM_obj_to_vect(vect)[i];
   free(ptr);
   SCM_obj_to_vect(vect)[i] = 0L;

   return SCM_undef;
}

SCM_obj xDfreeDgc( vect, gcs, gc_index )
SCM_obj vect;
SCM_obj gcs;
SCM_obj gc_index;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * gcs ---------------- MOD --- unscanned vector.
    * gc_index ----------- IN  --- index of GC in gcs, Scheme integer.
    * Returns SCM_undef.
    */

   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(gc_index);
   GC gc = (GC) SCM_obj_to_vect(gcs)[i];
   XFreeGC(dpy, gc);

   return SCM_undef;
}

SCM_obj xDnextDevent( vect, events, ev_index )
SCM_obj vect;
SCM_obj events;
SCM_obj ev_index;
{
   /*
    * vect[DISP_INDEX] --- IN  --- Display pointer.
    * events ------------- MOD --- unscanned vector.
    * ev_index ----------- IN  --- index of event in events, Scheme integer.
    * Returns SCM_undef.
    */

   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(ev_index);
   XEvent *event = (XEvent *) SCM_obj_to_vect(events)[i];
   XNextEvent(dpy, event);

   return SCM_undef;
}

SCM_obj xDcreateDsimpleDwindow(
   vect,
   parent_vector,
   parent_index,
   fxx, fxy,
   fxwidth, fxheight,
   fxborder_width,
   border_pixel_vector,
   border_pixel_index,
   bgd_pixel_vector,
   bgd_pixel_index,
   new_win_vector,
   new_win_index
   )
SCM_obj vect;
SCM_obj parent_vector;             /* Parent Window in this vector.     */
SCM_obj parent_index;              /* Parent Window index.              */
SCM_obj fxx;                       
SCM_obj fxy;                       
SCM_obj fxwidth;                   
SCM_obj fxheight;                  
SCM_obj fxborder_width;            
SCM_obj border_pixel_vector;
SCM_obj border_pixel_index;
SCM_obj bgd_pixel_vector;
SCM_obj bgd_pixel_index;
SCM_obj new_win_vector;            /* New window stored in this vector. */
SCM_obj new_win_index;             /* Index of new window.              */
{
   Window simple;
   int x, y, width, height, border_width;
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int new_win_i = SCM_obj_to_int(new_win_index);
   int parent_i  = SCM_obj_to_int(parent_index);
   Window parent = (Window) SCM_obj_to_vect(parent_vector)[parent_i];
   int border_i = SCM_obj_to_int(border_pixel_index);
   int bgd_i    = SCM_obj_to_int(bgd_pixel_index);
   unsigned long border_pixel = (unsigned long) SCM_obj_to_vect(border_pixel_vector)[border_i];
   unsigned long bgd_pixel    = (unsigned long) SCM_obj_to_vect(bgd_pixel_vector)[bgd_i];
   x            = SCM_obj_to_int(fxx);
   y            = SCM_obj_to_int(fxy);
   width        = SCM_obj_to_int(fxwidth);
   height       = SCM_obj_to_int(fxheight);
   border_width = SCM_obj_to_int(fxborder_width);
   simple = XCreateSimpleWindow(
      dpy, parent, x, y, width, height, border_width, border_pixel, bgd_pixel);
   SCM_obj_to_vect(new_win_vector)[new_win_i] = simple;
   
   return SCM_undef;
}

SCM_obj xDdestroyDwindow( vect, windows, win_index )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window   win = (Window)    SCM_obj_to_vect(windows)[i];
   XDestroyWindow(dpy, win);

   return SCM_undef;
}

SCM_obj xDsetDiconDname( vect, windows, win_index, str )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
SCM_obj str;
{
   char *mark = local_mark();           /* put mark on local C heap          */
   char *name = string_to_c_str( str ); /* convert Scheme string to C string */
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window win = (Window) SCM_obj_to_vect(windows)[i];
   XSetIconName(dpy, win, name);
   local_release( mark );           /* get rid of converted string       */

   return SCM_undef;
}

SCM_obj xDstoreDname( vect, windows, win_index, str )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
SCM_obj str;
{
   char *mark = local_mark();       /* put mark on local C heap          */
   char *name = string_to_c_str( str ); /* convert Scheme string to C string */
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window win = (Window) SCM_obj_to_vect(windows)[i];
   XStoreName(dpy, win, name);
   local_release( mark );           /* get rid of converted string       */

   return SCM_undef;
}

SCM_obj xDsetDnormalDhints( vect, windows, win_index, hints_vector, hints_index )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
SCM_obj hints_vector;
SCM_obj hints_index;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window win = (Window) SCM_obj_to_vect(windows)[i];
   int j = SCM_obj_to_int(hints_index);
   XSizeHints *hints = (XSizeHints *) SCM_obj_to_vect(hints_index)[j];
   XSetNormalHints(dpy, win, hints);

   return SCM_undef;
}

SCM_obj makeDxDsizeDhints( vect, index )
SCM_obj vect;
SCM_obj index;
{
   /*
    * The caller passes in an unscanned vector with at least six
    * 16-bit elements. (In C they are indexed as 32-bit elements. In
    * Gambit Scheme they are referenced as 16-bit elements.) A pointer
    * to the hints is returned as the fifth and sizth two 16-bit
    * elements of the vector. The first four 16-bit elements are
    * unused (they are usually the Display pointer and Window in a
    * call to XSetNormalHints, and so are reserved, see
    * x-set-normal-hints). The function returns SCM_undef.
    */

   int i = SCM_obj_to_int(index);
   XSizeHints *hints = (XSizeHints *)malloc4(sizeof(XSizeHints)); /* use c-free to free this space. */
   SCM_obj_to_vect(vect)[i] = (long) hints;
   
   return SCM_undef;
}

SCM_obj setDxDsizeDhintsDminDsizeB( vect, index, fxw, fxh )
SCM_obj vect;
SCM_obj index;
SCM_obj fxw;                    /* fixnum */
SCM_obj fxh;                    /* fixnum */
{
   int i = SCM_obj_to_int(index);
   XSizeHints *hints = (XSizeHints *) SCM_obj_to_vect(vect)[i];
   int w = SCM_obj_to_int(fxw);
   int h = SCM_obj_to_int(fxh);
   hints->min_width = w;
   hints->min_height = h;

   return SCM_undef;
}

SCM_obj setDxDsizeDhintsDmaxDsizeB( vect, index, fxw, fxh )
SCM_obj vect;
SCM_obj index;
SCM_obj fxw;                    /* fixnum */
SCM_obj fxh;                    /* fixnum */
{
   int i = SCM_obj_to_int(index);
   XSizeHints *hints = (XSizeHints *) SCM_obj_to_vect(vect)[i];
   int w = SCM_obj_to_int(fxw);
   int h = SCM_obj_to_int(fxh);
   hints->max_width = w;
   hints->max_height = h;

   return SCM_undef;
}

SCM_obj setDxDsizeDhintsDflagsB( vect, index, fxflags )
SCM_obj vect;
SCM_obj index;
SCM_obj fxflags;                /* fixnum */
{
   int i = SCM_obj_to_int(index);
   XSizeHints *hints = (XSizeHints *) SCM_obj_to_vect(vect)[i];
   int flags = SCM_obj_to_int(fxflags);
   hints->flags = flags;

   return SCM_undef;
}

/* More XSizeHints setters and getters to be implemented as needed... */

SCM_obj xDlookupDstring( events, ev_index, vect2 )
SCM_obj events;
SCM_obj ev_index;
SCM_obj vect2;
{
   /*
    * vect2 is a regular Scheme vector with at least two elements.
    * The first element is a Scheme string which will be used as a
    * buffer in the call to XLookupString. The second element will be
    * set to an integer, the number of meaningful characters in the
    * buffer.
    *
    * The return value is a Scheme integer, the keysym.
    */

   SCM_obj result;
   KeySym key;
   static XComposeStatus compose = {NULL, 0}; /* unused in R3,4,5? */
   int i = SCM_obj_to_int(ev_index);
   XEvent *event = (XEvent *) SCM_obj_to_vect(events)[i];
   SCM_obj str = SCM_obj_to_vect(vect2)[0];
   long len = string_length(str);
   int count = XLookupString(event, SCM_obj_to_str(str), len, &key, &compose);
   SCM_obj_to_vect(vect2)[1] = SCM_int_to_obj(count);
   result = SCM_int_to_obj((int) key);

   return result;
}

SCM_obj xDloadDqueryDfont( vect, str, fonts, findex )
SCM_obj vect;
SCM_obj str;
SCM_obj fonts;
SCM_obj findex;
{
   int i = SCM_obj_to_int(findex);
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   char *mark = local_mark();
   char *name = string_to_c_str(str);
   XFontStruct *font = XLoadQueryFont(dpy, name); /* Free with x-free-font. */
   local_release(mark);
   SCM_obj_to_vect(fonts)[i] = (long) font;

   return SCM_undef;
}

SCM_obj xDtextDwidth( fonts, findex, str, fxstart, fxend )
SCM_obj fonts;
SCM_obj findex;
SCM_obj str;
SCM_obj fxstart;                /* Starting index in the string, inclusive. */
SCM_obj fxend;                  /* Ending index in the string, exclusive.   */
{
   /* Return the width as a Scheme integer. */

   int w;
   int i = SCM_obj_to_int(findex);
   XFontStruct *font = (XFontStruct *) SCM_obj_to_vect(fonts)[i];
   int start = SCM_obj_to_int(fxstart);
   int end   = SCM_obj_to_int(fxend);
   char *start_ptr = SCM_obj_to_str(str);
   start_ptr += start;
   w = XTextWidth(font, start_ptr, end - start);

   return SCM_int_to_obj(w);
}

SCM_obj xDclearDwindow( vect, windows, win_index )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window win = (Window) SCM_obj_to_vect(windows)[i];
   XClearWindow(dpy, win);

   return SCM_undef;
}

SCM_obj xDmoveDwindow( vect, windows, win_index, fxx, fxy )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
SCM_obj fxx;                    /* fixnum */
SCM_obj fxy;                    /* fixnum */
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window win = (Window) SCM_obj_to_vect(windows)[i];
   int x = SCM_int_to_obj(fxx);
   int y = SCM_int_to_obj(fxy);
   XMoveWindow(dpy, win, x, y);

   return SCM_undef;
}

SCM_obj xDcreateDgc( vect, drawables, dindex, gcvalues, gindex, fxmask, gc_vector, gc_index )
SCM_obj vect;
SCM_obj drawables;              /* Can be a window or a pixmap at the given index. */
SCM_obj dindex;
SCM_obj gcvalues;
SCM_obj gindex;
SCM_obj fxmask;
SCM_obj gc_vector;
SCM_obj gc_index;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(dindex);
   XID drawable = (XID) SCM_obj_to_vect(drawables)[i];
   int j = SCM_obj_to_int(gindex);
   XGCValues *values = (XGCValues *) SCM_obj_to_vect(gcvalues)[j];
   int mask = SCM_obj_to_int(fxmask);
   GC gc = XCreateGC(dpy, drawable, mask, values);
   int k = SCM_obj_to_int(gc_index);
   SCM_obj_to_vect(gc_vector)[k] = (long) gc;
   
   return SCM_undef;
}

SCM_obj makeDxDgcDvalues( vect, index )
SCM_obj vect;
SCM_obj index;
{
   XGCValues *values = (XGCValues *)malloc4(sizeof(XGCValues));   /* use c-free to free this space. */
   int i = SCM_obj_to_int(index);
   SCM_obj_to_vect(vect)[i] = (long) values;
   
   return SCM_undef;
}

SCM_obj xDsetDforeground( vect, gc_vector, gc_index, pixel_vector, pixel_index )
SCM_obj vect;
SCM_obj gc_vector;
SCM_obj gc_index;
SCM_obj pixel_vector;
SCM_obj pixel_index;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(gc_index);
   int j = SCM_obj_to_int(pixel_index);
   GC gc = (GC) SCM_obj_to_vect(gc_vector)[i];
   unsigned long pixel = (unsigned long) SCM_obj_to_vect(pixel_vector)[j];
   XSetForeground(dpy, gc, pixel);
   
   return SCM_undef;
}

SCM_obj xDsetDbackground( vect, gc_vector, gc_index, pixel_vector, pixel_index )
SCM_obj vect;
SCM_obj gc_vector;
SCM_obj gc_index;
SCM_obj pixel_vector;
SCM_obj pixel_index;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(gc_index);
   int j = SCM_obj_to_int(pixel_index);
   GC gc = (GC) SCM_obj_to_vect(gc_vector)[i];
   unsigned long pixel = (unsigned long) SCM_obj_to_vect(pixel_vector)[j];
   XSetBackground(dpy, gc, pixel);
   
   return SCM_undef;
}

SCM_obj xDsetDfont( vect, gc_vector, gc_index, fonts, findex )
SCM_obj vect;
SCM_obj gc_vector;
SCM_obj gc_index;
SCM_obj fonts;
SCM_obj findex;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(gc_index);
   int j = SCM_obj_to_int(findex);
   GC gc = (GC) SCM_obj_to_vect(gc_vector)[i];
   XFontStruct *font = (XFontStruct *) SCM_obj_to_vect(fonts)[j];
   XSetFont(dpy, gc, font->fid);
   
   return SCM_undef;
}

SCM_obj xDdrawDrectangle( vect, drawables, dindex, gc_vector, gc_index, fxx, fxy, fxwidth, fxheight )
SCM_obj vect;
SCM_obj drawables;
SCM_obj dindex;
SCM_obj gc_vector;
SCM_obj gc_index;
SCM_obj fxx;
SCM_obj fxy;
SCM_obj fxwidth;
SCM_obj fxheight;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(dindex);
   XID drawable = (XID) SCM_obj_to_vect(drawables)[i];
   int j = SCM_obj_to_int(gc_index);
   GC gc = (GC) SCM_obj_to_vect(gc_vector)[j];
   int x = SCM_obj_to_int(fxx);
   int y = SCM_obj_to_int(fxy);
   int w = SCM_obj_to_int(fxwidth);
   int h = SCM_obj_to_int(fxheight);
   XDrawRectangle(dpy, drawable, gc, x, y, w, h);

   return SCM_undef;
}

SCM_obj xDdrawDline( vect, drawables, dindex, gc_vector, gc_index, fxx1, fxy1, fxx2, fxy2 )
SCM_obj vect;
SCM_obj drawables;
SCM_obj dindex;
SCM_obj gc_vector;
SCM_obj gc_index;
SCM_obj fxx1;
SCM_obj fxy1;
SCM_obj fxx2;
SCM_obj fxy2;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(dindex);
   XID drawable = (XID) SCM_obj_to_vect(drawables)[i];
   int j = SCM_obj_to_int(gc_index);
   GC gc = (GC) SCM_obj_to_vect(gc_vector)[j];
   int x1 = SCM_obj_to_int(fxx1);
   int y1 = SCM_obj_to_int(fxy1);
   int x2 = SCM_obj_to_int(fxx2);
   int y2 = SCM_obj_to_int(fxy2);
   XDrawLine(dpy, drawable, gc, x1, y1, x2, y2);

   return SCM_undef;
}

SCM_obj xDdrawDstring( vect, drawables, dindex, gc_vector, gc_index, fxx, fxy, str, fxstart, fxend )
SCM_obj vect;
SCM_obj drawables;
SCM_obj dindex;
SCM_obj gc_vector;
SCM_obj gc_index;
SCM_obj fxx;
SCM_obj fxy;
SCM_obj str;
SCM_obj fxstart;
SCM_obj fxend;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(dindex);
   XID drawable = (XID) SCM_obj_to_vect(drawables)[i];
   int j = SCM_obj_to_int(gc_index);
   GC gc = (GC) SCM_obj_to_vect(gc_vector)[j];
   int x = SCM_obj_to_int(fxx);
   int y = SCM_obj_to_int(fxy);
   int start = SCM_obj_to_int(fxstart);
   int end = SCM_obj_to_int(fxend);
   char *start_ptr = SCM_obj_to_str(str);
   start_ptr += start;
   XDrawString(dpy, drawable, gc, x, y, start_ptr, end - start);

   return SCM_undef;
}

SCM_obj xDdrawDimageDstring( vect, drawables, dindex, gc_vector, gc_index, fxx, fxy, str, fxstart, fxend )
SCM_obj vect;
SCM_obj drawables;
SCM_obj dindex;
SCM_obj gc_vector;
SCM_obj gc_index;
SCM_obj fxx;
SCM_obj fxy;
SCM_obj str;
SCM_obj fxstart;
SCM_obj fxend;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(dindex);
   XID drawable = (XID) SCM_obj_to_vect(drawables)[i];
   int j = SCM_obj_to_int(gc_index);
   GC gc = (GC) SCM_obj_to_vect(gc_vector)[j];
   int x = SCM_obj_to_int(fxx);
   int y = SCM_obj_to_int(fxy);
   int start = SCM_obj_to_int(fxstart);
   int end = SCM_obj_to_int(fxend);
   char *start_ptr = SCM_obj_to_str(str);
   start_ptr += start;

   XDrawImageString(dpy, drawable, gc, x, y, start_ptr, end - start);

   return SCM_undef;
}

SCM_obj xDgetDfontDproperty( vect1, index1, fxatom, vect2, index2 )
SCM_obj vect1;                  /* unscanned */
SCM_obj index1;
SCM_obj fxatom;
SCM_obj vect2;                  /* scanned   */
SCM_obj index2;
{
   /* #t is returned if the property was defined, otherwise #f. */
   /* The property value is stored in the Scheme vector at the  */
   /* given index (vect2, index2).                              */
   /* Are there property values other than integers?            */

   unsigned long val;
   int j;
   int i = SCM_obj_to_int(index1);
   XFontStruct *font = (XFontStruct *) SCM_obj_to_vect(vect1)[i];
   Atom atom = (Atom) SCM_obj_to_int(fxatom);

   if (XGetFontProperty(font, atom, &val)) {
      j = SCM_obj_to_int(index2);
      SCM_obj_to_vect(vect2)[j] = SCM_int_to_obj(val);
      return SCM_true;
   } else {
      return SCM_false;
   }
}

SCM_obj xDselectDinput( vect, windows, win_index, fxmask )
SCM_obj vect;
SCM_obj windows;
SCM_obj win_index;
SCM_obj fxmask;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(win_index);
   Window win = (Window) SCM_obj_to_vect(windows)[i];
   long mask = (long) SCM_obj_to_int(fxmask);
   XSelectInput(dpy, win, mask);
   
   return SCM_undef;
}

SCM_obj xDrefreshDkeyboardDmapping( vect, index )
SCM_obj vect;
SCM_obj index;
{
   int i = SCM_obj_to_int(index);
   XEvent *event = (XEvent *) SCM_obj_to_vect(vect)[i];
   XRefreshKeyboardMapping(&event);

   return SCM_undef;
}

SCM_obj xDcreateDregion( regionv, rindex )
SCM_obj regionv;
SCM_obj rindex;
{
   /* Create an empty region and store it at rindex in regionv. */

   int i = SCM_obj_to_int(rindex);
   Region region = XCreateRegion();
   SCM_obj_to_vect(regionv)[i] = (long) region;

   return SCM_undef;
}

SCM_obj xDdestroyDregion( regionv, rindex )
SCM_obj regionv;
SCM_obj rindex;
{
   int i = SCM_obj_to_int(rindex);
   Region region = (Region) SCM_obj_to_vect(regionv)[i];
   XDestroyRegion(region);

   return SCM_undef;
}

SCM_obj xDunionDrectDwithDregion( regionv, rindex, x, y, width, height )
SCM_obj regionv;
SCM_obj rindex;
SCM_obj x;
SCM_obj y;
SCM_obj width;
SCM_obj height;
{
   XRectangle rect;
   int i = SCM_obj_to_int(rindex);
   Region region = (Region) SCM_obj_to_vect(regionv)[i];
   rect.x = SCM_obj_to_int(x);
   rect.y = SCM_obj_to_int(y);
   rect.width = SCM_obj_to_int(width);
   rect.height = SCM_obj_to_int(height);
   XUnionRectWithRegion(&rect, region, region);
   
   return SCM_undef;
}

SCM_obj xDsetDregion( vect, gcv, gindex, regionv, rindex )
SCM_obj vect, gcv, gindex, regionv, rindex;
{
   Display *dpy = (Display *) SCM_obj_to_vect(vect)[DISP_INDEX];
   int i = SCM_obj_to_int(gindex);
   int j = SCM_obj_to_int(rindex);
   GC gc = (GC) SCM_obj_to_vect(gcv)[i];
   Region region = (Region) SCM_obj_to_vect(regionv)[j];
   XSetRegion(dpy, gc, region);

   return SCM_undef;
}

/* NEW ----------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/

/*
 * This section defines the list of procedures to export to the Scheme
 * world.  It must contain: a link initialization procedure and a flag
 * that indicates that this is a C file (as opposed to a Scheme object file).
 * The name of the (long) flag is the name of this file, prefixed with
 * 'link_sizeof_ofile_'.  The name of the (void) init procedure is the
 * name of this file, prefixed with 'link_ofile_'.
 *
 */


long link_sizeof_ofile_gxlib = -1; /* indicate this is a C file */


void link_ofile_gxlib()
{
   DEFINE_C_PROC(xDconnectionDnumber);
   DEFINE_C_PROC(xDdefaultDscreen);
   DEFINE_C_PROC(xDdefaultDrootDwindow);
   DEFINE_C_PROC(xDblackDpixel);
   DEFINE_C_PROC(xDwhiteDpixel);
   DEFINE_C_PROC(xDopenDdisplay);
   DEFINE_C_PROC(xDcloseDdisplay);
   DEFINE_C_PROC(xDsynchronize);
   DEFINE_C_PROC(xDmapDsubwindows);
   DEFINE_C_PROC(xDmapDraised);
   DEFINE_C_PROC(makeDxDevent);
   DEFINE_C_PROC(xDeventDtype);
   DEFINE_C_PROC(xDeventDwindow);
   DEFINE_C_PROC(xDeventDwindowDindex);
   DEFINE_C_PROC(xDbuttonDeventDx);
   DEFINE_C_PROC(xDbuttonDeventDy);
   DEFINE_C_PROC(xDeventDexposeDcount);
   DEFINE_C_PROC(xDnextDevent);
   DEFINE_C_PROC(xDcreateDsimpleDwindow);
   DEFINE_C_PROC(xDdestroyDwindow);
   DEFINE_C_PROC(xDsetDiconDname);
   DEFINE_C_PROC(xDstoreDname);
   DEFINE_C_PROC(xDsetDnormalDhints);
   DEFINE_C_PROC(makeDxDsizeDhints);
   DEFINE_C_PROC(setDxDsizeDhintsDminDsizeB);
   DEFINE_C_PROC(setDxDsizeDhintsDmaxDsizeB);
   DEFINE_C_PROC(setDxDsizeDhintsDflagsB);
   DEFINE_C_PROC(xDlookupDstring);
   DEFINE_C_PROC(xDloadDqueryDfont);
   DEFINE_C_PROC(xDtextDwidth);
   DEFINE_C_PROC(xDclearDwindow);
   DEFINE_C_PROC(xDmoveDwindow);
   DEFINE_C_PROC(xDcreateDgc);
   DEFINE_C_PROC(makeDxDgcDvalues);
   DEFINE_C_PROC(xDsetDforeground);
   DEFINE_C_PROC(xDsetDbackground);
   DEFINE_C_PROC(xDsetDfont);
   DEFINE_C_PROC(xDdrawDstring);
   DEFINE_C_PROC(xDdrawDrectangle);
   DEFINE_C_PROC(xDdrawDline);
   DEFINE_C_PROC(xDdrawDimageDstring);
   DEFINE_C_PROC(xDgetDfontDproperty);
   DEFINE_C_PROC(xDselectDinput);
   DEFINE_C_PROC(xDrefreshDkeyboardDmapping);
   DEFINE_C_PROC(cDfree);
   DEFINE_C_PROC(xDfreeDgc);
   DEFINE_C_PROC(xDcreateDregion);
   DEFINE_C_PROC(xDdestroyDregion);
   DEFINE_C_PROC(xDunionDrectDwithDregion);
   DEFINE_C_PROC(xDsetDregion);
}

/*---------------------------------------------------------------------------*/
