/*****************************************************************************
                         HyperScheme 1.0beta

    ELK-Scheme wrappers for HyperNeWS interface primitives.
    This file defines HyperNeWS object types and functions, and makes
    them transparent (and opaque where appropriate) to ELK-Scheme.

    Russell Ritchie, <russell@uk.co.igc>, Mon Nov 19 12:54:00 1990.
******************************************************************************/

/* Description:
 * 	A C-Interface between ELK-Scheme and HyperNeWS.
 *	All communication to and from HyperNeWS goes through the
 *	hn_any data-structure from "../hypernews.h". The 'hnscm_get_*'
 *	and 'hnscm_put_*' routines build/read this data structure from
 *	scheme using the hnscm_stack. All other interface routines use this
 *	stack to communicate their arguments/results from and to ELK-Scheme.
 */

#include <hypernews.h>		/* For the HyperNeWS definitions. */
#include <scheme.h>		/* For the ELK-Scheme definitions. */
#include "../util/string.h"	/* For Make_C_String and friends. */

#define HN_OBJECT_MAX hn_nr_object_types+1
char *hn_object_table[HN_OBJECT_MAX] =
{"any","stack","background","card","button","edittext","slider","pulldown","colorselect","canvas"};


/**********************************************************************
 * The HyperNeWS<=>ELK-Scheme Stack and it's operations.
 */

#define MAXDEPTH	64

hn_any *hnscm_stack[MAXDEPTH];
hn_any **hnscm_sp;

void hnscm_stack_init ()
{
  register int i;
  for (i=0; i<MAXDEPTH; i++)
    hnscm_stack[i] = (hn_any *) 0;
  hnscm_sp = hnscm_stack;
}

#define hnscm_push(any) (*hnscm_sp++ = (any))
#define hnscm_pop() ((hnscm_sp > hnscm_stack) ? *--hnscm_sp : NULL)

/**********************************************************************
 * Getting data from scheme onto the stack.
 */

Object P_Hnscm_put_null ()
     /* Push a HyperNeWS null onto stack. Returns #t if this was achieved.  */
{
  hnscm_push(hn_null);
  return True;
}

Object P_Hnscm_put_boolean (val) Object val;
     /* Push a HyperNeWS BOOL onto stack. Returns #t if this was achieved.  */
{
  Check_Type (val, T_Boolean);
  hnscm_push(hn_new_boolean(FIXNUM (val)));
  return True;
}

Object P_Hnscm_put_integer (val) Object val;
     /* Push a HyperNeWS INT onto stack. Returns #t if this was achieved.  */
{
  Check_Integer (val);
  hnscm_push(hn_new_integer(Get_Integer (val)));
  return True;
}

Object P_Hnscm_put_float (val) Object val;
     /*  Push a HyperNeWS FLOAT onto stack. Returns #t if this was achieved. */
{
  Check_Number (val);
  hnscm_push(hn_new_number(FLONUM (val)->val));
  return True;
}

Object P_Hnscm_put_string (val) Object val;
     /* Push a HyperNeWS STRING onto stack. Returns #t if this was achieved. */
{
  register char *s;
  Declare_C_Strings;

  Make_C_String (val, s);
  hnscm_push(hn_new_string(s));
  Dispose_C_Strings;
  return True;
}

Object P_Hnscm_put_name (val) Object val;
     /*  Push a HyperNeWS NAME onto stack. Returns #t if this was achieved. */
{
  register char *s;
  Declare_C_Strings;

  Make_C_String (val, s);
  hnscm_push(hn_new_name(s));
  Dispose_C_Strings;
  return True;
}

Object P_Hnscm_put_message (name) Object name;
     /* Push HyperNeWS MESSAGE onto stack. Returns #t if this was achieved. */
{
  hn_any *target = hnscm_pop();
  hn_any *args = hnscm_pop();
  register char *s;
  Declare_C_Strings;

  Make_C_String (name, s);
  hnscm_push(hn_new_message(target, s, args));
  Dispose_C_Strings;
  return True;
}

Object P_Hnscm_put_object (stack, type, name) Object stack, type, name;
     /* Push STACK, object TYPE and NAME. Return #t if this was achieved. */
{
  register char *s, *n;
  Declare_C_Strings;

  Make_C_String (stack, s);
  Check_Integer (type);
  Make_C_String (name, n);
  hnscm_push (hn_new_object(s, Get_Integer (type), n));
  Dispose_C_Strings;
  return True;
}

Object P_Hnscm_put_stack (name) Object name;
     /*  Push stack NAME onto stack. Return #t if this was achieved.  */
{
  register char *s;
  Declare_C_Strings;

  Make_C_String (name, s);
  hnscm_push(hn_new_object(NULL,hn_Stack,s));
  Dispose_C_Strings;
  return True;
}

Object P_Hnscm_put_array (len) Object len;
     /*  Push empty array[LEN] onto stack. Return #T if this was achieved.  */
{
  Check_Integer (len);
  hnscm_push(hn_new_array_body(Get_Integer (len)));
  return True;
}

Object P_Hnscm_put_array_elt (n) Object n;
     /*  Having stacked array/elt, make array[N] = elt. Return #T if ok. */
{
  hn_any *any = hnscm_pop();
  hn_any *array = hnscm_pop();

  Check_Integer (n);
  array->u.array.argv[Get_Integer (n)] = any;
  hnscm_push(array);
  return True;
}

/**********************************************************************
 * Getting data from the stack into scheme.
 */

Object hnscm_get_integer ()
{
  Object val;
  hn_any *any = hnscm_pop();

  if (any) {
    val = Make_Integer ((int)any->u.number);
    hn_free(any);
  }
  else
    return Null;
  return val;
}

Object hnscm_get_float ()
{
  Object val;
  hn_any *any = hnscm_pop();

  if (any) {
    val = Make_Reduced_Flonum ((double)any->u.number); 
    hn_free(any);
  }
  else
    return Null;
  return val;
}

Object hnscm_get_string ()
{
  Object val;
  hn_any *any = hnscm_pop();

  if (any) {
    val = Make_String (any->u.string, strlen (any->u.string));
    hn_free(any);
    }
  else
    return Null;
  return val;
}

Object hnscm_get_name ()
{
  Object val;
  hn_any *any = hnscm_pop();

  if (any) {
    val = Make_String (any->u.string, strlen (any->u.string));
    hn_free(any);
  } 
  else
    return Null;
  return val;
}

Object hnscm_get_message ()
{
  Object val;
  hn_any *any = hnscm_pop();

  if (any) {
    val = Make_String (any->u.message.name, strlen (any->u.message.name));
    hn_free(any);
  } 
  else
    return Null;
  return val;
}

Object hnscm_get_array_element (n)
int n;
{
  hn_any *any = hnscm_pop();

  if (any) {
    if (n < any->u.array.argc) {
      hnscm_push(any);
      hn_use(any->u.array.argv[n]);
      hnscm_push(any->u.array.argv[n]);
      return True;
    }
    else {
      hn_free(any);
      return Null;
    }}
  else
    return Null;
}

Object P_Hnscm_obj_type_name (match) Object match;
     /*  Map a HyperNeWS object to it's type number, or vice versa.  */
{
  register t = TYPE(match);
  register int i = 0;

  if ( t == T_Fixnum || t == T_Bignum ) {
    i = Get_Integer (match);
    if (i >= 0 && i <= hn_nr_object_types)
      return Make_String(hn_object_table[i], strlen (hn_object_table[i]));
    else
      return Null;
  }
  else
    if ( t == T_String ) {
      register char *s;
      register int done = 0;
      Object result = Null;
      Declare_C_Strings;

      Make_C_String (match, s);
      while ((! done) && (i < HN_OBJECT_MAX)) {
	/*  	for (i=0; i<HN_OBJECT_MAX; i++)  */
  	if (strcmp(s, hn_object_table[i]) == 0) {
	  done = 1;
	  result = Make_Integer(i);
	}
	i++;
      }
      Dispose_C_Strings;
      return result;
    }
    else
      return Null;
}

int hnscm_type_to_number (type)
     hn_type type;
{
  switch (type) {
  case hn_Stack:
    return 1;
  case hn_BackGround:
    return 2;
  case hn_Card:
    return 3;
  case hn_Button:
    return 4;
  case hn_EditText:
    return 5;
  case hn_Slider:
    return 6;
  case hn_PullDown:
    return 7;
  case hn_ColorSelect:
    return 8;
  case hn_Canvas:
    return 9;
  default:
    return 0;
  }
}

Object hnscm_get_object()
{  
  hn_any *any = hnscm_pop();  
       
  if (any) {  
    Object stack, name, objname, typenr, list_args[2], result;
    register char *s;
    Declare_C_Strings;
    
    if (!any->u.object.stack)
      stack = Make_String ("", 0);
    else
      stack = Make_String (any->u.object.stack, strlen (any->u.object.stack));
    name = Make_String (any->u.object.name, strlen (any->u.object.name));  
    typenr = Make_Integer (hnscm_type_to_number((int)any->u.object.type));
    hn_free(any);  
    objname = P_Hnscm_obj_type_name (typenr);
    list_args[0] = typenr;
    list_args[1] = name;

    Make_C_String (objname, s);
    if (strcmp (s, "stack") == 0) 
      result = P_List (2, list_args);
    else {
      register Object stacklist_args[2];

      stacklist_args[0] = stack;
      stacklist_args[1] = P_List (2, list_args);
      result = P_List (2, stacklist_args);
    }
    Dispose_C_Strings;
    return result;
  }
  return Null;
}  

Object hnscm_get_type ()
{
  hn_any *any = hnscm_pop();

  if (any) {
    switch ((int)any->type) {
    case hn_null_type:
      return Make_String ("null", 4);
    case hn_number_type:
      hnscm_push(any);
      if (any->u.number == (int)any->u.number)
	return Make_String ("integer", 7);
      else
	return Make_String ("float", 5);
    case hn_boolean_type:
      if (any->u.boolean)
	return Make_String ("true", 4);
      else
	return Make_String ("false", 5);
    case hn_string_type:
      hnscm_push(any);
      return Make_String ("string", 6);
    case hn_name_type:
      hnscm_push(any);
      return Make_String ("name", 4);
    case hn_message_type:
      hn_use(any->u.message.args);
      hnscm_push(any->u.message.args);
      hn_use(any->u.message.target);
      hnscm_push(any->u.message.target);
      hnscm_push(any);
      return Make_String ("message", 7);
    case hn_array_type:
      hnscm_push(any);
      return Make_String ("array", 5);
    case hn_object_type:
      hnscm_push(any);
      return Make_String ("object", 6);
    default:
      return Null;
    }}
  else 
    return Null;
}

/* The ELK-Scheme interface functions start here... */

Object P_Hnscm_start (client) Object client;
     /* Try to start a HyperNeWS connection for CLIENT. */
     /* Returns #t if the connection was established, #f otherwise. */
{
  register char *s;
  Object result;
  Declare_C_Strings;
    
  hnscm_stack_init();
  Make_C_String (client, s);
  if (hn_start(s))
    result = True;
  else
    result = Null;
  Dispose_C_Strings;
  return result;
}

Object P_Hnscm_stop () 
     /*  Disconnect from HyperNeWS. Returns #t if this was achieved."  */
{
  if (hn_stop())
    return True;
  else
    return Null;
}

Object P_Hnscm_flush () 
     /*  Flush output from HyperNeWS. Returns #t when finished.  */
{
  hn_flush();
  return True;
}

Object P_Hnscm_flush_input () 
     /*  Flush pending input to HyperNeWS. Returns #t when finished.  */
{
  hn_flush_input();
  return True;
}

Object P_Hnscm_ok () 
     /*  Check connection to HyperNeWS. Returns #t if everything's groovy.  */
{
  if (hn_ok())
    return True;
  else
    return Null;
}

Object P_Hnscm_verbose (verbose) Object verbose;
     /*  If ARG is nil/non-nil, tell HyperNeWS to shut/speak up, returns ARG.*/
{
  if (Nullp (verbose))
    hn_verbose = 0;
  else
    hn_verbose = 1;
  return verbose;
}

Object P_Hnscm_clear ()
     /*  Clear the stack. Returns #t if everything's groovy.  */
{
  hnscm_sp = hnscm_stack;
  if (hn_running)
    return True;
  else
    return Null;
}

Object P_Hnscm_ps (postscript) Object postscript;
     /* Send a string of raw (and I mean *raw*) PostScript to HyperNeWS. */
     /* Returns #t if the HyperNeWS ate the string, nil otherwise."  */
{
  register char *s;
  Object result;
  Declare_C_Strings;

  Make_C_String (postscript, s);
  if (hn_ok()) {
    hn_ps(" ");
    if (hn_ps(s)) {
      hn_ps("\n");
      hn_flush();
      result = True;
    } else
      result = Null;
  } else
    result = Null;
  Dispose_C_Strings;
  return result;
}

Object P_Hnscm_write () 
     /*  Write top of stack to HyperNeWS. Returns #t if everything's groovy.  */
{
  hn_any *any = hnscm_pop();
  Object result;

  if (hn_ok()) {
    hn_write(any);
    hn_flush();
    result = True;
  } else
    result = Null;
  hn_free(any);
  return result;
}

Object P_Hnscm_read (delay) Object delay;
     /*  Read from HyperNeWS stack. Returns #t if everything's groovy.  */
{
  Check_Integer (delay);
  if (hn_ok()) {
    hnscm_sp = hnscm_stack;
    hnscm_push(hn_read(Get_Integer (delay)));
    return True;
  } else
    return Null;
}

Object P_Hnscm_put_any (param) Object param;
     /*  Put string PARAM on HyperNeWS stack. Returns #t if all's well.  */
{
  Object result;
  hn_any *target = hnscm_pop();
  hn_any *value = hnscm_pop();
  register char *s;
  Declare_C_Strings;

  Make_C_String (param, s);
  if (hn_ok()) {
    if (hn_put(target, s, value)) {
      hn_flush();
      result = True;
    }else
      result = Null;
  } else
    result = Null;
  hn_free(target);
  hn_free(value);
  Dispose_C_Strings;
  return result;
}

Object P_Hnscm_get_any (param) Object param;
     /*  Get string PARAM from HyperNeWS stack. Returns #t if all's well.  */
{
  Object result;
  hn_any *target = hnscm_pop();
  register char *s;
  Declare_C_Strings;

  Make_C_String (param, s);
  if (hn_ok()) {
    hnscm_push(hn_get(target, s));
    result = True;
  } else
    result = Null;
  hn_free(target);
  Dispose_C_Strings;
  return result;
}

Object P_Hnscm_show ()
     /*  Show the top element of stack. Returns #t if everything's groovy.  */
{
  Object result;
  hn_any *target = hnscm_pop();

  if (hn_ok()) {
    if (hn_show(target)) {
      hn_flush();
      result = True;
    } else
      result = Null;
  } else
    result = Null;
  hn_free(target);
  return result;
}

Object P_Hnscm_hide () 
     /*  Hide the top element of stack. Returns #t if everything's groovy.  */
{
  Object result;
  hn_any *target = hnscm_pop();

  if (hn_ok()) {
    if (hn_hide(target)) {
      hn_flush();
      result = True;
    } else
      result = Null;
  } else
    result = Null;
  hn_free(target);
  return result;
}

Object P_Hnscm_exists ()
     /*  Returns #t if the top element of the stack exists.  */
{
  Object result;
  hn_any *target = hnscm_pop();

  if (hn_ok()) {
    if (hn_exists(target)) {
      result = True;
    } else
      result = Null;
  } else
    result = Null;
  hn_free(target);
  return result;
}

Object P_Hnscm_set_drawing (file) Object file;
     /*  Set current Stack's drawing to FILE. Returns #t if all's well.  */
{
  Object result;
  hn_any *target = hnscm_pop();
  register char *s;
  Declare_C_Strings;

  Make_C_String (file, s);
  if (hn_ok()) {
    if (hn_set_drawing(target, s)) {
      hn_flush();
      result = True;
    } else
      result = Null;
  } else
    result = Null;
  hn_free(target);
  Dispose_C_Strings;
  return result;
}

Object P_Hnscm_rename (name) Object name;
     /*  Rename the current stack to NAME. Returns #t if all's well.  */
{
  Object result;
  hn_any *target = hnscm_pop(), *n;
  register char *s;
  Declare_C_Strings;

  Make_C_String (name, s);
  if (hn_ok()) {
    if (n = hn_rename(target, s)) {
      hnscm_push(n);
      result = True;
    } else
      result = Null;
  } else
    result = Null;
  hn_free(target);
  Dispose_C_Strings;
  return result;
}

Object P_Hnscm_stack_copy ()
     /*  Copy the top element of the stack. Returns #t if this was achieved. */
{
  Object result;
  hn_any *target = hnscm_pop(), *n;

  if (hn_ok()) {
    if (n = hn_stack_copy(target)) {
      hnscm_push(n);
      result = True;
    } else
      result = Null;
  } else
    result = Null;
  hn_free(target);
  return result;
}

extern Object P_Hnscm_get(); 

Object hnscm_get_array (n)
     int n;
{
  if (Nullp (hnscm_get_array_element (n)))
    return Null;
  else {
    Object list_arg[1], append_args[2];

    list_arg[0] = P_Hnscm_get (hnscm_get_type ());
    append_args[0] = P_List (1, list_arg);
    append_args[1] = hnscm_get_array (++n);
    return P_Append (2, append_args);
   }
}

Object P_Hnscm_get (type) Object type;
     /* Get object of TYPE from top of the stack. Returns #t if all's well. */
{
  if (Nullp (type)) {
    type = hnscm_get_type ();
    if (Nullp (type))
      return Null;
    return P_Hnscm_get (type);
  }
  else {
    register char *s;
    Object result;
    Declare_C_Strings;

    Make_C_String (type, s);
    if (strcmp (s, "message") == 0) {
      Object msg = hnscm_get_message ();
      if (Nullp (msg))
	result = Null;
      else {
	Object fromtype = hnscm_get_type ();
	if (Nullp (fromtype))
	  result = Null;
	else {
	  Object from = P_Hnscm_get (fromtype);
	  Object argstype = hnscm_get_type ();
	  if (Nullp (argstype))
	    result = Null;
	  else {
	    Object list_args[3];

	    list_args[0] = from;
	    list_args[1] = msg;
	    list_args[2] = P_Hnscm_get (argstype);
	    result = P_List (3, list_args);
	}}}}
    else 
      if (strcmp (s, "array") == 0) 
	result = hnscm_get_array (0);
      else
	if (strcmp (s, "object") == 0) 
	  result = hnscm_get_object ();
	else
	  if (strcmp (s, "true") == 0)
	    result = Make_String ("true", 4);
	  else
	    if (strcmp (s, "false") == 0)
	      result = Make_String ("false", 5);
	    else
	      if (strcmp (s, "null") == 0)
		result = Make_String ("null", 4);
	      else
		if (strcmp (s, "integer") == 0)
		  result = hnscm_get_integer ();
		else
		  if (strcmp (s, "float") == 0)
		    result = hnscm_get_float ();
		  else	 
		    if (strcmp (s, "string") == 0)
		      result = hnscm_get_string ();
		    else
		      if (strcmp (s, "name") == 0)
			result = hnscm_get_name ();
    Dispose_C_Strings;
    return result;
  }
}

init_lib_HyperNeWS () {
  Define_Primitive (P_Hnscm_put_null, "hnscm-put-null", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_put_boolean, "hnscm-put-boolean", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_integer, "hnscm-put-integer", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_float, "hnscm-put-float", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_string, "hnscm-put-string", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_name, "hnscm-put-name", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_message, "hnscm-put-message", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_object, "hnscm-put-object", 3, 3, EVAL);
  Define_Primitive (P_Hnscm_put_stack, "hnscm-put-stack", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_array, "hnscm-put-array", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_array_elt, "hnscm-put-array-elt", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_obj_type_name, "hnscm-obj-type-name", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_start, "hnscm-start", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_stop, "hnscm-stop", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_flush, "hnscm-flush", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_flush_input, "hnscm-flush-input", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_ok, "hnscm-flush-ok", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_verbose, "hnscm-verbose", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_clear, "hnscm-clear", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_ps, "hnscm-ps", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_write, "hnscm-write", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_read, "hnscm-read", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_put_any, "hnscm-put-any", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_get_any, "hnscm-get-any", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_show, "hnscm-show", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_hide, "hnscm-hide", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_exists, "hnscm-exists", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_set_drawing, "hnscm-set-drawing", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_rename, "hnscm-rename", 1, 1, EVAL);
  Define_Primitive (P_Hnscm_stack_copy, "hnscm-stack-copy", 0, 0, EVAL);
  Define_Primitive (P_Hnscm_get, "hnscm-get", 1, 1, EVAL);

  P_Provide (Intern ("HyperNeWS.o"));
}

