; -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; File node.scm / Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Node abstraction

;+++ Make it abstract at some point.

; Standard type order (8):
;  constant variable LAMBDA LETREC IF BEGIN SET! call

(define (node? obj)
  (and (vector? obj)
       (>= (vector-length obj) 1)
       (memq (vector-ref obj 0)
	     '(constant variable lambda letrec if begin set! call))))

(define (node-type node)
  (vector-ref node 0))

(define (node-predicate type)
  (lambda (node)
    (eq? (node-type node) type)))

(define (node-accessor type index)
  (lambda (node)
    (if (not (eq? (node-type node) type))
	(error "wrong node type" node type))
    (vector-ref node index)))

(define (node-updater type index)
  (lambda (node new-val)
    (if (not (eq? (node-type node) type))
	(error "wrong node type" node type))
    (vector-set! node index new-val)))

; Constant

(define (make-constant val)
  (vector 'constant val))

(define constant? (node-predicate 'constant))

(define constant-value (node-accessor 'constant 1))

; LAMBDA

(define (make-lambda vars body-node)
  (vector 'lambda vars body-node))

(define lambda? (node-predicate 'lambda))

(define lambda-vars (node-accessor 'lambda 1))
(define lambda-body (node-accessor 'lambda 2))

(define (n-ary? proc)
  (not (proper-list? (lambda-vars proc))))

(define (proper-list? thing)
  (or (null? thing)
      (and (pair? thing)
	   (null? (cdr (last-pair thing))))))

(define (proper-listify thing)
  (cond ((null? thing) '())
	((pair? thing) (cons (car thing) (proper-listify (cdr thing))))
	(else (list thing))))

(define (map-bvl proc bvl)
  (cond ((null? bvl) '())
	((pair? bvl)
	 (cons (proc (car bvl)) (map-bvl proc (cdr bvl))))
	(else (proc bvl))))

; LETREC

(define (make-letrec vars val-nodes body-node)
  (vector 'letrec vars val-nodes body-node #f))

(define letrec? (node-predicate 'letrec))

(define letrec-vars (node-accessor 'letrec 1))
(define letrec-vals (node-accessor 'letrec 2))
(define letrec-body (node-accessor 'letrec 3))
(define letrec-strategy (node-accessor 'letrec 4))

(define set-letrec-strategy! (node-updater 'letrec 4))

; IF

(define (make-if test con alt)
  (vector 'if test con alt))

(define if? (node-predicate 'if))

(define if-test (node-accessor 'if 1))
(define if-con  (node-accessor 'if 2))
(define if-alt  (node-accessor 'if 3))

; BEGIN

(define (make-begin first second)
  (vector 'begin first second))
(define begin? (node-predicate 'begin))
(define begin-first  (node-accessor 'begin 1))
(define begin-second (node-accessor 'begin 2))

; SET!

(define (make-set! lhs rhs)
  (vector 'set! lhs rhs))
(define set!? (node-predicate 'set!))
(define set!-lhs (node-accessor 'set! 1))
(define set!-rhs (node-accessor 'set! 2))

; Call

(define (make-call proc-node arg-nodes)
  (vector 'call proc-node arg-nodes))

(define call? (node-predicate 'call))
(define call-proc (node-accessor 'call 1))
(define call-args (node-accessor 'call 2))

; Definition

(define (make-define lhs rhs)
  (vector 'define lhs rhs))
(define define? (node-predicate 'define))
(define define-lhs (node-accessor 'define 1))
(define define-rhs (node-accessor 'define 2))

; Variable (global or local)

(define (make-variable uname status path)
  (vector 'variable
	  uname				;1 user's name
	  status			;2 status
	  #f				;3 substitution
	  path				;4 path (for globals only)
	  ;; The following fields are only used for local variables.
	  ;; (Good application for variant records??)
	  #f				;5 value-refs?
	  #f				;6 proc-refs?
	  #f				;7 assigned?
	  #f				;8 closed-over?
	  ))

; status:  LOCAL   means lambda- or letrec-bound
;	   FREE    means unbound (may be altered to be DEFINED)
;	   DEFINED means defined in top-level env

(define variable? (node-predicate 'variable))

(define variable-name         (node-accessor 'variable 1))
(define variable-status       (node-accessor 'variable 2))
(define variable-substitution (node-accessor 'variable 3))

(define set-status!       (node-updater 'variable 2))
(define set-substitution! (node-updater 'variable 3))

(define variable-path         (node-accessor 'variable 4))
(define variable-value-refs?  (node-accessor 'variable 5))
(define variable-proc-refs?   (node-accessor 'variable 6))
(define variable-assigned?    (node-accessor 'variable 7))
(define variable-closed-over? (node-accessor 'variable 8))

(define (set-value-refs!  var) ((node-updater 'variable 5) var #t))
(define (set-proc-refs!   var) ((node-updater 'variable 6) var #t))
(define (set-assigned!    var) ((node-updater 'variable 7) var #t))
(define (set-closed-over! var) ((node-updater 'variable 8) var #t))

; Locals

(define (make-local-variable uname)
  (make-variable uname 'local #f))

(define (local-variable? obj)
  (and (variable? obj)
       (eq? (variable-status obj) 'local)))

; Globals

(define (make-global-variable name path)
  (make-variable name 'free path))

(define (global-variable? obj)
  (and (variable? obj)
       (not (eq? (variable-status obj) 'local))))

(define global-variable-name variable-name)
