;;; Copyright (c) 1993 by Olin Shivers.
;;; Job control code.

;;; Fork off a process that runs in its own process group. The process
;;; is (1) placed in its own process group and (2) suspended before
;;; the process's actual work code is executed, and before FORK-JOB
;;; returns to the parent. The next time the job is resumed, it will
;;; begin its actual work.
(define (fork-job . maybe-thunk)
  (let ((child (fork)))

    (cond ((zero? child) ; CHILD
	   (let lp ()	; Block until we are put in our own proc group.
	     (signal-process 0 signal/stop)
	     (if (not (= (pid) (process-group))) (lp)))

	   (if (pair? maybe-thunk)
	       (call-terminally (car maybe-thunk))))

	  (else
	   ;; PARENT -- wait for child to stop and then set its proc group.
	   (let ((status (wait child wait/stopped-children)))
	     (if (not (status:stop-sig status)) ; Make sure it didn't die.
		 (error "premature job death" status))) ; error call right?
	   (set-process-group child child)))

    child))


;;; Foreground a suspended job.
(define (resume-job proc-group)
  (set-terminal-proc-group stdin proc-group)
  (signal-process-group proc-group signal/cont)
  (wait-job proc-group))				; Should we wait here?

;;; The guy that waits for the job should maybe take responsibility for
;;; taking the tty back.

;;; Background a suspended job.
(define (background-job proc-group)
  (signal-process-group proc-group signal/cont)
  proc-group)


(define-simple-syntax (run . epf)
  (resume-job fork-job (lambda () (exec-epf . epf))))

(define-simple-syntax (& . epf)
  (background-job (fork-job (lambda () (exec-epf . epf)))))


;;; Need repl loop that manages some kind of a job table, 
;;; and grabs the terminal back after running a job.
;;; A lot of this is managed by the WAIT-JOB procedure, which needs defined.
;;; Need a (CONTROL-TTY) procedure?



;;; (wait pid [flags]) => status
;;; (wait-any [flags]) => [pid status]
;;; (wait-proc-group [pgrp flags]) => [pid status]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; The wait interface:
;;; If PGRP is zero, it means the process' own process group.
;;; POSIX flags: wait/stopped-children, wait/poll
;;; If wait/poll is specified, these procedures return #f and [#f _]
;;; instead of blocking.
;;;
;;; Returned status values can be decoded with these procedures:
;;;     (status:exit-val status)	
;;;     (status:term-sig status) 
;;;     (status:stop-sig status)  
;;; These procedures serve as predicates, returning true if the child process
;;; exited, terminated due to a signal, or was suspended. In addition, the
;;; particular true value returned is the process' exit code, terminating 
;;; signal, or suspending signal.

